#!/usr/bin/env python3
# encoding: utf-8
#
# ** header v3.0
# This file is a part of the CaosDB Project.
#
# Copyright (C) 2022 Alexander Schlemmer <alexander.schlemmer@ds.mpg.de>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program. If not, see <https://www.gnu.org/licenses/>.
#
# ** end header
#

"""
test the converters module
"""

from caoscrawler.converters import Converter
from caoscrawler.stores import GeneralStore
from caoscrawler.converters import (ConverterValidationError,
                                    DictConverter, XLSXTableConverter, CSVTableConverter)
from caoscrawler.structure_elements import Directory
from caoscrawler.structure_elements import (File, DictTextElement,
                                            DictListElement, DictElement,
                                            DictBooleanElement, DictDictElement,
                                            DictIntegerElement, DictFloatElement)

from os.path import join, dirname, basename

from caoscrawler.identifiable_adapters import IdentifiableAdapter, LocalStorageIdentifiableAdapter

import pytest
import os
import importlib

import math

from caoscrawler import Crawler

import caosdb as db


@pytest.fixture
def converter_registry():
    converter_registry: dict[str, dict[str, str]] = {
        "Directory": {
            "converter": "DirectoryConverter",
            "package": "caoscrawler.converters"},
        "CSVTableConverter": {
            "converter": "CSVTableConverter",
            "package": "caoscrawler.converters"},
        "XLSXTableConverter": {
            "converter": "XLSXTableConverter",
            "package": "caoscrawler.converters"},

        "DictDictElement": {
            "converter": "DictDictElementConverter",
            "package": "caoscrawler.converters"},
        "DictTextElement": {
            "converter": "DictTextElementConverter",
            "package": "caoscrawler.converters"},
        "DictIntegerElement": {
            "converter": "DictIntegerElementConverter",
            "package": "caoscrawler.converters"},
        "DictFloatElement": {
            "converter": "DictFloatElementConverter",
            "package": "caoscrawler.converters"},
    }


def rfp(*pathcomponents):
    """
    Return full path.
    Shorthand convenience function.
    """
    return join(dirname(__file__), *pathcomponents)


def dircheckstr(*pathcomponents):
    """
    Return the debug tree identifier for a given path.
    """
    return "caoscrawler.structure_elements.File: " + basename(join(*pathcomponents)) + ", " + rfp("test_directories", "examples_tables", "ExperimentalData", *pathcomponents)


@pytest.fixture
def crawler():
    crawler = Crawler(debug=True)
    crawler.crawl_directory(rfp("test_directories", "examples_tables", "ExperimentalData"),
                            rfp("test_directories", "examples_tables", "crawler_for_tables.yml"))
    return crawler


def test_convert_table(converter_registry):
    extentions = ["xlsx", "csv", "tsv"]
    if importlib.util.find_spec("odf") is not None:
        extentions.append("ods")
    for file_ext in extentions:
        def_opt = {"skiprows": ["1", "2"], "header": 0}
        if file_ext == "tsv":
            def_opt["sep"] = "\t"
        if file_ext in ["csv", "tsv"]:
            converter = CSVTableConverter(
                def_opt,
                "Tab",
                converter_registry)
        else:
            converter = XLSXTableConverter(
                def_opt,
                "Tab",
                converter_registry)
        store = GeneralStore()
        file_element = File("table." + file_ext,
                            rfp("test_tables", "test1." + file_ext))
        res = converter.create_children(store,
                                        file_element)
        assert len(res) == 5
        for i in range(5):
            assert res[i].name == str(i)
            assert type(res[i].name) == str
            assert type(res[i].value) == dict
            assert len(res[i].value) == 4
            assert type(res[i].value["Col_1"]) == int
            assert res[i].value["Col_1"] == i
            assert type(res[i].value["Col_2"]) == float
            assert type(res[i].value["Col_3"]) == int
            if i != 3:
                assert type(res[i].value["text"]) == str
            else:
                assert type(res[i].value["text"]) == float  # the nan value
                assert math.isnan(res[i].value["text"])

    # Using an index col:
    converter = XLSXTableConverter(
        {"skiprows": ["1", "2"], "header": 0, "index_col": "3"},
        "XLSXTable",
        converter_registry)
    store = GeneralStore()
    file_element = File("table.xlsx",
                        rfp("test_tables", "test1.xlsx"))
    res = converter.create_children(store,
                                    file_element)
    assert res[0].name == "jdsfkljadskf"


def test_crawl_csv_table(crawler):
    for file_ext in ["xlsx", "csv"]:
        subd = crawler.debug_tree[dircheckstr("test1." + file_ext)]
        record_experiment = subd[1]["Experiment"]
        assert isinstance(record_experiment, db.Record)
        assert isinstance(record_experiment.get_property("Measurements").value, list)
        assert len(record_experiment.get_property("Measurements").value) == 5
        prop_measure = record_experiment.get_property("Measurements").value[2]
        assert isinstance(prop_measure, db.Record)
        assert prop_measure.get_property("Col_1").value == "2"
