#!/usr/bin/env python3
# encoding: utf-8
#
# ** header v3.0
# This file is a part of the CaosDB Project.
#
# Copyright (C) 2021 Indiscale GmbH <info@indiscale.com>
# Copyright (C) 2021 Henrik tom Wörden <h.tomwoerden@indiscale.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program. If not, see <https://www.gnu.org/licenses/>.
#
# ** end header
#

"""
test identifiable_adapters module
"""

import os
from datetime import datetime
from caoscrawler.identifiable_adapters import (
    CaosDBIdentifiableAdapter, IdentifiableAdapter)
import caosdb as db


def test_create_query_for_identifiable():
    query = IdentifiableAdapter.create_query_for_identifiable(
        db.Record().add_parent("Person")
        .add_property("first_name", value="A")
        .add_property("last_name", value="B"))
    assert query.lower() == "find record person with 'first_name'='a' and 'last_name'='b' "

    query = IdentifiableAdapter.create_query_for_identifiable(
        db.Record(name="A").add_parent("B")
        .add_property("c", value="c")
        .add_property("d", value=5)
        .add_property("e", value=5.5)
        .add_property("f", value=datetime(2020, 10, 10))
        .add_property("g", value=True)
        .add_property("h", value=db.Record(id=1111))
        .add_property("i", value=db.File(id=1112))
        .add_property("j", value=[2222, db.Record(id=3333)]))
    assert (query.lower() == "find record b with name='a' and 'c'='c' and 'd'='5' and 'e'='5.5'"
            " and 'f'='2020-10-10t00:00:00' and 'g'='true' and 'h'='1111' and 'i'='1112' and "
            "'j'='2222' and 'j'='3333' ")

    # The name can be the only identifiable
    query = IdentifiableAdapter.create_query_for_identifiable(
        db.Record(name="TestRecord").add_parent("TestType"))
    assert query.lower() == "find record testtype with name='testrecord'"


def test_load_from_yaml_file():
    ident = CaosDBIdentifiableAdapter()
    ident.load_from_yaml_definition(
        os.path.join(os.path.dirname(__file__), "test_directories",
                     "single_file_test_data", "identifiables.yml")
    )

    person_i = ident.get_registered_identifiable(
        db.Record().add_parent("Person"))
    assert person_i is not None
    assert person_i.get_property("full_name") is not None

    keyword_i = ident.get_registered_identifiable(
        db.Record().add_parent("Keyword"))
    assert keyword_i is not None
    assert keyword_i.get_property("name") is not None

    project_i = ident.get_registered_identifiable(
        db.Record().add_parent("Project"))
    assert project_i is not None
    assert project_i.get_property("project_id") is not None
    assert project_i.get_property("title") is not None
