package org.caosdb.server.grpc;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.util.TimeZone;
import org.caosdb.datetime.DateTimeFactory2;
import org.caosdb.server.datatype.GenericValue;
import org.caosdb.server.datatype.Value;
import org.caosdb.server.entity.EntityID;
import org.caosdb.server.entity.FileProperties;
import org.caosdb.server.entity.Message;
import org.caosdb.server.entity.RetrieveEntity;
import org.caosdb.server.entity.Role;
import org.caosdb.server.entity.StatementStatus;
import org.caosdb.server.entity.wrapper.Parent;
import org.caosdb.server.entity.wrapper.Property;
import org.caosdb.server.entity.xml.IdAndServerMessagesOnlyStrategy;
import org.junit.jupiter.api.Test;

public class CaosDBToGrpcConvertersTest {

  @Test
  public void testConvertScalarValue_Datetime() {
    TimeZone timeZone = TimeZone.getTimeZone("UTC");
    DateTimeFactory2 factory = new DateTimeFactory2(timeZone);
    CaosDBToGrpcConverters converters = new CaosDBToGrpcConverters(timeZone);
    Value value = null;
    assertNull(converters.convertScalarValue(value));
    value = factory.parse("2022");
    assertEquals(converters.convertScalarValue(value).toString(), "string_value: \"2022\"\n");
    value = factory.parse("2022-12");
    assertEquals(converters.convertScalarValue(value).toString(), "string_value: \"2022-12\"\n");
    value = factory.parse("2022-12-24");
    assertEquals(converters.convertScalarValue(value).toString(), "string_value: \"2022-12-24\"\n");
    value = factory.parse("2022-12-24T18:15:00");
    assertEquals(
        converters.convertScalarValue(value).toString(),
        "string_value: \"2022-12-24T18:15:00+0000\"\n");
    value = factory.parse("2022-12-24T18:15:00.999999");
    assertEquals(
        converters.convertScalarValue(value).toString(),
        "string_value: \"2022-12-24T18:15:00.999999+0000\"\n");
    value = factory.parse("2022-12-24T18:15:00.999999UTC");
    assertEquals(
        converters.convertScalarValue(value).toString(),
        "string_value: \"2022-12-24T18:15:00.999999+0000\"\n");
    value = factory.parse("2022-12-24T18:15:00.999999+0200");
    assertEquals(
        converters.convertScalarValue(value).toString(),
        "string_value: \"2022-12-24T16:15:00.999999+0000\"\n");
  }

  @Test
  public void testConvertEntity_FileDescriptor() {
    RetrieveEntity entity = new RetrieveEntity();
    CaosDBToGrpcConverters converters = new CaosDBToGrpcConverters(null);
    assertEquals(converters.convert(entity).toString(), "entity {\n}\n");
    entity.setFileProperties(new FileProperties("checksum1234", "the/path", 1024L));
    assertEquals(
        converters.convert(entity).toString(),
        "entity {\n  file_descriptor {\n    path: \"the/path\"\n    size: 1024\n  }\n}\n");
  }

  @Test
  public void testIdServerMessagesOnlyStrategy() {
    // @review Florian Spreckelsen 2022-03-22
    RetrieveEntity entity = new RetrieveEntity();

    // must be printed
    entity.setId(new EntityID(1234));
    entity.addInfo("info");
    entity.addWarning(new Message("warning"));
    entity.addError(new Message("error"));

    // must not be printed
    Parent par = new Parent(new RetrieveEntity());
    par.setName("dont print parent");
    entity.addParent(par);
    entity.setName("dont print");
    entity.setDescription("dont print");
    entity.setRole(Role.File);
    entity.setFileProperties(new FileProperties("dont print checksum", "dont print path", 1234L));
    Property p = new Property(new RetrieveEntity());
    p.setStatementStatus(StatementStatus.FIX);
    p.setName("dont print property");
    p.setDatatype("TEXT");
    p.setValue(new GenericValue("don print"));
    entity.addProperty(p);

    CaosDBToGrpcConverters converters = new CaosDBToGrpcConverters(null);

    // first test the normal SerializeFieldStrategy instead
    entity.setSerializeFieldStrategy(null);
    assertTrue(converters.convert(entity).toString().contains("dont print"));

    // now suppress all fields but id and server messages.
    entity.setSerializeFieldStrategy(new IdAndServerMessagesOnlyStrategy());
    assertEquals(
        converters.convert(entity).toString(),
        "entity {\n  id: \"1234\"\n}\nerrors {\n  code: 1\n  description: \"error\"\n}\nwarnings {\n  code: 1\n  description: \"warning\"\n}\ninfos {\n  code: 1\n  description: \"info\"\n}\n");
  }
}
