/*
 * ** header v3.0
 * This file is a part of the CaosDB Project.
 *
 * Copyright (C) 2018 Research Group Biomedical Physics,
 * Max-Planck-Institute for Dynamics and Self-Organization Göttingen
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 *
 * ** end header
 */
package org.caosdb.server.database.backend.transaction;

import java.util.ArrayList;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import org.caosdb.server.database.BackendTransaction;
import org.caosdb.server.database.DatabaseUtils;
import org.caosdb.server.database.backend.interfaces.InsertEntityPropertiesImpl;
import org.caosdb.server.database.exceptions.TransactionException;
import org.caosdb.server.database.proto.FlatProperty;
import org.caosdb.server.datatype.AbstractCollectionDatatype;
import org.caosdb.server.datatype.AbstractDatatype.Table;
import org.caosdb.server.datatype.CollectionValue;
import org.caosdb.server.datatype.IndexedSingleValue;
import org.caosdb.server.datatype.SingleValue;
import org.caosdb.server.entity.EntityID;
import org.caosdb.server.entity.EntityInterface;
import org.caosdb.server.entity.Role;
import org.caosdb.server.entity.StatementStatus;
import org.caosdb.server.entity.wrapper.Property;

public class InsertEntityProperties extends BackendTransaction {

  private final EntityInterface entity;

  public InsertEntityProperties(final EntityInterface entity) {
    this.entity = entity;
  }

  @Override
  public void execute() throws TransactionException {

    final InsertEntityPropertiesImpl t = getImplementation(InsertEntityPropertiesImpl.class);

    final ArrayList<EntityInterface> stage1Inserts = new ArrayList<EntityInterface>();
    final ArrayList<EntityInterface> stage2Inserts = new ArrayList<EntityInterface>();

    DatabaseUtils.deriveStage1Inserts(stage1Inserts, this.entity);

    final int domainCount = DatabaseUtils.deriveStage2Inserts(stage2Inserts, stage1Inserts);

    final Deque<EntityID> domainIds = execute(new RegisterSubDomain(domainCount)).getDomains();

    insertStages(t, domainIds, stage1Inserts, this.entity.getDomain(), this.entity.getId());
    insertStages(t, domainIds, stage2Inserts, this.entity.getId(), null);
  }

  private void insertStages(
      final InsertEntityPropertiesImpl t,
      final Deque<EntityID> domainIds,
      final List<EntityInterface> stage1Inserts,
      final EntityID domain,
      final EntityID entity)
      throws TransactionException {

    for (final EntityInterface property : stage1Inserts) {
      if (property.hasRole() && property.getRole() == Role.Domain && !property.hasId()) {
        property.setId(domainIds.removeFirst());
      }
      int pIdx;
      if (property instanceof Property) {
        // this is a normal property
        pIdx = ((Property) property).getPIdx();
      } else {
        // this is a replacement
        pIdx = 0;
      }

      // prepare flat property
      final FlatProperty fp = new FlatProperty();
      Table table = Table.null_data;
      Long unit_sig = null;
      fp.id = property.getId().toInteger();
      fp.idx = pIdx;

      if (property.hasReplacement()) {
        if (!property.getReplacement().hasId()) {
          property.getReplacement().setId(domainIds.removeFirst());
        }

        fp.value = property.getReplacement().getId().toString();
        fp.status = StatementStatus.REPLACEMENT.toString();

        table = Table.reference_data;
      } else {
        if (property.hasUnit()) {
          unit_sig = property.getUnit().getSignature();
        }
        fp.status = property.getStatementStatus().toString();
        if (property.hasValue()) {
          if (property.getValue() instanceof CollectionValue) {
            // insert collection of values
            final CollectionValue v = (CollectionValue) property.getValue();
            final Iterator<IndexedSingleValue> iterator = v.iterator();
            final SingleValue firstValue = iterator.next();

            // insert 2nd to nth item
            for (int i = 1; i < v.size(); i++) {
              final SingleValue vi = iterator.next();
              fp.idx = i;
              if (vi == null) {
                fp.value = null;
                table = Table.null_data;
              } else {
                fp.value = vi.toDatabaseString();
                table = vi.getTable();
              }
              t.execute(
                  domain, entity != null ? entity : property.getDomain(), fp, table, unit_sig);
            }

            // insert first item
            fp.idx = 0;
            if (firstValue == null) {
              fp.value = null;
              table = Table.null_data;
            } else {
              fp.value = firstValue.toDatabaseString();
              table = firstValue.getTable();
            }

          } else {
            // insert single value
            fp.value = ((SingleValue) property.getValue()).toDatabaseString();
            if (property instanceof Property && ((Property) property).isName()) {
              table = Table.name_data;
            } else {
              table = ((SingleValue) property.getValue()).getTable();
            }
          }
        }
        if (property.isNameOverride()) {
          fp.name = property.getName();
        }
        if (property.isDescOverride()) {
          fp.desc = property.getDescription();
        }
        if (property.isDatatypeOverride()) {
          if (property.getDatatype() instanceof AbstractCollectionDatatype) {
            fp.type =
                ((AbstractCollectionDatatype) property.getDatatype())
                    .getDatatype()
                    .getId()
                    .toString();
            fp.collection =
                ((AbstractCollectionDatatype) property.getDatatype()).getCollectionName();
          } else {
            fp.type = property.getDatatype().getId().toString();
          }
        }
      }

      t.execute(domain, entity != null ? entity : property.getDomain(), fp, table, unit_sig);
    }
  }
}
