/*
 * ** header v3.0
 * This file is a part of the CaosDB Project.
 *
 * Copyright (C) 2018 Research Group Biomedical Physics,
 * Max-Planck-Institute for Dynamics and Self-Organization Göttingen
 * Copyright (C) 2020 Indiscale GmbH <info@indiscale.com>
 * Copyright (C) 2020 Timm Fitschen <t.fitschen@indiscale.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 *
 * ** end header
 */
package org.caosdb.server.accessControl;

import java.util.Arrays;
import java.util.Collection;
import org.apache.shiro.authc.AuthenticationException;
import org.apache.shiro.authc.AuthenticationToken;
import org.caosdb.server.utils.Utils;
import org.eclipse.jetty.util.ajax.JSON;

/**
 * These AuthenticationTokens are characterized by the following properties:
 *
 * <ul>
 *   <li>date: The creation time.
 *   <li>timeout: How long this token is valid after creation.
 *   <li>checksum: The checksum is calculated from all relevant parts of the authentication token
 *       (including the salt, timeout, permissions, roles, and date) and most importantly, the
 *       pepper which serves as a randomized password of the server. The salt makes it hard to guess
 *       the pepper by creating a rainbow table with plausible values for the other properties.
 *   <li>salt: Salt for the password checksum, may be used by inheriting classes.
 *   <li>pepper: A static property, generated when class is loaded and used until the server
 *       reboots. It servers as randomized password of the server. "In cryptography, a pepper is a
 *       secret added to an input such as a password prior to being hashed with a cryptographic hash
 *       function." (from: Pepper (cryptography),
 *       https://en.wikipedia.org/w/index.php?title=Pepper_(cryptography)&oldid=960047694 (last
 *       visited July 7, 2020)) In our case, the pepper is added to the token before hashing, but
 *       not exposed to the public, while the salt is. That also means that the resulting hash
 *       cannot be generated by any client nor be validated by any client, and that all tokens of
 *       this kind invalidate when the server reboots.
 */
public abstract class SelfValidatingAuthenticationToken extends Principal
    implements AuthenticationToken {

  protected static final transient String PEPPER = Utils.getSecureFilename(32);
  private static final long serialVersionUID = -7469302791917310460L;
  // date is the token creation time, in ms since 1970
  protected final long date;
  // token validity duration
  protected final long timeout;
  protected final String checksum;
  protected final String salt;
  protected final String[] permissions;
  protected final String[] roles;

  public Collection<String> getPermissions() {
    return Arrays.asList(this.permissions);
  }

  public Collection<String> getRoles() {
    return Arrays.asList(this.roles);
  }

  public static final String getFreshSalt() {
    // salt should be at least 8 octets long. https://www.ietf.org/rfc/rfc2898.txt
    // let's double that
    return Utils.getSecureFilename(16);
  }

  protected static <T> T defaultIfNull(T val, T def) {
    if (val != null) {
      return val;
    }
    return def;
  }

  public SelfValidatingAuthenticationToken(
      final Principal principal,
      final long date,
      final long timeout,
      final String salt,
      final String checksum,
      final String[] permissions,
      final String[] roles) {
    this(principal, date, timeout, salt, permissions, roles, checksum, false);
  }

  private SelfValidatingAuthenticationToken(
      final Principal principal,
      final long date,
      final long timeout,
      final String salt,
      final String[] permissions,
      final String[] roles,
      String checksum,
      boolean newChecksum,
      Object... fields) {
    super(principal);
    this.date = date;
    this.timeout = timeout;
    this.salt = salt;
    this.permissions = defaultIfNull(permissions, new String[] {});
    this.roles = defaultIfNull(roles, new String[] {});
    if (fields.length > 0) setFields(fields);
    // only calculate a checksum iff none is given and newChecksum is explicitly 'true'.
    this.checksum = checksum == null && newChecksum ? calcChecksum() : checksum;
  }

  /** Customizable customization method, will be called with the remaining constructor arguments. */
  protected abstract void setFields(Object[] fields);

  public SelfValidatingAuthenticationToken(
      final Principal principal,
      final long timeout,
      final String[] permissions,
      final String[] roles,
      Object... fields) {
    this(
        principal,
        System.currentTimeMillis(),
        timeout,
        getFreshSalt(),
        permissions,
        roles,
        null,
        true,
        fields);
  }

  public final String calcChecksum() {
    return calcChecksum(PEPPER);
  }

  @Override
  public abstract String toString();

  /**
   * Implementation specific version of a peppered checksum.
   *
   * <p>For secure operation, implementing classes must make sure that the pepper is actually used
   * in calculating the checksum and that the checksum can not be used to infer information about
   * the pepper. This can be achieved for example by using the {@link calcChecksum(final Object...
   * fields)} method.
   */
  public abstract String calcChecksum(String pepper);

  /** No credentials (returns null), since this token is self-validating. */
  @Override
  public Object getCredentials() {
    return null;
  }

  public long getExpires() {
    return this.date + this.timeout;
  }

  public long getTimeout() {
    return this.timeout;
  }

  public boolean isExpired() {
    return System.currentTimeMillis() >= getExpires();
  }

  /**
   * Test if the hash stored in `checksum` is equal to the one calculated using the secret pepper.
   */
  public boolean isHashValid() {
    final String other = calcChecksum();
    return this.checksum != null && this.checksum.equals(other);
  }

  public boolean isValid() {
    return !isExpired() && isHashValid();
  }

  /** Return the hash (SHA512) of the stringified arguments. */
  protected static String calcChecksum(final Object... fields) {
    final StringBuilder sb = new StringBuilder();
    for (final Object field : fields) {
      if (field != null) {
        sb.append(field.toString());
      }
      sb.append(":");
    }
    return Utils.sha512(sb.toString(), null, 1);
  }

  protected static String[] toStringArray(final Object[] array) {
    final String[] ret = new String[array.length];
    for (int i = 0; i < ret.length; i++) {
      ret[i] = (String) array[i];
    }
    return ret;
  }

  /**
   * Parse a JSON string and return the generated token. Depending on the first element of the JSON,
   * this is either (if it is "O") a OneTimeAuthenticationToken or (if it is "S") a SessionToken.
   *
   * @throws AuthenticationToken if the string could not be parsed into a token.
   */
  public static SelfValidatingAuthenticationToken parse(String token) {
    Object[] array = (Object[]) JSON.parse(token);
    switch (array[0].toString()) {
      case "O":
        return OneTimeAuthenticationToken.parse(array);
      case "S":
        return SessionToken.parse(array);
      default:
        throw new AuthenticationException("Could not parse the authtoken string (unknown type).");
    }
  }

  /** No "other" identity, so this returns itself. */
  @Override
  public SelfValidatingAuthenticationToken getPrincipal() {
    return this;
  }
}
