#include "cppdefs.h"
!-----------------------------------------------------------------------
!BOP
!
! !MODULE: output_processing
!
! !INTERFACE:
   module output_processing
!
! !DESCRIPTION:
!  This modules serves as a container for processing output variables. 

! !USES:
   use parameters, only: rk
   use domain, only: imin,imax,jmin,jmax,kmax
   use field_manager
   IMPLICIT NONE
!
   private
!
! !PUBLIC DATA FUNCTIONS:
   public init_output_processing, do_output_processing
   public register_processed_variables, finalize_register_processed_variables
!
! !PUBLIC DATA MEMBERS:
   integer, public  :: zbins=0
   REALTYPE, public :: zbinmin=_ZERO_
   REALTYPE, public :: zbinmax=_ZERO_
!
! !PRIVATE DATA MEMBERS:
   REALTYPE, dimension(:,:),   allocatable, target :: u2d, v2d
   REALTYPE, dimension(:,:,:), allocatable, target :: u3d, v3d
   REALTYPE, dimension(:,:),   allocatable, target :: u2d_destag, v2d_destag
   REALTYPE, dimension(:,:,:), allocatable, target :: u3d_destag, v3d_destag

   REALTYPE, dimension(:), allocatable, target :: zi_z
   REALTYPE, dimension(:,:,:), allocatable, target :: h_z, hS_z
   REALTYPE, dimension(:,:,:), allocatable, target :: hu_z, hv_z
   REALTYPE, dimension(:,:,:), allocatable, target :: uu_z, vv_z
   REALTYPE, dimension(:,:,:), allocatable, target :: Sfluxu_z, Sfluxv_z
   REALTYPE, dimension(:,:,:), allocatable, target :: hpmS_z, hnmS_z
   REALTYPE, dimension(:,:,:), allocatable, target :: h_z_mean, hS_z_mean
   REALTYPE, dimension(:,:,:), allocatable, target :: hu_z_mean, hv_z_mean
   REALTYPE, dimension(:,:,:), allocatable, target :: uu_z_mean, vv_z_mean
   REALTYPE, dimension(:,:,:), allocatable, target :: Sfluxu_z_mean, Sfluxv_z_mean
   REALTYPE, dimension(:,:,:), allocatable, target :: hpmS_z_mean, hnmS_z_mean

#ifndef NO_BAROCLINIC
   REALTYPE,dimension(:,:,:),allocatable,target :: h_s_mean, hS_s_mean, hS2_s_mean
   REALTYPE,dimension(:,:,:),allocatable,target :: hu_s_mean, hv_s_mean
   REALTYPE,dimension(:,:,:),allocatable,target :: uu_s_mean, vv_s_mean
   REALTYPE,dimension(:,:,:),allocatable,target :: wdia_s_mean
   REALTYPE,dimension(:,:,:),allocatable,target :: Sfluxu_s_mean, Sfluxv_s_mean
   REALTYPE,dimension(:,:,:),allocatable,target :: f3dia_s_mean
   REALTYPE,dimension(:,:,:),allocatable,target :: hpmS_s_mean, hnmS_s_mean
   REALTYPE,dimension(:,:,:),allocatable,target :: fwf_s_int, phymix_S_fwf_s_mean
   REALTYPE,dimension(:,:,:),allocatable,target :: rvol_s_int, phymix_S_riv_s_mean
#endif

   logical         :: u2d_used, v2d_used
   logical         :: u3d_used=.false., v3d_used=.false.
   logical         :: u2d_destag_used, v2d_destag_used
   logical, target :: u2d_now, v2d_now
   logical, target :: u3d_now=.false., v3d_now=.false.
   logical, target :: u2d_destag_now, v2d_destag_now
   logical, target:: u3d_destag_use, v3d_destag_use

   logical         :: process_zbin=.false.
   logical         :: h_z_used=.false.
   logical         :: hS_z_used=.false.
   logical         :: hu_z_used=.false.
   logical         :: hv_z_used=.false.
   logical         :: uu_z_used=.false.
   logical         :: vv_z_used=.false.
   logical         :: Sfluxu_z_used=.false.
   logical         :: Sfluxv_z_used=.false.
   logical         :: hpmS_z_used=.false.
   logical         :: hnmS_z_used=.false.

   logical         :: process_zbinmean=.false.
   logical         :: h_z_mean_used=.false.
   logical         :: hS_z_mean_used=.false.
   logical         :: hu_z_mean_used=.false.
   logical         :: hv_z_mean_used=.false.
   logical         :: uu_z_mean_used=.false.
   logical         :: vv_z_mean_used=.false.
   logical         :: Sfluxu_z_mean_used=.false.
   logical         :: Sfluxv_z_mean_used=.false.
   logical         :: hpmS_z_mean_used=.false.
   logical         :: hnmS_z_mean_used=.false.
   logical, target :: h_z_mean_now=.false.
   logical, target :: hS_z_mean_now=.false.
   logical, target :: hu_z_mean_now=.false.
   logical, target :: hv_z_mean_now=.false.
   logical, target :: uu_z_mean_now=.false.
   logical, target :: vv_z_mean_now=.false.
   logical, target :: Sfluxu_z_mean_now=.false.
   logical, target :: Sfluxv_z_mean_now=.false.
   logical, target :: hpmS_z_mean_now=.false.
   logical, target :: hnmS_z_mean_now=.false.

#ifndef NO_BAROCLINIC
   logical                             :: process_TEFmean=.false.
   logical                             :: h_s_mean_used=.false.
   logical                             :: hS_s_mean_used=.false.
   logical                             :: hS2_s_mean_used=.false.
   logical                             :: hu_s_mean_used=.false.
   logical                             :: hv_s_mean_used=.false.
   logical                             :: uu_s_mean_used=.false.
   logical                             :: vv_s_mean_used=.false.
   logical                             :: wdia_s_mean_used=.false.
   logical                             :: Sfluxu_s_mean_used=.false.
   logical                             :: Sfluxv_s_mean_used=.false.
   logical                             :: f3dia_s_mean_used=.false.
   logical                             :: hpmS_s_mean_used=.false.
   logical                             :: hnmS_s_mean_used=.false.
   logical                             :: fwf_s_int_used=.false.
   logical                             :: phymix_S_fwf_s_mean_used=.false.
   logical                             :: rvol_s_int_used=.false.
   logical                             :: phymix_S_riv_s_mean_used=.false.
   logical, target                     :: h_s_mean_now=.false.
   logical, target                     :: hS_s_mean_now=.false.
   logical, target                     :: hS2_s_mean_now=.false.
   logical, target                     :: hu_s_mean_now=.false.
   logical, target                     :: hv_s_mean_now=.false.
   logical, target                     :: uu_s_mean_now=.false.
   logical, target                     :: vv_s_mean_now=.false.
   logical, target                     :: wdia_s_mean_now=.false.
   logical, target                     :: Sfluxu_s_mean_now=.false.
   logical, target                     :: Sfluxv_s_mean_now=.false.
   logical, target                     :: f3dia_s_mean_now=.false.
   logical, target                     :: hpmS_s_mean_now=.false.
   logical, target                     :: hnmS_s_mean_now=.false.
   logical, target                     :: fwf_s_int_now=.false.
   logical, target                     :: phymix_S_fwf_s_mean_now=.false.
   logical, target                     :: rvol_s_int_now=.false.
   logical, target                     :: phymix_S_riv_s_mean_now=.false.
#endif

!
! !REVISION HISTORY:
!  Original author(s): Karsten Bolding & Hans Burchard
!
!EOP
!-----------------------------------------------------------------------

   contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_output_processing - read required variables
!
! !INTERFACE:
   subroutine init_output_processing
!
! !USES:
#ifndef NO_BAROCLINIC
   use variables_3d
#endif
   IMPLICIT NONE
!
! !DESCRIPTION:
!
! !INPUT PARAMETERS:
!    character(len=*), intent(in)        :: filename
!
! !REVISION HISTORY:
!  Original author(s): Karsten Bolding
!
! !LOCAL VARIABLES:
   integer                             :: l
   integer                             :: rc
!EOP
!-------------------------------------------------------------------------
!BOC

#if 0
   allocate(u3d_destag(I3DFIELD),stat=rc)
   if (rc /= 0) stop 'init_output_processing: Error allocating memory (u3d_destag)'
   allocate(v3d_destag(I3DFIELD),stat=rc)
   if (rc /= 0) stop 'init_output_processing: Error allocating memory (v3d_destag)'
#endif

   if (process_zbinmean) then
      if (h_z_mean_used) then
         h_z_used = .true.
         allocate(h_z_mean(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'finalize_register_processed_variables: Error allocating memory (h_z_mean)'
         h_z_mean = _ZERO_
      end if
      if (hS_z_mean_used) then
         hS_z_used = .true.
         allocate(hS_z_mean(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'finalize_register_processed_variables: Error allocating memory (hS_z_mean)'
         hS_z_mean = _ZERO_
      end if
      if (hu_z_mean_used) then
         hu_z_used = .true.
         allocate(hu_z_mean(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'finalize_register_processed_variables: Error allocating memory (hu_z_mean)'
         hu_z_mean = _ZERO_
      end if
      if (hv_z_mean_used) then
         hv_z_used = .true.
         allocate(hv_z_mean(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'finalize_register_processed_variables: Error allocating memory (hv_z_mean)'
         hv_z_mean = _ZERO_
      end if
      if (uu_z_mean_used) then
         uu_z_used = .true.
         allocate(uu_z_mean(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'finalize_register_processed_variables: Error allocating memory (uu_z_mean)'
         uu_z_mean = _ZERO_
      end if
      if (vv_z_mean_used) then
         vv_z_used = .true.
         allocate(vv_z_mean(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'finalize_register_processed_variables: Error allocating memory (vv_z_mean)'
         vv_z_mean = _ZERO_
      end if
      if (Sfluxu_z_mean_used) then
         Sfluxu_z_used = .true.
         allocate(Sfluxu_z_mean(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'finalize_register_processed_variables: Error allocating memory (Sfluxu_z_mean)'
         Sfluxu_z_mean = _ZERO_
      end if
      if (Sfluxv_z_mean_used) then
         Sfluxv_z_used = .true.
         allocate(Sfluxv_z_mean(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'finalize_register_processed_variables: Error allocating memory (Sfluxv_z_mean)'
         Sfluxv_z_mean = _ZERO_
      end if
      if (hpmS_z_mean_used) then
         hpmS_z_used = .true.
         allocate(hpmS_z_mean(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'finalize_register_processed_variables: Error allocating memory (hpmS_z_mean)'
         hpmS_z_mean = _ZERO_
      end if
      if (hnmS_z_mean_used) then
         hnmS_z_used = .true.
         allocate(hnmS_z_mean(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'finalize_register_processed_variables: Error allocating memory (hnmS_z_mean)'
         hnmS_z_mean = _ZERO_
      end if
   end if

   if (process_zbin) then
      LEVEL3 'z space for mapping:',real(zbinmin),real(zbinmax),'(',zbins,'bins)'
      allocate(zi_z(0:zbins),stat=rc)
      if (rc /= 0) stop 'register_processed_variables: Error allocating memory (zi_z)'
      do l=0,zbins
         zi_z(l) = zbinmin + (zbinmax-zbinmin)/zbins * l
      end do

      if (h_z_used) then
         allocate(h_z(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'postinit_3d: Error allocating memory (h_z)'
         h_z = _ZERO_
      end if
      if (hS_z_used) then
         allocate(hS_z(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'postinit_3d: Error allocating memory (hS_z)'
         hS_z = _ZERO_
      end if
      if (hu_z_used) then
         allocate(hu_z(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'postinit_3d: Error allocating memory (hu_z)'
         hu_z = _ZERO_
      end if
      if (hv_z_used) then
         allocate(hv_z(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'postinit_3d: Error allocating memory (hv_z)'
         hv_z = _ZERO_
      end if
      if (uu_z_used) then
         allocate(uu_z(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'postinit_3d: Error allocating memory (uu_z)'
         uu_z = _ZERO_
      end if
      if (vv_z_used) then
         allocate(vv_z(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'postinit_3d: Error allocating memory (vv_z)'
         vv_z = _ZERO_
      end if
      if (Sfluxu_z_used) then
         save_Sfluxu = .true.
         allocate(Sfluxu_z(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'postinit_3d: Error allocating memory (Sfluxu_z)'
         Sfluxu_z = _ZERO_
      end if
      if (Sfluxv_z_used) then
         save_Sfluxv = .true.
         allocate(Sfluxv_z(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'postinit_3d: Error allocating memory (Sfluxv_z)'
         Sfluxv_z = _ZERO_
      end if
      if (hpmS_z_used) then
         save_phymix_S = .true.
         allocate(hpmS_z(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'postinit_3d: Error allocating memory (hpmS_z)'
         hpmS_z = _ZERO_
      end if
      if (hnmS_z_used) then
         save_nummix_S = .true.
         allocate(hnmS_z(I2DFIELD,0:zbins),stat=rc)
         if (rc /= 0) stop 'postinit_3d: Error allocating memory (hnmS_z)'
         hnmS_z = _ZERO_
      end if
   end if

!  Note (KK): this needs to be called
!             AFTER register_processed_variables()
!             and BEFORE postinit_variables_3d() !!!
#ifndef NO_BAROCLINIC
   if (process_TEFmean) then
      if (h_s_mean_used) then
         h_s_used = .true.
         allocate(h_s_mean(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_variables_3d: Error allocating memory (h_s_mean)'
         h_s_mean = _ZERO_
      end if
      if (hS_s_mean_used) then
         hS_s_used = .true.
         allocate(hS_s_mean(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_variables_3d: Error allocating memory (hS_s_mean)'
         hS_s_mean = _ZERO_
      end if
      if (hS2_s_mean_used) then
         hS2_s_used = .true.
         allocate(hS2_s_mean(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_variables_3d: Error allocating memory (hS2_s_mean)'
         hS2_s_mean = _ZERO_
      end if
      if (hu_s_mean_used) then
         hu_s_used = .true.
         allocate(hu_s_mean(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_variables_3d: Error allocating memory (hu_s_mean)'
         hu_s_mean = _ZERO_
      end if
      if (hv_s_mean_used) then
         hv_s_used = .true.
         allocate(hv_s_mean(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_variables_3d: Error allocating memory (hv_s_mean)'
         hv_s_mean = _ZERO_
      end if
      if (uu_s_mean_used) then
         uu_s_used = .true.
         allocate(uu_s_mean(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_variables_3d: Error allocating memory (uu_s_mean)'
         uu_s_mean = _ZERO_
      end if
      if (vv_s_mean_used) then
         vv_s_used = .true.
         allocate(vv_s_mean(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_variables_3d: Error allocating memory (vv_s_mean)'
         vv_s_mean = _ZERO_
      end if
      if (wdia_s_mean_used) then
         wdia_s_used = .true.
         allocate(wdia_s_mean(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_variables_3d: Error allocating memory (wdia_s_mean)'
         wdia_s_mean = _ZERO_
      end if
      if (Sfluxu_s_mean_used) then
         Sfluxu_s_used = .true.
         allocate(Sfluxu_s_mean(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_variables_3d: Error allocating memory (Sfluxu_s_mean)'
         Sfluxu_s_mean = _ZERO_
      end if
      if (Sfluxv_s_mean_used) then
         Sfluxv_s_used = .true.
         allocate(Sfluxv_s_mean(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_variables_3d: Error allocating memory (Sfluxv_s_mean)'
         Sfluxv_s_mean = _ZERO_
      end if
      if (f3dia_s_mean_used) then
         f3dia_s_used = .true.
         allocate(f3dia_s_mean(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_variables_3d: Error allocating memory (f3dia_s_mean)'
         f3dia_s_mean = _ZERO_
      end if
      if (hpmS_s_mean_used) then
         hpmS_s_used = .true.
         allocate(hpmS_s_mean(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_variables_3d: Error allocating memory (hpmS_s_mean)'
         hpmS_s_mean = _ZERO_
      end if
      if (hnmS_s_mean_used) then
         hnmS_s_used = .true.
         allocate(hnmS_s_mean(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_variables_3d: Error allocating memory (hnmS_s_mean)'
         hnmS_s_mean = _ZERO_
      end if
      if (fwf_s_int_used) then
         fwf_s_used = .true.
         allocate(fwf_s_int(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_variables_3d: Error allocating memory (fwf_s_int)'
         fwf_s_int = _ZERO_
      end if
      if (phymix_S_fwf_s_mean_used) then
         phymix_S_fwf_s_used = .true.
         allocate(phymix_S_fwf_s_mean(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_3d: Error allocating memory (phymix_S_fwf_s_mean)'
         phymix_S_fwf_s_mean = _ZERO_
      end if
      if (rvol_s_int_used) then
         rvol_s_used = .true.
         allocate(rvol_s_int(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_3d: Error allocating memory (rvol_s_int)'
         rvol_s_int = _ZERO_
      end if
      if (phymix_S_riv_s_mean_used) then
         phymix_S_riv_s_used = .true.
         allocate(phymix_S_riv_s_mean(I2DFIELD,0:saltbins),stat=rc)
         if (rc /= 0) stop 'postinit_3d: Error allocating memory (phymix_S_riv_s_mean)'
         phymix_S_riv_s_mean = _ZERO_
      end if
   end if
#endif

   return
   end subroutine init_output_processing
!EOC

!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: register_processed_variables()
!
! !INTERFACE:
   subroutine register_processed_variables(fm,runtype)
!
! !DESCRIPTION:
!
! !USES:
#ifndef NO_BAROCLINIC
   use m3d, only: update_salt
   use variables_3d, only: saltbins
   use variables_3d, only: id_dim_salt
#endif
   use meteo       , only: fwf_method
   use domain      , only: nriverl
   IMPLICIT NONE
!
! !INPUT PARAMETERS:
   type (type_field_manager) :: fm
   integer, intent(in)       :: runtype
!
! !REVISION HISTORY:
!  Original author(s): Karsten Bolding & Jorn Bruggeman
!
! !LOCAL VARIABLES:
   integer :: id_dim_zbins
!EOP
!-----------------------------------------------------------------------
!BOC
   LEVEL2 'register_processed_variables()'

   call fm%register('u2d', 'm/s', 'velocity in local x-direction', standard_name='', fill_value=-9999._rk, category='velocities', output_level=output_level_debug, used=u2d_used, used_now=u2d_now)
   call fm%register('v2d', 'm/s', 'velocity in local y-direction', standard_name='', fill_value=-9999._rk, category='velocities', output_level=output_level_debug, used=v2d_used, used_now=v2d_now)

#ifndef NO_3D
   if (runtype .ge. 2) then
   call fm%register('u3d', 'm/s', 'velocity in local x-direction (3D)', standard_name='', dimensions=(/id_dim_z/), fill_value=-9999._rk, category='velocities', output_level=output_level_debug, used=u3d_used, used_now=u3d_now)
   call fm%register('v3d', 'm/s', 'velocity in local y-direction (3D)', standard_name='', dimensions=(/id_dim_z/), fill_value=-9999._rk, category='velocities', output_level=output_level_debug, used=v3d_used, used_now=v3d_now)
   end if
#endif


   call fm%register('u2d-destag', 'm/s', 'velocity in local x-direction(destag)', standard_name='', fill_value=-9999._rk, category='velocities',output_level=output_level_debug, used=u2d_destag_used, used_now=u2d_destag_now)
   call fm%register('v2d-destag', 'm/s', 'velocity in local y-direction(destag)', standard_name='', fill_value=-9999._rk, category='velocities',output_level=output_level_debug, used=v2d_destag_used, used_now=v2d_destag_now)

#ifndef NO_3D
   if (runtype .ge. 2) then
      if (zbins .gt. 0) then

         process_zbin = .true.

         call fm%register_dimension('zi_z', zbins+1, newid=id_dim_zbins)
         call fm%register('zi_z', 'm', 'interface z (zbin)', dimensions=(/id_dim_zbins/), no_default_dimensions=.true., coordinate_dimension=id_dim_zbins, category='zbin', output_level=output_level_debug)
         call fm%register('h_z', 'm', 'thickness (zbin)', dimensions=(/id_dim_zbins/), category='zbin', output_level=output_level_debug, used=h_z_used)
         call fm%register('hS_z', 'm*(g/kg)', 'salt content (zbin)', dimensions=(/id_dim_zbins/), category='zbin', output_level=output_level_debug, used=hS_z_used)
         call fm%register('hu_z', 'm', 'thickness at U-points (zbin)', dimensions=(/id_dim_zbins/), category='zbin', output_level=output_level_debug, used=hu_z_used)
         call fm%register('hv_z', 'm', 'thickness at V-points (zbin)', dimensions=(/id_dim_zbins/), category='zbin', output_level=output_level_debug, used=hv_z_used)
         call fm%register('uu_z', 'm2/s', 'transport in local x-direction (zbin)', dimensions=(/id_dim_zbins/), category='zbin', output_level=output_level_debug, used=uu_z_used)
         call fm%register('vv_z', 'm2/s', 'transport in local y-direction (zbin)', dimensions=(/id_dim_zbins/), category='zbin', output_level=output_level_debug, used=vv_z_used)
         call fm%register('Sfluxu_z', '(g/kg)*m3/s', 'salt flux in local x-direction (zbin)', dimensions=(/id_dim_zbins/), category='zbin', output_level=output_level_debug, used=Sfluxu_z_used)
         call fm%register('Sfluxv_z', '(g/kg)*m3/s', 'salt flux in local y-direction (zbin)', dimensions=(/id_dim_zbins/), category='zbin', output_level=output_level_debug, used=Sfluxv_z_used)
         call fm%register('hpmS_z', 'm*(g/kg)**2/s', 'phymixS content (zbin)', dimensions=(/id_dim_zbins/), category='zbin', output_level=output_level_debug, used=hpmS_z_used)
         call fm%register('hnmS_z', 'm*(g/kg)**2/s', 'nummixS content (zbin)', dimensions=(/id_dim_zbins/), category='zbin', output_level=output_level_debug, used=hnmS_z_used)

         process_zbinmean = .true.

         call fm%register('h_z_mean', 'm', 'mean thickness (zbin)', dimensions=(/id_dim_zbins/), category='zbinmean', output_level=output_level_debug, used=h_z_mean_used, used_now=h_z_mean_now)
         call fm%register('hS_z_mean', 'm*(g/kg)', 'mean salt content (zbin)', dimensions=(/id_dim_zbins/), category='zbinmean', output_level=output_level_debug, used=hS_z_mean_used, used_now=hS_z_mean_now)
         call fm%register('hu_z_mean', 'm', 'mean thickness at U-points (zbin)', dimensions=(/id_dim_zbins/), category='zbinmean', output_level=output_level_debug, used=hu_z_mean_used, used_now=hu_z_mean_now)
         call fm%register('hv_z_mean', 'm', 'mean thickness at V-points (zbin)', dimensions=(/id_dim_zbins/), category='zbinmean', output_level=output_level_debug, used=hv_z_mean_used, used_now=hv_z_mean_now)
         call fm%register('uu_z_mean', 'm2/s', 'mean transport in local x-direction (zbin)', dimensions=(/id_dim_zbins/), category='zbinmean', output_level=output_level_debug, used=uu_z_mean_used, used_now=uu_z_mean_now)
         call fm%register('vv_z_mean', 'm2/s', 'mean transport in local y-direction (zbin)', dimensions=(/id_dim_zbins/), category='zbinmean', output_level=output_level_debug, used=vv_z_mean_used, used_now=vv_z_mean_now)
         call fm%register('Sfluxu_z_mean', '(g/kg)*m3/s', 'mean salt flux in local x-direction (zbin)', dimensions=(/id_dim_zbins/), category='zbinmean', output_level=output_level_debug, used=Sfluxu_z_mean_used, used_now=Sfluxu_z_mean_now)
         call fm%register('Sfluxv_z_mean', '(g/kg)*m3/s', 'mean salt flux in local y-direction (zbin)', dimensions=(/id_dim_zbins/), category='zbinmean', output_level=output_level_debug, used=Sfluxv_z_mean_used, used_now=Sfluxv_z_mean_now)
         call fm%register('hpmS_z_mean', 'm*(g/kg)**2/s', 'mean phymixS content (zbin)', dimensions=(/id_dim_zbins/), category='zbinmean', output_level=output_level_debug, used=hpmS_z_mean_used, used_now=hpmS_z_mean_now)
         call fm%register('hnmS_z_mean', 'm*(g/kg)**2/s', 'mean nummixS content (zbin)', dimensions=(/id_dim_zbins/), category='zbinmean', output_level=output_level_debug, used=hnmS_z_mean_used, used_now=hnmS_z_mean_now)

      end if
   end if

#ifndef NO_BAROCLINIC
   if (runtype .ge. 3) then
      if (update_salt) then
         if (saltbins .gt. 0) then

         process_TEFmean = .true.

         call fm%register('h_s_mean', 'm', 'mean bin thickness', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=h_s_mean_used, used_now=h_s_mean_now)
         call fm%register('hS_s_mean', 'm*(g/kg)', 'mean salt content (bin)', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=hS_s_mean_used, used_now=hS_s_mean_now)
         call fm%register('hS2_s_mean', 'm*(g/kg)**2', 'mean salt2 content (bin)', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=hS2_s_mean_used, used_now=hS2_s_mean_now)
         call fm%register('hu_s_mean', 'm', 'mean bin thickness at U-points', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=hu_s_mean_used, used_now=hu_s_mean_now)
         call fm%register('hv_s_mean', 'm', 'mean bin thickness at V-points', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=hv_s_mean_used, used_now=hv_s_mean_now)
         call fm%register('uu_s_mean', 'm2/s', 'mean transport in local x-direction (bin)', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=uu_s_mean_used, used_now=uu_s_mean_now)
         call fm%register('vv_s_mean', 'm2/s', 'mean transport in local y-direction (bin)', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=vv_s_mean_used, used_now=vv_s_mean_now)
         call fm%register('wdia_s_mean', 'm/s', 'mean vertical diahaline velocity (isohaline)', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=wdia_s_mean_used, used_now=wdia_s_mean_now)
         call fm%register('Sfluxu_s_mean', '(g/kg)*m3/s', 'mean salt flux in local x-direction (bin)', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=Sfluxu_s_mean_used, used_now=Sfluxu_s_mean_now)
         call fm%register('Sfluxv_s_mean', '(g/kg)*m3/s', 'mean salt flux in local y-direction (bin)', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=Sfluxv_s_mean_used, used_now=Sfluxv_s_mean_now)
         call fm%register('f3dia_s_mean', '(g/kg)*m/s', 'mean vertical diahaline salt flux (isohaline)', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=f3dia_s_mean_used, used_now=f3dia_s_mean_now)
         call fm%register('hpmS_s_mean', 'm*(g/kg)**2/s', 'mean phymixS content (bin)', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=hpmS_s_mean_used, used_now=hpmS_s_mean_now)
         call fm%register('hnmS_s_mean', 'm*(g/kg)**2/s', 'mean nummixS content (bin)', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=hnmS_s_mean_used, used_now=hnmS_s_mean_now)
         if (fwf_method .ge. 1) then
            call fm%register('fwf_s_int', 'm', 'integrated fresh water fluxes (bin)', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=fwf_s_int_used, used_now=fwf_s_int_now)
            call fm%register('phymix_S_fwf_s_mean', 'm*(g/kg)**2/s', 'mean physical mixing content of salinity (fwf) (bin)', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=phymix_S_fwf_s_mean_used, used_now=phymix_S_fwf_s_mean_now)
         end if
         if (nriverl .gt. 0) then
            call fm%register('rvol_s_int', 'm**3', 'integrated river discharge (3D) (bin)', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=rvol_s_int_used, used_now=rvol_s_int_now)
            call fm%register('phymix_S_riv_s_mean', 'm*(g/kg)**2/s', 'mean physical mixing content of salinity (riv) (bin)', dimensions=(/id_dim_salt/), category='TEFmean', output_level=output_level_debug, used=phymix_S_riv_s_mean_used, used_now=phymix_S_riv_s_mean_now)
         end if

         end if
      end if
   end if
#endif
#endif

   return
   end subroutine register_processed_variables
!EOC

!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: finalize_register_processed_variables() - send optional variables.
!
! !INTERFACE:
   subroutine finalize_register_processed_variables(fm)
!
! !DESCRIPTION:
!
! !USES:
   use field_manager
   IMPLICIT NONE
!
! !INPUT PARAMETERS:
   type (type_field_manager) :: fm
!
! !REVISION HISTORY:
!  Original author(s): Knut Klingbeil
!
! !LOCAL VARIABLES:
   integer                             :: rc
!EOP
!-----------------------------------------------------------------------
!BOC
   LEVEL1 'finalize_register_processed_variables()'

   if (u2d_used) then
      allocate(u2d(E2DFIELD),stat=rc)
      if (rc /= 0) stop 'finalize_register_processed_variables: Error allocating memory (u2d)'
      u2d = 0._rk
      call fm%send_data('u2d', u2d(_2D_W_))
   end if

   if (v2d_used) then
      allocate(v2d(E2DFIELD),stat=rc)
      if (rc /= 0) stop 'finalize_register_processed_variables: Error allocating memory (v2d)'
      v2d = 0._rk
      call fm%send_data('v2d', v2d(_2D_W_))
   end if

#ifndef NO_3D
   if (u3d_used) then
      allocate(u3d(I3DFIELD),stat=rc)
      if (rc /= 0) stop 'finalize_register_processed_variables: Error allocating memory (u3d)'
      u3d = 0._rk
      call fm%send_data('u3d', u3d(_3D_W_))
   end if

   if (v3d_used) then
      allocate(v3d(I3DFIELD),stat=rc)
      if (rc /= 0) stop 'finalize_register_processed_variables: Error allocating memory (v3d)'
      v3d = 0._rk
      call fm%send_data('v3d', v3d(_3D_W_))
   end if
#endif

   if (u2d_destag_used) then
      allocate(u2d_destag(E2DFIELD),stat=rc)
      if (rc /= 0) stop 'finalize_register_processed_variables: Error allocating memory (u2d_destag)'
      u2d_destag = 0._rk
      call fm%send_data('u2d_destag', u2d_destag(_2D_W_))
   end if

   if (v2d_destag_used) then
      allocate(v2d_destag(E2DFIELD),stat=rc)
      if (rc /= 0) stop 'finalize_register_processed_variables: Error allocating memory (v2d_destag)'
      v2d_destag = 0._rk
      call fm%send_data('v2d_destag', v2d_destag(_2D_W_))
   end if


   if (process_zbin) then
      if (allocated(zi_z    )) call fm%send_data('zi_z'    , zi_z              )
      if (allocated(h_z     )) call fm%send_data('h_z'     , h_z     (_2D_W_,:))
      if (allocated(hS_z    )) call fm%send_data('hS_z'    , hS_z    (_2D_W_,:))
      if (allocated(hu_z    )) call fm%send_data('hu_z'    , hu_z    (_2D_W_,:))
      if (allocated(hv_z    )) call fm%send_data('hv_z'    , hv_z    (_2D_W_,:))
      if (allocated(uu_z    )) call fm%send_data('uu_z'    , uu_z    (_2D_W_,:))
      if (allocated(vv_z    )) call fm%send_data('vv_z'    , vv_z    (_2D_W_,:))
      if (allocated(Sfluxu_z)) call fm%send_data('Sfluxu_z', Sfluxu_z(_2D_W_,:))
      if (allocated(Sfluxv_z)) call fm%send_data('Sfluxv_z', Sfluxv_z(_2D_W_,:))
      if (allocated(hpmS_z  )) call fm%send_data('hpmS_z'  , hpmS_z  (_2D_W_,:))
      if (allocated(hnmS_z  )) call fm%send_data('hnmS_z'  , hnmS_z  (_2D_W_,:))
   end if

   if (process_zbinmean) then
      if (allocated(h_z_mean     )) call fm%send_data('h_z_mean'     , h_z_mean     (_2D_W_,:))
      if (allocated(hS_z_mean    )) call fm%send_data('hS_z_mean'    , hS_z_mean    (_2D_W_,:))
      if (allocated(hu_z_mean    )) call fm%send_data('hu_z_mean'    , hu_z_mean    (_2D_W_,:))
      if (allocated(hv_z_mean    )) call fm%send_data('hv_z_mean'    , hv_z_mean    (_2D_W_,:))
      if (allocated(uu_z_mean    )) call fm%send_data('uu_z_mean'    , uu_z_mean    (_2D_W_,:))
      if (allocated(vv_z_mean    )) call fm%send_data('vv_z_mean'    , vv_z_mean    (_2D_W_,:))
      if (allocated(Sfluxu_z_mean)) call fm%send_data('Sfluxu_z_mean', Sfluxu_z_mean(_2D_W_,:))
      if (allocated(Sfluxv_z_mean)) call fm%send_data('Sfluxv_z_mean', Sfluxv_z_mean(_2D_W_,:))
      if (allocated(hpmS_z_mean  )) call fm%send_data('hpmS_z_mean'  , hpmS_z_mean  (_2D_W_,:))
      if (allocated(hnmS_z_mean  )) call fm%send_data('hnmS_z_mean'  , hnmS_z_mean  (_2D_W_,:))
   end if

#ifndef NO_BAROCLINIC
   if (process_TEFmean) then
      if (allocated(h_s_mean     )) call fm%send_data('h_s_mean'     , h_s_mean     (_2D_W_,:))
      if (allocated(hS_s_mean    )) call fm%send_data('hS_s_mean'    , hS_s_mean    (_2D_W_,:))
      if (allocated(hS2_s_mean   )) call fm%send_data('hS2_s_mean'   , hS2_s_mean   (_2D_W_,:))
      if (allocated(hu_s_mean    )) call fm%send_data('hu_s_mean'    , hu_s_mean    (_2D_W_,:))
      if (allocated(hv_s_mean    )) call fm%send_data('hv_s_mean'    , hv_s_mean    (_2D_W_,:))
      if (allocated(uu_s_mean    )) call fm%send_data('uu_s_mean'    , uu_s_mean    (_2D_W_,:))
      if (allocated(vv_s_mean    )) call fm%send_data('vv_s_mean'    , vv_s_mean    (_2D_W_,:))
      if (allocated(wdia_s_mean  )) call fm%send_data('wdia_s_mean'  , wdia_s_mean  (_2D_W_,:))
      if (allocated(Sfluxu_s_mean)) call fm%send_data('Sfluxu_s_mean', Sfluxu_s_mean(_2D_W_,:))
      if (allocated(Sfluxv_s_mean)) call fm%send_data('Sfluxv_s_mean', Sfluxv_s_mean(_2D_W_,:))
      if (allocated(f3dia_s_mean )) call fm%send_data('f3dia_s_mean' , f3dia_s_mean (_2D_W_,:))
      if (allocated(hpmS_s_mean  )) call fm%send_data('hpmS_s_mean'  , hpmS_s_mean  (_2D_W_,:))
      if (allocated(hnmS_s_mean  )) call fm%send_data('hnmS_s_mean'  , hnmS_s_mean  (_2D_W_,:))

      if (allocated(fwf_s_int)) call fm%send_data('fwf_s_int', fwf_s_int (_2D_W_,:))
      if (allocated(phymix_S_fwf_s_mean)) call fm%send_data('phymix_S_fwf_s_mean', phymix_S_fwf_s_mean(_2D_W_,:))
      if (allocated(rvol_s_int)) call fm%send_data('rvol_s_int', rvol_s_int(_2D_W_,:))
      if (allocated(phymix_S_riv_s_mean)) call fm%send_data('phymix_S_riv_s_mean', phymix_S_riv_s_mean(_2D_W_,:))
   end if
#endif

   return
   end subroutine finalize_register_processed_variables
!EOC

!-----------------------------------------------------------------------
!BOP
! !IROUTINE: do_output_processing - read required variables
!
! !INTERFACE:
   subroutine do_output_processing(loop)
!
! !USES:
   use domain, only: az, au, av
   use variables_2d, only: z,D
   use variables_2d, only: U,V,DU,DV
#ifndef NO_3D
   use m3d, only: M
   use variables_3d
#endif
   IMPLICIT NONE
!
! !DESCRIPTION:
!
! !INPUT PARAMETERS:
   integer, intent(in) :: loop
!
! !REVISION HISTORY:
!  Original author(s): Karsten Bolding
!
! !LOCAL VARIABLES:
   REALTYPE, parameter                 :: vel_missing        =-9999.
   logical       :: do_3d
   REALTYPE      :: dzlk(0:zbins,0:kmax)
   integer       :: lmin(0:kmax)
   integer       :: lmax(0:kmax)
   integer, save :: n_h_z=0
   integer, save :: n_hS_z=0
   integer, save :: n_hu_z=0
   integer, save :: n_hv_z=0
   integer, save :: n_uu_z=0
   integer, save :: n_vv_z=0
   integer, save :: n_Sfluxu_z=0
   integer, save :: n_Sfluxv_z=0
   integer, save :: n_hpmS_z=0
   integer, save :: n_hnmS_z=0
#ifndef NO_BAROCLINIC
   integer, save :: n_h_s=0
   integer, save :: n_hS_s=0
   integer, save :: n_hS2_s=0
   integer, save :: n_hu_s=0
   integer, save :: n_hv_s=0
   integer, save :: n_uu_s=0
   integer, save :: n_vv_s=0
   integer, save :: n_wdia_s=0
   integer, save :: n_Sfluxu_s=0
   integer, save :: n_Sfluxv_s=0
   integer, save :: n_f3dia_s=0
   integer, save :: n_hpmS_s=0
   integer, save :: n_hnmS_s=0
   integer, save :: n_fwf_s=0
   integer, save :: n_phymix_S_fwf_s=0
   integer, save :: n_rvol_s=0
   integer, save :: n_phymix_S_riv_s=0
#endif
   integer       :: i,j,k,l
!EOP
!-------------------------------------------------------------------------
!BOC

#ifdef _POINTER_ZZO_
!  update pointers after pointer swap in sealevel()
   !call fm%send_data('z' , z (_2D_W_))
   !call fm%send_data('zo', zo(_2D_W_))
#endif

!  2D - velocities

   if (u2d_now) then
      call to_2d_vel(imin,jmin,imax,jmax,au,U,DU,vel_missing,       &
                     imin,jmin,imax,jmax,u2d)
   end if

   if (v2d_now) then
      call to_2d_vel(imin,jmin,imax,jmax,av,V,DV,vel_missing,       &
                     imin,jmin,imax,jmax,v2d)
   end if

!  3D - velocities
#ifndef NO_3D
   if (u3d_now) then
      call to_3d_uu(imin,jmin,imax,jmax,kmin,kmax,az, &
                    hun,uu,vel_missing,u3d)
   end if

   if (v3d_now) then
      call to_3d_vv (imin,jmin,imax,jmax,kmin,kmax,az, &
                     hvn,vv,vel_missing,v3d)
   end if
#endif


   if (u2d_destag_now .and. v2d_destag_now) then
      call to_2d_u(imin,jmin,imax,jmax,az,U,DU,vel_missing,      &
                   imin,jmin,imax,jmax,u2d_destag)
      call to_2d_v(imin,jmin,imax,jmax,az,V,DV,vel_missing,      &
                   imin,jmin,imax,jmax,v2d_destag)
   end if

#if 0
#ifndef NO_3D
   if (allocated(u3d_destag) .and. allocated(v3d_destag)) then
      call to_3d_vel(imin,jmin,imax,jmax,kmin,kmax,au, &
                     hun,uu,vel_missing,u3d_destag)
      call to_3d_vel(imin,jmin,imax,jmax,kmin,kmax,av, &
                     hvn,vv,vel_missing,v3d_destag)
   end if
#endif
#endif

#ifndef NO_3D
   do_3d = (mod(loop,M) .eq. 0)

   if (do_3d) then
      if (process_zbin) then
         if (allocated(h_z).or.allocated(hS_z).or.allocated(hpmS_z).or.allocated(hnmS_z)) then
            if (allocated(h_z   )) h_z    = _ZERO_
            if (allocated(hS_z  )) hS_z   = _ZERO_
            if (allocated(hpmS_z)) hpmS_z = _ZERO_
            if (allocated(hnmS_z)) hnmS_z = _ZERO_
            do j = jmin,jmax
               do i = imin,imax
                  if (az(i,j) .eq. 1) then
                     call remap1(kmax,zwn(i,j,:),zbins,zi_z,dzlk(:,1:),lmin(1:),lmax(1:))
                     do k=1,kmax
                        do l=lmin(k),lmax(k)
                           if (allocated(h_z   )) h_z   (i,j,l) = h_z   (i,j,l) + dzlk(l,k)
#ifndef NO_BAROCLINIC
                           if (allocated(hS_z  )) hS_z  (i,j,l) = hS_z  (i,j,l) + dzlk(l,k)*S(i,j,k)
                           if (allocated(hpmS_z)) hpmS_z(i,j,l) = hpmS_z(i,j,l) + dzlk(l,k)*phymix_S(i,j,k)/hn(i,j,k)
                           if (allocated(hnmS_z)) hnmS_z(i,j,l) = hnmS_z(i,j,l) + dzlk(l,k)*nummix_S(i,j,k)/hn(i,j,k)
#endif
                        end do
                     end do
                  end if
               end do
            end do
         end if
         if (allocated(hu_z).or.allocated(uu_z).or.allocated(Sfluxu_z)) then
            if (allocated(hu_z    )) hu_z     = _ZERO_
            if (allocated(uu_z    )) uu_z     = _ZERO_
            if (allocated(Sfluxu_z)) Sfluxu_z = _ZERO_
            do j = jmin,jmax
               do i = imin,imax
                  if (au(i,j).eq.1 .or. au(i,j).eq.2) then
                     call remap1(kmax,zwu(i,j,:),zbins,zi_z,dzlk(:,1:),lmin(1:),lmax(1:))
                     do k=1,kmax
                        do l=lmin(k),lmax(k)
                           if (allocated(hu_z    )) hu_z    (i,j,l) = hu_z    (i,j,l) + dzlk(l,k)
                           if (allocated(uu_z    )) uu_z    (i,j,l) = uu_z    (i,j,l) + dzlk(l,k)*velu3d(i,j,k)
                           if (allocated(Sfluxu_z)) Sfluxu_z(i,j,l) = Sfluxu_z(i,j,l) + dzlk(l,k)*Sfluxu(i,j,k)/hun(i,j,k)
                        end do
                     end do
                  end if
               end do
            end do
         end if
         if (allocated(hv_z).or.allocated(vv_z).or.allocated(Sfluxv_z)) then
            if (allocated(hv_z    )) hv_z     = _ZERO_
            if (allocated(vv_z    )) vv_z     = _ZERO_
            if (allocated(Sfluxv_z)) Sfluxv_z = _ZERO_
            do j = jmin,jmax
               do i = imin,imax
                  if (av(i,j).eq.1 .or. av(i,j).eq.2) then
                     call remap1(kmax,zwv(i,j,:),zbins,zi_z,dzlk(:,1:),lmin(1:),lmax(1:))
                     do k=1,kmax
                        do l=lmin(k),lmax(k)
                           if (allocated(hv_z    )) hv_z    (i,j,l) = hv_z    (i,j,l) + dzlk(l,k)
                           if (allocated(vv_z    )) vv_z    (i,j,l) = vv_z    (i,j,l) + dzlk(l,k)*velv3d(i,j,k)
                           if (allocated(Sfluxv_z)) Sfluxv_z(i,j,l) = Sfluxv_z(i,j,l) + dzlk(l,k)*Sfluxv(i,j,k)/hvn(i,j,k)
                        end do
                     end do
                  end if
               end do
            end do
         end if
      end if
   end if

   if (process_zbinmean) then
      call calc_mean(h_z     , h_z_mean_now     , n_h_z     , h_z_mean     )
      call calc_mean(hS_z    , hS_z_mean_now    , n_hS_z    , hS_z_mean    )
      call calc_mean(hu_z    , hu_z_mean_now    , n_hu_z    , hu_z_mean    )
      call calc_mean(hv_z    , hv_z_mean_now    , n_hv_z    , hv_z_mean    )
      call calc_mean(uu_z    , uu_z_mean_now    , n_uu_z    , uu_z_mean    )
      call calc_mean(vv_z    , vv_z_mean_now    , n_vv_z    , vv_z_mean    )
      call calc_mean(Sfluxu_z, Sfluxu_z_mean_now, n_Sfluxu_z, Sfluxu_z_mean)
      call calc_mean(Sfluxv_z, Sfluxv_z_mean_now, n_Sfluxv_z, Sfluxv_z_mean)
      call calc_mean(hpmS_z  , hpmS_z_mean_now  , n_hpmS_z  , hpmS_z_mean  )
      call calc_mean(hnmS_z  , hnmS_z_mean_now  , n_hnmS_z  , hnmS_z_mean  )
   end if
#endif

#ifndef NO_BAROCLINIC
   if (process_TEFmean) then
      call calc_mean(h_s     , h_s_mean_now     , n_h_s     , h_s_mean     )
      call calc_mean(hS_s    , hS_s_mean_now    , n_hS_s    , hS_s_mean    )
      call calc_mean(hS2_s   , hS2_s_mean_now   , n_hS2_s   , hS2_s_mean   )
      call calc_mean(hu_s    , hu_s_mean_now    , n_hu_s    , hu_s_mean    )
      call calc_mean(hv_s    , hv_s_mean_now    , n_hv_s    , hv_s_mean    )
      call calc_mean(uu_s    , uu_s_mean_now    , n_uu_s    , uu_s_mean    )
      call calc_mean(vv_s    , vv_s_mean_now    , n_vv_s    , vv_s_mean    )
      call calc_mean(wdia_s  , wdia_s_mean_now  , n_wdia_s  , wdia_s_mean  )
      call calc_mean(Sfluxu_s, Sfluxu_s_mean_now, n_Sfluxu_s, Sfluxu_s_mean)
      call calc_mean(Sfluxv_s, Sfluxv_s_mean_now, n_Sfluxv_s, Sfluxv_s_mean)
      call calc_mean(f3dia_s , f3dia_s_mean_now , n_f3dia_s , f3dia_s_mean )
      call calc_mean(hpmS_s  , hpmS_s_mean_now  , n_hpmS_s  , hpmS_s_mean  )
      call calc_mean(hnmS_s  , hnmS_s_mean_now  , n_hnmS_s  , hnmS_s_mean  )

      call calc_mean(fwf_s, fwf_s_int_now, n_fwf_s , fwf_s_int, no_avg=.true.)
      call calc_mean(phymix_S_fwf_s, phymix_S_fwf_s_mean_now, n_phymix_S_fwf_s, phymix_S_fwf_s_mean)
      call calc_mean(rvol_s, rvol_s_int_now , n_rvol_s, rvol_s_int, no_avg=.true.)
      call calc_mean(phymix_S_riv_s, phymix_S_riv_s_mean_now, n_phymix_S_riv_s, phymix_S_riv_s_mean)
   end if
#endif

   return

   contains
#ifndef NO_3D
      subroutine calc_mean(snapshot, now, n, mean, no_avg)
         REALTYPE, allocatable, intent(in)    :: snapshot(:,:,:)
         logical              , intent(in)    :: now
         integer              , intent(inout) :: n
         REALTYPE, allocatable, intent(inout) :: mean(:,:,:)
         logical , optional   , intent(in)    :: no_avg
         logical                              :: do_avg
         logical , save                       :: first=.true.
         if (allocated(mean)) then
            if (first) then
               LEVEL0 "WARNING: Using GETM-internal mean calculation for flexible output."
               LEVEL0 "         Make sure that each mean quantity is requested with only one output frequency!"
               first = .false.
            end if
            if (n .eq. 0) mean = _ZERO_
            if (do_3d) then
               mean = mean + snapshot
               n = n + 1
            end if
            if (now) then
               do_avg = .true.
               if (present(no_avg)) do_avg = (.not. no_avg)
               if (do_avg .and. n.gt.0) mean = mean / n
               n = 0
            end if
         end if
      end subroutine calc_mean
#endif

   end subroutine do_output_processing
!EOC

   end module output_processing

!-----------------------------------------------------------------------
! Copyright (C) 2019 - Karsten Bolding & Jorn Bruggeman (BB)           !
!-----------------------------------------------------------------------
