#include "cppdefs.h"
module glacial_ice

use parameters, only: rk
use parameters_ice
use domain, only: imin, imax, jmin, jmax
use domain, only: sdom

IMPLICIT NONE
private

public init_glacial_ice
public do_glacial_ice
public register_glacial_ice_variables

logical, public :: have_glacial_ice=.false.
REALTYPE, dimension(:,:), allocatable, public :: Tb
REALTYPE, dimension(:,:), allocatable, public :: Sb
REALTYPE, dimension(:,:), allocatable, public :: vm
REALTYPE, dimension(:,:), allocatable, public :: bmf_heat
REALTYPE, dimension(:,:), allocatable, public :: bmf_salt

REALTYPE, dimension(:,:), allocatable :: glIceD

!  KK-TODO: assumed-size array should be turned into explict-shape array
   interface
      subroutine get_2d_field(fn,varname,il,ih,jl,jh,break_on_missing,f)
         character(len=*),intent(in)   :: fn,varname
         integer, intent(in)           :: il,ih,jl,jh
         logical, intent(in)           :: break_on_missing
         REALTYPE, intent(out)         :: f(:,:)
      end subroutine get_2d_field
   end interface

contains

   subroutine init_glacial_ice(runtype, ice_file, icep)

   use parameters, only: grav=>g, rho_0
   use domain, only: az
   use domain, only: ill, ihl, jll, jhl, ilg, ihg, jlg, jhg
   use domain, only: z0s

   IMPLICIT NONE

   integer, intent(in)                        :: runtype
   character(len=*), intent(in)               :: ice_file
   REALTYPE, dimension(E2DFIELD), intent(out) :: icep

   LEVEL2 'init_glacial_ice'

   allocate(glIceD(E2DFIELD))

   if (runtype .gt. 2) then
      allocate(Tb(E2DFIELD))
      allocate(Sb(E2DFIELD))
      allocate(vm(E2DFIELD))
      vm = _ZERO_
      allocate(bmf_heat(E2DFIELD))
      bmf_heat = _ZERO_
      allocate(bmf_salt(E2DFIELD))
      bmf_salt = _ZERO_
   end if

   call get_2d_field( trim(ice_file), "glIceD", ilg, ihg, jlg, jhg, .true., glIceD(ill:ihl,jll:jhl) )
   where (az .eq. 0) glIceD = -9999.0_rk

   where (glIceD .gt. _ZERO_) sdom = 3

   where (sdom .eq. 3)
      z0s = z0_ice
      icep = rho_ice * grav * glIceD
   end where

   end subroutine init_glacial_ice


   subroutine do_glacial_ice(runtype, ustar, T, S, h, z0, zb)

   IMPLICIT NONE

   integer, intent(in)                       :: runtype
   REALTYPE, dimension(E2DFIELD), intent(in) :: ustar
   REALTYPE, dimension(E2DFIELD), intent(in) :: T
   REALTYPE, dimension(E2DFIELD), intent(in) :: S
   REALTYPE, dimension(E2DFIELD), intent(in) :: h
   REALTYPE, dimension(E2DFIELD), intent(in) :: z0
   REALTYPE, dimension(E2DFIELD), intent(in) :: zb

   integer :: i,j

#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'do_glacial_ice() # ',Ncall
#endif

   if (runtype .gt. 2) then
   do j=jmin-HALO,jmax+HALO
      do i=imin-HALO,imax+HALO
         if ( sdom(i,j) .eq. 3 ) call fluxes_under_ice(ustar(i,j), T(i,j), S(i,j), h(i,j), z0(i,j), zb(i,j), Tb(i,j), Sb(i,j), vm(i,j), bmf_heat(i,j), bmf_salt(i,j))
      end do
   end do
   end if

!  TODO: update ice thickness glIceD and ice pressure icep

#ifdef DEBUG
   write(debug,*) 'Leaving do_glacial_ice()'
   write(debug,*)
#endif

   end subroutine do_glacial_ice


   subroutine register_glacial_ice_variables(fm, runtype)

   use field_manager
   use domain, only: imin, imax, jmin, jmax
   IMPLICIT NONE

   type (type_field_manager) :: fm
   integer, intent(in)       :: runtype

   LEVEL2 'register_glacial_ice_variables'

   call fm%register('glIceD', 'm', 'glacial ice thickness', dimensions=(/id_dim_lon,id_dim_lat/), no_default_dimensions=.true., data2d=glIceD(_2D_W_), category='glacial_ice', fill_value=-9999.0_rk, output_level=output_level_debug)

   if (runtype .gt. 2) then
      call fm%register('Tb', 'degC', 'melt layer temperature', data2d=Tb(_2D_W_), category='glacial_ice', fill_value=-9999.0_rk, output_level=output_level_debug)
      call fm%register('Sb', 'g kg-1', 'melt layer salinity', data2d=Sb(_2D_W_), category='glacial_ice', fill_value=-9999.0_rk, output_level=output_level_debug)
      call fm%register('vm', 'm s-1', 'melt rate', data2d=vm(_2D_W_), category='glacial_ice', fill_value=-9999.0_rk, output_level=output_level_debug)
      call fm%register('bmf_heat', 'W m-2', 'basal heat flux', data2d=bmf_heat(_2D_W_), category='glacial_ice', fill_value=-9999.0_rk, output_level=output_level_debug)
      call fm%register('bmf_salt', 'g kg-1 m s-1', 'basal salt flux', data2d=bmf_salt(_2D_W_), category='glacial_ice', fill_value=-9999.0_rk, output_level=output_level_debug)
   end if

   end subroutine register_glacial_ice_variables

end module
