#include "cppdefs.h"
      pure subroutine fluxes_under_ice(ustar, T, S, h, z0, zb, Tb, Sb, vm, fH, fS)

      use parameters, only: nu=>avmmol, kappa, c=>cp, rho0=>rho_0
      use parameters_ice, only: rhoi=>rho_ice

      implicit none

      !input variables
      REALTYPE, intent(in) :: ustar  ! upper layer friction velocity
      REALTYPE, intent(in) :: T      ! upper layer potential temperature
      REALTYPE, intent(in) :: S      ! upper layer salinity
      REALTYPE, intent(in) :: h      ! upper layer thickness
      REALTYPE, intent(in) :: z0     ! roughness length under ice
      REALTYPE, intent(in) :: zb     ! depth of ice-ocean interface

      !output variables
      REALTYPE, intent(out) :: Tb  ! melt layer temperature
      REALTYPE, intent(out) :: Sb  ! melt layer salinity
      REALTYPE, intent(out) :: vm  ! melt rate                ! m / y
      REALTYPE, intent(out) :: fH  ! upward heat flux         ! W / m2
      REALTYPE, intent(out) :: fS  ! upward salinity flux     ! (m/s)*(g/kg)

      !empirical parameters
      !REALTYPE, parameter :: nu     = 1.95e-6 ! molecular viscosity [m2/s]
      !REALTYPE, parameter :: kappa  = 0.4     ! von Karman constant [-]
      REALTYPE, parameter :: Bs     = 8.5     ! constant for tracer roughness [-]
      REALTYPE, parameter :: Pr_turb= 0.7     ! turbulent Prandtl number [-]
      REALTYPE, parameter :: Pr_temp= 13.8    ! molecular Prandtl for temp [-]
      REALTYPE, parameter :: Pr_salt= 2432.   ! turbulent Prandtl for salt [-]
      !REALTYPE, parameter :: c      = 4180.   ! heat capacity of sea water [J/(kg K)]
      REALTYPE, parameter :: ci     = 1995.   ! heat capacity of glacial ice [J/(kg K)]
      REALTYPE, parameter :: Ti     = -20.    ! ice core temperature [deg C]
      REALTYPE, parameter :: Li     = 3.33e5  ! latent heat of fusion [J/kg]
      !REALTYPE, parameter :: rho0   = 1030.   ! reference density of sea water [kg/m3]
      !REALTYPE, parameter :: rhoi   =  920.   ! reference density of glacial ice [kg/m3]

      ! param. for freezing in-situ temperature eq.
      !REALTYPE, parameter :: l1 =  -0.0575
      !REALTYPE, parameter :: l2 =   0.0901
      !REALTYPE, parameter :: l3 =  7.61d-4

      ! param. for freezing potential temperature eq.
      REALTYPE, parameter :: l1 = -5.6705121472405570d-002
      REALTYPE, parameter :: l2 =  7.5436448744204881d-002
      REALTYPE, parameter :: l3 =  7.6828512903539831d-004

      !local parameters
      REALTYPE log_zk_p_z0_z0t,log_zk_p_z0_z0s
      REALTYPE betaT,betaS
      REALTYPE A1T,A1S,LL
      REALTYPE s1,s2,s3,pp,qq

      if (ustar.lt.1e-10) then ! here we are also for z0.le.0
         fH = 0.0
         fS = 0.0
         vm = 0.0
         return
      end if

      betaT=0.55*exp(0.5*kappa*Bs)*sqrt(z0*ustar/nu)
      betaT=betaT*(Pr_temp**(2./3.)-0.2)
      betaT=betaT-Pr_turb*Bs+9.5

      betaS=0.55*exp(0.5*kappa*Bs)*sqrt(z0*ustar/nu)
      betaS=betaS*(Pr_salt**(2./3.)-0.2)
      betaS=betaS-Pr_turb*Bs+9.5

      log_zk_p_z0_z0t=log((0.5*h+z0)/z0)+kappa/Pr_turb*betaT
      log_zk_p_z0_z0s=log((0.5*h+z0)/z0)+kappa/Pr_turb*betaS

      A1T=kappa*ustar/(Pr_turb*log_zk_p_z0_z0t)
      A1S=kappa*ustar/(Pr_turb*log_zk_p_z0_z0s)
      LL=l2+l3*zb

      s1=l1*(A1T-A1S*ci/c)
      s2=A1S*S*l1*ci/c-A1S/c*(ci*(LL-Ti)+Li)-A1T*(T-LL)
      s3=A1S*S/c*(ci*(LL-Ti)+Li)
      pp=s2/s1
      qq=s3/s1

      Sb = -0.5*pp+sqrt(0.25*pp**2-qq)     ! melt layer salinity
      Tb = l1*Sb+LL                        ! melt layer temperature

      fH = A1T*(T-Tb)*c*rho0               ! upward heat flux
      fS = A1S*(S-Sb)                      ! upward salinity flux

      vm = A1T*(T-Tb)/(ci/c*(Tb-Ti)+Li/c)  ! melt rate in m/s
                                           ! (>0 for melting)
                                           ! (rate of incoming ocean water)

!     fluxes for free-surface models here!
!     KK-TODO: check normalization with rhoi and rho0

      fH = (A1T*(T-Tb)-vm*Tb)*c*rho0       ! upward heat flux in W/m2
                                           ! (>0 for melting)

      fS = A1S*(S-Sb)-vm*Sb                ! upward salinity flux
                                           ! (should be zero)

      return
      end subroutine fluxes_under_ice
