#include "cppdefs.h"
!-----------------------------------------------------------------------
!BOP
!
! !MODULE:  getm_nuopc - routines to integrate GETM into NUOPC
!
! !INTERFACE:
   module getm_nuopc
!
! !DESCRIPTION:
!
! !USES:
   use esmf
   use NUOPC

   IMPLICIT NONE
   private
!
! !PUBLIC DATA MEMBERS:
   public SetServices ! (NUOPC requires this name), must be public
   public do_getm_nuopc
   public FieldDictionarySetup

!  Note (KK): __FILE__ includes full path, thus too long for log
   character(len=*),parameter :: FILENAME="getm_nuopc.F90"
!EOP
!-----------------------------------------------------------------------

   contains

!-----------------------------------------------------------------------

   subroutine SetServices(model, rc)
   IMPLICIT NONE
   type(ESMF_GridComp)  :: model
   integer, intent(out) :: rc
   call model_SetServices(model, rc)
   end subroutine SetServices

!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: do_getm_nuopc
!
! !INTERFACE:
   subroutine do_getm_nuopc()
!
! !USES: 
   use getm_esmf, only: createGetmClock
   use getm_timers, only: tic, toc, TIM_ESMF
   IMPLICIT NONE
!
! !LOCAL VARIABLES:
   type(ESMF_Clock)    :: mainClock
   type(ESMF_GridComp) :: getmDriver
   integer             :: rc, userRc
   logical             :: abort
!EOP
!-----------------------------------------------------------------------
!BOC
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'do_getm_nuopc() # ',Ncall
#endif

   call tic(TIM_ESMF)

   call ESMF_Initialize(defaultCalKind=ESMF_CALKIND_GREGORIAN,         &
                        logappendflag=.false.)
   call ESMF_LogSet(flush=.true.,                                      &
                    logmsgList=(/ESMF_LOGMSG_NOTRACE/))

   mainClock = createGetmClock()

   getmDriver = ESMF_GridCompCreate(name="getmDriver", clock=mainClock, rc=rc)
   abort = ESMF_LogFoundError(rc,line=__LINE__,file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

!  NUOPC-provided routines only set userRc, but do not break in case of errors
   call ESMF_GridCompSetServices(getmDriver, driver_SetServices, userRc=userRc, rc=rc)
   abort = ESMF_LogFoundError(userRc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call toc(TIM_ESMF)

   call ESMF_GridCompInitialize(getmDriver, userRc=userRc, rc=rc)
   abort = ESMF_LogFoundError(userRc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call ESMF_GridCompRun(getmDriver, userRc=userRc, rc=rc)
   abort = ESMF_LogFoundError(userRc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call ESMF_GridCompFinalize(getmDriver)

   call tic(TIM_ESMF)

   call ESMF_GridCompDestroy(getmDriver)

   call ESMF_Finalize()

   call toc(TIM_ESMF)

#ifdef DEBUG
   write(debug,*) 'Leaving do_getm_nuopc()'
   write(debug,*)
#endif
   return

   end subroutine do_getm_nuopc
!EOC
!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: driver_SetServices
!
! !INTERFACE:
   subroutine driver_SetServices(driver, rc)
!
! !DESCRIPTION:
!  Derive driver from generic NUOPC driver and attach specializing
!  routines.
!  required: SetModelServices
!  optional: SetRunSequence
!            ModifyInitializePhaseMap
!            ModifyCplLists
!            SetRunClock
!            ExecuteRunSequence
!            Finalize
!
! !USES:
   use NUOPC_Driver, only: driverSS => SetServices, label_SetModelServices
   use getm_timers, only: tic, toc, TIM_ESMF
   IMPLICIT NONE
!
! !INPUT/OUTPUT PARAMETERS:
   type(ESMF_GridComp) :: driver
!
! !OUTPUT PARAMETERS:
   integer,intent(out) :: rc
!
! !LOCAL VARIABLES:
   logical             :: abort
!EOP
!-----------------------------------------------------------------------
!BOC
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'driver_SetServices() # ',Ncall
#endif

   call tic(TIM_ESMF)

   call NUOPC_CompDerive(driver, driverSS, rc=rc)
   abort = ESMF_LogFoundError(rc,line=__LINE__,file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call NUOPC_CompAttributeSet(driver, name="Verbosity", value="25089", rc=rc)

!  required specialization
   call NUOPC_CompSpecialize(driver, specLabel = label_SetModelServices, &
                             specRoutine = driver_SetModelServices, rc=rc)
   abort = ESMF_LogFoundError(rc,line=__LINE__,file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call toc(TIM_ESMF)

#ifdef DEBUG
   write(debug,*) 'Leaving driver_SetServices()'
   write(debug,*)
#endif
   return

   end subroutine driver_SetServices
!EOC
!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: driver_SetModelServices
!
! !INTERFACE:
  subroutine driver_SetModelServices(driver, rc)
!
! !DESCRIPTION:
!  Add child components to driver.
!  Use NUOPC_CompAttributeSet() or NUOPC_CompAttributeIngest() to set
!  attributes on child components.
!  Optionally create or replace default driverClock (copy of clock
!  provided to GridCompCreate(driver), if valid).
!
! !USES:
   use NUOPC_Driver, only: NUOPC_DriverAddComp, NUOPC_DriverEgestRunSequence
   use getm_timers, only: tic, toc, TIM_ESMF
   IMPLICIT NONE
!
! !INPUT/OUTPUT PARAMETERS:
    type(ESMF_GridComp) :: driver
!
! !OUTPUT PARAMETERS:
   integer,intent(out)  :: rc
!
! !LOCAL VARIABLES:
   type(ESMF_Clock)       :: clock
   type(ESMF_GridComp)    :: getmModel
   type(NUOPC_FreeFormat) :: freeFormat
!EOP
!-----------------------------------------------------------------------
!BOC
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'driver_SetModelServices() # ',Ncall
#endif

   call tic(TIM_ESMF)

   call NUOPC_DriverAddComp(driver, "GETM", model_SetServices, comp=getmModel, rc=rc)
   call NUOPC_CompAttributeSet(getmModel, name="Verbosity", value="25089", rc=rc)

   call NUOPC_DriverEgestRunSequence(driver, freeFormat)
   call ESMF_LogWrite("", ESMF_LOGMSG_INFO)
   call ESMF_LogWrite("RUN SEQUENCE:", ESMF_LOGMSG_INFO)
   call ESMF_LogWrite("", ESMF_LOGMSG_INFO)
   call NUOPC_FreeFormatLog(freeFormat)
   call ESMF_LogWrite("", ESMF_LOGMSG_INFO)

!  Usually the driverClock must be set here...
   !call ESMF_GridCompSet(driver, clock=clock, rc=rc)
   !abort = ESMF_LogFoundError(rc,line=__LINE__,file=FILENAME)
   !if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call driver_FieldDictionarySetup()

   call NUOPC_FreeFormatDestroy(freeFormat)

   call toc(TIM_ESMF)

#ifdef DEBUG
   write(debug,*) 'Leaving driver_SetModelServices()'
   write(debug,*)
#endif
   return

   contains

      subroutine driver_FieldDictionarySetup()

      use NUOPC_FieldDictionaryApi, only: NUOPC_FieldDictionaryIsSetup
      use NUOPC_FieldDictionaryDef, only: NUOPC_FieldDictionaryCreateI

      !call NUOPC_FieldDictionarySetupEmpty()
      call NUOPC_FieldDictionaryCreateI(NUOPC_FieldDictionary)
      NUOPC_FieldDictionaryIsSetup = .true.
      call NUOPC_FieldDictionarySetAutoAdd(.true.)

      call FieldDictionarySetup()
      call NUOPC_FieldDictionaryEgest(freeFormat)
      call ESMF_LogWrite("", ESMF_LOGMSG_INFO)
      call ESMF_LogWrite("FIELD DICTIONARY:", ESMF_LOGMSG_INFO)
      call ESMF_LogWrite("", ESMF_LOGMSG_INFO)
      call NUOPC_FreeFormatLog(freeFormat)
      call ESMF_LogWrite("", ESMF_LOGMSG_INFO)

      end subroutine driver_FieldDictionarySetup

   end subroutine driver_SetModelServices
!EOC
!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: model_SetServices - register GriddedComponent GETM
!
! !INTERFACE:
   subroutine model_SetServices(model, rc)
!
! !DESCRIPTION:
!  Register user-code subroutines.
!  For the NUOPC Layer this routine must be named SetServices().
!  The toplevel component can inquire rc via optional keyword argument
!  userRc to ESMF_GridCompSetServices().
!
! !USES:
   use NUOPC_Model, only: modelSS=>SetServices, label_Advance
   use NUOPC_Model, only: label_AdvanceClock, label_Finalize
   use getm_esmf,   only: InitializeP1, InitializeP2
   use getm_timers, only: tic, toc, TIM_ESMF
   IMPLICIT NONE
!
! !INPUT/OUTPUT PARAMETERS:
   type(ESMF_GridComp)  :: model
!
! !OUTPUT PARAMETERS:
   integer, intent(out) :: rc
!
! !REVISION HISTORY:
!
! !LOCAL VARIABLES
   logical                    :: abort
   character(len=ESMF_MAXSTR) :: name
!
!EOP
!-----------------------------------------------------------------------
!BOC
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'model_SetServices() # ',Ncall
#endif

   call tic(TIM_ESMF)

   call NUOPC_CompGet(model, name=name, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call ESMF_LogWrite(trim(name)//"::model_SetServices...", ESMF_LOGMSG_TRACE)

   call NUOPC_CompDerive(model , modelSS, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

!  TODO: calling NUOPC_CompSetEntryPoint() is deprecated.
!        change to Specialize(label_Advertise)!
   call NUOPC_CompSetEntryPoint(model, ESMF_METHOD_INITIALIZE,         &
                                phaseLabelList = (/"IPDv00p1"/),       &
                                userRoutine = InitializeP1, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

!  TODO: calling NUOPC_CompSetEntryPoint() is deprecated.
!        change to Specialize(label_RealizeAccepted)!
   call NUOPC_CompSetEntryPoint(model, ESMF_METHOD_INITIALIZE,         &
                                phaseLabelList = (/ "IPDv00p2" /),     &
                                userRoutine = InitializeP2, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

!  required specialization
   call NUOPC_CompSpecialize(model, specLabel = label_Advance,         &
                             specRoutine = Advance, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

!  clock is already advanced in getm_esmf.F90::RunP1()
   call NUOPC_CompSpecialize(model, specLabel = label_AdvanceClock,    &
                             specRoutine = NUOPC_NoOp, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call NUOPC_CompSpecialize(model, specLabel = label_Finalize,        &
                             specRoutine = Finalize, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call toc(TIM_ESMF)

#ifdef DEBUG
   write(debug,*) 'Leaving model_SetServices()'
   write(debug,*)
#endif
   return

   end subroutine model_SetServices
!EOC
!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: Advance - advance the model forward in time
!
! !INTERFACE:
   subroutine Advance(model, rc)
!
! !DESCRIPTION:
!  A specialization method that advances the model (including clock!)
!  forward in time by one timestep of the internal clock.
!  This method will be called iteratively by the default run phase
!  until reaching the stop time on the internal clock.
!  If not overridden, the default method SetRunClock() will check that
!  the internal clock and incoming clock agree on the current time and
!  that the time step of the incoming clock is a multiple of the
!  internal clock time step.
!  Under these conditions the internal stop time is set to one time step
!  interval of the incoming clock.
!  Otherwise exit with error, flagging an incompatibility.
!
! !USES:
   use NUOPC_Model, only: NUOPC_ModelGet
   use getm_esmf,   only: RunP1
   use getm_timers, only: tic, toc, TIM_ESMF
   implicit none
!
! !INPUT/OUTPUT PARAMETERS:
   type(ESMF_GridComp)  :: model
!
! !OUTPUT PARAMETERS:
   integer, intent(out) :: rc
!
! !REVISION HISTORY:
!
! !LOCAL VARIABLES
   type(ESMF_State)           :: importState, exportState
   type(ESMF_Clock)           :: modelClock, clock
   type(ESMF_Time)            :: stopTime, currTime
   type(ESMF_TimeInterval)    :: timeStep
   logical                    :: abort
   character(len=ESMF_MAXSTR) :: name
!
!EOP
!-----------------------------------------------------------------------
!BOC
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'Advance() # ',Ncall
#endif

   call tic(TIM_ESMF)

   call NUOPC_CompGet(model, name=name, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call ESMF_LogWrite(trim(name)//"::Advance...",ESMF_LOGMSG_TRACE)

   call NUOPC_ModelGet(model, modelClock = modelClock,                 &
                       importState = importState,                      &
                       exportState = exportState, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   clock = ESMF_ClockCreate(modelClock, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

!  getm_esmf.F90::RunP1() advances until nextTime of incoming clock.
!  Set up incoming clock to cover whole driverTimeStep instead of only
!  one modelTimeStep such that driver needs to call this Advance() only
!  once and data in states need to be im/exported only once as well.
   call ESMF_ClockGet(clock, stopTime=stopTime, currTime=currTime, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   timeStep = stopTime - currTime
   call ESMF_ClockSet(clock, timeStep=timeStep, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call toc(TIM_ESMF)

   call RunP1(model, clock = clock, importState = importState,         &
              exportState = exportState, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call tic(TIM_ESMF)

   call ESMF_ClockDestroy(clock, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call toc(TIM_ESMF)

#ifdef DEBUG
   write(debug,*) 'Leaving Advance()'
   write(debug,*)
#endif
   return

   end subroutine Advance
!EOC
!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: Finalize - finalize the model
!
! !INTERFACE:
   subroutine Finalize(model, rc)
!
! !DESCRIPTION:
!  An optional specialization method.
!
! !USES:
   use NUOPC_Model, only: NUOPC_ModelGet
   use getm_esmf,   only: FinalizeP1
   use getm_timers, only: tic, toc, TIM_ESMF
   implicit none
!
! !INPUT/OUTPUT PARAMETERS:
   type(ESMF_GridComp)  :: model
!
! !OUTPUT PARAMETERS:
   integer, intent(out) :: rc
!
! !REVISION HISTORY:
!
! !LOCAL VARIABLES
   type(ESMF_State)           :: importState, exportState
   type(ESMF_Clock)           :: modelClock
   logical                    :: abort
   character(len=ESMF_MAXSTR) :: name
!
!EOP
!-----------------------------------------------------------------------
!BOC
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'Finalize() # ',Ncall
#endif

   call tic(TIM_ESMF)

   call NUOPC_CompGet(model, name=name, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call ESMF_LogWrite(trim(name)//"::Finalize...",ESMF_LOGMSG_TRACE)

   call NUOPC_ModelGet(model, modelClock = modelClock,                 &
                       importState = importState,                      &
                       exportState = exportState, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call toc(TIM_ESMF)

   call FinalizeP1(model, clock = modelClock, importState = importState, &
                   exportState = exportState, rc=rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

#ifdef DEBUG
   write(debug,*) 'Leaving Finalize()'
   write(debug,*)
#endif
   return

   end subroutine Finalize
!EOC
!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: FieldDictionarySetup - setup dictionary for GETM
!
! !INTERFACE:
   subroutine FieldDictionarySetup(rc)
!
! !DESCRIPTION:
!
! !USES:
   implicit none
!
! !OUTPUT PARAMETERS:
   integer, intent(out), optional :: rc
!
! !REVISION HISTORY:
!
! !LOCAL VARIABLES
   logical                    :: abort
!
!EOP
!-----------------------------------------------------------------------
!BOC
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'FieldDictionarySetup() # ',Ncall
#endif

!  GETM import: wind_x_velocity_at_10m
!  MOSSCO-GETM export: wind_x_velocity_at_10m
!  ICONGETM export: wind_x_velocity_at_10m
!  WW3 import: eastward_wind_at_10m_height
   call FieldDictionaryAddEntrySetSyno((/"wind_x_velocity_at_10m     ",   &
                                         "eastward_wind_at_10m_height"/), &
                                        "m s-1", rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

!  GETM import: wind_y_velocity_at_10m
!  MOSSCO-GETM export: wind_y_velocity_at_10m
!  ICONGETM export: wind_y_velocity_at_10m
!  WW3 import: northward_wind_at_10m_height
   call FieldDictionaryAddEntrySetSyno((/"wind_y_velocity_at_10m      ",   &
                                         "northward_wind_at_10m_height"/), &
                                       "m s-1", rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

#ifdef DEBUG
   write(debug,*) 'Leaving FieldDictionarySetup()'
   write(debug,*)
#endif
   return

   end subroutine FieldDictionarySetup
!EOC
!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: FieldDictionaryAddEntrySetSyno - add synonyms to dictionary
!
! !INTERFACE:
   subroutine FieldDictionaryAddEntrySetSyno(standardNames, units, rc)
!
! !DESCRIPTION:
!
! !USES:
   implicit none
!
! !INPUT PARAMETERS:
   character(*), intent(in)       :: standardNames(:)
   character(*), intent(in)       :: units

! !OUTPUT PARAMETERS:
   integer, intent(out), optional :: rc
!
! !REVISION HISTORY:
!
! !LOCAL VARIABLES
   integer :: i
   logical :: abort
!
!EOP
!-----------------------------------------------------------------------
!BOC
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'FieldDictionaryAddEntrySetSyno() # ',Ncall
#endif

   do i = 1, size(standardNames)

      if (NUOPC_FieldDictionaryHasEntry(standardNames(i))) cycle

      call NUOPC_FieldDictionaryAddEntry(trim(standardNames(i)), units, rc)
      abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
      if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   end do

   call NUOPC_FieldDictionarySetSyno(standardNames, rc)
   abort = ESMF_LogFoundError(rc, line=__LINE__, file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

#ifdef DEBUG
   write(debug,*) 'Leaving FieldDictionaryAddEntrySetSynp()'
   write(debug,*)
#endif
   return

   end subroutine FieldDictionaryAddEntrySetSyno
!EOC
!-----------------------------------------------------------------------



#if 0
  subroutine InitializeP1(gridComp,importState,exportState,clock,rc)
    call getmCmp_init_variables()
    ! advertise export fields
    if (associated(sst)) then
      call advertise_field("sea_surface_temperature",sst,exportState)
    end if
    call getmCmp_update_exportState()
  end subroutine InitializeP1

  subroutine InitializeP2(gridComp,importState,exportState,clock,rc)
    if (associated(sst)) then
       call realize_field("sea_surface_temperature",exportState)!,sst)
    end if
   end subroutine InitializeP2


!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: getmCmp_init_variables
!
! !INTERFACE:
#undef  ESMF_METHOD
#define ESMF_METHOD "getmCmp_init_variables"
   subroutine getmCmp_init_variables()
!
! !DESCRIPTION:
!
! !USES:
   use domain, only: imin, jmin, imax, jmax
#ifndef NO_3D
#ifndef NO_BAROCLINIC
   use m3d   , only: calc_temp
#endif
#endif

   IMPLICIT NONE
!
! !INPUT/OUTPUT PARAMETERS:
!
! !REVISION HISTORY:
!  Original Author(s): Knut Klingbeil
!
! !LOCAL VARIABLES
   REALTYPE :: getmreal
!EOP
!-----------------------------------------------------------------------
!BOC
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'getmCmp_init_variables() # ',Ncall
#endif

   noKindMatch = ( kind(getmreal) .ne. ESMF_KIND_R8 )

#ifndef NO_3D
#ifndef NO_BAROCLINIC
   if (calc_temp) then
      allocate(sst(I2DFIELD))
   end if
#endif
#endif

#ifdef DEBUG
   write(debug,*) 'getmCmp_init_variables()'
   write(debug,*)
#endif
   return

   end subroutine getmCmp_init_variables
!EOC
!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: getmCmp_update_exportState -
!
! !INTERFACE:
#undef  ESMF_METHOD
#define ESMF_METHOD "getmCmp_update_exportState"
   subroutine getmCmp_update_exportState()
!
! !DESCRIPTION:
!
! !USES:
   use domain         ,only: kmax
   use initialise     ,only: runtype
#ifndef NO_3D
#ifndef NO_BAROCLINIC
   use m3d            ,only: calc_temp
   use variables_3d   ,only: T
#endif
#endif

   IMPLICIT NONE
!
! !INPUT/OUTPUT PARAMETERS:
!
! !REVISION HISTORY:
!  Original Author(s): Knut Klingbeil
!
! !LOCAL VARIABLES
!
!EOP
!-----------------------------------------------------------------------
!BOC
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'getmCmp_update_exportState() # ',Ncall
#endif

#ifndef NO_3D
      if (runtype .gt. 1) then
#ifndef NO_BAROCLINIC
         if (calc_temp) then
!           transform from degC to K
            sst = T(:,:,kmax) + 273.15d0
         end if
#endif
      end if
#endif

#ifdef DEBUG
   write(debug,*) 'getmCmp_update_exportState()'
   write(debug,*)
#endif
   return

   end subroutine getmCmp_update_exportState
!EOC
!BOP
!
! !ROUTINE: advertise_field_2d -
!
! !INTERFACE:
   subroutine advertise_field_2d(name,p2d,state,kwe,units,staggerloc)
!
! !DESCRIPTION:
!
! !USES:
   IMPLICIT NONE
!
! !INPUT PARAMETERS:
   character(len=*), intent(in)                            :: name
   real(ESMF_KIND_R8), dimension(:,:), pointer, intent(in) :: p2d
   logical, intent(in), optional :: kwe !keyword-enforcer
   character(len=*), intent(in), optional                  :: units
   type(ESMF_StaggerLoc), intent(in), optional             :: staggerloc
!
! !INPUT/OUTPUT PARAMETERS:
   type(ESMF_State), intent(inout)                         :: state
!
! !REVISION HISTORY:
!  Original Author(s): Knut Klingbeil
!
! !LOCAL VARIABLES
   type(ESMF_Field) :: field
   integer,target   :: elb(2),eub(2)
   integer          :: rc
   logical          :: abort
!
!EOP
!-----------------------------------------------------------------------
!BOC

   call NUOPC_Advertise(state, trim(name), units=units, rc=rc)
   abort = ESMF_LogFoundError(rc,line=__LINE__,file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call ESMF_StateGet(state, trim(name), field=field, rc=rc)
   abort = ESMF_LogFoundError(rc,line=__LINE__,file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call ESMF_FieldEmptySet(field,getmGrid2D,staggerloc=staggerloc,rc=rc)
   abort = ESMF_LogFoundError(rc,line=__LINE__,file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call ESMF_GridGetFieldBounds(getmGrid2D,staggerloc=staggerloc,            &
                                totalLBound=elb(1:2),totalUBound=eub(1:2),rc=rc)
   abort = ESMF_LogFoundError(rc,line=__LINE__,file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

!  in contrast to ESMF_ArrayCreate() no automatic determination of total[L|U]Width
#if 1
!  Note (KK): in former times ESMF_FieldCreateGridDataPtr<rank><type><kind>() failed
   call ESMF_FieldEmptyComplete(field,p2d,                             &
#else
!  internal call to ESMF_FieldCreateGridData<rank><type><kind>()
!  forced by indexflag argument.
   call ESMF_FieldEmptyComplete(field,p2d,ESMF_INDEX_DELOCAL,          &
#endif
                                totalLWidth=int(elb(1:2)-lbound(p2d)), &
                                totalUWidth=int(ubound(p2d)-eub(1:2)), &
                                rc=rc)
   abort = ESMF_LogFoundError(rc,line=__LINE__,file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   return

   end subroutine advertise_field_2d
!EOC
!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: advertise_field_3d -
!
! !INTERFACE:
   subroutine advertise_field_3d(name,p3d,state,kwe,units,staggerloc)
!
! !DESCRIPTION:
!
! !USES:
   IMPLICIT NONE
!
! !INPUT PARAMETERS:
   character(len=*), intent(in)                              :: name
   real(ESMF_KIND_R8), dimension(:,:,:), pointer, intent(in) :: p3d
   logical, intent(in), optional :: kwe !keyword-enforcer
   character(len=*), intent(in), optional                    :: units
   type(ESMF_StaggerLoc), intent(in), optional               :: staggerloc
!
! !INPUT/OUTPUT PARAMETERS:
   type(ESMF_State), intent(inout)                           :: state
!
! !REVISION HISTORY:
!  Original Author(s): Knut Klingbeil
!
! !LOCAL VARIABLES
   type(ESMF_Field) :: field
   integer,target   :: elb(3),eub(3)
   integer          :: rc
   logical          :: abort
!
!EOP
!-----------------------------------------------------------------------
!BOC

   call NUOPC_Advertise(state, trim(name), units=units, rc=rc)
   abort = ESMF_LogFoundError(rc,line=__LINE__,file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call ESMF_StateGet(state, trim(name), field=field, rc=rc)
   abort = ESMF_LogFoundError(rc,line=__LINE__,file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call ESMF_FieldEmptySet(field,getmGrid3D,staggerloc=staggerloc,rc=rc)
   abort = ESMF_LogFoundError(rc,line=__LINE__,file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   call ESMF_GridGetFieldBounds(getmGrid3D,staggerloc=staggerloc,            &
                                totalLBound=elb(1:3),totalUBound=eub(1:3),rc=rc)
   abort = ESMF_LogFoundError(rc,line=__LINE__,file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

!  in contrast to ESMF_ArrayCreate() no automatic determination of total[L|U]Width
#if 1
!  Note (KK): in former times ESMF_FieldCreateGridDataPtr<rank><type><kind>() failed
   call ESMF_FieldEmptyComplete(field,p3d,                             &
#else
!  internal call to ESMF_FieldCreateGridData<rank><type><kind>()
!  forced by indexflag argument.
   call ESMF_FieldEmptyComplete(field,p3d,ESMF_INDEX_DELOCAL,          &
#endif
                                totalLWidth=int(elb(1:3)-lbound(p3d)), &
                                totalUWidth=int(ubound(p3d)-eub(1:3)), &
                                rc=rc)
   abort = ESMF_LogFoundError(rc,line=__LINE__,file=FILENAME)
   if (abort) call ESMF_Finalize(endflag=ESMF_END_ABORT)

   return

   end subroutine advertise_field_3d
!EOC
#endif
!-----------------------------------------------------------------------

   end module getm_nuopc

!-----------------------------------------------------------------------
! Copyright (C) 2020 - Knut Klingbeil (IOW)                            !
!-----------------------------------------------------------------------
