#include "cppdefs.h"
!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: gotm - a wrapper to call GOTM \label{sec-gotm}
!
! !INTERFACE:
   subroutine gotm()
!
! !DESCRIPTION:
!
! Here, the turbulence module of the General Ocean Turbulence Model (GOTM,
! see {\tt www.gotm.net} and \cite{UMLAUFea05}) is called. First, all
! necessary parameters are transformed to suit with a 1D water column model,
! i.e., 3D fields are transformed to a vertical vector, 2D horizontal
! fields are converted to a scalar. The transformed 3D fields are
! the layer heights {\tt hn $\rightarrow$ h}, the shear squared
! {\tt SS $\rightarrow$ SS1d},
! the buoyancy frequency squared {\tt NN $\rightarrow$ NN1d},
! the turbulent kinetic energy {\tt tke $\rightarrow$ tke1d},
! the dissipation rate {\tt eps $\rightarrow$ eps1d}
! (from which the integral length scale {\tt L1d} is calculated), the
! eddy viscosity {\tt num $\rightarrow$ num1d}, and the eddy diffusivity
! {\tt nuh $\rightarrow$ nuh1d}. The scalars are the surface and bottom friction
! velocities, {\tt u\_taus} and {\tt u\_taub}, respectively, the
! surface roughness parameter {\tt z0s} (which is currently hard-coded),
! and the bottom roughess parameter {\tt z0b}.
! Then, the GOTM turbulence module {\tt do\_turbulence} is called with
! all the transformed parameters discussed above. Finally, the
! vertical vectors {\tt tke1d}, {\tt eps1d}, {\tt num1d} and {\tt nuh1d}
! are transformed back to 3D fields.
!
! In case that the compiler option {\tt STRUCTURE\_FRICTION} is switched on,
! the additional turbulence production by structures in the water column is calculated
! by calculating the total production as
! \begin{equation}
! P_{tot} = P +C \left(u^2+v^2\right)^{3/2},
! \end{equation}
! with the shear production $P$, and the structure friction coefficient $C$. The
! latter is calculated in the routine {\tt structure\_friction\_3d.F90}.
!
! There are furthermore a number of compiler options provided, e.g.\
! for an older GOTM version, for barotropic calcuations,
! and for simple parabolic viscosity profiles circumventing the GOTM
! turbulence module.
!
! !USES:
   use halo_zones, only: update_3d_halo,wait_halo,H_TAG
   use domain, only: imin,imax,jmin,jmax,kmax,az,min_depth,crit_depth,z0
   use domain, only: z0surf=>z0s
   use variables_2d, only: D,z
   use variables_3d, only: dt,kmin,ho,hn,tke,eps,SS,num,u_taus,taub,zub,zvb
#ifndef NO_BAROCLINIC
   use variables_3d, only: NN,nuh
#endif
   use variables_3d, only: avmback,avhback
#ifdef STRUCTURE_FRICTION
   use variables_3d, only: velu3d=>veluEuler3d,velv3d=>velvEuler3d,hun,hvn,sf
#endif
   use turbulence, only: do_turbulence,cde
   use turbulence, only: tke1d => tke, eps1d => eps, L1d => L
   use turbulence, only: num1d => num, nuh1d => nuh
   use getm_timers, only: tic, toc, TIM_GOTM, TIM_GOTMTURB, TIM_GOTMH
   use meteo,      only: wind
   IMPLICIT NONE
!
! !REVISION HISTORY:
!  Original author(s): Karsten Bolding & Hans Burchard
!
! !LOCAL VARIABLES:
   integer                   :: i,j,k
   REALTYPE                  :: u_taub,z0s,z0b
   REALTYPE                  :: h(0:kmax),dry,zz
   REALTYPE                  :: NN1d(0:kmax),SS1d(0:kmax)
   REALTYPE                  :: xP(0:kmax)
!EOP
!-----------------------------------------------------------------------
!BOC
! Note: For ifort we need to explicitly state that this routine is
! single-thread only. Presently I don't know why that is necessary,
! but if I use ifort -omp without any OMP-statements in this file,
! then the result is garbage.
! The OMP SINGLE or OMP MASTER statements helps, but sometimes it *still*
! messes up, in the sense that NaN "suddenly" appears on output.
! Apparently, writing out array-copy explicitly helps.
!    BJB 2009-09-17.
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'gotm() # ',Ncall
#endif
   call tic(TIM_GOTM)

   xP = _ZERO_
#ifdef NO_BAROCLINIC
   NN1d = _ZERO_
#endif
   do j=jmin,jmax
      do i=imin,imax

         if (az(i,j) .ge. 1 ) then

            xP = _ZERO_

#ifdef STRUCTURE_FRICTION
! BJB-TODO: Change all constants to double
            do k=1,kmax
               xP(k)= _QUART_*(                                                &
               (velu3d(i  ,j  ,k))**2*(sf(i  ,j  ,k)+sf(i+1,j  ,k)) &
              +(velu3d(i-1,j  ,k))**2*(sf(i-1,j  ,k)+sf(i  ,j  ,k)) &
              +(velv3d(i  ,j  ,k))**2*(sf(i  ,j  ,k)+sf(i  ,j+1,k)) &
              +(velv3d(i  ,j-1,k))**2*(sf(i  ,j-1,k)+sf(i  ,j  ,k)))
            end do
#endif

            u_taub = sqrt(taub(i,j))
            do k=0,kmax
               h(k)    = hn(i,j,k)
               SS1d(k) = SS(i,j,k)
#ifndef NO_BAROCLINIC
               NN1d(k) = NN(i,j,k)
#endif
               tke1d(k)=tke(i,j,k)
               eps1d(k)=eps(i,j,k)
               L1d(k)  =cde*tke1d(k)**1.5/eps1d(k)
               num1d(k)=num(i,j,k)
#ifndef NO_BAROCLINIC
               nuh1d(k)=nuh(i,j,k)
#endif
            end do
            z0s = z0surf(i,j)
            z0b = _HALF_*( max( z0(i,j) , zub(i-1,j  ) , zub(i,j) ) &
                          +max( z0(i,j) , zvb(i  ,j-1) , zvb(i,j) ) )
            if (z0s .le. _ZERO_) z0s = _TENTH_ ! default value from old GETM
            if (z0s .gt. D(i,j)/10.) z0s= D(i,j)/10.

#ifndef NO_BAROCLINIC
            call langmuir(kmax,wind(i,j),h,NN1d,xP)
#endif

#ifdef PARABOLIC_VISCOSITY
            zz = _ZERO_
            do k=1,kmax-1
               zz=zz+hn(i,j,k)
! BJB-TODO: Get rid of **1.5 and **2
               tke1d(k)=max(1.e-10,3.333333*taub(i,j)*(_ONE_-zz/D(i,j)))
               L1d(k)=0.4*(zz+z0b)*sqrt(_ONE_-zz/D(i,j))
               eps1d(k)=0.16431677*sqrt(tke1d(k)*tke1d(k)*tke1d(k))/L1d(k)
               num1d(k)=0.09*tke1d(k)*tke1d(k)/eps1d(k)
#ifndef NO_BAROCLINIC
               nuh1d(k)=num1d(k)
#endif
            end do
#else
            ! If we do tic/toc for do_turbulence, then we can
            ! easily get into the millions of system_clock calls,
            ! as the call is deeply in loops
            !call tic(TIM_GOTMTURB)
            call do_turbulence(kmax,dt,D(i,j),u_taus(i,j),u_taub,z0s,z0b,h, &
                               NN1d,SS1d,xP)
            !call toc(TIM_GOTMTURB)
#endif
            do k=0,kmax
               tke(i,j,k) = tke1d(k)
               eps(i,j,k) = eps1d(k)
               num(i,j,k) = num1d(k) + avmback
#ifndef NO_BAROCLINIC
               nuh(i,j,k) = nuh1d(k) + avhback
#endif
            end do
         end if
      end do
   end do

   call tic(TIM_GOTMH)
   call update_3d_halo(num,num,az,imin,jmin,imax,jmax,kmax,H_TAG)
   call wait_halo(H_TAG)
#ifndef NO_BAROCLINIC
   call update_3d_halo(nuh,nuh,az,imin,jmin,imax,jmax,kmax,H_TAG)
   call wait_halo(H_TAG)
#endif
   call toc(TIM_GOTMH)

   call toc(TIM_GOTM)
#ifdef DEBUG
   write(debug,*) 'Leaving gotm()'
   write(debug,*)
#endif
   return
   end subroutine gotm
!EOC

!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: langmuir - source term parameterisation for Langmuir turbulence
!
! !INTERFACE:
   subroutine langmuir(nlev,wind,h,NN,xP)
!
! !DESCRIPTION:
!
! !USES:
   use m3d, only: c_lc
   IMPLICIT NONE
!
! !INPUT PARAMETERS:
   integer , intent(in)  :: nlev
   REALTYPE, intent(in)  :: wind
   REALTYPE, intent(in)  :: h(0:nlev)
   REALTYPE, intent(in)  :: NN(0:nlev)
!
! !INPUT/OUTPUT PARAMETERS:
   REALTYPE, intent(inout) :: xP(0:nlev)
!
! !REVISION HISTORY:
!  Original author(s): Knut Klingbeil
!
! !LOCAL VARIABLES:
   integer, parameter        :: rk = kind(_ONE_)
   REALTYPE, parameter       :: grav = 9.81_rk
   REALTYPE, parameter       :: pi = 3.14159265358979323846_rk
   integer                   :: i
   REALTYPE                  :: zi(0:nlev),L_lc,us,epot,H_lc,ekin,w_lc,H_lc_inv
!   REALTYPE, parameter       :: c_lc=0.125_rk
!EOP
!-----------------------------------------------------------------------
!BOC
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'langmuir() # ',Ncall
#endif

!  Axell (2002)
   if (c_lc .le. _ZERO_) return

   ! calculate the depth z from the cell heights (negative, downward from the surface)
   zi(nlev) = _ZERO_
   do i=nlev-1,0,-1
     zi(i) = zi(i+1) - h(i+1)
   end do

   epot = _ZERO_
   H_lc = _ZERO_
   us   = 0.016_rk * wind          ! Stokes drift velocity at the surface (m/s)
   ekin = 0.5_rk * us*us           ! kinetic energy of water parcel at surface (m2/s2)

   ! see how deep a surface water parcel could travel using its potential energy
   do i=nlev-1,1,-1
      if ( epot .gt. ekin ) then
         exit
      end if
      epot = epot - NN(i)*zi(i)*h(i+1)  !add (remember that z<0) potential energy required to pass layer i+1 (m2/s2)
      H_lc = H_lc + h(i+1)
   end do

   H_lc_inv = _ONE_ / H_lc
   do i=1,nlev-1
      if ( -H_lc .lt. zi(i) ) then
         w_lc = abs(c_lc * us * sin( -pi * zi(i) * H_lc_inv ))   ! velocity scale (m/s)
         xP(i) = xP(i) + w_lc*w_lc*w_lc * H_lc_inv                   ! TKE production (m2/s3)
      end if
   end do

#ifdef DEBUG
   write(debug,*) 'Leaving langmuir()'
   write(debug,*)
#endif
   return
   end subroutine langmuir
!EOC

!-----------------------------------------------------------------------
! Copyright (C) 2001 - Hans Burchard and Karsten Bolding               !
!-----------------------------------------------------------------------
