#include "cppdefs.h"
!-----------------------------------------------------------------------
!BOP
!
! !MODULE: variables_2d - global variables for 2D model
!
! !INTERFACE:
   module variables_2d
!
! !DESCRIPTION:
!  This modules contains declarations for all variables related to 2D
!  hydrodynamical calculations. Information about the calculation domain
!  is included from the {\tt domain} module.
!  The module contains public subroutines to initialise and cleanup.
!  Depending whether the compiler option {STATIC} is set or not,
!  memory for 2D variables is statically or dynamically allocated, see
!  {\tt PUBLIC DATA MEMBERS}.
!
! !USES:
   use domain, only: imin,imax,jmin,jmax,az
   use domain, only: bottfric_method,rdrag
   use waves , only: waveforcing_method,waves_method,NO_WAVES,WAVES_VF
   IMPLICIT NONE
!
! !PUBLIC DATA MEMBERS:
   integer, parameter                  :: rk = kind(_ONE_)
   REALTYPE                            :: dtm
#ifdef _POINTER_ZZO_
   REALTYPE,dimension(:,:),pointer     :: zo,z
#endif
   logical                             :: deformC=.false.
   logical                             :: deformX=.false.
   logical                             :: deformUV=.false.
   logical                             :: calc_taubmax=.false.
   logical                             :: save_numdis_2d=.false.
   logical                             :: save_phydis_2d=.false.
   logical                             :: save_rvol=.false.

#ifdef STATIC
#include "static_2d.h"
#else
#include "dynamic_declarations_2d.h"
#endif

!  the following fields will be allocated for waves
   REALTYPE,dimension(:,:),pointer,contiguous :: UEuler=>NULL(),VEuler=>NULL()
   REALTYPE,dimension(:,:),pointer,contiguous :: UEulerInt=>NULL(),VEulerInt=>NULL()
   REALTYPE,dimension(:,:),pointer,contiguous :: veluf=>NULL(),velvf=>NULL()
   REALTYPE,dimension(:,:),pointer,contiguous :: veluEuler=>NULL(),velvEuler=>NULL()
   REALTYPE,dimension(:,:),pointer,contiguous :: velxEuler=>NULL(),velyEuler=>NULL()
   REALTYPE,dimension(:,:),pointer,contiguous :: taubmax=>NULL()
!  the following fields will only be allocated if deformCX=.true.
   REALTYPE,dimension(:,:),allocatable :: dudxC,dvdyC
   REALTYPE,dimension(:,:),pointer,contiguous :: dudyX=>null()
   REALTYPE,dimension(:,:),pointer,contiguous :: dvdxX=>null()
   REALTYPE,dimension(:,:),allocatable :: shearX
!  the following fields will only be allocated if deformUV=.true.
   REALTYPE,dimension(:,:),allocatable :: dudxV,dvdyU,shearU
!  the followinf fields will only be allocated if An_method=2
   REALTYPE,dimension(:,:),allocatable :: AnC,AnX

!  KK-TODO: make *dis_2d allocatable
   REALTYPE,dimension(:,:),contiguous,pointer :: numdis_2d=>null()
   REALTYPE,dimension(:,:),contiguous,pointer :: phydis_2d=>null()

   REALTYPE,dimension(:,:),allocatable :: rvol

   integer                             :: size2d_field
   integer                             :: mem2d
!
! !REVISION HISTORY:
!  Original author(s): Karsten Bolding & Hans Burchard
!
!EOP
!-----------------------------------------------------------------------

   contains

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: init_variables_2d - initialise 2D related stuff.
!
! !INTERFACE:
   subroutine init_variables_2d(runtype,no_2d)
   IMPLICIT NONE
!
! !DESCRIPTION:
!  Allocates memory (unless {\tt STATIC} is set) for 2D related fields,
!  by an include statement. Furthermore all public 2D variables are
!  initialised to zero. Those are listed in table \ref{table_2d_variables}
!  on page \pageref{table_2d_variables}.
!
!  \begin{table}[h]
!  \begin{center}
!  \begin{tabular}{lll}
! {\tt z } & sea surface elevation in T-point & [m] \\
! {\tt U } & $x$ component of transport in U-point & [m$^2$s$^{-1}$] \\
! {\tt DU } & water depth in U-point & [m] \\
! {\tt fU } & Coriolis term for $V$-equation in V-point & [m$^2$s$^{-2}$] \\
! {\tt SlUx } & slow term for $U$-equation in U-point & [m$^2$s$^{-2}$] \\
! {\tt Slru } &slow bottom friction for $U$-equation in U-point &
! [m$^2$s$^{-2}$]\\
! {\tt V } & $y$ component of transport in V-point & [m$^2$s$^{-1}$]\\
! {\tt DV } & water depth in V-point & [m] \\
! {\tt fV } &  Coriolis term for $U$-equation in U-point & [m$^2$s$^{-2}$]\\
! {\tt SlVx } & slow term for $V$-equation in V-point & [m$^2$s$^{-2}$] \\
! {\tt Slrv } &slow bottom friction for $V$-equation in V-point &
! [m$^2$s$^{-2}$]\\
! {\tt Uint } & $x$-component of mean transport in U-point & [m$^2$s$^{-1}$]\\
! {\tt Vint } & $y$-component of mean transport in V-point & [m$^2$s$^{-1}$]\\
! {\tt UEx } & sum of explicit terms for for $U$-equation in U-point & [m$^2$s$^{-2}$]\\
! {\tt VEx } &sum of explicit terms for for $V$-equation in V-point & [m$^2$s$^{-2}$]\\
! {\tt ru } & bottom friction for $U$-equation in U-point & [m$^2$s$^{-2}$]\\
! {\tt rv } &bottom friction for $V$-equation in V-point & [m$^2$s$^{-2}$]\\
! {\tt res\_du } & residual depth in U-point & [m]\\
! {\tt res\_u } & $x$-component of residual transport in U-point &
! [m$^2$s$^{-1}$]\\
! {\tt res\_dv } & residual depth in V-point & [m] \\
! {\tt res\_v } &$y$-component of residual transport in V-point &
! [m$^2$s$^{-1}$]\\
! \end{tabular}
! \caption{Public 2D variables.}
! \label{table_2d_variables}
! \end{center}
! \end{table}
!
! !INPUT PARAMETERS:
   integer, intent(in)                 :: runtype
   logical, intent(in)                 :: no_2d
!
! !LOCAL VARIABLES:
   integer                   :: rc
!EOP
!-------------------------------------------------------------------------
!BOC
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'init_variables_2d() # ',Ncall
#endif

   LEVEL2 'init_variables_2d'
   size2d_field=((imax+HALO)-(imin+HALO)+1)*((jmax+HALO)-(jmin+HALO)+1)
   mem2d=n2d_fields*size2d_field*REAL_SIZE

!  Allocates memory for the public data members - if not static
#ifndef STATIC
#include "dynamic_allocations_2d.h"
#endif

#ifdef _POINTER_ZZO_
   zo => t_zo ; z => t_z
#endif

#ifdef USE_BREAKS
   break_stat = 0
#endif

   UEuler    => U    ; VEuler    => V
   UEulerInt => Uint ; VEulerInt => Vint
   veluf     => velu ; velvf     => velv
   veluEuler => velu ; velvEuler => velv
   velxEuler => velx ; velyEuler => vely


   if (waveforcing_method .ne. NO_WAVES) then

      allocate(UEuler(E2DFIELD),stat=rc)
      if (rc /= 0) stop 'init_2d: Error allocating memory (UEuler)'
      UEuler = _ZERO_
      allocate(VEuler(E2DFIELD),stat=rc)
      if (rc /= 0) stop 'init_2d: Error allocating memory (VEuler)'
      VEuler = _ZERO_
      allocate(veluEuler(E2DFIELD),stat=rc)
      if (rc /= 0) stop 'init_2d: Error allocating memory (veluEuler)'
      veluEuler = _ZERO_
      allocate(velvEuler(E2DFIELD),stat=rc)
      if (rc /= 0) stop 'init_2d: Error allocating memory (velvEuler)'
      velvEuler = _ZERO_
      allocate(velxEuler(E2DFIELD),stat=rc)
      if (rc /= 0) stop 'init_2d: Error allocating memory (velxEuler)'
      velxEuler = _ZERO_
      allocate(velyEuler(E2DFIELD),stat=rc)
      if (rc /= 0) stop 'init_2d: Error allocating memory (velyEuler)'
      velyEuler = _ZERO_

      if (runtype .gt. 1) then
         allocate(UEulerInt(E2DFIELD),stat=rc)
         if (rc /= 0) stop 'init_2d: Error allocating memory (UEulerInt)'
         UEulerInt = _ZERO_
         allocate(VEulerInt(E2DFIELD),stat=rc)
         if (rc /= 0) stop 'init_2d: Error allocating memory (VEulerInt)'
         VEulerInt = _ZERO_
      end if

      if (waves_method .eq. WAVES_VF) then
         veluf => veluEuler ; velvf => velvEuler
      end if

   end if


   z  = -9999._rk ; zo =_ZERO_

   U = _ZERO_; DU = _ZERO_; Uint = _ZERO_; UEx = _ZERO_
   V = _ZERO_; DV = _ZERO_; Vint = _ZERO_; VEx = _ZERO_
   velu = _ZERO_  ; velv = _ZERO_
   velx = -9999._rk ; vely = -9999._rk
   where (az .gt. 0)
      velx = _ZERO_ ; vely = _ZERO_
   end where


   if (bottfric_method .eq. 1) then
      ru = rdrag
      rv = rdrag
   else
      ru = _ZERO_
      rv = _ZERO_
   end if
   rds = _ZERO_

   res_du = _ZERO_; res_u = _ZERO_
   res_dv = _ZERO_; res_v = _ZERO_

   SlUx=_ZERO_; Slru=_ZERO_
   SlVx=_ZERO_; Slrv=_ZERO_

   fwf     = _ZERO_
   fwf_int = _ZERO_

   EWbdy=_ZERO_
   ENbdy=_ZERO_
   EEbdy=_ZERO_
   ESbdy=_ZERO_

#ifdef DEBUG
   write(debug,*) 'Leaving init_variables_2d()'
   write(debug,*)
#endif
   return
   end subroutine init_variables_2d
!EOC

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: postinit_variables_2d - re-initialise some 2D stuff.
!
! !INTERFACE:
   subroutine postinit_variables_2d(no_2d)
   IMPLICIT NONE
!
! !INPUT PARAMETERS:
   logical, intent(in)                 :: no_2d
!
! !DESCRIPTION:
!
! !LOCAL VARIABLES:
   integer                   :: rc
!
!EOP
!-----------------------------------------------------------------------
!BOC
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'postinit_variables_2d() # ',Ncall
#endif

!  must be allocated in postinit because of do_numerical_analyses_2d

   if (.not. no_2d) then

      if (deformC) then
         allocate(dudxC(E2DFIELD),stat=rc)
         if (rc /= 0) stop 'postinit_variables_2d: Error allocating memory (dudxC)'
         dudxC=_ZERO_
#ifndef SLICE_MODEL
         allocate(dvdyC(E2DFIELD),stat=rc)
         if (rc /= 0) stop 'postinit_variables_2d: Error allocating memory (dvdyC)'
         dvdyC=_ZERO_
#endif
      end if
      if (deformX) then
         allocate(shearX(E2DFIELD),stat=rc)
         if (rc /= 0) stop 'postinit_variables_2d: Error allocating memory (shearX)'
         shearX=_ZERO_
         if (save_phydis_2d) then
            allocate(dvdxX(E2DFIELD),stat=rc)
            if (rc /= 0) stop 'postinit_variables_2d: Error allocating memory (dvdxX)'
            dvdxX=_ZERO_
#ifndef SLICE_MODEL
            allocate(dudyX(E2DFIELD),stat=rc)
            if (rc /= 0) stop 'postinit_variables_2d: Error allocating memory (dudyX)'
            dudyX=_ZERO_
#endif
         end if
      end if
      if (deformUV) then
         allocate(dudxV(E2DFIELD),stat=rc)
         if (rc /= 0) stop 'postinit_variables_2d: Error allocating memory (dudxV)'
         dudxV=_ZERO_
#ifndef SLICE_MODEL
         allocate(dvdyU(E2DFIELD),stat=rc)
         if (rc /= 0) stop 'postinit_variables_2d: Error allocating memory (dvdyU)'
         dvdyU=_ZERO_
#endif
         allocate(shearU(E2DFIELD),stat=rc)
         if (rc /= 0) stop 'postinit_variables_2d: Error allocating memory (shearU)'
         shearU=_ZERO_
      end if

   if (save_phydis_2d) then
      allocate(phydis_2d(E2DFIELD),stat=rc)
      if (rc /= 0) stop 'postinit_2d: Error allocating memory (phydis_2d)'
      phydis_2d = _ZERO_
   end if
   if (save_numdis_2d) then
      allocate(numdis_2d(E2DFIELD),stat=rc)
      if (rc /= 0) stop 'postinit_2d: Error allocating memory (numdis_2d)'
      numdis_2d = _ZERO_
   end if

      if (calc_taubmax) then
         allocate(taubmax(E2DFIELD),stat=rc)
         if (rc /= 0) stop 'postinit_2d: Error allocating memory (taubmax)'
         taubmax = _ZERO_
      end if

   if (save_rvol) then
      allocate(rvol(E2DFIELD),stat=rc)
      if (rc /= 0) stop 'postinit_2d: Error allocating memory (rvol)'
      rvol = _ZERO_
   end if

   end if

#ifdef DEBUG
   write(debug,*) 'Leaving postinit_variables_2d()'
   write(debug,*)
#endif
   return
   end subroutine postinit_variables_2d
!EOC

!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: register_2d_variables() - register GETM variables.
!
! !INTERFACE:
   subroutine register_2d_variables(fm)
!
! !DESCRIPTION:
!
! !USES:
   use field_manager
   use meteo, only: fwf_method
   use domain, only: nriverl
   IMPLICIT NONE
!
! !INPUT PARAMETERS:
   type (type_field_manager) :: fm
!
! !REVISION HISTORY:
!  Original author(s): Karsten Bolding & Jorn Bruggeman
!
! !LOCAL VARIABLES:
   logical :: used
!EOP
!-----------------------------------------------------------------------
!BOC
   LEVEL2 'register_2d_variables()'

!D(E2DFIELD)
!DU,DV
!z(E2DFIELD)
!zo(E2DFIELD)
!U(E2DFIELD)
!V(E2DFIELD)
!UEx(E2DFIELD)
!VEx(E2DFIELD)
!fU(E2DFIELD)
!fV(E2DFIELD)
!ru(E2DFIELD)
!rv(E2DFIELD)
!Uint(E2DFIELD)
!Vint(E2DFIELD)
!Uinto(E2DFIELD)
!Vinto(E2DFIELD)
!res_du(E2DFIELD)
!res_u(E2DFIELD)
!res_dv(E2DFIELD)
!res_v(E2DFIELD)
!kbk
!SlUx(E2DFIELD)
!SlVx(E2DFIELD)
!Slru(E2DFIELD)
!Slrv(E2DFIELD)
!zub(E2DFIELD)
!zvb(E2DFIELD)
!zub0(E2DFIELD)
!zvb0(E2DFIELD)
!An(E2DFIELD)
!AnX(E2DFIELD)
!fwf(E2DFIELD)
!fwf_int(E2DFIELD)
!EWbdy(jmax),ENbdy(imax),EEbdy(jmax),ESbdy(imax)


!  category - 2d
   call fm%register('z', 'm', 'sea surface elevation', standard_name='sea surface elevation', fill_value=-9999.0_rk, data2d=z(_2D_W_), category="2d", part_of_state=.true.)
   call fm%register('zo', 'm', 'sea surface elevation', standard_name='sea surface elevation', fill_value=-9999.0_rk, data2d=zo(_2D_W_), category="2d", output_level=output_level_debug)
   call fm%register('D', 'm', 'water depth', standard_name='water depth', fill_value=-10009.0_rk, data2d=D(_2D_W_), category="2d", part_of_state=.true.)
   call fm%register('DU', 'm', 'water depth - U-points', standard_name='water depth', data2d=DU(_2D_W_), category='2d', output_level=output_level_debug)
   call fm%register('DV', 'm', 'water depth - V-points', standard_name='water depth', data2d=DV(_2D_W_), category='2d', output_level=output_level_debug)
   call fm%register('U', 'm2/s', 'transport in local x-direction', standard_name='', data2d=U(_2D_W_), category='2d', output_level=output_level_debug, part_of_state=.true.)
   call fm%register('V', 'm2/s', 'transport in local y-direction', standard_name='', data2d=V(_2D_W_), category='2d', output_level=output_level_debug, part_of_state=.true.)
   call fm%register('velx', 'm/s', 'velocity in global x-direction', standard_name='', data2d=velx(_2D_W_), category='2d', fill_value=-9999.0_rk, output_level=output_level_debug)
   call fm%register('vely', 'm/s', 'velocity in global y-direction', standard_name='', data2d=vely(_2D_W_), category='2d', fill_value=-9999.0_rk, output_level=output_level_debug)

   call fm%register('taubmax', 'm2/s2', 'max. bottom stress', standard_name='', category='2d', fill_value=-9999.0_rk, output_level=output_level_debug, used=used)
   if (.not. calc_taubmax) calc_taubmax = used

   call fm%register('numdis_2d', 'm*W/kg', 'numerical dissipation content', standard_name='', category='2d', output_level=output_level_debug, used=save_numdis_2d)
   call fm%register('phydis_2d', 'W/kg', 'physical dissipation' , standard_name='', category='2d', output_level=output_level_debug, used=save_phydis_2d)

   if (fwf_method .ge. 1) then
   call fm%register('fwf', 'm', 'surface freshwater fluxes', standard_name='', data2d=fwf(_2D_W_), category='2d', output_level=output_level_debug)
   end if

   if (nriverl .gt. 0) then
   call fm%register('rvol', 'm**3', 'river discharge', category='2d', fill_value=_ZERO_, output_level=output_level_debug, used=save_rvol)
   end if

   return
   end subroutine register_2d_variables
!EOC

!-----------------------------------------------------------------------
!BOP
!
! !ROUTINE: finalize_register_2d_variables() - send optional variables.
!
! !INTERFACE:
   subroutine finalize_register_2d_variables(fm)
!
! !DESCRIPTION:
!
! !USES:
   use field_manager
   IMPLICIT NONE
!
! !INPUT PARAMETERS:
   type (type_field_manager) :: fm
!
! !REVISION HISTORY:
!  Original author(s): Karsten Bolding & Jorn Bruggeman
!
! !LOCAL VARIABLES:
!EOP
!-----------------------------------------------------------------------
!BOC
   LEVEL1 'finalize_register_2d_variables()'

   if (associated(taubmax)) call fm%send_data('taubmax', taubmax(_2D_W_))
   if (associated(numdis_2d)) call fm%send_data('numdis_2d', numdis_2d(_2D_W_))
   if (associated(phydis_2d)) call fm%send_data('phydis_2d', phydis_2d(_2D_W_))
   if (allocated(rvol)) call fm%send_data('rvol', rvol(_2D_W_))

   return
   end subroutine finalize_register_2d_variables
!EOC

!-----------------------------------------------------------------------
!BOP
!
! !IROUTINE: clean_variables_2d - cleanup after 2D run.
!
! !INTERFACE:
   subroutine clean_variables_2d()
   IMPLICIT NONE
!
! !DESCRIPTION:
!  This routine is currently empty.
!
! !LOCAL VARIABLES:
!
!EOP
!-----------------------------------------------------------------------
!BOC
#ifdef DEBUG
   integer, save :: Ncall = 0
   Ncall = Ncall+1
   write(debug,*) 'clean_variables_2d() # ',Ncall
#endif

#ifdef DEBUG
   write(debug,*) 'Leaving clean_variables_2d()'
   write(debug,*)
#endif
   return
   end subroutine clean_variables_2d
!EOC

!-----------------------------------------------------------------------

   end module variables_2d

!-----------------------------------------------------------------------
! Copyright (C) 2001 - Hans Burchard and Karsten Bolding (BBH)         !
!-----------------------------------------------------------------------
