import os.path
import os
import glob
import time
import numpy as np
from fpdf import FPDF
import matplotlib as mpl
import matplotlib.pyplot as plt
import cartopy.crs as ccrs
from matplotlib.ticker import FormatStrFormatter
import numpy as np
from mpl_toolkits.axes_grid1 import make_axes_locatable


def plotter(X, Y, Z, fname: str, title: str, unit: int, vmin, vmax):

    pc = ccrs.PlateCarree()
    fig = plt.figure(figsize=(16, 8))
    ax = plt.axes(projection=pc)
    ax.set_extent([-180, 180,
                   -90, 90], crs=pc)
    im = ax.imshow(Z, origin='lower',
                   extent=[-180, 180,
                           -90, 90],
                   interpolation='nearest', aspect='auto', vmin=vmin, vmax=vmax,
                   cmap=mpl.colormaps['rainbow'])
    ax.coastlines(resolution='50m', color='black', linewidth=4)
    # fixing tickers
    x_ticks = np.arange(-180,
                        180, 40)
    x_labels = np.linspace(-180, 80, np.size(x_ticks))
    ax.set_xticks(x_ticks)
    ax.set_xticklabels(x_labels, fontsize=18)
    ax.xaxis.set_major_formatter(FormatStrFormatter('%.1f'))
    y_ticks = np.arange(-90, 90, 20)
    y_labels = np.linspace(-90, 90, np.size(y_ticks))
    ax.set_yticks(y_ticks)
    ax.set_yticklabels(y_labels, fontsize=18)
    ax.yaxis.set_major_formatter(FormatStrFormatter('%.1f'))
    # plotting lat and lon
    plt.xlabel('Lon', fontsize=20)
    plt.ylabel('Lat', fontsize=20)
    cbar = plt.colorbar(im)
    cbar.ax.tick_params(labelsize=18)
    if unit == 1:
        cbar.set_label(r'$[ \times 10^{15}molec.cm^{-2}] $', fontsize=18)
    elif unit == 2:
        cbar.set_label('$ [Unitless] $', fontsize=18)
    plt.title(title, loc='left', fontweight='bold', fontsize=20)
    plt.tight_layout()
    fig.savefig(fname, format='png', dpi=300)
    plt.close()


def topdf(fname: str, folder:str, pdf_output: str):
    ''' 
    save all pngs to a pdf report
    '''
    def header(pdfobj, title, fsize=22):
        # Arial bold 15
        pdfobj.set_font('Arial', 'B', fsize)
        # Calculate width of title and position
        w = pdfobj.get_string_width(title) + 6
        pdfobj.set_x((210 - w) / 2)
        pdfobj.set_fill_color(255, 255, 255)
        pdfobj.set_text_color(0, 0, 0)
        # Thickness of frame (1 mm)
        pdfobj.set_line_width(1)
        # Title
        pdfobj.cell(w, 9, title, 1, 1, 'C', 1)
        # Line break
        pdfobj.ln(10)
        return w

    def body(pdfobj, bd1):
        # Times 12
        pdfobj.set_font('Times', '', 12)
        # Output justified text
        pdfobj.multi_cell(0, 5, bd1)
        # Line break
        pdfobj.ln(1)

    # call the fpdf obj
    pdf = FPDF(orientation="landscape")
    pdf.add_page()
    title = 'The Optimal Interpolation Report generated by the OI-SAT-GMI tool'
    pdf.set_font('Arial', 'B', 16)
    pdf.cell(280, 100, txt=title, border=0, ln=1, align="C")
    pdf.cell(280, 20, txt='Amir H. Souri', border=0, ln=1, align="C")
    pdf.cell(280, 20, txt='Contact: ahsouri@gmail.com',
             border=0, ln=1, align="C")
    t = time.localtime()
    current_time = time.strftime("%Y-%m-%d %H:%M:%S", t)
    pdf.cell(280, 20, txt='Created at: ' +
             current_time, border=0, ln=1, align="C")

    # printing grayscales
    map_plots = sorted(glob.glob('temp/*' + fname + '*.png'))
    for fname in map_plots:
        pdf.add_page()
        pdf.image(fname, h=pdf.eph, w=pdf.epw+20)

    # writing
    if not os.path.exists(folder):
        os.makedirs(folder)

    pdf.output(folder + '/' + pdf_output, 'F')


def report(lon: np.ndarray, lat: np.ndarray, ctm_vcd_before: np.ndarray, ctm_vcd_after: np.ndarray,
           sat_vcd: np.ndarray, sat_err: np.ndarray, increment: np.ndarray, averaging_kernel: np.ndarray, error_OI: np.ndarray,
           new_amf: np.ndarray, old_amf: np.ndarray, fname:str, ffolder: str):
    '''
    '''
    if not os.path.exists('temp'):
        os.makedirs('temp')
    plotter(lon, lat, ctm_vcd_before, 'temp/ctm_vcd_before_' +
            fname + '.png', 'CTM VCD (prior)', 1, 0.0, 10.0)
    plotter(lon, lat, ctm_vcd_after, 'temp/ctm_vcd_after_' +
            fname + '.png', 'CTM VCD (posterior)', 1, 0.0, 10.0)
    plotter(lon, lat, sat_vcd, 'temp/ctm_vcd_sat_used_' + fname +
            '.png', 'Satellite Observation (Y)', 1, 0.0, 10.0)
    plotter(lon, lat, sat_err, 'temp/ctm_vcd_sat_zerr_' + fname +
            '.png', 'Satellite Error (So)', 1, 0.0, 4.0)
    plotter(lon, lat, increment, 'temp/increment_' +
            fname + '.png', 'Increment', 1, -5.0, 5.0)
    plotter(lon, lat, averaging_kernel, 'temp/dak_' +
            fname + '.png', 'Averaging Kernels', 2, 0.0, 1.0)
    plotter(lon, lat, error_OI, 'temp/error_' +
            fname + '.png', 'OI estimate error', 1, 0.0, 4.0)
    plotter(lon, lat, new_amf, 'temp/new_amf_' +
            fname + '.png', 'new AMF', 2, 0.0, 4)
    plotter(lon, lat, old_amf, 'temp/old_amf_' +
            fname + '.png', 'old AMF', 2, 0.0, 4)

    topdf(fname, ffolder, 'OI_report_' + fname + '.pdf')
