(* Content-type: application/vnd.wolfram.mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 13.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       158,          7]
NotebookDataLength[    182301,       4740]
NotebookOptionsPosition[    170099,       4548]
NotebookOutlinePosition[    170513,       4564]
CellTagsIndexPosition[    170470,       4561]
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{
Cell[BoxData[
 RowBox[{"Quit", "[", "]"}]], "Input",
 CellChangeTimes->{{3.859571037171164*^9, 3.859571038592571*^9}},
 CellLabel->"In[27]:=",ExpressionUUID->"412f3c07-f6da-4e15-b3cb-2cae69dd58e2"],

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", 
   RowBox[{
    RowBox[{"LCDD$Instruction", " ", "=", " ", 
     RowBox[{"file", "\[Function]", 
      RowBox[{
       RowBox[{"FileNameJoin", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Environment", "[", "\"\<OneDrive\>\"", "]"}], ",", 
          "\"\<utilities\\\\libcdd2\\\\lcdd_gmp.exe\>\""}], "}"}], "]"}], 
       "<>", "\"\< \>\"", "<>", "file"}]}]}], ";"}], "*)"}], 
  "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"LCDD$Instruction", " ", "=", "  ", 
     RowBox[{"file", "\[Function]", " ", 
      RowBox[{"\"\<wsl cddexec --rep < \>\"", "<>", "file"}]}]}], ";"}], "\n",
    "\[IndentingNewLine]", 
   RowBox[{"(*", 
    RowBox[{"Common", " ", "path", " ", "locations", " ", 
     RowBox[{"are", ":"}]}], "*)"}], "\n", 
   RowBox[{"(*", 
    RowBox[{"ON", " ", 
     RowBox[{
     "MAC", ":", " ", 
      "\"\</usr/local/Cellar/cddlib/094h/bin/lcdd_gmp\>\""}]}], "*)"}], 
   "\[IndentingNewLine]", 
   RowBox[{"(*", 
    RowBox[{"ON", " ", 
     RowBox[{"WINDOWS", ":", " ", 
      RowBox[{
      "\"\<C:\\cygwin\\usr\\bin\\lcdd_gmp.exe\>\"", " ", "or", " ", 
       "\"\<cddexec --rep < \>\""}]}]}], "*)"}], "\n", 
   RowBox[{"(*", 
    RowBox[{"ON", " ", 
     RowBox[{"LINUX", ":", " ", 
      RowBox[{
      "\"\</usr/lib/cdd-tools/lcdd_gmp\>\"", " ", "or", " ", 
       "\"\<cddexec --rep < \>\""}]}]}], "*)"}]}]}]], "Code",
 CellChangeTimes->CompressedData["
1:eJwdzlsogwEABeDxwBINyW2ITU1yCVkaqRUpNbKmpZaifo2mGYvlVhjL5cHI
MrOV5MGlUItYKUtmSFsyreZWiLZmLS9MzX/+h9P3cDp1ctrlQiKSRqNlkIGy
qwr/zKmPvxXUhmADU8SYJV19mEiAJ/2cdBjw6yi/csNc+H5Iq4ZVXo0ACh0D
jbDb9tMGkzKaCLjnjhuCNR0myl/BpYHaGbXrcNGVtgPpuuhdmF+vsMJlo+gM
yomIO8gaVrvgrdj2BD2S5mfqz8XeK5R2xr7BR/NkyhzpgzKUCedUEhZkWS4o
HVwPB9L7z/Oo3v5ZAgsMNzw4X8serLT6+PH2Og30Hyn0UPQtXYHWP8UGNBH7
23CqlXcAU20flM7plugqUjF/MwbylOWJcFR9nwwZljUmDPR1saEs21sIg73J
RXCp2DlUTRoVLh2B+h7zuOvaxy9TvUzA4yyR1k1qHytagP/+Evef
  "],
 CellLabel->"In[1]:=",ExpressionUUID->"ab8f2682-73c3-4db1-9a2c-8af8f121607d"],

Cell[CellGroupData[{

Cell["Background Algorithms:", "Subsection",
 CellChangeTimes->{{3.851685419117911*^9, 3.851685437578148*^9}, {
  3.851685470990306*^9, 3.8516854782867985`*^9}, {3.8516855301907043`*^9, 
  3.851685548589467*^9}, {3.851685865521597*^9, 3.851685871412819*^9}, {
  3.852038776825276*^9, 
  3.852038782842166*^9}},ExpressionUUID->"2a368a57-ec22-4798-abc4-\
592fc1c28196"],

Cell[CellGroupData[{

Cell["On converting a matrix-based GPT to a vector-based GPT", "Subsubsection",
 CellChangeTimes->{{3.856008503334812*^9, 3.8560085331340485`*^9}, {
  3.859549247974127*^9, 
  3.8595492826946697`*^9}},ExpressionUUID->"1e2e38ee-3bbf-48b4-8cf9-\
7453fd1feacb"],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"GellMannMatrixFamily", "[", "0", "]"}], "[", "n_", "]"}], ":=", 
   RowBox[{
    RowBox[{
     FractionBox["#", 
      SqrtBox[
       RowBox[{"Tr", "[", 
        RowBox[{"#", ".", 
         RowBox[{"#", "\[ConjugateTranspose]"}]}], "]"}]]], "&"}], "/@", 
    RowBox[{"SparseArray", "[", 
     RowBox[{"{", 
      RowBox[{"IdentityMatrix", "[", 
       RowBox[{"n", ",", "SparseArray"}], "]"}], "}"}], "]"}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"GellMannMatrixFamily", "[", "1", "]"}], "[", "n_", "]"}], ":=", 
  RowBox[{
   RowBox[{
    FractionBox["#", 
     SqrtBox[
      RowBox[{"Tr", "[", 
       RowBox[{"#", ".", 
        RowBox[{"#", "\[ConjugateTranspose]"}]}], "]"}]]], "&"}], "/@", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"Flatten", "[", 
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"SymmetrizedArray", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"{", 
            RowBox[{"j", ",", "k"}], "}"}], "->", "1"}], ",", 
          RowBox[{"{", 
           RowBox[{"n", ",", "n"}], "}"}], ",", 
          RowBox[{"Symmetric", "[", 
           RowBox[{"{", 
            RowBox[{"1", ",", "2"}], "}"}], "]"}]}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"k", ",", "2", ",", "n"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"j", ",", "1", ",", 
          RowBox[{"k", "-", "1"}]}], "}"}]}], "]"}], ",", "1"}], "]"}], 
    "]"}]}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"GellMannMatrixFamily", "[", "2", "]"}], "[", "n_", "]"}], ":=", 
  RowBox[{
   RowBox[{
    FractionBox["#", 
     SqrtBox[
      RowBox[{"Tr", "[", 
       RowBox[{"#", ".", 
        RowBox[{"#", "\[ConjugateTranspose]"}]}], "]"}]]], "&"}], "/@", 
   RowBox[{"SparseArray", "[", 
    RowBox[{"Flatten", "[", 
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"SymmetrizedArray", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"{", 
            RowBox[{"j", ",", "k"}], "}"}], "->", 
           RowBox[{"-", "\[ImaginaryI]"}]}], ",", 
          RowBox[{"{", 
           RowBox[{"n", ",", "n"}], "}"}], ",", 
          RowBox[{"Antisymmetric", "[", 
           RowBox[{"{", 
            RowBox[{"1", ",", "2"}], "}"}], "]"}]}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"{", 
         RowBox[{"k", ",", "2", ",", "n"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"j", ",", "1", ",", 
          RowBox[{"k", "-", "1"}]}], "}"}]}], "]"}], ",", "1"}], "]"}], 
    "]"}]}]}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"GellMannMatrixFamily", "[", "3", "]"}], "[", "n_", "]"}], ":=", 
   RowBox[{
    RowBox[{
     FractionBox["#", 
      SqrtBox[
       RowBox[{"Tr", "[", 
        RowBox[{"#", ".", 
         RowBox[{"#", "\[ConjugateTranspose]"}]}], "]"}]]], "&"}], "/@", 
    RowBox[{"SparseArray", "[", 
     RowBox[{"Table", "[", 
      RowBox[{"(*", 
       SqrtBox[
        FractionBox["2", 
         RowBox[{"l", 
          RowBox[{"(", 
           RowBox[{"l", "+", "1"}], ")"}]}]]], "*)"}], 
      RowBox[{
       RowBox[{"SparseArray", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{
             RowBox[{"{", 
              RowBox[{"j_", ",", "j_"}], "}"}], "/;", 
             RowBox[{"j", "<=", "l"}]}], "->", "1"}], ",", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{
              RowBox[{"l", "+", "1"}], ",", 
              RowBox[{"l", "+", "1"}]}], "}"}], "->", 
            RowBox[{"-", "l"}]}]}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"n", ",", "n"}], "}"}]}], "]"}], ",", "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{"l", ",", "1", ",", 
         RowBox[{"n", "-", "1"}]}], "}"}]}], "]"}], "]"}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"GellMannMatrices", "[", "n_", "]"}], ":=", 
  RowBox[{"Join", "[", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"GellMannMatrixFamily", "[", "0", "]"}], "[", "n", "]"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"GellMannMatrixFamily", "[", "1", "]"}], "[", "n", "]"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"GellMannMatrixFamily", "[", "2", "]"}], "[", "n", "]"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"GellMannMatrixFamily", "[", "3", "]"}], "[", "n", "]"}]}], 
   "\[IndentingNewLine]", "]"}]}]}], "Code",
 CellChangeTimes->{{3.8593743681124177`*^9, 3.8593744180113325`*^9}, {
  3.859374477563774*^9, 3.859374947577633*^9}, {3.859374980128759*^9, 
  3.8593750114413347`*^9}, {3.859375100414237*^9, 3.8593751159216948`*^9}, {
  3.8593759798139367`*^9, 3.859376082421482*^9}, {3.8593762841086416`*^9, 
  3.8593762876611238`*^9}, {3.859376521452162*^9, 3.859376534940526*^9}, {
  3.8593766036605625`*^9, 3.859376608700052*^9}, {3.8593766502514067`*^9, 
  3.8593766505949306`*^9}, {3.8593770450492926`*^9, 3.859377048649145*^9}, {
  3.8593771644328613`*^9, 3.859377166353325*^9}, {3.8595450338966274`*^9, 
  3.8595450848320465`*^9}, {3.859549293408511*^9, 3.859549297615225*^9}, {
  3.8595496571207037`*^9, 3.859549665097819*^9}},
 CellLabel->"In[2]:=",ExpressionUUID->"0cfc615b-2bf0-419c-b196-eb4e7142d926"],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"ConvertMatrixListToVectorList", "[", "ListOfMatrices_", "]"}], ":=",
    "\[IndentingNewLine]", 
   RowBox[{"With", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"d", "=", 
       RowBox[{"Last", "[", 
        RowBox[{"Dimensions", "[", "ListOfMatrices", "]"}], "]"}]}], "}"}], 
     ",", "\[IndentingNewLine]", 
     RowBox[{"Outer", "[", 
      RowBox[{
       RowBox[{"Tr", "@*", "Dot"}], ",", "ListOfMatrices", ",", 
       RowBox[{"GellMannMatrices", "[", "d", "]"}], ",", "1"}], "]"}]}], 
    "\[IndentingNewLine]", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"ConvertMatrixToVectorList", "[", "mat_", "]"}], ":=", 
  RowBox[{"With", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"d", "=", 
      RowBox[{"Last", "[", 
       RowBox[{"Dimensions", "[", "mat", "]"}], "]"}]}], "}"}], ",", 
    RowBox[{
     RowBox[{
      RowBox[{"Tr", "[", 
       RowBox[{"Dot", "[", 
        RowBox[{"mat", ",", "#"}], "]"}], "]"}], "&"}], "/@", 
     RowBox[{"GellMannMatrices", "[", "d", "]"}]}]}], "]"}]}]}], "Code",
 CellChangeTimes->{{3.859549389600199*^9, 3.8595494592875414`*^9}, {
  3.859549964742053*^9, 3.859550041766463*^9}},
 CellLabel->"In[7]:=",ExpressionUUID->"fb96447f-72f5-47f7-9d73-290a3d98ca71"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"MaximallyMixedVector", "[", "d_", "]"}], ":=", 
  RowBox[{"Normal", "[", 
   RowBox[{"SparseArray", "[", 
    RowBox[{
     RowBox[{"1", "->", 
      FractionBox["1", 
       SqrtBox["d"]]}], ",", 
     RowBox[{"{", 
      SuperscriptBox["d", "2"], "}"}]}], "]"}], "]"}]}], "\n", 
 RowBox[{
  RowBox[{"UnitEffectVector", "[", "d_", "]"}], ":=", 
  RowBox[{"Normal", "[", 
   RowBox[{"SparseArray", "[", 
    RowBox[{
     RowBox[{"1", "->", 
      SqrtBox["d"]}], ",", 
     RowBox[{"{", 
      SuperscriptBox["d", "2"], "}"}]}], "]"}], "]"}]}]}], "Code",
 CellChangeTimes->{{3.8595494856729584`*^9, 3.8595495276154766`*^9}, {
  3.8595497130471535`*^9, 3.8595497138787136`*^9}, {3.859549762326404*^9, 
  3.859549767358387*^9}, {3.8599352404432745`*^9, 3.8599352583323946`*^9}},
 CellLabel->"In[9]:=",ExpressionUUID->"e4919f2e-da76-49fe-8692-d8803cc9f3ad"]
}, Closed]],

Cell[CellGroupData[{

Cell["On coordinate transformation to span vector space.", "Subsubsection",
 CellChangeTimes->{{3.856008503334812*^9, 
  3.8560085331340485`*^9}},ExpressionUUID->"9686e296-f094-42c7-9c98-\
da54c454dac5"],

Cell[CellGroupData[{

Cell["3 Methods to obtain Inclusion matrix.", "Item",
 CellChangeTimes->{{3.8560064639143085`*^9, 
  3.856006471290798*^9}},ExpressionUUID->"5c0a7f13-b477-4387-90f5-\
10627d2ba81d"],

Cell[BoxData[
 RowBox[{
  RowBox[{"NewMandMap", "[", "mat_", "]"}], ":=", "\[IndentingNewLine]", 
  RowBox[{"Block", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
     "q", ",", "r", ",", "proj$map", ",", "inc$map", ",", "projected$m"}], 
     "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{"q", ",", "r"}], "}"}], "=", 
      RowBox[{"QRDecomposition", "[", 
       RowBox[{"mat", "\[ConjugateTranspose]"}], "]"}]}], ";", "\n", 
     RowBox[{"proj$map", " ", "=", " ", 
      RowBox[{"q", "\[ConjugateTranspose]"}]}], ";", "\n", 
     RowBox[{"inc$map", " ", "=", " ", 
      RowBox[{"q", "\[ConjugateTranspose]"}]}], ";", "\n", 
     RowBox[{"projected$m", " ", "=", " ", 
      RowBox[{"r", "\[ConjugateTranspose]"}]}], ";", "\n", 
     RowBox[{"(*", 
      RowBox[{
       RowBox[{"invq", "=", 
        RowBox[{"q", "\[Transpose]"}]}], ";"}], "*)"}], "\[IndentingNewLine]", 
     RowBox[{"Return", "[", 
      RowBox[{"{", 
       RowBox[{"projected$m", ",", "inc$map", ",", "proj$map", ",", 
        RowBox[{"projected$m", ".", 
         RowBox[{"inc$map", "\[ConjugateTranspose]"}]}]}], "}"}], "]"}]}]}], 
   "]"}]}]], "Code",
 CellChangeTimes->{{3.8560064791012826`*^9, 3.856006529123526*^9}, {
  3.856007825608018*^9, 3.8560078469524045`*^9}, {3.856013879779849*^9, 
  3.8560138983881736`*^9}, {3.8560142757237277`*^9, 3.856014276107254*^9}, {
  3.856258767016102*^9, 3.856258769846775*^9}, {3.85644353719722*^9, 
  3.856443672061204*^9}, {3.8564437092125616`*^9, 3.856443712333285*^9}, {
  3.856444702895355*^9, 3.8564447171253295`*^9}},
 CellLabel->"In[11]:=",ExpressionUUID->"b2d9a55c-552e-41c7-b02b-c5f6e57853ea"],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"NewMandMap2", "[", "mat_", "]"}], ":=", "\[IndentingNewLine]", 
   RowBox[{"Block", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
      "q", ",", "r", ",", "d", ",", "proj$map", ",", "inc$map", ",", 
       "projected$m"}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"q", "=", 
       RowBox[{"RowReduce", "[", "mat", "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"d", "=", 
       RowBox[{
        RowBox[{"CountDistinct", "[", 
         RowBox[{
          RowBox[{"SparseArray", "[", "q", "]"}], "[", "\"\<RowPointers\>\"", 
          "]"}], "]"}], "-", "1"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"q", "=", 
       RowBox[{"q", "\[LeftDoubleBracket]", 
        RowBox[{";;", "d"}], "\[RightDoubleBracket]"}]}], ";", "\n", 
      RowBox[{"inc$map", " ", "=", " ", 
       RowBox[{"q", "\[ConjugateTranspose]"}]}], ";", "\n", 
      RowBox[{"proj$map", " ", "=", " ", 
       RowBox[{"PseudoInverse", "[", "q", "]"}]}], ";", "\n", 
      RowBox[{"projected$m", " ", "=", " ", 
       RowBox[{"mat", ".", "proj$map"}]}], ";", "\n", 
      RowBox[{"(*", "\n", 
       RowBox[{
        RowBox[{"invq", "=", 
         RowBox[{"PseudoInverse", "[", "q", "]"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"r", "=", 
         RowBox[{"Transpose", "[", 
          RowBox[{"mat", ".", "invq"}], "]"}]}], ";"}], "*)"}], 
      "\[IndentingNewLine]", 
      RowBox[{"Return", "[", 
       RowBox[{"{", 
        RowBox[{"projected$m", ",", "inc$map", ",", "proj$map", ",", 
         RowBox[{"projected$m", ".", 
          RowBox[{"inc$map", "\[ConjugateTranspose]"}]}]}], "}"}], "]"}]}]}], 
    "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"NewMandMap2b", "[", "mat_", "]"}], ":=", "\[IndentingNewLine]", 
   RowBox[{"Block", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
      "q", ",", "r", ",", "d", ",", "proj$map", ",", "inc$map", ",", 
       "projected$m"}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"q", "=", 
       RowBox[{"RowReduce", "[", "mat", "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"d", "=", 
       RowBox[{
        RowBox[{"CountDistinct", "[", 
         RowBox[{
          RowBox[{"SparseArray", "[", "q", "]"}], "[", "\"\<RowPointers\>\"", 
          "]"}], "]"}], "-", "1"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"q", "=", 
       RowBox[{"q", "\[LeftDoubleBracket]", 
        RowBox[{";;", "d"}], "\[RightDoubleBracket]"}]}], ";", "\n", 
      RowBox[{"proj$map", " ", "=", " ", 
       RowBox[{"q", "\[ConjugateTranspose]"}]}], ";", "\n", 
      RowBox[{"inc$map", " ", "=", " ", 
       RowBox[{"PseudoInverse", "[", "q", "]"}]}], ";", "\n", 
      RowBox[{"projected$m", " ", "=", " ", 
       RowBox[{"mat", ".", "proj$map"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"Return", "[", 
       RowBox[{"{", 
        RowBox[{"projected$m", ",", "inc$map", ",", "proj$map", ",", 
         RowBox[{"projected$m", ".", 
          RowBox[{"inc$map", "\[ConjugateTranspose]"}]}]}], "}"}], "]"}]}]}], 
    "]"}]}], ";"}]}], "Code",
 CellChangeTimes->{{3.8560067459864626`*^9, 3.8560068119702682`*^9}, {
   3.8560069200505886`*^9, 3.856006932738624*^9}, {3.856006994226719*^9, 
   3.856007000434091*^9}, {3.8560078857937613`*^9, 3.8560079372803597`*^9}, {
   3.8560138880127583`*^9, 3.8560139028603*^9}, {3.8560142729650326`*^9, 
   3.8560142732512674`*^9}, {3.8562587735511227`*^9, 3.856258776374461*^9}, {
   3.8562651192737174`*^9, 3.856265148296669*^9}, {3.8564369807607603`*^9, 
   3.85643698859317*^9}, {3.8564442791973877`*^9, 3.856444335957054*^9}, 
   3.8564444372846217`*^9, {3.8564445173168364`*^9, 3.8564445797335043`*^9}, 
   3.8564446237247725`*^9, {3.8564447266200924`*^9, 3.856444736693222*^9}, {
   3.8564447767011642`*^9, 3.85644478960441*^9}},
 CellLabel->"In[12]:=",ExpressionUUID->"e51affbf-d070-40f3-9d80-2eb585689f69"],

Cell[BoxData[
 RowBox[{
  RowBox[{"NewMandMap3", "[", "mat_", "]"}], ":=", "\[IndentingNewLine]", 
  RowBox[{"Block", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
     "prenull", ",", "draw", ",", "null", ",", "coordtoelim", ",", "vecs", 
      ",", "i", ",", "proj$map", ",", "inc$map", ",", "projected$m"}], "}"}], 
    ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"prenull", "=", 
      RowBox[{"NullSpace", "[", "mat", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"draw", "=", 
      RowBox[{"Last", "[", 
       RowBox[{"Dimensions", "[", "mat", "]"}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"If", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"Length", "[", "prenull", "]"}], "==", "0"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"projected$m", "=", 
         RowBox[{"mat", "\[Transpose]"}]}], ";", "\[IndentingNewLine]", 
        RowBox[{"proj$map", "=", 
         RowBox[{"IdentityMatrix", "[", "draw", "]"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"inc$map", "=", "proj$map"}]}], ",", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"null", "=", 
         RowBox[{"RowReduce", "[", "prenull", "]"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"coordtoelim", "=", 
         RowBox[{
          RowBox[{
           RowBox[{"SparseArray", "[", "null", "]"}], "[", 
           "\"\<AdjacencyLists\>\"", "]"}], "\[LeftDoubleBracket]", 
          RowBox[{"All", ",", "1"}], "\[RightDoubleBracket]"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"vecs", "=", 
         RowBox[{"-", 
          RowBox[{
           RowBox[{"Delete", "[", 
            RowBox[{
             RowBox[{"null", "\[Transpose]"}], ",", 
             RowBox[{
              RowBox[{"{", "coordtoelim", "}"}], "\[Transpose]"}]}], "]"}], 
           "\[Transpose]"}]}]}], ";", "\[IndentingNewLine]", 
        RowBox[{"inc$map", "=", 
         RowBox[{"IdentityMatrix", "[", 
          RowBox[{"draw", "-", 
           RowBox[{"Length", "[", "coordtoelim", "]"}]}], "]"}]}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"For", "[", 
         RowBox[{
          RowBox[{"i", "=", "1"}], ",", 
          RowBox[{"i", "<=", 
           RowBox[{"Length", "[", "coordtoelim", "]"}]}], ",", 
          RowBox[{"i", "++"}], ",", "\[IndentingNewLine]", 
          RowBox[{
           RowBox[{"inc$map", "=", 
            RowBox[{"Insert", "[", 
             RowBox[{"inc$map", ",", 
              RowBox[{
               RowBox[{
               "vecs", "\[LeftDoubleBracket]", "i", "\[RightDoubleBracket]"}],
                "//", "Conjugate"}], ",", 
              RowBox[{
              "coordtoelim", "\[LeftDoubleBracket]", "i", 
               "\[RightDoubleBracket]"}]}], "]"}]}], ";"}]}], "]"}], ";", 
        "\[IndentingNewLine]", 
        RowBox[{"proj$map", "=", 
         RowBox[{"PseudoInverse", "[", 
          RowBox[{"inc$map", "\[ConjugateTranspose]"}], "]"}]}], ";", "\n", 
        RowBox[{"projected$m", "=", 
         RowBox[{"mat", ".", "proj$map"}]}], ";"}]}], "\[IndentingNewLine]", 
      RowBox[{"(*", 
       RowBox[{
        RowBox[{"r", "=", 
         RowBox[{"ConjugateTranspose", "[", 
          RowBox[{"mat", ".", "invq"}], "]"}]}], ";"}], "*)"}], 
      "\[IndentingNewLine]", "]"}], ";", "\n", 
     RowBox[{"(*", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"M", "'"}], ",", 
         SuperscriptBox["I", "LI"], ",", "I", ",", "M"}]}], "]"}], "*)"}], 
     "\[IndentingNewLine]", 
     RowBox[{"Return", "[", 
      RowBox[{"{", 
       RowBox[{"projected$m", ",", "inc$map", ",", "proj$map", ",", 
        RowBox[{"projected$m", ".", 
         RowBox[{"inc$map", "\[ConjugateTranspose]"}]}]}], "}"}], "]"}]}]}], 
   "]"}]}]], "Code",
 CellChangeTimes->{
  3.856007075770269*^9, {3.8560071921454415`*^9, 3.8560072239300656`*^9}, {
   3.8560072838978767`*^9, 3.856007367169465*^9}, {3.856007411465172*^9, 
   3.856007617450018*^9}, {3.856007687170168*^9, 3.8560076920481663`*^9}, {
   3.8560077239952707`*^9, 3.85600773015273*^9}, {3.8560079927042007`*^9, 
   3.8560080274322805`*^9}, {3.8560080594403152`*^9, 3.856008110840375*^9}, {
   3.8560082414901867`*^9, 3.8560082621523237`*^9}, {3.8560083516637187`*^9, 
   3.8560084385771074`*^9}, {3.8560138372208214`*^9, 3.8560138638847337`*^9}, 
   3.8560139054365635`*^9, {3.8560142665639553`*^9, 3.8560142668937693`*^9}, {
   3.8562587824484982`*^9, 3.8562587844152303`*^9}, {3.85625923309338*^9, 
   3.8562592713890915`*^9}, {3.856264672043073*^9, 3.85626469231602*^9}, 
   3.8562647624601345`*^9, {3.856264825515631*^9, 3.856264862002469*^9}, {
   3.8564369913048*^9, 3.856437026484538*^9}, {3.8564371407538223`*^9, 
   3.8564371430127544`*^9}, {3.856444855469286*^9, 3.8564448985419874`*^9}, {
   3.8564450215326524`*^9, 3.8564450273164577`*^9}, {3.856445059725251*^9, 
   3.8564451014693017`*^9}},
 CellLabel->"In[14]:=",ExpressionUUID->"eaa25f48-8f9c-455e-b249-2745887f794f"],

Cell[BoxData[
 RowBox[{
  RowBox[{"testM", "=", 
   RowBox[{"(", GridBox[{
      {"1", "1", "1", "1", "0"},
      {"1", 
       RowBox[{"-", "1"}], 
       RowBox[{"3", "\[ImaginaryI]"}], "1", "0"},
      {
       RowBox[{"-", "1"}], 
       RowBox[{"-", "1"}], 
       RowBox[{"3", "\[ImaginaryI]"}], "1", "0"},
      {
       RowBox[{"-", "1"}], "1", "1", "1", "0"},
      {"1", 
       RowBox[{"-", "1"}], 
       RowBox[{"3", "\[ImaginaryI]"}], "1", "0"},
      {"1", 
       RowBox[{"-", "1"}], 
       RowBox[{"3", "\[ImaginaryI]"}], "1", "0"}
     }], ")"}]}], ";"}]], "Input",
 CellChangeTimes->{{3.856006548661586*^9, 3.8560066625235825`*^9}, {
   3.856006693074054*^9, 3.856006702602775*^9}, {3.8562645202031326`*^9, 
   3.8562645322105274`*^9}, {3.8564368640240946`*^9, 3.856436864453625*^9}, {
   3.8564368979289923`*^9, 3.8564369039737577`*^9}, {3.8564378139973164`*^9, 
   3.856437834665079*^9}, 3.856443728109231*^9},
 CellLabel->"In[23]:=",ExpressionUUID->"6e83dff4-3b47-4ab6-a26a-8b03316b75e7"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Dep", "=", 
  RowBox[{
   RowBox[{
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "0", ",", "1", ",", "0"}], "}"}], "}"}], 
    "\[Transpose]"}], ".", 
   RowBox[{"{", 
    RowBox[{"Mean", "[", "testM", "]"}], "}"}]}]}]], "Input",
 CellChangeTimes->{{3.8590202322113466`*^9, 3.8590202845935025`*^9}},
 CellLabel->"In[25]:=",ExpressionUUID->"e9568eb1-d887-4dd1-ad00-31179d34ffad"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["1", "3"], ",", 
     RowBox[{"-", 
      FractionBox["1", "3"]}], ",", 
     RowBox[{
      FractionBox["1", "3"], "+", 
      RowBox[{"2", " ", "\[ImaginaryI]"}]}], ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}]}], 
  "}"}]], "Output",
 CellChangeTimes->{{3.8590202811379504`*^9, 3.8590202849922004`*^9}},
 CellLabel->"Out[25]=",ExpressionUUID->"dbbf2b4f-ffcf-49be-863f-5a9632b35603"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"testM", ".", "Dep"}], "//", "MatrixForm"}]], "Input",
 CellChangeTimes->{{3.8590202954806333`*^9, 3.8590203023569045`*^9}},
 CellLabel->"In[27]:=",ExpressionUUID->"bce56010-3076-4918-990e-48e30ffe8044"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      FractionBox["1", "3"], 
      RowBox[{"-", 
       FractionBox["1", "3"]}], 
      RowBox[{
       FractionBox["1", "3"], "+", 
       RowBox[{"2", " ", "\[ImaginaryI]"}]}], "1", "0"},
     {
      FractionBox["1", "3"], 
      RowBox[{"-", 
       FractionBox["1", "3"]}], 
      RowBox[{
       FractionBox["1", "3"], "+", 
       RowBox[{"2", " ", "\[ImaginaryI]"}]}], "1", "0"},
     {
      FractionBox["1", "3"], 
      RowBox[{"-", 
       FractionBox["1", "3"]}], 
      RowBox[{
       FractionBox["1", "3"], "+", 
       RowBox[{"2", " ", "\[ImaginaryI]"}]}], "1", "0"},
     {
      FractionBox["1", "3"], 
      RowBox[{"-", 
       FractionBox["1", "3"]}], 
      RowBox[{
       FractionBox["1", "3"], "+", 
       RowBox[{"2", " ", "\[ImaginaryI]"}]}], "1", "0"},
     {
      FractionBox["1", "3"], 
      RowBox[{"-", 
       FractionBox["1", "3"]}], 
      RowBox[{
       FractionBox["1", "3"], "+", 
       RowBox[{"2", " ", "\[ImaginaryI]"}]}], "1", "0"},
     {
      FractionBox["1", "3"], 
      RowBox[{"-", 
       FractionBox["1", "3"]}], 
      RowBox[{
       FractionBox["1", "3"], "+", 
       RowBox[{"2", " ", "\[ImaginaryI]"}]}], "1", "0"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 CellChangeTimes->{{3.859020297100423*^9, 3.8590203028336344`*^9}},
 CellLabel->
  "Out[27]//MatrixForm=",ExpressionUUID->"e7efca14-8e8e-4a4c-9a75-\
0b883ff8bd01"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"MatrixForm", "/@", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"mp", ",", "inc", ",", "proj", ",", "m"}], "}"}], "=", 
      RowBox[{"Simplify", "[", 
       RowBox[{"NewMandMap", "[", "testM", "]"}], "]"}]}], ")"}]}], "//", 
   "Echo"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"mp", ".", 
     RowBox[{"inc", "\[ConjugateTranspose]"}]}], "==", "m"}], ",", 
   RowBox[{
    RowBox[{"m", ".", "proj"}], "==", "mp"}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"inc", "\[ConjugateTranspose]"}], ".", "proj"}], "==", 
    RowBox[{"IdentityMatrix", "[", 
     RowBox[{"MatrixRank", "[", "m", "]"}], "]"}]}]}], "}"}]}], "Input",
 CellLabel->"In[30]:=",ExpressionUUID->"e183fb1a-8bcd-44b1-a584-556c2468d2d9"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"2", "0", "0"},
       {
        RowBox[{
         FractionBox["1", "2"], "+", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "2"]}], 
        SqrtBox[
         FractionBox["19", "2"]], "0"},
       {
        RowBox[{
         RowBox[{"-", 
          FractionBox["1", "2"]}], "+", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "2"]}], 
        FractionBox[
         RowBox[{"16", "-", 
          RowBox[{"3", " ", "\[ImaginaryI]"}]}], 
         SqrtBox["38"]], 
        RowBox[{"4", " ", 
         SqrtBox[
          FractionBox["3", "19"]]}]},
       {"1", 
        RowBox[{"-", 
         FractionBox[
          RowBox[{"3", "+", 
           RowBox[{"3", " ", "\[ImaginaryI]"}]}], 
          SqrtBox["38"]]}], 
        RowBox[{"4", " ", 
         SqrtBox[
          FractionBox["3", "19"]]}]},
       {
        RowBox[{
         FractionBox["1", "2"], "+", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "2"]}], 
        SqrtBox[
         FractionBox["19", "2"]], "0"},
       {
        RowBox[{
         FractionBox["1", "2"], "+", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "2"]}], 
        SqrtBox[
         FractionBox["19", "2"]], "0"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {
        FractionBox["1", "2"], 
        FractionBox[
         RowBox[{
          FractionBox["3", "2"], "+", 
          FractionBox[
           RowBox[{"3", " ", "\[ImaginaryI]"}], "2"]}], 
         SqrtBox["38"]], 
        RowBox[{
         RowBox[{"-", "2"}], " ", 
         SqrtBox[
          FractionBox["3", "19"]]}]},
       {
        FractionBox["1", "2"], 
        RowBox[{"-", 
         FractionBox[
          RowBox[{
           FractionBox["5", "2"], "-", 
           FractionBox[
            RowBox[{"3", " ", "\[ImaginaryI]"}], "2"]}], 
          SqrtBox["38"]]}], 
        FractionBox[
         RowBox[{"2", "+", 
          FractionBox[
           RowBox[{"3", " ", "\[ImaginaryI]"}], "2"]}], 
         SqrtBox["57"]]},
       {
        FractionBox["1", "2"], 
        RowBox[{"-", 
         FractionBox[
          RowBox[{
           FractionBox["1", "2"], "+", 
           FractionBox[
            RowBox[{"9", " ", "\[ImaginaryI]"}], "2"]}], 
          SqrtBox["38"]]}], 
        FractionBox[
         RowBox[{
          FractionBox["1", "2"], "-", 
          FractionBox[
           RowBox[{"3", " ", "\[ImaginaryI]"}], "2"]}], 
         SqrtBox["57"]]},
       {
        FractionBox["1", "2"], 
        FractionBox[
         RowBox[{
          FractionBox["3", "2"], "+", 
          FractionBox[
           RowBox[{"3", " ", "\[ImaginaryI]"}], "2"]}], 
         SqrtBox["38"]], 
        FractionBox["7", 
         RowBox[{"2", " ", 
          SqrtBox["57"]}]]},
       {"0", "0", "0"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {
        FractionBox["1", "2"], 
        FractionBox[
         RowBox[{
          FractionBox["3", "2"], "+", 
          FractionBox[
           RowBox[{"3", " ", "\[ImaginaryI]"}], "2"]}], 
         SqrtBox["38"]], 
        RowBox[{
         RowBox[{"-", "2"}], " ", 
         SqrtBox[
          FractionBox["3", "19"]]}]},
       {
        FractionBox["1", "2"], 
        RowBox[{"-", 
         FractionBox[
          RowBox[{
           FractionBox["5", "2"], "-", 
           FractionBox[
            RowBox[{"3", " ", "\[ImaginaryI]"}], "2"]}], 
          SqrtBox["38"]]}], 
        FractionBox[
         RowBox[{"2", "+", 
          FractionBox[
           RowBox[{"3", " ", "\[ImaginaryI]"}], "2"]}], 
         SqrtBox["57"]]},
       {
        FractionBox["1", "2"], 
        RowBox[{"-", 
         FractionBox[
          RowBox[{
           FractionBox["1", "2"], "+", 
           FractionBox[
            RowBox[{"9", " ", "\[ImaginaryI]"}], "2"]}], 
          SqrtBox["38"]]}], 
        FractionBox[
         RowBox[{
          FractionBox["1", "2"], "-", 
          FractionBox[
           RowBox[{"3", " ", "\[ImaginaryI]"}], "2"]}], 
         SqrtBox["57"]]},
       {
        FractionBox["1", "2"], 
        FractionBox[
         RowBox[{
          FractionBox["3", "2"], "+", 
          FractionBox[
           RowBox[{"3", " ", "\[ImaginaryI]"}], "2"]}], 
         SqrtBox["38"]], 
        FractionBox["7", 
         RowBox[{"2", " ", 
          SqrtBox["57"]}]]},
       {"0", "0", "0"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1", "1", "1", "1", "0"},
       {"1", 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"},
       {
        RowBox[{"-", "1"}], 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"},
       {
        RowBox[{"-", "1"}], "1", "1", "1", "0"},
       {"1", 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"},
       {"1", 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]}], "}"}]], "Echo",
 CellChangeTimes->{
  3.8564454541674137`*^9},ExpressionUUID->"e2882545-e6d5-4618-827e-\
a0cb1d477c8a"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"True", ",", "True", ",", "True"}], "}"}]], "Output",
 CellChangeTimes->{3.8564441930614004`*^9, 3.856445454232224*^9},
 CellLabel->"Out[31]=",ExpressionUUID->"5c9c3ab5-75c4-4c76-a7a7-21b1d660fa44"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"MatrixForm", "/@", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"mp", ",", "inc", ",", "proj", ",", "m"}], "}"}], "=", 
      RowBox[{"Simplify", "[", 
       RowBox[{"NewMandMap2", "[", "testM", "]"}], "]"}]}], ")"}]}], "//", 
   "Echo"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"mp", ".", 
     RowBox[{"inc", "\[ConjugateTranspose]"}]}], "==", "m"}], ",", 
   RowBox[{
    RowBox[{"m", ".", "proj"}], "==", "mp"}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"inc", "\[ConjugateTranspose]"}], ".", "proj"}], "==", 
    RowBox[{"IdentityMatrix", "[", 
     RowBox[{"MatrixRank", "[", "m", "]"}], "]"}]}]}], "}"}]}], "Input",
 CellChangeTimes->{3.856444589124835*^9},
 CellLabel->"In[32]:=",ExpressionUUID->"2f4e45b3-7fa6-4592-8ff3-36def4589d14"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1", "1", "1"},
       {"1", 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}]},
       {
        RowBox[{"-", "1"}], 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}]},
       {
        RowBox[{"-", "1"}], "1", "1"},
       {"1", 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}]},
       {"1", 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}]}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1", "0", "0"},
       {"0", "1", "0"},
       {"0", "0", "1"},
       {"0", 
        RowBox[{
         FractionBox["4", "5"], "-", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "5"]}], 
        RowBox[{
         FractionBox["1", "5"], "+", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "5"]}]},
       {"0", "0", "0"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1", "0", "0"},
       {"0", 
        FractionBox["7", "12"], 
        RowBox[{
         FractionBox["1", "12"], "-", 
         FractionBox["\[ImaginaryI]", "4"]}]},
       {"0", 
        RowBox[{
         FractionBox["1", "12"], "+", 
         FractionBox["\[ImaginaryI]", "4"]}], 
        FractionBox["5", "6"]},
       {"0", 
        RowBox[{
         FractionBox["1", "3"], "-", 
         FractionBox["\[ImaginaryI]", "4"]}], 
        RowBox[{
         FractionBox["1", "12"], "+", 
         FractionBox["\[ImaginaryI]", "4"]}]},
       {"0", "0", "0"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1", "1", "1", "1", "0"},
       {"1", 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"},
       {
        RowBox[{"-", "1"}], 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"},
       {
        RowBox[{"-", "1"}], "1", "1", "1", "0"},
       {"1", 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"},
       {"1", 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]}], "}"}]], "Echo",
 CellChangeTimes->{
  3.856445456380024*^9},ExpressionUUID->"4d019157-92e3-41ae-9ef1-\
43c4ff332e91"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"True", ",", "True", ",", "True"}], "}"}]], "Output",
 CellChangeTimes->{3.856444590052809*^9, 3.8564446280947685`*^9, 
  3.8564454564131303`*^9},
 CellLabel->"Out[33]=",ExpressionUUID->"35e711e8-d3ad-4d63-89f4-c1a5500f2b41"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"MatrixForm", "/@", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"mp", ",", "inc", ",", "proj", ",", "m"}], "}"}], "=", 
      RowBox[{"Simplify", "[", 
       RowBox[{"NewMandMap2b", "[", "testM", "]"}], "]"}]}], ")"}]}], "//", 
   "Echo"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"mp", ".", 
     RowBox[{"inc", "\[ConjugateTranspose]"}]}], "==", "m"}], ",", 
   RowBox[{
    RowBox[{"m", ".", "proj"}], "==", "mp"}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"inc", "\[ConjugateTranspose]"}], ".", "proj"}], "==", 
    RowBox[{"IdentityMatrix", "[", 
     RowBox[{"MatrixRank", "[", "m", "]"}], "]"}]}]}], "}"}]}], "Input",
 CellChangeTimes->{3.8564447980686445`*^9},
 CellLabel->"In[34]:=",ExpressionUUID->"85bc0d3e-6657-4c92-9542-a765df9add69"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1", 
        RowBox[{
         FractionBox["9", "5"], "-", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "5"]}], 
        RowBox[{
         FractionBox["6", "5"], "+", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "5"]}]},
       {"1", 
        RowBox[{
         RowBox[{"-", 
          FractionBox["1", "5"]}], "-", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "5"]}], 
        RowBox[{
         FractionBox["1", "5"], "+", 
         FractionBox[
          RowBox[{"18", " ", "\[ImaginaryI]"}], "5"]}]},
       {
        RowBox[{"-", "1"}], 
        RowBox[{
         RowBox[{"-", 
          FractionBox["1", "5"]}], "-", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "5"]}], 
        RowBox[{
         FractionBox["1", "5"], "+", 
         FractionBox[
          RowBox[{"18", " ", "\[ImaginaryI]"}], "5"]}]},
       {
        RowBox[{"-", "1"}], 
        RowBox[{
         FractionBox["9", "5"], "-", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "5"]}], 
        RowBox[{
         FractionBox["6", "5"], "+", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "5"]}]},
       {"1", 
        RowBox[{
         RowBox[{"-", 
          FractionBox["1", "5"]}], "-", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "5"]}], 
        RowBox[{
         FractionBox["1", "5"], "+", 
         FractionBox[
          RowBox[{"18", " ", "\[ImaginaryI]"}], "5"]}]},
       {"1", 
        RowBox[{
         RowBox[{"-", 
          FractionBox["1", "5"]}], "-", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "5"]}], 
        RowBox[{
         FractionBox["1", "5"], "+", 
         FractionBox[
          RowBox[{"18", " ", "\[ImaginaryI]"}], "5"]}]}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1", "0", "0"},
       {"0", 
        FractionBox["7", "12"], 
        RowBox[{
         FractionBox["1", "12"], "-", 
         FractionBox["\[ImaginaryI]", "4"]}]},
       {"0", 
        RowBox[{
         FractionBox["1", "12"], "+", 
         FractionBox["\[ImaginaryI]", "4"]}], 
        FractionBox["5", "6"]},
       {"0", 
        RowBox[{
         FractionBox["1", "3"], "-", 
         FractionBox["\[ImaginaryI]", "4"]}], 
        RowBox[{
         FractionBox["1", "12"], "+", 
         FractionBox["\[ImaginaryI]", "4"]}]},
       {"0", "0", "0"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1", "0", "0"},
       {"0", "1", "0"},
       {"0", "0", "1"},
       {"0", 
        RowBox[{
         FractionBox["4", "5"], "-", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "5"]}], 
        RowBox[{
         FractionBox["1", "5"], "+", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "5"]}]},
       {"0", "0", "0"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1", "1", "1", "1", "0"},
       {"1", 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"},
       {
        RowBox[{"-", "1"}], 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"},
       {
        RowBox[{"-", "1"}], "1", "1", "1", "0"},
       {"1", 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"},
       {"1", 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]}], "}"}]], "Echo",
 CellChangeTimes->{
  3.8564454591098824`*^9},ExpressionUUID->"507bf542-093a-4784-88cc-\
de0a4f78055f"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"True", ",", "True", ",", "True"}], "}"}]], "Output",
 CellChangeTimes->{3.856444798284848*^9, 3.856445459135992*^9},
 CellLabel->"Out[35]=",ExpressionUUID->"a67cd722-6230-4f97-be14-687d7f14c0b6"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"1", "1", "1", "1", "0"},
     {"1", 
      RowBox[{"-", "1"}], 
      RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"},
     {
      RowBox[{"-", "1"}], 
      RowBox[{"-", "1"}], 
      RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"},
     {
      RowBox[{"-", "1"}], "1", "1", "1", "0"},
     {"1", 
      RowBox[{"-", "1"}], 
      RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"},
     {"1", 
      RowBox[{"-", "1"}], 
      RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}], "//", "RowReduce"}]], "Input",
 CellChangeTimes->{{3.8564530475917883`*^9, 3.856453049323265*^9}},
 CellLabel->"In[47]:=",ExpressionUUID->"ba5c82a0-088c-4cf1-a1ad-0986fca7455a"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0", ",", 
     RowBox[{
      FractionBox["4", "5"], "+", 
      FractionBox[
       RowBox[{"3", " ", "\[ImaginaryI]"}], "5"]}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", 
     RowBox[{
      FractionBox["1", "5"], "-", 
      FractionBox[
       RowBox[{"3", " ", "\[ImaginaryI]"}], "5"]}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}]}], 
  "}"}]], "Output",
 CellChangeTimes->{3.8564530495805693`*^9},
 CellLabel->"Out[47]=",ExpressionUUID->"6ee9cc06-7364-4385-a233-2ef377874ea1"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"MatrixForm", "/@", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"mp", ",", "inc", ",", "proj", ",", "m"}], "}"}], "=", 
      RowBox[{"Simplify", "[", 
       RowBox[{"NewMandMap3", "[", "testM", "]"}], "]"}]}], ")"}]}], "//", 
   "Echo"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"mp", ".", 
     RowBox[{"inc", "\[ConjugateTranspose]"}]}], "==", "m"}], ",", 
   RowBox[{
    RowBox[{"m", ".", "proj"}], "==", "mp"}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"inc", "\[ConjugateTranspose]"}], ".", "proj"}], "==", 
    RowBox[{"IdentityMatrix", "[", 
     RowBox[{"MatrixRank", "[", "m", "]"}], "]"}]}]}], "}"}]}], "Input",
 CellChangeTimes->{3.85644511104425*^9},
 CellLabel->"In[36]:=",ExpressionUUID->"1e49dcbf-fe57-4a2f-a7ee-a37555f78f70"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1", "1", "1"},
       {"1", 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1"},
       {
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1"},
       {
        RowBox[{"-", "1"}], "1", "1"},
       {"1", 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1"},
       {"1", 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1", "0", "0"},
       {"0", 
        RowBox[{
         FractionBox["1", "5"], "-", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "5"]}], 
        RowBox[{
         FractionBox["4", "5"], "+", 
         FractionBox[
          RowBox[{"3", " ", "\[ImaginaryI]"}], "5"]}]},
       {"0", "1", "0"},
       {"0", "0", "1"},
       {"0", "0", "0"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1", "0", "0"},
       {"0", 
        RowBox[{
         FractionBox["1", "12"], "-", 
         FractionBox["\[ImaginaryI]", "4"]}], 
        RowBox[{
         FractionBox["1", "3"], "+", 
         FractionBox["\[ImaginaryI]", "4"]}]},
       {"0", 
        FractionBox["5", "6"], 
        RowBox[{
         FractionBox["1", "12"], "-", 
         FractionBox["\[ImaginaryI]", "4"]}]},
       {"0", 
        RowBox[{
         FractionBox["1", "12"], "+", 
         FractionBox["\[ImaginaryI]", "4"]}], 
        FractionBox["7", "12"]},
       {"0", "0", "0"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1", "1", "1", "1", "0"},
       {"1", 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"},
       {
        RowBox[{"-", "1"}], 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"},
       {
        RowBox[{"-", "1"}], "1", "1", "1", "0"},
       {"1", 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"},
       {"1", 
        RowBox[{"-", "1"}], 
        RowBox[{"3", " ", "\[ImaginaryI]"}], "1", "0"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]}], "}"}]], "Echo",
 CellChangeTimes->{
  3.8564454607310877`*^9},ExpressionUUID->"950ab129-9f5a-4e73-97f9-\
f8d428f2d3b9"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"True", ",", "True", ",", "True"}], "}"}]], "Output",
 CellChangeTimes->{3.856445111373085*^9, 3.856445460755069*^9},
 CellLabel->"Out[37]=",ExpressionUUID->"b56b39b8-1aeb-4878-9c1f-92e365a94bbd"]
}, Open  ]]
}, Open  ]],

Cell[" ", "Text",
 Editable->False,
 Selectable->False,
 CellFrame->{{0, 0}, {0, 3}},
 ShowCellBracket->False,
 CellMargins->{{0, 0}, {1, 1}},
 CellElementSpacings->{"CellMinHeight"->1},
 CellFrameMargins->0,
 CellFrameColor->RGBColor[0, 0, 1],
 CellSize->{
  Inherited, 5},ExpressionUUID->"fe506bc8-0dff-4042-97dc-1e293f4c795a"]
}, Closed]],

Cell[CellGroupData[{

Cell["Mathematica Row Rescaling To Integers Code:", "Subsubsection",
 CellChangeTimes->{{3.851685419117911*^9, 3.851685437578148*^9}, {
  3.851685470990306*^9, 
  3.8516854782867985`*^9}},ExpressionUUID->"4fb92e0e-4de9-4989-970a-\
9a445a85dcad"],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"RenormalizeByGCD", "[", 
    RowBox[{"matrix_", "?", "MatrixQ"}], "]"}], ":=", 
   RowBox[{"With", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"gcd", "=", 
       RowBox[{"PolynomialGCD", "@@@", "matrix"}]}], "}"}], ",", 
     RowBox[{
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"Head", "[", "matrix", "]"}], "===", "SparseArray"}], ",", 
        "SparseArray", ",", "Developer`ToPackedArray"}], "]"}], "[", 
      FractionBox["matrix", 
       RowBox[{"Abs", "[", 
        RowBox[{"gcd", "+", "1", "-", 
         RowBox[{"Unitize", "[", "gcd", "]"}]}], "]"}]], "]"}]}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"RealMatrixToIntegerMatrix", "=", 
   RowBox[{
    RowBox[{"RenormalizeByGCD", "[", 
     RowBox[{"SparseArray", "[", 
      RowBox[{"Rationalize", "[", 
       RowBox[{
        RowBox[{"#", "//", "Normal"}], ",", 
        SuperscriptBox["10", 
         RowBox[{"-", "8"}]]}], "]"}], "]"}], "]"}], "&"}]}], ";"}]}], "Code",
 CellChangeTimes->{{3.851685659782112*^9, 3.851685660104169*^9}},
 CellLabel->"In[15]:=",ExpressionUUID->"163ed304-c504-4a7f-9be8-d918cf3dbdd8"]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[{
 "Mathematica Convex Hull (from ",
 StyleBox["rays",
  FontSlant->"Italic"],
 ")",
 StyleBox[" ",
  FontSlant->"Italic"],
 "code:"
}], "Subsubsection",
 CellChangeTimes->{{3.851685419117911*^9, 3.851685437578148*^9}, {
  3.851685470990306*^9, 3.8516854782867985`*^9}, {3.8516855301907043`*^9, 
  3.851685548589467*^9}},ExpressionUUID->"98c92339-349f-4e01-95ac-\
eb4b36a6aa40"],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"VMatrixToCDDString", "[", 
     RowBox[{"mat_", "?", "MatrixQ"}], "]"}], ":=", 
    RowBox[{
     RowBox[{"With", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"renormalized$mat", "=", 
         RowBox[{"mat", "//", "RenormalizeByGCD"}]}], "}"}], ",", 
       RowBox[{"With", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"rational", "=", 
           RowBox[{"MatrixQ", "[", 
            RowBox[{"renormalized$mat", ",", "IntegerQ"}], "]"}]}], "}"}], 
         ",", "\n", 
         RowBox[{
          RowBox[{"ExportString", "[", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{"\"\<V-representation\>\"", ",", "\"\<begin\>\"", ",", 
              RowBox[{"Append", "[", 
               RowBox[{
                RowBox[{"Dimensions", "[", "mat", "]"}], ",", 
                RowBox[{"If", "[", 
                 RowBox[{
                 "rational", ",", "\"\<integer\>\"", ",", "\"\<real\>\""}], 
                 "]"}]}], "]"}]}], "}"}], ",", "\"\<Table\>\"", ",", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"\"\<FieldSeparators\>\"", "\[Rule]", "\"\< \>\""}], 
              ",", 
              RowBox[{"\"\<LineSeparators\>\"", "\[Rule]", "\"\<\\n\>\""}]}], 
             "}"}]}], "]"}], "<>", "\"\<\\n\>\"", "<>", "\[IndentingNewLine]", 
          RowBox[{"ExportString", "[", 
           RowBox[{
            RowBox[{"If", "[", 
             RowBox[{"rational", ",", "renormalized$mat", ",", 
              RowBox[{"renormalized$mat", "//", "N"}]}], "]"}], ",", 
            "\"\<Table\>\"", ",", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{"\"\<FieldSeparators\>\"", "\[Rule]", "\"\< \>\""}], 
              ",", 
              RowBox[{"\"\<LineSeparators\>\"", "\[Rule]", "\"\<\\n\>\""}]}], 
             "}"}]}], "]"}]}]}], "]"}]}], "]"}], "<>", "\"\<\\n\>\"", "<>", 
     "\[IndentingNewLine]", 
     RowBox[{"ExportString", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", "\"\<end\>\"", "}"}], "}"}], ",", "\"\<Table\>\"", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"\"\<FieldSeparators\>\"", "\[Rule]", "\"\< \>\""}], ",", 
         RowBox[{"\"\<LineSeparators\>\"", "\[Rule]", "\"\<\\n\>\""}]}], 
        "}"}]}], "]"}]}]}], ";"}], "\n", "\n", 
  RowBox[{"(*", 
   RowBox[{"UPDATED", " ", 
    RowBox[{
     RowBox[{"9", "/", "25"}], "/", "2017"}], " ", "to", " ", "return", " ", 
    "equalities", " ", "from", " ", "CDD", " ", "as", " ", "extra", " ", 
    "flipped", " ", 
    RowBox[{"inequalities", ".", " ", "Use"}], " ", "of", " ", 
    "FirstPosition", " ", "requires", " ", "Mathematica", " ", "10.0"}], 
   "*)"}]}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"UseBinaryFuncOnMatrix", "[", "binary_Function", "]"}], "[", 
   RowBox[{"mat_", "?", "MatrixQ"}], "]"}], ":=", "\[IndentingNewLine]", 
  RowBox[{"With", "[", 
   RowBox[{
    RowBox[{"{", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"inputstring", "=", 
       RowBox[{"VMatrixToCDDString", "[", "mat", "]"}]}], ",", "\n", 
      RowBox[{"infilename", "=", "\"\<forscdd.ext\>\""}]}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"Export", "[", 
      RowBox[{
       RowBox[{"FileNameJoin", "[", 
        RowBox[{"{", 
         RowBox[{"$TemporaryDirectory", ",", "infilename"}], "}"}], "]"}], 
       ",", "inputstring", ",", "\"\<Text\>\""}], "]"}], ";", "\n", 
     RowBox[{"SetDirectory", "[", "$TemporaryDirectory", "]"}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"With", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"rawcddout", "=", 
         RowBox[{"Import", "[", 
          RowBox[{
           RowBox[{"\"\<!\>\"", "<>", 
            RowBox[{"binary", "[", "infilename", "]"}]}], ",", 
           "\"\<Table\>\""}], "]"}]}], "}"}], ",", 
       RowBox[{"With", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"rawineqs", "=", 
            RowBox[{
             RowBox[{"ToExpression", "[", 
              RowBox[{"rawcddout", "\[LeftDoubleBracket]", 
               RowBox[{
                RowBox[{"(", 
                 RowBox[{
                  RowBox[{
                   RowBox[{"FirstPosition", "[", 
                    RowBox[{"rawcddout", ",", 
                    RowBox[{"{", "\"\<begin\>\"", "}"}], ",", 
                    RowBox[{"{", 
                    RowBox[{"{", "}"}], "}"}], ",", 
                    RowBox[{"{", "1", "}"}]}], "]"}], "\[LeftDoubleBracket]", 
                   "1", "\[RightDoubleBracket]"}], "+", "2"}], ")"}], ";;", 
                RowBox[{"(", 
                 RowBox[{
                  RowBox[{
                   RowBox[{"FirstPosition", "[", 
                    RowBox[{"rawcddout", ",", 
                    RowBox[{"{", "\"\<end\>\"", "}"}], ",", 
                    RowBox[{"{", 
                    RowBox[{"{", "}"}], "}"}], ",", 
                    RowBox[{"{", "1", "}"}]}], "]"}], "\[LeftDoubleBracket]", 
                   "1", "\[RightDoubleBracket]"}], "-", "1"}], ")"}]}], 
               "\[RightDoubleBracket]"}], "]"}], "//", 
             "Developer`ToPackedArray"}]}], ",", 
           RowBox[{"linearityline", "=", 
            RowBox[{
             RowBox[{"FirstPosition", "[", 
              RowBox[{"rawcddout", ",", 
               RowBox[{"{", 
                RowBox[{"\"\<linearity\>\"", ",", "__Integer"}], "}"}], ",", 
               RowBox[{"{", 
                RowBox[{"{", "}"}], "}"}], ",", 
               RowBox[{"{", "1", "}"}]}], "]"}], "\[LeftDoubleBracket]", "1", 
             "\[RightDoubleBracket]"}]}]}], "}"}], ",", "\n", 
         RowBox[{
          RowBox[{"ResetDirectory", "[", "]"}], ";", "\n", 
          RowBox[{"(*", 
           RowBox[{
            RowBox[{"globalsave", "=", "rawcddout"}], ";"}], "*)"}], "\n", 
          RowBox[{
           RowBox[{"With", "[", 
            RowBox[{
             RowBox[{"{", 
              RowBox[{"equalitypositions", "=", 
               RowBox[{"If", "[", 
                RowBox[{
                 RowBox[{"IntegerQ", "[", "linearityline", "]"}], ",", 
                 RowBox[{"Drop", "[", 
                  RowBox[{
                   RowBox[{
                   "rawcddout", "\[LeftDoubleBracket]", "linearityline", 
                    "\[RightDoubleBracket]"}], ",", "2"}], "]"}], ",", 
                 RowBox[{"{", "}"}]}], "]"}]}], "}"}], ",", "\n", 
             RowBox[{"(*", 
              RowBox[{
               RowBox[{"Print", "[", 
                RowBox[{
                 RowBox[{"Length", "[", "equalitypositions", "]"}], ",", 
                 "\"\< Equality Positions:\\t\>\"", ",", 
                 "equalitypositions"}], "]"}], ";"}], "*)"}], "\n", 
             RowBox[{"SparseArray", "[", 
              RowBox[{"Join", "[", 
               RowBox[{"rawineqs", ",", 
                RowBox[{"-", 
                 RowBox[{
                 "rawineqs", "\[LeftDoubleBracket]", "equalitypositions", 
                  "\[RightDoubleBracket]"}]}]}], "]"}], "]"}]}], "]"}], "//", 
           "RealMatrixToIntegerMatrix"}]}]}], "]"}]}], "]"}]}]}], 
   "]"}]}]}], "Code",
 CellChangeTimes->{{3.7035137622830443`*^9, 3.7035138074718447`*^9}, {
   3.7045481326520195`*^9, 3.7045481529611406`*^9}, {3.70639220841425*^9, 
   3.7063922423023376`*^9}, {3.706392947708648*^9, 3.7063929480896883`*^9}, {
   3.7153760688375425`*^9, 3.7153760714625244`*^9}, 3.7153761212466702`*^9, {
   3.7153773428872414`*^9, 3.71537744465875*^9}, {3.7153775423484235`*^9, 
   3.715377591185216*^9}, {3.7153777529692383`*^9, 3.7153777641856585`*^9}, {
   3.7153778209294577`*^9, 3.7153778254174175`*^9}, {3.7153778868165855`*^9, 
   3.7153779237778544`*^9}, {3.715377954121503*^9, 3.7153780329700794`*^9}, {
   3.7153781057384796`*^9, 3.7153781649779716`*^9}, {3.715378264960758*^9, 
   3.7153782893225117`*^9}, {3.715525069333522*^9, 3.715525075619923*^9}, {
   3.7155251598274837`*^9, 3.715525173514492*^9}, {3.715526152344182*^9, 
   3.715526163648197*^9}, {3.7155263084321814`*^9, 3.715526322288012*^9}, {
   3.71552637431935*^9, 3.715526394798994*^9}, {3.7155264746701803`*^9, 
   3.715526489285564*^9}, {3.715526693757826*^9, 3.715526695857361*^9}, {
   3.7155267262531843`*^9, 3.7155267310578823`*^9}, {3.7155267993489513`*^9, 
   3.715526800833161*^9}, {3.7155269841400623`*^9, 3.7155269921883297`*^9}, 
   3.715527348697876*^9, {3.7155284985642347`*^9, 3.7155285124683104`*^9}, {
   3.71724470979348*^9, 3.7172447103915105`*^9}, {3.717248592991723*^9, 
   3.717248593457056*^9}, {3.7172486422497077`*^9, 3.7172486424328375`*^9}, {
   3.7172486793922358`*^9, 3.7172487019202633`*^9}, {3.7172487347536244`*^9, 
   3.7172487699070187`*^9}, {3.717248803071617*^9, 3.7172488243527584`*^9}, {
   3.717248983067834*^9, 3.7172489832129393`*^9}, {3.7172490877949996`*^9, 
   3.7172490879250927`*^9}, 3.717249193527242*^9, {3.7172492457895107`*^9, 
   3.717249267106694*^9}, {3.7172492973955007`*^9, 3.717249297550628*^9}, {
   3.7172601342944117`*^9, 3.717260135187046*^9}, {3.717261536749022*^9, 
   3.7172615377797556`*^9}, {3.809689317319308*^9, 3.809689318208872*^9}, {
   3.851685958338287*^9, 3.851685962334543*^9}, {3.8516860069660935`*^9, 
   3.8516860102096443`*^9}, {3.8595695470946894`*^9, 3.859569550289259*^9}, {
   3.859570252933361*^9, 3.8595702805885506`*^9}, {3.8595705026226006`*^9, 
   3.859570504380865*^9}, {3.859570710827712*^9, 3.8595707430525866`*^9}, {
   3.859570796876666*^9, 3.859570801709857*^9}, {3.8595708568587265`*^9, 
   3.8595708575022397`*^9}, {3.85957094406839*^9, 3.859570945125911*^9}, {
   3.859570988621497*^9, 3.8595709895302677`*^9}, 3.8595710630422754`*^9, {
   3.8599302793874445`*^9, 3.8599302830932565`*^9}, {3.8599310274264207`*^9, 
   3.8599310517108574`*^9}, {3.859931108110671*^9, 3.8599311724241705`*^9}, {
   3.8599314577308316`*^9, 3.85993148727211*^9}, {3.866902308942631*^9, 
   3.866902314359599*^9}, {3.8669023599438877`*^9, 3.8669024770134735`*^9}},
 CellLabel->"In[60]:=",ExpressionUUID->"68aaac52-9924-4309-9b72-7ef068a73ed2"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FileNameJoin", "[", 
  RowBox[{"{", 
   RowBox[{"$TemporaryDirectory", ",", "\"\<forscdd.ext\>\""}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[59]:=",ExpressionUUID->"f4d8bc79-bb13-4395-8feb-5c90dcacc067"],

Cell[BoxData["\<\"D:\\\\TEMP\\\\forscdd.ext\"\>"], "Output",
 CellChangeTimes->{3.85993066137354*^9},
 CellLabel->"Out[59]=",ExpressionUUID->"48ce37e5-0690-43d4-b5d7-cb60bb044661"]
}, Open  ]],

Cell[BoxData[""], "Input",
 CellChangeTimes->{{3.8595694849418845`*^9, 
  3.8595695667761927`*^9}},ExpressionUUID->"e4d47643-85f4-47d5-9a29-\
9d1a7ec26265"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"ConvexHullRays$CDD", ":=", 
   RowBox[{"UseBinaryFuncOnMatrix", "[", "LCDD$Instruction", "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"(*", 
   RowBox[{
    RowBox[{"ConvexHullRays$CDD", ":=", 
     RowBox[{
      RowBox[{"UseBinaryOnMatrix", "[", 
       RowBox[{"If", "[", 
        RowBox[{
         RowBox[{"$OperatingSystem", "===", "\"\<Windows\>\""}], ",", 
         "WINDOWScdd", ",", "MACcdd"}], "]"}], "]"}], "/*", 
      "RealMatrixToIntegerMatrix"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"ConvexHullRays$CDDGMP", ":=", 
     RowBox[{"UseBinaryOnMatrix", "[", 
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{"$OperatingSystem", "===", "\"\<Windows\>\""}], ",", 
        "WINDOWScddgmp", ",", "MACcddgmp"}], "]"}], "]"}]}], ";", "\n", 
    RowBox[{"RemoveRedundant$CDD", ":=", 
     RowBox[{"UseBinaryOnMatrix", "[", 
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{"$OperatingSystem", "===", "\"\<Windows\>\""}], ",", 
        "WINDOWSredcheck", ",", "MACredcheck"}], "]"}], "]"}]}], ";"}], 
   "*)"}]}]}], "Code",
 CellChangeTimes->{{3.7063924891679993`*^9, 3.7063925035361757`*^9}, {
   3.706392541027841*^9, 3.7063926207959166`*^9}, {3.706392655255499*^9, 
   3.7063926707166014`*^9}, {3.7155284280038905`*^9, 3.715528488140982*^9}, {
   3.7155285763971114`*^9, 3.7155285855147505`*^9}, 3.8516882025582685`*^9, 
   3.851688260681367*^9, {3.8595695539273453`*^9, 3.8595695697436686`*^9}, {
   3.859570298267518*^9, 3.8595703017798753`*^9}, 3.859570446191043*^9, 
   3.8595708370020275`*^9, 3.8599305088306074`*^9},
 CellLabel->"In[19]:=",ExpressionUUID->"7031cc09-7c58-45f6-b504-e3788f770f78"]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Algorithm for embedding:", "Subsection",
 CellChangeTimes->{{3.851685419117911*^9, 3.851685437578148*^9}, {
  3.851685470990306*^9, 3.8516854782867985`*^9}, {3.8516855301907043`*^9, 
  3.851685548589467*^9}, {3.851685865521597*^9, 
  3.851685871412819*^9}},ExpressionUUID->"e61e3b31-1149-4ffc-beb2-\
262e22387740"],

Cell[CellGroupData[{

Cell["High-Level Summary", "Subsubsection",
 CellChangeTimes->{{3.8520407597064786`*^9, 3.8520407722904377`*^9}, {
  3.857820606498135*^9, 3.8578206082190332`*^9}, {3.8654906733005567`*^9, 
  3.865490678420291*^9}},ExpressionUUID->"5a91f733-0312-4255-955d-\
606b5f1e8bc3"],

Cell[TextData[{
 StyleBox["Stage 0 [OPTIONAL]: ",
  FontWeight->"Bold"],
 "Convert quantum ",
 StyleBox["density-matrix-based representation to GPT",
  FontColor->RGBColor[1, 0, 0]],
 " real-vector-based representation"
}], "Item",
 CellChangeTimes->{{3.865490747445092*^9, 3.865490882963541*^9}, {
  3.865491141043458*^9, 
  3.8654912030467176`*^9}},ExpressionUUID->"a1b1cf93-0bb1-4013-8654-\
39344ab81283"],

Cell[CellGroupData[{

Cell[TextData[{
 StyleBox["Stage 1: ",
  FontWeight->"Bold"],
 "Given a GPT fragment, ",
 StyleBox["find an ",
  FontColor->RGBColor[1, 0, 0]],
 StyleBox["accessible",
  FontSlant->"Italic",
  FontColor->RGBColor[1, 0, 0]],
 StyleBox[" GPT fragment",
  FontColor->RGBColor[1, 0, 0]],
 ". That is, ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"{", " ", 
    RowBox[{
     TemplateBox[{"V", "\[CapitalOmega]", "F"},
      "Subsuperscript"], ",", 
     TemplateBox[{"V", "\[Xi]", "F"},
      "Subsuperscript"], ",", 
     TemplateBox[{
       SuperscriptBox["u", "F"]},
      "Ket"]}], " ", "}"}], " ", "->", " ", 
   RowBox[{"{", " ", 
    RowBox[{
     TemplateBox[{"V", "\[CapitalOmega]", "A"},
      "Subsuperscript"], ",", 
     TemplateBox[{"V", "\[Xi]", "A"},
      "Subsuperscript"], ",", " ", 
     SubscriptBox["I", "\[CapitalOmega]"], ",", " ", 
     SubscriptBox["I", "\[Xi]"], ",", 
     TemplateBox[{
       SuperscriptBox["u", "A"]},
      "Ket"]}], " ", "}"}], " "}]],
  CellChangeTimes->{3.865490472348921*^9},ExpressionUUID->
  "f469c12e-5bac-4f73-968f-20fb16e2ff1d"]
}], "Item",
 CellChangeTimes->{{3.865490747445092*^9, 3.865490882963541*^9}, {
  3.865491141043458*^9, 3.865491209402831*^9}, {3.8654913740743685`*^9, 
  3.865491447676219*^9}, {3.865491540561975*^9, 
  3.8654915419082437`*^9}},ExpressionUUID->"4f70ab94-55c9-41cd-8cfe-\
8f2a1b9d0fac"],

Cell[TextData[{
 Cell[BoxData[
  TemplateBox[{"V", "\[CapitalOmega]", "A"},
   "Subsuperscript"]],ExpressionUUID->"b58d7991-c292-42b8-8d46-d0f23f3ea1a0"],
 "  and ",
 Cell[BoxData[
  TemplateBox[{"V", "\[Xi]", "A"},
   "Subsuperscript"]],ExpressionUUID->"1282351e-5a63-44cb-a235-bfd0c0181ea9"],
 "  are ",
 StyleBox["full column rank ",
  FontSlant->"Italic"],
 "matrices, even if ",
 Cell[BoxData[
  TemplateBox[{"V", "\[CapitalOmega]", "F"},
   "Subsuperscript"]],ExpressionUUID->"91f6319a-7e00-426c-a551-1eccf84a7bab"],
 " and/or ",
 Cell[BoxData[
  TemplateBox[{"V", "\[Xi]", "F"},
   "Subsuperscript"]],ExpressionUUID->"dce60a30-b51e-4112-9588-aa40dd3e23cb"],
 " are not."
}], "Subitem",
 CellChangeTimes->{{3.865490747445092*^9, 3.865490882963541*^9}, {
  3.865491141043458*^9, 3.865491209402831*^9}, {3.8654913740743685`*^9, 
  3.865491447676219*^9}, {3.865491540561975*^9, 3.8654916314911222`*^9}, {
  3.865491662209387*^9, 
  3.865491907929203*^9}},ExpressionUUID->"a2606422-7faf-45c7-9cee-\
49f2af703c2c"],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   RowBox[{
    TemplateBox[{"V", "\[CapitalOmega]", "A"},
     "Subsuperscript"], "\[CenterDot]", 
    RowBox[{
     SubscriptBox["I", "\[CapitalOmega]"], "\[Transpose]"}]}], "=", 
   TemplateBox[{"V", "\[CapitalOmega]", "F"},
    "Subsuperscript"]}]],ExpressionUUID->
  "0848e5ce-eb3a-46f5-a0c8-8875414b6d16"],
 "  and ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    TemplateBox[{"V", "\[Xi]", "A"},
     "Subsuperscript"], "\[CenterDot]", 
    RowBox[{
     SubscriptBox["I", "\[Xi]"], "\[Transpose]"}]}], "=", 
   TemplateBox[{"V", "\[Xi]", "F"},
    "Subsuperscript"]}]],ExpressionUUID->
  "84383549-9ad8-456f-b0e8-72297b47e81a"],
 " ,    such that",
 Cell[BoxData[
  RowBox[{
   TemplateBox[{
     SuperscriptBox["u", "F"]},
    "Ket"], "=", 
   RowBox[{
    SubscriptBox["I", "\[Xi]"], "\[CenterDot]", 
    TemplateBox[{
      SuperscriptBox["u", "A"]},
     "Ket"]}]}]],
  CellChangeTimes->{3.865490472348921*^9},ExpressionUUID->
  "b39dd0b9-f951-4991-a1dd-5be5473ca4c7"]
}], "Subitem",
 CellChangeTimes->{{3.865490747445092*^9, 3.865490882963541*^9}, {
  3.865491141043458*^9, 3.865491209402831*^9}, {3.8654913740743685`*^9, 
  3.865491447676219*^9}, {3.865491540561975*^9, 3.8654916314911222`*^9}, {
  3.865491662209387*^9, 3.865491831465187*^9}, {3.865491997243719*^9, 
  3.8654919972477193`*^9}},ExpressionUUID->"2cc485da-5ffc-4d97-ab25-\
df6984fd937f"],

Cell[TextData[{
 "Given an (optional-to-specify) ",
 StyleBox["maximally mixed state",
  FontSlant->"Italic"],
 ", we can construct the ",
 StyleBox["depolarizing map",
  FontWeight->"Bold"],
 " which takes every state in ",
 Cell[BoxData[
  TemplateBox[{"V", "\[CapitalOmega]", "A"},
   "Subsuperscript"]],ExpressionUUID->"9c1f1f1b-8b4d-42dd-93af-14f7e87a6707"],
 " to the (projected version) of the maximally mixed state."
}], "Subitem",
 CellChangeTimes->{{3.865490747445092*^9, 3.865490882963541*^9}, {
  3.865491141043458*^9, 3.865491209402831*^9}, {3.8654913740743685`*^9, 
  3.865491447676219*^9}, {3.865491540561975*^9, 3.8654916314911222`*^9}, {
  3.865491662209387*^9, 3.865491831465187*^9}, {3.8654920000245914`*^9, 
  3.8654920591525393`*^9}, {3.8654920979443235`*^9, 
  3.865492117112265*^9}},ExpressionUUID->"abcf28db-03c9-4433-9833-\
90f16d81c9a3"]
}, Open  ]],

Cell[TextData[{
 StyleBox["Stage 2: ",
  FontWeight->"Bold"],
 "Given an accessible GPT Fragment ",
 Cell[BoxData[
  RowBox[{"{", " ", 
   RowBox[{
    TemplateBox[{"V", "\[CapitalOmega]", "A"},
     "Subsuperscript"], ",", 
    TemplateBox[{"V", "\[Xi]", "A"},
     "Subsuperscript"], ",", " ", 
    SubscriptBox["I", "\[CapitalOmega]"], ",", " ", 
    SubscriptBox["I", "\[Xi]"], ",", 
    TemplateBox[{
      SuperscriptBox["u", "A"]},
     "Ket"]}], " ", "}"}]],
  CellChangeTimes->{3.865490472348921*^9},ExpressionUUID->
  "7311600d-8a85-48a0-818b-6e21fd813b2f"],
 ", ",
 StyleBox["find ",
  FontColor->RGBColor[0, 0, 1]],
 Cell[BoxData[
  RowBox[{"{", " ", 
   RowBox[{
    SubscriptBox["H", "\[CapitalOmega]"], ",", " ", 
    SubscriptBox["H", "\[Xi]"]}], "  ", "}"}]],
  CellChangeTimes->{3.8654904726343393`*^9},
  FontColor->RGBColor[0, 0, 1],ExpressionUUID->
  "49bff072-c8aa-43b0-8d09-9fe577dd70fc"],
 " such that  ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    SubscriptBox["H", "\[CapitalOmega]"], "\[CenterDot]", 
    RowBox[{
     TemplateBox[{"V", "\[CapitalOmega]", "A"},
      "Subsuperscript"], "\[Transpose]"}]}], ">=", "0"}]],
  CellChangeTimes->{3.8654904726343393`*^9},ExpressionUUID->
  "12aa4a59-20b7-4f43-b1a7-4c91c7ac2dfe"],
 " and ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    SubscriptBox["H", "\[Xi]"], "\[CenterDot]", 
    RowBox[{
     TemplateBox[{"V", "\[Xi]", "A"},
      "Subsuperscript"], "\[Transpose]"}]}], ">=", "0"}]],
  CellChangeTimes->{3.8654904726343393`*^9},ExpressionUUID->
  "43d90b18-777d-4d21-a2d5-baa14385aa84"],
 " and ",
 Cell[BoxData[
  RowBox[{
   TemplateBox[{"s"},
    "Ket"], "\[Element]", 
   RowBox[{"Cone", "[", 
    TemplateBox[{"V", "\[CapitalOmega]", "A"},
     "Subsuperscript"], "]"}]}]],ExpressionUUID->
  "c1a8eb71-4a59-4965-b212-26853d1fadaa"],
 "  ",
 StyleBox["iff",
  FontSlant->"Italic"],
 "  ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    SubscriptBox["H", "\[Xi]"], "\[CenterDot]", 
    TemplateBox[{"s"},
     "Ket"]}], ">=", "0"}]],
  CellChangeTimes->{3.8654904726343393`*^9},ExpressionUUID->
  "0afcb53a-bd2e-40ef-8abf-d9778e4b2467"],
 " etc."
}], "Item",
 CellChangeTimes->{{3.865490747445092*^9, 3.865490873604588*^9}, {
  3.8654909767556047`*^9, 3.8654910872965326`*^9}, {3.8654911317330494`*^9, 
  3.865491133325512*^9}, {3.865491349692706*^9, 3.865491369418964*^9}, {
  3.8654914533056955`*^9, 3.8654914696558733`*^9}, {3.865492438881692*^9, 
  3.8654924563133383`*^9}, {3.865494835001817*^9, 
  3.865494922673046*^9}},ExpressionUUID->"cdf64d17-89af-4acc-adc4-\
8c840f100665"],

Cell[CellGroupData[{

Cell[TextData[{
 StyleBox["Stage 3: ",
  FontWeight->"Bold"],
 "Determine ",
 StyleBox["if there exists any ",
  FontColor->RGBColor[0, 0, 1]],
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Sigma]", ">=", "0"}], TraditionalForm]],
  FormatType->TraditionalForm,
  FontColor->RGBColor[0, 0, 1],ExpressionUUID->
  "9d86d949-f79e-450e-9d6a-0414dd2f3373"],
 " such that ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    RowBox[{
     SubscriptBox["I", "\[Xi]"], "\[Transpose]"}], "\[CenterDot]", 
    SubscriptBox["I", "\[CapitalOmega]"]}], " ", "=", " ", 
   RowBox[{
    RowBox[{
     SubscriptBox["H", "\[Xi]"], "\[Transpose]"}], "\[CenterDot]", "\[Sigma]",
     "\[CenterDot]", 
    SubscriptBox["H", "\[CapitalOmega]"]}]}]],
  CellChangeTimes->{3.865490472348921*^9},ExpressionUUID->
  "ec2be2c7-d2df-4c84-bc11-cce168bab45d"],
 "."
}], "Item",
 CellChangeTimes->{{3.865490747445092*^9, 3.865490873604588*^9}, {
  3.8654909767556047`*^9, 3.8654910872965326`*^9}, {3.8654911317330494`*^9, 
  3.865491133325512*^9}, {3.865491349692706*^9, 3.865491369418964*^9}, {
  3.8654914533056955`*^9, 3.865491472577675*^9}, {3.865492131504867*^9, 
  3.8654921645448575`*^9}, {3.865492397868293*^9, 
  3.8654925065298758`*^9}},ExpressionUUID->"cbb0b52f-b1c6-4025-b95a-\
ea8c2b6d1a09"],

Cell["\<\
More precisely, we determine how much noise is required to make the \
accessible GPT fragment simplex-embeddable, that is:\
\>", "Subitem",
 CellChangeTimes->{{3.865490747445092*^9, 3.865490873604588*^9}, {
  3.8654909767556047`*^9, 3.8654910872965326`*^9}, {3.8654911317330494`*^9, 
  3.865491133325512*^9}, {3.865491349692706*^9, 3.865491369418964*^9}, {
  3.8654914533056955`*^9, 3.865491472577675*^9}, {3.865492131504867*^9, 
  3.8654921645448575`*^9}, {3.865492397868293*^9, 
  3.8654925439319534`*^9}},ExpressionUUID->"282a3d75-e159-4fef-830b-\
a7f0765fb544"],

Cell[TextData[{
 "Minimize ",
 StyleBox["r  ",
  FontSlant->"Italic"],
 "such that ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    RowBox[{
     SubscriptBox["I", "\[Xi]"], "\[Transpose]"}], "\[CenterDot]", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"r", "\[Times]", 
       RowBox[{"DepMap", "\[Transpose]"}]}], "+", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"1", "-", "r"}], ")"}], "\[Times]", "IdentityMap"}]}], ")"}], 
    "\[CenterDot]", 
    SubscriptBox["I", "\[CapitalOmega]"]}], " ", "=", " ", 
   RowBox[{
    RowBox[{
     SubscriptBox["H", "\[Xi]"], "\[Transpose]"}], "\[CenterDot]", "\[Sigma]",
     "\[CenterDot]", 
    SubscriptBox["H", "\[CapitalOmega]"]}]}]],
  CellChangeTimes->{3.865490472348921*^9},ExpressionUUID->
  "32734565-230c-43ee-9bd5-020687a63385"],
 "\[LineSeparator]"
}], "Subitem",
 CellChangeTimes->{{3.865490747445092*^9, 3.865490873604588*^9}, {
  3.8654909767556047`*^9, 3.8654910872965326`*^9}, {3.8654911317330494`*^9, 
  3.865491133325512*^9}, {3.865491349692706*^9, 3.865491369418964*^9}, {
  3.8654914533056955`*^9, 3.865491472577675*^9}, {3.865492131504867*^9, 
  3.8654921645448575`*^9}, {3.865492397868293*^9, 
  3.865492621488507*^9}},ExpressionUUID->"2031ef2c-0b8e-47e7-b437-\
69f2b9b9f985"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
 StyleBox["Stage 4:",
  FontWeight->"Bold"],
 " Explicitly reconstruct embedding maps ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    TemplateBox[{"\[Tau]", "\[CapitalOmega]", "A"},
     "Subsuperscript"], ",", 
    TemplateBox[{"\[Tau]", "\[Xi]", "A"},
     "Subsuperscript"]}], " ", "}"}]],
  CellChangeTimes->{3.8654927982727547`*^9},ExpressionUUID->
  "b7dc22a8-68b6-486a-b678-f5b9a2e5d82d"],
 " such that  "
}], "Item",
 CellChangeTimes->{{3.865490747445092*^9, 3.865490873604588*^9}, {
  3.8654909767556047`*^9, 3.8654910872965326`*^9}, {3.8654911317330494`*^9, 
  3.865491133325512*^9}, {3.865491349692706*^9, 3.865491369418964*^9}, {
  3.8654914533056955`*^9, 3.865491472577675*^9}, {3.865492131504867*^9, 
  3.8654921645448575`*^9}, {3.865492397868293*^9, 3.8654925327423553`*^9}, {
  3.8654928981579437`*^9, 3.8654929199340696`*^9}, {3.865493180278265*^9, 
  3.865493184533617*^9}, {3.865493226467978*^9, 
  3.865493277132266*^9}},ExpressionUUID->"66481e06-9583-44de-b1aa-\
f532c7420d9d"],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   RowBox[{
    TemplateBox[{"V", "\[CapitalOmega]", "A"},
     "Subsuperscript"], "\[CenterDot]", 
    RowBox[{
     TemplateBox[{"\[Tau]", "\[CapitalOmega]", "A"},
      "Subsuperscript"], "\[Transpose]"}]}], ">=", "0"}]],ExpressionUUID->
  "2fe3a44c-16c6-4a06-961b-0f35490cbc18"],
 " and ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    TemplateBox[{"V", "\[Xi]", "A"},
     "Subsuperscript"], "\[CenterDot]", 
    RowBox[{
     TemplateBox[{"\[Tau]", "\[Xi]", "A"},
      "Subsuperscript"], "\[Transpose]"}]}], ">=", "0"}]],ExpressionUUID->
  "77e5a33d-ef8a-4fb0-a028-987bd3406677"],
 "  and   ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    RowBox[{
     TemplateBox[{"\[Tau]", "\[Xi]", "A"},
      "Subsuperscript"], "\[Transpose]"}], "\[CenterDot]", 
    TemplateBox[{"\[Tau]", "\[CapitalOmega]", "A"},
     "Subsuperscript"]}], "=", 
   RowBox[{
    RowBox[{
     SubscriptBox["I", "\[Xi]"], "\[Transpose]"}], "\[CenterDot]", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"r", "\[Times]", 
       RowBox[{"DepMap", "\[Transpose]"}]}], "+", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"1", "-", "r"}], ")"}], "\[Times]", "IdentityMap"}]}], ")"}], 
    "\[CenterDot]", 
    SubscriptBox["I", "\[CapitalOmega]"]}]}]],ExpressionUUID->
  "1a7d008c-24a1-4e36-b0a1-40fca51e96fc"]
}], "Subitem",
 CellChangeTimes->{{3.865490747445092*^9, 3.865490873604588*^9}, {
  3.8654909767556047`*^9, 3.8654910872965326`*^9}, {3.8654911317330494`*^9, 
  3.865491133325512*^9}, {3.865491349692706*^9, 3.865491369418964*^9}, {
  3.8654914533056955`*^9, 3.865491472577675*^9}, {3.865492131504867*^9, 
  3.8654921645448575`*^9}, {3.865492397868293*^9, 3.8654925327423553`*^9}, {
  3.8654928981579437`*^9, 3.8654929199340696`*^9}, {3.865493180278265*^9, 
  3.865493184533617*^9}, {3.865493226467978*^9, 3.865493321182929*^9}, {
  3.8654933635662823`*^9, 3.8654934496568522`*^9}, {3.865493501367443*^9, 
  3.8654935013724427`*^9}},ExpressionUUID->"ba598b30-98fd-4cbe-bb02-\
59567de99dbb"],

Cell[TextData[{
 "Normalization:  ",
 Cell[BoxData[
  RowBox[{
   TemplateBox[{"\[Tau]", "\[Xi]", "A"},
    "Subsuperscript"], "\[CenterDot]", 
   TemplateBox[{
     SuperscriptBox["u", "A"]},
    "Ket"]}]],ExpressionUUID->"7fd148fa-1225-4876-953c-79bc0ca93ba1"],
 " yields an inner product of 1 on every ",
 Cell[BoxData[
  RowBox[{
   TemplateBox[{
     SuperscriptBox["s", "NC"]},
    "Bra"], "\[Element]", 
   RowBox[{
    TemplateBox[{"V", "\[CapitalOmega]", "A"},
     "Subsuperscript"], "\[CenterDot]", 
    RowBox[{
     TemplateBox[{"\[Tau]", "\[CapitalOmega]", "A"},
      "Subsuperscript"], "\[Transpose]"}]}]}]],ExpressionUUID->
  "c2c64b42-a70c-45a3-9772-3b64d2ec9bf5"]
}], "Subitem",
 CellChangeTimes->{{3.865490747445092*^9, 3.865490873604588*^9}, {
  3.8654909767556047`*^9, 3.8654910872965326`*^9}, {3.8654911317330494`*^9, 
  3.865491133325512*^9}, {3.865491349692706*^9, 3.865491369418964*^9}, {
  3.8654914533056955`*^9, 3.865491472577675*^9}, {3.865492131504867*^9, 
  3.8654921645448575`*^9}, {3.865492397868293*^9, 3.8654925327423553`*^9}, {
  3.8654928981579437`*^9, 3.8654929199340696`*^9}, {3.865493180278265*^9, 
  3.865493184533617*^9}, {3.865493226467978*^9, 3.865493321182929*^9}, {
  3.8654933635662823`*^9, 3.8654934496568522`*^9}, {3.86549350258101*^9, 
  3.865493547052122*^9}, {3.865493751354251*^9, 3.865493770312453*^9}, {
  3.865493800543441*^9, 
  3.865493800550439*^9}},ExpressionUUID->"9f7f1cab-14fd-407e-974f-\
a090dc0d795f"],

Cell[TextData[{
 StyleBox["Reconstruct NC model",
  FontColor->RGBColor[1, 0, 0]],
 " such that for all ",
 Cell[BoxData[
  RowBox[{
   TemplateBox[{
     SuperscriptBox["s", "F"]},
    "Ket"], "\[Element]", 
   RowBox[{
    TemplateBox[{"V", "\[CapitalOmega]", "F"},
     "Subsuperscript"], "\[Transpose]"}]}]],ExpressionUUID->
  "329ded5a-5af6-4cf4-b311-1a99556addf0"],
 " and ",
 Cell[BoxData[
  RowBox[{
   TemplateBox[{
     SuperscriptBox["e", "F"]},
    "Ket"], "\[Element]", 
   RowBox[{
    TemplateBox[{"V", "\[Xi]", "F"},
     "Subsuperscript"], "\[Transpose]"}]}]],ExpressionUUID->
  "e271e77a-420c-46f8-a906-9b6e73e2be43"],
 "  we find associated ontological states and effects  ",
 Cell[BoxData[
  RowBox[{
   TemplateBox[{
     SuperscriptBox["s", "NC"]},
    "Ket"], "\[Element]", 
   RowBox[{
    TemplateBox[{"V", "\[CapitalOmega]", "NC"},
     "Subsuperscript"], "\[Transpose]"}]}]],ExpressionUUID->
  "f15e1061-45f3-4e48-9206-a1ff9ba86d18"],
 " and ",
 Cell[BoxData[
  RowBox[{
   TemplateBox[{
     SuperscriptBox["e", "NC"]},
    "Ket"], "\[Element]", 
   RowBox[{
    TemplateBox[{"V", "\[Xi]", "NC"},
     "Subsuperscript"], "\[Transpose]"}]}]],ExpressionUUID->
  "ed788090-789e-4ad1-aedc-8468b8692286"],
 "\[LineSeparator]such that ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    TemplateBox[{
      SuperscriptBox["s", "F"]},
     "Bra"], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"r", "\[Times]", 
       RowBox[{"DepMap", "\[Transpose]"}]}], "+", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"1", "-", "r"}], ")"}], "\[Times]", "IdentityMap"}]}], ")"}], 
    TemplateBox[{
      SuperscriptBox["e", "F"]},
     "Ket"]}], "=", 
   TemplateBox[{
     SuperscriptBox["s", "NC"], 
     SuperscriptBox["e", "NC"]},
    "BraKet"]}]],ExpressionUUID->"58af64ca-488b-4180-916c-272e86daa2e7"]
}], "Subitem",
 CellChangeTimes->{{3.865490747445092*^9, 3.865490873604588*^9}, {
  3.8654909767556047`*^9, 3.8654910872965326`*^9}, {3.8654911317330494`*^9, 
  3.865491133325512*^9}, {3.865491349692706*^9, 3.865491369418964*^9}, {
  3.8654914533056955`*^9, 3.865491472577675*^9}, {3.865492131504867*^9, 
  3.8654921645448575`*^9}, {3.865492397868293*^9, 3.8654925327423553`*^9}, {
  3.8654928981579437`*^9, 3.8654929199340696`*^9}, {3.865493180278265*^9, 
  3.865493184533617*^9}, {3.865493226467978*^9, 3.865493321182929*^9}, {
  3.8654933635662823`*^9, 3.8654934496568522`*^9}, {3.86549350258101*^9, 
  3.865493547052122*^9}, {3.865493751354251*^9, 3.865493770312453*^9}, {
  3.8654938017883444`*^9, 
  3.865494046797858*^9}},ExpressionUUID->"f70bfd1c-61b6-4b34-b551-\
03fe955663a2"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Main Code", "Subsubsection",
 CellChangeTimes->{{3.8520407597064786`*^9, 3.8520407722904377`*^9}, {
  3.857820606498135*^9, 
  3.8578206082190332`*^9}},ExpressionUUID->"c5d2a4d3-cd48-4d57-ab2b-\
4a600ec661d7"],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"DepolarizingMap", "[", 
    RowBox[{"mms$F_", ",", "u$F_"}], "]"}], ":=", 
   RowBox[{"With", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"uKet", "=", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"Flatten", "[", "u$F", "]"}], "}"}], "\[Transpose]"}]}], 
       ",", 
       RowBox[{"mmsBra", "=", 
        RowBox[{"{", 
         RowBox[{"Flatten", "[", "mms$F", "]"}], "}"}]}]}], "}"}], ",", 
     RowBox[{"uKet", ".", "mmsBra"}]}], "]"}]}], "\n", "\n", 
  RowBox[{"(*", 
   RowBox[{
   "From", " ", "Formulation", " ", "1", " ", "to", " ", "Formulation", " ", 
    "2"}], "*)"}]}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"ImplicitUnitEffect", ",", "ImplicitMaximallyMixedState"}], 
      "}"}], ",", "\n", 
     RowBox[{
      RowBox[{"DiscoverEmbedding", "[", 
       RowBox[{"\[CapitalOmega]$F_", ",", "\[Xi]$F_", ",", 
        RowBox[{"u$F_", ":", "ImplicitUnitEffect"}], ",", 
        RowBox[{"mms$F_", ":", "ImplicitMaximallyMixedState"}]}], "]"}], ":=",
       "\n", 
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"ArrayDepth", "[", "\[CapitalOmega]$F", "]"}], "==", "3"}], 
        ",", "\n", 
        RowBox[{
         RowBox[{"Echo", "[", 
          RowBox[{"Style", "[", 
           RowBox[{
           "\"\<Stage 0: Construncting GPT from Quantum Epistriction.\>\"", 
            ",", "Bold"}], "]"}], "]"}], ";", "\n", "\n", 
         RowBox[{"Module", "[", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{
             RowBox[{"\[CapitalOmega]$FGPT", "=", 
              RowBox[{
              "\[CapitalOmega]$F", "//", "ConvertMatrixListToVectorList"}]}], 
             ",", "\n", 
             RowBox[{"\[Xi]$FGPT", "=", 
              RowBox[{"\[Xi]$F", "//", "ConvertMatrixListToVectorList"}]}], 
             ",", "\n", 
             RowBox[{"dQ", "=", 
              RowBox[{
               RowBox[{"\[CapitalOmega]$F", "//", "Dimensions"}], "//", 
               "Last"}]}]}], "}"}], ",", "\n", 
           RowBox[{"With", "[", 
            RowBox[{
             RowBox[{"{", 
              RowBox[{
               RowBox[{"u$FGPT", "=", 
                RowBox[{"UnitEffectVector", "[", "dQ", "]"}]}], ",", "\n", 
               RowBox[{"mms$FGTP", "=", 
                RowBox[{"MaximallyMixedVector", "[", "dQ", "]"}]}]}], "}"}], 
             ",", "\n", 
             RowBox[{
              RowBox[{"Echo", "[", 
               RowBox[{"Row", "[", 
                RowBox[{
                 RowBox[{"{", 
                  RowBox[{
                   RowBox[{"Row", "[", 
                    RowBox[{"{", 
                    RowBox[{
                    "\"\<\!\(\*TemplateBox[{\"V\", \"\[CapitalOmega]\", \
\"F\"},\"Subsuperscript\"]\)=\>\"", ",", 
                    RowBox[{
                    "MatrixForm", "[", "\[CapitalOmega]$FGPT", "]"}]}], "}"}],
                     "]"}], ",", 
                   RowBox[{"Row", "[", 
                    RowBox[{"{", 
                    RowBox[{
                    "\"\<\!\(\*TemplateBox[{\"V\", \"\[Xi]\", \
\"F\"},\"Subsuperscript\"]\)=\>\"", ",", 
                    RowBox[{"MatrixForm", "[", "\[Xi]$FGPT", "]"}]}], "}"}], 
                    "]"}], ",", 
                   RowBox[{"Row", "[", 
                    RowBox[{"{", 
                    RowBox[{"\"\<\!\(\*TemplateBox[{
SuperscriptBox[\"u\", \"F\"]},\"Ket\"]\)=\>\"", ",", 
                    RowBox[{"MatrixForm", "[", 
                    RowBox[{
                    RowBox[{"{", "u$FGPT", "}"}], "\[Transpose]"}], "]"}]}], 
                    "}"}], "]"}]}], "}"}], ",", "\"\<, \>\""}], "]"}], "]"}], 
              ";", "\n", 
              RowBox[{"DiscoverEmbedding", "[", "\n", 
               RowBox[{
               "\[CapitalOmega]$FGPT", ",", "\n", "\[Xi]$FGPT", ",", "\n", 
                "u$FGPT", ",", "\n", "mms$FGTP"}], "]"}]}]}], "]"}]}], 
          "]"}]}], ",", "\n", 
        RowBox[{"With", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"d\[CapitalOmega]", "=", 
             RowBox[{
              RowBox[{"Dimensions", "[", "\[CapitalOmega]$F", "]"}], 
              "\[LeftDoubleBracket]", 
              RowBox[{"-", "1"}], "\[RightDoubleBracket]"}]}], ",", 
            RowBox[{"d\[Xi]", "=", 
             RowBox[{
              RowBox[{"Dimensions", "[", "\[Xi]$F", "]"}], 
              "\[LeftDoubleBracket]", 
              RowBox[{"-", "1"}], "\[RightDoubleBracket]"}]}]}], "}"}], ",", 
          "\n", 
          RowBox[{
           RowBox[{"Echo", "[", 
            RowBox[{"Style", "[", 
             RowBox[{
             "\"\<Stage 1: Constructing an accessible GPT fragment.\>\"", ",",
               "Bold"}], "]"}], "]"}], ";", "\[IndentingNewLine]", "\t", 
           RowBox[{"If", "[", 
            RowBox[{
             RowBox[{"d\[CapitalOmega]", "==", "d\[Xi]"}], ",", "\n", "\t\t", 
             RowBox[{"Block", "[", 
              RowBox[{
               RowBox[{"{", "\n", "\t\t", 
                RowBox[{
                 RowBox[{"ImplicitMaximallyMixedState", "=", 
                  RowBox[{"Mean", "[", "\[CapitalOmega]$F", "]"}]}], ",", 
                 "\n", "\t\t", 
                 RowBox[{"ImplicitUnitEffect", "=", 
                  RowBox[{"LinearSolve", "[", 
                   RowBox[{"\[CapitalOmega]$F", ",", 
                    RowBox[{"ConstantArray", "[", 
                    RowBox[{"1", ",", 
                    RowBox[{"Length", "[", "\[CapitalOmega]$F", "]"}]}], 
                    "]"}]}], "]"}]}]}], "}"}], ",", "\n", "\t\t", 
               RowBox[{"Module", "[", 
                RowBox[{
                 RowBox[{"{", 
                  RowBox[{
                  "\[CapitalOmega]$A", ",", "\[Xi]$A", ",", "u$AKet", ",", 
                   "Inc\[CapitalOmega]", ",", "Inc\[Xi]", ",", 
                   "Proj\[CapitalOmega]", ",", "Proj\[Xi]", ",", 
                   RowBox[{"DepMap", "=", 
                    RowBox[{"DepolarizingMap", "[", 
                    RowBox[{"mms$F", ",", "u$F"}], "]"}]}]}], "}"}], ",", 
                 "\n", "\t\t\t", 
                 RowBox[{
                  RowBox[{
                   RowBox[{"{", 
                    RowBox[{
                    "\[CapitalOmega]$A", ",", "Inc\[CapitalOmega]", ",", 
                    "Proj\[CapitalOmega]"}], "}"}], "=", 
                   RowBox[{
                    RowBox[{"NewMandMap2", "[", "\[CapitalOmega]$F", "]"}], 
                    "\[LeftDoubleBracket]", 
                    RowBox[{"{", 
                    RowBox[{"1", ",", "2", ",", "3"}], "}"}], 
                    "\[RightDoubleBracket]"}]}], ";", "\n", "\t\t\t", 
                  RowBox[{
                   RowBox[{"{", 
                    RowBox[{"\[Xi]$A", ",", "Inc\[Xi]", ",", "Proj\[Xi]"}], 
                    "}"}], "=", 
                   RowBox[{
                    RowBox[{"NewMandMap2", "[", "\[Xi]$F", "]"}], 
                    "\[LeftDoubleBracket]", 
                    RowBox[{"{", 
                    RowBox[{"1", ",", "2", ",", "3"}], "}"}], 
                    "\[RightDoubleBracket]"}]}], ";", "\n", "\t\t\t", 
                  RowBox[{"u$AKet", "=", 
                   RowBox[{
                    RowBox[{"Proj\[Xi]", "\[ConjugateTranspose]"}], ".", 
                    RowBox[{
                    RowBox[{"{", 
                    RowBox[{"Flatten", "[", "u$F", "]"}], "}"}], 
                    "\[Transpose]"}]}]}], ";", "\n", "\t\t\t", "\n", "\t\t", 
                  RowBox[{"Echo", "[", 
                   RowBox[{"Row", "[", 
                    RowBox[{
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"Row", "[", 
                    RowBox[{"{", 
                    RowBox[{
                    "\"\<\!\(\*SubscriptBox[\(Proj\), \(\[CapitalOmega]\)]\)=\
\>\"", ",", 
                    RowBox[{"MatrixForm", "[", "Proj\[CapitalOmega]", "]"}]}],
                     "}"}], "]"}], ",", 
                    RowBox[{"Row", "[", 
                    RowBox[{"{", 
                    RowBox[{
                    "\"\<\!\(\*SubscriptBox[\(Proj\), \(\[Xi]\)]\)=\>\"", ",", 
                    RowBox[{"MatrixForm", "[", "Proj\[Xi]", "]"}]}], "}"}], 
                    "]"}]}], "}"}], ",", "\"\<, \>\""}], "]"}], "]"}], ";", 
                  "\n", "\t\t", 
                  RowBox[{"Echo", "[", 
                   RowBox[{"Row", "[", 
                    RowBox[{
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"Row", "[", 
                    RowBox[{"{", 
                    RowBox[{
                    "\"\<\!\(\*SubscriptBox[\(Inc\), \
\(\[CapitalOmega]\)]\)=\>\"", ",", 
                    RowBox[{"MatrixForm", "[", "Inc\[CapitalOmega]", "]"}]}], 
                    "}"}], "]"}], ",", 
                    RowBox[{"Row", "[", 
                    RowBox[{"{", 
                    RowBox[{
                    "\"\<\!\(\*SubscriptBox[\(Inc\), \(\[Xi]\)]\)=\>\"", ",", 
                    RowBox[{"MatrixForm", "[", "Inc\[Xi]", "]"}]}], "}"}], 
                    "]"}]}], "}"}], ",", "\"\<, \>\""}], "]"}], "]"}], ";", 
                  "\n", "\t\t", 
                  RowBox[{"Echo", "[", 
                   RowBox[{"Row", "[", 
                    RowBox[{
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"Row", "[", 
                    RowBox[{"{", 
                    RowBox[{
                    "\"\<\!\(\*TemplateBox[{\"V\", \"\[CapitalOmega]\", \
\"A\"},\"Subsuperscript\"]\)=\>\"", ",", 
                    RowBox[{"MatrixForm", "[", "\[CapitalOmega]$A", "]"}]}], 
                    "}"}], "]"}], ",", 
                    RowBox[{"Row", "[", 
                    RowBox[{"{", 
                    RowBox[{
                    "\"\<\!\(\*TemplateBox[{\"V\", \"\[Xi]\", \
\"A\"},\"Subsuperscript\"]\)=\>\"", ",", 
                    RowBox[{"MatrixForm", "[", "\[Xi]$A", "]"}]}], "}"}], 
                    "]"}], ",", 
                    RowBox[{"Row", "[", 
                    RowBox[{"{", 
                    RowBox[{"\"\<\!\(\*TemplateBox[{
SuperscriptBox[\"u\", \"A\"]},\"Ket\"]\)=\>\"", ",", 
                    RowBox[{"MatrixForm", "[", "u$AKet", "]"}]}], "}"}], 
                    "]"}]}], "}"}], ",", "\"\<, \>\""}], "]"}], "]"}], ";", 
                  "\n", "\t\t", 
                  RowBox[{"Echo", "[", 
                   RowBox[{"Row", "[", 
                    RowBox[{"{", 
                    RowBox[{"\"\<DepolarizingMap=\>\"", ",", 
                    RowBox[{"MatrixForm", "[", "DepMap", "]"}]}], "}"}], 
                    "]"}], "]"}], ";", "\[IndentingNewLine]", "\t\t", 
                  RowBox[{"InternalDiscoverEmbedding", "[", 
                   RowBox[{
                   "\[CapitalOmega]$A", ",", "\[Xi]$A", ",", "u$AKet", ",", 
                    "Inc\[Xi]", ",", "Inc\[CapitalOmega]", ",", "DepMap"}], 
                   "]"}]}]}], "]"}]}], "]"}], ",", "\[IndentingNewLine]", 
             "\t\t", 
             RowBox[{
              RowBox[{
              "Print", "[", 
               "\"\<Error: Dimension mismatch, and no inner product matrix \
specified.\>\"", "]"}], ";", 
              RowBox[{"Return", "[", "False", "]"}]}]}], 
            "\[IndentingNewLine]", "]"}]}]}], "]"}]}], "]"}]}]}], "]"}], 
   ";"}], "\n", "\n", 
  RowBox[{"(*", 
   RowBox[{
   "From", " ", "Formulation", " ", "2", " ", "to", " ", "Formulation", " ", 
    "3"}], "*)"}]}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"InternalDiscoverEmbedding", "[", 
     RowBox[{
     "\[CapitalOmega]$A_", ",", "\[Xi]$A_", ",", "u$AKet_", ",", "Inc\[Xi]_", 
      ",", "Inc\[CapitalOmega]_", ",", "DepMap_"}], "]"}], ":=", "\n", 
    RowBox[{"Module", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
       "H\[CapitalOmega]", ",", "H\[Xi]", ",", "B$id", ",", "B$identity", ",",
         "B$depolarizing", ",", "r", ",", "\[Iota]", ",", "\[Kappa]", ",", 
        "quasi\[CapitalOmega]", ",", "quasi\[Xi]"}], "}"}], ",", "\n", 
      RowBox[{
       RowBox[{"Echo", "[", 
        RowBox[{"Style", "[", 
         RowBox[{
         "\"\<Stage 2: Finding the facets of the GPT state and effect \
hypercones.\>\"", ",", "Bold"}], "]"}], "]"}], ";", "\n", 
       RowBox[{"H\[CapitalOmega]", "=", 
        RowBox[{"SortBy", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"ConvexHullRays$CDD", "[", 
            RowBox[{
            "RealMatrixToIntegerMatrix", "[", "\[CapitalOmega]$A", "]"}], 
            "]"}], "//", "Normal"}], ",", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"Sort", "@*", "Minus", "@*", "Abs"}], ",", "Abs", ",", 
            "Minus"}], "}"}]}], "]"}]}], ";", "\[IndentingNewLine]", 
       RowBox[{"H\[Xi]", "=", 
        RowBox[{"SortBy", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"ConvexHullRays$CDD", "[", 
            RowBox[{"RealMatrixToIntegerMatrix", "[", "\[Xi]$A", "]"}], "]"}],
            "//", "Normal"}], ",", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"Sort", "@*", "Minus", "@*", "Abs"}], ",", "Abs", ",", 
            "Minus"}], "}"}]}], "]"}]}], ";", "\n", 
       RowBox[{"Echo", "[", 
        RowBox[{"Row", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"Row", "[", 
             RowBox[{"{", 
              RowBox[{
              "\"\<\!\(\*SubscriptBox[\(H\), \(\[CapitalOmega]\)]\)=\>\"", 
               ",", 
               RowBox[{"MatrixForm", "[", "H\[CapitalOmega]", "]"}]}], "}"}], 
             "]"}], ",", 
            RowBox[{"Row", "[", 
             RowBox[{"{", 
              RowBox[{"\"\<\!\(\*SubscriptBox[\(H\), \(\[Xi]\)]\)=\>\"", ",", 
               RowBox[{"MatrixForm", "[", "H\[Xi]", "]"}]}], "}"}], "]"}]}], 
           "}"}], ",", "\"\<, \>\""}], "]"}], "]"}], ";", "\n", 
       RowBox[{"B$identity", "=", 
        RowBox[{
         RowBox[{"Inc\[Xi]", "\[ConjugateTranspose]"}], ".", 
         "Inc\[CapitalOmega]"}]}], ";", "\n", 
       RowBox[{"B$depolarizing", "=", 
        RowBox[{
         RowBox[{"Inc\[Xi]", "\[ConjugateTranspose]"}], ".", 
         RowBox[{"DepMap", "\[Transpose]"}], ".", "Inc\[CapitalOmega]"}]}], 
       ";", "\n", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"r", ",", 
          RowBox[{"{", 
           RowBox[{"\[Iota]", ",", "\[Kappa]"}], "}"}]}], "}"}], "=", 
        RowBox[{"DiscoverEmbeddingLP", "[", 
         RowBox[{
         "H\[CapitalOmega]", ",", "H\[Xi]", ",", "u$AKet", ",", "B$identity", 
          ",", "B$depolarizing"}], "]"}]}], ";", "\n", 
       RowBox[{"quasi\[CapitalOmega]", "=", 
        RowBox[{"\[CapitalOmega]$A", ".", 
         RowBox[{"\[Iota]", "\[Transpose]"}]}]}], ";", "\n", 
       RowBox[{"quasi\[Xi]", "=", 
        RowBox[{"\[Xi]$A", ".", 
         RowBox[{"\[Kappa]", "\[Transpose]"}]}]}], ";", "\n", 
       RowBox[{"Echo", "[", 
        RowBox[{"Row", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"Row", "[", 
             RowBox[{"{", 
              RowBox[{"\"\<DepolarizedNCModel[\[CapitalOmega]]=\>\"", ",", 
               RowBox[{"MatrixForm", "[", "quasi\[CapitalOmega]", "]"}]}], 
              "}"}], "]"}], ",", 
            RowBox[{"Row", "[", 
             RowBox[{"{", 
              RowBox[{"\"\<DepolarizedNCModel[\[Xi]]=\>\"", ",", 
               RowBox[{"MatrixForm", "[", "quasi\[Xi]", "]"}]}], "}"}], 
             "]"}]}], "}"}], ",", "\"\<, \>\""}], "]"}], "]"}], ";", "\n", 
       RowBox[{"{", 
        RowBox[{"r", ",", 
         RowBox[{"{", 
          RowBox[{"quasi\[CapitalOmega]", ",", "quasi\[Xi]"}], "}"}]}], 
        "}"}]}]}], "\n", "]"}]}], ";"}], "\n", "\n", 
  RowBox[{"(*", 
   RowBox[{
   "From", " ", "Formulation", " ", "2", " ", "to", " ", "Formulation", " ", 
    "4"}], "*)"}]}], "\n", 
 RowBox[{
  RowBox[{"DiscoverEmbeddingLP", "[", 
   RowBox[{"H\[CapitalOmega]_", ",", "H\[Xi]_", ",", 
    RowBox[{"u$effectiveKet_", ":", "0"}], ",", "B$identity_", ",", 
    "B$depolarizing_"}], "]"}], ":=", "\n", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
     "\[Sigma]Symbolic", ",", "lpposvars", ",", "HeldLP", ",", "r", ",", 
      "\[Sigma]", ",", "B"}], "}"}], ",", "\n", 
    RowBox[{
     RowBox[{"Echo", "[", 
      RowBox[{"Style", "[", 
       RowBox[{
       "\"\<Stage 3: Finding the Robustness of Nonembeddability to \
Depolarizing Noise.\>\"", ",", "Bold"}], "]"}], "]"}], ";", "\n", 
     RowBox[{"\[Sigma]Symbolic", "=", 
      RowBox[{"Array", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"Indexed", "[", 
          RowBox[{"\[FormalS]", ",", 
           RowBox[{"{", "##", "}"}]}], "]"}], "&"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Length", "[", "H\[Xi]", "]"}], ",", 
          RowBox[{"Length", "[", "H\[CapitalOmega]", "]"}]}], "}"}]}], 
       "]"}]}], ";", "\n", 
     RowBox[{"lpposvars", "=", 
      RowBox[{"Append", "[", 
       RowBox[{
        RowBox[{"Flatten", "[", "\[Sigma]Symbolic", "]"}], ",", 
        "\[FormalR]"}], "]"}]}], ";", "\n", 
     RowBox[{"HeldLP", "=", 
      RowBox[{
       RowBox[{"Hold", "[", "LinearOptimization", "]"}], "[", 
       "\[IndentingNewLine]", 
       RowBox[{"\[FormalR]", ",", "\[IndentingNewLine]", 
        RowBox[{"Join", "[", "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"Thread", "[", 
           RowBox[{
            RowBox[{"Flatten", "[", 
             RowBox[{
              RowBox[{
               RowBox[{"H\[Xi]", "\[Transpose]"}], ".", "\[Sigma]Symbolic", 
               ".", "H\[CapitalOmega]"}], "-", 
              RowBox[{"(", 
               RowBox[{
                RowBox[{
                 RowBox[{"(", 
                  RowBox[{"1", "-", "\[FormalR]"}], ")"}], "*", 
                 "B$identity"}], "+", 
                RowBox[{
                 RowBox[{"(", "\[FormalR]", ")"}], "B$depolarizing"}]}], 
               ")"}]}], "]"}], "==", "0"}], "]"}], ",", "\[IndentingNewLine]", 
          RowBox[{"Thread", "[", 
           RowBox[{"lpposvars", ">=", "0"}], "]"}]}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"Join", "[", "lpposvars", "]"}], ",", 
        "\"\<PrimalMinimizerRules\>\"", ",", "\[IndentingNewLine]", 
        RowBox[{"Method", "\[Rule]", "\n", 
         RowBox[{"If", "[", 
          RowBox[{
           RowBox[{"And", "[", 
            RowBox[{
             RowBox[{"MatrixQ", "[", 
              RowBox[{"B$depolarizing", ",", "IntegerQ"}], "]"}], ",", "\n", 
             RowBox[{"MatrixQ", "[", 
              RowBox[{"B$identity", ",", "IntegerQ"}], "]"}]}], "]"}], ",", 
           "\n", "\"\<Simplex\>\"", ",", "\"\<CLP\>\""}], "]"}]}]}], "]"}]}], 
     ";", "\n", "\n", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"r", ",", "\[Sigma]"}], "}"}], "=", 
      RowBox[{"Rationalize", "[", 
       RowBox[{"Chop", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"{", 
           RowBox[{"\[FormalR]", ",", "\[Sigma]Symbolic"}], "}"}], "/.", 
          RowBox[{"ReleaseHold", "[", "HeldLP", "]"}]}], ",", 
         RowBox[{"10", "^", 
          RowBox[{"-", "8"}]}]}], "]"}], "]"}]}], ";", "\n", 
     RowBox[{"B", "=", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         RowBox[{"(", 
          RowBox[{"1", "-", "r"}], ")"}], "*", "B$identity"}], "+", 
        RowBox[{
         RowBox[{"(", "r", ")"}], "B$depolarizing"}]}], ")"}]}], ";", "\n", 
     RowBox[{"Echo", "[", 
      RowBox[{"Row", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Row", "[", 
           RowBox[{"{", 
            RowBox[{"\"\<\[Sigma]=\>\"", ",", 
             RowBox[{"MatrixForm", "[", "\[Sigma]", "]"}]}], "}"}], "]"}], 
          ",", 
          RowBox[{"Row", "[", 
           RowBox[{"{", 
            RowBox[{"\"\<r=\>\"", ",", "r"}], "}"}], "]"}], ",", " ", 
          RowBox[{"If", "[", 
           RowBox[{
            RowBox[{"r", ">", "0"}], ",", 
            RowBox[{"Row", "[", 
             RowBox[{"{", 
              RowBox[{
              "\"\<\!\(\*TemplateBox[{\"B\", \"embeddable\", \
\"A\"},\"Subsuperscript\"]\)=\>\"", ",", 
               RowBox[{"MatrixForm", "[", "B", "]"}]}], "}"}], "]"}], ",", 
            "Nothing"}], "]"}]}], "}"}], ",", "\"\<, \>\""}], "]"}], "]"}], 
     ";", "\n", 
     RowBox[{"{", 
      RowBox[{"r", ",", 
       RowBox[{"RescaleEmbedding", "[", 
        RowBox[{
        "H\[CapitalOmega]", ",", "H\[Xi]", ",", "\[Sigma]", ",", 
         "u$effectiveKet"}], "]"}]}], "}"}]}]}], "\n", "]"}]}]}], "Code",
 CellChangeTimes->{{3.851685027242819*^9, 3.8516850521127114`*^9}, {
   3.8516851777203555`*^9, 3.8516853948473077`*^9}, {3.851686020455269*^9, 
   3.8516860281682386`*^9}, {3.8516877926804404`*^9, 3.851687901000943*^9}, {
   3.851687935982847*^9, 3.851688163047453*^9}, {3.8516882096470647`*^9, 
   3.8516882359294543`*^9}, {3.8516883095039206`*^9, 3.851688323071621*^9}, {
   3.8516887925935755`*^9, 3.851688831725793*^9}, {3.851688926796523*^9, 
   3.851688932724241*^9}, {3.8516890319415684`*^9, 3.851689059437289*^9}, {
   3.8516892322919946`*^9, 3.8516893887516713`*^9}, {3.851689530933708*^9, 
   3.8516895850185127`*^9}, {3.851689713162363*^9, 3.851689725883181*^9}, {
   3.851689783052549*^9, 3.851689862034442*^9}, {3.851689939740202*^9, 
   3.8516899825378695`*^9}, {3.8516903609614525`*^9, 
   3.8516904890809717`*^9}, {3.8516906365612774`*^9, 
   3.8516908040187693`*^9}, {3.8516911598234687`*^9, 3.8516911992636366`*^9}, 
   3.8516912791017046`*^9, 3.8516913371745925`*^9, {3.8516914654219627`*^9, 
   3.851691505613835*^9}, 3.8516915537246346`*^9, {3.8516916187224154`*^9, 
   3.851691753805719*^9}, {3.85169180212906*^9, 3.8516918128453197`*^9}, {
   3.851786207502663*^9, 3.8517862554779825`*^9}, {3.851786889051985*^9, 
   3.8517869306997967`*^9}, {3.8520302160252495`*^9, 
   3.8520302225100355`*^9}, {3.852030646405367*^9, 3.8520307145381346`*^9}, {
   3.852030869141343*^9, 3.852030874983594*^9}, {3.8520360565501184`*^9, 
   3.852036120987689*^9}, {3.852036200876336*^9, 3.852036280634906*^9}, {
   3.852036322917176*^9, 3.852036325645074*^9}, {3.8520363635173817`*^9, 
   3.8520363776592555`*^9}, 3.8520364582600117`*^9, {3.852036643683382*^9, 
   3.8520366738749075`*^9}, {3.852036705186039*^9, 3.85203674365851*^9}, {
   3.8520368220734262`*^9, 3.8520368555463223`*^9}, {3.852036905057602*^9, 
   3.852036905914102*^9}, {3.8520373797124796`*^9, 3.8520374825121202`*^9}, {
   3.852037513384708*^9, 3.8520375289273434`*^9}, {3.852037565624374*^9, 
   3.8520375789917583`*^9}, {3.8520376332640495`*^9, 
   3.8520376561034703`*^9}, {3.8520377162394156`*^9, 
   3.8520377196722784`*^9}, {3.8520377813272815`*^9, 3.852037783967321*^9}, {
   3.8520378318705063`*^9, 3.8520378844152985`*^9}, {3.8520379725348797`*^9, 
   3.8520380929424696`*^9}, {3.8520381263604584`*^9, 
   3.8520381999604125`*^9}, {3.852038242727743*^9, 3.8520383277818975`*^9}, {
   3.8520384271573687`*^9, 3.8520386108620825`*^9}, {3.852038648268362*^9, 
   3.852038729572224*^9}, {3.852038836460517*^9, 3.8520388400284653`*^9}, {
   3.852038887332788*^9, 3.852038891819769*^9}, {3.8520389279552164`*^9, 
   3.8520390362362747`*^9}, {3.852039225155672*^9, 3.852039274642191*^9}, 
   3.852039318498988*^9, {3.8520404766323824`*^9, 3.8520404847036915`*^9}, {
   3.852040593303506*^9, 3.8520406310784903`*^9}, {3.8520407198749027`*^9, 
   3.8520407225978174`*^9}, 3.8520447972421827`*^9, {3.8520448406110363`*^9, 
   3.8520448689978976`*^9}, {3.852045329102182*^9, 3.852045343148321*^9}, {
   3.852047884468964*^9, 3.852047887316987*^9}, {3.855392234529628*^9, 
   3.855392237224928*^9}, {3.8553923013532133`*^9, 3.855392311160785*^9}, {
   3.855392358442201*^9, 3.855392375314316*^9}, {3.855395880669408*^9, 
   3.855395930235879*^9}, {3.8553959657323523`*^9, 3.85539613955562*^9}, {
   3.855396174740655*^9, 3.8553961777086644`*^9}, {3.8560086389914055`*^9, 
   3.8560087072165613`*^9}, {3.8560087402151585`*^9, 
   3.8560087821136613`*^9}, {3.856008822374682*^9, 3.85600883359171*^9}, {
   3.856008912015394*^9, 3.856008934782257*^9}, {3.8560143037490673`*^9, 
   3.856014309804495*^9}, {3.856265845215176*^9, 3.8562658570717077`*^9}, {
   3.8562659048967795`*^9, 3.856266044439046*^9}, {3.8562660889682198`*^9, 
   3.856266110663081*^9}, 3.8562705421610107`*^9, {3.8564451880374155`*^9, 
   3.8564453339176908`*^9}, {3.8578187148817263`*^9, 
   3.8578187154251146`*^9}, {3.8578187720109816`*^9, 3.857818774256955*^9}, {
   3.8578188118335514`*^9, 3.857818840329586*^9}, 3.8578190344566293`*^9, {
   3.857819155127031*^9, 3.857819168991051*^9}, {3.857819252223004*^9, 
   3.8578195231895337`*^9}, {3.857819565109845*^9, 3.8578196348455667`*^9}, {
   3.857819703390459*^9, 3.8578197088711433`*^9}, {3.8578198247019615`*^9, 
   3.8578200720449543`*^9}, {3.857820104757904*^9, 3.857820126669478*^9}, {
   3.857820165524902*^9, 3.857820180015134*^9}, {3.8578202503481927`*^9, 
   3.8578202521563587`*^9}, 3.8578203135115952`*^9, {3.857820355564739*^9, 
   3.8578203837406807`*^9}, 3.857820598244559*^9, {3.8578207893077097`*^9, 
   3.857820798172414*^9}, {3.8578211299480543`*^9, 3.857821151435853*^9}, {
   3.8578212058762646`*^9, 3.8578213816420174`*^9}, {3.8578214565296216`*^9, 
   3.8578215206818495`*^9}, {3.8578216945380874`*^9, 
   3.8578217317372694`*^9}, {3.8578217773316784`*^9, 
   3.8578218296115527`*^9}, {3.857821890689782*^9, 3.8578218910994616`*^9}, {
   3.8578234830455427`*^9, 3.857823546836153*^9}, {3.85806724933897*^9, 
   3.858067250462943*^9}, {3.8580712588536024`*^9, 3.8580713025635324`*^9}, {
   3.858071619746681*^9, 3.8580716316586423`*^9}, {3.8580716735296535`*^9, 
   3.858071756217123*^9}, {3.8580718061117463`*^9, 3.858071862032937*^9}, {
   3.8580718966717405`*^9, 3.8580719282640667`*^9}, {3.858071978625491*^9, 
   3.858071988424292*^9}, {3.8580720258877215`*^9, 3.858072173783414*^9}, {
   3.85807222316016*^9, 3.8580723536791153`*^9}, {3.8580724352565584`*^9, 
   3.858072480662404*^9}, {3.858073457580406*^9, 3.858073537211959*^9}, {
   3.8580735842527037`*^9, 3.858073646259405*^9}, {3.858073778533165*^9, 
   3.858073810716627*^9}, {3.8580739441511836`*^9, 3.8580739823325047`*^9}, {
   3.859020350485526*^9, 3.859020464332153*^9}, {3.8590205388539257`*^9, 
   3.8590206722607546`*^9}, {3.859020839339039*^9, 3.8590209297167053`*^9}, {
   3.859021014321864*^9, 3.8590210332017574`*^9}, {3.8590210839767876`*^9, 
   3.8590211779286203`*^9}, {3.8590212097697024`*^9, 3.859021339177455*^9}, {
   3.859021371167891*^9, 3.859021392768396*^9}, {3.85902143914671*^9, 
   3.859021479329636*^9}, {3.8590216102182417`*^9, 3.8590216628732724`*^9}, {
   3.859021766392772*^9, 3.859021798048918*^9}, {3.85902188911261*^9, 
   3.859021917712829*^9}, {3.859021975096426*^9, 3.8590220071123905`*^9}, {
   3.859023389479328*^9, 3.8590233897669973`*^9}, {3.859023422487667*^9, 
   3.8590234255920258`*^9}, {3.859377978216378*^9, 3.8593779877834444`*^9}, {
   3.8593781294793596`*^9, 3.859378137623133*^9}, {3.859409742269972*^9, 
   3.8594098682049427`*^9}, {3.859549914521356*^9, 3.8595499408660016`*^9}, {
   3.85955006189903*^9, 3.859550106889106*^9}, {3.8595501549058857`*^9, 
   3.859550156235386*^9}, {3.8595502967700896`*^9, 3.8595503002563906`*^9}, {
   3.8595504115600967`*^9, 3.8595505488716264`*^9}, {3.8595505790005827`*^9, 
   3.859550632784728*^9}, {3.8595507245024347`*^9, 3.8595507589270816`*^9}, {
   3.859550822423871*^9, 3.859550850951764*^9}, {3.859550985735213*^9, 
   3.8595510044383507`*^9}, {3.859551182033825*^9, 3.859551414873742*^9}, {
   3.8595695791857405`*^9, 3.8595695821357193`*^9}, {3.8599346954366508`*^9, 
   3.859934720427026*^9}, 3.8599352175106516`*^9, {3.8599352682162104`*^9, 
   3.85993550512816*^9}, {3.8599355673509784`*^9, 3.8599356369908113`*^9}, {
   3.859935796391165*^9, 3.8599357987988997`*^9}, {3.8654906198870564`*^9, 
   3.865490646845976*^9}, 3.865492790435232*^9, {3.8654954565677223`*^9, 
   3.865495470149846*^9}, {3.866902635837757*^9, 3.86690268667719*^9}, {
   3.8669027170471087`*^9, 3.866902720502009*^9}, {3.866902813372343*^9, 
   3.866902826781226*^9}},
 CellLabel->"In[68]:=",ExpressionUUID->"e9cd84a4-60a1-48cc-af30-a6509b2aa33d"]
}, Closed]],

Cell[CellGroupData[{

Cell["From Simplicial-Cone Embedding to Simplex Embedding", "Subsubsection",
 CellChangeTimes->{{3.8520407597064786`*^9, 3.8520407635226927`*^9}, {
  3.8578202819639473`*^9, 3.8578203001397963`*^9}, {3.8595506975415945`*^9, 
  3.8595506982763977`*^9}},ExpressionUUID->"74cf22c0-cebc-4688-a884-\
758007aa1a73"],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"RescaleEmbedding", "[", 
    RowBox[{
    "H\[CapitalOmega]_", ",", "H\[Xi]_", ",", "\[Sigma]_", ",", 
     "u$effectiveKet_"}], "]"}], ":=", "\n", 
   RowBox[{"With", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"raw\[Iota]", "=", "H\[CapitalOmega]"}], ",", 
       RowBox[{"raw\[Kappa]", "=", 
        RowBox[{
         RowBox[{"\[Sigma]", "\[Transpose]"}], ".", "H\[Xi]"}]}]}], "}"}], 
     ",", "\n", 
     RowBox[{
      RowBox[{"Echo", "[", 
       RowBox[{"Style", "[", 
        RowBox[{
        "\"\<Stage 4: Constructing a Simplex Embedding from a Simplicial-Cone \
Embedding.\>\"", ",", "Bold"}], "]"}], "]"}], ";", "\[IndentingNewLine]", 
      RowBox[{"With", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"rescalingvec", "=", 
          RowBox[{"Flatten", "[", 
           RowBox[{"raw\[Kappa]", ".", "u$effectiveKet"}], "]"}]}], "}"}], 
        ",", "\n", 
        RowBox[{"With", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            RowBox[{"subspace", "=", 
             RowBox[{"Flatten", "[", 
              RowBox[{"Position", "[", 
               RowBox[{
                RowBox[{"Unitize", "[", "rescalingvec", "]"}], ",", "1"}], 
               "]"}], "]"}]}], ",", "\n", "\t", 
            RowBox[{"embeddingdimension", "=", 
             RowBox[{"Length", "[", "rescalingvec", "]"}]}]}], "}"}], ",", 
          "\n", 
          RowBox[{"With", "[", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{
              RowBox[{"averagerescaling", "=", 
               RowBox[{"Mean", "[", 
                RowBox[{
                "rescalingvec", "\[LeftDoubleBracket]", "subspace", 
                 "\[RightDoubleBracket]"}], "]"}]}], ",", "\n", "\t", 
              RowBox[{"rescalingmat", "=", 
               RowBox[{
                RowBox[{"DiagonalMatrix", "[", "rescalingvec", "]"}], 
                "\[LeftDoubleBracket]", 
                RowBox[{"All", ",", "All"}], "\[RightDoubleBracket]"}]}]}], 
             "}"}], ",", "\n", 
            RowBox[{"Module", "[", 
             RowBox[{
              RowBox[{"{", 
               RowBox[{"invertiblerescalingvec", "=", 
                RowBox[{"ConstantArray", "[", 
                 RowBox[{"averagerescaling", ",", "embeddingdimension"}], 
                 "]"}]}], "}"}], ",", "\n", "\t", 
              RowBox[{
               RowBox[{
                RowBox[{
                "invertiblerescalingvec", "\[LeftDoubleBracket]", "subspace", 
                 "\[RightDoubleBracket]"}], "=", 
                RowBox[{
                "rescalingvec", "\[LeftDoubleBracket]", "subspace", 
                 "\[RightDoubleBracket]"}]}], ";", "\n", 
               RowBox[{"With", "[", 
                RowBox[{
                 RowBox[{"{", 
                  RowBox[{
                   RowBox[{"\[Iota]", "=", 
                    RowBox[{"Transpose", "[", 
                    RowBox[{
                    RowBox[{"raw\[Iota]", "\[Transpose]"}], ".", 
                    "rescalingmat"}], "]"}]}], ",", " ", 
                   RowBox[{"\[Kappa]", "=", 
                    RowBox[{
                    RowBox[{"PseudoInverse", "[", "rescalingmat", "]"}], ".", 
                    "raw\[Kappa]"}]}]}], "}"}], ",", "\n", 
                 RowBox[{
                  RowBox[{"Echo", "[", 
                   RowBox[{"Row", "[", 
                    RowBox[{
                    RowBox[{"{", 
                    RowBox[{
                    RowBox[{"Row", "[", 
                    RowBox[{"{", 
                    RowBox[{
                    "\"\<\!\(\*TemplateBox[{\"\[Tau]\", \"\[CapitalOmega]\", \
\"A\"},\"Subsuperscript\"]\)=\>\"", ",", 
                    RowBox[{"MatrixForm", "[", "\[Iota]", "]"}]}], "}"}], 
                    "]"}], ",", 
                    RowBox[{"Row", "[", 
                    RowBox[{"{", 
                    RowBox[{
                    
                    "\"\<\!\(\*TemplateBox[{\"\[Tau]\", \"\[Xi]\", \
\"A\"},\"Subsuperscript\"]\)=\>\"", ",", 
                    RowBox[{"MatrixForm", "[", "\[Kappa]", "]"}]}], "}"}], 
                    "]"}]}], "}"}], ",", "\"\<, \>\""}], "]"}], "]"}], ";", 
                  "\n", 
                  RowBox[{"Return", "[", 
                   RowBox[{"{", 
                    RowBox[{"\[Iota]", ",", "\[Kappa]"}], "}"}], "]"}], 
                  ";"}]}], "\n", "]"}]}]}], "]"}]}], "]"}]}], "]"}]}], 
       "]"}]}]}], "]"}]}], "\[IndentingNewLine]"}]], "Code",
 CellChangeTimes->{{3.8520408096211257`*^9, 3.8520408154674587`*^9}, {
   3.852040877682639*^9, 3.8520409378427057`*^9}, {3.852041139202009*^9, 
   3.852041144698636*^9}, {3.852041212033887*^9, 3.852041282578456*^9}, {
   3.8520413239947042`*^9, 3.852041333882022*^9}, {3.8520415645223517`*^9, 
   3.8520415662414637`*^9}, {3.852045400294651*^9, 3.852045419736435*^9}, {
   3.8535931300191154`*^9, 3.853593162346599*^9}, {3.8535936200676937`*^9, 
   3.8535936244186263`*^9}, {3.857820315076685*^9, 3.8578203160275817`*^9}, {
   3.8578203691326046`*^9, 3.8578205950759335`*^9}, 3.857820967603745*^9, {
   3.857821576562649*^9, 3.8578216037059364`*^9}, {3.859377819192107*^9, 
   3.859377947865396*^9}, 3.8593779995763893`*^9, {3.8593780805199175`*^9, 
   3.8593780982634506`*^9}, {3.8594098857891827`*^9, 
   3.8594099137638273`*^9}, {3.859550252079241*^9, 3.859550258807335*^9}, 
   3.859550719431184*^9, {3.865493646400417*^9, 3.8654936468067756`*^9}},
 CellLabel->"In[24]:=",ExpressionUUID->"a892f1fc-a62c-47fb-b653-7cf1298ed7e4"]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Manuscript Examples:", "Subsection",
 CellChangeTimes->{{3.851685419117911*^9, 3.851685437578148*^9}, {
  3.851685470990306*^9, 3.8516854782867985`*^9}, {3.8516855301907043`*^9, 
  3.851685548589467*^9}, {3.851685865521597*^9, 3.8516858923817043`*^9}, {
  3.859279253960359*^9, 
  3.8592792568876286`*^9}},ExpressionUUID->"97fb38f9-3784-484b-820c-\
1bb55037754c"],

Cell[CellGroupData[{

Cell["Boxworld / Gbit (Example #3 in text)", "Subsubsection",
 CellChangeTimes->{{3.8516760170260105`*^9, 3.8516760293322372`*^9}, {
  3.8516860826211653`*^9, 3.851686087092563*^9}, {3.859279258855912*^9, 
  3.859279263536249*^9}},ExpressionUUID->"8aa2f22d-1245-4b09-81c5-\
be719ae469e7"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"DiscoverEmbedding", "[", 
   RowBox[{"(*", 
    RowBox[{
     TemplateBox[{"V", "\[CapitalOmega]", "F"},
      "Subsuperscript"], "="}], "*)"}], 
   RowBox[{
    RowBox[{"(", GridBox[{
       {"1", "1", "0"},
       {"1", "0", "1"},
       {"1", 
        RowBox[{"-", "1"}], "0"},
       {"1", "0", 
        RowBox[{"-", "1"}]}
      }], ")"}], ",", 
    RowBox[{"(*", 
     RowBox[{
      TemplateBox[{"V", "\[Xi]", "F"},
       "Subsuperscript"], "="}], "*)"}], 
    RowBox[{
     TagBox[
      RowBox[{"(", "\[NoBreak]", GridBox[{
         {"0.5", 
          RowBox[{"-", "0.5"}], 
          RowBox[{"-", "0.5"}]},
         {"0.5", "0.5", 
          RowBox[{"-", "0.5"}]},
         {"0.5", "0.5", "0.5"},
         {"0.5", 
          RowBox[{"-", "0.5"}], "0.5"}
        },
        GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
        GridBoxSpacings->{"Columns" -> {
            Offset[0.27999999999999997`], {
             Offset[0.7]}, 
            Offset[0.27999999999999997`]}, "Rows" -> {
            Offset[0.2], {
             Offset[0.4]}, 
            Offset[0.2]}}], "\[NoBreak]", ")"}],
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]], "//", "Rationalize"}], ",", 
    RowBox[{"(*", 
     RowBox[{
      RowBox[{"unit", " ", "effect"}], "="}], "*)"}], 
    RowBox[{"{", 
     RowBox[{"1", ",", "0", ",", "0"}], "}"}]}], "]"}], ";"}]], "Input",
 CellChangeTimes->{{3.8516860629335*^9, 3.8516860721578007`*^9}, {
   3.851687626168855*^9, 3.8516876473204665`*^9}, 3.8520311292817597`*^9, 
   3.852031163723178*^9, {3.8553961462918334`*^9, 3.85539614676361*^9}, 
   3.8554018243245935`*^9, 3.8560090007767324`*^9, 3.8578206245567*^9, {
   3.8578206757325206`*^9, 3.8578207022779083`*^9}, {3.858068421160929*^9, 
   3.8580684319295807`*^9}, {3.859279267467314*^9, 3.859279327635626*^9}, 
   3.8599356573908434`*^9, {3.859935714347586*^9, 3.859935717153613*^9}, {
   3.8599376794963913`*^9, 3.8599376949989576`*^9}, {3.865492773370363*^9, 
   3.865492784279158*^9}},
 CellLabel->"In[25]:=",ExpressionUUID->"f16b41d7-b408-42be-a88f-633b207c9144"],

Cell[CellGroupData[{

Cell[BoxData[
 StyleBox["\<\"Stage 1: Constructing an accessible GPT fragment.\"\>",
  StripOnInput->False,
  FontWeight->Bold]], "Echo",
 CellChangeTimes->{
  3.8669017549093523`*^9},ExpressionUUID->"107b2363-5e18-446b-9acf-\
a487a65724be"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{
    "\"\\!\\(\\*SubscriptBox[\\(Proj\\), \\(\[CapitalOmega]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "0", "0"}, {"0", "1", "0"}, {"0", "0", "1"}}, 
         RowSpacings -> 1, ColumnSpacings -> 1, RowAlignments -> Baseline, 
         ColumnAlignments -> Center], "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{"\"\\!\\(\\*SubscriptBox[\\(Proj\\), \\(\[Xi]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "0", "0"}, {"0", "1", "0"}, {"0", "0", "1"}}, 
         RowSpacings -> 1, ColumnSpacings -> 1, RowAlignments -> Baseline, 
         ColumnAlignments -> Center], "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.866901754947352*^9},ExpressionUUID->"0542b29c-ca6a-404b-bfdc-\
532a19def008"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{
    "\"\\!\\(\\*SubscriptBox[\\(Inc\\), \\(\[CapitalOmega]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "0", "0"}, {"0", "1", "0"}, {"0", "0", "1"}}, 
         RowSpacings -> 1, ColumnSpacings -> 1, RowAlignments -> Baseline, 
         ColumnAlignments -> Center], "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{"\"\\!\\(\\*SubscriptBox[\\(Inc\\), \\(\[Xi]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "0", "0"}, {"0", "1", "0"}, {"0", "0", "1"}}, 
         RowSpacings -> 1, ColumnSpacings -> 1, RowAlignments -> Baseline, 
         ColumnAlignments -> Center], "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.866901754976385*^9},ExpressionUUID->"c63b6e89-2083-4312-995c-\
42095547d372"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{\\\"V\\\", \\\"\[CapitalOmega]\\\", \
\\\"A\\\"},\\\"Subsuperscript\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "1", "0"}, {"1", "0", "1"}, {"1", 
           RowBox[{"-", "1"}], "0"}, {"1", "0", 
           RowBox[{"-", "1"}]}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{\\\"V\\\", \\\"\[Xi]\\\", \
\\\"A\\\"},\\\"Subsuperscript\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{
           FractionBox["1", "2"], 
           RowBox[{"-", 
             FractionBox["1", "2"]}], 
           RowBox[{"-", 
             FractionBox["1", "2"]}]}, {
           FractionBox["1", "2"], 
           FractionBox["1", "2"], 
           RowBox[{"-", 
             FractionBox["1", "2"]}]}, {
           FractionBox["1", "2"], 
           FractionBox["1", "2"], 
           FractionBox["1", "2"]}, {
           FractionBox["1", "2"], 
           RowBox[{"-", 
             FractionBox["1", "2"]}], 
           FractionBox["1", "2"]}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{SuperscriptBox[\\\"u\\\", \
\\\"A\\\"]},\\\"Ket\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1"}, {"0"}, {"0"}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.866901755004114*^9},ExpressionUUID->"bdbeaee5-50df-4c40-94d1-\
3f8b54722542"],

Cell[BoxData[
 TemplateBox[{"\"DepolarizingMap=\"", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", 
      GridBox[{{"1", "0", "0"}, {"0", "0", "0"}, {"0", "0", "0"}}, 
       RowSpacings -> 1, ColumnSpacings -> 1, RowAlignments -> Baseline, 
       ColumnAlignments -> Center], "\[NoBreak]", ")"}], 
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]},
  "RowDefault"]], "Echo",
 CellChangeTimes->{
  3.866901755039114*^9},ExpressionUUID->"a16b7ce4-6721-4cfa-a5a2-\
d62cc36d0d40"],

Cell[BoxData[
 StyleBox["\<\"Stage 2: Finding the facets of the GPT state and effect \
hypercones.\"\>",
  StripOnInput->False,
  FontWeight->Bold]], "Echo",
 CellChangeTimes->{
  3.866901755046114*^9},ExpressionUUID->"b9c7d616-2964-4856-bd1f-\
9d4a098cc3ea"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{
    "\"\\!\\(\\*SubscriptBox[\\(H\\), \\(\[CapitalOmega]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "1", "1"}, {"1", "1", 
           RowBox[{"-", "1"}]}, {"1", 
           RowBox[{"-", "1"}], "1"}, {"1", 
           RowBox[{"-", "1"}], 
           RowBox[{"-", "1"}]}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{"\"\\!\\(\\*SubscriptBox[\\(H\\), \\(\[Xi]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "0", "1"}, {"1", "0", 
           RowBox[{"-", "1"}]}, {"1", "1", "0"}, {"1", 
           RowBox[{"-", "1"}], "0"}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.866901755522532*^9},ExpressionUUID->"40a58280-7fb6-485c-9383-\
74c40d716bbd"],

Cell[BoxData[
 StyleBox["\<\"Stage 3: Finding the Robustness of Nonembeddability to \
Depolarizing Noise.\"\>",
  StripOnInput->False,
  FontWeight->Bold]], "Echo",
 CellChangeTimes->{
  3.8669017555535316`*^9},ExpressionUUID->"279b4444-05c3-436c-9b56-\
9386b5e99ce8"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{"\"\[Sigma]=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{
           FractionBox["1", "8"], "0", 
           FractionBox["1", "8"], "0"}, {"0", 
           FractionBox["1", "8"], "0", 
           FractionBox["1", "8"]}, {
           FractionBox["1", "8"], 
           FractionBox["1", "8"], "0", "0"}, {"0", "0", 
           FractionBox["1", "8"], 
           FractionBox["1", "8"]}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{"\"r=\"", 
     FractionBox["1", "2"]}, "RowDefault"], 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{\\\"B\\\", \\\"embeddable\\\", \
\\\"A\\\"},\\\"Subsuperscript\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "0", "0"}, {"0", 
           FractionBox["1", "2"], "0"}, {"0", "0", 
           FractionBox["1", "2"]}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.866901755641718*^9},ExpressionUUID->"e7e9ea27-7e22-438d-ba76-\
89ca8b6b3c09"],

Cell[BoxData[
 StyleBox["\<\"Stage 4: Constructing a Simplex Embedding from a \
Simplicial-Cone Embedding.\"\>",
  StripOnInput->False,
  FontWeight->Bold]], "Echo",
 CellChangeTimes->{
  3.8669017556720724`*^9},ExpressionUUID->"46284eeb-2275-4d6a-a543-\
78bf7fac104e"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{\\\"\[Tau]\\\", \\\"\[CapitalOmega]\\\", \\\"A\\\
\"},\\\"Subsuperscript\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{
           FractionBox["1", "4"], 
           FractionBox["1", "4"], 
           FractionBox["1", "4"]}, {
           FractionBox["1", "4"], 
           FractionBox["1", "4"], 
           RowBox[{"-", 
             FractionBox["1", "4"]}]}, {
           FractionBox["1", "4"], 
           RowBox[{"-", 
             FractionBox["1", "4"]}], 
           FractionBox["1", "4"]}, {
           FractionBox["1", "4"], 
           RowBox[{"-", 
             FractionBox["1", "4"]}], 
           RowBox[{"-", 
             FractionBox["1", "4"]}]}}, RowSpacings -> 1, ColumnSpacings -> 1,
          RowAlignments -> Baseline, ColumnAlignments -> Center], 
        "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{\\\"\[Tau]\\\", \\\"\[Xi]\\\", \
\\\"A\\\"},\\\"Subsuperscript\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", 
           FractionBox["1", "2"], 
           FractionBox["1", "2"]}, {"1", 
           FractionBox["1", "2"], 
           RowBox[{"-", 
             FractionBox["1", "2"]}]}, {"1", 
           RowBox[{"-", 
             FractionBox["1", "2"]}], 
           FractionBox["1", "2"]}, {"1", 
           RowBox[{"-", 
             FractionBox["1", "2"]}], 
           RowBox[{"-", 
             FractionBox["1", "2"]}]}}, RowSpacings -> 1, ColumnSpacings -> 1,
          RowAlignments -> Baseline, ColumnAlignments -> Center], 
        "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.8669017557035804`*^9},ExpressionUUID->"1c96d6c6-60eb-4ac7-a1f7-\
1a8509411346"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{"\"DepolarizedNCModel[\[CapitalOmega]]=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{
           FractionBox["1", "2"], 
           FractionBox["1", "2"], "0", "0"}, {
           FractionBox["1", "2"], "0", 
           FractionBox["1", "2"], "0"}, {"0", "0", 
           FractionBox["1", "2"], 
           FractionBox["1", "2"]}, {"0", 
           FractionBox["1", "2"], "0", 
           FractionBox["1", "2"]}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{"\"DepolarizedNCModel[\[Xi]]=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"0", 
           FractionBox["1", "2"], 
           FractionBox["1", "2"], "1"}, {
           FractionBox["1", "2"], "1", "0", 
           FractionBox["1", "2"]}, {"1", 
           FractionBox["1", "2"], 
           FractionBox["1", "2"], "0"}, {
           FractionBox["1", "2"], "0", "1", 
           FractionBox["1", "2"]}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.8669017557419925`*^9},ExpressionUUID->"ba8e633f-0236-46d7-885b-\
abe38d220436"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Example #2 in text", "Subsubsection",
 CellChangeTimes->{{3.8516760170260105`*^9, 3.8516760293322372`*^9}, {
  3.8516860826211653`*^9, 3.851686087092563*^9}, {3.859279258855912*^9, 
  3.859279263536249*^9}, {3.8592793664709287`*^9, 
  3.8592793701830215`*^9}},ExpressionUUID->"3c4a4526-ad0e-4758-84ec-\
f4b445c3f524"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"DiscoverEmbedding", "[", 
   RowBox[{"(*", 
    RowBox[{
     TemplateBox[{"V", "\[CapitalOmega]", "F"},
      "Subsuperscript"], "="}], "*)"}], 
   RowBox[{
    RowBox[{"(", GridBox[{
       {"1", "0", "0", "0"},
       {"0", "1", "0", "0"},
       {"0", "0", "1", "0"},
       {"0", "0", "0", "1"}
      }], ")"}], ",", 
    RowBox[{"(*", 
     RowBox[{
      TemplateBox[{"V", "\[Xi]", "F"},
       "Subsuperscript"], "="}], "*)"}], 
    TagBox[
     RowBox[{"(", "\[NoBreak]", GridBox[{
        {"1", "1", "0", "0"},
        {"0", "1", "1", "0"},
        {"0", "0", "1", "1"},
        {"1", "0", "0", "1"}
       },
       GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
       GridBoxSpacings->{"Columns" -> {
           Offset[0.27999999999999997`], {
            Offset[0.7]}, 
           Offset[0.27999999999999997`]}, "Rows" -> {
           Offset[0.2], {
            Offset[0.4]}, 
           Offset[0.2]}}], "\[NoBreak]", ")"}],
     Function[BoxForm`e$, 
      MatrixForm[BoxForm`e$]]], ",", 
    RowBox[{"(*", 
     RowBox[{
      RowBox[{"unit", " ", "effect"}], "="}], "*)"}], 
    RowBox[{"{", GridBox[{
       {"1", "1", "1", "1"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "}"}]}], "]"}], ";"}]], "Input",
 CellChangeTimes->{{3.859279375111204*^9, 3.8592794589683003`*^9}, 
   3.8599356977014513`*^9, {3.8654912555315986`*^9, 3.8654912720621896`*^9}, {
   3.8654927122936707`*^9, 3.8654927451508226`*^9}},
 CellLabel->"In[44]:=",ExpressionUUID->"e16f26f2-b4eb-4bcf-aaeb-69c197d2986d"],

Cell[CellGroupData[{

Cell[BoxData[
 StyleBox["\<\"Stage 1: Constructing an accessible GPT fragment.\"\>",
  StripOnInput->False,
  FontWeight->Bold]], "Echo",
 CellChangeTimes->{
  3.8654927979512835`*^9},ExpressionUUID->"15495224-2a80-487a-bdfa-\
c2f3e263efa5"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{
    "\"\\!\\(\\*SubscriptBox[\\(Proj\\), \\(\[CapitalOmega]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "0", "0", "0"}, {"0", "1", "0", "0"}, {
          "0", "0", "1", "0"}, {"0", "0", "0", "1"}}, RowSpacings -> 1, 
         ColumnSpacings -> 1, RowAlignments -> Baseline, ColumnAlignments -> 
         Center], "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{"\"\\!\\(\\*SubscriptBox[\\(Proj\\), \\(\[Xi]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{
           FractionBox["3", "4"], 
           FractionBox["1", "4"], 
           RowBox[{"-", 
             FractionBox["1", "4"]}]}, {
           FractionBox["1", "4"], 
           FractionBox["3", "4"], 
           FractionBox["1", "4"]}, {
           RowBox[{"-", 
             FractionBox["1", "4"]}], 
           FractionBox["1", "4"], 
           FractionBox["3", "4"]}, {
           FractionBox["1", "4"], 
           RowBox[{"-", 
             FractionBox["1", "4"]}], 
           FractionBox["1", "4"]}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.8654927979714055`*^9},ExpressionUUID->"0204440c-39ec-43ef-a48e-\
dff542e87843"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{
    "\"\\!\\(\\*SubscriptBox[\\(Inc\\), \\(\[CapitalOmega]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "0", "0", "0"}, {"0", "1", "0", "0"}, {
          "0", "0", "1", "0"}, {"0", "0", "0", "1"}}, RowSpacings -> 1, 
         ColumnSpacings -> 1, RowAlignments -> Baseline, ColumnAlignments -> 
         Center], "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{"\"\\!\\(\\*SubscriptBox[\\(Inc\\), \\(\[Xi]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "0", "0"}, {"0", "1", "0"}, {"0", "0", "1"}, {"1", 
           RowBox[{"-", "1"}], "1"}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.865492798003473*^9},ExpressionUUID->"69905599-de13-4f63-afda-\
3d5e39ac4288"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{\\\"V\\\", \\\"\[CapitalOmega]\\\", \
\\\"A\\\"},\\\"Subsuperscript\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "0", "0", "0"}, {"0", "1", "0", "0"}, {
          "0", "0", "1", "0"}, {"0", "0", "0", "1"}}, RowSpacings -> 1, 
         ColumnSpacings -> 1, RowAlignments -> Baseline, ColumnAlignments -> 
         Center], "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{\\\"V\\\", \\\"\[Xi]\\\", \
\\\"A\\\"},\\\"Subsuperscript\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "1", "0"}, {"0", "1", "1"}, {"0", "0", "1"}, {
          "1", "0", "0"}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{SuperscriptBox[\\\"u\\\", \
\\\"A\\\"]},\\\"Ket\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1"}, {"1"}, {"1"}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.86549279801474*^9},ExpressionUUID->"bfa85604-abb8-4eb8-8d43-be966474b1e5"],

Cell[BoxData[
 TemplateBox[{"\"DepolarizingMap=\"", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", 
      GridBox[{{
         FractionBox["1", "4"], 
         FractionBox["1", "4"], 
         FractionBox["1", "4"], 
         FractionBox["1", "4"]}, {
         FractionBox["1", "4"], 
         FractionBox["1", "4"], 
         FractionBox["1", "4"], 
         FractionBox["1", "4"]}, {
         FractionBox["1", "4"], 
         FractionBox["1", "4"], 
         FractionBox["1", "4"], 
         FractionBox["1", "4"]}, {
         FractionBox["1", "4"], 
         FractionBox["1", "4"], 
         FractionBox["1", "4"], 
         FractionBox["1", "4"]}}, RowSpacings -> 1, ColumnSpacings -> 1, 
       RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]", 
      ")"}], 
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]},
  "RowDefault"]], "Echo",
 CellChangeTimes->{
  3.8654927980449724`*^9},ExpressionUUID->"a8bbe2ec-c1a5-4fdf-9640-\
1adcd1b69473"],

Cell[BoxData[
 StyleBox["\<\"Stage 2: Finding the facets of the GPT state and effect \
hypercones.\"\>",
  StripOnInput->False,
  FontWeight->Bold]], "Echo",
 CellChangeTimes->{
  3.8654927980715036`*^9},ExpressionUUID->"840d72f9-c547-403a-9431-\
80b285279dfc"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{
    "\"\\!\\(\\*SubscriptBox[\\(H\\), \\(\[CapitalOmega]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"0", "0", "0", "1"}, {"0", "0", "1", "0"}, {
          "0", "1", "0", "0"}, {"1", "0", "0", "0"}}, RowSpacings -> 1, 
         ColumnSpacings -> 1, RowAlignments -> Baseline, ColumnAlignments -> 
         Center], "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{"\"\\!\\(\\*SubscriptBox[\\(H\\), \\(\[Xi]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", 
           RowBox[{"-", "1"}], "1"}, {"0", "0", "1"}, {"0", "1", "0"}, {
          "1", "0", "0"}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.865492798188737*^9},ExpressionUUID->"535e6b09-c1da-477b-a83a-\
5a1697422df9"],

Cell[BoxData[
 StyleBox["\<\"Stage 3: Finding the Robustness of Nonembeddability to \
Depolarizing Noise.\"\>",
  StripOnInput->False,
  FontWeight->Bold]], "Echo",
 CellChangeTimes->{
  3.8654927982144613`*^9},ExpressionUUID->"93d1a33c-af2f-4041-92be-\
cb29bb2930c2"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{"\"\[Sigma]=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "0", "0", "0"}, {"0", "1", "0", "0"}, {
          "0", "0", "1", "0"}, {"0", "0", "0", "1"}}, RowSpacings -> 1, 
         ColumnSpacings -> 1, RowAlignments -> Baseline, ColumnAlignments -> 
         Center], "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{"\"r=\"", "0"}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.8654927982386556`*^9},ExpressionUUID->"f93ea8c9-c7df-4f90-8e5b-\
8969acf0945e"],

Cell[BoxData[
 StyleBox["\<\"Stage 4: Constructing a Simplex Embedding from a \
Simplicial-Cone Embedding.\"\>",
  StripOnInput->False,
  FontWeight->Bold]], "Echo",
 CellChangeTimes->{
  3.8654927982506638`*^9},ExpressionUUID->"5e696a28-fa77-4e46-aaf3-\
fdcfde39f711"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{\\\"\[Tau]\\\", \\\"\[CapitalOmega]\\\", \\\"A\\\
\"},\\\"Subsuperscript\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"0", "0", "0", "1"}, {"0", "0", "1", "0"}, {
          "0", "1", "0", "0"}, {"1", "0", "0", "0"}}, RowSpacings -> 1, 
         ColumnSpacings -> 1, RowAlignments -> Baseline, ColumnAlignments -> 
         Center], "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{\\\"\[Tau]\\\", \\\"\[Xi]\\\", \
\\\"A\\\"},\\\"Subsuperscript\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", 
           RowBox[{"-", "1"}], "1"}, {"0", "0", "1"}, {"0", "1", "0"}, {
          "1", "0", "0"}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.8654927982727547`*^9},ExpressionUUID->"654885e7-d940-4f7b-8b1b-\
8de9a877301f"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{"\"DepolarizedNCModel[\[CapitalOmega]]=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"0", "0", "0", "1"}, {"0", "0", "1", "0"}, {
          "0", "1", "0", "0"}, {"1", "0", "0", "0"}}, RowSpacings -> 1, 
         ColumnSpacings -> 1, RowAlignments -> Baseline, ColumnAlignments -> 
         Center], "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{"\"DepolarizedNCModel[\[Xi]]=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"0", "0", "1", "1"}, {"0", "1", "1", "0"}, {
          "1", "1", "0", "0"}, {"1", "0", "0", "1"}}, RowSpacings -> 1, 
         ColumnSpacings -> 1, RowAlignments -> Baseline, ColumnAlignments -> 
         Center], "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.8654927983007417`*^9},ExpressionUUID->"681e009a-b8ef-414d-8209-\
23fee8ea1943"]
}, Open  ]]
}, Open  ]],

Cell["", "PageBreak",
 PageBreakBelow->True,ExpressionUUID->"16253e6b-f67e-4cbe-af55-e4c25e1174c8"]
}, Closed]],

Cell[CellGroupData[{

Cell["Example #1 in text", "Subsubsection",
 CellChangeTimes->{{3.8516760170260105`*^9, 3.8516760293322372`*^9}, {
  3.8516860826211653`*^9, 3.851686087092563*^9}, {3.859279258855912*^9, 
  3.859279263536249*^9}, {3.8592793664709287`*^9, 3.8592793701830215`*^9}, {
  3.859377727486562*^9, 3.859377731309953*^9}, {3.859378152972926*^9, 
  3.859378153796678*^9}, {3.859378676722635*^9, 3.8593786978994007`*^9}, {
  3.8595511076441946`*^9, 
  3.859551109059647*^9}},ExpressionUUID->"697bbcb7-b5f0-4792-8f5f-\
338d2e4caa49"],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"Clear", "[", "Ket", "]"}], ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"(", 
     RowBox[{"Ket", "/@", 
      RowBox[{"{", 
       RowBox[{"\"\<0\>\"", ",", "\"\<1\>\""}], "}"}]}], ")"}], "]"}], "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"Transpose", "@*", "List", "@*", "Normalize"}], ")"}], "/@", 
    RowBox[{"Eigenvectors", "[", 
     RowBox[{"PauliMatrix", "[", "3", "]"}], "]"}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"(", 
     RowBox[{"Ket", "/@", 
      RowBox[{"{", 
       RowBox[{"\"\<+\>\"", ",", "\"\<-\>\""}], "}"}]}], ")"}], "]"}], "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"Transpose", "@*", "List", "@*", "Normalize"}], ")"}], "/@", 
    RowBox[{"Eigenvectors", "[", 
     RowBox[{"PauliMatrix", "[", "2", "]"}], "]"}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   TemplateBox[{
     RowBox[{"Except", "[", 
       RowBox[{"_Integer", ",", "\[Psi]_"}], "]"}]},
    "Bra"], ":=", 
   RowBox[{
    TemplateBox[{"\[Psi]"},
     "Ket"], "\[ConjugateTranspose]"}]}], ";"}], "\[IndentingNewLine]", 
 RowBox[{"MatrixForm", "/@", 
  RowBox[{"(", 
   RowBox[{"RebitProjectors", "=", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{
       TemplateBox[{"s"},
        "Ket"], ".", 
       TemplateBox[{"s"},
        "Bra"]}], ",", 
      RowBox[{"{", 
       RowBox[{"s", ",", 
        RowBox[{"{", 
         RowBox[{
         "\"\<0\>\"", ",", "\"\<1\>\"", ",", "\"\<+\>\"", ",", "\"\<-\>\""}], 
         "}"}]}], "}"}]}], "]"}]}], ")"}]}]}], "Input",
 CellChangeTimes->{{3.859551080739842*^9, 3.859551129973133*^9}, {
  3.8654943118220243`*^9, 3.8654943358672495`*^9}},
 CellLabel->"In[72]:=",ExpressionUUID->"bcbac566-c006-444b-b4b1-229762bdaa47"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"0", "0"},
       {"0", "1"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1", "0"},
       {"0", "0"}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {
        FractionBox["1", "2"], 
        FractionBox["\[ImaginaryI]", "2"]},
       {
        RowBox[{"-", 
         FractionBox["\[ImaginaryI]", "2"]}], 
        FractionBox["1", "2"]}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]], ",", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {
        FractionBox["1", "2"], 
        RowBox[{"-", 
         FractionBox["\[ImaginaryI]", "2"]}]},
       {
        FractionBox["\[ImaginaryI]", "2"], 
        FractionBox["1", "2"]}
      },
      GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
      GridBoxSpacings->{"Columns" -> {
          Offset[0.27999999999999997`], {
           Offset[0.7]}, 
          Offset[0.27999999999999997`]}, "Rows" -> {
          Offset[0.2], {
           Offset[0.4]}, 
          Offset[0.2]}}], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]}], "}"}]], "Output",
 CellChangeTimes->{
  3.8595511002961693`*^9, 3.8595511308544827`*^9, 3.859935729111212*^9, 
   3.865490471039726*^9, {3.8654943129792366`*^9, 3.86549434551108*^9}},
 CellLabel->"Out[76]=",ExpressionUUID->"b2b88efa-1557-44aa-8f25-e274a0082bb8"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"DiscoverEmbedding", "[", 
   RowBox[{"(*", 
    RowBox[{
     TemplateBox[{"V", "\[CapitalOmega]", "Q"},
      "Subsuperscript"], "="}], "*)"}], 
   RowBox[{"RebitProjectors", ",", 
    RowBox[{"(*", 
     RowBox[{
      TemplateBox[{"V", "\[Xi]", "Q"},
       "Subsuperscript"], "="}], "*)"}], "RebitProjectors"}], "\[NoBreak]", 
   "]"}], ";"}]], "Input",
 CellChangeTimes->{{3.8593774363075137`*^9, 3.85937744977532*^9}, {
  3.8593776034996443`*^9, 3.8593776049674196`*^9}, {3.859377640712638*^9, 
  3.859377698774996*^9}, {3.859377768163028*^9, 3.8593777783285627`*^9}, {
  3.8593787248116617`*^9, 3.859378732196188*^9}, {3.859551139479594*^9, 
  3.8595511539881153`*^9}, {3.8599357350017967`*^9, 3.8599357356177216`*^9}, {
  3.8654943981186795`*^9, 3.8654944137105017`*^9}},
 CellLabel->"In[78]:=",ExpressionUUID->"1699be7d-510b-4f88-81fe-c1c8178c4a9d"],

Cell[CellGroupData[{

Cell[BoxData[
 StyleBox["\<\"Stage 0: Construncting GPT from Quantum Epistriction.\"\>",
  StripOnInput->False,
  FontWeight->Bold]], "Echo",
 CellChangeTimes->{
  3.865496038804189*^9},ExpressionUUID->"4293c3f0-a331-4f53-8e1d-\
a6ea9b8c672f"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{\\\"V\\\", \\\"\[CapitalOmega]\\\", \
\\\"F\\\"},\\\"Subsuperscript\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{
           FractionBox["1", 
            SqrtBox["2"]], "0", "0", 
           RowBox[{"-", 
             FractionBox["1", 
              SqrtBox["2"]]}]}, {
           FractionBox["1", 
            SqrtBox["2"]], "0", "0", 
           FractionBox["1", 
            SqrtBox["2"]]}, {
           FractionBox["1", 
            SqrtBox["2"]], "0", 
           RowBox[{"-", 
             FractionBox["1", 
              SqrtBox["2"]]}], "0"}, {
           FractionBox["1", 
            SqrtBox["2"]], "0", 
           FractionBox["1", 
            SqrtBox["2"]], "0"}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{\\\"V\\\", \\\"\[Xi]\\\", \
\\\"F\\\"},\\\"Subsuperscript\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{
           FractionBox["1", 
            SqrtBox["2"]], "0", "0", 
           RowBox[{"-", 
             FractionBox["1", 
              SqrtBox["2"]]}]}, {
           FractionBox["1", 
            SqrtBox["2"]], "0", "0", 
           FractionBox["1", 
            SqrtBox["2"]]}, {
           FractionBox["1", 
            SqrtBox["2"]], "0", 
           RowBox[{"-", 
             FractionBox["1", 
              SqrtBox["2"]]}], "0"}, {
           FractionBox["1", 
            SqrtBox["2"]], "0", 
           FractionBox["1", 
            SqrtBox["2"]], "0"}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{SuperscriptBox[\\\"u\\\", \
\\\"F\\\"]},\\\"Ket\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{
           SqrtBox["2"]}, {"0"}, {"0"}, {"0"}}, RowSpacings -> 1, 
         ColumnSpacings -> 1, RowAlignments -> Baseline, ColumnAlignments -> 
         Center], "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.865496038839061*^9},ExpressionUUID->"c595c5c5-dfef-4fdf-9a65-\
9e57782d5fd2"],

Cell[BoxData[
 StyleBox["\<\"Stage 1: Constructing an accessible GPT fragment.\"\>",
  StripOnInput->False,
  FontWeight->Bold]], "Echo",
 CellChangeTimes->{
  3.865496038906377*^9},ExpressionUUID->"31d4ccca-c866-43a4-a6a9-\
386248974e55"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{
    "\"\\!\\(\\*SubscriptBox[\\(Proj\\), \\(\[CapitalOmega]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "0", "0"}, {"0", "0", "0"}, {"0", "1", "0"}, {
          "0", "0", "1"}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{"\"\\!\\(\\*SubscriptBox[\\(Proj\\), \\(\[Xi]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "0", "0"}, {"0", "0", "0"}, {"0", "1", "0"}, {
          "0", "0", "1"}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.865496038962572*^9},ExpressionUUID->"f7c9fe0a-75bf-4a93-a63e-\
7d737e47536e"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{
    "\"\\!\\(\\*SubscriptBox[\\(Inc\\), \\(\[CapitalOmega]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "0", "0"}, {"0", "0", "0"}, {"0", "1", "0"}, {
          "0", "0", "1"}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{"\"\\!\\(\\*SubscriptBox[\\(Inc\\), \\(\[Xi]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "0", "0"}, {"0", "0", "0"}, {"0", "1", "0"}, {
          "0", "0", "1"}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.865496039023316*^9},ExpressionUUID->"3416f7da-521a-4dea-946f-\
c2ff0329c0af"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{\\\"V\\\", \\\"\[CapitalOmega]\\\", \
\\\"A\\\"},\\\"Subsuperscript\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{
           FractionBox["1", 
            SqrtBox["2"]], "0", 
           RowBox[{"-", 
             FractionBox["1", 
              SqrtBox["2"]]}]}, {
           FractionBox["1", 
            SqrtBox["2"]], "0", 
           FractionBox["1", 
            SqrtBox["2"]]}, {
           FractionBox["1", 
            SqrtBox["2"]], 
           RowBox[{"-", 
             FractionBox["1", 
              SqrtBox["2"]]}], "0"}, {
           FractionBox["1", 
            SqrtBox["2"]], 
           FractionBox["1", 
            SqrtBox["2"]], "0"}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{\\\"V\\\", \\\"\[Xi]\\\", \
\\\"A\\\"},\\\"Subsuperscript\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{
           FractionBox["1", 
            SqrtBox["2"]], "0", 
           RowBox[{"-", 
             FractionBox["1", 
              SqrtBox["2"]]}]}, {
           FractionBox["1", 
            SqrtBox["2"]], "0", 
           FractionBox["1", 
            SqrtBox["2"]]}, {
           FractionBox["1", 
            SqrtBox["2"]], 
           RowBox[{"-", 
             FractionBox["1", 
              SqrtBox["2"]]}], "0"}, {
           FractionBox["1", 
            SqrtBox["2"]], 
           FractionBox["1", 
            SqrtBox["2"]], "0"}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{SuperscriptBox[\\\"u\\\", \
\\\"A\\\"]},\\\"Ket\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{
           SqrtBox["2"]}, {"0"}, {"0"}}, RowSpacings -> 1, ColumnSpacings -> 
         1, RowAlignments -> Baseline, ColumnAlignments -> Center], 
        "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.8654960391070786`*^9},ExpressionUUID->"47d4fcd7-629d-431f-a475-\
0e37c42151d7"],

Cell[BoxData[
 TemplateBox[{"\"DepolarizingMap=\"", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", 
      GridBox[{{"1", "0", "0", "0"}, {"0", "0", "0", "0"}, {
        "0", "0", "0", "0"}, {"0", "0", "0", "0"}}, RowSpacings -> 1, 
       ColumnSpacings -> 1, RowAlignments -> Baseline, ColumnAlignments -> 
       Center], "\[NoBreak]", ")"}], 
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]},
  "RowDefault"]], "Echo",
 CellChangeTimes->{
  3.8654960392090907`*^9},ExpressionUUID->"340a3586-fc18-4be1-a1d6-\
d10f16616fce"],

Cell[BoxData[
 StyleBox["\<\"Stage 2: Finding the facets of the GPT state and effect \
hypercones.\"\>",
  StripOnInput->False,
  FontWeight->Bold]], "Echo",
 CellChangeTimes->{
  3.8654960392811117`*^9},ExpressionUUID->"bdc22c76-1752-4c6b-a3b7-\
8b3a487ae9dd"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{
    "\"\\!\\(\\*SubscriptBox[\\(H\\), \\(\[CapitalOmega]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "1", "1"}, {"1", "1", 
           RowBox[{"-", "1"}]}, {"1", 
           RowBox[{"-", "1"}], "1"}, {"1", 
           RowBox[{"-", "1"}], 
           RowBox[{"-", "1"}]}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{"\"\\!\\(\\*SubscriptBox[\\(H\\), \\(\[Xi]\\)]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"1", "1", "1"}, {"1", "1", 
           RowBox[{"-", "1"}]}, {"1", 
           RowBox[{"-", "1"}], "1"}, {"1", 
           RowBox[{"-", "1"}], 
           RowBox[{"-", "1"}]}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.865496039577292*^9},ExpressionUUID->"c8ea05bb-ee51-4935-9781-\
ce9c44d560de"],

Cell[BoxData[
 StyleBox["\<\"Stage 3: Finding the Robustness of Nonembeddability to \
Depolarizing Noise.\"\>",
  StripOnInput->False,
  FontWeight->Bold]], "Echo",
 CellChangeTimes->{
  3.8654960396303005`*^9},ExpressionUUID->"c8deaa57-c20c-467e-8db5-\
3ce498a31635"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{"\"\[Sigma]=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{
           FractionBox["1", "4"], "0", "0", "0"}, {"0", 
           FractionBox["1", "4"], "0", "0"}, {"0", "0", 
           FractionBox["1", "4"], "0"}, {"0", "0", "0", 
           FractionBox["1", "4"]}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{"\"r=\"", "0"}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.8654960396907315`*^9},ExpressionUUID->"5d310034-2528-4834-ae0b-\
68479e2614ce"],

Cell[BoxData[
 StyleBox["\<\"Stage 4: Constructing a Simplex Embedding from a \
Simplicial-Cone Embedding.\"\>",
  StripOnInput->False,
  FontWeight->Bold]], "Echo",
 CellChangeTimes->{
  3.865496039750878*^9},ExpressionUUID->"760d47b0-39a1-489d-81d6-\
334b21c327e9"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{\\\"\[Tau]\\\", \\\"\[CapitalOmega]\\\", \\\"A\\\
\"},\\\"Subsuperscript\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{
           FractionBox["1", 
            RowBox[{"2", " ", 
              SqrtBox["2"]}]], 
           FractionBox["1", 
            RowBox[{"2", " ", 
              SqrtBox["2"]}]], 
           FractionBox["1", 
            RowBox[{"2", " ", 
              SqrtBox["2"]}]]}, {
           FractionBox["1", 
            RowBox[{"2", " ", 
              SqrtBox["2"]}]], 
           FractionBox["1", 
            RowBox[{"2", " ", 
              SqrtBox["2"]}]], 
           RowBox[{"-", 
             FractionBox["1", 
              RowBox[{"2", " ", 
                SqrtBox["2"]}]]}]}, {
           FractionBox["1", 
            RowBox[{"2", " ", 
              SqrtBox["2"]}]], 
           RowBox[{"-", 
             FractionBox["1", 
              RowBox[{"2", " ", 
                SqrtBox["2"]}]]}], 
           FractionBox["1", 
            RowBox[{"2", " ", 
              SqrtBox["2"]}]]}, {
           FractionBox["1", 
            RowBox[{"2", " ", 
              SqrtBox["2"]}]], 
           RowBox[{"-", 
             FractionBox["1", 
              RowBox[{"2", " ", 
                SqrtBox["2"]}]]}], 
           RowBox[{"-", 
             FractionBox["1", 
              RowBox[{"2", " ", 
                SqrtBox["2"]}]]}]}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{
    "\"\\!\\(\\*TemplateBox[{\\\"\[Tau]\\\", \\\"\[Xi]\\\", \
\\\"A\\\"},\\\"Subsuperscript\\\"]\\)=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{
           FractionBox["1", 
            SqrtBox["2"]], 
           FractionBox["1", 
            SqrtBox["2"]], 
           FractionBox["1", 
            SqrtBox["2"]]}, {
           FractionBox["1", 
            SqrtBox["2"]], 
           FractionBox["1", 
            SqrtBox["2"]], 
           RowBox[{"-", 
             FractionBox["1", 
              SqrtBox["2"]]}]}, {
           FractionBox["1", 
            SqrtBox["2"]], 
           RowBox[{"-", 
             FractionBox["1", 
              SqrtBox["2"]]}], 
           FractionBox["1", 
            SqrtBox["2"]]}, {
           FractionBox["1", 
            SqrtBox["2"]], 
           RowBox[{"-", 
             FractionBox["1", 
              SqrtBox["2"]]}], 
           RowBox[{"-", 
             FractionBox["1", 
              SqrtBox["2"]]}]}}, RowSpacings -> 1, ColumnSpacings -> 1, 
         RowAlignments -> Baseline, ColumnAlignments -> Center], "\[NoBreak]",
         ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.865496039818553*^9},ExpressionUUID->"12a91a10-1331-4345-89eb-\
8294d5840c9f"],

Cell[BoxData[
 TemplateBox[{", ", "\", \"", 
   TemplateBox[{"\"DepolarizedNCModel[\[CapitalOmega]]=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"0", 
           FractionBox["1", "2"], "0", 
           FractionBox["1", "2"]}, {
           FractionBox["1", "2"], "0", 
           FractionBox["1", "2"], "0"}, {"0", "0", 
           FractionBox["1", "2"], 
           FractionBox["1", "2"]}, {
           FractionBox["1", "2"], 
           FractionBox["1", "2"], "0", "0"}}, RowSpacings -> 1, 
         ColumnSpacings -> 1, RowAlignments -> Baseline, ColumnAlignments -> 
         Center], "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"], 
   TemplateBox[{"\"DepolarizedNCModel[\[Xi]]=\"", 
     TagBox[
      RowBox[{"(", "\[NoBreak]", 
        GridBox[{{"0", "1", "0", "1"}, {"1", "0", "1", "0"}, {
          "0", "0", "1", "1"}, {"1", "1", "0", "0"}}, RowSpacings -> 1, 
         ColumnSpacings -> 1, RowAlignments -> Baseline, ColumnAlignments -> 
         Center], "\[NoBreak]", ")"}], 
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]]}, "RowDefault"]},
  "RowWithSeparators"]], "Echo",
 CellChangeTimes->{
  3.8654960398933525`*^9},ExpressionUUID->"f71d9c80-ceec-47c1-a6a8-\
f8e84e2171a6"]
}, Open  ]]
}, Open  ]],

Cell["", "PageBreak",
 PageBreakBelow->True,ExpressionUUID->"54d0e004-1bc6-4e65-aaf2-6100249b0846"]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell["See Also:", "Subsection",
 CellChangeTimes->{{3.851685419117911*^9, 3.851685437578148*^9}, {
  3.851685470990306*^9, 3.8516854782867985`*^9}, {3.8516855301907043`*^9, 
  3.851685548589467*^9}, {3.851685865521597*^9, 3.8516858923817043`*^9}, {
  3.859279253960359*^9, 3.8592792568876286`*^9}, {3.865494761224945*^9, 
  3.8654947657525926`*^9}},ExpressionUUID->"2f730ccd-7488-4a45-b980-\
6b1f82ac468e"],

Cell["\<\
Main article about this code on the arXiv at https://arxiv.org/abs/2204.11905\
\>", "Item",
 CellChangeTimes->{{3.865494767568616*^9, 
  3.865494784637094*^9}},ExpressionUUID->"d44a0f46-4b70-413d-b46c-\
15868d79368a"],

Cell["\<\
Special thanks to Mathew Weiss who has made a lot of progress porting this to \
Python.\[LineSeparator]https://github.com/heyredhat/pySimplexEmbedding\
\>", "Item",
 CellChangeTimes->{{3.8654946929706974`*^9, 
  3.865494740363988*^9}},ExpressionUUID->"06705ce5-539f-45f6-88bb-\
b3bf11a44e19"]
}, Closed]]
},
WindowSize->{900., 1377.75},
WindowMargins->{{Automatic, -2346}, {-387.75, Automatic}},
FrontEndVersion->"13.0 for Microsoft Windows (64-bit) (December 2, 2021)",
StyleDefinitions->"Default.nb",
ExpressionUUID->"37a1e262-0f6c-4a90-8360-ca0d95ff3b57"
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[558, 20, 197, 3, 28, "Input",ExpressionUUID->"412f3c07-f6da-4e15-b3cb-2cae69dd58e2"],
Cell[758, 25, 2016, 51, 156, "Code",ExpressionUUID->"ab8f2682-73c3-4db1-9a2c-8af8f121607d"],
Cell[CellGroupData[{
Cell[2799, 80, 367, 6, 54, "Subsection",ExpressionUUID->"2a368a57-ec22-4798-abc4-592fc1c28196"],
Cell[CellGroupData[{
Cell[3191, 90, 258, 4, 45, "Subsubsection",ExpressionUUID->"1e2e38ee-3bbf-48b4-8cf9-7453fd1feacb"],
Cell[3452, 96, 5362, 149, 366, "Code",ExpressionUUID->"0cfc615b-2bf0-419c-b196-eb4e7142d926"],
Cell[8817, 247, 1262, 33, 121, "Code",ExpressionUUID->"fb96447f-72f5-47f7-9d73-290a3d98ca71"],
Cell[10082, 282, 891, 23, 94, "Code",ExpressionUUID->"e4919f2e-da76-49fe-8692-d8803cc9f3ad"]
}, Closed]],
Cell[CellGroupData[{
Cell[11010, 310, 203, 3, 37, "Subsubsection",ExpressionUUID->"9686e296-f094-42c7-9c98-da54c454dac5"],
Cell[CellGroupData[{
Cell[11238, 317, 181, 3, 32, "Item",ExpressionUUID->"5c0a7f13-b477-4387-90f5-10627d2ba81d"],
Cell[11422, 322, 1680, 37, 173, "Code",ExpressionUUID->"b2d9a55c-552e-41c7-b02b-c5f6e57853ea"],
Cell[13105, 361, 3912, 86, 402, "Code",ExpressionUUID->"e51affbf-d070-40f3-9d80-2eb585689f69"],
Cell[17020, 449, 4985, 112, 391, "Code",ExpressionUUID->"eaa25f48-8f9c-455e-b249-2745887f794f"],
Cell[22008, 563, 1011, 26, 113, "Input",ExpressionUUID->"6e83dff4-3b47-4ab6-a26a-8b03316b75e7"],
Cell[CellGroupData[{
Cell[23044, 593, 427, 11, 28, "Input",ExpressionUUID->"e9568eb1-d887-4dd1-ad00-31179d34ffad"],
Cell[23474, 606, 810, 21, 48, "Output",ExpressionUUID->"dbbf2b4f-ffcf-49be-863f-5a9632b35603"]
}, Open  ]],
Cell[CellGroupData[{
Cell[24321, 632, 237, 4, 28, "Input",ExpressionUUID->"bce56010-3076-4918-990e-48e30ffe8044"],
Cell[24561, 638, 1800, 59, 177, "Output",ExpressionUUID->"e7efca14-8e8e-4a4c-9a75-0b883ff8bd01"]
}, Open  ]],
Cell[CellGroupData[{
Cell[26398, 702, 817, 23, 48, "Input",ExpressionUUID->"e183fb1a-8bcd-44b1-a584-556c2468d2d9"],
Cell[27218, 727, 6767, 223, 180, "Echo",ExpressionUUID->"e2882545-e6d5-4618-827e-a0cb1d477c8a"],
Cell[33988, 952, 237, 4, 32, "Output",ExpressionUUID->"5c9c3ab5-75c4-4c76-a7a7-21b1d660fa44"]
}, Open  ]],
Cell[CellGroupData[{
Cell[34262, 961, 860, 24, 48, "Input",ExpressionUUID->"2f4e45b3-7fa6-4592-8ff3-36def4589d14"],
Cell[35125, 987, 3883, 121, 111, "Echo",ExpressionUUID->"4d019157-92e3-41ae-9ef1-43c4ff332e91"],
Cell[39011, 1110, 264, 5, 32, "Output",ExpressionUUID->"35e711e8-d3ad-4d63-89f4-c1a5500f2b41"]
}, Open  ]],
Cell[CellGroupData[{
Cell[39312, 1120, 863, 24, 48, "Input",ExpressionUUID->"85bc0d3e-6657-4c92-9542-a765df9add69"],
Cell[40178, 1146, 5269, 165, 154, "Echo",ExpressionUUID->"507bf542-093a-4784-88cc-de0a4f78055f"],
Cell[45450, 1313, 235, 4, 32, "Output",ExpressionUUID->"a67cd722-6230-4f97-be14-687d7f14c0b6"]
}, Open  ]],
Cell[CellGroupData[{
Cell[45722, 1322, 1043, 29, 113, "Input",ExpressionUUID->"ba5c82a0-088c-4cf1-a1ad-0986fca7455a"],
Cell[46768, 1353, 945, 25, 48, "Output",ExpressionUUID->"6ee9cc06-7364-4385-a233-2ef377874ea1"]
}, Open  ]],
Cell[CellGroupData[{
Cell[47750, 1383, 859, 24, 48, "Input",ExpressionUUID->"1e49dcbf-fe57-4a2f-a7ee-a37555f78f70"],
Cell[48612, 1409, 3789, 117, 111, "Echo",ExpressionUUID->"950ab129-9f5a-4e73-97f9-f8d428f2d3b9"],
Cell[52404, 1528, 235, 4, 32, "Output",ExpressionUUID->"b56b39b8-1aeb-4878-9c1f-92e365a94bbd"]
}, Open  ]]
}, Open  ]],
Cell[52666, 1536, 329, 10, 10, "Text",ExpressionUUID->"fe506bc8-0dff-4042-97dc-1e293f4c795a"]
}, Closed]],
Cell[CellGroupData[{
Cell[53032, 1551, 245, 4, 37, "Subsubsection",ExpressionUUID->"4fb92e0e-4de9-4989-970a-9a445a85dcad"],
Cell[53280, 1557, 1174, 32, 93, "Code",ExpressionUUID->"163ed304-c504-4a7f-9be8-d918cf3dbdd8"]
}, Closed]],
Cell[CellGroupData[{
Cell[54491, 1594, 392, 12, 37, "Subsubsection",ExpressionUUID->"98c92339-349f-4e01-95ac-eb4b36a6aa40"],
Cell[54886, 1608, 10106, 211, 349, "Code",ExpressionUUID->"68aaac52-9924-4309-9b72-7ef068a73ed2"],
Cell[CellGroupData[{
Cell[65017, 1823, 230, 5, 28, "Input",ExpressionUUID->"f4d8bc79-bb13-4395-8feb-5c90dcacc067"],
Cell[65250, 1830, 180, 2, 32, "Output",ExpressionUUID->"48ce37e5-0690-43d4-b5d7-cb60bb044661"]
}, Open  ]],
Cell[65445, 1835, 156, 3, 28, "Input",ExpressionUUID->"e4d47643-85f4-47d5-9a29-9d1a7ec26265"],
Cell[65604, 1840, 1669, 36, 103, "Code",ExpressionUUID->"7031cc09-7c58-45f6-b504-e3788f770f78"]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[67322, 1882, 320, 5, 54, "Subsection",ExpressionUUID->"e61e3b31-1149-4ffc-beb2-262e22387740"],
Cell[CellGroupData[{
Cell[67667, 1891, 272, 4, 45, "Subsubsection",ExpressionUUID->"5a91f733-0312-4255-955d-606b5f1e8bc3"],
Cell[67942, 1897, 408, 11, 32, "Item",ExpressionUUID->"a1b1cf93-0bb1-4013-8654-39344ab81283"],
Cell[CellGroupData[{
Cell[68375, 1912, 1363, 41, 55, "Item",ExpressionUUID->"4f70ab94-55c9-41cd-8cfe-8f2a1b9d0fac"],
Cell[69741, 1955, 1015, 26, 28, "Subitem",ExpressionUUID->"a2606422-7faf-45c7-9cee-49f2af703c2c"],
Cell[70759, 1983, 1404, 41, 29, "Subitem",ExpressionUUID->"2cc485da-5ffc-4d97-ab25-df6984fd937f"],
Cell[72166, 2026, 863, 19, 45, "Subitem",ExpressionUUID->"abcf28db-03c9-4433-9833-90f16d81c9a3"]
}, Open  ]],
Cell[73044, 2048, 2556, 78, 60, "Item",ExpressionUUID->"cdf64d17-89af-4acc-adc4-8c840f100665"],
Cell[CellGroupData[{
Cell[75625, 2130, 1259, 34, 32, "Item",ExpressionUUID->"cbb0b52f-b1c6-4025-b95a-ea8c2b6d1a09"],
Cell[76887, 2166, 575, 10, 26, "Subitem",ExpressionUUID->"282a3d75-e159-4fef-830b-a7f0765fb544"],
Cell[77465, 2178, 1249, 34, 46, "Subitem",ExpressionUUID->"2031ef2c-0b8e-47e7-b437-69f2b9b9f985"]
}, Open  ]],
Cell[CellGroupData[{
Cell[78751, 2217, 1016, 23, 34, "Item",ExpressionUUID->"66481e06-9583-44de-b1aa-f532c7420d9d"],
Cell[79770, 2242, 2015, 52, 28, "Subitem",ExpressionUUID->"ba598b30-98fd-4cbe-bb02-59567de99dbb"],
Cell[81788, 2296, 1467, 34, 29, "Subitem",ExpressionUUID->"9f7f1cab-14fd-407e-974f-a090dc0d795f"],
Cell[83258, 2332, 2607, 76, 75, "Subitem",ExpressionUUID->"f70bfd1c-61b6-4b34-b551-03fe955663a2"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[85914, 2414, 215, 4, 45, "Subsubsection",ExpressionUUID->"c5d2a4d3-cd48-4d57-ab2b-4a600ec661d7"],
Cell[86132, 2420, 28364, 605, 1439, "Code",ExpressionUUID->"e9cd84a4-60a1-48cc-af30-a6509b2aa33d"]
}, Closed]],
Cell[CellGroupData[{
Cell[114533, 3030, 309, 4, 37, "Subsubsection",ExpressionUUID->"74cf22c0-cebc-4688-a884-758007aa1a73"],
Cell[114845, 3036, 5595, 125, 317, "Code",ExpressionUUID->"a892f1fc-a62c-47fb-b653-7cf1298ed7e4"]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[120489, 3167, 369, 6, 54, "Subsection",ExpressionUUID->"97fb38f9-3784-484b-820c-1bb55037754c"],
Cell[CellGroupData[{
Cell[120883, 3177, 288, 4, 45, "Subsubsection",ExpressionUUID->"8aa2f22d-1245-4b09-81c5-be719ae469e7"],
Cell[CellGroupData[{
Cell[121196, 3185, 2124, 56, 78, "Input",ExpressionUUID->"f16b41d7-b408-42be-a88f-633b207c9144"],
Cell[CellGroupData[{
Cell[123345, 3245, 241, 6, 24, "Echo",ExpressionUUID->"107b2363-5e18-446b-9acf-a487a65724be"],
Cell[123589, 3253, 1006, 22, 52, "Echo",ExpressionUUID->"0542b29c-ca6a-404b-bfdc-532a19def008"],
Cell[124598, 3277, 1004, 22, 52, "Echo",ExpressionUUID->"c63b6e89-2083-4312-995c-42095547d372"],
Cell[125605, 3301, 2057, 53, 104, "Echo",ExpressionUUID->"bdbeaee5-50df-4c40-94d1-3f8b54722542"],
Cell[127665, 3356, 481, 12, 52, "Echo",ExpressionUUID->"a16b7ce4-6721-4cfa-a5a2-d62cc36d0d40"],
Cell[128149, 3370, 259, 7, 24, "Echo",ExpressionUUID->"b9c7d616-2964-4856-bd1f-9d4a098cc3ea"],
Cell[128411, 3379, 1194, 28, 68, "Echo",ExpressionUUID->"40a58280-7fb6-485c-9383-74c40d716bbd"],
Cell[129608, 3409, 268, 7, 24, "Echo",ExpressionUUID->"279b4444-05c3-436c-9b56-9386b5e99ce8"],
Cell[129879, 3418, 1405, 35, 104, "Echo",ExpressionUUID->"e7e9ea27-7e22-438d-ba76-89ca8b6b3c09"],
Cell[131287, 3455, 269, 7, 24, "Echo",ExpressionUUID->"46284eeb-2275-4d6a-a543-78bf7fac104e"],
Cell[131559, 3464, 1963, 53, 104, "Echo",ExpressionUUID->"1c96d6c6-60eb-4ac7-a1f7-1a8509411346"],
Cell[133525, 3519, 1507, 37, 104, "Echo",ExpressionUUID->"ba8e633f-0236-46d7-885b-abe38d220436"]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[135093, 3563, 323, 5, 45, "Subsubsection",ExpressionUUID->"3c4a4526-ad0e-4758-84ec-f4b445c3f524"],
Cell[CellGroupData[{
Cell[135441, 3572, 1836, 52, 78, "Input",ExpressionUUID->"e16f26f2-b4eb-4bcf-aaeb-69c197d2986d"],
Cell[CellGroupData[{
Cell[137302, 3628, 241, 6, 24, "Echo",ExpressionUUID->"15495224-2a80-487a-bdfa-c2f3e263efa5"],
Cell[137546, 3636, 1519, 38, 104, "Echo",ExpressionUUID->"0204440c-39ec-43ef-a48e-dff542e87843"],
Cell[139068, 3676, 1095, 24, 68, "Echo",ExpressionUUID->"69905599-de13-4f63-afda-3d5e39ac4288"],
Cell[140166, 3702, 1567, 36, 68, "Echo",ExpressionUUID->"bfa85604-abb8-4eb8-8d43-be966474b1e5"],
Cell[141736, 3740, 967, 28, 104, "Echo",ExpressionUUID->"a8bbe2ec-c1a5-4fdf-9640-1adcd1b69473"],
Cell[142706, 3770, 261, 7, 24, "Echo",ExpressionUUID->"840d72f9-c547-403a-9431-80b285279dfc"],
Cell[142970, 3779, 1102, 25, 68, "Echo",ExpressionUUID->"535e6b09-c1da-477b-a83a-5a1697422df9"],
Cell[144075, 3806, 268, 7, 24, "Echo",ExpressionUUID->"93d1a33c-af2f-4041-92be-cb29bb2930c2"],
Cell[144346, 3815, 642, 15, 68, "Echo",ExpressionUUID->"f93ea8c9-c7df-4f90-8e5b-8969acf0945e"],
Cell[144991, 3832, 269, 7, 24, "Echo",ExpressionUUID->"5e696a28-fa77-4e46-aaf3-fdcfde39f711"],
Cell[145263, 3841, 1201, 28, 68, "Echo",ExpressionUUID->"654885e7-d940-4f7b-8b1b-8de9a877301f"],
Cell[146467, 3871, 1051, 23, 68, "Echo",ExpressionUUID->"681e009a-b8ef-414d-8209-23fee8ea1943"]
}, Open  ]]
}, Open  ]],
Cell[147545, 3898, 99, 1, 4, "PageBreak",ExpressionUUID->"16253e6b-f67e-4cbe-af55-e4c25e1174c8",
 PageBreakBelow->True]
}, Closed]],
Cell[CellGroupData[{
Cell[147681, 3904, 520, 8, 37, "Subsubsection",ExpressionUUID->"697bbcb7-b5f0-4792-8f5f-338d2e4caa49"],
Cell[CellGroupData[{
Cell[148226, 3916, 1842, 56, 113, "Input",ExpressionUUID->"bcbac566-c006-444b-b4b1-229762bdaa47"],
Cell[150071, 3974, 2588, 76, 64, "Output",ExpressionUUID->"b2b88efa-1557-44aa-8f25-e274a0082bb8"]
}, Open  ]],
Cell[CellGroupData[{
Cell[152696, 4055, 893, 19, 33, "Input",ExpressionUUID->"1699be7d-510b-4f88-81fe-c1c8178c4a9d"],
Cell[CellGroupData[{
Cell[153614, 4078, 243, 6, 24, "Echo",ExpressionUUID->"4293c3f0-a331-4f53-8e1d-a6ea9b8c672f"],
Cell[153860, 4086, 2552, 72, 112, "Echo",ExpressionUUID->"c595c5c5-dfef-4fdf-9a65-9e57782d5fd2"],
Cell[156415, 4160, 239, 6, 24, "Echo",ExpressionUUID->"31d4ccca-c866-43a4-a6a9-386248974e55"],
Cell[156657, 4168, 1060, 24, 68, "Echo",ExpressionUUID->"f7c9fe0a-75bf-4a93-a63e-7d737e47536e"],
Cell[157720, 4194, 1058, 24, 68, "Echo",ExpressionUUID->"3416f7da-521a-4dea-946f-c2ff0329c0af"],
Cell[158781, 4220, 2506, 72, 112, "Echo",ExpressionUUID->"47d4fcd7-629d-431f-a475-0e37c42151d7"],
Cell[161290, 4294, 529, 13, 68, "Echo",ExpressionUUID->"340a3586-fc18-4be1-a1d6-d10f16616fce"],
Cell[161822, 4309, 261, 7, 24, "Echo",ExpressionUUID->"bdc22c76-1752-4c6b-a3b7-8b3a487ae9dd"],
Cell[162086, 4318, 1248, 30, 68, "Echo",ExpressionUUID->"c8ea05bb-ee51-4935-9781-ce9c44d560de"],
Cell[163337, 4350, 268, 7, 24, "Echo",ExpressionUUID->"c8deaa57-c20c-467e-8db5-3ce498a31635"],
Cell[163608, 4359, 750, 18, 104, "Echo",ExpressionUUID->"5d310034-2528-4834-ae0b-68479e2614ce"],
Cell[164361, 4379, 267, 7, 24, "Echo",ExpressionUUID->"760d47b0-39a1-489d-81d6-334b21c327e9"],
Cell[164631, 4388, 3053, 93, 112, "Echo",ExpressionUUID->"12a91a10-1331-4345-89eb-8294d5840c9f"],
Cell[167687, 4483, 1280, 30, 104, "Echo",ExpressionUUID->"f71d9c80-ceec-47c1-a6a8-f8e84e2171a6"]
}, Open  ]]
}, Open  ]],
Cell[168994, 4517, 99, 1, 4, "PageBreak",ExpressionUUID->"54d0e004-1bc6-4e65-aaf2-6100249b0846",
 PageBreakBelow->True]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[169142, 4524, 406, 6, 54, "Subsection",ExpressionUUID->"2f730ccd-7488-4a45-b980-6b1f82ac468e"],
Cell[169551, 4532, 227, 5, 32, "Item",ExpressionUUID->"d44a0f46-4b70-413d-b46c-15868d79368a"],
Cell[169781, 4539, 302, 6, 53, "Item",ExpressionUUID->"06705ce5-539f-45f6-88bb-b3bf11a44e19"]
}, Closed]]
}
]
*)

