library(coxme)
library("survival")
library(survminer)
library(ggplot2)
library(dplyr)

target.dat=read.csv("Target_survival_data.csv", header = TRUE)

# Treatment = target treatment code (see below)
# PC = photocode
# B = block (random effect of different transects, each with 60 trees)
# T = survival time in hours
# 0 = censored data (alive after last check or taken by ant, spider etc.), 1 = experienced event (predation)
# cens = target censored or not

target.dat$B=as.factor(target.dat$B)
target.dat$Treatment<-as.factor(target.dat$Treatment)

### Note treatment codes:

# A = 3D continuous surface (labelled iii in paper)
# B = 2D continuous surface (unpatterned) (vi)
# C = 2D disruptive (iv)
# D = 2D enhanced edge (v)
# E = 3D physical disruption (i)
# F = 3D physical disruption (ii)

# 1. Run full survival model with all six treatments

fm = coxme(Surv(T, event=cens) ~ Treatment + (1|B), target.dat)
fullsum = summary(fm,rcoef = T)
fullsum

# 2. Compare to null model
nm = coxme(Surv(T, event=cens) ~ (1|B), target.dat)
nmsum = summary(nm,rcoef = T)
anova(fm,nm) # Compare models

# Manually fit contrasts between all treatment combinations
# Code very kindly provided by a statistics referee
# adapted from # https://stats.stackexchange.com/questions/141255/categorical-variablecoding-
 # to-compare-all-levels-to-all-levels/269746#269746

cmat = matrix(0, 10, 5)
nam = character(length=10)
row = 0
for (i in 1:4) for (j in (i+1):5) {
  treatments =c("B","C","D","E","F")
  row = row+1
  nam[row] = paste("treatment[", treatments[i], "]-treatment[", treatments[j], "]",
                   sep="")
  cmat[row, c(i, j)] <- c(1, -1)
}
rownames(cmat) <- nam
co = coef(fm)
CV = vcov(fm)
se = sqrt( diag( cmat %*% CV %*% t(cmat) ))
contr = cmat %*% co
p = 2*pnorm(-abs(unname(contr[,1] / se)))
ci = cbind.data.frame(diff=contr,se=se,z=contr/se,p=p)
ci # note: no adjustment for familywise error rate

# Adjust familywise error rate according to a priori planned comparisons:

#1.  Do 3D prey with disrupted surfaces survive longer than those with continuous surfaces?
AEF = c(0.247513982, 0.190, 0.013) # pvalues of E vs F, E vs A and F vs A. 
p.adjust(AEF,"fdr")


# 2. Is there a difference in survival between unpatterned (2D) prey, and those with disruptive patterns or enhanced edges? (compare BCD)

BCD = c(0.475320562,0.789682353,0.656882416) # p-values for B vs C, B vs D and C vs D from full model above
p.adjust(BCD,"fdr") # Adjusted p-values

# 3. Is there a difference in survival between 3D and 2D body shapes (A vs B)?

target.dat$Treatment<-as.factor(target.dat$Treatment)
target.dat$Treatment2<-dplyr::recode(target.dat$Treatment, A = "3D", F = "3D", E = "3D", B =
                                       "2D", C = "2D", D = "2D") # Recode target types

fm2 =coxme(Surv(T, event=cens) ~ Treatment2 + (1|B), target.dat)

summary(fm2)

fm2nm = coxme(Surv(T, event=cens) ~ (1|B), target.dat)
summary(fm2nm)
anova(fm2, fm2nm)

fm2.fit<-survfit(Surv(T, event= cens) ~ Treatment2, data=target.dat) 
ggsurvplot(fm2.fit, data=target.dat, pval = FALSE, size = 1.5, palette = "d3")

## Re-analysis of data for which image information (luminance) is available

image.dat<-read.csv('Image_and_target_data.csv')
str(image.dat)

image.dat$B<-as.factor(image.dat$B)
image.dat$Treatment<-as.factor(image.dat$Treatment)

# 4. Do differences in luminance (e.g. shadows) account for differences in survival of 3D prey?

mean.lum<-mean(image.dat$wing.dblMean) # Calculate mean wing luminance
std.lum<-image.dat$wing.dblMean/mean.lum # Standardize covariate by dividing by mean

treat_and_lum=coxme(Surv(T, event=cens) ~ Treatment + std.lum + (1|B), image.dat) # Treatment and luminance model 
treat_and_lum_sum = summary(treat_and_lum, rcoef=T)

treat_only = coxme(Surv(T, event=cens) ~ Treatment + (1|B), image.dat)
fullsum.treat_only = summary(treat_only,rcoef = T)
fullsum.treat_only # Treatment only model 

lum_only=coxme(Surv(T, event=cens) ~ std.lum + (1|B), image.dat) #luminance only model
fullsum.lum_only = summary(lum_only,rcoef = T)
fullsum.lum_only 

nm.image = coxme(Surv(T, event=cens) ~ (1|B), image.dat) # null model

# Compare fits
anova(treat_and_lum, treat_only) ## Is luminance and treatment model better than treatment only?
anova(lum_only, nm.image) # Is a luminance only model better than the null model?
anova(treat_and_lum, lum_only) # Is the luminance and treatment model better than the luminance only model?
anova(treat_and_lum, nm.image) # Is the final model better than the null?
print(treat_and_lum) #Print final model

# Run all comparisons (treatment only model) for image dataset

cmat = matrix(0, 10, 5)
nam = character(length=10)
row = 0
for (i in 1:4) for (j in (i+1):5) {
  treatments =c("B","C","D","E","F")
  row = row+1
  nam[row] = paste("treatment[", treatments[i], "]-treatment[", treatments[j], "]",
                   sep="")
  cmat[row, c(i, j)] <- c(1, -1)
}
rownames(cmat) <- nam
co = coef(treat_only)
CV = vcov(treat_only)
se = sqrt( diag( cmat %*% CV %*% t(cmat) ))
contr = cmat %*% co
p = 2*pnorm(-abs(unname(contr[,1] / se)))
ci = cbind.data.frame(diff=contr,se=se,z=contr/se,p=p)
ci # note: no adjustment for familywise error rate

# Adjust familywise error rate according to planned comparisons:
AEF.image = c(0.76578930, 0.034, 0.051, 0.010) # pvalues of E vs F, E vs A and F vs A and std lum
p.adjust(AEF.image,"fdr")






