# Single-marker analysis using a damselfly, Ischnura elegans

######
# data analysis

# load library
library(lme4)
library(ggplot); library(patchwork)

# data generated by Takahashi et al. (2014) Nat. Comm. https://doi.org/10.1038/ncomms5468
# available upon request for Y. Takahashi (takahashi.yum@gmail.com)
d = read.csv("fitness_data_forSato.csv", header=TRUE)

d$morph[d$morph=="i"] = -1 #infuscans-type gynomorph as ressesive
d$morph[d$morph=="a"] = +1 #andromorph as dominant
d$morph = as.numeric(d$morph)

# exchange frequency conditions into morph similarities
f_sim = c(NA, nrow(d))
f_sim[(d$morph==1)&(d$frequency=="0.8")] = 0.6
f_sim[(d$morph==1)&(d$frequency=="0.5")] = 0.0
f_sim[(d$morph==1)&(d$frequency=="0.2")] = -0.6

f_sim[(d$morph==-1)&(d$frequency=="0.8")] = -0.6
f_sim[(d$morph==-1)&(d$frequency=="0.5")] = 0.0
f_sim[(d$morph==-1)&(d$frequency=="0.2")] = 0.6

d = data.frame(d,f_sim)

# Poisson GLMMs
summary(glmer(matureeggnumber~morph*f_sim+density+(1|cageID/experimentID),data=d,family=poisson)) #morph x f_sim not signif.
summary(glmer(matureeggnumber~morph+f_sim+density+(1|cageID/experimentID),data=d,family=poisson))

######
# Figure

# observed
pch = rep(NA,length(d$morph))
pch[d$morph==-1] = 1; pch[d$morph==1] = 16
p2 = ggplot(d, aes(x=frequency,y=log(matureeggnumber))) + geom_jitter(pch=pch,colour="grey",width=0.05) +
  theme_classic() + ylab("log(no. of eggs)") + xlab("phenotype-level frequency of andromorph") + xlim(0,1)

# predicted (see Appendix S3)
# b0 = 4.55; b1 = 0.021; b2 = -0.878; b12 = 0.04
# f_star = 0.5-(b1/(2*b2))

plt_Ie = function(b0,b1,b2,b12) {
  f_star = 0.5-(b1/(2*b2))
  p2 = ggplot(d, aes(x=frequency,y=log(matureeggnumber))) + geom_jitter(pch=pch,colour="grey",width=0.05) +
    theme_classic() + ylab("Egg production") + xlab("phenotype-level frequency of andromorph") + xlim(0,1) +
    geom_function(aes(x=1,y=1),fun=function(x,b0,b1,b2,b12) { (b12+b2)*(2*x-1)+b0+b1 }, args=list(b0,b1,b2,b12)) + 
    geom_function(aes(x=1,y=1),fun=function(x,b0,b1,b2,b12) { (b12-b2)*(2*x-1)+b0-b1 }, args=list(b0,b1,b2,b12), colour=grey(0.0,0.33)) +
    geom_function(aes(x=1,y=1),fun=function(x,b0,b1,b2,b12) { 2*b2*x*(2*x-1)+(b12-b2)*(2*x-1)+b0-b1+2*b1*x }, args=list(b0,b1,b2,b12), lty=2) +
    geom_point(aes(x=f_star,y=(b12+b2)*(2*f_star-1)+b0+b1),pch=16,size=3) + labs(title=substitute(paste(italic('Ischnura elegans'))))
  
  return(p2)
}

p2 = plt_Ie(b0 = 4.55, b1 = 0.021, b2 = -0.878, b12 = 0.04)

######
# combined figs
p1 = readRDS("Ahal_plot.rds")
p = p1 + p2 + plot_annotation(tag_levels = "a")
ggsave(p, filename="Ah_Ie_plots.pdf",width=8,height=4)
