## get data

lineages_by_ltla_and_week <- read_tsv("data/lineages_by_ltla_and_week.tsv")

lineages_by_ltla_and_week %<>% 
    group_by(WeekEndDate, Lineage) %>%
    summarise(Count = sum(Count), .groups = 'drop')

lineages_by_ltla_and_week %<>% 
    mutate(Lineage_2 = case_when(
        Lineage == "None"                         ~ "None",
        grepl(pattern = "AY",        x = Lineage) ~ "Delta",
        grepl(pattern = "B.1.617.2", x = Lineage) ~ "Delta",
        grepl(pattern = "B.1.1.7",   x = Lineage) ~ "Alpha",
        TRUE                                      ~ "Other"
    ))

lineage_cols <- c("Alpha" = 'lightskyblue',# rgb(118,166,213,maxColorValue = 255),
                  "Delta" = rgb(74,186,153,maxColorValue = 255),
                  "Other" = "grey80",
                  "None"  = "grey40")

data_to_plot <- lineages_by_ltla_and_week %>% 
    #filter(Lineage_2 %in% c("Alpha", "Delta")) %>%
    group_by(WeekEndDate, Lineage_2) %>%
    summarise(Count = sum(Count), .groups = 'drop') %>%
    group_by(WeekEndDate) %>%
    mutate(N = sum(Count),
           p = Count/N)  %>%
    ungroup %>%
    mutate(Lineage_2 = factor(Lineage_2,
                              levels = c( "Other", "None", "Alpha", "Delta")))  %>%
    mutate(Source = "Data")

data_plot <- data_to_plot %>%
    ggplot(data = .,
           aes(x = WeekEndDate, y = p)) +
    geom_col(aes(fill = Lineage_2),
             color = 'black', size = .1, width = 6) +
    xlab("Time (week ending date)") +
    scale_x_date(limits = range(dat_clean$SAMPLE_DT) + c(-3,3) ,
                 date_breaks = "1 month") +
    theme(axis.text.x = element_text(hjust = 1, angle = 45),
          panel.grid.minor = element_blank()) +
    scale_fill_manual(values = lineage_cols,
                      name = "Variant") +
    ylab("Proportion")

p_load(patchwork)

lineages_by_ltla_and_week_to_model <- 
    lineages_by_ltla_and_week %>% 
    filter(Lineage_2 %in% c("Alpha", "Delta")) %>%
    group_by(WeekEndDate, Lineage_2) %>%
    summarise(Count = sum(Count), .groups = 'drop') %>%
    group_by(WeekEndDate) %>%
    mutate(N = sum(Count),
           p = Count/N)  %>%
    ungroup %>%
    mutate(time = as.numeric(WeekEndDate - as.Date('2020-01-01')))

lineages_by_ltla_and_week_model <- 
    lineages_by_ltla_and_week_to_model %>% 
    filter(Lineage_2 == "Delta")  %>%
    glm(data = .,
        p ~ time, weights = N, family = binomial())

model_to_plot <- lineages_by_ltla_and_week_to_model %>%
    distinct(WeekEndDate, time) %>%
    mutate(pred = predict(newdata = ., 
                          object = lineages_by_ltla_and_week_model,
                          type = 'response')) %>%
    left_join(select(lineages_by_ltla_and_week_to_model, WeekEndDate, p, Lineage_2) %>%
                  filter(Lineage_2 == "Delta") ) %>%
    mutate(Lineage_2 = "Delta") %>%
    mutate(Lineage_2 = factor(Lineage_2,
                              levels = c( "Other", "None", "Alpha", "Delta"))) %>%
    mutate(Source = "Model")

model_plot <- model_to_plot %>%
    ggplot(data = ., aes(x = WeekEndDate, y = p, color = Lineage_2)) +
    geom_point() +
    geom_line(aes(y = pred)) +
    # ylab("P(Lineage is Delta|lineage is Alpha or Delta)") +
    ylab("Proportion") +
    xlab("Time (week ending date)") +
    scale_x_date(limits = range(dat_clean$SAMPLE_DT) + c(-3,3),
                 date_breaks = "1 month") +
    theme(axis.text.x = element_text(hjust = 1, angle = 45),
          panel.grid.minor = element_blank()) +
    scale_color_manual(values = lineage_cols, name = "Variant")


combo_plot <-
    ggplot(data = data_to_plot,
           aes(x = WeekEndDate, y = p)) +
    geom_col(aes(fill = Lineage_2),
             color = 'black', size = .1, width = 6) +
    geom_point(data = model_to_plot, aes(color = Lineage_2),
               show.legend = F) +
    geom_line(data  = model_to_plot, aes(color = Lineage_2, y = pred),
              show.legend = F) +
    xlab("Time (week ending date)") +
    scale_x_date(limits = range(dat_clean$SAMPLE_DT) + c(-3,3),
                 date_breaks = "1 month") +
    theme(axis.text.x = element_text(hjust = 1, angle = 45),
          panel.grid.minor = element_blank(),
          legend.position = 'bottom') +
    scale_fill_manual(values = lineage_cols,
                      name = "Variant") +
    scale_color_manual(values = lineage_cols,
                       name = "Variant") +
    ylab("Proportion") +
    facet_grid(Source ~ .)

ggsave(filename = "figures/combo_plot.pdf", combo_plot, 
       device = cairo_pdf, width = 4, height = 4)


Dates <- lineages_by_ltla_and_week %>% distinct(WeekEndDate) %>% pull(WeekEndDate) 

Variants <- map2(.x = dat_to_analyse, 
                 .y = Swabs, 
                 .f = calculate_variant_prob,
                 model = lineages_by_ltla_and_week_model)

lineages_plot <- Swabs$Sensitivity %>%
    mutate(HOCONUMBER = sub(pattern = '(^.*)/.*',
                            replacement = "\\1", 
                            x = NickID)) %>%
    group_by(HOCONUMBER) %>%
    summarise(SAMPLE_DT = min(SAMPLE_DT)) %>%
    inner_join(Variants$Sensitivity) %>%
    mutate(WeekEndDate = cut(SAMPLE_DT, breaks = Dates),
           WeekEndDate = as.Date(WeekEndDate) + 7) %>%
    arrange(WeekEndDate, desc(p_delta)) %>%
    group_by(WeekEndDate) %>%
    mutate(y = 1:n()) %>%
    ggplot(data = ., aes(x = WeekEndDate, y = y)) +
    geom_point(aes(color = p_delta,
                   shape = Variant %in% c("Alpha", "Delta"))) +
    scale_color_gradient(low = lineage_cols["Alpha"],
                         high = lineage_cols["Delta"],
                         name = "p(Delta|Alpha or Delta)\n") +
    scale_shape_manual(values = c(`TRUE`  = 19,
                                  `FALSE` = 1), 
                       labels = c(`TRUE`  = "Yes",
                                  `FALSE` = "No"),
                       name = "Sequence available") +   
    theme(legend.position = 'bottom') +
    ylab("Number of households\nenrolled this week") + 
    xlab("Time (week ending date)") +
    scale_x_date(limits = range(dat_clean$SAMPLE_DT) + c(-3,3),
                 date_breaks = "1 month") +
    theme(axis.text.x = element_text(hjust = 1, angle = 45),
          panel.grid.minor = element_blank()) +
    ylim(c(0, NA))

ggsave(filename = "figures/lineages_plot.pdf", lineages_plot, 
       device = cairo_pdf, width = 8, height = 4)


lineages_by_ltla_and_week %>%
    group_by(WeekEndDate) %>%
    mutate(N = sum(Count),
           p = Count/N) %>%
    filter(WeekEndDate > as.Date("2020-03-30"),
           WeekEndDate < as.Date("2020-11-17")) %>%
    ungroup %>%
    arrange(desc(p)) %>%
    group_by(Lineage) %>%
    mutate(P    = max(p),
           flag = P > 0.05) %>%
    mutate(Lineage_2 = ifelse(flag,
                              Lineage,
                              "Other")) %>%
    arrange(P) %>%
    mutate(Lineage_2 = fct_reorder(factor(Lineage_2), .x = P, .fun = median)) %>%
    mutate(Lineage_2 = fct_relevel(Lineage_2, 'Lineage data suppressed', "Other", after = Inf)) %>%
    ggplot(data = ., aes(x = WeekEndDate, y = p)) +
    geom_col() +
    facet_wrap(~Lineage_2, nrow = 2)



vaccine_plot <- dat_clean %>%
    filter(ReCodedVaccineType != "None") %>%
    distinct(HOCONUMBER, NickID, RoundedAge, VaccDate1, ReCodedVaccineType) %>%
    mutate(#VaccDate1 = cut(VaccDate1, breaks = Dates),
        #VaccDate1 = as.Date(VaccDate1) + 7,
        Age = cut(RoundedAge, c(0, 30, 40, 88),
                  include.lowest = FALSE)) %>%
    # arrange(VaccDate1, Age, ReCodedVaccineType) %>%
    # group_by(VaccDate1, Age) %>%
    # mutate(y = 1:n()) %>%
    ggplot(data = ., aes(x = VaccDate1)) +
    geom_bar(aes(fill = ReCodedVaccineType)) +
    theme(legend.position = 'bottom') +
    ylab("Number of individuals vaccinated") + 
    xlab("Time (date of first vaccination)") +
    scale_x_date(#limits = range(dat_clean$SAMPLE_DT) + c(-3,3),
        date_breaks = "1 month") +
    theme(axis.text.x = element_text(hjust = 1, angle = 45),
          panel.grid.minor = element_blank()) +
    facet_grid(Age ~ ., scales = 'free_y', space = 'free_y') +
    scale_y_continuous(limits = c(0, NA)#,
                       #breaks = seq(1, 10, by = 5)
    ) +
    # scale_shape_manual(values = c("ChAdOx1" = 19, "BNT162b2" = 1),
    #                    name = "Vaccine product received") +
    scale_fill_manual(values = c("BNT162b2" = "red", "ChAdOx1" = "black"),
                      breaks = c("ChAdOx1", "BNT162b2"),
                       name = "Vaccine product received") +
    geom_vline(aes(xintercept = VaccDate1),
               lty = 2,
               data = data.frame(Age       = c('(0,30]','(30,40]'),
                                 VaccDate1 = as.Date(c('2021-04-07', '2021-05-07')))) +
    theme(axis.text.y = element_blank(),
          panel.grid.major.y = element_blank())

ggsave(filename = "figures/vaccine_plot.pdf", vaccine_plot, 
       device = cairo_pdf, width = 8, height = 6)
