# These are model summary results as Odds Ratios

vas_ORs <- map_df(vas_fit,
                  calculate_ORs, 
                  .id = "Analysis", 
                  conf_level = c(0.5, 0.95)) 

# how much more transmissible is Delta?
vas_fit %>%
    map_df(~mmcc:::tidy.mcmc.list(.x$vas_pst) %>%
               filter(parameter == 'delta'),
           .id = 'Analysis') %>%
    mutate_at(.vars = vars(-c(Analysis, parameter, sd)),
              .funs = exp) %>%
    group_by(Analysis) %>%
    transmute(`Odds Ratio` = sprintf("%0.1f (%0.1f, %0.1f)", median, `2.5%`, `97.5%`)) %>%
    kable('simple', align = 'lr', 
          caption = "Increased transmissibility of Delta variant (Odds ratio and 95% CrI)")

# age effects as odds ratios
vas_fit %>%
    map_df(~mmcc:::tidy.mcmc.list(.x$vas_pst, conf_level = c(0.5, 0.95)) %>%
               filter(grepl(pattern = 'epsilon', x = parameter)),
           .id = 'Analysis') %>%
    mutate_at(.vars = vars(-c(Analysis, parameter, sd)),
              .funs = exp) %>%
    mutate(Status = sub(pattern = 'epsilon_', 
                        replacement = "", x = parameter) %>%
               str_to_title) %>%
    mutate(Analysis = factor(Analysis, levels = names(min_SIs))) %>%
    group_by(Analysis, Status) %>%
    transmute(`Odds Ratio` = sprintf("%0.2f (%0.2f, %0.2f)", median, `2.5%`, `97.5%`)) %>%
    spread(Status, `Odds Ratio`) %>%
    kable('simple', align = 'lrr', 
          caption = "Age effects for transmissibility of 50+ case and susceptibility of <18 contact")

vas_fit %>%
    map_df(~mmcc:::tidy.mcmc.list(.x$vas_pst, conf_level = c(0.5, 0.95)) %>%
               filter(grepl(pattern = 'epsilon', x = parameter)),
           .id = 'Analysis') %>%
    mutate(Status = sub(pattern = 'epsilon_', 
                        replacement = "", x = parameter) %>%
               str_to_title) %>%
    mutate(Analysis = factor(Analysis, levels = names(min_SIs))) %>%
    group_by(Analysis, Status) %>%
    transmute(Precision = 1/sd^2) %>%
    spread(Status, Precision) %>%
    kable(format = 'simple', digits = 1, caption = "Posterior precision of age effects")

pretty_ors(vas_ORs) %>%
    unite(col = 'Vaccine', Doses, Vaccine, sep = " ", remove = T) %>%
    knitr::kable(format = 'simple')

write_csv(pretty_ors(vas_ORs), 'figures/table_ORs.csv')

pretty_ors(vas_ORs) %>%
    rename(`Inf. prot.` = `Infection protection`,
           `Trans. red.` = `Transmission reduction`) %>%
    knitr::kable(booktabs = TRUE, label = "ORs",
                 format = 'latex', linesep="", align = 'lllrrr',
                 caption = "Median odds ratios and 95\\% credible intervals for infection protection in contacts and transmission reduction in cases, by vaccination status.") %>%
    write_file('figures/table_ORs.tex')

plot_vas_ORs <-
    vas_ORs %>%
    filter(Analysis == "Main") %>%
    mutate(Variant = fct_drop(Variant)) %>%
    mutate_if(is.numeric, ~pmax(., 0.01)) %>%
    arrange(Effect, Vaccine, Dose) %>%
    unite(col = "Vaccine status", Dose, Vaccine, sep = " ") %>%
    mutate(`Vaccine status` = fct_rev(fct_inorder(factor(`Vaccine status`)))) %>%
    ggplot(data = .,
           aes(x = `Vaccine status`, y = median,
               color = Variant,
               group = rev(Variant))) +
    geom_linerange(aes(ymin = `2.5%`,  ymax = `97.5%`),
                   position = position_dodge(w = 0.5)) +
    geom_linerange(aes(ymin = `25.0%`, ymax = `75.0%`), 
                   size = 2,
                   position = position_dodge(w = 0.5)) +
    geom_point(pch = 124, size = 2, color = 'white', 
               position = position_dodge(w = 0.5)) +
    geom_hline(yintercept = 1, lty = 2) +
    facet_grid(. ~ Effect) + #,
    coord_flip() + 
    # labeller = labeller(Effect = function(x){sub(x = x, pattern = " ",
    # replacement = "\n")})) +
    scale_color_manual(values = lineage_cols[c('Alpha', 'Delta')],
                       name = "Variant", 
                       drop = TRUE) +
    scale_y_continuous(trans = 'log10', labels = function(x){sprintf("%g",x)}) +
    theme(panel.grid.minor = element_blank(),
          axis.ticks = element_line()#,
          #legend.position = 'none'
    ) +
    xlab(NULL) + 
    #annotation_logticks(sides = 'b') +
    ylab("Odds ratio") 

ggsave("figures/plot_vas_ORs.pdf",
       plot = plot_vas_ORs,
       device = cairo_pdf,
       width = 8, height = 3)