import java.io.File;
import java.io.IOException;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.regex.Pattern;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;

public class VulnerabilityManager {
	
	//https://csrc.nist.gov/CSRC/media/Projects/National-Vulnerability-Database/documents/web%20service%20documentation/Automation%20Support%20for%20CVE%20Retrieval.pdf
	private static final String nvdBaseUrl = "https://services.nvd.nist.gov/rest/json/";

	private HashMap<String, Software> software;
	private HashMap<String, Vulnerability> vulnerabilities;
	
	private static String cvePattern = "CVE-\\d\\d\\d\\d-\\d\\d\\d\\d+";

	public VulnerabilityManager() {
		software = new HashMap<String, Software>();
		vulnerabilities = new HashMap<String, Vulnerability>();
	}
	
	public VulnerabilityManager(HashMap<String, Software> software) {
		this();
		setSoftware(software);
	}

	public HashMap<String, Software> getSoftware() {
		return software;
	}

	public void setSoftware(HashMap<String, Software> software) {
		this.software = software;
	}
	
	public HashMap<String, Vulnerability> getVulnerabilities() {
		return vulnerabilities;
	}
	
	public void refreshVulnerabilities(boolean force) {
		for (Software s : software.values()) {
			ModeSwitcher.log("refreshVulnerabilitiesPerProduct " + s.getVendor() + " " + s.getProduct() + " " + s.getVersion());
			refreshVulnerabilitiesPerProduct(s.getVendor(),s.getProduct(),s.getVersion(), force);
		}
	}	

	public void refreshVulnerabilitiesPerProduct(String vendor, String product, String version, boolean force) {
		//https://services.nvd.nist.gov/rest/json/cpes/1.0?cpeMatchString=cpe:2.3:a:apache:http_server:2.4.38:*:*:*:*:*:*:*&addOns=cves
		//result.cpes[0].vulnerabilities
		String filename = ModeSwitcher.CPESDIR + File.separator + vendor + "_" + product + "_" + version + ".json";
		File f = new File(filename);		
		if(!f.exists() || force == true) {
			FileUtils.loadFileFromUrl(nvdBaseUrl + "cpes/1.0?cpeMatchString=cpe:2.3:a:"+vendor+":"+product+":"+version+":*:*:*:*:*:*:*&addOns=cves", filename);
		}
		
		if(f.exists()) {
		
		ObjectMapper objectMapper = new ObjectMapper();
		/*URL url = null;
		try {
			url = new URL(nvdBaseUrl + "cpes/1.0?cpeMatchString=cpe:2.3:a:"+vendor+":"+product+":"+version+":*:*:*:*:*:*:*&addOns=cves");
			ModeSwitcher.log("cpe url: " + url);
		} catch (MalformedURLException e1) {
			// TODO Auto-generated catch block
			e1.printStackTrace();
		}
		if(url != null) {*/
			try {
			    //JsonNode jsonNode = objectMapper.readValue(url,JsonNode.class);
				JsonNode jsonNode = objectMapper.readValue(f,JsonNode.class);
				
			    JsonNode cpesNode = jsonNode.at("/result/cpes");
			    if(cpesNode.isArray()) {
			    	ArrayNode cpesArray = (ArrayNode) cpesNode;
			    	//JsonNode cpeNode = cpesArray.get(0);
			    	for (JsonNode cpeNode : cpesArray){
			    		ModeSwitcher.log("Vulnerabilities for " + cpeNode.get("cpe23Uri").asText());
					    JsonNode vulnerabilitiesNode = cpeNode.get("vulnerabilities");
					    if(vulnerabilitiesNode.isArray()){
					        ArrayNode arrayNode = (ArrayNode) vulnerabilitiesNode;
					        
					        //ModeSwitcher.log("CVE             published lastModified Score");
					        //String format = "%-14s %10s %12s %5.1f\n";
					        for(int i = 0; i < arrayNode.size(); i++) {
					            String cveId = arrayNode.get(i).asText(); //CVE-1999-0236
						        //ModeSwitcher.log(cveId); 
						        Vulnerability vuln = getCveDetails(cveId);
						        if(vuln != null) {
						        	/*System.out.format(format, 
						        			cveId,
						        			vuln.getPublishDate(),
						        			vuln.getPublishDate(),
						        			vuln.getScore()
									);*/
						        	addVulnerability(cveId, vuln, "cpe:/:"+vendor+":"+product+":"+version);
						        }
					        }
					        ModeSwitcher.log("----------------------------------------------------------------------------");
					    }
			    	}
			    }
			    
			} catch (Exception e) {
			    e.printStackTrace();
			}
		}
	}
	
	public boolean checkCVEString(String cve) {
		return Pattern.matches(cvePattern, cve);
	}
	
	public Vulnerability getCveDetails(String cve) {
		
		if(checkCVEString(cve)) {
			
			//ModeSwitcher.log("getCveDetails for " + cve);
		
			//https://services.nvd.nist.gov/rest/json/cve/1.0/CVE-2009-0796
			//result.CVE_Items[0].cve.impact.baseMetricV3.cvssV3.baseScore
			//result.CVE_Items[0].cve.publishedDate
			//result.CVE_Items[0].cve.lastModifiedDate
			
			String filename = ModeSwitcher.CVESDIR + File.separator + cve + ".json"; //CVE-2009-0796.json
			File f = new File(filename);
			if(!f.exists()) {
				FileUtils.loadFileFromUrl(nvdBaseUrl + "cve/1.0/" + cve, filename);
			}
			
			if(f.exists()) {
			
			ObjectMapper objectMapper = new ObjectMapper();
			JsonNode baseScoreNode = null;
			JsonNode publishedDateNode = null;
			JsonNode lastModifiedDateNode = null;
			
				try {
				    JsonNode jsonNode = objectMapper.readValue(f,JsonNode.class);
				    JsonNode CVE_ItemsNode = jsonNode.at("/result/CVE_Items");
				    if(CVE_ItemsNode.isArray()) {
				    	ArrayNode CVE_ItemsArray = (ArrayNode) CVE_ItemsNode;
					    JsonNode cveNode = CVE_ItemsArray.get(0);
					    publishedDateNode = cveNode.get("publishedDate"); //2020-08-07T16:15Z
					    lastModifiedDateNode = cveNode.get("lastModifiedDate"); //2020-08-07T16:15Z
					    if (cveNode.has("/impact/baseMetricV3/cvssV3/baseScore")) {
					    	baseScoreNode = cveNode.at("/impact/baseMetricV3/cvssV3/baseScore");
					    }else {
					    	baseScoreNode = cveNode.at("/impact/baseMetricV2/cvssV2/baseScore");
					    }
				    }
				    
				} catch (IOException e) {
				    e.printStackTrace();
				}
				if(publishedDateNode != null && lastModifiedDateNode != null) {
					return new Vulnerability(
							publishedDateNode.asText("9999-12-31T12:00Z").substring(0, 10),
							lastModifiedDateNode.asText("9999-12-31T12:00Z").substring(0, 10),
							cve,
							baseScoreNode.asDouble(0)
					);
				}
			}
		}else {
			ModeSwitcher.log("cve " + cve + " is invalid!");
		}
		return null;
	}	
	
	public boolean addVulnerability(String date, String cveId, double score, String cpe) {
		Vulnerability vuln = new Vulnerability(date, cveId, score);
		addVulnerability(cveId, vuln, cpe);
		return false;
	}
	
	public void addVulnerability(String date, String modifiedDate, String cveId, double score, String cpe) {
		Vulnerability vuln = new Vulnerability(date, modifiedDate, cveId, score);
		addVulnerability(cveId, vuln, cpe);
	}
	
	public boolean addVulnerability(String cveId, Vulnerability vuln, String cpe) {
		boolean changes = false;
		DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy");
		if(checkCVEString(cveId) && vuln != null) {
			if(vuln.getPublishDate().getYear() < ModeSwitcher.getOS().getReleaseDate().getYear()) {
				ModeSwitcher.log("WARNING: "+ cveId + " was ignored, because it was published "+ formatter.format(vuln.getPublishDate()) + " before the release year of the distribution");
			}else {
				if(this.vulnerabilities.containsKey(cveId)){
					System.out.print("[" + cveId + "] already known");
					if(this.vulnerabilities.get(cveId).getModifiedDate() == null) {
						//do nothing
						ModeSwitcher.log(", nothing changed!");
					}else { // this.vulnerabilities.get(cveId).getModifiedDate() != null
						if(vuln.getModifiedDate() != null && this.vulnerabilities.get(cveId).getModifiedDate().isEqual(vuln.getModifiedDate())) {
							ModeSwitcher.log(", nothing changed!");
						}else {
							ModeSwitcher.log(", override with newer version!");
							changes = putVulnerability(cveId, vuln, cpe);
						}
					}
				}else { //new
					changes = putVulnerability(cveId, vuln, cpe);
				}
			}
		}
		return changes;
	}
	
	private boolean putVulnerability(String cveId, Vulnerability vuln, String cpe){
		DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
		this.vulnerabilities.put(cveId, vuln); //override existing
		if(this.software.containsKey(cpe)) {
			Software sw = this.software.get(cpe);
			sw.addVulnerability(vuln);
			ModeSwitcher.log(formatter.format(vuln.getPublishDate()) + " " + sw + " [" + cveId + "] new vulnerability with score:" + vuln.getScore());
			return true;
		}else {
			ModeSwitcher.log("WARNING: " + cpe + " not found");
		}
		return true;
	}
		
	public boolean addPatch(String date, String cveId, String cpe) {
		if(this.vulnerabilities.containsKey(cveId)) {
			if(vulnerabilities.get(cveId).getResolvedDate() != null) {
				ModeSwitcher.log("[" + cveId + "] already resolved!");
			}else {
				vulnerabilities.get(cveId).setResolvedDate(date);
				if(this.software.containsKey(cpe)) {
					Software sw = this.software.get(cpe);
					ModeSwitcher.log(date + " " + sw + " [" + cveId + "] new patch");
					return true;
				}else {
					ModeSwitcher.log("WARNING: " + cpe + " not found");
				}
			}
		}else {
			ModeSwitcher.log("WARNING: Could not add patch! " + cveId + " not found");
		}
		return false;
	}

	public void setVulnerabilities(HashMap<String, Vulnerability> vulnerabilities) {
		this.vulnerabilities = vulnerabilities;
	}
	
}
