# Scripts to reproduce the results from
#
# T. Richter, R Ulrich, M. Janczyk:
#    "Diffusion models with time-dependent parameters:
#     Comparing the computation effort and accuracy
#     of different numerical methods"
#
# Thomas Richter
# Otto-von-Guericke University of Magdeburg
# 39106 Magdeburg, Germany
# thomas.richter@ovgu.de
#
# You can use this code under ther terms of the
# Creative Commons Attribution 4.0 License


import numpy as np
from PythonTools import tools
from PythonTools import kfe
from PythonTools import randomwalk
from PythonTools import integral
import matplotlib.pyplot as plt
import time


### Model defining Case 4,
class TestCase4:
   
    sigma = 4
    
    def mu(self,t,params):
        return (0.5+params['Adrift']/params['tau']*np.exp(1.0-t/params['tau'])*(1-t/params['tau']))

    def muINT(self,t,params):
        return 0.5*t/params['tau']*(2.*params['Adrift']*np.exp(1.-t/params['tau'])+params['tau'])

    def b(self,t,parameters = None):
        return 75.0 * (1.0-0.6*(t)/(t+150.0))

    def dt_b(self,t,parameters = None):
        return 75.0 * (-0.6) * (1.0/(t+150.0) - t/(t+150.0)/(t+150.0))


model = TestCase4()


### Parameters to define the time-depending drift
# For TestCase 2 they are fixed, no parameter identification
params = {
    'mu0'    : 0.5,
    'Adrift' : 20,
    'tau'    : 150.0,
    
    # alpha=0 for initial value at x=0
    'alpha'  : 0.0,
    }

disc = {
        'T' : 1000,
        'dt' : 0,
        'dx' : 0
        }


# space Discretization parameters used in Table 1.
NX_kfe = np.array([10,20,40,80,160])#,320,640])  ## Number of spatial steps

# step size for kfe
DX_kfe = 2.0/NX_kfe                           ## step sizes dx (based on ALE-Inverval [-1,1])


## TT_kfe/DT_kfe are the number of time steps / step sizes used for the KFE simulations. 
NT_kfe     = 5*NX_kfe
DT_kfe     = disc['T'] / NT_kfe


## Time steps for Random Walks
NT_rw      = NX_kfe*NX_kfe//4*3
DT_rw      = 1000.0/NT_rw

DX_rw = np.sqrt(DT_rw) * model.sigma
NX_rw = (150.0/DX_rw).astype(int)
DX_rw = 150.0/NX_rw




print('Simulation with varying boundary on the fixed reference domain\n')


### Compute C++ - Reference Solution
disc['dt'] = 0.05
disc['dx'] = 0.02
disc['theta'] = 0.5+0.*disc['dt']
print('Compute Reference Solution for C++-Comparison')

[pdf_u_ref,pdf_l_ref,fullsolution]   = kfe.kfe_ale(model, disc, params)
cdf_u_ref   = tools.pdf2cdf(pdf_u_ref, disc)  # cdf upper
cdf_l_ref   = tools.pdf2cdf(pdf_l_ref, disc)  # cdf lower

np.savetxt('results/testcase4-refu-cdf.txt',np.array([np.linspace(0,1000,len(cdf_u_ref)),cdf_u_ref]).transpose())
np.savetxt('results/testcase4-refl-cdf.txt',np.array([np.linspace(0,1000,len(cdf_l_ref)),cdf_l_ref]).transpose())


### Compute Reference Solution
disc['dt'] = DT_rw[-1]/4
disc['dx'] = DX_kfe[-1]/4.
disc['theta'] = 0.5+0.*disc['dt']

if params['alpha'] == 0:   # stabilized version of CN for Diriac initial
    disc['theta'] = 0.5+0.01*disc['dt']
    
print('Compute the reference solution using dx={0:3.3f} and dt={1:3.3f}'
      .format(disc['dx'], disc['dt']))

[pdf_u_ref,pdf_l_ref,fs]   = kfe.kfe_ale(model, disc, params)
cdf_u_ref   = tools.pdf2cdf(pdf_u_ref, disc)  # cdf upper
cdf_l_ref   = tools.pdf2cdf(pdf_l_ref, disc)  # cdf lower


### run all simulations and print and save the errors to file
LOG = []

print('Random Walk')
print('dx\t\tdt\t\ttime\t\tErr')

disc['bmax'] = 75

for i in range(len(DX_rw)):
    disc['dt']   = DT_rw[i]/4
    disc['dx']   = DX_rw[i]
    t1 = time.perf_counter()
    [pdf_u,pdf_l]=randomwalk.randomwalk_variable(model,disc,params)
    t2 = time.perf_counter()

    cdf_u = tools.pdf2cdf(pdf_u, disc)  # cdf upper
    cdf_l = tools.pdf2cdf(pdf_l, disc)  # cdf lower
    
#    plt.plot(np.linspace(0,1000,len(pdf_u)),pdf_u)
#    plt.plot(np.linspace(0,1000,len(pdf_u_ref)),pdf_u_ref)
#    plt.show()

    # Compute Error in the cummulative probability density of reaching upper boundary
    Ep_u = tools.computeerrors(pdf_u_ref,pdf_u)
    Ep_l = tools.computeerrors(pdf_l_ref,pdf_l)

    Ec_u = tools.computeerrors(cdf_u_ref,cdf_u)
    Ec_l = tools.computeerrors(cdf_l_ref,cdf_l)

    print('{0:2.5f}\t{1:2.5f}\t{2:2.5f}\t{3:2.5f}\t{4:2.5f}\t{5:2.5f}\t{6:2.5f}'.format(disc['dx'], disc['dt'], t2-t1, Ec_u, Ec_l, Ep_u, Ep_l ))
    LOG.append([disc['dx'],disc['dt'] ,t2-t1, Ec_u, Ec_l, Ep_u, Ep_l])

np.savetxt('results/testcase4-rw.txt',LOG) # save to file
LOG = []



print('KFE')
print('dx\t\tdt\t\ttime\t\tErr_u\tErr_l')

disc['bmax'] = 0

for i in range(len(DX_kfe)):
    disc['dt']   = DT_kfe[i]/2
    disc['dx']   = DX_kfe[i]/2
    disc['theta'] = 0.5+0.0*disc['dt']
    t1 = time.perf_counter()
    [pdf_u,pdf_l,fs]   = kfe.kfe_ale(model, disc, params)
    t2 = time.perf_counter()

    cdf_u = tools.pdf2cdf(pdf_u, disc)  # cdf upper
    cdf_l = tools.pdf2cdf(pdf_l, disc)  # cdf lower
    
    
    # Compute Error in the probability density of reaching upper boundary
    Ep_u = tools.computeerrors(pdf_u_ref,pdf_u)
    Ep_l = tools.computeerrors(pdf_l_ref,pdf_l)

    Ec_u = tools.computeerrors(cdf_u_ref,cdf_u)
    Ec_l = tools.computeerrors(cdf_l_ref,cdf_l)

    print('{0:2.5f}\t{1:2.5f}\t{2:2.5f}\t{3:2.5f}\t{4:2.5f}\t{5:2.5f}\t{6:2.5f}'.format(disc['dx'], disc['dt'], t2-t1, Ec_u, Ec_l, Ep_u, Ep_l))
    LOG.append([disc['dx'],disc['dt'] ,t2-t1, Ec_u, Ec_l, Ep_u, Ep_l])

np.savetxt('results/testcase4-kfe.txt',LOG) # save to file
LOG = []

# Integral
print('\nAccuracy and computational effort for the Integral approach:')
print('dx      dt       t       cdf_u    cdf_l    pdf_u   pdf_l')
for i in range(len(DT_kfe)):
    disc['dt']   = DT_kfe[i]
    disc['dx']   = -1
    disc['nX']   = NX_kfe[i]*2
            
    # Solve
    t0 = time.perf_counter()
    [pdf_u,pdf_l]=integral.imzero(model,params, disc)
    cdf_u = tools.pdf2cdf(pdf_u, disc)  # cdf upper
    cdf_l = tools.pdf2cdf(pdf_l, disc)  # cdf lower
    
    t1 = time.perf_counter()
    
    # Compute Errors
    Ec_u = tools.computeerrors(cdf_u_ref,cdf_u)
    Ec_l = tools.computeerrors(cdf_l_ref,cdf_l)
    Ep_u  = tools.computeerrors(pdf_u_ref,pdf_u)
    Ep_l  = tools.computeerrors(pdf_l_ref,pdf_l)

    LOG.append([disc['dx'],disc['dt'] ,t1-t0, Ec_u, Ec_l, Ep_u, Ep_l])
    print('{0:3.3f}\t{1:3.3f}\t{2:2.4f}\t{3:2.4f}\t{4:2.4f}\t{5:2.4f}\t{6:2.4f}'
           .format(disc['dx'],disc['dt'] ,t1-t0, Ec_u, Ec_l, Ep_u, Ep_l))

np.savetxt('results/testcase4-int.txt',LOG) # save to file
LOG = []


