# Scripts to reproduce the results from
#
# T. Richter, R Ulrich, M. Janczyk:
#    "Diffusion models with time-dependent parameters:
#     Comparing the computation effort and accuracy
#     of different numerical methods"
#
# Thomas Richter
# Otto-von-Guericke University of Magdeburg
# 39106 Magdeburg, Germany
# thomas.richter@ovgu.de
#
# You can use this code under ther terms of the
# Creative Commons Attribution 4.0 License


import numpy as np
from PythonTools import tools
from PythonTools import kfe
from PythonTools import randomwalk
from PythonTools import integral
import matplotlib.pyplot as plt
import time


### Model defining Case 4,
class TestCase4:
   
    sigma = 4
    
    def mu(self,t,params):
        return (0.+0*params['Adrift']/params['tau']*np.exp(1.0-t/params['tau'])*(1-t/params['tau']))

    def b(self,t,parameters = None):
        return 75.0 * (1.0-0.*(t)/(t+150.0))

    def dt_b(self,t,parameters = None):
        return 75.0 * (-0.) * (1.0/(t+150.0) - t/(t+150.0)/(t+150.0))


model = TestCase4()


### Parameters to define the time-depending drift
# For TestCase 2 they are fixed, no parameter identification
params = {
    'mu0'    : 0.5,
    'Adrift' : 20,
    'tau'    : 150.0,
    
    # alpha=0 for initial value at x=0
    'alpha'  : 0.0,
    }

disc = {
        'T' : 10,
        'dt' : 0,
        'dx' : 0
        }


# space Discretization parameters used in Table 1.
NX_kfe = np.array([10,20,40,80,160])#,320,640])  ## Number of spatial steps

# step size for kfe
DX_kfe = 2.0/NX_kfe                           ## step sizes dx (based on ALE-Inverval [-1,1])


## TT_kfe/DT_kfe are the number of time steps / step sizes used for the KFE simulations. 
NT_kfe     = 5*NX_kfe
DT_kfe     = disc['T'] / NT_kfe


## Time steps for Random Walks
NT_rw      = NX_kfe*NX_kfe//4*3
DT_rw      = 1000.0/NT_rw

DX_rw = np.sqrt(DT_rw) * model.sigma
NX_rw = (150.0/DX_rw).astype(int)
DX_rw = 150.0/NX_rw




print('Simulation with varying boundary on the fixed reference domain\n')


### Compute C++ - Reference Solution
disc['dt'] = 1.0
disc['dx'] = 0.025
disc['theta'] = 0.5
print('Compute Reference Solution for C++-Comparison')

nnx = int(2./disc['dx']+1.e-8)
nnt = int(disc['T']/disc['dt']+1.e-8)

fs = np.zeros( (nnt+1,nnx+1) )
[pdf_u_ref,pdf_l_ref,fullsolution]   = kfe.kfe_ale(model, disc, params, fs)
cdf_u_ref   = tools.pdf2cdf(pdf_u_ref, disc)  # cdf upper
cdf_l_ref   = tools.pdf2cdf(pdf_l_ref, disc)  # cdf lower

fs = fs[:nnt+1,2*nnx//6:(4*nnx)//6]

### heatmap of sol
from mpl_toolkits import mplot3d
tt = np.linspace(0, disc['T'],nnt+1)
xx = np.linspace(-.333, .333, 2*nnx//6+1)
ttt,xxx = np.meshgrid(tt, xx)

print(ttt.shape,xxx.shape,fs.shape)
fig = plt.figure()
ax = plt.axes(projection='3d')
ax.set_aspect('auto')
#fs[0,:] = fs[0,:]/2
ax.set_box_aspect((1,1,0.5))
ax.plot_surface(ttt, xxx, np.transpose(fs),cmap='turbo', edgecolor='none')
ax.set_title('Surface plot')
locs, labels = plt.yticks()  #
plt.yticks([-0.33,-0.167,0,0.167,0.33], ['-25','-12.5', '0', '12.5','25'])

plt.title('')
plt.xlabel('time')
plt.ylabel('x')
plt.savefig('cn-0.025.png',dpi=300)
plt.show()

