# Scripts to reproduce the results from
#
# T. Richter, R Ulrich, M. Janczyk:
#    "Diffusion models with time-dependent parameters:
#     Comparing the computation effort and accuracy
#     of different numerical methods"
#
# Thomas Richter
# Otto-von-Guericke University of Magdeburg
# 39106 Magdeburg, Germany
# thomas.richter@ovgu.de
#
# You can use this code under ther terms of the
# Creative Commons Attribution 4.0 License

import numpy as np
from PythonTools import tools
from PythonTools import kfe
from PythonTools import randomwalk
from PythonTools import integral
import matplotlib.pyplot as plt
import time

plt.rcParams['text.usetex'] = True


### Model defining Case 3
# KFE and random ralks are used to simulate the problem
# for varying values of tau

class TestCase2:
   
    sigma = 4
    alpha = 5.
    def mu(self,t,params):
        return 0.5+params['Adrift']/params['tau']*np.exp(1.0-t/params['tau'])*(1-t/params['tau'])

 
    def muINT(self,t,params):
        return 0.5*t/params['tau']*(2.*params['Adrift']*np.exp(1.-t/params['tau'])+params['tau'])
    
    def b(self,t,parameters = None):
        return 75.0 + t * 0 # 

    def dt_b(self,t,parameters = None):
        return 0./8.0*np.exp(5.0-t/80.0)/((1.0+np.exp(5.0-t/80.0))**2.0)


model = TestCase2()


### Parameters to define the time-depending drift
# For TestCase 2 they are fixed, no parameter identification
params = {
    'mu0'    : 0.5,
    'Adrift' : 40,
    'tau'    : 150.0,
    
    # alpha=0 for initial value at x=0
    'alpha'  : 5.0,
    }

disc = {
        'T' : 1000,
        'dt' : 0,
        'dx' : 0
        }




dt = 1.0
nt = int(disc['T']/dt + 1.e-8)

dx_rw = np.sqrt(dt) * model.sigma
nx    = (150.0/dx_rw).astype(int)
nx    = 2*(nx//2-1)
dx_rw = 150.0/nx
dx_kfe = 2.0/nx/2
dx_inte = 150.0/nx

### run all simulations and print and z̄ the errors to file
LOG = []

taus = 10


print('Random walks\ntau\tdx\t\tdt\t\ttime\t\tErr')

for tau in [160, 80, 40, 20, 10, 5]:
    params['tau'] = tau

    ### Compute Reference solution for random walks
    disc['dt'] = dt     / 8.
    disc['dx'] = dx_kfe / 8.
    disc['theta'] = 0.5#+0.2*disc['dt']
    
    
    [pdf_u_ref,pdf_l_ref,fs]   = kfe.kfe_ale(model, disc, params)
    cdf_u_ref   = tools.pdf2cdf(pdf_u_ref, disc)  # cdf upper
    cdf_l_ref   = tools.pdf2cdf(pdf_l_ref, disc)  # cdf lower

    disc['bmax'] = 75
    disc['dt']   = dt
    disc['dx']   = dx_rw
    
    t1 = time.perf_counter()
    [pdf_u,pdf_l]=randomwalk.randomwalk_variabledrift(model,disc,params)
    t2 = time.perf_counter()

    cdf_u = tools.pdf2cdf(pdf_u, disc)  # cdf upper
    cdf_l = tools.pdf2cdf(pdf_l, disc)  # cdf lower

    if tau==taus:
        pdfurw = pdf_u[:51]
        pdflrw = pdf_l[:51]
    
    # Compute Error in the cummulative probability density of reaching upper boundary
    Ep_u = tools.computeerrors(pdf_u_ref,pdf_u)
    Ep_l = tools.computeerrors(pdf_l_ref,pdf_l)

    Ec_u = tools.computeerrors(cdf_u_ref,cdf_u)
    Ec_l = tools.computeerrors(cdf_l_ref,cdf_l)

    print('{0}\t{1:2.5f}\t{2:2.5f}\t{3:2.5f}\t{4:2.5f}\t{5:2.5f}\t{6:2.5f}\t{7:2.5f}'.format(params['tau'], disc['dx'], disc['dt'], t2-t1, Ec_u, Ec_l, Ep_u, Ep_l ))
    LOG.append([params['tau'] , disc['dx'],disc['dt'] ,t2-t1, Ec_u, Ec_l, Ep_u, Ep_l])

np.savetxt('results/testcase3-rw.txt',LOG) # save to file
LOG = []

print('KFE\ntau\tdx\t\tdt\t\ttime\t\tErr')

for tau in [160, 80, 40, 20, 10, 5]:
    params['tau'] = tau

    ### Compute Reference solution for random walks
    disc['dt'] = dt     / 8.
    disc['dx'] = dx_kfe / 8.
    disc['theta'] = 0.5#+0.2*disc['dt']
    
    [pdf_u_ref,pdf_l_ref,fs]   = kfe.kfe_ale(model, disc, params)
    cdf_u_ref   = tools.pdf2cdf(pdf_u_ref, disc)  # cdf upper
    cdf_l_ref   = tools.pdf2cdf(pdf_l_ref, disc)  # cdf lower

    np.savetxt('results/testcase3-refu-cdf-{0}.txt'.format(tau),np.array([np.linspace(0,1000,len(cdf_u_ref)),cdf_u_ref]).transpose())
    np.savetxt('results/testcase3-refl-cdf-{0}.txt'.format(tau),np.array([np.linspace(0,1000,len(cdf_l_ref)),cdf_l_ref]).transpose())


    disc['bmax'] = 75

    disc['dt']   = dt
    disc['dx']   = dx_kfe
    disc['theta'] = 0.5#+0.2*disc['dt']
    
    
    t1 = time.perf_counter()
    [pdf_u,pdf_l,fs]=kfe.kfe_ale(model,disc,params)
    t2 = time.perf_counter()

    cdf_u = tools.pdf2cdf(pdf_u, disc)  # cdf upper
    cdf_l = tools.pdf2cdf(pdf_l, disc)  # cdf lower

    if tau==taus:
        pdfukferef = pdf_u_ref[:401]
        pdflkferef = pdf_l_ref[:401]
        pdfukfe = pdf_u[:51]
        pdflkfe = pdf_l[:51]

    
    # Compute Error in the cummulative probability density of reaching upper boundary
    Ep_u = tools.computeerrors(pdf_u_ref,pdf_u)
    Ep_l = tools.computeerrors(pdf_l_ref,pdf_l)

    Ec_u = tools.computeerrors(cdf_u_ref,cdf_u)
    Ec_l = tools.computeerrors(cdf_l_ref,cdf_l)

    print('{0}\t{1:2.5f}\t{2:2.5f}\t{3:2.5f}\t{4:2.5f}\t{5:2.5f}\t{6:2.5f}\t{7:2.5f}'.format(params['tau'], disc['dx'], disc['dt'], t2-t1, Ec_u, Ec_l, Ep_u, Ep_l ))
    LOG.append([params['tau'], disc['dx'],disc['dt'] ,t2-t1, Ec_u, Ec_l, Ep_u, Ep_l])

np.savetxt('results/testcase3-kfe.txt',LOG) # save to file
LOG = []





print('Integral\ntau\tdx\t\tdt\t\ttime\t\tErr')

for tau in [160, 80, 40, 20, 10, 5]:
    params['tau'] = tau

    ### Compute Reference solution for random walks
    disc['dt'] = dt     / 8.
    disc['dx'] = dx_kfe / 8.
    disc['theta'] = 0.5#+0.2*disc['dt']

    
    [pdf_u_ref,pdf_l_ref,fs]   = kfe.kfe_ale(model, disc, params)
    cdf_u_ref   = tools.pdf2cdf(pdf_u_ref, disc)  # cdf upper
    cdf_l_ref   = tools.pdf2cdf(pdf_l_ref, disc)  # cdf lower


    disc['bmax'] = 75

    disc['dt']   = dt
    disc['dx']   = dx_rw
    disc['nX']   = nx
    disc['theta'] = 0.5#+0.2*disc['dt']
    disc['T'] = 1000.
    
    
    t1 = time.perf_counter()
    [pdf_u,pdf_l]= integral.im(model,params,disc)
    t2 = time.perf_counter()

    cdf_u = tools.pdf2cdf(pdf_u, disc)  # cdf upper
    cdf_l = tools.pdf2cdf(pdf_l, disc)  # cdf lower

    if tau==taus:
        pdfukferef = pdf_u_ref[:401]
        pdflkferef = pdf_l_ref[:401]
        pdfuint = pdf_u[:51]
        pdflint = pdf_l[:51]

    
    # Compute Error in the cummulative probability density of reaching upper boundary
    Ep_u = tools.computeerrors(pdf_u_ref,pdf_u)
    Ep_l = tools.computeerrors(pdf_l_ref,pdf_l)

    Ec_u = tools.computeerrors(cdf_u_ref,cdf_u)
    Ec_l = tools.computeerrors(cdf_l_ref,cdf_l)

    print('{0}\t{1:2.5f}\t{2:2.5f}\t{3:2.5f}\t{4:2.5f}\t{5:2.5f}\t{6:2.5f}\t{7:2.5f}'.format(params['tau'], disc['dx'], disc['dt'], t2-t1, Ec_u, Ec_l, Ep_u, Ep_l ))
    LOG.append([params['tau'], disc['dx'],disc['dt'] ,t2-t1, Ec_u, Ec_l, Ep_u, Ep_l])

np.savetxt('results/testcase3-int.txt',LOG) # save to file
LOG = []





plt.figure(figsize=(10, 2))

plt.subplots_adjust(left=None, bottom=None, right=None, top=None, wspace=0.15, hspace=0.4)

plt.subplot(121)
plt.xlabel('$\displaystyle t$', labelpad=-10)

plt.title('Reaching upper threshold (PDF, $\\tau=10$) ')
plt.plot(np.linspace(0,50,51),pdfuint,label='Integral Equation')
plt.plot(np.linspace(0,50,51),pdfukfe,label='Kolmogorov Forward Equation')
plt.plot(np.linspace(0,50,51),pdfurw,label='Random Walks')
plt.plot(np.linspace(0,50,401),pdfukferef,color='gray',label='Exact Solution')
plt.plot(np.zeros(51),'--',color='gray')

#plt.legend()

plt.subplot(122)
plt.xlabel('$\displaystyle t$', labelpad=-10)
plt.title('Reaching lower threshold (PDF, $\\tau=10$)')
plt.plot(np.linspace(0,50,51),pdflkfe)
plt.plot(np.linspace(0,50,51),pdflint)
plt.plot(np.linspace(0,50,51),pdflrw)
plt.plot(np.linspace(0,50,401),pdflkferef,color='gray')
plt.plot(np.zeros(51),'--',color='gray')
current_values = plt.gca().get_yticks()
plt.gca().set_yticklabels(['{:2.0e}'.format(x) for x in current_values])

plt.savefig('pics/testcase3-2.png', dpi=300,bbox_inches = 'tight')

plt.show()
