#!/usr/bin/env python3
# -*- coding: utf-8 -*-


# Scripts to reproduce the results from
#
# T. Richter, R Ulrich, M. Janczyk:
#    "Diffusion models with time-dependent parameters:
#     Comparing the computation effort and accuracy
#     of different numerical methods"
#
# Thomas Richter
# Otto-von-Guericke University of Magdeburg
# 39106 Magdeburg, Germany
# thomas.richter@ovgu.de
#
# You can use this code under ther terms of the
# Creative Commons Attribution 4.0 License



import numpy as np
import time
from PythonTools import tools
from PythonTools import kfe
from PythonTools import randomwalk 
from PythonTools import integral 
import matplotlib.pyplot as plt

#################################################
# Reproduces Test Case I of
# Richter, Janczyk, Ulrich: Diffusion models with time-dependent
# parameters: Analysis of computational
# effort and accuracy of different numerical
# methods
#


# The class collects are the necessary information to define the model
# problem of test case 1.
class TestCase1:
    sigma = 4
    b     = 75
    mu    = 0.5
    alpha = 5
   
model = TestCase1()


### Parameters of the discretization
disc = {
        'T' : 1000,
        'dt' : 0,
        'dx' : 0
        }

### Defines the spatial discretization parameters
# We use the step size dx = 2b/xx = 150/xx
# dx = [10,5,2.5,1.25,0.625]
XX = np.array([16,32,64,128,256])
XX = np.array([10,20,40,80,160])*2
DX_kfe = 2*model.b / XX   


### Defines the time step sizes
# We specifiy in TT the number of steps in [0,T=1000]
# For the KFE we choose approximately dt = dx
#   which, to get integer numbers, corresponds to TT = 6 XX
# For the random walk approach we must satisfy the parabolic
#   time step condition dt < dx^2 sigma^2
#   With dx = 2b/xx and dt = T/tt this relates to
#   TT = XX^2/(4b^2) * sigma^2
TT_kfe = XX*10  
DT_kfe = disc['T'] / TT_kfe

### Random walks must satisfy the parabolic cfl condition
# dT  < dx^2 / sigma^2
# NT  > 1000 * sigma^2 / (dx^2) = sigma^2 / (2b/X)^2 = sigma^2 X^2 / (2b)^2
TT_rw  = XX*XX #32*32*XX*XX*model.sigma*model.sigma//(4*model.b*model.b)

### Compute C++ - Reference Solution
disc['dt'] = 0.25
disc['dx'] = 0.25
print('Compute Reference Solution for C++-Comparison with dx={0:4.4f} and dt={1:4.4f}'.format(disc['dx'], disc['dt'] ))
[pdf_u_ref,pdf_l_ref]   = kfe.kfe(model, disc)
cdf_u_ref   = tools.pdf2cdf(pdf_u_ref, disc)  # cdf upper
cdf_l_ref   = tools.pdf2cdf(pdf_l_ref, disc)  # cdf lower
np.savetxt('results/testcase1-refu-cdf.txt',np.array([np.linspace(0,1000,len(cdf_u_ref)),cdf_u_ref]).transpose())
np.savetxt('results/testcase1-refl-cdf.txt',np.array([np.linspace(0,1000,len(cdf_l_ref)),cdf_l_ref]).transpose())


### Reference solution
# We start by computing an accurate solution for comparison. Nr. of reference time-steps
# must be a multiple of the KFE and the RW solution 
print(TT_rw[-1],TT_kfe[-1])


# We compute density distributions for the reference discretization
#disc['dt'] = DT_kfe[-1]/4
#disc['dx'] = DX_kfe[-1]/4
disc['dt'] = 1000/TT_rw[-1]
disc['dx'] = DX_kfe[-1]


print('Computing the reference solution with KFE and dx={0:4.4f}, dt={1:4.4f}'.format(disc['dx'], disc['dt']))
[pdf_u_ref,pdf_l_ref]=kfe.kfe(model, disc)
cdf_u_ref   = tools.pdf2cdf(pdf_u_ref, disc)  # cdf upper
cdf_l_ref   = tools.pdf2cdf(pdf_l_ref, disc)  # cdf lower


### Plot the PDF and the CDF

#plt.title('Probability density functions of first passage times')
#plt.plot(np.linspace(0,disc['T'],len(cdf_l_ref)), pdf_l_ref, label='lower margin')
#plt.plot(np.linspace(0,disc['T'],len(cdf_l_ref)), pdf_u_ref, label='upper margin')
#plt.legend()
#plt.show()

#plt.title('Cummulative probability density functions of first passage times')
#plt.plot(np.linspace(0,disc['T'],len(cdf_l_ref)), cdf_l_ref, label='lower margin')
#plt.plot(np.linspace(0,disc['T'],len(cdf_l_ref)), cdf_u_ref, label='upper margin')
#plt.legend()
#plt.show()


### run all simulations and print and save the errors to file
LOG = []

# KFE
print('\nAccuracy and computational effort for the KFE approach:')
print('dx      dt       t       cdf_u    cdf_l    pdf_u   pdf_l')
for ti in range(len(TT_kfe)-1):
    nT = TT_kfe[ti]
    disc['dt'] = 1000/nT
        
    nX = XX[ti]
    disc['dx'] = 2.0*model.b/nX
    
    # Solve
    t0 = time.perf_counter()
    [pdf_u,pdf_l]=kfe.kfe(model, disc)
    cdf_u = tools.pdf2cdf(pdf_u, disc)  # cdf upper
    cdf_l = tools.pdf2cdf(pdf_l, disc)  # cdf lower

    t1 = time.perf_counter()
    
    # Compute Errors
    Ec_u = tools.computeerrors(cdf_u_ref,cdf_u)
    Ec_l = tools.computeerrors(cdf_l_ref,cdf_l)
    Ep_u  = tools.computeerrors(pdf_u_ref,pdf_u)
    Ep_l  = tools.computeerrors(pdf_l_ref,pdf_l)

    LOG.append([disc['dx'],disc['dt'] ,t1-t0, Ec_u, Ec_l, Ep_u, Ep_l])
    print('{0:3.3f}\t{1:3.3f}\t{2:2.4f}\t{3:2.4f}\t{4:2.4f}\t{5:2.4f}\t{6:2.4f}'
           .format(disc['dx'],disc['dt'] ,t1-t0, Ec_u, Ec_l, Ep_u, Ep_l))

np.savetxt('results/testcase1-kfe.txt',LOG) # save to file
LOG = []


# Integral
print('\nAccuracy and computational effort for the Integral approach:')
print('dx      dt       t       cdf_u    cdf_l    pdf_u   pdf_l')
for ti in range(len(TT_kfe)-1):
    nT = TT_kfe[ti]//4
    disc['dt'] = 1000/nT

    nX = XX[ti]
    disc['dx'] = 2*model.b/nX
    disc['nX']   = nX
            
    # Solve
    t0 = time.perf_counter()
    [pdf_u,pdf_l]=integral.imfixed(model, disc)
    cdf_u = tools.pdf2cdf(pdf_u, disc)  # cdf upper
    cdf_l = tools.pdf2cdf(pdf_l, disc)  # cdf lower

    t1 = time.perf_counter()
    
    # Compute Errors
    Ec_u = tools.computeerrors(cdf_u_ref,cdf_u)
    Ec_l = tools.computeerrors(cdf_l_ref,cdf_l)
    Ep_u  = tools.computeerrors(pdf_u_ref,pdf_u)
    Ep_l  = tools.computeerrors(pdf_l_ref,pdf_l)

    LOG.append([disc['dx'],disc['dt'] ,t1-t0, Ec_u, Ec_l, Ep_u, Ep_l])
    print('{0:3.3f}\t{1:3.3f}\t{2:2.4f}\t{3:2.4f}\t{4:2.4f}\t{5:2.4f}\t{6:2.4f}'
           .format(disc['dx'],disc['dt'] ,t1-t0, Ec_u, Ec_l, Ep_u, Ep_l))

np.savetxt('results/testcase1-int.txt',LOG) # save to file
LOG = []




# Random walks

# We recompute the reference solution on a finer temporal mesh as comparing the
# errors only works if the number of time steps are multiples of each other. This
# is just a technical limitation
disc['dt'] = disc['T']/(TT_rw[-1])
disc['dx'] = 2.0*model.b / XX[-1]/4.0
print('\nRecompute a reference solution with KFE and dx={0:4.4f}, dt={1:4.4f}'.format(disc['dx'], disc['dt']))

[pdf_u_ref,pdf_l_ref]=kfe.kfe(model, disc) 
cdf_u_ref = tools.pdf2cdf(pdf_u_ref, disc)  # cdf upper
cdf_l_ref = tools.pdf2cdf(pdf_l_ref, disc)  # cdf lower

print('\nAccuracy and computational effort for the random walk approach:')
print('dx      dt       t       cdf_u    cdf_l    pdf_u   pdf_l')

for ti in range(len(TT_rw)-1):
    nT = TT_rw[ti]
    disc['dt'] = disc['T']/nT
        
    nX = XX[ti]
    disc['dx'] = 2*model.b/nX
    
    
    
    # Solve
    t0 = time.perf_counter()
    [pdf_u,pdf_l]=randomwalk.randomwalk(model, disc)
    cdf_u = tools.pdf2cdf(pdf_u, disc)  # cdf upper
    cdf_l = tools.pdf2cdf(pdf_l, disc)  # cdf lower
    t1 = time.perf_counter()
    
    # Compute Errors
    Ec_u = tools.computeerrors(cdf_u_ref,cdf_u)
    Ec_l = tools.computeerrors(cdf_l_ref,cdf_l)
    Ep_u  = tools.computeerrors(pdf_u_ref,pdf_u)
    Ep_l  = tools.computeerrors(pdf_l_ref,pdf_l)

    LOG.append([disc['dx'],disc['dt'] ,t1-t0, Ec_u, Ec_l, Ep_u, Ep_l])
    print('{0:3.3f}\t{1:3.3f}\t{2:2.4f}\t{3:2.4f}\t{4:2.4f}\t{5:2.4f}\t{6:2.4f}'
           .format(disc['dx'],disc['dt'] ,t1-t0, Ec_u, Ec_l, Ep_u, Ep_l))

np.savetxt('results/testcase1-rw.txt',LOG) # save to file
