#!/usr/bin/env python3
# -*- coding: utf-8 -*-


# Scripts to reproduce the results from
#
# T. Richter, R Ulrich, M. Janczyk:
#    "Diffusion models with time-dependent parameters:
#     Comparing the computation effort and accuracy
#     of different numerical methods"
#
# Thomas Richter
# Otto-von-Guericke University of Magdeburg
# 39106 Magdeburg, Germany
# thomas.richter@ovgu.de
#
# You can use this code under ther terms of the
# Creative Commons Attribution 4.0 License


import numpy as np
from scipy.linalg import solve_banded
import matplotlib.pyplot as plt

### Solve a linear system of equations with a tridiagonal matrix A
# The matrix has the shape
# [1 0 ....... 0]
# [0 b c 0 ... 0]
# [0 a b c 0 . 0]
# [| \ \ \ \ \  ]
# [0 . 0 a b c 0]
# [0 ... 0 a b 0]
# [0 ....... 0 1]
#
def tridiag_fixed(f,a,b,c):
    n = len(f)
        
    # See the SciPy solve_banded tutorial for the strange matrix format
    # https://docs.scipy.org/doc/scipy/reference/generated/scipy.linalg.solve_banded.html
    ab = np.zeros( (3,n ))
    ab[1,:] = b #*np.ones(n)
    ab[1,0] = 1
    ab[1,-1] = 1
    ab[0,2:-1] = c# np.ones(n-3)*c
    ab[2,1:-2] = a#np.ones(n-3)*a
    x = solve_banded( (1,1),ab,f)
    return x

def tridiag(f,a,b,c):
    n = len(f)
        
    # See the SciPy solve_banded tutorial for the strange matrix format
    # https://docs.scipy.org/doc/scipy/reference/generated/scipy.linalg.solve_banded.html
    ab = np.zeros( (3,n ))
    ab[1,:] = b #*np.ones(n)
    ab[1,0] = 1
    ab[1,-1] = 1
    ab[0,2:-1] = c[2:-1]# np.ones(n-3)*c
    ab[2,1:-2] = a[1:-2]#np.ones(n-3)*a
    x = solve_banded( (1,1),ab,f)
    return x

# transforms a pdf to a cdf using 2nd order integration
def pdf2cdf(pdf,disc):
    cdf = disc['dt'] * np.cumsum(pdf)
    cdf[1:] = 0.5*(cdf[:-1]+cdf[1:])
    return cdf


# compute the maximum error between two vectors, scaled by the maximum of the reference
def computeerrors(ref,low):
    nR = ref.shape[0]-1
    nL = low.shape[0]-1
    
    assert nR%nL==0,'reference steps not a multiple'

    st =int(nR//nL)
    err = 100.0*np.abs(ref[::st]-low)

    return np.linalg.norm(err,np.inf)/np.linalg.norm(ref,np.inf)


# adds the residual distribution with mean muR and standard deviation sigmaR
def add_residual(pdf, disc, p):
    nT = len(pdf)
    tt = np.linspace(0,disc['T'],nT)  # time steps from 0 to T 
    # normal distribution with mean muR and deviation sigmaR
    nv =  1.0/p['sigmaR']/np.sqrt(2.0*np.pi) * np.exp( -0.5*((tt-p['muR'])/p['sigmaR'])**2.0)
    return np.convolve(pdf,nv)*disc['dt']


# returns the mean passage time
def mean_passage_time(pdf,disc):
    T = disc['dt'] * len(pdf)
    return disc['dt'] * np.sum(np.linspace(0,T,len(pdf))*pdf)

