#!/usr/bin/env python3
# -*- coding: utf-8 -*-


# Scripts to reproduce the results from
#
# T. Richter, R Ulrich, M. Janczyk:
#    "Diffusion models with time-dependent parameters:
#     Comparing the computation effort and accuracy
#     of different numerical methods"
#
# Thomas Richter
# Otto-von-Guericke University of Magdeburg
# 39106 Magdeburg, Germany
# thomas.richter@ovgu.de
#
# You can use this code under ther terms of the
# Creative Commons Attribution 4.0 License

import numpy as np
from PythonTools import tools
import matplotlib.pyplot as plt


# solves the Kolmogorov forward equation using fixed margins (-b,+b) 
#
def kfe(model,disc):
    T = disc['T']
    assert T>0,    'max time must be postive'
    
    b = model.b
    assert b>0,    'threshold must be positive'
    
    dt = disc['dt']
    assert dt>0,   'time steps must be postive'
    
    nT = int(T/dt+1.e-10)
    
    dx = disc['dx']
    assert dx>0,   'space steps must be positive'
    
    nX = int(2*b/dx+1.e-12)
    assert nX%2==0,'number of space steps must be even'

    L  = 1./dx * model.sigma*model.sigma/2.0             # laplace matrix
    x = np.zeros( (nX+1) )    # solution vector
    f = np.zeros( (nX+1) )    # rhs vector

    if model.alpha>0:
        xx = np.linspace(0,1.,nX+1) # initial value Beta-Distribution
        x = xx**(model.alpha-1)*(1.0-xx)**(model.alpha-1)
        x=x/np.sum(x)               # normalize
    else:
        x[nX//2] = 1


    pdf_u = np.zeros(nT+1)
    pdf_l = np.zeros(nT+1)

    for n in range(nT):       # time loop
        theta = 0.5 # Crank-Nicolson without any stabilization
                
        f = 2.0/3.0*dx*x
        f[1:-1] += 1.0/6.0 * dx * x[0:-2]
        f[1:-1] += 1.0/6.0 * dx * x[2:]
        
        f[1:-1] = f[1:-1] + (theta-1)*dt*(   -L-0.5*model.mu)*x[0:-2]
        f[1:-1] = f[1:-1] + (theta-1)*dt*(2.0*L)             *x[1:-1]
        f[1:-1] = f[1:-1] + (theta-1)*dt*(   -L+0.5*model.mu)*x[2:]

        assert f[0]  == 0, 'f nicht null?'
        assert f[-1] == 0, 'f nicht null?'

        x=tools.tridiag_fixed(f,
                1.0/6.0*dx + dt*theta * (   -L - 0.5*model.mu),
                2.0/3.0*dx + dt*theta * (2.0*L  ),
                1.0/6.0*dx + dt*theta * (   -L + 0.5*model.mu))
        pdf_u[n+1]=.5*model.sigma**2.0/dx/dx*(3.0*x[-2] - 1.5*x[-3] + 1.0/3.*x[-4])
        pdf_l[n+1]=.5*model.sigma**2.0/dx/dx*(3.0*x[1]  - 1.5*x[2]  + 1.0/3.*x[3] )


    # We ignore all trials that do not reach a threshold
    scale = np.sum(pdf_u)*dt + np.sum(pdf_l) * dt
    return [pdf_u/scale,pdf_l/scale]


# Solves the Kolmogorov Forward Equation with variable boundaries
#
def kfe_ale(model,disc,params,fullsolution=np.array([[]])):
    T    = disc['T']
    
    dt   = disc['dt']
    dx   = disc['dx']
    
    if 'theta' in disc:
        theta = disc['theta']
    else:
        theta = 0.5
#    assert theta>=0.5, 'time stepping parameter theta must be larger or equal to 0.5'
    
    nT   = int(T/dt+1.e-8)
    assert abs(dt * nT - T)<1.e-10,'Final time not multiple of time step'
    nX   = int(2./dx+1.e-8)
    assert abs(dx * nX - 2)<1.e-10,'2 (ref domain size) not multiple of space step'
       
    
    assert T>0,    'max time must be postive'
    assert nT>0,   'time steps must be postive'
    assert nX>0,   'space steps must be positive'
    assert nX%2==0,'number of space steps must be even'

    dt = T/nT       # time step size
    dx = 2.0/nX     # space step sizes on [-1,1]

    x = np.zeros( (nX+1) )      # solution vector
    f = np.zeros( (nX+1) )      # rhs vector

    if params['alpha']>0:
        xx = np.linspace(0,1.,nX+1) # initial value Beta-Distribution
        x = xx**(params['alpha']-1)*(1.0-xx)**(params['alpha']-1)
        x=x/np.sum(x)               # normalize
    else:
        x[nX//2] = 1

    pdf_u = np.zeros(nT+1)
    pdf_l = np.zeros(nT+1)
    
    xx = np.linspace(-1.,1.,nX+1)
    
    if fullsolution.shape[0]==nT+1:
        fullsolution[0,:]=x


    for n in range(nT):       # time loop
# Rannacher time-marching, change theta in the first steps.
#        if n<2:
#            theta = 1.0
#        else:
#            theta = 0.5
#        theta = 0.5
        t = (n+1)*dt
        J_old = model.b(t-dt,params)
        mu_old = np.ones(nX+1) * model.mu(t-dt,params)/J_old - model.dt_b(t-dt,params)/J_old*xx
        sigma_old = model.sigma/J_old
        L_old = 1.0/dx*sigma_old*sigma_old/2.0
        
        J_new  = model.b(t,params)
        mu_new = np.ones(nX+1) * model.mu(t,params)/J_new    - model.dt_b(t,params)/J_new*xx
        sigma_new = model.sigma/J_new
        L_new = 1.0/dx*sigma_new*sigma_new/2.0
        
        
        f = 2.0/3.0*dx*x
        f[1:-1] += 1.0/6.0 * dx * x[0:-2]
        f[1:-1] += 1.0/6.0 * dx * x[2:]
        
        f[1:-1] = f[1:-1] + (theta-1)*dt*(   -L_old-0.5*mu_old[0:-2])*x[0:-2]
        f[1:-1] = f[1:-1] + (theta-1)*dt*(2.0*L_old)                 *x[1:-1]
        f[1:-1] = f[1:-1] + (theta-1)*dt*(   -L_old+0.5*mu_old[2:]  )*x[2:]

        assert f[0]  == 0, 'f nicht null?'
        assert f[-1] == 0, 'f nicht null?'

        x=tools.tridiag(f,
                1.0/6.0*dx + dt*theta * (   -L_new - 0.5*mu_new),
                2.0/3.0*dx + dt*theta * (2.0*L_new  ),
                1.0/6.0*dx + dt*theta * (   -L_new + 0.5*mu_new))

        if fullsolution.shape[0]==nT+1:
            fullsolution[n+1,:]=x
        
        pdf_u[n+1]=.5*sigma_new**2.0/dx/dx*(3.0*x[-2] - 1.5*x[-3] + 1.0/3.*x[-4])
        pdf_l[n+1]=.5*sigma_new**2.0/dx/dx*(3.0*x[1]  - 1.5*x[2]  + 1.0/3.*x[3] )

    # omit all states that did not reach a threshold
    scale = np.sum(pdf_u)*dt + np.sum(pdf_l) * dt
    pdf_u=pdf_u/scale
    pdf_l=pdf_l/scale
    
    return [pdf_u,pdf_l,fullsolution]

### Some further required functions

