//////////////////////////////////////////////////
//
// Scripts to reproduce the results from
//
// T. Richter, R Ulrich, M. Janczyk:
//    "Diffusion models with time-dependent parameters:
//     Comparing the computation effort and accuracy
//     of different numerical methods"
//
// Thomas Richter
// Otto-von-Guericke University of Magdeburg
// 39106 Magdeburg, Germany
// thomas.richter@ovgu.de
//
// You can use this code under ther terms of the
// Creative Commons Attribution 4.0 License
//
//////////////////////////////////////////////////

/*----------------------------   tools.h     ---------------------------*/
/*      $Id:$                 */
#ifndef __tools_H
#define __tools_H
/*----------------------------   tools.h     ---------------------------*/

#include <cassert>
#include <iostream>
#include <string>
#include <vector>
#include <algorithm>
#include <random> // Funktionen fuer Zufallszahlen
#include <chrono>
#include <ctime> // Funktionen zur Zeitmessung
#include <fstream>
#include <map>
#include <ratio>

using namespace std;


class RandomNumbers
{
private:
  default_random_engine generator;

public:
  // Startet die Zufallszahlensuche mit einem "zufaelligem" Wert
  void initsimple()
  {
    int X = clock() + 1;
    generator.seed(X);
  }
  void init(double s)
  {
    int X = s*10000.0;
    generator.seed(X);
  }

  // liefert Zufallszahl im definierten Bereich
  int operator()(int min, int max)
  {
    return uniform_int_distribution<int>{min, max}(generator);
  }
  // liefert Zufallszahl im definierten Bereich
  double randomdouble(double min, double max)
  {
    return uniform_real_distribution<double>{min, max}(generator);
  }
  double normal(double a, double b)
  {
    return normal_distribution<double>{a, b}(generator);
  }
  double gamma(double a, double b)
  {
    return gamma_distribution<double> {a,b}    (generator);
  }
  double beta(double a, double b)
  {
    double x = gamma(a,1.0);
    double y = gamma(b,1.0);
    return x/(x+y);
  }

};

//////////
////////// Stopuhr
//////////

class StopUhr
{
private:
  chrono::high_resolution_clock::time_point _start;
  chrono::high_resolution_clock::time_point _stop;
  double _elapsed_time;
  bool _running;

public:
  void reset()
  {
    _running = false;
    _elapsed_time = 0.0;
  }


  StopUhr()
  {
    reset();
  }

  void start()
  {
    assert(_running == false);
    _running = true;
    _start = std::chrono::high_resolution_clock::now();
  }
  void stop()
  {
    assert(_running);
    _running = false;
    _stop = std::chrono::high_resolution_clock::now();
    _elapsed_time += std::chrono::duration<double>{_stop - _start}.count();
  }

  double operator()() const
  {
    return _elapsed_time;
  }
};



//
// Compute the relative error:  max(cdf - reference) / max(reference) in %
//
double computeerror(const vector<double>& cdf, const vector<double> ref)
{
  assert(cdf.size() == 1000);
  assert(ref.size() == 1001);

  double err = 0.0;
  double scale = 0.0;
  for (int i=0;i<ref.size();++i)
    scale = std::max(scale,fabs(ref[i]));

  for (int i=0;i<1000;++i)
    {
      err = std::max(err,fabs(ref[i+1]-cdf[i]));
    }
  return err/scale*100.0;
}


//
// transform the pdf into a cdf 
//
void pdf2cdf(vector<double>& cdf,
	     const vector<double> &pdf, int NEX)
{
  assert(cdf.size() == 1000);
  assert(pdf.size() == 1000);

  cdf[0]=pdf[0]/NEX;
  for (int i=1;i<1000;++i)
    {
      cdf[i]=static_cast<double>(pdf[i])/NEX + cdf[i-1];
    }
}


//
// 
//
void readreference(int tmax, string name,vector<double>& tt, vector<double>& ref)
{
  //  std::cerr << "Reading reference solution " << name << std::flush;
  ifstream IN(name.c_str());
  if (!IN.is_open())
    {
      std::cerr << "file " << name << " could not be found for input! " << std::endl;
      abort();
    }
  assert (IN.is_open());
  tt.clear();
  ref.clear();
  double t,d;
  while (!IN.eof())
    {
      IN >> t >> d;
      if (IN.eof())
	break;
      tt.push_back(t);
      ref.push_back(d);
    }
  IN.close();

  // some checks for consistency
  assert(ref.size() == tt.size());
  assert((tt.size()-1)%tmax == 0);
  assert(fabs(tt[tt.size()-1]-tmax)<1.e-10);
  assert(ref.size()>tmax);

  // shorten reference solution
  int step = (tt.size()-1)/tmax;
  for (int i=0;i<tmax;++i)
    {
      ref[i]=ref[i*step];
      tt[i]=tt[i*step];
    }
  ref[tmax] = ref[ref.size()-1];
  tt[tmax] = tt[tt.size()-1];
  ref.resize(tmax+1);
  tt.resize(tmax+1);
}



/*----------------------------   tools.h     ---------------------------*/
/* end of #ifndef __tools_H */
#endif
/*----------------------------   tools.h     ---------------------------*/
