//////////////////////////////////////////////////
//
// Scripts to reproduce the results from
//
// T. Richter, R Ulrich, M. Janczyk:
//    "Diffusion models with time-dependent parameters:
//     Comparing the computation effort and accuracy
//     of different numerical methods"
//
// Thomas Richter
// Otto-von-Guericke University of Magdeburg
// 39106 Magdeburg, Germany
// thomas.richter@ovgu.de
//
// You can use this code under ther terms of the
// Creative Commons Attribution 4.0 License
//
//////////////////////////////////////////////////

#include "tools.h"
#include <numeric>

using namespace std;



// time step 
double dt;

// Final time. Specified as (int) since used as vector length
int TMAX = 1000; 


//////////////////////////////////////////////////
// Process paramters and core SDE-routine 
constexpr double mu     = 0.5;
constexpr double A      = 20.0;
constexpr double tau    = 150.0;
constexpr double sigma  = 4.0;

// returns the reaction time t<TMAX and sets (expected) to true
// if upper threshold is reached, to false otherwise
// The experiment is repeated until a decision is reached within
// t<TMAX
double process(RandomNumbers &Z,
               bool &expected)
{
  double T =0.0;
  double xc=0.0;

  double b=75.0; // variable boundary
  double m=0;    // variable drift
  do
    {
      T=0.0;

      xc=2.0*b*(Z.beta(5.,5.)-0.5); // Beta-distributed initial in [-b1, b1]
      
      while ((fabs(xc)< b)&&(T<TMAX))
	{
	  T += dt;

	  b = 75.0 - 60.0 / (1.0 + exp(5.0-T/80.0)); // adjust boundary
	  m = mu + A/tau * exp(1.0-T/tau) * (1.0-T/tau);
	  
	  xc += dt * m + sqrt(dt) * Z.normal(0.0, sigma);
	}
      
    }
  while (T>=TMAX);
  expected = (xc > 0.);
  return T;//  + Z.normal(mur, sigmar) + Z.normal(0.0,1.e-14);
}
//////////////////////////////////////////////////


//
// Run EX experiments and count result in pos/neg
//
void experiments(RandomNumbers& Z0,
		 vector<double>& pos,
		 vector<double>& neg, int EX)
{
  RandomNumbers Z;
  Z.init(Z0.normal(0,1000));
  
  for (int i = 0; i < EX; ++i)
    {
      bool ex = false;
      
      double r = process(Z, ex);
      if (r<=TMAX)
	{
	  if (ex)
	    pos[static_cast<int>(r)]++;
	  else
	    neg[static_cast<int>(r)]++;
	}
    }
  
}

//
// 
//
int main(int argc, char **argv)
{
  //
  // read reference solution (coming from testcase1.py)
  //

  if (argc<2)
    {
      cerr << "Aufruf mit: testcase3 tau" << endl;
      abort();
    }

  double tau = atoi(argv[1]);
  assert(tau>0);

  int itau = static_cast<int> (tau+1.e-10);
  char su[100], sl[100];
  sprintf(su, "../Python/results/testcase3-refu-cdf-%i.txt", itau);
  sprintf(sl, "../Python/results/testcase3-refl-cdf-%i.txt", itau);

  vector<double> refupper,reflower,reft;
  readreference(TMAX,sl,reft,reflower);
  readreference(TMAX,su,reft,refupper);

  assert(reflower.size()-1 == 1000);
  assert(refupper.size()-1 == 1000);
  assert(TMAX == 1000);
  
  dt     = 0.125;
  int EX = 800000;

  int MACRO = 64;
  vector<vector<double> >    pdfl(MACRO,vector<double> (1000,0.0));
  vector<vector<double> >    pdfu(MACRO,vector<double> (1000,0.0));
  
  vector<vector<double> >    cdfl(MACRO,vector<double> (1000,0.0));
  vector<vector<double> >    cdfu(MACRO,vector<double> (1000,0.0));
  vector<double> erru(MACRO),errl(MACRO);

  StopUhr S; // for time measurement
  S.reset();	
  S.start();

  
#pragma omp parallel for
  for (int i = 0; i<MACRO; ++i)
    {
      
      RandomNumbers Z0; //initialize random numbers
      Z0.initsimple();
	
      experiments(Z0, pdfu[i] , pdfl[i] , EX); // run EX experiemnts

      pdf2cdf(cdfu[i],pdfu[i], EX); // transform pdf's to cdf's
      pdf2cdf(cdfl[i],pdfl[i], EX);

      erru[i] = computeerror(cdfu[i],refupper); // compute errors
      errl[i] = computeerror(cdfl[i],reflower);
    }

  S.stop();
  
  // get the average error
  double eu=0,el=0;
  for (int i=0;i<MACRO;++i)
    {
      eu += erru[i]/MACRO;
      el += errl[i]/MACRO;
    }

  std::cout << tau << "\t" << EX << "\t" << dt << "\t" << S()/MACRO << "\t" << eu << "\t" << el << std::endl;
  
  return 0;
}
