#!/usr/bin/env perl
use strict;
use warnings;
use utf8;
use feature qw{ postderef say signatures state };
no warnings qw{ experimental::postderef experimental::signatures };

use Data::Dumper qw(Dumper);
use DBI;

if(@ARGV < 2) {
    say STDERR "Require inputs! Both globbing (\"./*/\") and direct lists are supported. The last path is the file to write the database to.";
    exit;
}

my @folders;
for(my $i = 0; $i < $#ARGV; $i++) {
    my $a = $ARGV[$i];
    my @globresult = glob $a;
    foreach(@globresult) {
        if(not(-d $_)) {
            say STDERR "Path \"$_\" resulting from glob \"$a\" is not a folder! Ignoring.";
            next;
        }
        push @folders, $_;
    }
}

my $dbpath = $ARGV[$#ARGV];

# Connect to database.
my $dbh = DBI->connect("DBI:SQLite:dbname=$dbpath", "", "", { 
        RaiseError => 0, 
        AutoCommit => 0 })
    or die $DBI::errstr;

$dbh->do("PRAGMA synchronous = OFF;");

# SQL
sub ParseAndInsertFile {
    my $stmt = $_[0];
    my $file = $_[1];

    open my $fh, $file or die "    Could not open $file: $!";

    my $problem = "";
    my $name = "";
    my $startdate = "";
    my $task = 0;
    my $executable = "";
    my $host = "";
    my $commit = "";
    my $result = "";
    my $status = "";
    my $resultcode = "";
    my $real = 0;
    my $time = 0;
    my $load = 0;
    my $utilization = 0;
    my $space = 0;
    my $samples = 0;

    while(my $line = <$fh>)  {
        if($line =~ m/^c submit\.pl: host:\s*(\w+)$/) {
            $host = $1;
        } elsif($line =~ m/^c submit\.pl: commit:\s*(\w+)$/) {
            $commit = $1;
        } elsif($line =~ m/^c submit\.pl: name:\s*(\w+)$/) {
            $name = $1;
        } elsif($line =~ m/^c submit\.pl: task:\s*(\d+)$/) {
            $task = $1;
        } elsif($line =~ m/^c submit\.pl: start:\s*(.+)$/) {
            $startdate = $1;
        } elsif($line =~ m/^c submit\.pl: executable:\s*(.+)$/) {
            $executable = $1;
        } elsif($line =~ m/^c Running job [\w\-]+ with problem ([^\s]+) of name ([^!]+)/) {
            $problem = $1;
        } elsif($line =~ m/^s SAT/) {
            $result = "SATISFIABLE";
        } elsif($line =~ m/^s UNSAT/) {
            $result = "UNSATISFIABLE";
        } elsif($line =~ m/^SAT$/) {
            $result = "SATISFIABLE";
        } elsif($line =~ m/^UNSAT$/) {
            $result = "UNSATISFIABLE";
        } elsif($line =~ m/^s UNKNOWN/) {
            $result = "UNKNOWN";
            $status = "unknown";
        } elsif($line =~ m/^[rsc] Satisfiable/) {
            $result = "SATISFIABLE";
        } elsif($line =~ m/^[rsc] Unsatisfiable/) {
            $result = "UNSATISFIABLE";
        } elsif($line =~ m/^[rs] (\w+)/) {
            $result = $1;
        } elsif($line =~ m/^\[runlim\] status:\s*(\w+)/ and $status eq "") {
            $status = $1;
        } elsif($line =~ m/^\[runlim\] result:\s*(\d+)/) {
            $resultcode = $1;
        } elsif($line =~ m/^\[runlim\] real:\s*(\d+(?:\.\d+))/) {
            $real = $1;
        } elsif($line =~ m/^\[runlim\] time:\s*(\d+(?:\.\d+))/) {
            $time = $1;
        } elsif($line =~ m/^\[runlim\] space:\s*(\d+)/) {
            $space = $1;
        } elsif($line =~ m/^\[runlim\] load:\s*(\d+(?:\.\d+))/) {
            $load = $1;
        } elsif($line =~ m/^\[runlim\] samples:\s*(\d+)/) {
            $samples = $1;
        } elsif($line =~ m/Segmentation fault/) {
            $result = "SEGFAULT";
            $status = "segfault";
        } elsif($line =~ m/: Assertion /) {
            $result = "ABORTED";
            $status = "aborted";
        }
    }

    close $fh;

    if($real != 0) {
        $utilization = $time / $real;

        $stmt->execute(
            $task,
            $problem,
            $file,
            $name,
            $startdate,
            $executable,
            $host,
            $commit,
            $result,
            $status,
            $resultcode,
            $real,
            $time,
            $load,
            $utilization,
            $space,
            $samples);
    }
    # Debugging Log
    # say "    Inserted \"$file\"";
}

# Process Folder
sub ProcessFolder {
    my $folder = $_[0];

    my $tblname = $folder;
    $tblname =~ tr/-/_/;
    $tblname =~ tr/ /_/;
    $tblname =~ tr/+/_/;
    $tblname =~ tr/./_/;
    $tblname =~ tr/\//_/;

    say "Processing folder \"$folder\" into table \"$tblname\"...";
    
    my $stmt_create_table = qq(CREATE TABLE IF NOT EXISTS $tblname (
        id INTEGER PRIMARY KEY NOT NULL,
        problem TEXT UNIQUE NOT NULL,
        logfile TEXT UNIQUE NOT NULL,
        name TEXT NOT NULL,
        startdate TEXT NOT NULL,
        executable TEXT NOT NULL,
        host TEXT NOT NULL,
        comm TEXT NOT NULL,
        result TEXT NOT NULL,
        status TEXT NOT NULL,
        resultcode INTEGER NOT NULL,
        real REAL NOT NULL,
        time REAL NOT NULL,
        load REAL,
        utilization REAL NOT NULL,
        space REAL NOT NULL,
        samples INTEGER NOT NULL);
    );
    my $rv = $dbh->do($stmt_create_table);
    if($rv < 0) {
        say $DBI::errstr;
    } else {
        say "    Table $tblname created successfully.";
    }

    my $stmt_insert = "INSERT INTO $tblname VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    my $stmt_insert_prepared = $dbh->prepare($stmt_insert);

    my @logs = glob "$folder/*.log";
    foreach my $l (@logs) {
        ParseAndInsertFile($stmt_insert_prepared, $l);
    }

    $dbh->do("CREATE UNIQUE INDEX IF NOT EXISTS idx_problem_$tblname ON $tblname (problem);");
    $dbh->do("COMMIT;");
}

foreach(@folders) {
    say $_;
    ProcessFolder($_);
}

$dbh->disconnect();

