#pragma once

#include "veri_file.h"
#include "VeriModule.h"
#include "VeriId.h"
#include "VeriMisc.h"
#include "VeriExpression.h"
#include "veri_tokens.h"
#include "VeriVisitor.h"
#include "VeriStatement.h"
#include "VeriScope.h"      // Definitions of the 'VeriScope' class.
#include "VeriConstVal.h"
#include "Strings.h"
#include "hdl_file_sort.h"
#include "Array.h"          // a dynamic array
#include "Set.h"            // a simple hash table
#include "Map.h"   
#include "utils/result.h"
#include <vector>
#include <string>
#include <iostream>
#include <sstream>
#include <math.h>
#include "scanner.h"

#ifdef VERIFIC_NAMESPACE
using namespace Verific ;
#endif

extern std::ofstream log_detail;extern std::ofstream log_detail, log_results;
extern std::string filepath, module_name;
extern Result result;


// =============================CWE 1245 Detector============================//

int hamming_distance(int a, int b);

class VulCaseStatement{
public:
    VulCaseStatement(){};
    VulCaseStatement(int l, bool d, bool it){
        loc = l;
        no_default_stmt = d;
        not_enough_items = it;
    }
    int loc;
    bool no_default_stmt = false;
    bool not_enough_items = false;

    void print(){
        // std::cout<<"\nCase statement at line "<<loc<<" has neither a default statement nor enough case items\n";
        log_detail<<"\nCase statement at line "<<loc<<" has neither a default statement nor enough case items\n";

    }
};

extern std::vector<VulCaseStatement> vec_vul_case_statements;

class Fsm_transition{

public:
    Fsm_transition() {};
    Fsm_transition(std::string fsm_sv, int ival, std::string ival_enum, int fval, std::string fval_enum, int l, std::string cond){
        state_var = fsm_sv;
        initial_value = ival;
        initial_value_enum = ival_enum;
        final_value = fval;
        final_value_enum = fval_enum;
        loc = l;
        condition = cond;
    }

    void print_transition(){
        std::cout<<"  "<<state_var<<" transitions from "<<initial_value_enum<<" ("<< initial_value << ") to "<<final_value_enum<<" ("<<final_value<<") on condition "<< condition<<" , line "<<loc<<std::endl;
        log_detail<<"  "<<state_var<<" transitions from "<<initial_value_enum<<" ("<< initial_value << ") to "<<final_value_enum<<" ("<<final_value<<") on condition "<< condition<<" , line "<<loc<<std::endl;
    }

    std::string state_var;
    int initial_value;
    int final_value;
    std::string initial_value_enum;
    std::string final_value_enum;
    int loc;
    std::string condition;

};

class Fsm_state_reset{

public:
    Fsm_state_reset() {};
    Fsm_state_reset(std::string fsm_sv, int rv, std::string rv_enum, int l){
        state_var = fsm_sv;
        reset_value = rv;
        reset_value_enum = rv_enum;
        loc = l;
    }

    void print_reset(){
        std::cout<<"  "<<state_var<<" resets to "<<reset_value_enum<<" ("<< reset_value << ") , line "<<loc<<std::endl;
        log_detail<<"  "<<state_var<<" resets to "<<reset_value_enum<<" ("<< reset_value << "), line "<<loc<<std::endl;

    }

    std::string state_var;
    int reset_value;
    std::string reset_value_enum;
    int loc;

};

class Fsm_obj{

public:
    Fsm_obj() {};
    Fsm_obj(std::string n){
        module_name = n;
    }

    std::string module_name;
    std::vector<std::string> state_var_names; 
    std::vector<Fsm_transition> fsm_transitions;
    std::vector<Fsm_state_reset> fsm_state_resets;

    void print_fsm(){
        std::cout<<"\nFSM: \nmodule: "<<module_name<<std::endl;
        std::cout<<"variables: ";
        for (auto state_var_name = state_var_names.begin(); state_var_name != state_var_names.end(); ++state_var_name)
            std::cout<<" "<<(*state_var_name);
        std::cout<<"\nresets: \n";
        for (auto r_it = fsm_state_resets.begin(); r_it != fsm_state_resets.end(); ++r_it)
            (*r_it).print_reset();
        std::cout<<"transitions: \n";
        for (auto tr = fsm_transitions.begin();  tr!= fsm_transitions.end(); ++tr)
            ( *tr ).print_transition();
        
    }

    void analyze_transitions(){

        // check if FSM relies on enumeration only
        bool enum_only = true;
        for (auto tr = fsm_transitions.begin();  tr!= fsm_transitions.end(); ++tr){
            if (!(tr->initial_value==0 && tr->final_value==0)){
                enum_only=false;
                break;
            } 
        }

        // log_detail<<"\nenum_only: "<<enum_only<<"\n";

        if(enum_only){
            // std::cout<<"\nAnalyzing state transitions for missing state transitions \n";
            bool transition_found = false;
            for (auto tr = fsm_transitions.begin();  tr!= fsm_transitions.end(); ++tr){
                std::string i_val = tr->initial_value_enum; // get the intial value for the transition            
                // check whether i_val is present in final_val of any other transition
                transition_found = false;
                for (auto tr1 = fsm_transitions.begin();  tr1!= fsm_transitions.end(); ++tr1){
                    if ( tr1->final_value_enum == i_val && tr1->initial_value_enum != i_val){
                        transition_found = true;
                        break;
                    }
                }
                // check whether i_val is present in reset value of state
                for (auto tr2 = fsm_state_resets.begin(); tr2!= fsm_state_resets.end(); ++tr2){
                    if (tr2->reset_value_enum == i_val){
                        transition_found = true;
                        break;
                    }
                }
                // warning if transition is not found
                if(!transition_found){
                    if(!result.cwe_1245){
                        result.cwe_1245 = true;
                        log_detail << "potential cwe 1245 detected\n";
                    }
                    log_detail<<"missing transition for state "<<i_val<<" i.e. FSM has a transition to some other state from "<<i_val<<", but no way to reach "<<i_val <<"\n";
                    std::string cwe_type = "1245_missing-transition";
                    std::string line_start = std::to_string(tr->loc);
                    std::string line_end = std::to_string(tr->loc);
                    log_results<<"x"<<","<<filepath<<","<<module_name<<","<<cwe_type<<","<<line_start<<","<<line_end<<std::endl;
                }
            }

            // std::cout<<"\nAnalyzing state transitions for FSM deadlock \n";
            transition_found = false;
            for (auto tr = fsm_transitions.begin();  tr!= fsm_transitions.end(); ++tr){
                std::string f_val = tr->final_value_enum; // get the final value for the transition            
                // check whether f_val is present in initial_val of any other transition
                transition_found = false;
                for (auto tr1 = fsm_transitions.begin();  tr1!= fsm_transitions.end(); ++tr1){
                    if ( tr1->initial_value_enum == f_val && tr1->final_value_enum != f_val){
                        transition_found = true;
                        break;
                    }
                }
                // warning if transition is not found
                if(!transition_found){
                    if(!result.cwe_1245){
                        result.cwe_1245 = true;
                        log_detail << "potential cwe 1245 detected\n";
                    }
                    log_detail<<"FSM deadlock at "<<f_val<<" i.e. FSM can not transition to some other state on from this state\n";

                    // ID,filepath,module,cwe,line-start,line-end
                    std::string cwe_type = "1245_deadlock";
                    std::string line_start = std::to_string(tr->loc);
                    std::string line_end = std::to_string(tr->loc);
                    log_results<<"x"<<","<<filepath<<","<<module_name<<","<<cwe_type<<","<<line_start<<","<<line_end<<std::endl;
                }
            }
        }

        else{
            // std::cout<<"\nAnalyzing state transitions for missing state transitions \n";
            bool transition_found = false;
            for (auto tr = fsm_transitions.begin();  tr!= fsm_transitions.end(); ++tr){
                int i_val = tr->initial_value; // get the intial value for the transition            
                // check whether i_val is present in final_val of any other transition
                transition_found = false;
                for (auto tr1 = fsm_transitions.begin();  tr1!= fsm_transitions.end(); ++tr1){
                    if ( tr1->final_value == i_val && tr1->initial_value != i_val){
                        transition_found = true;
                        break;
                    }
                }
                // check whether i_val is present in reset value of state
                for (auto tr2 = fsm_state_resets.begin(); tr2!= fsm_state_resets.end(); ++tr2){
                    if (tr2->reset_value == i_val){
                        transition_found = true;
                        break;
                    }
                }
                // warning if transition is not found
                if(!transition_found){
                    if(!result.cwe_1245){
                        result.cwe_1245 = true;
                        log_detail << "potential cwe 1245 detected\n";
                    }
                    log_detail<<"missing transition for state "<<i_val<<" i.e. FSM has a transition to some other state from "<<i_val<<", but no way to reach "<<i_val <<"\n";
                    std::string cwe_type = "1245_missing-transition";
                    std::string line_start = std::to_string(tr->loc);
                    std::string line_end = std::to_string(tr->loc);
                    log_results<<"x"<<","<<filepath<<","<<module_name<<","<<cwe_type<<","<<line_start<<","<<line_end<<std::endl;
                
                }
            }

            // std::cout<<"\nAnalyzing state transitions for FSM deadlock \n";
            transition_found = false;
            for (auto tr = fsm_transitions.begin();  tr!= fsm_transitions.end(); ++tr){
                int f_val = tr->final_value; // get the final value for the transition            
                // check whether f_val is present in initial_val of any other transition
                transition_found = false;
                for (auto tr1 = fsm_transitions.begin();  tr1!= fsm_transitions.end(); ++tr1){
                    if ( tr1->initial_value == f_val && tr1->final_value != f_val){
                        transition_found = true;
                        break;
                    }
                }
                // warning if transition is not found
                if(!transition_found){
                    if(!result.cwe_1245){
                        result.cwe_1245 = true;
                        log_detail << "potential cwe 1245 detected\n";
                    }
                    log_detail<<"FSM deadlock at "<<f_val<<" i.e. FSM can not transition to some other state on from this state\n";
                    std::string cwe_type = "1245_deadlock";
                    std::string line_start = std::to_string(tr->loc);
                    std::string line_end = std::to_string(tr->loc);
                    log_results<<"x"<<","<<filepath<<","<<module_name<<","<<cwe_type<<","<<line_start<<","<<line_end<<std::endl;
                }
            }

            // std::cout<<"\nAnalyzing state transitions for vulnerable transitions \n";
            // for (auto tr = fsm_transitions.begin();  tr!= fsm_transitions.end(); ++tr){
            //     int ham_dist = hamming_distance( ( *tr ).initial_value, (*tr).final_value );
            //     if (ham_dist >1 ){
            //         if(!result.cwe_1245){
            //             result.cwe_1245 = true;
            //             log_detail << "potential cwe 1245 detected\n";
            //         }
            //         (*tr).print_transition();
            //         // std::cout<<"    transition could be vulnerable to a timing violation-based fault injection attack. It has hamming distance "<<ham_dist<<std::endl;
            //         log_detail<<"    transition could be vulnerable to a timing violation-based fault injection attack. It has hamming distance "<<ham_dist<<std::endl;

            //     }
            // }
        }
    }
        
    void clear(){
        module_name = "";
        state_var_names.clear();
        fsm_transitions.clear();
        fsm_state_resets.clear();

    }

};

// create an FSM for module [BG]
extern Fsm_obj fsm;

class TransitionVisitor : public VeriVisitor
{
public :
    TransitionVisitor() : VeriVisitor(),
        _present_state_var(0),
        _present_state(0),
        _is_clocked(0),
        _conditions(POINTER_HASH) { } ;

    ~TransitionVisitor() {} ;

    virtual void VERI_VISIT(VeriInitialConstruct, node) {
        // node.Info("skipping this initial block") ;
    }

    // Collect event control statements :
    virtual void VERI_VISIT(VeriEventControlStatement, event_control_stmt) {

        result.cwe_1245_relevant_nodes++;
        // event_control_stmt.Info("visiting this event control statement") ;

        // Check if this event control statement is clocked :
        _is_clocked = 0 ;
        // Look at the sensitivity list (the "@()" clause), and check if it has 'edge' expressions :
        Array *sens_list = event_control_stmt.GetAt() ;
        unsigned i ;
        VeriExpression *event_expr ;
        FOREACH_ARRAY_ITEM(sens_list, i, event_expr) {
            if (!event_expr) break ; // invalid sensitivity list
            if (event_expr->IsEdge(0/*any edge (pos or neg)*/)) {
                _is_clocked = 1 ;
                break ; // no need to check the other sensitivity list items. This statement is clocked !
            }
        }

        // Dive into the statement
        VeriStatement *stmt = event_control_stmt.GetStmt() ;
        if (stmt) stmt->Accept(*this) ;

        // reset clocking information
        _is_clocked = 0 ;
    }

    // Catch IF-statements :
    virtual void VERI_VISIT(VeriConditionalStatement, if_statement) {

        result.cwe_1245_relevant_nodes++;

        // Get the condition expression :
        VeriExpression *if_condition = if_statement.GetIfExpr() ;
        if (!if_condition) return ; // something bad happened
        VeriStatement *then_stmt = if_statement.GetThenStmt() ;
        VeriStatement *else_stmt = if_statement.GetElseStmt() ;

        // FIX ME: Check if this is a :
        //   - set/reset/clocked condition (if _present_state is not set and no state var compare)
        //       also we could put limit on set/reset condition depth..
        //       and check if set/reset is async or sync.

        // Check if this is a state-compare :'if (statevar==stateconstant) ..'
        if (if_condition->OperType()==VERI_LOGEQ) {
            // This is a == compare condition.
            // Check if it is a fsm-state compare :
            VeriExpression *left = if_condition->GetLeft() ;
            VeriExpression *right = if_condition->GetRight() ;
            VeriIdDef *left_id = (left) ? left->FullId() : 0 ;
            VeriIdDef *right_id = (right) ? right->FullId() : 0 ;

            if ((left_id && left_id->CanBeOnehot()) || (right_id && right_id->CanBeOnehot())) {
                // Yes. This is a state variable compare.
                // If we are already under a state compare, then warn :
                if (_present_state_var) {
                    // If these is already a present_state_var set, then we two nested state compares.
                    // That is suspicious. Warn about this (and consider below that this second compare is a 'normal' (input-condition) case compare
                     if_condition->Warning("multiple nested FSM state conditions detected, on variable %s", _present_state_var->Name()) ;
                }

                // Set as the 'present_state' for the 'if' part.
                _present_state_var = (left_id->CanBeOnehot()) ? left_id : right_id ;
                _present_state = (left_id->CanBeOnehot()) ? right : left ;

                // Traverse 'then_stmt' with this :
                if (then_stmt) then_stmt->Accept(*this) ;

                _present_state_var = 0 ;
                _present_state = 0 ;

                // Now traverse 'else_stmt' without this state compare set.
                if (else_stmt) else_stmt->Accept(*this) ;

                // done
                return ;
            }
        }

        // This is not a state-comparing 'if'. Treat as normal condition area :
        // Generate a string representation of the 'if' condition :

        // Push the condition on the stack
        PushIfCondition(if_condition) ;

        // Traverse the 'then' statements :
        if (then_stmt) then_stmt->Accept(*this) ;

        // Pop the condition :
        PopCondition() ;

        // Now push the 'NOT' condition on the stack
        PushElseCondition(if_condition) ;

        // if_statement.Info("found else condition %s",ns.c_str()) ;

        // Traverse the 'else' statements :
        if (else_stmt) else_stmt->Accept(*this) ;

        // Pop the condition :
        PopCondition() ;
    }

    // Catch CASE statements :
    virtual void VERI_VISIT(VeriCaseStatement, case_statement) {

        result.cwe_1245_relevant_nodes++;
        // case_statement.Info("found case statement") ;

        // Get the case condition :
        VeriExpression *case_condition = case_statement.GetCondition() ;

        // Check if this is a state-comparing case statement.
        // The case expression should be a IdRef of the 'present_state' variable :
        VeriIdDef *present_state_id = (case_condition) ? case_condition->FullId() : 0 ;
        if (present_state_id && present_state_id->CanBeOnehot()) {
            // Yes. This is a present_state comparing case statement.
            if (_present_state_var) {
                // If these is already a present_state_var set, then we two nested state compares.
                // That is suspicious. Warn about this (and consider below that this second compare is a 'normal' (input-condition) case compare
                 case_condition->Warning("multiple nested FSM state conditions detected, on variable %s and %s", present_state_id->Name(), _present_state_var->Name()) ;
            }
            // Each case-item will (should) contain the present_state constant.
        } else {
            // No. This is a regular case statement.
            if (!_present_state_var) {
                // If there is no '_present_state_var' set, then this case statement is NOT part of an FSM.
                // So don't analyse it.
                return ; // done.
            }
        }

        // The case statement is pertinent to an FSM. Analyze its condition for cwe [BG]
        analyze_casestatement(&case_statement);

        // Get the case items :
        Array *case_items = case_statement.GetCaseItems() ;
        // and iterate over them
        unsigned i ;
        VeriCaseItem *case_item ;
        FOREACH_ARRAY_ITEM(case_items, i, case_item) {
            if (!case_item) continue ; // something wrong..

            // Get the statement and the (OR-ed) conditions for this item :
            VeriStatement *stmt = case_item->GetStmt() ;
            if (!stmt) continue ; // no statement, no action

            Array *conditions = case_item->GetConditions() ;
            // and iterate over them :
            unsigned j ;
            VeriExpression *case_item_value ;
            FOREACH_ARRAY_ITEM(conditions, j, case_item_value) {
                if (!case_item_value) continue ;

                // Now execute 'stmt' under condition 'case_item_value'=='case_condition'.
                // Check if this case statement is a present-state comparison,
                // or if it is a normal case statement.
                if (!_present_state_var && present_state_id) {
                    // This is a new present_state comparing case statement.
                    // The present state (of the statement here) is 'case_item_value'.
                    _present_state = case_item_value ;
                    _present_state_var = present_state_id ;
                    // recur into the statement :
                    stmt->Accept(*this) ;
                    // reset the state comparison
                    _present_state = 0 ;
                    _present_state_var = 0 ;
                } else {
                    // This is a case statement comparing input variables.
                    // Push the case condition 'case_item_value'==='case_condition' on the track
                    PushCaseCondition(case_condition,case_item_value) ;

                    stmt->Accept(*this) ;

                    PopCondition() ;
                }
            }
        }
    }

    // Catch non-Blocking assignments :
    virtual void VERI_VISIT(VeriNonBlockingAssign, assign_statement) {

        result.cwe_1245_relevant_nodes++;

        // See if there is a state variable on the LHS of this assignment :
        VeriExpression *lval = assign_statement.GetLVal() ;
        VeriIdDef *lhs_id = (lval) ? lval->FullId() : 0 ;
        if (lhs_id && lhs_id->CanBeOnehot()) {
            // OK. We are assigning to state variable.
            // assign_statement.Info("found non-blocking assign to onehot var %s",lhs_id->Name()) ;

            // Get the (new state) on the rhs.
            VeriExpression *rhs = assign_statement.GetValue() ;

            VeriIdDef *rhs_id = (rhs) ? rhs->FullId() : 0 ;
            if ((rhs && rhs->IsConst()) || (rhs_id && rhs_id->IsParam())) {
                // This is an async reset assignment if there is no 'current-state'
                // This is a FSM state transition if there is a 'current-state'

                // Get the name and the (integer) value for this state :
                const char *next_state_name = (rhs_id) ? rhs_id->Name() : 0 ; // state only has a name if it is a parameter.
                int next_state_value = (rhs_id && rhs_id->GetInitialValue()) ? rhs_id->GetInitialValue()->Integer() : rhs->Integer() ; // value of parameter is in its initial value.

                if (_present_state) {
                    // Check if present state is a 'parameter' (or a constant)
                    VeriIdDef *present_id = _present_state->FullId() ;
                    if (present_id && !present_id->IsParam()) present_id = 0 ;

                    // Get the name and the (integer) value for this state :
                    const char *present_state_name = (present_id) ? present_id->Name() : 0 ; // state only has a name if it is a parameter.
                    int present_state_value = (present_id && present_id->GetInitialValue()) ? present_id->GetInitialValue()->Integer() : _present_state->Integer() ; // value of parameter is in its initial value.

                    char *condition = PrintCondition() ; // Get the current overall condition in string form.
                    // assign_statement.Info("TRANSITION variable %s from state %s(%d) to %s(%d) under condition \"%s\"",lhs_id->Name(),(present_state_name)?present_state_name:"",present_state_value,(next_state_name)?next_state_name:"",next_state_value, (condition)?condition:"") ;
                    
                    // Add the transition to out custom fsm object [BG]
                    Fsm_transition fsm_transition( lhs_id->Name(), present_state_value, (present_state_name)?present_state_name:"" , next_state_value, (next_state_name)?next_state_name:"", LineFile::GetLineNo(assign_statement.Linefile()) , (condition)?condition:"" );
                    fsm.fsm_transitions.push_back( fsm_transition );

                    Strings::free(condition) ;
                } else {
                    // assign_statement.Info("RESET variable %s to state %s(%d)",lhs_id->Name(),(next_state_name)?next_state_name:"",next_state_value) ;
                    
                    // Add the reset to our custom fsm object [BG]
                    Fsm_state_reset fsm_state_reset( lhs_id->Name(), next_state_value, (next_state_name)?next_state_name:"", LineFile::GetLineNo(assign_statement.Linefile()) );
                    fsm.fsm_state_resets.push_back( fsm_state_reset );
                }
            } else if (rhs_id && rhs_id->IsVar()) {
                // This is a variable to variable assignment.

                // Verific's fsm extraction guarantees that 'rhs_id' is an fsm variable.
                VERIFIC_ASSERT(rhs_id->CanBeOnehot()) ;

                // This assignment makes 'lhs_id' the 'present-state' variable and 'rhs_id' the 'next-state' variable in this FSM.
                if (_is_clocked) {
                    // assign_statement.Info("CLOCKED assignment from %s to %s",rhs_id->Name(), lhs_id->Name()) ;
                } else {
                    // assign_statement.Info("UNCLOCKED assignment from %s to %s",rhs_id->Name(), lhs_id->Name()) ;
                }

                // Check me : There should be no 'current_state' ?
            }
        }
    }

    // Catch blocking assignments :
    virtual void VERI_VISIT(VeriBlockingAssign, assign_statement) {

        result.cwe_1245_relevant_nodes++;

        // See if there is a state variable on the LHS of this assignment :
        VeriExpression *lval = assign_statement.GetLVal() ;
        VeriIdDef *lhs_id = (lval) ? lval->FullId() : 0 ;
        if (lhs_id && lhs_id->CanBeOnehot()) {
            // OK. We are assigning to state variable.
            // assign_statement.Info("found blocking assign to onehot var %s",lhs_id->Name()) ;

            // Get the (new state) on the rhs.
            VeriExpression *rhs = assign_statement.GetValue() ;

            VeriIdDef *rhs_id = (rhs) ? rhs->FullId() : 0 ;
            if ((rhs && rhs->IsConst()) || (rhs_id && rhs_id->IsParam())) {
                // This is an async reset assignment if there is no 'current-state'
                // This is a FSM state transition if there is a 'current-state'

                // Get the name and the (integer) value for this state :
                const char *next_state_name = (rhs_id) ? rhs_id->Name() : 0 ; // state only has a name if it is a parameter.
                int next_state_value = (rhs_id && rhs_id->GetInitialValue()) ? rhs_id->GetInitialValue()->Integer() : rhs->Integer() ; // value of parameter is in its initial value.

                if (_present_state) {
                    // Check if present state is a 'parameter' (or a constant)
                    VeriIdDef *present_id = _present_state->FullId() ;
                    if (present_id && !present_id->IsParam()) present_id = 0 ;

                    // Get the name and the (integer) value for this state :
                    const char *present_state_name = (present_id) ? present_id->Name() : 0 ; // state only has a name if it is a parameter.
                    int present_state_value = (present_id && present_id->GetInitialValue()) ? present_id->GetInitialValue()->Integer() : _present_state->Integer() ; // value of parameter is in its initial value.

                    char *condition = PrintCondition() ; // Get the current overall condition in string form.
                    // assign_statement.Info("TRANSITION variable %s from state %s(%d) to %s(%d) under condition \"%s\"",lhs_id->Name(),(present_state_name)?present_state_name:"",present_state_value,(next_state_name)?next_state_name:"",next_state_value, (condition)?condition:"") ;
                    
                    // Add the transition to out custom fsm object [BG]
                    Fsm_transition fsm_transition( lhs_id->Name(), present_state_value, (present_state_name)?present_state_name:"" , next_state_value, (next_state_name)?next_state_name:"", LineFile::GetLineNo(assign_statement.Linefile()) , (condition)?condition:"" );
                    fsm.fsm_transitions.push_back( fsm_transition );
                    
                    Strings::free(condition) ;
                } else {
                    // assign_statement.Info("RESET variable %s to state %s(%d)",lhs_id->Name(),(next_state_name)?next_state_name:"",next_state_value) ;
                
                    // Add the reset to our custom fsm object [BG]
                    Fsm_state_reset fsm_state_reset( lhs_id->Name(), next_state_value, (next_state_name)?next_state_name:"", LineFile::GetLineNo(assign_statement.Linefile()) );
                    fsm.fsm_state_resets.push_back( fsm_state_reset );
                
                }
            } else if (rhs_id && rhs_id->IsVar()) {
                // This is a variable to variable assignment.

                // Verific's fsm extraction guarantees that 'rhs_id' is an fsm variable.
                VERIFIC_ASSERT(rhs_id->CanBeOnehot()) ;

                // This assignment makes 'lhs_id' the 'present-state' variable and 'rhs_id' the 'next-state' variable in this FSM.
                if (_is_clocked) {
                    // assign_statement.Info("CLOCKED assignment from %s to %s",rhs_id->Name(), lhs_id->Name()) ;
                } else {
                    // assign_statement.Info("UNCLOCKED assignment from %s to %s",rhs_id->Name(), lhs_id->Name()) ;
                }

                // Check me : There should be no 'current_state' ?
            }
        }
    }

    // The 'input' condition stack.
    // This is modeled as a Map, so we can insert 'if', 'else' and 'case' conditions.
    void PushIfCondition(VeriExpression *cond) {
        VERIFIC_ASSERT(cond) ;
        (void) _conditions.Insert(cond,0,0,1) ; // expression goes to the 'key'
    }
    void PushElseCondition(VeriExpression *not_cond) {
        VERIFIC_ASSERT(not_cond) ;
        (void) _conditions.Insert(0,not_cond,0,1) ; // expression goes to the 'value'
    }
    void PushCaseCondition(VeriExpression *case_cond, VeriExpression *case_item) {
        VERIFIC_ASSERT(case_cond && case_item) ;
        (void) _conditions.Insert(case_cond,case_item,0,1) ; // case-expression and item go 'key' and 'value' respectively.
    }
    void PopCondition() {
        // Blindly remove the last inserted item. Will assert if you attempt to pop an empty stack.
        (void) _conditions.Remove(_conditions.GetItemAt(_conditions.Size()-1)) ;
    }

    char *PrintCondition() {
        // Print the condition stack as a char* image. Take the AND of all conditions in the stack.

        // Build the result in a stringstream, because we want to use PrettyPrint() to print the expressions.
        std::ostringstream ss ;

        // Iterate over the conditions :
        MapIter mi ;
        VeriExpression *condition ;
        VeriExpression *other_condition ;
        FOREACH_MAP_ITEM(&_conditions,mi,&condition,&other_condition) {
             if (ss.tellp()) {
                 // There is already a condition in this stream.
                 // So put an AND operator (&&) before writing the new one :
                 ss << " && " ;
             }

             // Check if this was a 'if' condition, and 'else' condition or a 'case' condition :
             if (condition && !other_condition) {
                 // regular single 'if' condition :
                 condition->PrettyPrint(ss,0) ;
             } else if (!condition && other_condition) {
                 // single 'else' condition. Put NOT operator (!) in front and parenthesize :
                 ss << "!(" ;
                 other_condition->PrettyPrint(ss,0) ;
                 ss << ")" ;
             } else if (condition && other_condition) {
                 // This is a case condition. Put a case-equal operator (===) in between :
                 condition->PrettyPrint(ss,0) ;
                 ss << "===" ;
                 other_condition->PrettyPrint(ss,0) ;
             } else {
                 VERIFIC_ASSERT(0) ; // should have have 0 - 0 item.
             }
        }

        // Now turn the stream into a string and return a 'saved' char* :
        std::string s = ss.str() ;
        return Strings::save(s.c_str()) ;
    }

    // [BG] Analyze case statement to check if either the number of case items match the size of the case condition or there is a default condition
    void analyze_casestatement(VeriCaseStatement* case_statement){

        VeriExpression *case_condition = case_statement->GetCondition() ;
        VeriIdDef *present_state_id = (case_condition) ? case_condition->FullId() : 0 ;
        // Get the case items :
        Array *case_items = case_statement->GetCaseItems() ;

        // std::string str = case_statement->GetPrettyPrintedString();
        // std::cout<<str;
        // std::cout<<"\ncase condition: "<<case_condition->GetPrettyPrintedString();
        // std::cout<<"\nsize of condition: "<<std::to_string( present_state_id->GetDimensionSize() );
        // std::cout<<"\nnumber of case items: "<<case_items->Size();

        bool has_enough_items = false;
        bool has_default_stmt = false;

        if (present_state_id){
            if (case_items->Size() >= pow(2,present_state_id->GetDimensionSize()) )
                has_enough_items = true;
        }

        // and iterate over case items
        unsigned i ;
        VeriCaseItem *case_item ;
        FOREACH_ARRAY_ITEM(case_items, i, case_item) {
            if (!case_item) continue ; // something wrong..
            Array *conditions = case_item->GetConditions();
            
            if(!conditions) //default statement
            {
                has_default_stmt = true;
                break;
            }
        }

        // std::cout<<"\nhas enough items: "<<has_enough_items<<" has default: "<<has_default_stmt<<std::endl;
        if (!has_enough_items && !has_default_stmt){
            // std::cout<<"\n\ncwe here\n\n";
            VulCaseStatement vcs( LineFile::GetLineNo(case_statement->Linefile()), has_default_stmt, has_enough_items );
            vec_vul_case_statements.push_back(vcs);
            if(!result.cwe_1245){
                result.cwe_1245 = true;
                log_detail << "potential cwe 1245 detected\n";
                std::string cwe_type = "1245_vul-case-statement";
                std::string line_start = std::to_string(case_statement->Linefile()->GetLeftLine());
                // std::string line_start = std::to_string(LineFile::GetLineNo(case_statement->Linefile()));
                std::string line_end = std::to_string(LineFile::GetLineNo(case_statement->Linefile()));
                log_results<<"x"<<","<<filepath<<","<<module_name<<","<<cwe_type<<","<<line_start<<","<<line_end<<std::endl;
            }
        }
    }

public :
    VeriIdDef *_present_state_var ; // The present state variable
    VeriExpression *_present_state ; // The condition (state) in which the state variable currently is.
    unsigned _is_clocked:1 ; // flag if current event control statement is clocked.
    // input conditions :
    Map _conditions ; // Map of pair of VeriExpression conditions, indicating the condition under which statements execute.
} ;

void detector1245( VeriModule* mod);

class Scanner_CWE1245 : public Scanner {
    public:
    void scan(VeriModule * mod){
        detector1245( mod);
    }
};