(* ::Package:: *)

(* RigiComp
 * Georg Grasegger
 * V1.0 December 2022
 *  
 * RigiComp is a collection of code for graph rigidity theory.
 * It contains functions for rigid graph constructions, rigidity testing and counting realizations.
 * RigiComp is a Package for Wolfram Mathematica written in Wolfram Language.
 * The package was tested with Mathematica 12
 *)


(* This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version, see
 *  https://www.gnu.org/licenses/gpl-3.0.html
 *
 * This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *)


(* ::Section:: *)
(*Settings*)


Protect[x,y];


(* ::Section:: *)
(*Graphs and Representations*)


(* ::Subsubsection::Closed:: *)
(*Representations*)


(* We represent a graph by the integer obtained from flattening the upper triangle
 * of its adjacency matrix and interpreting this sequence as binary digits.
 *)


(* To and from integer representaion *)
Mat2G::usage="Mat2G[m] transforms adjacency matrix m to integer representation of the graph.";
Mat2G[mat_] := FromDigits[Flatten[MapIndexed[Drop[#1, #2[[1]]]&, mat]], 2];

G2Mat::usage="G2Mat[g] transforms integer representation g to adjacency matrix of the graph.";
G2Mat[g_Integer, n_Integer] := (# + Transpose[#])&[PadLeft[
  Table[Take[#, {(2n-i)*(i-1)/2+1, (2n-i-1)*i/2}], {i, n}]&[PadLeft[IntegerDigits[g, 2], n*(n-1)/2]], {n, n}]];
G2Mat[g_Integer] := Quiet[If[g==0,{},G2Mat[g, Floor[(3 + Sqrt[8*Floor[Log[2, g]] + 1]) / 2]]],{Floor::meprec,N::meprec}];

(* Transform a graph to a list of edges and vice versa. *)
G2Edges::usage="G2Edges[g] Transforms integer representation g to edge list of the graph.";
G2Edges[g_Integer] := Select[Position[G2Mat[g], 1], Less @@ # &];

Mat2Edges::usage="Mat2Edges[m] transforms adjacency matrix m to edge list of the graph.";
Mat2Edges[mat_] := Select[Position[mat, 1], Less @@ # &];

Edges2Mat::usage="Edges2Mat[e] transforms edge list e to adjacency matrix of the graph.";
Edges2Mat[ed_List] :=
Module[{n = Max[ed], mat},
  If[Length[ed]==0,Return[{}]];
  mat = Table[0, {n}, {n}];
  (mat[[##]] = 1) & @@@ ed;
  Return[mat + Transpose[mat]];
];

Edges2G::usage="Edges2G[e] transforms edge list e to integer representation of the graph.";
Edges2G[ed_List] := Mat2G[Edges2Mat[ed]];


G2Graph::usage="G2Graph[g] transforms the integer representation g of a graph to a graph object. It takes the same options as Graph.";
Options[G2Graph]=Options[Graph];
SyntaxInformation[G2Graph]={"ArgumentsPattern"->{_,OptionsPattern[]}};
G2Graph[g_Integer,OptionsPattern[Graph]]:=AdjacencyGraph[G2Mat[g],First[#]->OptionValue[First[#]]&/@Options[Graph]]

G2GraphL::usage="G2GraphL[g] transforms the integer representation g of a graph to a graph object with automatic vertex labels.";
G2GraphL[g_Integer]:=AdjacencyGraph[G2Mat[g],VertexLabels->"Name"]

Graph2G::usage="Graph2G[g] transforms a graph object g to its integer representation.";
Graph2G[g_Graph]:=Edges2G[GEdges[g]]


(* edges and lists *)
GEdges::usage="GEdges[G] gives a list of pairs representing the edges of the graph G.";
GEdges[G_Graph]:={First[#],Last[#]}&/@EdgeList[G]

Pairs2Edges::usage="Pairs2Edges[{{v1,v2},...}] transforms the list of pairs to a list of edges of the form {v1<->v2,...}.";
Pairs2Edges[pairs_List]:=First[#]<->Last[#]&/@pairs

Edges2Pairs::usage="Edges2Pairs[{v1<->v2,...}] transforms the list of pairs to a list of edges of the form {{v1,v2},...}.";
Edges2Pairs[edges_List]:={First[#],Last[#]}&/@edges


(* Graph 6 format *)
Gs2Graph::usage="Gs2Graph[g] transforms Graph6 data type to a Graph object.";
Gs2Graph[gs_String]:=ImportString[gs,"Graph6"]

Gs2G::usage="Gs2G[g] transforms Graph6 data type to its integer representation.";
Gs2G[gs_String]:=Graph2G[Gs2Graph[gs]]

G2Gs::usage="G2Gs[g] transforms integer or Graph representation to Graph6 representation.";
G2Gs[g_Integer]:=G2Gs[G2Graph[g]]
G2Gs[G_Graph]:=StringTake[ExportString[G,"Graph6"],11;;-1]


(* Equality of graphs *)
EqualGraphQ::usage="EqualGraphQ[g1,g2] checks whether two graph objects g1, g2 are equal.";
EqualGraphQ[g1_Graph,g2_Graph]:=Sort[Sort/@GEdges[g1]]==Sort[Sort/@GEdges[g2]]


(* ::Subsubsection::Closed:: *)
(*Normal Form*)


(* Normal forms. *)
GraphNormalForm::usage="GraphNormalForm[g] computes a unique representative of the graph g within its class of isomorphic graphs.
	Here g can be an integer representation or a graph object. GraphNormalForm returns an integer representation.
	The input can also be an adjacency matrix, then the output is a matrix as well.";
GraphNormalForm[g_Graph] := Edges2G[GEdges[CanonicalGraph[g]]];
GraphNormalForm[g_Integer] := GraphNormalForm[G2Graph[g]];
GraphNormalForm[gr1_List] := GraphNormalForm[Graph[Mat2Edges[gr1]]];

StandardGraph::usage="StandardGraph[G] assures that the vertices of g are {1,2,...,n}.";
StandardGraph[G_Graph]:=Graph[Range[VertexCount[G]],EdgeList[G]/.Thread[Sort[VertexList[G]]->Range[VertexCount[G]]]]


(* ::Subsubsection::Closed:: *)
(*Graph Properties*)


VertexDegreeI::usage="VertexDegreeI[g] computes degrees for all vertices in the graph g (integer representation)";
VertexDegreeI[graphI_Integer]:=VertexDegree[G2Graph[graphI]]

VertexCountI::usage="VertexCountI[g] returns the number of vertices in the graph g (integer representation)";
VertexCountI[graphI_Integer]:=VertexCount[G2Graph[graphI]]

VertexListI::usage="VertexListI[g] returns the list of vertices in the graph g (integer representation)";
VertexListI[graphI_Integer]:=VertexList[G2Graph[graphI]]

EdgeCountI::usage="EdgeCountI[g] returns the number of edges in the graph g (integer representation)";
EdgeCountI[graphI_Integer]:=Length[G2Edges[graphI]]

EdgeListI::usage="EdgeListI[g] returns the list of edges (as pairs) in the graph g (integer representation)";
EdgeListI[graphI_Integer]:=G2Edges[graphI]


(* ::Subsubsection::Closed:: *)
(*Distance Equations*)


VertexPositions::usage="VertexPositions[g] gives the vertex positions of the given graph layout.";
VertexPositions[g_Integer]:=VertexPositions[G2Graph[g]]
VertexPositions[G_Graph]:=Last[First[AbsoluteOptions[G,VertexCoordinates]]]


DistanceEquations::usage="DistanceEquations[g,d] returns the equations determined by edges of the graph g.
The right hand side can be Symbolic, Random or Automatic (set by the RHS option)";
Options[DistanceEquations]={RHS->"Symbolic",Var->x,VarR->"a",RandomMin->1,RandomMax->15};
SyntaxInformation[DistanceEquations]={"ArgumentsPattern"->{_,_.,OptionsPattern[]}};
(* dim=2 *)
DistanceEquations[graph_Integer,opts:OptionsPattern[]]:=DistanceEquations[G2Graph[graph],FilterRules[{opts},Options[DistanceEquations]]]
DistanceEquations[graph_Graph,opts:OptionsPattern[]]:=DistanceEquations[graph,2,FilterRules[{opts},Options[DistanceEquations]]]/.{OptionValue[Var][a_,1]->x[a],OptionValue[Var][a_,2]->y[a]}
(* general dimension *)
DistanceEquations[graph_Integer,dim_Integer,opts:OptionsPattern[]]:=DistanceEquations[G2Graph[graph],dim,FilterRules[{opts},Options[DistanceEquations]]]
DistanceEquations[graph_Graph,dim_Integer,OptionsPattern[]]:=Module[{vertices,var,eq,pos},
  vertices=VertexList[graph];
  var=OptionValue[Var];
  pos=VertexPositions[graph];
  eq=Sum[(var[First[#],i]-var[Last[#],i])^2,{i,1,dim}]-
    If[OptionValue[RHS]=="Symbolic",
      OptionValue[VarR][#],
      If[OptionValue[RHS]=="Random",
        RandomInteger[{OptionValue[RandomMin],OptionValue[RandomMax]}],
        Sqrt[Plus@@((pos[[First[#]]]-pos[[Last[#]]])^2)]
      ]
    ]&/@(Sort/@GEdges[graph]);
  Return[eq];
]


EdgeLengthsFromPosition::usage="EdgeLengthsFromPosition[g] returns the lengths of the edges given by the graph object g.
EdgeLengthsFromPosition[g,p] returns the lengths of the edges given by the vertex positions from p.";
EdgeLengthsFromPosition[G_Graph]:=EdgeLengthsFromPosition[G,VertexPositions[G]]
EdgeLengthsFromPosition[G_Graph,pos_List]:=Module[{edges},
  edges=GEdges[G];
  Return[Sqrt[Plus@@((pos[[First[#]]]-pos[[Last[#]]])^2)]&/@edges];
]


(* ::Section::Closed:: *)
(*Tools*)


ParallelSelect::usage="ParallelSelect[list,func] works as Select[list,func] but uses parallel computation";
ParallelSelect[L_List,Q_Function]:=ParallelCombine[Select[#,Q]&,L]


(*Test whether a graph is an induced subgraph of given graph*)

SubgraphQ::usage="SubgraphQ[g,h] tests whether h is an induced subgraph of g.";
SubgraphQ::method="ReturnValue Option is not valid.";
Options[SubgraphQ]={ReturnValue->"Boolean",QMethod->"Triangles"};
SyntaxInformation[SubgraphQ]={"ArgumentsPattern"->{_,_,OptionsPattern[]}};
SubgraphQ[graphI_Integer,subI_Integer,OptionsPattern[]]:=SubgraphQ[G2Graph[graphI], G2Graph[subI],First[#]->OptionValue[First[#]]&/@Options[SubgraphQ]]
SubgraphQ[graph_Graph,sub_,OptionsPattern[]]:=Module[{res},
  res=Switch[OptionValue[ReturnValue],
    "Boolean",
    If[OptionValue[QMethod]==="Triangles",
      SubgraphTriQ[graph,sub],
      SubgraphQbool[graph,sub]
    ],
    "Certificate",
    SubgraphQcert[graph,sub],
    "All",
    SubgraphQall[graph,sub],
    _,
    Message[SubgraphQ::method]
  ];
  Return[res];
]

SubgraphQbool::usage="SubgraphQbool[g,h] tests whether h is an induced subgraph of g. Return is a Boolean.";
SubgraphQbool[graph_Graph,sub_Graph]:=Module[{k,V,Vsub,current},
  k=VertexCount[sub];
  V=VertexList[graph];
  Vsub=Subsets[V,{k}];
  While[Length[Vsub]>0,
    current=First[Vsub];
    If[IsomorphicGraphQ[Subgraph[graph,current],sub],
      Return[True];
    ];
    Vsub=Rest[Vsub];
  ];
  Return[False];
]

SubgraphQcert::usage="SubgraphQcert[g,h] tests whether h is an induced subgraph of g. If true, returns certificate.";
SubgraphQcert[graph_,sub_]:=Module[{k,V,Vsub,current},
  k=VertexCount[sub];
  V=VertexList[graph];
  Vsub=Subsets[V,{k}];
  While[Length[Vsub]>0,
    current=First[Vsub];
    If[IsomorphicGraphQ[Subgraph[graph,current],sub],
      Return[{True,current}];
    ];
    Vsub=Rest[Vsub];
  ];
  Return[{False}];
]

(*finds all ocuurences of the subgraph*)
SubgraphQall::usage="SubgraphQall[g,h] tests whether h is an induced subgraph of g. Return is a list of vertices that form these subgraph.";
SubgraphQall[graph_Graph,sub_Graph]:=Module[{k,V,Vsub,current,subL={}},
  k=VertexCount[sub];
  V=VertexList[graph];
  Vsub=Subsets[V,{k}];
  While[Length[Vsub]>0,
    current=First[Vsub];
    If[IsomorphicGraphQ[Subgraph[graph,current],sub],
      subL=Append[subL,current];
    ];
    Vsub=Rest[Vsub];
  ];
  Return[subL];
]

(*Faster Version using triangle Subgraphs of subI*)
SubgraphTriQ::usage="SubgraphTriQ[g,h] tests whether h is an induced subgraph of g. The method uses triangle subgraphs to be faster.";
SubgraphTriQ[graphI_Integer,subI_Integer]:=SubgraphTriQ[G2Graph[graphI],G2Graph[subI]]
SubgraphTriQ[graph_Graph,sub_Graph]:=Module[{cyc,subL,V,Vsub,current,k},
  If[IsomorphicGraphQ[sub,G2Graph[7]],
    cyc=FindCycle[graph,{3},1];
    If[Length[cyc]>0,Return[True],Return[False]]
  ];
  If[SubgraphTriQ[sub,G2Graph[7]],
    cyc=FindCycle[graph,{3},All];   
    subL=VertexList/@cyc;
    If[Length[subL]==0,
      Return[False]];
    k=VertexCount[sub];
    V=VertexList[graph];
    Vsub=Flatten[Function[tri,Sort@Join[tri,#]&/@Subsets[Complement[V,tri],{k-3}]]/@subL,1];
    While[Length[Vsub]>0,
      current=First[Vsub];
      If[IsomorphicGraphQ[Subgraph[graph,current],sub],
        Return[True];
      ];
      Vsub=Rest[Vsub];
    ];
    Return[False],
    Return[SubgraphQ[graph,sub,QMethod->"Normal"]]
  ];
]

(*Select from a List all graphs which have given subgraph*)
SubgraphSelect::usage="SubgraphSelect[list,h] selects those graphs in list that contain h as an induced subgraph.";
SubgraphSelect[graphs_List,sub_Integer]:=ParallelSelect[graphs,SubgraphTriQ[#,sub]&]


DeleteIsolatedVertices::usage="DeleteIsolatedVertices[g] removes all vertices with degree 0 (vertex names might be renamed).";
DeleteIsolatedVertices[G_Graph]:=StandardGraph[VertexDelete[G,Select[VertexList[G],VertexDegree[G,#]==0&]]]


VertexNeighbors::usage="VertexNeighbors[g,v] returns the neighbors of v in g.";
VertexNeighbors[G_Graph,v_]:=Complement[VertexList[NeighborhoodGraph[G,v]],{v}]


DegreeKVertices[G_Graph,k_Integer]:=Select[VertexList[G],VertexDegree[G,#]==k&]


(* ::Section:: *)
(*Constructions*)


(* ::Subsection::Closed:: *)
(*Extensions*)


(* k-extensions or Henneberg steps are constructions that play an important role in rigidity theory
 * 0-extensions and 1-extensions preserve rigidity
 * In a k-extension k edges are removed and a new vertex is added together with edges to the vertices of the deleted edges. Additional edges from the new vertex are added to obtain the right edge count.
 * Here, we do not allow multi-edges and loops.
 * All graphs are assumed to be connected.
 *)


KExtensions::usage="KExtensions[g,d] constructs all graphs that can be obtained from g (in integer representation) by a single k-extension step in dimension d,
 where k runs from SetStart (default: 0) to SetLimit (default: All; representing d-1).";

Options[KExtensions]={SetLimit->All,SetStart->0,UseSubgraphOnChosenVertices->False,UseIntersectionNumberOfChosenVertices->False};
KExtensions::sub="The given input `1 ` for option UseSubgraphOnChosenVertices is neither a graph nor False.";
KExtensions::inter="The given input `1 ` for option UseIntersectionNumberOfChosenVertices is neither a non-negative integer nor False.";
KExtensions[g_Integer,opts:OptionsPattern[]] :=KExtensions[g,2,FilterRules[{opts},Options[KExtensions]]]
KExtensions[g_Integer,dim_,OptionsPattern[]]:= Module[{gr,n,edges,allsteps,k,stepd,delV,num,edpk,lim,sub,inter},
  gr = G2Mat[g];
  If[(lim=OptionValue[SetLimit])==All,lim=dim-1];
  n = Length[gr]+1;
  gr = PadRight[gr, {n, n}];
  edges = G2Edges[g];
  allsteps={};
  For[k=OptionValue[SetStart],k+dim<=n&&k<=lim,k++,(*k defines which extension step*)
    num=Binomial[Length[edges],k];
    stepd=DeleteCases[Union[Flatten[
      (* Perform all possible type k-extensions. *)
      Table[
         edpk=Subsets[edges,{k},{i}];
         delV=Union[Flatten[edpk]]; 
         If[(inter=OptionValue[UseIntersectionNumberOfChosenVertices])===False||(If[Head[inter]===Integer&&inter>=0,True,Message[KExtensions::inter,inter];False]&&Length[delV]==2k-inter),                 
         Function[points,
          (*Check for option UseSubgraphOnChosenVertices*)
          check=Catch[If[(sub=OptionValue[UseSubgraphOnChosenVertices])=!=False,
            If[Head[sub]===Integer,sub=G2Graph[sub]];
            If[Head[sub]===Graph,
              If[Not[IsomorphicGraphQ[DeleteIsolatedVertices[Subgraph[Graph[edges],Union[points,delV]]],sub]],Throw[False]]
              ,
              Message[KExtensions::sub,OptionValue[UseSubgraphOnChosenVertices]];
              Break[]
            ];
          ]];
          If[check===False,Null,
          (*main part*)
          GraphNormalForm[ReplacePart[gr, 
            Flatten[{{#->0,Reverse[#]->0}&/@edpk,
             {{#, n} -> 1, {n, #} -> 1}&/@Union[points,delV]
            }]
          ]]]
         ]/@ Subsets[Complement[Range[n - 1], delV],{dim+k-Length[delV]}]
        ],
        {i,num}
      ]
    ]],Null];
    allsteps=Union[allsteps,stepd];
 ];
  Return[allsteps];
];


K0Extensions::usage="K0Extensions[g,d] applies all 0-extensions to a graph g in dimension d.";
K0Extensions[g_Integer]:=K0Extensions[g,2]
K0Extensions[g_Integer,dim_Integer]:=KExtensions[g,dim,SetLimit->0]

K1Extensions::usage="K1Extensions[g,d] applies all 1-extensions to a graph g in dimension d.";
K1Extensions[g_Integer]:=K1Extensions[g,2]
K1Extensions[g_Integer,dim_Integer]:=KExtensions[g,dim,SetLimit->1,SetStart->1]


K0Extension::usage="K0Extension[g,{v1,v2,...,vd}] applies a 0-extension to a graph g on the vertices v1 to vd in dimension d.";
K0Extension::vertex="Input vertex `1` is not a part of the graph.";
K0Extension[g_Integer,vert_List]:=Check[Graph2G[Graph[K0Extension[G2Edges[g],vert]]],Null]
K0Extension[G_Graph,vert_List]:=Check[Graph[K0Extension[GEdges[G],vert]],Null]
K0Extension[edges_List,vert_List]:=Module[{n,VV,test},
  n=Max[edges];
  VV=Union@@edges;
  test=If[Not[MemberQ[VV,#]],Message[K0Extension::vertex,#];True,False]&/@vert;
  If[Or@@test,Return[]];
  Return[Join[edges,{#,n+1}&/@vert]];
]

K1Extension::usage="K1Extension[g,{v_1,v_2},{v_3,...,v_{d+1}}] applies a 1-extension to a graph g deleting edge {v_1,v_2} and adding edges for the vertices v_3 to v_{d+1} in dimension d.";
K1Extension::vertex="Input vertex `1` is not a part of the graph.";
K1Extension::edge="Input edge `1` is not a member of the graph edges `2`.";
K1Extension[g_Integer,edge_List,vert_List]:=Check[Graph2G[Graph[K1Extension[G2Edges[g],edge,vert]]],Null]
K1Extension[G_Graph,edge_List,vert_List]:=Check[Graph[K1Extension[GEdges[G],edge,vert]],Null]
K1Extension[edges_List,edge_List,vert_List]:=Module[{n,VV,test},
  n=Max[edges];
  VV=Union@@edges;
  If[Not[MemberQ[Sort/@edges,Sort[edge]]],Return[Message[K1Extension::edge,edge,edges]]];
  test=If[Not[MemberQ[VV,#]],Message[K1Extension::vertex,#];True,False]&/@vert;
  If[Or@@test,Return[]];
  Return[Join[Complement[Sort/@edges,{Sort[edge]}],{#,n+1}&/@Join[edge,vert]]];
]


(* ::Subsection::Closed:: *)
(*Specific Extensions*)


K0ExtensionD2Sub1::usage="K0ExtensionD2Sub1[g] constructs all graphs that can be obtained from g (in integer representation) by a single 0-extension step in dimension 2, where the chosen vertices are connected by an edge.";
K0ExtensionD2Sub1[g_Integer]:=Module[{edges},
  edges=G2Edges[g];
  Return[DeleteDuplicates[GraphNormalForm[Edges2G[K0Extension[edges,#]]]&/@edges]];
]
K0ExtensionD2Sub0::usage="K0ExtensionD2Sub0[g] constructs all graphs that can be obtained from g (in integer representation) by a single 0-extension step in dimension 2, where the chosen vertices are not connected by an edge.";
K0ExtensionD2Sub0[g_Integer]:=Module[{vertices,edges},
  edges=G2Edges[g];
  vertices=Union@@edges;
  Return[DeleteDuplicates[GraphNormalForm[Edges2G[K0Extension[edges,#]]]&/@Complement[Sort/@Subsets[vertices,{2}],Sort/@edges]]];
]

K1ExtensionD2Sub7::usage="K1ExtensionD2Sub7[g] constructs all graphs that can be obtained from g (in integer representation) by a single 1-extension step in dimension 2, where the chosen additional vertex is connected to both vertices of the chosen edge.";
K1ExtensionD2Sub7[g_Integer]:=Module[{edges},
  edges=G2Edges[g];
  Return[DeleteDuplicates[Flatten[(GraphNormalForm[Edges2G[K1Extension[edges,#[[1;;2]],{#[[3]]}]]])&/@Table[RotateLeft[#,i],{i,0,2}]&/@Select[(Union@@#)&/@Subsets[edges,{3}],Length[#]==3&]]]];
]
K1ExtensionD2Sub3::usage="K1ExtensionD2Sub3[g] constructs all graphs that can be obtained from g (in integer representation) by a single 1-extension step in dimension 2, where the chosen additional vertex is connected to exactly one vertex of the chosen edge.";
K1ExtensionD2Sub3[g_Integer]:=Module[{edges},
  edges=G2Edges[g];
  Return[DeleteDuplicates[Flatten[{GraphNormalForm[Edges2G[K1Extension[edges,#[[1]],Complement[#[[2]],#[[1]]]]]],GraphNormalForm[Edges2G[K1Extension[edges,#[[2]],Complement[#[[1]],#[[2]]]]]]}&/@Select[Subsets[edges,{2}],Length[Union@@#]==3&&Not[MemberQ[edges,Sort[(Complement[Union[Flatten[#]],Intersection[First[#],Last[#]]]&)[#]]]]&]]]];
]
K1ExtensionD2Sub1::usage="K1ExtensionD2Sub1[g] constructs all graphs that can be obtained from g (in integer representation) by a single 1-extension step in dimension 2, where the chosen additional vertex is connected to none of the vertices of the chosen edge.";
K1ExtensionD2Sub1[g_Integer]:=Module[{edges,vertices},
  edges=G2Edges[g];
  vertices=DeleteDuplicates[Flatten[edges]];
  Return[DeleteCases[DeleteDuplicates[Flatten[Function[edge,Function[vertex,If[Not[MemberQ[edges,Sort[{edge[[1]],vertex}]]||MemberQ[edges,Sort[{edge[[2]],vertex}]]],GraphNormalForm[Edges2G[K1Extension[edges,edge,{vertex}]]]]]/@Complement[vertices,edge]]/@edges]],Null]];
]


XReplacement::usage="XReplacement[g,d] constructs all graphs that can be obtained from g (in integer representation) by a single X-Replacement (i.e. 2-extension where the chosen edges have no vertex in common).";
XReplacement[g_Integer,dim_Integer] :=KExtensions[g,dim,UseIntersectionNumberOfChosenVertices->0,SetStart->2,SetLimit->2]

VReplacement::usage="VReplacement[g,d] constructs all graphs that can be obtained from g (in integer representation) by a single V-Replacement (i.e. 2-extension where the chosen edges have a vertex in common).";
VReplacement[g_Integer,dim_Integer] :=KExtensions[g,dim,UseIntersectionNumberOfChosenVertices->1,SetStart->2,SetLimit->2]


(* ::Subsection::Closed:: *)
(*Coning*)


Coning::usage="Coning[g] gives a graph with one more vertex that is connected to all the existing ones.";
Coning[g_Integer]:=GraphNormalForm[Coning[G2Graph[g]]]
Coning[G_Graph]:=Graph[Join[EdgeList[G],UndirectedEdge[VertexCount[G]+1,#]&/@VertexList[G]]]


(* ::Subsection::Closed:: *)
(*Splittings*)


(* ::Subsubsection::Closed:: *)
(*Vertex Splitting*)


VertexSplitting::usage="VertexSplitting[g,d,v,W,set1] does a vertex splitting construction on a graph g (Integer or Graph) in dimension d for a specified vertex v and two neighbors W of v and a set of additional neighbors set1.
VertexSplitting[g,d] constructs all graphs achievable by vertex splitting in g (integer representation) in dimension d.";
VertexSplitting::noneighbour="Some vertex in `1` is not adjacent to `2`.";
VertexSplitting::neighbourlist="The list `1` is not of correct size.";
VertexSplitting::notvertex="The arguement `1` is not a vertex of the graph `2`.";
VertexSplitting::overlap="Vertex `1` is member of the set `2`.";
VertexSplitting[g_Integer,dim_Integer,v_,W_List,N1_List]:=VertexSplitting[G2Graph[g],dim,v,W,N1]
VertexSplitting[G_Graph,dim_Integer,v_,W_List,N1_List]:=Module[{edges,new,neigh,n1edges,w},
  edges=GEdges[G];
  new=Max[edges]+1;
  neigh=DeleteCases[Flatten[Select[edges,MemberQ[#,v]&]],v];
  If[Not[VertexQ[G,v]],Message[VertexSplitting::notvertex,v,G];Return[]];
  If[Not[Length[W]==dim-1],Message[VertexSplitting::neighbourlist,W];Return[]];
  Do[
    If[Not[VertexQ[G,w]],Message[VertexSplitting::notvertex,w,G];Return[]];
    If[Not[MemberQ[neigh,w]],Message[VertexSplitting::noneighbour,{w},v];Return[]];
    If[MemberQ[N1,w],Message[VertexSplitting::overlap,w,N1];Return[]],
  {w,W}];
  If[Not[SubsetQ[neigh,N1]],Message[VertexSplitting::noneighbour,N1,v];Return[]];
  n1edges=Select[edges,MemberQ[#,v]&&Length[Intersection[#,N1]]>0&];
  If[Length[n1edges]>0,
   Return[EdgeAdd[EdgeDelete[G,Pairs2Edges[n1edges]],Join[new<->#&/@W,{new<->v},Pairs2Edges[(n1edges/.v->new)]]]],
   Return[EdgeAdd[G,Join[new<->#&/@W,{new<->v},Pairs2Edges[(n1edges/.v->new)]]]]
  ];
]

VertexSplitting[graph_Integer]:=VertexSplitting[graph,2]
VertexSplitting[G_Graph]:=VertexSplitting[G,2]
VertexSplitting[graph_Integer,dim_Integer]:=VertexSplitting[G2Graph[graph],dim]
VertexSplitting[G_Graph,dim_Integer]:=Module[{V,neigh, neigh2,res},
  V=VertexList[G];
  res=Function[vertex,
    neigh=DeleteCases[VertexList[NeighborhoodGraph[G,vertex]],vertex];
    Function[W,
      neigh2=Complement[neigh,W];
      Function[subset1,
        VertexSplitting[G,dim,vertex,W,subset1]
      ]/@Subsets[neigh2]
    ]/@Subsets[neigh,{dim-1}]
  ]/@V;
  Return[DeleteDuplicates[GraphNormalForm/@Flatten[res]]];
]

StrictVertexSplitting::usage="StrictVertexSplitting[g,d] constructs all graphs achievable by vertex splitting in g (integer representation) in dimension d where the two chosen neighboring sets are non-empty";
StrictVertexSplitting[graph_Integer]:=StrictVertexSplitting[graph,2]
StrictVertexSplitting[graph_Integer,dim_Integer]:=StrictVertexSplitting[G2Graph[graph],dim]
StrictVertexSplitting[G_Graph,dim_Integer]:=Module[{V,neigh, neigh2,res},
  V=VertexList[G];
  res=Function[vertex,
    neigh=DeleteCases[VertexList[NeighborhoodGraph[G,vertex]],vertex];
    Function[W,
      neigh2=Complement[neigh,W];
      Function[subset1,
        VertexSplitting[G,dim,vertex,W,subset1]
      ]/@Subsets[neigh2,{1,Length[neigh2]-1}]
    ]/@Subsets[neigh,{dim-1}]
  ]/@V;
  Return[DeleteDuplicates[GraphNormalForm/@Flatten[res]]];
]


(* ::Subsubsection::Closed:: *)
(*VertexToK22d/VertexToC4/Diamond/Spider Splitting*)


SpiderSplitting::usage="SpiderSplitting[g,v,W,S] does a spider splitting construction on a graph g (Integer or Graph) for a specified vertex v and a set of neighbors W of v and a set of additional neighbors S.
SpiderSplitting[g] constructs all graphs achievable by spider splitting in g";
SpiderSplitting::noneighbour="Some vertex in `1` is not adjacent to `2`.";
SpiderSplitting::notvertex="The arguement `1` is not a vertex of the graph `2`.";
SpiderSplitting::overlap="Vertex `1` is member of the set `2`.";
SpiderSplitting[g_Integer,v_,W_List,N1_List]:=SpiderSplitting[G2Graph[g],v,W,N1]
SpiderSplitting[G_Graph,v_,W_List,N1_List]:=Module[{edges,new,neigh,n1edges,check},
  edges=GEdges[G];
  new=Max[edges]+1;
  neigh=DeleteCases[Flatten[Select[edges,MemberQ[#,v]&]],v];
  check=Catch[
    If[Not[VertexQ[G,v]],Message[SpiderSplitting::notvertex,v,G];Throw[False]];
    If[Not[VertexQ[G,#]],Message[SpiderSplitting::notvertex,#,G];Throw[False]]&/@W;
    If[Not[MemberQ[neigh,#]],Message[SpiderSplitting::noneighbour,{#},v];Throw[False]]&/@W;
    If[Not[SubsetQ[neigh,N1]],Message[SpiderSplitting::noneighbour,N1,v];Throw[False]];
    If[MemberQ[N1,#],Message[SpiderSplitting::overlap,#,N1];Throw[False]]&/@W;
  ];
  n1edges=Select[edges,MemberQ[#,v]&&Length[Intersection[#,N1]]>0&];
  If[Length[n1edges]>0,
   Return[EdgeAdd[EdgeDelete[G,Pairs2Edges[n1edges]],Join[new<->#&/@W,Pairs2Edges[(n1edges/.v->new)]]]],
   Return[EdgeAdd[G,Join[new<->#&/@W,Pairs2Edges[(n1edges/.v->new)]]]]
  ];
]

SpiderSplitting[graph_Integer]:=SpiderSplitting[G2Graph[graph],2]
SpiderSplitting[G_Graph]:=SpiderSplitting[G,2]
SpiderSplitting[graph_Integer,dim_Integer]:=SpiderSplitting[G2Graph[graph],dim]
SpiderSplitting[G_Graph,dim_Integer]:=Module[{V,neigh, neigh2,res},
  V=VertexList[G];
  res=Function[vertex,
    neigh=DeleteCases[VertexList[NeighborhoodGraph[G,vertex]],vertex];
    Function[W,
      neigh2=Complement[neigh,W];
      Function[subset1,
        SpiderSplitting[G,vertex,W,subset1]
      ]/@Subsets[neigh2]
    ]/@Subsets[neigh,{dim}]
  ]/@V;
  Return[DeleteDuplicates[GraphNormalForm/@Flatten[res]]];
]


(*Synonyms*)
DiamondSplitting::usage=StringReplace[SpiderSplitting::usage,{"Spider"->"Diamond","spider"->"diamond"}];
DiamondSplitting[g_Integer,v_,W_List,N1_List]:=SpiderSplitting[G2Graph[g],v,W,N1]
DiamondSplitting[G_Graph,v_,W_List,N1_List]:=SpiderSplitting[G,v,W,N1]
DiamondSplitting[graph_Integer]:=SpiderSplitting[G2Graph[graph],2]
DiamondSplitting[G_Graph]:=SpiderSplitting[G,2]
DiamondSplitting[graph_Integer,dim_Integer]:=SpiderSplitting[G2Graph[graph],dim]
DiamondSplitting[G_Graph,dim_Integer]:=SpiderSplitting[G,dim]

VertexToK22dSplitting::usage=StringReplace[SpiderSplitting::usage,{"Spider"->"VertexToK2d","spider"->"VertexToK2d"}];
VertexToK22dSplitting[g_Integer,v_,W_List,N1_List]:=SpiderSplitting[G2Graph[g],v,W,N1]
VertexToK22dSplitting[G_Graph,v_,W_List,N1_List]:=SpiderSplitting[G,v,W,N1]
VertexToK22dSplitting[graph_Integer]:=SpiderSplitting[G2Graph[graph],2]
VertexToK22dSplitting[G_Graph]:=SpiderSplitting[G,2]
VertexToK22dSplitting[graph_Integer,dim_Integer]:=SpiderSplitting[G2Graph[graph],dim]
VertexToK22dSplitting[G_Graph,dim_Integer]:=SpiderSplitting[G,dim]

VertexToC4Splitting::usage=StringReplace[SpiderSplitting::usage,{"Spider"->"VertexToC4","spider"->"VertexToC4"}];
VertexToC4Splitting[graph_Integer]:=SpiderSplitting[G2Graph[graph],2]
VertexToC4Splitting[G_Graph]:=SpiderSplitting[G,2]


(* ::Subsection::Closed:: *)
(*Reductions*)


K0Reduction::usage="K0Reduction[g,d,v] applies a 0-reduction on g with vertex v in dimension d.";
K0Reduction::vertex="Input `1` is not a vertex of the graph or does not have the right degree.";
K0Reduction[g_Integer,v_]:=K0Reduction[g,2,v]
K0Reduction[G_Graph,v_]:=K0Reduction[G,2,v]
K0Reduction[g_Integer,dim_Integer,v_]:=Graph2G[K0Reduction[G2Graph[g],dim,v]]
K0Reduction[G_Graph,dim_Integer,v_]:=Module[{G2},
  If[VertexDegree[G,v]!=dim,Message[K0Reduction::vertex,v];Return[]];
  G2=VertexDelete[G,v];
  Return[G2]
]

K1Reduction::usage="K1Reduction[g,d,v] applies a 1-reduction on g with vertex v in dimension d.
With the option PickEdges it can be specified how many/which of the possible reductions are considered.";
K1Reduction::vertex="Input `1` is not a vertex of the graph or does not have the right degree.";
K1Reduction::index="Index `1` in PickEdges is out of bound.";
K1Reduction::pick="Option value for PickEdges is not valid.";
Options[K1Reduction]={PickEdges->All,ShowDetails->False};
K1Reduction[g_Integer,v_,opts:OptionsPattern[]]:=K1Reduction[g,2,v,opts]
K1Reduction[G_Graph,v_,opts:OptionsPattern[]]:=K1Reduction[G,2,v,opts]
K1Reduction[g_Integer,dim_Integer,v_,opts:OptionsPattern[]]:=Graph2G/@K1Reduction[G2Graph[g],dim,v,opts]
K1Reduction[G_Graph,dim_Integer,v_,opts:OptionsPattern[]]:=Module[{G2,neigh,sedges,freeedges,pick},
  If[VertexDegree[G,v]!=dim+1,Message[K1Reduction::vertex,v];Return[]];
  neigh=Cases[VertexList[NeighborhoodGraph[G,v]],Except[v]];
  sedges=GEdges[Subgraph[G,neigh]];
  freeedges=Complement[Subsets[neigh,{2}],sedges];
  Switch[
    (pick=OptionValue[PickEdges]),
    _Integer,
      If[pick>0,freeedges=freeedges[[1;;UpTo[pick]]],Message[K1Reduction::index,pick]],
    _List,
      freeedges=freeedges[[pick]],
    All,
     Nothing,
    _,
      Message[K1Reduction::pick]
  ];
  G2=VertexDelete[G,v];
  G2=EdgeAdd[G2,UndirectedEdge@@#]&/@freeedges;
  Return[G2]
]


K0Reductions::usage="K0Reductions[g,d] gives all 0-reductions of the graph g in dimension d.";
Options[K0Reductions]={ShowDetails->False,UseNormalForm->False,Unify->False};
K0Reductions[g_Integer,opts:OptionsPattern[]]:=K0Reductions[g,2,opts]
K0Reductions[G_Graph,opts:OptionsPattern[]]:=K0Reductions[G,2,opts]
K0Reductions[g_Integer,dim_Integer,opts:OptionsPattern[]]:=If[OptionValue[UseNormalForm],Identity,Graph2G]/@K0Reductions[G2Graph[g],dim,opts]
K0Reductions[G_Graph,dim_Integer,opts:OptionsPattern[]]:=
  If[OptionValue[Unify],Union,Identity]@
  If[OptionValue[ShowDetails],
    {If[OptionValue[UseNormalForm],GraphNormalForm,Identity]@K0Reduction[G,dim,#],Graph2G[VertexDelete[NeighborhoodGraph[G,#],#]]}&/@DegreeKVertices[G,dim],
    If[OptionValue[UseNormalForm],GraphNormalForm,Identity]@K0Reduction[G,dim,#]&/@DegreeKVertices[G,dim]
  ]

K1Reductions::usage="K1Reductions[g,d] gives all 0-reductions of the graph g in dimension d.";
Options[K1Reductions]={ShowDetails->False,UseNormalForm->False,Unify->False};
K1Reductions[g_Integer,opts:OptionsPattern[]]:=K1Reductions[g,2,opts]
K1Reductions[G_Graph,opts:OptionsPattern[]]:=K1Reductions[G,2,opts]
K1Reductions[g_Integer,dim_Integer,opts:OptionsPattern[]]:=If[OptionValue[UseNormalForm],Identity,Graph2G]/@K1Reductions[G2Graph[g],dim,opts]
K1Reductions[G_Graph,dim_Integer,opts:OptionsPattern[]]:=
  If[OptionValue[Unify],Union,Identity]@
  Flatten[If[OptionValue[ShowDetails],
    Function[v,{#,GraphNormalForm[VertexDelete[NeighborhoodGraph[G,v],v]]}&/@(If[OptionValue[UseNormalForm],GraphNormalForm,Identity]/@K1Reduction[G,dim,v])]/@DegreeKVertices[G,dim+1],
    If[OptionValue[UseNormalForm],GraphNormalForm,Identity]/@K1Reduction[G,dim,#]&/@DegreeKVertices[G,dim+1]
  ],1]


(* ::Section:: *)
(*Rigidity Check*)


(* ::Subsection::Closed:: *)
(*Rigidity Matrix*)


RigidityMatrix::usage="RigidityMatrix[g,r] computes the rigidity matrix for graph g with placement r of vertices.";
RigidityMatrix[G_Integer,rho_List]:=RigidityMatrix[G2Graph[G],rho]
RigidityMatrix[G_Graph,rho_List]:=Module[{R,EE},
  R=Table[0*First[rho],{i,EdgeCount[G]},{j,VertexCount[G]}];
  EE=GEdges[G];
  MapIndexed[(
    R[[First[#2],First[#1]]]=rho[[First[#1]]]-rho[[Last[#1]]];
    R[[First[#2],Last[#1]]]=rho[[Last[#1]]]-rho[[First[#1]]]
  )&,EE];
  Return[Flatten/@R]
]

RandomRigidityMatrix::usage="RandomRigidityMatrix[g,d] computes the rigidity matrix for graph g with a random placement of vertices in R^d.";
Options[RandomRigidityMatrix]={RandomRange->Automatic,RandomSet->"Reals"};
RandomRigidityMatrix[G_Integer,opts:OptionsPattern[]]:=RandomRigidityMatrix[G,2,FilterRules[{opts},Options[RandomRigidityMatrix]]]
RandomRigidityMatrix[G_Graph,opts:OptionsPattern[]]:=RandomRigidityMatrix[G,2,FilterRules[{opts},Options[RandomRigidityMatrix]]]
RandomRigidityMatrix[G_Integer,dim_Integer,opts:OptionsPattern[]]:=RandomRigidityMatrix[G2Graph[G],dim,FilterRules[{opts},Options[RandomRigidityMatrix]]]
RandomRigidityMatrix[G_Graph,dim_Integer,OptionsPattern[]]:=RigidityMatrix[G,
Switch[
  OptionValue[RandomSet],
  "Integers",
    RandomInteger[{-1,1}*If[NumberQ[OptionValue[RandomRange]],OptionValue[RandomRange],10^6*dim*VertexCount[G]],{VertexCount[G],dim}],(*Chosen according to [5]*)
  _,
    RandomReal[{-1,1}*If[NumberQ[OptionValue[RandomRange]],OptionValue[RandomRange],10^6*dim*VertexCount[G]],{VertexCount[G],dim}]
  ]
]
  
  

SymbolicRigidityMatrix::usage="SymbolicRigidityMatrix[g,d,x] computes the rigidity matrix for graph g with a symbolic placement of vertices in R^d using x as a symbol for the variable.";
SymbolicRigidityMatrix[G_Integer,x_Symbol]:=SymbolicRigidityMatrix[G,2,x]
SymbolicRigidityMatrix[G_Graph,x_Symbol]:=SymbolicRigidityMatrix[G,2,x]
SymbolicRigidityMatrix[g_Integer,dim_Integer,x_Symbol]:=SymbolicRigidityMatrix[G2Graph[g],dim,x]
SymbolicRigidityMatrix[G_Graph,dim_Integer,x_Symbol]:=RigidityMatrix[G,Table[x[i,#]&/@Range[dim],{i,VertexCount[G]}]]


(* ::Subsection::Closed:: *)
(*Rigid Graph*)


RigidGraphQ::usage="RigidGraphQ[g,d] checks whether the Graph represented by g is rigid in dimension d.
Methods available are:
	RigidityMatrix: all dimensions, probabilistic (false negatives possible)";
RigidGraphQ::method="Method is not valid.";
Options[RigidGraphQ]={Method->"RandomRigidityMatrix",UseCount->True,RandomRigidityMatrixRange->Automatic,RandomSet->"Reals"};
SyntaxInformation[RigidGraphQ]={"ArgumentsPattern"->{_,_.,OptionsPattern[]}};
RigidGraphQ[g_Integer,opts:OptionsPattern[]]:=RigidGraphQ[g,2,FilterRules[{opts},Options[RigidGraphQ]]]
RigidGraphQ[G_Graph,opts:OptionsPattern[]]:=RigidGraphQ[G,2,FilterRules[{opts},Options[RigidGraphQ]]]
RigidGraphQ[g_Integer,dim_Integer,opts:OptionsPattern[]]:=RigidGraphQ[G2Graph[g],dim,FilterRules[{opts},Options[RigidGraphQ]]]
RigidGraphQ[G_Graph,dim_Integer,opts:OptionsPattern[]]:=Module[{},
  If[OptionValue[UseCount],If[EdgeCount[G]<dim VertexCount[G]-Binomial[dim+1,2],Return[False]]];
  Return[Switch[
    OptionValue[Method],
    "RandomRigidityMatrix",
      RigidGraphQRandomMatrix[G,dim,FilterRules[{opts},Options[RigidGraphQRandomMatrix]]],
    "SymbolicRigidityMatrix",
      RigidGraphQSymbolicMatrix[G,dim,FilterRules[{opts},Options[RigidGraphQSymbolicMatrix]]],
    _,
      Message[RigidGraphQ::method]
  ]];
]


RigidGraphQRandomMatrix::usage="RigidGraphQRandomMatrix[g,d] checks whether a random rigidity matrix has correct rank for being rigid in dimension d for a random placement of vertices.";
Options[RigidGraphQRandomMatrix]={RandomRigidityMatrixRange->Automatic,RandomSet->"Reals"};
RigidGraphQRandomMatrix[g_Integer,opts:OptionsPattern[]]:=RigidGraphQRandomMatrix[g,2,FilterRules[{opts},Options[RigidGraphQRandomMatrix]]]
RigidGraphQRandomMatrix[G_Graph,opts:OptionsPattern[]]:=RigidGraphQRandomMatrix[G,2,FilterRules[{opts},Options[RigidGraphQRandomMatrix]]]
RigidGraphQRandomMatrix[g_Integer,dim_Integer,opts:OptionsPattern[]]:=RigidGraphQRandomMatrix[G2Graph[g],dim,FilterRules[{opts},Options[RigidGraphQRandomMatrix]]]
RigidGraphQRandomMatrix[G_Graph,dim_Integer,OptionsPattern[]]:= EdgeCount[G]>0&&(MatrixRank[RandomRigidityMatrix[G,dim,RandomRange->OptionValue[RandomRigidityMatrixRange],RandomSet->OptionValue[RandomSet]]]==dim VertexCount[G]-Binomial[dim+1,2])


RigidGraphQSymbolicMatrix::usage="RigidGraphQSymbolicMatrix[g,d] checks whether the symbolic rigidity matrix has correct rank for being rigid in dimension d for a random placement of vertices.";
Options[RigidGraphQSymbolicMatrix]={Var->x};
RigidGraphQSymbolicMatrix[g_Integer,opts:OptionsPattern[]]:=RigidGraphQSymbolicMatrix[g,2,FilterRules[{opts},Options[RigidGraphQSymbolicMatrix]]]
RigidGraphQSymbolicMatrix[G_Graph,opts:OptionsPattern[]]:=RigidGraphQSymbolicMatrix[G,2,FilterRules[{opts},Options[RigidGraphQSymbolicMatrix]]]
RigidGraphQSymbolicMatrix[g_Integer,dim_Integer,opts:OptionsPattern[]]:=RigidGraphQSymbolicMatrix[G2Graph[g],dim,FilterRules[{opts},Options[RigidGraphQSymbolicMatrix]]]
RigidGraphQSymbolicMatrix[G_Graph,dim_Integer,OptionsPattern[]]:= EdgeCount[G]>0&&(MatrixRank[SymbolicRigidityMatrix[G,dim,OptionValue[Var]]]==dim VertexCount[G]-Binomial[dim+1,2])


(* ::Subsection::Closed:: *)
(*Rigid Framework*)


InfinitesimallyRigidFrameworkQ::usage="InfinitesimallyRigidFrameworkQ[g,r] checks whether the rigidity matrix has correct rank for being rigid for a given placement r of vertices.";
InfinitesimallyRigidFrameworkQ::length="Placement has invalid length or dimensions.";
InfinitesimallyRigidFrameworkQ[g_Integer,r_List]:=InfinitesimallyRigidFrameworkQ[G2Graph[g],r]
InfinitesimallyRigidFrameworkQ[G_Graph,r_List]:=Module[{n,dim},
  n=VertexCount[G];
  dim=Length[First[r]];
  If[Length[r]==n&&(And@@(Length[#]==dim&/@r)),MatrixRank[RigidityMatrix[G,r]]==dim*n-Binomial[dim+1,2],Message[RigidFrameworkQ::length];False]
]


(* ::Subsection::Closed:: *)
(*Minimally Rigid Graph*)


MinRigidGraphQ::usage="MinRigidGraphQ[g,d] checks whether the Graph represented by g is minimally rigid in dimension d.
Methods available are
	RandomRigidityMatrix (default): all dimensions, probabilistic (false negatives possible); checks the rank of a random rigidity matrix.
	SymbolicRigidityMatrix: all dimensions, deterministic (slow); checks the rank of the symbolic rigidity matrix.
	RealizationsCount: all dimensions but only few vertices, probabilistic (false negatives possible, except dimension 2); computes the number of realizations and checks for positive finiteness.
	Sequence: dimension 2, deterministic; searches for a sequence of 0- and 1-extensions.
	Subgraph: dimension 2, deterministic; checks whether the graph is tight by edge counting on subgraphs.
	Pebble (recommended): dimension 2, deterministic; checks whether the graph is tight by the pebble game algorithm.
	TwoSpanningTrees: dimension 2, deterministic; checks whether the graph has an edge-disjoint decomposition of two spanning trees.";
MinRigidGraphQ::method="Method is not valid.";
MinRigidGraphQ::method2="Method is not available for dimension `1`.";
MinRigidGraphQ::limit="The graph has too many vertices. If you are sure you have more computing power use the Option VertexLimit->False.";
Options[MinRigidGraphQ]={Method->"RandomRigidityMatrix",UseCount->True,VertexLimit->True,RandomRigidityMatrixRange->Automatic,RandomSet->"Reals"};
SyntaxInformation[MinRigidGraphQ]={"ArgumentsPattern"->{_,_.,OptionsPattern[]}};
MinRigidGraphQ[g_Integer,opts:OptionsPattern[]]:=MinRigidGraphQ[g,2,FilterRules[{opts},Options[MinRigidGraphQ]]]
MinRigidGraphQ[G_Graph,opts:OptionsPattern[]]:=MinRigidGraphQ[G,2,FilterRules[{opts},Options[MinRigidGraphQ]]]
MinRigidGraphQ[g_Integer,dim_Integer,opts:OptionsPattern[]]:=MinRigidGraphQ[G2Graph[g],dim,FilterRules[{opts},Options[MinRigidGraphQ]]]
MinRigidGraphQ[G_Graph,dim_Integer,opts:OptionsPattern[]]:=Module[{},
  If[OptionValue[UseCount],If[EdgeCount[G]!=dim VertexCount[G]-Binomial[dim+1,2],Return[False]]];
  Return[Switch[
    OptionValue[Method],
    x_String/;MemberQ[{"Henneberg","Extension","Sequence"},x],
      If[dim==2,
        MinRigidGraphQSequence[G],
        Message[MinRigidGraphQ::method2,dim]
      ],
    x_String/;MemberQ[{"Count","Subgraph"},x],
      If[dim==2,
        TightGraphQ[G,2,Method->"General"],
        Message[MinRigidGraphQ::method2,dim]
      ],
    x_String/;MemberQ[{"Tight","PebbleGame"},x],
      If[dim==2,
        PebbleGame[G,2,3,OutputInfo->"Tight"],
        Message[MinRigidGraphQ::method2,dim]
      ],
    "TwoSpanningTrees",
      If[dim==2,
        LengthWhile[GEdges[G],Length[TwoSpanningTrees[G,#]]>0&]==EdgeCount[G],
        Message[MinRigidGraphQ::method2,dim]
      ],
    "ThreeTrees",
      If[dim==2,
        Length[ThreeTrees[G]]>0,
        Message[MinRigidGraphQ::method2,dim]
      ],
    "RealizationCount",
      If[Not[OptionValue[VertexLimit]]||VertexCount[G]<10,MinRigidGraphQRealizationCount[G,dim],Message[MinRigidGraphQ::limit]],
    "RandomRigidityMatrix",
      MinRigidGraphQRandomMatrix[G,dim,FilterRules[{opts},Options[MinRigidGraphQRandomMatrix]]],
    "SymbolicRigidityMatrix",
      MinRigidGraphQSymbolicMatrix[G,dim],
    _,
      Message[MinRigidGraphQ::method]
  ]];
]


MinRigidGraphQRandomMatrix::usage="MinRigidGraphQRandomMatrix[g,d] checks whether a random rigidity matrix has correct rank for being minimially rigid in dimension d for a random placement of vertices.";
Options[MinRigidGraphQRandomMatrix]={RandomRigidityMatrixRange->Automatic,RandomSet->"Reals"};
MinRigidGraphQRandomMatrix[g_Integer,dim_,opts:OptionsPattern[]]:=MinRigidGraphQRandomMatrix[G2Graph[g],dim,opts]
MinRigidGraphQRandomMatrix[G_Graph,dim_,opts:OptionsPattern[]]:=(EdgeCount[G]==dim VertexCount[G]-Binomial[dim+1,2]) && RigidGraphQRandomMatrix[G,dim,FilterRules[{opts},Options[RigidGraphQRandomMatrix]]]


MinRigidGraphQSymbolicMatrix::usage="MinRigidGraphQSymbolicMatrix[g,d] checks whether the symbolic rigidity matrix has correct rank for being minimially rigid in dimension d for a random placement of vertices.";
MinRigidGraphQSymbolicMatrix[g_Integer,dim_]:=MinRigidGraphQSymbolicMatrix[G2Graph[g],dim]
MinRigidGraphQSymbolicMatrix[G_Graph,dim_]:=(EdgeCount[G]==dim VertexCount[G]-Binomial[dim+1,2]) && RigidGraphQSymbolicMatrix[G,dim]


(*Test by computing the number of embeddings whether a graph can be a minimally rigid*)
MinRigidGraphQRealizationCount::usage="MinRigidGraphQRealizationCount[g,d] checks whether there is a finite number of realizations in dimension d.";
MinRigidGraphQRealizationCount[G_Graph,dim_Integer]:=MinRigidGraphQRealizationCount[Graph2G[G],dim]
MinRigidGraphQRealizationCount[g_Integer,dim_]:=Module[{tmp},
	If[EdgeCountI[g]=!=dim VertexCountI[g] -Binomial[dim+1,2],Return[False]];
	If[tmp=Catch[ComplexRealizationCount[g,dim]];StringQ[tmp]||tmp===\[Infinity]||tmp==0,Return[False],Return[True]];
]


(* This is a function from [1]*)
MinRigidGraphQSequence::usage="MinRigidGraphQSequence[g] checks whether there is a sequence of 0- and 1-extensions to construct the graph g from an edge. Returns True iff the graph is minimally rigid in dimension 2.";
MinRigidGraphQSequence[g_Graph] := MinRigidGraphQSequence[Normal[AdjacencyMatrix[g]]]
MinRigidGraphQSequence[g_Integer] := MinRigidGraphQSequence[G2Mat[g]];
MinRigidGraphQSequence[gr1_List] :=
Module[{gr = gr1, p, s, vs, lq},
  If[gr === {{0, 1}, {1, 0}}, Return[True]];
  p = Position[Total /@ gr, 2];
  If[p =!= {}, (* Henneberg type I move *) p = p[[1, 1]]; Return[MinRigidGraphQSequence[Delete[Transpose[Delete[gr, p]], p]]]];
  p = Position[Total /@ gr, 3];
  If[p =!= {}, (* Henneberg type II move *)
    p = p[[1, 1]];
    vs = Flatten[Position[gr[[p]], 1]];
    s = Select[Subsets[vs, {2}], gr[[#[[1]], #[[2]]]] === 0 &];
    If[s === {}, Return[False]];
    Do[
      gr = gr1;
      (gr[[##]] = 1)& @@@ Permutations[s[[i]]];
      If[(lq = MinRigidGraphQSequence[Delete[Transpose[Delete[gr, p]], p]]), Break[]];
    , {i, Length[s]}];
    Return[lq];
  ];
  Return[False];
];


(* ::Subsection::Closed:: *)
(*Pebble Game*)


PebbleGame::usage="PebbleGame[g,k,l] runs the pebble game for (k,l) on graph g (see [3]).
Depending on the Options OutputInfo it checks for (k-l)-tightness, (k,l)-sparsity or gives text output. Options are
	Tight (default): Returns whether the graph is (k-l)-tight.
	Sparse: Returns whether the graph is (k-l)-sparse.
	Detailed: Tells whether the graphs is well/over/under-constrained with respect to (k,l) or none of them.";
Options[PebbleGame]={OutputInfo->"Tight"};
SyntaxInformation[PebbleGame]={"ArgumentsPattern"->{_,_,_,OptionsPattern[]}};
PebbleGame::pebble="Pebble game is not available for k=`1` and l=`2`.";
PebbleGame[g_Integer,k_Integer,ell_Integer,opts:OptionsPattern[]]:=PebbleGame[G2Graph[g],k,ell,FilterRules[{opts},Options[PebbleGame]]]
PebbleGame[G_Graph,k_Integer,ell_Integer,OptionsPattern[]]:=Module[{G1,edges,n,pebbles,DG,res},
  If[ell>=2k||Not[IntegerQ[k]]||Not[IntegerQ[ell]],Return[Message[PebbleGame::pebble,k,ell]]];
  G1=StandardGraph[G];
  edges=GEdges[G1];
  n=VertexCount[G1];
  pebbles=Table[k,n];
  DG=Graph[Range[n],{}];
  Do[
    (*Print["Edge ",ee,"\tpebbles ",pebbles,"\tdirected ",EdgeList[DG]];*)
    While[Total[pebbles[[ee]]]<ell+1,
      res=PebbleSearch[DG,pebbles,ee];
      If[ListQ[res],{DG,pebbles}=res,Break[]];
    ];
    If[Total[pebbles[[ee]]]>=ell+1,
      If[pebbles[[First[ee]]]>0,
        pebbles[[First[ee]]]-=1;
        DG=EdgeAdd[DG,DirectedEdge@@ee],
        pebbles[[Last[ee]]]-=1;
        DG=EdgeAdd[DG,DirectedEdge@@Reverse[ee]]
      ],
      If[OptionValue[OutputInfo]=="Boolean"||OptionValue[OutputInfo]=="Tight"||OptionValue[OutputInfo]=="Sparse",Return[False]]
    ]
  ,
  {ee,edges}
  ];
  If[OptionValue[OutputInfo]=="Boolean"||OptionValue[OutputInfo]=="Tight",
    Return[If[Total[pebbles]==ell,
      EdgeCount[DG]==EdgeCount[G],
      False
    ]]
  ];
  If[OptionValue[OutputInfo]=="Sparse",
    Return[EdgeCount[DG]==EdgeCount[G]]
  ];
  Return[If[Total[pebbles]==ell,
    If[EdgeCount[DG]==EdgeCount[G],"Well-constrained","Over-constrained"],
    If[EdgeCount[DG]==EdgeCount[G],"Under-constrained","Other"]
  ]];
]

PebbleSearch::usage="PebbleSearch[g,p,e] performs a depth first search for pebbles in a directed graph g with pebbles p and starting vertices from e.";
PebbleSearch[DG_Graph,pebbles_List,ee_List]:=Module[{path,start,startv,pathedges,DG2,pebbles2,found},
  path=Reap[found=Catch[DepthFirstScan[VertexDelete[DG,Last[ee]],First[ee],{"FrontierEdge"->Sow,"DiscoverVertex"->((If[pebbles[[#1]]>0&&#1!=First[ee],Throw[#1]])&)}]]];
  start=1;
  If[ListQ[First[path]](*This means that no pebble was found in the previous search*),
    path=Reap[found=Catch[DepthFirstScan[VertexDelete[DG,First[ee]],Last[ee],{"FrontierEdge"->Sow,"DiscoverVertex"->((If[pebbles[[#1]]>0&&#1!=Last[ee],Throw[#1]])&)}]]];
    start=-1;
    If[ListQ[First[path]](*This means that no pebble was found*),
      Return[False]
    ];
  ];
  startv=ee[[start]];
  pathedges=If[Length[Last[path]]==0,
    {DirectedEdge[startv,First[path]]},
    Append[First[Last[path]],DirectedEdge[Last[Last[First[Last[path]]]],First[path]]]
  ];
  DG2=EdgeAdd[EdgeDelete[DG,pathedges],Reverse/@pathedges];
  pebbles2=pebbles;
  pebbles2[[startv]]+=1;
  pebbles2[[First[path]]]-=1;
  Return[{DG2,pebbles2}];
]


(* ::Subsection::Closed:: *)
(*Tree Decompositions*)


TwoSpanningTrees::usage="TwoSpanningTrees[G] finds all pairs of two spanning trees of G such that their union is G with some edge doubbled.
TwoSpanningTrees[G,e] finds all pairs of two spanning trees of G such that their union is G where e doubbled if it is already an edge or added if not.
By setting the option RemoveEquivalent to False one gets all trees otherwise by default the equivalent trees are omited (i.e. those by swapping color/order).
Note that graphs symmetry is not taken into account.
Option OutputStyle (Data or Graph): Output is either a list of pairs of trees given by edges or a graph with highlighted trees.";
TwoSpanningTrees::output="OutputStyle not available. Chose Data or Graph.";
Options[TwoSpanningTrees]={OutputStyle->"Data",RemoveEquivalent->True};
SyntaxInformation[TwoSpanningTrees]={"ArgumentsPattern"->{__,OptionsPattern[]}};
TwoSpanningTrees[G_Graph,fix_List,OptionsPattern[]]:=Module[{edges,colors,rest,green,black,resg,resb,nonedge=False},
  edges=GEdges[G];
  rest=Complement[edges,{fix}];
  If[Length[rest]==Length[edges],
    colors={{{},{}}};
    rest=Join[edges,{fix}];
    nonedge=True,
    colors={{{fix},{fix}}};
  ];
  Do[
    colors=Flatten[
      (green=First[#];
      black=Last[#];
      resg=If[AcyclicGraphQ[Graph[Join[green,{edge}]]],{Join[green,{edge}],black}];
      resb=If[AcyclicGraphQ[Graph[Join[black,{edge}]]],{green,Join[black,{edge}]}];
      DeleteCases[{resg,resb},Null]
      )&/@colors
    ,1]
  ,{edge,rest}];
  If[OptionValue[RemoveEquivalent],
    colors=Union[Sort/@colors]
  ];
  (*spanning ?*)
  colors = Select[colors, Length[Union@@First[#]]==VertexCount[G] && Length[Union@@Last[#]]==VertexCount[G] &];
  (* tree ?*)
  colors = Select[colors, TreeGraphQ[Graph[First[#]]] && TreeGraphQ[Graph[Last[#]]] &];
  Switch[
    OptionValue[OutputStyle],
    "Data",
    Return[colors],
    "Graph",
    If[nonedge,
      Return[HighlightGraph[Join[edges,{fix}],{Style[#,Darker[Green],Thick]&/@Pairs2Edges[First[#]],Style[#,Darker[Blue],Thick]&/@Pairs2Edges[Last[#]],Style[First[fix]<->Last[fix],Dashed]}]&/@colors],
      Return[HighlightGraph[Join[{Style[First[fix]<->Last[fix],Dashed,Thick]},Pairs2Edges[edges]],{Style[#,Darker[Green],Thick]&/@Rest[Pairs2Edges[First[#]]],Style[#,Darker[Blue],Thick]&/@Rest[Pairs2Edges[Last[#]]]}]&/@colors]
    ],
    _,
    Message[TwoSpanningTrees::output];
    Return[]
  ];  
]
TwoSpanningTrees[g_Integer,opts:OptionsPattern[]]:=TwoSpanningTrees[G2Graph[g],opts]
TwoSpanningTrees[G_Graph,opts:OptionsPattern[]]:=Join@@(TwoSpanningTrees[G,#,opts]&/@GEdges[G])



ThreeTrees::usage="ThreeTrees[g] finds all decompositions of g into three trees such that every vertex is contained in exactly two of them.
By setting the option RemoveEquivalent to False one gets all trees otherwise by default the equivalent trees are omited (i.e. those by swapping color/order).
Note that graphs symmetry is not taken into account.
Option OutputStyle (Data or Graph): Output is either a list of triples of trees given by edges or a graph with highlighted trees.";
ThreeTrees::output="OutputStyle not available. Chose Data or Graph.";
Options[ThreeTrees]={OutputStyle->"Data",RemoveEquivalent->True};
SyntaxInformation[ThreeTrees]={"ArgumentsPattern"->{_,OptionsPattern[]}};
ThreeTrees[g_Integer,opts:OptionsPattern[]]:=ThreeTrees[G2Graph[g],opts]
ThreeTrees[G_Graph,OptionsPattern[]]:=Module[{edges,colors,green,black,red,resg,resb,resr,jb,jg,jr,tmp},
  If[Not[ConnectedGraphQ[G]],Return[{}]];
  edges=GEdges[G];
  colors={{{},{},{}}};
  Do[
    colors=Flatten[
      (green=First[#];
      black=#[[2]];
      red=Last[#];
      jg=Join[green,{edge}];
      jb=Join[black,{edge}];
      jr=Join[red,{edge}];
      resg=If[AcyclicGraphQ[Graph[jg]]&&Length[Intersection[Flatten[jg],Flatten[black],Flatten[red]]]==0,{jg,black,red}];
      resb=If[AcyclicGraphQ[Graph[jb]]&&Length[Intersection[Flatten[green],Flatten[jb],Flatten[red]]]==0,{green,jb,red}];
      resr=If[AcyclicGraphQ[Graph[jr]]&&Length[Intersection[Flatten[green],Flatten[black],Flatten[jr]]]==0,{green,black,jr}];
      DeleteCases[{resg,resb,resr},Null]
      )&/@colors
    ,1]
  ,{edge,edges}];
  If[OptionValue[RemoveEquivalent],
    colors=Union[Sort/@colors]
  ];
  (*remove those where some part is a forest of single vertices*)
  colors = Select[colors, (tmp=Count[Last/@Tally[Join[Union@@#[[1]],Union@@#[[2]],Union@@#[[3]]]],1]; tmp==0 || (tmp==1 && Count[Length/@#,0]==1) || (tmp==2 && Count[Length/@#,0]==2)) &];
  (*remove those where some part is not connected*)
  colors = Select[colors, ConnectedGraphQ[Graph[Union@@#[[1]],Pairs2Edges[#[[1]]]]]&&ConnectedGraphQ[Graph[Union@@#[[2]],Pairs2Edges[#[[2]]]]]&&ConnectedGraphQ[Graph[Union@@#[[3]],Pairs2Edges[#[[3]]]]]&];
  (*check vertex induced subtrees*)
  colors = Select[colors, 
    Function[col,
      And@@
      Function[vv,
        tmp={Length[Select[col[[1]],Length[Intersection[#,vv]]==2&]],Length[Select[col[[2]],Length[Intersection[#,vv]]==2&]],Length[Select[col[[3]],Length[Intersection[#,vv]]==2&]]};
        Count[tmp,Length[vv]-1]!=2 || Count[tmp,0]!=1
      ]/@Subsets[VertexList[G],{4,Length[VertexList[G]]}]
    ]
  ];
  Switch[
    OptionValue[OutputStyle],
    "Data",
    Return[colors],
    "Graph",
    Return[HighlightGraph[G,{Style[#,Darker[Green],Thick]&/@Pairs2Edges[First[#]],Style[#,Darker[Blue],Thick]&/@Pairs2Edges[#[[2]]],Style[#,Orange,Thick]&/@Pairs2Edges[Last[#]]}]&/@colors],
    _,
    Message[ThreeTrees::output];
    Return[]
  ];  
]


(* ::Subsection::Closed:: *)
(*Redundantly Rigid Graph*)


KRedundantlyRigidGraphQ::usage="KRedundantlyRigidGraphQ[G,k,d] checks whether the graph G is k-edge-redundantly rigid in dimension d.";
Options[KRedundantlyRigidGraphQ]={CheckInputRigidity->True,RandomRigidityMatrixRange->Automatic,Method->"RandomRigidityMatrix",UseCount->True,RandomSet->"Reals"};
SyntaxInformation[KRedundantlyRigidGraphQ]={"ArgumentsPattern"->{_,_,_,OptionsPattern[]}};
KRedundantlyRigidGraphQ[g_Integer,k_Integer,dim_Integer,opts:OptionsPattern[]]:=KRedundantlyRigidGraphQ[G2Graph[g],k,dim,FilterRules[{opts},Options[KRedundantlyRigidGraphQ]]]
KRedundantlyRigidGraphQ[G_Graph,k_Integer,dim_Integer,opts:OptionsPattern[]]:=(
  EdgeCount[G]>=dim*VertexCount[G]-Binomial[dim+1,2]+k)&&
  (Min[VertexDegree[G]]>=dim+k)&&
  If[OptionValue[CheckInputRigidity],RigidGraphQ[G,dim,FilterRules[{opts},Options[RigidGraphQ]]],True]&&
  (LengthWhile[Subsets[EdgeList[G],{k}],RigidGraphQ[EdgeDelete[G,#],dim,FilterRules[{opts},Options[RigidGraphQ]]]&]==Binomial[EdgeCount[G],k]
)

MinimallyKRedundantlyRigidGraphQ::usage="MinimallyKRedundantlyRigidGraphQ[G,k,d] checks whether the graph G is minimally k-edge-redundantly rigid in dimension d.";
Options[MinimallyKRedundantlyRigidGraphQ]={CheckInputRedundancy->True,RandomRigidityMatrixRange->Automatic,Method->"RandomRigidityMatrix",UseCount->True,RandomSet->"Reals"};
SyntaxInformation[MinimallyKRedundantlyRigidGraphQ]={"ArgumentsPattern"->{_,_,_,OptionsPattern[]}};
MinimallyKRedundantlyRigidGraphQ[g_Integer,k_Integer,dim_Integer,opts:OptionsPattern[]]:=MinimallyKRedundantlyRigidGraphQ[G2Graph[g],k,dim,FilterRules[{opts},Options[MinimallyKRedundantlyRigidGraphQ]]]
MinimallyKRedundantlyRigidGraphQ[G_Graph,k_Integer,dim_Integer,opts:OptionsPattern[]]:=(
  If[OptionValue[CheckInputRedundancy],KRedundantlyRigidGraphQ[G,k,dim,FilterRules[{opts},Options[RigidGraphQ]]],True]&&
  (LengthWhile[EdgeList[G],Not[KRedundantlyRigidGraphQ[EdgeDelete[G,#],k,dim,FilterRules[{opts},Options[RigidGraphQ]]]]&]==EdgeCount[G]) (*Note, the option CheckInputRedundancy should not be passed on (if set to False)*)
)

RedundantlyRigidGraphQ::usage="RedundantlyRigidGraphQ[G,d] checks whether the graph G is redundantly rigid in dimension d.";
Options[RedundantlyRigidGraphQ]={CheckInputRigidity->True,RandomRigidityMatrixRange->Automatic,Method->"RandomRigidityMatrix",UseCount->True,RandomSet->"Reals"};
SyntaxInformation[RedundantlyRigidGraphQ]={"ArgumentsPattern"->{_,_.,OptionsPattern[]}};
RedundantlyRigidGraphQ[g_Integer,opts:OptionsPattern[]]:=KRedundantlyRigidGraphQ[g,1,2,FilterRules[{opts},Options[KRedundantlyRigidGraphQ]]]
RedundantlyRigidGraphQ[g_Graph,opts:OptionsPattern[]]:=KRedundantlyRigidGraphQ[g,1,2,FilterRules[{opts},Options[KRedundantlyRigidGraphQ]]]
RedundantlyRigidGraphQ[g_Integer,dim_Integer,opts:OptionsPattern[]]:=KRedundantlyRigidGraphQ[g,1,dim,FilterRules[{opts},Options[KRedundantlyRigidGraphQ]]]
RedundantlyRigidGraphQ[g_Graph,dim_Integer,opts:OptionsPattern[]]:=KRedundantlyRigidGraphQ[g,1,dim,FilterRules[{opts},Options[KRedundantlyRigidGraphQ]]]


(* ::Subsection::Closed:: *)
(*Vertex Redundantly Rigid Graph*)


KVertexRedundantlyRigidGraphQ::usage="KVertexRedundantlyRigidGraphQ[G,k,d] checks whether the graph G is k-vertex-redundantly rigid in dimension d.";
Options[KVertexRedundantlyRigidGraphQ]={CheckInputRigidity->True,RandomRigidityMatrixRange->Automatic};
SyntaxInformation[KVertexRedundantlyRigidGraphQ]={"ArgumentsPattern"->{_,_,_,OptionsPattern[]}};
KVertexRedundantlyRigidGraphQ[g_Integer,k_Integer,dim_Integer,opts:OptionsPattern[]]:=KVertexRedundantlyRigidGraphQ[G2Graph[g],k,dim,FilterRules[{opts},Options[KVertexRedundantlyRigidGraphQ]]]
KVertexRedundantlyRigidGraphQ[G_Graph,k_Integer,dim_Integer,opts:OptionsPattern[]]:=(
  If[OptionValue[CheckInputRigidity],RigidGraphQ[G,dim,FilterRules[{opts},Options[RigidGraphQ]]],True]&&
  (LengthWhile[Subsets[VertexList[G],{k}],RigidGraphQ[StandardGraph[VertexDelete[G,#]],dim,FilterRules[{opts},Options[RigidGraphQ]]]&]==Binomial[VertexCount[G],k])
)

MinimallyKVertexRedundantlyRigidGraphQ::usage="MinimallyKVertexRedundantlyRigidGraphQ[G,k,d] checks whether the graph G is minimally k-vertex-redundantly rigid in dimension d.";
Options[MinimallyKVertexRedundantlyRigidGraphQ]={CheckInputRedundancy->True,RandomRigidityMatrixRange->Automatic};
SyntaxInformation[MinimallyKVertexRedundantlyRigidGraphQ]={"ArgumentsPattern"->{_,_,_,OptionsPattern[]}};
MinimallyKVertexRedundantlyRigidGraphQ[g_Integer,k_Integer,dim_Integer,opts:OptionsPattern[]]:=MinimallyKVertexRedundantlyRigidGraphQ[G2Graph[g],k,dim,FilterRules[{opts},Options[MinimallyKVertexRedundantlyRigidGraphQ]]]
MinimallyKVertexRedundantlyRigidGraphQ[G_Graph,k_Integer,dim_Integer,opts:OptionsPattern[]]:=(
  If[OptionValue[CheckInputRedundancy],KVertexRedundantlyRigidGraphQ[G,k,dim,FilterRules[{opts},Options[RigidGraphQ]]],True]&&
  (LengthWhile[EdgeList[G],Not[KVertexRedundantlyRigidGraphQ[EdgeDelete[G,#],k,dim,FilterRules[{opts},Options[RigidGraphQ]]]]&]==EdgeCount[G]) (*Note, the option CheckInputRedundancy should not be passed on (if set to False)*)
)

VertexRedundantlyRigidGraphQ::usage="VertexRedundantlyRigidGraphQ[G,d] checks whether the graph G is vertex-redundantly rigid in dimension d.";
Options[VertexRedundantlyRigidGraphQ]={CheckInputRigidity->True,RandomRigidityMatrixRange->Automatic};
SyntaxInformation[VertexRedundantlyRigidGraphQ]={"ArgumentsPattern"->{_,_.,OptionsPattern[]}};
VertexRedundantlyRigidGraphQ[g_Integer,opts:OptionsPattern[]]:=KVertexRedundantlyRigidGraphQ[g,1,2,FilterRules[{opts},Options[KVertexRedundantlyRigidGraphQ]]]
VertexRedundantlyRigidGraphQ[g_Graph,opts:OptionsPattern[]]:=KVertexRedundantlyRigidGraphQ[g,1,2,FilterRules[{opts},Options[KVertexRedundantlyRigidGraphQ]]]
VertexRedundantlyRigidGraphQ[g_Integer,dim_Integer,opts:OptionsPattern[]]:=KVertexRedundantlyRigidGraphQ[g,1,dim,FilterRules[{opts},Options[KVertexRedundantlyRigidGraphQ]]]
VertexRedundantlyRigidGraphQ[g_Graph,dim_Integer,opts:OptionsPattern[]]:=KVertexRedundantlyRigidGraphQ[g,1,dim,FilterRules[{opts},Options[KVertexRedundantlyRigidGraphQ]]]


(* ::Subsection::Closed:: *)
(*Globally Rigid Graph*)


GloballyRigidGraphQ::usage="GloballyRigidGraphQ[g,d] checks whether the Graph represented by g is globally rigid in dimension d.
Methods available are:
	RandomStress: all dimensions, probabilistic (false negatives possible).
	SymbolicStress: all dimensions, deterministic.
	ConnectivityAndRedundancy: dimension 2, deterministic.";
GloballyRigidGraphQ::method="Method is not valid.";
GloballyRigidGraphQ::method2="Method is not possible for dimension `1`.";
Options[GloballyRigidGraphQ]={Method->"RandomStress",RMethod->"RandomRigidityMatrix",UseCount->True,RandomRigidityMatrixRange->Automatic};
SyntaxInformation[GloballyRigidGraphQ]={"ArgumentsPattern"->{_,_.,OptionsPattern[]}};
GloballyRigidGraphQ[g_Integer,opts:OptionsPattern[]]:=GloballyRigidGraphQ[g,2,FilterRules[{opts},Options[GloballyRigidGraphQ]]]
GloballyRigidGraphQ[G_Graph,opts:OptionsPattern[]]:=GloballyRigidGraphQ[G,2,FilterRules[{opts},Options[GloballyRigidGraphQ]]]
GloballyRigidGraphQ[g_Integer,dim_Integer,opts:OptionsPattern[]]:=GloballyRigidGraphQ[G2Graph[g],dim,FilterRules[{opts},Options[GloballyRigidGraphQ]]]
GloballyRigidGraphQ[G_Graph,dim_Integer,opts:OptionsPattern[]]:=Module[{n,m},
  n=VertexCount[G];
  m=EdgeCount[G];
  If[n<=dim+1,
    Return[2m==n(n-1)]
  ];
  If[OptionValue[UseCount],If[(Min[VertexDegree[G]]<dim+1)||(EdgeCount[G]<dim VertexCount[G]-Binomial[dim+1,2]+1&&VertexCount[G]!=dim+1),Return[False]]];
  Return[Switch[
    OptionValue[Method],
    "RandomStress",
      GloballyRigidGraphQRandomStress[G,dim,FilterRules[{opts},Options[GloballyRigidGraphQRandomStress]]],
    "SymbolicStress",
      GloballyRigidGraphQSymbolicStress[G,dim,FilterRules[{opts},Options[GloballyRigidGraphQSymbolicStress]]],
    "ConnectivityAndRedundancy",
      If[dim==2,
        VertexConnectivity[G]>=dim+1&&RedundantlyRigidGraphQ[G,dim,FilterRules[{opts},Options[RedundantlyRigidGraphQ]]/."ConnectivityAndRedundancy"->OptionValue[RMethod]],
        Message[GloballyRigidGraphQ::method2,dim]
      ],
    _,
      Message[GloballyRigidGraphQ::method]
  ]];
]


GloballyRigidGraphQRandomStress::usage="GloballyRigidGraphQRandomStress[g,d] checks whether the Graph represented by g is globally rigid in dimension d by computing random stresses.";
Options[GloballyRigidGraphQRandomStress]={RandomRigidityMatrixRange->Automatic};
GloballyRigidGraphQRandomStress[g_Integer,dim_Integer,opts:OptionsPattern[]]:=GloballyRigidGraphQRandomStress[G2Graph[g],dim,FilterRules[{opts},Options[GloballyRigidGraphQRandomStress]]]
GloballyRigidGraphQRandomStress[G_Graph,dim_Integer,opts:OptionsPattern[]]:=Module[{n,m,rm,sm,rand},
  n=VertexCount[G];
  m=EdgeCount[G];
  rm=RandomRigidityMatrix[G,dim,RandomRange->OptionValue[RandomRigidityMatrixRange],RandomSet->"Integers"];
  If[MatrixRank[rm]!=dim*n-Binomial[dim+1,2],Return[False]];
  sm=StressMatrixFromStress[G,RandomStressFromRigidityMatrix[rm,RandomRange->If[NumberQ[rand=OptionValue[RandomRigidityMatrixRange]],rand,10^6*n*m*dim]]]; (*Chosen according to [5]*)
  Return[MatrixRank[sm]==n-dim-1]
]

RandomStressFromRigidityMatrix::usage="RandomStressFromRigidityMatrix[r] computes a random stress given a rigidity matrix r.";
Options[RandomStressFromRigidityMatrix]={RandomRange->Automatic};
RandomStressFromRigidityMatrix[rm_List,OptionsPattern[]]:=Module[{mat,nspace,cs,vec,stress},
  mat=Transpose[rm];
  nspace=NullSpace[mat];
  If[Length[nspace]>0,
    cs=Length[nspace];
    vec=RandomInteger[If[NumberQ[OptionValue[RandomRange]],OptionValue[RandomRange],10^6*Length[rm]],cs];
    stress=Plus@@(nspace*vec),
    stress=Table[0,Length[rm]]
  ];
  Return[stress];
]

StressMatrixFromStress::usage="StressMatrixFromStress[g,s] computes the stress matrix for a graph g and a stress s.";
StressMatrixFromStress[G_Graph,stress_List]:=Module[{n,edges,mat},
  n=VertexCount[G];
  edges=GEdges[G];
  mat=Table[Table[0,n],n];
  MapIndexed[
    (mat[[Sequence@@#1]]-= stress[[First[#2]]];
    mat[[Sequence@@Reverse[#1]]]-= stress[[First[#2]]];
    mat[[First[#1],First[#1]]]+= stress[[First[#2]]];
    mat[[Last[#1],Last[#1]]]+= stress[[First[#2]]];
    )&,
    edges
  ];
  Return[mat];
]


GloballyRigidGraphQSymbolicStress::usage="GloballyRigidGraphQSymbolicStress[g,d] checks whether the Graph represented by g is globally rigid in dimension d by computing symbolic stresses.";
Options[GloballyRigidGraphQSymbolicStress]={};
GloballyRigidGraphQSymbolicStress[g_Integer,dim_Integer,opts:OptionsPattern[]]:=GloballyRigidGraphQSymbolicStress[G2Graph[g],dim,FilterRules[{opts},Options[GloballyRigidGraphQRandomStress]]]
GloballyRigidGraphQSymbolicStress[G_Graph,dim_Integer,opts:OptionsPattern[]]:=Module[{n,m,rm,sm,a,x},
  n=VertexCount[G];
  m=EdgeCount[G];
  rm=SymbolicRigidityMatrix[G,dim,x];
  If[MatrixRank[rm]!=dim*n-Binomial[dim+1,2],Return[False]];
  sm=StressMatrixFromStress[G,SymbolicStressFromRigidityMatrix[rm,Var->a]];
  Return[MatrixRank[sm]==n-dim-1]
]

SymbolicStressFromRigidityMatrix::usage="SymbolicStressFromRigidityMatrix[r] computes a symbolic stress given a rigidity matrix r, where the variable is given by the option Var (default: x).";
Options[SymbolicStressFromRigidityMatrix]={Var->x};
SymbolicStressFromRigidityMatrix[rm_List,OptionsPattern[]]:=Module[{mat,nspace,cs,vec,stress},
  mat=Transpose[rm];
  nspace=NullSpace[mat];
  If[Length[nspace]>0,
    cs=Length[nspace];
    vec=Table[OptionValue[Var][i],{i,cs}];
    stress=Plus@@(nspace*vec),
    stress=Table[0,Length[rm]]
  ];
  Return[stress];
]


MinimallyGloballyRigidGraphQ::usage="MinimallyGloballyRigidGraphQ[g,d] checks whether the graph g is minimally globally rigid in dimension d.";
Options[MinimallyGloballyRigidGraphQ]={CheckInputGlobalRigidity->True,RandomRigidityMatrixRange->Automatic,Method->"RandomStress"};
SyntaxInformation[MinimallyGloballyRigidGraphQ]={"ArgumentsPattern"->{_,_.,OptionsPattern[]}};
MinimallyGloballyRigidGraphQ[g_Integer,opts:OptionsPattern[]]:=MinimallyGloballyRigidGraphQ[g,2,FilterRules[{opts},Options[MinimallyGloballyRigidGraphQ]]]
MinimallyGloballyRigidGraphQ[g_Graph,opts:OptionsPattern[]]:=MinimallyGloballyRigidGraphQ[g,2,FilterRules[{opts},Options[MinimallyGloballyRigidGraphQ]]]
MinimallyGloballyRigidGraphQ[g_Integer,dim_Integer,opts:OptionsPattern[]]:=MinimallyGloballyRigidGraphQ[G2Graph[g],dim,FilterRules[{opts},Options[MinimallyGloballyRigidGraphQ]]]
MinimallyGloballyRigidGraphQ[G_Graph,dim_Integer,opts:OptionsPattern[]]:=(
  If[OptionValue[CheckInputGlobalRigidity],GloballyRigidGraphQ[G,dim],True]&&
  (LengthWhile[EdgeList[G],Not[GloballyRigidGraphQ[EdgeDelete[G,#],dim,FilterRules[{opts},Options[GloballyRigidGraphQ]]]]&]==EdgeCount[G])
)



KRedundantlyGloballyRigidGraphQ::usage="KRedundantlyGloballyRigidGraphQ[G,k,d] checks whether the graph G is k-redundantly globally rigid in dimension d.";
Options[KRedundantlyGloballyRigidGraphQ]={CheckInputGlobalRigidity->True,RandomRigidityMatrixRange->Automatic,Method->"RandomStress"};
SyntaxInformation[KRedundantlyGloballyRigidGraphQ]={"ArgumentsPattern"->{_,_,_,OptionsPattern[]}};
KRedundantlyGloballyRigidGraphQ[g_Integer,k_Integer,dim_Integer,opts:OptionsPattern[]]:=KRedundantlyGloballyRigidGraphQ[G2Graph[g],k,dim,FilterRules[{opts},Options[KRedundantlyGloballyRigidGraphQ]]]
KRedundantlyGloballyRigidGraphQ[G_Graph,k_Integer,dim_Integer,opts:OptionsPattern[]]:=(
  EdgeCount[G]>=dim*VertexCount[G]-Binomial[dim+1,2]+k)&&
  (Min[VertexDegree[G]]>=dim+k)&&
  If[OptionValue[CheckInputGlobalRigidity],GloballyRigidGraphQ[G,dim,FilterRules[{opts},Options[GloballyRigidGraphQ]]],True]&&
  (LengthWhile[Subsets[EdgeList[G],{k}],GloballyRigidGraphQ[EdgeDelete[G,#],dim,FilterRules[{opts},Options[GloballyRigidGraphQ]]]&]==Binomial[EdgeCount[G],k]
)


MinimallyKRedundantlyGloballyRigidGraphQ::usage="MinimallyKRedundantlyGloballyRigidGraphQ[G,k,d] checks whether the graph G is minimally k-redundantly globally rigid in dimension d.";
Options[MinimallyKRedundantlyGloballyRigidGraphQ]={CheckInputRedundancy->True,RandomRigidityMatrixRange->Automatic};
SyntaxInformation[MinimallyKRedundantlyGloballyRigidGraphQ]={"ArgumentsPattern"->{_,_,_,OptionsPattern[]}};
MinimallyKRedundantlyGloballyRigidGraphQ[g_Integer,k_Integer,dim_Integer,opts:OptionsPattern[]]:=MinimallyKRedundantlyGloballyRigidGraphQ[G2Graph[g],k,dim,FilterRules[{opts},Options[MinimallyKRedundantlyGloballyRigidGraphQ]]]
MinimallyKRedundantlyGloballyRigidGraphQ[G_Graph,k_Integer,dim_Integer,opts:OptionsPattern[]]:=(
  If[OptionValue[CheckInputRedundancy],KRedundantlyGloballyRigidGraphQ[G,k,dim,FilterRules[{opts},Options[RigidGraphQ]]],True]&&
  (LengthWhile[EdgeList[G],Not[KRedundantlyGloballyRigidGraphQ[EdgeDelete[G,#],k,dim,FilterRules[{opts},Options[GloballyRigidGraphQ]]]]&]==EdgeCount[G]) (*Note, the option CheckInputRedundancy should not be passed on (if set to False)*)
)


RedundantlyGloballyRigidGraphQ::usage="RedundantlyGloballyRigidGraphQ[G,d] checks whether the graph G is redundantly globally rigid in dimension d.";
Options[RedundantlyGloballyRigidGraphQ]={CheckInputGlobalRigidity->True,RandomRigidityMatrixRange->Automatic};
SyntaxInformation[RedundantlyGloballyRigidGraphQ]={"ArgumentsPattern"->{_,_.,OptionsPattern[]}};
RedundantlyGloballyRigidGraphQ[g_Integer,opts:OptionsPattern[]]:=KRedundantlyGloballyRigidGraphQ[g,1,2,FilterRules[{opts},Options[KRedundantlyGloballyRigidGraphQ]]]
RedundantlyGloballyRigidGraphQ[g_Graph,opts:OptionsPattern[]]:=KRedundantlyGloballyRigidGraphQ[g,1,2,FilterRules[{opts},Options[KRedundantlyGloballyRigidGraphQ]]]
RedundantlyGloballyRigidGraphQ[g_Integer,dim_Integer,opts:OptionsPattern[]]:=KRedundantlyGloballyRigidGraphQ[g,1,dim,FilterRules[{opts},Options[KRedundantlyGloballyRigidGraphQ]]]
RedundantlyGloballyRigidGraphQ[g_Graph,dim_Integer,opts:OptionsPattern[]]:=KRedundantlyGloballyRigidGraphQ[g,1,dim,FilterRules[{opts},Options[KRedundantlyGloballyRigidGraphQ]]]


KVertexRedundantlyGloballyRigidGraphQ::usage="KVertexRedundantlyGloballyRigidGraphQ[G,k,d] checks whether the graph G is k-vertex-redundantly globally rigid in dimension d.";
Options[KVertexRedundantlyGloballyRigidGraphQ]={CheckInputGlobalRigidity->True,RandomRigidityMatrixRange->Automatic,Method->"RandomStress"};
SyntaxInformation[KVertexRedundantlyGloballyRigidGraphQ]={"ArgumentsPattern"->{_,_,_,OptionsPattern[]}};
KVertexRedundantlyGloballyRigidGraphQ[g_Integer,k_Integer,dim_Integer,opts:OptionsPattern[]]:=KVertexRedundantlyGloballyRigidGraphQ[G2Graph[g],k,dim,FilterRules[{opts},Options[KVertexRedundantlyGloballyRigidGraphQ]]]
KVertexRedundantlyGloballyRigidGraphQ[G_Graph,k_Integer,dim_Integer,opts:OptionsPattern[]]:=(
  If[OptionValue[CheckInputGlobalRigidity],GloballyRigidGraphQ[G,dim,FilterRules[{opts},Options[GloballyRigidGraphQ]]],True]&&
  (LengthWhile[Subsets[VertexList[G],{k}],GloballyRigidGraphQ[StandardGraph[VertexDelete[G,#]],dim,FilterRules[{opts},Options[GloballyRigidGraphQ]]]&]==Binomial[VertexCount[G],k])
)


MinimallyKVertexRedundantlyGloballyRigidGraphQ::usage="MinimallyKVertexRedundantlyGloballyRigidGraphQ[G,k,d] checks whether the graph G is minimally k-vertex-redundantly globally rigid in dimension d.";
Options[MinimallyKVertexRedundantlyGloballyRigidGraphQ]={CheckInputRedundancy->True,RandomRigidityMatrixRange->Automatic,Method->"RandomStress"};
SyntaxInformation[MinimallyKVertexRedundantlyGloballyRigidGraphQ]={"ArgumentsPattern"->{_,_,_,OptionsPattern[]}};
MinimallyKVertexRedundantlyGloballyRigidGraphQ[g_Integer,k_Integer,dim_Integer,opts:OptionsPattern[]]:=MinimallyKVertexRedundantlyGloballyRigidGraphQ[G2Graph[g],k,dim,FilterRules[{opts},Options[MinimallyKVertexRedundantlyGloballyRigidGraphQ]]]
MinimallyKVertexRedundantlyGloballyRigidGraphQ[G_Graph,k_Integer,dim_Integer,opts:OptionsPattern[]]:=(
  If[OptionValue[CheckInputRedundancy],KVertexRedundantlyGloballyRigidGraphQ[G,k,dim,FilterRules[{opts},Options[GloballyRigidGraphQ]]],True]&&
  (LengthWhile[EdgeList[G],Not[KVertexRedundantlyGloballyRigidGraphQ[EdgeDelete[G,#],k,dim,FilterRules[{opts},Options[GloballyRigidGraphQ]]]]&]==EdgeCount[G]) (*Note, the option CheckInputRedundancy should not be passed on (if set to False)*)
)


VertexRedundantlyGloballyRigidGraphQ::usage="VertexRedundantlyGloballyRigidGraphQ[G,d] checks whether the graph G is vertex-redundantly globally rigid in dimension d.";
Options[VertexRedundantlyGloballyRigidGraphQ]={CheckInputGlobalRigidity->True,RandomRigidityMatrixRange->Automatic,Method->"RandomStress"};
SyntaxInformation[VertexRedundantlyGloballyRigidGraphQ]={"ArgumentsPattern"->{_,_.,OptionsPattern[]}};
VertexRedundantlyGloballyRigidGraphQ[g_Integer,opts:OptionsPattern[]]:=KVertexRedundantlyGloballyRigidGraphQ[g,1,2,FilterRules[{opts},Options[KVertexRedundantlyGloballyRigidGraphQ]]]
VertexRedundantlyGloballyRigidGraphQ[g_Graph,opts:OptionsPattern[]]:=KVertexRedundantlyGloballyRigidGraphQ[g,1,2,FilterRules[{opts},Options[KVertexRedundantlyGloballyRigidGraphQ]]]
VertexRedundantlyGloballyRigidGraphQ[g_Integer,dim_Integer,opts:OptionsPattern[]]:=KVertexRedundantlyGloballyRigidGraphQ[g,1,dim,FilterRules[{opts},Options[KVertexRedundantlyGloballyRigidGraphQ]]]
VertexRedundantlyGloballyRigidGraphQ[g_Graph,dim_Integer,opts:OptionsPattern[]]:=KVertexRedundantlyGloballyRigidGraphQ[g,1,dim,FilterRules[{opts},Options[KVertexRedundantlyGloballyRigidGraphQ]]]


(* ::Subsection::Closed:: *)
(*Sparse and Tight Graphs*)


SparseGraphQ::usage="SparseGraphQ[G,k,l] checks whether a graph is (k,l)-sparse.
SparseGraphQ[G,d] checks whether a graph is sparse in dimension d, i.e. (d,Binomial[d+1,2])-sparse.";
Options[SparseGraphQ]={Method->"Pebble"};
SparseGraphQ::pebble="Pebble game does not work for (`1`,`2`)-sparsity.";
SyntaxInformation[SparseGraphQ]={"ArgumentsPattern"->{_,_,_.,OptionsPattern[]}};
SparseGraphQ[g_Integer,k_/;NumberQ[k],l_/;NumberQ[l],opts:OptionsPattern[]]:=SparseGraphQ[G2Graph[g],k,l,FilterRules[{opts},Options[SparseGraphQ]]]
SparseGraphQ[g_Graph,k_/;NumberQ[k],l_/;NumberQ[l],OptionsPattern[]]:=Module[{sub},
  If[OptionValue[Method]=="Strict",If[l>2k,Return[False]]];
  If[OptionValue[Method]=="Pebble",If[l>=2k||Not[IntegerQ[k]]||Not[IntegerQ[l]],Return[Message[SparseGraphQ::pebble,k,l]],Return[PebbleGame[g,k,l,OutputInfo->"Sparse"]]]];
  (*Return[And[EdgeCount[g]<=k VertexCount[g]-l,And@@(Function[subg,EdgeCount[subg]<=k VertexCount[subg]-l]/@(Flatten[Subgraph[g,#]&/@Subsets[VertexList[g],{k+1,VertexCount[g]-1}]]))]]*)
  Return[And[
    EdgeCount[g]<=k VertexCount[g]-l,
    (sub=Subgraph[g,#]&/@Subsets[VertexList[g],{k+1,VertexCount[g]-1}];
    LengthWhile[sub,EdgeCount[#]<=k VertexCount[#]-l&]==Length[sub]
    )
  ]]
];

SparseGraphQ[g_Integer,d_Integer,opts:OptionsPattern[]]:=SparseGraphQ[G2Graph[g],d,Binomial[d+1,2],FilterRules[{opts},Options[SparseGraphQ]]]
SparseGraphQ[g_Graph,d_Integer,opts:OptionsPattern[]]:=SparseGraphQ[g,d,Binomial[d+1,2],FilterRules[{opts},Options[SparseGraphQ]]]


TightGraphQ::usage="TightGraphQ[G,k,l] checks whether a graph is (k,l)-tight.
TightGraphQ[G,d] checks whether a graph is tight in dimension d, i.e. (d,Binomial[d+1,2])-tight.";
Options[TightGraphQ]={Method->"Pebble"};
TightGraphQ::pebble="Pebble game does not work for (`1`,`2`)-tightness.";
SyntaxInformation[TightGraphQ]={"ArgumentsPattern"->{_,_,_.,OptionsPattern[]}};
TightGraphQ[g_Integer,k_/;NumberQ[k],l_/;NumberQ[l],opts:OptionsPattern[]]:=TightGraphQ[G2Graph[g],k,l,FilterRules[{opts},Options[TightGraphQ]]]
TightGraphQ[g_Graph,k_/;NumberQ[k],l_/;NumberQ[l],opts:OptionsPattern[]]:=Module[{},
  If[OptionValue[Method]=="Strict",If[l>2k,Return[False]]];
  If[OptionValue[Method]=="Pebble",If[l>=2k||Not[IntegerQ[k]]||Not[IntegerQ[l]],Return[Message[TightGraphQ::pebble,k,l]],Return[PebbleGame[g,k,l,OutputInfo->"Tight"]]]];
  Return[And[EdgeCount[g]==k VertexCount[g]-l,SparseGraphQ[g,k,l,FilterRules[{opts},Options[SparseGraphQ]]]]]
];

TightGraphQ[g_Integer,d_Integer,opts:OptionsPattern[]]:=TightGraphQ[G2Graph[g],d,Binomial[d+1,2],FilterRules[{opts},Options[TightGraphQ]]]
TightGraphQ[g_Graph,d_Integer,opts:OptionsPattern[]]:=TightGraphQ[g,d,Binomial[d+1,2],FilterRules[{opts},Options[TightGraphQ]]]


(* ::Subsection::Closed:: *)
(*Dependence and Circuits*)


DependentGraphQ::usage="DependentGraphQ[G,d] checks whether the graph G is dependent in dimension d.";
DependentGraphQ::method="Method is not valid.";
Options[DependentGraphQ]={Method->"RandomRigidityMatrix",RandomRigidityMatrixRange->Automatic};
SyntaxInformation[DependentGraphQ]={"ArgumentsPattern"->{_,_,OptionsPattern[]}};
DependentGraphQ[graph_Integer,dim_Integer,opts:OptionsPattern[]]:=DependentGraphQ[G2Graph[graph],dim,FilterRules[{opts},Options[DependentGraphQ]]]
DependentGraphQ[graph_Graph,dim_Integer,opts:OptionsPattern[]]:=Module[{x},
  Switch[
    OptionValue[Method],
    "RandomRigidityMatrix",
      MatrixRank[RandomRigidityMatrix[graph,dim,RandomRange->OptionValue[RandomRigidityMatrixRange]]]<EdgeCount[graph],
    "SymbolicRigidityMatrix",
      MatrixRank[SymbolicRigidityMatrix[graph,dim,x]]<EdgeCount[graph],
    _,
      Message[DependentGraphQ::method]
  ]
]

IndependentGraphQ::usage="IndependentGraphQ[G,d] checks whether the graph G is independent in dimension d.";
IndependentGraphQ::method="Method is not valid.";
Options[IndependentGraphQ]={Method->"RandomRigidityMatrix",RandomRigidityMatrixRange->Automatic};
SyntaxInformation[IndependentGraphQ]={"ArgumentsPattern"->{_,_,OptionsPattern[]}};
IndependentGraphQ[graph_Integer,dim_Integer,opts:OptionsPattern[]]:=IndependentGraphQ[G2Graph[graph],dim,FilterRules[{opts},Options[IndependentGraphQ]]]
IndependentGraphQ[graph_Graph,dim_Integer,opts:OptionsPattern[]]:=Module[{x},
  Switch[
    OptionValue[Method],
    "RandomRigidityMatrix",
      MatrixRank[RandomRigidityMatrix[graph,dim,RandomRange->OptionValue[RandomRigidityMatrixRange]]]==EdgeCount[graph],
    "SymbolicRigidityMatrix",
      MatrixRank[SymbolicRigidityMatrix[graph,dim,x]]==EdgeCount[graph],
    _,
      Message[DependentGraphQ::method]
  ]
]

CircuitQ::usage="CircuitQ[G,d] checks whether the graph G is a circuit in dimension d.";
Options[CircuitQ]={Method->"RandomRigidityMatrix",RandomRigidityMatrixRange->Automatic};
SyntaxInformation[CircuitQ]={"ArgumentsPattern"->{_,_,OptionsPattern[]}};
CircuitQ[graph_Integer,dim_Integer,opts:OptionsPattern[]]:=CircuitQ[G2Graph[graph],dim,FilterRules[{opts},Options[CircuitQ]]]
CircuitQ[graph_Graph,dim_Integer,opts:OptionsPattern[]]:=Module[{edges,edgedel},
  If[Not[DependentGraphQ[graph,dim,FilterRules[{opts},Options[DependentGraphQ]]]],Return[False]];
  edges=GEdges[graph];
  edgedel=Table[Graph[ReplacePart[edges,i->Nothing]],{i,1,Length[edges]}];
  Return[And@@(IndependentGraphQ[#,dim,FilterRules[{opts},Options[IndependentGraphQ]]]&/@edgedel)];
]


(* ::Subsection::Closed:: *)
(*Rigid Subgraph Free*)


RigidSubgraphFreeMinRigidGraphQ::usage="RigidSubgraphFreeMinRigidGraphQ[g,d} checks whether the graph g is minimally rigid and has non-trivial minimally rigid induced subgraphs in dimension d.
By default subgraphs on less than d+1 vertices are ignored (IgnoreSmallSubgraphs->True).
If IgnoreSmallSubgraphs is set to False all subgraphs with more than two vertices are considered.";
Options[RigidSubgraphFreeMinRigidGraphQ]={IgnoreSmallSubgraphs->True,CheckMinRigidity->True,Method->"RandomRigidityMatrix",UseCount->True,RandomRigidityMatrixRange->Automatic,RandomSet->"Reals"};
RigidSubgraphFreeMinRigidGraphQ[g_Integer,dim_Integer,opts:OptionsPattern[]]:=RigidSubgraphFreeMinRigidGraphQ[G2Graph[g],dim,FilterRules[{opts},{Options[RigidSubgraphFreeMinRigidGraphQ],Options[MinRigidGraphQ]}]]
RigidSubgraphFreeMinRigidGraphQ[G_Graph,dim_Integer,opts:OptionsPattern[]]:=Module[{n,subgraphs,subgraphs2},
  If[OptionValue[CheckMinRigidity],If[Not[MinRigidGraphQ[G,dim,FilterRules[{opts},Options[MinRigidGraphQ]]]],Return[False]]];
  n=VertexCount[G];
  If[Min[VertexDegree[G]]==dim,Return[False]];
  subgraphs=Subgraph[G,#]&/@Subsets[VertexList[G],{If[OptionValue[IgnoreSmallSubgraphs],dim+1,3],n-1}];
  subgraphs=Select[subgraphs,EdgeCount[#]==dim VertexCount[#]-Binomial[dim+1,2]&];
  subgraphs2=LengthWhile[subgraphs,Not[MinRigidGraphQ[StandardGraph[#],dim,FilterRules[{opts},Options[MinRigidGraphQ]]]]&];
  Return[Length[subgraphs]==subgraphs2];
]


(* ::Section:: *)
(*Realization Count*)


(* ::Subsection::Closed:: *)
(*Complex Space*)


ComplexRealizationCount::usage="ComplexRealizationCount[g] computes the number of complex realizations in d dimensional space.
The following Methods are available:
	Automatic (default): choses Combinatorial for dimension 2 and Groebner otherwise.
	Groebner: all dimensions, probabilistic; generates a system of equations and counts the complex solutions thereof.
	Combinatorial: dimension 2, deterministic; uses the algorithm from [2].";
Options[ComplexRealizationCount]={Method->Automatic,RemoveDegD->True};
ComplexRealizationCount::method="The chosen Method is invalid.";
ComplexRealizationCount::method2="Method is not available for dimension `1`.";
SyntaxInformation[ComplexRealizationCount]={"ArgumentsPattern"->{_,_.,OptionsPattern[]}};
ComplexRealizationCount[g_Integer,opts:OptionsPattern[]]:=ComplexRealizationCount[g,2,opts]
ComplexRealizationCount[G_Graph,opts:OptionsPattern[]]:=ComplexRealizationCount[G,2,opts]
ComplexRealizationCount[g_Integer,dim_Integer,opts:OptionsPattern[]]:=ComplexRealizationCount[G2Graph[g],dim,opts]
ComplexRealizationCount[G_Graph,dim_Integer,opts:OptionsPattern[]]:=Module[{Gs,cc,pos,switch},
  
  Gs=StandardGraph[G];
  (* If there are vertices of valency less than dim it cannot be rigid*)
  If[Min[pos=VertexDegree[Gs]]<dim&&VertexCount[Gs]>dim,Return[\[Infinity]]];
  
  (*check for vertices with degree dim *)
  cc=1;
  If[OptionValue[RemoveDegD],
   While[Min[pos=VertexDegree[Gs]]==dim&&VertexCount[Gs]>dim+1,
     pos=Flatten[Position[pos,dim]];
     Gs=VertexDelete[Gs,pos];
     Gs=StandardGraph[Gs];
     cc*=2^Length[pos];
   ];
  ];
  If[EdgeCount[Gs]==Binomial[VertexCount[Gs],2],Return[Switch[VertexCount[Gs],dim,cc,dim+1,2*cc,x_/;x>dim+1,0,x_/;x<dim,Infinity]]];
  
  (* Apply counting method *)
  If[(switch=OptionValue[Method])===Automatic,switch=If[dim==2,"Combinatorial","Groebner"]];
  Return[cc*Switch[
    switch,
    "Groebner",
      RealizationCountGroebner[Gs,dim],
    "Combinatorial",
      If[dim==2,
        BigraphCount[GEdges[Gs],GEdges[Gs],FilterRules[{opts},Options[BigraphCount]]],
        Message[MinRigidGraphQ::method2,dim]
      ],
    _,
      Message[ComplexRealizationCount::method]
  ]]
]


(* ::Subsubsection::Closed:: *)
(*Gr\[ODoubleDot]bner Basis Approach*)


RealizationCountGroebner::usage="RealizationCountGroebner[g,d] computes the number of complex realizations of a graph g in dimension d by counting solutions of a system of equations with Gr\[ODoubleDot]bner basis.";
RealizationCountGroebner[G_Graph,dim_Integer]:=Module[{eqs,var,fix,x,gb,sol},
  eqs=DistanceEquations[G,dim,RHS->"Random",RandomMax->10^6,Var->x];
  fix=Range[dim];
  MapIndexed[Function[{c,i},
    eqs = eqs /.Thread[(x[c,#]&/@Range[dim-First[i]+1])->0]],
    fix
  ];
  var=Variables[eqs];
  eqs=Select[eqs,Head[#]=!=Integer&];
  If[eqs === {}, Return[1]];
  gb = GroebnerBasis[eqs, var, MonomialOrder -> DegreeReverseLexicographic, Modulus -> NextPrime[2^31, -2]];
  sol = CountUnderTheStaircase[LeadingMonomial[#, var] & /@ gb,var];
  Return[sol/2^(dim-1)]
]


(*Subroutines*)
LeadingMonomial::usage="(internal function)\nLeadingMonomial[p,V] gives the leading monomial of the polynomial p with variables V with respect to degree reverse lexicographic ordering";
LeadingMonomial[poly_,vars_List]:=Module[{term},
  term=First[MonomialList[poly,vars,"DegreeReverseLexicographic"]];
  Return[term/(term/.Thread[vars->1])]
]

CountUnderTheStaircase::usage="(internal function)\nCountUnderTheStaircase[m,v] counts the irreducible monomials for a Gr\[ODoubleDot]bner basis represented by its leading monomials m in variables v.";
CountUnderTheStaircase[mon_List,var_List]:=Module[{exp,exps,dim,max,mm,pos,count},
  exp=Exponent[#,var]&/@mon;
  dim=Length[var];
  exps=Select[exp,Count[#,0]==dim-1&];
  (* check whether every variable appears purely *)
  If[Length[Union@@(Position[#,_?Positive]&/@exps)]!=dim,Return[Infinity]];
  
  (* run through all exponents that are under the staircase
   * start with all 0 and increase step by step (starting from last index)
   *)
  max=Max[First/@exps];
  mm=Table[0,dim];
  pos=dim;
  count=0;
  While[First[mm]<max&&pos>0,
    If[LengthWhile[exp,Or@@Thread[mm<#]&]==Length[exp],
      count++;
      pos=dim;
      mm[[pos]]++,
    (*else*)
      mm[[pos]]=0;
      pos--;
      If[pos>0,mm[[pos]]++]
    ];
  ];
  Return[count];
]


(* ::Subsubsection::Closed:: *)
(*Combinatorial Algorithm*)


BigraphCount::usage="BigraphCount[e1,e2] returns the 'number of realizations' in the plane of the bigraph with components given by the edges e1, and e2, respectively.
This uses the algorithm from [2] and is based on the implmentation of [1].";
Options[BigraphCount]={StoreSmall->True,SmallThreshold->4,UseTriangleSimplification->True,UseQuadrilateralSimplification->True,TriangleThreshold->5};
BigraphCount[e1_List, e2_List,OptionsPattern[]] :=
Module[{k, e = 1, sum, l, v1, v2, t1, t2, q1, q2, subsets},
  If[Not[SameQ @@ (Length /@ {e1, e2})], Throw["Not the same number of edges."]];
  If[(k = Length[e1]) === 1, Return[1]];
  {v1, v2} = Union@@#& /@ {e1, e2};
  {t1, t2} = Length /@ {v1, v2};
  
  (* If there are few edges, normalize the input so that we have fewer cases to remember. *)
  If[OptionValue[StoreSmall]&& k <= OptionValue[SmallThreshold],    
    If[v1 =!= Range[t1] || v2 =!= Range[t2] || e1 =!= Sort[e1],
      q1 = e1 /. Thread[v1 -> Range[t1]];
      q2 = e2 /. Thread[v2 -> Range[t2]];
      {q1, q2} = Transpose[SortBy[Transpose[{q1, q2}], First]];
      Return[BigraphCount[q1, q2] = BigraphCount[q1, q2]];   (*This is done in order to store results for few vertices*)
    ];
  ];
  
  (* Criterion on components *)
  If[t1 - Length[Components[e1]] + t2 - Length[Components[e2]] =!= k + 1, Return[0]];
  
  If[OptionValue[UseTriangleSimplification]===False || k < OptionValue[TriangleThreshold],
    subsets = Subsets[Rest[Range[k]], {1, k - 2}];
  ,
    {t1, t2} = Triangles /@ {e1, e2};
    (* Choose a "good" edge e. *)
    If[(e = Complement[Range[k], Union[Flatten[{t1, t2}]]]) === {},
      e = SortBy[Tally[Flatten[{t1, t2}]], Last][[1,1]];
    ,
      e = e[[1]];
    ];
    subsets = Subsets[Delete[Range[k], e], {1, k - 2}];
    (* Exclude subsets for which M resp. N involves precisely 2 edges of a triangle in e2 resp. e1. *)
    Function[t, subsets = Select[subsets, Length[Intersection[#, t]] =!= 1 &];] /@ t1; 
    Function[t, With[{a = If[MemberQ[t, e], 1, 2]},
      subsets = Select[subsets, Length[Intersection[#, t]] =!= a &]];] /@ t2;
    (* Exclude subsets for which M resp. N involves some but not all members of a multiple edge. *)
    {t1, t2} = Function[e3, Flatten[Position[e3, #]]& /@ Cases[Tally[e3], {a_, b_ /; b > 1} :> a]] /@ {e1, e2};
    Function[t, subsets = Select[subsets, MatchQ[Length[Intersection[#, t]], 0 | Length[t]]&];] /@ t1;
    Function[t, With[{a = If[MemberQ[t, e], 1, 0]},
      subsets = Select[subsets, MatchQ[Length[Intersection[#, t]] + a, 0 | Length[t]]&]];] /@ t2;
  ];
  If[OptionValue[UseQuadrilateralSimplification] && k >= 9 && Length[subsets] >= 62,
    (* Exclude subsets for which M resp. N involves precisely 3 edges of a quadrilateral in e2 resp. e1. *)
    {t1, t2} = Quadrilaterals /@ {e1, e2};
    Function[t, subsets = Select[subsets, Length[Intersection[#, t]] =!= 1 &];] /@ t1;
    Function[t, With[{a = If[MemberQ[t, e], 2, 3]}, subsets = Select[subsets, Length[Intersection[#, t]] =!= a &]];] /@ t2;
  ];
  
  (* Main part *)
  sum = With[{M = Append[#, e], N = Complement[Range[k], #]},
      If[(Length[M] <= k / 2 && (q1 = QuotientEdges[e1, N]) =!= {} && (q2 = QuotientEdges[e2, M]) =!= {}) ||
         (Length[M] >  k / 2 && (q2 = QuotientEdges[e2, M]) =!= {} && (q1 = QuotientEdges[e1, N]) =!= {}),
        If[(l = BigraphCount[q1, Delete[e2, Transpose[{N}]]]) === 0, 0,
          l * BigraphCount[Delete[e1, Transpose[{M}]], q2]], 0]]& /@ subsets;
  sum = Total[sum];
  l = If[(q2 = QuotientEdges[e2, {e}]) =!= {}, BigraphCount[Delete[e1, e], q2], 0];
  sum += If[e1 === e2, 2 * l, l + If[(q1 = QuotientEdges[e1, {e}]) =!= {}, BigraphCount[q1, Delete[e2, e]], 0]];
  Return[sum];
];


(* Subroutines for the combinatorial algorithm from [1]*)

(* Components of a graph, given as lists of vertices. *)
Components::usage="(internal function)\nComponents[edges] computes the connected components of the graph given by its edges (ignoring isolated vertices).";
Components[e_List] := Components[Union[Flatten[e]], e];
Components[v_List, e_List] :=
Module[{comps = {}, c},
  While[(c = Complement[v, Union[Flatten[comps]]]) =!= {},
    AppendTo[comps, FixedPoint[Union[#, Cases[e, ({a_, Alternatives @@ #} | {Alternatives @@ #, a_}) :> a]]&, Take[c, 1]]];
  ];
  Return[comps];
];

(* Quotient a graph (given by a list of edges) by a subset of the edges (given by their positions in the list). *)
(* also checks for loops *)
QuotientEdges::usage="(internal function)\nQuotientEdges[edges,pos] computes the quotient graph of the graph given by edges and quotient by the edges at positions pos.";
QuotientEdges[e_List, p_List] :=
  With[{e1 = Delete[e, Transpose[{p}]] /. Flatten[MapIndexed[(#1 -> #2[[1]])&, Components[Union[Flatten[e]], e[[p]]], {2}]]},
    If[Cases[e1, {a_, a_}] =!= {}, {}, Sort /@ e1]];

(* List of triangles in a graph (given by its edge list).
 * The triangles are given by the positions of their edges in the list. *)
Triangles::usage="(internal function)\nTriangles[edges] computes the triangle subgraphs of the graph given by edges.";
Triangles[e_List] :=
  Union[Sort /@ (Join @@ MapThread[
    Function[{a, b, i},
      {i, Position[e, {a, #} | {#, a}][[1,1]], Position[e, {b, #} | {#, b}][[1,1]]}& /@
        Intersection[Cases[e, ({a, c_} | {c_, a}) :> c], Cases[e, ({b, c_} | {c_, b}) :> c]]
    ], Append[Transpose[e], Range[Length[e]]]])];

(* List of quadrilaterals in a graph (given by its edge list).
 * The quadrilaterals are given by the positions of their edges in the list. *)
Quadrilaterals::usage="(internal function)\nQuadrilaterals[edges] computes the quadrilateral subgraphs of the graph given by edges.";
Quadrilaterals[e_List] :=
Module[{vert = Union[Flatten[e]], p},
  p = Join @@ (Function[v, {#1, v, #2}& @@@ Subsets[Cases[e, {v, a_} | {a_, v} :> a], {2}]] /@ vert);
  p = Join @@ (Function[{a, d, b},
      {Position[e, {a, d} | {d, a}][[1,1]], Position[e, {d, b} | {b, d}][[1,1]],
       Position[e, {b, #} | {#, b}][[1,1]], Position[e, {a, #} | {#, a}][[1,1]]}& /@
        Intersection[Cases[e, ({a, c_} | {c_, a}) :> c], Cases[e, ({b, c_} | {c_, b}) :> c]]
    ] @@@ p);
  p = Select[p, Length[Union[#]] === 4 &];
  p = Union[RotateLeft[#, Position[#, Min[#]][[1,1]] - 1]& /@ p, SameTest -> (Union[#1] === Union[#2]&)];
  Return[p];
];


(* ::Subsection::Closed:: *)
(*Complex Sphere*)


ComplexRealizationCountSphere::usage="ComplexRealizationCountSphere[g] computes the number of complex realizations in d dimensional space.
The following Methods are available:
	Automatic (default): choses Combinatorial for dimension 2 and Groebner otherwise.
	Groebner: all dimensions, probabilistic; generates a system of equations and counts the complex solutions thereof.
	Combinatorial: dimension 2, deterministic; uses the algorithm from [4].";
Options[ComplexRealizationCountSphere]={Method->Automatic,RemoveDegD->True};
ComplexRealizationCountSphere::method="The chosen Method is invalid.";
ComplexRealizationCountSphere::method2="Method is not available for dimension `1`.";
SyntaxInformation[ComplexRealizationCountSphere]={"ArgumentsPattern"->{_,_.,OptionsPattern[]}};
ComplexRealizationCountSphere[g_Integer,opts:OptionsPattern[]]:=ComplexRealizationCountSphere[g,2,opts]
ComplexRealizationCountSphere[G_Graph,opts:OptionsPattern[]]:=ComplexRealizationCountSphere[G,2,opts]
ComplexRealizationCountSphere[g_Integer,dim_Integer,opts:OptionsPattern[]]:=ComplexRealizationCountSphere[G2Graph[g],dim,opts]
ComplexRealizationCountSphere[G_Graph,dim_Integer,opts:OptionsPattern[]]:=Module[{Gs,cc,pos,switch},
  
  Gs=StandardGraph[G];
  (* If there are vertices of valency less than dim it cannot be rigid*)
  If[Min[pos=VertexDegree[Gs]]<dim&&VertexCount[Gs]>dim,Return[\[Infinity]]];
  
  (*check for vertices with degree dim *)
  cc=1;
  If[OptionValue[RemoveDegD],
   While[Min[pos=VertexDegree[Gs]]==dim&&VertexCount[Gs]>dim+1,
     pos=Flatten[Position[pos,dim]];
     Gs=VertexDelete[Gs,pos];
     Gs=StandardGraph[Gs];
     cc*=2^Length[pos];
   ];
  ];
  If[EdgeCount[Gs]==Binomial[VertexCount[Gs],2],Return[Switch[VertexCount[Gs],dim,cc,dim+1,2*cc,x_/;x>dim+1,0,x_/;x<dim,Infinity]]];
  
  (* Apply counting method *)
  If[(switch=OptionValue[Method])===Automatic,switch=If[dim==2,"Combinatorial","Groebner"]];
  Return[cc*Switch[
    switch,
    "Groebner",
      RealizationCountGroebnerSphere[Gs,dim],
    "Combinatorial",
      If[dim==2,
        CombinatorialCountSphere[Gs],
        Message[MinRigidGraphQ::method2,dim]
      ],
    _,
      Message[ComplexRealizationCount::method]
  ]]
]


(* ::Subsubsection::Closed:: *)
(*Gr\[ODoubleDot]bner Basis Approach*)


RealizationCountGroebnerSphere::usage="RealizationCountGroebnerSphere[g,d] computes the number of complex realizations of a graph g in dimension d by counting solutions of a system of equations with Gr\[ODoubleDot]bner basis.";
RealizationCountGroebnerSphere[G_Graph,dim_Integer]:=Module[{eqs,var,fix,x,gb,sol},
  eqs=DistanceEquations[G,dim+1,RHS->"Random",RandomMax->10^6,Var->x];
  eqs=Join[eqs,SphereEquations[VertexCount[G],dim+1,x]];
  fix=Range[dim];
  MapIndexed[Function[{c,i},
    eqs = eqs /.Thread[(x[c,#]&/@Range[dim-First[i]+1])->0]],
    fix
  ];
  var=Variables[eqs];
  eqs=Select[eqs,Head[#]=!=Integer&];
  If[eqs === {}, Return[1]];
  gb = GroebnerBasis[eqs, var, MonomialOrder -> DegreeReverseLexicographic, Modulus -> NextPrime[2^31, -2]];
  sol = CountUnderTheStaircase[LeadingMonomial[#, var] & /@ gb,var];
  Return[sol/2^dim]
]


(*subroutines*)
SphereEquations::usage="SphereEquations[n,d,x] gives the equations that bind a point to the d dimensional sphere. n is the number of points and x the variable name.";
SphereEquations[n_Integer,dim_Integer,x_Symbol]:=Table[Sum[x[j,i]^2,{i,dim}]-1,{j,n}]


(* ::Subsubsection::Closed:: *)
(*Combinatorial Algorithm*)


CombinatorialCountSphere::usage="CombinatorialCountSphere[g] computes the spherical number of complex realizations for the graph g.";
CombinatorialCountSphere[G_Graph]:=CombinatorialCountSphere[GEdges[G],GEdges[G]]
CombinatorialCountSphere[g_Integer]:=CombinatorialCountSphere[G2Edges[g],G2Edges[g]]

CombinatorialCountSphere[EG_List,EH_List]:=CombinatorialCountSphere[Join[Union@@EG,Union@@(#+Table[Length[Union@@EG],2]&/@EH)],Join@@#&/@Transpose[{EG,#+Table[Length[Union@@EG],2]&/@EH}],1]
CombinatorialCountSphere[N_List,Q_List,fix_]:=Module[{Qp,res,II,JJ,Q22,Q34,IIn,Q01,JJn,sets,subs},
  If[Length[N]==3||Length[N]==4,Return[1]];
  If[Length[N]==2,Return[0]];
  Qp=Delete[Q,fix];
  sets=FindAllSetsIJ[#[[1;;2]]&/@Q,#[[3;;4]]&/@Q,N,fix];
  res=0;
  res=Plus@@Map[
    Function[set,
      II=First[set];
      JJ=Last[set];
      Q22=Select[Qp,Length[Intersection[#,II]]==2&];
      If[Length[Q22]==0,
        Q34=Select[Qp,MemberQ[{3,4},Length[Intersection[#,II]]]&]/.Thread[JJ->"x"];
        IIn=Append[II,"x"];
        subs=Thread[IIn->Range[Length[IIn]]];
        Q34=Q34/.subs;
        IIn=IIn/.subs;

        If[Length[IIn]-Length[Q34]==3,
          Q01=Select[Qp,MemberQ[{0,1},Length[Intersection[#,II]]]&]/.Thread[II->"x"];
          JJn=Append[JJ,"x"];
          subs=Thread[JJn->Range[Length[JJn]]];
          Q01=Q01/.subs;
          JJn=JJn/.subs;

          If[Length[JJn]-Length[Q01]==3,
            CombinatorialCountSphere[IIn,Q34,fix]*CombinatorialCountSphere[JJn,Q01,fix],0
          ],0
        ],0
      ]
    ]
  ,sets];
  Return[res];
]


(* subroutines *)
FindAllSetsIJ::usage="(internal function)\nFindAllSetsIJ[EG,EH,N,sei] computes all the possible pairs (I,J)";
FindAllSetsIJ[EG_List,EH_List,N_List,sei_Integer]:=Module[{Nn},
  Nn=Complement[N,Join[EG[[sei]],EH[[sei]]]];
  Return[{Join[EG[[sei]],#],Join[EH[[sei]],Complement[Nn,#]]}&/@Subsets[Nn]];
]


(* ::Section:: *)
(*References*)


(* [1] J. Capco, M. Gallet, G. Grasegger, C. Koutschan, N. Lubbes, and J. Schicho
 *     An algorithm for computing the number of realizations of a Laman graph
 *     Zenodo, 2018
 *     doi: 10.5281/zenodo.1245506
 *     Implementing [2]
 * [2] J. Capco, M. Gallet, G. Grasegger, C. Koutschan, N. Lubbes, and J. Schicho
 *     The number of realizations of a Laman graph
 *     SIAM Journal on Applied Algebra and Geometry, 2(1):94-125, 2018
 *     doi: 10.1137/17M1118312
 * [3] A. Lee, I. Streinu
 *     Pebble game algorithms and sparse graphs
 *     Discrete Mathematics, 308(8):1425-1437, 2008
 *     doi: 10.1016/j.disc.2007.07.104
 * [4] M. Gallet, G. Grasegger, J. Schicho
 *     Counting realizations of Laman graphs on the sphere
 *     The Electronic Journal of Combinatorics, 27(2):P2.5, 2020
 *     doi: 10.37236/8548
 * [5] S.J. Gortler, A.D. Healy, D.P. Thurston 
 *     Characterizing generic global rigidity
 *     American Journal of Mathematics 132(4):897-939, 2010
 *     10.1353/ajm.0.0132
 *)
