# script that creates all Figures for the publication 
# Friend et al 2022,

#Authors: Andrew Friend, Annemarie Eckes-Shephard

#For guidance, Nature’s standard figure sizes are 90 mm (single column) and 
#180 mm (double column) and the full depth of the page is 170 mm.
res= 1000 #1000 change for high-quality images after acceptance
cex.legend = 0.5
cex.lab = 0.7
file_path <- "/Users/annemarie/Library/CloudStorage/OneDrive-LundUniversity/3_Other_Projects/2021_Friend_eckes-Shephard/RINGS_v2_07.07.2022/CF_FIGS/"
obs_path  <- "Observations/"
file_path <- ""
#obs_path <- ""
figs_path <- paste0(file_path,"Figures/")
#specs similar across plots:
runs_col   <- safe_colorblind_palette <- c("#E6AB02","#1B9E77", "#D95F02", "#7570B3", "#666666","#E7298A", "#66A61E", "#E6AB02", "#D55E00","#56B4E9")

# Parameters taken from the literature and calculated from observations
# used for conversions in this script
ttl = 50.1 # cell tangential Length [mu m]
tal = 2680. # Cell axial length [ mu m]
##############################################################################
#Functions used in data aggregation and plotting

##################################################################################
#bin_and_running_mean
#input: model_config_output numeric vector,
#input: bin_size numeric, desired size used for binning across the radial file
#input: maxlength_radial_file numeric, maximal radial file length that has come 
# out of the simulations, it is used together with the bin size to derive the
# total number of bins for the aggregation.
#input: varname, string, whether binning is done over cell mass density ("den") 
# or cell lengths ("L")
#output: a list of three numeric vectors with either cell length ('L') or 
# cell mass density ('den'), together with cell distance from the cambium ('Di')
# and number of datapoints in the bin ('n')
bin_and_running_mean <- function(model_config_output,bin_size,maxlength_radial_file,varname){
  # author of original code: Andrew F, translated to R, Annemarie ES
#binning:
  bs = bin_size
  mlen = maxlength_radial_file
  den_a = vector()
  pos_a = vector()
  ns_a   = vector()
  a = .0
  b = a + bs
  while (b < mlen+bs){
    n = length(t(model_config_output[varname]))
    i = 1
    j = .0
    dena_sum = .0
    while (i <=n){
      d = model_config_output$Di[i]
      # print(i)
      # print(d)
      if (d >= a & d < b){
        dena_sum = dena_sum + model_config_output[i,varname]
        j = j + 1.
      }
      i = i + 1
    }
    if (j > 0){
      den_a <- append(den_a,dena_sum/j)
      ns_a   <- append(ns_a,j)  # record how many data points were actually used to derive the number in this bin
      pos_a <- append(pos_a,a+(b-a)/2.)
    }
    
    a = a + bs
    b = b + bs
  }

n = length(den_a)
run = 3
den_ar = vector()
den_ar = append(den_ar,den_a[1])
# running means across bins:
for (i in 2:(n-1)){
  sden = .0
  for (j in (i-1):(i+1)){
    sden = sden + den_a [j]
  }
  den_ar = append(den_ar,(sden/run))
}
den_ar = append(den_ar,den_a[n])

#prep output:
if(varname=="L"){
  out <- list(den_ar,pos_a,ns_a)
  names(out)[1:3] <- c("L","Di","n")
  return(out)
}
if(varname=="den"){
  out <- list(den_ar,pos_a,ns_a)
  names(out)[1:3] <- c("den","Di","n")
  return(out)
}

}
###################makeTransparent
#To show where overlap between predicted cells is greatest
#note: always pass alpha on the 0-255 scale
makeTransparent <- function(someColor, alpha=100){
  newColor<-col2rgb(someColor)
  apply(newColor, 2, function(curcoldata){rgb(red=curcoldata[1], green=curcoldata[2],
                                              blue=curcoldata[3],alpha=alpha, maxColorValue=255)})
}  
################################ add_legend
add_legend <- function(...) {
  opar <- par(fig=c(0, 1, 0, 1), oma=c(0, 0, 0, 0), 
              mar=c(0, 0, 0, 0), new=TRUE)
  on.exit(par(opar))
  plot(0, 0, type='n', bty='n', xaxt='n', yaxt='n')
  legend(...)
}
###################add_cellNumberColumn
# quick and dirty addition of column to highlight each radial file for later 
# year-specific binning and plotting
# input: model_output RINGS_v2 model output from fort.97 table read in.
# output model output, with additional column
add_cellNumberColumn <- function(model_output){
  pos=1
  rfile = 1
  model_output$celln[1] <- pos 
  model_output$row[1] <- rfile
  for(i in 2:dim(model_output)[1]){
    pos=pos+1
    if(model_output$Di[i-1] < model_output$Di[i]){
      model_output$celln[i] <- pos
      model_output$row[i] <- rfile
    }else{
      pos = 1
      rfile = rfile+1
      model_output$celln[i] <- pos
      model_output$row[i] <- rfile
    }
  }
  return(model_output)
}



###########FIGURE1:#############################################################
#FIGURE1:
# we normalis every simulated set of radial files for each model configuration run
bs = 0.02 # bin size (frac)
mlen = 1.0 # length to process (frac)


# Di = a tracheid's distance from the cambium [mu]
# L  = a tracheid's length ( radial diameter), this includes cell walls [mu m]
# M  = a tracheid's mass [g]
#load data for each model configuration run:

full_model <- read.table(paste0(file_path,'fort.97_20'))
names(full_model) <- c("Di", "L ", "M")
# add another cellnumber column, for analysis with tgram
full_model <- add_cellNumberColumn(full_model)

fixed_temp <- read.table(paste0(file_path,'fort.97_20_nT') )
names(fixed_temp) <-  c("Di", "L ", "M")
# add another cellnumber column, for analysis with tgram
fixed_temp <- add_cellNumberColumn(fixed_temp)

fixed_zonewidth <- read.table(paste0(file_path,'fort.97_20_fzw')) 
names(fixed_zonewidth)<-  c("Di", "L ", "M")
# add another cellnumber column, for analysis with tgram
fixed_zonewidth <- add_cellNumberColumn(fixed_zonewidth)

fixed_zonewidth_nodormancy <- read.table(paste0(file_path,'fort.97_20_fzw_nd'))
names(fixed_zonewidth_nodormancy) <- c("Di", "L ", "M")
# add another cellnumber column, for analysis with tgram
fixed_zonewidth_nodormancy <- add_cellNumberColumn(fixed_zonewidth_nodormancy)

sugar_saturation <- read.table(paste0(file_path,'fort.97_20_sS'))
names(sugar_saturation) <-  c("Di", "L ", "M")
# add another cellnumber column, for analysis with tgram
sugar_saturation <- add_cellNumberColumn(sugar_saturation)



# some data manipulation and aggregation:

#turn radial files around
full_model$Di                 <-  1- full_model$Di
fixed_temp$Di                 <-  1- fixed_temp$Di
fixed_zonewidth$Di            <-  1- fixed_zonewidth$Di
fixed_zonewidth_nodormancy$Di <-  1- fixed_zonewidth_nodormancy$Di
sugar_saturation$Di           <-  1- sugar_saturation$Di

# list model output objects for easy looping access:
runs <- list(full_model,fixed_temp,fixed_zonewidth,fixed_zonewidth_nodormancy,sugar_saturation)
names(runs) <- c("full_model","fixed_temp","fixed_zonewidth","fixed_zonewidth_nodormancy","sugar_saturation")

# loop through all model configurations and:
# add new variable-columns, that will hold derived variables Vc and den, from output variables:
# create binned tracheidograms for plotting

#initialise list of lists for data storage and easy retrieval in loops for plotting:
collect_stdTrach_single <- list(list(),list(),list(),list(),list())
names(collect_stdTrach_single) <- c("full_model","fixed_temp","fixed_zonewidth","fixed_zonewidth_nodormancy","sugar_saturation")

for (e in names(runs)){
  # add new vars:
  print(e)
  runs[[e]]$Vc  <- runs[[e]]$L * ttl * tal / 1.E12 # ml    eq 11 in manuscript
  runs[[e]]$den <- 1.E-6 *  runs[[e]]$M /  runs[[e]]$Vc # g/cm3 
  # create binned trachs:
  collect_stdTrach_single[[e]] <- bin_and_running_mean(model_config_output = runs[[e]],
                                                       bin_size = bs,maxlength_radial_file = mlen,varname="den")
}

# plot:

jpeg(filename=paste0(figs_path,'Fig1.jpeg'),units="mm",width=90, height=60, res=res)

par(mfrow=c(2,1),mar=c(0,2,0,0),oma=c(1,0,0.02,4),cex.axis=0.7, cex.lab=0.7, cex.main=1, cex.sub=1,ps=8,xpd=TRUE)

#Fig 1a)
e="full_model"

# start with all simulated density values that were simulated across all 20 years:
plot(runs[[e]]$Di,runs[[e]]$den,pch=16,cex=0.12,ylab="",xlab="",yaxt="n",col=makeTransparent('grey',alpha=50),
     ylim=c(0,1.2),xaxt="n")

# add all the tracheidograms across all 20 years to plot
# grey lines are the (binned) dens - tracheidograms of the different years at that site.
# 100 radial files were modelled for each year between the simulation years 1976 to 1995. 
# (note "row" is equivalent to an individually simulated radial file)
for(row in 1:20){
  n=100
  tmp <- bin_and_running_mean(model_config_output = runs[[e]][which(runs[[e]]$row > row*n-99 & runs[[e]]$row <= row*n),], 
                              bin_size = bs,maxlength_radial_file = mlen,varname="den")
  lines(tmp$Di, tmp$den,type="l",col="black",lwd=0.2)
}

# add binned tracheidogram derived from all years' simulations:
lines(collect_stdTrach_single[[e]]$Di,collect_stdTrach_single[[e]]$den,lwd=1)


# add observations:
# Source:
# Cuny, H. E., Rathgeber, C. B. K., Frank, D., Fonti, P. & Fournier, M. 
# Kinetics of tracheid development explain conifer tree-ring structure. 
# New Phytol. 203, 1231–1241, DOI: 10.1111/nph. 12871 (2014).
e3 = read.table(file = paste0(obs_path,"cuny_3e.dat"),col.names = c("d_3e", "den_3e"),sep = ",")
f3 = read.table(file = paste0(obs_path,"cuny_3f.dat"),col.names = c("d_3f", "den_3f"),sep = ",")

#unit conversion to g cm-3
e3$d_3e   = e3$d_3e / 1.0E2
f3$d_3f   = f3$d_3f / 1.0E2
e3$den_3e = e3$den_3e / 1.0E3
f3$den_3f = f3$den_3f / 1.0E3

#Source: Southern finnland (SFinP) and Northern Finnland (NFinP):
#  Decoux, V., Varcin, E. & Leban, J.-M. Relationships between the intra-ring wood 
#  density assessed by X-ray densitometry and optical anatomical measurements in conifers.
#  Consequences for the cell wall apparent density determination.
#  Annals For. Sci. 61, 251–262, DOI: 10.1051/forest:2004018 (2004).

NFinP= c(433,371,352,361,362,354,366,376,375,378,396,425,479,530,628,746,826,921,959,875)
SFinP= c(508,374,355,346,348,360,383,398,406,427,453,530,706,817,845,859,975,997,998,862)
NfP  = c(1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20)
x = NfP / 20. - .5 / 20.
#unit conversion to g cm-3
yN = NFinP / 1000.
yS = SFinP / 1000.

#France:
lines(e3$d_3e ,e3$den_3e, col='black', lty=2 ,lwd=1)
lines(f3$d_3f ,f3$den_3f, col='black', lty=3 ,lwd=1)

#Finnland:
lines(x,yN, col='black', lty=4 ,lwd=1)
lines(x,yS, col='black', lty=5 ,lwd=1)

## layout and legend:
axis(side = 2,lwd = 0, line = -0.8, las = 2)
axis(side = 2, tck = -.015, labels = NA)
axis(side = 1, tck = .015, labels = NA)
#mtext(side = 3 , "a)",line=-1, adj=0.05 ,outer = FALSE)
mtext(side=1,'a)',line=-5.7, adj=0.015,col='black',cex=0.7)


# Fig 1 b)
# start with all simulated density values that were simulated across all 20 years for the full model:
e="full_model"
plot(runs[[e]]$Di,runs[[e]]$den,pch=16,cex=0.1,ylab="",xlab="",yaxt="n",
     col=makeTransparent('grey',alpha=50),ylim=c(0,1.2),xaxt="n")

# add all the binned (annual) dens - tracheidograms to plot
# grey lines are the (binned) dens - tracheidograms of the different years at that site.
for(row in 1:20){
  n=100
  tmp <- bin_and_running_mean(model_config_output = runs[[e]][which(runs[[e]]$row > row*n-99 & runs[[e]]$row <= row*n),], 
                              bin_size = bs,maxlength_radial_file = mlen,varname="den")
  lines(tmp$Di, tmp$den,type="l",col="black",lwd=0.2)
}

#add other model configurations- but only their binned tracheidogram derived from binning across all 20 years:
lines(collect_stdTrach_single[["fixed_temp"]]$Di,
      collect_stdTrach_single[["fixed_temp"]]$den,type="l",col=runs_col[1], lwd=1)
points(collect_stdTrach_single[["fixed_zonewidth"]]$Di,
       collect_stdTrach_single[["fixed_zonewidth"]]$den,type="l",col=runs_col[2], lwd=1)
points(collect_stdTrach_single[["fixed_zonewidth_nodormancy"]]$Di,
       collect_stdTrach_single[["fixed_zonewidth_nodormancy"]]$den,type="l",col=runs_col[3], lwd=1)
points(collect_stdTrach_single[["sugar_saturation"]]$Di,
       collect_stdTrach_single[["sugar_saturation"]]$den,pch=2,type="l",col=runs_col[4], lwd=1)

# add full_model binned tracheidogram derived from all years' binned tracheidograms:
# binned tracheidogram derived from all years' simulations:
lines(collect_stdTrach_single[[e]]$Di,collect_stdTrach_single[[e]]$den,lwd=1.4)


axis(side = 2,lwd = 0, line = -0.8, las = 2)
axis(side = 2, tck = -.015, labels = NA)
axis(side = 1, tck = -.015, labels = NA)
axis(side = 1,lwd = 0, line = -1.3,cex=0.7)
mtext(side = 2 , expression(Density ~(g ~cm^-3)),line=-1 ,outer = TRUE,cex = cex.lab)

mtext(outer=FALSE,side=1,expression(Distance~from~inner~edge~(fraction~of~ring)),line=0.2,cex = cex.lab)


#mtext(side = 3 , "b)",line=-1, adj=0.05 ,outer = FALSE,cex=0.7)
mtext(side=1,'b)',line=-5.7, adj=0.015,col='black',cex = cex.lab)

# Add legend between a&b, outside plot region
add_legend(0.62,0.25,xpd=TRUE,  legend = c("density/cell","tracheidograms"), title = "Model output",
           pch = c(16,NA), lty = c(NA,1), lwd= c(NA,1.5), col='grey',box.lwd = 0, cex = cex.legend, bg = "transparent")

# Add legend to the side of Fig 1 b), outside plot region
add_legend(0.58,-0.25, xpd=TRUE, legend = c("full model","fixed temp","fixed zw","fixed zw, no dorm. ","sugar saturation"), cex = cex.legend,
           lty = c(1,1,1,1,1), col = c(1,runs_col[1:4]), lwd = 1.5,
           box.lwd = 0, title = "Model configurations", bg = "transparent")

# Add legend to the side of fig 1 a), outside plot region
add_legend(0.58,0.95, xpd=TRUE, 
           legend = c("full model","Obs1 (France)","Obs2 (France)","Obs3 (Finland)","Obs4 (Finland)"),
           lty = c(1,2,3,4,5), box.lwd = 0, cex = cex.legend , title = "Model vs. observation", bg = "transparent")

dev.off()
################################################################################



############APPENDIX FIGURE 1###################################################
#APPENDIX FIGURE 1 - plots all binned dens - radial files and the simulated densities
# from Figure 1b) from the model configurations that were not included in the graph
# plot all binned tracheidograms for the appendix, this time with a graph each 
# year containing all model configurations' output.

pdf(file = paste0(figs_path,'appdx.pdf'), width = 10, height = 6)

#par(mfrow=c(5,4),mar=c(0,0,0,0),oma=c(4,4,0.5,0.5))
par(mfrow = c(4,6), mar = c(0,0,0,0), oma = c(4,4,0.5,0.5))

years <- seq(1976,1995,1)

for(row in 1:20){
  c=1
  for (e in names(runs)){
    #print(e)
    if(e =="full_model"){
      tmp_save <- bin_and_running_mean(model_config_output = runs[[e]][which(runs[[e]]$row > row*n-99 & runs[[e]]$row <= row*n),], bin_size = bs,maxlength_radial_file = mlen,varname="den")
      
      plot(runs[[e]][which(runs[[e]]$row > row*n-99 & runs[[e]]$row <= row*n),]$Di,runs[[e]][which(runs[[e]]$row > row*n-99 & runs[[e]]$row <= row*n),]$den,pch=16,cex=0.5,ylab="",xlab="",yaxt="n",col="grey",
           ylim=c(0,1.2),xlim=c(0,1),axes = FALSE)
      lines(tmp_save$Di, tmp_save$den,type="l",col="black",lwd=1.2)
      
      
    }else{  
    tmp <- bin_and_running_mean(model_config_output = runs[[e]][which(runs[[e]]$row > row*n-99 & runs[[e]]$row <= row*n),], bin_size = bs,maxlength_radial_file = mlen,varname="den")
    
    points(runs[[e]][which(runs[[e]]$row > row*n-99 & runs[[e]]$row <= row*n),]$Di,runs[[e]][which(runs[[e]]$row > row*n-99 & runs[[e]]$row <= row*n),]$den,pch=16,cex=0.5,ylab="",xlab="",yaxt="n",col="grey",
         ylim=c(0,1.2),xlim=c(0,1),axes = FALSE)
    lines(tmp$Di, tmp$den,type="l",col=runs_col[c],lwd=1.2)
    c = c + 1 # switch to next colour (= model assumption)
    }
    # redraw full_model line again, as it is covered quite a bit:
    lines(tmp_save$Di, tmp_save$den,type="l",col="black",lwd=1.2)
    

  }
  #axes on the outer edges only 

    #if(row==1|| row==5 || row==9 ||row==13||row==17){
  if(row==1|row==7|row==13|row==19){
      axis(side=2,las=2)
    }
    #if(row>=17){
   if(row>=15){
      axis(side=1,las=2)
    }
  
  box()
  par(new=TRUE)
  mtext(paste(years[row]),side=1,line=-1.7,adj=0.4)

  
}
# add legend
plot.new()
legend("center",c("Single cells","Binned profile:"),
       lty=c(NA,1),pch=c(16,16),pt.cex=c(2,0),lwd=c(0,1),col=c("grey","black"),
       text.font=c(1,0.5),bty="n",cex=0.9)
plot.new()
legend("bottomleft",legend = names(runs), col = c(1,runs_col),
       lty=rep(1,length(runs_col)),cex=0.75, box.lwd = 0, 
       title = "Model configurations", bg = "transparent")


mtext(outer=TRUE,side=1,"Distance from inner edge (fraction of ring)",line=2.4)
mtext(outer=TRUE,side=2,expression(Density ~(g ~cm^-3)),line=2.1)


dev.off()
################################################################################



############FIGURE2:##############################################


#load data for each model configuration run:

full_model <- read.table(paste0(file_path,'fort.94.def'))
names(full_model) <- c("Di", "L", "M")
#full_model <- add_cellNumberColumn(full_model)

full_temp_incr<- read.table(paste0(file_path,'fort.94.two'))
names(full_temp_incr) <- c("Di", "L", "M")
#full_temp_incr <- add_cellNumberColumn(full_temp_incr)

THKOnly_temp_incr <- read.table(paste0(file_path,'fort.94.twob'))
names(THKOnly_temp_incr) <- c("Di", "L", "M")
#THKOnly_temp_incr <- add_cellNumberColumn(THKOnly_temp_incr)

SucSat_model <- read.table(paste0(file_path,'fort.94.defC'))
names(SucSat_model) <- c("Di", "L", "M")
#SucSat_model <- add_cellNumberColumn(SucSat_model)

SucSat_full_temp_incr<- read.table(paste0(file_path,'fort.94.twoC'))
names(SucSat_full_temp_incr) <- c("Di", "L", "M")
#SucSat_full_temp_incr <- add_cellNumberColumn(SucSat_full_temp_incr)

# some data manipulation and aggregation:

bs = 40. # bin size (um)
mlen = 1850. # length to process (um)
maxlen=mlen
runs <- list(full_model,full_temp_incr,THKOnly_temp_incr,SucSat_full_temp_incr,SucSat_model)
names(runs) <- c("full_model","full_temp_incr","THKOnly_temp_incr","SucSat_full_temp_incr","SucSat_model")

# loop through all model configurations and:
# add new variable-columns, that will hold derived variables Vc and den, from output variables:
# create binned tracheidograms for plotting
#initialise list of lists for data storage and easy retrieval in loops for plotting:
collect_stdTrach_single <- list(list(),list(),list(),list(),list())
names(collect_stdTrach_single) <- c("full_model","full_temp_incr","THKOnly_temp_incr","SucSat_full_temp_incr","SucSat_model")

for (e in names(runs)){
  # add new vars:
  print(e)
  runs[[e]]$Vc  <- runs[[e]]$L * ttl * tal / 1.E12 # ml    eq 11 in manuscript
  runs[[e]]$den <- 1.E-6 *  runs[[e]]$M /  runs[[e]]$Vc # g/cm3 
  # create binned trachs:
  collect_stdTrach_single[[e]] <- bin_and_running_mean(model_config_output = runs[[e]], 
                                                       bin_size = bs,maxlength_radial_file = mlen,varname="den")
}

#plot:

jpeg(filename=paste0(figs_path,'Fig2.jpeg'),units="mm",width=90, height=60, res=res)

par(mfrow=c(2,1),mar=c(0,2,0,0),oma=c(1,0,0.02,4),cex.axis=0.7, cex.lab=0.7, cex.main=1, cex.sub=1,ps=8,xpd=TRUE)

# start with all simulated density positions:
plot(runs[["full_model"]]$Di,runs[["full_model"]]$den,
     pch=16,cex=0.5,col=makeTransparent('grey',alpha=50),
     xlim=c(maxlen,0),ylim=c(0,1.5),ylab='',xlab='',xaxt='n',yaxt='n')
points(runs[["full_temp_incr"]]$Di,runs[["full_temp_incr"]]$den,
       pch=16,cex=0.5,col=makeTransparent(runs_col[7],alpha=50))
points(runs[["THKOnly_temp_incr"]]$Di,runs[["THKOnly_temp_incr"]]$den,
       pch=16,cex=0.5,col=makeTransparent(runs_col[6],alpha=50))
points(runs[["SucSat_full_temp_incr"]]$Di,runs[["SucSat_full_temp_incr"]]$den,
       pch=16,cex=0.5,col=makeTransparent(runs_col[8],alpha=50))

#add binned tracheidograms
lines( collect_stdTrach_single[["full_model"]]$Di,
       collect_stdTrach_single[["full_model"]]$den,type="l",col="black", lwd=2)
points( collect_stdTrach_single[["full_temp_incr"]]$Di,
        collect_stdTrach_single[["full_temp_incr"]]$den,type="l",col=runs_col[7], lwd=2)
points( collect_stdTrach_single[["THKOnly_temp_incr"]]$Di,
        collect_stdTrach_single[["THKOnly_temp_incr"]]$den,type="l",col=runs_col[6], lwd=2)
points( collect_stdTrach_single[["SucSat_full_temp_incr"]]$Di,
        collect_stdTrach_single[["SucSat_full_temp_incr"]]$den,pch=2,type="l",col=runs_col[8], lwd=2)


axis(side = 2,lwd = 0, line = -0.8, las = 2)
axis(side = 2, tck = -.015, labels = NA)
axis(1, at = seq(0,maxlen,by=200), labels = NA,tck= .015)

mtext(outer=FALSE,side=2,expression(Cell~mass~density),line=1.3,cex= cex.lab)
mtext(outer=FALSE,side=2,expression((g ~cm^-3)),line=0.8,cex= cex.lab,adj=0.5)

mtext(side=1,'a)',line=-5.7, adj=0.015,col='black',cex=0.7)



#Fig 2b)

# We need to compare the difference in cell mass density effect across the radial 
# files between model configurations under a temperature increase
# It happens that due to the stochastic properties of RINGS, radial files have 
# slightly different cell mass densities at different locations, which  means that
# when performing the binning especially for density-related variables, 
# towards the end of the radial files, fewer datapoints can be collected 
# in each bin, to the extent that in some cases below, 
# the last bin can e.g. only be filled with 3 datapoints.

# As the model configurations produce radial files of different lengths (max Di)
#, in  order to calculate the difference between the radial files of different
# model configurations and the default (full) model, we need to select the 
# minimum vector length for which all binned radial files have a reasonnable 
# amount of datapoints. We determine that a reasonnable minimum number of 
# datapoints in a bin is 10% of the maximum number of datapoints found in any 
# bin (renders a minimum vector length of 38)
# going with this makes sense because this is equivalent to assuming that the 
# maximum number of datapoints found in any bin is the population size
# and in statistics, 10% of the population size is a good sample size to get
# a reasonnable estimate of the population.

#find maximum datapoints in any bins
maxpoints=max(collect_stdTrach_single[["full_model"]]$n, 
      collect_stdTrach_single[["SucSat_full_temp_incr"]]$n, 
      collect_stdTrach_single[["SucSat_model"]]$n,
      collect_stdTrach_single[["full_temp_incr"]]$n,
      collect_stdTrach_single[["THKOnly_temp_incr"]]$n)

# get 10% of population size
npoints = floor(maxpoints-(maxpoints*0.9))
# determine minimum number of bins which all contain more or equal to a sample 
# size that is 10% of the population size
n=min(length(which(collect_stdTrach_single[["full_model"]]$n>=npoints)), 
      length(which(collect_stdTrach_single[["SucSat_full_temp_incr"]]$n>=npoints)), 
      length(which(collect_stdTrach_single[["SucSat_model"]]$n>=npoints)),
      length(which(collect_stdTrach_single[["full_temp_incr"]]$n>=npoints)),
      length(which(collect_stdTrach_single[["THKOnly_temp_incr"]]$n>=npoints)) )
# so, only select [1:n] of the vector lengths for plotting

#plot:
lwd=1.4
plot(NULL,ylim=c(-0.10,0.15),xlim=c(maxlen,0),ylab='',yaxt='n',xlab= '',xaxt='n',axes=FALSE)
par(xpd=FALSE,cex.axis=0.7, cex.lab=0.7, cex.main=1, cex.sub=1,ps=8)
abline(h=0,col='grey',lwd=1.4) # signifying the 'full model'
mtext(side=1,'neutral \neffect',line=-2.7, adj=0.01,cex=0.6,col='grey')
lines(collect_stdTrach_single[["SucSat_full_temp_incr"]]$Di[1:n],
      (collect_stdTrach_single[["SucSat_full_temp_incr"]]$den[1:n] -
         collect_stdTrach_single[["SucSat_model"]]$den[1:n])/2,xlim=c(2500,0),
      col= "black",lty=2,lwd=lwd)
lines(collect_stdTrach_single[["full_temp_incr"]]$Di[1:n],
      (collect_stdTrach_single[["full_temp_incr"]]$den[1:n] -
         collect_stdTrach_single[["full_model"]]$den[1:n])/2,xlim=c(2500,0),
      col= "black",lty=1,lwd=lwd)
lines(collect_stdTrach_single[["THKOnly_temp_incr"]]$Di[1:n],
      (collect_stdTrach_single[["THKOnly_temp_incr"]]$den[1:n] -
         collect_stdTrach_single[["full_model"]]$den[1:n])/2,xlim=c(2500,0),
      col= "black",lty=4,lwd=lwd)

####select the area of more carbon/temp (positive values), to colour in red:
idx <-  collect_stdTrach_single[["full_temp_incr"]]$den[1:n]-
        collect_stdTrach_single[["full_model"]]$den[1:n] > 0

# select for positive values: 
x <- collect_stdTrach_single[["full_temp_incr"]]$Di[1:n][idx]
y <- (collect_stdTrach_single[["full_temp_incr"]]$den[1:n]-
        collect_stdTrach_single[["full_model"]]$den[1:n])[idx]/2

x.poly = c(x,tail(x, n=1),x[1])
y.poly = c(y,0,0)
polygon(x.poly, y.poly, col=makeTransparent("red",alpha=60), border=NA)

# select for negative values: 

x <- collect_stdTrach_single[["full_temp_incr"]]$Di[1:n][!idx]
y <- (collect_stdTrach_single[["full_temp_incr"]]$den[1:n]-
        collect_stdTrach_single[["full_model"]]$den[1:n])[!idx]/2

x.poly = c(x,rev(x))
y.poly = c(y,rep(0,length(y)))
polygon(x.poly, y.poly, col=gray(0.9), border=NA)#gray(0.7)

#redraw black line
lines(collect_stdTrach_single[["full_temp_incr"]]$Di[1:n],
      (collect_stdTrach_single[["full_temp_incr"]]$den[1:n] -
         collect_stdTrach_single[["full_model"]]$den[1:n])/2,xlim=c(2500,0),
      col= "black",lty=1,lwd=lwd)

axis(side = 2, lwd = 0, line = -0.8, las = 2)
axis(side = 2, tck = -.015, labels = NA)
#axis(side = 1, tck = -.015, labels = NA)
axis(1, at = seq(0,maxlen,by=200), labels = NA,tck= -.015, tick = TRUE)
axis(1, at = seq(0,maxlen,by=200),line = -1.3,lwd=0)

box()

mtext(outer=FALSE,side=2,expression(Delta*Carbon~deposition),line=1.3,cex= cex.lab,adj=0.1)
mtext(outer=FALSE,side=2,expression((g ~cm^-3) /Delta*K),line=0.8,cex= cex.lab,adj=0.33)

mtext(side=1,'b)',line=-5.7, adj=0.015,col='black',cex=0.7)
mtext(outer=FALSE,side=1,expression(Distance~from~cambium~(~mu*m)),line=0.2,cex= cex.lab)

par(fig=c(7,10,1,6)/10)
par(new=T)
add_legend(0.58,-0.30, xpd=TRUE,legend=c('full model','sugar saturation','wall thickening only'),
           lty=c(1,2,4),col="black", bty='n',cex = cex.legend)

add_legend(0.58,0.8, xpd=TRUE,legend=c('full model','full model +2K','sugar saturation +2K','wall thickening only +2K'),
           fill=c('black',runs_col[7],runs_col[8],runs_col[6]), bty='n',cex = cex.legend,title="Model configurations")

dev.off()
################################################################################


############FIGURE3#############################################################
#maximum distance has changed, as these model configurations lead to 
#slightly longer files
maxlen = mlen = 1910 #file length to process(um)
bs =100 #bin size (um)


#load data for each model configuration run:

full_model <- read.table(paste0(file_path,'fort.94.def'))
names(full_model) <- c("Di", "L", "M")
#full_model <- add_cellNumberColumn(full_model)

full_temp_incr<- read.table(paste0(file_path,'fort.94.two'))
names(full_temp_incr) <- c("Di", "L", "M")
#full_temp_incr <- add_cellNumberColumn(full_temp_incr)

Prolif_temp_incr<- read.table(paste0(file_path,'fort.94.twoP'))
names(Prolif_temp_incr) <- c("Di", "L", "M")
#Prolif_temp_incr <- add_cellNumberColumn(Prolif_temp_incr)

SucSat_model <- read.table(paste0(file_path,'fort.94.defC'))
names(SucSat_model) <- c("Di", "L", "M")
#SucSat_model <- add_cellNumberColumn(SucSat_model)

Enl_temp_incr<- read.table(paste0(file_path,'fort.94.twoE'))
names(Enl_temp_incr) <- c("Di", "L", "M")
#Enl_temp_incr <- add_cellNumberColumn(Enl_temp_incr)


# some data manipulation and aggregation:

runs <- list(full_model,full_temp_incr,Enl_temp_incr,Prolif_temp_incr)
names(runs) <- c("full_model","full_temp_incr","Enl_temp_incr","Prolif_temp_incr")

# loop through all model configurations and:
# add new variable-columns, that will hold derived variables Vc and den, from output variables:
# create binned tracheidograms for plotting
#initialise list of lists for data storage and easy retrieval in loops for plotting:
collect_stdTrach_single <- list(list(),list(),list(),list(),list())
names(collect_stdTrach_single) <- c("full_model","full_temp_incr","Enl_temp_incr","Prolif_temp_incr")

for (e in names(runs)){
  # add new vars:
  print(e)
  runs[[e]]$Vc  <- runs[[e]]$L * ttl * tal / 1.E12 # ml    eq 11 in manuscript
  runs[[e]]$den <- 1.E-6 *  runs[[e]]$M /  runs[[e]]$Vc # g/cm3 
  # create binned trachs:
  collect_stdTrach_single[[e]] <- bin_and_running_mean(model_config_output = runs[[e]], 
                                                       bin_size = bs,maxlength_radial_file = mlen,varname="L")
}



#plot:
jpeg(filename=paste0(figs_path,'Fig3.jpeg'),units="mm",width=90, height=60, res=res)

par(mfrow=c(2,1),mar=c(0,2,0,0),oma=c(1,0,0.02,4),cex.axis=0.7, cex.lab=0.7, cex.main=1, cex.sub=1,ps=8,xpd=TRUE)

# add all simulated length positions:
plot(runs[["full_model"]]$Di,runs[["full_model"]]$L,pch=16,cex=0.5,col=makeTransparent('grey',alpha=50),xlim=c(maxlen,0),ylim=c(0,160),ylab='',xlab='',xaxt='n',yaxt='n')
points(runs[["full_temp_incr"]]$Di,runs[["full_temp_incr"]]$L,pch=16,cex=0.5,col=makeTransparent(runs_col[7],alpha=50))
points(runs[["Prolif_temp_incr"]]$Di,runs[["Prolif_temp_incr"]]$L,pch=16,cex=0.5,col=makeTransparent(runs_col[9],alpha=50))
points(runs[["Enl_temp_incr"]]$Di,runs[["Enl_temp_incr"]]$L,pch=16,cex=0.5,col=makeTransparent(runs_col[10],alpha=50))

# add binned tracheidograms
lines(collect_stdTrach_single[["full_model"]]$Di, collect_stdTrach_single[["full_model"]]$L,type="l",col="black", lwd=2)
points(collect_stdTrach_single[["full_temp_incr"]]$Di,  collect_stdTrach_single[["full_temp_incr"]]$L,type="l",col=runs_col[7], lwd=2)
points(collect_stdTrach_single[["Prolif_temp_incr"]]$Di,collect_stdTrach_single[["Prolif_temp_incr"]]$L,pch=2,type="l",col=runs_col[9], lwd=2)
points(collect_stdTrach_single[["Enl_temp_incr"]]$Di,  collect_stdTrach_single[["Enl_temp_incr"]]$L,type="l",col=runs_col[10], lwd=2)

axis(side = 2,lwd = 0, line = -0.8, las = 2)
axis(side = 2, tck = -.015, labels = NA)
axis(1, at = seq(0,maxlen,by=200), labels = NA,tck= .015)

mtext(side=1,'a)',line=-5.7, adj=0.015,col='black',cex=0.7)

mtext(outer=FALSE,side=2,expression(Cell~length ~(mu*m)),line=1,cex = cex.lab)

# We need to compare the difference in cell length effect across the radial files 
# between model configurations under a temperature increase
# It happens that due to the stochastic properties of RINGS, radial files have 
# different lengths, which  means that when performing the binning especially 
# for length-related variables, towards the end of the radial files, fewer 
# datapoints can be collected in each bin, to the extent that in some cases below, 
# the last bin is only filled with 1 datapoint.

# As the model configurations produce radial files of different lengths, in order 
# to calculate the difference between the radial files of different model 
# configurations and the default (full) model, we need to select the minimum 
# vector length for which all binned radial files have a reasonnable amount of 
# datapoints. We deem appropriate the minimum number of datapoints in a bin is 10% of 
# the maximum number of datapoints found in any bin (renders a minimum vector 
# length of 16).
# Going with this makes sense because this is equivalent to assuming that the 
# maximum number of datapoints found in any bin is the population size
# and in statistics, 10% of the population size is a good sample size to get 
# a reasonnable estimate of the population.

#find maximum datapoints in any bins
maxpoints=max(collect_stdTrach_single[["full_model"]]$n, 
              collect_stdTrach_single[["Prolif_temp_incr"]]$n, 
              collect_stdTrach_single[["full_temp_incr"]]$n,
              collect_stdTrach_single[["Enl_temp_incr"]]$n)

# get 10% of population size
npoints = floor(maxpoints-(maxpoints*0.9))
# determine minimum number of bins which all contain more or equal to a sample 
# size that is 10% of the population size
n=min(length(which(collect_stdTrach_single[["full_model"]]$n>=npoints)), 
      length(which(collect_stdTrach_single[["Prolif_temp_incr"]]$n>=npoints)), 
      length(which(collect_stdTrach_single[["full_temp_incr"]]$n>=npoints)),
      length(which(collect_stdTrach_single[["Enl_temp_incr"]]$n>=npoints)) )
# so, only select [1:n] of the vector lengths for plotting

#plot;
lwd=1.4
par(xpd=FALSE)
plot(NULL,ylim=c(-4,8),xlim=c(maxlen,0),ylab='',yaxt='n',xlab= '',xaxt='n')
abline(h=0,col='grey',lwd=1) # signifying the 'full model'
mtext(side=1,'neutral \neffect',line=-2.4, adj=0.05,cex=0.6,col='grey')
lines(collect_stdTrach_single[["Prolif_temp_incr"]]$Di[1:n],
      (collect_stdTrach_single[["Prolif_temp_incr"]]$L[1:n] -
         collect_stdTrach_single[["full_model"]]$L[1:n])/2,xlim=c(2500,0),
      col= "black",lty=2,lwd=lwd)
lines(collect_stdTrach_single[["Enl_temp_incr"]]$Di[1:n],
      (collect_stdTrach_single[["Enl_temp_incr"]]$L[1:n] -
         collect_stdTrach_single[["full_model"]]$L[1:n])/2,xlim=c(2500,0),
      col= "black",lty=4,lwd=lwd)
lines(collect_stdTrach_single[["full_temp_incr"]]$Di[1:n],
      (collect_stdTrach_single[["full_temp_incr"]]$L[1:n] -
         collect_stdTrach_single[["full_model"]]$L[1:n])/2,xlim=c(2500,0),
      col= "black",lty=1,lwd=lwd)


####select the area of more  length/temp incr. (positive values), to colour in red:
idx <-  collect_stdTrach_single[["full_temp_incr"]]$L[1:n] - collect_stdTrach_single[["full_model"]]$L[1:n] > 0
idx[16:17] <- FALSE # set these false here, as they will be plotted as their own polygon further below.

# select for the first round of positive values: 
x <- seq(min(collect_stdTrach_single[["full_temp_incr"]]$Di[1:n]),max(collect_stdTrach_single[["full_temp_incr"]]$Di[1:n]),length.out = length(collect_stdTrach_single[["full_temp_incr"]]$L[1:n]))[idx]
y <- (collect_stdTrach_single[["full_temp_incr"]]$L[1:n]-
        collect_stdTrach_single[["full_model"]]$L[1:n])[idx]/2

x.poly = c(x,tail(x, n=1),x[1])
y.poly = c(y,0,0)
polygon(x.poly, y.poly, col=makeTransparent("red",alpha=60), border=NA)

# select for negative values: 
idx <-  collect_stdTrach_single[["full_temp_incr"]]$L[1:n] - collect_stdTrach_single[["full_model"]]$L[1:n] <= 0
# the polygon has to start from the last positive value, for the correct plotting, so selecting one additional index
idx[4] <- TRUE
x <- seq(min(collect_stdTrach_single[["full_temp_incr"]]$Di[1:n]),max(collect_stdTrach_single[["full_temp_incr"]]$Di[1:n]),length.out = length(collect_stdTrach_single[["full_temp_incr"]]$L[1:n]))[idx]
y <- (collect_stdTrach_single[["full_temp_incr"]]$L[1:n]-
        collect_stdTrach_single[["full_model"]]$L[1:n])[idx]/2

x.poly = c(x,1500,x[1])#tail(x, n=1) manually set to 1500, because line crosses there in graph.
y.poly = c(y,0,0)
polygon(x.poly, y.poly, col=gray(0.9), border=NA)


####select the area of length/temp(positive values), to colour in red:
idx <-  collect_stdTrach_single[["full_temp_incr"]]$L[1:n] - collect_stdTrach_single[["full_model"]]$L[1:n] > 0
idx[1:4] <- FALSE # to make the polygon function work properly, define its own polygon for this second bit of red shaded area
idx[15]  <- TRUE # the polygon has to start from the last negative value,
# select for the second round of positive values: 
x <- seq(min(collect_stdTrach_single[["full_temp_incr"]]$Di[1:n]),max(collect_stdTrach_single[["full_temp_incr"]]$Di[1:n]),length.out = length(collect_stdTrach_single[["full_temp_incr"]]$L[1:n]))[idx]
y <- (collect_stdTrach_single[["full_temp_incr"]]$L[1:n]-
      collect_stdTrach_single[["full_model"]]$L[1:n])[idx]/2

x.poly = c(x,tail(x, n=1),1500)#x[1] manually set to 1500, because line crosses there in graph.
y.poly = c(y,0,0)
polygon(x.poly, y.poly, col=makeTransparent("red",alpha=60), border=NA)


#redraw black line
lines(collect_stdTrach_single[["full_temp_incr"]]$Di[1:n],
      (collect_stdTrach_single[["full_temp_incr"]]$L[1:n] -
      collect_stdTrach_single[["full_model"]]$L[1:n])/2,xlim=c(2500,0),
      col= "black",lty=1,lwd=lwd) 


mtext(side=1,'b)',line=-5.7, adj=0.015,col='black',cex=0.7)
axis(side = 2, lwd = 0, line = -0.8, las = 2)
axis(side = 2, tck = -.015, labels = NA)
axis(1, at = seq(0,maxlen,by=200), labels = NA,tck= -.015, tick = TRUE)
axis(1, at = seq(0,maxlen,by=200),line = -1.3,lwd=0)

mtext(outer=FALSE,side=1,expression(Distance~from~cambium~(~mu*m)),line=0.2,cex= cex.lab)
mtext(outer=FALSE,side=2,expression(Delta*Cell~length~change),line=1.2,cex= cex.lab,adj=0.3)
mtext(outer=FALSE,side=2,expression(~with~temperature~(mu*m) /Delta*K),line=0.7,cex= cex.lab,adj=0.33)


par(fig=c(7,10,0,6)/10)
par(new=T)
add_legend(0.6,0.8, xpd=TRUE,legend=c('full model','full model +2K','Prolif. +2K','Enlarg. +2K'),
           fill=c('black',runs_col[7],runs_col[9],runs_col[10]), bty='n',cex = cex.legend,title="Model configurations")


add_legend(0.6,-0.30, xpd=TRUE,legend=c('full model','Prolif.','Enlarg.'),
           lty=c(1,2,4),col="black", bty='n',cex = cex.legend)

dev.off()
################################################################################


############FIGURE4#############################################################
cols <- c("#66C2A5", "#FC8D62", "#8DA0CB", "#E78AC3", "#A6D854", "#FFD92F", "#E5C494", "#B3B3B3")

full_model <- read.table(paste0(file_path,'fort.93_def'))
names(full_model) <- c("dT", "trw", "tM","trho","MXD","tN")

C_model <- read.table(paste0(file_path,'fort.93_C'))
names(C_model) <- c("dTC", "trwC", "tMC","trhoC","MXDC","tNC")

# dTC or dT     -- Temperature diff. added or subtracted to 
# default daily temperatures 
# the below are averaged over 100 radial files per simulation:
# trw or trwC   -- annual tree ring width (mm)
# tM or tMC     -- total annual ring mass (g[DM])
# trho or trhoC -- mean annual ring density (g[DM] cm-3)
# MXD or MXDXAnnual  --max. den (g[DM] cm-3)
# tN or tNC  --  number of cells in tree ring
# DM = dry mattter

#normalise against default conditions (dT=0)
full_model$trw  <- full_model$trw / full_model$trw[11]  
full_model$tM   <- full_model$tM / full_model$tM[11]  
full_model$trho <- full_model$trho / full_model$trho[11]  
full_model$MXD  <- full_model$MXD / full_model$MXD[11]  
full_model$tN   <- full_model$tN / full_model$tN[11]  
C_model$tMC     <- C_model$tMC / C_model$tMC[11]  

# use running mean for smoother lines (does not make difference to result, 
# just to smooth out any effect of the stochasticity that is intrinsic to the model)
running_mean <- function(model_config_output){
  den_a <- model_config_output
  n = length(den_a)
  k = 3
  den_ar = vector()
  den_ar = append(den_ar,den_a[1])
  # running means
  for (i in 2:(n-1)){
    sden = .0
    for (j in (i-1):(i+1)){ # k=3
      sden = sden + den_a [j]
    }
    den_ar = append(den_ar,(sden/k))
  }
  den_ar = append(den_ar,den_a[n])
  out <- den_ar
  return(out)
}


#plot:
jpeg(filename=paste0(figs_path,'Fig4.jpeg'),units="mm",width=90, height=90, res=res)


par(mfrow=c(1,1),oma=c(1,1,0.1,0.1),mar=c(1,1,0,0),cex.axis=0.7, cex.lab=0.7,ps=8)
par(xpd=FALSE)
wd=1.1
plot(C_model$dTC, running_mean(C_model$tMC) ,col=runs_col[4], lty=2, type='l',
     ylim=c(0.5,1.6),xlim=c(-4,6),ylab='',xlab='',lwd=wd,yaxt="n",xaxt="n")
abline(h=0.6,col='light grey')
abline(h=0.8,col='light grey')
abline(h=1,col='light grey')
abline(h=1.2,col='light grey')
abline(h=1.4,col='light grey')
lines(full_model$dT, running_mean(full_model$trho), col=cols[7],lwd=wd)
lines(full_model$dT, running_mean(full_model$MXD) ,col=cols[2],lwd=wd)
lines(full_model$dT, running_mean(full_model$tN)  ,col =cols[1],lwd=wd) 
lines(full_model$dT, running_mean(full_model$tM)  ,col = runs_col[4],lwd=wd)
lines(full_model$dT, running_mean(full_model$trw) ,col= cols[8],lwd=wd)
mtext(side=1, "Temperature anomaly (K)",line=0.7,cex = cex.lab)
mtext(side=2, "Relative temperature response",line=1.1,cex = cex.lab)
axis(side = 2,lwd = 0, line = -0.7, las = 2)
axis(side = 2, tck = -.015, labels = NA)
axis(side = 1, tck = -.015, labels = NA)
axis(side = 1,lwd = 0, line = -1.05)


legend(legend=c("Ring width", "Cell numbers", "Maximum density", "Mean density",
                "Mass", "Mass, sugar \nsaturation"),
       col=c(cols[8],cols[1],cols[2],cols[7],runs_col[4],runs_col[4]),
       lty=c(1,1,1,1,1,2),lwd=lwd,"topleft",box.lwd = 0,
       cex=cex.legend+0.3,bg="transparent")
box()
wd=1
dev.off()

##########Appendix:
# scripts largely by Andrew Friend

############FIGURE sugars#############################################################

jpeg(filename=paste0(figs_path,'appdx_sugars.jpeg'),units="mm",width=90, height=90, res=res)
par(mfrow=c(1,1),oma=c(1,1,0.1,0.1),mar=c(1,1,0,0),cex.axis=0.7, cex.lab=0.7,ps=8)

#####################################################################
#section numbers for tree 1; aug 19 carbohydrate obs.
obs_s_1=c(1,4,7,9,11,13,14,15,16,17,18,19,20,21,22,23,25,27,29,33,37,42)
#convert to distances from 1/3 way into cz, assumed to be xylem initial, 18th.
obs_d_1=(obs_s_1-.5)*30.-17.5*30.
#total carbohydrate concentration in each section, ug/cm2
#data from Uggla et al. 2001
obs_s_1=c(42.5806451612903,29.6774193548387,120,287.741935483871,313.548387096774,
          300.645161290323,313.548387096774,236.129032258065,339.354838709677,
          300.645161290323,158.709677419355,264.516129032258,89.0322580645162,
          69.0322580645161,54.1935483870968,98.0645161290323,73.5483870967742,
          60.6451612903226,52.9032258064516,37.4193548387097,
          60.6451612903226,5.80645161290323)
#convert to ug/ml
obs_s_1=obs_s_1*10./30.
#####################################################################

#####################################################################
#section numbers for tree 3; aug 19 carbohydrate obs.
obs_s_3=c(1,3,6,9,11,13,14,15,16,17,18,19,20,21,22,23,24,26,30,35,39)
#convert to distances from 1/3 way into cz, assumed to be xylem initial, 15th.
obs_d_3=(obs_s_3-.5)*30.-15.5*30.
#total carbohydrate concentration in each section, ug/cm2
# data from Uggla et al. 2001
obs_s_3=c(74.1935483870968,136.774193548387,269.677419354839,256.774193548387,
          230.967741935484,269.677419354839,273.548387096774,223.870967741936,
          189.032258064516,111.612903225806,85.8064516129032,78.7096774193549,
          90.3225806451613,100,89.0322580645161,85.1612903225807,72.258064516129,
          56.7741935483871,62.5806451612903,28.3870967741935,26.4516129032258)
#convert to ug/ml
obs_s_3=obs_s_3*10./30.
#####################################################################

a1=obs_d_1
b=obs_s_1
a2=obs_d_3
c=obs_s_3

# Make a basic graph
plot( b~a1 , type="b" , bty="l" , 
      col=rgb(0.2,0.4,0.1,0.7) , lwd=1 , pch=17 , 
      ylim=c(0,150),xaxt='n',yaxt='n')


#####################################################################
#add block showing cambial zone
x=c(0-63/3,2*63/3)
y=c(0,150)
rect(x[1],y[1],x[2],y[2],col = rgb(0,1.0,0,alpha=0.1))

#add block showing enlarging zone
x=c(2*63/3,2*63/3+13)
y=c(0,150)
rect(x[1],y[1],x[2],y[2],col = rgb(0,0,1.0,alpha=0.1))

#add block showing thickening zone
x=c(2*63/3+13,2*63/3+13+560)
y=c(0,150)
rect(x[1],y[1],x[2],y[2],col = rgb(1.0,0,0,alpha=0.1))
#####################################################################

cells <- read.table(paste0(file_path,"cells_231.out"),header=T,sep="")
attach(cells)

points(dist,SUC,col = makeTransparent('grey',alpha=50),
       pch = 16, cex = 1,ylab='',xlab='',lwd=wd,yaxt="n",xaxt="n")

lines(b ~a1 , col=rgb(0.2,0.4,0.1,0.7) , lwd=1 , pch=17 , type="b" )
lines(c ~a2 , col=rgb(0.8,0.4,0.1,0.7) , lwd=1 , pch=19 , type="b" )



box()
mtext(side=1,expression(paste("Distance from cambial initial (",mu,"m)")), 
      line = 0.8, cex=cex.lab)
mtext(side=2,expression(paste("Total carbohydrates (mg ml"^"-1",")")), 
      line = 1, cex=cex.lab )

axis(side = 2,lwd = 0, line = -0.7, las = 2)
axis(side = 2, tck = -.015, labels = NA)
axis(side = 1, tck = -.015, labels = NA)
axis(side = 1,lwd = 0, line = -1.05)

#####################################################################
#modelled sugars
#sugars <- read.table("sugars.out",header=T,sep="")
#attach(sugars)
#lines(dist,sugar,col='red',lwd=4)
#####################################################################

# Add a legend
add_legend(-0.81,1.05, legend = c("Tree 1", "Tree 3", "Model"), 
           col = c(rgb(0.2,0.4,0.1,0.7), 
                   rgb(0.8,0.4,0.1,0.7),
                   makeTransparent('grey',alpha=80)),
           pch = c(17,19,16),
           bty = "n", 
           cex = cex.legend+0.2, 
           text.col = "black", 
           horiz = F , 
           inset = c(0.1, 0.1)
           )
dev.off()


############FIGURE zones#############################################################
print("generating appendix figures")

jpeg(filename=paste0(figs_path,'appdx_zones.jpeg'),units="mm",width=90, height=90, res=res)
par(mfrow=c(1,1),oma=c(1,1,0.1,1),mar=c(1,1,0,1),cex.axis=0.7, cex.lab=0.7,ps=8)


#                   1         2         3         4          5         6        7         8
#                blue    orange     green     mauve light_grey dark_grey   yellow     brown
cb_colors =c('#7b85d4','#f37738','#83c995','#d7369e','#c4c9d8','#859795','#e9d043','#ad5b50')
cb1_colors=c('#7273e2','#e16526','#71b783','#d6148c','#d4d9e8','#738583','#d7b831','#9b4a39')

zones <- read.table(paste0(file_path,"zones_v2.out"),header=TRUE,sep="")
names(zones)[8] <- c("Temp")
#rm (Year,kday,pz,ez,tz,mz,ns,T)
#attach(zones)

######################################################################################
plot(zones$kday,zones$Temp,axes=FALSE,xlab="",ylab="",ylim=c(0,40),type="n",tck=.01,yaxt="n",xaxt="n")
lines(zones$kday,zones$Temp)
polygon(zones$kday, zones$Temp, col=cb_colors[7])
mtext("Daily mean temperature (degree C)",side=4,line=0.8,adj=0.1,cex = cex.lab)
axis(4,ylim=c(0,40),at=c(0,5,10,15,20),las=2,lwd=0,line=-0.7)
axis(side = 4, tck = -.015,ylim=c(0,40), at=c(0,5,10,15,20),labels = NA)

par(new=TRUE)
######################################################################################

######################################################################################
xbeg =    0
xend =  365
ybeg =   .0
yend = 1700.
plot (zones$kday,zones$pz,
      col='black',xlim=c(xbeg,xend),ylim=c(ybeg,yend),type='n',yaxt="n",xaxt="n")

polygon(c(109,zones$kday[109:365],365),c(.0,zones$tz[109:365],.0),col=adjustcolor("pink",alpha.f=0.5))
polygon(c(109,zones$kday[109:231],231),c(.0,zones$ez[109:231],.0),col=adjustcolor("lightblue",alpha.f=0.5))
polygon(c(109,zones$kday[109:231],231),c(.0,zones$pz[109:231],.0),col=adjustcolor("lightgreen",alpha.f=0.5))
polygon(c(  1,zones$kday  [1:108],108),c(.0,zones$pz  [1:108],.0),col='brown')
polygon(c(232,zones$kday[232:365],365),c(.0,zones$pz[232:365],.0),col='brown')
polygon(c( 50,      100,172,109),c(150,1040.7,1040.7,110.),col='white',border='white')
######################################################################################

######################################################################################
cells <- read.table(paste0(file_path,"cells_v2.out"),header=TRUE,sep="")

######################################################################################

######################################################################################

l = length (cells$kday)
ad = c(4,13,29,53,84,120,156)
a = cells$kday[1]
j = 1
for (i in 1:l){
  if (cells$kday[i]>a){
    ad[j] = i - 1
    j = j + 1
    a = cells$kday[i]
  }
}
ad [j]=l
print (ad)
#ad = sort(ad)
a1 = array (1:6)
a2 = array (1:6)
for (i in 1:6){
  a1[i] = ad[i]
  a2[i] = ad[i+1]-ad[i]
}
for (i in 1:6){
  pz = 40.
  y0 = 10000.
  j = 0
  while (y0 > pz){
    a  = a1[i] - j
    x0 = cells$kday [a]
    y0 = cells$D [a]
    a  = a + a2[i]
    x1 = cells$kday [a]
    y1 = cells$D [a]
    arrows (x0, y0, x1, y1, col = makeTransparent('black',alpha=70), angle=0, lwd=1)
    j = j + 1
  }
}
######################################################################################

######################################################################################
Sc = (xend - xbeg) / (yend - ybeg)
n = length (cells$kday)
print(n)
i = 1
while (i < n+1) {
  xsize = Sc * 50.1
  ysize = cells$L [i]
  kd = cells$kday [i]
  xl = kd - xsize / 2
  xr = kd + xsize / 2
  yb = cells$D [i] - ysize / 2
  yt = cells$D [i] + ysize / 2
  rect (xl, yb, xr, yt, col = cb_colors [2], lwd=1)
  ################
  a = 1
  b = -(50.1 + cells$L [i])
  c = cells$L [i] * 50.1 * (1. - cells$f_Vl [i])
  thk = (-b - sqrt (b * b - 4. * a * c)) / (2. * a) / 2.
  xl = xl + Sc * thk
  xr = xr - Sc * thk
  yb = yb + thk
  yt = yt - thk
  if (cells$D [i] > zones$tz [cells$kday [i]]){
    rect (xl, yb, xr, yt, col = cb1_colors [5], lwd=.5)
  } else {
    rect (xl, yb, xr, yt, col = "green", lwd=.5)
  }
  ################
  i = i + 1
}
######################################################################################

######################################################################################
# add dots to show observed zone withs  at that site at simulation year
# From Uggla et al. 2001
doy <- c(185,210,231)
opz=c(100, 94, 63)
oez=c(331,200, 76)
otz=c(1007,818,668)
omz=c(1007+20,1436.8,1588.4)

points(doy,omz,cex=1,pch=16)
points(doy,otz,col='red',cex=1,pch=16)
points(doy,oez,col='blue',cex=1,pch=16)
points(doy,opz,col='darkgreen',cex=1,pch=16)
######################################################################################

######################################################################################
text (170,1400,'Mature',font=1,cex = cex.lab)
text (107,800,'Wall thickening',col='red',font=2,cex = cex.lab)
text (90,310,'Enlarging',col='blue',font=1,cex = cex.lab)
text (175,-35,'Proliferating',col='darkgreen',font=1,cex = cex.lab)
text ( 56,35,'Dormant',col="white",font=1,cex=.8)
text (300,35,'Dormant',col="white",font=1,cex=.8)
######################################################################################


axis(side = 2,lwd = 0, line = -0.7, las = 2)
axis(side = 2, tck = -.015, labels = NA)
axis(side = 1, tck = -.015, labels = NA)
axis(side = 1,lwd = 0, line = -1.05)
mtext(side=1,"Day of year",line=0.8,cex = cex.lab)
mtext(side=2,expression(paste("Distance from phloem (",mu,"m)")),line=1,cex = cex.lab )
                
                
dev.off()





############FIGURE Temperature wall activation energy###########################

jpeg(filename=paste0(figs_path,'appdx_dT_Eaw.jpeg'),units="mm",width=90, height=90, res=res)
par(mfrow=c(1,1),oma=c(1,1,0.1,0.1),mar=c(1,1,0,0),cex.axis=0.7, cex.lab=0.7,ps=8)


# calculate temperatures on the fly from forcing data:
out_temp= read.table(paste0(file_path,'torn_clm_crujrav2.2.txt'))
names(out_temp) <- c("kyr_clm", "kt", "Temp" )

jja = list() #mean jja temperature for each year (degC)
nyr = 2004 - 1901 + 1
nmo = 12
ndays = c(31,28,31,30,31,30,31,31,30,31,30,31)
m = 1
sum = 0
for (i in 1:nyr){ # for each year
  for (j in 1:nmo){ # for each month
    for (k in 1:ndays[j]){ # for days in the month
      for (l in 1:4){ #each day has four 6 hour timesteps, so loop over them, too
        # if the month is june july august, sum up temperatures
        if ( (j == 6) | (j == 7)| (j == 8)){
          sum = sum + out_temp$Temp[m] - 273.15
        }
        m = m + 1
      }
      
    }
  }
  jja = append( jja,(sum/ (4*(ndays[6]+ndays[7]+ndays[8]) ) ) )
  sum = 0
}
  

#extract years of relevance for plotting jja
jja_obs = vector()

l_obs = nyr
ib = 1901
ie = 2004
j = ib
for (i in 1:l_obs){
  if (j <= ie){
    jja_obs= append(jja_obs,jja[i])
  }
  j = j + 1
}
  
### load up Torneträsk width and density observation data
#Source:
#Grudd, H. Torneträsk tree-ring width and density ad 500–2004: a test of climatic 
#sensitivity and a new 1500-year reconstruction of north Fennoscandian summers.
# Clim. Dyn. 31, 843–857, DOI:10.1007/s00382-007-0358-2 (2008).
#  ftp://ftp.ncdc.noaa.gov/pub/data/paleo/treering/reconstructions/
#  europe/sweden/tornetrask-temperature2008.txt

out2 = read.table(paste0(obs_path,"torn_tree_data.txt"))
names(out2) = c("Year", "MXDrcs", "MXDlo", "MXDlowh", "MXDlowt", 
                "TRWrcs", "TRWlow", "TRWlowb", "TRWlowt")
out2 <- out2[which(out2$Year >= 1901),]

obs = list()
pre =  list()

l_obs = length(out2$Year)
ib = 1901
ie = 2004
j = ib
for (i in  1:l_obs){
  if (out2$Year[i]==j & j <= ie){
    # rcs = regional curve standardisation  ( see Grudd 2008 above)
    #MXD = maximum density
   obs = append(obs, out2$MXDrcs[i] * 0.615)
  }
   
  j = j + 1
}
  
# add model output
out = read.table(paste0(file_path,"TRW_torn.dat"))
names(out) = c("kyr", "trw", "mxd")


l_pre = length(out$kyr)
ib = 1901
ie = 2004
j = ib
for (i in 1:l_pre){
  if (out$kyr[i] == j & j <= ie){
    pre = append(pre,out$mxd[i])
    j = j + 1
  }
}
  
plot(jja_obs,obs,col=makeTransparent('dark green',alpha=200),type='p',pch=16,
     ylim=c(0.4,0.8),ylab="",xlab="",yaxt="n",xaxt="n",cex=0.8)
lines(type='p',jja_obs,pre, col='grey',pch=16,cex=0.8)

mtext(side=1, 'Mean June-July-August temperature (degree C)',line=0.7,
      cex = cex.lab)
mtext(side=2, expression(paste("Maximum density (g cm"^"-3",")")),
      line=1.1,cex = cex.lab)
axis(side = 2,lwd = 0, line = -0.7, las = 2)
axis(side = 2, tck = -.015, labels = NA)
axis(side = 1, tck = -.015, labels = NA)
axis(side = 1,lwd = 0, line = -1.05)



##add regression lines
#prepare for lm function:
data = data.frame(jja_obs=as.vector(unlist(jja_obs)),obs=as.vector(unlist(obs))
                  ,pre=as.vector(unlist(pre)))
abline(lm(obs~jja_obs,data=data),col=makeTransparent('dark green',alpha=180))
abline(lm(pre~jja_obs,data=data),col='grey')

add_legend(-0.81,1.05, legend=c("Observed","Simulated"),
           bty = "n", col = c(makeTransparent('dark green',alpha=200),
                              "grey"),
           lty=c(1,1),cex=cex.legend)



dev.off()

######TRW-plot

# prepare observations:

obs = list()
pre =  list()

l_obs = length(out2$Year)
ib = 1901
ie = 2004
j = ib
for (i in  1:l_obs){
  if (out2$Year[i]==j & j <= ie){
    # rcs = regional curve standardisation ( see Grudd 2008 above)
    # TRW = tree ring width
    obs = append(obs, out2$TRWrcs[i])
  }
  
  j = j + 1
}


l_pre = length(out$kyr)
ib = 1901
ie = 2004
j = ib
for (i in 1:l_pre){
  if (out$kyr[i] == j & j <= ie){
    pre = append(pre,(out$trw[i]/1000 ))
    j = j + 1
  }
}

############FIGURE Temperature enlargement activation energy####################

jpeg(filename=paste0(figs_path,'appdx_dT_Ea.jpeg'),units="mm",width=90, height=90, res=res)
par(mfrow=c(1,1),oma=c(1,1,1,1),mar=c(1,1,1,1),cex.axis=0.7, cex.lab=0.7,ps=8)

plot(jja_obs,unlist(obs),col=makeTransparent('dark green',alpha=200),type='p',pch=16, 
     ylim=c(0.5,2.1),ylab="",xlab="",yaxt="n",xaxt="n",cex=0.8)
lines(type='p',jja_obs,pre, col='grey',pch=16,cex=0.8)

mtext(side=1, 'Mean June-July-August temperature (degree C)',line=0.7,
      cex = cex.lab)
mtext(side=2, "Ring width (mm)",line=1.1,cex = cex.lab)
axis(side = 2,lwd = 0, line = -0.7, las = 2)
axis(side = 2, tck = -.015, labels = NA)
axis(side = 1, tck = -.015, labels = NA)
axis(side = 1,lwd = 0, line = -1.05)
axis(side = 4,lwd = 0, line = -0.7, las = 2,col="dark green")
axis(side = 4, tck = -.015, labels = NA,col="dark green")
mtext(side=4, "Ring index (-)",line=0.8,cex = cex.lab,col="dark green")

##add regression lines
#prepare for lm function:
data = data.frame(jja_obs=as.vector(unlist(jja_obs)),obs=as.vector(unlist(obs)),
                  pre=as.vector(unlist(pre)))
abline(lm(obs~jja_obs,data=data),col=makeTransparent('dark green',alpha=180))
abline(lm(pre~jja_obs,data=data),col='grey')

add_legend(-0.81,0.90, legend=c("Observed","Simulated"),
           bty = "n", col = c(makeTransparent('dark green',alpha=200),
                              "grey"),
           lty=c(1,1),cex=cex.legend)



dev.off()





