#ifndef actual_rhs_H
#define actual_rhs_H

#include <AMReX_REAL.H>
#include <AMReX_Array.H>

#include <extern_parameters.H>
#include <actual_network.H>
#include <burn_type.H>
#include <jacobian_utilities.H>
#include <screen.H>
#include <sneut5.H>
#include <reaclib_rates.H>
#include <table_rates.H>
#include <extern_parameters.H>

using namespace amrex;
using namespace ArrayUtil;

using namespace Species;
using namespace Rates;

using namespace rate_tables;


template<class T>
AMREX_GPU_HOST_DEVICE AMREX_INLINE
void ener_gener_rate(T const& dydt, Real& enuc)
{

    // Computes the instantaneous energy generation rate (from the nuclei)

    // This is basically e = m c**2

    enuc = 0.0_rt;

    for (int n = 1; n <= NumSpec; ++n) {
        enuc += dydt(n) * network::mion(n);
    }

    enuc *= C::Legacy::enuc_conv2;
}


AMREX_GPU_HOST_DEVICE AMREX_INLINE
void evaluate_rates(const burn_t& state, rate_eval_t& rate_eval) {


    // create molar fractions

    Array1D<Real, 1, NumSpec> Y;
    for (int n = 1; n <= NumSpec; ++n) {
        Y(n) = state.xn[n-1] * aion_inv[n-1];
    }

    Real rhoy = state.rho * state.y_e;

    // Calculate Reaclib rates

    plasma_state_t pstate;
    fill_plasma_state(pstate, state.T, state.rho, Y);

    tf_t tfactors = evaluate_tfactors(state.T);

    fill_reaclib_rates(tfactors, rate_eval);



    // Evaluate screening factors

    Real ratraw, dratraw_dT;
    Real scor, dscor_dt, dscor_dd;
    Real scor2, dscor2_dt, dscor2_dd;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 1.0_rt, 1.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_p__d__weak__bet_pos_);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p__d__weak__bet_pos_);
    rate_eval.screened_rates(k_p_p__d__weak__bet_pos_) *= scor;
    rate_eval.dscreened_rates_dT(k_p_p__d__weak__bet_pos_) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_p__d__weak__electron_capture);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p__d__weak__electron_capture);
    rate_eval.screened_rates(k_p_p__d__weak__electron_capture) *= scor;
    rate_eval.dscreened_rates_dT(k_p_p__d__weak__electron_capture) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_n_p_p__p_d);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_n_p_p__p_d);
    rate_eval.screened_rates(k_n_p_p__p_d) *= scor;
    rate_eval.dscreened_rates_dT(k_n_p_p__p_d) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_p_he4_he4__he3_be7);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p_he4_he4__he3_be7);
    rate_eval.screened_rates(k_p_p_he4_he4__he3_be7) *= scor;
    rate_eval.dscreened_rates_dT(k_p_p_he4_he4__he3_be7) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 1.0_rt, 2.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_d__he3);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_d__he3);
    rate_eval.screened_rates(k_p_d__he3) *= scor;
    rate_eval.dscreened_rates_dT(k_p_d__he3) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_d__n_p_p);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_d__n_p_p);
    rate_eval.screened_rates(k_p_d__n_p_p) *= scor;
    rate_eval.dscreened_rates_dT(k_p_d__n_p_p) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 1.0_rt, 2.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_d_d__he4);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_d_d__he4);
    rate_eval.screened_rates(k_d_d__he4) *= scor;
    rate_eval.dscreened_rates_dT(k_d_d__he4) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_d_d__n_he3);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_d_d__n_he3);
    rate_eval.screened_rates(k_d_d__n_he3) *= scor;
    rate_eval.dscreened_rates_dT(k_d_d__n_he3) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 2.0_rt, 4.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_d__li6);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_d__li6);
    rate_eval.screened_rates(k_he4_d__li6) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_d__li6) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 2.0_rt, 3.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_he3__he4__weak__bet_pos_);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_he3__he4__weak__bet_pos_);
    rate_eval.screened_rates(k_p_he3__he4__weak__bet_pos_) *= scor;
    rate_eval.dscreened_rates_dT(k_p_he3__he4__weak__bet_pos_) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 2.0_rt, 3.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_he3__be7);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_he3__be7);
    rate_eval.screened_rates(k_he4_he3__be7) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_he3__be7) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_he3__p_li6);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_he3__p_li6);
    rate_eval.screened_rates(k_he4_he3__p_li6) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_he3__p_li6) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 3.0_rt, 6.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_li6__be7);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_li6__be7);
    rate_eval.screened_rates(k_p_li6__be7) *= scor;
    rate_eval.dscreened_rates_dT(k_p_li6__be7) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_li6__he4_he3);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_li6__he4_he3);
    rate_eval.screened_rates(k_p_li6__he4_he3) *= scor;
    rate_eval.dscreened_rates_dT(k_p_li6__he4_he3) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 3.0_rt, 6.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_li6__b10);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_li6__b10);
    rate_eval.screened_rates(k_he4_li6__b10) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_li6__b10) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_li6__p_be9);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_li6__p_be9);
    rate_eval.screened_rates(k_he4_li6__p_be9) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_li6__p_be9) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 3.0_rt, 7.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_li7__b11);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_li7__b11);
    rate_eval.screened_rates(k_he4_li7__b11) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_li7__b11) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_li7__n_b10);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_li7__n_b10);
    rate_eval.screened_rates(k_he4_li7__n_b10) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_li7__n_b10) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 4.0_rt, 7.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_be7__b8);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_be7__b8);
    rate_eval.screened_rates(k_p_be7__b8) *= scor;
    rate_eval.dscreened_rates_dT(k_p_be7__b8) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 4.0_rt, 9.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_be9__b10);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_be9__b10);
    rate_eval.screened_rates(k_p_be9__b10) *= scor;
    rate_eval.dscreened_rates_dT(k_p_be9__b10) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_be9__he4_li6);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_be9__he4_li6);
    rate_eval.screened_rates(k_p_be9__he4_li6) *= scor;
    rate_eval.dscreened_rates_dT(k_p_be9__he4_li6) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_be9__d_he4_he4);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_be9__d_he4_he4);
    rate_eval.screened_rates(k_p_be9__d_he4_he4) *= scor;
    rate_eval.dscreened_rates_dT(k_p_be9__d_he4_he4) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_be9__n_p_he4_he4);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_be9__n_p_he4_he4);
    rate_eval.screened_rates(k_p_be9__n_p_he4_he4) *= scor;
    rate_eval.dscreened_rates_dT(k_p_be9__n_p_he4_he4) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 5.0_rt, 11.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_b11__c12);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_b11__c12);
    rate_eval.screened_rates(k_p_b11__c12) *= scor;
    rate_eval.dscreened_rates_dT(k_p_b11__c12) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_b11__he4_he4_he4);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_b11__he4_he4_he4);
    rate_eval.screened_rates(k_p_b11__he4_he4_he4) *= scor;
    rate_eval.dscreened_rates_dT(k_p_b11__he4_he4_he4) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 6.0_rt, 12.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_c12__n13);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_c12__n13);
    rate_eval.screened_rates(k_p_c12__n13) *= scor;
    rate_eval.dscreened_rates_dT(k_p_c12__n13) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 6.0_rt, 12.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_c12__o16);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_c12__o16);
    rate_eval.screened_rates(k_he4_c12__o16) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_c12__o16) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_c12__n_o15);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_c12__n_o15);
    rate_eval.screened_rates(k_he4_c12__n_o15) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_c12__n_o15) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_c12__p_n15);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_c12__p_n15);
    rate_eval.screened_rates(k_he4_c12__p_n15) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_c12__p_n15) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 6.0_rt, 13.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_c13__n14);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_c13__n14);
    rate_eval.screened_rates(k_p_c13__n14) *= scor;
    rate_eval.dscreened_rates_dT(k_p_c13__n14) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_c13__n_n13);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_c13__n_n13);
    rate_eval.screened_rates(k_p_c13__n_n13) *= scor;
    rate_eval.dscreened_rates_dT(k_p_c13__n_n13) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_c13__he4_b10);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_c13__he4_b10);
    rate_eval.screened_rates(k_p_c13__he4_b10) *= scor;
    rate_eval.dscreened_rates_dT(k_p_c13__he4_b10) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 6.0_rt, 14.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_c14__n15);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_c14__n15);
    rate_eval.screened_rates(k_p_c14__n15) *= scor;
    rate_eval.dscreened_rates_dT(k_p_c14__n15) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_c14__n_n14);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_c14__n_n14);
    rate_eval.screened_rates(k_p_c14__n_n14) *= scor;
    rate_eval.dscreened_rates_dT(k_p_c14__n_n14) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_c14__he4_b11);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_c14__he4_b11);
    rate_eval.screened_rates(k_p_c14__he4_b11) *= scor;
    rate_eval.dscreened_rates_dT(k_p_c14__he4_b11) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 6.0_rt, 14.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_c14__o18);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_c14__o18);
    rate_eval.screened_rates(k_he4_c14__o18) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_c14__o18) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_c14__n_o17);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_c14__n_o17);
    rate_eval.screened_rates(k_he4_c14__n_o17) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_c14__n_o17) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 7.0_rt, 13.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_n13__o14);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_n13__o14);
    rate_eval.screened_rates(k_p_n13__o14) *= scor;
    rate_eval.dscreened_rates_dT(k_p_n13__o14) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 7.0_rt, 14.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_n14__o15);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_n14__o15);
    rate_eval.screened_rates(k_p_n14__o15) *= scor;
    rate_eval.dscreened_rates_dT(k_p_n14__o15) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_n14__n_o14);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_n14__n_o14);
    rate_eval.screened_rates(k_p_n14__n_o14) *= scor;
    rate_eval.dscreened_rates_dT(k_p_n14__n_o14) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 7.0_rt, 14.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_n14__f18);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_n14__f18);
    rate_eval.screened_rates(k_he4_n14__f18) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_n14__f18) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_n14__n_f17);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_n14__n_f17);
    rate_eval.screened_rates(k_he4_n14__n_f17) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_n14__n_f17) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_n14__p_o17);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_n14__p_o17);
    rate_eval.screened_rates(k_he4_n14__p_o17) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_n14__p_o17) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 7.0_rt, 15.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_n15__o16);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_n15__o16);
    rate_eval.screened_rates(k_p_n15__o16) *= scor;
    rate_eval.dscreened_rates_dT(k_p_n15__o16) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_n15__n_o15);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_n15__n_o15);
    rate_eval.screened_rates(k_p_n15__n_o15) *= scor;
    rate_eval.dscreened_rates_dT(k_p_n15__n_o15) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_n15__he4_c12);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_n15__he4_c12);
    rate_eval.screened_rates(k_p_n15__he4_c12) *= scor;
    rate_eval.dscreened_rates_dT(k_p_n15__he4_c12) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 7.0_rt, 15.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_n15__f19);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_n15__f19);
    rate_eval.screened_rates(k_he4_n15__f19) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_n15__f19) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_n15__n_f18);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_n15__n_f18);
    rate_eval.screened_rates(k_he4_n15__n_f18) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_n15__n_f18) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_n15__p_o18);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_n15__p_o18);
    rate_eval.screened_rates(k_he4_n15__p_o18) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_n15__p_o18) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 8.0_rt, 14.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_o14__ne18);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_o14__ne18);
    rate_eval.screened_rates(k_he4_o14__ne18) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_o14__ne18) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_o14__p_f17);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_o14__p_f17);
    rate_eval.screened_rates(k_he4_o14__p_f17) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_o14__p_f17) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 8.0_rt, 15.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_o15__ne19);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_o15__ne19);
    rate_eval.screened_rates(k_he4_o15__ne19) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_o15__ne19) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_o15__n_ne18);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_o15__n_ne18);
    rate_eval.screened_rates(k_he4_o15__n_ne18) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_o15__n_ne18) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_o15__p_f18);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_o15__p_f18);
    rate_eval.screened_rates(k_he4_o15__p_f18) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_o15__p_f18) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 8.0_rt, 16.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_o16__f17);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_o16__f17);
    rate_eval.screened_rates(k_p_o16__f17) *= scor;
    rate_eval.dscreened_rates_dT(k_p_o16__f17) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_o16__he4_n13);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_o16__he4_n13);
    rate_eval.screened_rates(k_p_o16__he4_n13) *= scor;
    rate_eval.dscreened_rates_dT(k_p_o16__he4_n13) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 8.0_rt, 16.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_o16__ne20);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_o16__ne20);
    rate_eval.screened_rates(k_he4_o16__ne20) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_o16__ne20) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_o16__n_ne19);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_o16__n_ne19);
    rate_eval.screened_rates(k_he4_o16__n_ne19) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_o16__n_ne19) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_o16__p_f19);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_o16__p_f19);
    rate_eval.screened_rates(k_he4_o16__p_f19) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_o16__p_f19) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 8.0_rt, 17.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_o17__f18);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_o17__f18);
    rate_eval.screened_rates(k_p_o17__f18) *= scor;
    rate_eval.dscreened_rates_dT(k_p_o17__f18) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_o17__n_f17);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_o17__n_f17);
    rate_eval.screened_rates(k_p_o17__n_f17) *= scor;
    rate_eval.dscreened_rates_dT(k_p_o17__n_f17) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_o17__he4_n14);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_o17__he4_n14);
    rate_eval.screened_rates(k_p_o17__he4_n14) *= scor;
    rate_eval.dscreened_rates_dT(k_p_o17__he4_n14) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 8.0_rt, 17.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_o17__ne21);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_o17__ne21);
    rate_eval.screened_rates(k_he4_o17__ne21) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_o17__ne21) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_o17__n_ne20);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_o17__n_ne20);
    rate_eval.screened_rates(k_he4_o17__n_ne20) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_o17__n_ne20) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 8.0_rt, 18.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_o18__f19);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_o18__f19);
    rate_eval.screened_rates(k_p_o18__f19) *= scor;
    rate_eval.dscreened_rates_dT(k_p_o18__f19) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_o18__n_f18);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_o18__n_f18);
    rate_eval.screened_rates(k_p_o18__n_f18) *= scor;
    rate_eval.dscreened_rates_dT(k_p_o18__n_f18) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_o18__he4_n15);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_o18__he4_n15);
    rate_eval.screened_rates(k_p_o18__he4_n15) *= scor;
    rate_eval.dscreened_rates_dT(k_p_o18__he4_n15) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 8.0_rt, 18.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_o18__ne22);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_o18__ne22);
    rate_eval.screened_rates(k_he4_o18__ne22) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_o18__ne22) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_o18__n_ne21);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_o18__n_ne21);
    rate_eval.screened_rates(k_he4_o18__n_ne21) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_o18__n_ne21) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 9.0_rt, 17.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_f17__ne18);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_f17__ne18);
    rate_eval.screened_rates(k_p_f17__ne18) *= scor;
    rate_eval.dscreened_rates_dT(k_p_f17__ne18) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_f17__he4_o14);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_f17__he4_o14);
    rate_eval.screened_rates(k_p_f17__he4_o14) *= scor;
    rate_eval.dscreened_rates_dT(k_p_f17__he4_o14) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 9.0_rt, 17.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_f17__na21);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_f17__na21);
    rate_eval.screened_rates(k_he4_f17__na21) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_f17__na21) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_f17__p_ne20);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_f17__p_ne20);
    rate_eval.screened_rates(k_he4_f17__p_ne20) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_f17__p_ne20) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 9.0_rt, 18.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_f18__ne19);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_f18__ne19);
    rate_eval.screened_rates(k_p_f18__ne19) *= scor;
    rate_eval.dscreened_rates_dT(k_p_f18__ne19) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_f18__n_ne18);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_f18__n_ne18);
    rate_eval.screened_rates(k_p_f18__n_ne18) *= scor;
    rate_eval.dscreened_rates_dT(k_p_f18__n_ne18) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_f18__he4_o15);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_f18__he4_o15);
    rate_eval.screened_rates(k_p_f18__he4_o15) *= scor;
    rate_eval.dscreened_rates_dT(k_p_f18__he4_o15) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 9.0_rt, 18.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_f18__na22);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_f18__na22);
    rate_eval.screened_rates(k_he4_f18__na22) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_f18__na22) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_f18__n_na21);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_f18__n_na21);
    rate_eval.screened_rates(k_he4_f18__n_na21) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_f18__n_na21) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_f18__p_ne21);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_f18__p_ne21);
    rate_eval.screened_rates(k_he4_f18__p_ne21) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_f18__p_ne21) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 9.0_rt, 19.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_f19__ne20);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_f19__ne20);
    rate_eval.screened_rates(k_p_f19__ne20) *= scor;
    rate_eval.dscreened_rates_dT(k_p_f19__ne20) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_f19__n_ne19);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_f19__n_ne19);
    rate_eval.screened_rates(k_p_f19__n_ne19) *= scor;
    rate_eval.dscreened_rates_dT(k_p_f19__n_ne19) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_f19__he4_o16);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_f19__he4_o16);
    rate_eval.screened_rates(k_p_f19__he4_o16) *= scor;
    rate_eval.dscreened_rates_dT(k_p_f19__he4_o16) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 9.0_rt, 19.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_f19__na23);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_f19__na23);
    rate_eval.screened_rates(k_he4_f19__na23) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_f19__na23) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_f19__n_na22);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_f19__n_na22);
    rate_eval.screened_rates(k_he4_f19__n_na22) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_f19__n_na22) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_f19__p_ne22);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_f19__p_ne22);
    rate_eval.screened_rates(k_he4_f19__p_ne22) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_f19__p_ne22) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 10.0_rt, 19.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ne19__mg23);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ne19__mg23);
    rate_eval.screened_rates(k_he4_ne19__mg23) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ne19__mg23) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ne19__p_na22);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ne19__p_na22);
    rate_eval.screened_rates(k_he4_ne19__p_na22) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ne19__p_na22) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 10.0_rt, 20.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ne20__na21);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ne20__na21);
    rate_eval.screened_rates(k_p_ne20__na21) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ne20__na21) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ne20__he4_f17);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ne20__he4_f17);
    rate_eval.screened_rates(k_p_ne20__he4_f17) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ne20__he4_f17) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 10.0_rt, 20.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ne20__mg24);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ne20__mg24);
    rate_eval.screened_rates(k_he4_ne20__mg24) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ne20__mg24) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ne20__n_mg23);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ne20__n_mg23);
    rate_eval.screened_rates(k_he4_ne20__n_mg23) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ne20__n_mg23) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ne20__p_na23);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ne20__p_na23);
    rate_eval.screened_rates(k_he4_ne20__p_na23) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ne20__p_na23) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ne20__c12_c12);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ne20__c12_c12);
    rate_eval.screened_rates(k_he4_ne20__c12_c12) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ne20__c12_c12) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 10.0_rt, 21.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ne21__na22);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ne21__na22);
    rate_eval.screened_rates(k_p_ne21__na22) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ne21__na22) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ne21__n_na21);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ne21__n_na21);
    rate_eval.screened_rates(k_p_ne21__n_na21) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ne21__n_na21) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ne21__he4_f18);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ne21__he4_f18);
    rate_eval.screened_rates(k_p_ne21__he4_f18) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ne21__he4_f18) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 10.0_rt, 21.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ne21__mg25);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ne21__mg25);
    rate_eval.screened_rates(k_he4_ne21__mg25) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ne21__mg25) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ne21__n_mg24);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ne21__n_mg24);
    rate_eval.screened_rates(k_he4_ne21__n_mg24) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ne21__n_mg24) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 10.0_rt, 22.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ne22__na23);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ne22__na23);
    rate_eval.screened_rates(k_p_ne22__na23) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ne22__na23) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ne22__n_na22);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ne22__n_na22);
    rate_eval.screened_rates(k_p_ne22__n_na22) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ne22__n_na22) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ne22__he4_f19);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ne22__he4_f19);
    rate_eval.screened_rates(k_p_ne22__he4_f19) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ne22__he4_f19) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 10.0_rt, 22.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ne22__mg26);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ne22__mg26);
    rate_eval.screened_rates(k_he4_ne22__mg26) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ne22__mg26) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ne22__n_mg25);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ne22__n_mg25);
    rate_eval.screened_rates(k_he4_ne22__n_mg25) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ne22__n_mg25) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 11.0_rt, 21.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_na21__al25);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_na21__al25);
    rate_eval.screened_rates(k_he4_na21__al25) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_na21__al25) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_na21__p_mg24);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_na21__p_mg24);
    rate_eval.screened_rates(k_he4_na21__p_mg24) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_na21__p_mg24) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 11.0_rt, 22.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_na22__mg23);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_na22__mg23);
    rate_eval.screened_rates(k_p_na22__mg23) *= scor;
    rate_eval.dscreened_rates_dT(k_p_na22__mg23) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_na22__he4_ne19);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_na22__he4_ne19);
    rate_eval.screened_rates(k_p_na22__he4_ne19) *= scor;
    rate_eval.dscreened_rates_dT(k_p_na22__he4_ne19) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 11.0_rt, 22.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_na22__al26);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_na22__al26);
    rate_eval.screened_rates(k_he4_na22__al26) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_na22__al26) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_na22__n_al25);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_na22__n_al25);
    rate_eval.screened_rates(k_he4_na22__n_al25) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_na22__n_al25) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_na22__p_mg25);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_na22__p_mg25);
    rate_eval.screened_rates(k_he4_na22__p_mg25) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_na22__p_mg25) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 11.0_rt, 23.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_na23__mg24);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_na23__mg24);
    rate_eval.screened_rates(k_p_na23__mg24) *= scor;
    rate_eval.dscreened_rates_dT(k_p_na23__mg24) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_na23__n_mg23);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_na23__n_mg23);
    rate_eval.screened_rates(k_p_na23__n_mg23) *= scor;
    rate_eval.dscreened_rates_dT(k_p_na23__n_mg23) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_na23__he4_ne20);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_na23__he4_ne20);
    rate_eval.screened_rates(k_p_na23__he4_ne20) *= scor;
    rate_eval.dscreened_rates_dT(k_p_na23__he4_ne20) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_na23__c12_c12);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_na23__c12_c12);
    rate_eval.screened_rates(k_p_na23__c12_c12) *= scor;
    rate_eval.dscreened_rates_dT(k_p_na23__c12_c12) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 11.0_rt, 23.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_na23__al27);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_na23__al27);
    rate_eval.screened_rates(k_he4_na23__al27) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_na23__al27) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_na23__n_al26);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_na23__n_al26);
    rate_eval.screened_rates(k_he4_na23__n_al26) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_na23__n_al26) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_na23__p_mg26);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_na23__p_mg26);
    rate_eval.screened_rates(k_he4_na23__p_mg26) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_na23__p_mg26) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 12.0_rt, 24.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_mg24__al25);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mg24__al25);
    rate_eval.screened_rates(k_p_mg24__al25) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mg24__al25) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_mg24__he4_na21);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mg24__he4_na21);
    rate_eval.screened_rates(k_p_mg24__he4_na21) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mg24__he4_na21) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 12.0_rt, 24.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_mg24__si28);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mg24__si28);
    rate_eval.screened_rates(k_he4_mg24__si28) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mg24__si28) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_mg24__p_al27);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mg24__p_al27);
    rate_eval.screened_rates(k_he4_mg24__p_al27) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mg24__p_al27) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_mg24__c12_o16);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mg24__c12_o16);
    rate_eval.screened_rates(k_he4_mg24__c12_o16) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mg24__c12_o16) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 12.0_rt, 25.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_mg25__al26);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mg25__al26);
    rate_eval.screened_rates(k_p_mg25__al26) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mg25__al26) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_mg25__n_al25);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mg25__n_al25);
    rate_eval.screened_rates(k_p_mg25__n_al25) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mg25__n_al25) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_mg25__he4_na22);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mg25__he4_na22);
    rate_eval.screened_rates(k_p_mg25__he4_na22) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mg25__he4_na22) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 12.0_rt, 25.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_mg25__si29);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mg25__si29);
    rate_eval.screened_rates(k_he4_mg25__si29) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mg25__si29) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_mg25__n_si28);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mg25__n_si28);
    rate_eval.screened_rates(k_he4_mg25__n_si28) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mg25__n_si28) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 12.0_rt, 26.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_mg26__al27);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mg26__al27);
    rate_eval.screened_rates(k_p_mg26__al27) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mg26__al27) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_mg26__n_al26);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mg26__n_al26);
    rate_eval.screened_rates(k_p_mg26__n_al26) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mg26__n_al26) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_mg26__he4_na23);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mg26__he4_na23);
    rate_eval.screened_rates(k_p_mg26__he4_na23) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mg26__he4_na23) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 12.0_rt, 26.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_mg26__si30);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mg26__si30);
    rate_eval.screened_rates(k_he4_mg26__si30) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mg26__si30) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_mg26__n_si29);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mg26__n_si29);
    rate_eval.screened_rates(k_he4_mg26__n_si29) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mg26__n_si29) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 13.0_rt, 25.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_al25__p29);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_al25__p29);
    rate_eval.screened_rates(k_he4_al25__p29) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_al25__p29) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_al25__p_si28);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_al25__p_si28);
    rate_eval.screened_rates(k_he4_al25__p_si28) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_al25__p_si28) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 13.0_rt, 26.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_al26__p30);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_al26__p30);
    rate_eval.screened_rates(k_he4_al26__p30) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_al26__p30) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_al26__n_p29);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_al26__n_p29);
    rate_eval.screened_rates(k_he4_al26__n_p29) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_al26__n_p29) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_al26__p_si29);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_al26__p_si29);
    rate_eval.screened_rates(k_he4_al26__p_si29) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_al26__p_si29) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 13.0_rt, 27.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_al27__si28);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_al27__si28);
    rate_eval.screened_rates(k_p_al27__si28) *= scor;
    rate_eval.dscreened_rates_dT(k_p_al27__si28) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_al27__he4_mg24);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_al27__he4_mg24);
    rate_eval.screened_rates(k_p_al27__he4_mg24) *= scor;
    rate_eval.dscreened_rates_dT(k_p_al27__he4_mg24) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_al27__c12_o16);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_al27__c12_o16);
    rate_eval.screened_rates(k_p_al27__c12_o16) *= scor;
    rate_eval.dscreened_rates_dT(k_p_al27__c12_o16) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 13.0_rt, 27.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_al27__p31);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_al27__p31);
    rate_eval.screened_rates(k_he4_al27__p31) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_al27__p31) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_al27__n_p30);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_al27__n_p30);
    rate_eval.screened_rates(k_he4_al27__n_p30) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_al27__n_p30) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_al27__p_si30);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_al27__p_si30);
    rate_eval.screened_rates(k_he4_al27__p_si30) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_al27__p_si30) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 14.0_rt, 28.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_si28__p29);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_si28__p29);
    rate_eval.screened_rates(k_p_si28__p29) *= scor;
    rate_eval.dscreened_rates_dT(k_p_si28__p29) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_si28__he4_al25);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_si28__he4_al25);
    rate_eval.screened_rates(k_p_si28__he4_al25) *= scor;
    rate_eval.dscreened_rates_dT(k_p_si28__he4_al25) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 14.0_rt, 28.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_si28__s32);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_si28__s32);
    rate_eval.screened_rates(k_he4_si28__s32) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_si28__s32) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_si28__p_p31);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_si28__p_p31);
    rate_eval.screened_rates(k_he4_si28__p_p31) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_si28__p_p31) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_si28__c12_ne20);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_si28__c12_ne20);
    rate_eval.screened_rates(k_he4_si28__c12_ne20) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_si28__c12_ne20) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_si28__o16_o16);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_si28__o16_o16);
    rate_eval.screened_rates(k_he4_si28__o16_o16) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_si28__o16_o16) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 14.0_rt, 29.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_si29__p30);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_si29__p30);
    rate_eval.screened_rates(k_p_si29__p30) *= scor;
    rate_eval.dscreened_rates_dT(k_p_si29__p30) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_si29__n_p29);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_si29__n_p29);
    rate_eval.screened_rates(k_p_si29__n_p29) *= scor;
    rate_eval.dscreened_rates_dT(k_p_si29__n_p29) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_si29__he4_al26);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_si29__he4_al26);
    rate_eval.screened_rates(k_p_si29__he4_al26) *= scor;
    rate_eval.dscreened_rates_dT(k_p_si29__he4_al26) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 14.0_rt, 29.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_si29__s33);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_si29__s33);
    rate_eval.screened_rates(k_he4_si29__s33) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_si29__s33) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_si29__n_s32);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_si29__n_s32);
    rate_eval.screened_rates(k_he4_si29__n_s32) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_si29__n_s32) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_si29__p_p32);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_si29__p_p32);
    rate_eval.screened_rates(k_he4_si29__p_p32) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_si29__p_p32) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 14.0_rt, 30.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_si30__p31);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_si30__p31);
    rate_eval.screened_rates(k_p_si30__p31) *= scor;
    rate_eval.dscreened_rates_dT(k_p_si30__p31) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_si30__n_p30);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_si30__n_p30);
    rate_eval.screened_rates(k_p_si30__n_p30) *= scor;
    rate_eval.dscreened_rates_dT(k_p_si30__n_p30) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_si30__he4_al27);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_si30__he4_al27);
    rate_eval.screened_rates(k_p_si30__he4_al27) *= scor;
    rate_eval.dscreened_rates_dT(k_p_si30__he4_al27) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 14.0_rt, 30.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_si30__s34);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_si30__s34);
    rate_eval.screened_rates(k_he4_si30__s34) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_si30__s34) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_si30__n_s33);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_si30__n_s33);
    rate_eval.screened_rates(k_he4_si30__n_s33) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_si30__n_s33) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_si30__p_p33);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_si30__p_p33);
    rate_eval.screened_rates(k_he4_si30__p_p33) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_si30__p_p33) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 14.0_rt, 31.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_si31__p32);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_si31__p32);
    rate_eval.screened_rates(k_p_si31__p32) *= scor;
    rate_eval.dscreened_rates_dT(k_p_si31__p32) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_si31__n_p31);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_si31__n_p31);
    rate_eval.screened_rates(k_p_si31__n_p31) *= scor;
    rate_eval.dscreened_rates_dT(k_p_si31__n_p31) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 14.0_rt, 31.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_si31__s35);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_si31__s35);
    rate_eval.screened_rates(k_he4_si31__s35) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_si31__s35) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_si31__n_s34);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_si31__n_s34);
    rate_eval.screened_rates(k_he4_si31__n_s34) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_si31__n_s34) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 14.0_rt, 32.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_si32__p33);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_si32__p33);
    rate_eval.screened_rates(k_p_si32__p33) *= scor;
    rate_eval.dscreened_rates_dT(k_p_si32__p33) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_si32__n_p32);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_si32__n_p32);
    rate_eval.screened_rates(k_p_si32__n_p32) *= scor;
    rate_eval.dscreened_rates_dT(k_p_si32__n_p32) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 14.0_rt, 32.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_si32__s36);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_si32__s36);
    rate_eval.screened_rates(k_he4_si32__s36) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_si32__s36) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_si32__n_s35);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_si32__n_s35);
    rate_eval.screened_rates(k_he4_si32__n_s35) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_si32__n_s35) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 15.0_rt, 29.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_p29__cl33);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_p29__cl33);
    rate_eval.screened_rates(k_he4_p29__cl33) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_p29__cl33) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_p29__p_s32);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_p29__p_s32);
    rate_eval.screened_rates(k_he4_p29__p_s32) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_p29__p_s32) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 15.0_rt, 30.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_p30__cl34);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_p30__cl34);
    rate_eval.screened_rates(k_he4_p30__cl34) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_p30__cl34) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_p30__n_cl33);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_p30__n_cl33);
    rate_eval.screened_rates(k_he4_p30__n_cl33) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_p30__n_cl33) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_p30__p_s33);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_p30__p_s33);
    rate_eval.screened_rates(k_he4_p30__p_s33) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_p30__p_s33) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 15.0_rt, 31.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_p31__s32);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p31__s32);
    rate_eval.screened_rates(k_p_p31__s32) *= scor;
    rate_eval.dscreened_rates_dT(k_p_p31__s32) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_p31__he4_si28);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p31__he4_si28);
    rate_eval.screened_rates(k_p_p31__he4_si28) *= scor;
    rate_eval.dscreened_rates_dT(k_p_p31__he4_si28) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_p31__c12_ne20);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p31__c12_ne20);
    rate_eval.screened_rates(k_p_p31__c12_ne20) *= scor;
    rate_eval.dscreened_rates_dT(k_p_p31__c12_ne20) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_p31__o16_o16);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p31__o16_o16);
    rate_eval.screened_rates(k_p_p31__o16_o16) *= scor;
    rate_eval.dscreened_rates_dT(k_p_p31__o16_o16) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 15.0_rt, 31.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_p31__cl35);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_p31__cl35);
    rate_eval.screened_rates(k_he4_p31__cl35) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_p31__cl35) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_p31__n_cl34);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_p31__n_cl34);
    rate_eval.screened_rates(k_he4_p31__n_cl34) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_p31__n_cl34) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_p31__p_s34);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_p31__p_s34);
    rate_eval.screened_rates(k_he4_p31__p_s34) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_p31__p_s34) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 15.0_rt, 32.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_p32__s33);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p32__s33);
    rate_eval.screened_rates(k_p_p32__s33) *= scor;
    rate_eval.dscreened_rates_dT(k_p_p32__s33) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_p32__n_s32);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p32__n_s32);
    rate_eval.screened_rates(k_p_p32__n_s32) *= scor;
    rate_eval.dscreened_rates_dT(k_p_p32__n_s32) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_p32__he4_si29);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p32__he4_si29);
    rate_eval.screened_rates(k_p_p32__he4_si29) *= scor;
    rate_eval.dscreened_rates_dT(k_p_p32__he4_si29) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 15.0_rt, 32.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_p32__cl36);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_p32__cl36);
    rate_eval.screened_rates(k_he4_p32__cl36) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_p32__cl36) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_p32__n_cl35);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_p32__n_cl35);
    rate_eval.screened_rates(k_he4_p32__n_cl35) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_p32__n_cl35) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_p32__p_s35);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_p32__p_s35);
    rate_eval.screened_rates(k_he4_p32__p_s35) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_p32__p_s35) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 15.0_rt, 33.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_p33__s34);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p33__s34);
    rate_eval.screened_rates(k_p_p33__s34) *= scor;
    rate_eval.dscreened_rates_dT(k_p_p33__s34) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_p33__n_s33);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p33__n_s33);
    rate_eval.screened_rates(k_p_p33__n_s33) *= scor;
    rate_eval.dscreened_rates_dT(k_p_p33__n_s33) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_p33__he4_si30);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p33__he4_si30);
    rate_eval.screened_rates(k_p_p33__he4_si30) *= scor;
    rate_eval.dscreened_rates_dT(k_p_p33__he4_si30) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 15.0_rt, 33.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_p33__cl37);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_p33__cl37);
    rate_eval.screened_rates(k_he4_p33__cl37) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_p33__cl37) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_p33__n_cl36);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_p33__n_cl36);
    rate_eval.screened_rates(k_he4_p33__n_cl36) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_p33__n_cl36) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_p33__p_s36);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_p33__p_s36);
    rate_eval.screened_rates(k_he4_p33__p_s36) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_p33__p_s36) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 16.0_rt, 32.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_s32__cl33);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_s32__cl33);
    rate_eval.screened_rates(k_p_s32__cl33) *= scor;
    rate_eval.dscreened_rates_dT(k_p_s32__cl33) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_s32__he4_p29);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_s32__he4_p29);
    rate_eval.screened_rates(k_p_s32__he4_p29) *= scor;
    rate_eval.dscreened_rates_dT(k_p_s32__he4_p29) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 16.0_rt, 32.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_s32__ar36);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_s32__ar36);
    rate_eval.screened_rates(k_he4_s32__ar36) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_s32__ar36) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_s32__p_cl35);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_s32__p_cl35);
    rate_eval.screened_rates(k_he4_s32__p_cl35) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_s32__p_cl35) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 16.0_rt, 33.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_s33__cl34);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_s33__cl34);
    rate_eval.screened_rates(k_p_s33__cl34) *= scor;
    rate_eval.dscreened_rates_dT(k_p_s33__cl34) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_s33__n_cl33);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_s33__n_cl33);
    rate_eval.screened_rates(k_p_s33__n_cl33) *= scor;
    rate_eval.dscreened_rates_dT(k_p_s33__n_cl33) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_s33__he4_p30);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_s33__he4_p30);
    rate_eval.screened_rates(k_p_s33__he4_p30) *= scor;
    rate_eval.dscreened_rates_dT(k_p_s33__he4_p30) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 16.0_rt, 33.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_s33__ar37);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_s33__ar37);
    rate_eval.screened_rates(k_he4_s33__ar37) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_s33__ar37) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_s33__n_ar36);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_s33__n_ar36);
    rate_eval.screened_rates(k_he4_s33__n_ar36) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_s33__n_ar36) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_s33__p_cl36);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_s33__p_cl36);
    rate_eval.screened_rates(k_he4_s33__p_cl36) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_s33__p_cl36) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 16.0_rt, 34.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_s34__cl35);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_s34__cl35);
    rate_eval.screened_rates(k_p_s34__cl35) *= scor;
    rate_eval.dscreened_rates_dT(k_p_s34__cl35) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_s34__n_cl34);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_s34__n_cl34);
    rate_eval.screened_rates(k_p_s34__n_cl34) *= scor;
    rate_eval.dscreened_rates_dT(k_p_s34__n_cl34) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_s34__he4_p31);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_s34__he4_p31);
    rate_eval.screened_rates(k_p_s34__he4_p31) *= scor;
    rate_eval.dscreened_rates_dT(k_p_s34__he4_p31) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 16.0_rt, 34.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_s34__ar38);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_s34__ar38);
    rate_eval.screened_rates(k_he4_s34__ar38) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_s34__ar38) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_s34__n_ar37);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_s34__n_ar37);
    rate_eval.screened_rates(k_he4_s34__n_ar37) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_s34__n_ar37) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_s34__p_cl37);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_s34__p_cl37);
    rate_eval.screened_rates(k_he4_s34__p_cl37) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_s34__p_cl37) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 16.0_rt, 35.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_s35__cl36);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_s35__cl36);
    rate_eval.screened_rates(k_p_s35__cl36) *= scor;
    rate_eval.dscreened_rates_dT(k_p_s35__cl36) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_s35__n_cl35);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_s35__n_cl35);
    rate_eval.screened_rates(k_p_s35__n_cl35) *= scor;
    rate_eval.dscreened_rates_dT(k_p_s35__n_cl35) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_s35__he4_p32);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_s35__he4_p32);
    rate_eval.screened_rates(k_p_s35__he4_p32) *= scor;
    rate_eval.dscreened_rates_dT(k_p_s35__he4_p32) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 16.0_rt, 35.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_s35__ar39);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_s35__ar39);
    rate_eval.screened_rates(k_he4_s35__ar39) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_s35__ar39) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_s35__n_ar38);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_s35__n_ar38);
    rate_eval.screened_rates(k_he4_s35__n_ar38) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_s35__n_ar38) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 16.0_rt, 36.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_s36__cl37);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_s36__cl37);
    rate_eval.screened_rates(k_p_s36__cl37) *= scor;
    rate_eval.dscreened_rates_dT(k_p_s36__cl37) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_s36__n_cl36);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_s36__n_cl36);
    rate_eval.screened_rates(k_p_s36__n_cl36) *= scor;
    rate_eval.dscreened_rates_dT(k_p_s36__n_cl36) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_s36__he4_p33);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_s36__he4_p33);
    rate_eval.screened_rates(k_p_s36__he4_p33) *= scor;
    rate_eval.dscreened_rates_dT(k_p_s36__he4_p33) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 16.0_rt, 36.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_s36__ar40);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_s36__ar40);
    rate_eval.screened_rates(k_he4_s36__ar40) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_s36__ar40) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_s36__n_ar39);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_s36__n_ar39);
    rate_eval.screened_rates(k_he4_s36__n_ar39) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_s36__n_ar39) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 17.0_rt, 33.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cl33__k37);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cl33__k37);
    rate_eval.screened_rates(k_he4_cl33__k37) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cl33__k37) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cl33__p_ar36);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cl33__p_ar36);
    rate_eval.screened_rates(k_he4_cl33__p_ar36) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cl33__p_ar36) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 17.0_rt, 34.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cl34__k38);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cl34__k38);
    rate_eval.screened_rates(k_he4_cl34__k38) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cl34__k38) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cl34__n_k37);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cl34__n_k37);
    rate_eval.screened_rates(k_he4_cl34__n_k37) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cl34__n_k37) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cl34__p_ar37);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cl34__p_ar37);
    rate_eval.screened_rates(k_he4_cl34__p_ar37) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cl34__p_ar37) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 17.0_rt, 35.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cl35__ar36);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cl35__ar36);
    rate_eval.screened_rates(k_p_cl35__ar36) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cl35__ar36) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cl35__he4_s32);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cl35__he4_s32);
    rate_eval.screened_rates(k_p_cl35__he4_s32) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cl35__he4_s32) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 17.0_rt, 35.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cl35__k39);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cl35__k39);
    rate_eval.screened_rates(k_he4_cl35__k39) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cl35__k39) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cl35__n_k38);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cl35__n_k38);
    rate_eval.screened_rates(k_he4_cl35__n_k38) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cl35__n_k38) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cl35__p_ar38);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cl35__p_ar38);
    rate_eval.screened_rates(k_he4_cl35__p_ar38) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cl35__p_ar38) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 17.0_rt, 36.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cl36__ar37);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cl36__ar37);
    rate_eval.screened_rates(k_p_cl36__ar37) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cl36__ar37) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cl36__n_ar36);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cl36__n_ar36);
    rate_eval.screened_rates(k_p_cl36__n_ar36) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cl36__n_ar36) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cl36__he4_s33);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cl36__he4_s33);
    rate_eval.screened_rates(k_p_cl36__he4_s33) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cl36__he4_s33) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 17.0_rt, 36.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cl36__k40);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cl36__k40);
    rate_eval.screened_rates(k_he4_cl36__k40) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cl36__k40) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cl36__n_k39);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cl36__n_k39);
    rate_eval.screened_rates(k_he4_cl36__n_k39) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cl36__n_k39) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cl36__p_ar39);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cl36__p_ar39);
    rate_eval.screened_rates(k_he4_cl36__p_ar39) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cl36__p_ar39) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 17.0_rt, 37.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cl37__ar38);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cl37__ar38);
    rate_eval.screened_rates(k_p_cl37__ar38) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cl37__ar38) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cl37__n_ar37);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cl37__n_ar37);
    rate_eval.screened_rates(k_p_cl37__n_ar37) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cl37__n_ar37) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cl37__he4_s34);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cl37__he4_s34);
    rate_eval.screened_rates(k_p_cl37__he4_s34) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cl37__he4_s34) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 17.0_rt, 37.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cl37__k41);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cl37__k41);
    rate_eval.screened_rates(k_he4_cl37__k41) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cl37__k41) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cl37__n_k40);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cl37__n_k40);
    rate_eval.screened_rates(k_he4_cl37__n_k40) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cl37__n_k40) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cl37__p_ar40);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cl37__p_ar40);
    rate_eval.screened_rates(k_he4_cl37__p_ar40) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cl37__p_ar40) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 18.0_rt, 36.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ar36__k37);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ar36__k37);
    rate_eval.screened_rates(k_p_ar36__k37) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ar36__k37) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ar36__he4_cl33);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ar36__he4_cl33);
    rate_eval.screened_rates(k_p_ar36__he4_cl33) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ar36__he4_cl33) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 18.0_rt, 36.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ar36__ca40);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ar36__ca40);
    rate_eval.screened_rates(k_he4_ar36__ca40) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ar36__ca40) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ar36__p_k39);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ar36__p_k39);
    rate_eval.screened_rates(k_he4_ar36__p_k39) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ar36__p_k39) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 18.0_rt, 37.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ar37__k38);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ar37__k38);
    rate_eval.screened_rates(k_p_ar37__k38) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ar37__k38) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ar37__n_k37);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ar37__n_k37);
    rate_eval.screened_rates(k_p_ar37__n_k37) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ar37__n_k37) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ar37__he4_cl34);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ar37__he4_cl34);
    rate_eval.screened_rates(k_p_ar37__he4_cl34) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ar37__he4_cl34) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 18.0_rt, 37.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ar37__ca41);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ar37__ca41);
    rate_eval.screened_rates(k_he4_ar37__ca41) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ar37__ca41) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ar37__n_ca40);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ar37__n_ca40);
    rate_eval.screened_rates(k_he4_ar37__n_ca40) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ar37__n_ca40) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ar37__p_k40);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ar37__p_k40);
    rate_eval.screened_rates(k_he4_ar37__p_k40) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ar37__p_k40) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 18.0_rt, 38.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ar38__k39);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ar38__k39);
    rate_eval.screened_rates(k_p_ar38__k39) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ar38__k39) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ar38__n_k38);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ar38__n_k38);
    rate_eval.screened_rates(k_p_ar38__n_k38) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ar38__n_k38) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ar38__he4_cl35);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ar38__he4_cl35);
    rate_eval.screened_rates(k_p_ar38__he4_cl35) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ar38__he4_cl35) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 18.0_rt, 38.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ar38__ca42);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ar38__ca42);
    rate_eval.screened_rates(k_he4_ar38__ca42) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ar38__ca42) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ar38__n_ca41);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ar38__n_ca41);
    rate_eval.screened_rates(k_he4_ar38__n_ca41) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ar38__n_ca41) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ar38__p_k41);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ar38__p_k41);
    rate_eval.screened_rates(k_he4_ar38__p_k41) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ar38__p_k41) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 18.0_rt, 39.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ar39__k40);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ar39__k40);
    rate_eval.screened_rates(k_p_ar39__k40) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ar39__k40) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ar39__n_k39);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ar39__n_k39);
    rate_eval.screened_rates(k_p_ar39__n_k39) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ar39__n_k39) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ar39__he4_cl36);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ar39__he4_cl36);
    rate_eval.screened_rates(k_p_ar39__he4_cl36) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ar39__he4_cl36) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 18.0_rt, 39.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ar39__ca43);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ar39__ca43);
    rate_eval.screened_rates(k_he4_ar39__ca43) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ar39__ca43) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ar39__n_ca42);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ar39__n_ca42);
    rate_eval.screened_rates(k_he4_ar39__n_ca42) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ar39__n_ca42) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 18.0_rt, 40.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ar40__k41);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ar40__k41);
    rate_eval.screened_rates(k_p_ar40__k41) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ar40__k41) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ar40__n_k40);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ar40__n_k40);
    rate_eval.screened_rates(k_p_ar40__n_k40) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ar40__n_k40) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ar40__he4_cl37);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ar40__he4_cl37);
    rate_eval.screened_rates(k_p_ar40__he4_cl37) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ar40__he4_cl37) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 18.0_rt, 40.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ar40__ca44);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ar40__ca44);
    rate_eval.screened_rates(k_he4_ar40__ca44) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ar40__ca44) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ar40__n_ca43);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ar40__n_ca43);
    rate_eval.screened_rates(k_he4_ar40__n_ca43) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ar40__n_ca43) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 19.0_rt, 39.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_k39__ca40);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_k39__ca40);
    rate_eval.screened_rates(k_p_k39__ca40) *= scor;
    rate_eval.dscreened_rates_dT(k_p_k39__ca40) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_k39__he4_ar36);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_k39__he4_ar36);
    rate_eval.screened_rates(k_p_k39__he4_ar36) *= scor;
    rate_eval.dscreened_rates_dT(k_p_k39__he4_ar36) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 19.0_rt, 39.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_k39__sc43);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_k39__sc43);
    rate_eval.screened_rates(k_he4_k39__sc43) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_k39__sc43) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_k39__p_ca42);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_k39__p_ca42);
    rate_eval.screened_rates(k_he4_k39__p_ca42) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_k39__p_ca42) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 19.0_rt, 40.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_k40__ca41);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_k40__ca41);
    rate_eval.screened_rates(k_p_k40__ca41) *= scor;
    rate_eval.dscreened_rates_dT(k_p_k40__ca41) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_k40__n_ca40);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_k40__n_ca40);
    rate_eval.screened_rates(k_p_k40__n_ca40) *= scor;
    rate_eval.dscreened_rates_dT(k_p_k40__n_ca40) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_k40__he4_ar37);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_k40__he4_ar37);
    rate_eval.screened_rates(k_p_k40__he4_ar37) *= scor;
    rate_eval.dscreened_rates_dT(k_p_k40__he4_ar37) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 19.0_rt, 40.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_k40__sc44);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_k40__sc44);
    rate_eval.screened_rates(k_he4_k40__sc44) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_k40__sc44) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_k40__n_sc43);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_k40__n_sc43);
    rate_eval.screened_rates(k_he4_k40__n_sc43) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_k40__n_sc43) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_k40__p_ca43);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_k40__p_ca43);
    rate_eval.screened_rates(k_he4_k40__p_ca43) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_k40__p_ca43) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 19.0_rt, 41.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_k41__ca42);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_k41__ca42);
    rate_eval.screened_rates(k_p_k41__ca42) *= scor;
    rate_eval.dscreened_rates_dT(k_p_k41__ca42) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_k41__n_ca41);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_k41__n_ca41);
    rate_eval.screened_rates(k_p_k41__n_ca41) *= scor;
    rate_eval.dscreened_rates_dT(k_p_k41__n_ca41) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_k41__he4_ar38);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_k41__he4_ar38);
    rate_eval.screened_rates(k_p_k41__he4_ar38) *= scor;
    rate_eval.dscreened_rates_dT(k_p_k41__he4_ar38) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 19.0_rt, 41.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_k41__sc45);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_k41__sc45);
    rate_eval.screened_rates(k_he4_k41__sc45) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_k41__sc45) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_k41__n_sc44);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_k41__n_sc44);
    rate_eval.screened_rates(k_he4_k41__n_sc44) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_k41__n_sc44) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_k41__p_ca44);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_k41__p_ca44);
    rate_eval.screened_rates(k_he4_k41__p_ca44) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_k41__p_ca44) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 40.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ca40__ti44);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca40__ti44);
    rate_eval.screened_rates(k_he4_ca40__ti44) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca40__ti44) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ca40__p_sc43);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca40__p_sc43);
    rate_eval.screened_rates(k_he4_ca40__p_sc43) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca40__p_sc43) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 41.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ca41__ti45);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca41__ti45);
    rate_eval.screened_rates(k_he4_ca41__ti45) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca41__ti45) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ca41__n_ti44);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca41__n_ti44);
    rate_eval.screened_rates(k_he4_ca41__n_ti44) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca41__n_ti44) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ca41__p_sc44);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca41__p_sc44);
    rate_eval.screened_rates(k_he4_ca41__p_sc44) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca41__p_sc44) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 42.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ca42__sc43);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca42__sc43);
    rate_eval.screened_rates(k_p_ca42__sc43) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca42__sc43) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ca42__he4_k39);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca42__he4_k39);
    rate_eval.screened_rates(k_p_ca42__he4_k39) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca42__he4_k39) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 42.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ca42__ti46);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca42__ti46);
    rate_eval.screened_rates(k_he4_ca42__ti46) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca42__ti46) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ca42__n_ti45);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca42__n_ti45);
    rate_eval.screened_rates(k_he4_ca42__n_ti45) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca42__n_ti45) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ca42__p_sc45);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca42__p_sc45);
    rate_eval.screened_rates(k_he4_ca42__p_sc45) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca42__p_sc45) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 43.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ca43__sc44);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca43__sc44);
    rate_eval.screened_rates(k_p_ca43__sc44) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca43__sc44) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ca43__n_sc43);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca43__n_sc43);
    rate_eval.screened_rates(k_p_ca43__n_sc43) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca43__n_sc43) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ca43__he4_k40);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca43__he4_k40);
    rate_eval.screened_rates(k_p_ca43__he4_k40) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca43__he4_k40) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 43.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ca43__ti47);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca43__ti47);
    rate_eval.screened_rates(k_he4_ca43__ti47) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca43__ti47) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ca43__n_ti46);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca43__n_ti46);
    rate_eval.screened_rates(k_he4_ca43__n_ti46) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca43__n_ti46) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ca43__p_sc46);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca43__p_sc46);
    rate_eval.screened_rates(k_he4_ca43__p_sc46) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca43__p_sc46) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 44.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ca44__sc45);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca44__sc45);
    rate_eval.screened_rates(k_p_ca44__sc45) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca44__sc45) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ca44__n_sc44);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca44__n_sc44);
    rate_eval.screened_rates(k_p_ca44__n_sc44) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca44__n_sc44) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ca44__he4_k41);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca44__he4_k41);
    rate_eval.screened_rates(k_p_ca44__he4_k41) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca44__he4_k41) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 44.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ca44__ti48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca44__ti48);
    rate_eval.screened_rates(k_he4_ca44__ti48) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca44__ti48) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ca44__n_ti47);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca44__n_ti47);
    rate_eval.screened_rates(k_he4_ca44__n_ti47) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca44__n_ti47) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ca44__p_sc47);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca44__p_sc47);
    rate_eval.screened_rates(k_he4_ca44__p_sc47) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca44__p_sc47) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 45.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ca45__sc46);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca45__sc46);
    rate_eval.screened_rates(k_p_ca45__sc46) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca45__sc46) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ca45__n_sc45);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca45__n_sc45);
    rate_eval.screened_rates(k_p_ca45__n_sc45) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca45__n_sc45) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 45.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ca45__ti49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca45__ti49);
    rate_eval.screened_rates(k_he4_ca45__ti49) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca45__ti49) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ca45__n_ti48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca45__n_ti48);
    rate_eval.screened_rates(k_he4_ca45__n_ti48) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca45__n_ti48) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ca45__p_sc48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca45__p_sc48);
    rate_eval.screened_rates(k_he4_ca45__p_sc48) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca45__p_sc48) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 46.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ca46__sc47);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca46__sc47);
    rate_eval.screened_rates(k_p_ca46__sc47) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca46__sc47) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ca46__n_sc46);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca46__n_sc46);
    rate_eval.screened_rates(k_p_ca46__n_sc46) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca46__n_sc46) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 46.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ca46__ti50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca46__ti50);
    rate_eval.screened_rates(k_he4_ca46__ti50) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca46__ti50) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ca46__n_ti49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca46__n_ti49);
    rate_eval.screened_rates(k_he4_ca46__n_ti49) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca46__n_ti49) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ca46__p_sc49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca46__p_sc49);
    rate_eval.screened_rates(k_he4_ca46__p_sc49) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca46__p_sc49) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 47.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ca47__sc48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca47__sc48);
    rate_eval.screened_rates(k_p_ca47__sc48) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca47__sc48) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ca47__n_sc47);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca47__n_sc47);
    rate_eval.screened_rates(k_p_ca47__n_sc47) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca47__n_sc47) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 47.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ca47__ti51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca47__ti51);
    rate_eval.screened_rates(k_he4_ca47__ti51) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca47__ti51) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ca47__n_ti50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca47__n_ti50);
    rate_eval.screened_rates(k_he4_ca47__n_ti50) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca47__n_ti50) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ca48__sc49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca48__sc49);
    rate_eval.screened_rates(k_p_ca48__sc49) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca48__sc49) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ca48__n_sc48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca48__n_sc48);
    rate_eval.screened_rates(k_p_ca48__n_sc48) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca48__n_sc48) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 21.0_rt, 43.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_sc43__ti44);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc43__ti44);
    rate_eval.screened_rates(k_p_sc43__ti44) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc43__ti44) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_sc43__he4_ca40);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc43__he4_ca40);
    rate_eval.screened_rates(k_p_sc43__he4_ca40) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc43__he4_ca40) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 21.0_rt, 43.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_sc43__v47);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc43__v47);
    rate_eval.screened_rates(k_he4_sc43__v47) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc43__v47) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_sc43__n_v46);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc43__n_v46);
    rate_eval.screened_rates(k_he4_sc43__n_v46) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc43__n_v46) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_sc43__p_ti46);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc43__p_ti46);
    rate_eval.screened_rates(k_he4_sc43__p_ti46) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc43__p_ti46) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 21.0_rt, 44.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_sc44__ti45);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc44__ti45);
    rate_eval.screened_rates(k_p_sc44__ti45) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc44__ti45) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_sc44__n_ti44);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc44__n_ti44);
    rate_eval.screened_rates(k_p_sc44__n_ti44) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc44__n_ti44) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_sc44__he4_ca41);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc44__he4_ca41);
    rate_eval.screened_rates(k_p_sc44__he4_ca41) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc44__he4_ca41) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 21.0_rt, 44.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_sc44__v48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc44__v48);
    rate_eval.screened_rates(k_he4_sc44__v48) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc44__v48) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_sc44__n_v47);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc44__n_v47);
    rate_eval.screened_rates(k_he4_sc44__n_v47) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc44__n_v47) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_sc44__p_ti47);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc44__p_ti47);
    rate_eval.screened_rates(k_he4_sc44__p_ti47) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc44__p_ti47) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 21.0_rt, 45.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_sc45__ti46);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc45__ti46);
    rate_eval.screened_rates(k_p_sc45__ti46) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc45__ti46) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_sc45__n_ti45);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc45__n_ti45);
    rate_eval.screened_rates(k_p_sc45__n_ti45) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc45__n_ti45) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_sc45__he4_ca42);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc45__he4_ca42);
    rate_eval.screened_rates(k_p_sc45__he4_ca42) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc45__he4_ca42) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 21.0_rt, 45.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_sc45__v49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc45__v49);
    rate_eval.screened_rates(k_he4_sc45__v49) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc45__v49) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_sc45__n_v48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc45__n_v48);
    rate_eval.screened_rates(k_he4_sc45__n_v48) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc45__n_v48) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_sc45__p_ti48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc45__p_ti48);
    rate_eval.screened_rates(k_he4_sc45__p_ti48) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc45__p_ti48) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 21.0_rt, 46.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_sc46__ti47);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc46__ti47);
    rate_eval.screened_rates(k_p_sc46__ti47) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc46__ti47) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_sc46__n_ti46);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc46__n_ti46);
    rate_eval.screened_rates(k_p_sc46__n_ti46) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc46__n_ti46) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_sc46__he4_ca43);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc46__he4_ca43);
    rate_eval.screened_rates(k_p_sc46__he4_ca43) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc46__he4_ca43) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 21.0_rt, 46.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_sc46__v50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc46__v50);
    rate_eval.screened_rates(k_he4_sc46__v50) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc46__v50) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_sc46__n_v49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc46__n_v49);
    rate_eval.screened_rates(k_he4_sc46__n_v49) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc46__n_v49) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_sc46__p_ti49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc46__p_ti49);
    rate_eval.screened_rates(k_he4_sc46__p_ti49) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc46__p_ti49) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 21.0_rt, 47.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_sc47__ti48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc47__ti48);
    rate_eval.screened_rates(k_p_sc47__ti48) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc47__ti48) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_sc47__n_ti47);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc47__n_ti47);
    rate_eval.screened_rates(k_p_sc47__n_ti47) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc47__n_ti47) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_sc47__he4_ca44);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc47__he4_ca44);
    rate_eval.screened_rates(k_p_sc47__he4_ca44) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc47__he4_ca44) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 21.0_rt, 47.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_sc47__v51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc47__v51);
    rate_eval.screened_rates(k_he4_sc47__v51) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc47__v51) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_sc47__n_v50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc47__n_v50);
    rate_eval.screened_rates(k_he4_sc47__n_v50) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc47__n_v50) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_sc47__p_ti50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc47__p_ti50);
    rate_eval.screened_rates(k_he4_sc47__p_ti50) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc47__p_ti50) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 21.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_sc48__ti49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc48__ti49);
    rate_eval.screened_rates(k_p_sc48__ti49) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc48__ti49) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_sc48__n_ti48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc48__n_ti48);
    rate_eval.screened_rates(k_p_sc48__n_ti48) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc48__n_ti48) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_sc48__he4_ca45);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc48__he4_ca45);
    rate_eval.screened_rates(k_p_sc48__he4_ca45) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc48__he4_ca45) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 21.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_sc48__v52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc48__v52);
    rate_eval.screened_rates(k_he4_sc48__v52) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc48__v52) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_sc48__n_v51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc48__n_v51);
    rate_eval.screened_rates(k_he4_sc48__n_v51) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc48__n_v51) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_sc48__p_ti51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc48__p_ti51);
    rate_eval.screened_rates(k_he4_sc48__p_ti51) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc48__p_ti51) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 21.0_rt, 49.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_sc49__ti50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc49__ti50);
    rate_eval.screened_rates(k_p_sc49__ti50) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc49__ti50) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_sc49__n_ti49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc49__n_ti49);
    rate_eval.screened_rates(k_p_sc49__n_ti49) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc49__n_ti49) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_sc49__he4_ca46);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_sc49__he4_ca46);
    rate_eval.screened_rates(k_p_sc49__he4_ca46) *= scor;
    rate_eval.dscreened_rates_dT(k_p_sc49__he4_ca46) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 22.0_rt, 44.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ti44__cr48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti44__cr48);
    rate_eval.screened_rates(k_he4_ti44__cr48) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti44__cr48) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ti44__p_v47);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti44__p_v47);
    rate_eval.screened_rates(k_he4_ti44__p_v47) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti44__p_v47) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 22.0_rt, 45.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ti45__v46);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti45__v46);
    rate_eval.screened_rates(k_p_ti45__v46) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti45__v46) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 22.0_rt, 45.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ti45__cr49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti45__cr49);
    rate_eval.screened_rates(k_he4_ti45__cr49) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti45__cr49) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ti45__n_cr48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti45__n_cr48);
    rate_eval.screened_rates(k_he4_ti45__n_cr48) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti45__n_cr48) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ti45__p_v48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti45__p_v48);
    rate_eval.screened_rates(k_he4_ti45__p_v48) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti45__p_v48) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 22.0_rt, 46.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ti46__v47);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti46__v47);
    rate_eval.screened_rates(k_p_ti46__v47) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti46__v47) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ti46__n_v46);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti46__n_v46);
    rate_eval.screened_rates(k_p_ti46__n_v46) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti46__n_v46) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ti46__he4_sc43);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti46__he4_sc43);
    rate_eval.screened_rates(k_p_ti46__he4_sc43) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti46__he4_sc43) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 22.0_rt, 46.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ti46__cr50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti46__cr50);
    rate_eval.screened_rates(k_he4_ti46__cr50) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti46__cr50) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ti46__n_cr49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti46__n_cr49);
    rate_eval.screened_rates(k_he4_ti46__n_cr49) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti46__n_cr49) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ti46__p_v49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti46__p_v49);
    rate_eval.screened_rates(k_he4_ti46__p_v49) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti46__p_v49) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 22.0_rt, 47.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ti47__v48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti47__v48);
    rate_eval.screened_rates(k_p_ti47__v48) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti47__v48) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ti47__n_v47);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti47__n_v47);
    rate_eval.screened_rates(k_p_ti47__n_v47) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti47__n_v47) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ti47__he4_sc44);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti47__he4_sc44);
    rate_eval.screened_rates(k_p_ti47__he4_sc44) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti47__he4_sc44) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 22.0_rt, 47.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ti47__cr51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti47__cr51);
    rate_eval.screened_rates(k_he4_ti47__cr51) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti47__cr51) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ti47__n_cr50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti47__n_cr50);
    rate_eval.screened_rates(k_he4_ti47__n_cr50) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti47__n_cr50) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ti47__p_v50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti47__p_v50);
    rate_eval.screened_rates(k_he4_ti47__p_v50) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti47__p_v50) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 22.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ti48__v49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti48__v49);
    rate_eval.screened_rates(k_p_ti48__v49) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti48__v49) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ti48__n_v48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti48__n_v48);
    rate_eval.screened_rates(k_p_ti48__n_v48) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti48__n_v48) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ti48__he4_sc45);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti48__he4_sc45);
    rate_eval.screened_rates(k_p_ti48__he4_sc45) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti48__he4_sc45) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 22.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ti48__cr52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti48__cr52);
    rate_eval.screened_rates(k_he4_ti48__cr52) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti48__cr52) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ti48__n_cr51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti48__n_cr51);
    rate_eval.screened_rates(k_he4_ti48__n_cr51) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti48__n_cr51) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ti48__p_v51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti48__p_v51);
    rate_eval.screened_rates(k_he4_ti48__p_v51) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti48__p_v51) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 22.0_rt, 49.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ti49__v50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti49__v50);
    rate_eval.screened_rates(k_p_ti49__v50) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti49__v50) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ti49__n_v49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti49__n_v49);
    rate_eval.screened_rates(k_p_ti49__n_v49) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti49__n_v49) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ti49__he4_sc46);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti49__he4_sc46);
    rate_eval.screened_rates(k_p_ti49__he4_sc46) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti49__he4_sc46) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 22.0_rt, 49.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ti49__cr53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti49__cr53);
    rate_eval.screened_rates(k_he4_ti49__cr53) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti49__cr53) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ti49__n_cr52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti49__n_cr52);
    rate_eval.screened_rates(k_he4_ti49__n_cr52) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti49__n_cr52) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ti49__p_v52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti49__p_v52);
    rate_eval.screened_rates(k_he4_ti49__p_v52) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti49__p_v52) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 22.0_rt, 50.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ti50__v51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti50__v51);
    rate_eval.screened_rates(k_p_ti50__v51) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti50__v51) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ti50__n_v50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti50__n_v50);
    rate_eval.screened_rates(k_p_ti50__n_v50) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti50__n_v50) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ti50__he4_sc47);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti50__he4_sc47);
    rate_eval.screened_rates(k_p_ti50__he4_sc47) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti50__he4_sc47) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 22.0_rt, 50.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ti50__cr54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti50__cr54);
    rate_eval.screened_rates(k_he4_ti50__cr54) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti50__cr54) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ti50__n_cr53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti50__n_cr53);
    rate_eval.screened_rates(k_he4_ti50__n_cr53) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti50__n_cr53) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 22.0_rt, 51.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ti51__v52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti51__v52);
    rate_eval.screened_rates(k_p_ti51__v52) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti51__v52) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ti51__n_v51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti51__n_v51);
    rate_eval.screened_rates(k_p_ti51__n_v51) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti51__n_v51) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ti51__he4_sc48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ti51__he4_sc48);
    rate_eval.screened_rates(k_p_ti51__he4_sc48) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ti51__he4_sc48) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 23.0_rt, 46.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_v46__mn50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v46__mn50);
    rate_eval.screened_rates(k_he4_v46__mn50) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v46__mn50) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_v46__p_cr49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v46__p_cr49);
    rate_eval.screened_rates(k_he4_v46__p_cr49) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v46__p_cr49) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 23.0_rt, 47.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_v47__cr48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v47__cr48);
    rate_eval.screened_rates(k_p_v47__cr48) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v47__cr48) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_v47__he4_ti44);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v47__he4_ti44);
    rate_eval.screened_rates(k_p_v47__he4_ti44) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v47__he4_ti44) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 23.0_rt, 47.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_v47__mn51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v47__mn51);
    rate_eval.screened_rates(k_he4_v47__mn51) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v47__mn51) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_v47__n_mn50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v47__n_mn50);
    rate_eval.screened_rates(k_he4_v47__n_mn50) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v47__n_mn50) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_v47__p_cr50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v47__p_cr50);
    rate_eval.screened_rates(k_he4_v47__p_cr50) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v47__p_cr50) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 23.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_v48__cr49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v48__cr49);
    rate_eval.screened_rates(k_p_v48__cr49) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v48__cr49) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_v48__n_cr48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v48__n_cr48);
    rate_eval.screened_rates(k_p_v48__n_cr48) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v48__n_cr48) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_v48__he4_ti45);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v48__he4_ti45);
    rate_eval.screened_rates(k_p_v48__he4_ti45) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v48__he4_ti45) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 23.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_v48__mn52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v48__mn52);
    rate_eval.screened_rates(k_he4_v48__mn52) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v48__mn52) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_v48__n_mn51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v48__n_mn51);
    rate_eval.screened_rates(k_he4_v48__n_mn51) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v48__n_mn51) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_v48__p_cr51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v48__p_cr51);
    rate_eval.screened_rates(k_he4_v48__p_cr51) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v48__p_cr51) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 23.0_rt, 49.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_v49__cr50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v49__cr50);
    rate_eval.screened_rates(k_p_v49__cr50) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v49__cr50) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_v49__n_cr49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v49__n_cr49);
    rate_eval.screened_rates(k_p_v49__n_cr49) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v49__n_cr49) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_v49__he4_ti46);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v49__he4_ti46);
    rate_eval.screened_rates(k_p_v49__he4_ti46) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v49__he4_ti46) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 23.0_rt, 49.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_v49__mn53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v49__mn53);
    rate_eval.screened_rates(k_he4_v49__mn53) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v49__mn53) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_v49__n_mn52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v49__n_mn52);
    rate_eval.screened_rates(k_he4_v49__n_mn52) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v49__n_mn52) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_v49__p_cr52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v49__p_cr52);
    rate_eval.screened_rates(k_he4_v49__p_cr52) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v49__p_cr52) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 23.0_rt, 50.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_v50__cr51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v50__cr51);
    rate_eval.screened_rates(k_p_v50__cr51) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v50__cr51) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_v50__n_cr50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v50__n_cr50);
    rate_eval.screened_rates(k_p_v50__n_cr50) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v50__n_cr50) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_v50__he4_ti47);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v50__he4_ti47);
    rate_eval.screened_rates(k_p_v50__he4_ti47) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v50__he4_ti47) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 23.0_rt, 50.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_v50__mn54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v50__mn54);
    rate_eval.screened_rates(k_he4_v50__mn54) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v50__mn54) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_v50__n_mn53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v50__n_mn53);
    rate_eval.screened_rates(k_he4_v50__n_mn53) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v50__n_mn53) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_v50__p_cr53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v50__p_cr53);
    rate_eval.screened_rates(k_he4_v50__p_cr53) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v50__p_cr53) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 23.0_rt, 51.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_v51__cr52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v51__cr52);
    rate_eval.screened_rates(k_p_v51__cr52) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v51__cr52) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_v51__n_cr51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v51__n_cr51);
    rate_eval.screened_rates(k_p_v51__n_cr51) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v51__n_cr51) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_v51__he4_ti48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v51__he4_ti48);
    rate_eval.screened_rates(k_p_v51__he4_ti48) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v51__he4_ti48) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 23.0_rt, 51.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_v51__mn55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v51__mn55);
    rate_eval.screened_rates(k_he4_v51__mn55) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v51__mn55) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_v51__n_mn54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v51__n_mn54);
    rate_eval.screened_rates(k_he4_v51__n_mn54) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v51__n_mn54) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_v51__p_cr54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v51__p_cr54);
    rate_eval.screened_rates(k_he4_v51__p_cr54) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v51__p_cr54) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 23.0_rt, 52.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_v52__cr53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v52__cr53);
    rate_eval.screened_rates(k_p_v52__cr53) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v52__cr53) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_v52__n_cr52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v52__n_cr52);
    rate_eval.screened_rates(k_p_v52__n_cr52) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v52__n_cr52) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_v52__he4_ti49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_v52__he4_ti49);
    rate_eval.screened_rates(k_p_v52__he4_ti49) *= scor;
    rate_eval.dscreened_rates_dT(k_p_v52__he4_ti49) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 24.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cr48__fe52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr48__fe52);
    rate_eval.screened_rates(k_he4_cr48__fe52) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr48__fe52) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cr48__p_mn51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr48__p_mn51);
    rate_eval.screened_rates(k_he4_cr48__p_mn51) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr48__p_mn51) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 24.0_rt, 49.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cr49__mn50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr49__mn50);
    rate_eval.screened_rates(k_p_cr49__mn50) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr49__mn50) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cr49__he4_v46);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr49__he4_v46);
    rate_eval.screened_rates(k_p_cr49__he4_v46) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr49__he4_v46) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 24.0_rt, 49.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cr49__fe53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr49__fe53);
    rate_eval.screened_rates(k_he4_cr49__fe53) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr49__fe53) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cr49__n_fe52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr49__n_fe52);
    rate_eval.screened_rates(k_he4_cr49__n_fe52) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr49__n_fe52) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cr49__p_mn52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr49__p_mn52);
    rate_eval.screened_rates(k_he4_cr49__p_mn52) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr49__p_mn52) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 24.0_rt, 50.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cr50__mn51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr50__mn51);
    rate_eval.screened_rates(k_p_cr50__mn51) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr50__mn51) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cr50__n_mn50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr50__n_mn50);
    rate_eval.screened_rates(k_p_cr50__n_mn50) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr50__n_mn50) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cr50__he4_v47);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr50__he4_v47);
    rate_eval.screened_rates(k_p_cr50__he4_v47) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr50__he4_v47) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 24.0_rt, 50.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cr50__fe54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr50__fe54);
    rate_eval.screened_rates(k_he4_cr50__fe54) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr50__fe54) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cr50__n_fe53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr50__n_fe53);
    rate_eval.screened_rates(k_he4_cr50__n_fe53) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr50__n_fe53) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cr50__p_mn53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr50__p_mn53);
    rate_eval.screened_rates(k_he4_cr50__p_mn53) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr50__p_mn53) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 24.0_rt, 51.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cr51__mn52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr51__mn52);
    rate_eval.screened_rates(k_p_cr51__mn52) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr51__mn52) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cr51__n_mn51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr51__n_mn51);
    rate_eval.screened_rates(k_p_cr51__n_mn51) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr51__n_mn51) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cr51__he4_v48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr51__he4_v48);
    rate_eval.screened_rates(k_p_cr51__he4_v48) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr51__he4_v48) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 24.0_rt, 51.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cr51__fe55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr51__fe55);
    rate_eval.screened_rates(k_he4_cr51__fe55) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr51__fe55) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cr51__n_fe54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr51__n_fe54);
    rate_eval.screened_rates(k_he4_cr51__n_fe54) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr51__n_fe54) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cr51__p_mn54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr51__p_mn54);
    rate_eval.screened_rates(k_he4_cr51__p_mn54) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr51__p_mn54) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 24.0_rt, 52.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cr52__mn53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr52__mn53);
    rate_eval.screened_rates(k_p_cr52__mn53) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr52__mn53) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cr52__n_mn52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr52__n_mn52);
    rate_eval.screened_rates(k_p_cr52__n_mn52) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr52__n_mn52) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cr52__he4_v49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr52__he4_v49);
    rate_eval.screened_rates(k_p_cr52__he4_v49) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr52__he4_v49) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 24.0_rt, 52.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cr52__fe56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr52__fe56);
    rate_eval.screened_rates(k_he4_cr52__fe56) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr52__fe56) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cr52__n_fe55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr52__n_fe55);
    rate_eval.screened_rates(k_he4_cr52__n_fe55) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr52__n_fe55) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cr52__p_mn55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr52__p_mn55);
    rate_eval.screened_rates(k_he4_cr52__p_mn55) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr52__p_mn55) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 24.0_rt, 53.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cr53__mn54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr53__mn54);
    rate_eval.screened_rates(k_p_cr53__mn54) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr53__mn54) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cr53__n_mn53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr53__n_mn53);
    rate_eval.screened_rates(k_p_cr53__n_mn53) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr53__n_mn53) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cr53__he4_v50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr53__he4_v50);
    rate_eval.screened_rates(k_p_cr53__he4_v50) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr53__he4_v50) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 24.0_rt, 53.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cr53__fe57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr53__fe57);
    rate_eval.screened_rates(k_he4_cr53__fe57) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr53__fe57) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cr53__n_fe56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr53__n_fe56);
    rate_eval.screened_rates(k_he4_cr53__n_fe56) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr53__n_fe56) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 24.0_rt, 54.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cr54__mn55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr54__mn55);
    rate_eval.screened_rates(k_p_cr54__mn55) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr54__mn55) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cr54__n_mn54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr54__n_mn54);
    rate_eval.screened_rates(k_p_cr54__n_mn54) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr54__n_mn54) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cr54__he4_v51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cr54__he4_v51);
    rate_eval.screened_rates(k_p_cr54__he4_v51) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cr54__he4_v51) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 24.0_rt, 54.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cr54__fe58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr54__fe58);
    rate_eval.screened_rates(k_he4_cr54__fe58) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr54__fe58) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cr54__n_fe57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cr54__n_fe57);
    rate_eval.screened_rates(k_he4_cr54__n_fe57) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cr54__n_fe57) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 25.0_rt, 50.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_mn50__co54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn50__co54);
    rate_eval.screened_rates(k_he4_mn50__co54) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn50__co54) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_mn50__n_co53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn50__n_co53);
    rate_eval.screened_rates(k_he4_mn50__n_co53) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn50__n_co53) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_mn50__p_fe53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn50__p_fe53);
    rate_eval.screened_rates(k_he4_mn50__p_fe53) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn50__p_fe53) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 25.0_rt, 51.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_mn51__fe52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mn51__fe52);
    rate_eval.screened_rates(k_p_mn51__fe52) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mn51__fe52) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_mn51__he4_cr48);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mn51__he4_cr48);
    rate_eval.screened_rates(k_p_mn51__he4_cr48) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mn51__he4_cr48) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 25.0_rt, 51.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_mn51__co55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn51__co55);
    rate_eval.screened_rates(k_he4_mn51__co55) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn51__co55) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_mn51__n_co54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn51__n_co54);
    rate_eval.screened_rates(k_he4_mn51__n_co54) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn51__n_co54) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_mn51__p_fe54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn51__p_fe54);
    rate_eval.screened_rates(k_he4_mn51__p_fe54) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn51__p_fe54) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 25.0_rt, 52.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_mn52__fe53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mn52__fe53);
    rate_eval.screened_rates(k_p_mn52__fe53) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mn52__fe53) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_mn52__n_fe52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mn52__n_fe52);
    rate_eval.screened_rates(k_p_mn52__n_fe52) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mn52__n_fe52) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_mn52__he4_cr49);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mn52__he4_cr49);
    rate_eval.screened_rates(k_p_mn52__he4_cr49) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mn52__he4_cr49) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 25.0_rt, 52.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_mn52__co56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn52__co56);
    rate_eval.screened_rates(k_he4_mn52__co56) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn52__co56) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_mn52__n_co55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn52__n_co55);
    rate_eval.screened_rates(k_he4_mn52__n_co55) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn52__n_co55) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_mn52__p_fe55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn52__p_fe55);
    rate_eval.screened_rates(k_he4_mn52__p_fe55) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn52__p_fe55) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 25.0_rt, 53.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_mn53__fe54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mn53__fe54);
    rate_eval.screened_rates(k_p_mn53__fe54) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mn53__fe54) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_mn53__n_fe53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mn53__n_fe53);
    rate_eval.screened_rates(k_p_mn53__n_fe53) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mn53__n_fe53) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_mn53__he4_cr50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mn53__he4_cr50);
    rate_eval.screened_rates(k_p_mn53__he4_cr50) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mn53__he4_cr50) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 25.0_rt, 53.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_mn53__co57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn53__co57);
    rate_eval.screened_rates(k_he4_mn53__co57) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn53__co57) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_mn53__n_co56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn53__n_co56);
    rate_eval.screened_rates(k_he4_mn53__n_co56) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn53__n_co56) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_mn53__p_fe56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn53__p_fe56);
    rate_eval.screened_rates(k_he4_mn53__p_fe56) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn53__p_fe56) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 25.0_rt, 54.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_mn54__fe55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mn54__fe55);
    rate_eval.screened_rates(k_p_mn54__fe55) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mn54__fe55) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_mn54__n_fe54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mn54__n_fe54);
    rate_eval.screened_rates(k_p_mn54__n_fe54) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mn54__n_fe54) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_mn54__he4_cr51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mn54__he4_cr51);
    rate_eval.screened_rates(k_p_mn54__he4_cr51) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mn54__he4_cr51) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 25.0_rt, 54.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_mn54__co58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn54__co58);
    rate_eval.screened_rates(k_he4_mn54__co58) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn54__co58) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_mn54__n_co57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn54__n_co57);
    rate_eval.screened_rates(k_he4_mn54__n_co57) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn54__n_co57) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_mn54__p_fe57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn54__p_fe57);
    rate_eval.screened_rates(k_he4_mn54__p_fe57) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn54__p_fe57) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 25.0_rt, 55.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_mn55__fe56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mn55__fe56);
    rate_eval.screened_rates(k_p_mn55__fe56) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mn55__fe56) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_mn55__n_fe55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mn55__n_fe55);
    rate_eval.screened_rates(k_p_mn55__n_fe55) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mn55__n_fe55) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_mn55__he4_cr52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_mn55__he4_cr52);
    rate_eval.screened_rates(k_p_mn55__he4_cr52) *= scor;
    rate_eval.dscreened_rates_dT(k_p_mn55__he4_cr52) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 25.0_rt, 55.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_mn55__co59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn55__co59);
    rate_eval.screened_rates(k_he4_mn55__co59) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn55__co59) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_mn55__n_co58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn55__n_co58);
    rate_eval.screened_rates(k_he4_mn55__n_co58) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn55__n_co58) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_mn55__p_fe58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mn55__p_fe58);
    rate_eval.screened_rates(k_he4_mn55__p_fe58) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mn55__p_fe58) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 26.0_rt, 52.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_fe52__co53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe52__co53);
    rate_eval.screened_rates(k_p_fe52__co53) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe52__co53) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 26.0_rt, 52.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_fe52__ni56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe52__ni56);
    rate_eval.screened_rates(k_he4_fe52__ni56) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe52__ni56) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_fe52__p_co55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe52__p_co55);
    rate_eval.screened_rates(k_he4_fe52__p_co55) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe52__p_co55) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 26.0_rt, 53.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_fe53__co54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe53__co54);
    rate_eval.screened_rates(k_p_fe53__co54) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe53__co54) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_fe53__n_co53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe53__n_co53);
    rate_eval.screened_rates(k_p_fe53__n_co53) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe53__n_co53) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_fe53__he4_mn50);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe53__he4_mn50);
    rate_eval.screened_rates(k_p_fe53__he4_mn50) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe53__he4_mn50) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 26.0_rt, 53.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_fe53__ni57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe53__ni57);
    rate_eval.screened_rates(k_he4_fe53__ni57) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe53__ni57) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_fe53__n_ni56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe53__n_ni56);
    rate_eval.screened_rates(k_he4_fe53__n_ni56) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe53__n_ni56) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_fe53__p_co56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe53__p_co56);
    rate_eval.screened_rates(k_he4_fe53__p_co56) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe53__p_co56) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 26.0_rt, 54.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_fe54__co55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe54__co55);
    rate_eval.screened_rates(k_p_fe54__co55) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe54__co55) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_fe54__n_co54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe54__n_co54);
    rate_eval.screened_rates(k_p_fe54__n_co54) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe54__n_co54) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_fe54__he4_mn51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe54__he4_mn51);
    rate_eval.screened_rates(k_p_fe54__he4_mn51) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe54__he4_mn51) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 26.0_rt, 54.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_fe54__ni58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe54__ni58);
    rate_eval.screened_rates(k_he4_fe54__ni58) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe54__ni58) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_fe54__n_ni57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe54__n_ni57);
    rate_eval.screened_rates(k_he4_fe54__n_ni57) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe54__n_ni57) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_fe54__p_co57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe54__p_co57);
    rate_eval.screened_rates(k_he4_fe54__p_co57) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe54__p_co57) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 26.0_rt, 55.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_fe55__co56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe55__co56);
    rate_eval.screened_rates(k_p_fe55__co56) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe55__co56) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_fe55__n_co55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe55__n_co55);
    rate_eval.screened_rates(k_p_fe55__n_co55) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe55__n_co55) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_fe55__he4_mn52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe55__he4_mn52);
    rate_eval.screened_rates(k_p_fe55__he4_mn52) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe55__he4_mn52) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 26.0_rt, 55.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_fe55__ni59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe55__ni59);
    rate_eval.screened_rates(k_he4_fe55__ni59) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe55__ni59) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_fe55__n_ni58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe55__n_ni58);
    rate_eval.screened_rates(k_he4_fe55__n_ni58) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe55__n_ni58) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_fe55__p_co58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe55__p_co58);
    rate_eval.screened_rates(k_he4_fe55__p_co58) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe55__p_co58) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 26.0_rt, 56.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_fe56__co57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe56__co57);
    rate_eval.screened_rates(k_p_fe56__co57) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe56__co57) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_fe56__n_co56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe56__n_co56);
    rate_eval.screened_rates(k_p_fe56__n_co56) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe56__n_co56) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_fe56__he4_mn53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe56__he4_mn53);
    rate_eval.screened_rates(k_p_fe56__he4_mn53) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe56__he4_mn53) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 26.0_rt, 56.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_fe56__ni60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe56__ni60);
    rate_eval.screened_rates(k_he4_fe56__ni60) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe56__ni60) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_fe56__n_ni59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe56__n_ni59);
    rate_eval.screened_rates(k_he4_fe56__n_ni59) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe56__n_ni59) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_fe56__p_co59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe56__p_co59);
    rate_eval.screened_rates(k_he4_fe56__p_co59) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe56__p_co59) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 26.0_rt, 57.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_fe57__co58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe57__co58);
    rate_eval.screened_rates(k_p_fe57__co58) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe57__co58) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_fe57__n_co57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe57__n_co57);
    rate_eval.screened_rates(k_p_fe57__n_co57) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe57__n_co57) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_fe57__he4_mn54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe57__he4_mn54);
    rate_eval.screened_rates(k_p_fe57__he4_mn54) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe57__he4_mn54) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 26.0_rt, 57.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_fe57__ni61);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe57__ni61);
    rate_eval.screened_rates(k_he4_fe57__ni61) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe57__ni61) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_fe57__n_ni60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe57__n_ni60);
    rate_eval.screened_rates(k_he4_fe57__n_ni60) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe57__n_ni60) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 26.0_rt, 58.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_fe58__co59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe58__co59);
    rate_eval.screened_rates(k_p_fe58__co59) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe58__co59) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_fe58__n_co58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe58__n_co58);
    rate_eval.screened_rates(k_p_fe58__n_co58) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe58__n_co58) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_fe58__he4_mn55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_fe58__he4_mn55);
    rate_eval.screened_rates(k_p_fe58__he4_mn55) *= scor;
    rate_eval.dscreened_rates_dT(k_p_fe58__he4_mn55) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 26.0_rt, 58.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_fe58__ni62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe58__ni62);
    rate_eval.screened_rates(k_he4_fe58__ni62) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe58__ni62) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_fe58__n_ni61);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_fe58__n_ni61);
    rate_eval.screened_rates(k_he4_fe58__n_ni61) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_fe58__n_ni61) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 27.0_rt, 53.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_co53__cu57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co53__cu57);
    rate_eval.screened_rates(k_he4_co53__cu57) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co53__cu57) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_co53__p_ni56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co53__p_ni56);
    rate_eval.screened_rates(k_he4_co53__p_ni56) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co53__p_ni56) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 27.0_rt, 54.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_co54__cu58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co54__cu58);
    rate_eval.screened_rates(k_he4_co54__cu58) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co54__cu58) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_co54__n_cu57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co54__n_cu57);
    rate_eval.screened_rates(k_he4_co54__n_cu57) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co54__n_cu57) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_co54__p_ni57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co54__p_ni57);
    rate_eval.screened_rates(k_he4_co54__p_ni57) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co54__p_ni57) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 27.0_rt, 55.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_co55__ni56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_co55__ni56);
    rate_eval.screened_rates(k_p_co55__ni56) *= scor;
    rate_eval.dscreened_rates_dT(k_p_co55__ni56) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_co55__he4_fe52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_co55__he4_fe52);
    rate_eval.screened_rates(k_p_co55__he4_fe52) *= scor;
    rate_eval.dscreened_rates_dT(k_p_co55__he4_fe52) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 27.0_rt, 55.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_co55__cu59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co55__cu59);
    rate_eval.screened_rates(k_he4_co55__cu59) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co55__cu59) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_co55__n_cu58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co55__n_cu58);
    rate_eval.screened_rates(k_he4_co55__n_cu58) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co55__n_cu58) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_co55__p_ni58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co55__p_ni58);
    rate_eval.screened_rates(k_he4_co55__p_ni58) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co55__p_ni58) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 27.0_rt, 56.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_co56__ni57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_co56__ni57);
    rate_eval.screened_rates(k_p_co56__ni57) *= scor;
    rate_eval.dscreened_rates_dT(k_p_co56__ni57) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_co56__n_ni56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_co56__n_ni56);
    rate_eval.screened_rates(k_p_co56__n_ni56) *= scor;
    rate_eval.dscreened_rates_dT(k_p_co56__n_ni56) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_co56__he4_fe53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_co56__he4_fe53);
    rate_eval.screened_rates(k_p_co56__he4_fe53) *= scor;
    rate_eval.dscreened_rates_dT(k_p_co56__he4_fe53) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 27.0_rt, 56.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_co56__cu60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co56__cu60);
    rate_eval.screened_rates(k_he4_co56__cu60) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co56__cu60) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_co56__n_cu59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co56__n_cu59);
    rate_eval.screened_rates(k_he4_co56__n_cu59) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co56__n_cu59) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_co56__p_ni59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co56__p_ni59);
    rate_eval.screened_rates(k_he4_co56__p_ni59) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co56__p_ni59) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 27.0_rt, 57.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_co57__ni58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_co57__ni58);
    rate_eval.screened_rates(k_p_co57__ni58) *= scor;
    rate_eval.dscreened_rates_dT(k_p_co57__ni58) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_co57__n_ni57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_co57__n_ni57);
    rate_eval.screened_rates(k_p_co57__n_ni57) *= scor;
    rate_eval.dscreened_rates_dT(k_p_co57__n_ni57) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_co57__he4_fe54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_co57__he4_fe54);
    rate_eval.screened_rates(k_p_co57__he4_fe54) *= scor;
    rate_eval.dscreened_rates_dT(k_p_co57__he4_fe54) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 27.0_rt, 57.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_co57__cu61);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co57__cu61);
    rate_eval.screened_rates(k_he4_co57__cu61) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co57__cu61) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_co57__n_cu60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co57__n_cu60);
    rate_eval.screened_rates(k_he4_co57__n_cu60) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co57__n_cu60) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_co57__p_ni60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co57__p_ni60);
    rate_eval.screened_rates(k_he4_co57__p_ni60) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co57__p_ni60) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 27.0_rt, 58.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_co58__ni59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_co58__ni59);
    rate_eval.screened_rates(k_p_co58__ni59) *= scor;
    rate_eval.dscreened_rates_dT(k_p_co58__ni59) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_co58__n_ni58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_co58__n_ni58);
    rate_eval.screened_rates(k_p_co58__n_ni58) *= scor;
    rate_eval.dscreened_rates_dT(k_p_co58__n_ni58) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_co58__he4_fe55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_co58__he4_fe55);
    rate_eval.screened_rates(k_p_co58__he4_fe55) *= scor;
    rate_eval.dscreened_rates_dT(k_p_co58__he4_fe55) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 27.0_rt, 58.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_co58__cu62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co58__cu62);
    rate_eval.screened_rates(k_he4_co58__cu62) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co58__cu62) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_co58__n_cu61);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co58__n_cu61);
    rate_eval.screened_rates(k_he4_co58__n_cu61) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co58__n_cu61) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_co58__p_ni61);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co58__p_ni61);
    rate_eval.screened_rates(k_he4_co58__p_ni61) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co58__p_ni61) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 27.0_rt, 59.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_co59__ni60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_co59__ni60);
    rate_eval.screened_rates(k_p_co59__ni60) *= scor;
    rate_eval.dscreened_rates_dT(k_p_co59__ni60) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_co59__n_ni59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_co59__n_ni59);
    rate_eval.screened_rates(k_p_co59__n_ni59) *= scor;
    rate_eval.dscreened_rates_dT(k_p_co59__n_ni59) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_co59__he4_fe56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_co59__he4_fe56);
    rate_eval.screened_rates(k_p_co59__he4_fe56) *= scor;
    rate_eval.dscreened_rates_dT(k_p_co59__he4_fe56) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 27.0_rt, 59.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_co59__cu63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co59__cu63);
    rate_eval.screened_rates(k_he4_co59__cu63) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co59__cu63) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_co59__n_cu62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co59__n_cu62);
    rate_eval.screened_rates(k_he4_co59__n_cu62) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co59__n_cu62) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_co59__p_ni62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_co59__p_ni62);
    rate_eval.screened_rates(k_he4_co59__p_ni62) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_co59__p_ni62) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 56.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ni56__cu57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni56__cu57);
    rate_eval.screened_rates(k_p_ni56__cu57) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni56__cu57) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ni56__he4_co53);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni56__he4_co53);
    rate_eval.screened_rates(k_p_ni56__he4_co53) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni56__he4_co53) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 28.0_rt, 56.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ni56__zn60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni56__zn60);
    rate_eval.screened_rates(k_he4_ni56__zn60) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni56__zn60) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ni56__n_zn59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni56__n_zn59);
    rate_eval.screened_rates(k_he4_ni56__n_zn59) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni56__n_zn59) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ni56__p_cu59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni56__p_cu59);
    rate_eval.screened_rates(k_he4_ni56__p_cu59) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni56__p_cu59) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 57.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ni57__cu58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni57__cu58);
    rate_eval.screened_rates(k_p_ni57__cu58) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni57__cu58) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ni57__n_cu57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni57__n_cu57);
    rate_eval.screened_rates(k_p_ni57__n_cu57) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni57__n_cu57) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ni57__he4_co54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni57__he4_co54);
    rate_eval.screened_rates(k_p_ni57__he4_co54) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni57__he4_co54) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 28.0_rt, 57.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ni57__zn61);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni57__zn61);
    rate_eval.screened_rates(k_he4_ni57__zn61) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni57__zn61) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ni57__n_zn60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni57__n_zn60);
    rate_eval.screened_rates(k_he4_ni57__n_zn60) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni57__n_zn60) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ni57__p_cu60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni57__p_cu60);
    rate_eval.screened_rates(k_he4_ni57__p_cu60) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni57__p_cu60) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 58.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ni58__cu59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni58__cu59);
    rate_eval.screened_rates(k_p_ni58__cu59) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni58__cu59) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ni58__n_cu58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni58__n_cu58);
    rate_eval.screened_rates(k_p_ni58__n_cu58) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni58__n_cu58) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ni58__he4_co55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni58__he4_co55);
    rate_eval.screened_rates(k_p_ni58__he4_co55) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni58__he4_co55) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 28.0_rt, 58.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ni58__zn62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni58__zn62);
    rate_eval.screened_rates(k_he4_ni58__zn62) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni58__zn62) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ni58__n_zn61);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni58__n_zn61);
    rate_eval.screened_rates(k_he4_ni58__n_zn61) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni58__n_zn61) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ni58__p_cu61);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni58__p_cu61);
    rate_eval.screened_rates(k_he4_ni58__p_cu61) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni58__p_cu61) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 59.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ni59__cu60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni59__cu60);
    rate_eval.screened_rates(k_p_ni59__cu60) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni59__cu60) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ni59__n_cu59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni59__n_cu59);
    rate_eval.screened_rates(k_p_ni59__n_cu59) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni59__n_cu59) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ni59__he4_co56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni59__he4_co56);
    rate_eval.screened_rates(k_p_ni59__he4_co56) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni59__he4_co56) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 28.0_rt, 59.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ni59__zn63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni59__zn63);
    rate_eval.screened_rates(k_he4_ni59__zn63) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni59__zn63) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ni59__n_zn62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni59__n_zn62);
    rate_eval.screened_rates(k_he4_ni59__n_zn62) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni59__n_zn62) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ni59__p_cu62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni59__p_cu62);
    rate_eval.screened_rates(k_he4_ni59__p_cu62) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni59__p_cu62) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 60.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ni60__cu61);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni60__cu61);
    rate_eval.screened_rates(k_p_ni60__cu61) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni60__cu61) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ni60__n_cu60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni60__n_cu60);
    rate_eval.screened_rates(k_p_ni60__n_cu60) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni60__n_cu60) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ni60__he4_co57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni60__he4_co57);
    rate_eval.screened_rates(k_p_ni60__he4_co57) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni60__he4_co57) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 28.0_rt, 60.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ni60__zn64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni60__zn64);
    rate_eval.screened_rates(k_he4_ni60__zn64) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni60__zn64) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ni60__n_zn63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni60__n_zn63);
    rate_eval.screened_rates(k_he4_ni60__n_zn63) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni60__n_zn63) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ni60__p_cu63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni60__p_cu63);
    rate_eval.screened_rates(k_he4_ni60__p_cu63) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni60__p_cu63) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 61.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ni61__cu62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni61__cu62);
    rate_eval.screened_rates(k_p_ni61__cu62) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni61__cu62) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ni61__n_cu61);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni61__n_cu61);
    rate_eval.screened_rates(k_p_ni61__n_cu61) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni61__n_cu61) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ni61__he4_co58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni61__he4_co58);
    rate_eval.screened_rates(k_p_ni61__he4_co58) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni61__he4_co58) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 28.0_rt, 61.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ni61__zn65);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni61__zn65);
    rate_eval.screened_rates(k_he4_ni61__zn65) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni61__zn65) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ni61__n_zn64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni61__n_zn64);
    rate_eval.screened_rates(k_he4_ni61__n_zn64) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni61__n_zn64) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ni61__p_cu64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni61__p_cu64);
    rate_eval.screened_rates(k_he4_ni61__p_cu64) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni61__p_cu64) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 62.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ni62__cu63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni62__cu63);
    rate_eval.screened_rates(k_p_ni62__cu63) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni62__cu63) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ni62__n_cu62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni62__n_cu62);
    rate_eval.screened_rates(k_p_ni62__n_cu62) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni62__n_cu62) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ni62__he4_co59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni62__he4_co59);
    rate_eval.screened_rates(k_p_ni62__he4_co59) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni62__he4_co59) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 28.0_rt, 62.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ni62__zn66);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni62__zn66);
    rate_eval.screened_rates(k_he4_ni62__zn66) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni62__zn66) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ni62__n_zn65);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni62__n_zn65);
    rate_eval.screened_rates(k_he4_ni62__n_zn65) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni62__n_zn65) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_ni62__p_cu65);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni62__p_cu65);
    rate_eval.screened_rates(k_he4_ni62__p_cu65) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni62__p_cu65) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 63.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ni63__cu64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni63__cu64);
    rate_eval.screened_rates(k_p_ni63__cu64) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni63__cu64) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ni63__n_cu63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni63__n_cu63);
    rate_eval.screened_rates(k_p_ni63__n_cu63) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni63__n_cu63) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 64.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ni64__cu65);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni64__cu65);
    rate_eval.screened_rates(k_p_ni64__cu65) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni64__cu65) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ni64__n_cu64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ni64__n_cu64);
    rate_eval.screened_rates(k_p_ni64__n_cu64) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ni64__n_cu64) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 29.0_rt, 58.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cu58__zn59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu58__zn59);
    rate_eval.screened_rates(k_p_cu58__zn59) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu58__zn59) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 29.0_rt, 58.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cu58__ga62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cu58__ga62);
    rate_eval.screened_rates(k_he4_cu58__ga62) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cu58__ga62) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cu58__p_zn61);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cu58__p_zn61);
    rate_eval.screened_rates(k_he4_cu58__p_zn61) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cu58__p_zn61) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 29.0_rt, 59.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cu59__zn60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu59__zn60);
    rate_eval.screened_rates(k_p_cu59__zn60) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu59__zn60) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cu59__n_zn59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu59__n_zn59);
    rate_eval.screened_rates(k_p_cu59__n_zn59) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu59__n_zn59) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cu59__he4_ni56);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu59__he4_ni56);
    rate_eval.screened_rates(k_p_cu59__he4_ni56) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu59__he4_ni56) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 29.0_rt, 59.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cu59__ga63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cu59__ga63);
    rate_eval.screened_rates(k_he4_cu59__ga63) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cu59__ga63) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cu59__n_ga62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cu59__n_ga62);
    rate_eval.screened_rates(k_he4_cu59__n_ga62) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cu59__n_ga62) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cu59__p_zn62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cu59__p_zn62);
    rate_eval.screened_rates(k_he4_cu59__p_zn62) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cu59__p_zn62) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 29.0_rt, 60.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cu60__zn61);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu60__zn61);
    rate_eval.screened_rates(k_p_cu60__zn61) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu60__zn61) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cu60__n_zn60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu60__n_zn60);
    rate_eval.screened_rates(k_p_cu60__n_zn60) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu60__n_zn60) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cu60__he4_ni57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu60__he4_ni57);
    rate_eval.screened_rates(k_p_cu60__he4_ni57) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu60__he4_ni57) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 29.0_rt, 60.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cu60__ga64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cu60__ga64);
    rate_eval.screened_rates(k_he4_cu60__ga64) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cu60__ga64) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cu60__n_ga63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cu60__n_ga63);
    rate_eval.screened_rates(k_he4_cu60__n_ga63) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cu60__n_ga63) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cu60__p_zn63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cu60__p_zn63);
    rate_eval.screened_rates(k_he4_cu60__p_zn63) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cu60__p_zn63) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 29.0_rt, 61.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cu61__zn62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu61__zn62);
    rate_eval.screened_rates(k_p_cu61__zn62) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu61__zn62) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cu61__n_zn61);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu61__n_zn61);
    rate_eval.screened_rates(k_p_cu61__n_zn61) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu61__n_zn61) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cu61__he4_ni58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu61__he4_ni58);
    rate_eval.screened_rates(k_p_cu61__he4_ni58) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu61__he4_ni58) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 29.0_rt, 62.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cu62__zn63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu62__zn63);
    rate_eval.screened_rates(k_p_cu62__zn63) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu62__zn63) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cu62__n_zn62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu62__n_zn62);
    rate_eval.screened_rates(k_p_cu62__n_zn62) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu62__n_zn62) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cu62__he4_ni59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu62__he4_ni59);
    rate_eval.screened_rates(k_p_cu62__he4_ni59) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu62__he4_ni59) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 29.0_rt, 63.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cu63__zn64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu63__zn64);
    rate_eval.screened_rates(k_p_cu63__zn64) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu63__zn64) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cu63__n_zn63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu63__n_zn63);
    rate_eval.screened_rates(k_p_cu63__n_zn63) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu63__n_zn63) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cu63__he4_ni60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu63__he4_ni60);
    rate_eval.screened_rates(k_p_cu63__he4_ni60) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu63__he4_ni60) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 29.0_rt, 64.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cu64__zn65);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu64__zn65);
    rate_eval.screened_rates(k_p_cu64__zn65) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu64__zn65) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cu64__n_zn64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu64__n_zn64);
    rate_eval.screened_rates(k_p_cu64__n_zn64) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu64__n_zn64) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cu64__he4_ni61);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu64__he4_ni61);
    rate_eval.screened_rates(k_p_cu64__he4_ni61) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu64__he4_ni61) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 29.0_rt, 65.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_cu65__zn66);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu65__zn66);
    rate_eval.screened_rates(k_p_cu65__zn66) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu65__zn66) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cu65__n_zn65);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu65__n_zn65);
    rate_eval.screened_rates(k_p_cu65__n_zn65) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu65__n_zn65) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_cu65__he4_ni62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_cu65__he4_ni62);
    rate_eval.screened_rates(k_p_cu65__he4_ni62) *= scor;
    rate_eval.dscreened_rates_dT(k_p_cu65__he4_ni62) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 30.0_rt, 59.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_zn59__ge63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_zn59__ge63);
    rate_eval.screened_rates(k_he4_zn59__ge63) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_zn59__ge63) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_zn59__p_ga62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_zn59__p_ga62);
    rate_eval.screened_rates(k_he4_zn59__p_ga62) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_zn59__p_ga62) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 30.0_rt, 60.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_zn60__ge64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_zn60__ge64);
    rate_eval.screened_rates(k_he4_zn60__ge64) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_zn60__ge64) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_zn60__n_ge63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_zn60__n_ge63);
    rate_eval.screened_rates(k_he4_zn60__n_ge63) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_zn60__n_ge63) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_zn60__p_ga63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_zn60__p_ga63);
    rate_eval.screened_rates(k_he4_zn60__p_ga63) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_zn60__p_ga63) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 30.0_rt, 61.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_zn61__ga62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_zn61__ga62);
    rate_eval.screened_rates(k_p_zn61__ga62) *= scor;
    rate_eval.dscreened_rates_dT(k_p_zn61__ga62) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_zn61__he4_cu58);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_zn61__he4_cu58);
    rate_eval.screened_rates(k_p_zn61__he4_cu58) *= scor;
    rate_eval.dscreened_rates_dT(k_p_zn61__he4_cu58) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 30.0_rt, 62.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_zn62__ga63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_zn62__ga63);
    rate_eval.screened_rates(k_p_zn62__ga63) *= scor;
    rate_eval.dscreened_rates_dT(k_p_zn62__ga63) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_zn62__n_ga62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_zn62__n_ga62);
    rate_eval.screened_rates(k_p_zn62__n_ga62) *= scor;
    rate_eval.dscreened_rates_dT(k_p_zn62__n_ga62) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_zn62__he4_cu59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_zn62__he4_cu59);
    rate_eval.screened_rates(k_p_zn62__he4_cu59) *= scor;
    rate_eval.dscreened_rates_dT(k_p_zn62__he4_cu59) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 30.0_rt, 63.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_zn63__ga64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_zn63__ga64);
    rate_eval.screened_rates(k_p_zn63__ga64) *= scor;
    rate_eval.dscreened_rates_dT(k_p_zn63__ga64) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_zn63__n_ga63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_zn63__n_ga63);
    rate_eval.screened_rates(k_p_zn63__n_ga63) *= scor;
    rate_eval.dscreened_rates_dT(k_p_zn63__n_ga63) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_zn63__he4_cu60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_zn63__he4_cu60);
    rate_eval.screened_rates(k_p_zn63__he4_cu60) *= scor;
    rate_eval.dscreened_rates_dT(k_p_zn63__he4_cu60) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 31.0_rt, 62.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ga62__ge63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ga62__ge63);
    rate_eval.screened_rates(k_p_ga62__ge63) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ga62__ge63) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ga62__he4_zn59);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ga62__he4_zn59);
    rate_eval.screened_rates(k_p_ga62__he4_zn59) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ga62__he4_zn59) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 31.0_rt, 63.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ga63__ge64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ga63__ge64);
    rate_eval.screened_rates(k_p_ga63__ge64) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ga63__ge64) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ga63__n_ge63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ga63__n_ge63);
    rate_eval.screened_rates(k_p_ga63__n_ge63) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ga63__n_ge63) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ga63__he4_zn60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ga63__he4_zn60);
    rate_eval.screened_rates(k_p_ga63__he4_zn60) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ga63__he4_zn60) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 2.0_rt, 3.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_d_he3__p_he4);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_d_he3__p_he4);
    rate_eval.screened_rates(k_d_he3__p_he4) *= scor;
    rate_eval.dscreened_rates_dT(k_d_he3__p_he4) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 2.0_rt, 4.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_he4__d_he3);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_he4__d_he3);
    rate_eval.screened_rates(k_p_he4__d_he3) *= scor;
    rate_eval.dscreened_rates_dT(k_p_he4__d_he3) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_n_p_he4__li6);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_n_p_he4__li6);
    rate_eval.screened_rates(k_n_p_he4__li6) *= scor;
    rate_eval.dscreened_rates_dT(k_n_p_he4__li6) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 2.0_rt, 4.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_he4__n_be7);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_he4__n_be7);
    rate_eval.screened_rates(k_he4_he4__n_be7) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_he4__n_be7) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_he4__p_li7);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_he4__p_li7);
    rate_eval.screened_rates(k_he4_he4__p_li7) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_he4__p_li7) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_n_he4_he4__be9);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_n_he4_he4__be9);
    rate_eval.screened_rates(k_n_he4_he4__be9) *= scor;
    rate_eval.dscreened_rates_dT(k_n_he4_he4__be9) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_n_he4_he4__d_li7);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_n_he4_he4__d_li7);
    rate_eval.screened_rates(k_n_he4_he4__d_li7) *= scor;
    rate_eval.dscreened_rates_dT(k_n_he4_he4__d_li7) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 3.0_rt, 6.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_d_li6__n_be7);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_d_li6__n_be7);
    rate_eval.screened_rates(k_d_li6__n_be7) *= scor;
    rate_eval.dscreened_rates_dT(k_d_li6__n_be7) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_d_li6__p_li7);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_d_li6__p_li7);
    rate_eval.screened_rates(k_d_li6__p_li7) *= scor;
    rate_eval.dscreened_rates_dT(k_d_li6__p_li7) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 3.0_rt, 7.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_li7__n_be7);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_li7__n_be7);
    rate_eval.screened_rates(k_p_li7__n_be7) *= scor;
    rate_eval.dscreened_rates_dT(k_p_li7__n_be7) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_li7__d_li6);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_li7__d_li6);
    rate_eval.screened_rates(k_p_li7__d_li6) *= scor;
    rate_eval.dscreened_rates_dT(k_p_li7__d_li6) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_li7__he4_he4);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_li7__he4_he4);
    rate_eval.screened_rates(k_p_li7__he4_he4) *= scor;
    rate_eval.dscreened_rates_dT(k_p_li7__he4_he4) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 4.0_rt, 7.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_be7__p_b10);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_be7__p_b10);
    rate_eval.screened_rates(k_he4_be7__p_b10) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_be7__p_b10) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 4.0_rt, 9.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_be9__n_c12);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_be9__n_c12);
    rate_eval.screened_rates(k_he4_be9__n_c12) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_be9__n_c12) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 5.0_rt, 10.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_b10__he4_be7);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_b10__he4_be7);
    rate_eval.screened_rates(k_p_b10__he4_be7) *= scor;
    rate_eval.dscreened_rates_dT(k_p_b10__he4_be7) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 5.0_rt, 10.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_b10__n_n13);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_b10__n_n13);
    rate_eval.screened_rates(k_he4_b10__n_n13) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_b10__n_n13) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_b10__p_c13);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_b10__p_c13);
    rate_eval.screened_rates(k_he4_b10__p_c13) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_b10__p_c13) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 5.0_rt, 11.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_b11__n_n14);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_b11__n_n14);
    rate_eval.screened_rates(k_he4_b11__n_n14) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_b11__n_n14) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_b11__p_c14);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_b11__p_c14);
    rate_eval.screened_rates(k_he4_b11__p_c14) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_b11__p_c14) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(6.0_rt, 12.0_rt, 6.0_rt, 12.0_rt);


        static_assert(scn_fac.z1 == 6.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_c12_c12__n_mg23);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_c12_c12__n_mg23);
    rate_eval.screened_rates(k_c12_c12__n_mg23) *= scor;
    rate_eval.dscreened_rates_dT(k_c12_c12__n_mg23) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_c12_c12__p_na23);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_c12_c12__p_na23);
    rate_eval.screened_rates(k_c12_c12__p_na23) *= scor;
    rate_eval.dscreened_rates_dT(k_c12_c12__p_na23) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_c12_c12__he4_ne20);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_c12_c12__he4_ne20);
    rate_eval.screened_rates(k_c12_c12__he4_ne20) *= scor;
    rate_eval.dscreened_rates_dT(k_c12_c12__he4_ne20) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 6.0_rt, 13.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_d_c13__n_n14);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_d_c13__n_n14);
    rate_eval.screened_rates(k_d_c13__n_n14) *= scor;
    rate_eval.dscreened_rates_dT(k_d_c13__n_n14) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 6.0_rt, 13.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_c13__n_o16);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_c13__n_o16);
    rate_eval.screened_rates(k_he4_c13__n_o16) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_c13__n_o16) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 6.0_rt, 14.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_d_c14__n_n15);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_d_c14__n_n15);
    rate_eval.screened_rates(k_d_c14__n_n15) *= scor;
    rate_eval.dscreened_rates_dT(k_d_c14__n_n15) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 7.0_rt, 13.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_n13__p_o16);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_n13__p_o16);
    rate_eval.screened_rates(k_he4_n13__p_o16) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_n13__p_o16) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(6.0_rt, 12.0_rt, 8.0_rt, 16.0_rt);


        static_assert(scn_fac.z1 == 6.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_c12_o16__p_al27);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_c12_o16__p_al27);
    rate_eval.screened_rates(k_c12_o16__p_al27) *= scor;
    rate_eval.dscreened_rates_dT(k_c12_o16__p_al27) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_c12_o16__he4_mg24);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_c12_o16__he4_mg24);
    rate_eval.screened_rates(k_c12_o16__he4_mg24) *= scor;
    rate_eval.dscreened_rates_dT(k_c12_o16__he4_mg24) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(8.0_rt, 16.0_rt, 8.0_rt, 16.0_rt);


        static_assert(scn_fac.z1 == 8.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_o16_o16__p_p31);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_o16_o16__p_p31);
    rate_eval.screened_rates(k_o16_o16__p_p31) *= scor;
    rate_eval.dscreened_rates_dT(k_o16_o16__p_p31) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_o16_o16__he4_si28);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_o16_o16__he4_si28);
    rate_eval.screened_rates(k_o16_o16__he4_si28) *= scor;
    rate_eval.dscreened_rates_dT(k_o16_o16__he4_si28) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 10.0_rt, 18.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ne18__p_na21);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ne18__p_na21);
    rate_eval.screened_rates(k_he4_ne18__p_na21) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ne18__p_na21) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(6.0_rt, 12.0_rt, 10.0_rt, 20.0_rt);


        static_assert(scn_fac.z1 == 6.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_c12_ne20__p_p31);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_c12_ne20__p_p31);
    rate_eval.screened_rates(k_c12_ne20__p_p31) *= scor;
    rate_eval.dscreened_rates_dT(k_c12_ne20__p_p31) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_c12_ne20__he4_si28);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_c12_ne20__he4_si28);
    rate_eval.screened_rates(k_c12_ne20__he4_si28) *= scor;
    rate_eval.dscreened_rates_dT(k_c12_ne20__he4_si28) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 11.0_rt, 21.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_na21__he4_ne18);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_na21__he4_ne18);
    rate_eval.screened_rates(k_p_na21__he4_ne18) *= scor;
    rate_eval.dscreened_rates_dT(k_p_na21__he4_ne18) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 12.0_rt, 23.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_mg23__p_al26);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_mg23__p_al26);
    rate_eval.screened_rates(k_he4_mg23__p_al26) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_mg23__p_al26) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 13.0_rt, 26.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_al26__he4_mg23);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_al26__he4_mg23);
    rate_eval.screened_rates(k_p_al26__he4_mg23) *= scor;
    rate_eval.dscreened_rates_dT(k_p_al26__he4_mg23) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 19.0_rt, 37.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_k37__p_ca40);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_k37__p_ca40);
    rate_eval.screened_rates(k_he4_k37__p_ca40) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_k37__p_ca40) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 19.0_rt, 38.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_k38__p_ca41);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_k38__p_ca41);
    rate_eval.screened_rates(k_he4_k38__p_ca41) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_k38__p_ca41) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 40.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ca40__he4_k37);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca40__he4_k37);
    rate_eval.screened_rates(k_p_ca40__he4_k37) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca40__he4_k37) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 41.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ca41__he4_k38);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ca41__he4_k38);
    rate_eval.screened_rates(k_p_ca41__he4_k38) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ca41__he4_k38) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ca48__n_ti51);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ca48__n_ti51);
    rate_eval.screened_rates(k_he4_ca48__n_ti51) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ca48__n_ti51) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 21.0_rt, 49.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_sc49__n_v52);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_sc49__n_v52);
    rate_eval.screened_rates(k_he4_sc49__n_v52) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_sc49__n_v52) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 22.0_rt, 51.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ti51__n_cr54);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ti51__n_cr54);
    rate_eval.screened_rates(k_he4_ti51__n_cr54) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ti51__n_cr54) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 23.0_rt, 52.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_v52__n_mn55);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_v52__n_mn55);
    rate_eval.screened_rates(k_he4_v52__n_mn55) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_v52__n_mn55) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 28.0_rt, 63.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_ni63__n_zn66);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_ni63__n_zn66);
    rate_eval.screened_rates(k_he4_ni63__n_zn66) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_ni63__n_zn66) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 29.0_rt, 57.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cu57__p_zn60);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cu57__p_zn60);
    rate_eval.screened_rates(k_he4_cu57__p_zn60) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cu57__p_zn60) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 29.0_rt, 61.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cu61__n_ga64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cu61__n_ga64);
    rate_eval.screened_rates(k_he4_cu61__n_ga64) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cu61__n_ga64) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_cu61__p_zn64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cu61__p_zn64);
    rate_eval.screened_rates(k_he4_cu61__p_zn64) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cu61__p_zn64) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 29.0_rt, 62.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cu62__p_zn65);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cu62__p_zn65);
    rate_eval.screened_rates(k_he4_cu62__p_zn65) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cu62__p_zn65) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 29.0_rt, 63.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_cu63__p_zn66);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_cu63__p_zn66);
    rate_eval.screened_rates(k_he4_cu63__p_zn66) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_cu63__p_zn66) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 30.0_rt, 60.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_zn60__he4_cu57);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_zn60__he4_cu57);
    rate_eval.screened_rates(k_p_zn60__he4_cu57) *= scor;
    rate_eval.dscreened_rates_dT(k_p_zn60__he4_cu57) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 30.0_rt, 61.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he4_zn61__n_ge64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_zn61__n_ge64);
    rate_eval.screened_rates(k_he4_zn61__n_ge64) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_zn61__n_ge64) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_he4_zn61__p_ga64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_zn61__p_ga64);
    rate_eval.screened_rates(k_he4_zn61__p_ga64) *= scor;
    rate_eval.dscreened_rates_dT(k_he4_zn61__p_ga64) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 30.0_rt, 64.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_zn64__n_ga64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_zn64__n_ga64);
    rate_eval.screened_rates(k_p_zn64__n_ga64) *= scor;
    rate_eval.dscreened_rates_dT(k_p_zn64__n_ga64) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_zn64__he4_cu61);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_zn64__he4_cu61);
    rate_eval.screened_rates(k_p_zn64__he4_cu61) *= scor;
    rate_eval.dscreened_rates_dT(k_p_zn64__he4_cu61) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 30.0_rt, 65.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_zn65__he4_cu62);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_zn65__he4_cu62);
    rate_eval.screened_rates(k_p_zn65__he4_cu62) *= scor;
    rate_eval.dscreened_rates_dT(k_p_zn65__he4_cu62) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 30.0_rt, 66.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_zn66__he4_cu63);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_zn66__he4_cu63);
    rate_eval.screened_rates(k_p_zn66__he4_cu63) *= scor;
    rate_eval.dscreened_rates_dT(k_p_zn66__he4_cu63) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 31.0_rt, 64.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_ga64__n_ge64);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ga64__n_ge64);
    rate_eval.screened_rates(k_p_ga64__n_ge64) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ga64__n_ge64) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_ga64__he4_zn61);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_ga64__he4_zn61);
    rate_eval.screened_rates(k_p_ga64__he4_zn61) *= scor;
    rate_eval.dscreened_rates_dT(k_p_ga64__he4_zn61) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 3.0_rt, 2.0_rt, 3.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he3_he3__p_p_he4);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he3_he3__p_p_he4);
    rate_eval.screened_rates(k_he3_he3__p_p_he4) *= scor;
    rate_eval.dscreened_rates_dT(k_he3_he3__p_p_he4) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 3.0_rt, 7.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_d_li7__n_he4_he4);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_d_li7__n_he4_he4);
    rate_eval.screened_rates(k_d_li7__n_he4_he4) *= scor;
    rate_eval.dscreened_rates_dT(k_d_li7__n_he4_he4) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 4.0_rt, 7.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_d_be7__p_he4_he4);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_d_be7__p_he4_he4);
    rate_eval.screened_rates(k_d_be7__p_he4_he4) *= scor;
    rate_eval.dscreened_rates_dT(k_d_be7__p_he4_he4) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 3.0_rt, 3.0_rt, 7.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he3_li7__n_p_he4_he4);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he3_li7__n_p_he4_he4);
    rate_eval.screened_rates(k_he3_li7__n_p_he4_he4) *= scor;
    rate_eval.dscreened_rates_dT(k_he3_li7__n_p_he4_he4) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 3.0_rt, 4.0_rt, 7.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_he3_be7__p_p_he4_he4);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he3_be7__p_p_he4_he4);
    rate_eval.screened_rates(k_he3_be7__p_p_he4_he4) *= scor;
    rate_eval.dscreened_rates_dT(k_he3_be7__p_p_he4_he4) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 2.0_rt, 4.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }



    {
        constexpr auto scn_fac2 = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 4.0_rt, 8.0_rt);


        static_assert(scn_fac2.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac2, scor2, dscor2_dt);

    }


    ratraw = rate_eval.screened_rates(k_he4_he4_he4__c12);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_he4_he4__c12);
    rate_eval.screened_rates(k_he4_he4_he4__c12) *= scor * scor2;
    rate_eval.dscreened_rates_dT(k_he4_he4_he4__c12) = ratraw * (scor * dscor2_dt + dscor_dt * scor2) + dratraw_dT * scor * scor2;

    ratraw = rate_eval.screened_rates(k_he4_he4_he4__p_b11);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_he4_he4_he4__p_b11);
    rate_eval.screened_rates(k_he4_he4_he4__p_b11) *= scor * scor2;
    rate_eval.dscreened_rates_dT(k_he4_he4_he4__p_b11) = ratraw * (scor * dscor2_dt + dscor_dt * scor2) + dratraw_dT * scor * scor2;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 1.0_rt, 1.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_p_he4__he3_he3);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p_he4__he3_he3);
    rate_eval.screened_rates(k_p_p_he4__he3_he3) *= scor;
    rate_eval.dscreened_rates_dT(k_p_p_he4__he3_he3) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 2.0_rt, 4.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_he4_he4__n_b8);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_he4_he4__n_b8);
    rate_eval.screened_rates(k_p_he4_he4__n_b8) *= scor;
    rate_eval.dscreened_rates_dT(k_p_he4_he4__n_b8) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_p_he4_he4__d_be7);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_p_he4_he4__d_be7);
    rate_eval.screened_rates(k_p_he4_he4__d_be7) *= scor;
    rate_eval.dscreened_rates_dT(k_p_he4_he4__d_be7) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_n_p_he4_he4__he3_li7);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_n_p_he4_he4__he3_li7);
    rate_eval.screened_rates(k_n_p_he4_he4__he3_li7) *= scor;
    rate_eval.dscreened_rates_dT(k_n_p_he4_he4__he3_li7) = ratraw * dscor_dt + dratraw_dT * scor;

    ratraw = rate_eval.screened_rates(k_n_p_he4_he4__p_be9);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_n_p_he4_he4__p_be9);
    rate_eval.screened_rates(k_n_p_he4_he4__p_be9) *= scor;
    rate_eval.dscreened_rates_dT(k_n_p_he4_he4__p_be9) = ratraw * dscor_dt + dratraw_dT * scor;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 2.0_rt, 4.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_d_he4_he4__p_be9);
    dratraw_dT = rate_eval.dscreened_rates_dT(k_d_he4_he4__p_be9);
    rate_eval.screened_rates(k_d_he4_he4__p_be9) *= scor;
    rate_eval.dscreened_rates_dT(k_d_he4_he4__p_be9) = ratraw * dscor_dt + dratraw_dT * scor;


    // Calculate tabular rates

    Real rate, drate_dt, edot_nu;


}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rhs_nuc(const burn_t& state,
             Array1D<Real, 1, neqs>& ydot_nuc,
             const Array1D<Real, 1, NumSpec>& Y,
             const Array1D<Real, 1, NumRates>& screened_rates) {

    using namespace Rates;

    ydot_nuc(N) =
        -screened_rates(k_n__p__weak__wc12)*Y(N) +
        (-screened_rates(k_n_p__d)*Y(N)*Y(H1)*state.rho + screened_rates(k_d__n_p)*Y(H2)) +
        (-screened_rates(k_n_he3__he4)*Y(He3)*Y(N)*state.rho + screened_rates(k_he4__n_he3)*Y(He4)) +
        (-screened_rates(k_n_li6__li7)*Y(Li6)*Y(N)*state.rho + screened_rates(k_li7__n_li6)*Y(Li7)) +
        (-screened_rates(k_n_b10__b11)*Y(B10)*Y(N)*state.rho + screened_rates(k_b11__n_b10)*Y(B11)) +
        (-screened_rates(k_n_c12__c13)*Y(C12)*Y(N)*state.rho + screened_rates(k_c13__n_c12)*Y(C13)) +
        (-screened_rates(k_n_c13__c14)*Y(C13)*Y(N)*state.rho + screened_rates(k_c14__n_c13)*Y(C14)) +
        (-screened_rates(k_n_n13__n14)*Y(N13)*Y(N)*state.rho + screened_rates(k_n14__n_n13)*Y(N14)) +
        (-screened_rates(k_n_n14__n15)*Y(N14)*Y(N)*state.rho + screened_rates(k_n15__n_n14)*Y(N15)) +
        (-screened_rates(k_n_o14__o15)*Y(N)*Y(O14)*state.rho + screened_rates(k_o15__n_o14)*Y(O15)) +
        (-screened_rates(k_n_o15__o16)*Y(N)*Y(O15)*state.rho + screened_rates(k_o16__n_o15)*Y(O16)) +
        (-screened_rates(k_n_o16__o17)*Y(N)*Y(O16)*state.rho + screened_rates(k_o17__n_o16)*Y(O17)) +
        (-screened_rates(k_n_o17__o18)*Y(N)*Y(O17)*state.rho + screened_rates(k_o18__n_o17)*Y(O18)) +
        (-screened_rates(k_n_f17__f18)*Y(F17)*Y(N)*state.rho + screened_rates(k_f18__n_f17)*Y(F18)) +
        (-screened_rates(k_n_f18__f19)*Y(F18)*Y(N)*state.rho + screened_rates(k_f19__n_f18)*Y(F19)) +
        (-screened_rates(k_n_ne18__ne19)*Y(N)*Y(Ne18)*state.rho + screened_rates(k_ne19__n_ne18)*Y(Ne19)) +
        (-screened_rates(k_n_ne19__ne20)*Y(N)*Y(Ne19)*state.rho + screened_rates(k_ne20__n_ne19)*Y(Ne20)) +
        (-screened_rates(k_n_ne20__ne21)*Y(N)*Y(Ne20)*state.rho + screened_rates(k_ne21__n_ne20)*Y(Ne21)) +
        (-screened_rates(k_n_ne21__ne22)*Y(N)*Y(Ne21)*state.rho + screened_rates(k_ne22__n_ne21)*Y(Ne22)) +
        (-screened_rates(k_n_na21__na22)*Y(N)*Y(Na21)*state.rho + screened_rates(k_na22__n_na21)*Y(Na22)) +
        (-screened_rates(k_n_na22__na23)*Y(N)*Y(Na22)*state.rho + screened_rates(k_na23__n_na22)*Y(Na23)) +
        (-screened_rates(k_n_mg23__mg24)*Y(Mg23)*Y(N)*state.rho + screened_rates(k_mg24__n_mg23)*Y(Mg24)) +
        (-screened_rates(k_n_mg24__mg25)*Y(Mg24)*Y(N)*state.rho + screened_rates(k_mg25__n_mg24)*Y(Mg25)) +
        (-screened_rates(k_n_mg25__mg26)*Y(Mg25)*Y(N)*state.rho + screened_rates(k_mg26__n_mg25)*Y(Mg26)) +
        (-screened_rates(k_n_al25__al26)*Y(Al25)*Y(N)*state.rho + screened_rates(k_al26__n_al25)*Y(Al26)) +
        (-screened_rates(k_n_al26__al27)*Y(Al26)*Y(N)*state.rho + screened_rates(k_al27__n_al26)*Y(Al27)) +
        (-screened_rates(k_n_si28__si29)*Y(N)*Y(Si28)*state.rho + screened_rates(k_si29__n_si28)*Y(Si29)) +
        (-screened_rates(k_n_si29__si30)*Y(N)*Y(Si29)*state.rho + screened_rates(k_si30__n_si29)*Y(Si30)) +
        (-screened_rates(k_n_si30__si31)*Y(N)*Y(Si30)*state.rho + screened_rates(k_si31__n_si30)*Y(Si31)) +
        (-screened_rates(k_n_si31__si32)*Y(N)*Y(Si31)*state.rho + screened_rates(k_si32__n_si31)*Y(Si32)) +
        (-screened_rates(k_n_p29__p30)*Y(N)*Y(P29)*state.rho + screened_rates(k_p30__n_p29)*Y(P30)) +
        (-screened_rates(k_n_p30__p31)*Y(N)*Y(P30)*state.rho + screened_rates(k_p31__n_p30)*Y(P31)) +
        (-screened_rates(k_n_p31__p32)*Y(N)*Y(P31)*state.rho + screened_rates(k_p32__n_p31)*Y(P32)) +
        (-screened_rates(k_n_p32__p33)*Y(N)*Y(P32)*state.rho + screened_rates(k_p33__n_p32)*Y(P33)) +
        (-screened_rates(k_n_s32__s33)*Y(N)*Y(S32)*state.rho + screened_rates(k_s33__n_s32)*Y(S33)) +
        (-screened_rates(k_n_s33__s34)*Y(N)*Y(S33)*state.rho + screened_rates(k_s34__n_s33)*Y(S34)) +
        (-screened_rates(k_n_s34__s35)*Y(N)*Y(S34)*state.rho + screened_rates(k_s35__n_s34)*Y(S35)) +
        (-screened_rates(k_n_s35__s36)*Y(N)*Y(S35)*state.rho + screened_rates(k_s36__n_s35)*Y(S36)) +
        (-screened_rates(k_n_cl33__cl34)*Y(Cl33)*Y(N)*state.rho + screened_rates(k_cl34__n_cl33)*Y(Cl34)) +
        (-screened_rates(k_n_cl34__cl35)*Y(Cl34)*Y(N)*state.rho + screened_rates(k_cl35__n_cl34)*Y(Cl35)) +
        (-screened_rates(k_n_cl35__cl36)*Y(Cl35)*Y(N)*state.rho + screened_rates(k_cl36__n_cl35)*Y(Cl36)) +
        (-screened_rates(k_n_cl36__cl37)*Y(Cl36)*Y(N)*state.rho + screened_rates(k_cl37__n_cl36)*Y(Cl37)) +
        (-screened_rates(k_n_ar36__ar37)*Y(Ar36)*Y(N)*state.rho + screened_rates(k_ar37__n_ar36)*Y(Ar37)) +
        (-screened_rates(k_n_ar37__ar38)*Y(Ar37)*Y(N)*state.rho + screened_rates(k_ar38__n_ar37)*Y(Ar38)) +
        (-screened_rates(k_n_ar38__ar39)*Y(Ar38)*Y(N)*state.rho + screened_rates(k_ar39__n_ar38)*Y(Ar39)) +
        (-screened_rates(k_n_ar39__ar40)*Y(Ar39)*Y(N)*state.rho + screened_rates(k_ar40__n_ar39)*Y(Ar40)) +
        (-screened_rates(k_n_k37__k38)*Y(K37)*Y(N)*state.rho + screened_rates(k_k38__n_k37)*Y(K38)) +
        (-screened_rates(k_n_k38__k39)*Y(K38)*Y(N)*state.rho + screened_rates(k_k39__n_k38)*Y(K39)) +
        (-screened_rates(k_n_k39__k40)*Y(K39)*Y(N)*state.rho + screened_rates(k_k40__n_k39)*Y(K40)) +
        (-screened_rates(k_n_k40__k41)*Y(K40)*Y(N)*state.rho + screened_rates(k_k41__n_k40)*Y(K41)) +
        (-screened_rates(k_n_ca40__ca41)*Y(Ca40)*Y(N)*state.rho + screened_rates(k_ca41__n_ca40)*Y(Ca41)) +
        (-screened_rates(k_n_ca41__ca42)*Y(Ca41)*Y(N)*state.rho + screened_rates(k_ca42__n_ca41)*Y(Ca42)) +
        (-screened_rates(k_n_ca42__ca43)*Y(Ca42)*Y(N)*state.rho + screened_rates(k_ca43__n_ca42)*Y(Ca43)) +
        (-screened_rates(k_n_ca43__ca44)*Y(Ca43)*Y(N)*state.rho + screened_rates(k_ca44__n_ca43)*Y(Ca44)) +
        (-screened_rates(k_n_ca44__ca45)*Y(Ca44)*Y(N)*state.rho + screened_rates(k_ca45__n_ca44)*Y(Ca45)) +
        (-screened_rates(k_n_ca45__ca46)*Y(Ca45)*Y(N)*state.rho + screened_rates(k_ca46__n_ca45)*Y(Ca46)) +
        (-screened_rates(k_n_ca46__ca47)*Y(Ca46)*Y(N)*state.rho + screened_rates(k_ca47__n_ca46)*Y(Ca47)) +
        (-screened_rates(k_n_ca47__ca48)*Y(Ca47)*Y(N)*state.rho + screened_rates(k_ca48__n_ca47)*Y(Ca48)) +
        (-screened_rates(k_n_sc43__sc44)*Y(N)*Y(Sc43)*state.rho + screened_rates(k_sc44__n_sc43)*Y(Sc44)) +
        (-screened_rates(k_n_sc44__sc45)*Y(N)*Y(Sc44)*state.rho + screened_rates(k_sc45__n_sc44)*Y(Sc45)) +
        (-screened_rates(k_n_sc45__sc46)*Y(N)*Y(Sc45)*state.rho + screened_rates(k_sc46__n_sc45)*Y(Sc46)) +
        (-screened_rates(k_n_sc46__sc47)*Y(N)*Y(Sc46)*state.rho + screened_rates(k_sc47__n_sc46)*Y(Sc47)) +
        (-screened_rates(k_n_sc47__sc48)*Y(N)*Y(Sc47)*state.rho + screened_rates(k_sc48__n_sc47)*Y(Sc48)) +
        (-screened_rates(k_n_sc48__sc49)*Y(N)*Y(Sc48)*state.rho + screened_rates(k_sc49__n_sc48)*Y(Sc49)) +
        (-screened_rates(k_n_ti44__ti45)*Y(N)*Y(Ti44)*state.rho + screened_rates(k_ti45__n_ti44)*Y(Ti45)) +
        (-screened_rates(k_n_ti45__ti46)*Y(N)*Y(Ti45)*state.rho + screened_rates(k_ti46__n_ti45)*Y(Ti46)) +
        (-screened_rates(k_n_ti46__ti47)*Y(N)*Y(Ti46)*state.rho + screened_rates(k_ti47__n_ti46)*Y(Ti47)) +
        (-screened_rates(k_n_ti47__ti48)*Y(N)*Y(Ti47)*state.rho + screened_rates(k_ti48__n_ti47)*Y(Ti48)) +
        (-screened_rates(k_n_ti48__ti49)*Y(N)*Y(Ti48)*state.rho + screened_rates(k_ti49__n_ti48)*Y(Ti49)) +
        (-screened_rates(k_n_ti49__ti50)*Y(N)*Y(Ti49)*state.rho + screened_rates(k_ti50__n_ti49)*Y(Ti50)) +
        (-screened_rates(k_n_ti50__ti51)*Y(N)*Y(Ti50)*state.rho + screened_rates(k_ti51__n_ti50)*Y(Ti51)) +
        (-screened_rates(k_n_v46__v47)*Y(N)*Y(V46)*state.rho + screened_rates(k_v47__n_v46)*Y(V47)) +
        (-screened_rates(k_n_v47__v48)*Y(N)*Y(V47)*state.rho + screened_rates(k_v48__n_v47)*Y(V48)) +
        (-screened_rates(k_n_v48__v49)*Y(N)*Y(V48)*state.rho + screened_rates(k_v49__n_v48)*Y(V49)) +
        (-screened_rates(k_n_v49__v50)*Y(N)*Y(V49)*state.rho + screened_rates(k_v50__n_v49)*Y(V50)) +
        (-screened_rates(k_n_v50__v51)*Y(N)*Y(V50)*state.rho + screened_rates(k_v51__n_v50)*Y(V51)) +
        (-screened_rates(k_n_v51__v52)*Y(N)*Y(V51)*state.rho + screened_rates(k_v52__n_v51)*Y(V52)) +
        (-screened_rates(k_n_cr48__cr49)*Y(Cr48)*Y(N)*state.rho + screened_rates(k_cr49__n_cr48)*Y(Cr49)) +
        (-screened_rates(k_n_cr49__cr50)*Y(Cr49)*Y(N)*state.rho + screened_rates(k_cr50__n_cr49)*Y(Cr50)) +
        (-screened_rates(k_n_cr50__cr51)*Y(Cr50)*Y(N)*state.rho + screened_rates(k_cr51__n_cr50)*Y(Cr51)) +
        (-screened_rates(k_n_cr51__cr52)*Y(Cr51)*Y(N)*state.rho + screened_rates(k_cr52__n_cr51)*Y(Cr52)) +
        (-screened_rates(k_n_cr52__cr53)*Y(Cr52)*Y(N)*state.rho + screened_rates(k_cr53__n_cr52)*Y(Cr53)) +
        (-screened_rates(k_n_cr53__cr54)*Y(Cr53)*Y(N)*state.rho + screened_rates(k_cr54__n_cr53)*Y(Cr54)) +
        (-screened_rates(k_n_mn50__mn51)*Y(Mn50)*Y(N)*state.rho + screened_rates(k_mn51__n_mn50)*Y(Mn51)) +
        (-screened_rates(k_n_mn51__mn52)*Y(Mn51)*Y(N)*state.rho + screened_rates(k_mn52__n_mn51)*Y(Mn52)) +
        (-screened_rates(k_n_mn52__mn53)*Y(Mn52)*Y(N)*state.rho + screened_rates(k_mn53__n_mn52)*Y(Mn53)) +
        (-screened_rates(k_n_mn53__mn54)*Y(Mn53)*Y(N)*state.rho + screened_rates(k_mn54__n_mn53)*Y(Mn54)) +
        (-screened_rates(k_n_mn54__mn55)*Y(Mn54)*Y(N)*state.rho + screened_rates(k_mn55__n_mn54)*Y(Mn55)) +
        (-screened_rates(k_n_fe52__fe53)*Y(Fe52)*Y(N)*state.rho + screened_rates(k_fe53__n_fe52)*Y(Fe53)) +
        (-screened_rates(k_n_fe53__fe54)*Y(Fe53)*Y(N)*state.rho + screened_rates(k_fe54__n_fe53)*Y(Fe54)) +
        (-screened_rates(k_n_fe54__fe55)*Y(Fe54)*Y(N)*state.rho + screened_rates(k_fe55__n_fe54)*Y(Fe55)) +
        (-screened_rates(k_n_fe55__fe56)*Y(Fe55)*Y(N)*state.rho + screened_rates(k_fe56__n_fe55)*Y(Fe56)) +
        (-screened_rates(k_n_fe56__fe57)*Y(Fe56)*Y(N)*state.rho + screened_rates(k_fe57__n_fe56)*Y(Fe57)) +
        (-screened_rates(k_n_fe57__fe58)*Y(Fe57)*Y(N)*state.rho + screened_rates(k_fe58__n_fe57)*Y(Fe58)) +
        (-screened_rates(k_n_co53__co54)*Y(Co53)*Y(N)*state.rho + screened_rates(k_co54__n_co53)*Y(Co54)) +
        (-screened_rates(k_n_co54__co55)*Y(Co54)*Y(N)*state.rho + screened_rates(k_co55__n_co54)*Y(Co55)) +
        (-screened_rates(k_n_co55__co56)*Y(Co55)*Y(N)*state.rho + screened_rates(k_co56__n_co55)*Y(Co56)) +
        (-screened_rates(k_n_co56__co57)*Y(Co56)*Y(N)*state.rho + screened_rates(k_co57__n_co56)*Y(Co57)) +
        (-screened_rates(k_n_co57__co58)*Y(Co57)*Y(N)*state.rho + screened_rates(k_co58__n_co57)*Y(Co58)) +
        (-screened_rates(k_n_co58__co59)*Y(Co58)*Y(N)*state.rho + screened_rates(k_co59__n_co58)*Y(Co59)) +
        (-screened_rates(k_n_ni56__ni57)*Y(N)*Y(Ni56)*state.rho + screened_rates(k_ni57__n_ni56)*Y(Ni57)) +
        (-screened_rates(k_n_ni57__ni58)*Y(N)*Y(Ni57)*state.rho + screened_rates(k_ni58__n_ni57)*Y(Ni58)) +
        (-screened_rates(k_n_ni58__ni59)*Y(N)*Y(Ni58)*state.rho + screened_rates(k_ni59__n_ni58)*Y(Ni59)) +
        (-screened_rates(k_n_ni59__ni60)*Y(N)*Y(Ni59)*state.rho + screened_rates(k_ni60__n_ni59)*Y(Ni60)) +
        (-screened_rates(k_n_ni60__ni61)*Y(N)*Y(Ni60)*state.rho + screened_rates(k_ni61__n_ni60)*Y(Ni61)) +
        (-screened_rates(k_n_ni61__ni62)*Y(N)*Y(Ni61)*state.rho + screened_rates(k_ni62__n_ni61)*Y(Ni62)) +
        (-screened_rates(k_n_ni62__ni63)*Y(N)*Y(Ni62)*state.rho + screened_rates(k_ni63__n_ni62)*Y(Ni63)) +
        (-screened_rates(k_n_ni63__ni64)*Y(N)*Y(Ni63)*state.rho + screened_rates(k_ni64__n_ni63)*Y(Ni64)) +
        (-screened_rates(k_n_cu57__cu58)*Y(Cu57)*Y(N)*state.rho + screened_rates(k_cu58__n_cu57)*Y(Cu58)) +
        (-screened_rates(k_n_cu58__cu59)*Y(Cu58)*Y(N)*state.rho + screened_rates(k_cu59__n_cu58)*Y(Cu59)) +
        (-screened_rates(k_n_cu59__cu60)*Y(Cu59)*Y(N)*state.rho + screened_rates(k_cu60__n_cu59)*Y(Cu60)) +
        (-screened_rates(k_n_cu60__cu61)*Y(Cu60)*Y(N)*state.rho + screened_rates(k_cu61__n_cu60)*Y(Cu61)) +
        (-screened_rates(k_n_cu61__cu62)*Y(Cu61)*Y(N)*state.rho + screened_rates(k_cu62__n_cu61)*Y(Cu62)) +
        (-screened_rates(k_n_cu62__cu63)*Y(Cu62)*Y(N)*state.rho + screened_rates(k_cu63__n_cu62)*Y(Cu63)) +
        (-screened_rates(k_n_cu63__cu64)*Y(Cu63)*Y(N)*state.rho + screened_rates(k_cu64__n_cu63)*Y(Cu64)) +
        (-screened_rates(k_n_cu64__cu65)*Y(Cu64)*Y(N)*state.rho + screened_rates(k_cu65__n_cu64)*Y(Cu65)) +
        (-screened_rates(k_n_zn59__zn60)*Y(N)*Y(Zn59)*state.rho + screened_rates(k_zn60__n_zn59)*Y(Zn60)) +
        (-screened_rates(k_n_zn60__zn61)*Y(N)*Y(Zn60)*state.rho + screened_rates(k_zn61__n_zn60)*Y(Zn61)) +
        (-screened_rates(k_n_zn61__zn62)*Y(N)*Y(Zn61)*state.rho + screened_rates(k_zn62__n_zn61)*Y(Zn62)) +
        (-screened_rates(k_n_zn62__zn63)*Y(N)*Y(Zn62)*state.rho + screened_rates(k_zn63__n_zn62)*Y(Zn63)) +
        (-screened_rates(k_n_zn63__zn64)*Y(N)*Y(Zn63)*state.rho + screened_rates(k_zn64__n_zn63)*Y(Zn64)) +
        (-screened_rates(k_n_zn64__zn65)*Y(N)*Y(Zn64)*state.rho + screened_rates(k_zn65__n_zn64)*Y(Zn65)) +
        (-screened_rates(k_n_zn65__zn66)*Y(N)*Y(Zn65)*state.rho + screened_rates(k_zn66__n_zn65)*Y(Zn66)) +
        (-screened_rates(k_n_ga62__ga63)*Y(Ga62)*Y(N)*state.rho + screened_rates(k_ga63__n_ga62)*Y(Ga63)) +
        (-screened_rates(k_n_ga63__ga64)*Y(Ga63)*Y(N)*state.rho + screened_rates(k_ga64__n_ga63)*Y(Ga64)) +
        (-screened_rates(k_n_ge63__ge64)*Y(Ge63)*Y(N)*state.rho + screened_rates(k_ge64__n_ge63)*Y(Ge64)) +
        (0.5*screened_rates(k_d_d__n_he3)*std::pow(Y(H2), 2)*state.rho + -screened_rates(k_n_he3__d_d)*Y(He3)*Y(N)*state.rho) +
        (screened_rates(k_d_li6__n_be7)*Y(H2)*Y(Li6)*state.rho + -screened_rates(k_n_be7__d_li6)*Y(Be7)*Y(N)*state.rho) +
        (-screened_rates(k_n_be7__p_li7)*Y(Be7)*Y(N)*state.rho + screened_rates(k_p_li7__n_be7)*Y(Li7)*Y(H1)*state.rho) +
        (-screened_rates(k_n_be7__he4_he4)*Y(Be7)*Y(N)*state.rho + 0.5*screened_rates(k_he4_he4__n_be7)*std::pow(Y(He4), 2)*state.rho) +
        (screened_rates(k_he4_be9__n_c12)*Y(Be9)*Y(He4)*state.rho + -screened_rates(k_n_c12__he4_be9)*Y(C12)*Y(N)*state.rho) +
        (-screened_rates(k_n_b10__he4_li7)*Y(B10)*Y(N)*state.rho + screened_rates(k_he4_li7__n_b10)*Y(He4)*Y(Li7)*state.rho) +
        (screened_rates(k_he4_b10__n_n13)*Y(B10)*Y(He4)*state.rho + -screened_rates(k_n_n13__he4_b10)*Y(N13)*Y(N)*state.rho) +
        (screened_rates(k_he4_b11__n_n14)*Y(B11)*Y(He4)*state.rho + -screened_rates(k_n_n14__he4_b11)*Y(N14)*Y(N)*state.rho) +
        (screened_rates(k_d_c13__n_n14)*Y(C13)*Y(H2)*state.rho + -screened_rates(k_n_n14__d_c13)*Y(N14)*Y(N)*state.rho) +
        (screened_rates(k_he4_c13__n_o16)*Y(C13)*Y(He4)*state.rho + -screened_rates(k_n_o16__he4_c13)*Y(N)*Y(O16)*state.rho) +
        (screened_rates(k_d_c14__n_n15)*Y(C14)*Y(H2)*state.rho + -screened_rates(k_n_n15__d_c14)*Y(N15)*Y(N)*state.rho) +
        (-screened_rates(k_n_n13__p_c13)*Y(N13)*Y(N)*state.rho + screened_rates(k_p_c13__n_n13)*Y(C13)*Y(H1)*state.rho) +
        (-screened_rates(k_n_n14__p_c14)*Y(N14)*Y(N)*state.rho + screened_rates(k_p_c14__n_n14)*Y(C14)*Y(H1)*state.rho) +
        (-screened_rates(k_n_o14__p_n14)*Y(N)*Y(O14)*state.rho + screened_rates(k_p_n14__n_o14)*Y(N14)*Y(H1)*state.rho) +
        (-screened_rates(k_n_o15__p_n15)*Y(N)*Y(O15)*state.rho + screened_rates(k_p_n15__n_o15)*Y(N15)*Y(H1)*state.rho) +
        (-screened_rates(k_n_o15__he4_c12)*Y(N)*Y(O15)*state.rho + screened_rates(k_he4_c12__n_o15)*Y(C12)*Y(He4)*state.rho) +
        (-screened_rates(k_n_o17__he4_c14)*Y(N)*Y(O17)*state.rho + screened_rates(k_he4_c14__n_o17)*Y(C14)*Y(He4)*state.rho) +
        (screened_rates(k_he4_o17__n_ne20)*Y(He4)*Y(O17)*state.rho + -screened_rates(k_n_ne20__he4_o17)*Y(N)*Y(Ne20)*state.rho) +
        (-screened_rates(k_n_f17__p_o17)*Y(F17)*Y(N)*state.rho + screened_rates(k_p_o17__n_f17)*Y(O17)*Y(H1)*state.rho) +
        (-screened_rates(k_n_f17__he4_n14)*Y(F17)*Y(N)*state.rho + screened_rates(k_he4_n14__n_f17)*Y(He4)*Y(N14)*state.rho) +
        (-screened_rates(k_n_f18__p_o18)*Y(F18)*Y(N)*state.rho + screened_rates(k_p_o18__n_f18)*Y(O18)*Y(H1)*state.rho) +
        (-screened_rates(k_n_f18__he4_n15)*Y(F18)*Y(N)*state.rho + screened_rates(k_he4_n15__n_f18)*Y(He4)*Y(N15)*state.rho) +
        (-screened_rates(k_n_ne18__p_f18)*Y(N)*Y(Ne18)*state.rho + screened_rates(k_p_f18__n_ne18)*Y(F18)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ne18__he4_o15)*Y(N)*Y(Ne18)*state.rho + screened_rates(k_he4_o15__n_ne18)*Y(He4)*Y(O15)*state.rho) +
        (-screened_rates(k_n_ne19__p_f19)*Y(N)*Y(Ne19)*state.rho + screened_rates(k_p_f19__n_ne19)*Y(F19)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ne19__he4_o16)*Y(N)*Y(Ne19)*state.rho + screened_rates(k_he4_o16__n_ne19)*Y(He4)*Y(O16)*state.rho) +
        (-screened_rates(k_n_ne21__he4_o18)*Y(N)*Y(Ne21)*state.rho + screened_rates(k_he4_o18__n_ne21)*Y(He4)*Y(O18)*state.rho) +
        (screened_rates(k_he4_ne21__n_mg24)*Y(He4)*Y(Ne21)*state.rho + -screened_rates(k_n_mg24__he4_ne21)*Y(Mg24)*Y(N)*state.rho) +
        (-screened_rates(k_n_na21__p_ne21)*Y(N)*Y(Na21)*state.rho + screened_rates(k_p_ne21__n_na21)*Y(Ne21)*Y(H1)*state.rho) +
        (-screened_rates(k_n_na21__he4_f18)*Y(N)*Y(Na21)*state.rho + screened_rates(k_he4_f18__n_na21)*Y(F18)*Y(He4)*state.rho) +
        (-screened_rates(k_n_na22__p_ne22)*Y(N)*Y(Na22)*state.rho + screened_rates(k_p_ne22__n_na22)*Y(Ne22)*Y(H1)*state.rho) +
        (-screened_rates(k_n_na22__he4_f19)*Y(N)*Y(Na22)*state.rho + screened_rates(k_he4_f19__n_na22)*Y(F19)*Y(He4)*state.rho) +
        (-screened_rates(k_n_mg23__p_na23)*Y(Mg23)*Y(N)*state.rho + screened_rates(k_p_na23__n_mg23)*Y(Na23)*Y(H1)*state.rho) +
        (-screened_rates(k_n_mg23__he4_ne20)*Y(Mg23)*Y(N)*state.rho + screened_rates(k_he4_ne20__n_mg23)*Y(He4)*Y(Ne20)*state.rho) +
        (-screened_rates(k_n_mg23__c12_c12)*Y(Mg23)*Y(N)*state.rho + 0.5*screened_rates(k_c12_c12__n_mg23)*std::pow(Y(C12), 2)*state.rho) +
        (-screened_rates(k_n_mg25__he4_ne22)*Y(Mg25)*Y(N)*state.rho + screened_rates(k_he4_ne22__n_mg25)*Y(He4)*Y(Ne22)*state.rho) +
        (screened_rates(k_he4_mg25__n_si28)*Y(He4)*Y(Mg25)*state.rho + -screened_rates(k_n_si28__he4_mg25)*Y(N)*Y(Si28)*state.rho) +
        (screened_rates(k_he4_mg26__n_si29)*Y(He4)*Y(Mg26)*state.rho + -screened_rates(k_n_si29__he4_mg26)*Y(N)*Y(Si29)*state.rho) +
        (-screened_rates(k_n_al25__p_mg25)*Y(Al25)*Y(N)*state.rho + screened_rates(k_p_mg25__n_al25)*Y(Mg25)*Y(H1)*state.rho) +
        (-screened_rates(k_n_al25__he4_na22)*Y(Al25)*Y(N)*state.rho + screened_rates(k_he4_na22__n_al25)*Y(He4)*Y(Na22)*state.rho) +
        (-screened_rates(k_n_al26__p_mg26)*Y(Al26)*Y(N)*state.rho + screened_rates(k_p_mg26__n_al26)*Y(Mg26)*Y(H1)*state.rho) +
        (-screened_rates(k_n_al26__he4_na23)*Y(Al26)*Y(N)*state.rho + screened_rates(k_he4_na23__n_al26)*Y(He4)*Y(Na23)*state.rho) +
        (screened_rates(k_p_si31__n_p31)*Y(H1)*Y(Si31)*state.rho + -screened_rates(k_n_p31__p_si31)*Y(N)*Y(P31)*state.rho) +
        (screened_rates(k_he4_si31__n_s34)*Y(He4)*Y(Si31)*state.rho + -screened_rates(k_n_s34__he4_si31)*Y(N)*Y(S34)*state.rho) +
        (-screened_rates(k_n_p29__p_si29)*Y(N)*Y(P29)*state.rho + screened_rates(k_p_si29__n_p29)*Y(H1)*Y(Si29)*state.rho) +
        (-screened_rates(k_n_p29__he4_al26)*Y(N)*Y(P29)*state.rho + screened_rates(k_he4_al26__n_p29)*Y(Al26)*Y(He4)*state.rho) +
        (-screened_rates(k_n_p30__p_si30)*Y(N)*Y(P30)*state.rho + screened_rates(k_p_si30__n_p30)*Y(H1)*Y(Si30)*state.rho) +
        (-screened_rates(k_n_p30__he4_al27)*Y(N)*Y(P30)*state.rho + screened_rates(k_he4_al27__n_p30)*Y(Al27)*Y(He4)*state.rho) +
        (-screened_rates(k_n_p32__p_si32)*Y(N)*Y(P32)*state.rho + screened_rates(k_p_si32__n_p32)*Y(H1)*Y(Si32)*state.rho) +
        (screened_rates(k_p_p32__n_s32)*Y(P32)*Y(H1)*state.rho + -screened_rates(k_n_s32__p_p32)*Y(N)*Y(S32)*state.rho) +
        (-screened_rates(k_n_s32__he4_si29)*Y(N)*Y(S32)*state.rho + screened_rates(k_he4_si29__n_s32)*Y(He4)*Y(Si29)*state.rho) +
        (-screened_rates(k_n_s33__p_p33)*Y(N)*Y(S33)*state.rho + screened_rates(k_p_p33__n_s33)*Y(P33)*Y(H1)*state.rho) +
        (-screened_rates(k_n_s33__he4_si30)*Y(N)*Y(S33)*state.rho + screened_rates(k_he4_si30__n_s33)*Y(He4)*Y(Si30)*state.rho) +
        (-screened_rates(k_n_s35__he4_si32)*Y(N)*Y(S35)*state.rho + screened_rates(k_he4_si32__n_s35)*Y(He4)*Y(Si32)*state.rho) +
        (screened_rates(k_he4_s35__n_ar38)*Y(He4)*Y(S35)*state.rho + -screened_rates(k_n_ar38__he4_s35)*Y(Ar38)*Y(N)*state.rho) +
        (-screened_rates(k_n_cl33__p_s33)*Y(Cl33)*Y(N)*state.rho + screened_rates(k_p_s33__n_cl33)*Y(H1)*Y(S33)*state.rho) +
        (-screened_rates(k_n_cl33__he4_p30)*Y(Cl33)*Y(N)*state.rho + screened_rates(k_he4_p30__n_cl33)*Y(He4)*Y(P30)*state.rho) +
        (-screened_rates(k_n_cl34__p_s34)*Y(Cl34)*Y(N)*state.rho + screened_rates(k_p_s34__n_cl34)*Y(H1)*Y(S34)*state.rho) +
        (-screened_rates(k_n_cl34__he4_p31)*Y(Cl34)*Y(N)*state.rho + screened_rates(k_he4_p31__n_cl34)*Y(He4)*Y(P31)*state.rho) +
        (-screened_rates(k_n_cl35__p_s35)*Y(Cl35)*Y(N)*state.rho + screened_rates(k_p_s35__n_cl35)*Y(H1)*Y(S35)*state.rho) +
        (-screened_rates(k_n_cl35__he4_p32)*Y(Cl35)*Y(N)*state.rho + screened_rates(k_he4_p32__n_cl35)*Y(He4)*Y(P32)*state.rho) +
        (-screened_rates(k_n_cl36__p_s36)*Y(Cl36)*Y(N)*state.rho + screened_rates(k_p_s36__n_cl36)*Y(H1)*Y(S36)*state.rho) +
        (-screened_rates(k_n_cl36__he4_p33)*Y(Cl36)*Y(N)*state.rho + screened_rates(k_he4_p33__n_cl36)*Y(He4)*Y(P33)*state.rho) +
        (-screened_rates(k_n_ar36__p_cl36)*Y(Ar36)*Y(N)*state.rho + screened_rates(k_p_cl36__n_ar36)*Y(Cl36)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ar36__he4_s33)*Y(Ar36)*Y(N)*state.rho + screened_rates(k_he4_s33__n_ar36)*Y(He4)*Y(S33)*state.rho) +
        (-screened_rates(k_n_ar37__p_cl37)*Y(Ar37)*Y(N)*state.rho + screened_rates(k_p_cl37__n_ar37)*Y(Cl37)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ar37__he4_s34)*Y(Ar37)*Y(N)*state.rho + screened_rates(k_he4_s34__n_ar37)*Y(He4)*Y(S34)*state.rho) +
        (-screened_rates(k_n_ar39__he4_s36)*Y(Ar39)*Y(N)*state.rho + screened_rates(k_he4_s36__n_ar39)*Y(He4)*Y(S36)*state.rho) +
        (-screened_rates(k_n_k37__p_ar37)*Y(K37)*Y(N)*state.rho + screened_rates(k_p_ar37__n_k37)*Y(Ar37)*Y(H1)*state.rho) +
        (-screened_rates(k_n_k37__he4_cl34)*Y(K37)*Y(N)*state.rho + screened_rates(k_he4_cl34__n_k37)*Y(Cl34)*Y(He4)*state.rho) +
        (-screened_rates(k_n_k38__p_ar38)*Y(K38)*Y(N)*state.rho + screened_rates(k_p_ar38__n_k38)*Y(Ar38)*Y(H1)*state.rho) +
        (-screened_rates(k_n_k38__he4_cl35)*Y(K38)*Y(N)*state.rho + screened_rates(k_he4_cl35__n_k38)*Y(Cl35)*Y(He4)*state.rho) +
        (-screened_rates(k_n_k39__p_ar39)*Y(K39)*Y(N)*state.rho + screened_rates(k_p_ar39__n_k39)*Y(Ar39)*Y(H1)*state.rho) +
        (-screened_rates(k_n_k39__he4_cl36)*Y(K39)*Y(N)*state.rho + screened_rates(k_he4_cl36__n_k39)*Y(Cl36)*Y(He4)*state.rho) +
        (-screened_rates(k_n_k40__p_ar40)*Y(K40)*Y(N)*state.rho + screened_rates(k_p_ar40__n_k40)*Y(Ar40)*Y(H1)*state.rho) +
        (-screened_rates(k_n_k40__he4_cl37)*Y(K40)*Y(N)*state.rho + screened_rates(k_he4_cl37__n_k40)*Y(Cl37)*Y(He4)*state.rho) +
        (screened_rates(k_p_k40__n_ca40)*Y(K40)*Y(H1)*state.rho + -screened_rates(k_n_ca40__p_k40)*Y(Ca40)*Y(N)*state.rho) +
        (-screened_rates(k_n_ca40__he4_ar37)*Y(Ca40)*Y(N)*state.rho + screened_rates(k_he4_ar37__n_ca40)*Y(Ar37)*Y(He4)*state.rho) +
        (-screened_rates(k_n_ca41__p_k41)*Y(Ca41)*Y(N)*state.rho + screened_rates(k_p_k41__n_ca41)*Y(K41)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ca41__he4_ar38)*Y(Ca41)*Y(N)*state.rho + screened_rates(k_he4_ar38__n_ca41)*Y(Ar38)*Y(He4)*state.rho) +
        (-screened_rates(k_n_ca42__he4_ar39)*Y(Ca42)*Y(N)*state.rho + screened_rates(k_he4_ar39__n_ca42)*Y(Ar39)*Y(He4)*state.rho) +
        (-screened_rates(k_n_ca43__he4_ar40)*Y(Ca43)*Y(N)*state.rho + screened_rates(k_he4_ar40__n_ca43)*Y(Ar40)*Y(He4)*state.rho) +
        (screened_rates(k_he4_ca43__n_ti46)*Y(Ca43)*Y(He4)*state.rho + -screened_rates(k_n_ti46__he4_ca43)*Y(N)*Y(Ti46)*state.rho) +
        (screened_rates(k_he4_ca45__n_ti48)*Y(Ca45)*Y(He4)*state.rho + -screened_rates(k_n_ti48__he4_ca45)*Y(N)*Y(Ti48)*state.rho) +
        (screened_rates(k_p_ca47__n_sc47)*Y(Ca47)*Y(H1)*state.rho + -screened_rates(k_n_sc47__p_ca47)*Y(N)*Y(Sc47)*state.rho) +
        (screened_rates(k_he4_ca47__n_ti50)*Y(Ca47)*Y(He4)*state.rho + -screened_rates(k_n_ti50__he4_ca47)*Y(N)*Y(Ti50)*state.rho) +
        (-screened_rates(k_n_sc43__p_ca43)*Y(N)*Y(Sc43)*state.rho + screened_rates(k_p_ca43__n_sc43)*Y(Ca43)*Y(H1)*state.rho) +
        (-screened_rates(k_n_sc43__he4_k40)*Y(N)*Y(Sc43)*state.rho + screened_rates(k_he4_k40__n_sc43)*Y(He4)*Y(K40)*state.rho) +
        (-screened_rates(k_n_sc44__p_ca44)*Y(N)*Y(Sc44)*state.rho + screened_rates(k_p_ca44__n_sc44)*Y(Ca44)*Y(H1)*state.rho) +
        (-screened_rates(k_n_sc44__he4_k41)*Y(N)*Y(Sc44)*state.rho + screened_rates(k_he4_k41__n_sc44)*Y(He4)*Y(K41)*state.rho) +
        (-screened_rates(k_n_sc45__p_ca45)*Y(N)*Y(Sc45)*state.rho + screened_rates(k_p_ca45__n_sc45)*Y(Ca45)*Y(H1)*state.rho) +
        (-screened_rates(k_n_sc46__p_ca46)*Y(N)*Y(Sc46)*state.rho + screened_rates(k_p_ca46__n_sc46)*Y(Ca46)*Y(H1)*state.rho) +
        (screened_rates(k_p_sc46__n_ti46)*Y(H1)*Y(Sc46)*state.rho + -screened_rates(k_n_ti46__p_sc46)*Y(N)*Y(Ti46)*state.rho) +
        (screened_rates(k_he4_sc46__n_v49)*Y(He4)*Y(Sc46)*state.rho + -screened_rates(k_n_v49__he4_sc46)*Y(N)*Y(V49)*state.rho) +
        (-screened_rates(k_n_sc48__p_ca48)*Y(N)*Y(Sc48)*state.rho + screened_rates(k_p_ca48__n_sc48)*Y(Ca48)*Y(H1)*state.rho) +
        (screened_rates(k_p_sc48__n_ti48)*Y(H1)*Y(Sc48)*state.rho + -screened_rates(k_n_ti48__p_sc48)*Y(N)*Y(Ti48)*state.rho) +
        (screened_rates(k_he4_sc48__n_v51)*Y(He4)*Y(Sc48)*state.rho + -screened_rates(k_n_v51__he4_sc48)*Y(N)*Y(V51)*state.rho) +
        (screened_rates(k_p_sc49__n_ti49)*Y(H1)*Y(Sc49)*state.rho + -screened_rates(k_n_ti49__p_sc49)*Y(N)*Y(Ti49)*state.rho) +
        (-screened_rates(k_n_ti44__p_sc44)*Y(N)*Y(Ti44)*state.rho + screened_rates(k_p_sc44__n_ti44)*Y(H1)*Y(Sc44)*state.rho) +
        (-screened_rates(k_n_ti44__he4_ca41)*Y(N)*Y(Ti44)*state.rho + screened_rates(k_he4_ca41__n_ti44)*Y(Ca41)*Y(He4)*state.rho) +
        (-screened_rates(k_n_ti45__p_sc45)*Y(N)*Y(Ti45)*state.rho + screened_rates(k_p_sc45__n_ti45)*Y(H1)*Y(Sc45)*state.rho) +
        (-screened_rates(k_n_ti45__he4_ca42)*Y(N)*Y(Ti45)*state.rho + screened_rates(k_he4_ca42__n_ti45)*Y(Ca42)*Y(He4)*state.rho) +
        (-screened_rates(k_n_ti47__p_sc47)*Y(N)*Y(Ti47)*state.rho + screened_rates(k_p_sc47__n_ti47)*Y(H1)*Y(Sc47)*state.rho) +
        (-screened_rates(k_n_ti47__he4_ca44)*Y(N)*Y(Ti47)*state.rho + screened_rates(k_he4_ca44__n_ti47)*Y(Ca44)*Y(He4)*state.rho) +
        (-screened_rates(k_n_ti49__he4_ca46)*Y(N)*Y(Ti49)*state.rho + screened_rates(k_he4_ca46__n_ti49)*Y(Ca46)*Y(He4)*state.rho) +
        (screened_rates(k_he4_ti49__n_cr52)*Y(He4)*Y(Ti49)*state.rho + -screened_rates(k_n_cr52__he4_ti49)*Y(Cr52)*Y(N)*state.rho) +
        (-screened_rates(k_n_ti51__he4_ca48)*Y(N)*Y(Ti51)*state.rho + screened_rates(k_he4_ca48__n_ti51)*Y(Ca48)*Y(He4)*state.rho) +
        (screened_rates(k_p_ti51__n_v51)*Y(H1)*Y(Ti51)*state.rho + -screened_rates(k_n_v51__p_ti51)*Y(N)*Y(V51)*state.rho) +
        (screened_rates(k_he4_ti51__n_cr54)*Y(He4)*Y(Ti51)*state.rho + -screened_rates(k_n_cr54__he4_ti51)*Y(Cr54)*Y(N)*state.rho) +
        (-screened_rates(k_n_v46__p_ti46)*Y(N)*Y(V46)*state.rho + screened_rates(k_p_ti46__n_v46)*Y(H1)*Y(Ti46)*state.rho) +
        (-screened_rates(k_n_v46__he4_sc43)*Y(N)*Y(V46)*state.rho + screened_rates(k_he4_sc43__n_v46)*Y(He4)*Y(Sc43)*state.rho) +
        (-screened_rates(k_n_v47__p_ti47)*Y(N)*Y(V47)*state.rho + screened_rates(k_p_ti47__n_v47)*Y(H1)*Y(Ti47)*state.rho) +
        (-screened_rates(k_n_v47__he4_sc44)*Y(N)*Y(V47)*state.rho + screened_rates(k_he4_sc44__n_v47)*Y(He4)*Y(Sc44)*state.rho) +
        (-screened_rates(k_n_v48__p_ti48)*Y(N)*Y(V48)*state.rho + screened_rates(k_p_ti48__n_v48)*Y(H1)*Y(Ti48)*state.rho) +
        (-screened_rates(k_n_v48__he4_sc45)*Y(N)*Y(V48)*state.rho + screened_rates(k_he4_sc45__n_v48)*Y(He4)*Y(Sc45)*state.rho) +
        (-screened_rates(k_n_v49__p_ti49)*Y(N)*Y(V49)*state.rho + screened_rates(k_p_ti49__n_v49)*Y(H1)*Y(Ti49)*state.rho) +
        (-screened_rates(k_n_v50__p_ti50)*Y(N)*Y(V50)*state.rho + screened_rates(k_p_ti50__n_v50)*Y(H1)*Y(Ti50)*state.rho) +
        (-screened_rates(k_n_v50__he4_sc47)*Y(N)*Y(V50)*state.rho + screened_rates(k_he4_sc47__n_v50)*Y(He4)*Y(Sc47)*state.rho) +
        (screened_rates(k_p_v50__n_cr50)*Y(H1)*Y(V50)*state.rho + -screened_rates(k_n_cr50__p_v50)*Y(Cr50)*Y(N)*state.rho) +
        (-screened_rates(k_n_v52__he4_sc49)*Y(N)*Y(V52)*state.rho + screened_rates(k_he4_sc49__n_v52)*Y(He4)*Y(Sc49)*state.rho) +
        (screened_rates(k_p_v52__n_cr52)*Y(H1)*Y(V52)*state.rho + -screened_rates(k_n_cr52__p_v52)*Y(Cr52)*Y(N)*state.rho) +
        (screened_rates(k_he4_v52__n_mn55)*Y(He4)*Y(V52)*state.rho + -screened_rates(k_n_mn55__he4_v52)*Y(Mn55)*Y(N)*state.rho) +
        (-screened_rates(k_n_cr48__p_v48)*Y(Cr48)*Y(N)*state.rho + screened_rates(k_p_v48__n_cr48)*Y(H1)*Y(V48)*state.rho) +
        (-screened_rates(k_n_cr48__he4_ti45)*Y(Cr48)*Y(N)*state.rho + screened_rates(k_he4_ti45__n_cr48)*Y(He4)*Y(Ti45)*state.rho) +
        (-screened_rates(k_n_cr49__p_v49)*Y(Cr49)*Y(N)*state.rho + screened_rates(k_p_v49__n_cr49)*Y(H1)*Y(V49)*state.rho) +
        (-screened_rates(k_n_cr49__he4_ti46)*Y(Cr49)*Y(N)*state.rho + screened_rates(k_he4_ti46__n_cr49)*Y(He4)*Y(Ti46)*state.rho) +
        (-screened_rates(k_n_cr50__he4_ti47)*Y(Cr50)*Y(N)*state.rho + screened_rates(k_he4_ti47__n_cr50)*Y(He4)*Y(Ti47)*state.rho) +
        (-screened_rates(k_n_cr51__p_v51)*Y(Cr51)*Y(N)*state.rho + screened_rates(k_p_v51__n_cr51)*Y(H1)*Y(V51)*state.rho) +
        (-screened_rates(k_n_cr51__he4_ti48)*Y(Cr51)*Y(N)*state.rho + screened_rates(k_he4_ti48__n_cr51)*Y(He4)*Y(Ti48)*state.rho) +
        (-screened_rates(k_n_cr53__he4_ti50)*Y(Cr53)*Y(N)*state.rho + screened_rates(k_he4_ti50__n_cr53)*Y(He4)*Y(Ti50)*state.rho) +
        (-screened_rates(k_n_mn50__p_cr50)*Y(Mn50)*Y(N)*state.rho + screened_rates(k_p_cr50__n_mn50)*Y(Cr50)*Y(H1)*state.rho) +
        (-screened_rates(k_n_mn50__he4_v47)*Y(Mn50)*Y(N)*state.rho + screened_rates(k_he4_v47__n_mn50)*Y(He4)*Y(V47)*state.rho) +
        (-screened_rates(k_n_mn51__p_cr51)*Y(Mn51)*Y(N)*state.rho + screened_rates(k_p_cr51__n_mn51)*Y(Cr51)*Y(H1)*state.rho) +
        (-screened_rates(k_n_mn51__he4_v48)*Y(Mn51)*Y(N)*state.rho + screened_rates(k_he4_v48__n_mn51)*Y(He4)*Y(V48)*state.rho) +
        (-screened_rates(k_n_mn52__p_cr52)*Y(Mn52)*Y(N)*state.rho + screened_rates(k_p_cr52__n_mn52)*Y(Cr52)*Y(H1)*state.rho) +
        (-screened_rates(k_n_mn52__he4_v49)*Y(Mn52)*Y(N)*state.rho + screened_rates(k_he4_v49__n_mn52)*Y(He4)*Y(V49)*state.rho) +
        (-screened_rates(k_n_mn53__p_cr53)*Y(Mn53)*Y(N)*state.rho + screened_rates(k_p_cr53__n_mn53)*Y(Cr53)*Y(H1)*state.rho) +
        (-screened_rates(k_n_mn53__he4_v50)*Y(Mn53)*Y(N)*state.rho + screened_rates(k_he4_v50__n_mn53)*Y(He4)*Y(V50)*state.rho) +
        (-screened_rates(k_n_mn54__p_cr54)*Y(Mn54)*Y(N)*state.rho + screened_rates(k_p_cr54__n_mn54)*Y(Cr54)*Y(H1)*state.rho) +
        (-screened_rates(k_n_mn54__he4_v51)*Y(Mn54)*Y(N)*state.rho + screened_rates(k_he4_v51__n_mn54)*Y(He4)*Y(V51)*state.rho) +
        (-screened_rates(k_n_fe52__p_mn52)*Y(Fe52)*Y(N)*state.rho + screened_rates(k_p_mn52__n_fe52)*Y(Mn52)*Y(H1)*state.rho) +
        (-screened_rates(k_n_fe52__he4_cr49)*Y(Fe52)*Y(N)*state.rho + screened_rates(k_he4_cr49__n_fe52)*Y(Cr49)*Y(He4)*state.rho) +
        (-screened_rates(k_n_fe53__p_mn53)*Y(Fe53)*Y(N)*state.rho + screened_rates(k_p_mn53__n_fe53)*Y(Mn53)*Y(H1)*state.rho) +
        (-screened_rates(k_n_fe53__he4_cr50)*Y(Fe53)*Y(N)*state.rho + screened_rates(k_he4_cr50__n_fe53)*Y(Cr50)*Y(He4)*state.rho) +
        (-screened_rates(k_n_fe54__p_mn54)*Y(Fe54)*Y(N)*state.rho + screened_rates(k_p_mn54__n_fe54)*Y(Mn54)*Y(H1)*state.rho) +
        (-screened_rates(k_n_fe54__he4_cr51)*Y(Fe54)*Y(N)*state.rho + screened_rates(k_he4_cr51__n_fe54)*Y(Cr51)*Y(He4)*state.rho) +
        (-screened_rates(k_n_fe55__p_mn55)*Y(Fe55)*Y(N)*state.rho + screened_rates(k_p_mn55__n_fe55)*Y(Mn55)*Y(H1)*state.rho) +
        (-screened_rates(k_n_fe55__he4_cr52)*Y(Fe55)*Y(N)*state.rho + screened_rates(k_he4_cr52__n_fe55)*Y(Cr52)*Y(He4)*state.rho) +
        (-screened_rates(k_n_fe56__he4_cr53)*Y(Fe56)*Y(N)*state.rho + screened_rates(k_he4_cr53__n_fe56)*Y(Cr53)*Y(He4)*state.rho) +
        (-screened_rates(k_n_fe57__he4_cr54)*Y(Fe57)*Y(N)*state.rho + screened_rates(k_he4_cr54__n_fe57)*Y(Cr54)*Y(He4)*state.rho) +
        (-screened_rates(k_n_co53__p_fe53)*Y(Co53)*Y(N)*state.rho + screened_rates(k_p_fe53__n_co53)*Y(Fe53)*Y(H1)*state.rho) +
        (-screened_rates(k_n_co53__he4_mn50)*Y(Co53)*Y(N)*state.rho + screened_rates(k_he4_mn50__n_co53)*Y(He4)*Y(Mn50)*state.rho) +
        (-screened_rates(k_n_co54__p_fe54)*Y(Co54)*Y(N)*state.rho + screened_rates(k_p_fe54__n_co54)*Y(Fe54)*Y(H1)*state.rho) +
        (-screened_rates(k_n_co54__he4_mn51)*Y(Co54)*Y(N)*state.rho + screened_rates(k_he4_mn51__n_co54)*Y(He4)*Y(Mn51)*state.rho) +
        (-screened_rates(k_n_co55__p_fe55)*Y(Co55)*Y(N)*state.rho + screened_rates(k_p_fe55__n_co55)*Y(Fe55)*Y(H1)*state.rho) +
        (-screened_rates(k_n_co55__he4_mn52)*Y(Co55)*Y(N)*state.rho + screened_rates(k_he4_mn52__n_co55)*Y(He4)*Y(Mn52)*state.rho) +
        (-screened_rates(k_n_co56__p_fe56)*Y(Co56)*Y(N)*state.rho + screened_rates(k_p_fe56__n_co56)*Y(Fe56)*Y(H1)*state.rho) +
        (-screened_rates(k_n_co56__he4_mn53)*Y(Co56)*Y(N)*state.rho + screened_rates(k_he4_mn53__n_co56)*Y(He4)*Y(Mn53)*state.rho) +
        (-screened_rates(k_n_co57__p_fe57)*Y(Co57)*Y(N)*state.rho + screened_rates(k_p_fe57__n_co57)*Y(Fe57)*Y(H1)*state.rho) +
        (-screened_rates(k_n_co57__he4_mn54)*Y(Co57)*Y(N)*state.rho + screened_rates(k_he4_mn54__n_co57)*Y(He4)*Y(Mn54)*state.rho) +
        (-screened_rates(k_n_co58__p_fe58)*Y(Co58)*Y(N)*state.rho + screened_rates(k_p_fe58__n_co58)*Y(Fe58)*Y(H1)*state.rho) +
        (-screened_rates(k_n_co58__he4_mn55)*Y(Co58)*Y(N)*state.rho + screened_rates(k_he4_mn55__n_co58)*Y(He4)*Y(Mn55)*state.rho) +
        (-screened_rates(k_n_ni56__p_co56)*Y(N)*Y(Ni56)*state.rho + screened_rates(k_p_co56__n_ni56)*Y(Co56)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ni56__he4_fe53)*Y(N)*Y(Ni56)*state.rho + screened_rates(k_he4_fe53__n_ni56)*Y(Fe53)*Y(He4)*state.rho) +
        (-screened_rates(k_n_ni57__p_co57)*Y(N)*Y(Ni57)*state.rho + screened_rates(k_p_co57__n_ni57)*Y(Co57)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ni57__he4_fe54)*Y(N)*Y(Ni57)*state.rho + screened_rates(k_he4_fe54__n_ni57)*Y(Fe54)*Y(He4)*state.rho) +
        (-screened_rates(k_n_ni58__p_co58)*Y(N)*Y(Ni58)*state.rho + screened_rates(k_p_co58__n_ni58)*Y(Co58)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ni58__he4_fe55)*Y(N)*Y(Ni58)*state.rho + screened_rates(k_he4_fe55__n_ni58)*Y(Fe55)*Y(He4)*state.rho) +
        (-screened_rates(k_n_ni59__p_co59)*Y(N)*Y(Ni59)*state.rho + screened_rates(k_p_co59__n_ni59)*Y(Co59)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ni59__he4_fe56)*Y(N)*Y(Ni59)*state.rho + screened_rates(k_he4_fe56__n_ni59)*Y(Fe56)*Y(He4)*state.rho) +
        (-screened_rates(k_n_ni60__he4_fe57)*Y(N)*Y(Ni60)*state.rho + screened_rates(k_he4_fe57__n_ni60)*Y(Fe57)*Y(He4)*state.rho) +
        (-screened_rates(k_n_ni61__he4_fe58)*Y(N)*Y(Ni61)*state.rho + screened_rates(k_he4_fe58__n_ni61)*Y(Fe58)*Y(He4)*state.rho) +
        (-screened_rates(k_n_cu57__p_ni57)*Y(Cu57)*Y(N)*state.rho + screened_rates(k_p_ni57__n_cu57)*Y(Ni57)*Y(H1)*state.rho) +
        (-screened_rates(k_n_cu57__he4_co54)*Y(Cu57)*Y(N)*state.rho + screened_rates(k_he4_co54__n_cu57)*Y(Co54)*Y(He4)*state.rho) +
        (-screened_rates(k_n_cu58__p_ni58)*Y(Cu58)*Y(N)*state.rho + screened_rates(k_p_ni58__n_cu58)*Y(Ni58)*Y(H1)*state.rho) +
        (-screened_rates(k_n_cu58__he4_co55)*Y(Cu58)*Y(N)*state.rho + screened_rates(k_he4_co55__n_cu58)*Y(Co55)*Y(He4)*state.rho) +
        (-screened_rates(k_n_cu59__p_ni59)*Y(Cu59)*Y(N)*state.rho + screened_rates(k_p_ni59__n_cu59)*Y(Ni59)*Y(H1)*state.rho) +
        (-screened_rates(k_n_cu59__he4_co56)*Y(Cu59)*Y(N)*state.rho + screened_rates(k_he4_co56__n_cu59)*Y(Co56)*Y(He4)*state.rho) +
        (-screened_rates(k_n_cu60__p_ni60)*Y(Cu60)*Y(N)*state.rho + screened_rates(k_p_ni60__n_cu60)*Y(Ni60)*Y(H1)*state.rho) +
        (-screened_rates(k_n_cu60__he4_co57)*Y(Cu60)*Y(N)*state.rho + screened_rates(k_he4_co57__n_cu60)*Y(Co57)*Y(He4)*state.rho) +
        (-screened_rates(k_n_cu61__p_ni61)*Y(Cu61)*Y(N)*state.rho + screened_rates(k_p_ni61__n_cu61)*Y(Ni61)*Y(H1)*state.rho) +
        (-screened_rates(k_n_cu61__he4_co58)*Y(Cu61)*Y(N)*state.rho + screened_rates(k_he4_co58__n_cu61)*Y(Co58)*Y(He4)*state.rho) +
        (-screened_rates(k_n_cu62__p_ni62)*Y(Cu62)*Y(N)*state.rho + screened_rates(k_p_ni62__n_cu62)*Y(Ni62)*Y(H1)*state.rho) +
        (-screened_rates(k_n_cu62__he4_co59)*Y(Cu62)*Y(N)*state.rho + screened_rates(k_he4_co59__n_cu62)*Y(Co59)*Y(He4)*state.rho) +
        (-screened_rates(k_n_cu63__p_ni63)*Y(Cu63)*Y(N)*state.rho + screened_rates(k_p_ni63__n_cu63)*Y(Ni63)*Y(H1)*state.rho) +
        (-screened_rates(k_n_cu64__p_ni64)*Y(Cu64)*Y(N)*state.rho + screened_rates(k_p_ni64__n_cu64)*Y(Ni64)*Y(H1)*state.rho) +
        (-screened_rates(k_n_zn59__p_cu59)*Y(N)*Y(Zn59)*state.rho + screened_rates(k_p_cu59__n_zn59)*Y(Cu59)*Y(H1)*state.rho) +
        (-screened_rates(k_n_zn59__he4_ni56)*Y(N)*Y(Zn59)*state.rho + screened_rates(k_he4_ni56__n_zn59)*Y(He4)*Y(Ni56)*state.rho) +
        (-screened_rates(k_n_zn60__p_cu60)*Y(N)*Y(Zn60)*state.rho + screened_rates(k_p_cu60__n_zn60)*Y(Cu60)*Y(H1)*state.rho) +
        (-screened_rates(k_n_zn60__he4_ni57)*Y(N)*Y(Zn60)*state.rho + screened_rates(k_he4_ni57__n_zn60)*Y(He4)*Y(Ni57)*state.rho) +
        (-screened_rates(k_n_zn61__p_cu61)*Y(N)*Y(Zn61)*state.rho + screened_rates(k_p_cu61__n_zn61)*Y(Cu61)*Y(H1)*state.rho) +
        (-screened_rates(k_n_zn61__he4_ni58)*Y(N)*Y(Zn61)*state.rho + screened_rates(k_he4_ni58__n_zn61)*Y(He4)*Y(Ni58)*state.rho) +
        (-screened_rates(k_n_zn62__p_cu62)*Y(N)*Y(Zn62)*state.rho + screened_rates(k_p_cu62__n_zn62)*Y(Cu62)*Y(H1)*state.rho) +
        (-screened_rates(k_n_zn62__he4_ni59)*Y(N)*Y(Zn62)*state.rho + screened_rates(k_he4_ni59__n_zn62)*Y(He4)*Y(Ni59)*state.rho) +
        (-screened_rates(k_n_zn63__p_cu63)*Y(N)*Y(Zn63)*state.rho + screened_rates(k_p_cu63__n_zn63)*Y(Cu63)*Y(H1)*state.rho) +
        (-screened_rates(k_n_zn63__he4_ni60)*Y(N)*Y(Zn63)*state.rho + screened_rates(k_he4_ni60__n_zn63)*Y(He4)*Y(Ni60)*state.rho) +
        (-screened_rates(k_n_zn64__p_cu64)*Y(N)*Y(Zn64)*state.rho + screened_rates(k_p_cu64__n_zn64)*Y(Cu64)*Y(H1)*state.rho) +
        (-screened_rates(k_n_zn64__he4_ni61)*Y(N)*Y(Zn64)*state.rho + screened_rates(k_he4_ni61__n_zn64)*Y(He4)*Y(Ni61)*state.rho) +
        (-screened_rates(k_n_zn65__p_cu65)*Y(N)*Y(Zn65)*state.rho + screened_rates(k_p_cu65__n_zn65)*Y(Cu65)*Y(H1)*state.rho) +
        (-screened_rates(k_n_zn65__he4_ni62)*Y(N)*Y(Zn65)*state.rho + screened_rates(k_he4_ni62__n_zn65)*Y(He4)*Y(Ni62)*state.rho) +
        (-screened_rates(k_n_zn66__he4_ni63)*Y(N)*Y(Zn66)*state.rho + screened_rates(k_he4_ni63__n_zn66)*Y(He4)*Y(Ni63)*state.rho) +
        (-screened_rates(k_n_ga62__p_zn62)*Y(Ga62)*Y(N)*state.rho + screened_rates(k_p_zn62__n_ga62)*Y(H1)*Y(Zn62)*state.rho) +
        (-screened_rates(k_n_ga62__he4_cu59)*Y(Ga62)*Y(N)*state.rho + screened_rates(k_he4_cu59__n_ga62)*Y(Cu59)*Y(He4)*state.rho) +
        (-screened_rates(k_n_ga63__p_zn63)*Y(Ga63)*Y(N)*state.rho + screened_rates(k_p_zn63__n_ga63)*Y(H1)*Y(Zn63)*state.rho) +
        (-screened_rates(k_n_ga63__he4_cu60)*Y(Ga63)*Y(N)*state.rho + screened_rates(k_he4_cu60__n_ga63)*Y(Cu60)*Y(He4)*state.rho) +
        (-screened_rates(k_n_ga64__p_zn64)*Y(Ga64)*Y(N)*state.rho + screened_rates(k_p_zn64__n_ga64)*Y(H1)*Y(Zn64)*state.rho) +
        (-screened_rates(k_n_ga64__he4_cu61)*Y(Ga64)*Y(N)*state.rho + screened_rates(k_he4_cu61__n_ga64)*Y(Cu61)*Y(He4)*state.rho) +
        (-screened_rates(k_n_ge63__p_ga63)*Y(Ge63)*Y(N)*state.rho + screened_rates(k_p_ga63__n_ge63)*Y(Ga63)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ge63__he4_zn60)*Y(Ge63)*Y(N)*state.rho + screened_rates(k_he4_zn60__n_ge63)*Y(He4)*Y(Zn60)*state.rho) +
        (-screened_rates(k_n_ge64__p_ga64)*Y(Ge64)*Y(N)*state.rho + screened_rates(k_p_ga64__n_ge64)*Y(Ga64)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ge64__he4_zn61)*Y(Ge64)*Y(N)*state.rho + screened_rates(k_he4_zn61__n_ge64)*Y(He4)*Y(Zn61)*state.rho) +
        (screened_rates(k_d_li7__n_he4_he4)*Y(H2)*Y(Li7)*state.rho + -0.5*screened_rates(k_n_he4_he4__d_li7)*std::pow(Y(He4), 2)*Y(N)*std::pow(state.rho, 2)) +
        (-screened_rates(k_n_b8__p_he4_he4)*Y(B8)*Y(N)*state.rho + 0.5*screened_rates(k_p_he4_he4__n_b8)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 2)) +
        (screened_rates(k_he3_li7__n_p_he4_he4)*Y(He3)*Y(Li7)*state.rho + -0.5*screened_rates(k_n_p_he4_he4__he3_li7)*std::pow(Y(He4), 2)*Y(N)*Y(H1)*std::pow(state.rho, 3)) +
        (-screened_rates(k_n_p_he4__li6)*Y(He4)*Y(N)*Y(H1)*std::pow(state.rho, 2) + screened_rates(k_li6__n_p_he4)*Y(Li6)) +
        (-0.5*screened_rates(k_n_he4_he4__be9)*std::pow(Y(He4), 2)*Y(N)*std::pow(state.rho, 2) + screened_rates(k_be9__n_he4_he4)*Y(Be9)) +
        (-0.5*screened_rates(k_n_p_p__p_d)*Y(N)*std::pow(Y(H1), 2)*std::pow(state.rho, 2) + screened_rates(k_p_d__n_p_p)*Y(H2)*Y(H1)*state.rho) +
        (-0.5*screened_rates(k_n_p_he4_he4__p_be9)*std::pow(Y(He4), 2)*Y(N)*Y(H1)*std::pow(state.rho, 3) + screened_rates(k_p_be9__n_p_he4_he4)*Y(Be9)*Y(H1)*state.rho);

    ydot_nuc(H1) =
        screened_rates(k_n__p__weak__wc12)*Y(N) +
        screened_rates(k_zn59__p_ni58__weak__wc12)*Y(Zn59) +
        (-screened_rates(k_n_p__d)*Y(N)*Y(H1)*state.rho + screened_rates(k_d__n_p)*Y(H2)) +
        -screened_rates(k_p_p__d__weak__bet_pos_)*std::pow(Y(H1), 2)*state.rho +
        -screened_rates(k_p_p__d__weak__electron_capture)*std::pow(Y(H1), 2)*std::pow(state.rho, 2)*state.y_e +
        (-screened_rates(k_p_d__he3)*Y(H2)*Y(H1)*state.rho + screened_rates(k_he3__p_d)*Y(He3)) +
        -screened_rates(k_p_he3__he4__weak__bet_pos_)*Y(He3)*Y(H1)*state.rho +
        (-screened_rates(k_p_li6__be7)*Y(Li6)*Y(H1)*state.rho + screened_rates(k_be7__p_li6)*Y(Be7)) +
        (-screened_rates(k_p_be7__b8)*Y(Be7)*Y(H1)*state.rho + screened_rates(k_b8__p_be7)*Y(B8)) +
        (-screened_rates(k_p_be9__b10)*Y(Be9)*Y(H1)*state.rho + screened_rates(k_b10__p_be9)*Y(B10)) +
        (-screened_rates(k_p_b11__c12)*Y(B11)*Y(H1)*state.rho + screened_rates(k_c12__p_b11)*Y(C12)) +
        (-screened_rates(k_p_c12__n13)*Y(C12)*Y(H1)*state.rho + screened_rates(k_n13__p_c12)*Y(N13)) +
        (-screened_rates(k_p_c13__n14)*Y(C13)*Y(H1)*state.rho + screened_rates(k_n14__p_c13)*Y(N14)) +
        (-screened_rates(k_p_c14__n15)*Y(C14)*Y(H1)*state.rho + screened_rates(k_n15__p_c14)*Y(N15)) +
        (-screened_rates(k_p_n13__o14)*Y(N13)*Y(H1)*state.rho + screened_rates(k_o14__p_n13)*Y(O14)) +
        (-screened_rates(k_p_n14__o15)*Y(N14)*Y(H1)*state.rho + screened_rates(k_o15__p_n14)*Y(O15)) +
        (-screened_rates(k_p_n15__o16)*Y(N15)*Y(H1)*state.rho + screened_rates(k_o16__p_n15)*Y(O16)) +
        (-screened_rates(k_p_o16__f17)*Y(O16)*Y(H1)*state.rho + screened_rates(k_f17__p_o16)*Y(F17)) +
        (-screened_rates(k_p_o17__f18)*Y(O17)*Y(H1)*state.rho + screened_rates(k_f18__p_o17)*Y(F18)) +
        (-screened_rates(k_p_o18__f19)*Y(O18)*Y(H1)*state.rho + screened_rates(k_f19__p_o18)*Y(F19)) +
        (-screened_rates(k_p_f17__ne18)*Y(F17)*Y(H1)*state.rho + screened_rates(k_ne18__p_f17)*Y(Ne18)) +
        (-screened_rates(k_p_f18__ne19)*Y(F18)*Y(H1)*state.rho + screened_rates(k_ne19__p_f18)*Y(Ne19)) +
        (-screened_rates(k_p_f19__ne20)*Y(F19)*Y(H1)*state.rho + screened_rates(k_ne20__p_f19)*Y(Ne20)) +
        (-screened_rates(k_p_ne20__na21)*Y(Ne20)*Y(H1)*state.rho + screened_rates(k_na21__p_ne20)*Y(Na21)) +
        (-screened_rates(k_p_ne21__na22)*Y(Ne21)*Y(H1)*state.rho + screened_rates(k_na22__p_ne21)*Y(Na22)) +
        (-screened_rates(k_p_ne22__na23)*Y(Ne22)*Y(H1)*state.rho + screened_rates(k_na23__p_ne22)*Y(Na23)) +
        (-screened_rates(k_p_na22__mg23)*Y(Na22)*Y(H1)*state.rho + screened_rates(k_mg23__p_na22)*Y(Mg23)) +
        (-screened_rates(k_p_na23__mg24)*Y(Na23)*Y(H1)*state.rho + screened_rates(k_mg24__p_na23)*Y(Mg24)) +
        (-screened_rates(k_p_mg24__al25)*Y(Mg24)*Y(H1)*state.rho + screened_rates(k_al25__p_mg24)*Y(Al25)) +
        (-screened_rates(k_p_mg25__al26)*Y(Mg25)*Y(H1)*state.rho + screened_rates(k_al26__p_mg25)*Y(Al26)) +
        (-screened_rates(k_p_mg26__al27)*Y(Mg26)*Y(H1)*state.rho + screened_rates(k_al27__p_mg26)*Y(Al27)) +
        (-screened_rates(k_p_al27__si28)*Y(Al27)*Y(H1)*state.rho + screened_rates(k_si28__p_al27)*Y(Si28)) +
        (-screened_rates(k_p_si28__p29)*Y(H1)*Y(Si28)*state.rho + screened_rates(k_p29__p_si28)*Y(P29)) +
        (-screened_rates(k_p_si29__p30)*Y(H1)*Y(Si29)*state.rho + screened_rates(k_p30__p_si29)*Y(P30)) +
        (-screened_rates(k_p_si30__p31)*Y(H1)*Y(Si30)*state.rho + screened_rates(k_p31__p_si30)*Y(P31)) +
        (-screened_rates(k_p_si31__p32)*Y(H1)*Y(Si31)*state.rho + screened_rates(k_p32__p_si31)*Y(P32)) +
        (-screened_rates(k_p_si32__p33)*Y(H1)*Y(Si32)*state.rho + screened_rates(k_p33__p_si32)*Y(P33)) +
        (-screened_rates(k_p_p31__s32)*Y(P31)*Y(H1)*state.rho + screened_rates(k_s32__p_p31)*Y(S32)) +
        (-screened_rates(k_p_p32__s33)*Y(P32)*Y(H1)*state.rho + screened_rates(k_s33__p_p32)*Y(S33)) +
        (-screened_rates(k_p_p33__s34)*Y(P33)*Y(H1)*state.rho + screened_rates(k_s34__p_p33)*Y(S34)) +
        (-screened_rates(k_p_s32__cl33)*Y(H1)*Y(S32)*state.rho + screened_rates(k_cl33__p_s32)*Y(Cl33)) +
        (-screened_rates(k_p_s33__cl34)*Y(H1)*Y(S33)*state.rho + screened_rates(k_cl34__p_s33)*Y(Cl34)) +
        (-screened_rates(k_p_s34__cl35)*Y(H1)*Y(S34)*state.rho + screened_rates(k_cl35__p_s34)*Y(Cl35)) +
        (-screened_rates(k_p_s35__cl36)*Y(H1)*Y(S35)*state.rho + screened_rates(k_cl36__p_s35)*Y(Cl36)) +
        (-screened_rates(k_p_s36__cl37)*Y(H1)*Y(S36)*state.rho + screened_rates(k_cl37__p_s36)*Y(Cl37)) +
        (-screened_rates(k_p_cl35__ar36)*Y(Cl35)*Y(H1)*state.rho + screened_rates(k_ar36__p_cl35)*Y(Ar36)) +
        (-screened_rates(k_p_cl36__ar37)*Y(Cl36)*Y(H1)*state.rho + screened_rates(k_ar37__p_cl36)*Y(Ar37)) +
        (-screened_rates(k_p_cl37__ar38)*Y(Cl37)*Y(H1)*state.rho + screened_rates(k_ar38__p_cl37)*Y(Ar38)) +
        (-screened_rates(k_p_ar36__k37)*Y(Ar36)*Y(H1)*state.rho + screened_rates(k_k37__p_ar36)*Y(K37)) +
        (-screened_rates(k_p_ar37__k38)*Y(Ar37)*Y(H1)*state.rho + screened_rates(k_k38__p_ar37)*Y(K38)) +
        (-screened_rates(k_p_ar38__k39)*Y(Ar38)*Y(H1)*state.rho + screened_rates(k_k39__p_ar38)*Y(K39)) +
        (-screened_rates(k_p_ar39__k40)*Y(Ar39)*Y(H1)*state.rho + screened_rates(k_k40__p_ar39)*Y(K40)) +
        (-screened_rates(k_p_ar40__k41)*Y(Ar40)*Y(H1)*state.rho + screened_rates(k_k41__p_ar40)*Y(K41)) +
        (-screened_rates(k_p_k39__ca40)*Y(K39)*Y(H1)*state.rho + screened_rates(k_ca40__p_k39)*Y(Ca40)) +
        (-screened_rates(k_p_k40__ca41)*Y(K40)*Y(H1)*state.rho + screened_rates(k_ca41__p_k40)*Y(Ca41)) +
        (-screened_rates(k_p_k41__ca42)*Y(K41)*Y(H1)*state.rho + screened_rates(k_ca42__p_k41)*Y(Ca42)) +
        (-screened_rates(k_p_ca42__sc43)*Y(Ca42)*Y(H1)*state.rho + screened_rates(k_sc43__p_ca42)*Y(Sc43)) +
        (-screened_rates(k_p_ca43__sc44)*Y(Ca43)*Y(H1)*state.rho + screened_rates(k_sc44__p_ca43)*Y(Sc44)) +
        (-screened_rates(k_p_ca44__sc45)*Y(Ca44)*Y(H1)*state.rho + screened_rates(k_sc45__p_ca44)*Y(Sc45)) +
        (-screened_rates(k_p_ca45__sc46)*Y(Ca45)*Y(H1)*state.rho + screened_rates(k_sc46__p_ca45)*Y(Sc46)) +
        (-screened_rates(k_p_ca46__sc47)*Y(Ca46)*Y(H1)*state.rho + screened_rates(k_sc47__p_ca46)*Y(Sc47)) +
        (-screened_rates(k_p_ca47__sc48)*Y(Ca47)*Y(H1)*state.rho + screened_rates(k_sc48__p_ca47)*Y(Sc48)) +
        (-screened_rates(k_p_ca48__sc49)*Y(Ca48)*Y(H1)*state.rho + screened_rates(k_sc49__p_ca48)*Y(Sc49)) +
        (-screened_rates(k_p_sc43__ti44)*Y(H1)*Y(Sc43)*state.rho + screened_rates(k_ti44__p_sc43)*Y(Ti44)) +
        (-screened_rates(k_p_sc44__ti45)*Y(H1)*Y(Sc44)*state.rho + screened_rates(k_ti45__p_sc44)*Y(Ti45)) +
        (-screened_rates(k_p_sc45__ti46)*Y(H1)*Y(Sc45)*state.rho + screened_rates(k_ti46__p_sc45)*Y(Ti46)) +
        (-screened_rates(k_p_sc46__ti47)*Y(H1)*Y(Sc46)*state.rho + screened_rates(k_ti47__p_sc46)*Y(Ti47)) +
        (-screened_rates(k_p_sc47__ti48)*Y(H1)*Y(Sc47)*state.rho + screened_rates(k_ti48__p_sc47)*Y(Ti48)) +
        (-screened_rates(k_p_sc48__ti49)*Y(H1)*Y(Sc48)*state.rho + screened_rates(k_ti49__p_sc48)*Y(Ti49)) +
        (-screened_rates(k_p_sc49__ti50)*Y(H1)*Y(Sc49)*state.rho + screened_rates(k_ti50__p_sc49)*Y(Ti50)) +
        (-screened_rates(k_p_ti45__v46)*Y(H1)*Y(Ti45)*state.rho + screened_rates(k_v46__p_ti45)*Y(V46)) +
        (-screened_rates(k_p_ti46__v47)*Y(H1)*Y(Ti46)*state.rho + screened_rates(k_v47__p_ti46)*Y(V47)) +
        (-screened_rates(k_p_ti47__v48)*Y(H1)*Y(Ti47)*state.rho + screened_rates(k_v48__p_ti47)*Y(V48)) +
        (-screened_rates(k_p_ti48__v49)*Y(H1)*Y(Ti48)*state.rho + screened_rates(k_v49__p_ti48)*Y(V49)) +
        (-screened_rates(k_p_ti49__v50)*Y(H1)*Y(Ti49)*state.rho + screened_rates(k_v50__p_ti49)*Y(V50)) +
        (-screened_rates(k_p_ti50__v51)*Y(H1)*Y(Ti50)*state.rho + screened_rates(k_v51__p_ti50)*Y(V51)) +
        (-screened_rates(k_p_ti51__v52)*Y(H1)*Y(Ti51)*state.rho + screened_rates(k_v52__p_ti51)*Y(V52)) +
        (-screened_rates(k_p_v47__cr48)*Y(H1)*Y(V47)*state.rho + screened_rates(k_cr48__p_v47)*Y(Cr48)) +
        (-screened_rates(k_p_v48__cr49)*Y(H1)*Y(V48)*state.rho + screened_rates(k_cr49__p_v48)*Y(Cr49)) +
        (-screened_rates(k_p_v49__cr50)*Y(H1)*Y(V49)*state.rho + screened_rates(k_cr50__p_v49)*Y(Cr50)) +
        (-screened_rates(k_p_v50__cr51)*Y(H1)*Y(V50)*state.rho + screened_rates(k_cr51__p_v50)*Y(Cr51)) +
        (-screened_rates(k_p_v51__cr52)*Y(H1)*Y(V51)*state.rho + screened_rates(k_cr52__p_v51)*Y(Cr52)) +
        (-screened_rates(k_p_v52__cr53)*Y(H1)*Y(V52)*state.rho + screened_rates(k_cr53__p_v52)*Y(Cr53)) +
        (-screened_rates(k_p_cr49__mn50)*Y(Cr49)*Y(H1)*state.rho + screened_rates(k_mn50__p_cr49)*Y(Mn50)) +
        (-screened_rates(k_p_cr50__mn51)*Y(Cr50)*Y(H1)*state.rho + screened_rates(k_mn51__p_cr50)*Y(Mn51)) +
        (-screened_rates(k_p_cr51__mn52)*Y(Cr51)*Y(H1)*state.rho + screened_rates(k_mn52__p_cr51)*Y(Mn52)) +
        (-screened_rates(k_p_cr52__mn53)*Y(Cr52)*Y(H1)*state.rho + screened_rates(k_mn53__p_cr52)*Y(Mn53)) +
        (-screened_rates(k_p_cr53__mn54)*Y(Cr53)*Y(H1)*state.rho + screened_rates(k_mn54__p_cr53)*Y(Mn54)) +
        (-screened_rates(k_p_cr54__mn55)*Y(Cr54)*Y(H1)*state.rho + screened_rates(k_mn55__p_cr54)*Y(Mn55)) +
        (-screened_rates(k_p_mn51__fe52)*Y(Mn51)*Y(H1)*state.rho + screened_rates(k_fe52__p_mn51)*Y(Fe52)) +
        (-screened_rates(k_p_mn52__fe53)*Y(Mn52)*Y(H1)*state.rho + screened_rates(k_fe53__p_mn52)*Y(Fe53)) +
        (-screened_rates(k_p_mn53__fe54)*Y(Mn53)*Y(H1)*state.rho + screened_rates(k_fe54__p_mn53)*Y(Fe54)) +
        (-screened_rates(k_p_mn54__fe55)*Y(Mn54)*Y(H1)*state.rho + screened_rates(k_fe55__p_mn54)*Y(Fe55)) +
        (-screened_rates(k_p_mn55__fe56)*Y(Mn55)*Y(H1)*state.rho + screened_rates(k_fe56__p_mn55)*Y(Fe56)) +
        (-screened_rates(k_p_fe52__co53)*Y(Fe52)*Y(H1)*state.rho + screened_rates(k_co53__p_fe52)*Y(Co53)) +
        (-screened_rates(k_p_fe53__co54)*Y(Fe53)*Y(H1)*state.rho + screened_rates(k_co54__p_fe53)*Y(Co54)) +
        (-screened_rates(k_p_fe54__co55)*Y(Fe54)*Y(H1)*state.rho + screened_rates(k_co55__p_fe54)*Y(Co55)) +
        (-screened_rates(k_p_fe55__co56)*Y(Fe55)*Y(H1)*state.rho + screened_rates(k_co56__p_fe55)*Y(Co56)) +
        (-screened_rates(k_p_fe56__co57)*Y(Fe56)*Y(H1)*state.rho + screened_rates(k_co57__p_fe56)*Y(Co57)) +
        (-screened_rates(k_p_fe57__co58)*Y(Fe57)*Y(H1)*state.rho + screened_rates(k_co58__p_fe57)*Y(Co58)) +
        (-screened_rates(k_p_fe58__co59)*Y(Fe58)*Y(H1)*state.rho + screened_rates(k_co59__p_fe58)*Y(Co59)) +
        (-screened_rates(k_p_co55__ni56)*Y(Co55)*Y(H1)*state.rho + screened_rates(k_ni56__p_co55)*Y(Ni56)) +
        (-screened_rates(k_p_co56__ni57)*Y(Co56)*Y(H1)*state.rho + screened_rates(k_ni57__p_co56)*Y(Ni57)) +
        (-screened_rates(k_p_co57__ni58)*Y(Co57)*Y(H1)*state.rho + screened_rates(k_ni58__p_co57)*Y(Ni58)) +
        (-screened_rates(k_p_co58__ni59)*Y(Co58)*Y(H1)*state.rho + screened_rates(k_ni59__p_co58)*Y(Ni59)) +
        (-screened_rates(k_p_co59__ni60)*Y(Co59)*Y(H1)*state.rho + screened_rates(k_ni60__p_co59)*Y(Ni60)) +
        (-screened_rates(k_p_ni56__cu57)*Y(Ni56)*Y(H1)*state.rho + screened_rates(k_cu57__p_ni56)*Y(Cu57)) +
        (-screened_rates(k_p_ni57__cu58)*Y(Ni57)*Y(H1)*state.rho + screened_rates(k_cu58__p_ni57)*Y(Cu58)) +
        (-screened_rates(k_p_ni58__cu59)*Y(Ni58)*Y(H1)*state.rho + screened_rates(k_cu59__p_ni58)*Y(Cu59)) +
        (-screened_rates(k_p_ni59__cu60)*Y(Ni59)*Y(H1)*state.rho + screened_rates(k_cu60__p_ni59)*Y(Cu60)) +
        (-screened_rates(k_p_ni60__cu61)*Y(Ni60)*Y(H1)*state.rho + screened_rates(k_cu61__p_ni60)*Y(Cu61)) +
        (-screened_rates(k_p_ni61__cu62)*Y(Ni61)*Y(H1)*state.rho + screened_rates(k_cu62__p_ni61)*Y(Cu62)) +
        (-screened_rates(k_p_ni62__cu63)*Y(Ni62)*Y(H1)*state.rho + screened_rates(k_cu63__p_ni62)*Y(Cu63)) +
        (-screened_rates(k_p_ni63__cu64)*Y(Ni63)*Y(H1)*state.rho + screened_rates(k_cu64__p_ni63)*Y(Cu64)) +
        (-screened_rates(k_p_ni64__cu65)*Y(Ni64)*Y(H1)*state.rho + screened_rates(k_cu65__p_ni64)*Y(Cu65)) +
        (-screened_rates(k_p_cu58__zn59)*Y(Cu58)*Y(H1)*state.rho + screened_rates(k_zn59__p_cu58)*Y(Zn59)) +
        (-screened_rates(k_p_cu59__zn60)*Y(Cu59)*Y(H1)*state.rho + screened_rates(k_zn60__p_cu59)*Y(Zn60)) +
        (-screened_rates(k_p_cu60__zn61)*Y(Cu60)*Y(H1)*state.rho + screened_rates(k_zn61__p_cu60)*Y(Zn61)) +
        (-screened_rates(k_p_cu61__zn62)*Y(Cu61)*Y(H1)*state.rho + screened_rates(k_zn62__p_cu61)*Y(Zn62)) +
        (-screened_rates(k_p_cu62__zn63)*Y(Cu62)*Y(H1)*state.rho + screened_rates(k_zn63__p_cu62)*Y(Zn63)) +
        (-screened_rates(k_p_cu63__zn64)*Y(Cu63)*Y(H1)*state.rho + screened_rates(k_zn64__p_cu63)*Y(Zn64)) +
        (-screened_rates(k_p_cu64__zn65)*Y(Cu64)*Y(H1)*state.rho + screened_rates(k_zn65__p_cu64)*Y(Zn65)) +
        (-screened_rates(k_p_cu65__zn66)*Y(Cu65)*Y(H1)*state.rho + screened_rates(k_zn66__p_cu65)*Y(Zn66)) +
        (-screened_rates(k_p_zn61__ga62)*Y(H1)*Y(Zn61)*state.rho + screened_rates(k_ga62__p_zn61)*Y(Ga62)) +
        (-screened_rates(k_p_zn62__ga63)*Y(H1)*Y(Zn62)*state.rho + screened_rates(k_ga63__p_zn62)*Y(Ga63)) +
        (-screened_rates(k_p_zn63__ga64)*Y(H1)*Y(Zn63)*state.rho + screened_rates(k_ga64__p_zn63)*Y(Ga64)) +
        (-screened_rates(k_p_ga62__ge63)*Y(Ga62)*Y(H1)*state.rho + screened_rates(k_ge63__p_ga62)*Y(Ge63)) +
        (-screened_rates(k_p_ga63__ge64)*Y(Ga63)*Y(H1)*state.rho + screened_rates(k_ge64__p_ga63)*Y(Ge64)) +
        (screened_rates(k_d_he3__p_he4)*Y(H2)*Y(He3)*state.rho + -screened_rates(k_p_he4__d_he3)*Y(He4)*Y(H1)*state.rho) +
        (-screened_rates(k_p_li6__he4_he3)*Y(Li6)*Y(H1)*state.rho + screened_rates(k_he4_he3__p_li6)*Y(He3)*Y(He4)*state.rho) +
        (screened_rates(k_d_li6__p_li7)*Y(H2)*Y(Li6)*state.rho + -screened_rates(k_p_li7__d_li6)*Y(Li7)*Y(H1)*state.rho) +
        (-screened_rates(k_p_li7__he4_he4)*Y(Li7)*Y(H1)*state.rho + 0.5*screened_rates(k_he4_he4__p_li7)*std::pow(Y(He4), 2)*state.rho) +
        (screened_rates(k_n_be7__p_li7)*Y(Be7)*Y(N)*state.rho + -screened_rates(k_p_li7__n_be7)*Y(Li7)*Y(H1)*state.rho) +
        (-screened_rates(k_p_be9__he4_li6)*Y(Be9)*Y(H1)*state.rho + screened_rates(k_he4_li6__p_be9)*Y(He4)*Y(Li6)*state.rho) +
        (-screened_rates(k_p_b10__he4_be7)*Y(B10)*Y(H1)*state.rho + screened_rates(k_he4_be7__p_b10)*Y(Be7)*Y(He4)*state.rho) +
        (screened_rates(k_he4_b10__p_c13)*Y(B10)*Y(He4)*state.rho + -screened_rates(k_p_c13__he4_b10)*Y(C13)*Y(H1)*state.rho) +
        (screened_rates(k_he4_b11__p_c14)*Y(B11)*Y(He4)*state.rho + -screened_rates(k_p_c14__he4_b11)*Y(C14)*Y(H1)*state.rho) +
        (0.5*screened_rates(k_c12_c12__p_na23)*std::pow(Y(C12), 2)*state.rho + -screened_rates(k_p_na23__c12_c12)*Y(Na23)*Y(H1)*state.rho) +
        (screened_rates(k_n_n13__p_c13)*Y(N13)*Y(N)*state.rho + -screened_rates(k_p_c13__n_n13)*Y(C13)*Y(H1)*state.rho) +
        (screened_rates(k_he4_n13__p_o16)*Y(He4)*Y(N13)*state.rho + -screened_rates(k_p_o16__he4_n13)*Y(O16)*Y(H1)*state.rho) +
        (screened_rates(k_n_n14__p_c14)*Y(N14)*Y(N)*state.rho + -screened_rates(k_p_c14__n_n14)*Y(C14)*Y(H1)*state.rho) +
        (-screened_rates(k_p_n15__he4_c12)*Y(N15)*Y(H1)*state.rho + screened_rates(k_he4_c12__p_n15)*Y(C12)*Y(He4)*state.rho) +
        (screened_rates(k_n_o14__p_n14)*Y(N)*Y(O14)*state.rho + -screened_rates(k_p_n14__n_o14)*Y(N14)*Y(H1)*state.rho) +
        (screened_rates(k_he4_o14__p_f17)*Y(He4)*Y(O14)*state.rho + -screened_rates(k_p_f17__he4_o14)*Y(F17)*Y(H1)*state.rho) +
        (screened_rates(k_n_o15__p_n15)*Y(N)*Y(O15)*state.rho + -screened_rates(k_p_n15__n_o15)*Y(N15)*Y(H1)*state.rho) +
        (screened_rates(k_c12_o16__p_al27)*Y(C12)*Y(O16)*state.rho + -screened_rates(k_p_al27__c12_o16)*Y(Al27)*Y(H1)*state.rho) +
        (0.5*screened_rates(k_o16_o16__p_p31)*std::pow(Y(O16), 2)*state.rho + -screened_rates(k_p_p31__o16_o16)*Y(P31)*Y(H1)*state.rho) +
        (-screened_rates(k_p_o17__he4_n14)*Y(O17)*Y(H1)*state.rho + screened_rates(k_he4_n14__p_o17)*Y(He4)*Y(N14)*state.rho) +
        (-screened_rates(k_p_o18__he4_n15)*Y(O18)*Y(H1)*state.rho + screened_rates(k_he4_n15__p_o18)*Y(He4)*Y(N15)*state.rho) +
        (screened_rates(k_n_f17__p_o17)*Y(F17)*Y(N)*state.rho + -screened_rates(k_p_o17__n_f17)*Y(O17)*Y(H1)*state.rho) +
        (screened_rates(k_he4_f17__p_ne20)*Y(F17)*Y(He4)*state.rho + -screened_rates(k_p_ne20__he4_f17)*Y(Ne20)*Y(H1)*state.rho) +
        (screened_rates(k_n_f18__p_o18)*Y(F18)*Y(N)*state.rho + -screened_rates(k_p_o18__n_f18)*Y(O18)*Y(H1)*state.rho) +
        (-screened_rates(k_p_f18__he4_o15)*Y(F18)*Y(H1)*state.rho + screened_rates(k_he4_o15__p_f18)*Y(He4)*Y(O15)*state.rho) +
        (screened_rates(k_he4_f18__p_ne21)*Y(F18)*Y(He4)*state.rho + -screened_rates(k_p_ne21__he4_f18)*Y(Ne21)*Y(H1)*state.rho) +
        (-screened_rates(k_p_f19__he4_o16)*Y(F19)*Y(H1)*state.rho + screened_rates(k_he4_o16__p_f19)*Y(He4)*Y(O16)*state.rho) +
        (screened_rates(k_he4_f19__p_ne22)*Y(F19)*Y(He4)*state.rho + -screened_rates(k_p_ne22__he4_f19)*Y(Ne22)*Y(H1)*state.rho) +
        (screened_rates(k_n_ne18__p_f18)*Y(N)*Y(Ne18)*state.rho + -screened_rates(k_p_f18__n_ne18)*Y(F18)*Y(H1)*state.rho) +
        (screened_rates(k_he4_ne18__p_na21)*Y(He4)*Y(Ne18)*state.rho + -screened_rates(k_p_na21__he4_ne18)*Y(Na21)*Y(H1)*state.rho) +
        (screened_rates(k_n_ne19__p_f19)*Y(N)*Y(Ne19)*state.rho + -screened_rates(k_p_f19__n_ne19)*Y(F19)*Y(H1)*state.rho) +
        (screened_rates(k_he4_ne19__p_na22)*Y(He4)*Y(Ne19)*state.rho + -screened_rates(k_p_na22__he4_ne19)*Y(Na22)*Y(H1)*state.rho) +
        (screened_rates(k_c12_ne20__p_p31)*Y(C12)*Y(Ne20)*state.rho + -screened_rates(k_p_p31__c12_ne20)*Y(P31)*Y(H1)*state.rho) +
        (screened_rates(k_n_na21__p_ne21)*Y(N)*Y(Na21)*state.rho + -screened_rates(k_p_ne21__n_na21)*Y(Ne21)*Y(H1)*state.rho) +
        (screened_rates(k_he4_na21__p_mg24)*Y(He4)*Y(Na21)*state.rho + -screened_rates(k_p_mg24__he4_na21)*Y(Mg24)*Y(H1)*state.rho) +
        (screened_rates(k_n_na22__p_ne22)*Y(N)*Y(Na22)*state.rho + -screened_rates(k_p_ne22__n_na22)*Y(Ne22)*Y(H1)*state.rho) +
        (screened_rates(k_he4_na22__p_mg25)*Y(He4)*Y(Na22)*state.rho + -screened_rates(k_p_mg25__he4_na22)*Y(Mg25)*Y(H1)*state.rho) +
        (-screened_rates(k_p_na23__he4_ne20)*Y(Na23)*Y(H1)*state.rho + screened_rates(k_he4_ne20__p_na23)*Y(He4)*Y(Ne20)*state.rho) +
        (screened_rates(k_he4_na23__p_mg26)*Y(He4)*Y(Na23)*state.rho + -screened_rates(k_p_mg26__he4_na23)*Y(Mg26)*Y(H1)*state.rho) +
        (screened_rates(k_n_mg23__p_na23)*Y(Mg23)*Y(N)*state.rho + -screened_rates(k_p_na23__n_mg23)*Y(Na23)*Y(H1)*state.rho) +
        (screened_rates(k_he4_mg23__p_al26)*Y(He4)*Y(Mg23)*state.rho + -screened_rates(k_p_al26__he4_mg23)*Y(Al26)*Y(H1)*state.rho) +
        (screened_rates(k_n_al25__p_mg25)*Y(Al25)*Y(N)*state.rho + -screened_rates(k_p_mg25__n_al25)*Y(Mg25)*Y(H1)*state.rho) +
        (screened_rates(k_he4_al25__p_si28)*Y(Al25)*Y(He4)*state.rho + -screened_rates(k_p_si28__he4_al25)*Y(H1)*Y(Si28)*state.rho) +
        (screened_rates(k_n_al26__p_mg26)*Y(Al26)*Y(N)*state.rho + -screened_rates(k_p_mg26__n_al26)*Y(Mg26)*Y(H1)*state.rho) +
        (screened_rates(k_he4_al26__p_si29)*Y(Al26)*Y(He4)*state.rho + -screened_rates(k_p_si29__he4_al26)*Y(H1)*Y(Si29)*state.rho) +
        (-screened_rates(k_p_al27__he4_mg24)*Y(Al27)*Y(H1)*state.rho + screened_rates(k_he4_mg24__p_al27)*Y(He4)*Y(Mg24)*state.rho) +
        (screened_rates(k_he4_al27__p_si30)*Y(Al27)*Y(He4)*state.rho + -screened_rates(k_p_si30__he4_al27)*Y(H1)*Y(Si30)*state.rho) +
        (-screened_rates(k_p_si31__n_p31)*Y(H1)*Y(Si31)*state.rho + screened_rates(k_n_p31__p_si31)*Y(N)*Y(P31)*state.rho) +
        (screened_rates(k_n_p29__p_si29)*Y(N)*Y(P29)*state.rho + -screened_rates(k_p_si29__n_p29)*Y(H1)*Y(Si29)*state.rho) +
        (screened_rates(k_he4_p29__p_s32)*Y(He4)*Y(P29)*state.rho + -screened_rates(k_p_s32__he4_p29)*Y(H1)*Y(S32)*state.rho) +
        (screened_rates(k_n_p30__p_si30)*Y(N)*Y(P30)*state.rho + -screened_rates(k_p_si30__n_p30)*Y(H1)*Y(Si30)*state.rho) +
        (screened_rates(k_he4_p30__p_s33)*Y(He4)*Y(P30)*state.rho + -screened_rates(k_p_s33__he4_p30)*Y(H1)*Y(S33)*state.rho) +
        (-screened_rates(k_p_p31__he4_si28)*Y(P31)*Y(H1)*state.rho + screened_rates(k_he4_si28__p_p31)*Y(He4)*Y(Si28)*state.rho) +
        (screened_rates(k_he4_p31__p_s34)*Y(He4)*Y(P31)*state.rho + -screened_rates(k_p_s34__he4_p31)*Y(H1)*Y(S34)*state.rho) +
        (screened_rates(k_n_p32__p_si32)*Y(N)*Y(P32)*state.rho + -screened_rates(k_p_si32__n_p32)*Y(H1)*Y(Si32)*state.rho) +
        (-screened_rates(k_p_p32__n_s32)*Y(P32)*Y(H1)*state.rho + screened_rates(k_n_s32__p_p32)*Y(N)*Y(S32)*state.rho) +
        (-screened_rates(k_p_p32__he4_si29)*Y(P32)*Y(H1)*state.rho + screened_rates(k_he4_si29__p_p32)*Y(He4)*Y(Si29)*state.rho) +
        (-screened_rates(k_p_p33__he4_si30)*Y(P33)*Y(H1)*state.rho + screened_rates(k_he4_si30__p_p33)*Y(He4)*Y(Si30)*state.rho) +
        (screened_rates(k_n_s33__p_p33)*Y(N)*Y(S33)*state.rho + -screened_rates(k_p_p33__n_s33)*Y(P33)*Y(H1)*state.rho) +
        (-screened_rates(k_p_s35__he4_p32)*Y(H1)*Y(S35)*state.rho + screened_rates(k_he4_p32__p_s35)*Y(He4)*Y(P32)*state.rho) +
        (-screened_rates(k_p_s36__he4_p33)*Y(H1)*Y(S36)*state.rho + screened_rates(k_he4_p33__p_s36)*Y(He4)*Y(P33)*state.rho) +
        (screened_rates(k_n_cl33__p_s33)*Y(Cl33)*Y(N)*state.rho + -screened_rates(k_p_s33__n_cl33)*Y(H1)*Y(S33)*state.rho) +
        (screened_rates(k_he4_cl33__p_ar36)*Y(Cl33)*Y(He4)*state.rho + -screened_rates(k_p_ar36__he4_cl33)*Y(Ar36)*Y(H1)*state.rho) +
        (screened_rates(k_n_cl34__p_s34)*Y(Cl34)*Y(N)*state.rho + -screened_rates(k_p_s34__n_cl34)*Y(H1)*Y(S34)*state.rho) +
        (screened_rates(k_he4_cl34__p_ar37)*Y(Cl34)*Y(He4)*state.rho + -screened_rates(k_p_ar37__he4_cl34)*Y(Ar37)*Y(H1)*state.rho) +
        (screened_rates(k_n_cl35__p_s35)*Y(Cl35)*Y(N)*state.rho + -screened_rates(k_p_s35__n_cl35)*Y(H1)*Y(S35)*state.rho) +
        (-screened_rates(k_p_cl35__he4_s32)*Y(Cl35)*Y(H1)*state.rho + screened_rates(k_he4_s32__p_cl35)*Y(He4)*Y(S32)*state.rho) +
        (screened_rates(k_he4_cl35__p_ar38)*Y(Cl35)*Y(He4)*state.rho + -screened_rates(k_p_ar38__he4_cl35)*Y(Ar38)*Y(H1)*state.rho) +
        (screened_rates(k_n_cl36__p_s36)*Y(Cl36)*Y(N)*state.rho + -screened_rates(k_p_s36__n_cl36)*Y(H1)*Y(S36)*state.rho) +
        (-screened_rates(k_p_cl36__he4_s33)*Y(Cl36)*Y(H1)*state.rho + screened_rates(k_he4_s33__p_cl36)*Y(He4)*Y(S33)*state.rho) +
        (-screened_rates(k_p_cl37__he4_s34)*Y(Cl37)*Y(H1)*state.rho + screened_rates(k_he4_s34__p_cl37)*Y(He4)*Y(S34)*state.rho) +
        (screened_rates(k_n_ar36__p_cl36)*Y(Ar36)*Y(N)*state.rho + -screened_rates(k_p_cl36__n_ar36)*Y(Cl36)*Y(H1)*state.rho) +
        (screened_rates(k_n_ar37__p_cl37)*Y(Ar37)*Y(N)*state.rho + -screened_rates(k_p_cl37__n_ar37)*Y(Cl37)*Y(H1)*state.rho) +
        (-screened_rates(k_p_ar39__he4_cl36)*Y(Ar39)*Y(H1)*state.rho + screened_rates(k_he4_cl36__p_ar39)*Y(Cl36)*Y(He4)*state.rho) +
        (-screened_rates(k_p_ar40__he4_cl37)*Y(Ar40)*Y(H1)*state.rho + screened_rates(k_he4_cl37__p_ar40)*Y(Cl37)*Y(He4)*state.rho) +
        (screened_rates(k_n_k37__p_ar37)*Y(K37)*Y(N)*state.rho + -screened_rates(k_p_ar37__n_k37)*Y(Ar37)*Y(H1)*state.rho) +
        (screened_rates(k_he4_k37__p_ca40)*Y(He4)*Y(K37)*state.rho + -screened_rates(k_p_ca40__he4_k37)*Y(Ca40)*Y(H1)*state.rho) +
        (screened_rates(k_n_k38__p_ar38)*Y(K38)*Y(N)*state.rho + -screened_rates(k_p_ar38__n_k38)*Y(Ar38)*Y(H1)*state.rho) +
        (screened_rates(k_he4_k38__p_ca41)*Y(He4)*Y(K38)*state.rho + -screened_rates(k_p_ca41__he4_k38)*Y(Ca41)*Y(H1)*state.rho) +
        (screened_rates(k_n_k39__p_ar39)*Y(K39)*Y(N)*state.rho + -screened_rates(k_p_ar39__n_k39)*Y(Ar39)*Y(H1)*state.rho) +
        (-screened_rates(k_p_k39__he4_ar36)*Y(K39)*Y(H1)*state.rho + screened_rates(k_he4_ar36__p_k39)*Y(Ar36)*Y(He4)*state.rho) +
        (screened_rates(k_n_k40__p_ar40)*Y(K40)*Y(N)*state.rho + -screened_rates(k_p_ar40__n_k40)*Y(Ar40)*Y(H1)*state.rho) +
        (-screened_rates(k_p_k40__n_ca40)*Y(K40)*Y(H1)*state.rho + screened_rates(k_n_ca40__p_k40)*Y(Ca40)*Y(N)*state.rho) +
        (-screened_rates(k_p_k40__he4_ar37)*Y(K40)*Y(H1)*state.rho + screened_rates(k_he4_ar37__p_k40)*Y(Ar37)*Y(He4)*state.rho) +
        (screened_rates(k_he4_k40__p_ca43)*Y(He4)*Y(K40)*state.rho + -screened_rates(k_p_ca43__he4_k40)*Y(Ca43)*Y(H1)*state.rho) +
        (-screened_rates(k_p_k41__he4_ar38)*Y(K41)*Y(H1)*state.rho + screened_rates(k_he4_ar38__p_k41)*Y(Ar38)*Y(He4)*state.rho) +
        (screened_rates(k_he4_k41__p_ca44)*Y(He4)*Y(K41)*state.rho + -screened_rates(k_p_ca44__he4_k41)*Y(Ca44)*Y(H1)*state.rho) +
        (screened_rates(k_n_ca41__p_k41)*Y(Ca41)*Y(N)*state.rho + -screened_rates(k_p_k41__n_ca41)*Y(K41)*Y(H1)*state.rho) +
        (-screened_rates(k_p_ca42__he4_k39)*Y(Ca42)*Y(H1)*state.rho + screened_rates(k_he4_k39__p_ca42)*Y(He4)*Y(K39)*state.rho) +
        (-screened_rates(k_p_ca47__n_sc47)*Y(Ca47)*Y(H1)*state.rho + screened_rates(k_n_sc47__p_ca47)*Y(N)*Y(Sc47)*state.rho) +
        (screened_rates(k_n_sc43__p_ca43)*Y(N)*Y(Sc43)*state.rho + -screened_rates(k_p_ca43__n_sc43)*Y(Ca43)*Y(H1)*state.rho) +
        (-screened_rates(k_p_sc43__he4_ca40)*Y(H1)*Y(Sc43)*state.rho + screened_rates(k_he4_ca40__p_sc43)*Y(Ca40)*Y(He4)*state.rho) +
        (screened_rates(k_he4_sc43__p_ti46)*Y(He4)*Y(Sc43)*state.rho + -screened_rates(k_p_ti46__he4_sc43)*Y(H1)*Y(Ti46)*state.rho) +
        (screened_rates(k_n_sc44__p_ca44)*Y(N)*Y(Sc44)*state.rho + -screened_rates(k_p_ca44__n_sc44)*Y(Ca44)*Y(H1)*state.rho) +
        (-screened_rates(k_p_sc44__he4_ca41)*Y(H1)*Y(Sc44)*state.rho + screened_rates(k_he4_ca41__p_sc44)*Y(Ca41)*Y(He4)*state.rho) +
        (screened_rates(k_he4_sc44__p_ti47)*Y(He4)*Y(Sc44)*state.rho + -screened_rates(k_p_ti47__he4_sc44)*Y(H1)*Y(Ti47)*state.rho) +
        (screened_rates(k_n_sc45__p_ca45)*Y(N)*Y(Sc45)*state.rho + -screened_rates(k_p_ca45__n_sc45)*Y(Ca45)*Y(H1)*state.rho) +
        (-screened_rates(k_p_sc45__he4_ca42)*Y(H1)*Y(Sc45)*state.rho + screened_rates(k_he4_ca42__p_sc45)*Y(Ca42)*Y(He4)*state.rho) +
        (screened_rates(k_he4_sc45__p_ti48)*Y(He4)*Y(Sc45)*state.rho + -screened_rates(k_p_ti48__he4_sc45)*Y(H1)*Y(Ti48)*state.rho) +
        (screened_rates(k_n_sc46__p_ca46)*Y(N)*Y(Sc46)*state.rho + -screened_rates(k_p_ca46__n_sc46)*Y(Ca46)*Y(H1)*state.rho) +
        (-screened_rates(k_p_sc46__n_ti46)*Y(H1)*Y(Sc46)*state.rho + screened_rates(k_n_ti46__p_sc46)*Y(N)*Y(Ti46)*state.rho) +
        (-screened_rates(k_p_sc46__he4_ca43)*Y(H1)*Y(Sc46)*state.rho + screened_rates(k_he4_ca43__p_sc46)*Y(Ca43)*Y(He4)*state.rho) +
        (screened_rates(k_he4_sc46__p_ti49)*Y(He4)*Y(Sc46)*state.rho + -screened_rates(k_p_ti49__he4_sc46)*Y(H1)*Y(Ti49)*state.rho) +
        (-screened_rates(k_p_sc47__he4_ca44)*Y(H1)*Y(Sc47)*state.rho + screened_rates(k_he4_ca44__p_sc47)*Y(Ca44)*Y(He4)*state.rho) +
        (screened_rates(k_he4_sc47__p_ti50)*Y(He4)*Y(Sc47)*state.rho + -screened_rates(k_p_ti50__he4_sc47)*Y(H1)*Y(Ti50)*state.rho) +
        (screened_rates(k_n_sc48__p_ca48)*Y(N)*Y(Sc48)*state.rho + -screened_rates(k_p_ca48__n_sc48)*Y(Ca48)*Y(H1)*state.rho) +
        (-screened_rates(k_p_sc48__n_ti48)*Y(H1)*Y(Sc48)*state.rho + screened_rates(k_n_ti48__p_sc48)*Y(N)*Y(Ti48)*state.rho) +
        (-screened_rates(k_p_sc48__he4_ca45)*Y(H1)*Y(Sc48)*state.rho + screened_rates(k_he4_ca45__p_sc48)*Y(Ca45)*Y(He4)*state.rho) +
        (screened_rates(k_he4_sc48__p_ti51)*Y(He4)*Y(Sc48)*state.rho + -screened_rates(k_p_ti51__he4_sc48)*Y(H1)*Y(Ti51)*state.rho) +
        (-screened_rates(k_p_sc49__n_ti49)*Y(H1)*Y(Sc49)*state.rho + screened_rates(k_n_ti49__p_sc49)*Y(N)*Y(Ti49)*state.rho) +
        (-screened_rates(k_p_sc49__he4_ca46)*Y(H1)*Y(Sc49)*state.rho + screened_rates(k_he4_ca46__p_sc49)*Y(Ca46)*Y(He4)*state.rho) +
        (screened_rates(k_n_ti44__p_sc44)*Y(N)*Y(Ti44)*state.rho + -screened_rates(k_p_sc44__n_ti44)*Y(H1)*Y(Sc44)*state.rho) +
        (screened_rates(k_n_ti45__p_sc45)*Y(N)*Y(Ti45)*state.rho + -screened_rates(k_p_sc45__n_ti45)*Y(H1)*Y(Sc45)*state.rho) +
        (screened_rates(k_he4_ti45__p_v48)*Y(He4)*Y(Ti45)*state.rho + -screened_rates(k_p_v48__he4_ti45)*Y(H1)*Y(V48)*state.rho) +
        (screened_rates(k_n_ti47__p_sc47)*Y(N)*Y(Ti47)*state.rho + -screened_rates(k_p_sc47__n_ti47)*Y(H1)*Y(Sc47)*state.rho) +
        (-screened_rates(k_p_ti51__n_v51)*Y(H1)*Y(Ti51)*state.rho + screened_rates(k_n_v51__p_ti51)*Y(N)*Y(V51)*state.rho) +
        (screened_rates(k_n_v46__p_ti46)*Y(N)*Y(V46)*state.rho + -screened_rates(k_p_ti46__n_v46)*Y(H1)*Y(Ti46)*state.rho) +
        (screened_rates(k_he4_v46__p_cr49)*Y(He4)*Y(V46)*state.rho + -screened_rates(k_p_cr49__he4_v46)*Y(Cr49)*Y(H1)*state.rho) +
        (screened_rates(k_n_v47__p_ti47)*Y(N)*Y(V47)*state.rho + -screened_rates(k_p_ti47__n_v47)*Y(H1)*Y(Ti47)*state.rho) +
        (-screened_rates(k_p_v47__he4_ti44)*Y(H1)*Y(V47)*state.rho + screened_rates(k_he4_ti44__p_v47)*Y(He4)*Y(Ti44)*state.rho) +
        (screened_rates(k_he4_v47__p_cr50)*Y(He4)*Y(V47)*state.rho + -screened_rates(k_p_cr50__he4_v47)*Y(Cr50)*Y(H1)*state.rho) +
        (screened_rates(k_n_v48__p_ti48)*Y(N)*Y(V48)*state.rho + -screened_rates(k_p_ti48__n_v48)*Y(H1)*Y(Ti48)*state.rho) +
        (screened_rates(k_he4_v48__p_cr51)*Y(He4)*Y(V48)*state.rho + -screened_rates(k_p_cr51__he4_v48)*Y(Cr51)*Y(H1)*state.rho) +
        (screened_rates(k_n_v49__p_ti49)*Y(N)*Y(V49)*state.rho + -screened_rates(k_p_ti49__n_v49)*Y(H1)*Y(Ti49)*state.rho) +
        (-screened_rates(k_p_v49__he4_ti46)*Y(H1)*Y(V49)*state.rho + screened_rates(k_he4_ti46__p_v49)*Y(He4)*Y(Ti46)*state.rho) +
        (screened_rates(k_he4_v49__p_cr52)*Y(He4)*Y(V49)*state.rho + -screened_rates(k_p_cr52__he4_v49)*Y(Cr52)*Y(H1)*state.rho) +
        (screened_rates(k_n_v50__p_ti50)*Y(N)*Y(V50)*state.rho + -screened_rates(k_p_ti50__n_v50)*Y(H1)*Y(Ti50)*state.rho) +
        (-screened_rates(k_p_v50__n_cr50)*Y(H1)*Y(V50)*state.rho + screened_rates(k_n_cr50__p_v50)*Y(Cr50)*Y(N)*state.rho) +
        (-screened_rates(k_p_v50__he4_ti47)*Y(H1)*Y(V50)*state.rho + screened_rates(k_he4_ti47__p_v50)*Y(He4)*Y(Ti47)*state.rho) +
        (screened_rates(k_he4_v50__p_cr53)*Y(He4)*Y(V50)*state.rho + -screened_rates(k_p_cr53__he4_v50)*Y(Cr53)*Y(H1)*state.rho) +
        (-screened_rates(k_p_v51__he4_ti48)*Y(H1)*Y(V51)*state.rho + screened_rates(k_he4_ti48__p_v51)*Y(He4)*Y(Ti48)*state.rho) +
        (-screened_rates(k_p_v52__n_cr52)*Y(H1)*Y(V52)*state.rho + screened_rates(k_n_cr52__p_v52)*Y(Cr52)*Y(N)*state.rho) +
        (-screened_rates(k_p_v52__he4_ti49)*Y(H1)*Y(V52)*state.rho + screened_rates(k_he4_ti49__p_v52)*Y(He4)*Y(Ti49)*state.rho) +
        (screened_rates(k_n_cr48__p_v48)*Y(Cr48)*Y(N)*state.rho + -screened_rates(k_p_v48__n_cr48)*Y(H1)*Y(V48)*state.rho) +
        (screened_rates(k_he4_cr48__p_mn51)*Y(Cr48)*Y(He4)*state.rho + -screened_rates(k_p_mn51__he4_cr48)*Y(Mn51)*Y(H1)*state.rho) +
        (screened_rates(k_n_cr49__p_v49)*Y(Cr49)*Y(N)*state.rho + -screened_rates(k_p_v49__n_cr49)*Y(H1)*Y(V49)*state.rho) +
        (screened_rates(k_he4_cr49__p_mn52)*Y(Cr49)*Y(He4)*state.rho + -screened_rates(k_p_mn52__he4_cr49)*Y(Mn52)*Y(H1)*state.rho) +
        (screened_rates(k_n_cr51__p_v51)*Y(Cr51)*Y(N)*state.rho + -screened_rates(k_p_v51__n_cr51)*Y(H1)*Y(V51)*state.rho) +
        (-screened_rates(k_p_cr54__he4_v51)*Y(Cr54)*Y(H1)*state.rho + screened_rates(k_he4_v51__p_cr54)*Y(He4)*Y(V51)*state.rho) +
        (screened_rates(k_n_mn50__p_cr50)*Y(Mn50)*Y(N)*state.rho + -screened_rates(k_p_cr50__n_mn50)*Y(Cr50)*Y(H1)*state.rho) +
        (screened_rates(k_he4_mn50__p_fe53)*Y(He4)*Y(Mn50)*state.rho + -screened_rates(k_p_fe53__he4_mn50)*Y(Fe53)*Y(H1)*state.rho) +
        (screened_rates(k_n_mn51__p_cr51)*Y(Mn51)*Y(N)*state.rho + -screened_rates(k_p_cr51__n_mn51)*Y(Cr51)*Y(H1)*state.rho) +
        (screened_rates(k_he4_mn51__p_fe54)*Y(He4)*Y(Mn51)*state.rho + -screened_rates(k_p_fe54__he4_mn51)*Y(Fe54)*Y(H1)*state.rho) +
        (screened_rates(k_n_mn52__p_cr52)*Y(Mn52)*Y(N)*state.rho + -screened_rates(k_p_cr52__n_mn52)*Y(Cr52)*Y(H1)*state.rho) +
        (screened_rates(k_he4_mn52__p_fe55)*Y(He4)*Y(Mn52)*state.rho + -screened_rates(k_p_fe55__he4_mn52)*Y(Fe55)*Y(H1)*state.rho) +
        (screened_rates(k_n_mn53__p_cr53)*Y(Mn53)*Y(N)*state.rho + -screened_rates(k_p_cr53__n_mn53)*Y(Cr53)*Y(H1)*state.rho) +
        (-screened_rates(k_p_mn53__he4_cr50)*Y(Mn53)*Y(H1)*state.rho + screened_rates(k_he4_cr50__p_mn53)*Y(Cr50)*Y(He4)*state.rho) +
        (screened_rates(k_he4_mn53__p_fe56)*Y(He4)*Y(Mn53)*state.rho + -screened_rates(k_p_fe56__he4_mn53)*Y(Fe56)*Y(H1)*state.rho) +
        (screened_rates(k_n_mn54__p_cr54)*Y(Mn54)*Y(N)*state.rho + -screened_rates(k_p_cr54__n_mn54)*Y(Cr54)*Y(H1)*state.rho) +
        (-screened_rates(k_p_mn54__he4_cr51)*Y(Mn54)*Y(H1)*state.rho + screened_rates(k_he4_cr51__p_mn54)*Y(Cr51)*Y(He4)*state.rho) +
        (-screened_rates(k_p_mn55__he4_cr52)*Y(Mn55)*Y(H1)*state.rho + screened_rates(k_he4_cr52__p_mn55)*Y(Cr52)*Y(He4)*state.rho) +
        (screened_rates(k_n_fe52__p_mn52)*Y(Fe52)*Y(N)*state.rho + -screened_rates(k_p_mn52__n_fe52)*Y(Mn52)*Y(H1)*state.rho) +
        (screened_rates(k_he4_fe52__p_co55)*Y(Fe52)*Y(He4)*state.rho + -screened_rates(k_p_co55__he4_fe52)*Y(Co55)*Y(H1)*state.rho) +
        (screened_rates(k_n_fe53__p_mn53)*Y(Fe53)*Y(N)*state.rho + -screened_rates(k_p_mn53__n_fe53)*Y(Mn53)*Y(H1)*state.rho) +
        (screened_rates(k_he4_fe53__p_co56)*Y(Fe53)*Y(He4)*state.rho + -screened_rates(k_p_co56__he4_fe53)*Y(Co56)*Y(H1)*state.rho) +
        (screened_rates(k_n_fe54__p_mn54)*Y(Fe54)*Y(N)*state.rho + -screened_rates(k_p_mn54__n_fe54)*Y(Mn54)*Y(H1)*state.rho) +
        (screened_rates(k_n_fe55__p_mn55)*Y(Fe55)*Y(N)*state.rho + -screened_rates(k_p_mn55__n_fe55)*Y(Mn55)*Y(H1)*state.rho) +
        (-screened_rates(k_p_fe57__he4_mn54)*Y(Fe57)*Y(H1)*state.rho + screened_rates(k_he4_mn54__p_fe57)*Y(He4)*Y(Mn54)*state.rho) +
        (-screened_rates(k_p_fe58__he4_mn55)*Y(Fe58)*Y(H1)*state.rho + screened_rates(k_he4_mn55__p_fe58)*Y(He4)*Y(Mn55)*state.rho) +
        (screened_rates(k_n_co53__p_fe53)*Y(Co53)*Y(N)*state.rho + -screened_rates(k_p_fe53__n_co53)*Y(Fe53)*Y(H1)*state.rho) +
        (screened_rates(k_he4_co53__p_ni56)*Y(Co53)*Y(He4)*state.rho + -screened_rates(k_p_ni56__he4_co53)*Y(Ni56)*Y(H1)*state.rho) +
        (screened_rates(k_n_co54__p_fe54)*Y(Co54)*Y(N)*state.rho + -screened_rates(k_p_fe54__n_co54)*Y(Fe54)*Y(H1)*state.rho) +
        (screened_rates(k_he4_co54__p_ni57)*Y(Co54)*Y(He4)*state.rho + -screened_rates(k_p_ni57__he4_co54)*Y(Ni57)*Y(H1)*state.rho) +
        (screened_rates(k_n_co55__p_fe55)*Y(Co55)*Y(N)*state.rho + -screened_rates(k_p_fe55__n_co55)*Y(Fe55)*Y(H1)*state.rho) +
        (screened_rates(k_he4_co55__p_ni58)*Y(Co55)*Y(He4)*state.rho + -screened_rates(k_p_ni58__he4_co55)*Y(Ni58)*Y(H1)*state.rho) +
        (screened_rates(k_n_co56__p_fe56)*Y(Co56)*Y(N)*state.rho + -screened_rates(k_p_fe56__n_co56)*Y(Fe56)*Y(H1)*state.rho) +
        (screened_rates(k_he4_co56__p_ni59)*Y(Co56)*Y(He4)*state.rho + -screened_rates(k_p_ni59__he4_co56)*Y(Ni59)*Y(H1)*state.rho) +
        (screened_rates(k_n_co57__p_fe57)*Y(Co57)*Y(N)*state.rho + -screened_rates(k_p_fe57__n_co57)*Y(Fe57)*Y(H1)*state.rho) +
        (-screened_rates(k_p_co57__he4_fe54)*Y(Co57)*Y(H1)*state.rho + screened_rates(k_he4_fe54__p_co57)*Y(Fe54)*Y(He4)*state.rho) +
        (screened_rates(k_he4_co57__p_ni60)*Y(Co57)*Y(He4)*state.rho + -screened_rates(k_p_ni60__he4_co57)*Y(Ni60)*Y(H1)*state.rho) +
        (screened_rates(k_n_co58__p_fe58)*Y(Co58)*Y(N)*state.rho + -screened_rates(k_p_fe58__n_co58)*Y(Fe58)*Y(H1)*state.rho) +
        (-screened_rates(k_p_co58__he4_fe55)*Y(Co58)*Y(H1)*state.rho + screened_rates(k_he4_fe55__p_co58)*Y(Fe55)*Y(He4)*state.rho) +
        (-screened_rates(k_p_co59__he4_fe56)*Y(Co59)*Y(H1)*state.rho + screened_rates(k_he4_fe56__p_co59)*Y(Fe56)*Y(He4)*state.rho) +
        (screened_rates(k_n_ni56__p_co56)*Y(N)*Y(Ni56)*state.rho + -screened_rates(k_p_co56__n_ni56)*Y(Co56)*Y(H1)*state.rho) +
        (screened_rates(k_n_ni57__p_co57)*Y(N)*Y(Ni57)*state.rho + -screened_rates(k_p_co57__n_ni57)*Y(Co57)*Y(H1)*state.rho) +
        (screened_rates(k_n_ni58__p_co58)*Y(N)*Y(Ni58)*state.rho + -screened_rates(k_p_co58__n_ni58)*Y(Co58)*Y(H1)*state.rho) +
        (screened_rates(k_n_ni59__p_co59)*Y(N)*Y(Ni59)*state.rho + -screened_rates(k_p_co59__n_ni59)*Y(Co59)*Y(H1)*state.rho) +
        (-screened_rates(k_p_ni61__he4_co58)*Y(Ni61)*Y(H1)*state.rho + screened_rates(k_he4_co58__p_ni61)*Y(Co58)*Y(He4)*state.rho) +
        (-screened_rates(k_p_ni62__he4_co59)*Y(Ni62)*Y(H1)*state.rho + screened_rates(k_he4_co59__p_ni62)*Y(Co59)*Y(He4)*state.rho) +
        (screened_rates(k_n_cu57__p_ni57)*Y(Cu57)*Y(N)*state.rho + -screened_rates(k_p_ni57__n_cu57)*Y(Ni57)*Y(H1)*state.rho) +
        (screened_rates(k_he4_cu57__p_zn60)*Y(Cu57)*Y(He4)*state.rho + -screened_rates(k_p_zn60__he4_cu57)*Y(H1)*Y(Zn60)*state.rho) +
        (screened_rates(k_n_cu58__p_ni58)*Y(Cu58)*Y(N)*state.rho + -screened_rates(k_p_ni58__n_cu58)*Y(Ni58)*Y(H1)*state.rho) +
        (screened_rates(k_n_cu59__p_ni59)*Y(Cu59)*Y(N)*state.rho + -screened_rates(k_p_ni59__n_cu59)*Y(Ni59)*Y(H1)*state.rho) +
        (-screened_rates(k_p_cu59__he4_ni56)*Y(Cu59)*Y(H1)*state.rho + screened_rates(k_he4_ni56__p_cu59)*Y(He4)*Y(Ni56)*state.rho) +
        (screened_rates(k_n_cu60__p_ni60)*Y(Cu60)*Y(N)*state.rho + -screened_rates(k_p_ni60__n_cu60)*Y(Ni60)*Y(H1)*state.rho) +
        (-screened_rates(k_p_cu60__he4_ni57)*Y(Cu60)*Y(H1)*state.rho + screened_rates(k_he4_ni57__p_cu60)*Y(He4)*Y(Ni57)*state.rho) +
        (screened_rates(k_n_cu61__p_ni61)*Y(Cu61)*Y(N)*state.rho + -screened_rates(k_p_ni61__n_cu61)*Y(Ni61)*Y(H1)*state.rho) +
        (-screened_rates(k_p_cu61__he4_ni58)*Y(Cu61)*Y(H1)*state.rho + screened_rates(k_he4_ni58__p_cu61)*Y(He4)*Y(Ni58)*state.rho) +
        (screened_rates(k_n_cu62__p_ni62)*Y(Cu62)*Y(N)*state.rho + -screened_rates(k_p_ni62__n_cu62)*Y(Ni62)*Y(H1)*state.rho) +
        (-screened_rates(k_p_cu62__he4_ni59)*Y(Cu62)*Y(H1)*state.rho + screened_rates(k_he4_ni59__p_cu62)*Y(He4)*Y(Ni59)*state.rho) +
        (screened_rates(k_n_cu63__p_ni63)*Y(Cu63)*Y(N)*state.rho + -screened_rates(k_p_ni63__n_cu63)*Y(Ni63)*Y(H1)*state.rho) +
        (-screened_rates(k_p_cu63__he4_ni60)*Y(Cu63)*Y(H1)*state.rho + screened_rates(k_he4_ni60__p_cu63)*Y(He4)*Y(Ni60)*state.rho) +
        (screened_rates(k_n_cu64__p_ni64)*Y(Cu64)*Y(N)*state.rho + -screened_rates(k_p_ni64__n_cu64)*Y(Ni64)*Y(H1)*state.rho) +
        (-screened_rates(k_p_cu64__he4_ni61)*Y(Cu64)*Y(H1)*state.rho + screened_rates(k_he4_ni61__p_cu64)*Y(He4)*Y(Ni61)*state.rho) +
        (-screened_rates(k_p_cu65__he4_ni62)*Y(Cu65)*Y(H1)*state.rho + screened_rates(k_he4_ni62__p_cu65)*Y(He4)*Y(Ni62)*state.rho) +
        (screened_rates(k_n_zn59__p_cu59)*Y(N)*Y(Zn59)*state.rho + -screened_rates(k_p_cu59__n_zn59)*Y(Cu59)*Y(H1)*state.rho) +
        (screened_rates(k_n_zn60__p_cu60)*Y(N)*Y(Zn60)*state.rho + -screened_rates(k_p_cu60__n_zn60)*Y(Cu60)*Y(H1)*state.rho) +
        (screened_rates(k_n_zn61__p_cu61)*Y(N)*Y(Zn61)*state.rho + -screened_rates(k_p_cu61__n_zn61)*Y(Cu61)*Y(H1)*state.rho) +
        (-screened_rates(k_p_zn61__he4_cu58)*Y(H1)*Y(Zn61)*state.rho + screened_rates(k_he4_cu58__p_zn61)*Y(Cu58)*Y(He4)*state.rho) +
        (screened_rates(k_n_zn62__p_cu62)*Y(N)*Y(Zn62)*state.rho + -screened_rates(k_p_cu62__n_zn62)*Y(Cu62)*Y(H1)*state.rho) +
        (-screened_rates(k_p_zn62__he4_cu59)*Y(H1)*Y(Zn62)*state.rho + screened_rates(k_he4_cu59__p_zn62)*Y(Cu59)*Y(He4)*state.rho) +
        (screened_rates(k_n_zn63__p_cu63)*Y(N)*Y(Zn63)*state.rho + -screened_rates(k_p_cu63__n_zn63)*Y(Cu63)*Y(H1)*state.rho) +
        (-screened_rates(k_p_zn63__he4_cu60)*Y(H1)*Y(Zn63)*state.rho + screened_rates(k_he4_cu60__p_zn63)*Y(Cu60)*Y(He4)*state.rho) +
        (screened_rates(k_n_zn64__p_cu64)*Y(N)*Y(Zn64)*state.rho + -screened_rates(k_p_cu64__n_zn64)*Y(Cu64)*Y(H1)*state.rho) +
        (-screened_rates(k_p_zn64__he4_cu61)*Y(H1)*Y(Zn64)*state.rho + screened_rates(k_he4_cu61__p_zn64)*Y(Cu61)*Y(He4)*state.rho) +
        (screened_rates(k_n_zn65__p_cu65)*Y(N)*Y(Zn65)*state.rho + -screened_rates(k_p_cu65__n_zn65)*Y(Cu65)*Y(H1)*state.rho) +
        (-screened_rates(k_p_zn65__he4_cu62)*Y(H1)*Y(Zn65)*state.rho + screened_rates(k_he4_cu62__p_zn65)*Y(Cu62)*Y(He4)*state.rho) +
        (-screened_rates(k_p_zn66__he4_cu63)*Y(H1)*Y(Zn66)*state.rho + screened_rates(k_he4_cu63__p_zn66)*Y(Cu63)*Y(He4)*state.rho) +
        (screened_rates(k_n_ga62__p_zn62)*Y(Ga62)*Y(N)*state.rho + -screened_rates(k_p_zn62__n_ga62)*Y(H1)*Y(Zn62)*state.rho) +
        (-screened_rates(k_p_ga62__he4_zn59)*Y(Ga62)*Y(H1)*state.rho + screened_rates(k_he4_zn59__p_ga62)*Y(He4)*Y(Zn59)*state.rho) +
        (screened_rates(k_n_ga63__p_zn63)*Y(Ga63)*Y(N)*state.rho + -screened_rates(k_p_zn63__n_ga63)*Y(H1)*Y(Zn63)*state.rho) +
        (-screened_rates(k_p_ga63__he4_zn60)*Y(Ga63)*Y(H1)*state.rho + screened_rates(k_he4_zn60__p_ga63)*Y(He4)*Y(Zn60)*state.rho) +
        (screened_rates(k_n_ga64__p_zn64)*Y(Ga64)*Y(N)*state.rho + -screened_rates(k_p_zn64__n_ga64)*Y(H1)*Y(Zn64)*state.rho) +
        (-screened_rates(k_p_ga64__he4_zn61)*Y(Ga64)*Y(H1)*state.rho + screened_rates(k_he4_zn61__p_ga64)*Y(He4)*Y(Zn61)*state.rho) +
        (screened_rates(k_n_ge63__p_ga63)*Y(Ge63)*Y(N)*state.rho + -screened_rates(k_p_ga63__n_ge63)*Y(Ga63)*Y(H1)*state.rho) +
        (screened_rates(k_n_ge64__p_ga64)*Y(Ge64)*Y(N)*state.rho + -screened_rates(k_p_ga64__n_ge64)*Y(Ga64)*Y(H1)*state.rho) +
        (screened_rates(k_he3_he3__p_p_he4)*std::pow(Y(He3), 2)*state.rho + -screened_rates(k_p_p_he4__he3_he3)*Y(He4)*std::pow(Y(H1), 2)*std::pow(state.rho, 2)) +
        (screened_rates(k_d_be7__p_he4_he4)*Y(Be7)*Y(H2)*state.rho + -0.5*screened_rates(k_p_he4_he4__d_be7)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 2)) +
        (-screened_rates(k_p_be9__d_he4_he4)*Y(Be9)*Y(H1)*state.rho + 0.5*screened_rates(k_d_he4_he4__p_be9)*Y(H2)*std::pow(Y(He4), 2)*std::pow(state.rho, 2)) +
        (screened_rates(k_n_b8__p_he4_he4)*Y(B8)*Y(N)*state.rho + -0.5*screened_rates(k_p_he4_he4__n_b8)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 2)) +
        (-screened_rates(k_p_b11__he4_he4_he4)*Y(B11)*Y(H1)*state.rho + 0.16666666666666667*screened_rates(k_he4_he4_he4__p_b11)*std::pow(Y(He4), 3)*std::pow(state.rho, 2)) +
        (screened_rates(k_he3_li7__n_p_he4_he4)*Y(He3)*Y(Li7)*state.rho + -0.5*screened_rates(k_n_p_he4_he4__he3_li7)*std::pow(Y(He4), 2)*Y(N)*Y(H1)*std::pow(state.rho, 3)) +
        (2.0*screened_rates(k_he3_be7__p_p_he4_he4)*Y(Be7)*Y(He3)*state.rho + -0.5*screened_rates(k_p_p_he4_he4__he3_be7)*std::pow(Y(He4), 2)*std::pow(Y(H1), 2)*std::pow(state.rho, 3)) +
        (-screened_rates(k_n_p_he4__li6)*Y(He4)*Y(N)*Y(H1)*std::pow(state.rho, 2) + screened_rates(k_li6__n_p_he4)*Y(Li6)) +
        (-0.5*screened_rates(k_n_p_p__p_d)*Y(N)*std::pow(Y(H1), 2)*std::pow(state.rho, 2) + screened_rates(k_p_d__n_p_p)*Y(H2)*Y(H1)*state.rho) +
        (0.0e0_rt + 0.0e0_rt);

    ydot_nuc(H2) =
        (screened_rates(k_n_p__d)*Y(N)*Y(H1)*state.rho + -screened_rates(k_d__n_p)*Y(H2)) +
        0.5*screened_rates(k_p_p__d__weak__bet_pos_)*std::pow(Y(H1), 2)*state.rho +
        0.5*screened_rates(k_p_p__d__weak__electron_capture)*std::pow(Y(H1), 2)*std::pow(state.rho, 2)*state.y_e +
        (-screened_rates(k_p_d__he3)*Y(H2)*Y(H1)*state.rho + screened_rates(k_he3__p_d)*Y(He3)) +
        (-screened_rates(k_d_d__he4)*std::pow(Y(H2), 2)*state.rho + 2.0*screened_rates(k_he4__d_d)*Y(He4)) +
        (-screened_rates(k_he4_d__li6)*Y(H2)*Y(He4)*state.rho + screened_rates(k_li6__he4_d)*Y(Li6)) +
        (-screened_rates(k_d_d__n_he3)*std::pow(Y(H2), 2)*state.rho + 2.0*screened_rates(k_n_he3__d_d)*Y(He3)*Y(N)*state.rho) +
        (-screened_rates(k_d_he3__p_he4)*Y(H2)*Y(He3)*state.rho + screened_rates(k_p_he4__d_he3)*Y(He4)*Y(H1)*state.rho) +
        (-screened_rates(k_d_li6__n_be7)*Y(H2)*Y(Li6)*state.rho + screened_rates(k_n_be7__d_li6)*Y(Be7)*Y(N)*state.rho) +
        (-screened_rates(k_d_li6__p_li7)*Y(H2)*Y(Li6)*state.rho + screened_rates(k_p_li7__d_li6)*Y(Li7)*Y(H1)*state.rho) +
        (-screened_rates(k_d_c13__n_n14)*Y(C13)*Y(H2)*state.rho + screened_rates(k_n_n14__d_c13)*Y(N14)*Y(N)*state.rho) +
        (-screened_rates(k_d_c14__n_n15)*Y(C14)*Y(H2)*state.rho + screened_rates(k_n_n15__d_c14)*Y(N15)*Y(N)*state.rho) +
        (-screened_rates(k_d_li7__n_he4_he4)*Y(H2)*Y(Li7)*state.rho + 0.5*screened_rates(k_n_he4_he4__d_li7)*std::pow(Y(He4), 2)*Y(N)*std::pow(state.rho, 2)) +
        (-screened_rates(k_d_be7__p_he4_he4)*Y(Be7)*Y(H2)*state.rho + 0.5*screened_rates(k_p_he4_he4__d_be7)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 2)) +
        (screened_rates(k_p_be9__d_he4_he4)*Y(Be9)*Y(H1)*state.rho + -0.5*screened_rates(k_d_he4_he4__p_be9)*Y(H2)*std::pow(Y(He4), 2)*std::pow(state.rho, 2)) +
        (0.5*screened_rates(k_n_p_p__p_d)*Y(N)*std::pow(Y(H1), 2)*std::pow(state.rho, 2) + -screened_rates(k_p_d__n_p_p)*Y(H2)*Y(H1)*state.rho);

    ydot_nuc(He3) =
        (screened_rates(k_p_d__he3)*Y(H2)*Y(H1)*state.rho + -screened_rates(k_he3__p_d)*Y(He3)) +
        (-screened_rates(k_n_he3__he4)*Y(He3)*Y(N)*state.rho + screened_rates(k_he4__n_he3)*Y(He4)) +
        -screened_rates(k_p_he3__he4__weak__bet_pos_)*Y(He3)*Y(H1)*state.rho +
        (-screened_rates(k_he4_he3__be7)*Y(He3)*Y(He4)*state.rho + screened_rates(k_be7__he4_he3)*Y(Be7)) +
        (0.5*screened_rates(k_d_d__n_he3)*std::pow(Y(H2), 2)*state.rho + -screened_rates(k_n_he3__d_d)*Y(He3)*Y(N)*state.rho) +
        (-screened_rates(k_d_he3__p_he4)*Y(H2)*Y(He3)*state.rho + screened_rates(k_p_he4__d_he3)*Y(He4)*Y(H1)*state.rho) +
        (screened_rates(k_p_li6__he4_he3)*Y(Li6)*Y(H1)*state.rho + -screened_rates(k_he4_he3__p_li6)*Y(He3)*Y(He4)*state.rho) +
        (-screened_rates(k_he3_he3__p_p_he4)*std::pow(Y(He3), 2)*state.rho + screened_rates(k_p_p_he4__he3_he3)*Y(He4)*std::pow(Y(H1), 2)*std::pow(state.rho, 2)) +
        (-screened_rates(k_he3_li7__n_p_he4_he4)*Y(He3)*Y(Li7)*state.rho + 0.5*screened_rates(k_n_p_he4_he4__he3_li7)*std::pow(Y(He4), 2)*Y(N)*Y(H1)*std::pow(state.rho, 3)) +
        (-screened_rates(k_he3_be7__p_p_he4_he4)*Y(Be7)*Y(He3)*state.rho + 0.25*screened_rates(k_p_p_he4_he4__he3_be7)*std::pow(Y(He4), 2)*std::pow(Y(H1), 2)*std::pow(state.rho, 3));

    ydot_nuc(He4) =
        2.0*screened_rates(k_b8__he4_he4__weak__wc12)*Y(B8) +
        (0.5*screened_rates(k_d_d__he4)*std::pow(Y(H2), 2)*state.rho + -screened_rates(k_he4__d_d)*Y(He4)) +
        (-screened_rates(k_he4_d__li6)*Y(H2)*Y(He4)*state.rho + screened_rates(k_li6__he4_d)*Y(Li6)) +
        (screened_rates(k_n_he3__he4)*Y(He3)*Y(N)*state.rho + -screened_rates(k_he4__n_he3)*Y(He4)) +
        screened_rates(k_p_he3__he4__weak__bet_pos_)*Y(He3)*Y(H1)*state.rho +
        (-screened_rates(k_he4_he3__be7)*Y(He3)*Y(He4)*state.rho + screened_rates(k_be7__he4_he3)*Y(Be7)) +
        (-screened_rates(k_he4_li6__b10)*Y(He4)*Y(Li6)*state.rho + screened_rates(k_b10__he4_li6)*Y(B10)) +
        (-screened_rates(k_he4_li7__b11)*Y(He4)*Y(Li7)*state.rho + screened_rates(k_b11__he4_li7)*Y(B11)) +
        (-screened_rates(k_he4_c12__o16)*Y(C12)*Y(He4)*state.rho + screened_rates(k_o16__he4_c12)*Y(O16)) +
        (-screened_rates(k_he4_c14__o18)*Y(C14)*Y(He4)*state.rho + screened_rates(k_o18__he4_c14)*Y(O18)) +
        (-screened_rates(k_he4_n14__f18)*Y(He4)*Y(N14)*state.rho + screened_rates(k_f18__he4_n14)*Y(F18)) +
        (-screened_rates(k_he4_n15__f19)*Y(He4)*Y(N15)*state.rho + screened_rates(k_f19__he4_n15)*Y(F19)) +
        (-screened_rates(k_he4_o14__ne18)*Y(He4)*Y(O14)*state.rho + screened_rates(k_ne18__he4_o14)*Y(Ne18)) +
        (-screened_rates(k_he4_o15__ne19)*Y(He4)*Y(O15)*state.rho + screened_rates(k_ne19__he4_o15)*Y(Ne19)) +
        (-screened_rates(k_he4_o16__ne20)*Y(He4)*Y(O16)*state.rho + screened_rates(k_ne20__he4_o16)*Y(Ne20)) +
        (-screened_rates(k_he4_o17__ne21)*Y(He4)*Y(O17)*state.rho + screened_rates(k_ne21__he4_o17)*Y(Ne21)) +
        (-screened_rates(k_he4_o18__ne22)*Y(He4)*Y(O18)*state.rho + screened_rates(k_ne22__he4_o18)*Y(Ne22)) +
        (-screened_rates(k_he4_f17__na21)*Y(F17)*Y(He4)*state.rho + screened_rates(k_na21__he4_f17)*Y(Na21)) +
        (-screened_rates(k_he4_f18__na22)*Y(F18)*Y(He4)*state.rho + screened_rates(k_na22__he4_f18)*Y(Na22)) +
        (-screened_rates(k_he4_f19__na23)*Y(F19)*Y(He4)*state.rho + screened_rates(k_na23__he4_f19)*Y(Na23)) +
        (-screened_rates(k_he4_ne19__mg23)*Y(He4)*Y(Ne19)*state.rho + screened_rates(k_mg23__he4_ne19)*Y(Mg23)) +
        (-screened_rates(k_he4_ne20__mg24)*Y(He4)*Y(Ne20)*state.rho + screened_rates(k_mg24__he4_ne20)*Y(Mg24)) +
        (-screened_rates(k_he4_ne21__mg25)*Y(He4)*Y(Ne21)*state.rho + screened_rates(k_mg25__he4_ne21)*Y(Mg25)) +
        (-screened_rates(k_he4_ne22__mg26)*Y(He4)*Y(Ne22)*state.rho + screened_rates(k_mg26__he4_ne22)*Y(Mg26)) +
        (-screened_rates(k_he4_na21__al25)*Y(He4)*Y(Na21)*state.rho + screened_rates(k_al25__he4_na21)*Y(Al25)) +
        (-screened_rates(k_he4_na22__al26)*Y(He4)*Y(Na22)*state.rho + screened_rates(k_al26__he4_na22)*Y(Al26)) +
        (-screened_rates(k_he4_na23__al27)*Y(He4)*Y(Na23)*state.rho + screened_rates(k_al27__he4_na23)*Y(Al27)) +
        (-screened_rates(k_he4_mg24__si28)*Y(He4)*Y(Mg24)*state.rho + screened_rates(k_si28__he4_mg24)*Y(Si28)) +
        (-screened_rates(k_he4_mg25__si29)*Y(He4)*Y(Mg25)*state.rho + screened_rates(k_si29__he4_mg25)*Y(Si29)) +
        (-screened_rates(k_he4_mg26__si30)*Y(He4)*Y(Mg26)*state.rho + screened_rates(k_si30__he4_mg26)*Y(Si30)) +
        (-screened_rates(k_he4_al25__p29)*Y(Al25)*Y(He4)*state.rho + screened_rates(k_p29__he4_al25)*Y(P29)) +
        (-screened_rates(k_he4_al26__p30)*Y(Al26)*Y(He4)*state.rho + screened_rates(k_p30__he4_al26)*Y(P30)) +
        (-screened_rates(k_he4_al27__p31)*Y(Al27)*Y(He4)*state.rho + screened_rates(k_p31__he4_al27)*Y(P31)) +
        (-screened_rates(k_he4_si28__s32)*Y(He4)*Y(Si28)*state.rho + screened_rates(k_s32__he4_si28)*Y(S32)) +
        (-screened_rates(k_he4_si29__s33)*Y(He4)*Y(Si29)*state.rho + screened_rates(k_s33__he4_si29)*Y(S33)) +
        (-screened_rates(k_he4_si30__s34)*Y(He4)*Y(Si30)*state.rho + screened_rates(k_s34__he4_si30)*Y(S34)) +
        (-screened_rates(k_he4_si31__s35)*Y(He4)*Y(Si31)*state.rho + screened_rates(k_s35__he4_si31)*Y(S35)) +
        (-screened_rates(k_he4_si32__s36)*Y(He4)*Y(Si32)*state.rho + screened_rates(k_s36__he4_si32)*Y(S36)) +
        (-screened_rates(k_he4_p29__cl33)*Y(He4)*Y(P29)*state.rho + screened_rates(k_cl33__he4_p29)*Y(Cl33)) +
        (-screened_rates(k_he4_p30__cl34)*Y(He4)*Y(P30)*state.rho + screened_rates(k_cl34__he4_p30)*Y(Cl34)) +
        (-screened_rates(k_he4_p31__cl35)*Y(He4)*Y(P31)*state.rho + screened_rates(k_cl35__he4_p31)*Y(Cl35)) +
        (-screened_rates(k_he4_p32__cl36)*Y(He4)*Y(P32)*state.rho + screened_rates(k_cl36__he4_p32)*Y(Cl36)) +
        (-screened_rates(k_he4_p33__cl37)*Y(He4)*Y(P33)*state.rho + screened_rates(k_cl37__he4_p33)*Y(Cl37)) +
        (-screened_rates(k_he4_s32__ar36)*Y(He4)*Y(S32)*state.rho + screened_rates(k_ar36__he4_s32)*Y(Ar36)) +
        (-screened_rates(k_he4_s33__ar37)*Y(He4)*Y(S33)*state.rho + screened_rates(k_ar37__he4_s33)*Y(Ar37)) +
        (-screened_rates(k_he4_s34__ar38)*Y(He4)*Y(S34)*state.rho + screened_rates(k_ar38__he4_s34)*Y(Ar38)) +
        (-screened_rates(k_he4_s35__ar39)*Y(He4)*Y(S35)*state.rho + screened_rates(k_ar39__he4_s35)*Y(Ar39)) +
        (-screened_rates(k_he4_s36__ar40)*Y(He4)*Y(S36)*state.rho + screened_rates(k_ar40__he4_s36)*Y(Ar40)) +
        (-screened_rates(k_he4_cl33__k37)*Y(Cl33)*Y(He4)*state.rho + screened_rates(k_k37__he4_cl33)*Y(K37)) +
        (-screened_rates(k_he4_cl34__k38)*Y(Cl34)*Y(He4)*state.rho + screened_rates(k_k38__he4_cl34)*Y(K38)) +
        (-screened_rates(k_he4_cl35__k39)*Y(Cl35)*Y(He4)*state.rho + screened_rates(k_k39__he4_cl35)*Y(K39)) +
        (-screened_rates(k_he4_cl36__k40)*Y(Cl36)*Y(He4)*state.rho + screened_rates(k_k40__he4_cl36)*Y(K40)) +
        (-screened_rates(k_he4_cl37__k41)*Y(Cl37)*Y(He4)*state.rho + screened_rates(k_k41__he4_cl37)*Y(K41)) +
        (-screened_rates(k_he4_ar36__ca40)*Y(Ar36)*Y(He4)*state.rho + screened_rates(k_ca40__he4_ar36)*Y(Ca40)) +
        (-screened_rates(k_he4_ar37__ca41)*Y(Ar37)*Y(He4)*state.rho + screened_rates(k_ca41__he4_ar37)*Y(Ca41)) +
        (-screened_rates(k_he4_ar38__ca42)*Y(Ar38)*Y(He4)*state.rho + screened_rates(k_ca42__he4_ar38)*Y(Ca42)) +
        (-screened_rates(k_he4_ar39__ca43)*Y(Ar39)*Y(He4)*state.rho + screened_rates(k_ca43__he4_ar39)*Y(Ca43)) +
        (-screened_rates(k_he4_ar40__ca44)*Y(Ar40)*Y(He4)*state.rho + screened_rates(k_ca44__he4_ar40)*Y(Ca44)) +
        (-screened_rates(k_he4_k39__sc43)*Y(He4)*Y(K39)*state.rho + screened_rates(k_sc43__he4_k39)*Y(Sc43)) +
        (-screened_rates(k_he4_k40__sc44)*Y(He4)*Y(K40)*state.rho + screened_rates(k_sc44__he4_k40)*Y(Sc44)) +
        (-screened_rates(k_he4_k41__sc45)*Y(He4)*Y(K41)*state.rho + screened_rates(k_sc45__he4_k41)*Y(Sc45)) +
        (-screened_rates(k_he4_ca40__ti44)*Y(Ca40)*Y(He4)*state.rho + screened_rates(k_ti44__he4_ca40)*Y(Ti44)) +
        (-screened_rates(k_he4_ca41__ti45)*Y(Ca41)*Y(He4)*state.rho + screened_rates(k_ti45__he4_ca41)*Y(Ti45)) +
        (-screened_rates(k_he4_ca42__ti46)*Y(Ca42)*Y(He4)*state.rho + screened_rates(k_ti46__he4_ca42)*Y(Ti46)) +
        (-screened_rates(k_he4_ca43__ti47)*Y(Ca43)*Y(He4)*state.rho + screened_rates(k_ti47__he4_ca43)*Y(Ti47)) +
        (-screened_rates(k_he4_ca44__ti48)*Y(Ca44)*Y(He4)*state.rho + screened_rates(k_ti48__he4_ca44)*Y(Ti48)) +
        (-screened_rates(k_he4_ca45__ti49)*Y(Ca45)*Y(He4)*state.rho + screened_rates(k_ti49__he4_ca45)*Y(Ti49)) +
        (-screened_rates(k_he4_ca46__ti50)*Y(Ca46)*Y(He4)*state.rho + screened_rates(k_ti50__he4_ca46)*Y(Ti50)) +
        (-screened_rates(k_he4_ca47__ti51)*Y(Ca47)*Y(He4)*state.rho + screened_rates(k_ti51__he4_ca47)*Y(Ti51)) +
        (-screened_rates(k_he4_sc43__v47)*Y(He4)*Y(Sc43)*state.rho + screened_rates(k_v47__he4_sc43)*Y(V47)) +
        (-screened_rates(k_he4_sc44__v48)*Y(He4)*Y(Sc44)*state.rho + screened_rates(k_v48__he4_sc44)*Y(V48)) +
        (-screened_rates(k_he4_sc45__v49)*Y(He4)*Y(Sc45)*state.rho + screened_rates(k_v49__he4_sc45)*Y(V49)) +
        (-screened_rates(k_he4_sc46__v50)*Y(He4)*Y(Sc46)*state.rho + screened_rates(k_v50__he4_sc46)*Y(V50)) +
        (-screened_rates(k_he4_sc47__v51)*Y(He4)*Y(Sc47)*state.rho + screened_rates(k_v51__he4_sc47)*Y(V51)) +
        (-screened_rates(k_he4_sc48__v52)*Y(He4)*Y(Sc48)*state.rho + screened_rates(k_v52__he4_sc48)*Y(V52)) +
        (-screened_rates(k_he4_ti44__cr48)*Y(He4)*Y(Ti44)*state.rho + screened_rates(k_cr48__he4_ti44)*Y(Cr48)) +
        (-screened_rates(k_he4_ti45__cr49)*Y(He4)*Y(Ti45)*state.rho + screened_rates(k_cr49__he4_ti45)*Y(Cr49)) +
        (-screened_rates(k_he4_ti46__cr50)*Y(He4)*Y(Ti46)*state.rho + screened_rates(k_cr50__he4_ti46)*Y(Cr50)) +
        (-screened_rates(k_he4_ti47__cr51)*Y(He4)*Y(Ti47)*state.rho + screened_rates(k_cr51__he4_ti47)*Y(Cr51)) +
        (-screened_rates(k_he4_ti48__cr52)*Y(He4)*Y(Ti48)*state.rho + screened_rates(k_cr52__he4_ti48)*Y(Cr52)) +
        (-screened_rates(k_he4_ti49__cr53)*Y(He4)*Y(Ti49)*state.rho + screened_rates(k_cr53__he4_ti49)*Y(Cr53)) +
        (-screened_rates(k_he4_ti50__cr54)*Y(He4)*Y(Ti50)*state.rho + screened_rates(k_cr54__he4_ti50)*Y(Cr54)) +
        (-screened_rates(k_he4_v46__mn50)*Y(He4)*Y(V46)*state.rho + screened_rates(k_mn50__he4_v46)*Y(Mn50)) +
        (-screened_rates(k_he4_v47__mn51)*Y(He4)*Y(V47)*state.rho + screened_rates(k_mn51__he4_v47)*Y(Mn51)) +
        (-screened_rates(k_he4_v48__mn52)*Y(He4)*Y(V48)*state.rho + screened_rates(k_mn52__he4_v48)*Y(Mn52)) +
        (-screened_rates(k_he4_v49__mn53)*Y(He4)*Y(V49)*state.rho + screened_rates(k_mn53__he4_v49)*Y(Mn53)) +
        (-screened_rates(k_he4_v50__mn54)*Y(He4)*Y(V50)*state.rho + screened_rates(k_mn54__he4_v50)*Y(Mn54)) +
        (-screened_rates(k_he4_v51__mn55)*Y(He4)*Y(V51)*state.rho + screened_rates(k_mn55__he4_v51)*Y(Mn55)) +
        (-screened_rates(k_he4_cr48__fe52)*Y(Cr48)*Y(He4)*state.rho + screened_rates(k_fe52__he4_cr48)*Y(Fe52)) +
        (-screened_rates(k_he4_cr49__fe53)*Y(Cr49)*Y(He4)*state.rho + screened_rates(k_fe53__he4_cr49)*Y(Fe53)) +
        (-screened_rates(k_he4_cr50__fe54)*Y(Cr50)*Y(He4)*state.rho + screened_rates(k_fe54__he4_cr50)*Y(Fe54)) +
        (-screened_rates(k_he4_cr51__fe55)*Y(Cr51)*Y(He4)*state.rho + screened_rates(k_fe55__he4_cr51)*Y(Fe55)) +
        (-screened_rates(k_he4_cr52__fe56)*Y(Cr52)*Y(He4)*state.rho + screened_rates(k_fe56__he4_cr52)*Y(Fe56)) +
        (-screened_rates(k_he4_cr53__fe57)*Y(Cr53)*Y(He4)*state.rho + screened_rates(k_fe57__he4_cr53)*Y(Fe57)) +
        (-screened_rates(k_he4_cr54__fe58)*Y(Cr54)*Y(He4)*state.rho + screened_rates(k_fe58__he4_cr54)*Y(Fe58)) +
        (-screened_rates(k_he4_mn50__co54)*Y(He4)*Y(Mn50)*state.rho + screened_rates(k_co54__he4_mn50)*Y(Co54)) +
        (-screened_rates(k_he4_mn51__co55)*Y(He4)*Y(Mn51)*state.rho + screened_rates(k_co55__he4_mn51)*Y(Co55)) +
        (-screened_rates(k_he4_mn52__co56)*Y(He4)*Y(Mn52)*state.rho + screened_rates(k_co56__he4_mn52)*Y(Co56)) +
        (-screened_rates(k_he4_mn53__co57)*Y(He4)*Y(Mn53)*state.rho + screened_rates(k_co57__he4_mn53)*Y(Co57)) +
        (-screened_rates(k_he4_mn54__co58)*Y(He4)*Y(Mn54)*state.rho + screened_rates(k_co58__he4_mn54)*Y(Co58)) +
        (-screened_rates(k_he4_mn55__co59)*Y(He4)*Y(Mn55)*state.rho + screened_rates(k_co59__he4_mn55)*Y(Co59)) +
        (-screened_rates(k_he4_fe52__ni56)*Y(Fe52)*Y(He4)*state.rho + screened_rates(k_ni56__he4_fe52)*Y(Ni56)) +
        (-screened_rates(k_he4_fe53__ni57)*Y(Fe53)*Y(He4)*state.rho + screened_rates(k_ni57__he4_fe53)*Y(Ni57)) +
        (-screened_rates(k_he4_fe54__ni58)*Y(Fe54)*Y(He4)*state.rho + screened_rates(k_ni58__he4_fe54)*Y(Ni58)) +
        (-screened_rates(k_he4_fe55__ni59)*Y(Fe55)*Y(He4)*state.rho + screened_rates(k_ni59__he4_fe55)*Y(Ni59)) +
        (-screened_rates(k_he4_fe56__ni60)*Y(Fe56)*Y(He4)*state.rho + screened_rates(k_ni60__he4_fe56)*Y(Ni60)) +
        (-screened_rates(k_he4_fe57__ni61)*Y(Fe57)*Y(He4)*state.rho + screened_rates(k_ni61__he4_fe57)*Y(Ni61)) +
        (-screened_rates(k_he4_fe58__ni62)*Y(Fe58)*Y(He4)*state.rho + screened_rates(k_ni62__he4_fe58)*Y(Ni62)) +
        (-screened_rates(k_he4_co53__cu57)*Y(Co53)*Y(He4)*state.rho + screened_rates(k_cu57__he4_co53)*Y(Cu57)) +
        (-screened_rates(k_he4_co54__cu58)*Y(Co54)*Y(He4)*state.rho + screened_rates(k_cu58__he4_co54)*Y(Cu58)) +
        (-screened_rates(k_he4_co55__cu59)*Y(Co55)*Y(He4)*state.rho + screened_rates(k_cu59__he4_co55)*Y(Cu59)) +
        (-screened_rates(k_he4_co56__cu60)*Y(Co56)*Y(He4)*state.rho + screened_rates(k_cu60__he4_co56)*Y(Cu60)) +
        (-screened_rates(k_he4_co57__cu61)*Y(Co57)*Y(He4)*state.rho + screened_rates(k_cu61__he4_co57)*Y(Cu61)) +
        (-screened_rates(k_he4_co58__cu62)*Y(Co58)*Y(He4)*state.rho + screened_rates(k_cu62__he4_co58)*Y(Cu62)) +
        (-screened_rates(k_he4_co59__cu63)*Y(Co59)*Y(He4)*state.rho + screened_rates(k_cu63__he4_co59)*Y(Cu63)) +
        (-screened_rates(k_he4_ni56__zn60)*Y(He4)*Y(Ni56)*state.rho + screened_rates(k_zn60__he4_ni56)*Y(Zn60)) +
        (-screened_rates(k_he4_ni57__zn61)*Y(He4)*Y(Ni57)*state.rho + screened_rates(k_zn61__he4_ni57)*Y(Zn61)) +
        (-screened_rates(k_he4_ni58__zn62)*Y(He4)*Y(Ni58)*state.rho + screened_rates(k_zn62__he4_ni58)*Y(Zn62)) +
        (-screened_rates(k_he4_ni59__zn63)*Y(He4)*Y(Ni59)*state.rho + screened_rates(k_zn63__he4_ni59)*Y(Zn63)) +
        (-screened_rates(k_he4_ni60__zn64)*Y(He4)*Y(Ni60)*state.rho + screened_rates(k_zn64__he4_ni60)*Y(Zn64)) +
        (-screened_rates(k_he4_ni61__zn65)*Y(He4)*Y(Ni61)*state.rho + screened_rates(k_zn65__he4_ni61)*Y(Zn65)) +
        (-screened_rates(k_he4_ni62__zn66)*Y(He4)*Y(Ni62)*state.rho + screened_rates(k_zn66__he4_ni62)*Y(Zn66)) +
        (-screened_rates(k_he4_cu58__ga62)*Y(Cu58)*Y(He4)*state.rho + screened_rates(k_ga62__he4_cu58)*Y(Ga62)) +
        (-screened_rates(k_he4_cu59__ga63)*Y(Cu59)*Y(He4)*state.rho + screened_rates(k_ga63__he4_cu59)*Y(Ga63)) +
        (-screened_rates(k_he4_cu60__ga64)*Y(Cu60)*Y(He4)*state.rho + screened_rates(k_ga64__he4_cu60)*Y(Ga64)) +
        (-screened_rates(k_he4_zn59__ge63)*Y(He4)*Y(Zn59)*state.rho + screened_rates(k_ge63__he4_zn59)*Y(Ge63)) +
        (-screened_rates(k_he4_zn60__ge64)*Y(He4)*Y(Zn60)*state.rho + screened_rates(k_ge64__he4_zn60)*Y(Ge64)) +
        (screened_rates(k_d_he3__p_he4)*Y(H2)*Y(He3)*state.rho + -screened_rates(k_p_he4__d_he3)*Y(He4)*Y(H1)*state.rho) +
        (screened_rates(k_p_li6__he4_he3)*Y(Li6)*Y(H1)*state.rho + -screened_rates(k_he4_he3__p_li6)*Y(He3)*Y(He4)*state.rho) +
        (2.0*screened_rates(k_p_li7__he4_he4)*Y(Li7)*Y(H1)*state.rho + -screened_rates(k_he4_he4__p_li7)*std::pow(Y(He4), 2)*state.rho) +
        (2.0*screened_rates(k_n_be7__he4_he4)*Y(Be7)*Y(N)*state.rho + -screened_rates(k_he4_he4__n_be7)*std::pow(Y(He4), 2)*state.rho) +
        (screened_rates(k_p_be9__he4_li6)*Y(Be9)*Y(H1)*state.rho + -screened_rates(k_he4_li6__p_be9)*Y(He4)*Y(Li6)*state.rho) +
        (-screened_rates(k_he4_be9__n_c12)*Y(Be9)*Y(He4)*state.rho + screened_rates(k_n_c12__he4_be9)*Y(C12)*Y(N)*state.rho) +
        (screened_rates(k_n_b10__he4_li7)*Y(B10)*Y(N)*state.rho + -screened_rates(k_he4_li7__n_b10)*Y(He4)*Y(Li7)*state.rho) +
        (screened_rates(k_p_b10__he4_be7)*Y(B10)*Y(H1)*state.rho + -screened_rates(k_he4_be7__p_b10)*Y(Be7)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_b10__n_n13)*Y(B10)*Y(He4)*state.rho + screened_rates(k_n_n13__he4_b10)*Y(N13)*Y(N)*state.rho) +
        (-screened_rates(k_he4_b10__p_c13)*Y(B10)*Y(He4)*state.rho + screened_rates(k_p_c13__he4_b10)*Y(C13)*Y(H1)*state.rho) +
        (-screened_rates(k_he4_b11__n_n14)*Y(B11)*Y(He4)*state.rho + screened_rates(k_n_n14__he4_b11)*Y(N14)*Y(N)*state.rho) +
        (-screened_rates(k_he4_b11__p_c14)*Y(B11)*Y(He4)*state.rho + screened_rates(k_p_c14__he4_b11)*Y(C14)*Y(H1)*state.rho) +
        (0.5*screened_rates(k_c12_c12__he4_ne20)*std::pow(Y(C12), 2)*state.rho + -screened_rates(k_he4_ne20__c12_c12)*Y(He4)*Y(Ne20)*state.rho) +
        (-screened_rates(k_he4_c13__n_o16)*Y(C13)*Y(He4)*state.rho + screened_rates(k_n_o16__he4_c13)*Y(N)*Y(O16)*state.rho) +
        (-screened_rates(k_he4_n13__p_o16)*Y(He4)*Y(N13)*state.rho + screened_rates(k_p_o16__he4_n13)*Y(O16)*Y(H1)*state.rho) +
        (screened_rates(k_p_n15__he4_c12)*Y(N15)*Y(H1)*state.rho + -screened_rates(k_he4_c12__p_n15)*Y(C12)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_o14__p_f17)*Y(He4)*Y(O14)*state.rho + screened_rates(k_p_f17__he4_o14)*Y(F17)*Y(H1)*state.rho) +
        (screened_rates(k_n_o15__he4_c12)*Y(N)*Y(O15)*state.rho + -screened_rates(k_he4_c12__n_o15)*Y(C12)*Y(He4)*state.rho) +
        (screened_rates(k_c12_o16__he4_mg24)*Y(C12)*Y(O16)*state.rho + -screened_rates(k_he4_mg24__c12_o16)*Y(He4)*Y(Mg24)*state.rho) +
        (0.5*screened_rates(k_o16_o16__he4_si28)*std::pow(Y(O16), 2)*state.rho + -screened_rates(k_he4_si28__o16_o16)*Y(He4)*Y(Si28)*state.rho) +
        (screened_rates(k_n_o17__he4_c14)*Y(N)*Y(O17)*state.rho + -screened_rates(k_he4_c14__n_o17)*Y(C14)*Y(He4)*state.rho) +
        (screened_rates(k_p_o17__he4_n14)*Y(O17)*Y(H1)*state.rho + -screened_rates(k_he4_n14__p_o17)*Y(He4)*Y(N14)*state.rho) +
        (-screened_rates(k_he4_o17__n_ne20)*Y(He4)*Y(O17)*state.rho + screened_rates(k_n_ne20__he4_o17)*Y(N)*Y(Ne20)*state.rho) +
        (screened_rates(k_p_o18__he4_n15)*Y(O18)*Y(H1)*state.rho + -screened_rates(k_he4_n15__p_o18)*Y(He4)*Y(N15)*state.rho) +
        (screened_rates(k_n_f17__he4_n14)*Y(F17)*Y(N)*state.rho + -screened_rates(k_he4_n14__n_f17)*Y(He4)*Y(N14)*state.rho) +
        (-screened_rates(k_he4_f17__p_ne20)*Y(F17)*Y(He4)*state.rho + screened_rates(k_p_ne20__he4_f17)*Y(Ne20)*Y(H1)*state.rho) +
        (screened_rates(k_n_f18__he4_n15)*Y(F18)*Y(N)*state.rho + -screened_rates(k_he4_n15__n_f18)*Y(He4)*Y(N15)*state.rho) +
        (screened_rates(k_p_f18__he4_o15)*Y(F18)*Y(H1)*state.rho + -screened_rates(k_he4_o15__p_f18)*Y(He4)*Y(O15)*state.rho) +
        (-screened_rates(k_he4_f18__p_ne21)*Y(F18)*Y(He4)*state.rho + screened_rates(k_p_ne21__he4_f18)*Y(Ne21)*Y(H1)*state.rho) +
        (screened_rates(k_p_f19__he4_o16)*Y(F19)*Y(H1)*state.rho + -screened_rates(k_he4_o16__p_f19)*Y(He4)*Y(O16)*state.rho) +
        (-screened_rates(k_he4_f19__p_ne22)*Y(F19)*Y(He4)*state.rho + screened_rates(k_p_ne22__he4_f19)*Y(Ne22)*Y(H1)*state.rho) +
        (screened_rates(k_n_ne18__he4_o15)*Y(N)*Y(Ne18)*state.rho + -screened_rates(k_he4_o15__n_ne18)*Y(He4)*Y(O15)*state.rho) +
        (-screened_rates(k_he4_ne18__p_na21)*Y(He4)*Y(Ne18)*state.rho + screened_rates(k_p_na21__he4_ne18)*Y(Na21)*Y(H1)*state.rho) +
        (screened_rates(k_n_ne19__he4_o16)*Y(N)*Y(Ne19)*state.rho + -screened_rates(k_he4_o16__n_ne19)*Y(He4)*Y(O16)*state.rho) +
        (-screened_rates(k_he4_ne19__p_na22)*Y(He4)*Y(Ne19)*state.rho + screened_rates(k_p_na22__he4_ne19)*Y(Na22)*Y(H1)*state.rho) +
        (screened_rates(k_c12_ne20__he4_si28)*Y(C12)*Y(Ne20)*state.rho + -screened_rates(k_he4_si28__c12_ne20)*Y(He4)*Y(Si28)*state.rho) +
        (screened_rates(k_n_ne21__he4_o18)*Y(N)*Y(Ne21)*state.rho + -screened_rates(k_he4_o18__n_ne21)*Y(He4)*Y(O18)*state.rho) +
        (-screened_rates(k_he4_ne21__n_mg24)*Y(He4)*Y(Ne21)*state.rho + screened_rates(k_n_mg24__he4_ne21)*Y(Mg24)*Y(N)*state.rho) +
        (screened_rates(k_n_na21__he4_f18)*Y(N)*Y(Na21)*state.rho + -screened_rates(k_he4_f18__n_na21)*Y(F18)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_na21__p_mg24)*Y(He4)*Y(Na21)*state.rho + screened_rates(k_p_mg24__he4_na21)*Y(Mg24)*Y(H1)*state.rho) +
        (screened_rates(k_n_na22__he4_f19)*Y(N)*Y(Na22)*state.rho + -screened_rates(k_he4_f19__n_na22)*Y(F19)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_na22__p_mg25)*Y(He4)*Y(Na22)*state.rho + screened_rates(k_p_mg25__he4_na22)*Y(Mg25)*Y(H1)*state.rho) +
        (screened_rates(k_p_na23__he4_ne20)*Y(Na23)*Y(H1)*state.rho + -screened_rates(k_he4_ne20__p_na23)*Y(He4)*Y(Ne20)*state.rho) +
        (-screened_rates(k_he4_na23__p_mg26)*Y(He4)*Y(Na23)*state.rho + screened_rates(k_p_mg26__he4_na23)*Y(Mg26)*Y(H1)*state.rho) +
        (screened_rates(k_n_mg23__he4_ne20)*Y(Mg23)*Y(N)*state.rho + -screened_rates(k_he4_ne20__n_mg23)*Y(He4)*Y(Ne20)*state.rho) +
        (-screened_rates(k_he4_mg23__p_al26)*Y(He4)*Y(Mg23)*state.rho + screened_rates(k_p_al26__he4_mg23)*Y(Al26)*Y(H1)*state.rho) +
        (screened_rates(k_n_mg25__he4_ne22)*Y(Mg25)*Y(N)*state.rho + -screened_rates(k_he4_ne22__n_mg25)*Y(He4)*Y(Ne22)*state.rho) +
        (-screened_rates(k_he4_mg25__n_si28)*Y(He4)*Y(Mg25)*state.rho + screened_rates(k_n_si28__he4_mg25)*Y(N)*Y(Si28)*state.rho) +
        (-screened_rates(k_he4_mg26__n_si29)*Y(He4)*Y(Mg26)*state.rho + screened_rates(k_n_si29__he4_mg26)*Y(N)*Y(Si29)*state.rho) +
        (screened_rates(k_n_al25__he4_na22)*Y(Al25)*Y(N)*state.rho + -screened_rates(k_he4_na22__n_al25)*Y(He4)*Y(Na22)*state.rho) +
        (-screened_rates(k_he4_al25__p_si28)*Y(Al25)*Y(He4)*state.rho + screened_rates(k_p_si28__he4_al25)*Y(H1)*Y(Si28)*state.rho) +
        (screened_rates(k_n_al26__he4_na23)*Y(Al26)*Y(N)*state.rho + -screened_rates(k_he4_na23__n_al26)*Y(He4)*Y(Na23)*state.rho) +
        (-screened_rates(k_he4_al26__p_si29)*Y(Al26)*Y(He4)*state.rho + screened_rates(k_p_si29__he4_al26)*Y(H1)*Y(Si29)*state.rho) +
        (screened_rates(k_p_al27__he4_mg24)*Y(Al27)*Y(H1)*state.rho + -screened_rates(k_he4_mg24__p_al27)*Y(He4)*Y(Mg24)*state.rho) +
        (-screened_rates(k_he4_al27__p_si30)*Y(Al27)*Y(He4)*state.rho + screened_rates(k_p_si30__he4_al27)*Y(H1)*Y(Si30)*state.rho) +
        (-screened_rates(k_he4_si31__n_s34)*Y(He4)*Y(Si31)*state.rho + screened_rates(k_n_s34__he4_si31)*Y(N)*Y(S34)*state.rho) +
        (screened_rates(k_n_p29__he4_al26)*Y(N)*Y(P29)*state.rho + -screened_rates(k_he4_al26__n_p29)*Y(Al26)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_p29__p_s32)*Y(He4)*Y(P29)*state.rho + screened_rates(k_p_s32__he4_p29)*Y(H1)*Y(S32)*state.rho) +
        (screened_rates(k_n_p30__he4_al27)*Y(N)*Y(P30)*state.rho + -screened_rates(k_he4_al27__n_p30)*Y(Al27)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_p30__p_s33)*Y(He4)*Y(P30)*state.rho + screened_rates(k_p_s33__he4_p30)*Y(H1)*Y(S33)*state.rho) +
        (screened_rates(k_p_p31__he4_si28)*Y(P31)*Y(H1)*state.rho + -screened_rates(k_he4_si28__p_p31)*Y(He4)*Y(Si28)*state.rho) +
        (-screened_rates(k_he4_p31__p_s34)*Y(He4)*Y(P31)*state.rho + screened_rates(k_p_s34__he4_p31)*Y(H1)*Y(S34)*state.rho) +
        (screened_rates(k_p_p32__he4_si29)*Y(P32)*Y(H1)*state.rho + -screened_rates(k_he4_si29__p_p32)*Y(He4)*Y(Si29)*state.rho) +
        (screened_rates(k_p_p33__he4_si30)*Y(P33)*Y(H1)*state.rho + -screened_rates(k_he4_si30__p_p33)*Y(He4)*Y(Si30)*state.rho) +
        (screened_rates(k_n_s32__he4_si29)*Y(N)*Y(S32)*state.rho + -screened_rates(k_he4_si29__n_s32)*Y(He4)*Y(Si29)*state.rho) +
        (screened_rates(k_n_s33__he4_si30)*Y(N)*Y(S33)*state.rho + -screened_rates(k_he4_si30__n_s33)*Y(He4)*Y(Si30)*state.rho) +
        (screened_rates(k_n_s35__he4_si32)*Y(N)*Y(S35)*state.rho + -screened_rates(k_he4_si32__n_s35)*Y(He4)*Y(Si32)*state.rho) +
        (screened_rates(k_p_s35__he4_p32)*Y(H1)*Y(S35)*state.rho + -screened_rates(k_he4_p32__p_s35)*Y(He4)*Y(P32)*state.rho) +
        (-screened_rates(k_he4_s35__n_ar38)*Y(He4)*Y(S35)*state.rho + screened_rates(k_n_ar38__he4_s35)*Y(Ar38)*Y(N)*state.rho) +
        (screened_rates(k_p_s36__he4_p33)*Y(H1)*Y(S36)*state.rho + -screened_rates(k_he4_p33__p_s36)*Y(He4)*Y(P33)*state.rho) +
        (screened_rates(k_n_cl33__he4_p30)*Y(Cl33)*Y(N)*state.rho + -screened_rates(k_he4_p30__n_cl33)*Y(He4)*Y(P30)*state.rho) +
        (-screened_rates(k_he4_cl33__p_ar36)*Y(Cl33)*Y(He4)*state.rho + screened_rates(k_p_ar36__he4_cl33)*Y(Ar36)*Y(H1)*state.rho) +
        (screened_rates(k_n_cl34__he4_p31)*Y(Cl34)*Y(N)*state.rho + -screened_rates(k_he4_p31__n_cl34)*Y(He4)*Y(P31)*state.rho) +
        (-screened_rates(k_he4_cl34__p_ar37)*Y(Cl34)*Y(He4)*state.rho + screened_rates(k_p_ar37__he4_cl34)*Y(Ar37)*Y(H1)*state.rho) +
        (screened_rates(k_n_cl35__he4_p32)*Y(Cl35)*Y(N)*state.rho + -screened_rates(k_he4_p32__n_cl35)*Y(He4)*Y(P32)*state.rho) +
        (screened_rates(k_p_cl35__he4_s32)*Y(Cl35)*Y(H1)*state.rho + -screened_rates(k_he4_s32__p_cl35)*Y(He4)*Y(S32)*state.rho) +
        (-screened_rates(k_he4_cl35__p_ar38)*Y(Cl35)*Y(He4)*state.rho + screened_rates(k_p_ar38__he4_cl35)*Y(Ar38)*Y(H1)*state.rho) +
        (screened_rates(k_n_cl36__he4_p33)*Y(Cl36)*Y(N)*state.rho + -screened_rates(k_he4_p33__n_cl36)*Y(He4)*Y(P33)*state.rho) +
        (screened_rates(k_p_cl36__he4_s33)*Y(Cl36)*Y(H1)*state.rho + -screened_rates(k_he4_s33__p_cl36)*Y(He4)*Y(S33)*state.rho) +
        (screened_rates(k_p_cl37__he4_s34)*Y(Cl37)*Y(H1)*state.rho + -screened_rates(k_he4_s34__p_cl37)*Y(He4)*Y(S34)*state.rho) +
        (screened_rates(k_n_ar36__he4_s33)*Y(Ar36)*Y(N)*state.rho + -screened_rates(k_he4_s33__n_ar36)*Y(He4)*Y(S33)*state.rho) +
        (screened_rates(k_n_ar37__he4_s34)*Y(Ar37)*Y(N)*state.rho + -screened_rates(k_he4_s34__n_ar37)*Y(He4)*Y(S34)*state.rho) +
        (screened_rates(k_n_ar39__he4_s36)*Y(Ar39)*Y(N)*state.rho + -screened_rates(k_he4_s36__n_ar39)*Y(He4)*Y(S36)*state.rho) +
        (screened_rates(k_p_ar39__he4_cl36)*Y(Ar39)*Y(H1)*state.rho + -screened_rates(k_he4_cl36__p_ar39)*Y(Cl36)*Y(He4)*state.rho) +
        (screened_rates(k_p_ar40__he4_cl37)*Y(Ar40)*Y(H1)*state.rho + -screened_rates(k_he4_cl37__p_ar40)*Y(Cl37)*Y(He4)*state.rho) +
        (screened_rates(k_n_k37__he4_cl34)*Y(K37)*Y(N)*state.rho + -screened_rates(k_he4_cl34__n_k37)*Y(Cl34)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_k37__p_ca40)*Y(He4)*Y(K37)*state.rho + screened_rates(k_p_ca40__he4_k37)*Y(Ca40)*Y(H1)*state.rho) +
        (screened_rates(k_n_k38__he4_cl35)*Y(K38)*Y(N)*state.rho + -screened_rates(k_he4_cl35__n_k38)*Y(Cl35)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_k38__p_ca41)*Y(He4)*Y(K38)*state.rho + screened_rates(k_p_ca41__he4_k38)*Y(Ca41)*Y(H1)*state.rho) +
        (screened_rates(k_n_k39__he4_cl36)*Y(K39)*Y(N)*state.rho + -screened_rates(k_he4_cl36__n_k39)*Y(Cl36)*Y(He4)*state.rho) +
        (screened_rates(k_p_k39__he4_ar36)*Y(K39)*Y(H1)*state.rho + -screened_rates(k_he4_ar36__p_k39)*Y(Ar36)*Y(He4)*state.rho) +
        (screened_rates(k_n_k40__he4_cl37)*Y(K40)*Y(N)*state.rho + -screened_rates(k_he4_cl37__n_k40)*Y(Cl37)*Y(He4)*state.rho) +
        (screened_rates(k_p_k40__he4_ar37)*Y(K40)*Y(H1)*state.rho + -screened_rates(k_he4_ar37__p_k40)*Y(Ar37)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_k40__p_ca43)*Y(He4)*Y(K40)*state.rho + screened_rates(k_p_ca43__he4_k40)*Y(Ca43)*Y(H1)*state.rho) +
        (screened_rates(k_p_k41__he4_ar38)*Y(K41)*Y(H1)*state.rho + -screened_rates(k_he4_ar38__p_k41)*Y(Ar38)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_k41__p_ca44)*Y(He4)*Y(K41)*state.rho + screened_rates(k_p_ca44__he4_k41)*Y(Ca44)*Y(H1)*state.rho) +
        (screened_rates(k_n_ca40__he4_ar37)*Y(Ca40)*Y(N)*state.rho + -screened_rates(k_he4_ar37__n_ca40)*Y(Ar37)*Y(He4)*state.rho) +
        (screened_rates(k_n_ca41__he4_ar38)*Y(Ca41)*Y(N)*state.rho + -screened_rates(k_he4_ar38__n_ca41)*Y(Ar38)*Y(He4)*state.rho) +
        (screened_rates(k_n_ca42__he4_ar39)*Y(Ca42)*Y(N)*state.rho + -screened_rates(k_he4_ar39__n_ca42)*Y(Ar39)*Y(He4)*state.rho) +
        (screened_rates(k_p_ca42__he4_k39)*Y(Ca42)*Y(H1)*state.rho + -screened_rates(k_he4_k39__p_ca42)*Y(He4)*Y(K39)*state.rho) +
        (screened_rates(k_n_ca43__he4_ar40)*Y(Ca43)*Y(N)*state.rho + -screened_rates(k_he4_ar40__n_ca43)*Y(Ar40)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_ca43__n_ti46)*Y(Ca43)*Y(He4)*state.rho + screened_rates(k_n_ti46__he4_ca43)*Y(N)*Y(Ti46)*state.rho) +
        (-screened_rates(k_he4_ca45__n_ti48)*Y(Ca45)*Y(He4)*state.rho + screened_rates(k_n_ti48__he4_ca45)*Y(N)*Y(Ti48)*state.rho) +
        (-screened_rates(k_he4_ca47__n_ti50)*Y(Ca47)*Y(He4)*state.rho + screened_rates(k_n_ti50__he4_ca47)*Y(N)*Y(Ti50)*state.rho) +
        (screened_rates(k_n_sc43__he4_k40)*Y(N)*Y(Sc43)*state.rho + -screened_rates(k_he4_k40__n_sc43)*Y(He4)*Y(K40)*state.rho) +
        (screened_rates(k_p_sc43__he4_ca40)*Y(H1)*Y(Sc43)*state.rho + -screened_rates(k_he4_ca40__p_sc43)*Y(Ca40)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_sc43__p_ti46)*Y(He4)*Y(Sc43)*state.rho + screened_rates(k_p_ti46__he4_sc43)*Y(H1)*Y(Ti46)*state.rho) +
        (screened_rates(k_n_sc44__he4_k41)*Y(N)*Y(Sc44)*state.rho + -screened_rates(k_he4_k41__n_sc44)*Y(He4)*Y(K41)*state.rho) +
        (screened_rates(k_p_sc44__he4_ca41)*Y(H1)*Y(Sc44)*state.rho + -screened_rates(k_he4_ca41__p_sc44)*Y(Ca41)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_sc44__p_ti47)*Y(He4)*Y(Sc44)*state.rho + screened_rates(k_p_ti47__he4_sc44)*Y(H1)*Y(Ti47)*state.rho) +
        (screened_rates(k_p_sc45__he4_ca42)*Y(H1)*Y(Sc45)*state.rho + -screened_rates(k_he4_ca42__p_sc45)*Y(Ca42)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_sc45__p_ti48)*Y(He4)*Y(Sc45)*state.rho + screened_rates(k_p_ti48__he4_sc45)*Y(H1)*Y(Ti48)*state.rho) +
        (screened_rates(k_p_sc46__he4_ca43)*Y(H1)*Y(Sc46)*state.rho + -screened_rates(k_he4_ca43__p_sc46)*Y(Ca43)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_sc46__n_v49)*Y(He4)*Y(Sc46)*state.rho + screened_rates(k_n_v49__he4_sc46)*Y(N)*Y(V49)*state.rho) +
        (-screened_rates(k_he4_sc46__p_ti49)*Y(He4)*Y(Sc46)*state.rho + screened_rates(k_p_ti49__he4_sc46)*Y(H1)*Y(Ti49)*state.rho) +
        (screened_rates(k_p_sc47__he4_ca44)*Y(H1)*Y(Sc47)*state.rho + -screened_rates(k_he4_ca44__p_sc47)*Y(Ca44)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_sc47__p_ti50)*Y(He4)*Y(Sc47)*state.rho + screened_rates(k_p_ti50__he4_sc47)*Y(H1)*Y(Ti50)*state.rho) +
        (screened_rates(k_p_sc48__he4_ca45)*Y(H1)*Y(Sc48)*state.rho + -screened_rates(k_he4_ca45__p_sc48)*Y(Ca45)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_sc48__n_v51)*Y(He4)*Y(Sc48)*state.rho + screened_rates(k_n_v51__he4_sc48)*Y(N)*Y(V51)*state.rho) +
        (-screened_rates(k_he4_sc48__p_ti51)*Y(He4)*Y(Sc48)*state.rho + screened_rates(k_p_ti51__he4_sc48)*Y(H1)*Y(Ti51)*state.rho) +
        (screened_rates(k_p_sc49__he4_ca46)*Y(H1)*Y(Sc49)*state.rho + -screened_rates(k_he4_ca46__p_sc49)*Y(Ca46)*Y(He4)*state.rho) +
        (screened_rates(k_n_ti44__he4_ca41)*Y(N)*Y(Ti44)*state.rho + -screened_rates(k_he4_ca41__n_ti44)*Y(Ca41)*Y(He4)*state.rho) +
        (screened_rates(k_n_ti45__he4_ca42)*Y(N)*Y(Ti45)*state.rho + -screened_rates(k_he4_ca42__n_ti45)*Y(Ca42)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_ti45__p_v48)*Y(He4)*Y(Ti45)*state.rho + screened_rates(k_p_v48__he4_ti45)*Y(H1)*Y(V48)*state.rho) +
        (screened_rates(k_n_ti47__he4_ca44)*Y(N)*Y(Ti47)*state.rho + -screened_rates(k_he4_ca44__n_ti47)*Y(Ca44)*Y(He4)*state.rho) +
        (screened_rates(k_n_ti49__he4_ca46)*Y(N)*Y(Ti49)*state.rho + -screened_rates(k_he4_ca46__n_ti49)*Y(Ca46)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_ti49__n_cr52)*Y(He4)*Y(Ti49)*state.rho + screened_rates(k_n_cr52__he4_ti49)*Y(Cr52)*Y(N)*state.rho) +
        (screened_rates(k_n_ti51__he4_ca48)*Y(N)*Y(Ti51)*state.rho + -screened_rates(k_he4_ca48__n_ti51)*Y(Ca48)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_ti51__n_cr54)*Y(He4)*Y(Ti51)*state.rho + screened_rates(k_n_cr54__he4_ti51)*Y(Cr54)*Y(N)*state.rho) +
        (screened_rates(k_n_v46__he4_sc43)*Y(N)*Y(V46)*state.rho + -screened_rates(k_he4_sc43__n_v46)*Y(He4)*Y(Sc43)*state.rho) +
        (-screened_rates(k_he4_v46__p_cr49)*Y(He4)*Y(V46)*state.rho + screened_rates(k_p_cr49__he4_v46)*Y(Cr49)*Y(H1)*state.rho) +
        (screened_rates(k_n_v47__he4_sc44)*Y(N)*Y(V47)*state.rho + -screened_rates(k_he4_sc44__n_v47)*Y(He4)*Y(Sc44)*state.rho) +
        (screened_rates(k_p_v47__he4_ti44)*Y(H1)*Y(V47)*state.rho + -screened_rates(k_he4_ti44__p_v47)*Y(He4)*Y(Ti44)*state.rho) +
        (-screened_rates(k_he4_v47__p_cr50)*Y(He4)*Y(V47)*state.rho + screened_rates(k_p_cr50__he4_v47)*Y(Cr50)*Y(H1)*state.rho) +
        (screened_rates(k_n_v48__he4_sc45)*Y(N)*Y(V48)*state.rho + -screened_rates(k_he4_sc45__n_v48)*Y(He4)*Y(Sc45)*state.rho) +
        (-screened_rates(k_he4_v48__p_cr51)*Y(He4)*Y(V48)*state.rho + screened_rates(k_p_cr51__he4_v48)*Y(Cr51)*Y(H1)*state.rho) +
        (screened_rates(k_p_v49__he4_ti46)*Y(H1)*Y(V49)*state.rho + -screened_rates(k_he4_ti46__p_v49)*Y(He4)*Y(Ti46)*state.rho) +
        (-screened_rates(k_he4_v49__p_cr52)*Y(He4)*Y(V49)*state.rho + screened_rates(k_p_cr52__he4_v49)*Y(Cr52)*Y(H1)*state.rho) +
        (screened_rates(k_n_v50__he4_sc47)*Y(N)*Y(V50)*state.rho + -screened_rates(k_he4_sc47__n_v50)*Y(He4)*Y(Sc47)*state.rho) +
        (screened_rates(k_p_v50__he4_ti47)*Y(H1)*Y(V50)*state.rho + -screened_rates(k_he4_ti47__p_v50)*Y(He4)*Y(Ti47)*state.rho) +
        (-screened_rates(k_he4_v50__p_cr53)*Y(He4)*Y(V50)*state.rho + screened_rates(k_p_cr53__he4_v50)*Y(Cr53)*Y(H1)*state.rho) +
        (screened_rates(k_p_v51__he4_ti48)*Y(H1)*Y(V51)*state.rho + -screened_rates(k_he4_ti48__p_v51)*Y(He4)*Y(Ti48)*state.rho) +
        (screened_rates(k_n_v52__he4_sc49)*Y(N)*Y(V52)*state.rho + -screened_rates(k_he4_sc49__n_v52)*Y(He4)*Y(Sc49)*state.rho) +
        (screened_rates(k_p_v52__he4_ti49)*Y(H1)*Y(V52)*state.rho + -screened_rates(k_he4_ti49__p_v52)*Y(He4)*Y(Ti49)*state.rho) +
        (-screened_rates(k_he4_v52__n_mn55)*Y(He4)*Y(V52)*state.rho + screened_rates(k_n_mn55__he4_v52)*Y(Mn55)*Y(N)*state.rho) +
        (screened_rates(k_n_cr48__he4_ti45)*Y(Cr48)*Y(N)*state.rho + -screened_rates(k_he4_ti45__n_cr48)*Y(He4)*Y(Ti45)*state.rho) +
        (-screened_rates(k_he4_cr48__p_mn51)*Y(Cr48)*Y(He4)*state.rho + screened_rates(k_p_mn51__he4_cr48)*Y(Mn51)*Y(H1)*state.rho) +
        (screened_rates(k_n_cr49__he4_ti46)*Y(Cr49)*Y(N)*state.rho + -screened_rates(k_he4_ti46__n_cr49)*Y(He4)*Y(Ti46)*state.rho) +
        (-screened_rates(k_he4_cr49__p_mn52)*Y(Cr49)*Y(He4)*state.rho + screened_rates(k_p_mn52__he4_cr49)*Y(Mn52)*Y(H1)*state.rho) +
        (screened_rates(k_n_cr50__he4_ti47)*Y(Cr50)*Y(N)*state.rho + -screened_rates(k_he4_ti47__n_cr50)*Y(He4)*Y(Ti47)*state.rho) +
        (screened_rates(k_n_cr51__he4_ti48)*Y(Cr51)*Y(N)*state.rho + -screened_rates(k_he4_ti48__n_cr51)*Y(He4)*Y(Ti48)*state.rho) +
        (screened_rates(k_n_cr53__he4_ti50)*Y(Cr53)*Y(N)*state.rho + -screened_rates(k_he4_ti50__n_cr53)*Y(He4)*Y(Ti50)*state.rho) +
        (screened_rates(k_p_cr54__he4_v51)*Y(Cr54)*Y(H1)*state.rho + -screened_rates(k_he4_v51__p_cr54)*Y(He4)*Y(V51)*state.rho) +
        (screened_rates(k_n_mn50__he4_v47)*Y(Mn50)*Y(N)*state.rho + -screened_rates(k_he4_v47__n_mn50)*Y(He4)*Y(V47)*state.rho) +
        (-screened_rates(k_he4_mn50__p_fe53)*Y(He4)*Y(Mn50)*state.rho + screened_rates(k_p_fe53__he4_mn50)*Y(Fe53)*Y(H1)*state.rho) +
        (screened_rates(k_n_mn51__he4_v48)*Y(Mn51)*Y(N)*state.rho + -screened_rates(k_he4_v48__n_mn51)*Y(He4)*Y(V48)*state.rho) +
        (-screened_rates(k_he4_mn51__p_fe54)*Y(He4)*Y(Mn51)*state.rho + screened_rates(k_p_fe54__he4_mn51)*Y(Fe54)*Y(H1)*state.rho) +
        (screened_rates(k_n_mn52__he4_v49)*Y(Mn52)*Y(N)*state.rho + -screened_rates(k_he4_v49__n_mn52)*Y(He4)*Y(V49)*state.rho) +
        (-screened_rates(k_he4_mn52__p_fe55)*Y(He4)*Y(Mn52)*state.rho + screened_rates(k_p_fe55__he4_mn52)*Y(Fe55)*Y(H1)*state.rho) +
        (screened_rates(k_n_mn53__he4_v50)*Y(Mn53)*Y(N)*state.rho + -screened_rates(k_he4_v50__n_mn53)*Y(He4)*Y(V50)*state.rho) +
        (screened_rates(k_p_mn53__he4_cr50)*Y(Mn53)*Y(H1)*state.rho + -screened_rates(k_he4_cr50__p_mn53)*Y(Cr50)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_mn53__p_fe56)*Y(He4)*Y(Mn53)*state.rho + screened_rates(k_p_fe56__he4_mn53)*Y(Fe56)*Y(H1)*state.rho) +
        (screened_rates(k_n_mn54__he4_v51)*Y(Mn54)*Y(N)*state.rho + -screened_rates(k_he4_v51__n_mn54)*Y(He4)*Y(V51)*state.rho) +
        (screened_rates(k_p_mn54__he4_cr51)*Y(Mn54)*Y(H1)*state.rho + -screened_rates(k_he4_cr51__p_mn54)*Y(Cr51)*Y(He4)*state.rho) +
        (screened_rates(k_p_mn55__he4_cr52)*Y(Mn55)*Y(H1)*state.rho + -screened_rates(k_he4_cr52__p_mn55)*Y(Cr52)*Y(He4)*state.rho) +
        (screened_rates(k_n_fe52__he4_cr49)*Y(Fe52)*Y(N)*state.rho + -screened_rates(k_he4_cr49__n_fe52)*Y(Cr49)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_fe52__p_co55)*Y(Fe52)*Y(He4)*state.rho + screened_rates(k_p_co55__he4_fe52)*Y(Co55)*Y(H1)*state.rho) +
        (screened_rates(k_n_fe53__he4_cr50)*Y(Fe53)*Y(N)*state.rho + -screened_rates(k_he4_cr50__n_fe53)*Y(Cr50)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_fe53__p_co56)*Y(Fe53)*Y(He4)*state.rho + screened_rates(k_p_co56__he4_fe53)*Y(Co56)*Y(H1)*state.rho) +
        (screened_rates(k_n_fe54__he4_cr51)*Y(Fe54)*Y(N)*state.rho + -screened_rates(k_he4_cr51__n_fe54)*Y(Cr51)*Y(He4)*state.rho) +
        (screened_rates(k_n_fe55__he4_cr52)*Y(Fe55)*Y(N)*state.rho + -screened_rates(k_he4_cr52__n_fe55)*Y(Cr52)*Y(He4)*state.rho) +
        (screened_rates(k_n_fe56__he4_cr53)*Y(Fe56)*Y(N)*state.rho + -screened_rates(k_he4_cr53__n_fe56)*Y(Cr53)*Y(He4)*state.rho) +
        (screened_rates(k_n_fe57__he4_cr54)*Y(Fe57)*Y(N)*state.rho + -screened_rates(k_he4_cr54__n_fe57)*Y(Cr54)*Y(He4)*state.rho) +
        (screened_rates(k_p_fe57__he4_mn54)*Y(Fe57)*Y(H1)*state.rho + -screened_rates(k_he4_mn54__p_fe57)*Y(He4)*Y(Mn54)*state.rho) +
        (screened_rates(k_p_fe58__he4_mn55)*Y(Fe58)*Y(H1)*state.rho + -screened_rates(k_he4_mn55__p_fe58)*Y(He4)*Y(Mn55)*state.rho) +
        (screened_rates(k_n_co53__he4_mn50)*Y(Co53)*Y(N)*state.rho + -screened_rates(k_he4_mn50__n_co53)*Y(He4)*Y(Mn50)*state.rho) +
        (-screened_rates(k_he4_co53__p_ni56)*Y(Co53)*Y(He4)*state.rho + screened_rates(k_p_ni56__he4_co53)*Y(Ni56)*Y(H1)*state.rho) +
        (screened_rates(k_n_co54__he4_mn51)*Y(Co54)*Y(N)*state.rho + -screened_rates(k_he4_mn51__n_co54)*Y(He4)*Y(Mn51)*state.rho) +
        (-screened_rates(k_he4_co54__p_ni57)*Y(Co54)*Y(He4)*state.rho + screened_rates(k_p_ni57__he4_co54)*Y(Ni57)*Y(H1)*state.rho) +
        (screened_rates(k_n_co55__he4_mn52)*Y(Co55)*Y(N)*state.rho + -screened_rates(k_he4_mn52__n_co55)*Y(He4)*Y(Mn52)*state.rho) +
        (-screened_rates(k_he4_co55__p_ni58)*Y(Co55)*Y(He4)*state.rho + screened_rates(k_p_ni58__he4_co55)*Y(Ni58)*Y(H1)*state.rho) +
        (screened_rates(k_n_co56__he4_mn53)*Y(Co56)*Y(N)*state.rho + -screened_rates(k_he4_mn53__n_co56)*Y(He4)*Y(Mn53)*state.rho) +
        (-screened_rates(k_he4_co56__p_ni59)*Y(Co56)*Y(He4)*state.rho + screened_rates(k_p_ni59__he4_co56)*Y(Ni59)*Y(H1)*state.rho) +
        (screened_rates(k_n_co57__he4_mn54)*Y(Co57)*Y(N)*state.rho + -screened_rates(k_he4_mn54__n_co57)*Y(He4)*Y(Mn54)*state.rho) +
        (screened_rates(k_p_co57__he4_fe54)*Y(Co57)*Y(H1)*state.rho + -screened_rates(k_he4_fe54__p_co57)*Y(Fe54)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_co57__p_ni60)*Y(Co57)*Y(He4)*state.rho + screened_rates(k_p_ni60__he4_co57)*Y(Ni60)*Y(H1)*state.rho) +
        (screened_rates(k_n_co58__he4_mn55)*Y(Co58)*Y(N)*state.rho + -screened_rates(k_he4_mn55__n_co58)*Y(He4)*Y(Mn55)*state.rho) +
        (screened_rates(k_p_co58__he4_fe55)*Y(Co58)*Y(H1)*state.rho + -screened_rates(k_he4_fe55__p_co58)*Y(Fe55)*Y(He4)*state.rho) +
        (screened_rates(k_p_co59__he4_fe56)*Y(Co59)*Y(H1)*state.rho + -screened_rates(k_he4_fe56__p_co59)*Y(Fe56)*Y(He4)*state.rho) +
        (screened_rates(k_n_ni56__he4_fe53)*Y(N)*Y(Ni56)*state.rho + -screened_rates(k_he4_fe53__n_ni56)*Y(Fe53)*Y(He4)*state.rho) +
        (screened_rates(k_n_ni57__he4_fe54)*Y(N)*Y(Ni57)*state.rho + -screened_rates(k_he4_fe54__n_ni57)*Y(Fe54)*Y(He4)*state.rho) +
        (screened_rates(k_n_ni58__he4_fe55)*Y(N)*Y(Ni58)*state.rho + -screened_rates(k_he4_fe55__n_ni58)*Y(Fe55)*Y(He4)*state.rho) +
        (screened_rates(k_n_ni59__he4_fe56)*Y(N)*Y(Ni59)*state.rho + -screened_rates(k_he4_fe56__n_ni59)*Y(Fe56)*Y(He4)*state.rho) +
        (screened_rates(k_n_ni60__he4_fe57)*Y(N)*Y(Ni60)*state.rho + -screened_rates(k_he4_fe57__n_ni60)*Y(Fe57)*Y(He4)*state.rho) +
        (screened_rates(k_n_ni61__he4_fe58)*Y(N)*Y(Ni61)*state.rho + -screened_rates(k_he4_fe58__n_ni61)*Y(Fe58)*Y(He4)*state.rho) +
        (screened_rates(k_p_ni61__he4_co58)*Y(Ni61)*Y(H1)*state.rho + -screened_rates(k_he4_co58__p_ni61)*Y(Co58)*Y(He4)*state.rho) +
        (screened_rates(k_p_ni62__he4_co59)*Y(Ni62)*Y(H1)*state.rho + -screened_rates(k_he4_co59__p_ni62)*Y(Co59)*Y(He4)*state.rho) +
        (screened_rates(k_n_cu57__he4_co54)*Y(Cu57)*Y(N)*state.rho + -screened_rates(k_he4_co54__n_cu57)*Y(Co54)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_cu57__p_zn60)*Y(Cu57)*Y(He4)*state.rho + screened_rates(k_p_zn60__he4_cu57)*Y(H1)*Y(Zn60)*state.rho) +
        (screened_rates(k_n_cu58__he4_co55)*Y(Cu58)*Y(N)*state.rho + -screened_rates(k_he4_co55__n_cu58)*Y(Co55)*Y(He4)*state.rho) +
        (screened_rates(k_n_cu59__he4_co56)*Y(Cu59)*Y(N)*state.rho + -screened_rates(k_he4_co56__n_cu59)*Y(Co56)*Y(He4)*state.rho) +
        (screened_rates(k_p_cu59__he4_ni56)*Y(Cu59)*Y(H1)*state.rho + -screened_rates(k_he4_ni56__p_cu59)*Y(He4)*Y(Ni56)*state.rho) +
        (screened_rates(k_n_cu60__he4_co57)*Y(Cu60)*Y(N)*state.rho + -screened_rates(k_he4_co57__n_cu60)*Y(Co57)*Y(He4)*state.rho) +
        (screened_rates(k_p_cu60__he4_ni57)*Y(Cu60)*Y(H1)*state.rho + -screened_rates(k_he4_ni57__p_cu60)*Y(He4)*Y(Ni57)*state.rho) +
        (screened_rates(k_n_cu61__he4_co58)*Y(Cu61)*Y(N)*state.rho + -screened_rates(k_he4_co58__n_cu61)*Y(Co58)*Y(He4)*state.rho) +
        (screened_rates(k_p_cu61__he4_ni58)*Y(Cu61)*Y(H1)*state.rho + -screened_rates(k_he4_ni58__p_cu61)*Y(He4)*Y(Ni58)*state.rho) +
        (screened_rates(k_n_cu62__he4_co59)*Y(Cu62)*Y(N)*state.rho + -screened_rates(k_he4_co59__n_cu62)*Y(Co59)*Y(He4)*state.rho) +
        (screened_rates(k_p_cu62__he4_ni59)*Y(Cu62)*Y(H1)*state.rho + -screened_rates(k_he4_ni59__p_cu62)*Y(He4)*Y(Ni59)*state.rho) +
        (screened_rates(k_p_cu63__he4_ni60)*Y(Cu63)*Y(H1)*state.rho + -screened_rates(k_he4_ni60__p_cu63)*Y(He4)*Y(Ni60)*state.rho) +
        (screened_rates(k_p_cu64__he4_ni61)*Y(Cu64)*Y(H1)*state.rho + -screened_rates(k_he4_ni61__p_cu64)*Y(He4)*Y(Ni61)*state.rho) +
        (screened_rates(k_p_cu65__he4_ni62)*Y(Cu65)*Y(H1)*state.rho + -screened_rates(k_he4_ni62__p_cu65)*Y(He4)*Y(Ni62)*state.rho) +
        (screened_rates(k_n_zn59__he4_ni56)*Y(N)*Y(Zn59)*state.rho + -screened_rates(k_he4_ni56__n_zn59)*Y(He4)*Y(Ni56)*state.rho) +
        (screened_rates(k_n_zn60__he4_ni57)*Y(N)*Y(Zn60)*state.rho + -screened_rates(k_he4_ni57__n_zn60)*Y(He4)*Y(Ni57)*state.rho) +
        (screened_rates(k_n_zn61__he4_ni58)*Y(N)*Y(Zn61)*state.rho + -screened_rates(k_he4_ni58__n_zn61)*Y(He4)*Y(Ni58)*state.rho) +
        (screened_rates(k_p_zn61__he4_cu58)*Y(H1)*Y(Zn61)*state.rho + -screened_rates(k_he4_cu58__p_zn61)*Y(Cu58)*Y(He4)*state.rho) +
        (screened_rates(k_n_zn62__he4_ni59)*Y(N)*Y(Zn62)*state.rho + -screened_rates(k_he4_ni59__n_zn62)*Y(He4)*Y(Ni59)*state.rho) +
        (screened_rates(k_p_zn62__he4_cu59)*Y(H1)*Y(Zn62)*state.rho + -screened_rates(k_he4_cu59__p_zn62)*Y(Cu59)*Y(He4)*state.rho) +
        (screened_rates(k_n_zn63__he4_ni60)*Y(N)*Y(Zn63)*state.rho + -screened_rates(k_he4_ni60__n_zn63)*Y(He4)*Y(Ni60)*state.rho) +
        (screened_rates(k_p_zn63__he4_cu60)*Y(H1)*Y(Zn63)*state.rho + -screened_rates(k_he4_cu60__p_zn63)*Y(Cu60)*Y(He4)*state.rho) +
        (screened_rates(k_n_zn64__he4_ni61)*Y(N)*Y(Zn64)*state.rho + -screened_rates(k_he4_ni61__n_zn64)*Y(He4)*Y(Ni61)*state.rho) +
        (screened_rates(k_p_zn64__he4_cu61)*Y(H1)*Y(Zn64)*state.rho + -screened_rates(k_he4_cu61__p_zn64)*Y(Cu61)*Y(He4)*state.rho) +
        (screened_rates(k_n_zn65__he4_ni62)*Y(N)*Y(Zn65)*state.rho + -screened_rates(k_he4_ni62__n_zn65)*Y(He4)*Y(Ni62)*state.rho) +
        (screened_rates(k_p_zn65__he4_cu62)*Y(H1)*Y(Zn65)*state.rho + -screened_rates(k_he4_cu62__p_zn65)*Y(Cu62)*Y(He4)*state.rho) +
        (screened_rates(k_n_zn66__he4_ni63)*Y(N)*Y(Zn66)*state.rho + -screened_rates(k_he4_ni63__n_zn66)*Y(He4)*Y(Ni63)*state.rho) +
        (screened_rates(k_p_zn66__he4_cu63)*Y(H1)*Y(Zn66)*state.rho + -screened_rates(k_he4_cu63__p_zn66)*Y(Cu63)*Y(He4)*state.rho) +
        (screened_rates(k_n_ga62__he4_cu59)*Y(Ga62)*Y(N)*state.rho + -screened_rates(k_he4_cu59__n_ga62)*Y(Cu59)*Y(He4)*state.rho) +
        (screened_rates(k_p_ga62__he4_zn59)*Y(Ga62)*Y(H1)*state.rho + -screened_rates(k_he4_zn59__p_ga62)*Y(He4)*Y(Zn59)*state.rho) +
        (screened_rates(k_n_ga63__he4_cu60)*Y(Ga63)*Y(N)*state.rho + -screened_rates(k_he4_cu60__n_ga63)*Y(Cu60)*Y(He4)*state.rho) +
        (screened_rates(k_p_ga63__he4_zn60)*Y(Ga63)*Y(H1)*state.rho + -screened_rates(k_he4_zn60__p_ga63)*Y(He4)*Y(Zn60)*state.rho) +
        (screened_rates(k_n_ga64__he4_cu61)*Y(Ga64)*Y(N)*state.rho + -screened_rates(k_he4_cu61__n_ga64)*Y(Cu61)*Y(He4)*state.rho) +
        (screened_rates(k_p_ga64__he4_zn61)*Y(Ga64)*Y(H1)*state.rho + -screened_rates(k_he4_zn61__p_ga64)*Y(He4)*Y(Zn61)*state.rho) +
        (screened_rates(k_n_ge63__he4_zn60)*Y(Ge63)*Y(N)*state.rho + -screened_rates(k_he4_zn60__n_ge63)*Y(He4)*Y(Zn60)*state.rho) +
        (screened_rates(k_n_ge64__he4_zn61)*Y(Ge64)*Y(N)*state.rho + -screened_rates(k_he4_zn61__n_ge64)*Y(He4)*Y(Zn61)*state.rho) +
        (0.5*screened_rates(k_he3_he3__p_p_he4)*std::pow(Y(He3), 2)*state.rho + -0.5*screened_rates(k_p_p_he4__he3_he3)*Y(He4)*std::pow(Y(H1), 2)*std::pow(state.rho, 2)) +
        (2.0*screened_rates(k_d_li7__n_he4_he4)*Y(H2)*Y(Li7)*state.rho + -screened_rates(k_n_he4_he4__d_li7)*std::pow(Y(He4), 2)*Y(N)*std::pow(state.rho, 2)) +
        (2.0*screened_rates(k_d_be7__p_he4_he4)*Y(Be7)*Y(H2)*state.rho + -screened_rates(k_p_he4_he4__d_be7)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 2)) +
        (2.0*screened_rates(k_p_be9__d_he4_he4)*Y(Be9)*Y(H1)*state.rho + -screened_rates(k_d_he4_he4__p_be9)*Y(H2)*std::pow(Y(He4), 2)*std::pow(state.rho, 2)) +
        (2.0*screened_rates(k_n_b8__p_he4_he4)*Y(B8)*Y(N)*state.rho + -screened_rates(k_p_he4_he4__n_b8)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 2)) +
        (3.0*screened_rates(k_p_b11__he4_he4_he4)*Y(B11)*Y(H1)*state.rho + -0.5*screened_rates(k_he4_he4_he4__p_b11)*std::pow(Y(He4), 3)*std::pow(state.rho, 2)) +
        (2.0*screened_rates(k_he3_li7__n_p_he4_he4)*Y(He3)*Y(Li7)*state.rho + -screened_rates(k_n_p_he4_he4__he3_li7)*std::pow(Y(He4), 2)*Y(N)*Y(H1)*std::pow(state.rho, 3)) +
        (2.0*screened_rates(k_he3_be7__p_p_he4_he4)*Y(Be7)*Y(He3)*state.rho + -0.5*screened_rates(k_p_p_he4_he4__he3_be7)*std::pow(Y(He4), 2)*std::pow(Y(H1), 2)*std::pow(state.rho, 3)) +
        (-screened_rates(k_n_p_he4__li6)*Y(He4)*Y(N)*Y(H1)*std::pow(state.rho, 2) + screened_rates(k_li6__n_p_he4)*Y(Li6)) +
        (-screened_rates(k_n_he4_he4__be9)*std::pow(Y(He4), 2)*Y(N)*std::pow(state.rho, 2) + 2.0*screened_rates(k_be9__n_he4_he4)*Y(Be9)) +
        (-0.5*screened_rates(k_he4_he4_he4__c12)*std::pow(Y(He4), 3)*std::pow(state.rho, 2) + 3.0*screened_rates(k_c12__he4_he4_he4)*Y(C12)) +
        (-screened_rates(k_n_p_he4_he4__p_be9)*std::pow(Y(He4), 2)*Y(N)*Y(H1)*std::pow(state.rho, 3) + 2.0*screened_rates(k_p_be9__n_p_he4_he4)*Y(Be9)*Y(H1)*state.rho);

    ydot_nuc(Li6) =
        (screened_rates(k_he4_d__li6)*Y(H2)*Y(He4)*state.rho + -screened_rates(k_li6__he4_d)*Y(Li6)) +
        (-screened_rates(k_n_li6__li7)*Y(Li6)*Y(N)*state.rho + screened_rates(k_li7__n_li6)*Y(Li7)) +
        (-screened_rates(k_p_li6__be7)*Y(Li6)*Y(H1)*state.rho + screened_rates(k_be7__p_li6)*Y(Be7)) +
        (-screened_rates(k_he4_li6__b10)*Y(He4)*Y(Li6)*state.rho + screened_rates(k_b10__he4_li6)*Y(B10)) +
        (-screened_rates(k_p_li6__he4_he3)*Y(Li6)*Y(H1)*state.rho + screened_rates(k_he4_he3__p_li6)*Y(He3)*Y(He4)*state.rho) +
        (-screened_rates(k_d_li6__n_be7)*Y(H2)*Y(Li6)*state.rho + screened_rates(k_n_be7__d_li6)*Y(Be7)*Y(N)*state.rho) +
        (-screened_rates(k_d_li6__p_li7)*Y(H2)*Y(Li6)*state.rho + screened_rates(k_p_li7__d_li6)*Y(Li7)*Y(H1)*state.rho) +
        (screened_rates(k_p_be9__he4_li6)*Y(Be9)*Y(H1)*state.rho + -screened_rates(k_he4_li6__p_be9)*Y(He4)*Y(Li6)*state.rho) +
        (screened_rates(k_n_p_he4__li6)*Y(He4)*Y(N)*Y(H1)*std::pow(state.rho, 2) + -screened_rates(k_li6__n_p_he4)*Y(Li6));

    ydot_nuc(Li7) =
        screened_rates(k_be7__li7__weak__electron_capture)*Y(Be7)*state.rho*state.y_e +
        (screened_rates(k_n_li6__li7)*Y(Li6)*Y(N)*state.rho + -screened_rates(k_li7__n_li6)*Y(Li7)) +
        (-screened_rates(k_he4_li7__b11)*Y(He4)*Y(Li7)*state.rho + screened_rates(k_b11__he4_li7)*Y(B11)) +
        (screened_rates(k_d_li6__p_li7)*Y(H2)*Y(Li6)*state.rho + -screened_rates(k_p_li7__d_li6)*Y(Li7)*Y(H1)*state.rho) +
        (-screened_rates(k_p_li7__he4_he4)*Y(Li7)*Y(H1)*state.rho + 0.5*screened_rates(k_he4_he4__p_li7)*std::pow(Y(He4), 2)*state.rho) +
        (screened_rates(k_n_be7__p_li7)*Y(Be7)*Y(N)*state.rho + -screened_rates(k_p_li7__n_be7)*Y(Li7)*Y(H1)*state.rho) +
        (screened_rates(k_n_b10__he4_li7)*Y(B10)*Y(N)*state.rho + -screened_rates(k_he4_li7__n_b10)*Y(He4)*Y(Li7)*state.rho) +
        (-screened_rates(k_d_li7__n_he4_he4)*Y(H2)*Y(Li7)*state.rho + 0.5*screened_rates(k_n_he4_he4__d_li7)*std::pow(Y(He4), 2)*Y(N)*std::pow(state.rho, 2)) +
        (-screened_rates(k_he3_li7__n_p_he4_he4)*Y(He3)*Y(Li7)*state.rho + 0.5*screened_rates(k_n_p_he4_he4__he3_li7)*std::pow(Y(He4), 2)*Y(N)*Y(H1)*std::pow(state.rho, 3));

    ydot_nuc(Be7) =
        -screened_rates(k_be7__li7__weak__electron_capture)*Y(Be7)*state.rho*state.y_e +
        (screened_rates(k_he4_he3__be7)*Y(He3)*Y(He4)*state.rho + -screened_rates(k_be7__he4_he3)*Y(Be7)) +
        (screened_rates(k_p_li6__be7)*Y(Li6)*Y(H1)*state.rho + -screened_rates(k_be7__p_li6)*Y(Be7)) +
        (-screened_rates(k_p_be7__b8)*Y(Be7)*Y(H1)*state.rho + screened_rates(k_b8__p_be7)*Y(B8)) +
        (screened_rates(k_d_li6__n_be7)*Y(H2)*Y(Li6)*state.rho + -screened_rates(k_n_be7__d_li6)*Y(Be7)*Y(N)*state.rho) +
        (-screened_rates(k_n_be7__p_li7)*Y(Be7)*Y(N)*state.rho + screened_rates(k_p_li7__n_be7)*Y(Li7)*Y(H1)*state.rho) +
        (-screened_rates(k_n_be7__he4_he4)*Y(Be7)*Y(N)*state.rho + 0.5*screened_rates(k_he4_he4__n_be7)*std::pow(Y(He4), 2)*state.rho) +
        (screened_rates(k_p_b10__he4_be7)*Y(B10)*Y(H1)*state.rho + -screened_rates(k_he4_be7__p_b10)*Y(Be7)*Y(He4)*state.rho) +
        (-screened_rates(k_d_be7__p_he4_he4)*Y(Be7)*Y(H2)*state.rho + 0.5*screened_rates(k_p_he4_he4__d_be7)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 2)) +
        (-screened_rates(k_he3_be7__p_p_he4_he4)*Y(Be7)*Y(He3)*state.rho + 0.25*screened_rates(k_p_p_he4_he4__he3_be7)*std::pow(Y(He4), 2)*std::pow(Y(H1), 2)*std::pow(state.rho, 3));

    ydot_nuc(Be9) =
        (-screened_rates(k_p_be9__b10)*Y(Be9)*Y(H1)*state.rho + screened_rates(k_b10__p_be9)*Y(B10)) +
        (-screened_rates(k_p_be9__he4_li6)*Y(Be9)*Y(H1)*state.rho + screened_rates(k_he4_li6__p_be9)*Y(He4)*Y(Li6)*state.rho) +
        (-screened_rates(k_he4_be9__n_c12)*Y(Be9)*Y(He4)*state.rho + screened_rates(k_n_c12__he4_be9)*Y(C12)*Y(N)*state.rho) +
        (-screened_rates(k_p_be9__d_he4_he4)*Y(Be9)*Y(H1)*state.rho + 0.5*screened_rates(k_d_he4_he4__p_be9)*Y(H2)*std::pow(Y(He4), 2)*std::pow(state.rho, 2)) +
        (0.5*screened_rates(k_n_he4_he4__be9)*std::pow(Y(He4), 2)*Y(N)*std::pow(state.rho, 2) + -screened_rates(k_be9__n_he4_he4)*Y(Be9)) +
        (0.5*screened_rates(k_n_p_he4_he4__p_be9)*std::pow(Y(He4), 2)*Y(N)*Y(H1)*std::pow(state.rho, 3) + -screened_rates(k_p_be9__n_p_he4_he4)*Y(Be9)*Y(H1)*state.rho);

    ydot_nuc(B8) =
        -screened_rates(k_b8__he4_he4__weak__wc12)*Y(B8) +
        (screened_rates(k_p_be7__b8)*Y(Be7)*Y(H1)*state.rho + -screened_rates(k_b8__p_be7)*Y(B8)) +
        (-screened_rates(k_n_b8__p_he4_he4)*Y(B8)*Y(N)*state.rho + 0.5*screened_rates(k_p_he4_he4__n_b8)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 2));

    ydot_nuc(B10) =
        (screened_rates(k_he4_li6__b10)*Y(He4)*Y(Li6)*state.rho + -screened_rates(k_b10__he4_li6)*Y(B10)) +
        (screened_rates(k_p_be9__b10)*Y(Be9)*Y(H1)*state.rho + -screened_rates(k_b10__p_be9)*Y(B10)) +
        (-screened_rates(k_n_b10__b11)*Y(B10)*Y(N)*state.rho + screened_rates(k_b11__n_b10)*Y(B11)) +
        (-screened_rates(k_n_b10__he4_li7)*Y(B10)*Y(N)*state.rho + screened_rates(k_he4_li7__n_b10)*Y(He4)*Y(Li7)*state.rho) +
        (-screened_rates(k_p_b10__he4_be7)*Y(B10)*Y(H1)*state.rho + screened_rates(k_he4_be7__p_b10)*Y(Be7)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_b10__n_n13)*Y(B10)*Y(He4)*state.rho + screened_rates(k_n_n13__he4_b10)*Y(N13)*Y(N)*state.rho) +
        (-screened_rates(k_he4_b10__p_c13)*Y(B10)*Y(He4)*state.rho + screened_rates(k_p_c13__he4_b10)*Y(C13)*Y(H1)*state.rho);

    ydot_nuc(B11) =
        (screened_rates(k_he4_li7__b11)*Y(He4)*Y(Li7)*state.rho + -screened_rates(k_b11__he4_li7)*Y(B11)) +
        (screened_rates(k_n_b10__b11)*Y(B10)*Y(N)*state.rho + -screened_rates(k_b11__n_b10)*Y(B11)) +
        (-screened_rates(k_p_b11__c12)*Y(B11)*Y(H1)*state.rho + screened_rates(k_c12__p_b11)*Y(C12)) +
        (-screened_rates(k_he4_b11__n_n14)*Y(B11)*Y(He4)*state.rho + screened_rates(k_n_n14__he4_b11)*Y(N14)*Y(N)*state.rho) +
        (-screened_rates(k_he4_b11__p_c14)*Y(B11)*Y(He4)*state.rho + screened_rates(k_p_c14__he4_b11)*Y(C14)*Y(H1)*state.rho) +
        (-screened_rates(k_p_b11__he4_he4_he4)*Y(B11)*Y(H1)*state.rho + 0.16666666666666667*screened_rates(k_he4_he4_he4__p_b11)*std::pow(Y(He4), 3)*std::pow(state.rho, 2));

    ydot_nuc(C12) =
        (screened_rates(k_p_b11__c12)*Y(B11)*Y(H1)*state.rho + -screened_rates(k_c12__p_b11)*Y(C12)) +
        (-screened_rates(k_n_c12__c13)*Y(C12)*Y(N)*state.rho + screened_rates(k_c13__n_c12)*Y(C13)) +
        (-screened_rates(k_p_c12__n13)*Y(C12)*Y(H1)*state.rho + screened_rates(k_n13__p_c12)*Y(N13)) +
        (-screened_rates(k_he4_c12__o16)*Y(C12)*Y(He4)*state.rho + screened_rates(k_o16__he4_c12)*Y(O16)) +
        (screened_rates(k_he4_be9__n_c12)*Y(Be9)*Y(He4)*state.rho + -screened_rates(k_n_c12__he4_be9)*Y(C12)*Y(N)*state.rho) +
        (-screened_rates(k_c12_c12__p_na23)*std::pow(Y(C12), 2)*state.rho + 2.0*screened_rates(k_p_na23__c12_c12)*Y(Na23)*Y(H1)*state.rho) +
        (-screened_rates(k_c12_c12__he4_ne20)*std::pow(Y(C12), 2)*state.rho + 2.0*screened_rates(k_he4_ne20__c12_c12)*Y(He4)*Y(Ne20)*state.rho) +
        (screened_rates(k_p_n15__he4_c12)*Y(N15)*Y(H1)*state.rho + -screened_rates(k_he4_c12__p_n15)*Y(C12)*Y(He4)*state.rho) +
        (screened_rates(k_n_o15__he4_c12)*Y(N)*Y(O15)*state.rho + -screened_rates(k_he4_c12__n_o15)*Y(C12)*Y(He4)*state.rho) +
        (-screened_rates(k_c12_o16__p_al27)*Y(C12)*Y(O16)*state.rho + screened_rates(k_p_al27__c12_o16)*Y(Al27)*Y(H1)*state.rho) +
        (-screened_rates(k_c12_o16__he4_mg24)*Y(C12)*Y(O16)*state.rho + screened_rates(k_he4_mg24__c12_o16)*Y(He4)*Y(Mg24)*state.rho) +
        (-screened_rates(k_c12_ne20__p_p31)*Y(C12)*Y(Ne20)*state.rho + screened_rates(k_p_p31__c12_ne20)*Y(P31)*Y(H1)*state.rho) +
        (-screened_rates(k_c12_ne20__he4_si28)*Y(C12)*Y(Ne20)*state.rho + screened_rates(k_he4_si28__c12_ne20)*Y(He4)*Y(Si28)*state.rho) +
        (2.0*screened_rates(k_n_mg23__c12_c12)*Y(Mg23)*Y(N)*state.rho + -screened_rates(k_c12_c12__n_mg23)*std::pow(Y(C12), 2)*state.rho) +
        (0.16666666666666667*screened_rates(k_he4_he4_he4__c12)*std::pow(Y(He4), 3)*std::pow(state.rho, 2) + -screened_rates(k_c12__he4_he4_he4)*Y(C12));

    ydot_nuc(C13) =
        screened_rates(k_n13__c13__weak__wc12)*Y(N13) +
        (screened_rates(k_n_c12__c13)*Y(C12)*Y(N)*state.rho + -screened_rates(k_c13__n_c12)*Y(C13)) +
        (-screened_rates(k_n_c13__c14)*Y(C13)*Y(N)*state.rho + screened_rates(k_c14__n_c13)*Y(C14)) +
        (-screened_rates(k_p_c13__n14)*Y(C13)*Y(H1)*state.rho + screened_rates(k_n14__p_c13)*Y(N14)) +
        (screened_rates(k_he4_b10__p_c13)*Y(B10)*Y(He4)*state.rho + -screened_rates(k_p_c13__he4_b10)*Y(C13)*Y(H1)*state.rho) +
        (-screened_rates(k_d_c13__n_n14)*Y(C13)*Y(H2)*state.rho + screened_rates(k_n_n14__d_c13)*Y(N14)*Y(N)*state.rho) +
        (-screened_rates(k_he4_c13__n_o16)*Y(C13)*Y(He4)*state.rho + screened_rates(k_n_o16__he4_c13)*Y(N)*Y(O16)*state.rho) +
        (screened_rates(k_n_n13__p_c13)*Y(N13)*Y(N)*state.rho + -screened_rates(k_p_c13__n_n13)*Y(C13)*Y(H1)*state.rho);

    ydot_nuc(C14) =
        -screened_rates(k_c14__n14__weak__wc12)*Y(C14) +
        (screened_rates(k_n_c13__c14)*Y(C13)*Y(N)*state.rho + -screened_rates(k_c14__n_c13)*Y(C14)) +
        (-screened_rates(k_p_c14__n15)*Y(C14)*Y(H1)*state.rho + screened_rates(k_n15__p_c14)*Y(N15)) +
        (-screened_rates(k_he4_c14__o18)*Y(C14)*Y(He4)*state.rho + screened_rates(k_o18__he4_c14)*Y(O18)) +
        (screened_rates(k_he4_b11__p_c14)*Y(B11)*Y(He4)*state.rho + -screened_rates(k_p_c14__he4_b11)*Y(C14)*Y(H1)*state.rho) +
        (-screened_rates(k_d_c14__n_n15)*Y(C14)*Y(H2)*state.rho + screened_rates(k_n_n15__d_c14)*Y(N15)*Y(N)*state.rho) +
        (screened_rates(k_n_n14__p_c14)*Y(N14)*Y(N)*state.rho + -screened_rates(k_p_c14__n_n14)*Y(C14)*Y(H1)*state.rho) +
        (screened_rates(k_n_o17__he4_c14)*Y(N)*Y(O17)*state.rho + -screened_rates(k_he4_c14__n_o17)*Y(C14)*Y(He4)*state.rho);

    ydot_nuc(N13) =
        -screened_rates(k_n13__c13__weak__wc12)*Y(N13) +
        (screened_rates(k_p_c12__n13)*Y(C12)*Y(H1)*state.rho + -screened_rates(k_n13__p_c12)*Y(N13)) +
        (-screened_rates(k_n_n13__n14)*Y(N13)*Y(N)*state.rho + screened_rates(k_n14__n_n13)*Y(N14)) +
        (-screened_rates(k_p_n13__o14)*Y(N13)*Y(H1)*state.rho + screened_rates(k_o14__p_n13)*Y(O14)) +
        (screened_rates(k_he4_b10__n_n13)*Y(B10)*Y(He4)*state.rho + -screened_rates(k_n_n13__he4_b10)*Y(N13)*Y(N)*state.rho) +
        (-screened_rates(k_n_n13__p_c13)*Y(N13)*Y(N)*state.rho + screened_rates(k_p_c13__n_n13)*Y(C13)*Y(H1)*state.rho) +
        (-screened_rates(k_he4_n13__p_o16)*Y(He4)*Y(N13)*state.rho + screened_rates(k_p_o16__he4_n13)*Y(O16)*Y(H1)*state.rho);

    ydot_nuc(N14) =
        screened_rates(k_c14__n14__weak__wc12)*Y(C14) +
        screened_rates(k_o14__n14__weak__wc12)*Y(O14) +
        (screened_rates(k_p_c13__n14)*Y(C13)*Y(H1)*state.rho + -screened_rates(k_n14__p_c13)*Y(N14)) +
        (screened_rates(k_n_n13__n14)*Y(N13)*Y(N)*state.rho + -screened_rates(k_n14__n_n13)*Y(N14)) +
        (-screened_rates(k_n_n14__n15)*Y(N14)*Y(N)*state.rho + screened_rates(k_n15__n_n14)*Y(N15)) +
        (-screened_rates(k_p_n14__o15)*Y(N14)*Y(H1)*state.rho + screened_rates(k_o15__p_n14)*Y(O15)) +
        (-screened_rates(k_he4_n14__f18)*Y(He4)*Y(N14)*state.rho + screened_rates(k_f18__he4_n14)*Y(F18)) +
        (screened_rates(k_he4_b11__n_n14)*Y(B11)*Y(He4)*state.rho + -screened_rates(k_n_n14__he4_b11)*Y(N14)*Y(N)*state.rho) +
        (screened_rates(k_d_c13__n_n14)*Y(C13)*Y(H2)*state.rho + -screened_rates(k_n_n14__d_c13)*Y(N14)*Y(N)*state.rho) +
        (-screened_rates(k_n_n14__p_c14)*Y(N14)*Y(N)*state.rho + screened_rates(k_p_c14__n_n14)*Y(C14)*Y(H1)*state.rho) +
        (screened_rates(k_n_o14__p_n14)*Y(N)*Y(O14)*state.rho + -screened_rates(k_p_n14__n_o14)*Y(N14)*Y(H1)*state.rho) +
        (screened_rates(k_p_o17__he4_n14)*Y(O17)*Y(H1)*state.rho + -screened_rates(k_he4_n14__p_o17)*Y(He4)*Y(N14)*state.rho) +
        (screened_rates(k_n_f17__he4_n14)*Y(F17)*Y(N)*state.rho + -screened_rates(k_he4_n14__n_f17)*Y(He4)*Y(N14)*state.rho);

    ydot_nuc(N15) =
        screened_rates(k_o15__n15__weak__wc12)*Y(O15) +
        (screened_rates(k_p_c14__n15)*Y(C14)*Y(H1)*state.rho + -screened_rates(k_n15__p_c14)*Y(N15)) +
        (screened_rates(k_n_n14__n15)*Y(N14)*Y(N)*state.rho + -screened_rates(k_n15__n_n14)*Y(N15)) +
        (-screened_rates(k_p_n15__o16)*Y(N15)*Y(H1)*state.rho + screened_rates(k_o16__p_n15)*Y(O16)) +
        (-screened_rates(k_he4_n15__f19)*Y(He4)*Y(N15)*state.rho + screened_rates(k_f19__he4_n15)*Y(F19)) +
        (screened_rates(k_d_c14__n_n15)*Y(C14)*Y(H2)*state.rho + -screened_rates(k_n_n15__d_c14)*Y(N15)*Y(N)*state.rho) +
        (-screened_rates(k_p_n15__he4_c12)*Y(N15)*Y(H1)*state.rho + screened_rates(k_he4_c12__p_n15)*Y(C12)*Y(He4)*state.rho) +
        (screened_rates(k_n_o15__p_n15)*Y(N)*Y(O15)*state.rho + -screened_rates(k_p_n15__n_o15)*Y(N15)*Y(H1)*state.rho) +
        (screened_rates(k_p_o18__he4_n15)*Y(O18)*Y(H1)*state.rho + -screened_rates(k_he4_n15__p_o18)*Y(He4)*Y(N15)*state.rho) +
        (screened_rates(k_n_f18__he4_n15)*Y(F18)*Y(N)*state.rho + -screened_rates(k_he4_n15__n_f18)*Y(He4)*Y(N15)*state.rho);

    ydot_nuc(O14) =
        -screened_rates(k_o14__n14__weak__wc12)*Y(O14) +
        (screened_rates(k_p_n13__o14)*Y(N13)*Y(H1)*state.rho + -screened_rates(k_o14__p_n13)*Y(O14)) +
        (-screened_rates(k_n_o14__o15)*Y(N)*Y(O14)*state.rho + screened_rates(k_o15__n_o14)*Y(O15)) +
        (-screened_rates(k_he4_o14__ne18)*Y(He4)*Y(O14)*state.rho + screened_rates(k_ne18__he4_o14)*Y(Ne18)) +
        (-screened_rates(k_n_o14__p_n14)*Y(N)*Y(O14)*state.rho + screened_rates(k_p_n14__n_o14)*Y(N14)*Y(H1)*state.rho) +
        (-screened_rates(k_he4_o14__p_f17)*Y(He4)*Y(O14)*state.rho + screened_rates(k_p_f17__he4_o14)*Y(F17)*Y(H1)*state.rho);

    ydot_nuc(O15) =
        -screened_rates(k_o15__n15__weak__wc12)*Y(O15) +
        (screened_rates(k_p_n14__o15)*Y(N14)*Y(H1)*state.rho + -screened_rates(k_o15__p_n14)*Y(O15)) +
        (screened_rates(k_n_o14__o15)*Y(N)*Y(O14)*state.rho + -screened_rates(k_o15__n_o14)*Y(O15)) +
        (-screened_rates(k_n_o15__o16)*Y(N)*Y(O15)*state.rho + screened_rates(k_o16__n_o15)*Y(O16)) +
        (-screened_rates(k_he4_o15__ne19)*Y(He4)*Y(O15)*state.rho + screened_rates(k_ne19__he4_o15)*Y(Ne19)) +
        (-screened_rates(k_n_o15__p_n15)*Y(N)*Y(O15)*state.rho + screened_rates(k_p_n15__n_o15)*Y(N15)*Y(H1)*state.rho) +
        (-screened_rates(k_n_o15__he4_c12)*Y(N)*Y(O15)*state.rho + screened_rates(k_he4_c12__n_o15)*Y(C12)*Y(He4)*state.rho) +
        (screened_rates(k_p_f18__he4_o15)*Y(F18)*Y(H1)*state.rho + -screened_rates(k_he4_o15__p_f18)*Y(He4)*Y(O15)*state.rho) +
        (screened_rates(k_n_ne18__he4_o15)*Y(N)*Y(Ne18)*state.rho + -screened_rates(k_he4_o15__n_ne18)*Y(He4)*Y(O15)*state.rho);

    ydot_nuc(O16) =
        (screened_rates(k_he4_c12__o16)*Y(C12)*Y(He4)*state.rho + -screened_rates(k_o16__he4_c12)*Y(O16)) +
        (screened_rates(k_p_n15__o16)*Y(N15)*Y(H1)*state.rho + -screened_rates(k_o16__p_n15)*Y(O16)) +
        (screened_rates(k_n_o15__o16)*Y(N)*Y(O15)*state.rho + -screened_rates(k_o16__n_o15)*Y(O16)) +
        (-screened_rates(k_n_o16__o17)*Y(N)*Y(O16)*state.rho + screened_rates(k_o17__n_o16)*Y(O17)) +
        (-screened_rates(k_p_o16__f17)*Y(O16)*Y(H1)*state.rho + screened_rates(k_f17__p_o16)*Y(F17)) +
        (-screened_rates(k_he4_o16__ne20)*Y(He4)*Y(O16)*state.rho + screened_rates(k_ne20__he4_o16)*Y(Ne20)) +
        (screened_rates(k_he4_c13__n_o16)*Y(C13)*Y(He4)*state.rho + -screened_rates(k_n_o16__he4_c13)*Y(N)*Y(O16)*state.rho) +
        (screened_rates(k_he4_n13__p_o16)*Y(He4)*Y(N13)*state.rho + -screened_rates(k_p_o16__he4_n13)*Y(O16)*Y(H1)*state.rho) +
        (-screened_rates(k_c12_o16__p_al27)*Y(C12)*Y(O16)*state.rho + screened_rates(k_p_al27__c12_o16)*Y(Al27)*Y(H1)*state.rho) +
        (-screened_rates(k_c12_o16__he4_mg24)*Y(C12)*Y(O16)*state.rho + screened_rates(k_he4_mg24__c12_o16)*Y(He4)*Y(Mg24)*state.rho) +
        (-screened_rates(k_o16_o16__p_p31)*std::pow(Y(O16), 2)*state.rho + 2.0*screened_rates(k_p_p31__o16_o16)*Y(P31)*Y(H1)*state.rho) +
        (-screened_rates(k_o16_o16__he4_si28)*std::pow(Y(O16), 2)*state.rho + 2.0*screened_rates(k_he4_si28__o16_o16)*Y(He4)*Y(Si28)*state.rho) +
        (screened_rates(k_p_f19__he4_o16)*Y(F19)*Y(H1)*state.rho + -screened_rates(k_he4_o16__p_f19)*Y(He4)*Y(O16)*state.rho) +
        (screened_rates(k_n_ne19__he4_o16)*Y(N)*Y(Ne19)*state.rho + -screened_rates(k_he4_o16__n_ne19)*Y(He4)*Y(O16)*state.rho);

    ydot_nuc(O17) =
        screened_rates(k_f17__o17__weak__wc12)*Y(F17) +
        (screened_rates(k_n_o16__o17)*Y(N)*Y(O16)*state.rho + -screened_rates(k_o17__n_o16)*Y(O17)) +
        (-screened_rates(k_n_o17__o18)*Y(N)*Y(O17)*state.rho + screened_rates(k_o18__n_o17)*Y(O18)) +
        (-screened_rates(k_p_o17__f18)*Y(O17)*Y(H1)*state.rho + screened_rates(k_f18__p_o17)*Y(F18)) +
        (-screened_rates(k_he4_o17__ne21)*Y(He4)*Y(O17)*state.rho + screened_rates(k_ne21__he4_o17)*Y(Ne21)) +
        (-screened_rates(k_n_o17__he4_c14)*Y(N)*Y(O17)*state.rho + screened_rates(k_he4_c14__n_o17)*Y(C14)*Y(He4)*state.rho) +
        (-screened_rates(k_p_o17__he4_n14)*Y(O17)*Y(H1)*state.rho + screened_rates(k_he4_n14__p_o17)*Y(He4)*Y(N14)*state.rho) +
        (-screened_rates(k_he4_o17__n_ne20)*Y(He4)*Y(O17)*state.rho + screened_rates(k_n_ne20__he4_o17)*Y(N)*Y(Ne20)*state.rho) +
        (screened_rates(k_n_f17__p_o17)*Y(F17)*Y(N)*state.rho + -screened_rates(k_p_o17__n_f17)*Y(O17)*Y(H1)*state.rho);

    ydot_nuc(O18) =
        screened_rates(k_f18__o18__weak__wc12)*Y(F18) +
        (screened_rates(k_he4_c14__o18)*Y(C14)*Y(He4)*state.rho + -screened_rates(k_o18__he4_c14)*Y(O18)) +
        (screened_rates(k_n_o17__o18)*Y(N)*Y(O17)*state.rho + -screened_rates(k_o18__n_o17)*Y(O18)) +
        (-screened_rates(k_p_o18__f19)*Y(O18)*Y(H1)*state.rho + screened_rates(k_f19__p_o18)*Y(F19)) +
        (-screened_rates(k_he4_o18__ne22)*Y(He4)*Y(O18)*state.rho + screened_rates(k_ne22__he4_o18)*Y(Ne22)) +
        (-screened_rates(k_p_o18__he4_n15)*Y(O18)*Y(H1)*state.rho + screened_rates(k_he4_n15__p_o18)*Y(He4)*Y(N15)*state.rho) +
        (screened_rates(k_n_f18__p_o18)*Y(F18)*Y(N)*state.rho + -screened_rates(k_p_o18__n_f18)*Y(O18)*Y(H1)*state.rho) +
        (screened_rates(k_n_ne21__he4_o18)*Y(N)*Y(Ne21)*state.rho + -screened_rates(k_he4_o18__n_ne21)*Y(He4)*Y(O18)*state.rho);

    ydot_nuc(F17) =
        -screened_rates(k_f17__o17__weak__wc12)*Y(F17) +
        (screened_rates(k_p_o16__f17)*Y(O16)*Y(H1)*state.rho + -screened_rates(k_f17__p_o16)*Y(F17)) +
        (-screened_rates(k_n_f17__f18)*Y(F17)*Y(N)*state.rho + screened_rates(k_f18__n_f17)*Y(F18)) +
        (-screened_rates(k_p_f17__ne18)*Y(F17)*Y(H1)*state.rho + screened_rates(k_ne18__p_f17)*Y(Ne18)) +
        (-screened_rates(k_he4_f17__na21)*Y(F17)*Y(He4)*state.rho + screened_rates(k_na21__he4_f17)*Y(Na21)) +
        (screened_rates(k_he4_o14__p_f17)*Y(He4)*Y(O14)*state.rho + -screened_rates(k_p_f17__he4_o14)*Y(F17)*Y(H1)*state.rho) +
        (-screened_rates(k_n_f17__p_o17)*Y(F17)*Y(N)*state.rho + screened_rates(k_p_o17__n_f17)*Y(O17)*Y(H1)*state.rho) +
        (-screened_rates(k_n_f17__he4_n14)*Y(F17)*Y(N)*state.rho + screened_rates(k_he4_n14__n_f17)*Y(He4)*Y(N14)*state.rho) +
        (-screened_rates(k_he4_f17__p_ne20)*Y(F17)*Y(He4)*state.rho + screened_rates(k_p_ne20__he4_f17)*Y(Ne20)*Y(H1)*state.rho);

    ydot_nuc(F18) =
        -screened_rates(k_f18__o18__weak__wc12)*Y(F18) +
        screened_rates(k_ne18__f18__weak__wc12)*Y(Ne18) +
        (screened_rates(k_he4_n14__f18)*Y(He4)*Y(N14)*state.rho + -screened_rates(k_f18__he4_n14)*Y(F18)) +
        (screened_rates(k_p_o17__f18)*Y(O17)*Y(H1)*state.rho + -screened_rates(k_f18__p_o17)*Y(F18)) +
        (screened_rates(k_n_f17__f18)*Y(F17)*Y(N)*state.rho + -screened_rates(k_f18__n_f17)*Y(F18)) +
        (-screened_rates(k_n_f18__f19)*Y(F18)*Y(N)*state.rho + screened_rates(k_f19__n_f18)*Y(F19)) +
        (-screened_rates(k_p_f18__ne19)*Y(F18)*Y(H1)*state.rho + screened_rates(k_ne19__p_f18)*Y(Ne19)) +
        (-screened_rates(k_he4_f18__na22)*Y(F18)*Y(He4)*state.rho + screened_rates(k_na22__he4_f18)*Y(Na22)) +
        (-screened_rates(k_n_f18__p_o18)*Y(F18)*Y(N)*state.rho + screened_rates(k_p_o18__n_f18)*Y(O18)*Y(H1)*state.rho) +
        (-screened_rates(k_n_f18__he4_n15)*Y(F18)*Y(N)*state.rho + screened_rates(k_he4_n15__n_f18)*Y(He4)*Y(N15)*state.rho) +
        (-screened_rates(k_p_f18__he4_o15)*Y(F18)*Y(H1)*state.rho + screened_rates(k_he4_o15__p_f18)*Y(He4)*Y(O15)*state.rho) +
        (-screened_rates(k_he4_f18__p_ne21)*Y(F18)*Y(He4)*state.rho + screened_rates(k_p_ne21__he4_f18)*Y(Ne21)*Y(H1)*state.rho) +
        (screened_rates(k_n_ne18__p_f18)*Y(N)*Y(Ne18)*state.rho + -screened_rates(k_p_f18__n_ne18)*Y(F18)*Y(H1)*state.rho) +
        (screened_rates(k_n_na21__he4_f18)*Y(N)*Y(Na21)*state.rho + -screened_rates(k_he4_f18__n_na21)*Y(F18)*Y(He4)*state.rho);

    ydot_nuc(F19) =
        screened_rates(k_ne19__f19__weak__wc12)*Y(Ne19) +
        (screened_rates(k_he4_n15__f19)*Y(He4)*Y(N15)*state.rho + -screened_rates(k_f19__he4_n15)*Y(F19)) +
        (screened_rates(k_p_o18__f19)*Y(O18)*Y(H1)*state.rho + -screened_rates(k_f19__p_o18)*Y(F19)) +
        (screened_rates(k_n_f18__f19)*Y(F18)*Y(N)*state.rho + -screened_rates(k_f19__n_f18)*Y(F19)) +
        (-screened_rates(k_p_f19__ne20)*Y(F19)*Y(H1)*state.rho + screened_rates(k_ne20__p_f19)*Y(Ne20)) +
        (-screened_rates(k_he4_f19__na23)*Y(F19)*Y(He4)*state.rho + screened_rates(k_na23__he4_f19)*Y(Na23)) +
        (-screened_rates(k_p_f19__he4_o16)*Y(F19)*Y(H1)*state.rho + screened_rates(k_he4_o16__p_f19)*Y(He4)*Y(O16)*state.rho) +
        (-screened_rates(k_he4_f19__p_ne22)*Y(F19)*Y(He4)*state.rho + screened_rates(k_p_ne22__he4_f19)*Y(Ne22)*Y(H1)*state.rho) +
        (screened_rates(k_n_ne19__p_f19)*Y(N)*Y(Ne19)*state.rho + -screened_rates(k_p_f19__n_ne19)*Y(F19)*Y(H1)*state.rho) +
        (screened_rates(k_n_na22__he4_f19)*Y(N)*Y(Na22)*state.rho + -screened_rates(k_he4_f19__n_na22)*Y(F19)*Y(He4)*state.rho);

    ydot_nuc(Ne18) =
        -screened_rates(k_ne18__f18__weak__wc12)*Y(Ne18) +
        (screened_rates(k_he4_o14__ne18)*Y(He4)*Y(O14)*state.rho + -screened_rates(k_ne18__he4_o14)*Y(Ne18)) +
        (screened_rates(k_p_f17__ne18)*Y(F17)*Y(H1)*state.rho + -screened_rates(k_ne18__p_f17)*Y(Ne18)) +
        (-screened_rates(k_n_ne18__ne19)*Y(N)*Y(Ne18)*state.rho + screened_rates(k_ne19__n_ne18)*Y(Ne19)) +
        (-screened_rates(k_n_ne18__p_f18)*Y(N)*Y(Ne18)*state.rho + screened_rates(k_p_f18__n_ne18)*Y(F18)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ne18__he4_o15)*Y(N)*Y(Ne18)*state.rho + screened_rates(k_he4_o15__n_ne18)*Y(He4)*Y(O15)*state.rho) +
        (-screened_rates(k_he4_ne18__p_na21)*Y(He4)*Y(Ne18)*state.rho + screened_rates(k_p_na21__he4_ne18)*Y(Na21)*Y(H1)*state.rho);

    ydot_nuc(Ne19) =
        -screened_rates(k_ne19__f19__weak__wc12)*Y(Ne19) +
        (screened_rates(k_he4_o15__ne19)*Y(He4)*Y(O15)*state.rho + -screened_rates(k_ne19__he4_o15)*Y(Ne19)) +
        (screened_rates(k_p_f18__ne19)*Y(F18)*Y(H1)*state.rho + -screened_rates(k_ne19__p_f18)*Y(Ne19)) +
        (screened_rates(k_n_ne18__ne19)*Y(N)*Y(Ne18)*state.rho + -screened_rates(k_ne19__n_ne18)*Y(Ne19)) +
        (-screened_rates(k_n_ne19__ne20)*Y(N)*Y(Ne19)*state.rho + screened_rates(k_ne20__n_ne19)*Y(Ne20)) +
        (-screened_rates(k_he4_ne19__mg23)*Y(He4)*Y(Ne19)*state.rho + screened_rates(k_mg23__he4_ne19)*Y(Mg23)) +
        (-screened_rates(k_n_ne19__p_f19)*Y(N)*Y(Ne19)*state.rho + screened_rates(k_p_f19__n_ne19)*Y(F19)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ne19__he4_o16)*Y(N)*Y(Ne19)*state.rho + screened_rates(k_he4_o16__n_ne19)*Y(He4)*Y(O16)*state.rho) +
        (-screened_rates(k_he4_ne19__p_na22)*Y(He4)*Y(Ne19)*state.rho + screened_rates(k_p_na22__he4_ne19)*Y(Na22)*Y(H1)*state.rho);

    ydot_nuc(Ne20) =
        (screened_rates(k_he4_o16__ne20)*Y(He4)*Y(O16)*state.rho + -screened_rates(k_ne20__he4_o16)*Y(Ne20)) +
        (screened_rates(k_p_f19__ne20)*Y(F19)*Y(H1)*state.rho + -screened_rates(k_ne20__p_f19)*Y(Ne20)) +
        (screened_rates(k_n_ne19__ne20)*Y(N)*Y(Ne19)*state.rho + -screened_rates(k_ne20__n_ne19)*Y(Ne20)) +
        (-screened_rates(k_n_ne20__ne21)*Y(N)*Y(Ne20)*state.rho + screened_rates(k_ne21__n_ne20)*Y(Ne21)) +
        (-screened_rates(k_p_ne20__na21)*Y(Ne20)*Y(H1)*state.rho + screened_rates(k_na21__p_ne20)*Y(Na21)) +
        (-screened_rates(k_he4_ne20__mg24)*Y(He4)*Y(Ne20)*state.rho + screened_rates(k_mg24__he4_ne20)*Y(Mg24)) +
        (0.5*screened_rates(k_c12_c12__he4_ne20)*std::pow(Y(C12), 2)*state.rho + -screened_rates(k_he4_ne20__c12_c12)*Y(He4)*Y(Ne20)*state.rho) +
        (screened_rates(k_he4_o17__n_ne20)*Y(He4)*Y(O17)*state.rho + -screened_rates(k_n_ne20__he4_o17)*Y(N)*Y(Ne20)*state.rho) +
        (screened_rates(k_he4_f17__p_ne20)*Y(F17)*Y(He4)*state.rho + -screened_rates(k_p_ne20__he4_f17)*Y(Ne20)*Y(H1)*state.rho) +
        (-screened_rates(k_c12_ne20__p_p31)*Y(C12)*Y(Ne20)*state.rho + screened_rates(k_p_p31__c12_ne20)*Y(P31)*Y(H1)*state.rho) +
        (-screened_rates(k_c12_ne20__he4_si28)*Y(C12)*Y(Ne20)*state.rho + screened_rates(k_he4_si28__c12_ne20)*Y(He4)*Y(Si28)*state.rho) +
        (screened_rates(k_p_na23__he4_ne20)*Y(Na23)*Y(H1)*state.rho + -screened_rates(k_he4_ne20__p_na23)*Y(He4)*Y(Ne20)*state.rho) +
        (screened_rates(k_n_mg23__he4_ne20)*Y(Mg23)*Y(N)*state.rho + -screened_rates(k_he4_ne20__n_mg23)*Y(He4)*Y(Ne20)*state.rho);

    ydot_nuc(Ne21) =
        screened_rates(k_na21__ne21__weak__wc12)*Y(Na21) +
        (screened_rates(k_he4_o17__ne21)*Y(He4)*Y(O17)*state.rho + -screened_rates(k_ne21__he4_o17)*Y(Ne21)) +
        (screened_rates(k_n_ne20__ne21)*Y(N)*Y(Ne20)*state.rho + -screened_rates(k_ne21__n_ne20)*Y(Ne21)) +
        (-screened_rates(k_n_ne21__ne22)*Y(N)*Y(Ne21)*state.rho + screened_rates(k_ne22__n_ne21)*Y(Ne22)) +
        (-screened_rates(k_p_ne21__na22)*Y(Ne21)*Y(H1)*state.rho + screened_rates(k_na22__p_ne21)*Y(Na22)) +
        (-screened_rates(k_he4_ne21__mg25)*Y(He4)*Y(Ne21)*state.rho + screened_rates(k_mg25__he4_ne21)*Y(Mg25)) +
        (screened_rates(k_he4_f18__p_ne21)*Y(F18)*Y(He4)*state.rho + -screened_rates(k_p_ne21__he4_f18)*Y(Ne21)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ne21__he4_o18)*Y(N)*Y(Ne21)*state.rho + screened_rates(k_he4_o18__n_ne21)*Y(He4)*Y(O18)*state.rho) +
        (-screened_rates(k_he4_ne21__n_mg24)*Y(He4)*Y(Ne21)*state.rho + screened_rates(k_n_mg24__he4_ne21)*Y(Mg24)*Y(N)*state.rho) +
        (screened_rates(k_n_na21__p_ne21)*Y(N)*Y(Na21)*state.rho + -screened_rates(k_p_ne21__n_na21)*Y(Ne21)*Y(H1)*state.rho);

    ydot_nuc(Ne22) =
        screened_rates(k_na22__ne22__weak__wc12)*Y(Na22) +
        (screened_rates(k_he4_o18__ne22)*Y(He4)*Y(O18)*state.rho + -screened_rates(k_ne22__he4_o18)*Y(Ne22)) +
        (screened_rates(k_n_ne21__ne22)*Y(N)*Y(Ne21)*state.rho + -screened_rates(k_ne22__n_ne21)*Y(Ne22)) +
        (-screened_rates(k_p_ne22__na23)*Y(Ne22)*Y(H1)*state.rho + screened_rates(k_na23__p_ne22)*Y(Na23)) +
        (-screened_rates(k_he4_ne22__mg26)*Y(He4)*Y(Ne22)*state.rho + screened_rates(k_mg26__he4_ne22)*Y(Mg26)) +
        (screened_rates(k_he4_f19__p_ne22)*Y(F19)*Y(He4)*state.rho + -screened_rates(k_p_ne22__he4_f19)*Y(Ne22)*Y(H1)*state.rho) +
        (screened_rates(k_n_na22__p_ne22)*Y(N)*Y(Na22)*state.rho + -screened_rates(k_p_ne22__n_na22)*Y(Ne22)*Y(H1)*state.rho) +
        (screened_rates(k_n_mg25__he4_ne22)*Y(Mg25)*Y(N)*state.rho + -screened_rates(k_he4_ne22__n_mg25)*Y(He4)*Y(Ne22)*state.rho);

    ydot_nuc(Na21) =
        -screened_rates(k_na21__ne21__weak__wc12)*Y(Na21) +
        (screened_rates(k_he4_f17__na21)*Y(F17)*Y(He4)*state.rho + -screened_rates(k_na21__he4_f17)*Y(Na21)) +
        (screened_rates(k_p_ne20__na21)*Y(Ne20)*Y(H1)*state.rho + -screened_rates(k_na21__p_ne20)*Y(Na21)) +
        (-screened_rates(k_n_na21__na22)*Y(N)*Y(Na21)*state.rho + screened_rates(k_na22__n_na21)*Y(Na22)) +
        (-screened_rates(k_he4_na21__al25)*Y(He4)*Y(Na21)*state.rho + screened_rates(k_al25__he4_na21)*Y(Al25)) +
        (screened_rates(k_he4_ne18__p_na21)*Y(He4)*Y(Ne18)*state.rho + -screened_rates(k_p_na21__he4_ne18)*Y(Na21)*Y(H1)*state.rho) +
        (-screened_rates(k_n_na21__p_ne21)*Y(N)*Y(Na21)*state.rho + screened_rates(k_p_ne21__n_na21)*Y(Ne21)*Y(H1)*state.rho) +
        (-screened_rates(k_n_na21__he4_f18)*Y(N)*Y(Na21)*state.rho + screened_rates(k_he4_f18__n_na21)*Y(F18)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_na21__p_mg24)*Y(He4)*Y(Na21)*state.rho + screened_rates(k_p_mg24__he4_na21)*Y(Mg24)*Y(H1)*state.rho);

    ydot_nuc(Na22) =
        -screened_rates(k_na22__ne22__weak__wc12)*Y(Na22) +
        (screened_rates(k_he4_f18__na22)*Y(F18)*Y(He4)*state.rho + -screened_rates(k_na22__he4_f18)*Y(Na22)) +
        (screened_rates(k_p_ne21__na22)*Y(Ne21)*Y(H1)*state.rho + -screened_rates(k_na22__p_ne21)*Y(Na22)) +
        (screened_rates(k_n_na21__na22)*Y(N)*Y(Na21)*state.rho + -screened_rates(k_na22__n_na21)*Y(Na22)) +
        (-screened_rates(k_n_na22__na23)*Y(N)*Y(Na22)*state.rho + screened_rates(k_na23__n_na22)*Y(Na23)) +
        (-screened_rates(k_p_na22__mg23)*Y(Na22)*Y(H1)*state.rho + screened_rates(k_mg23__p_na22)*Y(Mg23)) +
        (-screened_rates(k_he4_na22__al26)*Y(He4)*Y(Na22)*state.rho + screened_rates(k_al26__he4_na22)*Y(Al26)) +
        (screened_rates(k_he4_ne19__p_na22)*Y(He4)*Y(Ne19)*state.rho + -screened_rates(k_p_na22__he4_ne19)*Y(Na22)*Y(H1)*state.rho) +
        (-screened_rates(k_n_na22__p_ne22)*Y(N)*Y(Na22)*state.rho + screened_rates(k_p_ne22__n_na22)*Y(Ne22)*Y(H1)*state.rho) +
        (-screened_rates(k_n_na22__he4_f19)*Y(N)*Y(Na22)*state.rho + screened_rates(k_he4_f19__n_na22)*Y(F19)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_na22__p_mg25)*Y(He4)*Y(Na22)*state.rho + screened_rates(k_p_mg25__he4_na22)*Y(Mg25)*Y(H1)*state.rho) +
        (screened_rates(k_n_al25__he4_na22)*Y(Al25)*Y(N)*state.rho + -screened_rates(k_he4_na22__n_al25)*Y(He4)*Y(Na22)*state.rho);

    ydot_nuc(Na23) =
        screened_rates(k_mg23__na23__weak__wc12)*Y(Mg23) +
        (screened_rates(k_he4_f19__na23)*Y(F19)*Y(He4)*state.rho + -screened_rates(k_na23__he4_f19)*Y(Na23)) +
        (screened_rates(k_p_ne22__na23)*Y(Ne22)*Y(H1)*state.rho + -screened_rates(k_na23__p_ne22)*Y(Na23)) +
        (screened_rates(k_n_na22__na23)*Y(N)*Y(Na22)*state.rho + -screened_rates(k_na23__n_na22)*Y(Na23)) +
        (-screened_rates(k_p_na23__mg24)*Y(Na23)*Y(H1)*state.rho + screened_rates(k_mg24__p_na23)*Y(Mg24)) +
        (-screened_rates(k_he4_na23__al27)*Y(He4)*Y(Na23)*state.rho + screened_rates(k_al27__he4_na23)*Y(Al27)) +
        (0.5*screened_rates(k_c12_c12__p_na23)*std::pow(Y(C12), 2)*state.rho + -screened_rates(k_p_na23__c12_c12)*Y(Na23)*Y(H1)*state.rho) +
        (-screened_rates(k_p_na23__he4_ne20)*Y(Na23)*Y(H1)*state.rho + screened_rates(k_he4_ne20__p_na23)*Y(He4)*Y(Ne20)*state.rho) +
        (-screened_rates(k_he4_na23__p_mg26)*Y(He4)*Y(Na23)*state.rho + screened_rates(k_p_mg26__he4_na23)*Y(Mg26)*Y(H1)*state.rho) +
        (screened_rates(k_n_mg23__p_na23)*Y(Mg23)*Y(N)*state.rho + -screened_rates(k_p_na23__n_mg23)*Y(Na23)*Y(H1)*state.rho) +
        (screened_rates(k_n_al26__he4_na23)*Y(Al26)*Y(N)*state.rho + -screened_rates(k_he4_na23__n_al26)*Y(He4)*Y(Na23)*state.rho);

    ydot_nuc(Mg23) =
        -screened_rates(k_mg23__na23__weak__wc12)*Y(Mg23) +
        (screened_rates(k_he4_ne19__mg23)*Y(He4)*Y(Ne19)*state.rho + -screened_rates(k_mg23__he4_ne19)*Y(Mg23)) +
        (screened_rates(k_p_na22__mg23)*Y(Na22)*Y(H1)*state.rho + -screened_rates(k_mg23__p_na22)*Y(Mg23)) +
        (-screened_rates(k_n_mg23__mg24)*Y(Mg23)*Y(N)*state.rho + screened_rates(k_mg24__n_mg23)*Y(Mg24)) +
        (-screened_rates(k_n_mg23__p_na23)*Y(Mg23)*Y(N)*state.rho + screened_rates(k_p_na23__n_mg23)*Y(Na23)*Y(H1)*state.rho) +
        (-screened_rates(k_n_mg23__he4_ne20)*Y(Mg23)*Y(N)*state.rho + screened_rates(k_he4_ne20__n_mg23)*Y(He4)*Y(Ne20)*state.rho) +
        (-screened_rates(k_n_mg23__c12_c12)*Y(Mg23)*Y(N)*state.rho + 0.5*screened_rates(k_c12_c12__n_mg23)*std::pow(Y(C12), 2)*state.rho) +
        (-screened_rates(k_he4_mg23__p_al26)*Y(He4)*Y(Mg23)*state.rho + screened_rates(k_p_al26__he4_mg23)*Y(Al26)*Y(H1)*state.rho);

    ydot_nuc(Mg24) =
        (screened_rates(k_he4_ne20__mg24)*Y(He4)*Y(Ne20)*state.rho + -screened_rates(k_mg24__he4_ne20)*Y(Mg24)) +
        (screened_rates(k_p_na23__mg24)*Y(Na23)*Y(H1)*state.rho + -screened_rates(k_mg24__p_na23)*Y(Mg24)) +
        (screened_rates(k_n_mg23__mg24)*Y(Mg23)*Y(N)*state.rho + -screened_rates(k_mg24__n_mg23)*Y(Mg24)) +
        (-screened_rates(k_n_mg24__mg25)*Y(Mg24)*Y(N)*state.rho + screened_rates(k_mg25__n_mg24)*Y(Mg25)) +
        (-screened_rates(k_p_mg24__al25)*Y(Mg24)*Y(H1)*state.rho + screened_rates(k_al25__p_mg24)*Y(Al25)) +
        (-screened_rates(k_he4_mg24__si28)*Y(He4)*Y(Mg24)*state.rho + screened_rates(k_si28__he4_mg24)*Y(Si28)) +
        (screened_rates(k_c12_o16__he4_mg24)*Y(C12)*Y(O16)*state.rho + -screened_rates(k_he4_mg24__c12_o16)*Y(He4)*Y(Mg24)*state.rho) +
        (screened_rates(k_he4_ne21__n_mg24)*Y(He4)*Y(Ne21)*state.rho + -screened_rates(k_n_mg24__he4_ne21)*Y(Mg24)*Y(N)*state.rho) +
        (screened_rates(k_he4_na21__p_mg24)*Y(He4)*Y(Na21)*state.rho + -screened_rates(k_p_mg24__he4_na21)*Y(Mg24)*Y(H1)*state.rho) +
        (screened_rates(k_p_al27__he4_mg24)*Y(Al27)*Y(H1)*state.rho + -screened_rates(k_he4_mg24__p_al27)*Y(He4)*Y(Mg24)*state.rho);

    ydot_nuc(Mg25) =
        screened_rates(k_al25__mg25__weak__wc12)*Y(Al25) +
        (screened_rates(k_he4_ne21__mg25)*Y(He4)*Y(Ne21)*state.rho + -screened_rates(k_mg25__he4_ne21)*Y(Mg25)) +
        (screened_rates(k_n_mg24__mg25)*Y(Mg24)*Y(N)*state.rho + -screened_rates(k_mg25__n_mg24)*Y(Mg25)) +
        (-screened_rates(k_n_mg25__mg26)*Y(Mg25)*Y(N)*state.rho + screened_rates(k_mg26__n_mg25)*Y(Mg26)) +
        (-screened_rates(k_p_mg25__al26)*Y(Mg25)*Y(H1)*state.rho + screened_rates(k_al26__p_mg25)*Y(Al26)) +
        (-screened_rates(k_he4_mg25__si29)*Y(He4)*Y(Mg25)*state.rho + screened_rates(k_si29__he4_mg25)*Y(Si29)) +
        (screened_rates(k_he4_na22__p_mg25)*Y(He4)*Y(Na22)*state.rho + -screened_rates(k_p_mg25__he4_na22)*Y(Mg25)*Y(H1)*state.rho) +
        (-screened_rates(k_n_mg25__he4_ne22)*Y(Mg25)*Y(N)*state.rho + screened_rates(k_he4_ne22__n_mg25)*Y(He4)*Y(Ne22)*state.rho) +
        (-screened_rates(k_he4_mg25__n_si28)*Y(He4)*Y(Mg25)*state.rho + screened_rates(k_n_si28__he4_mg25)*Y(N)*Y(Si28)*state.rho) +
        (screened_rates(k_n_al25__p_mg25)*Y(Al25)*Y(N)*state.rho + -screened_rates(k_p_mg25__n_al25)*Y(Mg25)*Y(H1)*state.rho);

    ydot_nuc(Mg26) =
        screened_rates(k_al26__mg26__weak__wc12)*Y(Al26) +
        (screened_rates(k_he4_ne22__mg26)*Y(He4)*Y(Ne22)*state.rho + -screened_rates(k_mg26__he4_ne22)*Y(Mg26)) +
        (screened_rates(k_n_mg25__mg26)*Y(Mg25)*Y(N)*state.rho + -screened_rates(k_mg26__n_mg25)*Y(Mg26)) +
        (-screened_rates(k_p_mg26__al27)*Y(Mg26)*Y(H1)*state.rho + screened_rates(k_al27__p_mg26)*Y(Al27)) +
        (-screened_rates(k_he4_mg26__si30)*Y(He4)*Y(Mg26)*state.rho + screened_rates(k_si30__he4_mg26)*Y(Si30)) +
        (screened_rates(k_he4_na23__p_mg26)*Y(He4)*Y(Na23)*state.rho + -screened_rates(k_p_mg26__he4_na23)*Y(Mg26)*Y(H1)*state.rho) +
        (-screened_rates(k_he4_mg26__n_si29)*Y(He4)*Y(Mg26)*state.rho + screened_rates(k_n_si29__he4_mg26)*Y(N)*Y(Si29)*state.rho) +
        (screened_rates(k_n_al26__p_mg26)*Y(Al26)*Y(N)*state.rho + -screened_rates(k_p_mg26__n_al26)*Y(Mg26)*Y(H1)*state.rho);

    ydot_nuc(Al25) =
        -screened_rates(k_al25__mg25__weak__wc12)*Y(Al25) +
        (screened_rates(k_he4_na21__al25)*Y(He4)*Y(Na21)*state.rho + -screened_rates(k_al25__he4_na21)*Y(Al25)) +
        (screened_rates(k_p_mg24__al25)*Y(Mg24)*Y(H1)*state.rho + -screened_rates(k_al25__p_mg24)*Y(Al25)) +
        (-screened_rates(k_n_al25__al26)*Y(Al25)*Y(N)*state.rho + screened_rates(k_al26__n_al25)*Y(Al26)) +
        (-screened_rates(k_he4_al25__p29)*Y(Al25)*Y(He4)*state.rho + screened_rates(k_p29__he4_al25)*Y(P29)) +
        (-screened_rates(k_n_al25__p_mg25)*Y(Al25)*Y(N)*state.rho + screened_rates(k_p_mg25__n_al25)*Y(Mg25)*Y(H1)*state.rho) +
        (-screened_rates(k_n_al25__he4_na22)*Y(Al25)*Y(N)*state.rho + screened_rates(k_he4_na22__n_al25)*Y(He4)*Y(Na22)*state.rho) +
        (-screened_rates(k_he4_al25__p_si28)*Y(Al25)*Y(He4)*state.rho + screened_rates(k_p_si28__he4_al25)*Y(H1)*Y(Si28)*state.rho);

    ydot_nuc(Al26) =
        -screened_rates(k_al26__mg26__weak__wc12)*Y(Al26) +
        (screened_rates(k_he4_na22__al26)*Y(He4)*Y(Na22)*state.rho + -screened_rates(k_al26__he4_na22)*Y(Al26)) +
        (screened_rates(k_p_mg25__al26)*Y(Mg25)*Y(H1)*state.rho + -screened_rates(k_al26__p_mg25)*Y(Al26)) +
        (screened_rates(k_n_al25__al26)*Y(Al25)*Y(N)*state.rho + -screened_rates(k_al26__n_al25)*Y(Al26)) +
        (-screened_rates(k_n_al26__al27)*Y(Al26)*Y(N)*state.rho + screened_rates(k_al27__n_al26)*Y(Al27)) +
        (-screened_rates(k_he4_al26__p30)*Y(Al26)*Y(He4)*state.rho + screened_rates(k_p30__he4_al26)*Y(P30)) +
        (screened_rates(k_he4_mg23__p_al26)*Y(He4)*Y(Mg23)*state.rho + -screened_rates(k_p_al26__he4_mg23)*Y(Al26)*Y(H1)*state.rho) +
        (-screened_rates(k_n_al26__p_mg26)*Y(Al26)*Y(N)*state.rho + screened_rates(k_p_mg26__n_al26)*Y(Mg26)*Y(H1)*state.rho) +
        (-screened_rates(k_n_al26__he4_na23)*Y(Al26)*Y(N)*state.rho + screened_rates(k_he4_na23__n_al26)*Y(He4)*Y(Na23)*state.rho) +
        (-screened_rates(k_he4_al26__p_si29)*Y(Al26)*Y(He4)*state.rho + screened_rates(k_p_si29__he4_al26)*Y(H1)*Y(Si29)*state.rho) +
        (screened_rates(k_n_p29__he4_al26)*Y(N)*Y(P29)*state.rho + -screened_rates(k_he4_al26__n_p29)*Y(Al26)*Y(He4)*state.rho);

    ydot_nuc(Al27) =
        (screened_rates(k_he4_na23__al27)*Y(He4)*Y(Na23)*state.rho + -screened_rates(k_al27__he4_na23)*Y(Al27)) +
        (screened_rates(k_p_mg26__al27)*Y(Mg26)*Y(H1)*state.rho + -screened_rates(k_al27__p_mg26)*Y(Al27)) +
        (screened_rates(k_n_al26__al27)*Y(Al26)*Y(N)*state.rho + -screened_rates(k_al27__n_al26)*Y(Al27)) +
        (-screened_rates(k_p_al27__si28)*Y(Al27)*Y(H1)*state.rho + screened_rates(k_si28__p_al27)*Y(Si28)) +
        (-screened_rates(k_he4_al27__p31)*Y(Al27)*Y(He4)*state.rho + screened_rates(k_p31__he4_al27)*Y(P31)) +
        (screened_rates(k_c12_o16__p_al27)*Y(C12)*Y(O16)*state.rho + -screened_rates(k_p_al27__c12_o16)*Y(Al27)*Y(H1)*state.rho) +
        (-screened_rates(k_p_al27__he4_mg24)*Y(Al27)*Y(H1)*state.rho + screened_rates(k_he4_mg24__p_al27)*Y(He4)*Y(Mg24)*state.rho) +
        (-screened_rates(k_he4_al27__p_si30)*Y(Al27)*Y(He4)*state.rho + screened_rates(k_p_si30__he4_al27)*Y(H1)*Y(Si30)*state.rho) +
        (screened_rates(k_n_p30__he4_al27)*Y(N)*Y(P30)*state.rho + -screened_rates(k_he4_al27__n_p30)*Y(Al27)*Y(He4)*state.rho);

    ydot_nuc(Si28) =
        (screened_rates(k_he4_mg24__si28)*Y(He4)*Y(Mg24)*state.rho + -screened_rates(k_si28__he4_mg24)*Y(Si28)) +
        (screened_rates(k_p_al27__si28)*Y(Al27)*Y(H1)*state.rho + -screened_rates(k_si28__p_al27)*Y(Si28)) +
        (-screened_rates(k_n_si28__si29)*Y(N)*Y(Si28)*state.rho + screened_rates(k_si29__n_si28)*Y(Si29)) +
        (-screened_rates(k_p_si28__p29)*Y(H1)*Y(Si28)*state.rho + screened_rates(k_p29__p_si28)*Y(P29)) +
        (-screened_rates(k_he4_si28__s32)*Y(He4)*Y(Si28)*state.rho + screened_rates(k_s32__he4_si28)*Y(S32)) +
        (0.5*screened_rates(k_o16_o16__he4_si28)*std::pow(Y(O16), 2)*state.rho + -screened_rates(k_he4_si28__o16_o16)*Y(He4)*Y(Si28)*state.rho) +
        (screened_rates(k_c12_ne20__he4_si28)*Y(C12)*Y(Ne20)*state.rho + -screened_rates(k_he4_si28__c12_ne20)*Y(He4)*Y(Si28)*state.rho) +
        (screened_rates(k_he4_mg25__n_si28)*Y(He4)*Y(Mg25)*state.rho + -screened_rates(k_n_si28__he4_mg25)*Y(N)*Y(Si28)*state.rho) +
        (screened_rates(k_he4_al25__p_si28)*Y(Al25)*Y(He4)*state.rho + -screened_rates(k_p_si28__he4_al25)*Y(H1)*Y(Si28)*state.rho) +
        (screened_rates(k_p_p31__he4_si28)*Y(P31)*Y(H1)*state.rho + -screened_rates(k_he4_si28__p_p31)*Y(He4)*Y(Si28)*state.rho);

    ydot_nuc(Si29) =
        screened_rates(k_p29__si29__weak__wc12)*Y(P29) +
        (screened_rates(k_he4_mg25__si29)*Y(He4)*Y(Mg25)*state.rho + -screened_rates(k_si29__he4_mg25)*Y(Si29)) +
        (screened_rates(k_n_si28__si29)*Y(N)*Y(Si28)*state.rho + -screened_rates(k_si29__n_si28)*Y(Si29)) +
        (-screened_rates(k_n_si29__si30)*Y(N)*Y(Si29)*state.rho + screened_rates(k_si30__n_si29)*Y(Si30)) +
        (-screened_rates(k_p_si29__p30)*Y(H1)*Y(Si29)*state.rho + screened_rates(k_p30__p_si29)*Y(P30)) +
        (-screened_rates(k_he4_si29__s33)*Y(He4)*Y(Si29)*state.rho + screened_rates(k_s33__he4_si29)*Y(S33)) +
        (screened_rates(k_he4_mg26__n_si29)*Y(He4)*Y(Mg26)*state.rho + -screened_rates(k_n_si29__he4_mg26)*Y(N)*Y(Si29)*state.rho) +
        (screened_rates(k_he4_al26__p_si29)*Y(Al26)*Y(He4)*state.rho + -screened_rates(k_p_si29__he4_al26)*Y(H1)*Y(Si29)*state.rho) +
        (screened_rates(k_n_p29__p_si29)*Y(N)*Y(P29)*state.rho + -screened_rates(k_p_si29__n_p29)*Y(H1)*Y(Si29)*state.rho) +
        (screened_rates(k_p_p32__he4_si29)*Y(P32)*Y(H1)*state.rho + -screened_rates(k_he4_si29__p_p32)*Y(He4)*Y(Si29)*state.rho) +
        (screened_rates(k_n_s32__he4_si29)*Y(N)*Y(S32)*state.rho + -screened_rates(k_he4_si29__n_s32)*Y(He4)*Y(Si29)*state.rho);

    ydot_nuc(Si30) =
        screened_rates(k_p30__si30__weak__wc12)*Y(P30) +
        (screened_rates(k_he4_mg26__si30)*Y(He4)*Y(Mg26)*state.rho + -screened_rates(k_si30__he4_mg26)*Y(Si30)) +
        (screened_rates(k_n_si29__si30)*Y(N)*Y(Si29)*state.rho + -screened_rates(k_si30__n_si29)*Y(Si30)) +
        (-screened_rates(k_n_si30__si31)*Y(N)*Y(Si30)*state.rho + screened_rates(k_si31__n_si30)*Y(Si31)) +
        (-screened_rates(k_p_si30__p31)*Y(H1)*Y(Si30)*state.rho + screened_rates(k_p31__p_si30)*Y(P31)) +
        (-screened_rates(k_he4_si30__s34)*Y(He4)*Y(Si30)*state.rho + screened_rates(k_s34__he4_si30)*Y(S34)) +
        (screened_rates(k_he4_al27__p_si30)*Y(Al27)*Y(He4)*state.rho + -screened_rates(k_p_si30__he4_al27)*Y(H1)*Y(Si30)*state.rho) +
        (screened_rates(k_n_p30__p_si30)*Y(N)*Y(P30)*state.rho + -screened_rates(k_p_si30__n_p30)*Y(H1)*Y(Si30)*state.rho) +
        (screened_rates(k_p_p33__he4_si30)*Y(P33)*Y(H1)*state.rho + -screened_rates(k_he4_si30__p_p33)*Y(He4)*Y(Si30)*state.rho) +
        (screened_rates(k_n_s33__he4_si30)*Y(N)*Y(S33)*state.rho + -screened_rates(k_he4_si30__n_s33)*Y(He4)*Y(Si30)*state.rho);

    ydot_nuc(Si31) =
        -screened_rates(k_si31__p31__weak__wc12)*Y(Si31) +
        (screened_rates(k_n_si30__si31)*Y(N)*Y(Si30)*state.rho + -screened_rates(k_si31__n_si30)*Y(Si31)) +
        (-screened_rates(k_n_si31__si32)*Y(N)*Y(Si31)*state.rho + screened_rates(k_si32__n_si31)*Y(Si32)) +
        (-screened_rates(k_p_si31__p32)*Y(H1)*Y(Si31)*state.rho + screened_rates(k_p32__p_si31)*Y(P32)) +
        (-screened_rates(k_he4_si31__s35)*Y(He4)*Y(Si31)*state.rho + screened_rates(k_s35__he4_si31)*Y(S35)) +
        (-screened_rates(k_p_si31__n_p31)*Y(H1)*Y(Si31)*state.rho + screened_rates(k_n_p31__p_si31)*Y(N)*Y(P31)*state.rho) +
        (-screened_rates(k_he4_si31__n_s34)*Y(He4)*Y(Si31)*state.rho + screened_rates(k_n_s34__he4_si31)*Y(N)*Y(S34)*state.rho);

    ydot_nuc(Si32) =
        -screened_rates(k_si32__p32__weak__wc12)*Y(Si32) +
        (screened_rates(k_n_si31__si32)*Y(N)*Y(Si31)*state.rho + -screened_rates(k_si32__n_si31)*Y(Si32)) +
        (-screened_rates(k_p_si32__p33)*Y(H1)*Y(Si32)*state.rho + screened_rates(k_p33__p_si32)*Y(P33)) +
        (-screened_rates(k_he4_si32__s36)*Y(He4)*Y(Si32)*state.rho + screened_rates(k_s36__he4_si32)*Y(S36)) +
        (screened_rates(k_n_p32__p_si32)*Y(N)*Y(P32)*state.rho + -screened_rates(k_p_si32__n_p32)*Y(H1)*Y(Si32)*state.rho) +
        (screened_rates(k_n_s35__he4_si32)*Y(N)*Y(S35)*state.rho + -screened_rates(k_he4_si32__n_s35)*Y(He4)*Y(Si32)*state.rho);

    ydot_nuc(P29) =
        -screened_rates(k_p29__si29__weak__wc12)*Y(P29) +
        (screened_rates(k_he4_al25__p29)*Y(Al25)*Y(He4)*state.rho + -screened_rates(k_p29__he4_al25)*Y(P29)) +
        (screened_rates(k_p_si28__p29)*Y(H1)*Y(Si28)*state.rho + -screened_rates(k_p29__p_si28)*Y(P29)) +
        (-screened_rates(k_n_p29__p30)*Y(N)*Y(P29)*state.rho + screened_rates(k_p30__n_p29)*Y(P30)) +
        (-screened_rates(k_he4_p29__cl33)*Y(He4)*Y(P29)*state.rho + screened_rates(k_cl33__he4_p29)*Y(Cl33)) +
        (-screened_rates(k_n_p29__p_si29)*Y(N)*Y(P29)*state.rho + screened_rates(k_p_si29__n_p29)*Y(H1)*Y(Si29)*state.rho) +
        (-screened_rates(k_n_p29__he4_al26)*Y(N)*Y(P29)*state.rho + screened_rates(k_he4_al26__n_p29)*Y(Al26)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_p29__p_s32)*Y(He4)*Y(P29)*state.rho + screened_rates(k_p_s32__he4_p29)*Y(H1)*Y(S32)*state.rho);

    ydot_nuc(P30) =
        -screened_rates(k_p30__si30__weak__wc12)*Y(P30) +
        (screened_rates(k_he4_al26__p30)*Y(Al26)*Y(He4)*state.rho + -screened_rates(k_p30__he4_al26)*Y(P30)) +
        (screened_rates(k_p_si29__p30)*Y(H1)*Y(Si29)*state.rho + -screened_rates(k_p30__p_si29)*Y(P30)) +
        (screened_rates(k_n_p29__p30)*Y(N)*Y(P29)*state.rho + -screened_rates(k_p30__n_p29)*Y(P30)) +
        (-screened_rates(k_n_p30__p31)*Y(N)*Y(P30)*state.rho + screened_rates(k_p31__n_p30)*Y(P31)) +
        (-screened_rates(k_he4_p30__cl34)*Y(He4)*Y(P30)*state.rho + screened_rates(k_cl34__he4_p30)*Y(Cl34)) +
        (-screened_rates(k_n_p30__p_si30)*Y(N)*Y(P30)*state.rho + screened_rates(k_p_si30__n_p30)*Y(H1)*Y(Si30)*state.rho) +
        (-screened_rates(k_n_p30__he4_al27)*Y(N)*Y(P30)*state.rho + screened_rates(k_he4_al27__n_p30)*Y(Al27)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_p30__p_s33)*Y(He4)*Y(P30)*state.rho + screened_rates(k_p_s33__he4_p30)*Y(H1)*Y(S33)*state.rho) +
        (screened_rates(k_n_cl33__he4_p30)*Y(Cl33)*Y(N)*state.rho + -screened_rates(k_he4_p30__n_cl33)*Y(He4)*Y(P30)*state.rho);

    ydot_nuc(P31) =
        screened_rates(k_si31__p31__weak__wc12)*Y(Si31) +
        (screened_rates(k_he4_al27__p31)*Y(Al27)*Y(He4)*state.rho + -screened_rates(k_p31__he4_al27)*Y(P31)) +
        (screened_rates(k_p_si30__p31)*Y(H1)*Y(Si30)*state.rho + -screened_rates(k_p31__p_si30)*Y(P31)) +
        (screened_rates(k_n_p30__p31)*Y(N)*Y(P30)*state.rho + -screened_rates(k_p31__n_p30)*Y(P31)) +
        (-screened_rates(k_n_p31__p32)*Y(N)*Y(P31)*state.rho + screened_rates(k_p32__n_p31)*Y(P32)) +
        (-screened_rates(k_p_p31__s32)*Y(P31)*Y(H1)*state.rho + screened_rates(k_s32__p_p31)*Y(S32)) +
        (-screened_rates(k_he4_p31__cl35)*Y(He4)*Y(P31)*state.rho + screened_rates(k_cl35__he4_p31)*Y(Cl35)) +
        (0.5*screened_rates(k_o16_o16__p_p31)*std::pow(Y(O16), 2)*state.rho + -screened_rates(k_p_p31__o16_o16)*Y(P31)*Y(H1)*state.rho) +
        (screened_rates(k_c12_ne20__p_p31)*Y(C12)*Y(Ne20)*state.rho + -screened_rates(k_p_p31__c12_ne20)*Y(P31)*Y(H1)*state.rho) +
        (screened_rates(k_p_si31__n_p31)*Y(H1)*Y(Si31)*state.rho + -screened_rates(k_n_p31__p_si31)*Y(N)*Y(P31)*state.rho) +
        (-screened_rates(k_p_p31__he4_si28)*Y(P31)*Y(H1)*state.rho + screened_rates(k_he4_si28__p_p31)*Y(He4)*Y(Si28)*state.rho) +
        (-screened_rates(k_he4_p31__p_s34)*Y(He4)*Y(P31)*state.rho + screened_rates(k_p_s34__he4_p31)*Y(H1)*Y(S34)*state.rho) +
        (screened_rates(k_n_cl34__he4_p31)*Y(Cl34)*Y(N)*state.rho + -screened_rates(k_he4_p31__n_cl34)*Y(He4)*Y(P31)*state.rho);

    ydot_nuc(P32) =
        screened_rates(k_si32__p32__weak__wc12)*Y(Si32) +
        -screened_rates(k_p32__s32__weak__wc12)*Y(P32) +
        (screened_rates(k_p_si31__p32)*Y(H1)*Y(Si31)*state.rho + -screened_rates(k_p32__p_si31)*Y(P32)) +
        (screened_rates(k_n_p31__p32)*Y(N)*Y(P31)*state.rho + -screened_rates(k_p32__n_p31)*Y(P32)) +
        (-screened_rates(k_n_p32__p33)*Y(N)*Y(P32)*state.rho + screened_rates(k_p33__n_p32)*Y(P33)) +
        (-screened_rates(k_p_p32__s33)*Y(P32)*Y(H1)*state.rho + screened_rates(k_s33__p_p32)*Y(S33)) +
        (-screened_rates(k_he4_p32__cl36)*Y(He4)*Y(P32)*state.rho + screened_rates(k_cl36__he4_p32)*Y(Cl36)) +
        (-screened_rates(k_n_p32__p_si32)*Y(N)*Y(P32)*state.rho + screened_rates(k_p_si32__n_p32)*Y(H1)*Y(Si32)*state.rho) +
        (-screened_rates(k_p_p32__n_s32)*Y(P32)*Y(H1)*state.rho + screened_rates(k_n_s32__p_p32)*Y(N)*Y(S32)*state.rho) +
        (-screened_rates(k_p_p32__he4_si29)*Y(P32)*Y(H1)*state.rho + screened_rates(k_he4_si29__p_p32)*Y(He4)*Y(Si29)*state.rho) +
        (screened_rates(k_p_s35__he4_p32)*Y(H1)*Y(S35)*state.rho + -screened_rates(k_he4_p32__p_s35)*Y(He4)*Y(P32)*state.rho) +
        (screened_rates(k_n_cl35__he4_p32)*Y(Cl35)*Y(N)*state.rho + -screened_rates(k_he4_p32__n_cl35)*Y(He4)*Y(P32)*state.rho);

    ydot_nuc(P33) =
        -screened_rates(k_p33__s33__weak__wc12)*Y(P33) +
        (screened_rates(k_p_si32__p33)*Y(H1)*Y(Si32)*state.rho + -screened_rates(k_p33__p_si32)*Y(P33)) +
        (screened_rates(k_n_p32__p33)*Y(N)*Y(P32)*state.rho + -screened_rates(k_p33__n_p32)*Y(P33)) +
        (-screened_rates(k_p_p33__s34)*Y(P33)*Y(H1)*state.rho + screened_rates(k_s34__p_p33)*Y(S34)) +
        (-screened_rates(k_he4_p33__cl37)*Y(He4)*Y(P33)*state.rho + screened_rates(k_cl37__he4_p33)*Y(Cl37)) +
        (-screened_rates(k_p_p33__he4_si30)*Y(P33)*Y(H1)*state.rho + screened_rates(k_he4_si30__p_p33)*Y(He4)*Y(Si30)*state.rho) +
        (screened_rates(k_n_s33__p_p33)*Y(N)*Y(S33)*state.rho + -screened_rates(k_p_p33__n_s33)*Y(P33)*Y(H1)*state.rho) +
        (screened_rates(k_p_s36__he4_p33)*Y(H1)*Y(S36)*state.rho + -screened_rates(k_he4_p33__p_s36)*Y(He4)*Y(P33)*state.rho) +
        (screened_rates(k_n_cl36__he4_p33)*Y(Cl36)*Y(N)*state.rho + -screened_rates(k_he4_p33__n_cl36)*Y(He4)*Y(P33)*state.rho);

    ydot_nuc(S32) =
        screened_rates(k_p32__s32__weak__wc12)*Y(P32) +
        (screened_rates(k_he4_si28__s32)*Y(He4)*Y(Si28)*state.rho + -screened_rates(k_s32__he4_si28)*Y(S32)) +
        (screened_rates(k_p_p31__s32)*Y(P31)*Y(H1)*state.rho + -screened_rates(k_s32__p_p31)*Y(S32)) +
        (-screened_rates(k_n_s32__s33)*Y(N)*Y(S32)*state.rho + screened_rates(k_s33__n_s32)*Y(S33)) +
        (-screened_rates(k_p_s32__cl33)*Y(H1)*Y(S32)*state.rho + screened_rates(k_cl33__p_s32)*Y(Cl33)) +
        (-screened_rates(k_he4_s32__ar36)*Y(He4)*Y(S32)*state.rho + screened_rates(k_ar36__he4_s32)*Y(Ar36)) +
        (screened_rates(k_he4_p29__p_s32)*Y(He4)*Y(P29)*state.rho + -screened_rates(k_p_s32__he4_p29)*Y(H1)*Y(S32)*state.rho) +
        (screened_rates(k_p_p32__n_s32)*Y(P32)*Y(H1)*state.rho + -screened_rates(k_n_s32__p_p32)*Y(N)*Y(S32)*state.rho) +
        (-screened_rates(k_n_s32__he4_si29)*Y(N)*Y(S32)*state.rho + screened_rates(k_he4_si29__n_s32)*Y(He4)*Y(Si29)*state.rho) +
        (screened_rates(k_p_cl35__he4_s32)*Y(Cl35)*Y(H1)*state.rho + -screened_rates(k_he4_s32__p_cl35)*Y(He4)*Y(S32)*state.rho);

    ydot_nuc(S33) =
        screened_rates(k_p33__s33__weak__wc12)*Y(P33) +
        screened_rates(k_cl33__s33__weak__wc12)*Y(Cl33) +
        (screened_rates(k_he4_si29__s33)*Y(He4)*Y(Si29)*state.rho + -screened_rates(k_s33__he4_si29)*Y(S33)) +
        (screened_rates(k_p_p32__s33)*Y(P32)*Y(H1)*state.rho + -screened_rates(k_s33__p_p32)*Y(S33)) +
        (screened_rates(k_n_s32__s33)*Y(N)*Y(S32)*state.rho + -screened_rates(k_s33__n_s32)*Y(S33)) +
        (-screened_rates(k_n_s33__s34)*Y(N)*Y(S33)*state.rho + screened_rates(k_s34__n_s33)*Y(S34)) +
        (-screened_rates(k_p_s33__cl34)*Y(H1)*Y(S33)*state.rho + screened_rates(k_cl34__p_s33)*Y(Cl34)) +
        (-screened_rates(k_he4_s33__ar37)*Y(He4)*Y(S33)*state.rho + screened_rates(k_ar37__he4_s33)*Y(Ar37)) +
        (screened_rates(k_he4_p30__p_s33)*Y(He4)*Y(P30)*state.rho + -screened_rates(k_p_s33__he4_p30)*Y(H1)*Y(S33)*state.rho) +
        (-screened_rates(k_n_s33__p_p33)*Y(N)*Y(S33)*state.rho + screened_rates(k_p_p33__n_s33)*Y(P33)*Y(H1)*state.rho) +
        (-screened_rates(k_n_s33__he4_si30)*Y(N)*Y(S33)*state.rho + screened_rates(k_he4_si30__n_s33)*Y(He4)*Y(Si30)*state.rho) +
        (screened_rates(k_n_cl33__p_s33)*Y(Cl33)*Y(N)*state.rho + -screened_rates(k_p_s33__n_cl33)*Y(H1)*Y(S33)*state.rho) +
        (screened_rates(k_p_cl36__he4_s33)*Y(Cl36)*Y(H1)*state.rho + -screened_rates(k_he4_s33__p_cl36)*Y(He4)*Y(S33)*state.rho) +
        (screened_rates(k_n_ar36__he4_s33)*Y(Ar36)*Y(N)*state.rho + -screened_rates(k_he4_s33__n_ar36)*Y(He4)*Y(S33)*state.rho);

    ydot_nuc(S34) =
        screened_rates(k_cl34__s34__weak__wc12)*Y(Cl34) +
        (screened_rates(k_he4_si30__s34)*Y(He4)*Y(Si30)*state.rho + -screened_rates(k_s34__he4_si30)*Y(S34)) +
        (screened_rates(k_p_p33__s34)*Y(P33)*Y(H1)*state.rho + -screened_rates(k_s34__p_p33)*Y(S34)) +
        (screened_rates(k_n_s33__s34)*Y(N)*Y(S33)*state.rho + -screened_rates(k_s34__n_s33)*Y(S34)) +
        (-screened_rates(k_n_s34__s35)*Y(N)*Y(S34)*state.rho + screened_rates(k_s35__n_s34)*Y(S35)) +
        (-screened_rates(k_p_s34__cl35)*Y(H1)*Y(S34)*state.rho + screened_rates(k_cl35__p_s34)*Y(Cl35)) +
        (-screened_rates(k_he4_s34__ar38)*Y(He4)*Y(S34)*state.rho + screened_rates(k_ar38__he4_s34)*Y(Ar38)) +
        (screened_rates(k_he4_si31__n_s34)*Y(He4)*Y(Si31)*state.rho + -screened_rates(k_n_s34__he4_si31)*Y(N)*Y(S34)*state.rho) +
        (screened_rates(k_he4_p31__p_s34)*Y(He4)*Y(P31)*state.rho + -screened_rates(k_p_s34__he4_p31)*Y(H1)*Y(S34)*state.rho) +
        (screened_rates(k_n_cl34__p_s34)*Y(Cl34)*Y(N)*state.rho + -screened_rates(k_p_s34__n_cl34)*Y(H1)*Y(S34)*state.rho) +
        (screened_rates(k_p_cl37__he4_s34)*Y(Cl37)*Y(H1)*state.rho + -screened_rates(k_he4_s34__p_cl37)*Y(He4)*Y(S34)*state.rho) +
        (screened_rates(k_n_ar37__he4_s34)*Y(Ar37)*Y(N)*state.rho + -screened_rates(k_he4_s34__n_ar37)*Y(He4)*Y(S34)*state.rho);

    ydot_nuc(S35) =
        -screened_rates(k_s35__cl35__weak__wc12)*Y(S35) +
        (screened_rates(k_he4_si31__s35)*Y(He4)*Y(Si31)*state.rho + -screened_rates(k_s35__he4_si31)*Y(S35)) +
        (screened_rates(k_n_s34__s35)*Y(N)*Y(S34)*state.rho + -screened_rates(k_s35__n_s34)*Y(S35)) +
        (-screened_rates(k_n_s35__s36)*Y(N)*Y(S35)*state.rho + screened_rates(k_s36__n_s35)*Y(S36)) +
        (-screened_rates(k_p_s35__cl36)*Y(H1)*Y(S35)*state.rho + screened_rates(k_cl36__p_s35)*Y(Cl36)) +
        (-screened_rates(k_he4_s35__ar39)*Y(He4)*Y(S35)*state.rho + screened_rates(k_ar39__he4_s35)*Y(Ar39)) +
        (-screened_rates(k_n_s35__he4_si32)*Y(N)*Y(S35)*state.rho + screened_rates(k_he4_si32__n_s35)*Y(He4)*Y(Si32)*state.rho) +
        (-screened_rates(k_p_s35__he4_p32)*Y(H1)*Y(S35)*state.rho + screened_rates(k_he4_p32__p_s35)*Y(He4)*Y(P32)*state.rho) +
        (-screened_rates(k_he4_s35__n_ar38)*Y(He4)*Y(S35)*state.rho + screened_rates(k_n_ar38__he4_s35)*Y(Ar38)*Y(N)*state.rho) +
        (screened_rates(k_n_cl35__p_s35)*Y(Cl35)*Y(N)*state.rho + -screened_rates(k_p_s35__n_cl35)*Y(H1)*Y(S35)*state.rho);

    ydot_nuc(S36) =
        screened_rates(k_cl36__s36__weak__wc12)*Y(Cl36) +
        (screened_rates(k_he4_si32__s36)*Y(He4)*Y(Si32)*state.rho + -screened_rates(k_s36__he4_si32)*Y(S36)) +
        (screened_rates(k_n_s35__s36)*Y(N)*Y(S35)*state.rho + -screened_rates(k_s36__n_s35)*Y(S36)) +
        (-screened_rates(k_p_s36__cl37)*Y(H1)*Y(S36)*state.rho + screened_rates(k_cl37__p_s36)*Y(Cl37)) +
        (-screened_rates(k_he4_s36__ar40)*Y(He4)*Y(S36)*state.rho + screened_rates(k_ar40__he4_s36)*Y(Ar40)) +
        (-screened_rates(k_p_s36__he4_p33)*Y(H1)*Y(S36)*state.rho + screened_rates(k_he4_p33__p_s36)*Y(He4)*Y(P33)*state.rho) +
        (screened_rates(k_n_cl36__p_s36)*Y(Cl36)*Y(N)*state.rho + -screened_rates(k_p_s36__n_cl36)*Y(H1)*Y(S36)*state.rho) +
        (screened_rates(k_n_ar39__he4_s36)*Y(Ar39)*Y(N)*state.rho + -screened_rates(k_he4_s36__n_ar39)*Y(He4)*Y(S36)*state.rho);

    ydot_nuc(Cl33) =
        -screened_rates(k_cl33__s33__weak__wc12)*Y(Cl33) +
        (screened_rates(k_he4_p29__cl33)*Y(He4)*Y(P29)*state.rho + -screened_rates(k_cl33__he4_p29)*Y(Cl33)) +
        (screened_rates(k_p_s32__cl33)*Y(H1)*Y(S32)*state.rho + -screened_rates(k_cl33__p_s32)*Y(Cl33)) +
        (-screened_rates(k_n_cl33__cl34)*Y(Cl33)*Y(N)*state.rho + screened_rates(k_cl34__n_cl33)*Y(Cl34)) +
        (-screened_rates(k_he4_cl33__k37)*Y(Cl33)*Y(He4)*state.rho + screened_rates(k_k37__he4_cl33)*Y(K37)) +
        (-screened_rates(k_n_cl33__p_s33)*Y(Cl33)*Y(N)*state.rho + screened_rates(k_p_s33__n_cl33)*Y(H1)*Y(S33)*state.rho) +
        (-screened_rates(k_n_cl33__he4_p30)*Y(Cl33)*Y(N)*state.rho + screened_rates(k_he4_p30__n_cl33)*Y(He4)*Y(P30)*state.rho) +
        (-screened_rates(k_he4_cl33__p_ar36)*Y(Cl33)*Y(He4)*state.rho + screened_rates(k_p_ar36__he4_cl33)*Y(Ar36)*Y(H1)*state.rho);

    ydot_nuc(Cl34) =
        -screened_rates(k_cl34__s34__weak__wc12)*Y(Cl34) +
        (screened_rates(k_he4_p30__cl34)*Y(He4)*Y(P30)*state.rho + -screened_rates(k_cl34__he4_p30)*Y(Cl34)) +
        (screened_rates(k_p_s33__cl34)*Y(H1)*Y(S33)*state.rho + -screened_rates(k_cl34__p_s33)*Y(Cl34)) +
        (screened_rates(k_n_cl33__cl34)*Y(Cl33)*Y(N)*state.rho + -screened_rates(k_cl34__n_cl33)*Y(Cl34)) +
        (-screened_rates(k_n_cl34__cl35)*Y(Cl34)*Y(N)*state.rho + screened_rates(k_cl35__n_cl34)*Y(Cl35)) +
        (-screened_rates(k_he4_cl34__k38)*Y(Cl34)*Y(He4)*state.rho + screened_rates(k_k38__he4_cl34)*Y(K38)) +
        (-screened_rates(k_n_cl34__p_s34)*Y(Cl34)*Y(N)*state.rho + screened_rates(k_p_s34__n_cl34)*Y(H1)*Y(S34)*state.rho) +
        (-screened_rates(k_n_cl34__he4_p31)*Y(Cl34)*Y(N)*state.rho + screened_rates(k_he4_p31__n_cl34)*Y(He4)*Y(P31)*state.rho) +
        (-screened_rates(k_he4_cl34__p_ar37)*Y(Cl34)*Y(He4)*state.rho + screened_rates(k_p_ar37__he4_cl34)*Y(Ar37)*Y(H1)*state.rho) +
        (screened_rates(k_n_k37__he4_cl34)*Y(K37)*Y(N)*state.rho + -screened_rates(k_he4_cl34__n_k37)*Y(Cl34)*Y(He4)*state.rho);

    ydot_nuc(Cl35) =
        screened_rates(k_s35__cl35__weak__wc12)*Y(S35) +
        (screened_rates(k_he4_p31__cl35)*Y(He4)*Y(P31)*state.rho + -screened_rates(k_cl35__he4_p31)*Y(Cl35)) +
        (screened_rates(k_p_s34__cl35)*Y(H1)*Y(S34)*state.rho + -screened_rates(k_cl35__p_s34)*Y(Cl35)) +
        (screened_rates(k_n_cl34__cl35)*Y(Cl34)*Y(N)*state.rho + -screened_rates(k_cl35__n_cl34)*Y(Cl35)) +
        (-screened_rates(k_n_cl35__cl36)*Y(Cl35)*Y(N)*state.rho + screened_rates(k_cl36__n_cl35)*Y(Cl36)) +
        (-screened_rates(k_p_cl35__ar36)*Y(Cl35)*Y(H1)*state.rho + screened_rates(k_ar36__p_cl35)*Y(Ar36)) +
        (-screened_rates(k_he4_cl35__k39)*Y(Cl35)*Y(He4)*state.rho + screened_rates(k_k39__he4_cl35)*Y(K39)) +
        (-screened_rates(k_n_cl35__p_s35)*Y(Cl35)*Y(N)*state.rho + screened_rates(k_p_s35__n_cl35)*Y(H1)*Y(S35)*state.rho) +
        (-screened_rates(k_n_cl35__he4_p32)*Y(Cl35)*Y(N)*state.rho + screened_rates(k_he4_p32__n_cl35)*Y(He4)*Y(P32)*state.rho) +
        (-screened_rates(k_p_cl35__he4_s32)*Y(Cl35)*Y(H1)*state.rho + screened_rates(k_he4_s32__p_cl35)*Y(He4)*Y(S32)*state.rho) +
        (-screened_rates(k_he4_cl35__p_ar38)*Y(Cl35)*Y(He4)*state.rho + screened_rates(k_p_ar38__he4_cl35)*Y(Ar38)*Y(H1)*state.rho) +
        (screened_rates(k_n_k38__he4_cl35)*Y(K38)*Y(N)*state.rho + -screened_rates(k_he4_cl35__n_k38)*Y(Cl35)*Y(He4)*state.rho);

    ydot_nuc(Cl36) =
        -screened_rates(k_cl36__ar36__weak__wc12)*Y(Cl36) +
        -screened_rates(k_cl36__s36__weak__wc12)*Y(Cl36) +
        (screened_rates(k_he4_p32__cl36)*Y(He4)*Y(P32)*state.rho + -screened_rates(k_cl36__he4_p32)*Y(Cl36)) +
        (screened_rates(k_p_s35__cl36)*Y(H1)*Y(S35)*state.rho + -screened_rates(k_cl36__p_s35)*Y(Cl36)) +
        (screened_rates(k_n_cl35__cl36)*Y(Cl35)*Y(N)*state.rho + -screened_rates(k_cl36__n_cl35)*Y(Cl36)) +
        (-screened_rates(k_n_cl36__cl37)*Y(Cl36)*Y(N)*state.rho + screened_rates(k_cl37__n_cl36)*Y(Cl37)) +
        (-screened_rates(k_p_cl36__ar37)*Y(Cl36)*Y(H1)*state.rho + screened_rates(k_ar37__p_cl36)*Y(Ar37)) +
        (-screened_rates(k_he4_cl36__k40)*Y(Cl36)*Y(He4)*state.rho + screened_rates(k_k40__he4_cl36)*Y(K40)) +
        (-screened_rates(k_n_cl36__p_s36)*Y(Cl36)*Y(N)*state.rho + screened_rates(k_p_s36__n_cl36)*Y(H1)*Y(S36)*state.rho) +
        (-screened_rates(k_n_cl36__he4_p33)*Y(Cl36)*Y(N)*state.rho + screened_rates(k_he4_p33__n_cl36)*Y(He4)*Y(P33)*state.rho) +
        (-screened_rates(k_p_cl36__he4_s33)*Y(Cl36)*Y(H1)*state.rho + screened_rates(k_he4_s33__p_cl36)*Y(He4)*Y(S33)*state.rho) +
        (screened_rates(k_n_ar36__p_cl36)*Y(Ar36)*Y(N)*state.rho + -screened_rates(k_p_cl36__n_ar36)*Y(Cl36)*Y(H1)*state.rho) +
        (screened_rates(k_p_ar39__he4_cl36)*Y(Ar39)*Y(H1)*state.rho + -screened_rates(k_he4_cl36__p_ar39)*Y(Cl36)*Y(He4)*state.rho) +
        (screened_rates(k_n_k39__he4_cl36)*Y(K39)*Y(N)*state.rho + -screened_rates(k_he4_cl36__n_k39)*Y(Cl36)*Y(He4)*state.rho);

    ydot_nuc(Cl37) =
        screened_rates(k_ar37__cl37__weak__wc12)*Y(Ar37) +
        (screened_rates(k_he4_p33__cl37)*Y(He4)*Y(P33)*state.rho + -screened_rates(k_cl37__he4_p33)*Y(Cl37)) +
        (screened_rates(k_p_s36__cl37)*Y(H1)*Y(S36)*state.rho + -screened_rates(k_cl37__p_s36)*Y(Cl37)) +
        (screened_rates(k_n_cl36__cl37)*Y(Cl36)*Y(N)*state.rho + -screened_rates(k_cl37__n_cl36)*Y(Cl37)) +
        (-screened_rates(k_p_cl37__ar38)*Y(Cl37)*Y(H1)*state.rho + screened_rates(k_ar38__p_cl37)*Y(Ar38)) +
        (-screened_rates(k_he4_cl37__k41)*Y(Cl37)*Y(He4)*state.rho + screened_rates(k_k41__he4_cl37)*Y(K41)) +
        (-screened_rates(k_p_cl37__he4_s34)*Y(Cl37)*Y(H1)*state.rho + screened_rates(k_he4_s34__p_cl37)*Y(He4)*Y(S34)*state.rho) +
        (screened_rates(k_n_ar37__p_cl37)*Y(Ar37)*Y(N)*state.rho + -screened_rates(k_p_cl37__n_ar37)*Y(Cl37)*Y(H1)*state.rho) +
        (screened_rates(k_p_ar40__he4_cl37)*Y(Ar40)*Y(H1)*state.rho + -screened_rates(k_he4_cl37__p_ar40)*Y(Cl37)*Y(He4)*state.rho) +
        (screened_rates(k_n_k40__he4_cl37)*Y(K40)*Y(N)*state.rho + -screened_rates(k_he4_cl37__n_k40)*Y(Cl37)*Y(He4)*state.rho);

    ydot_nuc(Ar36) =
        screened_rates(k_cl36__ar36__weak__wc12)*Y(Cl36) +
        (screened_rates(k_he4_s32__ar36)*Y(He4)*Y(S32)*state.rho + -screened_rates(k_ar36__he4_s32)*Y(Ar36)) +
        (screened_rates(k_p_cl35__ar36)*Y(Cl35)*Y(H1)*state.rho + -screened_rates(k_ar36__p_cl35)*Y(Ar36)) +
        (-screened_rates(k_n_ar36__ar37)*Y(Ar36)*Y(N)*state.rho + screened_rates(k_ar37__n_ar36)*Y(Ar37)) +
        (-screened_rates(k_p_ar36__k37)*Y(Ar36)*Y(H1)*state.rho + screened_rates(k_k37__p_ar36)*Y(K37)) +
        (-screened_rates(k_he4_ar36__ca40)*Y(Ar36)*Y(He4)*state.rho + screened_rates(k_ca40__he4_ar36)*Y(Ca40)) +
        (screened_rates(k_he4_cl33__p_ar36)*Y(Cl33)*Y(He4)*state.rho + -screened_rates(k_p_ar36__he4_cl33)*Y(Ar36)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ar36__p_cl36)*Y(Ar36)*Y(N)*state.rho + screened_rates(k_p_cl36__n_ar36)*Y(Cl36)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ar36__he4_s33)*Y(Ar36)*Y(N)*state.rho + screened_rates(k_he4_s33__n_ar36)*Y(He4)*Y(S33)*state.rho) +
        (screened_rates(k_p_k39__he4_ar36)*Y(K39)*Y(H1)*state.rho + -screened_rates(k_he4_ar36__p_k39)*Y(Ar36)*Y(He4)*state.rho);

    ydot_nuc(Ar37) =
        -screened_rates(k_ar37__cl37__weak__wc12)*Y(Ar37) +
        screened_rates(k_k37__ar37__weak__wc12)*Y(K37) +
        (screened_rates(k_he4_s33__ar37)*Y(He4)*Y(S33)*state.rho + -screened_rates(k_ar37__he4_s33)*Y(Ar37)) +
        (screened_rates(k_p_cl36__ar37)*Y(Cl36)*Y(H1)*state.rho + -screened_rates(k_ar37__p_cl36)*Y(Ar37)) +
        (screened_rates(k_n_ar36__ar37)*Y(Ar36)*Y(N)*state.rho + -screened_rates(k_ar37__n_ar36)*Y(Ar37)) +
        (-screened_rates(k_n_ar37__ar38)*Y(Ar37)*Y(N)*state.rho + screened_rates(k_ar38__n_ar37)*Y(Ar38)) +
        (-screened_rates(k_p_ar37__k38)*Y(Ar37)*Y(H1)*state.rho + screened_rates(k_k38__p_ar37)*Y(K38)) +
        (-screened_rates(k_he4_ar37__ca41)*Y(Ar37)*Y(He4)*state.rho + screened_rates(k_ca41__he4_ar37)*Y(Ca41)) +
        (screened_rates(k_he4_cl34__p_ar37)*Y(Cl34)*Y(He4)*state.rho + -screened_rates(k_p_ar37__he4_cl34)*Y(Ar37)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ar37__p_cl37)*Y(Ar37)*Y(N)*state.rho + screened_rates(k_p_cl37__n_ar37)*Y(Cl37)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ar37__he4_s34)*Y(Ar37)*Y(N)*state.rho + screened_rates(k_he4_s34__n_ar37)*Y(He4)*Y(S34)*state.rho) +
        (screened_rates(k_n_k37__p_ar37)*Y(K37)*Y(N)*state.rho + -screened_rates(k_p_ar37__n_k37)*Y(Ar37)*Y(H1)*state.rho) +
        (screened_rates(k_p_k40__he4_ar37)*Y(K40)*Y(H1)*state.rho + -screened_rates(k_he4_ar37__p_k40)*Y(Ar37)*Y(He4)*state.rho) +
        (screened_rates(k_n_ca40__he4_ar37)*Y(Ca40)*Y(N)*state.rho + -screened_rates(k_he4_ar37__n_ca40)*Y(Ar37)*Y(He4)*state.rho);

    ydot_nuc(Ar38) =
        screened_rates(k_k38__ar38__weak__wc12)*Y(K38) +
        (screened_rates(k_he4_s34__ar38)*Y(He4)*Y(S34)*state.rho + -screened_rates(k_ar38__he4_s34)*Y(Ar38)) +
        (screened_rates(k_p_cl37__ar38)*Y(Cl37)*Y(H1)*state.rho + -screened_rates(k_ar38__p_cl37)*Y(Ar38)) +
        (screened_rates(k_n_ar37__ar38)*Y(Ar37)*Y(N)*state.rho + -screened_rates(k_ar38__n_ar37)*Y(Ar38)) +
        (-screened_rates(k_n_ar38__ar39)*Y(Ar38)*Y(N)*state.rho + screened_rates(k_ar39__n_ar38)*Y(Ar39)) +
        (-screened_rates(k_p_ar38__k39)*Y(Ar38)*Y(H1)*state.rho + screened_rates(k_k39__p_ar38)*Y(K39)) +
        (-screened_rates(k_he4_ar38__ca42)*Y(Ar38)*Y(He4)*state.rho + screened_rates(k_ca42__he4_ar38)*Y(Ca42)) +
        (screened_rates(k_he4_s35__n_ar38)*Y(He4)*Y(S35)*state.rho + -screened_rates(k_n_ar38__he4_s35)*Y(Ar38)*Y(N)*state.rho) +
        (screened_rates(k_he4_cl35__p_ar38)*Y(Cl35)*Y(He4)*state.rho + -screened_rates(k_p_ar38__he4_cl35)*Y(Ar38)*Y(H1)*state.rho) +
        (screened_rates(k_n_k38__p_ar38)*Y(K38)*Y(N)*state.rho + -screened_rates(k_p_ar38__n_k38)*Y(Ar38)*Y(H1)*state.rho) +
        (screened_rates(k_p_k41__he4_ar38)*Y(K41)*Y(H1)*state.rho + -screened_rates(k_he4_ar38__p_k41)*Y(Ar38)*Y(He4)*state.rho) +
        (screened_rates(k_n_ca41__he4_ar38)*Y(Ca41)*Y(N)*state.rho + -screened_rates(k_he4_ar38__n_ca41)*Y(Ar38)*Y(He4)*state.rho);

    ydot_nuc(Ar39) =
        -screened_rates(k_ar39__k39__weak__wc12)*Y(Ar39) +
        (screened_rates(k_he4_s35__ar39)*Y(He4)*Y(S35)*state.rho + -screened_rates(k_ar39__he4_s35)*Y(Ar39)) +
        (screened_rates(k_n_ar38__ar39)*Y(Ar38)*Y(N)*state.rho + -screened_rates(k_ar39__n_ar38)*Y(Ar39)) +
        (-screened_rates(k_n_ar39__ar40)*Y(Ar39)*Y(N)*state.rho + screened_rates(k_ar40__n_ar39)*Y(Ar40)) +
        (-screened_rates(k_p_ar39__k40)*Y(Ar39)*Y(H1)*state.rho + screened_rates(k_k40__p_ar39)*Y(K40)) +
        (-screened_rates(k_he4_ar39__ca43)*Y(Ar39)*Y(He4)*state.rho + screened_rates(k_ca43__he4_ar39)*Y(Ca43)) +
        (-screened_rates(k_n_ar39__he4_s36)*Y(Ar39)*Y(N)*state.rho + screened_rates(k_he4_s36__n_ar39)*Y(He4)*Y(S36)*state.rho) +
        (-screened_rates(k_p_ar39__he4_cl36)*Y(Ar39)*Y(H1)*state.rho + screened_rates(k_he4_cl36__p_ar39)*Y(Cl36)*Y(He4)*state.rho) +
        (screened_rates(k_n_k39__p_ar39)*Y(K39)*Y(N)*state.rho + -screened_rates(k_p_ar39__n_k39)*Y(Ar39)*Y(H1)*state.rho) +
        (screened_rates(k_n_ca42__he4_ar39)*Y(Ca42)*Y(N)*state.rho + -screened_rates(k_he4_ar39__n_ca42)*Y(Ar39)*Y(He4)*state.rho);

    ydot_nuc(Ar40) =
        screened_rates(k_k40__ar40__weak__wc12)*Y(K40) +
        (screened_rates(k_he4_s36__ar40)*Y(He4)*Y(S36)*state.rho + -screened_rates(k_ar40__he4_s36)*Y(Ar40)) +
        (screened_rates(k_n_ar39__ar40)*Y(Ar39)*Y(N)*state.rho + -screened_rates(k_ar40__n_ar39)*Y(Ar40)) +
        (-screened_rates(k_p_ar40__k41)*Y(Ar40)*Y(H1)*state.rho + screened_rates(k_k41__p_ar40)*Y(K41)) +
        (-screened_rates(k_he4_ar40__ca44)*Y(Ar40)*Y(He4)*state.rho + screened_rates(k_ca44__he4_ar40)*Y(Ca44)) +
        (-screened_rates(k_p_ar40__he4_cl37)*Y(Ar40)*Y(H1)*state.rho + screened_rates(k_he4_cl37__p_ar40)*Y(Cl37)*Y(He4)*state.rho) +
        (screened_rates(k_n_k40__p_ar40)*Y(K40)*Y(N)*state.rho + -screened_rates(k_p_ar40__n_k40)*Y(Ar40)*Y(H1)*state.rho) +
        (screened_rates(k_n_ca43__he4_ar40)*Y(Ca43)*Y(N)*state.rho + -screened_rates(k_he4_ar40__n_ca43)*Y(Ar40)*Y(He4)*state.rho);

    ydot_nuc(K37) =
        -screened_rates(k_k37__ar37__weak__wc12)*Y(K37) +
        (screened_rates(k_he4_cl33__k37)*Y(Cl33)*Y(He4)*state.rho + -screened_rates(k_k37__he4_cl33)*Y(K37)) +
        (screened_rates(k_p_ar36__k37)*Y(Ar36)*Y(H1)*state.rho + -screened_rates(k_k37__p_ar36)*Y(K37)) +
        (-screened_rates(k_n_k37__k38)*Y(K37)*Y(N)*state.rho + screened_rates(k_k38__n_k37)*Y(K38)) +
        (-screened_rates(k_n_k37__p_ar37)*Y(K37)*Y(N)*state.rho + screened_rates(k_p_ar37__n_k37)*Y(Ar37)*Y(H1)*state.rho) +
        (-screened_rates(k_n_k37__he4_cl34)*Y(K37)*Y(N)*state.rho + screened_rates(k_he4_cl34__n_k37)*Y(Cl34)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_k37__p_ca40)*Y(He4)*Y(K37)*state.rho + screened_rates(k_p_ca40__he4_k37)*Y(Ca40)*Y(H1)*state.rho);

    ydot_nuc(K38) =
        -screened_rates(k_k38__ar38__weak__wc12)*Y(K38) +
        (screened_rates(k_he4_cl34__k38)*Y(Cl34)*Y(He4)*state.rho + -screened_rates(k_k38__he4_cl34)*Y(K38)) +
        (screened_rates(k_p_ar37__k38)*Y(Ar37)*Y(H1)*state.rho + -screened_rates(k_k38__p_ar37)*Y(K38)) +
        (screened_rates(k_n_k37__k38)*Y(K37)*Y(N)*state.rho + -screened_rates(k_k38__n_k37)*Y(K38)) +
        (-screened_rates(k_n_k38__k39)*Y(K38)*Y(N)*state.rho + screened_rates(k_k39__n_k38)*Y(K39)) +
        (-screened_rates(k_n_k38__p_ar38)*Y(K38)*Y(N)*state.rho + screened_rates(k_p_ar38__n_k38)*Y(Ar38)*Y(H1)*state.rho) +
        (-screened_rates(k_n_k38__he4_cl35)*Y(K38)*Y(N)*state.rho + screened_rates(k_he4_cl35__n_k38)*Y(Cl35)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_k38__p_ca41)*Y(He4)*Y(K38)*state.rho + screened_rates(k_p_ca41__he4_k38)*Y(Ca41)*Y(H1)*state.rho);

    ydot_nuc(K39) =
        screened_rates(k_ar39__k39__weak__wc12)*Y(Ar39) +
        (screened_rates(k_he4_cl35__k39)*Y(Cl35)*Y(He4)*state.rho + -screened_rates(k_k39__he4_cl35)*Y(K39)) +
        (screened_rates(k_p_ar38__k39)*Y(Ar38)*Y(H1)*state.rho + -screened_rates(k_k39__p_ar38)*Y(K39)) +
        (screened_rates(k_n_k38__k39)*Y(K38)*Y(N)*state.rho + -screened_rates(k_k39__n_k38)*Y(K39)) +
        (-screened_rates(k_n_k39__k40)*Y(K39)*Y(N)*state.rho + screened_rates(k_k40__n_k39)*Y(K40)) +
        (-screened_rates(k_p_k39__ca40)*Y(K39)*Y(H1)*state.rho + screened_rates(k_ca40__p_k39)*Y(Ca40)) +
        (-screened_rates(k_he4_k39__sc43)*Y(He4)*Y(K39)*state.rho + screened_rates(k_sc43__he4_k39)*Y(Sc43)) +
        (-screened_rates(k_n_k39__p_ar39)*Y(K39)*Y(N)*state.rho + screened_rates(k_p_ar39__n_k39)*Y(Ar39)*Y(H1)*state.rho) +
        (-screened_rates(k_n_k39__he4_cl36)*Y(K39)*Y(N)*state.rho + screened_rates(k_he4_cl36__n_k39)*Y(Cl36)*Y(He4)*state.rho) +
        (-screened_rates(k_p_k39__he4_ar36)*Y(K39)*Y(H1)*state.rho + screened_rates(k_he4_ar36__p_k39)*Y(Ar36)*Y(He4)*state.rho) +
        (screened_rates(k_p_ca42__he4_k39)*Y(Ca42)*Y(H1)*state.rho + -screened_rates(k_he4_k39__p_ca42)*Y(He4)*Y(K39)*state.rho);

    ydot_nuc(K40) =
        -screened_rates(k_k40__ca40__weak__wc12)*Y(K40) +
        -screened_rates(k_k40__ar40__weak__wc12)*Y(K40) +
        (screened_rates(k_he4_cl36__k40)*Y(Cl36)*Y(He4)*state.rho + -screened_rates(k_k40__he4_cl36)*Y(K40)) +
        (screened_rates(k_p_ar39__k40)*Y(Ar39)*Y(H1)*state.rho + -screened_rates(k_k40__p_ar39)*Y(K40)) +
        (screened_rates(k_n_k39__k40)*Y(K39)*Y(N)*state.rho + -screened_rates(k_k40__n_k39)*Y(K40)) +
        (-screened_rates(k_n_k40__k41)*Y(K40)*Y(N)*state.rho + screened_rates(k_k41__n_k40)*Y(K41)) +
        (-screened_rates(k_p_k40__ca41)*Y(K40)*Y(H1)*state.rho + screened_rates(k_ca41__p_k40)*Y(Ca41)) +
        (-screened_rates(k_he4_k40__sc44)*Y(He4)*Y(K40)*state.rho + screened_rates(k_sc44__he4_k40)*Y(Sc44)) +
        (-screened_rates(k_n_k40__p_ar40)*Y(K40)*Y(N)*state.rho + screened_rates(k_p_ar40__n_k40)*Y(Ar40)*Y(H1)*state.rho) +
        (-screened_rates(k_n_k40__he4_cl37)*Y(K40)*Y(N)*state.rho + screened_rates(k_he4_cl37__n_k40)*Y(Cl37)*Y(He4)*state.rho) +
        (-screened_rates(k_p_k40__n_ca40)*Y(K40)*Y(H1)*state.rho + screened_rates(k_n_ca40__p_k40)*Y(Ca40)*Y(N)*state.rho) +
        (-screened_rates(k_p_k40__he4_ar37)*Y(K40)*Y(H1)*state.rho + screened_rates(k_he4_ar37__p_k40)*Y(Ar37)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_k40__p_ca43)*Y(He4)*Y(K40)*state.rho + screened_rates(k_p_ca43__he4_k40)*Y(Ca43)*Y(H1)*state.rho) +
        (screened_rates(k_n_sc43__he4_k40)*Y(N)*Y(Sc43)*state.rho + -screened_rates(k_he4_k40__n_sc43)*Y(He4)*Y(K40)*state.rho);

    ydot_nuc(K41) =
        screened_rates(k_ca41__k41__weak__wc12)*Y(Ca41) +
        (screened_rates(k_he4_cl37__k41)*Y(Cl37)*Y(He4)*state.rho + -screened_rates(k_k41__he4_cl37)*Y(K41)) +
        (screened_rates(k_p_ar40__k41)*Y(Ar40)*Y(H1)*state.rho + -screened_rates(k_k41__p_ar40)*Y(K41)) +
        (screened_rates(k_n_k40__k41)*Y(K40)*Y(N)*state.rho + -screened_rates(k_k41__n_k40)*Y(K41)) +
        (-screened_rates(k_p_k41__ca42)*Y(K41)*Y(H1)*state.rho + screened_rates(k_ca42__p_k41)*Y(Ca42)) +
        (-screened_rates(k_he4_k41__sc45)*Y(He4)*Y(K41)*state.rho + screened_rates(k_sc45__he4_k41)*Y(Sc45)) +
        (-screened_rates(k_p_k41__he4_ar38)*Y(K41)*Y(H1)*state.rho + screened_rates(k_he4_ar38__p_k41)*Y(Ar38)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_k41__p_ca44)*Y(He4)*Y(K41)*state.rho + screened_rates(k_p_ca44__he4_k41)*Y(Ca44)*Y(H1)*state.rho) +
        (screened_rates(k_n_ca41__p_k41)*Y(Ca41)*Y(N)*state.rho + -screened_rates(k_p_k41__n_ca41)*Y(K41)*Y(H1)*state.rho) +
        (screened_rates(k_n_sc44__he4_k41)*Y(N)*Y(Sc44)*state.rho + -screened_rates(k_he4_k41__n_sc44)*Y(He4)*Y(K41)*state.rho);

    ydot_nuc(Ca40) =
        screened_rates(k_k40__ca40__weak__wc12)*Y(K40) +
        (screened_rates(k_he4_ar36__ca40)*Y(Ar36)*Y(He4)*state.rho + -screened_rates(k_ca40__he4_ar36)*Y(Ca40)) +
        (screened_rates(k_p_k39__ca40)*Y(K39)*Y(H1)*state.rho + -screened_rates(k_ca40__p_k39)*Y(Ca40)) +
        (-screened_rates(k_n_ca40__ca41)*Y(Ca40)*Y(N)*state.rho + screened_rates(k_ca41__n_ca40)*Y(Ca41)) +
        (-screened_rates(k_he4_ca40__ti44)*Y(Ca40)*Y(He4)*state.rho + screened_rates(k_ti44__he4_ca40)*Y(Ti44)) +
        (screened_rates(k_he4_k37__p_ca40)*Y(He4)*Y(K37)*state.rho + -screened_rates(k_p_ca40__he4_k37)*Y(Ca40)*Y(H1)*state.rho) +
        (screened_rates(k_p_k40__n_ca40)*Y(K40)*Y(H1)*state.rho + -screened_rates(k_n_ca40__p_k40)*Y(Ca40)*Y(N)*state.rho) +
        (-screened_rates(k_n_ca40__he4_ar37)*Y(Ca40)*Y(N)*state.rho + screened_rates(k_he4_ar37__n_ca40)*Y(Ar37)*Y(He4)*state.rho) +
        (screened_rates(k_p_sc43__he4_ca40)*Y(H1)*Y(Sc43)*state.rho + -screened_rates(k_he4_ca40__p_sc43)*Y(Ca40)*Y(He4)*state.rho);

    ydot_nuc(Ca41) =
        -screened_rates(k_ca41__k41__weak__wc12)*Y(Ca41) +
        (screened_rates(k_he4_ar37__ca41)*Y(Ar37)*Y(He4)*state.rho + -screened_rates(k_ca41__he4_ar37)*Y(Ca41)) +
        (screened_rates(k_p_k40__ca41)*Y(K40)*Y(H1)*state.rho + -screened_rates(k_ca41__p_k40)*Y(Ca41)) +
        (screened_rates(k_n_ca40__ca41)*Y(Ca40)*Y(N)*state.rho + -screened_rates(k_ca41__n_ca40)*Y(Ca41)) +
        (-screened_rates(k_n_ca41__ca42)*Y(Ca41)*Y(N)*state.rho + screened_rates(k_ca42__n_ca41)*Y(Ca42)) +
        (-screened_rates(k_he4_ca41__ti45)*Y(Ca41)*Y(He4)*state.rho + screened_rates(k_ti45__he4_ca41)*Y(Ti45)) +
        (screened_rates(k_he4_k38__p_ca41)*Y(He4)*Y(K38)*state.rho + -screened_rates(k_p_ca41__he4_k38)*Y(Ca41)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ca41__p_k41)*Y(Ca41)*Y(N)*state.rho + screened_rates(k_p_k41__n_ca41)*Y(K41)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ca41__he4_ar38)*Y(Ca41)*Y(N)*state.rho + screened_rates(k_he4_ar38__n_ca41)*Y(Ar38)*Y(He4)*state.rho) +
        (screened_rates(k_p_sc44__he4_ca41)*Y(H1)*Y(Sc44)*state.rho + -screened_rates(k_he4_ca41__p_sc44)*Y(Ca41)*Y(He4)*state.rho) +
        (screened_rates(k_n_ti44__he4_ca41)*Y(N)*Y(Ti44)*state.rho + -screened_rates(k_he4_ca41__n_ti44)*Y(Ca41)*Y(He4)*state.rho);

    ydot_nuc(Ca42) =
        (screened_rates(k_he4_ar38__ca42)*Y(Ar38)*Y(He4)*state.rho + -screened_rates(k_ca42__he4_ar38)*Y(Ca42)) +
        (screened_rates(k_p_k41__ca42)*Y(K41)*Y(H1)*state.rho + -screened_rates(k_ca42__p_k41)*Y(Ca42)) +
        (screened_rates(k_n_ca41__ca42)*Y(Ca41)*Y(N)*state.rho + -screened_rates(k_ca42__n_ca41)*Y(Ca42)) +
        (-screened_rates(k_n_ca42__ca43)*Y(Ca42)*Y(N)*state.rho + screened_rates(k_ca43__n_ca42)*Y(Ca43)) +
        (-screened_rates(k_p_ca42__sc43)*Y(Ca42)*Y(H1)*state.rho + screened_rates(k_sc43__p_ca42)*Y(Sc43)) +
        (-screened_rates(k_he4_ca42__ti46)*Y(Ca42)*Y(He4)*state.rho + screened_rates(k_ti46__he4_ca42)*Y(Ti46)) +
        (-screened_rates(k_n_ca42__he4_ar39)*Y(Ca42)*Y(N)*state.rho + screened_rates(k_he4_ar39__n_ca42)*Y(Ar39)*Y(He4)*state.rho) +
        (-screened_rates(k_p_ca42__he4_k39)*Y(Ca42)*Y(H1)*state.rho + screened_rates(k_he4_k39__p_ca42)*Y(He4)*Y(K39)*state.rho) +
        (screened_rates(k_p_sc45__he4_ca42)*Y(H1)*Y(Sc45)*state.rho + -screened_rates(k_he4_ca42__p_sc45)*Y(Ca42)*Y(He4)*state.rho) +
        (screened_rates(k_n_ti45__he4_ca42)*Y(N)*Y(Ti45)*state.rho + -screened_rates(k_he4_ca42__n_ti45)*Y(Ca42)*Y(He4)*state.rho);

    ydot_nuc(Ca43) =
        screened_rates(k_sc43__ca43__weak__wc12)*Y(Sc43) +
        (screened_rates(k_he4_ar39__ca43)*Y(Ar39)*Y(He4)*state.rho + -screened_rates(k_ca43__he4_ar39)*Y(Ca43)) +
        (screened_rates(k_n_ca42__ca43)*Y(Ca42)*Y(N)*state.rho + -screened_rates(k_ca43__n_ca42)*Y(Ca43)) +
        (-screened_rates(k_n_ca43__ca44)*Y(Ca43)*Y(N)*state.rho + screened_rates(k_ca44__n_ca43)*Y(Ca44)) +
        (-screened_rates(k_p_ca43__sc44)*Y(Ca43)*Y(H1)*state.rho + screened_rates(k_sc44__p_ca43)*Y(Sc44)) +
        (-screened_rates(k_he4_ca43__ti47)*Y(Ca43)*Y(He4)*state.rho + screened_rates(k_ti47__he4_ca43)*Y(Ti47)) +
        (screened_rates(k_he4_k40__p_ca43)*Y(He4)*Y(K40)*state.rho + -screened_rates(k_p_ca43__he4_k40)*Y(Ca43)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ca43__he4_ar40)*Y(Ca43)*Y(N)*state.rho + screened_rates(k_he4_ar40__n_ca43)*Y(Ar40)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_ca43__n_ti46)*Y(Ca43)*Y(He4)*state.rho + screened_rates(k_n_ti46__he4_ca43)*Y(N)*Y(Ti46)*state.rho) +
        (screened_rates(k_n_sc43__p_ca43)*Y(N)*Y(Sc43)*state.rho + -screened_rates(k_p_ca43__n_sc43)*Y(Ca43)*Y(H1)*state.rho) +
        (screened_rates(k_p_sc46__he4_ca43)*Y(H1)*Y(Sc46)*state.rho + -screened_rates(k_he4_ca43__p_sc46)*Y(Ca43)*Y(He4)*state.rho);

    ydot_nuc(Ca44) =
        screened_rates(k_sc44__ca44__weak__wc12)*Y(Sc44) +
        (screened_rates(k_he4_ar40__ca44)*Y(Ar40)*Y(He4)*state.rho + -screened_rates(k_ca44__he4_ar40)*Y(Ca44)) +
        (screened_rates(k_n_ca43__ca44)*Y(Ca43)*Y(N)*state.rho + -screened_rates(k_ca44__n_ca43)*Y(Ca44)) +
        (-screened_rates(k_n_ca44__ca45)*Y(Ca44)*Y(N)*state.rho + screened_rates(k_ca45__n_ca44)*Y(Ca45)) +
        (-screened_rates(k_p_ca44__sc45)*Y(Ca44)*Y(H1)*state.rho + screened_rates(k_sc45__p_ca44)*Y(Sc45)) +
        (-screened_rates(k_he4_ca44__ti48)*Y(Ca44)*Y(He4)*state.rho + screened_rates(k_ti48__he4_ca44)*Y(Ti48)) +
        (screened_rates(k_he4_k41__p_ca44)*Y(He4)*Y(K41)*state.rho + -screened_rates(k_p_ca44__he4_k41)*Y(Ca44)*Y(H1)*state.rho) +
        (screened_rates(k_n_sc44__p_ca44)*Y(N)*Y(Sc44)*state.rho + -screened_rates(k_p_ca44__n_sc44)*Y(Ca44)*Y(H1)*state.rho) +
        (screened_rates(k_p_sc47__he4_ca44)*Y(H1)*Y(Sc47)*state.rho + -screened_rates(k_he4_ca44__p_sc47)*Y(Ca44)*Y(He4)*state.rho) +
        (screened_rates(k_n_ti47__he4_ca44)*Y(N)*Y(Ti47)*state.rho + -screened_rates(k_he4_ca44__n_ti47)*Y(Ca44)*Y(He4)*state.rho);

    ydot_nuc(Ca45) =
        -screened_rates(k_ca45__sc45__weak__wc12)*Y(Ca45) +
        (screened_rates(k_n_ca44__ca45)*Y(Ca44)*Y(N)*state.rho + -screened_rates(k_ca45__n_ca44)*Y(Ca45)) +
        (-screened_rates(k_n_ca45__ca46)*Y(Ca45)*Y(N)*state.rho + screened_rates(k_ca46__n_ca45)*Y(Ca46)) +
        (-screened_rates(k_p_ca45__sc46)*Y(Ca45)*Y(H1)*state.rho + screened_rates(k_sc46__p_ca45)*Y(Sc46)) +
        (-screened_rates(k_he4_ca45__ti49)*Y(Ca45)*Y(He4)*state.rho + screened_rates(k_ti49__he4_ca45)*Y(Ti49)) +
        (-screened_rates(k_he4_ca45__n_ti48)*Y(Ca45)*Y(He4)*state.rho + screened_rates(k_n_ti48__he4_ca45)*Y(N)*Y(Ti48)*state.rho) +
        (screened_rates(k_n_sc45__p_ca45)*Y(N)*Y(Sc45)*state.rho + -screened_rates(k_p_ca45__n_sc45)*Y(Ca45)*Y(H1)*state.rho) +
        (screened_rates(k_p_sc48__he4_ca45)*Y(H1)*Y(Sc48)*state.rho + -screened_rates(k_he4_ca45__p_sc48)*Y(Ca45)*Y(He4)*state.rho);

    ydot_nuc(Ca46) =
        (screened_rates(k_n_ca45__ca46)*Y(Ca45)*Y(N)*state.rho + -screened_rates(k_ca46__n_ca45)*Y(Ca46)) +
        (-screened_rates(k_n_ca46__ca47)*Y(Ca46)*Y(N)*state.rho + screened_rates(k_ca47__n_ca46)*Y(Ca47)) +
        (-screened_rates(k_p_ca46__sc47)*Y(Ca46)*Y(H1)*state.rho + screened_rates(k_sc47__p_ca46)*Y(Sc47)) +
        (-screened_rates(k_he4_ca46__ti50)*Y(Ca46)*Y(He4)*state.rho + screened_rates(k_ti50__he4_ca46)*Y(Ti50)) +
        (screened_rates(k_n_sc46__p_ca46)*Y(N)*Y(Sc46)*state.rho + -screened_rates(k_p_ca46__n_sc46)*Y(Ca46)*Y(H1)*state.rho) +
        (screened_rates(k_p_sc49__he4_ca46)*Y(H1)*Y(Sc49)*state.rho + -screened_rates(k_he4_ca46__p_sc49)*Y(Ca46)*Y(He4)*state.rho) +
        (screened_rates(k_n_ti49__he4_ca46)*Y(N)*Y(Ti49)*state.rho + -screened_rates(k_he4_ca46__n_ti49)*Y(Ca46)*Y(He4)*state.rho);

    ydot_nuc(Ca47) =
        -screened_rates(k_ca47__sc47__weak__wc12)*Y(Ca47) +
        (screened_rates(k_n_ca46__ca47)*Y(Ca46)*Y(N)*state.rho + -screened_rates(k_ca47__n_ca46)*Y(Ca47)) +
        (-screened_rates(k_n_ca47__ca48)*Y(Ca47)*Y(N)*state.rho + screened_rates(k_ca48__n_ca47)*Y(Ca48)) +
        (-screened_rates(k_p_ca47__sc48)*Y(Ca47)*Y(H1)*state.rho + screened_rates(k_sc48__p_ca47)*Y(Sc48)) +
        (-screened_rates(k_he4_ca47__ti51)*Y(Ca47)*Y(He4)*state.rho + screened_rates(k_ti51__he4_ca47)*Y(Ti51)) +
        (-screened_rates(k_p_ca47__n_sc47)*Y(Ca47)*Y(H1)*state.rho + screened_rates(k_n_sc47__p_ca47)*Y(N)*Y(Sc47)*state.rho) +
        (-screened_rates(k_he4_ca47__n_ti50)*Y(Ca47)*Y(He4)*state.rho + screened_rates(k_n_ti50__he4_ca47)*Y(N)*Y(Ti50)*state.rho);

    ydot_nuc(Ca48) =
        -screened_rates(k_ca48__sc48__weak__mo03)*Y(Ca48) +
        (screened_rates(k_n_ca47__ca48)*Y(Ca47)*Y(N)*state.rho + -screened_rates(k_ca48__n_ca47)*Y(Ca48)) +
        (-screened_rates(k_p_ca48__sc49)*Y(Ca48)*Y(H1)*state.rho + screened_rates(k_sc49__p_ca48)*Y(Sc49)) +
        (screened_rates(k_n_sc48__p_ca48)*Y(N)*Y(Sc48)*state.rho + -screened_rates(k_p_ca48__n_sc48)*Y(Ca48)*Y(H1)*state.rho) +
        (screened_rates(k_n_ti51__he4_ca48)*Y(N)*Y(Ti51)*state.rho + -screened_rates(k_he4_ca48__n_ti51)*Y(Ca48)*Y(He4)*state.rho);

    ydot_nuc(Sc43) =
        -screened_rates(k_sc43__ca43__weak__wc12)*Y(Sc43) +
        (screened_rates(k_he4_k39__sc43)*Y(He4)*Y(K39)*state.rho + -screened_rates(k_sc43__he4_k39)*Y(Sc43)) +
        (screened_rates(k_p_ca42__sc43)*Y(Ca42)*Y(H1)*state.rho + -screened_rates(k_sc43__p_ca42)*Y(Sc43)) +
        (-screened_rates(k_n_sc43__sc44)*Y(N)*Y(Sc43)*state.rho + screened_rates(k_sc44__n_sc43)*Y(Sc44)) +
        (-screened_rates(k_p_sc43__ti44)*Y(H1)*Y(Sc43)*state.rho + screened_rates(k_ti44__p_sc43)*Y(Ti44)) +
        (-screened_rates(k_he4_sc43__v47)*Y(He4)*Y(Sc43)*state.rho + screened_rates(k_v47__he4_sc43)*Y(V47)) +
        (-screened_rates(k_n_sc43__p_ca43)*Y(N)*Y(Sc43)*state.rho + screened_rates(k_p_ca43__n_sc43)*Y(Ca43)*Y(H1)*state.rho) +
        (-screened_rates(k_n_sc43__he4_k40)*Y(N)*Y(Sc43)*state.rho + screened_rates(k_he4_k40__n_sc43)*Y(He4)*Y(K40)*state.rho) +
        (-screened_rates(k_p_sc43__he4_ca40)*Y(H1)*Y(Sc43)*state.rho + screened_rates(k_he4_ca40__p_sc43)*Y(Ca40)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_sc43__p_ti46)*Y(He4)*Y(Sc43)*state.rho + screened_rates(k_p_ti46__he4_sc43)*Y(H1)*Y(Ti46)*state.rho) +
        (screened_rates(k_n_v46__he4_sc43)*Y(N)*Y(V46)*state.rho + -screened_rates(k_he4_sc43__n_v46)*Y(He4)*Y(Sc43)*state.rho);

    ydot_nuc(Sc44) =
        -screened_rates(k_sc44__ca44__weak__wc12)*Y(Sc44) +
        screened_rates(k_ti44__sc44__weak__wc12)*Y(Ti44) +
        (screened_rates(k_he4_k40__sc44)*Y(He4)*Y(K40)*state.rho + -screened_rates(k_sc44__he4_k40)*Y(Sc44)) +
        (screened_rates(k_p_ca43__sc44)*Y(Ca43)*Y(H1)*state.rho + -screened_rates(k_sc44__p_ca43)*Y(Sc44)) +
        (screened_rates(k_n_sc43__sc44)*Y(N)*Y(Sc43)*state.rho + -screened_rates(k_sc44__n_sc43)*Y(Sc44)) +
        (-screened_rates(k_n_sc44__sc45)*Y(N)*Y(Sc44)*state.rho + screened_rates(k_sc45__n_sc44)*Y(Sc45)) +
        (-screened_rates(k_p_sc44__ti45)*Y(H1)*Y(Sc44)*state.rho + screened_rates(k_ti45__p_sc44)*Y(Ti45)) +
        (-screened_rates(k_he4_sc44__v48)*Y(He4)*Y(Sc44)*state.rho + screened_rates(k_v48__he4_sc44)*Y(V48)) +
        (-screened_rates(k_n_sc44__p_ca44)*Y(N)*Y(Sc44)*state.rho + screened_rates(k_p_ca44__n_sc44)*Y(Ca44)*Y(H1)*state.rho) +
        (-screened_rates(k_n_sc44__he4_k41)*Y(N)*Y(Sc44)*state.rho + screened_rates(k_he4_k41__n_sc44)*Y(He4)*Y(K41)*state.rho) +
        (-screened_rates(k_p_sc44__he4_ca41)*Y(H1)*Y(Sc44)*state.rho + screened_rates(k_he4_ca41__p_sc44)*Y(Ca41)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_sc44__p_ti47)*Y(He4)*Y(Sc44)*state.rho + screened_rates(k_p_ti47__he4_sc44)*Y(H1)*Y(Ti47)*state.rho) +
        (screened_rates(k_n_ti44__p_sc44)*Y(N)*Y(Ti44)*state.rho + -screened_rates(k_p_sc44__n_ti44)*Y(H1)*Y(Sc44)*state.rho) +
        (screened_rates(k_n_v47__he4_sc44)*Y(N)*Y(V47)*state.rho + -screened_rates(k_he4_sc44__n_v47)*Y(He4)*Y(Sc44)*state.rho);

    ydot_nuc(Sc45) =
        screened_rates(k_ca45__sc45__weak__wc12)*Y(Ca45) +
        screened_rates(k_ti45__sc45__weak__wc12)*Y(Ti45) +
        (screened_rates(k_he4_k41__sc45)*Y(He4)*Y(K41)*state.rho + -screened_rates(k_sc45__he4_k41)*Y(Sc45)) +
        (screened_rates(k_p_ca44__sc45)*Y(Ca44)*Y(H1)*state.rho + -screened_rates(k_sc45__p_ca44)*Y(Sc45)) +
        (screened_rates(k_n_sc44__sc45)*Y(N)*Y(Sc44)*state.rho + -screened_rates(k_sc45__n_sc44)*Y(Sc45)) +
        (-screened_rates(k_n_sc45__sc46)*Y(N)*Y(Sc45)*state.rho + screened_rates(k_sc46__n_sc45)*Y(Sc46)) +
        (-screened_rates(k_p_sc45__ti46)*Y(H1)*Y(Sc45)*state.rho + screened_rates(k_ti46__p_sc45)*Y(Ti46)) +
        (-screened_rates(k_he4_sc45__v49)*Y(He4)*Y(Sc45)*state.rho + screened_rates(k_v49__he4_sc45)*Y(V49)) +
        (-screened_rates(k_n_sc45__p_ca45)*Y(N)*Y(Sc45)*state.rho + screened_rates(k_p_ca45__n_sc45)*Y(Ca45)*Y(H1)*state.rho) +
        (-screened_rates(k_p_sc45__he4_ca42)*Y(H1)*Y(Sc45)*state.rho + screened_rates(k_he4_ca42__p_sc45)*Y(Ca42)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_sc45__p_ti48)*Y(He4)*Y(Sc45)*state.rho + screened_rates(k_p_ti48__he4_sc45)*Y(H1)*Y(Ti48)*state.rho) +
        (screened_rates(k_n_ti45__p_sc45)*Y(N)*Y(Ti45)*state.rho + -screened_rates(k_p_sc45__n_ti45)*Y(H1)*Y(Sc45)*state.rho) +
        (screened_rates(k_n_v48__he4_sc45)*Y(N)*Y(V48)*state.rho + -screened_rates(k_he4_sc45__n_v48)*Y(He4)*Y(Sc45)*state.rho);

    ydot_nuc(Sc46) =
        -screened_rates(k_sc46__ti46__weak__wc12)*Y(Sc46) +
        (screened_rates(k_p_ca45__sc46)*Y(Ca45)*Y(H1)*state.rho + -screened_rates(k_sc46__p_ca45)*Y(Sc46)) +
        (screened_rates(k_n_sc45__sc46)*Y(N)*Y(Sc45)*state.rho + -screened_rates(k_sc46__n_sc45)*Y(Sc46)) +
        (-screened_rates(k_n_sc46__sc47)*Y(N)*Y(Sc46)*state.rho + screened_rates(k_sc47__n_sc46)*Y(Sc47)) +
        (-screened_rates(k_p_sc46__ti47)*Y(H1)*Y(Sc46)*state.rho + screened_rates(k_ti47__p_sc46)*Y(Ti47)) +
        (-screened_rates(k_he4_sc46__v50)*Y(He4)*Y(Sc46)*state.rho + screened_rates(k_v50__he4_sc46)*Y(V50)) +
        (-screened_rates(k_n_sc46__p_ca46)*Y(N)*Y(Sc46)*state.rho + screened_rates(k_p_ca46__n_sc46)*Y(Ca46)*Y(H1)*state.rho) +
        (-screened_rates(k_p_sc46__n_ti46)*Y(H1)*Y(Sc46)*state.rho + screened_rates(k_n_ti46__p_sc46)*Y(N)*Y(Ti46)*state.rho) +
        (-screened_rates(k_p_sc46__he4_ca43)*Y(H1)*Y(Sc46)*state.rho + screened_rates(k_he4_ca43__p_sc46)*Y(Ca43)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_sc46__n_v49)*Y(He4)*Y(Sc46)*state.rho + screened_rates(k_n_v49__he4_sc46)*Y(N)*Y(V49)*state.rho) +
        (-screened_rates(k_he4_sc46__p_ti49)*Y(He4)*Y(Sc46)*state.rho + screened_rates(k_p_ti49__he4_sc46)*Y(H1)*Y(Ti49)*state.rho);

    ydot_nuc(Sc47) =
        screened_rates(k_ca47__sc47__weak__wc12)*Y(Ca47) +
        -screened_rates(k_sc47__ti47__weak__wc12)*Y(Sc47) +
        (screened_rates(k_p_ca46__sc47)*Y(Ca46)*Y(H1)*state.rho + -screened_rates(k_sc47__p_ca46)*Y(Sc47)) +
        (screened_rates(k_n_sc46__sc47)*Y(N)*Y(Sc46)*state.rho + -screened_rates(k_sc47__n_sc46)*Y(Sc47)) +
        (-screened_rates(k_n_sc47__sc48)*Y(N)*Y(Sc47)*state.rho + screened_rates(k_sc48__n_sc47)*Y(Sc48)) +
        (-screened_rates(k_p_sc47__ti48)*Y(H1)*Y(Sc47)*state.rho + screened_rates(k_ti48__p_sc47)*Y(Ti48)) +
        (-screened_rates(k_he4_sc47__v51)*Y(He4)*Y(Sc47)*state.rho + screened_rates(k_v51__he4_sc47)*Y(V51)) +
        (screened_rates(k_p_ca47__n_sc47)*Y(Ca47)*Y(H1)*state.rho + -screened_rates(k_n_sc47__p_ca47)*Y(N)*Y(Sc47)*state.rho) +
        (-screened_rates(k_p_sc47__he4_ca44)*Y(H1)*Y(Sc47)*state.rho + screened_rates(k_he4_ca44__p_sc47)*Y(Ca44)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_sc47__p_ti50)*Y(He4)*Y(Sc47)*state.rho + screened_rates(k_p_ti50__he4_sc47)*Y(H1)*Y(Ti50)*state.rho) +
        (screened_rates(k_n_ti47__p_sc47)*Y(N)*Y(Ti47)*state.rho + -screened_rates(k_p_sc47__n_ti47)*Y(H1)*Y(Sc47)*state.rho) +
        (screened_rates(k_n_v50__he4_sc47)*Y(N)*Y(V50)*state.rho + -screened_rates(k_he4_sc47__n_v50)*Y(He4)*Y(Sc47)*state.rho);

    ydot_nuc(Sc48) =
        screened_rates(k_ca48__sc48__weak__mo03)*Y(Ca48) +
        -screened_rates(k_sc48__ti48__weak__wc12)*Y(Sc48) +
        (screened_rates(k_p_ca47__sc48)*Y(Ca47)*Y(H1)*state.rho + -screened_rates(k_sc48__p_ca47)*Y(Sc48)) +
        (screened_rates(k_n_sc47__sc48)*Y(N)*Y(Sc47)*state.rho + -screened_rates(k_sc48__n_sc47)*Y(Sc48)) +
        (-screened_rates(k_n_sc48__sc49)*Y(N)*Y(Sc48)*state.rho + screened_rates(k_sc49__n_sc48)*Y(Sc49)) +
        (-screened_rates(k_p_sc48__ti49)*Y(H1)*Y(Sc48)*state.rho + screened_rates(k_ti49__p_sc48)*Y(Ti49)) +
        (-screened_rates(k_he4_sc48__v52)*Y(He4)*Y(Sc48)*state.rho + screened_rates(k_v52__he4_sc48)*Y(V52)) +
        (-screened_rates(k_n_sc48__p_ca48)*Y(N)*Y(Sc48)*state.rho + screened_rates(k_p_ca48__n_sc48)*Y(Ca48)*Y(H1)*state.rho) +
        (-screened_rates(k_p_sc48__n_ti48)*Y(H1)*Y(Sc48)*state.rho + screened_rates(k_n_ti48__p_sc48)*Y(N)*Y(Ti48)*state.rho) +
        (-screened_rates(k_p_sc48__he4_ca45)*Y(H1)*Y(Sc48)*state.rho + screened_rates(k_he4_ca45__p_sc48)*Y(Ca45)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_sc48__n_v51)*Y(He4)*Y(Sc48)*state.rho + screened_rates(k_n_v51__he4_sc48)*Y(N)*Y(V51)*state.rho) +
        (-screened_rates(k_he4_sc48__p_ti51)*Y(He4)*Y(Sc48)*state.rho + screened_rates(k_p_ti51__he4_sc48)*Y(H1)*Y(Ti51)*state.rho);

    ydot_nuc(Sc49) =
        -screened_rates(k_sc49__ti49__weak__wc12)*Y(Sc49) +
        (screened_rates(k_p_ca48__sc49)*Y(Ca48)*Y(H1)*state.rho + -screened_rates(k_sc49__p_ca48)*Y(Sc49)) +
        (screened_rates(k_n_sc48__sc49)*Y(N)*Y(Sc48)*state.rho + -screened_rates(k_sc49__n_sc48)*Y(Sc49)) +
        (-screened_rates(k_p_sc49__ti50)*Y(H1)*Y(Sc49)*state.rho + screened_rates(k_ti50__p_sc49)*Y(Ti50)) +
        (-screened_rates(k_p_sc49__n_ti49)*Y(H1)*Y(Sc49)*state.rho + screened_rates(k_n_ti49__p_sc49)*Y(N)*Y(Ti49)*state.rho) +
        (-screened_rates(k_p_sc49__he4_ca46)*Y(H1)*Y(Sc49)*state.rho + screened_rates(k_he4_ca46__p_sc49)*Y(Ca46)*Y(He4)*state.rho) +
        (screened_rates(k_n_v52__he4_sc49)*Y(N)*Y(V52)*state.rho + -screened_rates(k_he4_sc49__n_v52)*Y(He4)*Y(Sc49)*state.rho);

    ydot_nuc(Ti44) =
        -screened_rates(k_ti44__sc44__weak__wc12)*Y(Ti44) +
        (screened_rates(k_he4_ca40__ti44)*Y(Ca40)*Y(He4)*state.rho + -screened_rates(k_ti44__he4_ca40)*Y(Ti44)) +
        (screened_rates(k_p_sc43__ti44)*Y(H1)*Y(Sc43)*state.rho + -screened_rates(k_ti44__p_sc43)*Y(Ti44)) +
        (-screened_rates(k_n_ti44__ti45)*Y(N)*Y(Ti44)*state.rho + screened_rates(k_ti45__n_ti44)*Y(Ti45)) +
        (-screened_rates(k_he4_ti44__cr48)*Y(He4)*Y(Ti44)*state.rho + screened_rates(k_cr48__he4_ti44)*Y(Cr48)) +
        (-screened_rates(k_n_ti44__p_sc44)*Y(N)*Y(Ti44)*state.rho + screened_rates(k_p_sc44__n_ti44)*Y(H1)*Y(Sc44)*state.rho) +
        (-screened_rates(k_n_ti44__he4_ca41)*Y(N)*Y(Ti44)*state.rho + screened_rates(k_he4_ca41__n_ti44)*Y(Ca41)*Y(He4)*state.rho) +
        (screened_rates(k_p_v47__he4_ti44)*Y(H1)*Y(V47)*state.rho + -screened_rates(k_he4_ti44__p_v47)*Y(He4)*Y(Ti44)*state.rho);

    ydot_nuc(Ti45) =
        -screened_rates(k_ti45__sc45__weak__wc12)*Y(Ti45) +
        (screened_rates(k_he4_ca41__ti45)*Y(Ca41)*Y(He4)*state.rho + -screened_rates(k_ti45__he4_ca41)*Y(Ti45)) +
        (screened_rates(k_p_sc44__ti45)*Y(H1)*Y(Sc44)*state.rho + -screened_rates(k_ti45__p_sc44)*Y(Ti45)) +
        (screened_rates(k_n_ti44__ti45)*Y(N)*Y(Ti44)*state.rho + -screened_rates(k_ti45__n_ti44)*Y(Ti45)) +
        (-screened_rates(k_n_ti45__ti46)*Y(N)*Y(Ti45)*state.rho + screened_rates(k_ti46__n_ti45)*Y(Ti46)) +
        (-screened_rates(k_p_ti45__v46)*Y(H1)*Y(Ti45)*state.rho + screened_rates(k_v46__p_ti45)*Y(V46)) +
        (-screened_rates(k_he4_ti45__cr49)*Y(He4)*Y(Ti45)*state.rho + screened_rates(k_cr49__he4_ti45)*Y(Cr49)) +
        (-screened_rates(k_n_ti45__p_sc45)*Y(N)*Y(Ti45)*state.rho + screened_rates(k_p_sc45__n_ti45)*Y(H1)*Y(Sc45)*state.rho) +
        (-screened_rates(k_n_ti45__he4_ca42)*Y(N)*Y(Ti45)*state.rho + screened_rates(k_he4_ca42__n_ti45)*Y(Ca42)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_ti45__p_v48)*Y(He4)*Y(Ti45)*state.rho + screened_rates(k_p_v48__he4_ti45)*Y(H1)*Y(V48)*state.rho) +
        (screened_rates(k_n_cr48__he4_ti45)*Y(Cr48)*Y(N)*state.rho + -screened_rates(k_he4_ti45__n_cr48)*Y(He4)*Y(Ti45)*state.rho);

    ydot_nuc(Ti46) =
        screened_rates(k_sc46__ti46__weak__wc12)*Y(Sc46) +
        screened_rates(k_v46__ti46__weak__wc12)*Y(V46) +
        (screened_rates(k_he4_ca42__ti46)*Y(Ca42)*Y(He4)*state.rho + -screened_rates(k_ti46__he4_ca42)*Y(Ti46)) +
        (screened_rates(k_p_sc45__ti46)*Y(H1)*Y(Sc45)*state.rho + -screened_rates(k_ti46__p_sc45)*Y(Ti46)) +
        (screened_rates(k_n_ti45__ti46)*Y(N)*Y(Ti45)*state.rho + -screened_rates(k_ti46__n_ti45)*Y(Ti46)) +
        (-screened_rates(k_n_ti46__ti47)*Y(N)*Y(Ti46)*state.rho + screened_rates(k_ti47__n_ti46)*Y(Ti47)) +
        (-screened_rates(k_p_ti46__v47)*Y(H1)*Y(Ti46)*state.rho + screened_rates(k_v47__p_ti46)*Y(V47)) +
        (-screened_rates(k_he4_ti46__cr50)*Y(He4)*Y(Ti46)*state.rho + screened_rates(k_cr50__he4_ti46)*Y(Cr50)) +
        (screened_rates(k_he4_ca43__n_ti46)*Y(Ca43)*Y(He4)*state.rho + -screened_rates(k_n_ti46__he4_ca43)*Y(N)*Y(Ti46)*state.rho) +
        (screened_rates(k_he4_sc43__p_ti46)*Y(He4)*Y(Sc43)*state.rho + -screened_rates(k_p_ti46__he4_sc43)*Y(H1)*Y(Ti46)*state.rho) +
        (screened_rates(k_p_sc46__n_ti46)*Y(H1)*Y(Sc46)*state.rho + -screened_rates(k_n_ti46__p_sc46)*Y(N)*Y(Ti46)*state.rho) +
        (screened_rates(k_n_v46__p_ti46)*Y(N)*Y(V46)*state.rho + -screened_rates(k_p_ti46__n_v46)*Y(H1)*Y(Ti46)*state.rho) +
        (screened_rates(k_p_v49__he4_ti46)*Y(H1)*Y(V49)*state.rho + -screened_rates(k_he4_ti46__p_v49)*Y(He4)*Y(Ti46)*state.rho) +
        (screened_rates(k_n_cr49__he4_ti46)*Y(Cr49)*Y(N)*state.rho + -screened_rates(k_he4_ti46__n_cr49)*Y(He4)*Y(Ti46)*state.rho);

    ydot_nuc(Ti47) =
        screened_rates(k_sc47__ti47__weak__wc12)*Y(Sc47) +
        screened_rates(k_v47__ti47__weak__wc12)*Y(V47) +
        (screened_rates(k_he4_ca43__ti47)*Y(Ca43)*Y(He4)*state.rho + -screened_rates(k_ti47__he4_ca43)*Y(Ti47)) +
        (screened_rates(k_p_sc46__ti47)*Y(H1)*Y(Sc46)*state.rho + -screened_rates(k_ti47__p_sc46)*Y(Ti47)) +
        (screened_rates(k_n_ti46__ti47)*Y(N)*Y(Ti46)*state.rho + -screened_rates(k_ti47__n_ti46)*Y(Ti47)) +
        (-screened_rates(k_n_ti47__ti48)*Y(N)*Y(Ti47)*state.rho + screened_rates(k_ti48__n_ti47)*Y(Ti48)) +
        (-screened_rates(k_p_ti47__v48)*Y(H1)*Y(Ti47)*state.rho + screened_rates(k_v48__p_ti47)*Y(V48)) +
        (-screened_rates(k_he4_ti47__cr51)*Y(He4)*Y(Ti47)*state.rho + screened_rates(k_cr51__he4_ti47)*Y(Cr51)) +
        (screened_rates(k_he4_sc44__p_ti47)*Y(He4)*Y(Sc44)*state.rho + -screened_rates(k_p_ti47__he4_sc44)*Y(H1)*Y(Ti47)*state.rho) +
        (-screened_rates(k_n_ti47__p_sc47)*Y(N)*Y(Ti47)*state.rho + screened_rates(k_p_sc47__n_ti47)*Y(H1)*Y(Sc47)*state.rho) +
        (-screened_rates(k_n_ti47__he4_ca44)*Y(N)*Y(Ti47)*state.rho + screened_rates(k_he4_ca44__n_ti47)*Y(Ca44)*Y(He4)*state.rho) +
        (screened_rates(k_n_v47__p_ti47)*Y(N)*Y(V47)*state.rho + -screened_rates(k_p_ti47__n_v47)*Y(H1)*Y(Ti47)*state.rho) +
        (screened_rates(k_p_v50__he4_ti47)*Y(H1)*Y(V50)*state.rho + -screened_rates(k_he4_ti47__p_v50)*Y(He4)*Y(Ti47)*state.rho) +
        (screened_rates(k_n_cr50__he4_ti47)*Y(Cr50)*Y(N)*state.rho + -screened_rates(k_he4_ti47__n_cr50)*Y(He4)*Y(Ti47)*state.rho);

    ydot_nuc(Ti48) =
        screened_rates(k_sc48__ti48__weak__wc12)*Y(Sc48) +
        screened_rates(k_v48__ti48__weak__wc12)*Y(V48) +
        (screened_rates(k_he4_ca44__ti48)*Y(Ca44)*Y(He4)*state.rho + -screened_rates(k_ti48__he4_ca44)*Y(Ti48)) +
        (screened_rates(k_p_sc47__ti48)*Y(H1)*Y(Sc47)*state.rho + -screened_rates(k_ti48__p_sc47)*Y(Ti48)) +
        (screened_rates(k_n_ti47__ti48)*Y(N)*Y(Ti47)*state.rho + -screened_rates(k_ti48__n_ti47)*Y(Ti48)) +
        (-screened_rates(k_n_ti48__ti49)*Y(N)*Y(Ti48)*state.rho + screened_rates(k_ti49__n_ti48)*Y(Ti49)) +
        (-screened_rates(k_p_ti48__v49)*Y(H1)*Y(Ti48)*state.rho + screened_rates(k_v49__p_ti48)*Y(V49)) +
        (-screened_rates(k_he4_ti48__cr52)*Y(He4)*Y(Ti48)*state.rho + screened_rates(k_cr52__he4_ti48)*Y(Cr52)) +
        (screened_rates(k_he4_ca45__n_ti48)*Y(Ca45)*Y(He4)*state.rho + -screened_rates(k_n_ti48__he4_ca45)*Y(N)*Y(Ti48)*state.rho) +
        (screened_rates(k_he4_sc45__p_ti48)*Y(He4)*Y(Sc45)*state.rho + -screened_rates(k_p_ti48__he4_sc45)*Y(H1)*Y(Ti48)*state.rho) +
        (screened_rates(k_p_sc48__n_ti48)*Y(H1)*Y(Sc48)*state.rho + -screened_rates(k_n_ti48__p_sc48)*Y(N)*Y(Ti48)*state.rho) +
        (screened_rates(k_n_v48__p_ti48)*Y(N)*Y(V48)*state.rho + -screened_rates(k_p_ti48__n_v48)*Y(H1)*Y(Ti48)*state.rho) +
        (screened_rates(k_p_v51__he4_ti48)*Y(H1)*Y(V51)*state.rho + -screened_rates(k_he4_ti48__p_v51)*Y(He4)*Y(Ti48)*state.rho) +
        (screened_rates(k_n_cr51__he4_ti48)*Y(Cr51)*Y(N)*state.rho + -screened_rates(k_he4_ti48__n_cr51)*Y(He4)*Y(Ti48)*state.rho);

    ydot_nuc(Ti49) =
        screened_rates(k_sc49__ti49__weak__wc12)*Y(Sc49) +
        screened_rates(k_v49__ti49__weak__wc12)*Y(V49) +
        (screened_rates(k_he4_ca45__ti49)*Y(Ca45)*Y(He4)*state.rho + -screened_rates(k_ti49__he4_ca45)*Y(Ti49)) +
        (screened_rates(k_p_sc48__ti49)*Y(H1)*Y(Sc48)*state.rho + -screened_rates(k_ti49__p_sc48)*Y(Ti49)) +
        (screened_rates(k_n_ti48__ti49)*Y(N)*Y(Ti48)*state.rho + -screened_rates(k_ti49__n_ti48)*Y(Ti49)) +
        (-screened_rates(k_n_ti49__ti50)*Y(N)*Y(Ti49)*state.rho + screened_rates(k_ti50__n_ti49)*Y(Ti50)) +
        (-screened_rates(k_p_ti49__v50)*Y(H1)*Y(Ti49)*state.rho + screened_rates(k_v50__p_ti49)*Y(V50)) +
        (-screened_rates(k_he4_ti49__cr53)*Y(He4)*Y(Ti49)*state.rho + screened_rates(k_cr53__he4_ti49)*Y(Cr53)) +
        (screened_rates(k_he4_sc46__p_ti49)*Y(He4)*Y(Sc46)*state.rho + -screened_rates(k_p_ti49__he4_sc46)*Y(H1)*Y(Ti49)*state.rho) +
        (screened_rates(k_p_sc49__n_ti49)*Y(H1)*Y(Sc49)*state.rho + -screened_rates(k_n_ti49__p_sc49)*Y(N)*Y(Ti49)*state.rho) +
        (-screened_rates(k_n_ti49__he4_ca46)*Y(N)*Y(Ti49)*state.rho + screened_rates(k_he4_ca46__n_ti49)*Y(Ca46)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_ti49__n_cr52)*Y(He4)*Y(Ti49)*state.rho + screened_rates(k_n_cr52__he4_ti49)*Y(Cr52)*Y(N)*state.rho) +
        (screened_rates(k_n_v49__p_ti49)*Y(N)*Y(V49)*state.rho + -screened_rates(k_p_ti49__n_v49)*Y(H1)*Y(Ti49)*state.rho) +
        (screened_rates(k_p_v52__he4_ti49)*Y(H1)*Y(V52)*state.rho + -screened_rates(k_he4_ti49__p_v52)*Y(He4)*Y(Ti49)*state.rho);

    ydot_nuc(Ti50) =
        screened_rates(k_v50__ti50__weak__mo03)*Y(V50) +
        (screened_rates(k_he4_ca46__ti50)*Y(Ca46)*Y(He4)*state.rho + -screened_rates(k_ti50__he4_ca46)*Y(Ti50)) +
        (screened_rates(k_p_sc49__ti50)*Y(H1)*Y(Sc49)*state.rho + -screened_rates(k_ti50__p_sc49)*Y(Ti50)) +
        (screened_rates(k_n_ti49__ti50)*Y(N)*Y(Ti49)*state.rho + -screened_rates(k_ti50__n_ti49)*Y(Ti50)) +
        (-screened_rates(k_n_ti50__ti51)*Y(N)*Y(Ti50)*state.rho + screened_rates(k_ti51__n_ti50)*Y(Ti51)) +
        (-screened_rates(k_p_ti50__v51)*Y(H1)*Y(Ti50)*state.rho + screened_rates(k_v51__p_ti50)*Y(V51)) +
        (-screened_rates(k_he4_ti50__cr54)*Y(He4)*Y(Ti50)*state.rho + screened_rates(k_cr54__he4_ti50)*Y(Cr54)) +
        (screened_rates(k_he4_ca47__n_ti50)*Y(Ca47)*Y(He4)*state.rho + -screened_rates(k_n_ti50__he4_ca47)*Y(N)*Y(Ti50)*state.rho) +
        (screened_rates(k_he4_sc47__p_ti50)*Y(He4)*Y(Sc47)*state.rho + -screened_rates(k_p_ti50__he4_sc47)*Y(H1)*Y(Ti50)*state.rho) +
        (screened_rates(k_n_v50__p_ti50)*Y(N)*Y(V50)*state.rho + -screened_rates(k_p_ti50__n_v50)*Y(H1)*Y(Ti50)*state.rho) +
        (screened_rates(k_n_cr53__he4_ti50)*Y(Cr53)*Y(N)*state.rho + -screened_rates(k_he4_ti50__n_cr53)*Y(He4)*Y(Ti50)*state.rho);

    ydot_nuc(Ti51) =
        -screened_rates(k_ti51__v51__weak__wc12)*Y(Ti51) +
        (screened_rates(k_he4_ca47__ti51)*Y(Ca47)*Y(He4)*state.rho + -screened_rates(k_ti51__he4_ca47)*Y(Ti51)) +
        (screened_rates(k_n_ti50__ti51)*Y(N)*Y(Ti50)*state.rho + -screened_rates(k_ti51__n_ti50)*Y(Ti51)) +
        (-screened_rates(k_p_ti51__v52)*Y(H1)*Y(Ti51)*state.rho + screened_rates(k_v52__p_ti51)*Y(V52)) +
        (screened_rates(k_he4_sc48__p_ti51)*Y(He4)*Y(Sc48)*state.rho + -screened_rates(k_p_ti51__he4_sc48)*Y(H1)*Y(Ti51)*state.rho) +
        (-screened_rates(k_n_ti51__he4_ca48)*Y(N)*Y(Ti51)*state.rho + screened_rates(k_he4_ca48__n_ti51)*Y(Ca48)*Y(He4)*state.rho) +
        (-screened_rates(k_p_ti51__n_v51)*Y(H1)*Y(Ti51)*state.rho + screened_rates(k_n_v51__p_ti51)*Y(N)*Y(V51)*state.rho) +
        (-screened_rates(k_he4_ti51__n_cr54)*Y(He4)*Y(Ti51)*state.rho + screened_rates(k_n_cr54__he4_ti51)*Y(Cr54)*Y(N)*state.rho);

    ydot_nuc(V46) =
        -screened_rates(k_v46__ti46__weak__wc12)*Y(V46) +
        (screened_rates(k_p_ti45__v46)*Y(H1)*Y(Ti45)*state.rho + -screened_rates(k_v46__p_ti45)*Y(V46)) +
        (-screened_rates(k_n_v46__v47)*Y(N)*Y(V46)*state.rho + screened_rates(k_v47__n_v46)*Y(V47)) +
        (-screened_rates(k_he4_v46__mn50)*Y(He4)*Y(V46)*state.rho + screened_rates(k_mn50__he4_v46)*Y(Mn50)) +
        (-screened_rates(k_n_v46__p_ti46)*Y(N)*Y(V46)*state.rho + screened_rates(k_p_ti46__n_v46)*Y(H1)*Y(Ti46)*state.rho) +
        (-screened_rates(k_n_v46__he4_sc43)*Y(N)*Y(V46)*state.rho + screened_rates(k_he4_sc43__n_v46)*Y(He4)*Y(Sc43)*state.rho) +
        (-screened_rates(k_he4_v46__p_cr49)*Y(He4)*Y(V46)*state.rho + screened_rates(k_p_cr49__he4_v46)*Y(Cr49)*Y(H1)*state.rho);

    ydot_nuc(V47) =
        -screened_rates(k_v47__ti47__weak__wc12)*Y(V47) +
        (screened_rates(k_he4_sc43__v47)*Y(He4)*Y(Sc43)*state.rho + -screened_rates(k_v47__he4_sc43)*Y(V47)) +
        (screened_rates(k_p_ti46__v47)*Y(H1)*Y(Ti46)*state.rho + -screened_rates(k_v47__p_ti46)*Y(V47)) +
        (screened_rates(k_n_v46__v47)*Y(N)*Y(V46)*state.rho + -screened_rates(k_v47__n_v46)*Y(V47)) +
        (-screened_rates(k_n_v47__v48)*Y(N)*Y(V47)*state.rho + screened_rates(k_v48__n_v47)*Y(V48)) +
        (-screened_rates(k_p_v47__cr48)*Y(H1)*Y(V47)*state.rho + screened_rates(k_cr48__p_v47)*Y(Cr48)) +
        (-screened_rates(k_he4_v47__mn51)*Y(He4)*Y(V47)*state.rho + screened_rates(k_mn51__he4_v47)*Y(Mn51)) +
        (-screened_rates(k_n_v47__p_ti47)*Y(N)*Y(V47)*state.rho + screened_rates(k_p_ti47__n_v47)*Y(H1)*Y(Ti47)*state.rho) +
        (-screened_rates(k_n_v47__he4_sc44)*Y(N)*Y(V47)*state.rho + screened_rates(k_he4_sc44__n_v47)*Y(He4)*Y(Sc44)*state.rho) +
        (-screened_rates(k_p_v47__he4_ti44)*Y(H1)*Y(V47)*state.rho + screened_rates(k_he4_ti44__p_v47)*Y(He4)*Y(Ti44)*state.rho) +
        (-screened_rates(k_he4_v47__p_cr50)*Y(He4)*Y(V47)*state.rho + screened_rates(k_p_cr50__he4_v47)*Y(Cr50)*Y(H1)*state.rho) +
        (screened_rates(k_n_mn50__he4_v47)*Y(Mn50)*Y(N)*state.rho + -screened_rates(k_he4_v47__n_mn50)*Y(He4)*Y(V47)*state.rho);

    ydot_nuc(V48) =
        -screened_rates(k_v48__ti48__weak__wc12)*Y(V48) +
        screened_rates(k_cr48__v48__weak__wc12)*Y(Cr48) +
        (screened_rates(k_he4_sc44__v48)*Y(He4)*Y(Sc44)*state.rho + -screened_rates(k_v48__he4_sc44)*Y(V48)) +
        (screened_rates(k_p_ti47__v48)*Y(H1)*Y(Ti47)*state.rho + -screened_rates(k_v48__p_ti47)*Y(V48)) +
        (screened_rates(k_n_v47__v48)*Y(N)*Y(V47)*state.rho + -screened_rates(k_v48__n_v47)*Y(V48)) +
        (-screened_rates(k_n_v48__v49)*Y(N)*Y(V48)*state.rho + screened_rates(k_v49__n_v48)*Y(V49)) +
        (-screened_rates(k_p_v48__cr49)*Y(H1)*Y(V48)*state.rho + screened_rates(k_cr49__p_v48)*Y(Cr49)) +
        (-screened_rates(k_he4_v48__mn52)*Y(He4)*Y(V48)*state.rho + screened_rates(k_mn52__he4_v48)*Y(Mn52)) +
        (screened_rates(k_he4_ti45__p_v48)*Y(He4)*Y(Ti45)*state.rho + -screened_rates(k_p_v48__he4_ti45)*Y(H1)*Y(V48)*state.rho) +
        (-screened_rates(k_n_v48__p_ti48)*Y(N)*Y(V48)*state.rho + screened_rates(k_p_ti48__n_v48)*Y(H1)*Y(Ti48)*state.rho) +
        (-screened_rates(k_n_v48__he4_sc45)*Y(N)*Y(V48)*state.rho + screened_rates(k_he4_sc45__n_v48)*Y(He4)*Y(Sc45)*state.rho) +
        (-screened_rates(k_he4_v48__p_cr51)*Y(He4)*Y(V48)*state.rho + screened_rates(k_p_cr51__he4_v48)*Y(Cr51)*Y(H1)*state.rho) +
        (screened_rates(k_n_cr48__p_v48)*Y(Cr48)*Y(N)*state.rho + -screened_rates(k_p_v48__n_cr48)*Y(H1)*Y(V48)*state.rho) +
        (screened_rates(k_n_mn51__he4_v48)*Y(Mn51)*Y(N)*state.rho + -screened_rates(k_he4_v48__n_mn51)*Y(He4)*Y(V48)*state.rho);

    ydot_nuc(V49) =
        -screened_rates(k_v49__ti49__weak__wc12)*Y(V49) +
        screened_rates(k_cr49__v49__weak__wc12)*Y(Cr49) +
        (screened_rates(k_he4_sc45__v49)*Y(He4)*Y(Sc45)*state.rho + -screened_rates(k_v49__he4_sc45)*Y(V49)) +
        (screened_rates(k_p_ti48__v49)*Y(H1)*Y(Ti48)*state.rho + -screened_rates(k_v49__p_ti48)*Y(V49)) +
        (screened_rates(k_n_v48__v49)*Y(N)*Y(V48)*state.rho + -screened_rates(k_v49__n_v48)*Y(V49)) +
        (-screened_rates(k_n_v49__v50)*Y(N)*Y(V49)*state.rho + screened_rates(k_v50__n_v49)*Y(V50)) +
        (-screened_rates(k_p_v49__cr50)*Y(H1)*Y(V49)*state.rho + screened_rates(k_cr50__p_v49)*Y(Cr50)) +
        (-screened_rates(k_he4_v49__mn53)*Y(He4)*Y(V49)*state.rho + screened_rates(k_mn53__he4_v49)*Y(Mn53)) +
        (screened_rates(k_he4_sc46__n_v49)*Y(He4)*Y(Sc46)*state.rho + -screened_rates(k_n_v49__he4_sc46)*Y(N)*Y(V49)*state.rho) +
        (-screened_rates(k_n_v49__p_ti49)*Y(N)*Y(V49)*state.rho + screened_rates(k_p_ti49__n_v49)*Y(H1)*Y(Ti49)*state.rho) +
        (-screened_rates(k_p_v49__he4_ti46)*Y(H1)*Y(V49)*state.rho + screened_rates(k_he4_ti46__p_v49)*Y(He4)*Y(Ti46)*state.rho) +
        (-screened_rates(k_he4_v49__p_cr52)*Y(He4)*Y(V49)*state.rho + screened_rates(k_p_cr52__he4_v49)*Y(Cr52)*Y(H1)*state.rho) +
        (screened_rates(k_n_cr49__p_v49)*Y(Cr49)*Y(N)*state.rho + -screened_rates(k_p_v49__n_cr49)*Y(H1)*Y(V49)*state.rho) +
        (screened_rates(k_n_mn52__he4_v49)*Y(Mn52)*Y(N)*state.rho + -screened_rates(k_he4_v49__n_mn52)*Y(He4)*Y(V49)*state.rho);

    ydot_nuc(V50) =
        -screened_rates(k_v50__ti50__weak__mo03)*Y(V50) +
        (screened_rates(k_he4_sc46__v50)*Y(He4)*Y(Sc46)*state.rho + -screened_rates(k_v50__he4_sc46)*Y(V50)) +
        (screened_rates(k_p_ti49__v50)*Y(H1)*Y(Ti49)*state.rho + -screened_rates(k_v50__p_ti49)*Y(V50)) +
        (screened_rates(k_n_v49__v50)*Y(N)*Y(V49)*state.rho + -screened_rates(k_v50__n_v49)*Y(V50)) +
        (-screened_rates(k_n_v50__v51)*Y(N)*Y(V50)*state.rho + screened_rates(k_v51__n_v50)*Y(V51)) +
        (-screened_rates(k_p_v50__cr51)*Y(H1)*Y(V50)*state.rho + screened_rates(k_cr51__p_v50)*Y(Cr51)) +
        (-screened_rates(k_he4_v50__mn54)*Y(He4)*Y(V50)*state.rho + screened_rates(k_mn54__he4_v50)*Y(Mn54)) +
        (-screened_rates(k_n_v50__p_ti50)*Y(N)*Y(V50)*state.rho + screened_rates(k_p_ti50__n_v50)*Y(H1)*Y(Ti50)*state.rho) +
        (-screened_rates(k_n_v50__he4_sc47)*Y(N)*Y(V50)*state.rho + screened_rates(k_he4_sc47__n_v50)*Y(He4)*Y(Sc47)*state.rho) +
        (-screened_rates(k_p_v50__n_cr50)*Y(H1)*Y(V50)*state.rho + screened_rates(k_n_cr50__p_v50)*Y(Cr50)*Y(N)*state.rho) +
        (-screened_rates(k_p_v50__he4_ti47)*Y(H1)*Y(V50)*state.rho + screened_rates(k_he4_ti47__p_v50)*Y(He4)*Y(Ti47)*state.rho) +
        (-screened_rates(k_he4_v50__p_cr53)*Y(He4)*Y(V50)*state.rho + screened_rates(k_p_cr53__he4_v50)*Y(Cr53)*Y(H1)*state.rho) +
        (screened_rates(k_n_mn53__he4_v50)*Y(Mn53)*Y(N)*state.rho + -screened_rates(k_he4_v50__n_mn53)*Y(He4)*Y(V50)*state.rho);

    ydot_nuc(V51) =
        screened_rates(k_ti51__v51__weak__wc12)*Y(Ti51) +
        screened_rates(k_cr51__v51__weak__wc12)*Y(Cr51) +
        (screened_rates(k_he4_sc47__v51)*Y(He4)*Y(Sc47)*state.rho + -screened_rates(k_v51__he4_sc47)*Y(V51)) +
        (screened_rates(k_p_ti50__v51)*Y(H1)*Y(Ti50)*state.rho + -screened_rates(k_v51__p_ti50)*Y(V51)) +
        (screened_rates(k_n_v50__v51)*Y(N)*Y(V50)*state.rho + -screened_rates(k_v51__n_v50)*Y(V51)) +
        (-screened_rates(k_n_v51__v52)*Y(N)*Y(V51)*state.rho + screened_rates(k_v52__n_v51)*Y(V52)) +
        (-screened_rates(k_p_v51__cr52)*Y(H1)*Y(V51)*state.rho + screened_rates(k_cr52__p_v51)*Y(Cr52)) +
        (-screened_rates(k_he4_v51__mn55)*Y(He4)*Y(V51)*state.rho + screened_rates(k_mn55__he4_v51)*Y(Mn55)) +
        (screened_rates(k_he4_sc48__n_v51)*Y(He4)*Y(Sc48)*state.rho + -screened_rates(k_n_v51__he4_sc48)*Y(N)*Y(V51)*state.rho) +
        (screened_rates(k_p_ti51__n_v51)*Y(H1)*Y(Ti51)*state.rho + -screened_rates(k_n_v51__p_ti51)*Y(N)*Y(V51)*state.rho) +
        (-screened_rates(k_p_v51__he4_ti48)*Y(H1)*Y(V51)*state.rho + screened_rates(k_he4_ti48__p_v51)*Y(He4)*Y(Ti48)*state.rho) +
        (screened_rates(k_n_cr51__p_v51)*Y(Cr51)*Y(N)*state.rho + -screened_rates(k_p_v51__n_cr51)*Y(H1)*Y(V51)*state.rho) +
        (screened_rates(k_p_cr54__he4_v51)*Y(Cr54)*Y(H1)*state.rho + -screened_rates(k_he4_v51__p_cr54)*Y(He4)*Y(V51)*state.rho) +
        (screened_rates(k_n_mn54__he4_v51)*Y(Mn54)*Y(N)*state.rho + -screened_rates(k_he4_v51__n_mn54)*Y(He4)*Y(V51)*state.rho);

    ydot_nuc(V52) =
        -screened_rates(k_v52__cr52__weak__wc12)*Y(V52) +
        (screened_rates(k_he4_sc48__v52)*Y(He4)*Y(Sc48)*state.rho + -screened_rates(k_v52__he4_sc48)*Y(V52)) +
        (screened_rates(k_p_ti51__v52)*Y(H1)*Y(Ti51)*state.rho + -screened_rates(k_v52__p_ti51)*Y(V52)) +
        (screened_rates(k_n_v51__v52)*Y(N)*Y(V51)*state.rho + -screened_rates(k_v52__n_v51)*Y(V52)) +
        (-screened_rates(k_p_v52__cr53)*Y(H1)*Y(V52)*state.rho + screened_rates(k_cr53__p_v52)*Y(Cr53)) +
        (-screened_rates(k_n_v52__he4_sc49)*Y(N)*Y(V52)*state.rho + screened_rates(k_he4_sc49__n_v52)*Y(He4)*Y(Sc49)*state.rho) +
        (-screened_rates(k_p_v52__n_cr52)*Y(H1)*Y(V52)*state.rho + screened_rates(k_n_cr52__p_v52)*Y(Cr52)*Y(N)*state.rho) +
        (-screened_rates(k_p_v52__he4_ti49)*Y(H1)*Y(V52)*state.rho + screened_rates(k_he4_ti49__p_v52)*Y(He4)*Y(Ti49)*state.rho) +
        (-screened_rates(k_he4_v52__n_mn55)*Y(He4)*Y(V52)*state.rho + screened_rates(k_n_mn55__he4_v52)*Y(Mn55)*Y(N)*state.rho);

    ydot_nuc(Cr48) =
        -screened_rates(k_cr48__v48__weak__wc12)*Y(Cr48) +
        (screened_rates(k_he4_ti44__cr48)*Y(He4)*Y(Ti44)*state.rho + -screened_rates(k_cr48__he4_ti44)*Y(Cr48)) +
        (screened_rates(k_p_v47__cr48)*Y(H1)*Y(V47)*state.rho + -screened_rates(k_cr48__p_v47)*Y(Cr48)) +
        (-screened_rates(k_n_cr48__cr49)*Y(Cr48)*Y(N)*state.rho + screened_rates(k_cr49__n_cr48)*Y(Cr49)) +
        (-screened_rates(k_he4_cr48__fe52)*Y(Cr48)*Y(He4)*state.rho + screened_rates(k_fe52__he4_cr48)*Y(Fe52)) +
        (-screened_rates(k_n_cr48__p_v48)*Y(Cr48)*Y(N)*state.rho + screened_rates(k_p_v48__n_cr48)*Y(H1)*Y(V48)*state.rho) +
        (-screened_rates(k_n_cr48__he4_ti45)*Y(Cr48)*Y(N)*state.rho + screened_rates(k_he4_ti45__n_cr48)*Y(He4)*Y(Ti45)*state.rho) +
        (-screened_rates(k_he4_cr48__p_mn51)*Y(Cr48)*Y(He4)*state.rho + screened_rates(k_p_mn51__he4_cr48)*Y(Mn51)*Y(H1)*state.rho);

    ydot_nuc(Cr49) =
        -screened_rates(k_cr49__v49__weak__wc12)*Y(Cr49) +
        (screened_rates(k_he4_ti45__cr49)*Y(He4)*Y(Ti45)*state.rho + -screened_rates(k_cr49__he4_ti45)*Y(Cr49)) +
        (screened_rates(k_p_v48__cr49)*Y(H1)*Y(V48)*state.rho + -screened_rates(k_cr49__p_v48)*Y(Cr49)) +
        (screened_rates(k_n_cr48__cr49)*Y(Cr48)*Y(N)*state.rho + -screened_rates(k_cr49__n_cr48)*Y(Cr49)) +
        (-screened_rates(k_n_cr49__cr50)*Y(Cr49)*Y(N)*state.rho + screened_rates(k_cr50__n_cr49)*Y(Cr50)) +
        (-screened_rates(k_p_cr49__mn50)*Y(Cr49)*Y(H1)*state.rho + screened_rates(k_mn50__p_cr49)*Y(Mn50)) +
        (-screened_rates(k_he4_cr49__fe53)*Y(Cr49)*Y(He4)*state.rho + screened_rates(k_fe53__he4_cr49)*Y(Fe53)) +
        (screened_rates(k_he4_v46__p_cr49)*Y(He4)*Y(V46)*state.rho + -screened_rates(k_p_cr49__he4_v46)*Y(Cr49)*Y(H1)*state.rho) +
        (-screened_rates(k_n_cr49__p_v49)*Y(Cr49)*Y(N)*state.rho + screened_rates(k_p_v49__n_cr49)*Y(H1)*Y(V49)*state.rho) +
        (-screened_rates(k_n_cr49__he4_ti46)*Y(Cr49)*Y(N)*state.rho + screened_rates(k_he4_ti46__n_cr49)*Y(He4)*Y(Ti46)*state.rho) +
        (-screened_rates(k_he4_cr49__p_mn52)*Y(Cr49)*Y(He4)*state.rho + screened_rates(k_p_mn52__he4_cr49)*Y(Mn52)*Y(H1)*state.rho) +
        (screened_rates(k_n_fe52__he4_cr49)*Y(Fe52)*Y(N)*state.rho + -screened_rates(k_he4_cr49__n_fe52)*Y(Cr49)*Y(He4)*state.rho);

    ydot_nuc(Cr50) =
        screened_rates(k_mn50__cr50__weak__wc12)*Y(Mn50) +
        (screened_rates(k_he4_ti46__cr50)*Y(He4)*Y(Ti46)*state.rho + -screened_rates(k_cr50__he4_ti46)*Y(Cr50)) +
        (screened_rates(k_p_v49__cr50)*Y(H1)*Y(V49)*state.rho + -screened_rates(k_cr50__p_v49)*Y(Cr50)) +
        (screened_rates(k_n_cr49__cr50)*Y(Cr49)*Y(N)*state.rho + -screened_rates(k_cr50__n_cr49)*Y(Cr50)) +
        (-screened_rates(k_n_cr50__cr51)*Y(Cr50)*Y(N)*state.rho + screened_rates(k_cr51__n_cr50)*Y(Cr51)) +
        (-screened_rates(k_p_cr50__mn51)*Y(Cr50)*Y(H1)*state.rho + screened_rates(k_mn51__p_cr50)*Y(Mn51)) +
        (-screened_rates(k_he4_cr50__fe54)*Y(Cr50)*Y(He4)*state.rho + screened_rates(k_fe54__he4_cr50)*Y(Fe54)) +
        (screened_rates(k_he4_v47__p_cr50)*Y(He4)*Y(V47)*state.rho + -screened_rates(k_p_cr50__he4_v47)*Y(Cr50)*Y(H1)*state.rho) +
        (screened_rates(k_p_v50__n_cr50)*Y(H1)*Y(V50)*state.rho + -screened_rates(k_n_cr50__p_v50)*Y(Cr50)*Y(N)*state.rho) +
        (-screened_rates(k_n_cr50__he4_ti47)*Y(Cr50)*Y(N)*state.rho + screened_rates(k_he4_ti47__n_cr50)*Y(He4)*Y(Ti47)*state.rho) +
        (screened_rates(k_n_mn50__p_cr50)*Y(Mn50)*Y(N)*state.rho + -screened_rates(k_p_cr50__n_mn50)*Y(Cr50)*Y(H1)*state.rho) +
        (screened_rates(k_p_mn53__he4_cr50)*Y(Mn53)*Y(H1)*state.rho + -screened_rates(k_he4_cr50__p_mn53)*Y(Cr50)*Y(He4)*state.rho) +
        (screened_rates(k_n_fe53__he4_cr50)*Y(Fe53)*Y(N)*state.rho + -screened_rates(k_he4_cr50__n_fe53)*Y(Cr50)*Y(He4)*state.rho);

    ydot_nuc(Cr51) =
        -screened_rates(k_cr51__v51__weak__wc12)*Y(Cr51) +
        screened_rates(k_mn51__cr51__weak__wc12)*Y(Mn51) +
        (screened_rates(k_he4_ti47__cr51)*Y(He4)*Y(Ti47)*state.rho + -screened_rates(k_cr51__he4_ti47)*Y(Cr51)) +
        (screened_rates(k_p_v50__cr51)*Y(H1)*Y(V50)*state.rho + -screened_rates(k_cr51__p_v50)*Y(Cr51)) +
        (screened_rates(k_n_cr50__cr51)*Y(Cr50)*Y(N)*state.rho + -screened_rates(k_cr51__n_cr50)*Y(Cr51)) +
        (-screened_rates(k_n_cr51__cr52)*Y(Cr51)*Y(N)*state.rho + screened_rates(k_cr52__n_cr51)*Y(Cr52)) +
        (-screened_rates(k_p_cr51__mn52)*Y(Cr51)*Y(H1)*state.rho + screened_rates(k_mn52__p_cr51)*Y(Mn52)) +
        (-screened_rates(k_he4_cr51__fe55)*Y(Cr51)*Y(He4)*state.rho + screened_rates(k_fe55__he4_cr51)*Y(Fe55)) +
        (screened_rates(k_he4_v48__p_cr51)*Y(He4)*Y(V48)*state.rho + -screened_rates(k_p_cr51__he4_v48)*Y(Cr51)*Y(H1)*state.rho) +
        (-screened_rates(k_n_cr51__p_v51)*Y(Cr51)*Y(N)*state.rho + screened_rates(k_p_v51__n_cr51)*Y(H1)*Y(V51)*state.rho) +
        (-screened_rates(k_n_cr51__he4_ti48)*Y(Cr51)*Y(N)*state.rho + screened_rates(k_he4_ti48__n_cr51)*Y(He4)*Y(Ti48)*state.rho) +
        (screened_rates(k_n_mn51__p_cr51)*Y(Mn51)*Y(N)*state.rho + -screened_rates(k_p_cr51__n_mn51)*Y(Cr51)*Y(H1)*state.rho) +
        (screened_rates(k_p_mn54__he4_cr51)*Y(Mn54)*Y(H1)*state.rho + -screened_rates(k_he4_cr51__p_mn54)*Y(Cr51)*Y(He4)*state.rho) +
        (screened_rates(k_n_fe54__he4_cr51)*Y(Fe54)*Y(N)*state.rho + -screened_rates(k_he4_cr51__n_fe54)*Y(Cr51)*Y(He4)*state.rho);

    ydot_nuc(Cr52) =
        screened_rates(k_v52__cr52__weak__wc12)*Y(V52) +
        screened_rates(k_mn52__cr52__weak__wc12)*Y(Mn52) +
        (screened_rates(k_he4_ti48__cr52)*Y(He4)*Y(Ti48)*state.rho + -screened_rates(k_cr52__he4_ti48)*Y(Cr52)) +
        (screened_rates(k_p_v51__cr52)*Y(H1)*Y(V51)*state.rho + -screened_rates(k_cr52__p_v51)*Y(Cr52)) +
        (screened_rates(k_n_cr51__cr52)*Y(Cr51)*Y(N)*state.rho + -screened_rates(k_cr52__n_cr51)*Y(Cr52)) +
        (-screened_rates(k_n_cr52__cr53)*Y(Cr52)*Y(N)*state.rho + screened_rates(k_cr53__n_cr52)*Y(Cr53)) +
        (-screened_rates(k_p_cr52__mn53)*Y(Cr52)*Y(H1)*state.rho + screened_rates(k_mn53__p_cr52)*Y(Mn53)) +
        (-screened_rates(k_he4_cr52__fe56)*Y(Cr52)*Y(He4)*state.rho + screened_rates(k_fe56__he4_cr52)*Y(Fe56)) +
        (screened_rates(k_he4_ti49__n_cr52)*Y(He4)*Y(Ti49)*state.rho + -screened_rates(k_n_cr52__he4_ti49)*Y(Cr52)*Y(N)*state.rho) +
        (screened_rates(k_he4_v49__p_cr52)*Y(He4)*Y(V49)*state.rho + -screened_rates(k_p_cr52__he4_v49)*Y(Cr52)*Y(H1)*state.rho) +
        (screened_rates(k_p_v52__n_cr52)*Y(H1)*Y(V52)*state.rho + -screened_rates(k_n_cr52__p_v52)*Y(Cr52)*Y(N)*state.rho) +
        (screened_rates(k_n_mn52__p_cr52)*Y(Mn52)*Y(N)*state.rho + -screened_rates(k_p_cr52__n_mn52)*Y(Cr52)*Y(H1)*state.rho) +
        (screened_rates(k_p_mn55__he4_cr52)*Y(Mn55)*Y(H1)*state.rho + -screened_rates(k_he4_cr52__p_mn55)*Y(Cr52)*Y(He4)*state.rho) +
        (screened_rates(k_n_fe55__he4_cr52)*Y(Fe55)*Y(N)*state.rho + -screened_rates(k_he4_cr52__n_fe55)*Y(Cr52)*Y(He4)*state.rho);

    ydot_nuc(Cr53) =
        screened_rates(k_mn53__cr53__weak__wc12)*Y(Mn53) +
        (screened_rates(k_he4_ti49__cr53)*Y(He4)*Y(Ti49)*state.rho + -screened_rates(k_cr53__he4_ti49)*Y(Cr53)) +
        (screened_rates(k_p_v52__cr53)*Y(H1)*Y(V52)*state.rho + -screened_rates(k_cr53__p_v52)*Y(Cr53)) +
        (screened_rates(k_n_cr52__cr53)*Y(Cr52)*Y(N)*state.rho + -screened_rates(k_cr53__n_cr52)*Y(Cr53)) +
        (-screened_rates(k_n_cr53__cr54)*Y(Cr53)*Y(N)*state.rho + screened_rates(k_cr54__n_cr53)*Y(Cr54)) +
        (-screened_rates(k_p_cr53__mn54)*Y(Cr53)*Y(H1)*state.rho + screened_rates(k_mn54__p_cr53)*Y(Mn54)) +
        (-screened_rates(k_he4_cr53__fe57)*Y(Cr53)*Y(He4)*state.rho + screened_rates(k_fe57__he4_cr53)*Y(Fe57)) +
        (screened_rates(k_he4_v50__p_cr53)*Y(He4)*Y(V50)*state.rho + -screened_rates(k_p_cr53__he4_v50)*Y(Cr53)*Y(H1)*state.rho) +
        (-screened_rates(k_n_cr53__he4_ti50)*Y(Cr53)*Y(N)*state.rho + screened_rates(k_he4_ti50__n_cr53)*Y(He4)*Y(Ti50)*state.rho) +
        (screened_rates(k_n_mn53__p_cr53)*Y(Mn53)*Y(N)*state.rho + -screened_rates(k_p_cr53__n_mn53)*Y(Cr53)*Y(H1)*state.rho) +
        (screened_rates(k_n_fe56__he4_cr53)*Y(Fe56)*Y(N)*state.rho + -screened_rates(k_he4_cr53__n_fe56)*Y(Cr53)*Y(He4)*state.rho);

    ydot_nuc(Cr54) =
        screened_rates(k_mn54__cr54__weak__wc12)*Y(Mn54) +
        (screened_rates(k_he4_ti50__cr54)*Y(He4)*Y(Ti50)*state.rho + -screened_rates(k_cr54__he4_ti50)*Y(Cr54)) +
        (screened_rates(k_n_cr53__cr54)*Y(Cr53)*Y(N)*state.rho + -screened_rates(k_cr54__n_cr53)*Y(Cr54)) +
        (-screened_rates(k_p_cr54__mn55)*Y(Cr54)*Y(H1)*state.rho + screened_rates(k_mn55__p_cr54)*Y(Mn55)) +
        (-screened_rates(k_he4_cr54__fe58)*Y(Cr54)*Y(He4)*state.rho + screened_rates(k_fe58__he4_cr54)*Y(Fe58)) +
        (screened_rates(k_he4_ti51__n_cr54)*Y(He4)*Y(Ti51)*state.rho + -screened_rates(k_n_cr54__he4_ti51)*Y(Cr54)*Y(N)*state.rho) +
        (-screened_rates(k_p_cr54__he4_v51)*Y(Cr54)*Y(H1)*state.rho + screened_rates(k_he4_v51__p_cr54)*Y(He4)*Y(V51)*state.rho) +
        (screened_rates(k_n_mn54__p_cr54)*Y(Mn54)*Y(N)*state.rho + -screened_rates(k_p_cr54__n_mn54)*Y(Cr54)*Y(H1)*state.rho) +
        (screened_rates(k_n_fe57__he4_cr54)*Y(Fe57)*Y(N)*state.rho + -screened_rates(k_he4_cr54__n_fe57)*Y(Cr54)*Y(He4)*state.rho);

    ydot_nuc(Mn50) =
        -screened_rates(k_mn50__cr50__weak__wc12)*Y(Mn50) +
        (screened_rates(k_he4_v46__mn50)*Y(He4)*Y(V46)*state.rho + -screened_rates(k_mn50__he4_v46)*Y(Mn50)) +
        (screened_rates(k_p_cr49__mn50)*Y(Cr49)*Y(H1)*state.rho + -screened_rates(k_mn50__p_cr49)*Y(Mn50)) +
        (-screened_rates(k_n_mn50__mn51)*Y(Mn50)*Y(N)*state.rho + screened_rates(k_mn51__n_mn50)*Y(Mn51)) +
        (-screened_rates(k_he4_mn50__co54)*Y(He4)*Y(Mn50)*state.rho + screened_rates(k_co54__he4_mn50)*Y(Co54)) +
        (-screened_rates(k_n_mn50__p_cr50)*Y(Mn50)*Y(N)*state.rho + screened_rates(k_p_cr50__n_mn50)*Y(Cr50)*Y(H1)*state.rho) +
        (-screened_rates(k_n_mn50__he4_v47)*Y(Mn50)*Y(N)*state.rho + screened_rates(k_he4_v47__n_mn50)*Y(He4)*Y(V47)*state.rho) +
        (-screened_rates(k_he4_mn50__p_fe53)*Y(He4)*Y(Mn50)*state.rho + screened_rates(k_p_fe53__he4_mn50)*Y(Fe53)*Y(H1)*state.rho) +
        (screened_rates(k_n_co53__he4_mn50)*Y(Co53)*Y(N)*state.rho + -screened_rates(k_he4_mn50__n_co53)*Y(He4)*Y(Mn50)*state.rho);

    ydot_nuc(Mn51) =
        -screened_rates(k_mn51__cr51__weak__wc12)*Y(Mn51) +
        (screened_rates(k_he4_v47__mn51)*Y(He4)*Y(V47)*state.rho + -screened_rates(k_mn51__he4_v47)*Y(Mn51)) +
        (screened_rates(k_p_cr50__mn51)*Y(Cr50)*Y(H1)*state.rho + -screened_rates(k_mn51__p_cr50)*Y(Mn51)) +
        (screened_rates(k_n_mn50__mn51)*Y(Mn50)*Y(N)*state.rho + -screened_rates(k_mn51__n_mn50)*Y(Mn51)) +
        (-screened_rates(k_n_mn51__mn52)*Y(Mn51)*Y(N)*state.rho + screened_rates(k_mn52__n_mn51)*Y(Mn52)) +
        (-screened_rates(k_p_mn51__fe52)*Y(Mn51)*Y(H1)*state.rho + screened_rates(k_fe52__p_mn51)*Y(Fe52)) +
        (-screened_rates(k_he4_mn51__co55)*Y(He4)*Y(Mn51)*state.rho + screened_rates(k_co55__he4_mn51)*Y(Co55)) +
        (screened_rates(k_he4_cr48__p_mn51)*Y(Cr48)*Y(He4)*state.rho + -screened_rates(k_p_mn51__he4_cr48)*Y(Mn51)*Y(H1)*state.rho) +
        (-screened_rates(k_n_mn51__p_cr51)*Y(Mn51)*Y(N)*state.rho + screened_rates(k_p_cr51__n_mn51)*Y(Cr51)*Y(H1)*state.rho) +
        (-screened_rates(k_n_mn51__he4_v48)*Y(Mn51)*Y(N)*state.rho + screened_rates(k_he4_v48__n_mn51)*Y(He4)*Y(V48)*state.rho) +
        (-screened_rates(k_he4_mn51__p_fe54)*Y(He4)*Y(Mn51)*state.rho + screened_rates(k_p_fe54__he4_mn51)*Y(Fe54)*Y(H1)*state.rho) +
        (screened_rates(k_n_co54__he4_mn51)*Y(Co54)*Y(N)*state.rho + -screened_rates(k_he4_mn51__n_co54)*Y(He4)*Y(Mn51)*state.rho);

    ydot_nuc(Mn52) =
        -screened_rates(k_mn52__cr52__weak__wc12)*Y(Mn52) +
        screened_rates(k_fe52__mn52__weak__wc12)*Y(Fe52) +
        (screened_rates(k_he4_v48__mn52)*Y(He4)*Y(V48)*state.rho + -screened_rates(k_mn52__he4_v48)*Y(Mn52)) +
        (screened_rates(k_p_cr51__mn52)*Y(Cr51)*Y(H1)*state.rho + -screened_rates(k_mn52__p_cr51)*Y(Mn52)) +
        (screened_rates(k_n_mn51__mn52)*Y(Mn51)*Y(N)*state.rho + -screened_rates(k_mn52__n_mn51)*Y(Mn52)) +
        (-screened_rates(k_n_mn52__mn53)*Y(Mn52)*Y(N)*state.rho + screened_rates(k_mn53__n_mn52)*Y(Mn53)) +
        (-screened_rates(k_p_mn52__fe53)*Y(Mn52)*Y(H1)*state.rho + screened_rates(k_fe53__p_mn52)*Y(Fe53)) +
        (-screened_rates(k_he4_mn52__co56)*Y(He4)*Y(Mn52)*state.rho + screened_rates(k_co56__he4_mn52)*Y(Co56)) +
        (screened_rates(k_he4_cr49__p_mn52)*Y(Cr49)*Y(He4)*state.rho + -screened_rates(k_p_mn52__he4_cr49)*Y(Mn52)*Y(H1)*state.rho) +
        (-screened_rates(k_n_mn52__p_cr52)*Y(Mn52)*Y(N)*state.rho + screened_rates(k_p_cr52__n_mn52)*Y(Cr52)*Y(H1)*state.rho) +
        (-screened_rates(k_n_mn52__he4_v49)*Y(Mn52)*Y(N)*state.rho + screened_rates(k_he4_v49__n_mn52)*Y(He4)*Y(V49)*state.rho) +
        (-screened_rates(k_he4_mn52__p_fe55)*Y(He4)*Y(Mn52)*state.rho + screened_rates(k_p_fe55__he4_mn52)*Y(Fe55)*Y(H1)*state.rho) +
        (screened_rates(k_n_fe52__p_mn52)*Y(Fe52)*Y(N)*state.rho + -screened_rates(k_p_mn52__n_fe52)*Y(Mn52)*Y(H1)*state.rho) +
        (screened_rates(k_n_co55__he4_mn52)*Y(Co55)*Y(N)*state.rho + -screened_rates(k_he4_mn52__n_co55)*Y(He4)*Y(Mn52)*state.rho);

    ydot_nuc(Mn53) =
        -screened_rates(k_mn53__cr53__weak__wc12)*Y(Mn53) +
        screened_rates(k_fe53__mn53__weak__wc12)*Y(Fe53) +
        (screened_rates(k_he4_v49__mn53)*Y(He4)*Y(V49)*state.rho + -screened_rates(k_mn53__he4_v49)*Y(Mn53)) +
        (screened_rates(k_p_cr52__mn53)*Y(Cr52)*Y(H1)*state.rho + -screened_rates(k_mn53__p_cr52)*Y(Mn53)) +
        (screened_rates(k_n_mn52__mn53)*Y(Mn52)*Y(N)*state.rho + -screened_rates(k_mn53__n_mn52)*Y(Mn53)) +
        (-screened_rates(k_n_mn53__mn54)*Y(Mn53)*Y(N)*state.rho + screened_rates(k_mn54__n_mn53)*Y(Mn54)) +
        (-screened_rates(k_p_mn53__fe54)*Y(Mn53)*Y(H1)*state.rho + screened_rates(k_fe54__p_mn53)*Y(Fe54)) +
        (-screened_rates(k_he4_mn53__co57)*Y(He4)*Y(Mn53)*state.rho + screened_rates(k_co57__he4_mn53)*Y(Co57)) +
        (-screened_rates(k_n_mn53__p_cr53)*Y(Mn53)*Y(N)*state.rho + screened_rates(k_p_cr53__n_mn53)*Y(Cr53)*Y(H1)*state.rho) +
        (-screened_rates(k_n_mn53__he4_v50)*Y(Mn53)*Y(N)*state.rho + screened_rates(k_he4_v50__n_mn53)*Y(He4)*Y(V50)*state.rho) +
        (-screened_rates(k_p_mn53__he4_cr50)*Y(Mn53)*Y(H1)*state.rho + screened_rates(k_he4_cr50__p_mn53)*Y(Cr50)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_mn53__p_fe56)*Y(He4)*Y(Mn53)*state.rho + screened_rates(k_p_fe56__he4_mn53)*Y(Fe56)*Y(H1)*state.rho) +
        (screened_rates(k_n_fe53__p_mn53)*Y(Fe53)*Y(N)*state.rho + -screened_rates(k_p_mn53__n_fe53)*Y(Mn53)*Y(H1)*state.rho) +
        (screened_rates(k_n_co56__he4_mn53)*Y(Co56)*Y(N)*state.rho + -screened_rates(k_he4_mn53__n_co56)*Y(He4)*Y(Mn53)*state.rho);

    ydot_nuc(Mn54) =
        -screened_rates(k_mn54__cr54__weak__wc12)*Y(Mn54) +
        (screened_rates(k_he4_v50__mn54)*Y(He4)*Y(V50)*state.rho + -screened_rates(k_mn54__he4_v50)*Y(Mn54)) +
        (screened_rates(k_p_cr53__mn54)*Y(Cr53)*Y(H1)*state.rho + -screened_rates(k_mn54__p_cr53)*Y(Mn54)) +
        (screened_rates(k_n_mn53__mn54)*Y(Mn53)*Y(N)*state.rho + -screened_rates(k_mn54__n_mn53)*Y(Mn54)) +
        (-screened_rates(k_n_mn54__mn55)*Y(Mn54)*Y(N)*state.rho + screened_rates(k_mn55__n_mn54)*Y(Mn55)) +
        (-screened_rates(k_p_mn54__fe55)*Y(Mn54)*Y(H1)*state.rho + screened_rates(k_fe55__p_mn54)*Y(Fe55)) +
        (-screened_rates(k_he4_mn54__co58)*Y(He4)*Y(Mn54)*state.rho + screened_rates(k_co58__he4_mn54)*Y(Co58)) +
        (-screened_rates(k_n_mn54__p_cr54)*Y(Mn54)*Y(N)*state.rho + screened_rates(k_p_cr54__n_mn54)*Y(Cr54)*Y(H1)*state.rho) +
        (-screened_rates(k_n_mn54__he4_v51)*Y(Mn54)*Y(N)*state.rho + screened_rates(k_he4_v51__n_mn54)*Y(He4)*Y(V51)*state.rho) +
        (-screened_rates(k_p_mn54__he4_cr51)*Y(Mn54)*Y(H1)*state.rho + screened_rates(k_he4_cr51__p_mn54)*Y(Cr51)*Y(He4)*state.rho) +
        (screened_rates(k_n_fe54__p_mn54)*Y(Fe54)*Y(N)*state.rho + -screened_rates(k_p_mn54__n_fe54)*Y(Mn54)*Y(H1)*state.rho) +
        (screened_rates(k_p_fe57__he4_mn54)*Y(Fe57)*Y(H1)*state.rho + -screened_rates(k_he4_mn54__p_fe57)*Y(He4)*Y(Mn54)*state.rho) +
        (screened_rates(k_n_co57__he4_mn54)*Y(Co57)*Y(N)*state.rho + -screened_rates(k_he4_mn54__n_co57)*Y(He4)*Y(Mn54)*state.rho);

    ydot_nuc(Mn55) =
        screened_rates(k_fe55__mn55__weak__wc12)*Y(Fe55) +
        (screened_rates(k_he4_v51__mn55)*Y(He4)*Y(V51)*state.rho + -screened_rates(k_mn55__he4_v51)*Y(Mn55)) +
        (screened_rates(k_p_cr54__mn55)*Y(Cr54)*Y(H1)*state.rho + -screened_rates(k_mn55__p_cr54)*Y(Mn55)) +
        (screened_rates(k_n_mn54__mn55)*Y(Mn54)*Y(N)*state.rho + -screened_rates(k_mn55__n_mn54)*Y(Mn55)) +
        (-screened_rates(k_p_mn55__fe56)*Y(Mn55)*Y(H1)*state.rho + screened_rates(k_fe56__p_mn55)*Y(Fe56)) +
        (-screened_rates(k_he4_mn55__co59)*Y(He4)*Y(Mn55)*state.rho + screened_rates(k_co59__he4_mn55)*Y(Co59)) +
        (screened_rates(k_he4_v52__n_mn55)*Y(He4)*Y(V52)*state.rho + -screened_rates(k_n_mn55__he4_v52)*Y(Mn55)*Y(N)*state.rho) +
        (-screened_rates(k_p_mn55__he4_cr52)*Y(Mn55)*Y(H1)*state.rho + screened_rates(k_he4_cr52__p_mn55)*Y(Cr52)*Y(He4)*state.rho) +
        (screened_rates(k_n_fe55__p_mn55)*Y(Fe55)*Y(N)*state.rho + -screened_rates(k_p_mn55__n_fe55)*Y(Mn55)*Y(H1)*state.rho) +
        (screened_rates(k_p_fe58__he4_mn55)*Y(Fe58)*Y(H1)*state.rho + -screened_rates(k_he4_mn55__p_fe58)*Y(He4)*Y(Mn55)*state.rho) +
        (screened_rates(k_n_co58__he4_mn55)*Y(Co58)*Y(N)*state.rho + -screened_rates(k_he4_mn55__n_co58)*Y(He4)*Y(Mn55)*state.rho);

    ydot_nuc(Fe52) =
        -screened_rates(k_fe52__mn52__weak__wc12)*Y(Fe52) +
        (screened_rates(k_he4_cr48__fe52)*Y(Cr48)*Y(He4)*state.rho + -screened_rates(k_fe52__he4_cr48)*Y(Fe52)) +
        (screened_rates(k_p_mn51__fe52)*Y(Mn51)*Y(H1)*state.rho + -screened_rates(k_fe52__p_mn51)*Y(Fe52)) +
        (-screened_rates(k_n_fe52__fe53)*Y(Fe52)*Y(N)*state.rho + screened_rates(k_fe53__n_fe52)*Y(Fe53)) +
        (-screened_rates(k_p_fe52__co53)*Y(Fe52)*Y(H1)*state.rho + screened_rates(k_co53__p_fe52)*Y(Co53)) +
        (-screened_rates(k_he4_fe52__ni56)*Y(Fe52)*Y(He4)*state.rho + screened_rates(k_ni56__he4_fe52)*Y(Ni56)) +
        (-screened_rates(k_n_fe52__p_mn52)*Y(Fe52)*Y(N)*state.rho + screened_rates(k_p_mn52__n_fe52)*Y(Mn52)*Y(H1)*state.rho) +
        (-screened_rates(k_n_fe52__he4_cr49)*Y(Fe52)*Y(N)*state.rho + screened_rates(k_he4_cr49__n_fe52)*Y(Cr49)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_fe52__p_co55)*Y(Fe52)*Y(He4)*state.rho + screened_rates(k_p_co55__he4_fe52)*Y(Co55)*Y(H1)*state.rho);

    ydot_nuc(Fe53) =
        -screened_rates(k_fe53__mn53__weak__wc12)*Y(Fe53) +
        screened_rates(k_co53__fe53__weak__wc12)*Y(Co53) +
        (screened_rates(k_he4_cr49__fe53)*Y(Cr49)*Y(He4)*state.rho + -screened_rates(k_fe53__he4_cr49)*Y(Fe53)) +
        (screened_rates(k_p_mn52__fe53)*Y(Mn52)*Y(H1)*state.rho + -screened_rates(k_fe53__p_mn52)*Y(Fe53)) +
        (screened_rates(k_n_fe52__fe53)*Y(Fe52)*Y(N)*state.rho + -screened_rates(k_fe53__n_fe52)*Y(Fe53)) +
        (-screened_rates(k_n_fe53__fe54)*Y(Fe53)*Y(N)*state.rho + screened_rates(k_fe54__n_fe53)*Y(Fe54)) +
        (-screened_rates(k_p_fe53__co54)*Y(Fe53)*Y(H1)*state.rho + screened_rates(k_co54__p_fe53)*Y(Co54)) +
        (-screened_rates(k_he4_fe53__ni57)*Y(Fe53)*Y(He4)*state.rho + screened_rates(k_ni57__he4_fe53)*Y(Ni57)) +
        (screened_rates(k_he4_mn50__p_fe53)*Y(He4)*Y(Mn50)*state.rho + -screened_rates(k_p_fe53__he4_mn50)*Y(Fe53)*Y(H1)*state.rho) +
        (-screened_rates(k_n_fe53__p_mn53)*Y(Fe53)*Y(N)*state.rho + screened_rates(k_p_mn53__n_fe53)*Y(Mn53)*Y(H1)*state.rho) +
        (-screened_rates(k_n_fe53__he4_cr50)*Y(Fe53)*Y(N)*state.rho + screened_rates(k_he4_cr50__n_fe53)*Y(Cr50)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_fe53__p_co56)*Y(Fe53)*Y(He4)*state.rho + screened_rates(k_p_co56__he4_fe53)*Y(Co56)*Y(H1)*state.rho) +
        (screened_rates(k_n_co53__p_fe53)*Y(Co53)*Y(N)*state.rho + -screened_rates(k_p_fe53__n_co53)*Y(Fe53)*Y(H1)*state.rho) +
        (screened_rates(k_n_ni56__he4_fe53)*Y(N)*Y(Ni56)*state.rho + -screened_rates(k_he4_fe53__n_ni56)*Y(Fe53)*Y(He4)*state.rho);

    ydot_nuc(Fe54) =
        screened_rates(k_co54__fe54__weak__wc12)*Y(Co54) +
        (screened_rates(k_he4_cr50__fe54)*Y(Cr50)*Y(He4)*state.rho + -screened_rates(k_fe54__he4_cr50)*Y(Fe54)) +
        (screened_rates(k_p_mn53__fe54)*Y(Mn53)*Y(H1)*state.rho + -screened_rates(k_fe54__p_mn53)*Y(Fe54)) +
        (screened_rates(k_n_fe53__fe54)*Y(Fe53)*Y(N)*state.rho + -screened_rates(k_fe54__n_fe53)*Y(Fe54)) +
        (-screened_rates(k_n_fe54__fe55)*Y(Fe54)*Y(N)*state.rho + screened_rates(k_fe55__n_fe54)*Y(Fe55)) +
        (-screened_rates(k_p_fe54__co55)*Y(Fe54)*Y(H1)*state.rho + screened_rates(k_co55__p_fe54)*Y(Co55)) +
        (-screened_rates(k_he4_fe54__ni58)*Y(Fe54)*Y(He4)*state.rho + screened_rates(k_ni58__he4_fe54)*Y(Ni58)) +
        (screened_rates(k_he4_mn51__p_fe54)*Y(He4)*Y(Mn51)*state.rho + -screened_rates(k_p_fe54__he4_mn51)*Y(Fe54)*Y(H1)*state.rho) +
        (-screened_rates(k_n_fe54__p_mn54)*Y(Fe54)*Y(N)*state.rho + screened_rates(k_p_mn54__n_fe54)*Y(Mn54)*Y(H1)*state.rho) +
        (-screened_rates(k_n_fe54__he4_cr51)*Y(Fe54)*Y(N)*state.rho + screened_rates(k_he4_cr51__n_fe54)*Y(Cr51)*Y(He4)*state.rho) +
        (screened_rates(k_n_co54__p_fe54)*Y(Co54)*Y(N)*state.rho + -screened_rates(k_p_fe54__n_co54)*Y(Fe54)*Y(H1)*state.rho) +
        (screened_rates(k_p_co57__he4_fe54)*Y(Co57)*Y(H1)*state.rho + -screened_rates(k_he4_fe54__p_co57)*Y(Fe54)*Y(He4)*state.rho) +
        (screened_rates(k_n_ni57__he4_fe54)*Y(N)*Y(Ni57)*state.rho + -screened_rates(k_he4_fe54__n_ni57)*Y(Fe54)*Y(He4)*state.rho);

    ydot_nuc(Fe55) =
        -screened_rates(k_fe55__mn55__weak__wc12)*Y(Fe55) +
        screened_rates(k_co55__fe55__weak__wc12)*Y(Co55) +
        (screened_rates(k_he4_cr51__fe55)*Y(Cr51)*Y(He4)*state.rho + -screened_rates(k_fe55__he4_cr51)*Y(Fe55)) +
        (screened_rates(k_p_mn54__fe55)*Y(Mn54)*Y(H1)*state.rho + -screened_rates(k_fe55__p_mn54)*Y(Fe55)) +
        (screened_rates(k_n_fe54__fe55)*Y(Fe54)*Y(N)*state.rho + -screened_rates(k_fe55__n_fe54)*Y(Fe55)) +
        (-screened_rates(k_n_fe55__fe56)*Y(Fe55)*Y(N)*state.rho + screened_rates(k_fe56__n_fe55)*Y(Fe56)) +
        (-screened_rates(k_p_fe55__co56)*Y(Fe55)*Y(H1)*state.rho + screened_rates(k_co56__p_fe55)*Y(Co56)) +
        (-screened_rates(k_he4_fe55__ni59)*Y(Fe55)*Y(He4)*state.rho + screened_rates(k_ni59__he4_fe55)*Y(Ni59)) +
        (screened_rates(k_he4_mn52__p_fe55)*Y(He4)*Y(Mn52)*state.rho + -screened_rates(k_p_fe55__he4_mn52)*Y(Fe55)*Y(H1)*state.rho) +
        (-screened_rates(k_n_fe55__p_mn55)*Y(Fe55)*Y(N)*state.rho + screened_rates(k_p_mn55__n_fe55)*Y(Mn55)*Y(H1)*state.rho) +
        (-screened_rates(k_n_fe55__he4_cr52)*Y(Fe55)*Y(N)*state.rho + screened_rates(k_he4_cr52__n_fe55)*Y(Cr52)*Y(He4)*state.rho) +
        (screened_rates(k_n_co55__p_fe55)*Y(Co55)*Y(N)*state.rho + -screened_rates(k_p_fe55__n_co55)*Y(Fe55)*Y(H1)*state.rho) +
        (screened_rates(k_p_co58__he4_fe55)*Y(Co58)*Y(H1)*state.rho + -screened_rates(k_he4_fe55__p_co58)*Y(Fe55)*Y(He4)*state.rho) +
        (screened_rates(k_n_ni58__he4_fe55)*Y(N)*Y(Ni58)*state.rho + -screened_rates(k_he4_fe55__n_ni58)*Y(Fe55)*Y(He4)*state.rho);

    ydot_nuc(Fe56) =
        screened_rates(k_co56__fe56__weak__wc12)*Y(Co56) +
        (screened_rates(k_he4_cr52__fe56)*Y(Cr52)*Y(He4)*state.rho + -screened_rates(k_fe56__he4_cr52)*Y(Fe56)) +
        (screened_rates(k_p_mn55__fe56)*Y(Mn55)*Y(H1)*state.rho + -screened_rates(k_fe56__p_mn55)*Y(Fe56)) +
        (screened_rates(k_n_fe55__fe56)*Y(Fe55)*Y(N)*state.rho + -screened_rates(k_fe56__n_fe55)*Y(Fe56)) +
        (-screened_rates(k_n_fe56__fe57)*Y(Fe56)*Y(N)*state.rho + screened_rates(k_fe57__n_fe56)*Y(Fe57)) +
        (-screened_rates(k_p_fe56__co57)*Y(Fe56)*Y(H1)*state.rho + screened_rates(k_co57__p_fe56)*Y(Co57)) +
        (-screened_rates(k_he4_fe56__ni60)*Y(Fe56)*Y(He4)*state.rho + screened_rates(k_ni60__he4_fe56)*Y(Ni60)) +
        (screened_rates(k_he4_mn53__p_fe56)*Y(He4)*Y(Mn53)*state.rho + -screened_rates(k_p_fe56__he4_mn53)*Y(Fe56)*Y(H1)*state.rho) +
        (-screened_rates(k_n_fe56__he4_cr53)*Y(Fe56)*Y(N)*state.rho + screened_rates(k_he4_cr53__n_fe56)*Y(Cr53)*Y(He4)*state.rho) +
        (screened_rates(k_n_co56__p_fe56)*Y(Co56)*Y(N)*state.rho + -screened_rates(k_p_fe56__n_co56)*Y(Fe56)*Y(H1)*state.rho) +
        (screened_rates(k_p_co59__he4_fe56)*Y(Co59)*Y(H1)*state.rho + -screened_rates(k_he4_fe56__p_co59)*Y(Fe56)*Y(He4)*state.rho) +
        (screened_rates(k_n_ni59__he4_fe56)*Y(N)*Y(Ni59)*state.rho + -screened_rates(k_he4_fe56__n_ni59)*Y(Fe56)*Y(He4)*state.rho);

    ydot_nuc(Fe57) =
        screened_rates(k_co57__fe57__weak__wc12)*Y(Co57) +
        (screened_rates(k_he4_cr53__fe57)*Y(Cr53)*Y(He4)*state.rho + -screened_rates(k_fe57__he4_cr53)*Y(Fe57)) +
        (screened_rates(k_n_fe56__fe57)*Y(Fe56)*Y(N)*state.rho + -screened_rates(k_fe57__n_fe56)*Y(Fe57)) +
        (-screened_rates(k_n_fe57__fe58)*Y(Fe57)*Y(N)*state.rho + screened_rates(k_fe58__n_fe57)*Y(Fe58)) +
        (-screened_rates(k_p_fe57__co58)*Y(Fe57)*Y(H1)*state.rho + screened_rates(k_co58__p_fe57)*Y(Co58)) +
        (-screened_rates(k_he4_fe57__ni61)*Y(Fe57)*Y(He4)*state.rho + screened_rates(k_ni61__he4_fe57)*Y(Ni61)) +
        (-screened_rates(k_n_fe57__he4_cr54)*Y(Fe57)*Y(N)*state.rho + screened_rates(k_he4_cr54__n_fe57)*Y(Cr54)*Y(He4)*state.rho) +
        (-screened_rates(k_p_fe57__he4_mn54)*Y(Fe57)*Y(H1)*state.rho + screened_rates(k_he4_mn54__p_fe57)*Y(He4)*Y(Mn54)*state.rho) +
        (screened_rates(k_n_co57__p_fe57)*Y(Co57)*Y(N)*state.rho + -screened_rates(k_p_fe57__n_co57)*Y(Fe57)*Y(H1)*state.rho) +
        (screened_rates(k_n_ni60__he4_fe57)*Y(N)*Y(Ni60)*state.rho + -screened_rates(k_he4_fe57__n_ni60)*Y(Fe57)*Y(He4)*state.rho);

    ydot_nuc(Fe58) =
        screened_rates(k_co58__fe58__weak__wc12)*Y(Co58) +
        (screened_rates(k_he4_cr54__fe58)*Y(Cr54)*Y(He4)*state.rho + -screened_rates(k_fe58__he4_cr54)*Y(Fe58)) +
        (screened_rates(k_n_fe57__fe58)*Y(Fe57)*Y(N)*state.rho + -screened_rates(k_fe58__n_fe57)*Y(Fe58)) +
        (-screened_rates(k_p_fe58__co59)*Y(Fe58)*Y(H1)*state.rho + screened_rates(k_co59__p_fe58)*Y(Co59)) +
        (-screened_rates(k_he4_fe58__ni62)*Y(Fe58)*Y(He4)*state.rho + screened_rates(k_ni62__he4_fe58)*Y(Ni62)) +
        (-screened_rates(k_p_fe58__he4_mn55)*Y(Fe58)*Y(H1)*state.rho + screened_rates(k_he4_mn55__p_fe58)*Y(He4)*Y(Mn55)*state.rho) +
        (screened_rates(k_n_co58__p_fe58)*Y(Co58)*Y(N)*state.rho + -screened_rates(k_p_fe58__n_co58)*Y(Fe58)*Y(H1)*state.rho) +
        (screened_rates(k_n_ni61__he4_fe58)*Y(N)*Y(Ni61)*state.rho + -screened_rates(k_he4_fe58__n_ni61)*Y(Fe58)*Y(He4)*state.rho);

    ydot_nuc(Co53) =
        -screened_rates(k_co53__fe53__weak__wc12)*Y(Co53) +
        (screened_rates(k_p_fe52__co53)*Y(Fe52)*Y(H1)*state.rho + -screened_rates(k_co53__p_fe52)*Y(Co53)) +
        (-screened_rates(k_n_co53__co54)*Y(Co53)*Y(N)*state.rho + screened_rates(k_co54__n_co53)*Y(Co54)) +
        (-screened_rates(k_he4_co53__cu57)*Y(Co53)*Y(He4)*state.rho + screened_rates(k_cu57__he4_co53)*Y(Cu57)) +
        (-screened_rates(k_n_co53__p_fe53)*Y(Co53)*Y(N)*state.rho + screened_rates(k_p_fe53__n_co53)*Y(Fe53)*Y(H1)*state.rho) +
        (-screened_rates(k_n_co53__he4_mn50)*Y(Co53)*Y(N)*state.rho + screened_rates(k_he4_mn50__n_co53)*Y(He4)*Y(Mn50)*state.rho) +
        (-screened_rates(k_he4_co53__p_ni56)*Y(Co53)*Y(He4)*state.rho + screened_rates(k_p_ni56__he4_co53)*Y(Ni56)*Y(H1)*state.rho);

    ydot_nuc(Co54) =
        -screened_rates(k_co54__fe54__weak__wc12)*Y(Co54) +
        (screened_rates(k_he4_mn50__co54)*Y(He4)*Y(Mn50)*state.rho + -screened_rates(k_co54__he4_mn50)*Y(Co54)) +
        (screened_rates(k_p_fe53__co54)*Y(Fe53)*Y(H1)*state.rho + -screened_rates(k_co54__p_fe53)*Y(Co54)) +
        (screened_rates(k_n_co53__co54)*Y(Co53)*Y(N)*state.rho + -screened_rates(k_co54__n_co53)*Y(Co54)) +
        (-screened_rates(k_n_co54__co55)*Y(Co54)*Y(N)*state.rho + screened_rates(k_co55__n_co54)*Y(Co55)) +
        (-screened_rates(k_he4_co54__cu58)*Y(Co54)*Y(He4)*state.rho + screened_rates(k_cu58__he4_co54)*Y(Cu58)) +
        (-screened_rates(k_n_co54__p_fe54)*Y(Co54)*Y(N)*state.rho + screened_rates(k_p_fe54__n_co54)*Y(Fe54)*Y(H1)*state.rho) +
        (-screened_rates(k_n_co54__he4_mn51)*Y(Co54)*Y(N)*state.rho + screened_rates(k_he4_mn51__n_co54)*Y(He4)*Y(Mn51)*state.rho) +
        (-screened_rates(k_he4_co54__p_ni57)*Y(Co54)*Y(He4)*state.rho + screened_rates(k_p_ni57__he4_co54)*Y(Ni57)*Y(H1)*state.rho) +
        (screened_rates(k_n_cu57__he4_co54)*Y(Cu57)*Y(N)*state.rho + -screened_rates(k_he4_co54__n_cu57)*Y(Co54)*Y(He4)*state.rho);

    ydot_nuc(Co55) =
        -screened_rates(k_co55__fe55__weak__wc12)*Y(Co55) +
        (screened_rates(k_he4_mn51__co55)*Y(He4)*Y(Mn51)*state.rho + -screened_rates(k_co55__he4_mn51)*Y(Co55)) +
        (screened_rates(k_p_fe54__co55)*Y(Fe54)*Y(H1)*state.rho + -screened_rates(k_co55__p_fe54)*Y(Co55)) +
        (screened_rates(k_n_co54__co55)*Y(Co54)*Y(N)*state.rho + -screened_rates(k_co55__n_co54)*Y(Co55)) +
        (-screened_rates(k_n_co55__co56)*Y(Co55)*Y(N)*state.rho + screened_rates(k_co56__n_co55)*Y(Co56)) +
        (-screened_rates(k_p_co55__ni56)*Y(Co55)*Y(H1)*state.rho + screened_rates(k_ni56__p_co55)*Y(Ni56)) +
        (-screened_rates(k_he4_co55__cu59)*Y(Co55)*Y(He4)*state.rho + screened_rates(k_cu59__he4_co55)*Y(Cu59)) +
        (screened_rates(k_he4_fe52__p_co55)*Y(Fe52)*Y(He4)*state.rho + -screened_rates(k_p_co55__he4_fe52)*Y(Co55)*Y(H1)*state.rho) +
        (-screened_rates(k_n_co55__p_fe55)*Y(Co55)*Y(N)*state.rho + screened_rates(k_p_fe55__n_co55)*Y(Fe55)*Y(H1)*state.rho) +
        (-screened_rates(k_n_co55__he4_mn52)*Y(Co55)*Y(N)*state.rho + screened_rates(k_he4_mn52__n_co55)*Y(He4)*Y(Mn52)*state.rho) +
        (-screened_rates(k_he4_co55__p_ni58)*Y(Co55)*Y(He4)*state.rho + screened_rates(k_p_ni58__he4_co55)*Y(Ni58)*Y(H1)*state.rho) +
        (screened_rates(k_n_cu58__he4_co55)*Y(Cu58)*Y(N)*state.rho + -screened_rates(k_he4_co55__n_cu58)*Y(Co55)*Y(He4)*state.rho);

    ydot_nuc(Co56) =
        -screened_rates(k_co56__fe56__weak__wc12)*Y(Co56) +
        screened_rates(k_ni56__co56__weak__wc12)*Y(Ni56) +
        (screened_rates(k_he4_mn52__co56)*Y(He4)*Y(Mn52)*state.rho + -screened_rates(k_co56__he4_mn52)*Y(Co56)) +
        (screened_rates(k_p_fe55__co56)*Y(Fe55)*Y(H1)*state.rho + -screened_rates(k_co56__p_fe55)*Y(Co56)) +
        (screened_rates(k_n_co55__co56)*Y(Co55)*Y(N)*state.rho + -screened_rates(k_co56__n_co55)*Y(Co56)) +
        (-screened_rates(k_n_co56__co57)*Y(Co56)*Y(N)*state.rho + screened_rates(k_co57__n_co56)*Y(Co57)) +
        (-screened_rates(k_p_co56__ni57)*Y(Co56)*Y(H1)*state.rho + screened_rates(k_ni57__p_co56)*Y(Ni57)) +
        (-screened_rates(k_he4_co56__cu60)*Y(Co56)*Y(He4)*state.rho + screened_rates(k_cu60__he4_co56)*Y(Cu60)) +
        (screened_rates(k_he4_fe53__p_co56)*Y(Fe53)*Y(He4)*state.rho + -screened_rates(k_p_co56__he4_fe53)*Y(Co56)*Y(H1)*state.rho) +
        (-screened_rates(k_n_co56__p_fe56)*Y(Co56)*Y(N)*state.rho + screened_rates(k_p_fe56__n_co56)*Y(Fe56)*Y(H1)*state.rho) +
        (-screened_rates(k_n_co56__he4_mn53)*Y(Co56)*Y(N)*state.rho + screened_rates(k_he4_mn53__n_co56)*Y(He4)*Y(Mn53)*state.rho) +
        (-screened_rates(k_he4_co56__p_ni59)*Y(Co56)*Y(He4)*state.rho + screened_rates(k_p_ni59__he4_co56)*Y(Ni59)*Y(H1)*state.rho) +
        (screened_rates(k_n_ni56__p_co56)*Y(N)*Y(Ni56)*state.rho + -screened_rates(k_p_co56__n_ni56)*Y(Co56)*Y(H1)*state.rho) +
        (screened_rates(k_n_cu59__he4_co56)*Y(Cu59)*Y(N)*state.rho + -screened_rates(k_he4_co56__n_cu59)*Y(Co56)*Y(He4)*state.rho);

    ydot_nuc(Co57) =
        -screened_rates(k_co57__fe57__weak__wc12)*Y(Co57) +
        screened_rates(k_ni57__co57__weak__wc12)*Y(Ni57) +
        (screened_rates(k_he4_mn53__co57)*Y(He4)*Y(Mn53)*state.rho + -screened_rates(k_co57__he4_mn53)*Y(Co57)) +
        (screened_rates(k_p_fe56__co57)*Y(Fe56)*Y(H1)*state.rho + -screened_rates(k_co57__p_fe56)*Y(Co57)) +
        (screened_rates(k_n_co56__co57)*Y(Co56)*Y(N)*state.rho + -screened_rates(k_co57__n_co56)*Y(Co57)) +
        (-screened_rates(k_n_co57__co58)*Y(Co57)*Y(N)*state.rho + screened_rates(k_co58__n_co57)*Y(Co58)) +
        (-screened_rates(k_p_co57__ni58)*Y(Co57)*Y(H1)*state.rho + screened_rates(k_ni58__p_co57)*Y(Ni58)) +
        (-screened_rates(k_he4_co57__cu61)*Y(Co57)*Y(He4)*state.rho + screened_rates(k_cu61__he4_co57)*Y(Cu61)) +
        (-screened_rates(k_n_co57__p_fe57)*Y(Co57)*Y(N)*state.rho + screened_rates(k_p_fe57__n_co57)*Y(Fe57)*Y(H1)*state.rho) +
        (-screened_rates(k_n_co57__he4_mn54)*Y(Co57)*Y(N)*state.rho + screened_rates(k_he4_mn54__n_co57)*Y(He4)*Y(Mn54)*state.rho) +
        (-screened_rates(k_p_co57__he4_fe54)*Y(Co57)*Y(H1)*state.rho + screened_rates(k_he4_fe54__p_co57)*Y(Fe54)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_co57__p_ni60)*Y(Co57)*Y(He4)*state.rho + screened_rates(k_p_ni60__he4_co57)*Y(Ni60)*Y(H1)*state.rho) +
        (screened_rates(k_n_ni57__p_co57)*Y(N)*Y(Ni57)*state.rho + -screened_rates(k_p_co57__n_ni57)*Y(Co57)*Y(H1)*state.rho) +
        (screened_rates(k_n_cu60__he4_co57)*Y(Cu60)*Y(N)*state.rho + -screened_rates(k_he4_co57__n_cu60)*Y(Co57)*Y(He4)*state.rho);

    ydot_nuc(Co58) =
        -screened_rates(k_co58__fe58__weak__wc12)*Y(Co58) +
        (screened_rates(k_he4_mn54__co58)*Y(He4)*Y(Mn54)*state.rho + -screened_rates(k_co58__he4_mn54)*Y(Co58)) +
        (screened_rates(k_p_fe57__co58)*Y(Fe57)*Y(H1)*state.rho + -screened_rates(k_co58__p_fe57)*Y(Co58)) +
        (screened_rates(k_n_co57__co58)*Y(Co57)*Y(N)*state.rho + -screened_rates(k_co58__n_co57)*Y(Co58)) +
        (-screened_rates(k_n_co58__co59)*Y(Co58)*Y(N)*state.rho + screened_rates(k_co59__n_co58)*Y(Co59)) +
        (-screened_rates(k_p_co58__ni59)*Y(Co58)*Y(H1)*state.rho + screened_rates(k_ni59__p_co58)*Y(Ni59)) +
        (-screened_rates(k_he4_co58__cu62)*Y(Co58)*Y(He4)*state.rho + screened_rates(k_cu62__he4_co58)*Y(Cu62)) +
        (-screened_rates(k_n_co58__p_fe58)*Y(Co58)*Y(N)*state.rho + screened_rates(k_p_fe58__n_co58)*Y(Fe58)*Y(H1)*state.rho) +
        (-screened_rates(k_n_co58__he4_mn55)*Y(Co58)*Y(N)*state.rho + screened_rates(k_he4_mn55__n_co58)*Y(He4)*Y(Mn55)*state.rho) +
        (-screened_rates(k_p_co58__he4_fe55)*Y(Co58)*Y(H1)*state.rho + screened_rates(k_he4_fe55__p_co58)*Y(Fe55)*Y(He4)*state.rho) +
        (screened_rates(k_n_ni58__p_co58)*Y(N)*Y(Ni58)*state.rho + -screened_rates(k_p_co58__n_ni58)*Y(Co58)*Y(H1)*state.rho) +
        (screened_rates(k_p_ni61__he4_co58)*Y(Ni61)*Y(H1)*state.rho + -screened_rates(k_he4_co58__p_ni61)*Y(Co58)*Y(He4)*state.rho) +
        (screened_rates(k_n_cu61__he4_co58)*Y(Cu61)*Y(N)*state.rho + -screened_rates(k_he4_co58__n_cu61)*Y(Co58)*Y(He4)*state.rho);

    ydot_nuc(Co59) =
        screened_rates(k_ni59__co59__weak__wc12)*Y(Ni59) +
        (screened_rates(k_he4_mn55__co59)*Y(He4)*Y(Mn55)*state.rho + -screened_rates(k_co59__he4_mn55)*Y(Co59)) +
        (screened_rates(k_p_fe58__co59)*Y(Fe58)*Y(H1)*state.rho + -screened_rates(k_co59__p_fe58)*Y(Co59)) +
        (screened_rates(k_n_co58__co59)*Y(Co58)*Y(N)*state.rho + -screened_rates(k_co59__n_co58)*Y(Co59)) +
        (-screened_rates(k_p_co59__ni60)*Y(Co59)*Y(H1)*state.rho + screened_rates(k_ni60__p_co59)*Y(Ni60)) +
        (-screened_rates(k_he4_co59__cu63)*Y(Co59)*Y(He4)*state.rho + screened_rates(k_cu63__he4_co59)*Y(Cu63)) +
        (-screened_rates(k_p_co59__he4_fe56)*Y(Co59)*Y(H1)*state.rho + screened_rates(k_he4_fe56__p_co59)*Y(Fe56)*Y(He4)*state.rho) +
        (screened_rates(k_n_ni59__p_co59)*Y(N)*Y(Ni59)*state.rho + -screened_rates(k_p_co59__n_ni59)*Y(Co59)*Y(H1)*state.rho) +
        (screened_rates(k_p_ni62__he4_co59)*Y(Ni62)*Y(H1)*state.rho + -screened_rates(k_he4_co59__p_ni62)*Y(Co59)*Y(He4)*state.rho) +
        (screened_rates(k_n_cu62__he4_co59)*Y(Cu62)*Y(N)*state.rho + -screened_rates(k_he4_co59__n_cu62)*Y(Co59)*Y(He4)*state.rho);

    ydot_nuc(Ni56) =
        -screened_rates(k_ni56__co56__weak__wc12)*Y(Ni56) +
        (screened_rates(k_he4_fe52__ni56)*Y(Fe52)*Y(He4)*state.rho + -screened_rates(k_ni56__he4_fe52)*Y(Ni56)) +
        (screened_rates(k_p_co55__ni56)*Y(Co55)*Y(H1)*state.rho + -screened_rates(k_ni56__p_co55)*Y(Ni56)) +
        (-screened_rates(k_n_ni56__ni57)*Y(N)*Y(Ni56)*state.rho + screened_rates(k_ni57__n_ni56)*Y(Ni57)) +
        (-screened_rates(k_p_ni56__cu57)*Y(Ni56)*Y(H1)*state.rho + screened_rates(k_cu57__p_ni56)*Y(Cu57)) +
        (-screened_rates(k_he4_ni56__zn60)*Y(He4)*Y(Ni56)*state.rho + screened_rates(k_zn60__he4_ni56)*Y(Zn60)) +
        (screened_rates(k_he4_co53__p_ni56)*Y(Co53)*Y(He4)*state.rho + -screened_rates(k_p_ni56__he4_co53)*Y(Ni56)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ni56__p_co56)*Y(N)*Y(Ni56)*state.rho + screened_rates(k_p_co56__n_ni56)*Y(Co56)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ni56__he4_fe53)*Y(N)*Y(Ni56)*state.rho + screened_rates(k_he4_fe53__n_ni56)*Y(Fe53)*Y(He4)*state.rho) +
        (screened_rates(k_p_cu59__he4_ni56)*Y(Cu59)*Y(H1)*state.rho + -screened_rates(k_he4_ni56__p_cu59)*Y(He4)*Y(Ni56)*state.rho) +
        (screened_rates(k_n_zn59__he4_ni56)*Y(N)*Y(Zn59)*state.rho + -screened_rates(k_he4_ni56__n_zn59)*Y(He4)*Y(Ni56)*state.rho);

    ydot_nuc(Ni57) =
        -screened_rates(k_ni57__co57__weak__wc12)*Y(Ni57) +
        screened_rates(k_cu57__ni57__weak__wc12)*Y(Cu57) +
        (screened_rates(k_he4_fe53__ni57)*Y(Fe53)*Y(He4)*state.rho + -screened_rates(k_ni57__he4_fe53)*Y(Ni57)) +
        (screened_rates(k_p_co56__ni57)*Y(Co56)*Y(H1)*state.rho + -screened_rates(k_ni57__p_co56)*Y(Ni57)) +
        (screened_rates(k_n_ni56__ni57)*Y(N)*Y(Ni56)*state.rho + -screened_rates(k_ni57__n_ni56)*Y(Ni57)) +
        (-screened_rates(k_n_ni57__ni58)*Y(N)*Y(Ni57)*state.rho + screened_rates(k_ni58__n_ni57)*Y(Ni58)) +
        (-screened_rates(k_p_ni57__cu58)*Y(Ni57)*Y(H1)*state.rho + screened_rates(k_cu58__p_ni57)*Y(Cu58)) +
        (-screened_rates(k_he4_ni57__zn61)*Y(He4)*Y(Ni57)*state.rho + screened_rates(k_zn61__he4_ni57)*Y(Zn61)) +
        (screened_rates(k_he4_co54__p_ni57)*Y(Co54)*Y(He4)*state.rho + -screened_rates(k_p_ni57__he4_co54)*Y(Ni57)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ni57__p_co57)*Y(N)*Y(Ni57)*state.rho + screened_rates(k_p_co57__n_ni57)*Y(Co57)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ni57__he4_fe54)*Y(N)*Y(Ni57)*state.rho + screened_rates(k_he4_fe54__n_ni57)*Y(Fe54)*Y(He4)*state.rho) +
        (screened_rates(k_n_cu57__p_ni57)*Y(Cu57)*Y(N)*state.rho + -screened_rates(k_p_ni57__n_cu57)*Y(Ni57)*Y(H1)*state.rho) +
        (screened_rates(k_p_cu60__he4_ni57)*Y(Cu60)*Y(H1)*state.rho + -screened_rates(k_he4_ni57__p_cu60)*Y(He4)*Y(Ni57)*state.rho) +
        (screened_rates(k_n_zn60__he4_ni57)*Y(N)*Y(Zn60)*state.rho + -screened_rates(k_he4_ni57__n_zn60)*Y(He4)*Y(Ni57)*state.rho);

    ydot_nuc(Ni58) =
        screened_rates(k_cu58__ni58__weak__wc12)*Y(Cu58) +
        screened_rates(k_zn59__p_ni58__weak__wc12)*Y(Zn59) +
        (screened_rates(k_he4_fe54__ni58)*Y(Fe54)*Y(He4)*state.rho + -screened_rates(k_ni58__he4_fe54)*Y(Ni58)) +
        (screened_rates(k_p_co57__ni58)*Y(Co57)*Y(H1)*state.rho + -screened_rates(k_ni58__p_co57)*Y(Ni58)) +
        (screened_rates(k_n_ni57__ni58)*Y(N)*Y(Ni57)*state.rho + -screened_rates(k_ni58__n_ni57)*Y(Ni58)) +
        (-screened_rates(k_n_ni58__ni59)*Y(N)*Y(Ni58)*state.rho + screened_rates(k_ni59__n_ni58)*Y(Ni59)) +
        (-screened_rates(k_p_ni58__cu59)*Y(Ni58)*Y(H1)*state.rho + screened_rates(k_cu59__p_ni58)*Y(Cu59)) +
        (-screened_rates(k_he4_ni58__zn62)*Y(He4)*Y(Ni58)*state.rho + screened_rates(k_zn62__he4_ni58)*Y(Zn62)) +
        (screened_rates(k_he4_co55__p_ni58)*Y(Co55)*Y(He4)*state.rho + -screened_rates(k_p_ni58__he4_co55)*Y(Ni58)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ni58__p_co58)*Y(N)*Y(Ni58)*state.rho + screened_rates(k_p_co58__n_ni58)*Y(Co58)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ni58__he4_fe55)*Y(N)*Y(Ni58)*state.rho + screened_rates(k_he4_fe55__n_ni58)*Y(Fe55)*Y(He4)*state.rho) +
        (screened_rates(k_n_cu58__p_ni58)*Y(Cu58)*Y(N)*state.rho + -screened_rates(k_p_ni58__n_cu58)*Y(Ni58)*Y(H1)*state.rho) +
        (screened_rates(k_p_cu61__he4_ni58)*Y(Cu61)*Y(H1)*state.rho + -screened_rates(k_he4_ni58__p_cu61)*Y(He4)*Y(Ni58)*state.rho) +
        (screened_rates(k_n_zn61__he4_ni58)*Y(N)*Y(Zn61)*state.rho + -screened_rates(k_he4_ni58__n_zn61)*Y(He4)*Y(Ni58)*state.rho);

    ydot_nuc(Ni59) =
        -screened_rates(k_ni59__co59__weak__wc12)*Y(Ni59) +
        screened_rates(k_cu59__ni59__weak__wc12)*Y(Cu59) +
        (screened_rates(k_he4_fe55__ni59)*Y(Fe55)*Y(He4)*state.rho + -screened_rates(k_ni59__he4_fe55)*Y(Ni59)) +
        (screened_rates(k_p_co58__ni59)*Y(Co58)*Y(H1)*state.rho + -screened_rates(k_ni59__p_co58)*Y(Ni59)) +
        (screened_rates(k_n_ni58__ni59)*Y(N)*Y(Ni58)*state.rho + -screened_rates(k_ni59__n_ni58)*Y(Ni59)) +
        (-screened_rates(k_n_ni59__ni60)*Y(N)*Y(Ni59)*state.rho + screened_rates(k_ni60__n_ni59)*Y(Ni60)) +
        (-screened_rates(k_p_ni59__cu60)*Y(Ni59)*Y(H1)*state.rho + screened_rates(k_cu60__p_ni59)*Y(Cu60)) +
        (-screened_rates(k_he4_ni59__zn63)*Y(He4)*Y(Ni59)*state.rho + screened_rates(k_zn63__he4_ni59)*Y(Zn63)) +
        (screened_rates(k_he4_co56__p_ni59)*Y(Co56)*Y(He4)*state.rho + -screened_rates(k_p_ni59__he4_co56)*Y(Ni59)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ni59__p_co59)*Y(N)*Y(Ni59)*state.rho + screened_rates(k_p_co59__n_ni59)*Y(Co59)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ni59__he4_fe56)*Y(N)*Y(Ni59)*state.rho + screened_rates(k_he4_fe56__n_ni59)*Y(Fe56)*Y(He4)*state.rho) +
        (screened_rates(k_n_cu59__p_ni59)*Y(Cu59)*Y(N)*state.rho + -screened_rates(k_p_ni59__n_cu59)*Y(Ni59)*Y(H1)*state.rho) +
        (screened_rates(k_p_cu62__he4_ni59)*Y(Cu62)*Y(H1)*state.rho + -screened_rates(k_he4_ni59__p_cu62)*Y(He4)*Y(Ni59)*state.rho) +
        (screened_rates(k_n_zn62__he4_ni59)*Y(N)*Y(Zn62)*state.rho + -screened_rates(k_he4_ni59__n_zn62)*Y(He4)*Y(Ni59)*state.rho);

    ydot_nuc(Ni60) =
        screened_rates(k_cu60__ni60__weak__wc12)*Y(Cu60) +
        (screened_rates(k_he4_fe56__ni60)*Y(Fe56)*Y(He4)*state.rho + -screened_rates(k_ni60__he4_fe56)*Y(Ni60)) +
        (screened_rates(k_p_co59__ni60)*Y(Co59)*Y(H1)*state.rho + -screened_rates(k_ni60__p_co59)*Y(Ni60)) +
        (screened_rates(k_n_ni59__ni60)*Y(N)*Y(Ni59)*state.rho + -screened_rates(k_ni60__n_ni59)*Y(Ni60)) +
        (-screened_rates(k_n_ni60__ni61)*Y(N)*Y(Ni60)*state.rho + screened_rates(k_ni61__n_ni60)*Y(Ni61)) +
        (-screened_rates(k_p_ni60__cu61)*Y(Ni60)*Y(H1)*state.rho + screened_rates(k_cu61__p_ni60)*Y(Cu61)) +
        (-screened_rates(k_he4_ni60__zn64)*Y(He4)*Y(Ni60)*state.rho + screened_rates(k_zn64__he4_ni60)*Y(Zn64)) +
        (screened_rates(k_he4_co57__p_ni60)*Y(Co57)*Y(He4)*state.rho + -screened_rates(k_p_ni60__he4_co57)*Y(Ni60)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ni60__he4_fe57)*Y(N)*Y(Ni60)*state.rho + screened_rates(k_he4_fe57__n_ni60)*Y(Fe57)*Y(He4)*state.rho) +
        (screened_rates(k_n_cu60__p_ni60)*Y(Cu60)*Y(N)*state.rho + -screened_rates(k_p_ni60__n_cu60)*Y(Ni60)*Y(H1)*state.rho) +
        (screened_rates(k_p_cu63__he4_ni60)*Y(Cu63)*Y(H1)*state.rho + -screened_rates(k_he4_ni60__p_cu63)*Y(He4)*Y(Ni60)*state.rho) +
        (screened_rates(k_n_zn63__he4_ni60)*Y(N)*Y(Zn63)*state.rho + -screened_rates(k_he4_ni60__n_zn63)*Y(He4)*Y(Ni60)*state.rho);

    ydot_nuc(Ni61) =
        screened_rates(k_cu61__ni61__weak__wc12)*Y(Cu61) +
        (screened_rates(k_he4_fe57__ni61)*Y(Fe57)*Y(He4)*state.rho + -screened_rates(k_ni61__he4_fe57)*Y(Ni61)) +
        (screened_rates(k_n_ni60__ni61)*Y(N)*Y(Ni60)*state.rho + -screened_rates(k_ni61__n_ni60)*Y(Ni61)) +
        (-screened_rates(k_n_ni61__ni62)*Y(N)*Y(Ni61)*state.rho + screened_rates(k_ni62__n_ni61)*Y(Ni62)) +
        (-screened_rates(k_p_ni61__cu62)*Y(Ni61)*Y(H1)*state.rho + screened_rates(k_cu62__p_ni61)*Y(Cu62)) +
        (-screened_rates(k_he4_ni61__zn65)*Y(He4)*Y(Ni61)*state.rho + screened_rates(k_zn65__he4_ni61)*Y(Zn65)) +
        (-screened_rates(k_n_ni61__he4_fe58)*Y(N)*Y(Ni61)*state.rho + screened_rates(k_he4_fe58__n_ni61)*Y(Fe58)*Y(He4)*state.rho) +
        (-screened_rates(k_p_ni61__he4_co58)*Y(Ni61)*Y(H1)*state.rho + screened_rates(k_he4_co58__p_ni61)*Y(Co58)*Y(He4)*state.rho) +
        (screened_rates(k_n_cu61__p_ni61)*Y(Cu61)*Y(N)*state.rho + -screened_rates(k_p_ni61__n_cu61)*Y(Ni61)*Y(H1)*state.rho) +
        (screened_rates(k_p_cu64__he4_ni61)*Y(Cu64)*Y(H1)*state.rho + -screened_rates(k_he4_ni61__p_cu64)*Y(He4)*Y(Ni61)*state.rho) +
        (screened_rates(k_n_zn64__he4_ni61)*Y(N)*Y(Zn64)*state.rho + -screened_rates(k_he4_ni61__n_zn64)*Y(He4)*Y(Ni61)*state.rho);

    ydot_nuc(Ni62) =
        screened_rates(k_cu62__ni62__weak__wc12)*Y(Cu62) +
        (screened_rates(k_he4_fe58__ni62)*Y(Fe58)*Y(He4)*state.rho + -screened_rates(k_ni62__he4_fe58)*Y(Ni62)) +
        (screened_rates(k_n_ni61__ni62)*Y(N)*Y(Ni61)*state.rho + -screened_rates(k_ni62__n_ni61)*Y(Ni62)) +
        (-screened_rates(k_n_ni62__ni63)*Y(N)*Y(Ni62)*state.rho + screened_rates(k_ni63__n_ni62)*Y(Ni63)) +
        (-screened_rates(k_p_ni62__cu63)*Y(Ni62)*Y(H1)*state.rho + screened_rates(k_cu63__p_ni62)*Y(Cu63)) +
        (-screened_rates(k_he4_ni62__zn66)*Y(He4)*Y(Ni62)*state.rho + screened_rates(k_zn66__he4_ni62)*Y(Zn66)) +
        (-screened_rates(k_p_ni62__he4_co59)*Y(Ni62)*Y(H1)*state.rho + screened_rates(k_he4_co59__p_ni62)*Y(Co59)*Y(He4)*state.rho) +
        (screened_rates(k_n_cu62__p_ni62)*Y(Cu62)*Y(N)*state.rho + -screened_rates(k_p_ni62__n_cu62)*Y(Ni62)*Y(H1)*state.rho) +
        (screened_rates(k_p_cu65__he4_ni62)*Y(Cu65)*Y(H1)*state.rho + -screened_rates(k_he4_ni62__p_cu65)*Y(He4)*Y(Ni62)*state.rho) +
        (screened_rates(k_n_zn65__he4_ni62)*Y(N)*Y(Zn65)*state.rho + -screened_rates(k_he4_ni62__n_zn65)*Y(He4)*Y(Ni62)*state.rho);

    ydot_nuc(Ni63) =
        -screened_rates(k_ni63__cu63__weak__wc12)*Y(Ni63) +
        (screened_rates(k_n_ni62__ni63)*Y(N)*Y(Ni62)*state.rho + -screened_rates(k_ni63__n_ni62)*Y(Ni63)) +
        (-screened_rates(k_n_ni63__ni64)*Y(N)*Y(Ni63)*state.rho + screened_rates(k_ni64__n_ni63)*Y(Ni64)) +
        (-screened_rates(k_p_ni63__cu64)*Y(Ni63)*Y(H1)*state.rho + screened_rates(k_cu64__p_ni63)*Y(Cu64)) +
        (screened_rates(k_n_cu63__p_ni63)*Y(Cu63)*Y(N)*state.rho + -screened_rates(k_p_ni63__n_cu63)*Y(Ni63)*Y(H1)*state.rho) +
        (screened_rates(k_n_zn66__he4_ni63)*Y(N)*Y(Zn66)*state.rho + -screened_rates(k_he4_ni63__n_zn66)*Y(He4)*Y(Ni63)*state.rho);

    ydot_nuc(Ni64) =
        screened_rates(k_cu64__ni64__weak__wc12)*Y(Cu64) +
        (screened_rates(k_n_ni63__ni64)*Y(N)*Y(Ni63)*state.rho + -screened_rates(k_ni64__n_ni63)*Y(Ni64)) +
        (-screened_rates(k_p_ni64__cu65)*Y(Ni64)*Y(H1)*state.rho + screened_rates(k_cu65__p_ni64)*Y(Cu65)) +
        (screened_rates(k_n_cu64__p_ni64)*Y(Cu64)*Y(N)*state.rho + -screened_rates(k_p_ni64__n_cu64)*Y(Ni64)*Y(H1)*state.rho);

    ydot_nuc(Cu57) =
        -screened_rates(k_cu57__ni57__weak__wc12)*Y(Cu57) +
        (screened_rates(k_he4_co53__cu57)*Y(Co53)*Y(He4)*state.rho + -screened_rates(k_cu57__he4_co53)*Y(Cu57)) +
        (screened_rates(k_p_ni56__cu57)*Y(Ni56)*Y(H1)*state.rho + -screened_rates(k_cu57__p_ni56)*Y(Cu57)) +
        (-screened_rates(k_n_cu57__cu58)*Y(Cu57)*Y(N)*state.rho + screened_rates(k_cu58__n_cu57)*Y(Cu58)) +
        (-screened_rates(k_n_cu57__p_ni57)*Y(Cu57)*Y(N)*state.rho + screened_rates(k_p_ni57__n_cu57)*Y(Ni57)*Y(H1)*state.rho) +
        (-screened_rates(k_n_cu57__he4_co54)*Y(Cu57)*Y(N)*state.rho + screened_rates(k_he4_co54__n_cu57)*Y(Co54)*Y(He4)*state.rho) +
        (-screened_rates(k_he4_cu57__p_zn60)*Y(Cu57)*Y(He4)*state.rho + screened_rates(k_p_zn60__he4_cu57)*Y(H1)*Y(Zn60)*state.rho);

    ydot_nuc(Cu58) =
        -screened_rates(k_cu58__ni58__weak__wc12)*Y(Cu58) +
        (screened_rates(k_he4_co54__cu58)*Y(Co54)*Y(He4)*state.rho + -screened_rates(k_cu58__he4_co54)*Y(Cu58)) +
        (screened_rates(k_p_ni57__cu58)*Y(Ni57)*Y(H1)*state.rho + -screened_rates(k_cu58__p_ni57)*Y(Cu58)) +
        (screened_rates(k_n_cu57__cu58)*Y(Cu57)*Y(N)*state.rho + -screened_rates(k_cu58__n_cu57)*Y(Cu58)) +
        (-screened_rates(k_n_cu58__cu59)*Y(Cu58)*Y(N)*state.rho + screened_rates(k_cu59__n_cu58)*Y(Cu59)) +
        (-screened_rates(k_p_cu58__zn59)*Y(Cu58)*Y(H1)*state.rho + screened_rates(k_zn59__p_cu58)*Y(Zn59)) +
        (-screened_rates(k_he4_cu58__ga62)*Y(Cu58)*Y(He4)*state.rho + screened_rates(k_ga62__he4_cu58)*Y(Ga62)) +
        (-screened_rates(k_n_cu58__p_ni58)*Y(Cu58)*Y(N)*state.rho + screened_rates(k_p_ni58__n_cu58)*Y(Ni58)*Y(H1)*state.rho) +
        (-screened_rates(k_n_cu58__he4_co55)*Y(Cu58)*Y(N)*state.rho + screened_rates(k_he4_co55__n_cu58)*Y(Co55)*Y(He4)*state.rho) +
        (screened_rates(k_p_zn61__he4_cu58)*Y(H1)*Y(Zn61)*state.rho + -screened_rates(k_he4_cu58__p_zn61)*Y(Cu58)*Y(He4)*state.rho);

    ydot_nuc(Cu59) =
        -screened_rates(k_cu59__ni59__weak__wc12)*Y(Cu59) +
        screened_rates(k_zn59__cu59__weak__wc12)*Y(Zn59) +
        (screened_rates(k_he4_co55__cu59)*Y(Co55)*Y(He4)*state.rho + -screened_rates(k_cu59__he4_co55)*Y(Cu59)) +
        (screened_rates(k_p_ni58__cu59)*Y(Ni58)*Y(H1)*state.rho + -screened_rates(k_cu59__p_ni58)*Y(Cu59)) +
        (screened_rates(k_n_cu58__cu59)*Y(Cu58)*Y(N)*state.rho + -screened_rates(k_cu59__n_cu58)*Y(Cu59)) +
        (-screened_rates(k_n_cu59__cu60)*Y(Cu59)*Y(N)*state.rho + screened_rates(k_cu60__n_cu59)*Y(Cu60)) +
        (-screened_rates(k_p_cu59__zn60)*Y(Cu59)*Y(H1)*state.rho + screened_rates(k_zn60__p_cu59)*Y(Zn60)) +
        (-screened_rates(k_he4_cu59__ga63)*Y(Cu59)*Y(He4)*state.rho + screened_rates(k_ga63__he4_cu59)*Y(Ga63)) +
        (-screened_rates(k_n_cu59__p_ni59)*Y(Cu59)*Y(N)*state.rho + screened_rates(k_p_ni59__n_cu59)*Y(Ni59)*Y(H1)*state.rho) +
        (-screened_rates(k_n_cu59__he4_co56)*Y(Cu59)*Y(N)*state.rho + screened_rates(k_he4_co56__n_cu59)*Y(Co56)*Y(He4)*state.rho) +
        (-screened_rates(k_p_cu59__he4_ni56)*Y(Cu59)*Y(H1)*state.rho + screened_rates(k_he4_ni56__p_cu59)*Y(He4)*Y(Ni56)*state.rho) +
        (screened_rates(k_n_zn59__p_cu59)*Y(N)*Y(Zn59)*state.rho + -screened_rates(k_p_cu59__n_zn59)*Y(Cu59)*Y(H1)*state.rho) +
        (screened_rates(k_p_zn62__he4_cu59)*Y(H1)*Y(Zn62)*state.rho + -screened_rates(k_he4_cu59__p_zn62)*Y(Cu59)*Y(He4)*state.rho) +
        (screened_rates(k_n_ga62__he4_cu59)*Y(Ga62)*Y(N)*state.rho + -screened_rates(k_he4_cu59__n_ga62)*Y(Cu59)*Y(He4)*state.rho);

    ydot_nuc(Cu60) =
        -screened_rates(k_cu60__ni60__weak__wc12)*Y(Cu60) +
        screened_rates(k_zn60__cu60__weak__wc12)*Y(Zn60) +
        (screened_rates(k_he4_co56__cu60)*Y(Co56)*Y(He4)*state.rho + -screened_rates(k_cu60__he4_co56)*Y(Cu60)) +
        (screened_rates(k_p_ni59__cu60)*Y(Ni59)*Y(H1)*state.rho + -screened_rates(k_cu60__p_ni59)*Y(Cu60)) +
        (screened_rates(k_n_cu59__cu60)*Y(Cu59)*Y(N)*state.rho + -screened_rates(k_cu60__n_cu59)*Y(Cu60)) +
        (-screened_rates(k_n_cu60__cu61)*Y(Cu60)*Y(N)*state.rho + screened_rates(k_cu61__n_cu60)*Y(Cu61)) +
        (-screened_rates(k_p_cu60__zn61)*Y(Cu60)*Y(H1)*state.rho + screened_rates(k_zn61__p_cu60)*Y(Zn61)) +
        (-screened_rates(k_he4_cu60__ga64)*Y(Cu60)*Y(He4)*state.rho + screened_rates(k_ga64__he4_cu60)*Y(Ga64)) +
        (-screened_rates(k_n_cu60__p_ni60)*Y(Cu60)*Y(N)*state.rho + screened_rates(k_p_ni60__n_cu60)*Y(Ni60)*Y(H1)*state.rho) +
        (-screened_rates(k_n_cu60__he4_co57)*Y(Cu60)*Y(N)*state.rho + screened_rates(k_he4_co57__n_cu60)*Y(Co57)*Y(He4)*state.rho) +
        (-screened_rates(k_p_cu60__he4_ni57)*Y(Cu60)*Y(H1)*state.rho + screened_rates(k_he4_ni57__p_cu60)*Y(He4)*Y(Ni57)*state.rho) +
        (screened_rates(k_n_zn60__p_cu60)*Y(N)*Y(Zn60)*state.rho + -screened_rates(k_p_cu60__n_zn60)*Y(Cu60)*Y(H1)*state.rho) +
        (screened_rates(k_p_zn63__he4_cu60)*Y(H1)*Y(Zn63)*state.rho + -screened_rates(k_he4_cu60__p_zn63)*Y(Cu60)*Y(He4)*state.rho) +
        (screened_rates(k_n_ga63__he4_cu60)*Y(Ga63)*Y(N)*state.rho + -screened_rates(k_he4_cu60__n_ga63)*Y(Cu60)*Y(He4)*state.rho);

    ydot_nuc(Cu61) =
        -screened_rates(k_cu61__ni61__weak__wc12)*Y(Cu61) +
        screened_rates(k_zn61__cu61__weak__wc12)*Y(Zn61) +
        (screened_rates(k_he4_co57__cu61)*Y(Co57)*Y(He4)*state.rho + -screened_rates(k_cu61__he4_co57)*Y(Cu61)) +
        (screened_rates(k_p_ni60__cu61)*Y(Ni60)*Y(H1)*state.rho + -screened_rates(k_cu61__p_ni60)*Y(Cu61)) +
        (screened_rates(k_n_cu60__cu61)*Y(Cu60)*Y(N)*state.rho + -screened_rates(k_cu61__n_cu60)*Y(Cu61)) +
        (-screened_rates(k_n_cu61__cu62)*Y(Cu61)*Y(N)*state.rho + screened_rates(k_cu62__n_cu61)*Y(Cu62)) +
        (-screened_rates(k_p_cu61__zn62)*Y(Cu61)*Y(H1)*state.rho + screened_rates(k_zn62__p_cu61)*Y(Zn62)) +
        (-screened_rates(k_n_cu61__p_ni61)*Y(Cu61)*Y(N)*state.rho + screened_rates(k_p_ni61__n_cu61)*Y(Ni61)*Y(H1)*state.rho) +
        (-screened_rates(k_n_cu61__he4_co58)*Y(Cu61)*Y(N)*state.rho + screened_rates(k_he4_co58__n_cu61)*Y(Co58)*Y(He4)*state.rho) +
        (-screened_rates(k_p_cu61__he4_ni58)*Y(Cu61)*Y(H1)*state.rho + screened_rates(k_he4_ni58__p_cu61)*Y(He4)*Y(Ni58)*state.rho) +
        (screened_rates(k_n_zn61__p_cu61)*Y(N)*Y(Zn61)*state.rho + -screened_rates(k_p_cu61__n_zn61)*Y(Cu61)*Y(H1)*state.rho) +
        (screened_rates(k_p_zn64__he4_cu61)*Y(H1)*Y(Zn64)*state.rho + -screened_rates(k_he4_cu61__p_zn64)*Y(Cu61)*Y(He4)*state.rho) +
        (screened_rates(k_n_ga64__he4_cu61)*Y(Ga64)*Y(N)*state.rho + -screened_rates(k_he4_cu61__n_ga64)*Y(Cu61)*Y(He4)*state.rho);

    ydot_nuc(Cu62) =
        -screened_rates(k_cu62__ni62__weak__wc12)*Y(Cu62) +
        screened_rates(k_zn62__cu62__weak__wc12)*Y(Zn62) +
        (screened_rates(k_he4_co58__cu62)*Y(Co58)*Y(He4)*state.rho + -screened_rates(k_cu62__he4_co58)*Y(Cu62)) +
        (screened_rates(k_p_ni61__cu62)*Y(Ni61)*Y(H1)*state.rho + -screened_rates(k_cu62__p_ni61)*Y(Cu62)) +
        (screened_rates(k_n_cu61__cu62)*Y(Cu61)*Y(N)*state.rho + -screened_rates(k_cu62__n_cu61)*Y(Cu62)) +
        (-screened_rates(k_n_cu62__cu63)*Y(Cu62)*Y(N)*state.rho + screened_rates(k_cu63__n_cu62)*Y(Cu63)) +
        (-screened_rates(k_p_cu62__zn63)*Y(Cu62)*Y(H1)*state.rho + screened_rates(k_zn63__p_cu62)*Y(Zn63)) +
        (-screened_rates(k_n_cu62__p_ni62)*Y(Cu62)*Y(N)*state.rho + screened_rates(k_p_ni62__n_cu62)*Y(Ni62)*Y(H1)*state.rho) +
        (-screened_rates(k_n_cu62__he4_co59)*Y(Cu62)*Y(N)*state.rho + screened_rates(k_he4_co59__n_cu62)*Y(Co59)*Y(He4)*state.rho) +
        (-screened_rates(k_p_cu62__he4_ni59)*Y(Cu62)*Y(H1)*state.rho + screened_rates(k_he4_ni59__p_cu62)*Y(He4)*Y(Ni59)*state.rho) +
        (screened_rates(k_n_zn62__p_cu62)*Y(N)*Y(Zn62)*state.rho + -screened_rates(k_p_cu62__n_zn62)*Y(Cu62)*Y(H1)*state.rho) +
        (screened_rates(k_p_zn65__he4_cu62)*Y(H1)*Y(Zn65)*state.rho + -screened_rates(k_he4_cu62__p_zn65)*Y(Cu62)*Y(He4)*state.rho);

    ydot_nuc(Cu63) =
        screened_rates(k_ni63__cu63__weak__wc12)*Y(Ni63) +
        screened_rates(k_zn63__cu63__weak__wc12)*Y(Zn63) +
        (screened_rates(k_he4_co59__cu63)*Y(Co59)*Y(He4)*state.rho + -screened_rates(k_cu63__he4_co59)*Y(Cu63)) +
        (screened_rates(k_p_ni62__cu63)*Y(Ni62)*Y(H1)*state.rho + -screened_rates(k_cu63__p_ni62)*Y(Cu63)) +
        (screened_rates(k_n_cu62__cu63)*Y(Cu62)*Y(N)*state.rho + -screened_rates(k_cu63__n_cu62)*Y(Cu63)) +
        (-screened_rates(k_n_cu63__cu64)*Y(Cu63)*Y(N)*state.rho + screened_rates(k_cu64__n_cu63)*Y(Cu64)) +
        (-screened_rates(k_p_cu63__zn64)*Y(Cu63)*Y(H1)*state.rho + screened_rates(k_zn64__p_cu63)*Y(Zn64)) +
        (-screened_rates(k_n_cu63__p_ni63)*Y(Cu63)*Y(N)*state.rho + screened_rates(k_p_ni63__n_cu63)*Y(Ni63)*Y(H1)*state.rho) +
        (-screened_rates(k_p_cu63__he4_ni60)*Y(Cu63)*Y(H1)*state.rho + screened_rates(k_he4_ni60__p_cu63)*Y(He4)*Y(Ni60)*state.rho) +
        (screened_rates(k_n_zn63__p_cu63)*Y(N)*Y(Zn63)*state.rho + -screened_rates(k_p_cu63__n_zn63)*Y(Cu63)*Y(H1)*state.rho) +
        (screened_rates(k_p_zn66__he4_cu63)*Y(H1)*Y(Zn66)*state.rho + -screened_rates(k_he4_cu63__p_zn66)*Y(Cu63)*Y(He4)*state.rho);

    ydot_nuc(Cu64) =
        -screened_rates(k_cu64__ni64__weak__wc12)*Y(Cu64) +
        -screened_rates(k_cu64__zn64__weak__wc12)*Y(Cu64) +
        (screened_rates(k_p_ni63__cu64)*Y(Ni63)*Y(H1)*state.rho + -screened_rates(k_cu64__p_ni63)*Y(Cu64)) +
        (screened_rates(k_n_cu63__cu64)*Y(Cu63)*Y(N)*state.rho + -screened_rates(k_cu64__n_cu63)*Y(Cu64)) +
        (-screened_rates(k_n_cu64__cu65)*Y(Cu64)*Y(N)*state.rho + screened_rates(k_cu65__n_cu64)*Y(Cu65)) +
        (-screened_rates(k_p_cu64__zn65)*Y(Cu64)*Y(H1)*state.rho + screened_rates(k_zn65__p_cu64)*Y(Zn65)) +
        (-screened_rates(k_n_cu64__p_ni64)*Y(Cu64)*Y(N)*state.rho + screened_rates(k_p_ni64__n_cu64)*Y(Ni64)*Y(H1)*state.rho) +
        (-screened_rates(k_p_cu64__he4_ni61)*Y(Cu64)*Y(H1)*state.rho + screened_rates(k_he4_ni61__p_cu64)*Y(He4)*Y(Ni61)*state.rho) +
        (screened_rates(k_n_zn64__p_cu64)*Y(N)*Y(Zn64)*state.rho + -screened_rates(k_p_cu64__n_zn64)*Y(Cu64)*Y(H1)*state.rho);

    ydot_nuc(Cu65) =
        screened_rates(k_zn65__cu65__weak__wc12)*Y(Zn65) +
        (screened_rates(k_p_ni64__cu65)*Y(Ni64)*Y(H1)*state.rho + -screened_rates(k_cu65__p_ni64)*Y(Cu65)) +
        (screened_rates(k_n_cu64__cu65)*Y(Cu64)*Y(N)*state.rho + -screened_rates(k_cu65__n_cu64)*Y(Cu65)) +
        (-screened_rates(k_p_cu65__zn66)*Y(Cu65)*Y(H1)*state.rho + screened_rates(k_zn66__p_cu65)*Y(Zn66)) +
        (-screened_rates(k_p_cu65__he4_ni62)*Y(Cu65)*Y(H1)*state.rho + screened_rates(k_he4_ni62__p_cu65)*Y(He4)*Y(Ni62)*state.rho) +
        (screened_rates(k_n_zn65__p_cu65)*Y(N)*Y(Zn65)*state.rho + -screened_rates(k_p_cu65__n_zn65)*Y(Cu65)*Y(H1)*state.rho);

    ydot_nuc(Zn59) =
        -screened_rates(k_zn59__cu59__weak__wc12)*Y(Zn59) +
        -screened_rates(k_zn59__p_ni58__weak__wc12)*Y(Zn59) +
        (screened_rates(k_p_cu58__zn59)*Y(Cu58)*Y(H1)*state.rho + -screened_rates(k_zn59__p_cu58)*Y(Zn59)) +
        (-screened_rates(k_n_zn59__zn60)*Y(N)*Y(Zn59)*state.rho + screened_rates(k_zn60__n_zn59)*Y(Zn60)) +
        (-screened_rates(k_he4_zn59__ge63)*Y(He4)*Y(Zn59)*state.rho + screened_rates(k_ge63__he4_zn59)*Y(Ge63)) +
        (-screened_rates(k_n_zn59__p_cu59)*Y(N)*Y(Zn59)*state.rho + screened_rates(k_p_cu59__n_zn59)*Y(Cu59)*Y(H1)*state.rho) +
        (-screened_rates(k_n_zn59__he4_ni56)*Y(N)*Y(Zn59)*state.rho + screened_rates(k_he4_ni56__n_zn59)*Y(He4)*Y(Ni56)*state.rho) +
        (screened_rates(k_p_ga62__he4_zn59)*Y(Ga62)*Y(H1)*state.rho + -screened_rates(k_he4_zn59__p_ga62)*Y(He4)*Y(Zn59)*state.rho);

    ydot_nuc(Zn60) =
        -screened_rates(k_zn60__cu60__weak__wc12)*Y(Zn60) +
        (screened_rates(k_he4_ni56__zn60)*Y(He4)*Y(Ni56)*state.rho + -screened_rates(k_zn60__he4_ni56)*Y(Zn60)) +
        (screened_rates(k_p_cu59__zn60)*Y(Cu59)*Y(H1)*state.rho + -screened_rates(k_zn60__p_cu59)*Y(Zn60)) +
        (screened_rates(k_n_zn59__zn60)*Y(N)*Y(Zn59)*state.rho + -screened_rates(k_zn60__n_zn59)*Y(Zn60)) +
        (-screened_rates(k_n_zn60__zn61)*Y(N)*Y(Zn60)*state.rho + screened_rates(k_zn61__n_zn60)*Y(Zn61)) +
        (-screened_rates(k_he4_zn60__ge64)*Y(He4)*Y(Zn60)*state.rho + screened_rates(k_ge64__he4_zn60)*Y(Ge64)) +
        (screened_rates(k_he4_cu57__p_zn60)*Y(Cu57)*Y(He4)*state.rho + -screened_rates(k_p_zn60__he4_cu57)*Y(H1)*Y(Zn60)*state.rho) +
        (-screened_rates(k_n_zn60__p_cu60)*Y(N)*Y(Zn60)*state.rho + screened_rates(k_p_cu60__n_zn60)*Y(Cu60)*Y(H1)*state.rho) +
        (-screened_rates(k_n_zn60__he4_ni57)*Y(N)*Y(Zn60)*state.rho + screened_rates(k_he4_ni57__n_zn60)*Y(He4)*Y(Ni57)*state.rho) +
        (screened_rates(k_p_ga63__he4_zn60)*Y(Ga63)*Y(H1)*state.rho + -screened_rates(k_he4_zn60__p_ga63)*Y(He4)*Y(Zn60)*state.rho) +
        (screened_rates(k_n_ge63__he4_zn60)*Y(Ge63)*Y(N)*state.rho + -screened_rates(k_he4_zn60__n_ge63)*Y(He4)*Y(Zn60)*state.rho);

    ydot_nuc(Zn61) =
        -screened_rates(k_zn61__cu61__weak__wc12)*Y(Zn61) +
        (screened_rates(k_he4_ni57__zn61)*Y(He4)*Y(Ni57)*state.rho + -screened_rates(k_zn61__he4_ni57)*Y(Zn61)) +
        (screened_rates(k_p_cu60__zn61)*Y(Cu60)*Y(H1)*state.rho + -screened_rates(k_zn61__p_cu60)*Y(Zn61)) +
        (screened_rates(k_n_zn60__zn61)*Y(N)*Y(Zn60)*state.rho + -screened_rates(k_zn61__n_zn60)*Y(Zn61)) +
        (-screened_rates(k_n_zn61__zn62)*Y(N)*Y(Zn61)*state.rho + screened_rates(k_zn62__n_zn61)*Y(Zn62)) +
        (-screened_rates(k_p_zn61__ga62)*Y(H1)*Y(Zn61)*state.rho + screened_rates(k_ga62__p_zn61)*Y(Ga62)) +
        (-screened_rates(k_n_zn61__p_cu61)*Y(N)*Y(Zn61)*state.rho + screened_rates(k_p_cu61__n_zn61)*Y(Cu61)*Y(H1)*state.rho) +
        (-screened_rates(k_n_zn61__he4_ni58)*Y(N)*Y(Zn61)*state.rho + screened_rates(k_he4_ni58__n_zn61)*Y(He4)*Y(Ni58)*state.rho) +
        (-screened_rates(k_p_zn61__he4_cu58)*Y(H1)*Y(Zn61)*state.rho + screened_rates(k_he4_cu58__p_zn61)*Y(Cu58)*Y(He4)*state.rho) +
        (screened_rates(k_p_ga64__he4_zn61)*Y(Ga64)*Y(H1)*state.rho + -screened_rates(k_he4_zn61__p_ga64)*Y(He4)*Y(Zn61)*state.rho) +
        (screened_rates(k_n_ge64__he4_zn61)*Y(Ge64)*Y(N)*state.rho + -screened_rates(k_he4_zn61__n_ge64)*Y(He4)*Y(Zn61)*state.rho);

    ydot_nuc(Zn62) =
        -screened_rates(k_zn62__cu62__weak__wc12)*Y(Zn62) +
        screened_rates(k_ga62__zn62__weak__wc12)*Y(Ga62) +
        (screened_rates(k_he4_ni58__zn62)*Y(He4)*Y(Ni58)*state.rho + -screened_rates(k_zn62__he4_ni58)*Y(Zn62)) +
        (screened_rates(k_p_cu61__zn62)*Y(Cu61)*Y(H1)*state.rho + -screened_rates(k_zn62__p_cu61)*Y(Zn62)) +
        (screened_rates(k_n_zn61__zn62)*Y(N)*Y(Zn61)*state.rho + -screened_rates(k_zn62__n_zn61)*Y(Zn62)) +
        (-screened_rates(k_n_zn62__zn63)*Y(N)*Y(Zn62)*state.rho + screened_rates(k_zn63__n_zn62)*Y(Zn63)) +
        (-screened_rates(k_p_zn62__ga63)*Y(H1)*Y(Zn62)*state.rho + screened_rates(k_ga63__p_zn62)*Y(Ga63)) +
        (-screened_rates(k_n_zn62__p_cu62)*Y(N)*Y(Zn62)*state.rho + screened_rates(k_p_cu62__n_zn62)*Y(Cu62)*Y(H1)*state.rho) +
        (-screened_rates(k_n_zn62__he4_ni59)*Y(N)*Y(Zn62)*state.rho + screened_rates(k_he4_ni59__n_zn62)*Y(He4)*Y(Ni59)*state.rho) +
        (-screened_rates(k_p_zn62__he4_cu59)*Y(H1)*Y(Zn62)*state.rho + screened_rates(k_he4_cu59__p_zn62)*Y(Cu59)*Y(He4)*state.rho) +
        (screened_rates(k_n_ga62__p_zn62)*Y(Ga62)*Y(N)*state.rho + -screened_rates(k_p_zn62__n_ga62)*Y(H1)*Y(Zn62)*state.rho);

    ydot_nuc(Zn63) =
        -screened_rates(k_zn63__cu63__weak__wc12)*Y(Zn63) +
        screened_rates(k_ga63__zn63__weak__wc12)*Y(Ga63) +
        (screened_rates(k_he4_ni59__zn63)*Y(He4)*Y(Ni59)*state.rho + -screened_rates(k_zn63__he4_ni59)*Y(Zn63)) +
        (screened_rates(k_p_cu62__zn63)*Y(Cu62)*Y(H1)*state.rho + -screened_rates(k_zn63__p_cu62)*Y(Zn63)) +
        (screened_rates(k_n_zn62__zn63)*Y(N)*Y(Zn62)*state.rho + -screened_rates(k_zn63__n_zn62)*Y(Zn63)) +
        (-screened_rates(k_n_zn63__zn64)*Y(N)*Y(Zn63)*state.rho + screened_rates(k_zn64__n_zn63)*Y(Zn64)) +
        (-screened_rates(k_p_zn63__ga64)*Y(H1)*Y(Zn63)*state.rho + screened_rates(k_ga64__p_zn63)*Y(Ga64)) +
        (-screened_rates(k_n_zn63__p_cu63)*Y(N)*Y(Zn63)*state.rho + screened_rates(k_p_cu63__n_zn63)*Y(Cu63)*Y(H1)*state.rho) +
        (-screened_rates(k_n_zn63__he4_ni60)*Y(N)*Y(Zn63)*state.rho + screened_rates(k_he4_ni60__n_zn63)*Y(He4)*Y(Ni60)*state.rho) +
        (-screened_rates(k_p_zn63__he4_cu60)*Y(H1)*Y(Zn63)*state.rho + screened_rates(k_he4_cu60__p_zn63)*Y(Cu60)*Y(He4)*state.rho) +
        (screened_rates(k_n_ga63__p_zn63)*Y(Ga63)*Y(N)*state.rho + -screened_rates(k_p_zn63__n_ga63)*Y(H1)*Y(Zn63)*state.rho);

    ydot_nuc(Zn64) =
        screened_rates(k_cu64__zn64__weak__wc12)*Y(Cu64) +
        screened_rates(k_ga64__zn64__weak__wc12)*Y(Ga64) +
        (screened_rates(k_he4_ni60__zn64)*Y(He4)*Y(Ni60)*state.rho + -screened_rates(k_zn64__he4_ni60)*Y(Zn64)) +
        (screened_rates(k_p_cu63__zn64)*Y(Cu63)*Y(H1)*state.rho + -screened_rates(k_zn64__p_cu63)*Y(Zn64)) +
        (screened_rates(k_n_zn63__zn64)*Y(N)*Y(Zn63)*state.rho + -screened_rates(k_zn64__n_zn63)*Y(Zn64)) +
        (-screened_rates(k_n_zn64__zn65)*Y(N)*Y(Zn64)*state.rho + screened_rates(k_zn65__n_zn64)*Y(Zn65)) +
        (-screened_rates(k_n_zn64__p_cu64)*Y(N)*Y(Zn64)*state.rho + screened_rates(k_p_cu64__n_zn64)*Y(Cu64)*Y(H1)*state.rho) +
        (-screened_rates(k_n_zn64__he4_ni61)*Y(N)*Y(Zn64)*state.rho + screened_rates(k_he4_ni61__n_zn64)*Y(He4)*Y(Ni61)*state.rho) +
        (-screened_rates(k_p_zn64__he4_cu61)*Y(H1)*Y(Zn64)*state.rho + screened_rates(k_he4_cu61__p_zn64)*Y(Cu61)*Y(He4)*state.rho) +
        (screened_rates(k_n_ga64__p_zn64)*Y(Ga64)*Y(N)*state.rho + -screened_rates(k_p_zn64__n_ga64)*Y(H1)*Y(Zn64)*state.rho);

    ydot_nuc(Zn65) =
        -screened_rates(k_zn65__cu65__weak__wc12)*Y(Zn65) +
        (screened_rates(k_he4_ni61__zn65)*Y(He4)*Y(Ni61)*state.rho + -screened_rates(k_zn65__he4_ni61)*Y(Zn65)) +
        (screened_rates(k_p_cu64__zn65)*Y(Cu64)*Y(H1)*state.rho + -screened_rates(k_zn65__p_cu64)*Y(Zn65)) +
        (screened_rates(k_n_zn64__zn65)*Y(N)*Y(Zn64)*state.rho + -screened_rates(k_zn65__n_zn64)*Y(Zn65)) +
        (-screened_rates(k_n_zn65__zn66)*Y(N)*Y(Zn65)*state.rho + screened_rates(k_zn66__n_zn65)*Y(Zn66)) +
        (-screened_rates(k_n_zn65__p_cu65)*Y(N)*Y(Zn65)*state.rho + screened_rates(k_p_cu65__n_zn65)*Y(Cu65)*Y(H1)*state.rho) +
        (-screened_rates(k_n_zn65__he4_ni62)*Y(N)*Y(Zn65)*state.rho + screened_rates(k_he4_ni62__n_zn65)*Y(He4)*Y(Ni62)*state.rho) +
        (-screened_rates(k_p_zn65__he4_cu62)*Y(H1)*Y(Zn65)*state.rho + screened_rates(k_he4_cu62__p_zn65)*Y(Cu62)*Y(He4)*state.rho);

    ydot_nuc(Zn66) =
        (screened_rates(k_he4_ni62__zn66)*Y(He4)*Y(Ni62)*state.rho + -screened_rates(k_zn66__he4_ni62)*Y(Zn66)) +
        (screened_rates(k_p_cu65__zn66)*Y(Cu65)*Y(H1)*state.rho + -screened_rates(k_zn66__p_cu65)*Y(Zn66)) +
        (screened_rates(k_n_zn65__zn66)*Y(N)*Y(Zn65)*state.rho + -screened_rates(k_zn66__n_zn65)*Y(Zn66)) +
        (-screened_rates(k_n_zn66__he4_ni63)*Y(N)*Y(Zn66)*state.rho + screened_rates(k_he4_ni63__n_zn66)*Y(He4)*Y(Ni63)*state.rho) +
        (-screened_rates(k_p_zn66__he4_cu63)*Y(H1)*Y(Zn66)*state.rho + screened_rates(k_he4_cu63__p_zn66)*Y(Cu63)*Y(He4)*state.rho);

    ydot_nuc(Ga62) =
        -screened_rates(k_ga62__zn62__weak__wc12)*Y(Ga62) +
        (screened_rates(k_he4_cu58__ga62)*Y(Cu58)*Y(He4)*state.rho + -screened_rates(k_ga62__he4_cu58)*Y(Ga62)) +
        (screened_rates(k_p_zn61__ga62)*Y(H1)*Y(Zn61)*state.rho + -screened_rates(k_ga62__p_zn61)*Y(Ga62)) +
        (-screened_rates(k_n_ga62__ga63)*Y(Ga62)*Y(N)*state.rho + screened_rates(k_ga63__n_ga62)*Y(Ga63)) +
        (-screened_rates(k_p_ga62__ge63)*Y(Ga62)*Y(H1)*state.rho + screened_rates(k_ge63__p_ga62)*Y(Ge63)) +
        (-screened_rates(k_n_ga62__p_zn62)*Y(Ga62)*Y(N)*state.rho + screened_rates(k_p_zn62__n_ga62)*Y(H1)*Y(Zn62)*state.rho) +
        (-screened_rates(k_n_ga62__he4_cu59)*Y(Ga62)*Y(N)*state.rho + screened_rates(k_he4_cu59__n_ga62)*Y(Cu59)*Y(He4)*state.rho) +
        (-screened_rates(k_p_ga62__he4_zn59)*Y(Ga62)*Y(H1)*state.rho + screened_rates(k_he4_zn59__p_ga62)*Y(He4)*Y(Zn59)*state.rho);

    ydot_nuc(Ga63) =
        -screened_rates(k_ga63__zn63__weak__wc12)*Y(Ga63) +
        screened_rates(k_ge63__ga63__weak__wc12)*Y(Ge63) +
        (screened_rates(k_he4_cu59__ga63)*Y(Cu59)*Y(He4)*state.rho + -screened_rates(k_ga63__he4_cu59)*Y(Ga63)) +
        (screened_rates(k_p_zn62__ga63)*Y(H1)*Y(Zn62)*state.rho + -screened_rates(k_ga63__p_zn62)*Y(Ga63)) +
        (screened_rates(k_n_ga62__ga63)*Y(Ga62)*Y(N)*state.rho + -screened_rates(k_ga63__n_ga62)*Y(Ga63)) +
        (-screened_rates(k_n_ga63__ga64)*Y(Ga63)*Y(N)*state.rho + screened_rates(k_ga64__n_ga63)*Y(Ga64)) +
        (-screened_rates(k_p_ga63__ge64)*Y(Ga63)*Y(H1)*state.rho + screened_rates(k_ge64__p_ga63)*Y(Ge64)) +
        (-screened_rates(k_n_ga63__p_zn63)*Y(Ga63)*Y(N)*state.rho + screened_rates(k_p_zn63__n_ga63)*Y(H1)*Y(Zn63)*state.rho) +
        (-screened_rates(k_n_ga63__he4_cu60)*Y(Ga63)*Y(N)*state.rho + screened_rates(k_he4_cu60__n_ga63)*Y(Cu60)*Y(He4)*state.rho) +
        (-screened_rates(k_p_ga63__he4_zn60)*Y(Ga63)*Y(H1)*state.rho + screened_rates(k_he4_zn60__p_ga63)*Y(He4)*Y(Zn60)*state.rho) +
        (screened_rates(k_n_ge63__p_ga63)*Y(Ge63)*Y(N)*state.rho + -screened_rates(k_p_ga63__n_ge63)*Y(Ga63)*Y(H1)*state.rho);

    ydot_nuc(Ga64) =
        -screened_rates(k_ga64__zn64__weak__wc12)*Y(Ga64) +
        screened_rates(k_ge64__ga64__weak__wc12)*Y(Ge64) +
        (screened_rates(k_he4_cu60__ga64)*Y(Cu60)*Y(He4)*state.rho + -screened_rates(k_ga64__he4_cu60)*Y(Ga64)) +
        (screened_rates(k_p_zn63__ga64)*Y(H1)*Y(Zn63)*state.rho + -screened_rates(k_ga64__p_zn63)*Y(Ga64)) +
        (screened_rates(k_n_ga63__ga64)*Y(Ga63)*Y(N)*state.rho + -screened_rates(k_ga64__n_ga63)*Y(Ga64)) +
        (-screened_rates(k_n_ga64__p_zn64)*Y(Ga64)*Y(N)*state.rho + screened_rates(k_p_zn64__n_ga64)*Y(H1)*Y(Zn64)*state.rho) +
        (-screened_rates(k_n_ga64__he4_cu61)*Y(Ga64)*Y(N)*state.rho + screened_rates(k_he4_cu61__n_ga64)*Y(Cu61)*Y(He4)*state.rho) +
        (-screened_rates(k_p_ga64__he4_zn61)*Y(Ga64)*Y(H1)*state.rho + screened_rates(k_he4_zn61__p_ga64)*Y(He4)*Y(Zn61)*state.rho) +
        (screened_rates(k_n_ge64__p_ga64)*Y(Ge64)*Y(N)*state.rho + -screened_rates(k_p_ga64__n_ge64)*Y(Ga64)*Y(H1)*state.rho);

    ydot_nuc(Ge63) =
        -screened_rates(k_ge63__ga63__weak__wc12)*Y(Ge63) +
        (screened_rates(k_he4_zn59__ge63)*Y(He4)*Y(Zn59)*state.rho + -screened_rates(k_ge63__he4_zn59)*Y(Ge63)) +
        (screened_rates(k_p_ga62__ge63)*Y(Ga62)*Y(H1)*state.rho + -screened_rates(k_ge63__p_ga62)*Y(Ge63)) +
        (-screened_rates(k_n_ge63__ge64)*Y(Ge63)*Y(N)*state.rho + screened_rates(k_ge64__n_ge63)*Y(Ge64)) +
        (-screened_rates(k_n_ge63__p_ga63)*Y(Ge63)*Y(N)*state.rho + screened_rates(k_p_ga63__n_ge63)*Y(Ga63)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ge63__he4_zn60)*Y(Ge63)*Y(N)*state.rho + screened_rates(k_he4_zn60__n_ge63)*Y(He4)*Y(Zn60)*state.rho);

    ydot_nuc(Ge64) =
        -screened_rates(k_ge64__ga64__weak__wc12)*Y(Ge64) +
        (screened_rates(k_he4_zn60__ge64)*Y(He4)*Y(Zn60)*state.rho + -screened_rates(k_ge64__he4_zn60)*Y(Ge64)) +
        (screened_rates(k_p_ga63__ge64)*Y(Ga63)*Y(H1)*state.rho + -screened_rates(k_ge64__p_ga63)*Y(Ge64)) +
        (screened_rates(k_n_ge63__ge64)*Y(Ge63)*Y(N)*state.rho + -screened_rates(k_ge64__n_ge63)*Y(Ge64)) +
        (-screened_rates(k_n_ge64__p_ga64)*Y(Ge64)*Y(N)*state.rho + screened_rates(k_p_ga64__n_ge64)*Y(Ga64)*Y(H1)*state.rho) +
        (-screened_rates(k_n_ge64__he4_zn61)*Y(Ge64)*Y(N)*state.rho + screened_rates(k_he4_zn61__n_ge64)*Y(He4)*Y(Zn61)*state.rho);

}


AMREX_GPU_HOST_DEVICE AMREX_INLINE
void actual_rhs (burn_t& state, Array1D<Real, 1, neqs>& ydot)
{
    for (int i = 1; i <= neqs; ++i) {
        ydot(i) = 0.0_rt;
    }


    // Set molar abundances
    Array1D<Real, 1, NumSpec> Y;
    for (int i = 1; i <= NumSpec; ++i) {
        Y(i) = state.xn[i-1] * aion_inv[i-1];
    }

    // build the rates

    rate_eval_t rate_eval;

    evaluate_rates(state, rate_eval);

    rhs_nuc(state, ydot, Y, rate_eval.screened_rates);

    // ion binding energy contributions

    Real enuc;
    ener_gener_rate(ydot, enuc);

    // include reaction neutrino losses (non-thermal)

    // Get the thermal neutrino losses

    Real sneut, dsneutdt, dsneutdd, snuda, snudz;

    sneut5(state.T, state.rho, state.abar, state.zbar, sneut, dsneutdt, dsneutdd, snuda, snudz);

    // Append the energy equation (this is erg/g/s)

    ydot(net_ienuc) = enuc - sneut;

}


template<class MatrixType>
AMREX_GPU_HOST_DEVICE AMREX_INLINE
void jac_nuc(const burn_t& state,
             MatrixType& jac,
             const Array1D<Real, 1, NumSpec>& Y,
             const Array1D<Real, 1, NumRates>& screened_rates)
{

    Real scratch;

    scratch = -screened_rates(k_n__p__weak__wc12) - screened_rates(k_n_al25__al26)*Y(Al25)*state.rho - screened_rates(k_n_al25__he4_na22)*Y(Al25)*state.rho - screened_rates(k_n_al25__p_mg25)*Y(Al25)*state.rho - screened_rates(k_n_al26__al27)*Y(Al26)*state.rho - screened_rates(k_n_al26__he4_na23)*Y(Al26)*state.rho - screened_rates(k_n_al26__p_mg26)*Y(Al26)*state.rho - screened_rates(k_n_ar36__ar37)*Y(Ar36)*state.rho - screened_rates(k_n_ar36__he4_s33)*Y(Ar36)*state.rho - screened_rates(k_n_ar36__p_cl36)*Y(Ar36)*state.rho - screened_rates(k_n_ar37__ar38)*Y(Ar37)*state.rho - screened_rates(k_n_ar37__he4_s34)*Y(Ar37)*state.rho - screened_rates(k_n_ar37__p_cl37)*Y(Ar37)*state.rho - screened_rates(k_n_ar38__ar39)*Y(Ar38)*state.rho - screened_rates(k_n_ar38__he4_s35)*Y(Ar38)*state.rho - screened_rates(k_n_ar39__ar40)*Y(Ar39)*state.rho - screened_rates(k_n_ar39__he4_s36)*Y(Ar39)*state.rho - screened_rates(k_n_b10__b11)*Y(B10)*state.rho - screened_rates(k_n_b10__he4_li7)*Y(B10)*state.rho - screened_rates(k_n_b8__p_he4_he4)*Y(B8)*state.rho - screened_rates(k_n_be7__d_li6)*Y(Be7)*state.rho - screened_rates(k_n_be7__he4_he4)*Y(Be7)*state.rho - screened_rates(k_n_be7__p_li7)*Y(Be7)*state.rho - screened_rates(k_n_c12__c13)*Y(C12)*state.rho - screened_rates(k_n_c12__he4_be9)*Y(C12)*state.rho - screened_rates(k_n_c13__c14)*Y(C13)*state.rho - screened_rates(k_n_ca40__ca41)*Y(Ca40)*state.rho - screened_rates(k_n_ca40__he4_ar37)*Y(Ca40)*state.rho - screened_rates(k_n_ca40__p_k40)*Y(Ca40)*state.rho - screened_rates(k_n_ca41__ca42)*Y(Ca41)*state.rho - screened_rates(k_n_ca41__he4_ar38)*Y(Ca41)*state.rho - screened_rates(k_n_ca41__p_k41)*Y(Ca41)*state.rho - screened_rates(k_n_ca42__ca43)*Y(Ca42)*state.rho - screened_rates(k_n_ca42__he4_ar39)*Y(Ca42)*state.rho - screened_rates(k_n_ca43__ca44)*Y(Ca43)*state.rho - screened_rates(k_n_ca43__he4_ar40)*Y(Ca43)*state.rho - screened_rates(k_n_ca44__ca45)*Y(Ca44)*state.rho - screened_rates(k_n_ca45__ca46)*Y(Ca45)*state.rho - screened_rates(k_n_ca46__ca47)*Y(Ca46)*state.rho - screened_rates(k_n_ca47__ca48)*Y(Ca47)*state.rho - screened_rates(k_n_cl33__cl34)*Y(Cl33)*state.rho - screened_rates(k_n_cl33__he4_p30)*Y(Cl33)*state.rho - screened_rates(k_n_cl33__p_s33)*Y(Cl33)*state.rho - screened_rates(k_n_cl34__cl35)*Y(Cl34)*state.rho - screened_rates(k_n_cl34__he4_p31)*Y(Cl34)*state.rho - screened_rates(k_n_cl34__p_s34)*Y(Cl34)*state.rho - screened_rates(k_n_cl35__cl36)*Y(Cl35)*state.rho - screened_rates(k_n_cl35__he4_p32)*Y(Cl35)*state.rho - screened_rates(k_n_cl35__p_s35)*Y(Cl35)*state.rho - screened_rates(k_n_cl36__cl37)*Y(Cl36)*state.rho - screened_rates(k_n_cl36__he4_p33)*Y(Cl36)*state.rho - screened_rates(k_n_cl36__p_s36)*Y(Cl36)*state.rho - screened_rates(k_n_co53__co54)*Y(Co53)*state.rho - screened_rates(k_n_co53__he4_mn50)*Y(Co53)*state.rho - screened_rates(k_n_co53__p_fe53)*Y(Co53)*state.rho - screened_rates(k_n_co54__co55)*Y(Co54)*state.rho - screened_rates(k_n_co54__he4_mn51)*Y(Co54)*state.rho - screened_rates(k_n_co54__p_fe54)*Y(Co54)*state.rho - screened_rates(k_n_co55__co56)*Y(Co55)*state.rho - screened_rates(k_n_co55__he4_mn52)*Y(Co55)*state.rho - screened_rates(k_n_co55__p_fe55)*Y(Co55)*state.rho - screened_rates(k_n_co56__co57)*Y(Co56)*state.rho - screened_rates(k_n_co56__he4_mn53)*Y(Co56)*state.rho - screened_rates(k_n_co56__p_fe56)*Y(Co56)*state.rho - screened_rates(k_n_co57__co58)*Y(Co57)*state.rho - screened_rates(k_n_co57__he4_mn54)*Y(Co57)*state.rho - screened_rates(k_n_co57__p_fe57)*Y(Co57)*state.rho - screened_rates(k_n_co58__co59)*Y(Co58)*state.rho - screened_rates(k_n_co58__he4_mn55)*Y(Co58)*state.rho - screened_rates(k_n_co58__p_fe58)*Y(Co58)*state.rho - screened_rates(k_n_cr48__cr49)*Y(Cr48)*state.rho - screened_rates(k_n_cr48__he4_ti45)*Y(Cr48)*state.rho - screened_rates(k_n_cr48__p_v48)*Y(Cr48)*state.rho - screened_rates(k_n_cr49__cr50)*Y(Cr49)*state.rho - screened_rates(k_n_cr49__he4_ti46)*Y(Cr49)*state.rho - screened_rates(k_n_cr49__p_v49)*Y(Cr49)*state.rho - screened_rates(k_n_cr50__cr51)*Y(Cr50)*state.rho - screened_rates(k_n_cr50__he4_ti47)*Y(Cr50)*state.rho - screened_rates(k_n_cr50__p_v50)*Y(Cr50)*state.rho - screened_rates(k_n_cr51__cr52)*Y(Cr51)*state.rho - screened_rates(k_n_cr51__he4_ti48)*Y(Cr51)*state.rho - screened_rates(k_n_cr51__p_v51)*Y(Cr51)*state.rho - screened_rates(k_n_cr52__cr53)*Y(Cr52)*state.rho - screened_rates(k_n_cr52__he4_ti49)*Y(Cr52)*state.rho - screened_rates(k_n_cr52__p_v52)*Y(Cr52)*state.rho - screened_rates(k_n_cr53__cr54)*Y(Cr53)*state.rho - screened_rates(k_n_cr53__he4_ti50)*Y(Cr53)*state.rho - screened_rates(k_n_cr54__he4_ti51)*Y(Cr54)*state.rho - screened_rates(k_n_cu57__cu58)*Y(Cu57)*state.rho - screened_rates(k_n_cu57__he4_co54)*Y(Cu57)*state.rho - screened_rates(k_n_cu57__p_ni57)*Y(Cu57)*state.rho - screened_rates(k_n_cu58__cu59)*Y(Cu58)*state.rho - screened_rates(k_n_cu58__he4_co55)*Y(Cu58)*state.rho - screened_rates(k_n_cu58__p_ni58)*Y(Cu58)*state.rho - screened_rates(k_n_cu59__cu60)*Y(Cu59)*state.rho - screened_rates(k_n_cu59__he4_co56)*Y(Cu59)*state.rho - screened_rates(k_n_cu59__p_ni59)*Y(Cu59)*state.rho - screened_rates(k_n_cu60__cu61)*Y(Cu60)*state.rho - screened_rates(k_n_cu60__he4_co57)*Y(Cu60)*state.rho - screened_rates(k_n_cu60__p_ni60)*Y(Cu60)*state.rho - screened_rates(k_n_cu61__cu62)*Y(Cu61)*state.rho - screened_rates(k_n_cu61__he4_co58)*Y(Cu61)*state.rho - screened_rates(k_n_cu61__p_ni61)*Y(Cu61)*state.rho - screened_rates(k_n_cu62__cu63)*Y(Cu62)*state.rho - screened_rates(k_n_cu62__he4_co59)*Y(Cu62)*state.rho - screened_rates(k_n_cu62__p_ni62)*Y(Cu62)*state.rho - screened_rates(k_n_cu63__cu64)*Y(Cu63)*state.rho - screened_rates(k_n_cu63__p_ni63)*Y(Cu63)*state.rho - screened_rates(k_n_cu64__cu65)*Y(Cu64)*state.rho - screened_rates(k_n_cu64__p_ni64)*Y(Cu64)*state.rho - screened_rates(k_n_f17__f18)*Y(F17)*state.rho - screened_rates(k_n_f17__he4_n14)*Y(F17)*state.rho - screened_rates(k_n_f17__p_o17)*Y(F17)*state.rho - screened_rates(k_n_f18__f19)*Y(F18)*state.rho - screened_rates(k_n_f18__he4_n15)*Y(F18)*state.rho - screened_rates(k_n_f18__p_o18)*Y(F18)*state.rho - screened_rates(k_n_fe52__fe53)*Y(Fe52)*state.rho - screened_rates(k_n_fe52__he4_cr49)*Y(Fe52)*state.rho - screened_rates(k_n_fe52__p_mn52)*Y(Fe52)*state.rho - screened_rates(k_n_fe53__fe54)*Y(Fe53)*state.rho - screened_rates(k_n_fe53__he4_cr50)*Y(Fe53)*state.rho - screened_rates(k_n_fe53__p_mn53)*Y(Fe53)*state.rho - screened_rates(k_n_fe54__fe55)*Y(Fe54)*state.rho - screened_rates(k_n_fe54__he4_cr51)*Y(Fe54)*state.rho - screened_rates(k_n_fe54__p_mn54)*Y(Fe54)*state.rho - screened_rates(k_n_fe55__fe56)*Y(Fe55)*state.rho - screened_rates(k_n_fe55__he4_cr52)*Y(Fe55)*state.rho - screened_rates(k_n_fe55__p_mn55)*Y(Fe55)*state.rho - screened_rates(k_n_fe56__fe57)*Y(Fe56)*state.rho - screened_rates(k_n_fe56__he4_cr53)*Y(Fe56)*state.rho - screened_rates(k_n_fe57__fe58)*Y(Fe57)*state.rho - screened_rates(k_n_fe57__he4_cr54)*Y(Fe57)*state.rho - screened_rates(k_n_ga62__ga63)*Y(Ga62)*state.rho - screened_rates(k_n_ga62__he4_cu59)*Y(Ga62)*state.rho - screened_rates(k_n_ga62__p_zn62)*Y(Ga62)*state.rho - screened_rates(k_n_ga63__ga64)*Y(Ga63)*state.rho - screened_rates(k_n_ga63__he4_cu60)*Y(Ga63)*state.rho - screened_rates(k_n_ga63__p_zn63)*Y(Ga63)*state.rho - screened_rates(k_n_ga64__he4_cu61)*Y(Ga64)*state.rho - screened_rates(k_n_ga64__p_zn64)*Y(Ga64)*state.rho - screened_rates(k_n_ge63__ge64)*Y(Ge63)*state.rho - screened_rates(k_n_ge63__he4_zn60)*Y(Ge63)*state.rho - screened_rates(k_n_ge63__p_ga63)*Y(Ge63)*state.rho - screened_rates(k_n_ge64__he4_zn61)*Y(Ge64)*state.rho - screened_rates(k_n_ge64__p_ga64)*Y(Ge64)*state.rho - screened_rates(k_n_he3__d_d)*Y(He3)*state.rho - screened_rates(k_n_he3__he4)*Y(He3)*state.rho - 0.5*screened_rates(k_n_he4_he4__be9)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) - 0.5*screened_rates(k_n_he4_he4__d_li7)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) - screened_rates(k_n_k37__he4_cl34)*Y(K37)*state.rho - screened_rates(k_n_k37__k38)*Y(K37)*state.rho - screened_rates(k_n_k37__p_ar37)*Y(K37)*state.rho - screened_rates(k_n_k38__he4_cl35)*Y(K38)*state.rho - screened_rates(k_n_k38__k39)*Y(K38)*state.rho - screened_rates(k_n_k38__p_ar38)*Y(K38)*state.rho - screened_rates(k_n_k39__he4_cl36)*Y(K39)*state.rho - screened_rates(k_n_k39__k40)*Y(K39)*state.rho - screened_rates(k_n_k39__p_ar39)*Y(K39)*state.rho - screened_rates(k_n_k40__he4_cl37)*Y(K40)*state.rho - screened_rates(k_n_k40__k41)*Y(K40)*state.rho - screened_rates(k_n_k40__p_ar40)*Y(K40)*state.rho - screened_rates(k_n_li6__li7)*Y(Li6)*state.rho - screened_rates(k_n_mg23__c12_c12)*Y(Mg23)*state.rho - screened_rates(k_n_mg23__he4_ne20)*Y(Mg23)*state.rho - screened_rates(k_n_mg23__mg24)*Y(Mg23)*state.rho - screened_rates(k_n_mg23__p_na23)*Y(Mg23)*state.rho - screened_rates(k_n_mg24__he4_ne21)*Y(Mg24)*state.rho - screened_rates(k_n_mg24__mg25)*Y(Mg24)*state.rho - screened_rates(k_n_mg25__he4_ne22)*Y(Mg25)*state.rho - screened_rates(k_n_mg25__mg26)*Y(Mg25)*state.rho - screened_rates(k_n_mn50__he4_v47)*Y(Mn50)*state.rho - screened_rates(k_n_mn50__mn51)*Y(Mn50)*state.rho - screened_rates(k_n_mn50__p_cr50)*Y(Mn50)*state.rho - screened_rates(k_n_mn51__he4_v48)*Y(Mn51)*state.rho - screened_rates(k_n_mn51__mn52)*Y(Mn51)*state.rho - screened_rates(k_n_mn51__p_cr51)*Y(Mn51)*state.rho - screened_rates(k_n_mn52__he4_v49)*Y(Mn52)*state.rho - screened_rates(k_n_mn52__mn53)*Y(Mn52)*state.rho - screened_rates(k_n_mn52__p_cr52)*Y(Mn52)*state.rho - screened_rates(k_n_mn53__he4_v50)*Y(Mn53)*state.rho - screened_rates(k_n_mn53__mn54)*Y(Mn53)*state.rho - screened_rates(k_n_mn53__p_cr53)*Y(Mn53)*state.rho - screened_rates(k_n_mn54__he4_v51)*Y(Mn54)*state.rho - screened_rates(k_n_mn54__mn55)*Y(Mn54)*state.rho - screened_rates(k_n_mn54__p_cr54)*Y(Mn54)*state.rho - screened_rates(k_n_mn55__he4_v52)*Y(Mn55)*state.rho - screened_rates(k_n_n13__he4_b10)*Y(N13)*state.rho - screened_rates(k_n_n13__n14)*Y(N13)*state.rho - screened_rates(k_n_n13__p_c13)*Y(N13)*state.rho - screened_rates(k_n_n14__d_c13)*Y(N14)*state.rho - screened_rates(k_n_n14__he4_b11)*Y(N14)*state.rho - screened_rates(k_n_n14__n15)*Y(N14)*state.rho - screened_rates(k_n_n14__p_c14)*Y(N14)*state.rho - screened_rates(k_n_n15__d_c14)*Y(N15)*state.rho - screened_rates(k_n_na21__he4_f18)*Y(Na21)*state.rho - screened_rates(k_n_na21__na22)*Y(Na21)*state.rho - screened_rates(k_n_na21__p_ne21)*Y(Na21)*state.rho - screened_rates(k_n_na22__he4_f19)*Y(Na22)*state.rho - screened_rates(k_n_na22__na23)*Y(Na22)*state.rho - screened_rates(k_n_na22__p_ne22)*Y(Na22)*state.rho - screened_rates(k_n_ne18__he4_o15)*Y(Ne18)*state.rho - screened_rates(k_n_ne18__ne19)*Y(Ne18)*state.rho - screened_rates(k_n_ne18__p_f18)*Y(Ne18)*state.rho - screened_rates(k_n_ne19__he4_o16)*Y(Ne19)*state.rho - screened_rates(k_n_ne19__ne20)*Y(Ne19)*state.rho - screened_rates(k_n_ne19__p_f19)*Y(Ne19)*state.rho - screened_rates(k_n_ne20__he4_o17)*Y(Ne20)*state.rho - screened_rates(k_n_ne20__ne21)*Y(Ne20)*state.rho - screened_rates(k_n_ne21__he4_o18)*Y(Ne21)*state.rho - screened_rates(k_n_ne21__ne22)*Y(Ne21)*state.rho - screened_rates(k_n_ni56__he4_fe53)*Y(Ni56)*state.rho - screened_rates(k_n_ni56__ni57)*Y(Ni56)*state.rho - screened_rates(k_n_ni56__p_co56)*Y(Ni56)*state.rho - screened_rates(k_n_ni57__he4_fe54)*Y(Ni57)*state.rho - screened_rates(k_n_ni57__ni58)*Y(Ni57)*state.rho - screened_rates(k_n_ni57__p_co57)*Y(Ni57)*state.rho - screened_rates(k_n_ni58__he4_fe55)*Y(Ni58)*state.rho - screened_rates(k_n_ni58__ni59)*Y(Ni58)*state.rho - screened_rates(k_n_ni58__p_co58)*Y(Ni58)*state.rho - screened_rates(k_n_ni59__he4_fe56)*Y(Ni59)*state.rho - screened_rates(k_n_ni59__ni60)*Y(Ni59)*state.rho - screened_rates(k_n_ni59__p_co59)*Y(Ni59)*state.rho - screened_rates(k_n_ni60__he4_fe57)*Y(Ni60)*state.rho - screened_rates(k_n_ni60__ni61)*Y(Ni60)*state.rho - screened_rates(k_n_ni61__he4_fe58)*Y(Ni61)*state.rho - screened_rates(k_n_ni61__ni62)*Y(Ni61)*state.rho - screened_rates(k_n_ni62__ni63)*Y(Ni62)*state.rho - screened_rates(k_n_ni63__ni64)*Y(Ni63)*state.rho - screened_rates(k_n_o14__o15)*Y(O14)*state.rho - screened_rates(k_n_o14__p_n14)*Y(O14)*state.rho - screened_rates(k_n_o15__he4_c12)*Y(O15)*state.rho - screened_rates(k_n_o15__o16)*Y(O15)*state.rho - screened_rates(k_n_o15__p_n15)*Y(O15)*state.rho - screened_rates(k_n_o16__he4_c13)*Y(O16)*state.rho - screened_rates(k_n_o16__o17)*Y(O16)*state.rho - screened_rates(k_n_o17__he4_c14)*Y(O17)*state.rho - screened_rates(k_n_o17__o18)*Y(O17)*state.rho - screened_rates(k_n_p29__he4_al26)*Y(P29)*state.rho - screened_rates(k_n_p29__p30)*Y(P29)*state.rho - screened_rates(k_n_p29__p_si29)*Y(P29)*state.rho - screened_rates(k_n_p30__he4_al27)*Y(P30)*state.rho - screened_rates(k_n_p30__p31)*Y(P30)*state.rho - screened_rates(k_n_p30__p_si30)*Y(P30)*state.rho - screened_rates(k_n_p31__p32)*Y(P31)*state.rho - screened_rates(k_n_p31__p_si31)*Y(P31)*state.rho - screened_rates(k_n_p32__p33)*Y(P32)*state.rho - screened_rates(k_n_p32__p_si32)*Y(P32)*state.rho - screened_rates(k_n_p__d)*Y(H1)*state.rho - screened_rates(k_n_p_he4__li6)*Y(He4)*Y(H1)*std::pow(state.rho, 2) - 0.5*screened_rates(k_n_p_he4_he4__he3_li7)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 3) - 0.5*screened_rates(k_n_p_he4_he4__p_be9)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 3) - 0.5*screened_rates(k_n_p_p__p_d)*std::pow(Y(H1), 2)*std::pow(state.rho, 2) - screened_rates(k_n_s32__he4_si29)*Y(S32)*state.rho - screened_rates(k_n_s32__p_p32)*Y(S32)*state.rho - screened_rates(k_n_s32__s33)*Y(S32)*state.rho - screened_rates(k_n_s33__he4_si30)*Y(S33)*state.rho - screened_rates(k_n_s33__p_p33)*Y(S33)*state.rho - screened_rates(k_n_s33__s34)*Y(S33)*state.rho - screened_rates(k_n_s34__he4_si31)*Y(S34)*state.rho - screened_rates(k_n_s34__s35)*Y(S34)*state.rho - screened_rates(k_n_s35__he4_si32)*Y(S35)*state.rho - screened_rates(k_n_s35__s36)*Y(S35)*state.rho - screened_rates(k_n_sc43__he4_k40)*Y(Sc43)*state.rho - screened_rates(k_n_sc43__p_ca43)*Y(Sc43)*state.rho - screened_rates(k_n_sc43__sc44)*Y(Sc43)*state.rho - screened_rates(k_n_sc44__he4_k41)*Y(Sc44)*state.rho - screened_rates(k_n_sc44__p_ca44)*Y(Sc44)*state.rho - screened_rates(k_n_sc44__sc45)*Y(Sc44)*state.rho - screened_rates(k_n_sc45__p_ca45)*Y(Sc45)*state.rho - screened_rates(k_n_sc45__sc46)*Y(Sc45)*state.rho - screened_rates(k_n_sc46__p_ca46)*Y(Sc46)*state.rho - screened_rates(k_n_sc46__sc47)*Y(Sc46)*state.rho - screened_rates(k_n_sc47__p_ca47)*Y(Sc47)*state.rho - screened_rates(k_n_sc47__sc48)*Y(Sc47)*state.rho - screened_rates(k_n_sc48__p_ca48)*Y(Sc48)*state.rho - screened_rates(k_n_sc48__sc49)*Y(Sc48)*state.rho - screened_rates(k_n_si28__he4_mg25)*Y(Si28)*state.rho - screened_rates(k_n_si28__si29)*Y(Si28)*state.rho - screened_rates(k_n_si29__he4_mg26)*Y(Si29)*state.rho - screened_rates(k_n_si29__si30)*Y(Si29)*state.rho - screened_rates(k_n_si30__si31)*Y(Si30)*state.rho - screened_rates(k_n_si31__si32)*Y(Si31)*state.rho - screened_rates(k_n_ti44__he4_ca41)*Y(Ti44)*state.rho - screened_rates(k_n_ti44__p_sc44)*Y(Ti44)*state.rho - screened_rates(k_n_ti44__ti45)*Y(Ti44)*state.rho - screened_rates(k_n_ti45__he4_ca42)*Y(Ti45)*state.rho - screened_rates(k_n_ti45__p_sc45)*Y(Ti45)*state.rho - screened_rates(k_n_ti45__ti46)*Y(Ti45)*state.rho - screened_rates(k_n_ti46__he4_ca43)*Y(Ti46)*state.rho - screened_rates(k_n_ti46__p_sc46)*Y(Ti46)*state.rho - screened_rates(k_n_ti46__ti47)*Y(Ti46)*state.rho - screened_rates(k_n_ti47__he4_ca44)*Y(Ti47)*state.rho - screened_rates(k_n_ti47__p_sc47)*Y(Ti47)*state.rho - screened_rates(k_n_ti47__ti48)*Y(Ti47)*state.rho - screened_rates(k_n_ti48__he4_ca45)*Y(Ti48)*state.rho - screened_rates(k_n_ti48__p_sc48)*Y(Ti48)*state.rho - screened_rates(k_n_ti48__ti49)*Y(Ti48)*state.rho - screened_rates(k_n_ti49__he4_ca46)*Y(Ti49)*state.rho - screened_rates(k_n_ti49__p_sc49)*Y(Ti49)*state.rho - screened_rates(k_n_ti49__ti50)*Y(Ti49)*state.rho - screened_rates(k_n_ti50__he4_ca47)*Y(Ti50)*state.rho - screened_rates(k_n_ti50__ti51)*Y(Ti50)*state.rho - screened_rates(k_n_ti51__he4_ca48)*Y(Ti51)*state.rho - screened_rates(k_n_v46__he4_sc43)*Y(V46)*state.rho - screened_rates(k_n_v46__p_ti46)*Y(V46)*state.rho - screened_rates(k_n_v46__v47)*Y(V46)*state.rho - screened_rates(k_n_v47__he4_sc44)*Y(V47)*state.rho - screened_rates(k_n_v47__p_ti47)*Y(V47)*state.rho - screened_rates(k_n_v47__v48)*Y(V47)*state.rho - screened_rates(k_n_v48__he4_sc45)*Y(V48)*state.rho - screened_rates(k_n_v48__p_ti48)*Y(V48)*state.rho - screened_rates(k_n_v48__v49)*Y(V48)*state.rho - screened_rates(k_n_v49__he4_sc46)*Y(V49)*state.rho - screened_rates(k_n_v49__p_ti49)*Y(V49)*state.rho - screened_rates(k_n_v49__v50)*Y(V49)*state.rho - screened_rates(k_n_v50__he4_sc47)*Y(V50)*state.rho - screened_rates(k_n_v50__p_ti50)*Y(V50)*state.rho - screened_rates(k_n_v50__v51)*Y(V50)*state.rho - screened_rates(k_n_v51__he4_sc48)*Y(V51)*state.rho - screened_rates(k_n_v51__p_ti51)*Y(V51)*state.rho - screened_rates(k_n_v51__v52)*Y(V51)*state.rho - screened_rates(k_n_v52__he4_sc49)*Y(V52)*state.rho - screened_rates(k_n_zn59__he4_ni56)*Y(Zn59)*state.rho - screened_rates(k_n_zn59__p_cu59)*Y(Zn59)*state.rho - screened_rates(k_n_zn59__zn60)*Y(Zn59)*state.rho - screened_rates(k_n_zn60__he4_ni57)*Y(Zn60)*state.rho - screened_rates(k_n_zn60__p_cu60)*Y(Zn60)*state.rho - screened_rates(k_n_zn60__zn61)*Y(Zn60)*state.rho - screened_rates(k_n_zn61__he4_ni58)*Y(Zn61)*state.rho - screened_rates(k_n_zn61__p_cu61)*Y(Zn61)*state.rho - screened_rates(k_n_zn61__zn62)*Y(Zn61)*state.rho - screened_rates(k_n_zn62__he4_ni59)*Y(Zn62)*state.rho - screened_rates(k_n_zn62__p_cu62)*Y(Zn62)*state.rho - screened_rates(k_n_zn62__zn63)*Y(Zn62)*state.rho - screened_rates(k_n_zn63__he4_ni60)*Y(Zn63)*state.rho - screened_rates(k_n_zn63__p_cu63)*Y(Zn63)*state.rho - screened_rates(k_n_zn63__zn64)*Y(Zn63)*state.rho - screened_rates(k_n_zn64__he4_ni61)*Y(Zn64)*state.rho - screened_rates(k_n_zn64__p_cu64)*Y(Zn64)*state.rho - screened_rates(k_n_zn64__zn65)*Y(Zn64)*state.rho - screened_rates(k_n_zn65__he4_ni62)*Y(Zn65)*state.rho - screened_rates(k_n_zn65__p_cu65)*Y(Zn65)*state.rho - screened_rates(k_n_zn65__zn66)*Y(Zn65)*state.rho - screened_rates(k_n_zn66__he4_ni63)*Y(Zn66)*state.rho;
    jac.set(N, N, scratch);

    scratch = -screened_rates(k_n_p__d)*Y(N)*state.rho - screened_rates(k_n_p_he4__li6)*Y(He4)*Y(N)*std::pow(state.rho, 2) - 0.5*screened_rates(k_n_p_he4_he4__he3_li7)*std::pow(Y(He4), 2)*Y(N)*std::pow(state.rho, 3) - 0.5*screened_rates(k_n_p_he4_he4__p_be9)*std::pow(Y(He4), 2)*Y(N)*std::pow(state.rho, 3) - 1.0*screened_rates(k_n_p_p__p_d)*Y(N)*Y(H1)*std::pow(state.rho, 2) + screened_rates(k_p_ar37__n_k37)*Y(Ar37)*state.rho + screened_rates(k_p_ar38__n_k38)*Y(Ar38)*state.rho + screened_rates(k_p_ar39__n_k39)*Y(Ar39)*state.rho + screened_rates(k_p_ar40__n_k40)*Y(Ar40)*state.rho + screened_rates(k_p_be9__n_p_he4_he4)*Y(Be9)*state.rho + screened_rates(k_p_c13__n_n13)*Y(C13)*state.rho + screened_rates(k_p_c14__n_n14)*Y(C14)*state.rho + screened_rates(k_p_ca43__n_sc43)*Y(Ca43)*state.rho + screened_rates(k_p_ca44__n_sc44)*Y(Ca44)*state.rho + screened_rates(k_p_ca45__n_sc45)*Y(Ca45)*state.rho + screened_rates(k_p_ca46__n_sc46)*Y(Ca46)*state.rho + screened_rates(k_p_ca47__n_sc47)*Y(Ca47)*state.rho + screened_rates(k_p_ca48__n_sc48)*Y(Ca48)*state.rho + screened_rates(k_p_cl36__n_ar36)*Y(Cl36)*state.rho + screened_rates(k_p_cl37__n_ar37)*Y(Cl37)*state.rho + screened_rates(k_p_co56__n_ni56)*Y(Co56)*state.rho + screened_rates(k_p_co57__n_ni57)*Y(Co57)*state.rho + screened_rates(k_p_co58__n_ni58)*Y(Co58)*state.rho + screened_rates(k_p_co59__n_ni59)*Y(Co59)*state.rho + screened_rates(k_p_cr50__n_mn50)*Y(Cr50)*state.rho + screened_rates(k_p_cr51__n_mn51)*Y(Cr51)*state.rho + screened_rates(k_p_cr52__n_mn52)*Y(Cr52)*state.rho + screened_rates(k_p_cr53__n_mn53)*Y(Cr53)*state.rho + screened_rates(k_p_cr54__n_mn54)*Y(Cr54)*state.rho + screened_rates(k_p_cu59__n_zn59)*Y(Cu59)*state.rho + screened_rates(k_p_cu60__n_zn60)*Y(Cu60)*state.rho + screened_rates(k_p_cu61__n_zn61)*Y(Cu61)*state.rho + screened_rates(k_p_cu62__n_zn62)*Y(Cu62)*state.rho + screened_rates(k_p_cu63__n_zn63)*Y(Cu63)*state.rho + screened_rates(k_p_cu64__n_zn64)*Y(Cu64)*state.rho + screened_rates(k_p_cu65__n_zn65)*Y(Cu65)*state.rho + screened_rates(k_p_d__n_p_p)*Y(H2)*state.rho + screened_rates(k_p_f18__n_ne18)*Y(F18)*state.rho + screened_rates(k_p_f19__n_ne19)*Y(F19)*state.rho + screened_rates(k_p_fe53__n_co53)*Y(Fe53)*state.rho + screened_rates(k_p_fe54__n_co54)*Y(Fe54)*state.rho + screened_rates(k_p_fe55__n_co55)*Y(Fe55)*state.rho + screened_rates(k_p_fe56__n_co56)*Y(Fe56)*state.rho + screened_rates(k_p_fe57__n_co57)*Y(Fe57)*state.rho + screened_rates(k_p_fe58__n_co58)*Y(Fe58)*state.rho + screened_rates(k_p_ga63__n_ge63)*Y(Ga63)*state.rho + screened_rates(k_p_ga64__n_ge64)*Y(Ga64)*state.rho + 0.5*screened_rates(k_p_he4_he4__n_b8)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) + screened_rates(k_p_k40__n_ca40)*Y(K40)*state.rho + screened_rates(k_p_k41__n_ca41)*Y(K41)*state.rho + screened_rates(k_p_li7__n_be7)*Y(Li7)*state.rho + screened_rates(k_p_mg25__n_al25)*Y(Mg25)*state.rho + screened_rates(k_p_mg26__n_al26)*Y(Mg26)*state.rho + screened_rates(k_p_mn52__n_fe52)*Y(Mn52)*state.rho + screened_rates(k_p_mn53__n_fe53)*Y(Mn53)*state.rho + screened_rates(k_p_mn54__n_fe54)*Y(Mn54)*state.rho + screened_rates(k_p_mn55__n_fe55)*Y(Mn55)*state.rho + screened_rates(k_p_n14__n_o14)*Y(N14)*state.rho + screened_rates(k_p_n15__n_o15)*Y(N15)*state.rho + screened_rates(k_p_na23__n_mg23)*Y(Na23)*state.rho + screened_rates(k_p_ne21__n_na21)*Y(Ne21)*state.rho + screened_rates(k_p_ne22__n_na22)*Y(Ne22)*state.rho + screened_rates(k_p_ni57__n_cu57)*Y(Ni57)*state.rho + screened_rates(k_p_ni58__n_cu58)*Y(Ni58)*state.rho + screened_rates(k_p_ni59__n_cu59)*Y(Ni59)*state.rho + screened_rates(k_p_ni60__n_cu60)*Y(Ni60)*state.rho + screened_rates(k_p_ni61__n_cu61)*Y(Ni61)*state.rho + screened_rates(k_p_ni62__n_cu62)*Y(Ni62)*state.rho + screened_rates(k_p_ni63__n_cu63)*Y(Ni63)*state.rho + screened_rates(k_p_ni64__n_cu64)*Y(Ni64)*state.rho + screened_rates(k_p_o17__n_f17)*Y(O17)*state.rho + screened_rates(k_p_o18__n_f18)*Y(O18)*state.rho + screened_rates(k_p_p32__n_s32)*Y(P32)*state.rho + screened_rates(k_p_p33__n_s33)*Y(P33)*state.rho + screened_rates(k_p_s33__n_cl33)*Y(S33)*state.rho + screened_rates(k_p_s34__n_cl34)*Y(S34)*state.rho + screened_rates(k_p_s35__n_cl35)*Y(S35)*state.rho + screened_rates(k_p_s36__n_cl36)*Y(S36)*state.rho + screened_rates(k_p_sc44__n_ti44)*Y(Sc44)*state.rho + screened_rates(k_p_sc45__n_ti45)*Y(Sc45)*state.rho + screened_rates(k_p_sc46__n_ti46)*Y(Sc46)*state.rho + screened_rates(k_p_sc47__n_ti47)*Y(Sc47)*state.rho + screened_rates(k_p_sc48__n_ti48)*Y(Sc48)*state.rho + screened_rates(k_p_sc49__n_ti49)*Y(Sc49)*state.rho + screened_rates(k_p_si29__n_p29)*Y(Si29)*state.rho + screened_rates(k_p_si30__n_p30)*Y(Si30)*state.rho + screened_rates(k_p_si31__n_p31)*Y(Si31)*state.rho + screened_rates(k_p_si32__n_p32)*Y(Si32)*state.rho + screened_rates(k_p_ti46__n_v46)*Y(Ti46)*state.rho + screened_rates(k_p_ti47__n_v47)*Y(Ti47)*state.rho + screened_rates(k_p_ti48__n_v48)*Y(Ti48)*state.rho + screened_rates(k_p_ti49__n_v49)*Y(Ti49)*state.rho + screened_rates(k_p_ti50__n_v50)*Y(Ti50)*state.rho + screened_rates(k_p_ti51__n_v51)*Y(Ti51)*state.rho + screened_rates(k_p_v48__n_cr48)*Y(V48)*state.rho + screened_rates(k_p_v49__n_cr49)*Y(V49)*state.rho + screened_rates(k_p_v50__n_cr50)*Y(V50)*state.rho + screened_rates(k_p_v51__n_cr51)*Y(V51)*state.rho + screened_rates(k_p_v52__n_cr52)*Y(V52)*state.rho + screened_rates(k_p_zn62__n_ga62)*Y(Zn62)*state.rho + screened_rates(k_p_zn63__n_ga63)*Y(Zn63)*state.rho + screened_rates(k_p_zn64__n_ga64)*Y(Zn64)*state.rho;
    jac.set(N, H1, scratch);

    scratch = screened_rates(k_d__n_p) + screened_rates(k_d_c13__n_n14)*Y(C13)*state.rho + screened_rates(k_d_c14__n_n15)*Y(C14)*state.rho + 1.0*screened_rates(k_d_d__n_he3)*Y(H2)*state.rho + screened_rates(k_d_li6__n_be7)*Y(Li6)*state.rho + screened_rates(k_d_li7__n_he4_he4)*Y(Li7)*state.rho + screened_rates(k_p_d__n_p_p)*Y(H1)*state.rho;
    jac.set(N, H2, scratch);

    scratch = screened_rates(k_he3_li7__n_p_he4_he4)*Y(Li7)*state.rho - screened_rates(k_n_he3__d_d)*Y(N)*state.rho - screened_rates(k_n_he3__he4)*Y(N)*state.rho;
    jac.set(N, He3, scratch);

    scratch = screened_rates(k_he4__n_he3) + screened_rates(k_he4_al26__n_p29)*Y(Al26)*state.rho + screened_rates(k_he4_al27__n_p30)*Y(Al27)*state.rho + screened_rates(k_he4_ar37__n_ca40)*Y(Ar37)*state.rho + screened_rates(k_he4_ar38__n_ca41)*Y(Ar38)*state.rho + screened_rates(k_he4_ar39__n_ca42)*Y(Ar39)*state.rho + screened_rates(k_he4_ar40__n_ca43)*Y(Ar40)*state.rho + screened_rates(k_he4_b10__n_n13)*Y(B10)*state.rho + screened_rates(k_he4_b11__n_n14)*Y(B11)*state.rho + screened_rates(k_he4_be9__n_c12)*Y(Be9)*state.rho + screened_rates(k_he4_c12__n_o15)*Y(C12)*state.rho + screened_rates(k_he4_c13__n_o16)*Y(C13)*state.rho + screened_rates(k_he4_c14__n_o17)*Y(C14)*state.rho + screened_rates(k_he4_ca41__n_ti44)*Y(Ca41)*state.rho + screened_rates(k_he4_ca42__n_ti45)*Y(Ca42)*state.rho + screened_rates(k_he4_ca43__n_ti46)*Y(Ca43)*state.rho + screened_rates(k_he4_ca44__n_ti47)*Y(Ca44)*state.rho + screened_rates(k_he4_ca45__n_ti48)*Y(Ca45)*state.rho + screened_rates(k_he4_ca46__n_ti49)*Y(Ca46)*state.rho + screened_rates(k_he4_ca47__n_ti50)*Y(Ca47)*state.rho + screened_rates(k_he4_ca48__n_ti51)*Y(Ca48)*state.rho + screened_rates(k_he4_cl34__n_k37)*Y(Cl34)*state.rho + screened_rates(k_he4_cl35__n_k38)*Y(Cl35)*state.rho + screened_rates(k_he4_cl36__n_k39)*Y(Cl36)*state.rho + screened_rates(k_he4_cl37__n_k40)*Y(Cl37)*state.rho + screened_rates(k_he4_co54__n_cu57)*Y(Co54)*state.rho + screened_rates(k_he4_co55__n_cu58)*Y(Co55)*state.rho + screened_rates(k_he4_co56__n_cu59)*Y(Co56)*state.rho + screened_rates(k_he4_co57__n_cu60)*Y(Co57)*state.rho + screened_rates(k_he4_co58__n_cu61)*Y(Co58)*state.rho + screened_rates(k_he4_co59__n_cu62)*Y(Co59)*state.rho + screened_rates(k_he4_cr49__n_fe52)*Y(Cr49)*state.rho + screened_rates(k_he4_cr50__n_fe53)*Y(Cr50)*state.rho + screened_rates(k_he4_cr51__n_fe54)*Y(Cr51)*state.rho + screened_rates(k_he4_cr52__n_fe55)*Y(Cr52)*state.rho + screened_rates(k_he4_cr53__n_fe56)*Y(Cr53)*state.rho + screened_rates(k_he4_cr54__n_fe57)*Y(Cr54)*state.rho + screened_rates(k_he4_cu59__n_ga62)*Y(Cu59)*state.rho + screened_rates(k_he4_cu60__n_ga63)*Y(Cu60)*state.rho + screened_rates(k_he4_cu61__n_ga64)*Y(Cu61)*state.rho + screened_rates(k_he4_f18__n_na21)*Y(F18)*state.rho + screened_rates(k_he4_f19__n_na22)*Y(F19)*state.rho + screened_rates(k_he4_fe53__n_ni56)*Y(Fe53)*state.rho + screened_rates(k_he4_fe54__n_ni57)*Y(Fe54)*state.rho + screened_rates(k_he4_fe55__n_ni58)*Y(Fe55)*state.rho + screened_rates(k_he4_fe56__n_ni59)*Y(Fe56)*state.rho + screened_rates(k_he4_fe57__n_ni60)*Y(Fe57)*state.rho + screened_rates(k_he4_fe58__n_ni61)*Y(Fe58)*state.rho + 1.0*screened_rates(k_he4_he4__n_be7)*Y(He4)*state.rho + screened_rates(k_he4_k40__n_sc43)*Y(K40)*state.rho + screened_rates(k_he4_k41__n_sc44)*Y(K41)*state.rho + screened_rates(k_he4_li7__n_b10)*Y(Li7)*state.rho + screened_rates(k_he4_mg25__n_si28)*Y(Mg25)*state.rho + screened_rates(k_he4_mg26__n_si29)*Y(Mg26)*state.rho + screened_rates(k_he4_mn50__n_co53)*Y(Mn50)*state.rho + screened_rates(k_he4_mn51__n_co54)*Y(Mn51)*state.rho + screened_rates(k_he4_mn52__n_co55)*Y(Mn52)*state.rho + screened_rates(k_he4_mn53__n_co56)*Y(Mn53)*state.rho + screened_rates(k_he4_mn54__n_co57)*Y(Mn54)*state.rho + screened_rates(k_he4_mn55__n_co58)*Y(Mn55)*state.rho + screened_rates(k_he4_n14__n_f17)*Y(N14)*state.rho + screened_rates(k_he4_n15__n_f18)*Y(N15)*state.rho + screened_rates(k_he4_na22__n_al25)*Y(Na22)*state.rho + screened_rates(k_he4_na23__n_al26)*Y(Na23)*state.rho + screened_rates(k_he4_ne20__n_mg23)*Y(Ne20)*state.rho + screened_rates(k_he4_ne21__n_mg24)*Y(Ne21)*state.rho + screened_rates(k_he4_ne22__n_mg25)*Y(Ne22)*state.rho + screened_rates(k_he4_ni56__n_zn59)*Y(Ni56)*state.rho + screened_rates(k_he4_ni57__n_zn60)*Y(Ni57)*state.rho + screened_rates(k_he4_ni58__n_zn61)*Y(Ni58)*state.rho + screened_rates(k_he4_ni59__n_zn62)*Y(Ni59)*state.rho + screened_rates(k_he4_ni60__n_zn63)*Y(Ni60)*state.rho + screened_rates(k_he4_ni61__n_zn64)*Y(Ni61)*state.rho + screened_rates(k_he4_ni62__n_zn65)*Y(Ni62)*state.rho + screened_rates(k_he4_ni63__n_zn66)*Y(Ni63)*state.rho + screened_rates(k_he4_o15__n_ne18)*Y(O15)*state.rho + screened_rates(k_he4_o16__n_ne19)*Y(O16)*state.rho + screened_rates(k_he4_o17__n_ne20)*Y(O17)*state.rho + screened_rates(k_he4_o18__n_ne21)*Y(O18)*state.rho + screened_rates(k_he4_p30__n_cl33)*Y(P30)*state.rho + screened_rates(k_he4_p31__n_cl34)*Y(P31)*state.rho + screened_rates(k_he4_p32__n_cl35)*Y(P32)*state.rho + screened_rates(k_he4_p33__n_cl36)*Y(P33)*state.rho + screened_rates(k_he4_s33__n_ar36)*Y(S33)*state.rho + screened_rates(k_he4_s34__n_ar37)*Y(S34)*state.rho + screened_rates(k_he4_s35__n_ar38)*Y(S35)*state.rho + screened_rates(k_he4_s36__n_ar39)*Y(S36)*state.rho + screened_rates(k_he4_sc43__n_v46)*Y(Sc43)*state.rho + screened_rates(k_he4_sc44__n_v47)*Y(Sc44)*state.rho + screened_rates(k_he4_sc45__n_v48)*Y(Sc45)*state.rho + screened_rates(k_he4_sc46__n_v49)*Y(Sc46)*state.rho + screened_rates(k_he4_sc47__n_v50)*Y(Sc47)*state.rho + screened_rates(k_he4_sc48__n_v51)*Y(Sc48)*state.rho + screened_rates(k_he4_sc49__n_v52)*Y(Sc49)*state.rho + screened_rates(k_he4_si29__n_s32)*Y(Si29)*state.rho + screened_rates(k_he4_si30__n_s33)*Y(Si30)*state.rho + screened_rates(k_he4_si31__n_s34)*Y(Si31)*state.rho + screened_rates(k_he4_si32__n_s35)*Y(Si32)*state.rho + screened_rates(k_he4_ti45__n_cr48)*Y(Ti45)*state.rho + screened_rates(k_he4_ti46__n_cr49)*Y(Ti46)*state.rho + screened_rates(k_he4_ti47__n_cr50)*Y(Ti47)*state.rho + screened_rates(k_he4_ti48__n_cr51)*Y(Ti48)*state.rho + screened_rates(k_he4_ti49__n_cr52)*Y(Ti49)*state.rho + screened_rates(k_he4_ti50__n_cr53)*Y(Ti50)*state.rho + screened_rates(k_he4_ti51__n_cr54)*Y(Ti51)*state.rho + screened_rates(k_he4_v47__n_mn50)*Y(V47)*state.rho + screened_rates(k_he4_v48__n_mn51)*Y(V48)*state.rho + screened_rates(k_he4_v49__n_mn52)*Y(V49)*state.rho + screened_rates(k_he4_v50__n_mn53)*Y(V50)*state.rho + screened_rates(k_he4_v51__n_mn54)*Y(V51)*state.rho + screened_rates(k_he4_v52__n_mn55)*Y(V52)*state.rho + screened_rates(k_he4_zn60__n_ge63)*Y(Zn60)*state.rho + screened_rates(k_he4_zn61__n_ge64)*Y(Zn61)*state.rho - 1.0*screened_rates(k_n_he4_he4__be9)*Y(He4)*Y(N)*std::pow(state.rho, 2) - 1.0*screened_rates(k_n_he4_he4__d_li7)*Y(He4)*Y(N)*std::pow(state.rho, 2) - screened_rates(k_n_p_he4__li6)*Y(N)*Y(H1)*std::pow(state.rho, 2) - 1.0*screened_rates(k_n_p_he4_he4__he3_li7)*Y(He4)*Y(N)*Y(H1)*std::pow(state.rho, 3) - 1.0*screened_rates(k_n_p_he4_he4__p_be9)*Y(He4)*Y(N)*Y(H1)*std::pow(state.rho, 3) + 1.0*screened_rates(k_p_he4_he4__n_b8)*Y(He4)*Y(H1)*std::pow(state.rho, 2);
    jac.set(N, He4, scratch);

    scratch = screened_rates(k_d_li6__n_be7)*Y(H2)*state.rho + screened_rates(k_li6__n_p_he4) - screened_rates(k_n_li6__li7)*Y(N)*state.rho;
    jac.set(N, Li6, scratch);

    scratch = screened_rates(k_d_li7__n_he4_he4)*Y(H2)*state.rho + screened_rates(k_he3_li7__n_p_he4_he4)*Y(He3)*state.rho + screened_rates(k_he4_li7__n_b10)*Y(He4)*state.rho + screened_rates(k_li7__n_li6) + screened_rates(k_p_li7__n_be7)*Y(H1)*state.rho;
    jac.set(N, Li7, scratch);

    scratch = -screened_rates(k_n_be7__d_li6)*Y(N)*state.rho - screened_rates(k_n_be7__he4_he4)*Y(N)*state.rho - screened_rates(k_n_be7__p_li7)*Y(N)*state.rho;
    jac.set(N, Be7, scratch);

    scratch = screened_rates(k_be9__n_he4_he4) + screened_rates(k_he4_be9__n_c12)*Y(He4)*state.rho + screened_rates(k_p_be9__n_p_he4_he4)*Y(H1)*state.rho;
    jac.set(N, Be9, scratch);

    scratch = -screened_rates(k_n_b8__p_he4_he4)*Y(N)*state.rho;
    jac.set(N, B8, scratch);

    scratch = screened_rates(k_he4_b10__n_n13)*Y(He4)*state.rho - screened_rates(k_n_b10__b11)*Y(N)*state.rho - screened_rates(k_n_b10__he4_li7)*Y(N)*state.rho;
    jac.set(N, B10, scratch);

    scratch = screened_rates(k_b11__n_b10) + screened_rates(k_he4_b11__n_n14)*Y(He4)*state.rho;
    jac.set(N, B11, scratch);

    scratch = 1.0*screened_rates(k_c12_c12__n_mg23)*Y(C12)*state.rho + screened_rates(k_he4_c12__n_o15)*Y(He4)*state.rho - screened_rates(k_n_c12__c13)*Y(N)*state.rho - screened_rates(k_n_c12__he4_be9)*Y(N)*state.rho;
    jac.set(N, C12, scratch);

    scratch = screened_rates(k_c13__n_c12) + screened_rates(k_d_c13__n_n14)*Y(H2)*state.rho + screened_rates(k_he4_c13__n_o16)*Y(He4)*state.rho - screened_rates(k_n_c13__c14)*Y(N)*state.rho + screened_rates(k_p_c13__n_n13)*Y(H1)*state.rho;
    jac.set(N, C13, scratch);

    scratch = screened_rates(k_c14__n_c13) + screened_rates(k_d_c14__n_n15)*Y(H2)*state.rho + screened_rates(k_he4_c14__n_o17)*Y(He4)*state.rho + screened_rates(k_p_c14__n_n14)*Y(H1)*state.rho;
    jac.set(N, C14, scratch);

    scratch = -screened_rates(k_n_n13__he4_b10)*Y(N)*state.rho - screened_rates(k_n_n13__n14)*Y(N)*state.rho - screened_rates(k_n_n13__p_c13)*Y(N)*state.rho;
    jac.set(N, N13, scratch);

    scratch = screened_rates(k_he4_n14__n_f17)*Y(He4)*state.rho + screened_rates(k_n14__n_n13) - screened_rates(k_n_n14__d_c13)*Y(N)*state.rho - screened_rates(k_n_n14__he4_b11)*Y(N)*state.rho - screened_rates(k_n_n14__n15)*Y(N)*state.rho - screened_rates(k_n_n14__p_c14)*Y(N)*state.rho + screened_rates(k_p_n14__n_o14)*Y(H1)*state.rho;
    jac.set(N, N14, scratch);

    scratch = screened_rates(k_he4_n15__n_f18)*Y(He4)*state.rho + screened_rates(k_n15__n_n14) - screened_rates(k_n_n15__d_c14)*Y(N)*state.rho + screened_rates(k_p_n15__n_o15)*Y(H1)*state.rho;
    jac.set(N, N15, scratch);

    scratch = -screened_rates(k_n_o14__o15)*Y(N)*state.rho - screened_rates(k_n_o14__p_n14)*Y(N)*state.rho;
    jac.set(N, O14, scratch);

    scratch = screened_rates(k_he4_o15__n_ne18)*Y(He4)*state.rho - screened_rates(k_n_o15__he4_c12)*Y(N)*state.rho - screened_rates(k_n_o15__o16)*Y(N)*state.rho - screened_rates(k_n_o15__p_n15)*Y(N)*state.rho + screened_rates(k_o15__n_o14);
    jac.set(N, O15, scratch);

    scratch = screened_rates(k_he4_o16__n_ne19)*Y(He4)*state.rho - screened_rates(k_n_o16__he4_c13)*Y(N)*state.rho - screened_rates(k_n_o16__o17)*Y(N)*state.rho + screened_rates(k_o16__n_o15);
    jac.set(N, O16, scratch);

    scratch = screened_rates(k_he4_o17__n_ne20)*Y(He4)*state.rho - screened_rates(k_n_o17__he4_c14)*Y(N)*state.rho - screened_rates(k_n_o17__o18)*Y(N)*state.rho + screened_rates(k_o17__n_o16) + screened_rates(k_p_o17__n_f17)*Y(H1)*state.rho;
    jac.set(N, O17, scratch);

    scratch = screened_rates(k_he4_o18__n_ne21)*Y(He4)*state.rho + screened_rates(k_o18__n_o17) + screened_rates(k_p_o18__n_f18)*Y(H1)*state.rho;
    jac.set(N, O18, scratch);

    scratch = -screened_rates(k_n_f17__f18)*Y(N)*state.rho - screened_rates(k_n_f17__he4_n14)*Y(N)*state.rho - screened_rates(k_n_f17__p_o17)*Y(N)*state.rho;
    jac.set(N, F17, scratch);

    scratch = screened_rates(k_f18__n_f17) + screened_rates(k_he4_f18__n_na21)*Y(He4)*state.rho - screened_rates(k_n_f18__f19)*Y(N)*state.rho - screened_rates(k_n_f18__he4_n15)*Y(N)*state.rho - screened_rates(k_n_f18__p_o18)*Y(N)*state.rho + screened_rates(k_p_f18__n_ne18)*Y(H1)*state.rho;
    jac.set(N, F18, scratch);

    scratch = screened_rates(k_f19__n_f18) + screened_rates(k_he4_f19__n_na22)*Y(He4)*state.rho + screened_rates(k_p_f19__n_ne19)*Y(H1)*state.rho;
    jac.set(N, F19, scratch);

    scratch = -screened_rates(k_n_ne18__he4_o15)*Y(N)*state.rho - screened_rates(k_n_ne18__ne19)*Y(N)*state.rho - screened_rates(k_n_ne18__p_f18)*Y(N)*state.rho;
    jac.set(N, Ne18, scratch);

    scratch = -screened_rates(k_n_ne19__he4_o16)*Y(N)*state.rho - screened_rates(k_n_ne19__ne20)*Y(N)*state.rho - screened_rates(k_n_ne19__p_f19)*Y(N)*state.rho + screened_rates(k_ne19__n_ne18);
    jac.set(N, Ne19, scratch);

    scratch = screened_rates(k_he4_ne20__n_mg23)*Y(He4)*state.rho - screened_rates(k_n_ne20__he4_o17)*Y(N)*state.rho - screened_rates(k_n_ne20__ne21)*Y(N)*state.rho + screened_rates(k_ne20__n_ne19);
    jac.set(N, Ne20, scratch);

    scratch = screened_rates(k_he4_ne21__n_mg24)*Y(He4)*state.rho - screened_rates(k_n_ne21__he4_o18)*Y(N)*state.rho - screened_rates(k_n_ne21__ne22)*Y(N)*state.rho + screened_rates(k_ne21__n_ne20) + screened_rates(k_p_ne21__n_na21)*Y(H1)*state.rho;
    jac.set(N, Ne21, scratch);

    scratch = screened_rates(k_he4_ne22__n_mg25)*Y(He4)*state.rho + screened_rates(k_ne22__n_ne21) + screened_rates(k_p_ne22__n_na22)*Y(H1)*state.rho;
    jac.set(N, Ne22, scratch);

    scratch = -screened_rates(k_n_na21__he4_f18)*Y(N)*state.rho - screened_rates(k_n_na21__na22)*Y(N)*state.rho - screened_rates(k_n_na21__p_ne21)*Y(N)*state.rho;
    jac.set(N, Na21, scratch);

    scratch = screened_rates(k_he4_na22__n_al25)*Y(He4)*state.rho - screened_rates(k_n_na22__he4_f19)*Y(N)*state.rho - screened_rates(k_n_na22__na23)*Y(N)*state.rho - screened_rates(k_n_na22__p_ne22)*Y(N)*state.rho + screened_rates(k_na22__n_na21);
    jac.set(N, Na22, scratch);

    scratch = screened_rates(k_he4_na23__n_al26)*Y(He4)*state.rho + screened_rates(k_na23__n_na22) + screened_rates(k_p_na23__n_mg23)*Y(H1)*state.rho;
    jac.set(N, Na23, scratch);

    scratch = -screened_rates(k_n_mg23__c12_c12)*Y(N)*state.rho - screened_rates(k_n_mg23__he4_ne20)*Y(N)*state.rho - screened_rates(k_n_mg23__mg24)*Y(N)*state.rho - screened_rates(k_n_mg23__p_na23)*Y(N)*state.rho;
    jac.set(N, Mg23, scratch);

    scratch = screened_rates(k_mg24__n_mg23) - screened_rates(k_n_mg24__he4_ne21)*Y(N)*state.rho - screened_rates(k_n_mg24__mg25)*Y(N)*state.rho;
    jac.set(N, Mg24, scratch);

    scratch = screened_rates(k_he4_mg25__n_si28)*Y(He4)*state.rho + screened_rates(k_mg25__n_mg24) - screened_rates(k_n_mg25__he4_ne22)*Y(N)*state.rho - screened_rates(k_n_mg25__mg26)*Y(N)*state.rho + screened_rates(k_p_mg25__n_al25)*Y(H1)*state.rho;
    jac.set(N, Mg25, scratch);

    scratch = screened_rates(k_he4_mg26__n_si29)*Y(He4)*state.rho + screened_rates(k_mg26__n_mg25) + screened_rates(k_p_mg26__n_al26)*Y(H1)*state.rho;
    jac.set(N, Mg26, scratch);

    scratch = -screened_rates(k_n_al25__al26)*Y(N)*state.rho - screened_rates(k_n_al25__he4_na22)*Y(N)*state.rho - screened_rates(k_n_al25__p_mg25)*Y(N)*state.rho;
    jac.set(N, Al25, scratch);

    scratch = screened_rates(k_al26__n_al25) + screened_rates(k_he4_al26__n_p29)*Y(He4)*state.rho - screened_rates(k_n_al26__al27)*Y(N)*state.rho - screened_rates(k_n_al26__he4_na23)*Y(N)*state.rho - screened_rates(k_n_al26__p_mg26)*Y(N)*state.rho;
    jac.set(N, Al26, scratch);

    scratch = screened_rates(k_al27__n_al26) + screened_rates(k_he4_al27__n_p30)*Y(He4)*state.rho;
    jac.set(N, Al27, scratch);

    scratch = -screened_rates(k_n_si28__he4_mg25)*Y(N)*state.rho - screened_rates(k_n_si28__si29)*Y(N)*state.rho;
    jac.set(N, Si28, scratch);

    scratch = screened_rates(k_he4_si29__n_s32)*Y(He4)*state.rho - screened_rates(k_n_si29__he4_mg26)*Y(N)*state.rho - screened_rates(k_n_si29__si30)*Y(N)*state.rho + screened_rates(k_p_si29__n_p29)*Y(H1)*state.rho + screened_rates(k_si29__n_si28);
    jac.set(N, Si29, scratch);

    scratch = screened_rates(k_he4_si30__n_s33)*Y(He4)*state.rho - screened_rates(k_n_si30__si31)*Y(N)*state.rho + screened_rates(k_p_si30__n_p30)*Y(H1)*state.rho + screened_rates(k_si30__n_si29);
    jac.set(N, Si30, scratch);

    scratch = screened_rates(k_he4_si31__n_s34)*Y(He4)*state.rho - screened_rates(k_n_si31__si32)*Y(N)*state.rho + screened_rates(k_p_si31__n_p31)*Y(H1)*state.rho + screened_rates(k_si31__n_si30);
    jac.set(N, Si31, scratch);

    scratch = screened_rates(k_he4_si32__n_s35)*Y(He4)*state.rho + screened_rates(k_p_si32__n_p32)*Y(H1)*state.rho + screened_rates(k_si32__n_si31);
    jac.set(N, Si32, scratch);

    scratch = -screened_rates(k_n_p29__he4_al26)*Y(N)*state.rho - screened_rates(k_n_p29__p30)*Y(N)*state.rho - screened_rates(k_n_p29__p_si29)*Y(N)*state.rho;
    jac.set(N, P29, scratch);

    scratch = screened_rates(k_he4_p30__n_cl33)*Y(He4)*state.rho - screened_rates(k_n_p30__he4_al27)*Y(N)*state.rho - screened_rates(k_n_p30__p31)*Y(N)*state.rho - screened_rates(k_n_p30__p_si30)*Y(N)*state.rho + screened_rates(k_p30__n_p29);
    jac.set(N, P30, scratch);

    scratch = screened_rates(k_he4_p31__n_cl34)*Y(He4)*state.rho - screened_rates(k_n_p31__p32)*Y(N)*state.rho - screened_rates(k_n_p31__p_si31)*Y(N)*state.rho + screened_rates(k_p31__n_p30);
    jac.set(N, P31, scratch);

    scratch = screened_rates(k_he4_p32__n_cl35)*Y(He4)*state.rho - screened_rates(k_n_p32__p33)*Y(N)*state.rho - screened_rates(k_n_p32__p_si32)*Y(N)*state.rho + screened_rates(k_p32__n_p31) + screened_rates(k_p_p32__n_s32)*Y(H1)*state.rho;
    jac.set(N, P32, scratch);

    scratch = screened_rates(k_he4_p33__n_cl36)*Y(He4)*state.rho + screened_rates(k_p33__n_p32) + screened_rates(k_p_p33__n_s33)*Y(H1)*state.rho;
    jac.set(N, P33, scratch);

    scratch = -screened_rates(k_n_s32__he4_si29)*Y(N)*state.rho - screened_rates(k_n_s32__p_p32)*Y(N)*state.rho - screened_rates(k_n_s32__s33)*Y(N)*state.rho;
    jac.set(N, S32, scratch);

    scratch = screened_rates(k_he4_s33__n_ar36)*Y(He4)*state.rho - screened_rates(k_n_s33__he4_si30)*Y(N)*state.rho - screened_rates(k_n_s33__p_p33)*Y(N)*state.rho - screened_rates(k_n_s33__s34)*Y(N)*state.rho + screened_rates(k_p_s33__n_cl33)*Y(H1)*state.rho + screened_rates(k_s33__n_s32);
    jac.set(N, S33, scratch);

    scratch = screened_rates(k_he4_s34__n_ar37)*Y(He4)*state.rho - screened_rates(k_n_s34__he4_si31)*Y(N)*state.rho - screened_rates(k_n_s34__s35)*Y(N)*state.rho + screened_rates(k_p_s34__n_cl34)*Y(H1)*state.rho + screened_rates(k_s34__n_s33);
    jac.set(N, S34, scratch);

    scratch = screened_rates(k_he4_s35__n_ar38)*Y(He4)*state.rho - screened_rates(k_n_s35__he4_si32)*Y(N)*state.rho - screened_rates(k_n_s35__s36)*Y(N)*state.rho + screened_rates(k_p_s35__n_cl35)*Y(H1)*state.rho + screened_rates(k_s35__n_s34);
    jac.set(N, S35, scratch);

    scratch = screened_rates(k_he4_s36__n_ar39)*Y(He4)*state.rho + screened_rates(k_p_s36__n_cl36)*Y(H1)*state.rho + screened_rates(k_s36__n_s35);
    jac.set(N, S36, scratch);

    scratch = -screened_rates(k_n_cl33__cl34)*Y(N)*state.rho - screened_rates(k_n_cl33__he4_p30)*Y(N)*state.rho - screened_rates(k_n_cl33__p_s33)*Y(N)*state.rho;
    jac.set(N, Cl33, scratch);

    scratch = screened_rates(k_cl34__n_cl33) + screened_rates(k_he4_cl34__n_k37)*Y(He4)*state.rho - screened_rates(k_n_cl34__cl35)*Y(N)*state.rho - screened_rates(k_n_cl34__he4_p31)*Y(N)*state.rho - screened_rates(k_n_cl34__p_s34)*Y(N)*state.rho;
    jac.set(N, Cl34, scratch);

    scratch = screened_rates(k_cl35__n_cl34) + screened_rates(k_he4_cl35__n_k38)*Y(He4)*state.rho - screened_rates(k_n_cl35__cl36)*Y(N)*state.rho - screened_rates(k_n_cl35__he4_p32)*Y(N)*state.rho - screened_rates(k_n_cl35__p_s35)*Y(N)*state.rho;
    jac.set(N, Cl35, scratch);

    scratch = screened_rates(k_cl36__n_cl35) + screened_rates(k_he4_cl36__n_k39)*Y(He4)*state.rho - screened_rates(k_n_cl36__cl37)*Y(N)*state.rho - screened_rates(k_n_cl36__he4_p33)*Y(N)*state.rho - screened_rates(k_n_cl36__p_s36)*Y(N)*state.rho + screened_rates(k_p_cl36__n_ar36)*Y(H1)*state.rho;
    jac.set(N, Cl36, scratch);

    scratch = screened_rates(k_cl37__n_cl36) + screened_rates(k_he4_cl37__n_k40)*Y(He4)*state.rho + screened_rates(k_p_cl37__n_ar37)*Y(H1)*state.rho;
    jac.set(N, Cl37, scratch);

    scratch = -screened_rates(k_n_ar36__ar37)*Y(N)*state.rho - screened_rates(k_n_ar36__he4_s33)*Y(N)*state.rho - screened_rates(k_n_ar36__p_cl36)*Y(N)*state.rho;
    jac.set(N, Ar36, scratch);

    scratch = screened_rates(k_ar37__n_ar36) + screened_rates(k_he4_ar37__n_ca40)*Y(He4)*state.rho - screened_rates(k_n_ar37__ar38)*Y(N)*state.rho - screened_rates(k_n_ar37__he4_s34)*Y(N)*state.rho - screened_rates(k_n_ar37__p_cl37)*Y(N)*state.rho + screened_rates(k_p_ar37__n_k37)*Y(H1)*state.rho;
    jac.set(N, Ar37, scratch);

    scratch = screened_rates(k_ar38__n_ar37) + screened_rates(k_he4_ar38__n_ca41)*Y(He4)*state.rho - screened_rates(k_n_ar38__ar39)*Y(N)*state.rho - screened_rates(k_n_ar38__he4_s35)*Y(N)*state.rho + screened_rates(k_p_ar38__n_k38)*Y(H1)*state.rho;
    jac.set(N, Ar38, scratch);

    scratch = screened_rates(k_ar39__n_ar38) + screened_rates(k_he4_ar39__n_ca42)*Y(He4)*state.rho - screened_rates(k_n_ar39__ar40)*Y(N)*state.rho - screened_rates(k_n_ar39__he4_s36)*Y(N)*state.rho + screened_rates(k_p_ar39__n_k39)*Y(H1)*state.rho;
    jac.set(N, Ar39, scratch);

    scratch = screened_rates(k_ar40__n_ar39) + screened_rates(k_he4_ar40__n_ca43)*Y(He4)*state.rho + screened_rates(k_p_ar40__n_k40)*Y(H1)*state.rho;
    jac.set(N, Ar40, scratch);

    scratch = -screened_rates(k_n_k37__he4_cl34)*Y(N)*state.rho - screened_rates(k_n_k37__k38)*Y(N)*state.rho - screened_rates(k_n_k37__p_ar37)*Y(N)*state.rho;
    jac.set(N, K37, scratch);

    scratch = screened_rates(k_k38__n_k37) - screened_rates(k_n_k38__he4_cl35)*Y(N)*state.rho - screened_rates(k_n_k38__k39)*Y(N)*state.rho - screened_rates(k_n_k38__p_ar38)*Y(N)*state.rho;
    jac.set(N, K38, scratch);

    scratch = screened_rates(k_k39__n_k38) - screened_rates(k_n_k39__he4_cl36)*Y(N)*state.rho - screened_rates(k_n_k39__k40)*Y(N)*state.rho - screened_rates(k_n_k39__p_ar39)*Y(N)*state.rho;
    jac.set(N, K39, scratch);

    scratch = screened_rates(k_he4_k40__n_sc43)*Y(He4)*state.rho + screened_rates(k_k40__n_k39) - screened_rates(k_n_k40__he4_cl37)*Y(N)*state.rho - screened_rates(k_n_k40__k41)*Y(N)*state.rho - screened_rates(k_n_k40__p_ar40)*Y(N)*state.rho + screened_rates(k_p_k40__n_ca40)*Y(H1)*state.rho;
    jac.set(N, K40, scratch);

    scratch = screened_rates(k_he4_k41__n_sc44)*Y(He4)*state.rho + screened_rates(k_k41__n_k40) + screened_rates(k_p_k41__n_ca41)*Y(H1)*state.rho;
    jac.set(N, K41, scratch);

    scratch = -screened_rates(k_n_ca40__ca41)*Y(N)*state.rho - screened_rates(k_n_ca40__he4_ar37)*Y(N)*state.rho - screened_rates(k_n_ca40__p_k40)*Y(N)*state.rho;
    jac.set(N, Ca40, scratch);

    scratch = screened_rates(k_ca41__n_ca40) + screened_rates(k_he4_ca41__n_ti44)*Y(He4)*state.rho - screened_rates(k_n_ca41__ca42)*Y(N)*state.rho - screened_rates(k_n_ca41__he4_ar38)*Y(N)*state.rho - screened_rates(k_n_ca41__p_k41)*Y(N)*state.rho;
    jac.set(N, Ca41, scratch);

    scratch = screened_rates(k_ca42__n_ca41) + screened_rates(k_he4_ca42__n_ti45)*Y(He4)*state.rho - screened_rates(k_n_ca42__ca43)*Y(N)*state.rho - screened_rates(k_n_ca42__he4_ar39)*Y(N)*state.rho;
    jac.set(N, Ca42, scratch);

    scratch = screened_rates(k_ca43__n_ca42) + screened_rates(k_he4_ca43__n_ti46)*Y(He4)*state.rho - screened_rates(k_n_ca43__ca44)*Y(N)*state.rho - screened_rates(k_n_ca43__he4_ar40)*Y(N)*state.rho + screened_rates(k_p_ca43__n_sc43)*Y(H1)*state.rho;
    jac.set(N, Ca43, scratch);

    scratch = screened_rates(k_ca44__n_ca43) + screened_rates(k_he4_ca44__n_ti47)*Y(He4)*state.rho - screened_rates(k_n_ca44__ca45)*Y(N)*state.rho + screened_rates(k_p_ca44__n_sc44)*Y(H1)*state.rho;
    jac.set(N, Ca44, scratch);

    scratch = screened_rates(k_ca45__n_ca44) + screened_rates(k_he4_ca45__n_ti48)*Y(He4)*state.rho - screened_rates(k_n_ca45__ca46)*Y(N)*state.rho + screened_rates(k_p_ca45__n_sc45)*Y(H1)*state.rho;
    jac.set(N, Ca45, scratch);

    scratch = screened_rates(k_ca46__n_ca45) + screened_rates(k_he4_ca46__n_ti49)*Y(He4)*state.rho - screened_rates(k_n_ca46__ca47)*Y(N)*state.rho + screened_rates(k_p_ca46__n_sc46)*Y(H1)*state.rho;
    jac.set(N, Ca46, scratch);

    scratch = screened_rates(k_ca47__n_ca46) + screened_rates(k_he4_ca47__n_ti50)*Y(He4)*state.rho - screened_rates(k_n_ca47__ca48)*Y(N)*state.rho + screened_rates(k_p_ca47__n_sc47)*Y(H1)*state.rho;
    jac.set(N, Ca47, scratch);

    scratch = screened_rates(k_ca48__n_ca47) + screened_rates(k_he4_ca48__n_ti51)*Y(He4)*state.rho + screened_rates(k_p_ca48__n_sc48)*Y(H1)*state.rho;
    jac.set(N, Ca48, scratch);

    scratch = screened_rates(k_he4_sc43__n_v46)*Y(He4)*state.rho - screened_rates(k_n_sc43__he4_k40)*Y(N)*state.rho - screened_rates(k_n_sc43__p_ca43)*Y(N)*state.rho - screened_rates(k_n_sc43__sc44)*Y(N)*state.rho;
    jac.set(N, Sc43, scratch);

    scratch = screened_rates(k_he4_sc44__n_v47)*Y(He4)*state.rho - screened_rates(k_n_sc44__he4_k41)*Y(N)*state.rho - screened_rates(k_n_sc44__p_ca44)*Y(N)*state.rho - screened_rates(k_n_sc44__sc45)*Y(N)*state.rho + screened_rates(k_p_sc44__n_ti44)*Y(H1)*state.rho + screened_rates(k_sc44__n_sc43);
    jac.set(N, Sc44, scratch);

    scratch = screened_rates(k_he4_sc45__n_v48)*Y(He4)*state.rho - screened_rates(k_n_sc45__p_ca45)*Y(N)*state.rho - screened_rates(k_n_sc45__sc46)*Y(N)*state.rho + screened_rates(k_p_sc45__n_ti45)*Y(H1)*state.rho + screened_rates(k_sc45__n_sc44);
    jac.set(N, Sc45, scratch);

    scratch = screened_rates(k_he4_sc46__n_v49)*Y(He4)*state.rho - screened_rates(k_n_sc46__p_ca46)*Y(N)*state.rho - screened_rates(k_n_sc46__sc47)*Y(N)*state.rho + screened_rates(k_p_sc46__n_ti46)*Y(H1)*state.rho + screened_rates(k_sc46__n_sc45);
    jac.set(N, Sc46, scratch);

    scratch = screened_rates(k_he4_sc47__n_v50)*Y(He4)*state.rho - screened_rates(k_n_sc47__p_ca47)*Y(N)*state.rho - screened_rates(k_n_sc47__sc48)*Y(N)*state.rho + screened_rates(k_p_sc47__n_ti47)*Y(H1)*state.rho + screened_rates(k_sc47__n_sc46);
    jac.set(N, Sc47, scratch);

    scratch = screened_rates(k_he4_sc48__n_v51)*Y(He4)*state.rho - screened_rates(k_n_sc48__p_ca48)*Y(N)*state.rho - screened_rates(k_n_sc48__sc49)*Y(N)*state.rho + screened_rates(k_p_sc48__n_ti48)*Y(H1)*state.rho + screened_rates(k_sc48__n_sc47);
    jac.set(N, Sc48, scratch);

    scratch = screened_rates(k_he4_sc49__n_v52)*Y(He4)*state.rho + screened_rates(k_p_sc49__n_ti49)*Y(H1)*state.rho + screened_rates(k_sc49__n_sc48);
    jac.set(N, Sc49, scratch);

    scratch = -screened_rates(k_n_ti44__he4_ca41)*Y(N)*state.rho - screened_rates(k_n_ti44__p_sc44)*Y(N)*state.rho - screened_rates(k_n_ti44__ti45)*Y(N)*state.rho;
    jac.set(N, Ti44, scratch);

    scratch = screened_rates(k_he4_ti45__n_cr48)*Y(He4)*state.rho - screened_rates(k_n_ti45__he4_ca42)*Y(N)*state.rho - screened_rates(k_n_ti45__p_sc45)*Y(N)*state.rho - screened_rates(k_n_ti45__ti46)*Y(N)*state.rho + screened_rates(k_ti45__n_ti44);
    jac.set(N, Ti45, scratch);

    scratch = screened_rates(k_he4_ti46__n_cr49)*Y(He4)*state.rho - screened_rates(k_n_ti46__he4_ca43)*Y(N)*state.rho - screened_rates(k_n_ti46__p_sc46)*Y(N)*state.rho - screened_rates(k_n_ti46__ti47)*Y(N)*state.rho + screened_rates(k_p_ti46__n_v46)*Y(H1)*state.rho + screened_rates(k_ti46__n_ti45);
    jac.set(N, Ti46, scratch);

    scratch = screened_rates(k_he4_ti47__n_cr50)*Y(He4)*state.rho - screened_rates(k_n_ti47__he4_ca44)*Y(N)*state.rho - screened_rates(k_n_ti47__p_sc47)*Y(N)*state.rho - screened_rates(k_n_ti47__ti48)*Y(N)*state.rho + screened_rates(k_p_ti47__n_v47)*Y(H1)*state.rho + screened_rates(k_ti47__n_ti46);
    jac.set(N, Ti47, scratch);

    scratch = screened_rates(k_he4_ti48__n_cr51)*Y(He4)*state.rho - screened_rates(k_n_ti48__he4_ca45)*Y(N)*state.rho - screened_rates(k_n_ti48__p_sc48)*Y(N)*state.rho - screened_rates(k_n_ti48__ti49)*Y(N)*state.rho + screened_rates(k_p_ti48__n_v48)*Y(H1)*state.rho + screened_rates(k_ti48__n_ti47);
    jac.set(N, Ti48, scratch);

    scratch = screened_rates(k_he4_ti49__n_cr52)*Y(He4)*state.rho - screened_rates(k_n_ti49__he4_ca46)*Y(N)*state.rho - screened_rates(k_n_ti49__p_sc49)*Y(N)*state.rho - screened_rates(k_n_ti49__ti50)*Y(N)*state.rho + screened_rates(k_p_ti49__n_v49)*Y(H1)*state.rho + screened_rates(k_ti49__n_ti48);
    jac.set(N, Ti49, scratch);

    scratch = screened_rates(k_he4_ti50__n_cr53)*Y(He4)*state.rho - screened_rates(k_n_ti50__he4_ca47)*Y(N)*state.rho - screened_rates(k_n_ti50__ti51)*Y(N)*state.rho + screened_rates(k_p_ti50__n_v50)*Y(H1)*state.rho + screened_rates(k_ti50__n_ti49);
    jac.set(N, Ti50, scratch);

    scratch = screened_rates(k_he4_ti51__n_cr54)*Y(He4)*state.rho - screened_rates(k_n_ti51__he4_ca48)*Y(N)*state.rho + screened_rates(k_p_ti51__n_v51)*Y(H1)*state.rho + screened_rates(k_ti51__n_ti50);
    jac.set(N, Ti51, scratch);

    scratch = -screened_rates(k_n_v46__he4_sc43)*Y(N)*state.rho - screened_rates(k_n_v46__p_ti46)*Y(N)*state.rho - screened_rates(k_n_v46__v47)*Y(N)*state.rho;
    jac.set(N, V46, scratch);

    scratch = screened_rates(k_he4_v47__n_mn50)*Y(He4)*state.rho - screened_rates(k_n_v47__he4_sc44)*Y(N)*state.rho - screened_rates(k_n_v47__p_ti47)*Y(N)*state.rho - screened_rates(k_n_v47__v48)*Y(N)*state.rho + screened_rates(k_v47__n_v46);
    jac.set(N, V47, scratch);

    scratch = screened_rates(k_he4_v48__n_mn51)*Y(He4)*state.rho - screened_rates(k_n_v48__he4_sc45)*Y(N)*state.rho - screened_rates(k_n_v48__p_ti48)*Y(N)*state.rho - screened_rates(k_n_v48__v49)*Y(N)*state.rho + screened_rates(k_p_v48__n_cr48)*Y(H1)*state.rho + screened_rates(k_v48__n_v47);
    jac.set(N, V48, scratch);

    scratch = screened_rates(k_he4_v49__n_mn52)*Y(He4)*state.rho - screened_rates(k_n_v49__he4_sc46)*Y(N)*state.rho - screened_rates(k_n_v49__p_ti49)*Y(N)*state.rho - screened_rates(k_n_v49__v50)*Y(N)*state.rho + screened_rates(k_p_v49__n_cr49)*Y(H1)*state.rho + screened_rates(k_v49__n_v48);
    jac.set(N, V49, scratch);

    scratch = screened_rates(k_he4_v50__n_mn53)*Y(He4)*state.rho - screened_rates(k_n_v50__he4_sc47)*Y(N)*state.rho - screened_rates(k_n_v50__p_ti50)*Y(N)*state.rho - screened_rates(k_n_v50__v51)*Y(N)*state.rho + screened_rates(k_p_v50__n_cr50)*Y(H1)*state.rho + screened_rates(k_v50__n_v49);
    jac.set(N, V50, scratch);

    scratch = screened_rates(k_he4_v51__n_mn54)*Y(He4)*state.rho - screened_rates(k_n_v51__he4_sc48)*Y(N)*state.rho - screened_rates(k_n_v51__p_ti51)*Y(N)*state.rho - screened_rates(k_n_v51__v52)*Y(N)*state.rho + screened_rates(k_p_v51__n_cr51)*Y(H1)*state.rho + screened_rates(k_v51__n_v50);
    jac.set(N, V51, scratch);

    scratch = screened_rates(k_he4_v52__n_mn55)*Y(He4)*state.rho - screened_rates(k_n_v52__he4_sc49)*Y(N)*state.rho + screened_rates(k_p_v52__n_cr52)*Y(H1)*state.rho + screened_rates(k_v52__n_v51);
    jac.set(N, V52, scratch);

    scratch = -screened_rates(k_n_cr48__cr49)*Y(N)*state.rho - screened_rates(k_n_cr48__he4_ti45)*Y(N)*state.rho - screened_rates(k_n_cr48__p_v48)*Y(N)*state.rho;
    jac.set(N, Cr48, scratch);

    scratch = screened_rates(k_cr49__n_cr48) + screened_rates(k_he4_cr49__n_fe52)*Y(He4)*state.rho - screened_rates(k_n_cr49__cr50)*Y(N)*state.rho - screened_rates(k_n_cr49__he4_ti46)*Y(N)*state.rho - screened_rates(k_n_cr49__p_v49)*Y(N)*state.rho;
    jac.set(N, Cr49, scratch);

    scratch = screened_rates(k_cr50__n_cr49) + screened_rates(k_he4_cr50__n_fe53)*Y(He4)*state.rho - screened_rates(k_n_cr50__cr51)*Y(N)*state.rho - screened_rates(k_n_cr50__he4_ti47)*Y(N)*state.rho - screened_rates(k_n_cr50__p_v50)*Y(N)*state.rho + screened_rates(k_p_cr50__n_mn50)*Y(H1)*state.rho;
    jac.set(N, Cr50, scratch);

    scratch = screened_rates(k_cr51__n_cr50) + screened_rates(k_he4_cr51__n_fe54)*Y(He4)*state.rho - screened_rates(k_n_cr51__cr52)*Y(N)*state.rho - screened_rates(k_n_cr51__he4_ti48)*Y(N)*state.rho - screened_rates(k_n_cr51__p_v51)*Y(N)*state.rho + screened_rates(k_p_cr51__n_mn51)*Y(H1)*state.rho;
    jac.set(N, Cr51, scratch);

    scratch = screened_rates(k_cr52__n_cr51) + screened_rates(k_he4_cr52__n_fe55)*Y(He4)*state.rho - screened_rates(k_n_cr52__cr53)*Y(N)*state.rho - screened_rates(k_n_cr52__he4_ti49)*Y(N)*state.rho - screened_rates(k_n_cr52__p_v52)*Y(N)*state.rho + screened_rates(k_p_cr52__n_mn52)*Y(H1)*state.rho;
    jac.set(N, Cr52, scratch);

    scratch = screened_rates(k_cr53__n_cr52) + screened_rates(k_he4_cr53__n_fe56)*Y(He4)*state.rho - screened_rates(k_n_cr53__cr54)*Y(N)*state.rho - screened_rates(k_n_cr53__he4_ti50)*Y(N)*state.rho + screened_rates(k_p_cr53__n_mn53)*Y(H1)*state.rho;
    jac.set(N, Cr53, scratch);

    scratch = screened_rates(k_cr54__n_cr53) + screened_rates(k_he4_cr54__n_fe57)*Y(He4)*state.rho - screened_rates(k_n_cr54__he4_ti51)*Y(N)*state.rho + screened_rates(k_p_cr54__n_mn54)*Y(H1)*state.rho;
    jac.set(N, Cr54, scratch);

    scratch = screened_rates(k_he4_mn50__n_co53)*Y(He4)*state.rho - screened_rates(k_n_mn50__he4_v47)*Y(N)*state.rho - screened_rates(k_n_mn50__mn51)*Y(N)*state.rho - screened_rates(k_n_mn50__p_cr50)*Y(N)*state.rho;
    jac.set(N, Mn50, scratch);

    scratch = screened_rates(k_he4_mn51__n_co54)*Y(He4)*state.rho + screened_rates(k_mn51__n_mn50) - screened_rates(k_n_mn51__he4_v48)*Y(N)*state.rho - screened_rates(k_n_mn51__mn52)*Y(N)*state.rho - screened_rates(k_n_mn51__p_cr51)*Y(N)*state.rho;
    jac.set(N, Mn51, scratch);

    scratch = screened_rates(k_he4_mn52__n_co55)*Y(He4)*state.rho + screened_rates(k_mn52__n_mn51) - screened_rates(k_n_mn52__he4_v49)*Y(N)*state.rho - screened_rates(k_n_mn52__mn53)*Y(N)*state.rho - screened_rates(k_n_mn52__p_cr52)*Y(N)*state.rho + screened_rates(k_p_mn52__n_fe52)*Y(H1)*state.rho;
    jac.set(N, Mn52, scratch);

    scratch = screened_rates(k_he4_mn53__n_co56)*Y(He4)*state.rho + screened_rates(k_mn53__n_mn52) - screened_rates(k_n_mn53__he4_v50)*Y(N)*state.rho - screened_rates(k_n_mn53__mn54)*Y(N)*state.rho - screened_rates(k_n_mn53__p_cr53)*Y(N)*state.rho + screened_rates(k_p_mn53__n_fe53)*Y(H1)*state.rho;
    jac.set(N, Mn53, scratch);

    scratch = screened_rates(k_he4_mn54__n_co57)*Y(He4)*state.rho + screened_rates(k_mn54__n_mn53) - screened_rates(k_n_mn54__he4_v51)*Y(N)*state.rho - screened_rates(k_n_mn54__mn55)*Y(N)*state.rho - screened_rates(k_n_mn54__p_cr54)*Y(N)*state.rho + screened_rates(k_p_mn54__n_fe54)*Y(H1)*state.rho;
    jac.set(N, Mn54, scratch);

    scratch = screened_rates(k_he4_mn55__n_co58)*Y(He4)*state.rho + screened_rates(k_mn55__n_mn54) - screened_rates(k_n_mn55__he4_v52)*Y(N)*state.rho + screened_rates(k_p_mn55__n_fe55)*Y(H1)*state.rho;
    jac.set(N, Mn55, scratch);

    scratch = -screened_rates(k_n_fe52__fe53)*Y(N)*state.rho - screened_rates(k_n_fe52__he4_cr49)*Y(N)*state.rho - screened_rates(k_n_fe52__p_mn52)*Y(N)*state.rho;
    jac.set(N, Fe52, scratch);

    scratch = screened_rates(k_fe53__n_fe52) + screened_rates(k_he4_fe53__n_ni56)*Y(He4)*state.rho - screened_rates(k_n_fe53__fe54)*Y(N)*state.rho - screened_rates(k_n_fe53__he4_cr50)*Y(N)*state.rho - screened_rates(k_n_fe53__p_mn53)*Y(N)*state.rho + screened_rates(k_p_fe53__n_co53)*Y(H1)*state.rho;
    jac.set(N, Fe53, scratch);

    scratch = screened_rates(k_fe54__n_fe53) + screened_rates(k_he4_fe54__n_ni57)*Y(He4)*state.rho - screened_rates(k_n_fe54__fe55)*Y(N)*state.rho - screened_rates(k_n_fe54__he4_cr51)*Y(N)*state.rho - screened_rates(k_n_fe54__p_mn54)*Y(N)*state.rho + screened_rates(k_p_fe54__n_co54)*Y(H1)*state.rho;
    jac.set(N, Fe54, scratch);

    scratch = screened_rates(k_fe55__n_fe54) + screened_rates(k_he4_fe55__n_ni58)*Y(He4)*state.rho - screened_rates(k_n_fe55__fe56)*Y(N)*state.rho - screened_rates(k_n_fe55__he4_cr52)*Y(N)*state.rho - screened_rates(k_n_fe55__p_mn55)*Y(N)*state.rho + screened_rates(k_p_fe55__n_co55)*Y(H1)*state.rho;
    jac.set(N, Fe55, scratch);

    scratch = screened_rates(k_fe56__n_fe55) + screened_rates(k_he4_fe56__n_ni59)*Y(He4)*state.rho - screened_rates(k_n_fe56__fe57)*Y(N)*state.rho - screened_rates(k_n_fe56__he4_cr53)*Y(N)*state.rho + screened_rates(k_p_fe56__n_co56)*Y(H1)*state.rho;
    jac.set(N, Fe56, scratch);

    scratch = screened_rates(k_fe57__n_fe56) + screened_rates(k_he4_fe57__n_ni60)*Y(He4)*state.rho - screened_rates(k_n_fe57__fe58)*Y(N)*state.rho - screened_rates(k_n_fe57__he4_cr54)*Y(N)*state.rho + screened_rates(k_p_fe57__n_co57)*Y(H1)*state.rho;
    jac.set(N, Fe57, scratch);

    scratch = screened_rates(k_fe58__n_fe57) + screened_rates(k_he4_fe58__n_ni61)*Y(He4)*state.rho + screened_rates(k_p_fe58__n_co58)*Y(H1)*state.rho;
    jac.set(N, Fe58, scratch);

    scratch = -screened_rates(k_n_co53__co54)*Y(N)*state.rho - screened_rates(k_n_co53__he4_mn50)*Y(N)*state.rho - screened_rates(k_n_co53__p_fe53)*Y(N)*state.rho;
    jac.set(N, Co53, scratch);

    scratch = screened_rates(k_co54__n_co53) + screened_rates(k_he4_co54__n_cu57)*Y(He4)*state.rho - screened_rates(k_n_co54__co55)*Y(N)*state.rho - screened_rates(k_n_co54__he4_mn51)*Y(N)*state.rho - screened_rates(k_n_co54__p_fe54)*Y(N)*state.rho;
    jac.set(N, Co54, scratch);

    scratch = screened_rates(k_co55__n_co54) + screened_rates(k_he4_co55__n_cu58)*Y(He4)*state.rho - screened_rates(k_n_co55__co56)*Y(N)*state.rho - screened_rates(k_n_co55__he4_mn52)*Y(N)*state.rho - screened_rates(k_n_co55__p_fe55)*Y(N)*state.rho;
    jac.set(N, Co55, scratch);

    scratch = screened_rates(k_co56__n_co55) + screened_rates(k_he4_co56__n_cu59)*Y(He4)*state.rho - screened_rates(k_n_co56__co57)*Y(N)*state.rho - screened_rates(k_n_co56__he4_mn53)*Y(N)*state.rho - screened_rates(k_n_co56__p_fe56)*Y(N)*state.rho + screened_rates(k_p_co56__n_ni56)*Y(H1)*state.rho;
    jac.set(N, Co56, scratch);

    scratch = screened_rates(k_co57__n_co56) + screened_rates(k_he4_co57__n_cu60)*Y(He4)*state.rho - screened_rates(k_n_co57__co58)*Y(N)*state.rho - screened_rates(k_n_co57__he4_mn54)*Y(N)*state.rho - screened_rates(k_n_co57__p_fe57)*Y(N)*state.rho + screened_rates(k_p_co57__n_ni57)*Y(H1)*state.rho;
    jac.set(N, Co57, scratch);

    scratch = screened_rates(k_co58__n_co57) + screened_rates(k_he4_co58__n_cu61)*Y(He4)*state.rho - screened_rates(k_n_co58__co59)*Y(N)*state.rho - screened_rates(k_n_co58__he4_mn55)*Y(N)*state.rho - screened_rates(k_n_co58__p_fe58)*Y(N)*state.rho + screened_rates(k_p_co58__n_ni58)*Y(H1)*state.rho;
    jac.set(N, Co58, scratch);

    scratch = screened_rates(k_co59__n_co58) + screened_rates(k_he4_co59__n_cu62)*Y(He4)*state.rho + screened_rates(k_p_co59__n_ni59)*Y(H1)*state.rho;
    jac.set(N, Co59, scratch);

    scratch = screened_rates(k_he4_ni56__n_zn59)*Y(He4)*state.rho - screened_rates(k_n_ni56__he4_fe53)*Y(N)*state.rho - screened_rates(k_n_ni56__ni57)*Y(N)*state.rho - screened_rates(k_n_ni56__p_co56)*Y(N)*state.rho;
    jac.set(N, Ni56, scratch);

    scratch = screened_rates(k_he4_ni57__n_zn60)*Y(He4)*state.rho - screened_rates(k_n_ni57__he4_fe54)*Y(N)*state.rho - screened_rates(k_n_ni57__ni58)*Y(N)*state.rho - screened_rates(k_n_ni57__p_co57)*Y(N)*state.rho + screened_rates(k_ni57__n_ni56) + screened_rates(k_p_ni57__n_cu57)*Y(H1)*state.rho;
    jac.set(N, Ni57, scratch);

    scratch = screened_rates(k_he4_ni58__n_zn61)*Y(He4)*state.rho - screened_rates(k_n_ni58__he4_fe55)*Y(N)*state.rho - screened_rates(k_n_ni58__ni59)*Y(N)*state.rho - screened_rates(k_n_ni58__p_co58)*Y(N)*state.rho + screened_rates(k_ni58__n_ni57) + screened_rates(k_p_ni58__n_cu58)*Y(H1)*state.rho;
    jac.set(N, Ni58, scratch);

    scratch = screened_rates(k_he4_ni59__n_zn62)*Y(He4)*state.rho - screened_rates(k_n_ni59__he4_fe56)*Y(N)*state.rho - screened_rates(k_n_ni59__ni60)*Y(N)*state.rho - screened_rates(k_n_ni59__p_co59)*Y(N)*state.rho + screened_rates(k_ni59__n_ni58) + screened_rates(k_p_ni59__n_cu59)*Y(H1)*state.rho;
    jac.set(N, Ni59, scratch);

    scratch = screened_rates(k_he4_ni60__n_zn63)*Y(He4)*state.rho - screened_rates(k_n_ni60__he4_fe57)*Y(N)*state.rho - screened_rates(k_n_ni60__ni61)*Y(N)*state.rho + screened_rates(k_ni60__n_ni59) + screened_rates(k_p_ni60__n_cu60)*Y(H1)*state.rho;
    jac.set(N, Ni60, scratch);

    scratch = screened_rates(k_he4_ni61__n_zn64)*Y(He4)*state.rho - screened_rates(k_n_ni61__he4_fe58)*Y(N)*state.rho - screened_rates(k_n_ni61__ni62)*Y(N)*state.rho + screened_rates(k_ni61__n_ni60) + screened_rates(k_p_ni61__n_cu61)*Y(H1)*state.rho;
    jac.set(N, Ni61, scratch);

    scratch = screened_rates(k_he4_ni62__n_zn65)*Y(He4)*state.rho - screened_rates(k_n_ni62__ni63)*Y(N)*state.rho + screened_rates(k_ni62__n_ni61) + screened_rates(k_p_ni62__n_cu62)*Y(H1)*state.rho;
    jac.set(N, Ni62, scratch);

    scratch = screened_rates(k_he4_ni63__n_zn66)*Y(He4)*state.rho - screened_rates(k_n_ni63__ni64)*Y(N)*state.rho + screened_rates(k_ni63__n_ni62) + screened_rates(k_p_ni63__n_cu63)*Y(H1)*state.rho;
    jac.set(N, Ni63, scratch);

    scratch = screened_rates(k_ni64__n_ni63) + screened_rates(k_p_ni64__n_cu64)*Y(H1)*state.rho;
    jac.set(N, Ni64, scratch);

    scratch = -screened_rates(k_n_cu57__cu58)*Y(N)*state.rho - screened_rates(k_n_cu57__he4_co54)*Y(N)*state.rho - screened_rates(k_n_cu57__p_ni57)*Y(N)*state.rho;
    jac.set(N, Cu57, scratch);

    scratch = screened_rates(k_cu58__n_cu57) - screened_rates(k_n_cu58__cu59)*Y(N)*state.rho - screened_rates(k_n_cu58__he4_co55)*Y(N)*state.rho - screened_rates(k_n_cu58__p_ni58)*Y(N)*state.rho;
    jac.set(N, Cu58, scratch);

    scratch = screened_rates(k_cu59__n_cu58) + screened_rates(k_he4_cu59__n_ga62)*Y(He4)*state.rho - screened_rates(k_n_cu59__cu60)*Y(N)*state.rho - screened_rates(k_n_cu59__he4_co56)*Y(N)*state.rho - screened_rates(k_n_cu59__p_ni59)*Y(N)*state.rho + screened_rates(k_p_cu59__n_zn59)*Y(H1)*state.rho;
    jac.set(N, Cu59, scratch);

    scratch = screened_rates(k_cu60__n_cu59) + screened_rates(k_he4_cu60__n_ga63)*Y(He4)*state.rho - screened_rates(k_n_cu60__cu61)*Y(N)*state.rho - screened_rates(k_n_cu60__he4_co57)*Y(N)*state.rho - screened_rates(k_n_cu60__p_ni60)*Y(N)*state.rho + screened_rates(k_p_cu60__n_zn60)*Y(H1)*state.rho;
    jac.set(N, Cu60, scratch);

    scratch = screened_rates(k_cu61__n_cu60) + screened_rates(k_he4_cu61__n_ga64)*Y(He4)*state.rho - screened_rates(k_n_cu61__cu62)*Y(N)*state.rho - screened_rates(k_n_cu61__he4_co58)*Y(N)*state.rho - screened_rates(k_n_cu61__p_ni61)*Y(N)*state.rho + screened_rates(k_p_cu61__n_zn61)*Y(H1)*state.rho;
    jac.set(N, Cu61, scratch);

    scratch = screened_rates(k_cu62__n_cu61) - screened_rates(k_n_cu62__cu63)*Y(N)*state.rho - screened_rates(k_n_cu62__he4_co59)*Y(N)*state.rho - screened_rates(k_n_cu62__p_ni62)*Y(N)*state.rho + screened_rates(k_p_cu62__n_zn62)*Y(H1)*state.rho;
    jac.set(N, Cu62, scratch);

    scratch = screened_rates(k_cu63__n_cu62) - screened_rates(k_n_cu63__cu64)*Y(N)*state.rho - screened_rates(k_n_cu63__p_ni63)*Y(N)*state.rho + screened_rates(k_p_cu63__n_zn63)*Y(H1)*state.rho;
    jac.set(N, Cu63, scratch);

    scratch = screened_rates(k_cu64__n_cu63) - screened_rates(k_n_cu64__cu65)*Y(N)*state.rho - screened_rates(k_n_cu64__p_ni64)*Y(N)*state.rho + screened_rates(k_p_cu64__n_zn64)*Y(H1)*state.rho;
    jac.set(N, Cu64, scratch);

    scratch = screened_rates(k_cu65__n_cu64) + screened_rates(k_p_cu65__n_zn65)*Y(H1)*state.rho;
    jac.set(N, Cu65, scratch);

    scratch = -screened_rates(k_n_zn59__he4_ni56)*Y(N)*state.rho - screened_rates(k_n_zn59__p_cu59)*Y(N)*state.rho - screened_rates(k_n_zn59__zn60)*Y(N)*state.rho;
    jac.set(N, Zn59, scratch);

    scratch = screened_rates(k_he4_zn60__n_ge63)*Y(He4)*state.rho - screened_rates(k_n_zn60__he4_ni57)*Y(N)*state.rho - screened_rates(k_n_zn60__p_cu60)*Y(N)*state.rho - screened_rates(k_n_zn60__zn61)*Y(N)*state.rho + screened_rates(k_zn60__n_zn59);
    jac.set(N, Zn60, scratch);

    scratch = screened_rates(k_he4_zn61__n_ge64)*Y(He4)*state.rho - screened_rates(k_n_zn61__he4_ni58)*Y(N)*state.rho - screened_rates(k_n_zn61__p_cu61)*Y(N)*state.rho - screened_rates(k_n_zn61__zn62)*Y(N)*state.rho + screened_rates(k_zn61__n_zn60);
    jac.set(N, Zn61, scratch);

    scratch = -screened_rates(k_n_zn62__he4_ni59)*Y(N)*state.rho - screened_rates(k_n_zn62__p_cu62)*Y(N)*state.rho - screened_rates(k_n_zn62__zn63)*Y(N)*state.rho + screened_rates(k_p_zn62__n_ga62)*Y(H1)*state.rho + screened_rates(k_zn62__n_zn61);
    jac.set(N, Zn62, scratch);

    scratch = -screened_rates(k_n_zn63__he4_ni60)*Y(N)*state.rho - screened_rates(k_n_zn63__p_cu63)*Y(N)*state.rho - screened_rates(k_n_zn63__zn64)*Y(N)*state.rho + screened_rates(k_p_zn63__n_ga63)*Y(H1)*state.rho + screened_rates(k_zn63__n_zn62);
    jac.set(N, Zn63, scratch);

    scratch = -screened_rates(k_n_zn64__he4_ni61)*Y(N)*state.rho - screened_rates(k_n_zn64__p_cu64)*Y(N)*state.rho - screened_rates(k_n_zn64__zn65)*Y(N)*state.rho + screened_rates(k_p_zn64__n_ga64)*Y(H1)*state.rho + screened_rates(k_zn64__n_zn63);
    jac.set(N, Zn64, scratch);

    scratch = -screened_rates(k_n_zn65__he4_ni62)*Y(N)*state.rho - screened_rates(k_n_zn65__p_cu65)*Y(N)*state.rho - screened_rates(k_n_zn65__zn66)*Y(N)*state.rho + screened_rates(k_zn65__n_zn64);
    jac.set(N, Zn65, scratch);

    scratch = -screened_rates(k_n_zn66__he4_ni63)*Y(N)*state.rho + screened_rates(k_zn66__n_zn65);
    jac.set(N, Zn66, scratch);

    scratch = -screened_rates(k_n_ga62__ga63)*Y(N)*state.rho - screened_rates(k_n_ga62__he4_cu59)*Y(N)*state.rho - screened_rates(k_n_ga62__p_zn62)*Y(N)*state.rho;
    jac.set(N, Ga62, scratch);

    scratch = screened_rates(k_ga63__n_ga62) - screened_rates(k_n_ga63__ga64)*Y(N)*state.rho - screened_rates(k_n_ga63__he4_cu60)*Y(N)*state.rho - screened_rates(k_n_ga63__p_zn63)*Y(N)*state.rho + screened_rates(k_p_ga63__n_ge63)*Y(H1)*state.rho;
    jac.set(N, Ga63, scratch);

    scratch = screened_rates(k_ga64__n_ga63) - screened_rates(k_n_ga64__he4_cu61)*Y(N)*state.rho - screened_rates(k_n_ga64__p_zn64)*Y(N)*state.rho + screened_rates(k_p_ga64__n_ge64)*Y(H1)*state.rho;
    jac.set(N, Ga64, scratch);

    scratch = -screened_rates(k_n_ge63__ge64)*Y(N)*state.rho - screened_rates(k_n_ge63__he4_zn60)*Y(N)*state.rho - screened_rates(k_n_ge63__p_ga63)*Y(N)*state.rho;
    jac.set(N, Ge63, scratch);

    scratch = screened_rates(k_ge64__n_ge63) - screened_rates(k_n_ge64__he4_zn61)*Y(N)*state.rho - screened_rates(k_n_ge64__p_ga64)*Y(N)*state.rho;
    jac.set(N, Ge64, scratch);

    scratch = screened_rates(k_n__p__weak__wc12) + screened_rates(k_n_al25__p_mg25)*Y(Al25)*state.rho + screened_rates(k_n_al26__p_mg26)*Y(Al26)*state.rho + screened_rates(k_n_ar36__p_cl36)*Y(Ar36)*state.rho + screened_rates(k_n_ar37__p_cl37)*Y(Ar37)*state.rho + screened_rates(k_n_b8__p_he4_he4)*Y(B8)*state.rho + screened_rates(k_n_be7__p_li7)*Y(Be7)*state.rho + screened_rates(k_n_ca40__p_k40)*Y(Ca40)*state.rho + screened_rates(k_n_ca41__p_k41)*Y(Ca41)*state.rho + screened_rates(k_n_cl33__p_s33)*Y(Cl33)*state.rho + screened_rates(k_n_cl34__p_s34)*Y(Cl34)*state.rho + screened_rates(k_n_cl35__p_s35)*Y(Cl35)*state.rho + screened_rates(k_n_cl36__p_s36)*Y(Cl36)*state.rho + screened_rates(k_n_co53__p_fe53)*Y(Co53)*state.rho + screened_rates(k_n_co54__p_fe54)*Y(Co54)*state.rho + screened_rates(k_n_co55__p_fe55)*Y(Co55)*state.rho + screened_rates(k_n_co56__p_fe56)*Y(Co56)*state.rho + screened_rates(k_n_co57__p_fe57)*Y(Co57)*state.rho + screened_rates(k_n_co58__p_fe58)*Y(Co58)*state.rho + screened_rates(k_n_cr48__p_v48)*Y(Cr48)*state.rho + screened_rates(k_n_cr49__p_v49)*Y(Cr49)*state.rho + screened_rates(k_n_cr50__p_v50)*Y(Cr50)*state.rho + screened_rates(k_n_cr51__p_v51)*Y(Cr51)*state.rho + screened_rates(k_n_cr52__p_v52)*Y(Cr52)*state.rho + screened_rates(k_n_cu57__p_ni57)*Y(Cu57)*state.rho + screened_rates(k_n_cu58__p_ni58)*Y(Cu58)*state.rho + screened_rates(k_n_cu59__p_ni59)*Y(Cu59)*state.rho + screened_rates(k_n_cu60__p_ni60)*Y(Cu60)*state.rho + screened_rates(k_n_cu61__p_ni61)*Y(Cu61)*state.rho + screened_rates(k_n_cu62__p_ni62)*Y(Cu62)*state.rho + screened_rates(k_n_cu63__p_ni63)*Y(Cu63)*state.rho + screened_rates(k_n_cu64__p_ni64)*Y(Cu64)*state.rho + screened_rates(k_n_f17__p_o17)*Y(F17)*state.rho + screened_rates(k_n_f18__p_o18)*Y(F18)*state.rho + screened_rates(k_n_fe52__p_mn52)*Y(Fe52)*state.rho + screened_rates(k_n_fe53__p_mn53)*Y(Fe53)*state.rho + screened_rates(k_n_fe54__p_mn54)*Y(Fe54)*state.rho + screened_rates(k_n_fe55__p_mn55)*Y(Fe55)*state.rho + screened_rates(k_n_ga62__p_zn62)*Y(Ga62)*state.rho + screened_rates(k_n_ga63__p_zn63)*Y(Ga63)*state.rho + screened_rates(k_n_ga64__p_zn64)*Y(Ga64)*state.rho + screened_rates(k_n_ge63__p_ga63)*Y(Ge63)*state.rho + screened_rates(k_n_ge64__p_ga64)*Y(Ge64)*state.rho + screened_rates(k_n_k37__p_ar37)*Y(K37)*state.rho + screened_rates(k_n_k38__p_ar38)*Y(K38)*state.rho + screened_rates(k_n_k39__p_ar39)*Y(K39)*state.rho + screened_rates(k_n_k40__p_ar40)*Y(K40)*state.rho + screened_rates(k_n_mg23__p_na23)*Y(Mg23)*state.rho + screened_rates(k_n_mn50__p_cr50)*Y(Mn50)*state.rho + screened_rates(k_n_mn51__p_cr51)*Y(Mn51)*state.rho + screened_rates(k_n_mn52__p_cr52)*Y(Mn52)*state.rho + screened_rates(k_n_mn53__p_cr53)*Y(Mn53)*state.rho + screened_rates(k_n_mn54__p_cr54)*Y(Mn54)*state.rho + screened_rates(k_n_n13__p_c13)*Y(N13)*state.rho + screened_rates(k_n_n14__p_c14)*Y(N14)*state.rho + screened_rates(k_n_na21__p_ne21)*Y(Na21)*state.rho + screened_rates(k_n_na22__p_ne22)*Y(Na22)*state.rho + screened_rates(k_n_ne18__p_f18)*Y(Ne18)*state.rho + screened_rates(k_n_ne19__p_f19)*Y(Ne19)*state.rho + screened_rates(k_n_ni56__p_co56)*Y(Ni56)*state.rho + screened_rates(k_n_ni57__p_co57)*Y(Ni57)*state.rho + screened_rates(k_n_ni58__p_co58)*Y(Ni58)*state.rho + screened_rates(k_n_ni59__p_co59)*Y(Ni59)*state.rho + screened_rates(k_n_o14__p_n14)*Y(O14)*state.rho + screened_rates(k_n_o15__p_n15)*Y(O15)*state.rho + screened_rates(k_n_p29__p_si29)*Y(P29)*state.rho + screened_rates(k_n_p30__p_si30)*Y(P30)*state.rho + screened_rates(k_n_p31__p_si31)*Y(P31)*state.rho + screened_rates(k_n_p32__p_si32)*Y(P32)*state.rho - screened_rates(k_n_p__d)*Y(H1)*state.rho - screened_rates(k_n_p_he4__li6)*Y(He4)*Y(H1)*std::pow(state.rho, 2) - 0.5*screened_rates(k_n_p_he4_he4__he3_li7)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 3) - 1.0*screened_rates(k_n_p_p__p_d)*std::pow(Y(H1), 2)*std::pow(state.rho, 2) + screened_rates(k_n_s32__p_p32)*Y(S32)*state.rho + screened_rates(k_n_s33__p_p33)*Y(S33)*state.rho + screened_rates(k_n_sc43__p_ca43)*Y(Sc43)*state.rho + screened_rates(k_n_sc44__p_ca44)*Y(Sc44)*state.rho + screened_rates(k_n_sc45__p_ca45)*Y(Sc45)*state.rho + screened_rates(k_n_sc46__p_ca46)*Y(Sc46)*state.rho + screened_rates(k_n_sc47__p_ca47)*Y(Sc47)*state.rho + screened_rates(k_n_sc48__p_ca48)*Y(Sc48)*state.rho + screened_rates(k_n_ti44__p_sc44)*Y(Ti44)*state.rho + screened_rates(k_n_ti45__p_sc45)*Y(Ti45)*state.rho + screened_rates(k_n_ti46__p_sc46)*Y(Ti46)*state.rho + screened_rates(k_n_ti47__p_sc47)*Y(Ti47)*state.rho + screened_rates(k_n_ti48__p_sc48)*Y(Ti48)*state.rho + screened_rates(k_n_ti49__p_sc49)*Y(Ti49)*state.rho + screened_rates(k_n_v46__p_ti46)*Y(V46)*state.rho + screened_rates(k_n_v47__p_ti47)*Y(V47)*state.rho + screened_rates(k_n_v48__p_ti48)*Y(V48)*state.rho + screened_rates(k_n_v49__p_ti49)*Y(V49)*state.rho + screened_rates(k_n_v50__p_ti50)*Y(V50)*state.rho + screened_rates(k_n_v51__p_ti51)*Y(V51)*state.rho + screened_rates(k_n_zn59__p_cu59)*Y(Zn59)*state.rho + screened_rates(k_n_zn60__p_cu60)*Y(Zn60)*state.rho + screened_rates(k_n_zn61__p_cu61)*Y(Zn61)*state.rho + screened_rates(k_n_zn62__p_cu62)*Y(Zn62)*state.rho + screened_rates(k_n_zn63__p_cu63)*Y(Zn63)*state.rho + screened_rates(k_n_zn64__p_cu64)*Y(Zn64)*state.rho + screened_rates(k_n_zn65__p_cu65)*Y(Zn65)*state.rho;
    jac.set(H1, N, scratch);

    scratch = -screened_rates(k_n_p__d)*Y(N)*state.rho - screened_rates(k_n_p_he4__li6)*Y(He4)*Y(N)*std::pow(state.rho, 2) - 0.5*screened_rates(k_n_p_he4_he4__he3_li7)*std::pow(Y(He4), 2)*Y(N)*std::pow(state.rho, 3) - 2.0*screened_rates(k_n_p_p__p_d)*Y(N)*Y(H1)*std::pow(state.rho, 2) - screened_rates(k_p_al26__he4_mg23)*Y(Al26)*state.rho - screened_rates(k_p_al27__c12_o16)*Y(Al27)*state.rho - screened_rates(k_p_al27__he4_mg24)*Y(Al27)*state.rho - screened_rates(k_p_al27__si28)*Y(Al27)*state.rho - screened_rates(k_p_ar36__he4_cl33)*Y(Ar36)*state.rho - screened_rates(k_p_ar36__k37)*Y(Ar36)*state.rho - screened_rates(k_p_ar37__he4_cl34)*Y(Ar37)*state.rho - screened_rates(k_p_ar37__k38)*Y(Ar37)*state.rho - screened_rates(k_p_ar37__n_k37)*Y(Ar37)*state.rho - screened_rates(k_p_ar38__he4_cl35)*Y(Ar38)*state.rho - screened_rates(k_p_ar38__k39)*Y(Ar38)*state.rho - screened_rates(k_p_ar38__n_k38)*Y(Ar38)*state.rho - screened_rates(k_p_ar39__he4_cl36)*Y(Ar39)*state.rho - screened_rates(k_p_ar39__k40)*Y(Ar39)*state.rho - screened_rates(k_p_ar39__n_k39)*Y(Ar39)*state.rho - screened_rates(k_p_ar40__he4_cl37)*Y(Ar40)*state.rho - screened_rates(k_p_ar40__k41)*Y(Ar40)*state.rho - screened_rates(k_p_ar40__n_k40)*Y(Ar40)*state.rho - screened_rates(k_p_b10__he4_be7)*Y(B10)*state.rho - screened_rates(k_p_b11__c12)*Y(B11)*state.rho - screened_rates(k_p_b11__he4_he4_he4)*Y(B11)*state.rho - screened_rates(k_p_be7__b8)*Y(Be7)*state.rho - screened_rates(k_p_be9__b10)*Y(Be9)*state.rho - screened_rates(k_p_be9__d_he4_he4)*Y(Be9)*state.rho - screened_rates(k_p_be9__he4_li6)*Y(Be9)*state.rho - screened_rates(k_p_c12__n13)*Y(C12)*state.rho - screened_rates(k_p_c13__he4_b10)*Y(C13)*state.rho - screened_rates(k_p_c13__n14)*Y(C13)*state.rho - screened_rates(k_p_c13__n_n13)*Y(C13)*state.rho - screened_rates(k_p_c14__he4_b11)*Y(C14)*state.rho - screened_rates(k_p_c14__n15)*Y(C14)*state.rho - screened_rates(k_p_c14__n_n14)*Y(C14)*state.rho - screened_rates(k_p_ca40__he4_k37)*Y(Ca40)*state.rho - screened_rates(k_p_ca41__he4_k38)*Y(Ca41)*state.rho - screened_rates(k_p_ca42__he4_k39)*Y(Ca42)*state.rho - screened_rates(k_p_ca42__sc43)*Y(Ca42)*state.rho - screened_rates(k_p_ca43__he4_k40)*Y(Ca43)*state.rho - screened_rates(k_p_ca43__n_sc43)*Y(Ca43)*state.rho - screened_rates(k_p_ca43__sc44)*Y(Ca43)*state.rho - screened_rates(k_p_ca44__he4_k41)*Y(Ca44)*state.rho - screened_rates(k_p_ca44__n_sc44)*Y(Ca44)*state.rho - screened_rates(k_p_ca44__sc45)*Y(Ca44)*state.rho - screened_rates(k_p_ca45__n_sc45)*Y(Ca45)*state.rho - screened_rates(k_p_ca45__sc46)*Y(Ca45)*state.rho - screened_rates(k_p_ca46__n_sc46)*Y(Ca46)*state.rho - screened_rates(k_p_ca46__sc47)*Y(Ca46)*state.rho - screened_rates(k_p_ca47__n_sc47)*Y(Ca47)*state.rho - screened_rates(k_p_ca47__sc48)*Y(Ca47)*state.rho - screened_rates(k_p_ca48__n_sc48)*Y(Ca48)*state.rho - screened_rates(k_p_ca48__sc49)*Y(Ca48)*state.rho - screened_rates(k_p_cl35__ar36)*Y(Cl35)*state.rho - screened_rates(k_p_cl35__he4_s32)*Y(Cl35)*state.rho - screened_rates(k_p_cl36__ar37)*Y(Cl36)*state.rho - screened_rates(k_p_cl36__he4_s33)*Y(Cl36)*state.rho - screened_rates(k_p_cl36__n_ar36)*Y(Cl36)*state.rho - screened_rates(k_p_cl37__ar38)*Y(Cl37)*state.rho - screened_rates(k_p_cl37__he4_s34)*Y(Cl37)*state.rho - screened_rates(k_p_cl37__n_ar37)*Y(Cl37)*state.rho - screened_rates(k_p_co55__he4_fe52)*Y(Co55)*state.rho - screened_rates(k_p_co55__ni56)*Y(Co55)*state.rho - screened_rates(k_p_co56__he4_fe53)*Y(Co56)*state.rho - screened_rates(k_p_co56__n_ni56)*Y(Co56)*state.rho - screened_rates(k_p_co56__ni57)*Y(Co56)*state.rho - screened_rates(k_p_co57__he4_fe54)*Y(Co57)*state.rho - screened_rates(k_p_co57__n_ni57)*Y(Co57)*state.rho - screened_rates(k_p_co57__ni58)*Y(Co57)*state.rho - screened_rates(k_p_co58__he4_fe55)*Y(Co58)*state.rho - screened_rates(k_p_co58__n_ni58)*Y(Co58)*state.rho - screened_rates(k_p_co58__ni59)*Y(Co58)*state.rho - screened_rates(k_p_co59__he4_fe56)*Y(Co59)*state.rho - screened_rates(k_p_co59__n_ni59)*Y(Co59)*state.rho - screened_rates(k_p_co59__ni60)*Y(Co59)*state.rho - screened_rates(k_p_cr49__he4_v46)*Y(Cr49)*state.rho - screened_rates(k_p_cr49__mn50)*Y(Cr49)*state.rho - screened_rates(k_p_cr50__he4_v47)*Y(Cr50)*state.rho - screened_rates(k_p_cr50__mn51)*Y(Cr50)*state.rho - screened_rates(k_p_cr50__n_mn50)*Y(Cr50)*state.rho - screened_rates(k_p_cr51__he4_v48)*Y(Cr51)*state.rho - screened_rates(k_p_cr51__mn52)*Y(Cr51)*state.rho - screened_rates(k_p_cr51__n_mn51)*Y(Cr51)*state.rho - screened_rates(k_p_cr52__he4_v49)*Y(Cr52)*state.rho - screened_rates(k_p_cr52__mn53)*Y(Cr52)*state.rho - screened_rates(k_p_cr52__n_mn52)*Y(Cr52)*state.rho - screened_rates(k_p_cr53__he4_v50)*Y(Cr53)*state.rho - screened_rates(k_p_cr53__mn54)*Y(Cr53)*state.rho - screened_rates(k_p_cr53__n_mn53)*Y(Cr53)*state.rho - screened_rates(k_p_cr54__he4_v51)*Y(Cr54)*state.rho - screened_rates(k_p_cr54__mn55)*Y(Cr54)*state.rho - screened_rates(k_p_cr54__n_mn54)*Y(Cr54)*state.rho - screened_rates(k_p_cu58__zn59)*Y(Cu58)*state.rho - screened_rates(k_p_cu59__he4_ni56)*Y(Cu59)*state.rho - screened_rates(k_p_cu59__n_zn59)*Y(Cu59)*state.rho - screened_rates(k_p_cu59__zn60)*Y(Cu59)*state.rho - screened_rates(k_p_cu60__he4_ni57)*Y(Cu60)*state.rho - screened_rates(k_p_cu60__n_zn60)*Y(Cu60)*state.rho - screened_rates(k_p_cu60__zn61)*Y(Cu60)*state.rho - screened_rates(k_p_cu61__he4_ni58)*Y(Cu61)*state.rho - screened_rates(k_p_cu61__n_zn61)*Y(Cu61)*state.rho - screened_rates(k_p_cu61__zn62)*Y(Cu61)*state.rho - screened_rates(k_p_cu62__he4_ni59)*Y(Cu62)*state.rho - screened_rates(k_p_cu62__n_zn62)*Y(Cu62)*state.rho - screened_rates(k_p_cu62__zn63)*Y(Cu62)*state.rho - screened_rates(k_p_cu63__he4_ni60)*Y(Cu63)*state.rho - screened_rates(k_p_cu63__n_zn63)*Y(Cu63)*state.rho - screened_rates(k_p_cu63__zn64)*Y(Cu63)*state.rho - screened_rates(k_p_cu64__he4_ni61)*Y(Cu64)*state.rho - screened_rates(k_p_cu64__n_zn64)*Y(Cu64)*state.rho - screened_rates(k_p_cu64__zn65)*Y(Cu64)*state.rho - screened_rates(k_p_cu65__he4_ni62)*Y(Cu65)*state.rho - screened_rates(k_p_cu65__n_zn65)*Y(Cu65)*state.rho - screened_rates(k_p_cu65__zn66)*Y(Cu65)*state.rho - screened_rates(k_p_d__he3)*Y(H2)*state.rho + 2*screened_rates(k_p_d__n_p_p)*Y(H2)*state.rho - screened_rates(k_p_f17__he4_o14)*Y(F17)*state.rho - screened_rates(k_p_f17__ne18)*Y(F17)*state.rho - screened_rates(k_p_f18__he4_o15)*Y(F18)*state.rho - screened_rates(k_p_f18__n_ne18)*Y(F18)*state.rho - screened_rates(k_p_f18__ne19)*Y(F18)*state.rho - screened_rates(k_p_f19__he4_o16)*Y(F19)*state.rho - screened_rates(k_p_f19__n_ne19)*Y(F19)*state.rho - screened_rates(k_p_f19__ne20)*Y(F19)*state.rho - screened_rates(k_p_fe52__co53)*Y(Fe52)*state.rho - screened_rates(k_p_fe53__co54)*Y(Fe53)*state.rho - screened_rates(k_p_fe53__he4_mn50)*Y(Fe53)*state.rho - screened_rates(k_p_fe53__n_co53)*Y(Fe53)*state.rho - screened_rates(k_p_fe54__co55)*Y(Fe54)*state.rho - screened_rates(k_p_fe54__he4_mn51)*Y(Fe54)*state.rho - screened_rates(k_p_fe54__n_co54)*Y(Fe54)*state.rho - screened_rates(k_p_fe55__co56)*Y(Fe55)*state.rho - screened_rates(k_p_fe55__he4_mn52)*Y(Fe55)*state.rho - screened_rates(k_p_fe55__n_co55)*Y(Fe55)*state.rho - screened_rates(k_p_fe56__co57)*Y(Fe56)*state.rho - screened_rates(k_p_fe56__he4_mn53)*Y(Fe56)*state.rho - screened_rates(k_p_fe56__n_co56)*Y(Fe56)*state.rho - screened_rates(k_p_fe57__co58)*Y(Fe57)*state.rho - screened_rates(k_p_fe57__he4_mn54)*Y(Fe57)*state.rho - screened_rates(k_p_fe57__n_co57)*Y(Fe57)*state.rho - screened_rates(k_p_fe58__co59)*Y(Fe58)*state.rho - screened_rates(k_p_fe58__he4_mn55)*Y(Fe58)*state.rho - screened_rates(k_p_fe58__n_co58)*Y(Fe58)*state.rho - screened_rates(k_p_ga62__ge63)*Y(Ga62)*state.rho - screened_rates(k_p_ga62__he4_zn59)*Y(Ga62)*state.rho - screened_rates(k_p_ga63__ge64)*Y(Ga63)*state.rho - screened_rates(k_p_ga63__he4_zn60)*Y(Ga63)*state.rho - screened_rates(k_p_ga63__n_ge63)*Y(Ga63)*state.rho - screened_rates(k_p_ga64__he4_zn61)*Y(Ga64)*state.rho - screened_rates(k_p_ga64__n_ge64)*Y(Ga64)*state.rho - screened_rates(k_p_he3__he4__weak__bet_pos_)*Y(He3)*state.rho - screened_rates(k_p_he4__d_he3)*Y(He4)*state.rho - 0.5*screened_rates(k_p_he4_he4__d_be7)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) - 0.5*screened_rates(k_p_he4_he4__n_b8)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) - screened_rates(k_p_k39__ca40)*Y(K39)*state.rho - screened_rates(k_p_k39__he4_ar36)*Y(K39)*state.rho - screened_rates(k_p_k40__ca41)*Y(K40)*state.rho - screened_rates(k_p_k40__he4_ar37)*Y(K40)*state.rho - screened_rates(k_p_k40__n_ca40)*Y(K40)*state.rho - screened_rates(k_p_k41__ca42)*Y(K41)*state.rho - screened_rates(k_p_k41__he4_ar38)*Y(K41)*state.rho - screened_rates(k_p_k41__n_ca41)*Y(K41)*state.rho - screened_rates(k_p_li6__be7)*Y(Li6)*state.rho - screened_rates(k_p_li6__he4_he3)*Y(Li6)*state.rho - screened_rates(k_p_li7__d_li6)*Y(Li7)*state.rho - screened_rates(k_p_li7__he4_he4)*Y(Li7)*state.rho - screened_rates(k_p_li7__n_be7)*Y(Li7)*state.rho - screened_rates(k_p_mg24__al25)*Y(Mg24)*state.rho - screened_rates(k_p_mg24__he4_na21)*Y(Mg24)*state.rho - screened_rates(k_p_mg25__al26)*Y(Mg25)*state.rho - screened_rates(k_p_mg25__he4_na22)*Y(Mg25)*state.rho - screened_rates(k_p_mg25__n_al25)*Y(Mg25)*state.rho - screened_rates(k_p_mg26__al27)*Y(Mg26)*state.rho - screened_rates(k_p_mg26__he4_na23)*Y(Mg26)*state.rho - screened_rates(k_p_mg26__n_al26)*Y(Mg26)*state.rho - screened_rates(k_p_mn51__fe52)*Y(Mn51)*state.rho - screened_rates(k_p_mn51__he4_cr48)*Y(Mn51)*state.rho - screened_rates(k_p_mn52__fe53)*Y(Mn52)*state.rho - screened_rates(k_p_mn52__he4_cr49)*Y(Mn52)*state.rho - screened_rates(k_p_mn52__n_fe52)*Y(Mn52)*state.rho - screened_rates(k_p_mn53__fe54)*Y(Mn53)*state.rho - screened_rates(k_p_mn53__he4_cr50)*Y(Mn53)*state.rho - screened_rates(k_p_mn53__n_fe53)*Y(Mn53)*state.rho - screened_rates(k_p_mn54__fe55)*Y(Mn54)*state.rho - screened_rates(k_p_mn54__he4_cr51)*Y(Mn54)*state.rho - screened_rates(k_p_mn54__n_fe54)*Y(Mn54)*state.rho - screened_rates(k_p_mn55__fe56)*Y(Mn55)*state.rho - screened_rates(k_p_mn55__he4_cr52)*Y(Mn55)*state.rho - screened_rates(k_p_mn55__n_fe55)*Y(Mn55)*state.rho - screened_rates(k_p_n13__o14)*Y(N13)*state.rho - screened_rates(k_p_n14__n_o14)*Y(N14)*state.rho - screened_rates(k_p_n14__o15)*Y(N14)*state.rho - screened_rates(k_p_n15__he4_c12)*Y(N15)*state.rho - screened_rates(k_p_n15__n_o15)*Y(N15)*state.rho - screened_rates(k_p_n15__o16)*Y(N15)*state.rho - screened_rates(k_p_na21__he4_ne18)*Y(Na21)*state.rho - screened_rates(k_p_na22__he4_ne19)*Y(Na22)*state.rho - screened_rates(k_p_na22__mg23)*Y(Na22)*state.rho - screened_rates(k_p_na23__c12_c12)*Y(Na23)*state.rho - screened_rates(k_p_na23__he4_ne20)*Y(Na23)*state.rho - screened_rates(k_p_na23__mg24)*Y(Na23)*state.rho - screened_rates(k_p_na23__n_mg23)*Y(Na23)*state.rho - screened_rates(k_p_ne20__he4_f17)*Y(Ne20)*state.rho - screened_rates(k_p_ne20__na21)*Y(Ne20)*state.rho - screened_rates(k_p_ne21__he4_f18)*Y(Ne21)*state.rho - screened_rates(k_p_ne21__n_na21)*Y(Ne21)*state.rho - screened_rates(k_p_ne21__na22)*Y(Ne21)*state.rho - screened_rates(k_p_ne22__he4_f19)*Y(Ne22)*state.rho - screened_rates(k_p_ne22__n_na22)*Y(Ne22)*state.rho - screened_rates(k_p_ne22__na23)*Y(Ne22)*state.rho - screened_rates(k_p_ni56__cu57)*Y(Ni56)*state.rho - screened_rates(k_p_ni56__he4_co53)*Y(Ni56)*state.rho - screened_rates(k_p_ni57__cu58)*Y(Ni57)*state.rho - screened_rates(k_p_ni57__he4_co54)*Y(Ni57)*state.rho - screened_rates(k_p_ni57__n_cu57)*Y(Ni57)*state.rho - screened_rates(k_p_ni58__cu59)*Y(Ni58)*state.rho - screened_rates(k_p_ni58__he4_co55)*Y(Ni58)*state.rho - screened_rates(k_p_ni58__n_cu58)*Y(Ni58)*state.rho - screened_rates(k_p_ni59__cu60)*Y(Ni59)*state.rho - screened_rates(k_p_ni59__he4_co56)*Y(Ni59)*state.rho - screened_rates(k_p_ni59__n_cu59)*Y(Ni59)*state.rho - screened_rates(k_p_ni60__cu61)*Y(Ni60)*state.rho - screened_rates(k_p_ni60__he4_co57)*Y(Ni60)*state.rho - screened_rates(k_p_ni60__n_cu60)*Y(Ni60)*state.rho - screened_rates(k_p_ni61__cu62)*Y(Ni61)*state.rho - screened_rates(k_p_ni61__he4_co58)*Y(Ni61)*state.rho - screened_rates(k_p_ni61__n_cu61)*Y(Ni61)*state.rho - screened_rates(k_p_ni62__cu63)*Y(Ni62)*state.rho - screened_rates(k_p_ni62__he4_co59)*Y(Ni62)*state.rho - screened_rates(k_p_ni62__n_cu62)*Y(Ni62)*state.rho - screened_rates(k_p_ni63__cu64)*Y(Ni63)*state.rho - screened_rates(k_p_ni63__n_cu63)*Y(Ni63)*state.rho - screened_rates(k_p_ni64__cu65)*Y(Ni64)*state.rho - screened_rates(k_p_ni64__n_cu64)*Y(Ni64)*state.rho - screened_rates(k_p_o16__f17)*Y(O16)*state.rho - screened_rates(k_p_o16__he4_n13)*Y(O16)*state.rho - screened_rates(k_p_o17__f18)*Y(O17)*state.rho - screened_rates(k_p_o17__he4_n14)*Y(O17)*state.rho - screened_rates(k_p_o17__n_f17)*Y(O17)*state.rho - screened_rates(k_p_o18__f19)*Y(O18)*state.rho - screened_rates(k_p_o18__he4_n15)*Y(O18)*state.rho - screened_rates(k_p_o18__n_f18)*Y(O18)*state.rho - screened_rates(k_p_p31__c12_ne20)*Y(P31)*state.rho - screened_rates(k_p_p31__he4_si28)*Y(P31)*state.rho - screened_rates(k_p_p31__o16_o16)*Y(P31)*state.rho - screened_rates(k_p_p31__s32)*Y(P31)*state.rho - screened_rates(k_p_p32__he4_si29)*Y(P32)*state.rho - screened_rates(k_p_p32__n_s32)*Y(P32)*state.rho - screened_rates(k_p_p32__s33)*Y(P32)*state.rho - screened_rates(k_p_p33__he4_si30)*Y(P33)*state.rho - screened_rates(k_p_p33__n_s33)*Y(P33)*state.rho - screened_rates(k_p_p33__s34)*Y(P33)*state.rho - 2.0*screened_rates(k_p_p__d__weak__bet_pos_)*Y(H1)*state.rho - 2.0*screened_rates(k_p_p__d__weak__electron_capture)*Y(H1)*std::pow(state.rho, 2)*state.y_e - 2.0*screened_rates(k_p_p_he4__he3_he3)*Y(He4)*Y(H1)*std::pow(state.rho, 2) - 1.0*screened_rates(k_p_p_he4_he4__he3_be7)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 3) - screened_rates(k_p_s32__cl33)*Y(S32)*state.rho - screened_rates(k_p_s32__he4_p29)*Y(S32)*state.rho - screened_rates(k_p_s33__cl34)*Y(S33)*state.rho - screened_rates(k_p_s33__he4_p30)*Y(S33)*state.rho - screened_rates(k_p_s33__n_cl33)*Y(S33)*state.rho - screened_rates(k_p_s34__cl35)*Y(S34)*state.rho - screened_rates(k_p_s34__he4_p31)*Y(S34)*state.rho - screened_rates(k_p_s34__n_cl34)*Y(S34)*state.rho - screened_rates(k_p_s35__cl36)*Y(S35)*state.rho - screened_rates(k_p_s35__he4_p32)*Y(S35)*state.rho - screened_rates(k_p_s35__n_cl35)*Y(S35)*state.rho - screened_rates(k_p_s36__cl37)*Y(S36)*state.rho - screened_rates(k_p_s36__he4_p33)*Y(S36)*state.rho - screened_rates(k_p_s36__n_cl36)*Y(S36)*state.rho - screened_rates(k_p_sc43__he4_ca40)*Y(Sc43)*state.rho - screened_rates(k_p_sc43__ti44)*Y(Sc43)*state.rho - screened_rates(k_p_sc44__he4_ca41)*Y(Sc44)*state.rho - screened_rates(k_p_sc44__n_ti44)*Y(Sc44)*state.rho - screened_rates(k_p_sc44__ti45)*Y(Sc44)*state.rho - screened_rates(k_p_sc45__he4_ca42)*Y(Sc45)*state.rho - screened_rates(k_p_sc45__n_ti45)*Y(Sc45)*state.rho - screened_rates(k_p_sc45__ti46)*Y(Sc45)*state.rho - screened_rates(k_p_sc46__he4_ca43)*Y(Sc46)*state.rho - screened_rates(k_p_sc46__n_ti46)*Y(Sc46)*state.rho - screened_rates(k_p_sc46__ti47)*Y(Sc46)*state.rho - screened_rates(k_p_sc47__he4_ca44)*Y(Sc47)*state.rho - screened_rates(k_p_sc47__n_ti47)*Y(Sc47)*state.rho - screened_rates(k_p_sc47__ti48)*Y(Sc47)*state.rho - screened_rates(k_p_sc48__he4_ca45)*Y(Sc48)*state.rho - screened_rates(k_p_sc48__n_ti48)*Y(Sc48)*state.rho - screened_rates(k_p_sc48__ti49)*Y(Sc48)*state.rho - screened_rates(k_p_sc49__he4_ca46)*Y(Sc49)*state.rho - screened_rates(k_p_sc49__n_ti49)*Y(Sc49)*state.rho - screened_rates(k_p_sc49__ti50)*Y(Sc49)*state.rho - screened_rates(k_p_si28__he4_al25)*Y(Si28)*state.rho - screened_rates(k_p_si28__p29)*Y(Si28)*state.rho - screened_rates(k_p_si29__he4_al26)*Y(Si29)*state.rho - screened_rates(k_p_si29__n_p29)*Y(Si29)*state.rho - screened_rates(k_p_si29__p30)*Y(Si29)*state.rho - screened_rates(k_p_si30__he4_al27)*Y(Si30)*state.rho - screened_rates(k_p_si30__n_p30)*Y(Si30)*state.rho - screened_rates(k_p_si30__p31)*Y(Si30)*state.rho - screened_rates(k_p_si31__n_p31)*Y(Si31)*state.rho - screened_rates(k_p_si31__p32)*Y(Si31)*state.rho - screened_rates(k_p_si32__n_p32)*Y(Si32)*state.rho - screened_rates(k_p_si32__p33)*Y(Si32)*state.rho - screened_rates(k_p_ti45__v46)*Y(Ti45)*state.rho - screened_rates(k_p_ti46__he4_sc43)*Y(Ti46)*state.rho - screened_rates(k_p_ti46__n_v46)*Y(Ti46)*state.rho - screened_rates(k_p_ti46__v47)*Y(Ti46)*state.rho - screened_rates(k_p_ti47__he4_sc44)*Y(Ti47)*state.rho - screened_rates(k_p_ti47__n_v47)*Y(Ti47)*state.rho - screened_rates(k_p_ti47__v48)*Y(Ti47)*state.rho - screened_rates(k_p_ti48__he4_sc45)*Y(Ti48)*state.rho - screened_rates(k_p_ti48__n_v48)*Y(Ti48)*state.rho - screened_rates(k_p_ti48__v49)*Y(Ti48)*state.rho - screened_rates(k_p_ti49__he4_sc46)*Y(Ti49)*state.rho - screened_rates(k_p_ti49__n_v49)*Y(Ti49)*state.rho - screened_rates(k_p_ti49__v50)*Y(Ti49)*state.rho - screened_rates(k_p_ti50__he4_sc47)*Y(Ti50)*state.rho - screened_rates(k_p_ti50__n_v50)*Y(Ti50)*state.rho - screened_rates(k_p_ti50__v51)*Y(Ti50)*state.rho - screened_rates(k_p_ti51__he4_sc48)*Y(Ti51)*state.rho - screened_rates(k_p_ti51__n_v51)*Y(Ti51)*state.rho - screened_rates(k_p_ti51__v52)*Y(Ti51)*state.rho - screened_rates(k_p_v47__cr48)*Y(V47)*state.rho - screened_rates(k_p_v47__he4_ti44)*Y(V47)*state.rho - screened_rates(k_p_v48__cr49)*Y(V48)*state.rho - screened_rates(k_p_v48__he4_ti45)*Y(V48)*state.rho - screened_rates(k_p_v48__n_cr48)*Y(V48)*state.rho - screened_rates(k_p_v49__cr50)*Y(V49)*state.rho - screened_rates(k_p_v49__he4_ti46)*Y(V49)*state.rho - screened_rates(k_p_v49__n_cr49)*Y(V49)*state.rho - screened_rates(k_p_v50__cr51)*Y(V50)*state.rho - screened_rates(k_p_v50__he4_ti47)*Y(V50)*state.rho - screened_rates(k_p_v50__n_cr50)*Y(V50)*state.rho - screened_rates(k_p_v51__cr52)*Y(V51)*state.rho - screened_rates(k_p_v51__he4_ti48)*Y(V51)*state.rho - screened_rates(k_p_v51__n_cr51)*Y(V51)*state.rho - screened_rates(k_p_v52__cr53)*Y(V52)*state.rho - screened_rates(k_p_v52__he4_ti49)*Y(V52)*state.rho - screened_rates(k_p_v52__n_cr52)*Y(V52)*state.rho - screened_rates(k_p_zn60__he4_cu57)*Y(Zn60)*state.rho - screened_rates(k_p_zn61__ga62)*Y(Zn61)*state.rho - screened_rates(k_p_zn61__he4_cu58)*Y(Zn61)*state.rho - screened_rates(k_p_zn62__ga63)*Y(Zn62)*state.rho - screened_rates(k_p_zn62__he4_cu59)*Y(Zn62)*state.rho - screened_rates(k_p_zn62__n_ga62)*Y(Zn62)*state.rho - screened_rates(k_p_zn63__ga64)*Y(Zn63)*state.rho - screened_rates(k_p_zn63__he4_cu60)*Y(Zn63)*state.rho - screened_rates(k_p_zn63__n_ga63)*Y(Zn63)*state.rho - screened_rates(k_p_zn64__he4_cu61)*Y(Zn64)*state.rho - screened_rates(k_p_zn64__n_ga64)*Y(Zn64)*state.rho - screened_rates(k_p_zn65__he4_cu62)*Y(Zn65)*state.rho - screened_rates(k_p_zn66__he4_cu63)*Y(Zn66)*state.rho;
    jac.set(H1, H1, scratch);

    scratch = screened_rates(k_d__n_p) + screened_rates(k_d_be7__p_he4_he4)*Y(Be7)*state.rho + screened_rates(k_d_he3__p_he4)*Y(He3)*state.rho + 0.5*screened_rates(k_d_he4_he4__p_be9)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) + screened_rates(k_d_li6__p_li7)*Y(Li6)*state.rho - screened_rates(k_p_d__he3)*Y(H1)*state.rho + 2*screened_rates(k_p_d__n_p_p)*Y(H1)*state.rho;
    jac.set(H1, H2, scratch);

    scratch = screened_rates(k_d_he3__p_he4)*Y(H2)*state.rho + screened_rates(k_he3__p_d) + 2.0*screened_rates(k_he3_be7__p_p_he4_he4)*Y(Be7)*state.rho + 2.0*screened_rates(k_he3_he3__p_p_he4)*Y(He3)*state.rho + screened_rates(k_he3_li7__n_p_he4_he4)*Y(Li7)*state.rho + screened_rates(k_he4_he3__p_li6)*Y(He4)*state.rho - screened_rates(k_p_he3__he4__weak__bet_pos_)*Y(H1)*state.rho;
    jac.set(H1, He3, scratch);

    scratch = 1.0*screened_rates(k_d_he4_he4__p_be9)*Y(H2)*Y(He4)*std::pow(state.rho, 2) + screened_rates(k_he4_al25__p_si28)*Y(Al25)*state.rho + screened_rates(k_he4_al26__p_si29)*Y(Al26)*state.rho + screened_rates(k_he4_al27__p_si30)*Y(Al27)*state.rho + screened_rates(k_he4_ar36__p_k39)*Y(Ar36)*state.rho + screened_rates(k_he4_ar37__p_k40)*Y(Ar37)*state.rho + screened_rates(k_he4_ar38__p_k41)*Y(Ar38)*state.rho + screened_rates(k_he4_b10__p_c13)*Y(B10)*state.rho + screened_rates(k_he4_b11__p_c14)*Y(B11)*state.rho + screened_rates(k_he4_be7__p_b10)*Y(Be7)*state.rho + screened_rates(k_he4_c12__p_n15)*Y(C12)*state.rho + screened_rates(k_he4_ca40__p_sc43)*Y(Ca40)*state.rho + screened_rates(k_he4_ca41__p_sc44)*Y(Ca41)*state.rho + screened_rates(k_he4_ca42__p_sc45)*Y(Ca42)*state.rho + screened_rates(k_he4_ca43__p_sc46)*Y(Ca43)*state.rho + screened_rates(k_he4_ca44__p_sc47)*Y(Ca44)*state.rho + screened_rates(k_he4_ca45__p_sc48)*Y(Ca45)*state.rho + screened_rates(k_he4_ca46__p_sc49)*Y(Ca46)*state.rho + screened_rates(k_he4_cl33__p_ar36)*Y(Cl33)*state.rho + screened_rates(k_he4_cl34__p_ar37)*Y(Cl34)*state.rho + screened_rates(k_he4_cl35__p_ar38)*Y(Cl35)*state.rho + screened_rates(k_he4_cl36__p_ar39)*Y(Cl36)*state.rho + screened_rates(k_he4_cl37__p_ar40)*Y(Cl37)*state.rho + screened_rates(k_he4_co53__p_ni56)*Y(Co53)*state.rho + screened_rates(k_he4_co54__p_ni57)*Y(Co54)*state.rho + screened_rates(k_he4_co55__p_ni58)*Y(Co55)*state.rho + screened_rates(k_he4_co56__p_ni59)*Y(Co56)*state.rho + screened_rates(k_he4_co57__p_ni60)*Y(Co57)*state.rho + screened_rates(k_he4_co58__p_ni61)*Y(Co58)*state.rho + screened_rates(k_he4_co59__p_ni62)*Y(Co59)*state.rho + screened_rates(k_he4_cr48__p_mn51)*Y(Cr48)*state.rho + screened_rates(k_he4_cr49__p_mn52)*Y(Cr49)*state.rho + screened_rates(k_he4_cr50__p_mn53)*Y(Cr50)*state.rho + screened_rates(k_he4_cr51__p_mn54)*Y(Cr51)*state.rho + screened_rates(k_he4_cr52__p_mn55)*Y(Cr52)*state.rho + screened_rates(k_he4_cu57__p_zn60)*Y(Cu57)*state.rho + screened_rates(k_he4_cu58__p_zn61)*Y(Cu58)*state.rho + screened_rates(k_he4_cu59__p_zn62)*Y(Cu59)*state.rho + screened_rates(k_he4_cu60__p_zn63)*Y(Cu60)*state.rho + screened_rates(k_he4_cu61__p_zn64)*Y(Cu61)*state.rho + screened_rates(k_he4_cu62__p_zn65)*Y(Cu62)*state.rho + screened_rates(k_he4_cu63__p_zn66)*Y(Cu63)*state.rho + screened_rates(k_he4_f17__p_ne20)*Y(F17)*state.rho + screened_rates(k_he4_f18__p_ne21)*Y(F18)*state.rho + screened_rates(k_he4_f19__p_ne22)*Y(F19)*state.rho + screened_rates(k_he4_fe52__p_co55)*Y(Fe52)*state.rho + screened_rates(k_he4_fe53__p_co56)*Y(Fe53)*state.rho + screened_rates(k_he4_fe54__p_co57)*Y(Fe54)*state.rho + screened_rates(k_he4_fe55__p_co58)*Y(Fe55)*state.rho + screened_rates(k_he4_fe56__p_co59)*Y(Fe56)*state.rho + screened_rates(k_he4_he3__p_li6)*Y(He3)*state.rho + 1.0*screened_rates(k_he4_he4__p_li7)*Y(He4)*state.rho + 0.5*screened_rates(k_he4_he4_he4__p_b11)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) + screened_rates(k_he4_k37__p_ca40)*Y(K37)*state.rho + screened_rates(k_he4_k38__p_ca41)*Y(K38)*state.rho + screened_rates(k_he4_k39__p_ca42)*Y(K39)*state.rho + screened_rates(k_he4_k40__p_ca43)*Y(K40)*state.rho + screened_rates(k_he4_k41__p_ca44)*Y(K41)*state.rho + screened_rates(k_he4_li6__p_be9)*Y(Li6)*state.rho + screened_rates(k_he4_mg23__p_al26)*Y(Mg23)*state.rho + screened_rates(k_he4_mg24__p_al27)*Y(Mg24)*state.rho + screened_rates(k_he4_mn50__p_fe53)*Y(Mn50)*state.rho + screened_rates(k_he4_mn51__p_fe54)*Y(Mn51)*state.rho + screened_rates(k_he4_mn52__p_fe55)*Y(Mn52)*state.rho + screened_rates(k_he4_mn53__p_fe56)*Y(Mn53)*state.rho + screened_rates(k_he4_mn54__p_fe57)*Y(Mn54)*state.rho + screened_rates(k_he4_mn55__p_fe58)*Y(Mn55)*state.rho + screened_rates(k_he4_n13__p_o16)*Y(N13)*state.rho + screened_rates(k_he4_n14__p_o17)*Y(N14)*state.rho + screened_rates(k_he4_n15__p_o18)*Y(N15)*state.rho + screened_rates(k_he4_na21__p_mg24)*Y(Na21)*state.rho + screened_rates(k_he4_na22__p_mg25)*Y(Na22)*state.rho + screened_rates(k_he4_na23__p_mg26)*Y(Na23)*state.rho + screened_rates(k_he4_ne18__p_na21)*Y(Ne18)*state.rho + screened_rates(k_he4_ne19__p_na22)*Y(Ne19)*state.rho + screened_rates(k_he4_ne20__p_na23)*Y(Ne20)*state.rho + screened_rates(k_he4_ni56__p_cu59)*Y(Ni56)*state.rho + screened_rates(k_he4_ni57__p_cu60)*Y(Ni57)*state.rho + screened_rates(k_he4_ni58__p_cu61)*Y(Ni58)*state.rho + screened_rates(k_he4_ni59__p_cu62)*Y(Ni59)*state.rho + screened_rates(k_he4_ni60__p_cu63)*Y(Ni60)*state.rho + screened_rates(k_he4_ni61__p_cu64)*Y(Ni61)*state.rho + screened_rates(k_he4_ni62__p_cu65)*Y(Ni62)*state.rho + screened_rates(k_he4_o14__p_f17)*Y(O14)*state.rho + screened_rates(k_he4_o15__p_f18)*Y(O15)*state.rho + screened_rates(k_he4_o16__p_f19)*Y(O16)*state.rho + screened_rates(k_he4_p29__p_s32)*Y(P29)*state.rho + screened_rates(k_he4_p30__p_s33)*Y(P30)*state.rho + screened_rates(k_he4_p31__p_s34)*Y(P31)*state.rho + screened_rates(k_he4_p32__p_s35)*Y(P32)*state.rho + screened_rates(k_he4_p33__p_s36)*Y(P33)*state.rho + screened_rates(k_he4_s32__p_cl35)*Y(S32)*state.rho + screened_rates(k_he4_s33__p_cl36)*Y(S33)*state.rho + screened_rates(k_he4_s34__p_cl37)*Y(S34)*state.rho + screened_rates(k_he4_sc43__p_ti46)*Y(Sc43)*state.rho + screened_rates(k_he4_sc44__p_ti47)*Y(Sc44)*state.rho + screened_rates(k_he4_sc45__p_ti48)*Y(Sc45)*state.rho + screened_rates(k_he4_sc46__p_ti49)*Y(Sc46)*state.rho + screened_rates(k_he4_sc47__p_ti50)*Y(Sc47)*state.rho + screened_rates(k_he4_sc48__p_ti51)*Y(Sc48)*state.rho + screened_rates(k_he4_si28__p_p31)*Y(Si28)*state.rho + screened_rates(k_he4_si29__p_p32)*Y(Si29)*state.rho + screened_rates(k_he4_si30__p_p33)*Y(Si30)*state.rho + screened_rates(k_he4_ti44__p_v47)*Y(Ti44)*state.rho + screened_rates(k_he4_ti45__p_v48)*Y(Ti45)*state.rho + screened_rates(k_he4_ti46__p_v49)*Y(Ti46)*state.rho + screened_rates(k_he4_ti47__p_v50)*Y(Ti47)*state.rho + screened_rates(k_he4_ti48__p_v51)*Y(Ti48)*state.rho + screened_rates(k_he4_ti49__p_v52)*Y(Ti49)*state.rho + screened_rates(k_he4_v46__p_cr49)*Y(V46)*state.rho + screened_rates(k_he4_v47__p_cr50)*Y(V47)*state.rho + screened_rates(k_he4_v48__p_cr51)*Y(V48)*state.rho + screened_rates(k_he4_v49__p_cr52)*Y(V49)*state.rho + screened_rates(k_he4_v50__p_cr53)*Y(V50)*state.rho + screened_rates(k_he4_v51__p_cr54)*Y(V51)*state.rho + screened_rates(k_he4_zn59__p_ga62)*Y(Zn59)*state.rho + screened_rates(k_he4_zn60__p_ga63)*Y(Zn60)*state.rho + screened_rates(k_he4_zn61__p_ga64)*Y(Zn61)*state.rho - screened_rates(k_n_p_he4__li6)*Y(N)*Y(H1)*std::pow(state.rho, 2) - 1.0*screened_rates(k_n_p_he4_he4__he3_li7)*Y(He4)*Y(N)*Y(H1)*std::pow(state.rho, 3) - screened_rates(k_p_he4__d_he3)*Y(H1)*state.rho - 1.0*screened_rates(k_p_he4_he4__d_be7)*Y(He4)*Y(H1)*std::pow(state.rho, 2) - 1.0*screened_rates(k_p_he4_he4__n_b8)*Y(He4)*Y(H1)*std::pow(state.rho, 2) - screened_rates(k_p_p_he4__he3_he3)*std::pow(Y(H1), 2)*std::pow(state.rho, 2) - 1.0*screened_rates(k_p_p_he4_he4__he3_be7)*Y(He4)*std::pow(Y(H1), 2)*std::pow(state.rho, 3);
    jac.set(H1, He4, scratch);

    scratch = screened_rates(k_d_li6__p_li7)*Y(H2)*state.rho + screened_rates(k_he4_li6__p_be9)*Y(He4)*state.rho + screened_rates(k_li6__n_p_he4) - screened_rates(k_p_li6__be7)*Y(H1)*state.rho - screened_rates(k_p_li6__he4_he3)*Y(H1)*state.rho;
    jac.set(H1, Li6, scratch);

    scratch = screened_rates(k_he3_li7__n_p_he4_he4)*Y(He3)*state.rho - screened_rates(k_p_li7__d_li6)*Y(H1)*state.rho - screened_rates(k_p_li7__he4_he4)*Y(H1)*state.rho - screened_rates(k_p_li7__n_be7)*Y(H1)*state.rho;
    jac.set(H1, Li7, scratch);

    scratch = screened_rates(k_be7__p_li6) + screened_rates(k_d_be7__p_he4_he4)*Y(H2)*state.rho + 2.0*screened_rates(k_he3_be7__p_p_he4_he4)*Y(He3)*state.rho + screened_rates(k_he4_be7__p_b10)*Y(He4)*state.rho + screened_rates(k_n_be7__p_li7)*Y(N)*state.rho - screened_rates(k_p_be7__b8)*Y(H1)*state.rho;
    jac.set(H1, Be7, scratch);

    scratch = -screened_rates(k_p_be9__b10)*Y(H1)*state.rho - screened_rates(k_p_be9__d_he4_he4)*Y(H1)*state.rho - screened_rates(k_p_be9__he4_li6)*Y(H1)*state.rho;
    jac.set(H1, Be9, scratch);

    scratch = screened_rates(k_b8__p_be7) + screened_rates(k_n_b8__p_he4_he4)*Y(N)*state.rho;
    jac.set(H1, B8, scratch);

    scratch = screened_rates(k_b10__p_be9) + screened_rates(k_he4_b10__p_c13)*Y(He4)*state.rho - screened_rates(k_p_b10__he4_be7)*Y(H1)*state.rho;
    jac.set(H1, B10, scratch);

    scratch = screened_rates(k_he4_b11__p_c14)*Y(He4)*state.rho - screened_rates(k_p_b11__c12)*Y(H1)*state.rho - screened_rates(k_p_b11__he4_he4_he4)*Y(H1)*state.rho;
    jac.set(H1, B11, scratch);

    scratch = screened_rates(k_c12__p_b11) + 1.0*screened_rates(k_c12_c12__p_na23)*Y(C12)*state.rho + screened_rates(k_c12_ne20__p_p31)*Y(Ne20)*state.rho + screened_rates(k_c12_o16__p_al27)*Y(O16)*state.rho + screened_rates(k_he4_c12__p_n15)*Y(He4)*state.rho - screened_rates(k_p_c12__n13)*Y(H1)*state.rho;
    jac.set(H1, C12, scratch);

    scratch = -screened_rates(k_p_c13__he4_b10)*Y(H1)*state.rho - screened_rates(k_p_c13__n14)*Y(H1)*state.rho - screened_rates(k_p_c13__n_n13)*Y(H1)*state.rho;
    jac.set(H1, C13, scratch);

    scratch = -screened_rates(k_p_c14__he4_b11)*Y(H1)*state.rho - screened_rates(k_p_c14__n15)*Y(H1)*state.rho - screened_rates(k_p_c14__n_n14)*Y(H1)*state.rho;
    jac.set(H1, C14, scratch);

    scratch = screened_rates(k_he4_n13__p_o16)*Y(He4)*state.rho + screened_rates(k_n13__p_c12) + screened_rates(k_n_n13__p_c13)*Y(N)*state.rho - screened_rates(k_p_n13__o14)*Y(H1)*state.rho;
    jac.set(H1, N13, scratch);

    scratch = screened_rates(k_he4_n14__p_o17)*Y(He4)*state.rho + screened_rates(k_n14__p_c13) + screened_rates(k_n_n14__p_c14)*Y(N)*state.rho - screened_rates(k_p_n14__n_o14)*Y(H1)*state.rho - screened_rates(k_p_n14__o15)*Y(H1)*state.rho;
    jac.set(H1, N14, scratch);

    scratch = screened_rates(k_he4_n15__p_o18)*Y(He4)*state.rho + screened_rates(k_n15__p_c14) - screened_rates(k_p_n15__he4_c12)*Y(H1)*state.rho - screened_rates(k_p_n15__n_o15)*Y(H1)*state.rho - screened_rates(k_p_n15__o16)*Y(H1)*state.rho;
    jac.set(H1, N15, scratch);

    scratch = screened_rates(k_he4_o14__p_f17)*Y(He4)*state.rho + screened_rates(k_n_o14__p_n14)*Y(N)*state.rho + screened_rates(k_o14__p_n13);
    jac.set(H1, O14, scratch);

    scratch = screened_rates(k_he4_o15__p_f18)*Y(He4)*state.rho + screened_rates(k_n_o15__p_n15)*Y(N)*state.rho + screened_rates(k_o15__p_n14);
    jac.set(H1, O15, scratch);

    scratch = screened_rates(k_c12_o16__p_al27)*Y(C12)*state.rho + screened_rates(k_he4_o16__p_f19)*Y(He4)*state.rho + screened_rates(k_o16__p_n15) + 1.0*screened_rates(k_o16_o16__p_p31)*Y(O16)*state.rho - screened_rates(k_p_o16__f17)*Y(H1)*state.rho - screened_rates(k_p_o16__he4_n13)*Y(H1)*state.rho;
    jac.set(H1, O16, scratch);

    scratch = -screened_rates(k_p_o17__f18)*Y(H1)*state.rho - screened_rates(k_p_o17__he4_n14)*Y(H1)*state.rho - screened_rates(k_p_o17__n_f17)*Y(H1)*state.rho;
    jac.set(H1, O17, scratch);

    scratch = -screened_rates(k_p_o18__f19)*Y(H1)*state.rho - screened_rates(k_p_o18__he4_n15)*Y(H1)*state.rho - screened_rates(k_p_o18__n_f18)*Y(H1)*state.rho;
    jac.set(H1, O18, scratch);

    scratch = screened_rates(k_f17__p_o16) + screened_rates(k_he4_f17__p_ne20)*Y(He4)*state.rho + screened_rates(k_n_f17__p_o17)*Y(N)*state.rho - screened_rates(k_p_f17__he4_o14)*Y(H1)*state.rho - screened_rates(k_p_f17__ne18)*Y(H1)*state.rho;
    jac.set(H1, F17, scratch);

    scratch = screened_rates(k_f18__p_o17) + screened_rates(k_he4_f18__p_ne21)*Y(He4)*state.rho + screened_rates(k_n_f18__p_o18)*Y(N)*state.rho - screened_rates(k_p_f18__he4_o15)*Y(H1)*state.rho - screened_rates(k_p_f18__n_ne18)*Y(H1)*state.rho - screened_rates(k_p_f18__ne19)*Y(H1)*state.rho;
    jac.set(H1, F18, scratch);

    scratch = screened_rates(k_f19__p_o18) + screened_rates(k_he4_f19__p_ne22)*Y(He4)*state.rho - screened_rates(k_p_f19__he4_o16)*Y(H1)*state.rho - screened_rates(k_p_f19__n_ne19)*Y(H1)*state.rho - screened_rates(k_p_f19__ne20)*Y(H1)*state.rho;
    jac.set(H1, F19, scratch);

    scratch = screened_rates(k_he4_ne18__p_na21)*Y(He4)*state.rho + screened_rates(k_n_ne18__p_f18)*Y(N)*state.rho + screened_rates(k_ne18__p_f17);
    jac.set(H1, Ne18, scratch);

    scratch = screened_rates(k_he4_ne19__p_na22)*Y(He4)*state.rho + screened_rates(k_n_ne19__p_f19)*Y(N)*state.rho + screened_rates(k_ne19__p_f18);
    jac.set(H1, Ne19, scratch);

    scratch = screened_rates(k_c12_ne20__p_p31)*Y(C12)*state.rho + screened_rates(k_he4_ne20__p_na23)*Y(He4)*state.rho + screened_rates(k_ne20__p_f19) - screened_rates(k_p_ne20__he4_f17)*Y(H1)*state.rho - screened_rates(k_p_ne20__na21)*Y(H1)*state.rho;
    jac.set(H1, Ne20, scratch);

    scratch = -screened_rates(k_p_ne21__he4_f18)*Y(H1)*state.rho - screened_rates(k_p_ne21__n_na21)*Y(H1)*state.rho - screened_rates(k_p_ne21__na22)*Y(H1)*state.rho;
    jac.set(H1, Ne21, scratch);

    scratch = -screened_rates(k_p_ne22__he4_f19)*Y(H1)*state.rho - screened_rates(k_p_ne22__n_na22)*Y(H1)*state.rho - screened_rates(k_p_ne22__na23)*Y(H1)*state.rho;
    jac.set(H1, Ne22, scratch);

    scratch = screened_rates(k_he4_na21__p_mg24)*Y(He4)*state.rho + screened_rates(k_n_na21__p_ne21)*Y(N)*state.rho + screened_rates(k_na21__p_ne20) - screened_rates(k_p_na21__he4_ne18)*Y(H1)*state.rho;
    jac.set(H1, Na21, scratch);

    scratch = screened_rates(k_he4_na22__p_mg25)*Y(He4)*state.rho + screened_rates(k_n_na22__p_ne22)*Y(N)*state.rho + screened_rates(k_na22__p_ne21) - screened_rates(k_p_na22__he4_ne19)*Y(H1)*state.rho - screened_rates(k_p_na22__mg23)*Y(H1)*state.rho;
    jac.set(H1, Na22, scratch);

    scratch = screened_rates(k_he4_na23__p_mg26)*Y(He4)*state.rho + screened_rates(k_na23__p_ne22) - screened_rates(k_p_na23__c12_c12)*Y(H1)*state.rho - screened_rates(k_p_na23__he4_ne20)*Y(H1)*state.rho - screened_rates(k_p_na23__mg24)*Y(H1)*state.rho - screened_rates(k_p_na23__n_mg23)*Y(H1)*state.rho;
    jac.set(H1, Na23, scratch);

    scratch = screened_rates(k_he4_mg23__p_al26)*Y(He4)*state.rho + screened_rates(k_mg23__p_na22) + screened_rates(k_n_mg23__p_na23)*Y(N)*state.rho;
    jac.set(H1, Mg23, scratch);

    scratch = screened_rates(k_he4_mg24__p_al27)*Y(He4)*state.rho + screened_rates(k_mg24__p_na23) - screened_rates(k_p_mg24__al25)*Y(H1)*state.rho - screened_rates(k_p_mg24__he4_na21)*Y(H1)*state.rho;
    jac.set(H1, Mg24, scratch);

    scratch = -screened_rates(k_p_mg25__al26)*Y(H1)*state.rho - screened_rates(k_p_mg25__he4_na22)*Y(H1)*state.rho - screened_rates(k_p_mg25__n_al25)*Y(H1)*state.rho;
    jac.set(H1, Mg25, scratch);

    scratch = -screened_rates(k_p_mg26__al27)*Y(H1)*state.rho - screened_rates(k_p_mg26__he4_na23)*Y(H1)*state.rho - screened_rates(k_p_mg26__n_al26)*Y(H1)*state.rho;
    jac.set(H1, Mg26, scratch);

    scratch = screened_rates(k_al25__p_mg24) + screened_rates(k_he4_al25__p_si28)*Y(He4)*state.rho + screened_rates(k_n_al25__p_mg25)*Y(N)*state.rho;
    jac.set(H1, Al25, scratch);

    scratch = screened_rates(k_al26__p_mg25) + screened_rates(k_he4_al26__p_si29)*Y(He4)*state.rho + screened_rates(k_n_al26__p_mg26)*Y(N)*state.rho - screened_rates(k_p_al26__he4_mg23)*Y(H1)*state.rho;
    jac.set(H1, Al26, scratch);

    scratch = screened_rates(k_al27__p_mg26) + screened_rates(k_he4_al27__p_si30)*Y(He4)*state.rho - screened_rates(k_p_al27__c12_o16)*Y(H1)*state.rho - screened_rates(k_p_al27__he4_mg24)*Y(H1)*state.rho - screened_rates(k_p_al27__si28)*Y(H1)*state.rho;
    jac.set(H1, Al27, scratch);

    scratch = screened_rates(k_he4_si28__p_p31)*Y(He4)*state.rho - screened_rates(k_p_si28__he4_al25)*Y(H1)*state.rho - screened_rates(k_p_si28__p29)*Y(H1)*state.rho + screened_rates(k_si28__p_al27);
    jac.set(H1, Si28, scratch);

    scratch = screened_rates(k_he4_si29__p_p32)*Y(He4)*state.rho - screened_rates(k_p_si29__he4_al26)*Y(H1)*state.rho - screened_rates(k_p_si29__n_p29)*Y(H1)*state.rho - screened_rates(k_p_si29__p30)*Y(H1)*state.rho;
    jac.set(H1, Si29, scratch);

    scratch = screened_rates(k_he4_si30__p_p33)*Y(He4)*state.rho - screened_rates(k_p_si30__he4_al27)*Y(H1)*state.rho - screened_rates(k_p_si30__n_p30)*Y(H1)*state.rho - screened_rates(k_p_si30__p31)*Y(H1)*state.rho;
    jac.set(H1, Si30, scratch);

    scratch = -screened_rates(k_p_si31__n_p31)*Y(H1)*state.rho - screened_rates(k_p_si31__p32)*Y(H1)*state.rho;
    jac.set(H1, Si31, scratch);

    scratch = -screened_rates(k_p_si32__n_p32)*Y(H1)*state.rho - screened_rates(k_p_si32__p33)*Y(H1)*state.rho;
    jac.set(H1, Si32, scratch);

    scratch = screened_rates(k_he4_p29__p_s32)*Y(He4)*state.rho + screened_rates(k_n_p29__p_si29)*Y(N)*state.rho + screened_rates(k_p29__p_si28);
    jac.set(H1, P29, scratch);

    scratch = screened_rates(k_he4_p30__p_s33)*Y(He4)*state.rho + screened_rates(k_n_p30__p_si30)*Y(N)*state.rho + screened_rates(k_p30__p_si29);
    jac.set(H1, P30, scratch);

    scratch = screened_rates(k_he4_p31__p_s34)*Y(He4)*state.rho + screened_rates(k_n_p31__p_si31)*Y(N)*state.rho + screened_rates(k_p31__p_si30) - screened_rates(k_p_p31__c12_ne20)*Y(H1)*state.rho - screened_rates(k_p_p31__he4_si28)*Y(H1)*state.rho - screened_rates(k_p_p31__o16_o16)*Y(H1)*state.rho - screened_rates(k_p_p31__s32)*Y(H1)*state.rho;
    jac.set(H1, P31, scratch);

    scratch = screened_rates(k_he4_p32__p_s35)*Y(He4)*state.rho + screened_rates(k_n_p32__p_si32)*Y(N)*state.rho + screened_rates(k_p32__p_si31) - screened_rates(k_p_p32__he4_si29)*Y(H1)*state.rho - screened_rates(k_p_p32__n_s32)*Y(H1)*state.rho - screened_rates(k_p_p32__s33)*Y(H1)*state.rho;
    jac.set(H1, P32, scratch);

    scratch = screened_rates(k_he4_p33__p_s36)*Y(He4)*state.rho + screened_rates(k_p33__p_si32) - screened_rates(k_p_p33__he4_si30)*Y(H1)*state.rho - screened_rates(k_p_p33__n_s33)*Y(H1)*state.rho - screened_rates(k_p_p33__s34)*Y(H1)*state.rho;
    jac.set(H1, P33, scratch);

    scratch = screened_rates(k_he4_s32__p_cl35)*Y(He4)*state.rho + screened_rates(k_n_s32__p_p32)*Y(N)*state.rho - screened_rates(k_p_s32__cl33)*Y(H1)*state.rho - screened_rates(k_p_s32__he4_p29)*Y(H1)*state.rho + screened_rates(k_s32__p_p31);
    jac.set(H1, S32, scratch);

    scratch = screened_rates(k_he4_s33__p_cl36)*Y(He4)*state.rho + screened_rates(k_n_s33__p_p33)*Y(N)*state.rho - screened_rates(k_p_s33__cl34)*Y(H1)*state.rho - screened_rates(k_p_s33__he4_p30)*Y(H1)*state.rho - screened_rates(k_p_s33__n_cl33)*Y(H1)*state.rho + screened_rates(k_s33__p_p32);
    jac.set(H1, S33, scratch);

    scratch = screened_rates(k_he4_s34__p_cl37)*Y(He4)*state.rho - screened_rates(k_p_s34__cl35)*Y(H1)*state.rho - screened_rates(k_p_s34__he4_p31)*Y(H1)*state.rho - screened_rates(k_p_s34__n_cl34)*Y(H1)*state.rho + screened_rates(k_s34__p_p33);
    jac.set(H1, S34, scratch);

    scratch = -screened_rates(k_p_s35__cl36)*Y(H1)*state.rho - screened_rates(k_p_s35__he4_p32)*Y(H1)*state.rho - screened_rates(k_p_s35__n_cl35)*Y(H1)*state.rho;
    jac.set(H1, S35, scratch);

    scratch = -screened_rates(k_p_s36__cl37)*Y(H1)*state.rho - screened_rates(k_p_s36__he4_p33)*Y(H1)*state.rho - screened_rates(k_p_s36__n_cl36)*Y(H1)*state.rho;
    jac.set(H1, S36, scratch);

    scratch = screened_rates(k_cl33__p_s32) + screened_rates(k_he4_cl33__p_ar36)*Y(He4)*state.rho + screened_rates(k_n_cl33__p_s33)*Y(N)*state.rho;
    jac.set(H1, Cl33, scratch);

    scratch = screened_rates(k_cl34__p_s33) + screened_rates(k_he4_cl34__p_ar37)*Y(He4)*state.rho + screened_rates(k_n_cl34__p_s34)*Y(N)*state.rho;
    jac.set(H1, Cl34, scratch);

    scratch = screened_rates(k_cl35__p_s34) + screened_rates(k_he4_cl35__p_ar38)*Y(He4)*state.rho + screened_rates(k_n_cl35__p_s35)*Y(N)*state.rho - screened_rates(k_p_cl35__ar36)*Y(H1)*state.rho - screened_rates(k_p_cl35__he4_s32)*Y(H1)*state.rho;
    jac.set(H1, Cl35, scratch);

    scratch = screened_rates(k_cl36__p_s35) + screened_rates(k_he4_cl36__p_ar39)*Y(He4)*state.rho + screened_rates(k_n_cl36__p_s36)*Y(N)*state.rho - screened_rates(k_p_cl36__ar37)*Y(H1)*state.rho - screened_rates(k_p_cl36__he4_s33)*Y(H1)*state.rho - screened_rates(k_p_cl36__n_ar36)*Y(H1)*state.rho;
    jac.set(H1, Cl36, scratch);

    scratch = screened_rates(k_cl37__p_s36) + screened_rates(k_he4_cl37__p_ar40)*Y(He4)*state.rho - screened_rates(k_p_cl37__ar38)*Y(H1)*state.rho - screened_rates(k_p_cl37__he4_s34)*Y(H1)*state.rho - screened_rates(k_p_cl37__n_ar37)*Y(H1)*state.rho;
    jac.set(H1, Cl37, scratch);

    scratch = screened_rates(k_ar36__p_cl35) + screened_rates(k_he4_ar36__p_k39)*Y(He4)*state.rho + screened_rates(k_n_ar36__p_cl36)*Y(N)*state.rho - screened_rates(k_p_ar36__he4_cl33)*Y(H1)*state.rho - screened_rates(k_p_ar36__k37)*Y(H1)*state.rho;
    jac.set(H1, Ar36, scratch);

    scratch = screened_rates(k_ar37__p_cl36) + screened_rates(k_he4_ar37__p_k40)*Y(He4)*state.rho + screened_rates(k_n_ar37__p_cl37)*Y(N)*state.rho - screened_rates(k_p_ar37__he4_cl34)*Y(H1)*state.rho - screened_rates(k_p_ar37__k38)*Y(H1)*state.rho - screened_rates(k_p_ar37__n_k37)*Y(H1)*state.rho;
    jac.set(H1, Ar37, scratch);

    scratch = screened_rates(k_ar38__p_cl37) + screened_rates(k_he4_ar38__p_k41)*Y(He4)*state.rho - screened_rates(k_p_ar38__he4_cl35)*Y(H1)*state.rho - screened_rates(k_p_ar38__k39)*Y(H1)*state.rho - screened_rates(k_p_ar38__n_k38)*Y(H1)*state.rho;
    jac.set(H1, Ar38, scratch);

    scratch = -screened_rates(k_p_ar39__he4_cl36)*Y(H1)*state.rho - screened_rates(k_p_ar39__k40)*Y(H1)*state.rho - screened_rates(k_p_ar39__n_k39)*Y(H1)*state.rho;
    jac.set(H1, Ar39, scratch);

    scratch = -screened_rates(k_p_ar40__he4_cl37)*Y(H1)*state.rho - screened_rates(k_p_ar40__k41)*Y(H1)*state.rho - screened_rates(k_p_ar40__n_k40)*Y(H1)*state.rho;
    jac.set(H1, Ar40, scratch);

    scratch = screened_rates(k_he4_k37__p_ca40)*Y(He4)*state.rho + screened_rates(k_k37__p_ar36) + screened_rates(k_n_k37__p_ar37)*Y(N)*state.rho;
    jac.set(H1, K37, scratch);

    scratch = screened_rates(k_he4_k38__p_ca41)*Y(He4)*state.rho + screened_rates(k_k38__p_ar37) + screened_rates(k_n_k38__p_ar38)*Y(N)*state.rho;
    jac.set(H1, K38, scratch);

    scratch = screened_rates(k_he4_k39__p_ca42)*Y(He4)*state.rho + screened_rates(k_k39__p_ar38) + screened_rates(k_n_k39__p_ar39)*Y(N)*state.rho - screened_rates(k_p_k39__ca40)*Y(H1)*state.rho - screened_rates(k_p_k39__he4_ar36)*Y(H1)*state.rho;
    jac.set(H1, K39, scratch);

    scratch = screened_rates(k_he4_k40__p_ca43)*Y(He4)*state.rho + screened_rates(k_k40__p_ar39) + screened_rates(k_n_k40__p_ar40)*Y(N)*state.rho - screened_rates(k_p_k40__ca41)*Y(H1)*state.rho - screened_rates(k_p_k40__he4_ar37)*Y(H1)*state.rho - screened_rates(k_p_k40__n_ca40)*Y(H1)*state.rho;
    jac.set(H1, K40, scratch);

    scratch = screened_rates(k_he4_k41__p_ca44)*Y(He4)*state.rho + screened_rates(k_k41__p_ar40) - screened_rates(k_p_k41__ca42)*Y(H1)*state.rho - screened_rates(k_p_k41__he4_ar38)*Y(H1)*state.rho - screened_rates(k_p_k41__n_ca41)*Y(H1)*state.rho;
    jac.set(H1, K41, scratch);

    scratch = screened_rates(k_ca40__p_k39) + screened_rates(k_he4_ca40__p_sc43)*Y(He4)*state.rho + screened_rates(k_n_ca40__p_k40)*Y(N)*state.rho - screened_rates(k_p_ca40__he4_k37)*Y(H1)*state.rho;
    jac.set(H1, Ca40, scratch);

    scratch = screened_rates(k_ca41__p_k40) + screened_rates(k_he4_ca41__p_sc44)*Y(He4)*state.rho + screened_rates(k_n_ca41__p_k41)*Y(N)*state.rho - screened_rates(k_p_ca41__he4_k38)*Y(H1)*state.rho;
    jac.set(H1, Ca41, scratch);

    scratch = screened_rates(k_ca42__p_k41) + screened_rates(k_he4_ca42__p_sc45)*Y(He4)*state.rho - screened_rates(k_p_ca42__he4_k39)*Y(H1)*state.rho - screened_rates(k_p_ca42__sc43)*Y(H1)*state.rho;
    jac.set(H1, Ca42, scratch);

    scratch = screened_rates(k_he4_ca43__p_sc46)*Y(He4)*state.rho - screened_rates(k_p_ca43__he4_k40)*Y(H1)*state.rho - screened_rates(k_p_ca43__n_sc43)*Y(H1)*state.rho - screened_rates(k_p_ca43__sc44)*Y(H1)*state.rho;
    jac.set(H1, Ca43, scratch);

    scratch = screened_rates(k_he4_ca44__p_sc47)*Y(He4)*state.rho - screened_rates(k_p_ca44__he4_k41)*Y(H1)*state.rho - screened_rates(k_p_ca44__n_sc44)*Y(H1)*state.rho - screened_rates(k_p_ca44__sc45)*Y(H1)*state.rho;
    jac.set(H1, Ca44, scratch);

    scratch = screened_rates(k_he4_ca45__p_sc48)*Y(He4)*state.rho - screened_rates(k_p_ca45__n_sc45)*Y(H1)*state.rho - screened_rates(k_p_ca45__sc46)*Y(H1)*state.rho;
    jac.set(H1, Ca45, scratch);

    scratch = screened_rates(k_he4_ca46__p_sc49)*Y(He4)*state.rho - screened_rates(k_p_ca46__n_sc46)*Y(H1)*state.rho - screened_rates(k_p_ca46__sc47)*Y(H1)*state.rho;
    jac.set(H1, Ca46, scratch);

    scratch = -screened_rates(k_p_ca47__n_sc47)*Y(H1)*state.rho - screened_rates(k_p_ca47__sc48)*Y(H1)*state.rho;
    jac.set(H1, Ca47, scratch);

    scratch = -screened_rates(k_p_ca48__n_sc48)*Y(H1)*state.rho - screened_rates(k_p_ca48__sc49)*Y(H1)*state.rho;
    jac.set(H1, Ca48, scratch);

    scratch = screened_rates(k_he4_sc43__p_ti46)*Y(He4)*state.rho + screened_rates(k_n_sc43__p_ca43)*Y(N)*state.rho - screened_rates(k_p_sc43__he4_ca40)*Y(H1)*state.rho - screened_rates(k_p_sc43__ti44)*Y(H1)*state.rho + screened_rates(k_sc43__p_ca42);
    jac.set(H1, Sc43, scratch);

    scratch = screened_rates(k_he4_sc44__p_ti47)*Y(He4)*state.rho + screened_rates(k_n_sc44__p_ca44)*Y(N)*state.rho - screened_rates(k_p_sc44__he4_ca41)*Y(H1)*state.rho - screened_rates(k_p_sc44__n_ti44)*Y(H1)*state.rho - screened_rates(k_p_sc44__ti45)*Y(H1)*state.rho + screened_rates(k_sc44__p_ca43);
    jac.set(H1, Sc44, scratch);

    scratch = screened_rates(k_he4_sc45__p_ti48)*Y(He4)*state.rho + screened_rates(k_n_sc45__p_ca45)*Y(N)*state.rho - screened_rates(k_p_sc45__he4_ca42)*Y(H1)*state.rho - screened_rates(k_p_sc45__n_ti45)*Y(H1)*state.rho - screened_rates(k_p_sc45__ti46)*Y(H1)*state.rho + screened_rates(k_sc45__p_ca44);
    jac.set(H1, Sc45, scratch);

    scratch = screened_rates(k_he4_sc46__p_ti49)*Y(He4)*state.rho + screened_rates(k_n_sc46__p_ca46)*Y(N)*state.rho - screened_rates(k_p_sc46__he4_ca43)*Y(H1)*state.rho - screened_rates(k_p_sc46__n_ti46)*Y(H1)*state.rho - screened_rates(k_p_sc46__ti47)*Y(H1)*state.rho + screened_rates(k_sc46__p_ca45);
    jac.set(H1, Sc46, scratch);

    scratch = screened_rates(k_he4_sc47__p_ti50)*Y(He4)*state.rho + screened_rates(k_n_sc47__p_ca47)*Y(N)*state.rho - screened_rates(k_p_sc47__he4_ca44)*Y(H1)*state.rho - screened_rates(k_p_sc47__n_ti47)*Y(H1)*state.rho - screened_rates(k_p_sc47__ti48)*Y(H1)*state.rho + screened_rates(k_sc47__p_ca46);
    jac.set(H1, Sc47, scratch);

    scratch = screened_rates(k_he4_sc48__p_ti51)*Y(He4)*state.rho + screened_rates(k_n_sc48__p_ca48)*Y(N)*state.rho - screened_rates(k_p_sc48__he4_ca45)*Y(H1)*state.rho - screened_rates(k_p_sc48__n_ti48)*Y(H1)*state.rho - screened_rates(k_p_sc48__ti49)*Y(H1)*state.rho + screened_rates(k_sc48__p_ca47);
    jac.set(H1, Sc48, scratch);

    scratch = -screened_rates(k_p_sc49__he4_ca46)*Y(H1)*state.rho - screened_rates(k_p_sc49__n_ti49)*Y(H1)*state.rho - screened_rates(k_p_sc49__ti50)*Y(H1)*state.rho + screened_rates(k_sc49__p_ca48);
    jac.set(H1, Sc49, scratch);

    scratch = screened_rates(k_he4_ti44__p_v47)*Y(He4)*state.rho + screened_rates(k_n_ti44__p_sc44)*Y(N)*state.rho + screened_rates(k_ti44__p_sc43);
    jac.set(H1, Ti44, scratch);

    scratch = screened_rates(k_he4_ti45__p_v48)*Y(He4)*state.rho + screened_rates(k_n_ti45__p_sc45)*Y(N)*state.rho - screened_rates(k_p_ti45__v46)*Y(H1)*state.rho + screened_rates(k_ti45__p_sc44);
    jac.set(H1, Ti45, scratch);

    scratch = screened_rates(k_he4_ti46__p_v49)*Y(He4)*state.rho + screened_rates(k_n_ti46__p_sc46)*Y(N)*state.rho - screened_rates(k_p_ti46__he4_sc43)*Y(H1)*state.rho - screened_rates(k_p_ti46__n_v46)*Y(H1)*state.rho - screened_rates(k_p_ti46__v47)*Y(H1)*state.rho + screened_rates(k_ti46__p_sc45);
    jac.set(H1, Ti46, scratch);

    scratch = screened_rates(k_he4_ti47__p_v50)*Y(He4)*state.rho + screened_rates(k_n_ti47__p_sc47)*Y(N)*state.rho - screened_rates(k_p_ti47__he4_sc44)*Y(H1)*state.rho - screened_rates(k_p_ti47__n_v47)*Y(H1)*state.rho - screened_rates(k_p_ti47__v48)*Y(H1)*state.rho + screened_rates(k_ti47__p_sc46);
    jac.set(H1, Ti47, scratch);

    scratch = screened_rates(k_he4_ti48__p_v51)*Y(He4)*state.rho + screened_rates(k_n_ti48__p_sc48)*Y(N)*state.rho - screened_rates(k_p_ti48__he4_sc45)*Y(H1)*state.rho - screened_rates(k_p_ti48__n_v48)*Y(H1)*state.rho - screened_rates(k_p_ti48__v49)*Y(H1)*state.rho + screened_rates(k_ti48__p_sc47);
    jac.set(H1, Ti48, scratch);

    scratch = screened_rates(k_he4_ti49__p_v52)*Y(He4)*state.rho + screened_rates(k_n_ti49__p_sc49)*Y(N)*state.rho - screened_rates(k_p_ti49__he4_sc46)*Y(H1)*state.rho - screened_rates(k_p_ti49__n_v49)*Y(H1)*state.rho - screened_rates(k_p_ti49__v50)*Y(H1)*state.rho + screened_rates(k_ti49__p_sc48);
    jac.set(H1, Ti49, scratch);

    scratch = -screened_rates(k_p_ti50__he4_sc47)*Y(H1)*state.rho - screened_rates(k_p_ti50__n_v50)*Y(H1)*state.rho - screened_rates(k_p_ti50__v51)*Y(H1)*state.rho + screened_rates(k_ti50__p_sc49);
    jac.set(H1, Ti50, scratch);

    scratch = -screened_rates(k_p_ti51__he4_sc48)*Y(H1)*state.rho - screened_rates(k_p_ti51__n_v51)*Y(H1)*state.rho - screened_rates(k_p_ti51__v52)*Y(H1)*state.rho;
    jac.set(H1, Ti51, scratch);

    scratch = screened_rates(k_he4_v46__p_cr49)*Y(He4)*state.rho + screened_rates(k_n_v46__p_ti46)*Y(N)*state.rho + screened_rates(k_v46__p_ti45);
    jac.set(H1, V46, scratch);

    scratch = screened_rates(k_he4_v47__p_cr50)*Y(He4)*state.rho + screened_rates(k_n_v47__p_ti47)*Y(N)*state.rho - screened_rates(k_p_v47__cr48)*Y(H1)*state.rho - screened_rates(k_p_v47__he4_ti44)*Y(H1)*state.rho + screened_rates(k_v47__p_ti46);
    jac.set(H1, V47, scratch);

    scratch = screened_rates(k_he4_v48__p_cr51)*Y(He4)*state.rho + screened_rates(k_n_v48__p_ti48)*Y(N)*state.rho - screened_rates(k_p_v48__cr49)*Y(H1)*state.rho - screened_rates(k_p_v48__he4_ti45)*Y(H1)*state.rho - screened_rates(k_p_v48__n_cr48)*Y(H1)*state.rho + screened_rates(k_v48__p_ti47);
    jac.set(H1, V48, scratch);

    scratch = screened_rates(k_he4_v49__p_cr52)*Y(He4)*state.rho + screened_rates(k_n_v49__p_ti49)*Y(N)*state.rho - screened_rates(k_p_v49__cr50)*Y(H1)*state.rho - screened_rates(k_p_v49__he4_ti46)*Y(H1)*state.rho - screened_rates(k_p_v49__n_cr49)*Y(H1)*state.rho + screened_rates(k_v49__p_ti48);
    jac.set(H1, V49, scratch);

    scratch = screened_rates(k_he4_v50__p_cr53)*Y(He4)*state.rho + screened_rates(k_n_v50__p_ti50)*Y(N)*state.rho - screened_rates(k_p_v50__cr51)*Y(H1)*state.rho - screened_rates(k_p_v50__he4_ti47)*Y(H1)*state.rho - screened_rates(k_p_v50__n_cr50)*Y(H1)*state.rho + screened_rates(k_v50__p_ti49);
    jac.set(H1, V50, scratch);

    scratch = screened_rates(k_he4_v51__p_cr54)*Y(He4)*state.rho + screened_rates(k_n_v51__p_ti51)*Y(N)*state.rho - screened_rates(k_p_v51__cr52)*Y(H1)*state.rho - screened_rates(k_p_v51__he4_ti48)*Y(H1)*state.rho - screened_rates(k_p_v51__n_cr51)*Y(H1)*state.rho + screened_rates(k_v51__p_ti50);
    jac.set(H1, V51, scratch);

    scratch = -screened_rates(k_p_v52__cr53)*Y(H1)*state.rho - screened_rates(k_p_v52__he4_ti49)*Y(H1)*state.rho - screened_rates(k_p_v52__n_cr52)*Y(H1)*state.rho + screened_rates(k_v52__p_ti51);
    jac.set(H1, V52, scratch);

    scratch = screened_rates(k_cr48__p_v47) + screened_rates(k_he4_cr48__p_mn51)*Y(He4)*state.rho + screened_rates(k_n_cr48__p_v48)*Y(N)*state.rho;
    jac.set(H1, Cr48, scratch);

    scratch = screened_rates(k_cr49__p_v48) + screened_rates(k_he4_cr49__p_mn52)*Y(He4)*state.rho + screened_rates(k_n_cr49__p_v49)*Y(N)*state.rho - screened_rates(k_p_cr49__he4_v46)*Y(H1)*state.rho - screened_rates(k_p_cr49__mn50)*Y(H1)*state.rho;
    jac.set(H1, Cr49, scratch);

    scratch = screened_rates(k_cr50__p_v49) + screened_rates(k_he4_cr50__p_mn53)*Y(He4)*state.rho + screened_rates(k_n_cr50__p_v50)*Y(N)*state.rho - screened_rates(k_p_cr50__he4_v47)*Y(H1)*state.rho - screened_rates(k_p_cr50__mn51)*Y(H1)*state.rho - screened_rates(k_p_cr50__n_mn50)*Y(H1)*state.rho;
    jac.set(H1, Cr50, scratch);

    scratch = screened_rates(k_cr51__p_v50) + screened_rates(k_he4_cr51__p_mn54)*Y(He4)*state.rho + screened_rates(k_n_cr51__p_v51)*Y(N)*state.rho - screened_rates(k_p_cr51__he4_v48)*Y(H1)*state.rho - screened_rates(k_p_cr51__mn52)*Y(H1)*state.rho - screened_rates(k_p_cr51__n_mn51)*Y(H1)*state.rho;
    jac.set(H1, Cr51, scratch);

    scratch = screened_rates(k_cr52__p_v51) + screened_rates(k_he4_cr52__p_mn55)*Y(He4)*state.rho + screened_rates(k_n_cr52__p_v52)*Y(N)*state.rho - screened_rates(k_p_cr52__he4_v49)*Y(H1)*state.rho - screened_rates(k_p_cr52__mn53)*Y(H1)*state.rho - screened_rates(k_p_cr52__n_mn52)*Y(H1)*state.rho;
    jac.set(H1, Cr52, scratch);

    scratch = screened_rates(k_cr53__p_v52) - screened_rates(k_p_cr53__he4_v50)*Y(H1)*state.rho - screened_rates(k_p_cr53__mn54)*Y(H1)*state.rho - screened_rates(k_p_cr53__n_mn53)*Y(H1)*state.rho;
    jac.set(H1, Cr53, scratch);

    scratch = -screened_rates(k_p_cr54__he4_v51)*Y(H1)*state.rho - screened_rates(k_p_cr54__mn55)*Y(H1)*state.rho - screened_rates(k_p_cr54__n_mn54)*Y(H1)*state.rho;
    jac.set(H1, Cr54, scratch);

    scratch = screened_rates(k_he4_mn50__p_fe53)*Y(He4)*state.rho + screened_rates(k_mn50__p_cr49) + screened_rates(k_n_mn50__p_cr50)*Y(N)*state.rho;
    jac.set(H1, Mn50, scratch);

    scratch = screened_rates(k_he4_mn51__p_fe54)*Y(He4)*state.rho + screened_rates(k_mn51__p_cr50) + screened_rates(k_n_mn51__p_cr51)*Y(N)*state.rho - screened_rates(k_p_mn51__fe52)*Y(H1)*state.rho - screened_rates(k_p_mn51__he4_cr48)*Y(H1)*state.rho;
    jac.set(H1, Mn51, scratch);

    scratch = screened_rates(k_he4_mn52__p_fe55)*Y(He4)*state.rho + screened_rates(k_mn52__p_cr51) + screened_rates(k_n_mn52__p_cr52)*Y(N)*state.rho - screened_rates(k_p_mn52__fe53)*Y(H1)*state.rho - screened_rates(k_p_mn52__he4_cr49)*Y(H1)*state.rho - screened_rates(k_p_mn52__n_fe52)*Y(H1)*state.rho;
    jac.set(H1, Mn52, scratch);

    scratch = screened_rates(k_he4_mn53__p_fe56)*Y(He4)*state.rho + screened_rates(k_mn53__p_cr52) + screened_rates(k_n_mn53__p_cr53)*Y(N)*state.rho - screened_rates(k_p_mn53__fe54)*Y(H1)*state.rho - screened_rates(k_p_mn53__he4_cr50)*Y(H1)*state.rho - screened_rates(k_p_mn53__n_fe53)*Y(H1)*state.rho;
    jac.set(H1, Mn53, scratch);

    scratch = screened_rates(k_he4_mn54__p_fe57)*Y(He4)*state.rho + screened_rates(k_mn54__p_cr53) + screened_rates(k_n_mn54__p_cr54)*Y(N)*state.rho - screened_rates(k_p_mn54__fe55)*Y(H1)*state.rho - screened_rates(k_p_mn54__he4_cr51)*Y(H1)*state.rho - screened_rates(k_p_mn54__n_fe54)*Y(H1)*state.rho;
    jac.set(H1, Mn54, scratch);

    scratch = screened_rates(k_he4_mn55__p_fe58)*Y(He4)*state.rho + screened_rates(k_mn55__p_cr54) - screened_rates(k_p_mn55__fe56)*Y(H1)*state.rho - screened_rates(k_p_mn55__he4_cr52)*Y(H1)*state.rho - screened_rates(k_p_mn55__n_fe55)*Y(H1)*state.rho;
    jac.set(H1, Mn55, scratch);

    scratch = screened_rates(k_fe52__p_mn51) + screened_rates(k_he4_fe52__p_co55)*Y(He4)*state.rho + screened_rates(k_n_fe52__p_mn52)*Y(N)*state.rho - screened_rates(k_p_fe52__co53)*Y(H1)*state.rho;
    jac.set(H1, Fe52, scratch);

    scratch = screened_rates(k_fe53__p_mn52) + screened_rates(k_he4_fe53__p_co56)*Y(He4)*state.rho + screened_rates(k_n_fe53__p_mn53)*Y(N)*state.rho - screened_rates(k_p_fe53__co54)*Y(H1)*state.rho - screened_rates(k_p_fe53__he4_mn50)*Y(H1)*state.rho - screened_rates(k_p_fe53__n_co53)*Y(H1)*state.rho;
    jac.set(H1, Fe53, scratch);

    scratch = screened_rates(k_fe54__p_mn53) + screened_rates(k_he4_fe54__p_co57)*Y(He4)*state.rho + screened_rates(k_n_fe54__p_mn54)*Y(N)*state.rho - screened_rates(k_p_fe54__co55)*Y(H1)*state.rho - screened_rates(k_p_fe54__he4_mn51)*Y(H1)*state.rho - screened_rates(k_p_fe54__n_co54)*Y(H1)*state.rho;
    jac.set(H1, Fe54, scratch);

    scratch = screened_rates(k_fe55__p_mn54) + screened_rates(k_he4_fe55__p_co58)*Y(He4)*state.rho + screened_rates(k_n_fe55__p_mn55)*Y(N)*state.rho - screened_rates(k_p_fe55__co56)*Y(H1)*state.rho - screened_rates(k_p_fe55__he4_mn52)*Y(H1)*state.rho - screened_rates(k_p_fe55__n_co55)*Y(H1)*state.rho;
    jac.set(H1, Fe55, scratch);

    scratch = screened_rates(k_fe56__p_mn55) + screened_rates(k_he4_fe56__p_co59)*Y(He4)*state.rho - screened_rates(k_p_fe56__co57)*Y(H1)*state.rho - screened_rates(k_p_fe56__he4_mn53)*Y(H1)*state.rho - screened_rates(k_p_fe56__n_co56)*Y(H1)*state.rho;
    jac.set(H1, Fe56, scratch);

    scratch = -screened_rates(k_p_fe57__co58)*Y(H1)*state.rho - screened_rates(k_p_fe57__he4_mn54)*Y(H1)*state.rho - screened_rates(k_p_fe57__n_co57)*Y(H1)*state.rho;
    jac.set(H1, Fe57, scratch);

    scratch = -screened_rates(k_p_fe58__co59)*Y(H1)*state.rho - screened_rates(k_p_fe58__he4_mn55)*Y(H1)*state.rho - screened_rates(k_p_fe58__n_co58)*Y(H1)*state.rho;
    jac.set(H1, Fe58, scratch);

    scratch = screened_rates(k_co53__p_fe52) + screened_rates(k_he4_co53__p_ni56)*Y(He4)*state.rho + screened_rates(k_n_co53__p_fe53)*Y(N)*state.rho;
    jac.set(H1, Co53, scratch);

    scratch = screened_rates(k_co54__p_fe53) + screened_rates(k_he4_co54__p_ni57)*Y(He4)*state.rho + screened_rates(k_n_co54__p_fe54)*Y(N)*state.rho;
    jac.set(H1, Co54, scratch);

    scratch = screened_rates(k_co55__p_fe54) + screened_rates(k_he4_co55__p_ni58)*Y(He4)*state.rho + screened_rates(k_n_co55__p_fe55)*Y(N)*state.rho - screened_rates(k_p_co55__he4_fe52)*Y(H1)*state.rho - screened_rates(k_p_co55__ni56)*Y(H1)*state.rho;
    jac.set(H1, Co55, scratch);

    scratch = screened_rates(k_co56__p_fe55) + screened_rates(k_he4_co56__p_ni59)*Y(He4)*state.rho + screened_rates(k_n_co56__p_fe56)*Y(N)*state.rho - screened_rates(k_p_co56__he4_fe53)*Y(H1)*state.rho - screened_rates(k_p_co56__n_ni56)*Y(H1)*state.rho - screened_rates(k_p_co56__ni57)*Y(H1)*state.rho;
    jac.set(H1, Co56, scratch);

    scratch = screened_rates(k_co57__p_fe56) + screened_rates(k_he4_co57__p_ni60)*Y(He4)*state.rho + screened_rates(k_n_co57__p_fe57)*Y(N)*state.rho - screened_rates(k_p_co57__he4_fe54)*Y(H1)*state.rho - screened_rates(k_p_co57__n_ni57)*Y(H1)*state.rho - screened_rates(k_p_co57__ni58)*Y(H1)*state.rho;
    jac.set(H1, Co57, scratch);

    scratch = screened_rates(k_co58__p_fe57) + screened_rates(k_he4_co58__p_ni61)*Y(He4)*state.rho + screened_rates(k_n_co58__p_fe58)*Y(N)*state.rho - screened_rates(k_p_co58__he4_fe55)*Y(H1)*state.rho - screened_rates(k_p_co58__n_ni58)*Y(H1)*state.rho - screened_rates(k_p_co58__ni59)*Y(H1)*state.rho;
    jac.set(H1, Co58, scratch);

    scratch = screened_rates(k_co59__p_fe58) + screened_rates(k_he4_co59__p_ni62)*Y(He4)*state.rho - screened_rates(k_p_co59__he4_fe56)*Y(H1)*state.rho - screened_rates(k_p_co59__n_ni59)*Y(H1)*state.rho - screened_rates(k_p_co59__ni60)*Y(H1)*state.rho;
    jac.set(H1, Co59, scratch);

    scratch = screened_rates(k_he4_ni56__p_cu59)*Y(He4)*state.rho + screened_rates(k_n_ni56__p_co56)*Y(N)*state.rho + screened_rates(k_ni56__p_co55) - screened_rates(k_p_ni56__cu57)*Y(H1)*state.rho - screened_rates(k_p_ni56__he4_co53)*Y(H1)*state.rho;
    jac.set(H1, Ni56, scratch);

    scratch = screened_rates(k_he4_ni57__p_cu60)*Y(He4)*state.rho + screened_rates(k_n_ni57__p_co57)*Y(N)*state.rho + screened_rates(k_ni57__p_co56) - screened_rates(k_p_ni57__cu58)*Y(H1)*state.rho - screened_rates(k_p_ni57__he4_co54)*Y(H1)*state.rho - screened_rates(k_p_ni57__n_cu57)*Y(H1)*state.rho;
    jac.set(H1, Ni57, scratch);

    scratch = screened_rates(k_he4_ni58__p_cu61)*Y(He4)*state.rho + screened_rates(k_n_ni58__p_co58)*Y(N)*state.rho + screened_rates(k_ni58__p_co57) - screened_rates(k_p_ni58__cu59)*Y(H1)*state.rho - screened_rates(k_p_ni58__he4_co55)*Y(H1)*state.rho - screened_rates(k_p_ni58__n_cu58)*Y(H1)*state.rho;
    jac.set(H1, Ni58, scratch);

    scratch = screened_rates(k_he4_ni59__p_cu62)*Y(He4)*state.rho + screened_rates(k_n_ni59__p_co59)*Y(N)*state.rho + screened_rates(k_ni59__p_co58) - screened_rates(k_p_ni59__cu60)*Y(H1)*state.rho - screened_rates(k_p_ni59__he4_co56)*Y(H1)*state.rho - screened_rates(k_p_ni59__n_cu59)*Y(H1)*state.rho;
    jac.set(H1, Ni59, scratch);

    scratch = screened_rates(k_he4_ni60__p_cu63)*Y(He4)*state.rho + screened_rates(k_ni60__p_co59) - screened_rates(k_p_ni60__cu61)*Y(H1)*state.rho - screened_rates(k_p_ni60__he4_co57)*Y(H1)*state.rho - screened_rates(k_p_ni60__n_cu60)*Y(H1)*state.rho;
    jac.set(H1, Ni60, scratch);

    scratch = screened_rates(k_he4_ni61__p_cu64)*Y(He4)*state.rho - screened_rates(k_p_ni61__cu62)*Y(H1)*state.rho - screened_rates(k_p_ni61__he4_co58)*Y(H1)*state.rho - screened_rates(k_p_ni61__n_cu61)*Y(H1)*state.rho;
    jac.set(H1, Ni61, scratch);

    scratch = screened_rates(k_he4_ni62__p_cu65)*Y(He4)*state.rho - screened_rates(k_p_ni62__cu63)*Y(H1)*state.rho - screened_rates(k_p_ni62__he4_co59)*Y(H1)*state.rho - screened_rates(k_p_ni62__n_cu62)*Y(H1)*state.rho;
    jac.set(H1, Ni62, scratch);

    scratch = -screened_rates(k_p_ni63__cu64)*Y(H1)*state.rho - screened_rates(k_p_ni63__n_cu63)*Y(H1)*state.rho;
    jac.set(H1, Ni63, scratch);

    scratch = -screened_rates(k_p_ni64__cu65)*Y(H1)*state.rho - screened_rates(k_p_ni64__n_cu64)*Y(H1)*state.rho;
    jac.set(H1, Ni64, scratch);

    scratch = screened_rates(k_cu57__p_ni56) + screened_rates(k_he4_cu57__p_zn60)*Y(He4)*state.rho + screened_rates(k_n_cu57__p_ni57)*Y(N)*state.rho;
    jac.set(H1, Cu57, scratch);

    scratch = screened_rates(k_cu58__p_ni57) + screened_rates(k_he4_cu58__p_zn61)*Y(He4)*state.rho + screened_rates(k_n_cu58__p_ni58)*Y(N)*state.rho - screened_rates(k_p_cu58__zn59)*Y(H1)*state.rho;
    jac.set(H1, Cu58, scratch);

    scratch = screened_rates(k_cu59__p_ni58) + screened_rates(k_he4_cu59__p_zn62)*Y(He4)*state.rho + screened_rates(k_n_cu59__p_ni59)*Y(N)*state.rho - screened_rates(k_p_cu59__he4_ni56)*Y(H1)*state.rho - screened_rates(k_p_cu59__n_zn59)*Y(H1)*state.rho - screened_rates(k_p_cu59__zn60)*Y(H1)*state.rho;
    jac.set(H1, Cu59, scratch);

    scratch = screened_rates(k_cu60__p_ni59) + screened_rates(k_he4_cu60__p_zn63)*Y(He4)*state.rho + screened_rates(k_n_cu60__p_ni60)*Y(N)*state.rho - screened_rates(k_p_cu60__he4_ni57)*Y(H1)*state.rho - screened_rates(k_p_cu60__n_zn60)*Y(H1)*state.rho - screened_rates(k_p_cu60__zn61)*Y(H1)*state.rho;
    jac.set(H1, Cu60, scratch);

    scratch = screened_rates(k_cu61__p_ni60) + screened_rates(k_he4_cu61__p_zn64)*Y(He4)*state.rho + screened_rates(k_n_cu61__p_ni61)*Y(N)*state.rho - screened_rates(k_p_cu61__he4_ni58)*Y(H1)*state.rho - screened_rates(k_p_cu61__n_zn61)*Y(H1)*state.rho - screened_rates(k_p_cu61__zn62)*Y(H1)*state.rho;
    jac.set(H1, Cu61, scratch);

    scratch = screened_rates(k_cu62__p_ni61) + screened_rates(k_he4_cu62__p_zn65)*Y(He4)*state.rho + screened_rates(k_n_cu62__p_ni62)*Y(N)*state.rho - screened_rates(k_p_cu62__he4_ni59)*Y(H1)*state.rho - screened_rates(k_p_cu62__n_zn62)*Y(H1)*state.rho - screened_rates(k_p_cu62__zn63)*Y(H1)*state.rho;
    jac.set(H1, Cu62, scratch);

    scratch = screened_rates(k_cu63__p_ni62) + screened_rates(k_he4_cu63__p_zn66)*Y(He4)*state.rho + screened_rates(k_n_cu63__p_ni63)*Y(N)*state.rho - screened_rates(k_p_cu63__he4_ni60)*Y(H1)*state.rho - screened_rates(k_p_cu63__n_zn63)*Y(H1)*state.rho - screened_rates(k_p_cu63__zn64)*Y(H1)*state.rho;
    jac.set(H1, Cu63, scratch);

    scratch = screened_rates(k_cu64__p_ni63) + screened_rates(k_n_cu64__p_ni64)*Y(N)*state.rho - screened_rates(k_p_cu64__he4_ni61)*Y(H1)*state.rho - screened_rates(k_p_cu64__n_zn64)*Y(H1)*state.rho - screened_rates(k_p_cu64__zn65)*Y(H1)*state.rho;
    jac.set(H1, Cu64, scratch);

    scratch = screened_rates(k_cu65__p_ni64) - screened_rates(k_p_cu65__he4_ni62)*Y(H1)*state.rho - screened_rates(k_p_cu65__n_zn65)*Y(H1)*state.rho - screened_rates(k_p_cu65__zn66)*Y(H1)*state.rho;
    jac.set(H1, Cu65, scratch);

    scratch = screened_rates(k_he4_zn59__p_ga62)*Y(He4)*state.rho + screened_rates(k_n_zn59__p_cu59)*Y(N)*state.rho + screened_rates(k_zn59__p_cu58) + screened_rates(k_zn59__p_ni58__weak__wc12);
    jac.set(H1, Zn59, scratch);

    scratch = screened_rates(k_he4_zn60__p_ga63)*Y(He4)*state.rho + screened_rates(k_n_zn60__p_cu60)*Y(N)*state.rho - screened_rates(k_p_zn60__he4_cu57)*Y(H1)*state.rho + screened_rates(k_zn60__p_cu59);
    jac.set(H1, Zn60, scratch);

    scratch = screened_rates(k_he4_zn61__p_ga64)*Y(He4)*state.rho + screened_rates(k_n_zn61__p_cu61)*Y(N)*state.rho - screened_rates(k_p_zn61__ga62)*Y(H1)*state.rho - screened_rates(k_p_zn61__he4_cu58)*Y(H1)*state.rho + screened_rates(k_zn61__p_cu60);
    jac.set(H1, Zn61, scratch);

    scratch = screened_rates(k_n_zn62__p_cu62)*Y(N)*state.rho - screened_rates(k_p_zn62__ga63)*Y(H1)*state.rho - screened_rates(k_p_zn62__he4_cu59)*Y(H1)*state.rho - screened_rates(k_p_zn62__n_ga62)*Y(H1)*state.rho + screened_rates(k_zn62__p_cu61);
    jac.set(H1, Zn62, scratch);

    scratch = screened_rates(k_n_zn63__p_cu63)*Y(N)*state.rho - screened_rates(k_p_zn63__ga64)*Y(H1)*state.rho - screened_rates(k_p_zn63__he4_cu60)*Y(H1)*state.rho - screened_rates(k_p_zn63__n_ga63)*Y(H1)*state.rho + screened_rates(k_zn63__p_cu62);
    jac.set(H1, Zn63, scratch);

    scratch = screened_rates(k_n_zn64__p_cu64)*Y(N)*state.rho - screened_rates(k_p_zn64__he4_cu61)*Y(H1)*state.rho - screened_rates(k_p_zn64__n_ga64)*Y(H1)*state.rho + screened_rates(k_zn64__p_cu63);
    jac.set(H1, Zn64, scratch);

    scratch = screened_rates(k_n_zn65__p_cu65)*Y(N)*state.rho - screened_rates(k_p_zn65__he4_cu62)*Y(H1)*state.rho + screened_rates(k_zn65__p_cu64);
    jac.set(H1, Zn65, scratch);

    scratch = -screened_rates(k_p_zn66__he4_cu63)*Y(H1)*state.rho + screened_rates(k_zn66__p_cu65);
    jac.set(H1, Zn66, scratch);

    scratch = screened_rates(k_ga62__p_zn61) + screened_rates(k_n_ga62__p_zn62)*Y(N)*state.rho - screened_rates(k_p_ga62__ge63)*Y(H1)*state.rho - screened_rates(k_p_ga62__he4_zn59)*Y(H1)*state.rho;
    jac.set(H1, Ga62, scratch);

    scratch = screened_rates(k_ga63__p_zn62) + screened_rates(k_n_ga63__p_zn63)*Y(N)*state.rho - screened_rates(k_p_ga63__ge64)*Y(H1)*state.rho - screened_rates(k_p_ga63__he4_zn60)*Y(H1)*state.rho - screened_rates(k_p_ga63__n_ge63)*Y(H1)*state.rho;
    jac.set(H1, Ga63, scratch);

    scratch = screened_rates(k_ga64__p_zn63) + screened_rates(k_n_ga64__p_zn64)*Y(N)*state.rho - screened_rates(k_p_ga64__he4_zn61)*Y(H1)*state.rho - screened_rates(k_p_ga64__n_ge64)*Y(H1)*state.rho;
    jac.set(H1, Ga64, scratch);

    scratch = screened_rates(k_ge63__p_ga62) + screened_rates(k_n_ge63__p_ga63)*Y(N)*state.rho;
    jac.set(H1, Ge63, scratch);

    scratch = screened_rates(k_ge64__p_ga63) + screened_rates(k_n_ge64__p_ga64)*Y(N)*state.rho;
    jac.set(H1, Ge64, scratch);

    scratch = screened_rates(k_n_be7__d_li6)*Y(Be7)*state.rho + 2.0*screened_rates(k_n_he3__d_d)*Y(He3)*state.rho + 0.5*screened_rates(k_n_he4_he4__d_li7)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) + screened_rates(k_n_n14__d_c13)*Y(N14)*state.rho + screened_rates(k_n_n15__d_c14)*Y(N15)*state.rho + screened_rates(k_n_p__d)*Y(H1)*state.rho + 0.5*screened_rates(k_n_p_p__p_d)*std::pow(Y(H1), 2)*std::pow(state.rho, 2);
    jac.set(H2, N, scratch);

    scratch = screened_rates(k_n_p__d)*Y(N)*state.rho + 1.0*screened_rates(k_n_p_p__p_d)*Y(N)*Y(H1)*std::pow(state.rho, 2) + screened_rates(k_p_be9__d_he4_he4)*Y(Be9)*state.rho - screened_rates(k_p_d__he3)*Y(H2)*state.rho - screened_rates(k_p_d__n_p_p)*Y(H2)*state.rho + screened_rates(k_p_he4__d_he3)*Y(He4)*state.rho + 0.5*screened_rates(k_p_he4_he4__d_be7)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) + screened_rates(k_p_li7__d_li6)*Y(Li7)*state.rho + 1.0*screened_rates(k_p_p__d__weak__bet_pos_)*Y(H1)*state.rho + 1.0*screened_rates(k_p_p__d__weak__electron_capture)*Y(H1)*std::pow(state.rho, 2)*state.y_e;
    jac.set(H2, H1, scratch);

    scratch = -screened_rates(k_d__n_p) - screened_rates(k_d_be7__p_he4_he4)*Y(Be7)*state.rho - screened_rates(k_d_c13__n_n14)*Y(C13)*state.rho - screened_rates(k_d_c14__n_n15)*Y(C14)*state.rho - 2.0*screened_rates(k_d_d__he4)*Y(H2)*state.rho - 2.0*screened_rates(k_d_d__n_he3)*Y(H2)*state.rho - screened_rates(k_d_he3__p_he4)*Y(He3)*state.rho - 0.5*screened_rates(k_d_he4_he4__p_be9)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) - screened_rates(k_d_li6__n_be7)*Y(Li6)*state.rho - screened_rates(k_d_li6__p_li7)*Y(Li6)*state.rho - screened_rates(k_d_li7__n_he4_he4)*Y(Li7)*state.rho - screened_rates(k_he4_d__li6)*Y(He4)*state.rho - screened_rates(k_p_d__he3)*Y(H1)*state.rho - screened_rates(k_p_d__n_p_p)*Y(H1)*state.rho;
    jac.set(H2, H2, scratch);

    scratch = -screened_rates(k_d_he3__p_he4)*Y(H2)*state.rho + screened_rates(k_he3__p_d) + 2.0*screened_rates(k_n_he3__d_d)*Y(N)*state.rho;
    jac.set(H2, He3, scratch);

    scratch = -1.0*screened_rates(k_d_he4_he4__p_be9)*Y(H2)*Y(He4)*std::pow(state.rho, 2) + 2.0*screened_rates(k_he4__d_d) - screened_rates(k_he4_d__li6)*Y(H2)*state.rho + 1.0*screened_rates(k_n_he4_he4__d_li7)*Y(He4)*Y(N)*std::pow(state.rho, 2) + screened_rates(k_p_he4__d_he3)*Y(H1)*state.rho + 1.0*screened_rates(k_p_he4_he4__d_be7)*Y(He4)*Y(H1)*std::pow(state.rho, 2);
    jac.set(H2, He4, scratch);

    scratch = -screened_rates(k_d_li6__n_be7)*Y(H2)*state.rho - screened_rates(k_d_li6__p_li7)*Y(H2)*state.rho + screened_rates(k_li6__he4_d);
    jac.set(H2, Li6, scratch);

    scratch = -screened_rates(k_d_li7__n_he4_he4)*Y(H2)*state.rho + screened_rates(k_p_li7__d_li6)*Y(H1)*state.rho;
    jac.set(H2, Li7, scratch);

    scratch = -screened_rates(k_d_be7__p_he4_he4)*Y(H2)*state.rho + screened_rates(k_n_be7__d_li6)*Y(N)*state.rho;
    jac.set(H2, Be7, scratch);

    scratch = screened_rates(k_p_be9__d_he4_he4)*Y(H1)*state.rho;
    jac.set(H2, Be9, scratch);

    scratch = -screened_rates(k_d_c13__n_n14)*Y(H2)*state.rho;
    jac.set(H2, C13, scratch);

    scratch = -screened_rates(k_d_c14__n_n15)*Y(H2)*state.rho;
    jac.set(H2, C14, scratch);

    scratch = screened_rates(k_n_n14__d_c13)*Y(N)*state.rho;
    jac.set(H2, N14, scratch);

    scratch = screened_rates(k_n_n15__d_c14)*Y(N)*state.rho;
    jac.set(H2, N15, scratch);

    scratch = -screened_rates(k_n_he3__d_d)*Y(He3)*state.rho - screened_rates(k_n_he3__he4)*Y(He3)*state.rho + 0.5*screened_rates(k_n_p_he4_he4__he3_li7)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 3);
    jac.set(He3, N, scratch);

    scratch = 0.5*screened_rates(k_n_p_he4_he4__he3_li7)*std::pow(Y(He4), 2)*Y(N)*std::pow(state.rho, 3) + screened_rates(k_p_d__he3)*Y(H2)*state.rho - screened_rates(k_p_he3__he4__weak__bet_pos_)*Y(He3)*state.rho + screened_rates(k_p_he4__d_he3)*Y(He4)*state.rho + screened_rates(k_p_li6__he4_he3)*Y(Li6)*state.rho + 2.0*screened_rates(k_p_p_he4__he3_he3)*Y(He4)*Y(H1)*std::pow(state.rho, 2) + 0.5*screened_rates(k_p_p_he4_he4__he3_be7)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 3);
    jac.set(He3, H1, scratch);

    scratch = 1.0*screened_rates(k_d_d__n_he3)*Y(H2)*state.rho - screened_rates(k_d_he3__p_he4)*Y(He3)*state.rho + screened_rates(k_p_d__he3)*Y(H1)*state.rho;
    jac.set(He3, H2, scratch);

    scratch = -screened_rates(k_d_he3__p_he4)*Y(H2)*state.rho - screened_rates(k_he3__p_d) - screened_rates(k_he3_be7__p_p_he4_he4)*Y(Be7)*state.rho - 2.0*screened_rates(k_he3_he3__p_p_he4)*Y(He3)*state.rho - screened_rates(k_he3_li7__n_p_he4_he4)*Y(Li7)*state.rho - screened_rates(k_he4_he3__be7)*Y(He4)*state.rho - screened_rates(k_he4_he3__p_li6)*Y(He4)*state.rho - screened_rates(k_n_he3__d_d)*Y(N)*state.rho - screened_rates(k_n_he3__he4)*Y(N)*state.rho - screened_rates(k_p_he3__he4__weak__bet_pos_)*Y(H1)*state.rho;
    jac.set(He3, He3, scratch);

    scratch = screened_rates(k_he4__n_he3) - screened_rates(k_he4_he3__be7)*Y(He3)*state.rho - screened_rates(k_he4_he3__p_li6)*Y(He3)*state.rho + 1.0*screened_rates(k_n_p_he4_he4__he3_li7)*Y(He4)*Y(N)*Y(H1)*std::pow(state.rho, 3) + screened_rates(k_p_he4__d_he3)*Y(H1)*state.rho + screened_rates(k_p_p_he4__he3_he3)*std::pow(Y(H1), 2)*std::pow(state.rho, 2) + 0.5*screened_rates(k_p_p_he4_he4__he3_be7)*Y(He4)*std::pow(Y(H1), 2)*std::pow(state.rho, 3);
    jac.set(He3, He4, scratch);

    scratch = screened_rates(k_p_li6__he4_he3)*Y(H1)*state.rho;
    jac.set(He3, Li6, scratch);

    scratch = -screened_rates(k_he3_li7__n_p_he4_he4)*Y(He3)*state.rho;
    jac.set(He3, Li7, scratch);

    scratch = screened_rates(k_be7__he4_he3) - screened_rates(k_he3_be7__p_p_he4_he4)*Y(He3)*state.rho;
    jac.set(He3, Be7, scratch);

    scratch = screened_rates(k_n_al25__he4_na22)*Y(Al25)*state.rho + screened_rates(k_n_al26__he4_na23)*Y(Al26)*state.rho + screened_rates(k_n_ar36__he4_s33)*Y(Ar36)*state.rho + screened_rates(k_n_ar37__he4_s34)*Y(Ar37)*state.rho + screened_rates(k_n_ar38__he4_s35)*Y(Ar38)*state.rho + screened_rates(k_n_ar39__he4_s36)*Y(Ar39)*state.rho + screened_rates(k_n_b10__he4_li7)*Y(B10)*state.rho + 2.0*screened_rates(k_n_b8__p_he4_he4)*Y(B8)*state.rho + 2.0*screened_rates(k_n_be7__he4_he4)*Y(Be7)*state.rho + screened_rates(k_n_c12__he4_be9)*Y(C12)*state.rho + screened_rates(k_n_ca40__he4_ar37)*Y(Ca40)*state.rho + screened_rates(k_n_ca41__he4_ar38)*Y(Ca41)*state.rho + screened_rates(k_n_ca42__he4_ar39)*Y(Ca42)*state.rho + screened_rates(k_n_ca43__he4_ar40)*Y(Ca43)*state.rho + screened_rates(k_n_cl33__he4_p30)*Y(Cl33)*state.rho + screened_rates(k_n_cl34__he4_p31)*Y(Cl34)*state.rho + screened_rates(k_n_cl35__he4_p32)*Y(Cl35)*state.rho + screened_rates(k_n_cl36__he4_p33)*Y(Cl36)*state.rho + screened_rates(k_n_co53__he4_mn50)*Y(Co53)*state.rho + screened_rates(k_n_co54__he4_mn51)*Y(Co54)*state.rho + screened_rates(k_n_co55__he4_mn52)*Y(Co55)*state.rho + screened_rates(k_n_co56__he4_mn53)*Y(Co56)*state.rho + screened_rates(k_n_co57__he4_mn54)*Y(Co57)*state.rho + screened_rates(k_n_co58__he4_mn55)*Y(Co58)*state.rho + screened_rates(k_n_cr48__he4_ti45)*Y(Cr48)*state.rho + screened_rates(k_n_cr49__he4_ti46)*Y(Cr49)*state.rho + screened_rates(k_n_cr50__he4_ti47)*Y(Cr50)*state.rho + screened_rates(k_n_cr51__he4_ti48)*Y(Cr51)*state.rho + screened_rates(k_n_cr52__he4_ti49)*Y(Cr52)*state.rho + screened_rates(k_n_cr53__he4_ti50)*Y(Cr53)*state.rho + screened_rates(k_n_cr54__he4_ti51)*Y(Cr54)*state.rho + screened_rates(k_n_cu57__he4_co54)*Y(Cu57)*state.rho + screened_rates(k_n_cu58__he4_co55)*Y(Cu58)*state.rho + screened_rates(k_n_cu59__he4_co56)*Y(Cu59)*state.rho + screened_rates(k_n_cu60__he4_co57)*Y(Cu60)*state.rho + screened_rates(k_n_cu61__he4_co58)*Y(Cu61)*state.rho + screened_rates(k_n_cu62__he4_co59)*Y(Cu62)*state.rho + screened_rates(k_n_f17__he4_n14)*Y(F17)*state.rho + screened_rates(k_n_f18__he4_n15)*Y(F18)*state.rho + screened_rates(k_n_fe52__he4_cr49)*Y(Fe52)*state.rho + screened_rates(k_n_fe53__he4_cr50)*Y(Fe53)*state.rho + screened_rates(k_n_fe54__he4_cr51)*Y(Fe54)*state.rho + screened_rates(k_n_fe55__he4_cr52)*Y(Fe55)*state.rho + screened_rates(k_n_fe56__he4_cr53)*Y(Fe56)*state.rho + screened_rates(k_n_fe57__he4_cr54)*Y(Fe57)*state.rho + screened_rates(k_n_ga62__he4_cu59)*Y(Ga62)*state.rho + screened_rates(k_n_ga63__he4_cu60)*Y(Ga63)*state.rho + screened_rates(k_n_ga64__he4_cu61)*Y(Ga64)*state.rho + screened_rates(k_n_ge63__he4_zn60)*Y(Ge63)*state.rho + screened_rates(k_n_ge64__he4_zn61)*Y(Ge64)*state.rho + screened_rates(k_n_he3__he4)*Y(He3)*state.rho - screened_rates(k_n_he4_he4__be9)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) - screened_rates(k_n_he4_he4__d_li7)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) + screened_rates(k_n_k37__he4_cl34)*Y(K37)*state.rho + screened_rates(k_n_k38__he4_cl35)*Y(K38)*state.rho + screened_rates(k_n_k39__he4_cl36)*Y(K39)*state.rho + screened_rates(k_n_k40__he4_cl37)*Y(K40)*state.rho + screened_rates(k_n_mg23__he4_ne20)*Y(Mg23)*state.rho + screened_rates(k_n_mg24__he4_ne21)*Y(Mg24)*state.rho + screened_rates(k_n_mg25__he4_ne22)*Y(Mg25)*state.rho + screened_rates(k_n_mn50__he4_v47)*Y(Mn50)*state.rho + screened_rates(k_n_mn51__he4_v48)*Y(Mn51)*state.rho + screened_rates(k_n_mn52__he4_v49)*Y(Mn52)*state.rho + screened_rates(k_n_mn53__he4_v50)*Y(Mn53)*state.rho + screened_rates(k_n_mn54__he4_v51)*Y(Mn54)*state.rho + screened_rates(k_n_mn55__he4_v52)*Y(Mn55)*state.rho + screened_rates(k_n_n13__he4_b10)*Y(N13)*state.rho + screened_rates(k_n_n14__he4_b11)*Y(N14)*state.rho + screened_rates(k_n_na21__he4_f18)*Y(Na21)*state.rho + screened_rates(k_n_na22__he4_f19)*Y(Na22)*state.rho + screened_rates(k_n_ne18__he4_o15)*Y(Ne18)*state.rho + screened_rates(k_n_ne19__he4_o16)*Y(Ne19)*state.rho + screened_rates(k_n_ne20__he4_o17)*Y(Ne20)*state.rho + screened_rates(k_n_ne21__he4_o18)*Y(Ne21)*state.rho + screened_rates(k_n_ni56__he4_fe53)*Y(Ni56)*state.rho + screened_rates(k_n_ni57__he4_fe54)*Y(Ni57)*state.rho + screened_rates(k_n_ni58__he4_fe55)*Y(Ni58)*state.rho + screened_rates(k_n_ni59__he4_fe56)*Y(Ni59)*state.rho + screened_rates(k_n_ni60__he4_fe57)*Y(Ni60)*state.rho + screened_rates(k_n_ni61__he4_fe58)*Y(Ni61)*state.rho + screened_rates(k_n_o15__he4_c12)*Y(O15)*state.rho + screened_rates(k_n_o16__he4_c13)*Y(O16)*state.rho + screened_rates(k_n_o17__he4_c14)*Y(O17)*state.rho + screened_rates(k_n_p29__he4_al26)*Y(P29)*state.rho + screened_rates(k_n_p30__he4_al27)*Y(P30)*state.rho - screened_rates(k_n_p_he4__li6)*Y(He4)*Y(H1)*std::pow(state.rho, 2) - screened_rates(k_n_p_he4_he4__he3_li7)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 3) - screened_rates(k_n_p_he4_he4__p_be9)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 3) + screened_rates(k_n_s32__he4_si29)*Y(S32)*state.rho + screened_rates(k_n_s33__he4_si30)*Y(S33)*state.rho + screened_rates(k_n_s34__he4_si31)*Y(S34)*state.rho + screened_rates(k_n_s35__he4_si32)*Y(S35)*state.rho + screened_rates(k_n_sc43__he4_k40)*Y(Sc43)*state.rho + screened_rates(k_n_sc44__he4_k41)*Y(Sc44)*state.rho + screened_rates(k_n_si28__he4_mg25)*Y(Si28)*state.rho + screened_rates(k_n_si29__he4_mg26)*Y(Si29)*state.rho + screened_rates(k_n_ti44__he4_ca41)*Y(Ti44)*state.rho + screened_rates(k_n_ti45__he4_ca42)*Y(Ti45)*state.rho + screened_rates(k_n_ti46__he4_ca43)*Y(Ti46)*state.rho + screened_rates(k_n_ti47__he4_ca44)*Y(Ti47)*state.rho + screened_rates(k_n_ti48__he4_ca45)*Y(Ti48)*state.rho + screened_rates(k_n_ti49__he4_ca46)*Y(Ti49)*state.rho + screened_rates(k_n_ti50__he4_ca47)*Y(Ti50)*state.rho + screened_rates(k_n_ti51__he4_ca48)*Y(Ti51)*state.rho + screened_rates(k_n_v46__he4_sc43)*Y(V46)*state.rho + screened_rates(k_n_v47__he4_sc44)*Y(V47)*state.rho + screened_rates(k_n_v48__he4_sc45)*Y(V48)*state.rho + screened_rates(k_n_v49__he4_sc46)*Y(V49)*state.rho + screened_rates(k_n_v50__he4_sc47)*Y(V50)*state.rho + screened_rates(k_n_v51__he4_sc48)*Y(V51)*state.rho + screened_rates(k_n_v52__he4_sc49)*Y(V52)*state.rho + screened_rates(k_n_zn59__he4_ni56)*Y(Zn59)*state.rho + screened_rates(k_n_zn60__he4_ni57)*Y(Zn60)*state.rho + screened_rates(k_n_zn61__he4_ni58)*Y(Zn61)*state.rho + screened_rates(k_n_zn62__he4_ni59)*Y(Zn62)*state.rho + screened_rates(k_n_zn63__he4_ni60)*Y(Zn63)*state.rho + screened_rates(k_n_zn64__he4_ni61)*Y(Zn64)*state.rho + screened_rates(k_n_zn65__he4_ni62)*Y(Zn65)*state.rho + screened_rates(k_n_zn66__he4_ni63)*Y(Zn66)*state.rho;
    jac.set(He4, N, scratch);

    scratch = -screened_rates(k_n_p_he4__li6)*Y(He4)*Y(N)*std::pow(state.rho, 2) - screened_rates(k_n_p_he4_he4__he3_li7)*std::pow(Y(He4), 2)*Y(N)*std::pow(state.rho, 3) - screened_rates(k_n_p_he4_he4__p_be9)*std::pow(Y(He4), 2)*Y(N)*std::pow(state.rho, 3) + screened_rates(k_p_al26__he4_mg23)*Y(Al26)*state.rho + screened_rates(k_p_al27__he4_mg24)*Y(Al27)*state.rho + screened_rates(k_p_ar36__he4_cl33)*Y(Ar36)*state.rho + screened_rates(k_p_ar37__he4_cl34)*Y(Ar37)*state.rho + screened_rates(k_p_ar38__he4_cl35)*Y(Ar38)*state.rho + screened_rates(k_p_ar39__he4_cl36)*Y(Ar39)*state.rho + screened_rates(k_p_ar40__he4_cl37)*Y(Ar40)*state.rho + screened_rates(k_p_b10__he4_be7)*Y(B10)*state.rho + 3.0*screened_rates(k_p_b11__he4_he4_he4)*Y(B11)*state.rho + 2.0*screened_rates(k_p_be9__d_he4_he4)*Y(Be9)*state.rho + screened_rates(k_p_be9__he4_li6)*Y(Be9)*state.rho + 2.0*screened_rates(k_p_be9__n_p_he4_he4)*Y(Be9)*state.rho + screened_rates(k_p_c13__he4_b10)*Y(C13)*state.rho + screened_rates(k_p_c14__he4_b11)*Y(C14)*state.rho + screened_rates(k_p_ca40__he4_k37)*Y(Ca40)*state.rho + screened_rates(k_p_ca41__he4_k38)*Y(Ca41)*state.rho + screened_rates(k_p_ca42__he4_k39)*Y(Ca42)*state.rho + screened_rates(k_p_ca43__he4_k40)*Y(Ca43)*state.rho + screened_rates(k_p_ca44__he4_k41)*Y(Ca44)*state.rho + screened_rates(k_p_cl35__he4_s32)*Y(Cl35)*state.rho + screened_rates(k_p_cl36__he4_s33)*Y(Cl36)*state.rho + screened_rates(k_p_cl37__he4_s34)*Y(Cl37)*state.rho + screened_rates(k_p_co55__he4_fe52)*Y(Co55)*state.rho + screened_rates(k_p_co56__he4_fe53)*Y(Co56)*state.rho + screened_rates(k_p_co57__he4_fe54)*Y(Co57)*state.rho + screened_rates(k_p_co58__he4_fe55)*Y(Co58)*state.rho + screened_rates(k_p_co59__he4_fe56)*Y(Co59)*state.rho + screened_rates(k_p_cr49__he4_v46)*Y(Cr49)*state.rho + screened_rates(k_p_cr50__he4_v47)*Y(Cr50)*state.rho + screened_rates(k_p_cr51__he4_v48)*Y(Cr51)*state.rho + screened_rates(k_p_cr52__he4_v49)*Y(Cr52)*state.rho + screened_rates(k_p_cr53__he4_v50)*Y(Cr53)*state.rho + screened_rates(k_p_cr54__he4_v51)*Y(Cr54)*state.rho + screened_rates(k_p_cu59__he4_ni56)*Y(Cu59)*state.rho + screened_rates(k_p_cu60__he4_ni57)*Y(Cu60)*state.rho + screened_rates(k_p_cu61__he4_ni58)*Y(Cu61)*state.rho + screened_rates(k_p_cu62__he4_ni59)*Y(Cu62)*state.rho + screened_rates(k_p_cu63__he4_ni60)*Y(Cu63)*state.rho + screened_rates(k_p_cu64__he4_ni61)*Y(Cu64)*state.rho + screened_rates(k_p_cu65__he4_ni62)*Y(Cu65)*state.rho + screened_rates(k_p_f17__he4_o14)*Y(F17)*state.rho + screened_rates(k_p_f18__he4_o15)*Y(F18)*state.rho + screened_rates(k_p_f19__he4_o16)*Y(F19)*state.rho + screened_rates(k_p_fe53__he4_mn50)*Y(Fe53)*state.rho + screened_rates(k_p_fe54__he4_mn51)*Y(Fe54)*state.rho + screened_rates(k_p_fe55__he4_mn52)*Y(Fe55)*state.rho + screened_rates(k_p_fe56__he4_mn53)*Y(Fe56)*state.rho + screened_rates(k_p_fe57__he4_mn54)*Y(Fe57)*state.rho + screened_rates(k_p_fe58__he4_mn55)*Y(Fe58)*state.rho + screened_rates(k_p_ga62__he4_zn59)*Y(Ga62)*state.rho + screened_rates(k_p_ga63__he4_zn60)*Y(Ga63)*state.rho + screened_rates(k_p_ga64__he4_zn61)*Y(Ga64)*state.rho + screened_rates(k_p_he3__he4__weak__bet_pos_)*Y(He3)*state.rho - screened_rates(k_p_he4__d_he3)*Y(He4)*state.rho - screened_rates(k_p_he4_he4__d_be7)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) - screened_rates(k_p_he4_he4__n_b8)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) + screened_rates(k_p_k39__he4_ar36)*Y(K39)*state.rho + screened_rates(k_p_k40__he4_ar37)*Y(K40)*state.rho + screened_rates(k_p_k41__he4_ar38)*Y(K41)*state.rho + screened_rates(k_p_li6__he4_he3)*Y(Li6)*state.rho + 2.0*screened_rates(k_p_li7__he4_he4)*Y(Li7)*state.rho + screened_rates(k_p_mg24__he4_na21)*Y(Mg24)*state.rho + screened_rates(k_p_mg25__he4_na22)*Y(Mg25)*state.rho + screened_rates(k_p_mg26__he4_na23)*Y(Mg26)*state.rho + screened_rates(k_p_mn51__he4_cr48)*Y(Mn51)*state.rho + screened_rates(k_p_mn52__he4_cr49)*Y(Mn52)*state.rho + screened_rates(k_p_mn53__he4_cr50)*Y(Mn53)*state.rho + screened_rates(k_p_mn54__he4_cr51)*Y(Mn54)*state.rho + screened_rates(k_p_mn55__he4_cr52)*Y(Mn55)*state.rho + screened_rates(k_p_n15__he4_c12)*Y(N15)*state.rho + screened_rates(k_p_na21__he4_ne18)*Y(Na21)*state.rho + screened_rates(k_p_na22__he4_ne19)*Y(Na22)*state.rho + screened_rates(k_p_na23__he4_ne20)*Y(Na23)*state.rho + screened_rates(k_p_ne20__he4_f17)*Y(Ne20)*state.rho + screened_rates(k_p_ne21__he4_f18)*Y(Ne21)*state.rho + screened_rates(k_p_ne22__he4_f19)*Y(Ne22)*state.rho + screened_rates(k_p_ni56__he4_co53)*Y(Ni56)*state.rho + screened_rates(k_p_ni57__he4_co54)*Y(Ni57)*state.rho + screened_rates(k_p_ni58__he4_co55)*Y(Ni58)*state.rho + screened_rates(k_p_ni59__he4_co56)*Y(Ni59)*state.rho + screened_rates(k_p_ni60__he4_co57)*Y(Ni60)*state.rho + screened_rates(k_p_ni61__he4_co58)*Y(Ni61)*state.rho + screened_rates(k_p_ni62__he4_co59)*Y(Ni62)*state.rho + screened_rates(k_p_o16__he4_n13)*Y(O16)*state.rho + screened_rates(k_p_o17__he4_n14)*Y(O17)*state.rho + screened_rates(k_p_o18__he4_n15)*Y(O18)*state.rho + screened_rates(k_p_p31__he4_si28)*Y(P31)*state.rho + screened_rates(k_p_p32__he4_si29)*Y(P32)*state.rho + screened_rates(k_p_p33__he4_si30)*Y(P33)*state.rho - 1.0*screened_rates(k_p_p_he4__he3_he3)*Y(He4)*Y(H1)*std::pow(state.rho, 2) - 1.0*screened_rates(k_p_p_he4_he4__he3_be7)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 3) + screened_rates(k_p_s32__he4_p29)*Y(S32)*state.rho + screened_rates(k_p_s33__he4_p30)*Y(S33)*state.rho + screened_rates(k_p_s34__he4_p31)*Y(S34)*state.rho + screened_rates(k_p_s35__he4_p32)*Y(S35)*state.rho + screened_rates(k_p_s36__he4_p33)*Y(S36)*state.rho + screened_rates(k_p_sc43__he4_ca40)*Y(Sc43)*state.rho + screened_rates(k_p_sc44__he4_ca41)*Y(Sc44)*state.rho + screened_rates(k_p_sc45__he4_ca42)*Y(Sc45)*state.rho + screened_rates(k_p_sc46__he4_ca43)*Y(Sc46)*state.rho + screened_rates(k_p_sc47__he4_ca44)*Y(Sc47)*state.rho + screened_rates(k_p_sc48__he4_ca45)*Y(Sc48)*state.rho + screened_rates(k_p_sc49__he4_ca46)*Y(Sc49)*state.rho + screened_rates(k_p_si28__he4_al25)*Y(Si28)*state.rho + screened_rates(k_p_si29__he4_al26)*Y(Si29)*state.rho + screened_rates(k_p_si30__he4_al27)*Y(Si30)*state.rho + screened_rates(k_p_ti46__he4_sc43)*Y(Ti46)*state.rho + screened_rates(k_p_ti47__he4_sc44)*Y(Ti47)*state.rho + screened_rates(k_p_ti48__he4_sc45)*Y(Ti48)*state.rho + screened_rates(k_p_ti49__he4_sc46)*Y(Ti49)*state.rho + screened_rates(k_p_ti50__he4_sc47)*Y(Ti50)*state.rho + screened_rates(k_p_ti51__he4_sc48)*Y(Ti51)*state.rho + screened_rates(k_p_v47__he4_ti44)*Y(V47)*state.rho + screened_rates(k_p_v48__he4_ti45)*Y(V48)*state.rho + screened_rates(k_p_v49__he4_ti46)*Y(V49)*state.rho + screened_rates(k_p_v50__he4_ti47)*Y(V50)*state.rho + screened_rates(k_p_v51__he4_ti48)*Y(V51)*state.rho + screened_rates(k_p_v52__he4_ti49)*Y(V52)*state.rho + screened_rates(k_p_zn60__he4_cu57)*Y(Zn60)*state.rho + screened_rates(k_p_zn61__he4_cu58)*Y(Zn61)*state.rho + screened_rates(k_p_zn62__he4_cu59)*Y(Zn62)*state.rho + screened_rates(k_p_zn63__he4_cu60)*Y(Zn63)*state.rho + screened_rates(k_p_zn64__he4_cu61)*Y(Zn64)*state.rho + screened_rates(k_p_zn65__he4_cu62)*Y(Zn65)*state.rho + screened_rates(k_p_zn66__he4_cu63)*Y(Zn66)*state.rho;
    jac.set(He4, H1, scratch);

    scratch = 2.0*screened_rates(k_d_be7__p_he4_he4)*Y(Be7)*state.rho + 1.0*screened_rates(k_d_d__he4)*Y(H2)*state.rho + screened_rates(k_d_he3__p_he4)*Y(He3)*state.rho - screened_rates(k_d_he4_he4__p_be9)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) + 2.0*screened_rates(k_d_li7__n_he4_he4)*Y(Li7)*state.rho - screened_rates(k_he4_d__li6)*Y(He4)*state.rho;
    jac.set(He4, H2, scratch);

    scratch = screened_rates(k_d_he3__p_he4)*Y(H2)*state.rho + 2.0*screened_rates(k_he3_be7__p_p_he4_he4)*Y(Be7)*state.rho + 1.0*screened_rates(k_he3_he3__p_p_he4)*Y(He3)*state.rho + 2.0*screened_rates(k_he3_li7__n_p_he4_he4)*Y(Li7)*state.rho - screened_rates(k_he4_he3__be7)*Y(He4)*state.rho - screened_rates(k_he4_he3__p_li6)*Y(He4)*state.rho + screened_rates(k_n_he3__he4)*Y(N)*state.rho + screened_rates(k_p_he3__he4__weak__bet_pos_)*Y(H1)*state.rho;
    jac.set(He4, He3, scratch);

    scratch = -2.0*screened_rates(k_d_he4_he4__p_be9)*Y(H2)*Y(He4)*std::pow(state.rho, 2) - screened_rates(k_he4__d_d) - screened_rates(k_he4__n_he3) - screened_rates(k_he4_al25__p29)*Y(Al25)*state.rho - screened_rates(k_he4_al25__p_si28)*Y(Al25)*state.rho - screened_rates(k_he4_al26__n_p29)*Y(Al26)*state.rho - screened_rates(k_he4_al26__p30)*Y(Al26)*state.rho - screened_rates(k_he4_al26__p_si29)*Y(Al26)*state.rho - screened_rates(k_he4_al27__n_p30)*Y(Al27)*state.rho - screened_rates(k_he4_al27__p31)*Y(Al27)*state.rho - screened_rates(k_he4_al27__p_si30)*Y(Al27)*state.rho - screened_rates(k_he4_ar36__ca40)*Y(Ar36)*state.rho - screened_rates(k_he4_ar36__p_k39)*Y(Ar36)*state.rho - screened_rates(k_he4_ar37__ca41)*Y(Ar37)*state.rho - screened_rates(k_he4_ar37__n_ca40)*Y(Ar37)*state.rho - screened_rates(k_he4_ar37__p_k40)*Y(Ar37)*state.rho - screened_rates(k_he4_ar38__ca42)*Y(Ar38)*state.rho - screened_rates(k_he4_ar38__n_ca41)*Y(Ar38)*state.rho - screened_rates(k_he4_ar38__p_k41)*Y(Ar38)*state.rho - screened_rates(k_he4_ar39__ca43)*Y(Ar39)*state.rho - screened_rates(k_he4_ar39__n_ca42)*Y(Ar39)*state.rho - screened_rates(k_he4_ar40__ca44)*Y(Ar40)*state.rho - screened_rates(k_he4_ar40__n_ca43)*Y(Ar40)*state.rho - screened_rates(k_he4_b10__n_n13)*Y(B10)*state.rho - screened_rates(k_he4_b10__p_c13)*Y(B10)*state.rho - screened_rates(k_he4_b11__n_n14)*Y(B11)*state.rho - screened_rates(k_he4_b11__p_c14)*Y(B11)*state.rho - screened_rates(k_he4_be7__p_b10)*Y(Be7)*state.rho - screened_rates(k_he4_be9__n_c12)*Y(Be9)*state.rho - screened_rates(k_he4_c12__n_o15)*Y(C12)*state.rho - screened_rates(k_he4_c12__o16)*Y(C12)*state.rho - screened_rates(k_he4_c12__p_n15)*Y(C12)*state.rho - screened_rates(k_he4_c13__n_o16)*Y(C13)*state.rho - screened_rates(k_he4_c14__n_o17)*Y(C14)*state.rho - screened_rates(k_he4_c14__o18)*Y(C14)*state.rho - screened_rates(k_he4_ca40__p_sc43)*Y(Ca40)*state.rho - screened_rates(k_he4_ca40__ti44)*Y(Ca40)*state.rho - screened_rates(k_he4_ca41__n_ti44)*Y(Ca41)*state.rho - screened_rates(k_he4_ca41__p_sc44)*Y(Ca41)*state.rho - screened_rates(k_he4_ca41__ti45)*Y(Ca41)*state.rho - screened_rates(k_he4_ca42__n_ti45)*Y(Ca42)*state.rho - screened_rates(k_he4_ca42__p_sc45)*Y(Ca42)*state.rho - screened_rates(k_he4_ca42__ti46)*Y(Ca42)*state.rho - screened_rates(k_he4_ca43__n_ti46)*Y(Ca43)*state.rho - screened_rates(k_he4_ca43__p_sc46)*Y(Ca43)*state.rho - screened_rates(k_he4_ca43__ti47)*Y(Ca43)*state.rho - screened_rates(k_he4_ca44__n_ti47)*Y(Ca44)*state.rho - screened_rates(k_he4_ca44__p_sc47)*Y(Ca44)*state.rho - screened_rates(k_he4_ca44__ti48)*Y(Ca44)*state.rho - screened_rates(k_he4_ca45__n_ti48)*Y(Ca45)*state.rho - screened_rates(k_he4_ca45__p_sc48)*Y(Ca45)*state.rho - screened_rates(k_he4_ca45__ti49)*Y(Ca45)*state.rho - screened_rates(k_he4_ca46__n_ti49)*Y(Ca46)*state.rho - screened_rates(k_he4_ca46__p_sc49)*Y(Ca46)*state.rho - screened_rates(k_he4_ca46__ti50)*Y(Ca46)*state.rho - screened_rates(k_he4_ca47__n_ti50)*Y(Ca47)*state.rho - screened_rates(k_he4_ca47__ti51)*Y(Ca47)*state.rho - screened_rates(k_he4_ca48__n_ti51)*Y(Ca48)*state.rho - screened_rates(k_he4_cl33__k37)*Y(Cl33)*state.rho - screened_rates(k_he4_cl33__p_ar36)*Y(Cl33)*state.rho - screened_rates(k_he4_cl34__k38)*Y(Cl34)*state.rho - screened_rates(k_he4_cl34__n_k37)*Y(Cl34)*state.rho - screened_rates(k_he4_cl34__p_ar37)*Y(Cl34)*state.rho - screened_rates(k_he4_cl35__k39)*Y(Cl35)*state.rho - screened_rates(k_he4_cl35__n_k38)*Y(Cl35)*state.rho - screened_rates(k_he4_cl35__p_ar38)*Y(Cl35)*state.rho - screened_rates(k_he4_cl36__k40)*Y(Cl36)*state.rho - screened_rates(k_he4_cl36__n_k39)*Y(Cl36)*state.rho - screened_rates(k_he4_cl36__p_ar39)*Y(Cl36)*state.rho - screened_rates(k_he4_cl37__k41)*Y(Cl37)*state.rho - screened_rates(k_he4_cl37__n_k40)*Y(Cl37)*state.rho - screened_rates(k_he4_cl37__p_ar40)*Y(Cl37)*state.rho - screened_rates(k_he4_co53__cu57)*Y(Co53)*state.rho - screened_rates(k_he4_co53__p_ni56)*Y(Co53)*state.rho - screened_rates(k_he4_co54__cu58)*Y(Co54)*state.rho - screened_rates(k_he4_co54__n_cu57)*Y(Co54)*state.rho - screened_rates(k_he4_co54__p_ni57)*Y(Co54)*state.rho - screened_rates(k_he4_co55__cu59)*Y(Co55)*state.rho - screened_rates(k_he4_co55__n_cu58)*Y(Co55)*state.rho - screened_rates(k_he4_co55__p_ni58)*Y(Co55)*state.rho - screened_rates(k_he4_co56__cu60)*Y(Co56)*state.rho - screened_rates(k_he4_co56__n_cu59)*Y(Co56)*state.rho - screened_rates(k_he4_co56__p_ni59)*Y(Co56)*state.rho - screened_rates(k_he4_co57__cu61)*Y(Co57)*state.rho - screened_rates(k_he4_co57__n_cu60)*Y(Co57)*state.rho - screened_rates(k_he4_co57__p_ni60)*Y(Co57)*state.rho - screened_rates(k_he4_co58__cu62)*Y(Co58)*state.rho - screened_rates(k_he4_co58__n_cu61)*Y(Co58)*state.rho - screened_rates(k_he4_co58__p_ni61)*Y(Co58)*state.rho - screened_rates(k_he4_co59__cu63)*Y(Co59)*state.rho - screened_rates(k_he4_co59__n_cu62)*Y(Co59)*state.rho - screened_rates(k_he4_co59__p_ni62)*Y(Co59)*state.rho - screened_rates(k_he4_cr48__fe52)*Y(Cr48)*state.rho - screened_rates(k_he4_cr48__p_mn51)*Y(Cr48)*state.rho - screened_rates(k_he4_cr49__fe53)*Y(Cr49)*state.rho - screened_rates(k_he4_cr49__n_fe52)*Y(Cr49)*state.rho - screened_rates(k_he4_cr49__p_mn52)*Y(Cr49)*state.rho - screened_rates(k_he4_cr50__fe54)*Y(Cr50)*state.rho - screened_rates(k_he4_cr50__n_fe53)*Y(Cr50)*state.rho - screened_rates(k_he4_cr50__p_mn53)*Y(Cr50)*state.rho - screened_rates(k_he4_cr51__fe55)*Y(Cr51)*state.rho - screened_rates(k_he4_cr51__n_fe54)*Y(Cr51)*state.rho - screened_rates(k_he4_cr51__p_mn54)*Y(Cr51)*state.rho - screened_rates(k_he4_cr52__fe56)*Y(Cr52)*state.rho - screened_rates(k_he4_cr52__n_fe55)*Y(Cr52)*state.rho - screened_rates(k_he4_cr52__p_mn55)*Y(Cr52)*state.rho - screened_rates(k_he4_cr53__fe57)*Y(Cr53)*state.rho - screened_rates(k_he4_cr53__n_fe56)*Y(Cr53)*state.rho - screened_rates(k_he4_cr54__fe58)*Y(Cr54)*state.rho - screened_rates(k_he4_cr54__n_fe57)*Y(Cr54)*state.rho - screened_rates(k_he4_cu57__p_zn60)*Y(Cu57)*state.rho - screened_rates(k_he4_cu58__ga62)*Y(Cu58)*state.rho - screened_rates(k_he4_cu58__p_zn61)*Y(Cu58)*state.rho - screened_rates(k_he4_cu59__ga63)*Y(Cu59)*state.rho - screened_rates(k_he4_cu59__n_ga62)*Y(Cu59)*state.rho - screened_rates(k_he4_cu59__p_zn62)*Y(Cu59)*state.rho - screened_rates(k_he4_cu60__ga64)*Y(Cu60)*state.rho - screened_rates(k_he4_cu60__n_ga63)*Y(Cu60)*state.rho - screened_rates(k_he4_cu60__p_zn63)*Y(Cu60)*state.rho - screened_rates(k_he4_cu61__n_ga64)*Y(Cu61)*state.rho - screened_rates(k_he4_cu61__p_zn64)*Y(Cu61)*state.rho - screened_rates(k_he4_cu62__p_zn65)*Y(Cu62)*state.rho - screened_rates(k_he4_cu63__p_zn66)*Y(Cu63)*state.rho - screened_rates(k_he4_d__li6)*Y(H2)*state.rho - screened_rates(k_he4_f17__na21)*Y(F17)*state.rho - screened_rates(k_he4_f17__p_ne20)*Y(F17)*state.rho - screened_rates(k_he4_f18__n_na21)*Y(F18)*state.rho - screened_rates(k_he4_f18__na22)*Y(F18)*state.rho - screened_rates(k_he4_f18__p_ne21)*Y(F18)*state.rho - screened_rates(k_he4_f19__n_na22)*Y(F19)*state.rho - screened_rates(k_he4_f19__na23)*Y(F19)*state.rho - screened_rates(k_he4_f19__p_ne22)*Y(F19)*state.rho - screened_rates(k_he4_fe52__ni56)*Y(Fe52)*state.rho - screened_rates(k_he4_fe52__p_co55)*Y(Fe52)*state.rho - screened_rates(k_he4_fe53__n_ni56)*Y(Fe53)*state.rho - screened_rates(k_he4_fe53__ni57)*Y(Fe53)*state.rho - screened_rates(k_he4_fe53__p_co56)*Y(Fe53)*state.rho - screened_rates(k_he4_fe54__n_ni57)*Y(Fe54)*state.rho - screened_rates(k_he4_fe54__ni58)*Y(Fe54)*state.rho - screened_rates(k_he4_fe54__p_co57)*Y(Fe54)*state.rho - screened_rates(k_he4_fe55__n_ni58)*Y(Fe55)*state.rho - screened_rates(k_he4_fe55__ni59)*Y(Fe55)*state.rho - screened_rates(k_he4_fe55__p_co58)*Y(Fe55)*state.rho - screened_rates(k_he4_fe56__n_ni59)*Y(Fe56)*state.rho - screened_rates(k_he4_fe56__ni60)*Y(Fe56)*state.rho - screened_rates(k_he4_fe56__p_co59)*Y(Fe56)*state.rho - screened_rates(k_he4_fe57__n_ni60)*Y(Fe57)*state.rho - screened_rates(k_he4_fe57__ni61)*Y(Fe57)*state.rho - screened_rates(k_he4_fe58__n_ni61)*Y(Fe58)*state.rho - screened_rates(k_he4_fe58__ni62)*Y(Fe58)*state.rho - screened_rates(k_he4_he3__be7)*Y(He3)*state.rho - screened_rates(k_he4_he3__p_li6)*Y(He3)*state.rho - 2.0*screened_rates(k_he4_he4__n_be7)*Y(He4)*state.rho - 2.0*screened_rates(k_he4_he4__p_li7)*Y(He4)*state.rho - 1.5*screened_rates(k_he4_he4_he4__c12)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) - 1.5*screened_rates(k_he4_he4_he4__p_b11)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) - screened_rates(k_he4_k37__p_ca40)*Y(K37)*state.rho - screened_rates(k_he4_k38__p_ca41)*Y(K38)*state.rho - screened_rates(k_he4_k39__p_ca42)*Y(K39)*state.rho - screened_rates(k_he4_k39__sc43)*Y(K39)*state.rho - screened_rates(k_he4_k40__n_sc43)*Y(K40)*state.rho - screened_rates(k_he4_k40__p_ca43)*Y(K40)*state.rho - screened_rates(k_he4_k40__sc44)*Y(K40)*state.rho - screened_rates(k_he4_k41__n_sc44)*Y(K41)*state.rho - screened_rates(k_he4_k41__p_ca44)*Y(K41)*state.rho - screened_rates(k_he4_k41__sc45)*Y(K41)*state.rho - screened_rates(k_he4_li6__b10)*Y(Li6)*state.rho - screened_rates(k_he4_li6__p_be9)*Y(Li6)*state.rho - screened_rates(k_he4_li7__b11)*Y(Li7)*state.rho - screened_rates(k_he4_li7__n_b10)*Y(Li7)*state.rho - screened_rates(k_he4_mg23__p_al26)*Y(Mg23)*state.rho - screened_rates(k_he4_mg24__c12_o16)*Y(Mg24)*state.rho - screened_rates(k_he4_mg24__p_al27)*Y(Mg24)*state.rho - screened_rates(k_he4_mg24__si28)*Y(Mg24)*state.rho - screened_rates(k_he4_mg25__n_si28)*Y(Mg25)*state.rho - screened_rates(k_he4_mg25__si29)*Y(Mg25)*state.rho - screened_rates(k_he4_mg26__n_si29)*Y(Mg26)*state.rho - screened_rates(k_he4_mg26__si30)*Y(Mg26)*state.rho - screened_rates(k_he4_mn50__co54)*Y(Mn50)*state.rho - screened_rates(k_he4_mn50__n_co53)*Y(Mn50)*state.rho - screened_rates(k_he4_mn50__p_fe53)*Y(Mn50)*state.rho - screened_rates(k_he4_mn51__co55)*Y(Mn51)*state.rho - screened_rates(k_he4_mn51__n_co54)*Y(Mn51)*state.rho - screened_rates(k_he4_mn51__p_fe54)*Y(Mn51)*state.rho - screened_rates(k_he4_mn52__co56)*Y(Mn52)*state.rho - screened_rates(k_he4_mn52__n_co55)*Y(Mn52)*state.rho - screened_rates(k_he4_mn52__p_fe55)*Y(Mn52)*state.rho - screened_rates(k_he4_mn53__co57)*Y(Mn53)*state.rho - screened_rates(k_he4_mn53__n_co56)*Y(Mn53)*state.rho - screened_rates(k_he4_mn53__p_fe56)*Y(Mn53)*state.rho - screened_rates(k_he4_mn54__co58)*Y(Mn54)*state.rho - screened_rates(k_he4_mn54__n_co57)*Y(Mn54)*state.rho - screened_rates(k_he4_mn54__p_fe57)*Y(Mn54)*state.rho - screened_rates(k_he4_mn55__co59)*Y(Mn55)*state.rho - screened_rates(k_he4_mn55__n_co58)*Y(Mn55)*state.rho - screened_rates(k_he4_mn55__p_fe58)*Y(Mn55)*state.rho - screened_rates(k_he4_n13__p_o16)*Y(N13)*state.rho - screened_rates(k_he4_n14__f18)*Y(N14)*state.rho - screened_rates(k_he4_n14__n_f17)*Y(N14)*state.rho - screened_rates(k_he4_n14__p_o17)*Y(N14)*state.rho - screened_rates(k_he4_n15__f19)*Y(N15)*state.rho - screened_rates(k_he4_n15__n_f18)*Y(N15)*state.rho - screened_rates(k_he4_n15__p_o18)*Y(N15)*state.rho - screened_rates(k_he4_na21__al25)*Y(Na21)*state.rho - screened_rates(k_he4_na21__p_mg24)*Y(Na21)*state.rho - screened_rates(k_he4_na22__al26)*Y(Na22)*state.rho - screened_rates(k_he4_na22__n_al25)*Y(Na22)*state.rho - screened_rates(k_he4_na22__p_mg25)*Y(Na22)*state.rho - screened_rates(k_he4_na23__al27)*Y(Na23)*state.rho - screened_rates(k_he4_na23__n_al26)*Y(Na23)*state.rho - screened_rates(k_he4_na23__p_mg26)*Y(Na23)*state.rho - screened_rates(k_he4_ne18__p_na21)*Y(Ne18)*state.rho - screened_rates(k_he4_ne19__mg23)*Y(Ne19)*state.rho - screened_rates(k_he4_ne19__p_na22)*Y(Ne19)*state.rho - screened_rates(k_he4_ne20__c12_c12)*Y(Ne20)*state.rho - screened_rates(k_he4_ne20__mg24)*Y(Ne20)*state.rho - screened_rates(k_he4_ne20__n_mg23)*Y(Ne20)*state.rho - screened_rates(k_he4_ne20__p_na23)*Y(Ne20)*state.rho - screened_rates(k_he4_ne21__mg25)*Y(Ne21)*state.rho - screened_rates(k_he4_ne21__n_mg24)*Y(Ne21)*state.rho - screened_rates(k_he4_ne22__mg26)*Y(Ne22)*state.rho - screened_rates(k_he4_ne22__n_mg25)*Y(Ne22)*state.rho - screened_rates(k_he4_ni56__n_zn59)*Y(Ni56)*state.rho - screened_rates(k_he4_ni56__p_cu59)*Y(Ni56)*state.rho - screened_rates(k_he4_ni56__zn60)*Y(Ni56)*state.rho - screened_rates(k_he4_ni57__n_zn60)*Y(Ni57)*state.rho - screened_rates(k_he4_ni57__p_cu60)*Y(Ni57)*state.rho - screened_rates(k_he4_ni57__zn61)*Y(Ni57)*state.rho - screened_rates(k_he4_ni58__n_zn61)*Y(Ni58)*state.rho - screened_rates(k_he4_ni58__p_cu61)*Y(Ni58)*state.rho - screened_rates(k_he4_ni58__zn62)*Y(Ni58)*state.rho - screened_rates(k_he4_ni59__n_zn62)*Y(Ni59)*state.rho - screened_rates(k_he4_ni59__p_cu62)*Y(Ni59)*state.rho - screened_rates(k_he4_ni59__zn63)*Y(Ni59)*state.rho - screened_rates(k_he4_ni60__n_zn63)*Y(Ni60)*state.rho - screened_rates(k_he4_ni60__p_cu63)*Y(Ni60)*state.rho - screened_rates(k_he4_ni60__zn64)*Y(Ni60)*state.rho - screened_rates(k_he4_ni61__n_zn64)*Y(Ni61)*state.rho - screened_rates(k_he4_ni61__p_cu64)*Y(Ni61)*state.rho - screened_rates(k_he4_ni61__zn65)*Y(Ni61)*state.rho - screened_rates(k_he4_ni62__n_zn65)*Y(Ni62)*state.rho - screened_rates(k_he4_ni62__p_cu65)*Y(Ni62)*state.rho - screened_rates(k_he4_ni62__zn66)*Y(Ni62)*state.rho - screened_rates(k_he4_ni63__n_zn66)*Y(Ni63)*state.rho - screened_rates(k_he4_o14__ne18)*Y(O14)*state.rho - screened_rates(k_he4_o14__p_f17)*Y(O14)*state.rho - screened_rates(k_he4_o15__n_ne18)*Y(O15)*state.rho - screened_rates(k_he4_o15__ne19)*Y(O15)*state.rho - screened_rates(k_he4_o15__p_f18)*Y(O15)*state.rho - screened_rates(k_he4_o16__n_ne19)*Y(O16)*state.rho - screened_rates(k_he4_o16__ne20)*Y(O16)*state.rho - screened_rates(k_he4_o16__p_f19)*Y(O16)*state.rho - screened_rates(k_he4_o17__n_ne20)*Y(O17)*state.rho - screened_rates(k_he4_o17__ne21)*Y(O17)*state.rho - screened_rates(k_he4_o18__n_ne21)*Y(O18)*state.rho - screened_rates(k_he4_o18__ne22)*Y(O18)*state.rho - screened_rates(k_he4_p29__cl33)*Y(P29)*state.rho - screened_rates(k_he4_p29__p_s32)*Y(P29)*state.rho - screened_rates(k_he4_p30__cl34)*Y(P30)*state.rho - screened_rates(k_he4_p30__n_cl33)*Y(P30)*state.rho - screened_rates(k_he4_p30__p_s33)*Y(P30)*state.rho - screened_rates(k_he4_p31__cl35)*Y(P31)*state.rho - screened_rates(k_he4_p31__n_cl34)*Y(P31)*state.rho - screened_rates(k_he4_p31__p_s34)*Y(P31)*state.rho - screened_rates(k_he4_p32__cl36)*Y(P32)*state.rho - screened_rates(k_he4_p32__n_cl35)*Y(P32)*state.rho - screened_rates(k_he4_p32__p_s35)*Y(P32)*state.rho - screened_rates(k_he4_p33__cl37)*Y(P33)*state.rho - screened_rates(k_he4_p33__n_cl36)*Y(P33)*state.rho - screened_rates(k_he4_p33__p_s36)*Y(P33)*state.rho - screened_rates(k_he4_s32__ar36)*Y(S32)*state.rho - screened_rates(k_he4_s32__p_cl35)*Y(S32)*state.rho - screened_rates(k_he4_s33__ar37)*Y(S33)*state.rho - screened_rates(k_he4_s33__n_ar36)*Y(S33)*state.rho - screened_rates(k_he4_s33__p_cl36)*Y(S33)*state.rho - screened_rates(k_he4_s34__ar38)*Y(S34)*state.rho - screened_rates(k_he4_s34__n_ar37)*Y(S34)*state.rho - screened_rates(k_he4_s34__p_cl37)*Y(S34)*state.rho - screened_rates(k_he4_s35__ar39)*Y(S35)*state.rho - screened_rates(k_he4_s35__n_ar38)*Y(S35)*state.rho - screened_rates(k_he4_s36__ar40)*Y(S36)*state.rho - screened_rates(k_he4_s36__n_ar39)*Y(S36)*state.rho - screened_rates(k_he4_sc43__n_v46)*Y(Sc43)*state.rho - screened_rates(k_he4_sc43__p_ti46)*Y(Sc43)*state.rho - screened_rates(k_he4_sc43__v47)*Y(Sc43)*state.rho - screened_rates(k_he4_sc44__n_v47)*Y(Sc44)*state.rho - screened_rates(k_he4_sc44__p_ti47)*Y(Sc44)*state.rho - screened_rates(k_he4_sc44__v48)*Y(Sc44)*state.rho - screened_rates(k_he4_sc45__n_v48)*Y(Sc45)*state.rho - screened_rates(k_he4_sc45__p_ti48)*Y(Sc45)*state.rho - screened_rates(k_he4_sc45__v49)*Y(Sc45)*state.rho - screened_rates(k_he4_sc46__n_v49)*Y(Sc46)*state.rho - screened_rates(k_he4_sc46__p_ti49)*Y(Sc46)*state.rho - screened_rates(k_he4_sc46__v50)*Y(Sc46)*state.rho - screened_rates(k_he4_sc47__n_v50)*Y(Sc47)*state.rho - screened_rates(k_he4_sc47__p_ti50)*Y(Sc47)*state.rho - screened_rates(k_he4_sc47__v51)*Y(Sc47)*state.rho - screened_rates(k_he4_sc48__n_v51)*Y(Sc48)*state.rho - screened_rates(k_he4_sc48__p_ti51)*Y(Sc48)*state.rho - screened_rates(k_he4_sc48__v52)*Y(Sc48)*state.rho - screened_rates(k_he4_sc49__n_v52)*Y(Sc49)*state.rho - screened_rates(k_he4_si28__c12_ne20)*Y(Si28)*state.rho - screened_rates(k_he4_si28__o16_o16)*Y(Si28)*state.rho - screened_rates(k_he4_si28__p_p31)*Y(Si28)*state.rho - screened_rates(k_he4_si28__s32)*Y(Si28)*state.rho - screened_rates(k_he4_si29__n_s32)*Y(Si29)*state.rho - screened_rates(k_he4_si29__p_p32)*Y(Si29)*state.rho - screened_rates(k_he4_si29__s33)*Y(Si29)*state.rho - screened_rates(k_he4_si30__n_s33)*Y(Si30)*state.rho - screened_rates(k_he4_si30__p_p33)*Y(Si30)*state.rho - screened_rates(k_he4_si30__s34)*Y(Si30)*state.rho - screened_rates(k_he4_si31__n_s34)*Y(Si31)*state.rho - screened_rates(k_he4_si31__s35)*Y(Si31)*state.rho - screened_rates(k_he4_si32__n_s35)*Y(Si32)*state.rho - screened_rates(k_he4_si32__s36)*Y(Si32)*state.rho - screened_rates(k_he4_ti44__cr48)*Y(Ti44)*state.rho - screened_rates(k_he4_ti44__p_v47)*Y(Ti44)*state.rho - screened_rates(k_he4_ti45__cr49)*Y(Ti45)*state.rho - screened_rates(k_he4_ti45__n_cr48)*Y(Ti45)*state.rho - screened_rates(k_he4_ti45__p_v48)*Y(Ti45)*state.rho - screened_rates(k_he4_ti46__cr50)*Y(Ti46)*state.rho - screened_rates(k_he4_ti46__n_cr49)*Y(Ti46)*state.rho - screened_rates(k_he4_ti46__p_v49)*Y(Ti46)*state.rho - screened_rates(k_he4_ti47__cr51)*Y(Ti47)*state.rho - screened_rates(k_he4_ti47__n_cr50)*Y(Ti47)*state.rho - screened_rates(k_he4_ti47__p_v50)*Y(Ti47)*state.rho - screened_rates(k_he4_ti48__cr52)*Y(Ti48)*state.rho - screened_rates(k_he4_ti48__n_cr51)*Y(Ti48)*state.rho - screened_rates(k_he4_ti48__p_v51)*Y(Ti48)*state.rho - screened_rates(k_he4_ti49__cr53)*Y(Ti49)*state.rho - screened_rates(k_he4_ti49__n_cr52)*Y(Ti49)*state.rho - screened_rates(k_he4_ti49__p_v52)*Y(Ti49)*state.rho - screened_rates(k_he4_ti50__cr54)*Y(Ti50)*state.rho - screened_rates(k_he4_ti50__n_cr53)*Y(Ti50)*state.rho - screened_rates(k_he4_ti51__n_cr54)*Y(Ti51)*state.rho - screened_rates(k_he4_v46__mn50)*Y(V46)*state.rho - screened_rates(k_he4_v46__p_cr49)*Y(V46)*state.rho - screened_rates(k_he4_v47__mn51)*Y(V47)*state.rho - screened_rates(k_he4_v47__n_mn50)*Y(V47)*state.rho - screened_rates(k_he4_v47__p_cr50)*Y(V47)*state.rho - screened_rates(k_he4_v48__mn52)*Y(V48)*state.rho - screened_rates(k_he4_v48__n_mn51)*Y(V48)*state.rho - screened_rates(k_he4_v48__p_cr51)*Y(V48)*state.rho - screened_rates(k_he4_v49__mn53)*Y(V49)*state.rho - screened_rates(k_he4_v49__n_mn52)*Y(V49)*state.rho - screened_rates(k_he4_v49__p_cr52)*Y(V49)*state.rho - screened_rates(k_he4_v50__mn54)*Y(V50)*state.rho - screened_rates(k_he4_v50__n_mn53)*Y(V50)*state.rho - screened_rates(k_he4_v50__p_cr53)*Y(V50)*state.rho - screened_rates(k_he4_v51__mn55)*Y(V51)*state.rho - screened_rates(k_he4_v51__n_mn54)*Y(V51)*state.rho - screened_rates(k_he4_v51__p_cr54)*Y(V51)*state.rho - screened_rates(k_he4_v52__n_mn55)*Y(V52)*state.rho - screened_rates(k_he4_zn59__ge63)*Y(Zn59)*state.rho - screened_rates(k_he4_zn59__p_ga62)*Y(Zn59)*state.rho - screened_rates(k_he4_zn60__ge64)*Y(Zn60)*state.rho - screened_rates(k_he4_zn60__n_ge63)*Y(Zn60)*state.rho - screened_rates(k_he4_zn60__p_ga63)*Y(Zn60)*state.rho - screened_rates(k_he4_zn61__n_ge64)*Y(Zn61)*state.rho - screened_rates(k_he4_zn61__p_ga64)*Y(Zn61)*state.rho - 2.0*screened_rates(k_n_he4_he4__be9)*Y(He4)*Y(N)*std::pow(state.rho, 2) - 2.0*screened_rates(k_n_he4_he4__d_li7)*Y(He4)*Y(N)*std::pow(state.rho, 2) - screened_rates(k_n_p_he4__li6)*Y(N)*Y(H1)*std::pow(state.rho, 2) - 2.0*screened_rates(k_n_p_he4_he4__he3_li7)*Y(He4)*Y(N)*Y(H1)*std::pow(state.rho, 3) - 2.0*screened_rates(k_n_p_he4_he4__p_be9)*Y(He4)*Y(N)*Y(H1)*std::pow(state.rho, 3) - screened_rates(k_p_he4__d_he3)*Y(H1)*state.rho - 2.0*screened_rates(k_p_he4_he4__d_be7)*Y(He4)*Y(H1)*std::pow(state.rho, 2) - 2.0*screened_rates(k_p_he4_he4__n_b8)*Y(He4)*Y(H1)*std::pow(state.rho, 2) - 0.5*screened_rates(k_p_p_he4__he3_he3)*std::pow(Y(H1), 2)*std::pow(state.rho, 2) - 1.0*screened_rates(k_p_p_he4_he4__he3_be7)*Y(He4)*std::pow(Y(H1), 2)*std::pow(state.rho, 3);
    jac.set(He4, He4, scratch);

    scratch = -screened_rates(k_he4_li6__b10)*Y(He4)*state.rho - screened_rates(k_he4_li6__p_be9)*Y(He4)*state.rho + screened_rates(k_li6__he4_d) + screened_rates(k_li6__n_p_he4) + screened_rates(k_p_li6__he4_he3)*Y(H1)*state.rho;
    jac.set(He4, Li6, scratch);

    scratch = 2.0*screened_rates(k_d_li7__n_he4_he4)*Y(H2)*state.rho + 2.0*screened_rates(k_he3_li7__n_p_he4_he4)*Y(He3)*state.rho - screened_rates(k_he4_li7__b11)*Y(He4)*state.rho - screened_rates(k_he4_li7__n_b10)*Y(He4)*state.rho + 2.0*screened_rates(k_p_li7__he4_he4)*Y(H1)*state.rho;
    jac.set(He4, Li7, scratch);

    scratch = screened_rates(k_be7__he4_he3) + 2.0*screened_rates(k_d_be7__p_he4_he4)*Y(H2)*state.rho + 2.0*screened_rates(k_he3_be7__p_p_he4_he4)*Y(He3)*state.rho - screened_rates(k_he4_be7__p_b10)*Y(He4)*state.rho + 2.0*screened_rates(k_n_be7__he4_he4)*Y(N)*state.rho;
    jac.set(He4, Be7, scratch);

    scratch = 2.0*screened_rates(k_be9__n_he4_he4) - screened_rates(k_he4_be9__n_c12)*Y(He4)*state.rho + 2.0*screened_rates(k_p_be9__d_he4_he4)*Y(H1)*state.rho + screened_rates(k_p_be9__he4_li6)*Y(H1)*state.rho + 2.0*screened_rates(k_p_be9__n_p_he4_he4)*Y(H1)*state.rho;
    jac.set(He4, Be9, scratch);

    scratch = 2.0*screened_rates(k_b8__he4_he4__weak__wc12) + 2.0*screened_rates(k_n_b8__p_he4_he4)*Y(N)*state.rho;
    jac.set(He4, B8, scratch);

    scratch = screened_rates(k_b10__he4_li6) - screened_rates(k_he4_b10__n_n13)*Y(He4)*state.rho - screened_rates(k_he4_b10__p_c13)*Y(He4)*state.rho + screened_rates(k_n_b10__he4_li7)*Y(N)*state.rho + screened_rates(k_p_b10__he4_be7)*Y(H1)*state.rho;
    jac.set(He4, B10, scratch);

    scratch = screened_rates(k_b11__he4_li7) - screened_rates(k_he4_b11__n_n14)*Y(He4)*state.rho - screened_rates(k_he4_b11__p_c14)*Y(He4)*state.rho + 3.0*screened_rates(k_p_b11__he4_he4_he4)*Y(H1)*state.rho;
    jac.set(He4, B11, scratch);

    scratch = 3.0*screened_rates(k_c12__he4_he4_he4) + 1.0*screened_rates(k_c12_c12__he4_ne20)*Y(C12)*state.rho + screened_rates(k_c12_ne20__he4_si28)*Y(Ne20)*state.rho + screened_rates(k_c12_o16__he4_mg24)*Y(O16)*state.rho - screened_rates(k_he4_c12__n_o15)*Y(He4)*state.rho - screened_rates(k_he4_c12__o16)*Y(He4)*state.rho - screened_rates(k_he4_c12__p_n15)*Y(He4)*state.rho + screened_rates(k_n_c12__he4_be9)*Y(N)*state.rho;
    jac.set(He4, C12, scratch);

    scratch = -screened_rates(k_he4_c13__n_o16)*Y(He4)*state.rho + screened_rates(k_p_c13__he4_b10)*Y(H1)*state.rho;
    jac.set(He4, C13, scratch);

    scratch = -screened_rates(k_he4_c14__n_o17)*Y(He4)*state.rho - screened_rates(k_he4_c14__o18)*Y(He4)*state.rho + screened_rates(k_p_c14__he4_b11)*Y(H1)*state.rho;
    jac.set(He4, C14, scratch);

    scratch = -screened_rates(k_he4_n13__p_o16)*Y(He4)*state.rho + screened_rates(k_n_n13__he4_b10)*Y(N)*state.rho;
    jac.set(He4, N13, scratch);

    scratch = -screened_rates(k_he4_n14__f18)*Y(He4)*state.rho - screened_rates(k_he4_n14__n_f17)*Y(He4)*state.rho - screened_rates(k_he4_n14__p_o17)*Y(He4)*state.rho + screened_rates(k_n_n14__he4_b11)*Y(N)*state.rho;
    jac.set(He4, N14, scratch);

    scratch = -screened_rates(k_he4_n15__f19)*Y(He4)*state.rho - screened_rates(k_he4_n15__n_f18)*Y(He4)*state.rho - screened_rates(k_he4_n15__p_o18)*Y(He4)*state.rho + screened_rates(k_p_n15__he4_c12)*Y(H1)*state.rho;
    jac.set(He4, N15, scratch);

    scratch = -screened_rates(k_he4_o14__ne18)*Y(He4)*state.rho - screened_rates(k_he4_o14__p_f17)*Y(He4)*state.rho;
    jac.set(He4, O14, scratch);

    scratch = -screened_rates(k_he4_o15__n_ne18)*Y(He4)*state.rho - screened_rates(k_he4_o15__ne19)*Y(He4)*state.rho - screened_rates(k_he4_o15__p_f18)*Y(He4)*state.rho + screened_rates(k_n_o15__he4_c12)*Y(N)*state.rho;
    jac.set(He4, O15, scratch);

    scratch = screened_rates(k_c12_o16__he4_mg24)*Y(C12)*state.rho - screened_rates(k_he4_o16__n_ne19)*Y(He4)*state.rho - screened_rates(k_he4_o16__ne20)*Y(He4)*state.rho - screened_rates(k_he4_o16__p_f19)*Y(He4)*state.rho + screened_rates(k_n_o16__he4_c13)*Y(N)*state.rho + screened_rates(k_o16__he4_c12) + 1.0*screened_rates(k_o16_o16__he4_si28)*Y(O16)*state.rho + screened_rates(k_p_o16__he4_n13)*Y(H1)*state.rho;
    jac.set(He4, O16, scratch);

    scratch = -screened_rates(k_he4_o17__n_ne20)*Y(He4)*state.rho - screened_rates(k_he4_o17__ne21)*Y(He4)*state.rho + screened_rates(k_n_o17__he4_c14)*Y(N)*state.rho + screened_rates(k_p_o17__he4_n14)*Y(H1)*state.rho;
    jac.set(He4, O17, scratch);

    scratch = -screened_rates(k_he4_o18__n_ne21)*Y(He4)*state.rho - screened_rates(k_he4_o18__ne22)*Y(He4)*state.rho + screened_rates(k_o18__he4_c14) + screened_rates(k_p_o18__he4_n15)*Y(H1)*state.rho;
    jac.set(He4, O18, scratch);

    scratch = -screened_rates(k_he4_f17__na21)*Y(He4)*state.rho - screened_rates(k_he4_f17__p_ne20)*Y(He4)*state.rho + screened_rates(k_n_f17__he4_n14)*Y(N)*state.rho + screened_rates(k_p_f17__he4_o14)*Y(H1)*state.rho;
    jac.set(He4, F17, scratch);

    scratch = screened_rates(k_f18__he4_n14) - screened_rates(k_he4_f18__n_na21)*Y(He4)*state.rho - screened_rates(k_he4_f18__na22)*Y(He4)*state.rho - screened_rates(k_he4_f18__p_ne21)*Y(He4)*state.rho + screened_rates(k_n_f18__he4_n15)*Y(N)*state.rho + screened_rates(k_p_f18__he4_o15)*Y(H1)*state.rho;
    jac.set(He4, F18, scratch);

    scratch = screened_rates(k_f19__he4_n15) - screened_rates(k_he4_f19__n_na22)*Y(He4)*state.rho - screened_rates(k_he4_f19__na23)*Y(He4)*state.rho - screened_rates(k_he4_f19__p_ne22)*Y(He4)*state.rho + screened_rates(k_p_f19__he4_o16)*Y(H1)*state.rho;
    jac.set(He4, F19, scratch);

    scratch = -screened_rates(k_he4_ne18__p_na21)*Y(He4)*state.rho + screened_rates(k_n_ne18__he4_o15)*Y(N)*state.rho + screened_rates(k_ne18__he4_o14);
    jac.set(He4, Ne18, scratch);

    scratch = -screened_rates(k_he4_ne19__mg23)*Y(He4)*state.rho - screened_rates(k_he4_ne19__p_na22)*Y(He4)*state.rho + screened_rates(k_n_ne19__he4_o16)*Y(N)*state.rho + screened_rates(k_ne19__he4_o15);
    jac.set(He4, Ne19, scratch);

    scratch = screened_rates(k_c12_ne20__he4_si28)*Y(C12)*state.rho - screened_rates(k_he4_ne20__c12_c12)*Y(He4)*state.rho - screened_rates(k_he4_ne20__mg24)*Y(He4)*state.rho - screened_rates(k_he4_ne20__n_mg23)*Y(He4)*state.rho - screened_rates(k_he4_ne20__p_na23)*Y(He4)*state.rho + screened_rates(k_n_ne20__he4_o17)*Y(N)*state.rho + screened_rates(k_ne20__he4_o16) + screened_rates(k_p_ne20__he4_f17)*Y(H1)*state.rho;
    jac.set(He4, Ne20, scratch);

    scratch = -screened_rates(k_he4_ne21__mg25)*Y(He4)*state.rho - screened_rates(k_he4_ne21__n_mg24)*Y(He4)*state.rho + screened_rates(k_n_ne21__he4_o18)*Y(N)*state.rho + screened_rates(k_ne21__he4_o17) + screened_rates(k_p_ne21__he4_f18)*Y(H1)*state.rho;
    jac.set(He4, Ne21, scratch);

    scratch = -screened_rates(k_he4_ne22__mg26)*Y(He4)*state.rho - screened_rates(k_he4_ne22__n_mg25)*Y(He4)*state.rho + screened_rates(k_ne22__he4_o18) + screened_rates(k_p_ne22__he4_f19)*Y(H1)*state.rho;
    jac.set(He4, Ne22, scratch);

    scratch = -screened_rates(k_he4_na21__al25)*Y(He4)*state.rho - screened_rates(k_he4_na21__p_mg24)*Y(He4)*state.rho + screened_rates(k_n_na21__he4_f18)*Y(N)*state.rho + screened_rates(k_na21__he4_f17) + screened_rates(k_p_na21__he4_ne18)*Y(H1)*state.rho;
    jac.set(He4, Na21, scratch);

    scratch = -screened_rates(k_he4_na22__al26)*Y(He4)*state.rho - screened_rates(k_he4_na22__n_al25)*Y(He4)*state.rho - screened_rates(k_he4_na22__p_mg25)*Y(He4)*state.rho + screened_rates(k_n_na22__he4_f19)*Y(N)*state.rho + screened_rates(k_na22__he4_f18) + screened_rates(k_p_na22__he4_ne19)*Y(H1)*state.rho;
    jac.set(He4, Na22, scratch);

    scratch = -screened_rates(k_he4_na23__al27)*Y(He4)*state.rho - screened_rates(k_he4_na23__n_al26)*Y(He4)*state.rho - screened_rates(k_he4_na23__p_mg26)*Y(He4)*state.rho + screened_rates(k_na23__he4_f19) + screened_rates(k_p_na23__he4_ne20)*Y(H1)*state.rho;
    jac.set(He4, Na23, scratch);

    scratch = -screened_rates(k_he4_mg23__p_al26)*Y(He4)*state.rho + screened_rates(k_mg23__he4_ne19) + screened_rates(k_n_mg23__he4_ne20)*Y(N)*state.rho;
    jac.set(He4, Mg23, scratch);

    scratch = -screened_rates(k_he4_mg24__c12_o16)*Y(He4)*state.rho - screened_rates(k_he4_mg24__p_al27)*Y(He4)*state.rho - screened_rates(k_he4_mg24__si28)*Y(He4)*state.rho + screened_rates(k_mg24__he4_ne20) + screened_rates(k_n_mg24__he4_ne21)*Y(N)*state.rho + screened_rates(k_p_mg24__he4_na21)*Y(H1)*state.rho;
    jac.set(He4, Mg24, scratch);

    scratch = -screened_rates(k_he4_mg25__n_si28)*Y(He4)*state.rho - screened_rates(k_he4_mg25__si29)*Y(He4)*state.rho + screened_rates(k_mg25__he4_ne21) + screened_rates(k_n_mg25__he4_ne22)*Y(N)*state.rho + screened_rates(k_p_mg25__he4_na22)*Y(H1)*state.rho;
    jac.set(He4, Mg25, scratch);

    scratch = -screened_rates(k_he4_mg26__n_si29)*Y(He4)*state.rho - screened_rates(k_he4_mg26__si30)*Y(He4)*state.rho + screened_rates(k_mg26__he4_ne22) + screened_rates(k_p_mg26__he4_na23)*Y(H1)*state.rho;
    jac.set(He4, Mg26, scratch);

    scratch = screened_rates(k_al25__he4_na21) - screened_rates(k_he4_al25__p29)*Y(He4)*state.rho - screened_rates(k_he4_al25__p_si28)*Y(He4)*state.rho + screened_rates(k_n_al25__he4_na22)*Y(N)*state.rho;
    jac.set(He4, Al25, scratch);

    scratch = screened_rates(k_al26__he4_na22) - screened_rates(k_he4_al26__n_p29)*Y(He4)*state.rho - screened_rates(k_he4_al26__p30)*Y(He4)*state.rho - screened_rates(k_he4_al26__p_si29)*Y(He4)*state.rho + screened_rates(k_n_al26__he4_na23)*Y(N)*state.rho + screened_rates(k_p_al26__he4_mg23)*Y(H1)*state.rho;
    jac.set(He4, Al26, scratch);

    scratch = screened_rates(k_al27__he4_na23) - screened_rates(k_he4_al27__n_p30)*Y(He4)*state.rho - screened_rates(k_he4_al27__p31)*Y(He4)*state.rho - screened_rates(k_he4_al27__p_si30)*Y(He4)*state.rho + screened_rates(k_p_al27__he4_mg24)*Y(H1)*state.rho;
    jac.set(He4, Al27, scratch);

    scratch = -screened_rates(k_he4_si28__c12_ne20)*Y(He4)*state.rho - screened_rates(k_he4_si28__o16_o16)*Y(He4)*state.rho - screened_rates(k_he4_si28__p_p31)*Y(He4)*state.rho - screened_rates(k_he4_si28__s32)*Y(He4)*state.rho + screened_rates(k_n_si28__he4_mg25)*Y(N)*state.rho + screened_rates(k_p_si28__he4_al25)*Y(H1)*state.rho + screened_rates(k_si28__he4_mg24);
    jac.set(He4, Si28, scratch);

    scratch = -screened_rates(k_he4_si29__n_s32)*Y(He4)*state.rho - screened_rates(k_he4_si29__p_p32)*Y(He4)*state.rho - screened_rates(k_he4_si29__s33)*Y(He4)*state.rho + screened_rates(k_n_si29__he4_mg26)*Y(N)*state.rho + screened_rates(k_p_si29__he4_al26)*Y(H1)*state.rho + screened_rates(k_si29__he4_mg25);
    jac.set(He4, Si29, scratch);

    scratch = -screened_rates(k_he4_si30__n_s33)*Y(He4)*state.rho - screened_rates(k_he4_si30__p_p33)*Y(He4)*state.rho - screened_rates(k_he4_si30__s34)*Y(He4)*state.rho + screened_rates(k_p_si30__he4_al27)*Y(H1)*state.rho + screened_rates(k_si30__he4_mg26);
    jac.set(He4, Si30, scratch);

    scratch = -screened_rates(k_he4_si31__n_s34)*Y(He4)*state.rho - screened_rates(k_he4_si31__s35)*Y(He4)*state.rho;
    jac.set(He4, Si31, scratch);

    scratch = -screened_rates(k_he4_si32__n_s35)*Y(He4)*state.rho - screened_rates(k_he4_si32__s36)*Y(He4)*state.rho;
    jac.set(He4, Si32, scratch);

    scratch = -screened_rates(k_he4_p29__cl33)*Y(He4)*state.rho - screened_rates(k_he4_p29__p_s32)*Y(He4)*state.rho + screened_rates(k_n_p29__he4_al26)*Y(N)*state.rho + screened_rates(k_p29__he4_al25);
    jac.set(He4, P29, scratch);

    scratch = -screened_rates(k_he4_p30__cl34)*Y(He4)*state.rho - screened_rates(k_he4_p30__n_cl33)*Y(He4)*state.rho - screened_rates(k_he4_p30__p_s33)*Y(He4)*state.rho + screened_rates(k_n_p30__he4_al27)*Y(N)*state.rho + screened_rates(k_p30__he4_al26);
    jac.set(He4, P30, scratch);

    scratch = -screened_rates(k_he4_p31__cl35)*Y(He4)*state.rho - screened_rates(k_he4_p31__n_cl34)*Y(He4)*state.rho - screened_rates(k_he4_p31__p_s34)*Y(He4)*state.rho + screened_rates(k_p31__he4_al27) + screened_rates(k_p_p31__he4_si28)*Y(H1)*state.rho;
    jac.set(He4, P31, scratch);

    scratch = -screened_rates(k_he4_p32__cl36)*Y(He4)*state.rho - screened_rates(k_he4_p32__n_cl35)*Y(He4)*state.rho - screened_rates(k_he4_p32__p_s35)*Y(He4)*state.rho + screened_rates(k_p_p32__he4_si29)*Y(H1)*state.rho;
    jac.set(He4, P32, scratch);

    scratch = -screened_rates(k_he4_p33__cl37)*Y(He4)*state.rho - screened_rates(k_he4_p33__n_cl36)*Y(He4)*state.rho - screened_rates(k_he4_p33__p_s36)*Y(He4)*state.rho + screened_rates(k_p_p33__he4_si30)*Y(H1)*state.rho;
    jac.set(He4, P33, scratch);

    scratch = -screened_rates(k_he4_s32__ar36)*Y(He4)*state.rho - screened_rates(k_he4_s32__p_cl35)*Y(He4)*state.rho + screened_rates(k_n_s32__he4_si29)*Y(N)*state.rho + screened_rates(k_p_s32__he4_p29)*Y(H1)*state.rho + screened_rates(k_s32__he4_si28);
    jac.set(He4, S32, scratch);

    scratch = -screened_rates(k_he4_s33__ar37)*Y(He4)*state.rho - screened_rates(k_he4_s33__n_ar36)*Y(He4)*state.rho - screened_rates(k_he4_s33__p_cl36)*Y(He4)*state.rho + screened_rates(k_n_s33__he4_si30)*Y(N)*state.rho + screened_rates(k_p_s33__he4_p30)*Y(H1)*state.rho + screened_rates(k_s33__he4_si29);
    jac.set(He4, S33, scratch);

    scratch = -screened_rates(k_he4_s34__ar38)*Y(He4)*state.rho - screened_rates(k_he4_s34__n_ar37)*Y(He4)*state.rho - screened_rates(k_he4_s34__p_cl37)*Y(He4)*state.rho + screened_rates(k_n_s34__he4_si31)*Y(N)*state.rho + screened_rates(k_p_s34__he4_p31)*Y(H1)*state.rho + screened_rates(k_s34__he4_si30);
    jac.set(He4, S34, scratch);

    scratch = -screened_rates(k_he4_s35__ar39)*Y(He4)*state.rho - screened_rates(k_he4_s35__n_ar38)*Y(He4)*state.rho + screened_rates(k_n_s35__he4_si32)*Y(N)*state.rho + screened_rates(k_p_s35__he4_p32)*Y(H1)*state.rho + screened_rates(k_s35__he4_si31);
    jac.set(He4, S35, scratch);

    scratch = -screened_rates(k_he4_s36__ar40)*Y(He4)*state.rho - screened_rates(k_he4_s36__n_ar39)*Y(He4)*state.rho + screened_rates(k_p_s36__he4_p33)*Y(H1)*state.rho + screened_rates(k_s36__he4_si32);
    jac.set(He4, S36, scratch);

    scratch = screened_rates(k_cl33__he4_p29) - screened_rates(k_he4_cl33__k37)*Y(He4)*state.rho - screened_rates(k_he4_cl33__p_ar36)*Y(He4)*state.rho + screened_rates(k_n_cl33__he4_p30)*Y(N)*state.rho;
    jac.set(He4, Cl33, scratch);

    scratch = screened_rates(k_cl34__he4_p30) - screened_rates(k_he4_cl34__k38)*Y(He4)*state.rho - screened_rates(k_he4_cl34__n_k37)*Y(He4)*state.rho - screened_rates(k_he4_cl34__p_ar37)*Y(He4)*state.rho + screened_rates(k_n_cl34__he4_p31)*Y(N)*state.rho;
    jac.set(He4, Cl34, scratch);

    scratch = screened_rates(k_cl35__he4_p31) - screened_rates(k_he4_cl35__k39)*Y(He4)*state.rho - screened_rates(k_he4_cl35__n_k38)*Y(He4)*state.rho - screened_rates(k_he4_cl35__p_ar38)*Y(He4)*state.rho + screened_rates(k_n_cl35__he4_p32)*Y(N)*state.rho + screened_rates(k_p_cl35__he4_s32)*Y(H1)*state.rho;
    jac.set(He4, Cl35, scratch);

    scratch = screened_rates(k_cl36__he4_p32) - screened_rates(k_he4_cl36__k40)*Y(He4)*state.rho - screened_rates(k_he4_cl36__n_k39)*Y(He4)*state.rho - screened_rates(k_he4_cl36__p_ar39)*Y(He4)*state.rho + screened_rates(k_n_cl36__he4_p33)*Y(N)*state.rho + screened_rates(k_p_cl36__he4_s33)*Y(H1)*state.rho;
    jac.set(He4, Cl36, scratch);

    scratch = screened_rates(k_cl37__he4_p33) - screened_rates(k_he4_cl37__k41)*Y(He4)*state.rho - screened_rates(k_he4_cl37__n_k40)*Y(He4)*state.rho - screened_rates(k_he4_cl37__p_ar40)*Y(He4)*state.rho + screened_rates(k_p_cl37__he4_s34)*Y(H1)*state.rho;
    jac.set(He4, Cl37, scratch);

    scratch = screened_rates(k_ar36__he4_s32) - screened_rates(k_he4_ar36__ca40)*Y(He4)*state.rho - screened_rates(k_he4_ar36__p_k39)*Y(He4)*state.rho + screened_rates(k_n_ar36__he4_s33)*Y(N)*state.rho + screened_rates(k_p_ar36__he4_cl33)*Y(H1)*state.rho;
    jac.set(He4, Ar36, scratch);

    scratch = screened_rates(k_ar37__he4_s33) - screened_rates(k_he4_ar37__ca41)*Y(He4)*state.rho - screened_rates(k_he4_ar37__n_ca40)*Y(He4)*state.rho - screened_rates(k_he4_ar37__p_k40)*Y(He4)*state.rho + screened_rates(k_n_ar37__he4_s34)*Y(N)*state.rho + screened_rates(k_p_ar37__he4_cl34)*Y(H1)*state.rho;
    jac.set(He4, Ar37, scratch);

    scratch = screened_rates(k_ar38__he4_s34) - screened_rates(k_he4_ar38__ca42)*Y(He4)*state.rho - screened_rates(k_he4_ar38__n_ca41)*Y(He4)*state.rho - screened_rates(k_he4_ar38__p_k41)*Y(He4)*state.rho + screened_rates(k_n_ar38__he4_s35)*Y(N)*state.rho + screened_rates(k_p_ar38__he4_cl35)*Y(H1)*state.rho;
    jac.set(He4, Ar38, scratch);

    scratch = screened_rates(k_ar39__he4_s35) - screened_rates(k_he4_ar39__ca43)*Y(He4)*state.rho - screened_rates(k_he4_ar39__n_ca42)*Y(He4)*state.rho + screened_rates(k_n_ar39__he4_s36)*Y(N)*state.rho + screened_rates(k_p_ar39__he4_cl36)*Y(H1)*state.rho;
    jac.set(He4, Ar39, scratch);

    scratch = screened_rates(k_ar40__he4_s36) - screened_rates(k_he4_ar40__ca44)*Y(He4)*state.rho - screened_rates(k_he4_ar40__n_ca43)*Y(He4)*state.rho + screened_rates(k_p_ar40__he4_cl37)*Y(H1)*state.rho;
    jac.set(He4, Ar40, scratch);

    scratch = -screened_rates(k_he4_k37__p_ca40)*Y(He4)*state.rho + screened_rates(k_k37__he4_cl33) + screened_rates(k_n_k37__he4_cl34)*Y(N)*state.rho;
    jac.set(He4, K37, scratch);

    scratch = -screened_rates(k_he4_k38__p_ca41)*Y(He4)*state.rho + screened_rates(k_k38__he4_cl34) + screened_rates(k_n_k38__he4_cl35)*Y(N)*state.rho;
    jac.set(He4, K38, scratch);

    scratch = -screened_rates(k_he4_k39__p_ca42)*Y(He4)*state.rho - screened_rates(k_he4_k39__sc43)*Y(He4)*state.rho + screened_rates(k_k39__he4_cl35) + screened_rates(k_n_k39__he4_cl36)*Y(N)*state.rho + screened_rates(k_p_k39__he4_ar36)*Y(H1)*state.rho;
    jac.set(He4, K39, scratch);

    scratch = -screened_rates(k_he4_k40__n_sc43)*Y(He4)*state.rho - screened_rates(k_he4_k40__p_ca43)*Y(He4)*state.rho - screened_rates(k_he4_k40__sc44)*Y(He4)*state.rho + screened_rates(k_k40__he4_cl36) + screened_rates(k_n_k40__he4_cl37)*Y(N)*state.rho + screened_rates(k_p_k40__he4_ar37)*Y(H1)*state.rho;
    jac.set(He4, K40, scratch);

    scratch = -screened_rates(k_he4_k41__n_sc44)*Y(He4)*state.rho - screened_rates(k_he4_k41__p_ca44)*Y(He4)*state.rho - screened_rates(k_he4_k41__sc45)*Y(He4)*state.rho + screened_rates(k_k41__he4_cl37) + screened_rates(k_p_k41__he4_ar38)*Y(H1)*state.rho;
    jac.set(He4, K41, scratch);

    scratch = screened_rates(k_ca40__he4_ar36) - screened_rates(k_he4_ca40__p_sc43)*Y(He4)*state.rho - screened_rates(k_he4_ca40__ti44)*Y(He4)*state.rho + screened_rates(k_n_ca40__he4_ar37)*Y(N)*state.rho + screened_rates(k_p_ca40__he4_k37)*Y(H1)*state.rho;
    jac.set(He4, Ca40, scratch);

    scratch = screened_rates(k_ca41__he4_ar37) - screened_rates(k_he4_ca41__n_ti44)*Y(He4)*state.rho - screened_rates(k_he4_ca41__p_sc44)*Y(He4)*state.rho - screened_rates(k_he4_ca41__ti45)*Y(He4)*state.rho + screened_rates(k_n_ca41__he4_ar38)*Y(N)*state.rho + screened_rates(k_p_ca41__he4_k38)*Y(H1)*state.rho;
    jac.set(He4, Ca41, scratch);

    scratch = screened_rates(k_ca42__he4_ar38) - screened_rates(k_he4_ca42__n_ti45)*Y(He4)*state.rho - screened_rates(k_he4_ca42__p_sc45)*Y(He4)*state.rho - screened_rates(k_he4_ca42__ti46)*Y(He4)*state.rho + screened_rates(k_n_ca42__he4_ar39)*Y(N)*state.rho + screened_rates(k_p_ca42__he4_k39)*Y(H1)*state.rho;
    jac.set(He4, Ca42, scratch);

    scratch = screened_rates(k_ca43__he4_ar39) - screened_rates(k_he4_ca43__n_ti46)*Y(He4)*state.rho - screened_rates(k_he4_ca43__p_sc46)*Y(He4)*state.rho - screened_rates(k_he4_ca43__ti47)*Y(He4)*state.rho + screened_rates(k_n_ca43__he4_ar40)*Y(N)*state.rho + screened_rates(k_p_ca43__he4_k40)*Y(H1)*state.rho;
    jac.set(He4, Ca43, scratch);

    scratch = screened_rates(k_ca44__he4_ar40) - screened_rates(k_he4_ca44__n_ti47)*Y(He4)*state.rho - screened_rates(k_he4_ca44__p_sc47)*Y(He4)*state.rho - screened_rates(k_he4_ca44__ti48)*Y(He4)*state.rho + screened_rates(k_p_ca44__he4_k41)*Y(H1)*state.rho;
    jac.set(He4, Ca44, scratch);

    scratch = -screened_rates(k_he4_ca45__n_ti48)*Y(He4)*state.rho - screened_rates(k_he4_ca45__p_sc48)*Y(He4)*state.rho - screened_rates(k_he4_ca45__ti49)*Y(He4)*state.rho;
    jac.set(He4, Ca45, scratch);

    scratch = -screened_rates(k_he4_ca46__n_ti49)*Y(He4)*state.rho - screened_rates(k_he4_ca46__p_sc49)*Y(He4)*state.rho - screened_rates(k_he4_ca46__ti50)*Y(He4)*state.rho;
    jac.set(He4, Ca46, scratch);

    scratch = -screened_rates(k_he4_ca47__n_ti50)*Y(He4)*state.rho - screened_rates(k_he4_ca47__ti51)*Y(He4)*state.rho;
    jac.set(He4, Ca47, scratch);

    scratch = -screened_rates(k_he4_ca48__n_ti51)*Y(He4)*state.rho;
    jac.set(He4, Ca48, scratch);

    scratch = -screened_rates(k_he4_sc43__n_v46)*Y(He4)*state.rho - screened_rates(k_he4_sc43__p_ti46)*Y(He4)*state.rho - screened_rates(k_he4_sc43__v47)*Y(He4)*state.rho + screened_rates(k_n_sc43__he4_k40)*Y(N)*state.rho + screened_rates(k_p_sc43__he4_ca40)*Y(H1)*state.rho + screened_rates(k_sc43__he4_k39);
    jac.set(He4, Sc43, scratch);

    scratch = -screened_rates(k_he4_sc44__n_v47)*Y(He4)*state.rho - screened_rates(k_he4_sc44__p_ti47)*Y(He4)*state.rho - screened_rates(k_he4_sc44__v48)*Y(He4)*state.rho + screened_rates(k_n_sc44__he4_k41)*Y(N)*state.rho + screened_rates(k_p_sc44__he4_ca41)*Y(H1)*state.rho + screened_rates(k_sc44__he4_k40);
    jac.set(He4, Sc44, scratch);

    scratch = -screened_rates(k_he4_sc45__n_v48)*Y(He4)*state.rho - screened_rates(k_he4_sc45__p_ti48)*Y(He4)*state.rho - screened_rates(k_he4_sc45__v49)*Y(He4)*state.rho + screened_rates(k_p_sc45__he4_ca42)*Y(H1)*state.rho + screened_rates(k_sc45__he4_k41);
    jac.set(He4, Sc45, scratch);

    scratch = -screened_rates(k_he4_sc46__n_v49)*Y(He4)*state.rho - screened_rates(k_he4_sc46__p_ti49)*Y(He4)*state.rho - screened_rates(k_he4_sc46__v50)*Y(He4)*state.rho + screened_rates(k_p_sc46__he4_ca43)*Y(H1)*state.rho;
    jac.set(He4, Sc46, scratch);

    scratch = -screened_rates(k_he4_sc47__n_v50)*Y(He4)*state.rho - screened_rates(k_he4_sc47__p_ti50)*Y(He4)*state.rho - screened_rates(k_he4_sc47__v51)*Y(He4)*state.rho + screened_rates(k_p_sc47__he4_ca44)*Y(H1)*state.rho;
    jac.set(He4, Sc47, scratch);

    scratch = -screened_rates(k_he4_sc48__n_v51)*Y(He4)*state.rho - screened_rates(k_he4_sc48__p_ti51)*Y(He4)*state.rho - screened_rates(k_he4_sc48__v52)*Y(He4)*state.rho + screened_rates(k_p_sc48__he4_ca45)*Y(H1)*state.rho;
    jac.set(He4, Sc48, scratch);

    scratch = -screened_rates(k_he4_sc49__n_v52)*Y(He4)*state.rho + screened_rates(k_p_sc49__he4_ca46)*Y(H1)*state.rho;
    jac.set(He4, Sc49, scratch);

    scratch = -screened_rates(k_he4_ti44__cr48)*Y(He4)*state.rho - screened_rates(k_he4_ti44__p_v47)*Y(He4)*state.rho + screened_rates(k_n_ti44__he4_ca41)*Y(N)*state.rho + screened_rates(k_ti44__he4_ca40);
    jac.set(He4, Ti44, scratch);

    scratch = -screened_rates(k_he4_ti45__cr49)*Y(He4)*state.rho - screened_rates(k_he4_ti45__n_cr48)*Y(He4)*state.rho - screened_rates(k_he4_ti45__p_v48)*Y(He4)*state.rho + screened_rates(k_n_ti45__he4_ca42)*Y(N)*state.rho + screened_rates(k_ti45__he4_ca41);
    jac.set(He4, Ti45, scratch);

    scratch = -screened_rates(k_he4_ti46__cr50)*Y(He4)*state.rho - screened_rates(k_he4_ti46__n_cr49)*Y(He4)*state.rho - screened_rates(k_he4_ti46__p_v49)*Y(He4)*state.rho + screened_rates(k_n_ti46__he4_ca43)*Y(N)*state.rho + screened_rates(k_p_ti46__he4_sc43)*Y(H1)*state.rho + screened_rates(k_ti46__he4_ca42);
    jac.set(He4, Ti46, scratch);

    scratch = -screened_rates(k_he4_ti47__cr51)*Y(He4)*state.rho - screened_rates(k_he4_ti47__n_cr50)*Y(He4)*state.rho - screened_rates(k_he4_ti47__p_v50)*Y(He4)*state.rho + screened_rates(k_n_ti47__he4_ca44)*Y(N)*state.rho + screened_rates(k_p_ti47__he4_sc44)*Y(H1)*state.rho + screened_rates(k_ti47__he4_ca43);
    jac.set(He4, Ti47, scratch);

    scratch = -screened_rates(k_he4_ti48__cr52)*Y(He4)*state.rho - screened_rates(k_he4_ti48__n_cr51)*Y(He4)*state.rho - screened_rates(k_he4_ti48__p_v51)*Y(He4)*state.rho + screened_rates(k_n_ti48__he4_ca45)*Y(N)*state.rho + screened_rates(k_p_ti48__he4_sc45)*Y(H1)*state.rho + screened_rates(k_ti48__he4_ca44);
    jac.set(He4, Ti48, scratch);

    scratch = -screened_rates(k_he4_ti49__cr53)*Y(He4)*state.rho - screened_rates(k_he4_ti49__n_cr52)*Y(He4)*state.rho - screened_rates(k_he4_ti49__p_v52)*Y(He4)*state.rho + screened_rates(k_n_ti49__he4_ca46)*Y(N)*state.rho + screened_rates(k_p_ti49__he4_sc46)*Y(H1)*state.rho + screened_rates(k_ti49__he4_ca45);
    jac.set(He4, Ti49, scratch);

    scratch = -screened_rates(k_he4_ti50__cr54)*Y(He4)*state.rho - screened_rates(k_he4_ti50__n_cr53)*Y(He4)*state.rho + screened_rates(k_n_ti50__he4_ca47)*Y(N)*state.rho + screened_rates(k_p_ti50__he4_sc47)*Y(H1)*state.rho + screened_rates(k_ti50__he4_ca46);
    jac.set(He4, Ti50, scratch);

    scratch = -screened_rates(k_he4_ti51__n_cr54)*Y(He4)*state.rho + screened_rates(k_n_ti51__he4_ca48)*Y(N)*state.rho + screened_rates(k_p_ti51__he4_sc48)*Y(H1)*state.rho + screened_rates(k_ti51__he4_ca47);
    jac.set(He4, Ti51, scratch);

    scratch = -screened_rates(k_he4_v46__mn50)*Y(He4)*state.rho - screened_rates(k_he4_v46__p_cr49)*Y(He4)*state.rho + screened_rates(k_n_v46__he4_sc43)*Y(N)*state.rho;
    jac.set(He4, V46, scratch);

    scratch = -screened_rates(k_he4_v47__mn51)*Y(He4)*state.rho - screened_rates(k_he4_v47__n_mn50)*Y(He4)*state.rho - screened_rates(k_he4_v47__p_cr50)*Y(He4)*state.rho + screened_rates(k_n_v47__he4_sc44)*Y(N)*state.rho + screened_rates(k_p_v47__he4_ti44)*Y(H1)*state.rho + screened_rates(k_v47__he4_sc43);
    jac.set(He4, V47, scratch);

    scratch = -screened_rates(k_he4_v48__mn52)*Y(He4)*state.rho - screened_rates(k_he4_v48__n_mn51)*Y(He4)*state.rho - screened_rates(k_he4_v48__p_cr51)*Y(He4)*state.rho + screened_rates(k_n_v48__he4_sc45)*Y(N)*state.rho + screened_rates(k_p_v48__he4_ti45)*Y(H1)*state.rho + screened_rates(k_v48__he4_sc44);
    jac.set(He4, V48, scratch);

    scratch = -screened_rates(k_he4_v49__mn53)*Y(He4)*state.rho - screened_rates(k_he4_v49__n_mn52)*Y(He4)*state.rho - screened_rates(k_he4_v49__p_cr52)*Y(He4)*state.rho + screened_rates(k_n_v49__he4_sc46)*Y(N)*state.rho + screened_rates(k_p_v49__he4_ti46)*Y(H1)*state.rho + screened_rates(k_v49__he4_sc45);
    jac.set(He4, V49, scratch);

    scratch = -screened_rates(k_he4_v50__mn54)*Y(He4)*state.rho - screened_rates(k_he4_v50__n_mn53)*Y(He4)*state.rho - screened_rates(k_he4_v50__p_cr53)*Y(He4)*state.rho + screened_rates(k_n_v50__he4_sc47)*Y(N)*state.rho + screened_rates(k_p_v50__he4_ti47)*Y(H1)*state.rho + screened_rates(k_v50__he4_sc46);
    jac.set(He4, V50, scratch);

    scratch = -screened_rates(k_he4_v51__mn55)*Y(He4)*state.rho - screened_rates(k_he4_v51__n_mn54)*Y(He4)*state.rho - screened_rates(k_he4_v51__p_cr54)*Y(He4)*state.rho + screened_rates(k_n_v51__he4_sc48)*Y(N)*state.rho + screened_rates(k_p_v51__he4_ti48)*Y(H1)*state.rho + screened_rates(k_v51__he4_sc47);
    jac.set(He4, V51, scratch);

    scratch = -screened_rates(k_he4_v52__n_mn55)*Y(He4)*state.rho + screened_rates(k_n_v52__he4_sc49)*Y(N)*state.rho + screened_rates(k_p_v52__he4_ti49)*Y(H1)*state.rho + screened_rates(k_v52__he4_sc48);
    jac.set(He4, V52, scratch);

    scratch = screened_rates(k_cr48__he4_ti44) - screened_rates(k_he4_cr48__fe52)*Y(He4)*state.rho - screened_rates(k_he4_cr48__p_mn51)*Y(He4)*state.rho + screened_rates(k_n_cr48__he4_ti45)*Y(N)*state.rho;
    jac.set(He4, Cr48, scratch);

    scratch = screened_rates(k_cr49__he4_ti45) - screened_rates(k_he4_cr49__fe53)*Y(He4)*state.rho - screened_rates(k_he4_cr49__n_fe52)*Y(He4)*state.rho - screened_rates(k_he4_cr49__p_mn52)*Y(He4)*state.rho + screened_rates(k_n_cr49__he4_ti46)*Y(N)*state.rho + screened_rates(k_p_cr49__he4_v46)*Y(H1)*state.rho;
    jac.set(He4, Cr49, scratch);

    scratch = screened_rates(k_cr50__he4_ti46) - screened_rates(k_he4_cr50__fe54)*Y(He4)*state.rho - screened_rates(k_he4_cr50__n_fe53)*Y(He4)*state.rho - screened_rates(k_he4_cr50__p_mn53)*Y(He4)*state.rho + screened_rates(k_n_cr50__he4_ti47)*Y(N)*state.rho + screened_rates(k_p_cr50__he4_v47)*Y(H1)*state.rho;
    jac.set(He4, Cr50, scratch);

    scratch = screened_rates(k_cr51__he4_ti47) - screened_rates(k_he4_cr51__fe55)*Y(He4)*state.rho - screened_rates(k_he4_cr51__n_fe54)*Y(He4)*state.rho - screened_rates(k_he4_cr51__p_mn54)*Y(He4)*state.rho + screened_rates(k_n_cr51__he4_ti48)*Y(N)*state.rho + screened_rates(k_p_cr51__he4_v48)*Y(H1)*state.rho;
    jac.set(He4, Cr51, scratch);

    scratch = screened_rates(k_cr52__he4_ti48) - screened_rates(k_he4_cr52__fe56)*Y(He4)*state.rho - screened_rates(k_he4_cr52__n_fe55)*Y(He4)*state.rho - screened_rates(k_he4_cr52__p_mn55)*Y(He4)*state.rho + screened_rates(k_n_cr52__he4_ti49)*Y(N)*state.rho + screened_rates(k_p_cr52__he4_v49)*Y(H1)*state.rho;
    jac.set(He4, Cr52, scratch);

    scratch = screened_rates(k_cr53__he4_ti49) - screened_rates(k_he4_cr53__fe57)*Y(He4)*state.rho - screened_rates(k_he4_cr53__n_fe56)*Y(He4)*state.rho + screened_rates(k_n_cr53__he4_ti50)*Y(N)*state.rho + screened_rates(k_p_cr53__he4_v50)*Y(H1)*state.rho;
    jac.set(He4, Cr53, scratch);

    scratch = screened_rates(k_cr54__he4_ti50) - screened_rates(k_he4_cr54__fe58)*Y(He4)*state.rho - screened_rates(k_he4_cr54__n_fe57)*Y(He4)*state.rho + screened_rates(k_n_cr54__he4_ti51)*Y(N)*state.rho + screened_rates(k_p_cr54__he4_v51)*Y(H1)*state.rho;
    jac.set(He4, Cr54, scratch);

    scratch = -screened_rates(k_he4_mn50__co54)*Y(He4)*state.rho - screened_rates(k_he4_mn50__n_co53)*Y(He4)*state.rho - screened_rates(k_he4_mn50__p_fe53)*Y(He4)*state.rho + screened_rates(k_mn50__he4_v46) + screened_rates(k_n_mn50__he4_v47)*Y(N)*state.rho;
    jac.set(He4, Mn50, scratch);

    scratch = -screened_rates(k_he4_mn51__co55)*Y(He4)*state.rho - screened_rates(k_he4_mn51__n_co54)*Y(He4)*state.rho - screened_rates(k_he4_mn51__p_fe54)*Y(He4)*state.rho + screened_rates(k_mn51__he4_v47) + screened_rates(k_n_mn51__he4_v48)*Y(N)*state.rho + screened_rates(k_p_mn51__he4_cr48)*Y(H1)*state.rho;
    jac.set(He4, Mn51, scratch);

    scratch = -screened_rates(k_he4_mn52__co56)*Y(He4)*state.rho - screened_rates(k_he4_mn52__n_co55)*Y(He4)*state.rho - screened_rates(k_he4_mn52__p_fe55)*Y(He4)*state.rho + screened_rates(k_mn52__he4_v48) + screened_rates(k_n_mn52__he4_v49)*Y(N)*state.rho + screened_rates(k_p_mn52__he4_cr49)*Y(H1)*state.rho;
    jac.set(He4, Mn52, scratch);

    scratch = -screened_rates(k_he4_mn53__co57)*Y(He4)*state.rho - screened_rates(k_he4_mn53__n_co56)*Y(He4)*state.rho - screened_rates(k_he4_mn53__p_fe56)*Y(He4)*state.rho + screened_rates(k_mn53__he4_v49) + screened_rates(k_n_mn53__he4_v50)*Y(N)*state.rho + screened_rates(k_p_mn53__he4_cr50)*Y(H1)*state.rho;
    jac.set(He4, Mn53, scratch);

    scratch = -screened_rates(k_he4_mn54__co58)*Y(He4)*state.rho - screened_rates(k_he4_mn54__n_co57)*Y(He4)*state.rho - screened_rates(k_he4_mn54__p_fe57)*Y(He4)*state.rho + screened_rates(k_mn54__he4_v50) + screened_rates(k_n_mn54__he4_v51)*Y(N)*state.rho + screened_rates(k_p_mn54__he4_cr51)*Y(H1)*state.rho;
    jac.set(He4, Mn54, scratch);

    scratch = -screened_rates(k_he4_mn55__co59)*Y(He4)*state.rho - screened_rates(k_he4_mn55__n_co58)*Y(He4)*state.rho - screened_rates(k_he4_mn55__p_fe58)*Y(He4)*state.rho + screened_rates(k_mn55__he4_v51) + screened_rates(k_n_mn55__he4_v52)*Y(N)*state.rho + screened_rates(k_p_mn55__he4_cr52)*Y(H1)*state.rho;
    jac.set(He4, Mn55, scratch);

    scratch = screened_rates(k_fe52__he4_cr48) - screened_rates(k_he4_fe52__ni56)*Y(He4)*state.rho - screened_rates(k_he4_fe52__p_co55)*Y(He4)*state.rho + screened_rates(k_n_fe52__he4_cr49)*Y(N)*state.rho;
    jac.set(He4, Fe52, scratch);

    scratch = screened_rates(k_fe53__he4_cr49) - screened_rates(k_he4_fe53__n_ni56)*Y(He4)*state.rho - screened_rates(k_he4_fe53__ni57)*Y(He4)*state.rho - screened_rates(k_he4_fe53__p_co56)*Y(He4)*state.rho + screened_rates(k_n_fe53__he4_cr50)*Y(N)*state.rho + screened_rates(k_p_fe53__he4_mn50)*Y(H1)*state.rho;
    jac.set(He4, Fe53, scratch);

    scratch = screened_rates(k_fe54__he4_cr50) - screened_rates(k_he4_fe54__n_ni57)*Y(He4)*state.rho - screened_rates(k_he4_fe54__ni58)*Y(He4)*state.rho - screened_rates(k_he4_fe54__p_co57)*Y(He4)*state.rho + screened_rates(k_n_fe54__he4_cr51)*Y(N)*state.rho + screened_rates(k_p_fe54__he4_mn51)*Y(H1)*state.rho;
    jac.set(He4, Fe54, scratch);

    scratch = screened_rates(k_fe55__he4_cr51) - screened_rates(k_he4_fe55__n_ni58)*Y(He4)*state.rho - screened_rates(k_he4_fe55__ni59)*Y(He4)*state.rho - screened_rates(k_he4_fe55__p_co58)*Y(He4)*state.rho + screened_rates(k_n_fe55__he4_cr52)*Y(N)*state.rho + screened_rates(k_p_fe55__he4_mn52)*Y(H1)*state.rho;
    jac.set(He4, Fe55, scratch);

    scratch = screened_rates(k_fe56__he4_cr52) - screened_rates(k_he4_fe56__n_ni59)*Y(He4)*state.rho - screened_rates(k_he4_fe56__ni60)*Y(He4)*state.rho - screened_rates(k_he4_fe56__p_co59)*Y(He4)*state.rho + screened_rates(k_n_fe56__he4_cr53)*Y(N)*state.rho + screened_rates(k_p_fe56__he4_mn53)*Y(H1)*state.rho;
    jac.set(He4, Fe56, scratch);

    scratch = screened_rates(k_fe57__he4_cr53) - screened_rates(k_he4_fe57__n_ni60)*Y(He4)*state.rho - screened_rates(k_he4_fe57__ni61)*Y(He4)*state.rho + screened_rates(k_n_fe57__he4_cr54)*Y(N)*state.rho + screened_rates(k_p_fe57__he4_mn54)*Y(H1)*state.rho;
    jac.set(He4, Fe57, scratch);

    scratch = screened_rates(k_fe58__he4_cr54) - screened_rates(k_he4_fe58__n_ni61)*Y(He4)*state.rho - screened_rates(k_he4_fe58__ni62)*Y(He4)*state.rho + screened_rates(k_p_fe58__he4_mn55)*Y(H1)*state.rho;
    jac.set(He4, Fe58, scratch);

    scratch = -screened_rates(k_he4_co53__cu57)*Y(He4)*state.rho - screened_rates(k_he4_co53__p_ni56)*Y(He4)*state.rho + screened_rates(k_n_co53__he4_mn50)*Y(N)*state.rho;
    jac.set(He4, Co53, scratch);

    scratch = screened_rates(k_co54__he4_mn50) - screened_rates(k_he4_co54__cu58)*Y(He4)*state.rho - screened_rates(k_he4_co54__n_cu57)*Y(He4)*state.rho - screened_rates(k_he4_co54__p_ni57)*Y(He4)*state.rho + screened_rates(k_n_co54__he4_mn51)*Y(N)*state.rho;
    jac.set(He4, Co54, scratch);

    scratch = screened_rates(k_co55__he4_mn51) - screened_rates(k_he4_co55__cu59)*Y(He4)*state.rho - screened_rates(k_he4_co55__n_cu58)*Y(He4)*state.rho - screened_rates(k_he4_co55__p_ni58)*Y(He4)*state.rho + screened_rates(k_n_co55__he4_mn52)*Y(N)*state.rho + screened_rates(k_p_co55__he4_fe52)*Y(H1)*state.rho;
    jac.set(He4, Co55, scratch);

    scratch = screened_rates(k_co56__he4_mn52) - screened_rates(k_he4_co56__cu60)*Y(He4)*state.rho - screened_rates(k_he4_co56__n_cu59)*Y(He4)*state.rho - screened_rates(k_he4_co56__p_ni59)*Y(He4)*state.rho + screened_rates(k_n_co56__he4_mn53)*Y(N)*state.rho + screened_rates(k_p_co56__he4_fe53)*Y(H1)*state.rho;
    jac.set(He4, Co56, scratch);

    scratch = screened_rates(k_co57__he4_mn53) - screened_rates(k_he4_co57__cu61)*Y(He4)*state.rho - screened_rates(k_he4_co57__n_cu60)*Y(He4)*state.rho - screened_rates(k_he4_co57__p_ni60)*Y(He4)*state.rho + screened_rates(k_n_co57__he4_mn54)*Y(N)*state.rho + screened_rates(k_p_co57__he4_fe54)*Y(H1)*state.rho;
    jac.set(He4, Co57, scratch);

    scratch = screened_rates(k_co58__he4_mn54) - screened_rates(k_he4_co58__cu62)*Y(He4)*state.rho - screened_rates(k_he4_co58__n_cu61)*Y(He4)*state.rho - screened_rates(k_he4_co58__p_ni61)*Y(He4)*state.rho + screened_rates(k_n_co58__he4_mn55)*Y(N)*state.rho + screened_rates(k_p_co58__he4_fe55)*Y(H1)*state.rho;
    jac.set(He4, Co58, scratch);

    scratch = screened_rates(k_co59__he4_mn55) - screened_rates(k_he4_co59__cu63)*Y(He4)*state.rho - screened_rates(k_he4_co59__n_cu62)*Y(He4)*state.rho - screened_rates(k_he4_co59__p_ni62)*Y(He4)*state.rho + screened_rates(k_p_co59__he4_fe56)*Y(H1)*state.rho;
    jac.set(He4, Co59, scratch);

    scratch = -screened_rates(k_he4_ni56__n_zn59)*Y(He4)*state.rho - screened_rates(k_he4_ni56__p_cu59)*Y(He4)*state.rho - screened_rates(k_he4_ni56__zn60)*Y(He4)*state.rho + screened_rates(k_n_ni56__he4_fe53)*Y(N)*state.rho + screened_rates(k_ni56__he4_fe52) + screened_rates(k_p_ni56__he4_co53)*Y(H1)*state.rho;
    jac.set(He4, Ni56, scratch);

    scratch = -screened_rates(k_he4_ni57__n_zn60)*Y(He4)*state.rho - screened_rates(k_he4_ni57__p_cu60)*Y(He4)*state.rho - screened_rates(k_he4_ni57__zn61)*Y(He4)*state.rho + screened_rates(k_n_ni57__he4_fe54)*Y(N)*state.rho + screened_rates(k_ni57__he4_fe53) + screened_rates(k_p_ni57__he4_co54)*Y(H1)*state.rho;
    jac.set(He4, Ni57, scratch);

    scratch = -screened_rates(k_he4_ni58__n_zn61)*Y(He4)*state.rho - screened_rates(k_he4_ni58__p_cu61)*Y(He4)*state.rho - screened_rates(k_he4_ni58__zn62)*Y(He4)*state.rho + screened_rates(k_n_ni58__he4_fe55)*Y(N)*state.rho + screened_rates(k_ni58__he4_fe54) + screened_rates(k_p_ni58__he4_co55)*Y(H1)*state.rho;
    jac.set(He4, Ni58, scratch);

    scratch = -screened_rates(k_he4_ni59__n_zn62)*Y(He4)*state.rho - screened_rates(k_he4_ni59__p_cu62)*Y(He4)*state.rho - screened_rates(k_he4_ni59__zn63)*Y(He4)*state.rho + screened_rates(k_n_ni59__he4_fe56)*Y(N)*state.rho + screened_rates(k_ni59__he4_fe55) + screened_rates(k_p_ni59__he4_co56)*Y(H1)*state.rho;
    jac.set(He4, Ni59, scratch);

    scratch = -screened_rates(k_he4_ni60__n_zn63)*Y(He4)*state.rho - screened_rates(k_he4_ni60__p_cu63)*Y(He4)*state.rho - screened_rates(k_he4_ni60__zn64)*Y(He4)*state.rho + screened_rates(k_n_ni60__he4_fe57)*Y(N)*state.rho + screened_rates(k_ni60__he4_fe56) + screened_rates(k_p_ni60__he4_co57)*Y(H1)*state.rho;
    jac.set(He4, Ni60, scratch);

    scratch = -screened_rates(k_he4_ni61__n_zn64)*Y(He4)*state.rho - screened_rates(k_he4_ni61__p_cu64)*Y(He4)*state.rho - screened_rates(k_he4_ni61__zn65)*Y(He4)*state.rho + screened_rates(k_n_ni61__he4_fe58)*Y(N)*state.rho + screened_rates(k_ni61__he4_fe57) + screened_rates(k_p_ni61__he4_co58)*Y(H1)*state.rho;
    jac.set(He4, Ni61, scratch);

    scratch = -screened_rates(k_he4_ni62__n_zn65)*Y(He4)*state.rho - screened_rates(k_he4_ni62__p_cu65)*Y(He4)*state.rho - screened_rates(k_he4_ni62__zn66)*Y(He4)*state.rho + screened_rates(k_ni62__he4_fe58) + screened_rates(k_p_ni62__he4_co59)*Y(H1)*state.rho;
    jac.set(He4, Ni62, scratch);

    scratch = -screened_rates(k_he4_ni63__n_zn66)*Y(He4)*state.rho;
    jac.set(He4, Ni63, scratch);

    scratch = screened_rates(k_cu57__he4_co53) - screened_rates(k_he4_cu57__p_zn60)*Y(He4)*state.rho + screened_rates(k_n_cu57__he4_co54)*Y(N)*state.rho;
    jac.set(He4, Cu57, scratch);

    scratch = screened_rates(k_cu58__he4_co54) - screened_rates(k_he4_cu58__ga62)*Y(He4)*state.rho - screened_rates(k_he4_cu58__p_zn61)*Y(He4)*state.rho + screened_rates(k_n_cu58__he4_co55)*Y(N)*state.rho;
    jac.set(He4, Cu58, scratch);

    scratch = screened_rates(k_cu59__he4_co55) - screened_rates(k_he4_cu59__ga63)*Y(He4)*state.rho - screened_rates(k_he4_cu59__n_ga62)*Y(He4)*state.rho - screened_rates(k_he4_cu59__p_zn62)*Y(He4)*state.rho + screened_rates(k_n_cu59__he4_co56)*Y(N)*state.rho + screened_rates(k_p_cu59__he4_ni56)*Y(H1)*state.rho;
    jac.set(He4, Cu59, scratch);

    scratch = screened_rates(k_cu60__he4_co56) - screened_rates(k_he4_cu60__ga64)*Y(He4)*state.rho - screened_rates(k_he4_cu60__n_ga63)*Y(He4)*state.rho - screened_rates(k_he4_cu60__p_zn63)*Y(He4)*state.rho + screened_rates(k_n_cu60__he4_co57)*Y(N)*state.rho + screened_rates(k_p_cu60__he4_ni57)*Y(H1)*state.rho;
    jac.set(He4, Cu60, scratch);

    scratch = screened_rates(k_cu61__he4_co57) - screened_rates(k_he4_cu61__n_ga64)*Y(He4)*state.rho - screened_rates(k_he4_cu61__p_zn64)*Y(He4)*state.rho + screened_rates(k_n_cu61__he4_co58)*Y(N)*state.rho + screened_rates(k_p_cu61__he4_ni58)*Y(H1)*state.rho;
    jac.set(He4, Cu61, scratch);

    scratch = screened_rates(k_cu62__he4_co58) - screened_rates(k_he4_cu62__p_zn65)*Y(He4)*state.rho + screened_rates(k_n_cu62__he4_co59)*Y(N)*state.rho + screened_rates(k_p_cu62__he4_ni59)*Y(H1)*state.rho;
    jac.set(He4, Cu62, scratch);

    scratch = screened_rates(k_cu63__he4_co59) - screened_rates(k_he4_cu63__p_zn66)*Y(He4)*state.rho + screened_rates(k_p_cu63__he4_ni60)*Y(H1)*state.rho;
    jac.set(He4, Cu63, scratch);

    scratch = screened_rates(k_p_cu64__he4_ni61)*Y(H1)*state.rho;
    jac.set(He4, Cu64, scratch);

    scratch = screened_rates(k_p_cu65__he4_ni62)*Y(H1)*state.rho;
    jac.set(He4, Cu65, scratch);

    scratch = -screened_rates(k_he4_zn59__ge63)*Y(He4)*state.rho - screened_rates(k_he4_zn59__p_ga62)*Y(He4)*state.rho + screened_rates(k_n_zn59__he4_ni56)*Y(N)*state.rho;
    jac.set(He4, Zn59, scratch);

    scratch = -screened_rates(k_he4_zn60__ge64)*Y(He4)*state.rho - screened_rates(k_he4_zn60__n_ge63)*Y(He4)*state.rho - screened_rates(k_he4_zn60__p_ga63)*Y(He4)*state.rho + screened_rates(k_n_zn60__he4_ni57)*Y(N)*state.rho + screened_rates(k_p_zn60__he4_cu57)*Y(H1)*state.rho + screened_rates(k_zn60__he4_ni56);
    jac.set(He4, Zn60, scratch);

    scratch = -screened_rates(k_he4_zn61__n_ge64)*Y(He4)*state.rho - screened_rates(k_he4_zn61__p_ga64)*Y(He4)*state.rho + screened_rates(k_n_zn61__he4_ni58)*Y(N)*state.rho + screened_rates(k_p_zn61__he4_cu58)*Y(H1)*state.rho + screened_rates(k_zn61__he4_ni57);
    jac.set(He4, Zn61, scratch);

    scratch = screened_rates(k_n_zn62__he4_ni59)*Y(N)*state.rho + screened_rates(k_p_zn62__he4_cu59)*Y(H1)*state.rho + screened_rates(k_zn62__he4_ni58);
    jac.set(He4, Zn62, scratch);

    scratch = screened_rates(k_n_zn63__he4_ni60)*Y(N)*state.rho + screened_rates(k_p_zn63__he4_cu60)*Y(H1)*state.rho + screened_rates(k_zn63__he4_ni59);
    jac.set(He4, Zn63, scratch);

    scratch = screened_rates(k_n_zn64__he4_ni61)*Y(N)*state.rho + screened_rates(k_p_zn64__he4_cu61)*Y(H1)*state.rho + screened_rates(k_zn64__he4_ni60);
    jac.set(He4, Zn64, scratch);

    scratch = screened_rates(k_n_zn65__he4_ni62)*Y(N)*state.rho + screened_rates(k_p_zn65__he4_cu62)*Y(H1)*state.rho + screened_rates(k_zn65__he4_ni61);
    jac.set(He4, Zn65, scratch);

    scratch = screened_rates(k_n_zn66__he4_ni63)*Y(N)*state.rho + screened_rates(k_p_zn66__he4_cu63)*Y(H1)*state.rho + screened_rates(k_zn66__he4_ni62);
    jac.set(He4, Zn66, scratch);

    scratch = screened_rates(k_ga62__he4_cu58) + screened_rates(k_n_ga62__he4_cu59)*Y(N)*state.rho + screened_rates(k_p_ga62__he4_zn59)*Y(H1)*state.rho;
    jac.set(He4, Ga62, scratch);

    scratch = screened_rates(k_ga63__he4_cu59) + screened_rates(k_n_ga63__he4_cu60)*Y(N)*state.rho + screened_rates(k_p_ga63__he4_zn60)*Y(H1)*state.rho;
    jac.set(He4, Ga63, scratch);

    scratch = screened_rates(k_ga64__he4_cu60) + screened_rates(k_n_ga64__he4_cu61)*Y(N)*state.rho + screened_rates(k_p_ga64__he4_zn61)*Y(H1)*state.rho;
    jac.set(He4, Ga64, scratch);

    scratch = screened_rates(k_ge63__he4_zn59) + screened_rates(k_n_ge63__he4_zn60)*Y(N)*state.rho;
    jac.set(He4, Ge63, scratch);

    scratch = screened_rates(k_ge64__he4_zn60) + screened_rates(k_n_ge64__he4_zn61)*Y(N)*state.rho;
    jac.set(He4, Ge64, scratch);

    scratch = screened_rates(k_n_be7__d_li6)*Y(Be7)*state.rho - screened_rates(k_n_li6__li7)*Y(Li6)*state.rho + screened_rates(k_n_p_he4__li6)*Y(He4)*Y(H1)*std::pow(state.rho, 2);
    jac.set(Li6, N, scratch);

    scratch = screened_rates(k_n_p_he4__li6)*Y(He4)*Y(N)*std::pow(state.rho, 2) + screened_rates(k_p_be9__he4_li6)*Y(Be9)*state.rho - screened_rates(k_p_li6__be7)*Y(Li6)*state.rho - screened_rates(k_p_li6__he4_he3)*Y(Li6)*state.rho + screened_rates(k_p_li7__d_li6)*Y(Li7)*state.rho;
    jac.set(Li6, H1, scratch);

    scratch = -screened_rates(k_d_li6__n_be7)*Y(Li6)*state.rho - screened_rates(k_d_li6__p_li7)*Y(Li6)*state.rho + screened_rates(k_he4_d__li6)*Y(He4)*state.rho;
    jac.set(Li6, H2, scratch);

    scratch = screened_rates(k_he4_he3__p_li6)*Y(He4)*state.rho;
    jac.set(Li6, He3, scratch);

    scratch = screened_rates(k_he4_d__li6)*Y(H2)*state.rho + screened_rates(k_he4_he3__p_li6)*Y(He3)*state.rho - screened_rates(k_he4_li6__b10)*Y(Li6)*state.rho - screened_rates(k_he4_li6__p_be9)*Y(Li6)*state.rho + screened_rates(k_n_p_he4__li6)*Y(N)*Y(H1)*std::pow(state.rho, 2);
    jac.set(Li6, He4, scratch);

    scratch = -screened_rates(k_d_li6__n_be7)*Y(H2)*state.rho - screened_rates(k_d_li6__p_li7)*Y(H2)*state.rho - screened_rates(k_he4_li6__b10)*Y(He4)*state.rho - screened_rates(k_he4_li6__p_be9)*Y(He4)*state.rho - screened_rates(k_li6__he4_d) - screened_rates(k_li6__n_p_he4) - screened_rates(k_n_li6__li7)*Y(N)*state.rho - screened_rates(k_p_li6__be7)*Y(H1)*state.rho - screened_rates(k_p_li6__he4_he3)*Y(H1)*state.rho;
    jac.set(Li6, Li6, scratch);

    scratch = screened_rates(k_li7__n_li6) + screened_rates(k_p_li7__d_li6)*Y(H1)*state.rho;
    jac.set(Li6, Li7, scratch);

    scratch = screened_rates(k_be7__p_li6) + screened_rates(k_n_be7__d_li6)*Y(N)*state.rho;
    jac.set(Li6, Be7, scratch);

    scratch = screened_rates(k_p_be9__he4_li6)*Y(H1)*state.rho;
    jac.set(Li6, Be9, scratch);

    scratch = screened_rates(k_b10__he4_li6);
    jac.set(Li6, B10, scratch);

    scratch = screened_rates(k_n_b10__he4_li7)*Y(B10)*state.rho + screened_rates(k_n_be7__p_li7)*Y(Be7)*state.rho + 0.5*screened_rates(k_n_he4_he4__d_li7)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) + screened_rates(k_n_li6__li7)*Y(Li6)*state.rho + 0.5*screened_rates(k_n_p_he4_he4__he3_li7)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 3);
    jac.set(Li7, N, scratch);

    scratch = 0.5*screened_rates(k_n_p_he4_he4__he3_li7)*std::pow(Y(He4), 2)*Y(N)*std::pow(state.rho, 3) - screened_rates(k_p_li7__d_li6)*Y(Li7)*state.rho - screened_rates(k_p_li7__he4_he4)*Y(Li7)*state.rho - screened_rates(k_p_li7__n_be7)*Y(Li7)*state.rho;
    jac.set(Li7, H1, scratch);

    scratch = screened_rates(k_d_li6__p_li7)*Y(Li6)*state.rho - screened_rates(k_d_li7__n_he4_he4)*Y(Li7)*state.rho;
    jac.set(Li7, H2, scratch);

    scratch = -screened_rates(k_he3_li7__n_p_he4_he4)*Y(Li7)*state.rho;
    jac.set(Li7, He3, scratch);

    scratch = 1.0*screened_rates(k_he4_he4__p_li7)*Y(He4)*state.rho - screened_rates(k_he4_li7__b11)*Y(Li7)*state.rho - screened_rates(k_he4_li7__n_b10)*Y(Li7)*state.rho + 1.0*screened_rates(k_n_he4_he4__d_li7)*Y(He4)*Y(N)*std::pow(state.rho, 2) + 1.0*screened_rates(k_n_p_he4_he4__he3_li7)*Y(He4)*Y(N)*Y(H1)*std::pow(state.rho, 3);
    jac.set(Li7, He4, scratch);

    scratch = screened_rates(k_d_li6__p_li7)*Y(H2)*state.rho + screened_rates(k_n_li6__li7)*Y(N)*state.rho;
    jac.set(Li7, Li6, scratch);

    scratch = -screened_rates(k_d_li7__n_he4_he4)*Y(H2)*state.rho - screened_rates(k_he3_li7__n_p_he4_he4)*Y(He3)*state.rho - screened_rates(k_he4_li7__b11)*Y(He4)*state.rho - screened_rates(k_he4_li7__n_b10)*Y(He4)*state.rho - screened_rates(k_li7__n_li6) - screened_rates(k_p_li7__d_li6)*Y(H1)*state.rho - screened_rates(k_p_li7__he4_he4)*Y(H1)*state.rho - screened_rates(k_p_li7__n_be7)*Y(H1)*state.rho;
    jac.set(Li7, Li7, scratch);

    scratch = screened_rates(k_be7__li7__weak__electron_capture)*state.rho*state.y_e + screened_rates(k_n_be7__p_li7)*Y(N)*state.rho;
    jac.set(Li7, Be7, scratch);

    scratch = screened_rates(k_n_b10__he4_li7)*Y(N)*state.rho;
    jac.set(Li7, B10, scratch);

    scratch = screened_rates(k_b11__he4_li7);
    jac.set(Li7, B11, scratch);

    scratch = -screened_rates(k_n_be7__d_li6)*Y(Be7)*state.rho - screened_rates(k_n_be7__he4_he4)*Y(Be7)*state.rho - screened_rates(k_n_be7__p_li7)*Y(Be7)*state.rho;
    jac.set(Be7, N, scratch);

    scratch = screened_rates(k_p_b10__he4_be7)*Y(B10)*state.rho - screened_rates(k_p_be7__b8)*Y(Be7)*state.rho + 0.5*screened_rates(k_p_he4_he4__d_be7)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) + screened_rates(k_p_li6__be7)*Y(Li6)*state.rho + screened_rates(k_p_li7__n_be7)*Y(Li7)*state.rho + 0.5*screened_rates(k_p_p_he4_he4__he3_be7)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 3);
    jac.set(Be7, H1, scratch);

    scratch = -screened_rates(k_d_be7__p_he4_he4)*Y(Be7)*state.rho + screened_rates(k_d_li6__n_be7)*Y(Li6)*state.rho;
    jac.set(Be7, H2, scratch);

    scratch = -screened_rates(k_he3_be7__p_p_he4_he4)*Y(Be7)*state.rho + screened_rates(k_he4_he3__be7)*Y(He4)*state.rho;
    jac.set(Be7, He3, scratch);

    scratch = -screened_rates(k_he4_be7__p_b10)*Y(Be7)*state.rho + screened_rates(k_he4_he3__be7)*Y(He3)*state.rho + 1.0*screened_rates(k_he4_he4__n_be7)*Y(He4)*state.rho + 1.0*screened_rates(k_p_he4_he4__d_be7)*Y(He4)*Y(H1)*std::pow(state.rho, 2) + 0.5*screened_rates(k_p_p_he4_he4__he3_be7)*Y(He4)*std::pow(Y(H1), 2)*std::pow(state.rho, 3);
    jac.set(Be7, He4, scratch);

    scratch = screened_rates(k_d_li6__n_be7)*Y(H2)*state.rho + screened_rates(k_p_li6__be7)*Y(H1)*state.rho;
    jac.set(Be7, Li6, scratch);

    scratch = screened_rates(k_p_li7__n_be7)*Y(H1)*state.rho;
    jac.set(Be7, Li7, scratch);

    scratch = -screened_rates(k_be7__he4_he3) - screened_rates(k_be7__li7__weak__electron_capture)*state.rho*state.y_e - screened_rates(k_be7__p_li6) - screened_rates(k_d_be7__p_he4_he4)*Y(H2)*state.rho - screened_rates(k_he3_be7__p_p_he4_he4)*Y(He3)*state.rho - screened_rates(k_he4_be7__p_b10)*Y(He4)*state.rho - screened_rates(k_n_be7__d_li6)*Y(N)*state.rho - screened_rates(k_n_be7__he4_he4)*Y(N)*state.rho - screened_rates(k_n_be7__p_li7)*Y(N)*state.rho - screened_rates(k_p_be7__b8)*Y(H1)*state.rho;
    jac.set(Be7, Be7, scratch);

    scratch = screened_rates(k_b8__p_be7);
    jac.set(Be7, B8, scratch);

    scratch = screened_rates(k_p_b10__he4_be7)*Y(H1)*state.rho;
    jac.set(Be7, B10, scratch);

    scratch = screened_rates(k_n_c12__he4_be9)*Y(C12)*state.rho + 0.5*screened_rates(k_n_he4_he4__be9)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) + 0.5*screened_rates(k_n_p_he4_he4__p_be9)*std::pow(Y(He4), 2)*Y(H1)*std::pow(state.rho, 3);
    jac.set(Be9, N, scratch);

    scratch = 0.5*screened_rates(k_n_p_he4_he4__p_be9)*std::pow(Y(He4), 2)*Y(N)*std::pow(state.rho, 3) - screened_rates(k_p_be9__b10)*Y(Be9)*state.rho - screened_rates(k_p_be9__d_he4_he4)*Y(Be9)*state.rho - screened_rates(k_p_be9__he4_li6)*Y(Be9)*state.rho - screened_rates(k_p_be9__n_p_he4_he4)*Y(Be9)*state.rho;
    jac.set(Be9, H1, scratch);

    scratch = 0.5*screened_rates(k_d_he4_he4__p_be9)*std::pow(Y(He4), 2)*std::pow(state.rho, 2);
    jac.set(Be9, H2, scratch);

    scratch = 1.0*screened_rates(k_d_he4_he4__p_be9)*Y(H2)*Y(He4)*std::pow(state.rho, 2) - screened_rates(k_he4_be9__n_c12)*Y(Be9)*state.rho + screened_rates(k_he4_li6__p_be9)*Y(Li6)*state.rho + 1.0*screened_rates(k_n_he4_he4__be9)*Y(He4)*Y(N)*std::pow(state.rho, 2) + 1.0*screened_rates(k_n_p_he4_he4__p_be9)*Y(He4)*Y(N)*Y(H1)*std::pow(state.rho, 3);
    jac.set(Be9, He4, scratch);

    scratch = screened_rates(k_he4_li6__p_be9)*Y(He4)*state.rho;
    jac.set(Be9, Li6, scratch);

    scratch = -screened_rates(k_be9__n_he4_he4) - screened_rates(k_he4_be9__n_c12)*Y(He4)*state.rho - screened_rates(k_p_be9__b10)*Y(H1)*state.rho - screened_rates(k_p_be9__d_he4_he4)*Y(H1)*state.rho - screened_rates(k_p_be9__he4_li6)*Y(H1)*state.rho - screened_rates(k_p_be9__n_p_he4_he4)*Y(H1)*state.rho;
    jac.set(Be9, Be9, scratch);

    scratch = screened_rates(k_b10__p_be9);
    jac.set(Be9, B10, scratch);

    scratch = screened_rates(k_n_c12__he4_be9)*Y(N)*state.rho;
    jac.set(Be9, C12, scratch);

    scratch = -screened_rates(k_n_b8__p_he4_he4)*Y(B8)*state.rho;
    jac.set(B8, N, scratch);

    scratch = screened_rates(k_p_be7__b8)*Y(Be7)*state.rho + 0.5*screened_rates(k_p_he4_he4__n_b8)*std::pow(Y(He4), 2)*std::pow(state.rho, 2);
    jac.set(B8, H1, scratch);

    scratch = 1.0*screened_rates(k_p_he4_he4__n_b8)*Y(He4)*Y(H1)*std::pow(state.rho, 2);
    jac.set(B8, He4, scratch);

    scratch = screened_rates(k_p_be7__b8)*Y(H1)*state.rho;
    jac.set(B8, Be7, scratch);

    scratch = -screened_rates(k_b8__he4_he4__weak__wc12) - screened_rates(k_b8__p_be7) - screened_rates(k_n_b8__p_he4_he4)*Y(N)*state.rho;
    jac.set(B8, B8, scratch);

    scratch = -screened_rates(k_n_b10__b11)*Y(B10)*state.rho - screened_rates(k_n_b10__he4_li7)*Y(B10)*state.rho + screened_rates(k_n_n13__he4_b10)*Y(N13)*state.rho;
    jac.set(B10, N, scratch);

    scratch = -screened_rates(k_p_b10__he4_be7)*Y(B10)*state.rho + screened_rates(k_p_be9__b10)*Y(Be9)*state.rho + screened_rates(k_p_c13__he4_b10)*Y(C13)*state.rho;
    jac.set(B10, H1, scratch);

    scratch = -screened_rates(k_he4_b10__n_n13)*Y(B10)*state.rho - screened_rates(k_he4_b10__p_c13)*Y(B10)*state.rho + screened_rates(k_he4_be7__p_b10)*Y(Be7)*state.rho + screened_rates(k_he4_li6__b10)*Y(Li6)*state.rho + screened_rates(k_he4_li7__n_b10)*Y(Li7)*state.rho;
    jac.set(B10, He4, scratch);

    scratch = screened_rates(k_he4_li6__b10)*Y(He4)*state.rho;
    jac.set(B10, Li6, scratch);

    scratch = screened_rates(k_he4_li7__n_b10)*Y(He4)*state.rho;
    jac.set(B10, Li7, scratch);

    scratch = screened_rates(k_he4_be7__p_b10)*Y(He4)*state.rho;
    jac.set(B10, Be7, scratch);

    scratch = screened_rates(k_p_be9__b10)*Y(H1)*state.rho;
    jac.set(B10, Be9, scratch);

    scratch = -screened_rates(k_b10__he4_li6) - screened_rates(k_b10__p_be9) - screened_rates(k_he4_b10__n_n13)*Y(He4)*state.rho - screened_rates(k_he4_b10__p_c13)*Y(He4)*state.rho - screened_rates(k_n_b10__b11)*Y(N)*state.rho - screened_rates(k_n_b10__he4_li7)*Y(N)*state.rho - screened_rates(k_p_b10__he4_be7)*Y(H1)*state.rho;
    jac.set(B10, B10, scratch);

    scratch = screened_rates(k_b11__n_b10);
    jac.set(B10, B11, scratch);

    scratch = screened_rates(k_p_c13__he4_b10)*Y(H1)*state.rho;
    jac.set(B10, C13, scratch);

    scratch = screened_rates(k_n_n13__he4_b10)*Y(N)*state.rho;
    jac.set(B10, N13, scratch);

    scratch = screened_rates(k_n_b10__b11)*Y(B10)*state.rho + screened_rates(k_n_n14__he4_b11)*Y(N14)*state.rho;
    jac.set(B11, N, scratch);

    scratch = -screened_rates(k_p_b11__c12)*Y(B11)*state.rho - screened_rates(k_p_b11__he4_he4_he4)*Y(B11)*state.rho + screened_rates(k_p_c14__he4_b11)*Y(C14)*state.rho;
    jac.set(B11, H1, scratch);

    scratch = -screened_rates(k_he4_b11__n_n14)*Y(B11)*state.rho - screened_rates(k_he4_b11__p_c14)*Y(B11)*state.rho + 0.5*screened_rates(k_he4_he4_he4__p_b11)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) + screened_rates(k_he4_li7__b11)*Y(Li7)*state.rho;
    jac.set(B11, He4, scratch);

    scratch = screened_rates(k_he4_li7__b11)*Y(He4)*state.rho;
    jac.set(B11, Li7, scratch);

    scratch = screened_rates(k_n_b10__b11)*Y(N)*state.rho;
    jac.set(B11, B10, scratch);

    scratch = -screened_rates(k_b11__he4_li7) - screened_rates(k_b11__n_b10) - screened_rates(k_he4_b11__n_n14)*Y(He4)*state.rho - screened_rates(k_he4_b11__p_c14)*Y(He4)*state.rho - screened_rates(k_p_b11__c12)*Y(H1)*state.rho - screened_rates(k_p_b11__he4_he4_he4)*Y(H1)*state.rho;
    jac.set(B11, B11, scratch);

    scratch = screened_rates(k_c12__p_b11);
    jac.set(B11, C12, scratch);

    scratch = screened_rates(k_p_c14__he4_b11)*Y(H1)*state.rho;
    jac.set(B11, C14, scratch);

    scratch = screened_rates(k_n_n14__he4_b11)*Y(N)*state.rho;
    jac.set(B11, N14, scratch);

    scratch = -screened_rates(k_n_c12__c13)*Y(C12)*state.rho - screened_rates(k_n_c12__he4_be9)*Y(C12)*state.rho + 2.0*screened_rates(k_n_mg23__c12_c12)*Y(Mg23)*state.rho + screened_rates(k_n_o15__he4_c12)*Y(O15)*state.rho;
    jac.set(C12, N, scratch);

    scratch = screened_rates(k_p_al27__c12_o16)*Y(Al27)*state.rho + screened_rates(k_p_b11__c12)*Y(B11)*state.rho - screened_rates(k_p_c12__n13)*Y(C12)*state.rho + screened_rates(k_p_n15__he4_c12)*Y(N15)*state.rho + 2.0*screened_rates(k_p_na23__c12_c12)*Y(Na23)*state.rho + screened_rates(k_p_p31__c12_ne20)*Y(P31)*state.rho;
    jac.set(C12, H1, scratch);

    scratch = screened_rates(k_he4_be9__n_c12)*Y(Be9)*state.rho - screened_rates(k_he4_c12__n_o15)*Y(C12)*state.rho - screened_rates(k_he4_c12__o16)*Y(C12)*state.rho - screened_rates(k_he4_c12__p_n15)*Y(C12)*state.rho + 0.5*screened_rates(k_he4_he4_he4__c12)*std::pow(Y(He4), 2)*std::pow(state.rho, 2) + screened_rates(k_he4_mg24__c12_o16)*Y(Mg24)*state.rho + 2.0*screened_rates(k_he4_ne20__c12_c12)*Y(Ne20)*state.rho + screened_rates(k_he4_si28__c12_ne20)*Y(Si28)*state.rho;
    jac.set(C12, He4, scratch);

    scratch = screened_rates(k_he4_be9__n_c12)*Y(He4)*state.rho;
    jac.set(C12, Be9, scratch);

    scratch = screened_rates(k_p_b11__c12)*Y(H1)*state.rho;
    jac.set(C12, B11, scratch);

    scratch = -screened_rates(k_c12__he4_he4_he4) - screened_rates(k_c12__p_b11) - 2.0*screened_rates(k_c12_c12__he4_ne20)*Y(C12)*state.rho - 2.0*screened_rates(k_c12_c12__n_mg23)*Y(C12)*state.rho - 2.0*screened_rates(k_c12_c12__p_na23)*Y(C12)*state.rho - screened_rates(k_c12_ne20__he4_si28)*Y(Ne20)*state.rho - screened_rates(k_c12_ne20__p_p31)*Y(Ne20)*state.rho - screened_rates(k_c12_o16__he4_mg24)*Y(O16)*state.rho - screened_rates(k_c12_o16__p_al27)*Y(O16)*state.rho - screened_rates(k_he4_c12__n_o15)*Y(He4)*state.rho - screened_rates(k_he4_c12__o16)*Y(He4)*state.rho - screened_rates(k_he4_c12__p_n15)*Y(He4)*state.rho - screened_rates(k_n_c12__c13)*Y(N)*state.rho - screened_rates(k_n_c12__he4_be9)*Y(N)*state.rho - screened_rates(k_p_c12__n13)*Y(H1)*state.rho;
    jac.set(C12, C12, scratch);

    scratch = screened_rates(k_c13__n_c12);
    jac.set(C12, C13, scratch);

    scratch = screened_rates(k_n13__p_c12);
    jac.set(C12, N13, scratch);

    scratch = screened_rates(k_p_n15__he4_c12)*Y(H1)*state.rho;
    jac.set(C12, N15, scratch);

    scratch = screened_rates(k_n_o15__he4_c12)*Y(N)*state.rho;
    jac.set(C12, O15, scratch);

    scratch = -screened_rates(k_c12_o16__he4_mg24)*Y(C12)*state.rho - screened_rates(k_c12_o16__p_al27)*Y(C12)*state.rho + screened_rates(k_o16__he4_c12);
    jac.set(C12, O16, scratch);

    scratch = -screened_rates(k_c12_ne20__he4_si28)*Y(C12)*state.rho - screened_rates(k_c12_ne20__p_p31)*Y(C12)*state.rho + 2.0*screened_rates(k_he4_ne20__c12_c12)*Y(He4)*state.rho;
    jac.set(C12, Ne20, scratch);

    scratch = 2.0*screened_rates(k_p_na23__c12_c12)*Y(H1)*state.rho;
    jac.set(C12, Na23, scratch);

    scratch = 2.0*screened_rates(k_n_mg23__c12_c12)*Y(N)*state.rho;
    jac.set(C12, Mg23, scratch);

    scratch = screened_rates(k_he4_mg24__c12_o16)*Y(He4)*state.rho;
    jac.set(C12, Mg24, scratch);

    scratch = screened_rates(k_p_al27__c12_o16)*Y(H1)*state.rho;
    jac.set(C12, Al27, scratch);

    scratch = screened_rates(k_he4_si28__c12_ne20)*Y(He4)*state.rho;
    jac.set(C12, Si28, scratch);

    scratch = screened_rates(k_p_p31__c12_ne20)*Y(H1)*state.rho;
    jac.set(C12, P31, scratch);

    scratch = screened_rates(k_n_c12__c13)*Y(C12)*state.rho - screened_rates(k_n_c13__c14)*Y(C13)*state.rho + screened_rates(k_n_n13__p_c13)*Y(N13)*state.rho + screened_rates(k_n_n14__d_c13)*Y(N14)*state.rho + screened_rates(k_n_o16__he4_c13)*Y(O16)*state.rho;
    jac.set(C13, N, scratch);

    scratch = -screened_rates(k_p_c13__he4_b10)*Y(C13)*state.rho - screened_rates(k_p_c13__n14)*Y(C13)*state.rho - screened_rates(k_p_c13__n_n13)*Y(C13)*state.rho;
    jac.set(C13, H1, scratch);

    scratch = -screened_rates(k_d_c13__n_n14)*Y(C13)*state.rho;
    jac.set(C13, H2, scratch);

    scratch = screened_rates(k_he4_b10__p_c13)*Y(B10)*state.rho - screened_rates(k_he4_c13__n_o16)*Y(C13)*state.rho;
    jac.set(C13, He4, scratch);

    scratch = screened_rates(k_he4_b10__p_c13)*Y(He4)*state.rho;
    jac.set(C13, B10, scratch);

    scratch = screened_rates(k_n_c12__c13)*Y(N)*state.rho;
    jac.set(C13, C12, scratch);

    scratch = -screened_rates(k_c13__n_c12) - screened_rates(k_d_c13__n_n14)*Y(H2)*state.rho - screened_rates(k_he4_c13__n_o16)*Y(He4)*state.rho - screened_rates(k_n_c13__c14)*Y(N)*state.rho - screened_rates(k_p_c13__he4_b10)*Y(H1)*state.rho - screened_rates(k_p_c13__n14)*Y(H1)*state.rho - screened_rates(k_p_c13__n_n13)*Y(H1)*state.rho;
    jac.set(C13, C13, scratch);

    scratch = screened_rates(k_c14__n_c13);
    jac.set(C13, C14, scratch);

    scratch = screened_rates(k_n13__c13__weak__wc12) + screened_rates(k_n_n13__p_c13)*Y(N)*state.rho;
    jac.set(C13, N13, scratch);

    scratch = screened_rates(k_n14__p_c13) + screened_rates(k_n_n14__d_c13)*Y(N)*state.rho;
    jac.set(C13, N14, scratch);

    scratch = screened_rates(k_n_o16__he4_c13)*Y(N)*state.rho;
    jac.set(C13, O16, scratch);

    scratch = screened_rates(k_n_c13__c14)*Y(C13)*state.rho + screened_rates(k_n_n14__p_c14)*Y(N14)*state.rho + screened_rates(k_n_n15__d_c14)*Y(N15)*state.rho + screened_rates(k_n_o17__he4_c14)*Y(O17)*state.rho;
    jac.set(C14, N, scratch);

    scratch = -screened_rates(k_p_c14__he4_b11)*Y(C14)*state.rho - screened_rates(k_p_c14__n15)*Y(C14)*state.rho - screened_rates(k_p_c14__n_n14)*Y(C14)*state.rho;
    jac.set(C14, H1, scratch);

    scratch = -screened_rates(k_d_c14__n_n15)*Y(C14)*state.rho;
    jac.set(C14, H2, scratch);

    scratch = screened_rates(k_he4_b11__p_c14)*Y(B11)*state.rho - screened_rates(k_he4_c14__n_o17)*Y(C14)*state.rho - screened_rates(k_he4_c14__o18)*Y(C14)*state.rho;
    jac.set(C14, He4, scratch);

    scratch = screened_rates(k_he4_b11__p_c14)*Y(He4)*state.rho;
    jac.set(C14, B11, scratch);

    scratch = screened_rates(k_n_c13__c14)*Y(N)*state.rho;
    jac.set(C14, C13, scratch);

    scratch = -screened_rates(k_c14__n14__weak__wc12) - screened_rates(k_c14__n_c13) - screened_rates(k_d_c14__n_n15)*Y(H2)*state.rho - screened_rates(k_he4_c14__n_o17)*Y(He4)*state.rho - screened_rates(k_he4_c14__o18)*Y(He4)*state.rho - screened_rates(k_p_c14__he4_b11)*Y(H1)*state.rho - screened_rates(k_p_c14__n15)*Y(H1)*state.rho - screened_rates(k_p_c14__n_n14)*Y(H1)*state.rho;
    jac.set(C14, C14, scratch);

    scratch = screened_rates(k_n_n14__p_c14)*Y(N)*state.rho;
    jac.set(C14, N14, scratch);

    scratch = screened_rates(k_n15__p_c14) + screened_rates(k_n_n15__d_c14)*Y(N)*state.rho;
    jac.set(C14, N15, scratch);

    scratch = screened_rates(k_n_o17__he4_c14)*Y(N)*state.rho;
    jac.set(C14, O17, scratch);

    scratch = screened_rates(k_o18__he4_c14);
    jac.set(C14, O18, scratch);

    scratch = -screened_rates(k_n_n13__he4_b10)*Y(N13)*state.rho - screened_rates(k_n_n13__n14)*Y(N13)*state.rho - screened_rates(k_n_n13__p_c13)*Y(N13)*state.rho;
    jac.set(N13, N, scratch);

    scratch = screened_rates(k_p_c12__n13)*Y(C12)*state.rho + screened_rates(k_p_c13__n_n13)*Y(C13)*state.rho - screened_rates(k_p_n13__o14)*Y(N13)*state.rho + screened_rates(k_p_o16__he4_n13)*Y(O16)*state.rho;
    jac.set(N13, H1, scratch);

    scratch = screened_rates(k_he4_b10__n_n13)*Y(B10)*state.rho - screened_rates(k_he4_n13__p_o16)*Y(N13)*state.rho;
    jac.set(N13, He4, scratch);

    scratch = screened_rates(k_he4_b10__n_n13)*Y(He4)*state.rho;
    jac.set(N13, B10, scratch);

    scratch = screened_rates(k_p_c12__n13)*Y(H1)*state.rho;
    jac.set(N13, C12, scratch);

    scratch = screened_rates(k_p_c13__n_n13)*Y(H1)*state.rho;
    jac.set(N13, C13, scratch);

    scratch = -screened_rates(k_he4_n13__p_o16)*Y(He4)*state.rho - screened_rates(k_n13__c13__weak__wc12) - screened_rates(k_n13__p_c12) - screened_rates(k_n_n13__he4_b10)*Y(N)*state.rho - screened_rates(k_n_n13__n14)*Y(N)*state.rho - screened_rates(k_n_n13__p_c13)*Y(N)*state.rho - screened_rates(k_p_n13__o14)*Y(H1)*state.rho;
    jac.set(N13, N13, scratch);

    scratch = screened_rates(k_n14__n_n13);
    jac.set(N13, N14, scratch);

    scratch = screened_rates(k_o14__p_n13);
    jac.set(N13, O14, scratch);

    scratch = screened_rates(k_p_o16__he4_n13)*Y(H1)*state.rho;
    jac.set(N13, O16, scratch);

    scratch = screened_rates(k_n_f17__he4_n14)*Y(F17)*state.rho + screened_rates(k_n_n13__n14)*Y(N13)*state.rho - screened_rates(k_n_n14__d_c13)*Y(N14)*state.rho - screened_rates(k_n_n14__he4_b11)*Y(N14)*state.rho - screened_rates(k_n_n14__n15)*Y(N14)*state.rho - screened_rates(k_n_n14__p_c14)*Y(N14)*state.rho + screened_rates(k_n_o14__p_n14)*Y(O14)*state.rho;
    jac.set(N14, N, scratch);

    scratch = screened_rates(k_p_c13__n14)*Y(C13)*state.rho + screened_rates(k_p_c14__n_n14)*Y(C14)*state.rho - screened_rates(k_p_n14__n_o14)*Y(N14)*state.rho - screened_rates(k_p_n14__o15)*Y(N14)*state.rho + screened_rates(k_p_o17__he4_n14)*Y(O17)*state.rho;
    jac.set(N14, H1, scratch);

    scratch = screened_rates(k_d_c13__n_n14)*Y(C13)*state.rho;
    jac.set(N14, H2, scratch);

    scratch = screened_rates(k_he4_b11__n_n14)*Y(B11)*state.rho - screened_rates(k_he4_n14__f18)*Y(N14)*state.rho - screened_rates(k_he4_n14__n_f17)*Y(N14)*state.rho - screened_rates(k_he4_n14__p_o17)*Y(N14)*state.rho;
    jac.set(N14, He4, scratch);

    scratch = screened_rates(k_he4_b11__n_n14)*Y(He4)*state.rho;
    jac.set(N14, B11, scratch);

    scratch = screened_rates(k_d_c13__n_n14)*Y(H2)*state.rho + screened_rates(k_p_c13__n14)*Y(H1)*state.rho;
    jac.set(N14, C13, scratch);

    scratch = screened_rates(k_c14__n14__weak__wc12) + screened_rates(k_p_c14__n_n14)*Y(H1)*state.rho;
    jac.set(N14, C14, scratch);

    scratch = screened_rates(k_n_n13__n14)*Y(N)*state.rho;
    jac.set(N14, N13, scratch);

    scratch = -screened_rates(k_he4_n14__f18)*Y(He4)*state.rho - screened_rates(k_he4_n14__n_f17)*Y(He4)*state.rho - screened_rates(k_he4_n14__p_o17)*Y(He4)*state.rho - screened_rates(k_n14__n_n13) - screened_rates(k_n14__p_c13) - screened_rates(k_n_n14__d_c13)*Y(N)*state.rho - screened_rates(k_n_n14__he4_b11)*Y(N)*state.rho - screened_rates(k_n_n14__n15)*Y(N)*state.rho - screened_rates(k_n_n14__p_c14)*Y(N)*state.rho - screened_rates(k_p_n14__n_o14)*Y(H1)*state.rho - screened_rates(k_p_n14__o15)*Y(H1)*state.rho;
    jac.set(N14, N14, scratch);

    scratch = screened_rates(k_n15__n_n14);
    jac.set(N14, N15, scratch);

    scratch = screened_rates(k_n_o14__p_n14)*Y(N)*state.rho + screened_rates(k_o14__n14__weak__wc12);
    jac.set(N14, O14, scratch);

    scratch = screened_rates(k_o15__p_n14);
    jac.set(N14, O15, scratch);

    scratch = screened_rates(k_p_o17__he4_n14)*Y(H1)*state.rho;
    jac.set(N14, O17, scratch);

    scratch = screened_rates(k_n_f17__he4_n14)*Y(N)*state.rho;
    jac.set(N14, F17, scratch);

    scratch = screened_rates(k_f18__he4_n14);
    jac.set(N14, F18, scratch);

    scratch = screened_rates(k_n_f18__he4_n15)*Y(F18)*state.rho + screened_rates(k_n_n14__n15)*Y(N14)*state.rho - screened_rates(k_n_n15__d_c14)*Y(N15)*state.rho + screened_rates(k_n_o15__p_n15)*Y(O15)*state.rho;
    jac.set(N15, N, scratch);

    scratch = screened_rates(k_p_c14__n15)*Y(C14)*state.rho - screened_rates(k_p_n15__he4_c12)*Y(N15)*state.rho - screened_rates(k_p_n15__n_o15)*Y(N15)*state.rho - screened_rates(k_p_n15__o16)*Y(N15)*state.rho + screened_rates(k_p_o18__he4_n15)*Y(O18)*state.rho;
    jac.set(N15, H1, scratch);

    scratch = screened_rates(k_d_c14__n_n15)*Y(C14)*state.rho;
    jac.set(N15, H2, scratch);

    scratch = screened_rates(k_he4_c12__p_n15)*Y(C12)*state.rho - screened_rates(k_he4_n15__f19)*Y(N15)*state.rho - screened_rates(k_he4_n15__n_f18)*Y(N15)*state.rho - screened_rates(k_he4_n15__p_o18)*Y(N15)*state.rho;
    jac.set(N15, He4, scratch);

    scratch = screened_rates(k_he4_c12__p_n15)*Y(He4)*state.rho;
    jac.set(N15, C12, scratch);

    scratch = screened_rates(k_d_c14__n_n15)*Y(H2)*state.rho + screened_rates(k_p_c14__n15)*Y(H1)*state.rho;
    jac.set(N15, C14, scratch);

    scratch = screened_rates(k_n_n14__n15)*Y(N)*state.rho;
    jac.set(N15, N14, scratch);

    scratch = -screened_rates(k_he4_n15__f19)*Y(He4)*state.rho - screened_rates(k_he4_n15__n_f18)*Y(He4)*state.rho - screened_rates(k_he4_n15__p_o18)*Y(He4)*state.rho - screened_rates(k_n15__n_n14) - screened_rates(k_n15__p_c14) - screened_rates(k_n_n15__d_c14)*Y(N)*state.rho - screened_rates(k_p_n15__he4_c12)*Y(H1)*state.rho - screened_rates(k_p_n15__n_o15)*Y(H1)*state.rho - screened_rates(k_p_n15__o16)*Y(H1)*state.rho;
    jac.set(N15, N15, scratch);

    scratch = screened_rates(k_n_o15__p_n15)*Y(N)*state.rho + screened_rates(k_o15__n15__weak__wc12);
    jac.set(N15, O15, scratch);

    scratch = screened_rates(k_o16__p_n15);
    jac.set(N15, O16, scratch);

    scratch = screened_rates(k_p_o18__he4_n15)*Y(H1)*state.rho;
    jac.set(N15, O18, scratch);

    scratch = screened_rates(k_n_f18__he4_n15)*Y(N)*state.rho;
    jac.set(N15, F18, scratch);

    scratch = screened_rates(k_f19__he4_n15);
    jac.set(N15, F19, scratch);

    scratch = -screened_rates(k_n_o14__o15)*Y(O14)*state.rho - screened_rates(k_n_o14__p_n14)*Y(O14)*state.rho;
    jac.set(O14, N, scratch);

    scratch = screened_rates(k_p_f17__he4_o14)*Y(F17)*state.rho + screened_rates(k_p_n13__o14)*Y(N13)*state.rho + screened_rates(k_p_n14__n_o14)*Y(N14)*state.rho;
    jac.set(O14, H1, scratch);

    scratch = -screened_rates(k_he4_o14__ne18)*Y(O14)*state.rho - screened_rates(k_he4_o14__p_f17)*Y(O14)*state.rho;
    jac.set(O14, He4, scratch);

    scratch = screened_rates(k_p_n13__o14)*Y(H1)*state.rho;
    jac.set(O14, N13, scratch);

    scratch = screened_rates(k_p_n14__n_o14)*Y(H1)*state.rho;
    jac.set(O14, N14, scratch);

    scratch = -screened_rates(k_he4_o14__ne18)*Y(He4)*state.rho - screened_rates(k_he4_o14__p_f17)*Y(He4)*state.rho - screened_rates(k_n_o14__o15)*Y(N)*state.rho - screened_rates(k_n_o14__p_n14)*Y(N)*state.rho - screened_rates(k_o14__n14__weak__wc12) - screened_rates(k_o14__p_n13);
    jac.set(O14, O14, scratch);

    scratch = screened_rates(k_o15__n_o14);
    jac.set(O14, O15, scratch);

    scratch = screened_rates(k_p_f17__he4_o14)*Y(H1)*state.rho;
    jac.set(O14, F17, scratch);

    scratch = screened_rates(k_ne18__he4_o14);
    jac.set(O14, Ne18, scratch);

    scratch = screened_rates(k_n_ne18__he4_o15)*Y(Ne18)*state.rho + screened_rates(k_n_o14__o15)*Y(O14)*state.rho - screened_rates(k_n_o15__he4_c12)*Y(O15)*state.rho - screened_rates(k_n_o15__o16)*Y(O15)*state.rho - screened_rates(k_n_o15__p_n15)*Y(O15)*state.rho;
    jac.set(O15, N, scratch);

    scratch = screened_rates(k_p_f18__he4_o15)*Y(F18)*state.rho + screened_rates(k_p_n14__o15)*Y(N14)*state.rho + screened_rates(k_p_n15__n_o15)*Y(N15)*state.rho;
    jac.set(O15, H1, scratch);

    scratch = screened_rates(k_he4_c12__n_o15)*Y(C12)*state.rho - screened_rates(k_he4_o15__n_ne18)*Y(O15)*state.rho - screened_rates(k_he4_o15__ne19)*Y(O15)*state.rho - screened_rates(k_he4_o15__p_f18)*Y(O15)*state.rho;
    jac.set(O15, He4, scratch);

    scratch = screened_rates(k_he4_c12__n_o15)*Y(He4)*state.rho;
    jac.set(O15, C12, scratch);

    scratch = screened_rates(k_p_n14__o15)*Y(H1)*state.rho;
    jac.set(O15, N14, scratch);

    scratch = screened_rates(k_p_n15__n_o15)*Y(H1)*state.rho;
    jac.set(O15, N15, scratch);

    scratch = screened_rates(k_n_o14__o15)*Y(N)*state.rho;
    jac.set(O15, O14, scratch);

    scratch = -screened_rates(k_he4_o15__n_ne18)*Y(He4)*state.rho - screened_rates(k_he4_o15__ne19)*Y(He4)*state.rho - screened_rates(k_he4_o15__p_f18)*Y(He4)*state.rho - screened_rates(k_n_o15__he4_c12)*Y(N)*state.rho - screened_rates(k_n_o15__o16)*Y(N)*state.rho - screened_rates(k_n_o15__p_n15)*Y(N)*state.rho - screened_rates(k_o15__n15__weak__wc12) - screened_rates(k_o15__n_o14) - screened_rates(k_o15__p_n14);
    jac.set(O15, O15, scratch);

    scratch = screened_rates(k_o16__n_o15);
    jac.set(O15, O16, scratch);

    scratch = screened_rates(k_p_f18__he4_o15)*Y(H1)*state.rho;
    jac.set(O15, F18, scratch);

    scratch = screened_rates(k_n_ne18__he4_o15)*Y(N)*state.rho;
    jac.set(O15, Ne18, scratch);

    scratch = screened_rates(k_ne19__he4_o15);
    jac.set(O15, Ne19, scratch);

    scratch = screened_rates(k_n_ne19__he4_o16)*Y(Ne19)*state.rho + screened_rates(k_n_o15__o16)*Y(O15)*state.rho - screened_rates(k_n_o16__he4_c13)*Y(O16)*state.rho - screened_rates(k_n_o16__o17)*Y(O16)*state.rho;
    jac.set(O16, N, scratch);

    scratch = screened_rates(k_p_al27__c12_o16)*Y(Al27)*state.rho + screened_rates(k_p_f19__he4_o16)*Y(F19)*state.rho + screened_rates(k_p_n15__o16)*Y(N15)*state.rho - screened_rates(k_p_o16__f17)*Y(O16)*state.rho - screened_rates(k_p_o16__he4_n13)*Y(O16)*state.rho + 2.0*screened_rates(k_p_p31__o16_o16)*Y(P31)*state.rho;
    jac.set(O16, H1, scratch);

    scratch = screened_rates(k_he4_c12__o16)*Y(C12)*state.rho + screened_rates(k_he4_c13__n_o16)*Y(C13)*state.rho + screened_rates(k_he4_mg24__c12_o16)*Y(Mg24)*state.rho + screened_rates(k_he4_n13__p_o16)*Y(N13)*state.rho - screened_rates(k_he4_o16__n_ne19)*Y(O16)*state.rho - screened_rates(k_he4_o16__ne20)*Y(O16)*state.rho - screened_rates(k_he4_o16__p_f19)*Y(O16)*state.rho + 2.0*screened_rates(k_he4_si28__o16_o16)*Y(Si28)*state.rho;
    jac.set(O16, He4, scratch);

    scratch = -screened_rates(k_c12_o16__he4_mg24)*Y(O16)*state.rho - screened_rates(k_c12_o16__p_al27)*Y(O16)*state.rho + screened_rates(k_he4_c12__o16)*Y(He4)*state.rho;
    jac.set(O16, C12, scratch);

    scratch = screened_rates(k_he4_c13__n_o16)*Y(He4)*state.rho;
    jac.set(O16, C13, scratch);

    scratch = screened_rates(k_he4_n13__p_o16)*Y(He4)*state.rho;
    jac.set(O16, N13, scratch);

    scratch = screened_rates(k_p_n15__o16)*Y(H1)*state.rho;
    jac.set(O16, N15, scratch);

    scratch = screened_rates(k_n_o15__o16)*Y(N)*state.rho;
    jac.set(O16, O15, scratch);

    scratch = -screened_rates(k_c12_o16__he4_mg24)*Y(C12)*state.rho - screened_rates(k_c12_o16__p_al27)*Y(C12)*state.rho - screened_rates(k_he4_o16__n_ne19)*Y(He4)*state.rho - screened_rates(k_he4_o16__ne20)*Y(He4)*state.rho - screened_rates(k_he4_o16__p_f19)*Y(He4)*state.rho - screened_rates(k_n_o16__he4_c13)*Y(N)*state.rho - screened_rates(k_n_o16__o17)*Y(N)*state.rho - screened_rates(k_o16__he4_c12) - screened_rates(k_o16__n_o15) - screened_rates(k_o16__p_n15) - 2.0*screened_rates(k_o16_o16__he4_si28)*Y(O16)*state.rho - 2.0*screened_rates(k_o16_o16__p_p31)*Y(O16)*state.rho - screened_rates(k_p_o16__f17)*Y(H1)*state.rho - screened_rates(k_p_o16__he4_n13)*Y(H1)*state.rho;
    jac.set(O16, O16, scratch);

    scratch = screened_rates(k_o17__n_o16);
    jac.set(O16, O17, scratch);

    scratch = screened_rates(k_f17__p_o16);
    jac.set(O16, F17, scratch);

    scratch = screened_rates(k_p_f19__he4_o16)*Y(H1)*state.rho;
    jac.set(O16, F19, scratch);

    scratch = screened_rates(k_n_ne19__he4_o16)*Y(N)*state.rho;
    jac.set(O16, Ne19, scratch);

    scratch = screened_rates(k_ne20__he4_o16);
    jac.set(O16, Ne20, scratch);

    scratch = screened_rates(k_he4_mg24__c12_o16)*Y(He4)*state.rho;
    jac.set(O16, Mg24, scratch);

    scratch = screened_rates(k_p_al27__c12_o16)*Y(H1)*state.rho;
    jac.set(O16, Al27, scratch);

    scratch = 2.0*screened_rates(k_he4_si28__o16_o16)*Y(He4)*state.rho;
    jac.set(O16, Si28, scratch);

    scratch = 2.0*screened_rates(k_p_p31__o16_o16)*Y(H1)*state.rho;
    jac.set(O16, P31, scratch);

    scratch = screened_rates(k_n_f17__p_o17)*Y(F17)*state.rho + screened_rates(k_n_ne20__he4_o17)*Y(Ne20)*state.rho + screened_rates(k_n_o16__o17)*Y(O16)*state.rho - screened_rates(k_n_o17__he4_c14)*Y(O17)*state.rho - screened_rates(k_n_o17__o18)*Y(O17)*state.rho;
    jac.set(O17, N, scratch);

    scratch = -screened_rates(k_p_o17__f18)*Y(O17)*state.rho - screened_rates(k_p_o17__he4_n14)*Y(O17)*state.rho - screened_rates(k_p_o17__n_f17)*Y(O17)*state.rho;
    jac.set(O17, H1, scratch);

    scratch = screened_rates(k_he4_c14__n_o17)*Y(C14)*state.rho + screened_rates(k_he4_n14__p_o17)*Y(N14)*state.rho - screened_rates(k_he4_o17__n_ne20)*Y(O17)*state.rho - screened_rates(k_he4_o17__ne21)*Y(O17)*state.rho;
    jac.set(O17, He4, scratch);

    scratch = screened_rates(k_he4_c14__n_o17)*Y(He4)*state.rho;
    jac.set(O17, C14, scratch);

    scratch = screened_rates(k_he4_n14__p_o17)*Y(He4)*state.rho;
    jac.set(O17, N14, scratch);

    scratch = screened_rates(k_n_o16__o17)*Y(N)*state.rho;
    jac.set(O17, O16, scratch);

    scratch = -screened_rates(k_he4_o17__n_ne20)*Y(He4)*state.rho - screened_rates(k_he4_o17__ne21)*Y(He4)*state.rho - screened_rates(k_n_o17__he4_c14)*Y(N)*state.rho - screened_rates(k_n_o17__o18)*Y(N)*state.rho - screened_rates(k_o17__n_o16) - screened_rates(k_p_o17__f18)*Y(H1)*state.rho - screened_rates(k_p_o17__he4_n14)*Y(H1)*state.rho - screened_rates(k_p_o17__n_f17)*Y(H1)*state.rho;
    jac.set(O17, O17, scratch);

    scratch = screened_rates(k_o18__n_o17);
    jac.set(O17, O18, scratch);

    scratch = screened_rates(k_f17__o17__weak__wc12) + screened_rates(k_n_f17__p_o17)*Y(N)*state.rho;
    jac.set(O17, F17, scratch);

    scratch = screened_rates(k_f18__p_o17);
    jac.set(O17, F18, scratch);

    scratch = screened_rates(k_n_ne20__he4_o17)*Y(N)*state.rho;
    jac.set(O17, Ne20, scratch);

    scratch = screened_rates(k_ne21__he4_o17);
    jac.set(O17, Ne21, scratch);

    scratch = screened_rates(k_n_f18__p_o18)*Y(F18)*state.rho + screened_rates(k_n_ne21__he4_o18)*Y(Ne21)*state.rho + screened_rates(k_n_o17__o18)*Y(O17)*state.rho;
    jac.set(O18, N, scratch);

    scratch = -screened_rates(k_p_o18__f19)*Y(O18)*state.rho - screened_rates(k_p_o18__he4_n15)*Y(O18)*state.rho - screened_rates(k_p_o18__n_f18)*Y(O18)*state.rho;
    jac.set(O18, H1, scratch);

    scratch = screened_rates(k_he4_c14__o18)*Y(C14)*state.rho + screened_rates(k_he4_n15__p_o18)*Y(N15)*state.rho - screened_rates(k_he4_o18__n_ne21)*Y(O18)*state.rho - screened_rates(k_he4_o18__ne22)*Y(O18)*state.rho;
    jac.set(O18, He4, scratch);

    scratch = screened_rates(k_he4_c14__o18)*Y(He4)*state.rho;
    jac.set(O18, C14, scratch);

    scratch = screened_rates(k_he4_n15__p_o18)*Y(He4)*state.rho;
    jac.set(O18, N15, scratch);

    scratch = screened_rates(k_n_o17__o18)*Y(N)*state.rho;
    jac.set(O18, O17, scratch);

    scratch = -screened_rates(k_he4_o18__n_ne21)*Y(He4)*state.rho - screened_rates(k_he4_o18__ne22)*Y(He4)*state.rho - screened_rates(k_o18__he4_c14) - screened_rates(k_o18__n_o17) - screened_rates(k_p_o18__f19)*Y(H1)*state.rho - screened_rates(k_p_o18__he4_n15)*Y(H1)*state.rho - screened_rates(k_p_o18__n_f18)*Y(H1)*state.rho;
    jac.set(O18, O18, scratch);

    scratch = screened_rates(k_f18__o18__weak__wc12) + screened_rates(k_n_f18__p_o18)*Y(N)*state.rho;
    jac.set(O18, F18, scratch);

    scratch = screened_rates(k_f19__p_o18);
    jac.set(O18, F19, scratch);

    scratch = screened_rates(k_n_ne21__he4_o18)*Y(N)*state.rho;
    jac.set(O18, Ne21, scratch);

    scratch = screened_rates(k_ne22__he4_o18);
    jac.set(O18, Ne22, scratch);

    scratch = -screened_rates(k_n_f17__f18)*Y(F17)*state.rho - screened_rates(k_n_f17__he4_n14)*Y(F17)*state.rho - screened_rates(k_n_f17__p_o17)*Y(F17)*state.rho;
    jac.set(F17, N, scratch);

    scratch = -screened_rates(k_p_f17__he4_o14)*Y(F17)*state.rho - screened_rates(k_p_f17__ne18)*Y(F17)*state.rho + screened_rates(k_p_ne20__he4_f17)*Y(Ne20)*state.rho + screened_rates(k_p_o16__f17)*Y(O16)*state.rho + screened_rates(k_p_o17__n_f17)*Y(O17)*state.rho;
    jac.set(F17, H1, scratch);

    scratch = -screened_rates(k_he4_f17__na21)*Y(F17)*state.rho - screened_rates(k_he4_f17__p_ne20)*Y(F17)*state.rho + screened_rates(k_he4_n14__n_f17)*Y(N14)*state.rho + screened_rates(k_he4_o14__p_f17)*Y(O14)*state.rho;
    jac.set(F17, He4, scratch);

    scratch = screened_rates(k_he4_n14__n_f17)*Y(He4)*state.rho;
    jac.set(F17, N14, scratch);

    scratch = screened_rates(k_he4_o14__p_f17)*Y(He4)*state.rho;
    jac.set(F17, O14, scratch);

    scratch = screened_rates(k_p_o16__f17)*Y(H1)*state.rho;
    jac.set(F17, O16, scratch);

    scratch = screened_rates(k_p_o17__n_f17)*Y(H1)*state.rho;
    jac.set(F17, O17, scratch);

    scratch = -screened_rates(k_f17__o17__weak__wc12) - screened_rates(k_f17__p_o16) - screened_rates(k_he4_f17__na21)*Y(He4)*state.rho - screened_rates(k_he4_f17__p_ne20)*Y(He4)*state.rho - screened_rates(k_n_f17__f18)*Y(N)*state.rho - screened_rates(k_n_f17__he4_n14)*Y(N)*state.rho - screened_rates(k_n_f17__p_o17)*Y(N)*state.rho - screened_rates(k_p_f17__he4_o14)*Y(H1)*state.rho - screened_rates(k_p_f17__ne18)*Y(H1)*state.rho;
    jac.set(F17, F17, scratch);

    scratch = screened_rates(k_f18__n_f17);
    jac.set(F17, F18, scratch);

    scratch = screened_rates(k_ne18__p_f17);
    jac.set(F17, Ne18, scratch);

    scratch = screened_rates(k_p_ne20__he4_f17)*Y(H1)*state.rho;
    jac.set(F17, Ne20, scratch);

    scratch = screened_rates(k_na21__he4_f17);
    jac.set(F17, Na21, scratch);

    scratch = screened_rates(k_n_f17__f18)*Y(F17)*state.rho - screened_rates(k_n_f18__f19)*Y(F18)*state.rho - screened_rates(k_n_f18__he4_n15)*Y(F18)*state.rho - screened_rates(k_n_f18__p_o18)*Y(F18)*state.rho + screened_rates(k_n_na21__he4_f18)*Y(Na21)*state.rho + screened_rates(k_n_ne18__p_f18)*Y(Ne18)*state.rho;
    jac.set(F18, N, scratch);

    scratch = -screened_rates(k_p_f18__he4_o15)*Y(F18)*state.rho - screened_rates(k_p_f18__n_ne18)*Y(F18)*state.rho - screened_rates(k_p_f18__ne19)*Y(F18)*state.rho + screened_rates(k_p_ne21__he4_f18)*Y(Ne21)*state.rho + screened_rates(k_p_o17__f18)*Y(O17)*state.rho + screened_rates(k_p_o18__n_f18)*Y(O18)*state.rho;
    jac.set(F18, H1, scratch);

    scratch = -screened_rates(k_he4_f18__n_na21)*Y(F18)*state.rho - screened_rates(k_he4_f18__na22)*Y(F18)*state.rho - screened_rates(k_he4_f18__p_ne21)*Y(F18)*state.rho + screened_rates(k_he4_n14__f18)*Y(N14)*state.rho + screened_rates(k_he4_n15__n_f18)*Y(N15)*state.rho + screened_rates(k_he4_o15__p_f18)*Y(O15)*state.rho;
    jac.set(F18, He4, scratch);

    scratch = screened_rates(k_he4_n14__f18)*Y(He4)*state.rho;
    jac.set(F18, N14, scratch);

    scratch = screened_rates(k_he4_n15__n_f18)*Y(He4)*state.rho;
    jac.set(F18, N15, scratch);

    scratch = screened_rates(k_he4_o15__p_f18)*Y(He4)*state.rho;
    jac.set(F18, O15, scratch);

    scratch = screened_rates(k_p_o17__f18)*Y(H1)*state.rho;
    jac.set(F18, O17, scratch);

    scratch = screened_rates(k_p_o18__n_f18)*Y(H1)*state.rho;
    jac.set(F18, O18, scratch);

    scratch = screened_rates(k_n_f17__f18)*Y(N)*state.rho;
    jac.set(F18, F17, scratch);

    scratch = -screened_rates(k_f18__he4_n14) - screened_rates(k_f18__n_f17) - screened_rates(k_f18__o18__weak__wc12) - screened_rates(k_f18__p_o17) - screened_rates(k_he4_f18__n_na21)*Y(He4)*state.rho - screened_rates(k_he4_f18__na22)*Y(He4)*state.rho - screened_rates(k_he4_f18__p_ne21)*Y(He4)*state.rho - screened_rates(k_n_f18__f19)*Y(N)*state.rho - screened_rates(k_n_f18__he4_n15)*Y(N)*state.rho - screened_rates(k_n_f18__p_o18)*Y(N)*state.rho - screened_rates(k_p_f18__he4_o15)*Y(H1)*state.rho - screened_rates(k_p_f18__n_ne18)*Y(H1)*state.rho - screened_rates(k_p_f18__ne19)*Y(H1)*state.rho;
    jac.set(F18, F18, scratch);

    scratch = screened_rates(k_f19__n_f18);
    jac.set(F18, F19, scratch);

    scratch = screened_rates(k_n_ne18__p_f18)*Y(N)*state.rho + screened_rates(k_ne18__f18__weak__wc12);
    jac.set(F18, Ne18, scratch);

    scratch = screened_rates(k_ne19__p_f18);
    jac.set(F18, Ne19, scratch);

    scratch = screened_rates(k_p_ne21__he4_f18)*Y(H1)*state.rho;
    jac.set(F18, Ne21, scratch);

    scratch = screened_rates(k_n_na21__he4_f18)*Y(N)*state.rho;
    jac.set(F18, Na21, scratch);

    scratch = screened_rates(k_na22__he4_f18);
    jac.set(F18, Na22, scratch);

    scratch = screened_rates(k_n_f18__f19)*Y(F18)*state.rho + screened_rates(k_n_na22__he4_f19)*Y(Na22)*state.rho + screened_rates(k_n_ne19__p_f19)*Y(Ne19)*state.rho;
    jac.set(F19, N, scratch);

    scratch = -screened_rates(k_p_f19__he4_o16)*Y(F19)*state.rho - screened_rates(k_p_f19__n_ne19)*Y(F19)*state.rho - screened_rates(k_p_f19__ne20)*Y(F19)*state.rho + screened_rates(k_p_ne22__he4_f19)*Y(Ne22)*state.rho + screened_rates(k_p_o18__f19)*Y(O18)*state.rho;
    jac.set(F19, H1, scratch);

    scratch = -screened_rates(k_he4_f19__n_na22)*Y(F19)*state.rho - screened_rates(k_he4_f19__na23)*Y(F19)*state.rho - screened_rates(k_he4_f19__p_ne22)*Y(F19)*state.rho + screened_rates(k_he4_n15__f19)*Y(N15)*state.rho + screened_rates(k_he4_o16__p_f19)*Y(O16)*state.rho;
    jac.set(F19, He4, scratch);

    scratch = screened_rates(k_he4_n15__f19)*Y(He4)*state.rho;
    jac.set(F19, N15, scratch);

    scratch = screened_rates(k_he4_o16__p_f19)*Y(He4)*state.rho;
    jac.set(F19, O16, scratch);

    scratch = screened_rates(k_p_o18__f19)*Y(H1)*state.rho;
    jac.set(F19, O18, scratch);

    scratch = screened_rates(k_n_f18__f19)*Y(N)*state.rho;
    jac.set(F19, F18, scratch);

    scratch = -screened_rates(k_f19__he4_n15) - screened_rates(k_f19__n_f18) - screened_rates(k_f19__p_o18) - screened_rates(k_he4_f19__n_na22)*Y(He4)*state.rho - screened_rates(k_he4_f19__na23)*Y(He4)*state.rho - screened_rates(k_he4_f19__p_ne22)*Y(He4)*state.rho - screened_rates(k_p_f19__he4_o16)*Y(H1)*state.rho - screened_rates(k_p_f19__n_ne19)*Y(H1)*state.rho - screened_rates(k_p_f19__ne20)*Y(H1)*state.rho;
    jac.set(F19, F19, scratch);

    scratch = screened_rates(k_n_ne19__p_f19)*Y(N)*state.rho + screened_rates(k_ne19__f19__weak__wc12);
    jac.set(F19, Ne19, scratch);

    scratch = screened_rates(k_ne20__p_f19);
    jac.set(F19, Ne20, scratch);

    scratch = screened_rates(k_p_ne22__he4_f19)*Y(H1)*state.rho;
    jac.set(F19, Ne22, scratch);

    scratch = screened_rates(k_n_na22__he4_f19)*Y(N)*state.rho;
    jac.set(F19, Na22, scratch);

    scratch = screened_rates(k_na23__he4_f19);
    jac.set(F19, Na23, scratch);

    scratch = -screened_rates(k_n_ne18__he4_o15)*Y(Ne18)*state.rho - screened_rates(k_n_ne18__ne19)*Y(Ne18)*state.rho - screened_rates(k_n_ne18__p_f18)*Y(Ne18)*state.rho;
    jac.set(Ne18, N, scratch);

    scratch = screened_rates(k_p_f17__ne18)*Y(F17)*state.rho + screened_rates(k_p_f18__n_ne18)*Y(F18)*state.rho + screened_rates(k_p_na21__he4_ne18)*Y(Na21)*state.rho;
    jac.set(Ne18, H1, scratch);

    scratch = -screened_rates(k_he4_ne18__p_na21)*Y(Ne18)*state.rho + screened_rates(k_he4_o14__ne18)*Y(O14)*state.rho + screened_rates(k_he4_o15__n_ne18)*Y(O15)*state.rho;
    jac.set(Ne18, He4, scratch);

    scratch = screened_rates(k_he4_o14__ne18)*Y(He4)*state.rho;
    jac.set(Ne18, O14, scratch);

    scratch = screened_rates(k_he4_o15__n_ne18)*Y(He4)*state.rho;
    jac.set(Ne18, O15, scratch);

    scratch = screened_rates(k_p_f17__ne18)*Y(H1)*state.rho;
    jac.set(Ne18, F17, scratch);

    scratch = screened_rates(k_p_f18__n_ne18)*Y(H1)*state.rho;
    jac.set(Ne18, F18, scratch);

    scratch = -screened_rates(k_he4_ne18__p_na21)*Y(He4)*state.rho - screened_rates(k_n_ne18__he4_o15)*Y(N)*state.rho - screened_rates(k_n_ne18__ne19)*Y(N)*state.rho - screened_rates(k_n_ne18__p_f18)*Y(N)*state.rho - screened_rates(k_ne18__f18__weak__wc12) - screened_rates(k_ne18__he4_o14) - screened_rates(k_ne18__p_f17);
    jac.set(Ne18, Ne18, scratch);

    scratch = screened_rates(k_ne19__n_ne18);
    jac.set(Ne18, Ne19, scratch);

    scratch = screened_rates(k_p_na21__he4_ne18)*Y(H1)*state.rho;
    jac.set(Ne18, Na21, scratch);

    scratch = screened_rates(k_n_ne18__ne19)*Y(Ne18)*state.rho - screened_rates(k_n_ne19__he4_o16)*Y(Ne19)*state.rho - screened_rates(k_n_ne19__ne20)*Y(Ne19)*state.rho - screened_rates(k_n_ne19__p_f19)*Y(Ne19)*state.rho;
    jac.set(Ne19, N, scratch);

    scratch = screened_rates(k_p_f18__ne19)*Y(F18)*state.rho + screened_rates(k_p_f19__n_ne19)*Y(F19)*state.rho + screened_rates(k_p_na22__he4_ne19)*Y(Na22)*state.rho;
    jac.set(Ne19, H1, scratch);

    scratch = -screened_rates(k_he4_ne19__mg23)*Y(Ne19)*state.rho - screened_rates(k_he4_ne19__p_na22)*Y(Ne19)*state.rho + screened_rates(k_he4_o15__ne19)*Y(O15)*state.rho + screened_rates(k_he4_o16__n_ne19)*Y(O16)*state.rho;
    jac.set(Ne19, He4, scratch);

    scratch = screened_rates(k_he4_o15__ne19)*Y(He4)*state.rho;
    jac.set(Ne19, O15, scratch);

    scratch = screened_rates(k_he4_o16__n_ne19)*Y(He4)*state.rho;
    jac.set(Ne19, O16, scratch);

    scratch = screened_rates(k_p_f18__ne19)*Y(H1)*state.rho;
    jac.set(Ne19, F18, scratch);

    scratch = screened_rates(k_p_f19__n_ne19)*Y(H1)*state.rho;
    jac.set(Ne19, F19, scratch);

    scratch = screened_rates(k_n_ne18__ne19)*Y(N)*state.rho;
    jac.set(Ne19, Ne18, scratch);

    scratch = -screened_rates(k_he4_ne19__mg23)*Y(He4)*state.rho - screened_rates(k_he4_ne19__p_na22)*Y(He4)*state.rho - screened_rates(k_n_ne19__he4_o16)*Y(N)*state.rho - screened_rates(k_n_ne19__ne20)*Y(N)*state.rho - screened_rates(k_n_ne19__p_f19)*Y(N)*state.rho - screened_rates(k_ne19__f19__weak__wc12) - screened_rates(k_ne19__he4_o15) - screened_rates(k_ne19__n_ne18) - screened_rates(k_ne19__p_f18);
    jac.set(Ne19, Ne19, scratch);

    scratch = screened_rates(k_ne20__n_ne19);
    jac.set(Ne19, Ne20, scratch);

    scratch = screened_rates(k_p_na22__he4_ne19)*Y(H1)*state.rho;
    jac.set(Ne19, Na22, scratch);

    scratch = screened_rates(k_mg23__he4_ne19);
    jac.set(Ne19, Mg23, scratch);

    scratch = screened_rates(k_n_mg23__he4_ne20)*Y(Mg23)*state.rho + screened_rates(k_n_ne19__ne20)*Y(Ne19)*state.rho - screened_rates(k_n_ne20__he4_o17)*Y(Ne20)*state.rho - screened_rates(k_n_ne20__ne21)*Y(Ne20)*state.rho;
    jac.set(Ne20, N, scratch);

    scratch = screened_rates(k_p_f19__ne20)*Y(F19)*state.rho + screened_rates(k_p_na23__he4_ne20)*Y(Na23)*state.rho - screened_rates(k_p_ne20__he4_f17)*Y(Ne20)*state.rho - screened_rates(k_p_ne20__na21)*Y(Ne20)*state.rho + screened_rates(k_p_p31__c12_ne20)*Y(P31)*state.rho;
    jac.set(Ne20, H1, scratch);

    scratch = screened_rates(k_he4_f17__p_ne20)*Y(F17)*state.rho - screened_rates(k_he4_ne20__c12_c12)*Y(Ne20)*state.rho - screened_rates(k_he4_ne20__mg24)*Y(Ne20)*state.rho - screened_rates(k_he4_ne20__n_mg23)*Y(Ne20)*state.rho - screened_rates(k_he4_ne20__p_na23)*Y(Ne20)*state.rho + screened_rates(k_he4_o16__ne20)*Y(O16)*state.rho + screened_rates(k_he4_o17__n_ne20)*Y(O17)*state.rho + screened_rates(k_he4_si28__c12_ne20)*Y(Si28)*state.rho;
    jac.set(Ne20, He4, scratch);

    scratch = 1.0*screened_rates(k_c12_c12__he4_ne20)*Y(C12)*state.rho - screened_rates(k_c12_ne20__he4_si28)*Y(Ne20)*state.rho - screened_rates(k_c12_ne20__p_p31)*Y(Ne20)*state.rho;
    jac.set(Ne20, C12, scratch);

    scratch = screened_rates(k_he4_o16__ne20)*Y(He4)*state.rho;
    jac.set(Ne20, O16, scratch);

    scratch = screened_rates(k_he4_o17__n_ne20)*Y(He4)*state.rho;
    jac.set(Ne20, O17, scratch);

    scratch = screened_rates(k_he4_f17__p_ne20)*Y(He4)*state.rho;
    jac.set(Ne20, F17, scratch);

    scratch = screened_rates(k_p_f19__ne20)*Y(H1)*state.rho;
    jac.set(Ne20, F19, scratch);

    scratch = screened_rates(k_n_ne19__ne20)*Y(N)*state.rho;
    jac.set(Ne20, Ne19, scratch);

    scratch = -screened_rates(k_c12_ne20__he4_si28)*Y(C12)*state.rho - screened_rates(k_c12_ne20__p_p31)*Y(C12)*state.rho - screened_rates(k_he4_ne20__c12_c12)*Y(He4)*state.rho - screened_rates(k_he4_ne20__mg24)*Y(He4)*state.rho - screened_rates(k_he4_ne20__n_mg23)*Y(He4)*state.rho - screened_rates(k_he4_ne20__p_na23)*Y(He4)*state.rho - screened_rates(k_n_ne20__he4_o17)*Y(N)*state.rho - screened_rates(k_n_ne20__ne21)*Y(N)*state.rho - screened_rates(k_ne20__he4_o16) - screened_rates(k_ne20__n_ne19) - screened_rates(k_ne20__p_f19) - screened_rates(k_p_ne20__he4_f17)*Y(H1)*state.rho - screened_rates(k_p_ne20__na21)*Y(H1)*state.rho;
    jac.set(Ne20, Ne20, scratch);

    scratch = screened_rates(k_ne21__n_ne20);
    jac.set(Ne20, Ne21, scratch);

    scratch = screened_rates(k_na21__p_ne20);
    jac.set(Ne20, Na21, scratch);

    scratch = screened_rates(k_p_na23__he4_ne20)*Y(H1)*state.rho;
    jac.set(Ne20, Na23, scratch);

    scratch = screened_rates(k_n_mg23__he4_ne20)*Y(N)*state.rho;
    jac.set(Ne20, Mg23, scratch);

    scratch = screened_rates(k_mg24__he4_ne20);
    jac.set(Ne20, Mg24, scratch);

    scratch = screened_rates(k_he4_si28__c12_ne20)*Y(He4)*state.rho;
    jac.set(Ne20, Si28, scratch);

    scratch = screened_rates(k_p_p31__c12_ne20)*Y(H1)*state.rho;
    jac.set(Ne20, P31, scratch);

    scratch = screened_rates(k_n_mg24__he4_ne21)*Y(Mg24)*state.rho + screened_rates(k_n_na21__p_ne21)*Y(Na21)*state.rho + screened_rates(k_n_ne20__ne21)*Y(Ne20)*state.rho - screened_rates(k_n_ne21__he4_o18)*Y(Ne21)*state.rho - screened_rates(k_n_ne21__ne22)*Y(Ne21)*state.rho;
    jac.set(Ne21, N, scratch);

    scratch = -screened_rates(k_p_ne21__he4_f18)*Y(Ne21)*state.rho - screened_rates(k_p_ne21__n_na21)*Y(Ne21)*state.rho - screened_rates(k_p_ne21__na22)*Y(Ne21)*state.rho;
    jac.set(Ne21, H1, scratch);

    scratch = screened_rates(k_he4_f18__p_ne21)*Y(F18)*state.rho - screened_rates(k_he4_ne21__mg25)*Y(Ne21)*state.rho - screened_rates(k_he4_ne21__n_mg24)*Y(Ne21)*state.rho + screened_rates(k_he4_o17__ne21)*Y(O17)*state.rho + screened_rates(k_he4_o18__n_ne21)*Y(O18)*state.rho;
    jac.set(Ne21, He4, scratch);

    scratch = screened_rates(k_he4_o17__ne21)*Y(He4)*state.rho;
    jac.set(Ne21, O17, scratch);

    scratch = screened_rates(k_he4_o18__n_ne21)*Y(He4)*state.rho;
    jac.set(Ne21, O18, scratch);

    scratch = screened_rates(k_he4_f18__p_ne21)*Y(He4)*state.rho;
    jac.set(Ne21, F18, scratch);

    scratch = screened_rates(k_n_ne20__ne21)*Y(N)*state.rho;
    jac.set(Ne21, Ne20, scratch);

    scratch = -screened_rates(k_he4_ne21__mg25)*Y(He4)*state.rho - screened_rates(k_he4_ne21__n_mg24)*Y(He4)*state.rho - screened_rates(k_n_ne21__he4_o18)*Y(N)*state.rho - screened_rates(k_n_ne21__ne22)*Y(N)*state.rho - screened_rates(k_ne21__he4_o17) - screened_rates(k_ne21__n_ne20) - screened_rates(k_p_ne21__he4_f18)*Y(H1)*state.rho - screened_rates(k_p_ne21__n_na21)*Y(H1)*state.rho - screened_rates(k_p_ne21__na22)*Y(H1)*state.rho;
    jac.set(Ne21, Ne21, scratch);

    scratch = screened_rates(k_ne22__n_ne21);
    jac.set(Ne21, Ne22, scratch);

    scratch = screened_rates(k_n_na21__p_ne21)*Y(N)*state.rho + screened_rates(k_na21__ne21__weak__wc12);
    jac.set(Ne21, Na21, scratch);

    scratch = screened_rates(k_na22__p_ne21);
    jac.set(Ne21, Na22, scratch);

    scratch = screened_rates(k_n_mg24__he4_ne21)*Y(N)*state.rho;
    jac.set(Ne21, Mg24, scratch);

    scratch = screened_rates(k_mg25__he4_ne21);
    jac.set(Ne21, Mg25, scratch);

    scratch = screened_rates(k_n_mg25__he4_ne22)*Y(Mg25)*state.rho + screened_rates(k_n_na22__p_ne22)*Y(Na22)*state.rho + screened_rates(k_n_ne21__ne22)*Y(Ne21)*state.rho;
    jac.set(Ne22, N, scratch);

    scratch = -screened_rates(k_p_ne22__he4_f19)*Y(Ne22)*state.rho - screened_rates(k_p_ne22__n_na22)*Y(Ne22)*state.rho - screened_rates(k_p_ne22__na23)*Y(Ne22)*state.rho;
    jac.set(Ne22, H1, scratch);

    scratch = screened_rates(k_he4_f19__p_ne22)*Y(F19)*state.rho - screened_rates(k_he4_ne22__mg26)*Y(Ne22)*state.rho - screened_rates(k_he4_ne22__n_mg25)*Y(Ne22)*state.rho + screened_rates(k_he4_o18__ne22)*Y(O18)*state.rho;
    jac.set(Ne22, He4, scratch);

    scratch = screened_rates(k_he4_o18__ne22)*Y(He4)*state.rho;
    jac.set(Ne22, O18, scratch);

    scratch = screened_rates(k_he4_f19__p_ne22)*Y(He4)*state.rho;
    jac.set(Ne22, F19, scratch);

    scratch = screened_rates(k_n_ne21__ne22)*Y(N)*state.rho;
    jac.set(Ne22, Ne21, scratch);

    scratch = -screened_rates(k_he4_ne22__mg26)*Y(He4)*state.rho - screened_rates(k_he4_ne22__n_mg25)*Y(He4)*state.rho - screened_rates(k_ne22__he4_o18) - screened_rates(k_ne22__n_ne21) - screened_rates(k_p_ne22__he4_f19)*Y(H1)*state.rho - screened_rates(k_p_ne22__n_na22)*Y(H1)*state.rho - screened_rates(k_p_ne22__na23)*Y(H1)*state.rho;
    jac.set(Ne22, Ne22, scratch);

    scratch = screened_rates(k_n_na22__p_ne22)*Y(N)*state.rho + screened_rates(k_na22__ne22__weak__wc12);
    jac.set(Ne22, Na22, scratch);

    scratch = screened_rates(k_na23__p_ne22);
    jac.set(Ne22, Na23, scratch);

    scratch = screened_rates(k_n_mg25__he4_ne22)*Y(N)*state.rho;
    jac.set(Ne22, Mg25, scratch);

    scratch = screened_rates(k_mg26__he4_ne22);
    jac.set(Ne22, Mg26, scratch);

    scratch = -screened_rates(k_n_na21__he4_f18)*Y(Na21)*state.rho - screened_rates(k_n_na21__na22)*Y(Na21)*state.rho - screened_rates(k_n_na21__p_ne21)*Y(Na21)*state.rho;
    jac.set(Na21, N, scratch);

    scratch = screened_rates(k_p_mg24__he4_na21)*Y(Mg24)*state.rho - screened_rates(k_p_na21__he4_ne18)*Y(Na21)*state.rho + screened_rates(k_p_ne20__na21)*Y(Ne20)*state.rho + screened_rates(k_p_ne21__n_na21)*Y(Ne21)*state.rho;
    jac.set(Na21, H1, scratch);

    scratch = screened_rates(k_he4_f17__na21)*Y(F17)*state.rho + screened_rates(k_he4_f18__n_na21)*Y(F18)*state.rho - screened_rates(k_he4_na21__al25)*Y(Na21)*state.rho - screened_rates(k_he4_na21__p_mg24)*Y(Na21)*state.rho + screened_rates(k_he4_ne18__p_na21)*Y(Ne18)*state.rho;
    jac.set(Na21, He4, scratch);

    scratch = screened_rates(k_he4_f17__na21)*Y(He4)*state.rho;
    jac.set(Na21, F17, scratch);

    scratch = screened_rates(k_he4_f18__n_na21)*Y(He4)*state.rho;
    jac.set(Na21, F18, scratch);

    scratch = screened_rates(k_he4_ne18__p_na21)*Y(He4)*state.rho;
    jac.set(Na21, Ne18, scratch);

    scratch = screened_rates(k_p_ne20__na21)*Y(H1)*state.rho;
    jac.set(Na21, Ne20, scratch);

    scratch = screened_rates(k_p_ne21__n_na21)*Y(H1)*state.rho;
    jac.set(Na21, Ne21, scratch);

    scratch = -screened_rates(k_he4_na21__al25)*Y(He4)*state.rho - screened_rates(k_he4_na21__p_mg24)*Y(He4)*state.rho - screened_rates(k_n_na21__he4_f18)*Y(N)*state.rho - screened_rates(k_n_na21__na22)*Y(N)*state.rho - screened_rates(k_n_na21__p_ne21)*Y(N)*state.rho - screened_rates(k_na21__he4_f17) - screened_rates(k_na21__ne21__weak__wc12) - screened_rates(k_na21__p_ne20) - screened_rates(k_p_na21__he4_ne18)*Y(H1)*state.rho;
    jac.set(Na21, Na21, scratch);

    scratch = screened_rates(k_na22__n_na21);
    jac.set(Na21, Na22, scratch);

    scratch = screened_rates(k_p_mg24__he4_na21)*Y(H1)*state.rho;
    jac.set(Na21, Mg24, scratch);

    scratch = screened_rates(k_al25__he4_na21);
    jac.set(Na21, Al25, scratch);

    scratch = screened_rates(k_n_al25__he4_na22)*Y(Al25)*state.rho + screened_rates(k_n_na21__na22)*Y(Na21)*state.rho - screened_rates(k_n_na22__he4_f19)*Y(Na22)*state.rho - screened_rates(k_n_na22__na23)*Y(Na22)*state.rho - screened_rates(k_n_na22__p_ne22)*Y(Na22)*state.rho;
    jac.set(Na22, N, scratch);

    scratch = screened_rates(k_p_mg25__he4_na22)*Y(Mg25)*state.rho - screened_rates(k_p_na22__he4_ne19)*Y(Na22)*state.rho - screened_rates(k_p_na22__mg23)*Y(Na22)*state.rho + screened_rates(k_p_ne21__na22)*Y(Ne21)*state.rho + screened_rates(k_p_ne22__n_na22)*Y(Ne22)*state.rho;
    jac.set(Na22, H1, scratch);

    scratch = screened_rates(k_he4_f18__na22)*Y(F18)*state.rho + screened_rates(k_he4_f19__n_na22)*Y(F19)*state.rho - screened_rates(k_he4_na22__al26)*Y(Na22)*state.rho - screened_rates(k_he4_na22__n_al25)*Y(Na22)*state.rho - screened_rates(k_he4_na22__p_mg25)*Y(Na22)*state.rho + screened_rates(k_he4_ne19__p_na22)*Y(Ne19)*state.rho;
    jac.set(Na22, He4, scratch);

    scratch = screened_rates(k_he4_f18__na22)*Y(He4)*state.rho;
    jac.set(Na22, F18, scratch);

    scratch = screened_rates(k_he4_f19__n_na22)*Y(He4)*state.rho;
    jac.set(Na22, F19, scratch);

    scratch = screened_rates(k_he4_ne19__p_na22)*Y(He4)*state.rho;
    jac.set(Na22, Ne19, scratch);

    scratch = screened_rates(k_p_ne21__na22)*Y(H1)*state.rho;
    jac.set(Na22, Ne21, scratch);

    scratch = screened_rates(k_p_ne22__n_na22)*Y(H1)*state.rho;
    jac.set(Na22, Ne22, scratch);

    scratch = screened_rates(k_n_na21__na22)*Y(N)*state.rho;
    jac.set(Na22, Na21, scratch);

    scratch = -screened_rates(k_he4_na22__al26)*Y(He4)*state.rho - screened_rates(k_he4_na22__n_al25)*Y(He4)*state.rho - screened_rates(k_he4_na22__p_mg25)*Y(He4)*state.rho - screened_rates(k_n_na22__he4_f19)*Y(N)*state.rho - screened_rates(k_n_na22__na23)*Y(N)*state.rho - screened_rates(k_n_na22__p_ne22)*Y(N)*state.rho - screened_rates(k_na22__he4_f18) - screened_rates(k_na22__n_na21) - screened_rates(k_na22__ne22__weak__wc12) - screened_rates(k_na22__p_ne21) - screened_rates(k_p_na22__he4_ne19)*Y(H1)*state.rho - screened_rates(k_p_na22__mg23)*Y(H1)*state.rho;
    jac.set(Na22, Na22, scratch);

    scratch = screened_rates(k_na23__n_na22);
    jac.set(Na22, Na23, scratch);

    scratch = screened_rates(k_mg23__p_na22);
    jac.set(Na22, Mg23, scratch);

    scratch = screened_rates(k_p_mg25__he4_na22)*Y(H1)*state.rho;
    jac.set(Na22, Mg25, scratch);

    scratch = screened_rates(k_n_al25__he4_na22)*Y(N)*state.rho;
    jac.set(Na22, Al25, scratch);

    scratch = screened_rates(k_al26__he4_na22);
    jac.set(Na22, Al26, scratch);

    scratch = screened_rates(k_n_al26__he4_na23)*Y(Al26)*state.rho + screened_rates(k_n_mg23__p_na23)*Y(Mg23)*state.rho + screened_rates(k_n_na22__na23)*Y(Na22)*state.rho;
    jac.set(Na23, N, scratch);

    scratch = screened_rates(k_p_mg26__he4_na23)*Y(Mg26)*state.rho - screened_rates(k_p_na23__c12_c12)*Y(Na23)*state.rho - screened_rates(k_p_na23__he4_ne20)*Y(Na23)*state.rho - screened_rates(k_p_na23__mg24)*Y(Na23)*state.rho - screened_rates(k_p_na23__n_mg23)*Y(Na23)*state.rho + screened_rates(k_p_ne22__na23)*Y(Ne22)*state.rho;
    jac.set(Na23, H1, scratch);

    scratch = screened_rates(k_he4_f19__na23)*Y(F19)*state.rho - screened_rates(k_he4_na23__al27)*Y(Na23)*state.rho - screened_rates(k_he4_na23__n_al26)*Y(Na23)*state.rho - screened_rates(k_he4_na23__p_mg26)*Y(Na23)*state.rho + screened_rates(k_he4_ne20__p_na23)*Y(Ne20)*state.rho;
    jac.set(Na23, He4, scratch);

    scratch = 1.0*screened_rates(k_c12_c12__p_na23)*Y(C12)*state.rho;
    jac.set(Na23, C12, scratch);

    scratch = screened_rates(k_he4_f19__na23)*Y(He4)*state.rho;
    jac.set(Na23, F19, scratch);

    scratch = screened_rates(k_he4_ne20__p_na23)*Y(He4)*state.rho;
    jac.set(Na23, Ne20, scratch);

    scratch = screened_rates(k_p_ne22__na23)*Y(H1)*state.rho;
    jac.set(Na23, Ne22, scratch);

    scratch = screened_rates(k_n_na22__na23)*Y(N)*state.rho;
    jac.set(Na23, Na22, scratch);

    scratch = -screened_rates(k_he4_na23__al27)*Y(He4)*state.rho - screened_rates(k_he4_na23__n_al26)*Y(He4)*state.rho - screened_rates(k_he4_na23__p_mg26)*Y(He4)*state.rho - screened_rates(k_na23__he4_f19) - screened_rates(k_na23__n_na22) - screened_rates(k_na23__p_ne22) - screened_rates(k_p_na23__c12_c12)*Y(H1)*state.rho - screened_rates(k_p_na23__he4_ne20)*Y(H1)*state.rho - screened_rates(k_p_na23__mg24)*Y(H1)*state.rho - screened_rates(k_p_na23__n_mg23)*Y(H1)*state.rho;
    jac.set(Na23, Na23, scratch);

    scratch = screened_rates(k_mg23__na23__weak__wc12) + screened_rates(k_n_mg23__p_na23)*Y(N)*state.rho;
    jac.set(Na23, Mg23, scratch);

    scratch = screened_rates(k_mg24__p_na23);
    jac.set(Na23, Mg24, scratch);

    scratch = screened_rates(k_p_mg26__he4_na23)*Y(H1)*state.rho;
    jac.set(Na23, Mg26, scratch);

    scratch = screened_rates(k_n_al26__he4_na23)*Y(N)*state.rho;
    jac.set(Na23, Al26, scratch);

    scratch = screened_rates(k_al27__he4_na23);
    jac.set(Na23, Al27, scratch);

    scratch = -screened_rates(k_n_mg23__c12_c12)*Y(Mg23)*state.rho - screened_rates(k_n_mg23__he4_ne20)*Y(Mg23)*state.rho - screened_rates(k_n_mg23__mg24)*Y(Mg23)*state.rho - screened_rates(k_n_mg23__p_na23)*Y(Mg23)*state.rho;
    jac.set(Mg23, N, scratch);

    scratch = screened_rates(k_p_al26__he4_mg23)*Y(Al26)*state.rho + screened_rates(k_p_na22__mg23)*Y(Na22)*state.rho + screened_rates(k_p_na23__n_mg23)*Y(Na23)*state.rho;
    jac.set(Mg23, H1, scratch);

    scratch = -screened_rates(k_he4_mg23__p_al26)*Y(Mg23)*state.rho + screened_rates(k_he4_ne19__mg23)*Y(Ne19)*state.rho + screened_rates(k_he4_ne20__n_mg23)*Y(Ne20)*state.rho;
    jac.set(Mg23, He4, scratch);

    scratch = 1.0*screened_rates(k_c12_c12__n_mg23)*Y(C12)*state.rho;
    jac.set(Mg23, C12, scratch);

    scratch = screened_rates(k_he4_ne19__mg23)*Y(He4)*state.rho;
    jac.set(Mg23, Ne19, scratch);

    scratch = screened_rates(k_he4_ne20__n_mg23)*Y(He4)*state.rho;
    jac.set(Mg23, Ne20, scratch);

    scratch = screened_rates(k_p_na22__mg23)*Y(H1)*state.rho;
    jac.set(Mg23, Na22, scratch);

    scratch = screened_rates(k_p_na23__n_mg23)*Y(H1)*state.rho;
    jac.set(Mg23, Na23, scratch);

    scratch = -screened_rates(k_he4_mg23__p_al26)*Y(He4)*state.rho - screened_rates(k_mg23__he4_ne19) - screened_rates(k_mg23__na23__weak__wc12) - screened_rates(k_mg23__p_na22) - screened_rates(k_n_mg23__c12_c12)*Y(N)*state.rho - screened_rates(k_n_mg23__he4_ne20)*Y(N)*state.rho - screened_rates(k_n_mg23__mg24)*Y(N)*state.rho - screened_rates(k_n_mg23__p_na23)*Y(N)*state.rho;
    jac.set(Mg23, Mg23, scratch);

    scratch = screened_rates(k_mg24__n_mg23);
    jac.set(Mg23, Mg24, scratch);

    scratch = screened_rates(k_p_al26__he4_mg23)*Y(H1)*state.rho;
    jac.set(Mg23, Al26, scratch);

    scratch = screened_rates(k_n_mg23__mg24)*Y(Mg23)*state.rho - screened_rates(k_n_mg24__he4_ne21)*Y(Mg24)*state.rho - screened_rates(k_n_mg24__mg25)*Y(Mg24)*state.rho;
    jac.set(Mg24, N, scratch);

    scratch = screened_rates(k_p_al27__he4_mg24)*Y(Al27)*state.rho - screened_rates(k_p_mg24__al25)*Y(Mg24)*state.rho - screened_rates(k_p_mg24__he4_na21)*Y(Mg24)*state.rho + screened_rates(k_p_na23__mg24)*Y(Na23)*state.rho;
    jac.set(Mg24, H1, scratch);

    scratch = -screened_rates(k_he4_mg24__c12_o16)*Y(Mg24)*state.rho - screened_rates(k_he4_mg24__p_al27)*Y(Mg24)*state.rho - screened_rates(k_he4_mg24__si28)*Y(Mg24)*state.rho + screened_rates(k_he4_na21__p_mg24)*Y(Na21)*state.rho + screened_rates(k_he4_ne20__mg24)*Y(Ne20)*state.rho + screened_rates(k_he4_ne21__n_mg24)*Y(Ne21)*state.rho;
    jac.set(Mg24, He4, scratch);

    scratch = screened_rates(k_c12_o16__he4_mg24)*Y(O16)*state.rho;
    jac.set(Mg24, C12, scratch);

    scratch = screened_rates(k_c12_o16__he4_mg24)*Y(C12)*state.rho;
    jac.set(Mg24, O16, scratch);

    scratch = screened_rates(k_he4_ne20__mg24)*Y(He4)*state.rho;
    jac.set(Mg24, Ne20, scratch);

    scratch = screened_rates(k_he4_ne21__n_mg24)*Y(He4)*state.rho;
    jac.set(Mg24, Ne21, scratch);

    scratch = screened_rates(k_he4_na21__p_mg24)*Y(He4)*state.rho;
    jac.set(Mg24, Na21, scratch);

    scratch = screened_rates(k_p_na23__mg24)*Y(H1)*state.rho;
    jac.set(Mg24, Na23, scratch);

    scratch = screened_rates(k_n_mg23__mg24)*Y(N)*state.rho;
    jac.set(Mg24, Mg23, scratch);

    scratch = -screened_rates(k_he4_mg24__c12_o16)*Y(He4)*state.rho - screened_rates(k_he4_mg24__p_al27)*Y(He4)*state.rho - screened_rates(k_he4_mg24__si28)*Y(He4)*state.rho - screened_rates(k_mg24__he4_ne20) - screened_rates(k_mg24__n_mg23) - screened_rates(k_mg24__p_na23) - screened_rates(k_n_mg24__he4_ne21)*Y(N)*state.rho - screened_rates(k_n_mg24__mg25)*Y(N)*state.rho - screened_rates(k_p_mg24__al25)*Y(H1)*state.rho - screened_rates(k_p_mg24__he4_na21)*Y(H1)*state.rho;
    jac.set(Mg24, Mg24, scratch);

    scratch = screened_rates(k_mg25__n_mg24);
    jac.set(Mg24, Mg25, scratch);

    scratch = screened_rates(k_al25__p_mg24);
    jac.set(Mg24, Al25, scratch);

    scratch = screened_rates(k_p_al27__he4_mg24)*Y(H1)*state.rho;
    jac.set(Mg24, Al27, scratch);

    scratch = screened_rates(k_si28__he4_mg24);
    jac.set(Mg24, Si28, scratch);

    scratch = screened_rates(k_n_al25__p_mg25)*Y(Al25)*state.rho + screened_rates(k_n_mg24__mg25)*Y(Mg24)*state.rho - screened_rates(k_n_mg25__he4_ne22)*Y(Mg25)*state.rho - screened_rates(k_n_mg25__mg26)*Y(Mg25)*state.rho + screened_rates(k_n_si28__he4_mg25)*Y(Si28)*state.rho;
    jac.set(Mg25, N, scratch);

    scratch = -screened_rates(k_p_mg25__al26)*Y(Mg25)*state.rho - screened_rates(k_p_mg25__he4_na22)*Y(Mg25)*state.rho - screened_rates(k_p_mg25__n_al25)*Y(Mg25)*state.rho;
    jac.set(Mg25, H1, scratch);

    scratch = -screened_rates(k_he4_mg25__n_si28)*Y(Mg25)*state.rho - screened_rates(k_he4_mg25__si29)*Y(Mg25)*state.rho + screened_rates(k_he4_na22__p_mg25)*Y(Na22)*state.rho + screened_rates(k_he4_ne21__mg25)*Y(Ne21)*state.rho + screened_rates(k_he4_ne22__n_mg25)*Y(Ne22)*state.rho;
    jac.set(Mg25, He4, scratch);

    scratch = screened_rates(k_he4_ne21__mg25)*Y(He4)*state.rho;
    jac.set(Mg25, Ne21, scratch);

    scratch = screened_rates(k_he4_ne22__n_mg25)*Y(He4)*state.rho;
    jac.set(Mg25, Ne22, scratch);

    scratch = screened_rates(k_he4_na22__p_mg25)*Y(He4)*state.rho;
    jac.set(Mg25, Na22, scratch);

    scratch = screened_rates(k_n_mg24__mg25)*Y(N)*state.rho;
    jac.set(Mg25, Mg24, scratch);

    scratch = -screened_rates(k_he4_mg25__n_si28)*Y(He4)*state.rho - screened_rates(k_he4_mg25__si29)*Y(He4)*state.rho - screened_rates(k_mg25__he4_ne21) - screened_rates(k_mg25__n_mg24) - screened_rates(k_n_mg25__he4_ne22)*Y(N)*state.rho - screened_rates(k_n_mg25__mg26)*Y(N)*state.rho - screened_rates(k_p_mg25__al26)*Y(H1)*state.rho - screened_rates(k_p_mg25__he4_na22)*Y(H1)*state.rho - screened_rates(k_p_mg25__n_al25)*Y(H1)*state.rho;
    jac.set(Mg25, Mg25, scratch);

    scratch = screened_rates(k_mg26__n_mg25);
    jac.set(Mg25, Mg26, scratch);

    scratch = screened_rates(k_al25__mg25__weak__wc12) + screened_rates(k_n_al25__p_mg25)*Y(N)*state.rho;
    jac.set(Mg25, Al25, scratch);

    scratch = screened_rates(k_al26__p_mg25);
    jac.set(Mg25, Al26, scratch);

    scratch = screened_rates(k_n_si28__he4_mg25)*Y(N)*state.rho;
    jac.set(Mg25, Si28, scratch);

    scratch = screened_rates(k_si29__he4_mg25);
    jac.set(Mg25, Si29, scratch);

    scratch = screened_rates(k_n_al26__p_mg26)*Y(Al26)*state.rho + screened_rates(k_n_mg25__mg26)*Y(Mg25)*state.rho + screened_rates(k_n_si29__he4_mg26)*Y(Si29)*state.rho;
    jac.set(Mg26, N, scratch);

    scratch = -screened_rates(k_p_mg26__al27)*Y(Mg26)*state.rho - screened_rates(k_p_mg26__he4_na23)*Y(Mg26)*state.rho - screened_rates(k_p_mg26__n_al26)*Y(Mg26)*state.rho;
    jac.set(Mg26, H1, scratch);

    scratch = -screened_rates(k_he4_mg26__n_si29)*Y(Mg26)*state.rho - screened_rates(k_he4_mg26__si30)*Y(Mg26)*state.rho + screened_rates(k_he4_na23__p_mg26)*Y(Na23)*state.rho + screened_rates(k_he4_ne22__mg26)*Y(Ne22)*state.rho;
    jac.set(Mg26, He4, scratch);

    scratch = screened_rates(k_he4_ne22__mg26)*Y(He4)*state.rho;
    jac.set(Mg26, Ne22, scratch);

    scratch = screened_rates(k_he4_na23__p_mg26)*Y(He4)*state.rho;
    jac.set(Mg26, Na23, scratch);

    scratch = screened_rates(k_n_mg25__mg26)*Y(N)*state.rho;
    jac.set(Mg26, Mg25, scratch);

    scratch = -screened_rates(k_he4_mg26__n_si29)*Y(He4)*state.rho - screened_rates(k_he4_mg26__si30)*Y(He4)*state.rho - screened_rates(k_mg26__he4_ne22) - screened_rates(k_mg26__n_mg25) - screened_rates(k_p_mg26__al27)*Y(H1)*state.rho - screened_rates(k_p_mg26__he4_na23)*Y(H1)*state.rho - screened_rates(k_p_mg26__n_al26)*Y(H1)*state.rho;
    jac.set(Mg26, Mg26, scratch);

    scratch = screened_rates(k_al26__mg26__weak__wc12) + screened_rates(k_n_al26__p_mg26)*Y(N)*state.rho;
    jac.set(Mg26, Al26, scratch);

    scratch = screened_rates(k_al27__p_mg26);
    jac.set(Mg26, Al27, scratch);

    scratch = screened_rates(k_n_si29__he4_mg26)*Y(N)*state.rho;
    jac.set(Mg26, Si29, scratch);

    scratch = screened_rates(k_si30__he4_mg26);
    jac.set(Mg26, Si30, scratch);

    scratch = -screened_rates(k_n_al25__al26)*Y(Al25)*state.rho - screened_rates(k_n_al25__he4_na22)*Y(Al25)*state.rho - screened_rates(k_n_al25__p_mg25)*Y(Al25)*state.rho;
    jac.set(Al25, N, scratch);

    scratch = screened_rates(k_p_mg24__al25)*Y(Mg24)*state.rho + screened_rates(k_p_mg25__n_al25)*Y(Mg25)*state.rho + screened_rates(k_p_si28__he4_al25)*Y(Si28)*state.rho;
    jac.set(Al25, H1, scratch);

    scratch = -screened_rates(k_he4_al25__p29)*Y(Al25)*state.rho - screened_rates(k_he4_al25__p_si28)*Y(Al25)*state.rho + screened_rates(k_he4_na21__al25)*Y(Na21)*state.rho + screened_rates(k_he4_na22__n_al25)*Y(Na22)*state.rho;
    jac.set(Al25, He4, scratch);

    scratch = screened_rates(k_he4_na21__al25)*Y(He4)*state.rho;
    jac.set(Al25, Na21, scratch);

    scratch = screened_rates(k_he4_na22__n_al25)*Y(He4)*state.rho;
    jac.set(Al25, Na22, scratch);

    scratch = screened_rates(k_p_mg24__al25)*Y(H1)*state.rho;
    jac.set(Al25, Mg24, scratch);

    scratch = screened_rates(k_p_mg25__n_al25)*Y(H1)*state.rho;
    jac.set(Al25, Mg25, scratch);

    scratch = -screened_rates(k_al25__he4_na21) - screened_rates(k_al25__mg25__weak__wc12) - screened_rates(k_al25__p_mg24) - screened_rates(k_he4_al25__p29)*Y(He4)*state.rho - screened_rates(k_he4_al25__p_si28)*Y(He4)*state.rho - screened_rates(k_n_al25__al26)*Y(N)*state.rho - screened_rates(k_n_al25__he4_na22)*Y(N)*state.rho - screened_rates(k_n_al25__p_mg25)*Y(N)*state.rho;
    jac.set(Al25, Al25, scratch);

    scratch = screened_rates(k_al26__n_al25);
    jac.set(Al25, Al26, scratch);

    scratch = screened_rates(k_p_si28__he4_al25)*Y(H1)*state.rho;
    jac.set(Al25, Si28, scratch);

    scratch = screened_rates(k_p29__he4_al25);
    jac.set(Al25, P29, scratch);

    scratch = screened_rates(k_n_al25__al26)*Y(Al25)*state.rho - screened_rates(k_n_al26__al27)*Y(Al26)*state.rho - screened_rates(k_n_al26__he4_na23)*Y(Al26)*state.rho - screened_rates(k_n_al26__p_mg26)*Y(Al26)*state.rho + screened_rates(k_n_p29__he4_al26)*Y(P29)*state.rho;
    jac.set(Al26, N, scratch);

    scratch = -screened_rates(k_p_al26__he4_mg23)*Y(Al26)*state.rho + screened_rates(k_p_mg25__al26)*Y(Mg25)*state.rho + screened_rates(k_p_mg26__n_al26)*Y(Mg26)*state.rho + screened_rates(k_p_si29__he4_al26)*Y(Si29)*state.rho;
    jac.set(Al26, H1, scratch);

    scratch = -screened_rates(k_he4_al26__n_p29)*Y(Al26)*state.rho - screened_rates(k_he4_al26__p30)*Y(Al26)*state.rho - screened_rates(k_he4_al26__p_si29)*Y(Al26)*state.rho + screened_rates(k_he4_mg23__p_al26)*Y(Mg23)*state.rho + screened_rates(k_he4_na22__al26)*Y(Na22)*state.rho + screened_rates(k_he4_na23__n_al26)*Y(Na23)*state.rho;
    jac.set(Al26, He4, scratch);

    scratch = screened_rates(k_he4_na22__al26)*Y(He4)*state.rho;
    jac.set(Al26, Na22, scratch);

    scratch = screened_rates(k_he4_na23__n_al26)*Y(He4)*state.rho;
    jac.set(Al26, Na23, scratch);

    scratch = screened_rates(k_he4_mg23__p_al26)*Y(He4)*state.rho;
    jac.set(Al26, Mg23, scratch);

    scratch = screened_rates(k_p_mg25__al26)*Y(H1)*state.rho;
    jac.set(Al26, Mg25, scratch);

    scratch = screened_rates(k_p_mg26__n_al26)*Y(H1)*state.rho;
    jac.set(Al26, Mg26, scratch);

    scratch = screened_rates(k_n_al25__al26)*Y(N)*state.rho;
    jac.set(Al26, Al25, scratch);

    scratch = -screened_rates(k_al26__he4_na22) - screened_rates(k_al26__mg26__weak__wc12) - screened_rates(k_al26__n_al25) - screened_rates(k_al26__p_mg25) - screened_rates(k_he4_al26__n_p29)*Y(He4)*state.rho - screened_rates(k_he4_al26__p30)*Y(He4)*state.rho - screened_rates(k_he4_al26__p_si29)*Y(He4)*state.rho - screened_rates(k_n_al26__al27)*Y(N)*state.rho - screened_rates(k_n_al26__he4_na23)*Y(N)*state.rho - screened_rates(k_n_al26__p_mg26)*Y(N)*state.rho - screened_rates(k_p_al26__he4_mg23)*Y(H1)*state.rho;
    jac.set(Al26, Al26, scratch);

    scratch = screened_rates(k_al27__n_al26);
    jac.set(Al26, Al27, scratch);

    scratch = screened_rates(k_p_si29__he4_al26)*Y(H1)*state.rho;
    jac.set(Al26, Si29, scratch);

    scratch = screened_rates(k_n_p29__he4_al26)*Y(N)*state.rho;
    jac.set(Al26, P29, scratch);

    scratch = screened_rates(k_p30__he4_al26);
    jac.set(Al26, P30, scratch);

    scratch = screened_rates(k_n_al26__al27)*Y(Al26)*state.rho + screened_rates(k_n_p30__he4_al27)*Y(P30)*state.rho;
    jac.set(Al27, N, scratch);

    scratch = -screened_rates(k_p_al27__c12_o16)*Y(Al27)*state.rho - screened_rates(k_p_al27__he4_mg24)*Y(Al27)*state.rho - screened_rates(k_p_al27__si28)*Y(Al27)*state.rho + screened_rates(k_p_mg26__al27)*Y(Mg26)*state.rho + screened_rates(k_p_si30__he4_al27)*Y(Si30)*state.rho;
    jac.set(Al27, H1, scratch);

    scratch = -screened_rates(k_he4_al27__n_p30)*Y(Al27)*state.rho - screened_rates(k_he4_al27__p31)*Y(Al27)*state.rho - screened_rates(k_he4_al27__p_si30)*Y(Al27)*state.rho + screened_rates(k_he4_mg24__p_al27)*Y(Mg24)*state.rho + screened_rates(k_he4_na23__al27)*Y(Na23)*state.rho;
    jac.set(Al27, He4, scratch);

    scratch = screened_rates(k_c12_o16__p_al27)*Y(O16)*state.rho;
    jac.set(Al27, C12, scratch);

    scratch = screened_rates(k_c12_o16__p_al27)*Y(C12)*state.rho;
    jac.set(Al27, O16, scratch);

    scratch = screened_rates(k_he4_na23__al27)*Y(He4)*state.rho;
    jac.set(Al27, Na23, scratch);

    scratch = screened_rates(k_he4_mg24__p_al27)*Y(He4)*state.rho;
    jac.set(Al27, Mg24, scratch);

    scratch = screened_rates(k_p_mg26__al27)*Y(H1)*state.rho;
    jac.set(Al27, Mg26, scratch);

    scratch = screened_rates(k_n_al26__al27)*Y(N)*state.rho;
    jac.set(Al27, Al26, scratch);

    scratch = -screened_rates(k_al27__he4_na23) - screened_rates(k_al27__n_al26) - screened_rates(k_al27__p_mg26) - screened_rates(k_he4_al27__n_p30)*Y(He4)*state.rho - screened_rates(k_he4_al27__p31)*Y(He4)*state.rho - screened_rates(k_he4_al27__p_si30)*Y(He4)*state.rho - screened_rates(k_p_al27__c12_o16)*Y(H1)*state.rho - screened_rates(k_p_al27__he4_mg24)*Y(H1)*state.rho - screened_rates(k_p_al27__si28)*Y(H1)*state.rho;
    jac.set(Al27, Al27, scratch);

    scratch = screened_rates(k_si28__p_al27);
    jac.set(Al27, Si28, scratch);

    scratch = screened_rates(k_p_si30__he4_al27)*Y(H1)*state.rho;
    jac.set(Al27, Si30, scratch);

    scratch = screened_rates(k_n_p30__he4_al27)*Y(N)*state.rho;
    jac.set(Al27, P30, scratch);

    scratch = screened_rates(k_p31__he4_al27);
    jac.set(Al27, P31, scratch);

    scratch = -screened_rates(k_n_si28__he4_mg25)*Y(Si28)*state.rho - screened_rates(k_n_si28__si29)*Y(Si28)*state.rho;
    jac.set(Si28, N, scratch);

    scratch = screened_rates(k_p_al27__si28)*Y(Al27)*state.rho + screened_rates(k_p_p31__he4_si28)*Y(P31)*state.rho - screened_rates(k_p_si28__he4_al25)*Y(Si28)*state.rho - screened_rates(k_p_si28__p29)*Y(Si28)*state.rho;
    jac.set(Si28, H1, scratch);

    scratch = screened_rates(k_he4_al25__p_si28)*Y(Al25)*state.rho + screened_rates(k_he4_mg24__si28)*Y(Mg24)*state.rho + screened_rates(k_he4_mg25__n_si28)*Y(Mg25)*state.rho - screened_rates(k_he4_si28__c12_ne20)*Y(Si28)*state.rho - screened_rates(k_he4_si28__o16_o16)*Y(Si28)*state.rho - screened_rates(k_he4_si28__p_p31)*Y(Si28)*state.rho - screened_rates(k_he4_si28__s32)*Y(Si28)*state.rho;
    jac.set(Si28, He4, scratch);

    scratch = screened_rates(k_c12_ne20__he4_si28)*Y(Ne20)*state.rho;
    jac.set(Si28, C12, scratch);

    scratch = 1.0*screened_rates(k_o16_o16__he4_si28)*Y(O16)*state.rho;
    jac.set(Si28, O16, scratch);

    scratch = screened_rates(k_c12_ne20__he4_si28)*Y(C12)*state.rho;
    jac.set(Si28, Ne20, scratch);

    scratch = screened_rates(k_he4_mg24__si28)*Y(He4)*state.rho;
    jac.set(Si28, Mg24, scratch);

    scratch = screened_rates(k_he4_mg25__n_si28)*Y(He4)*state.rho;
    jac.set(Si28, Mg25, scratch);

    scratch = screened_rates(k_he4_al25__p_si28)*Y(He4)*state.rho;
    jac.set(Si28, Al25, scratch);

    scratch = screened_rates(k_p_al27__si28)*Y(H1)*state.rho;
    jac.set(Si28, Al27, scratch);

    scratch = -screened_rates(k_he4_si28__c12_ne20)*Y(He4)*state.rho - screened_rates(k_he4_si28__o16_o16)*Y(He4)*state.rho - screened_rates(k_he4_si28__p_p31)*Y(He4)*state.rho - screened_rates(k_he4_si28__s32)*Y(He4)*state.rho - screened_rates(k_n_si28__he4_mg25)*Y(N)*state.rho - screened_rates(k_n_si28__si29)*Y(N)*state.rho - screened_rates(k_p_si28__he4_al25)*Y(H1)*state.rho - screened_rates(k_p_si28__p29)*Y(H1)*state.rho - screened_rates(k_si28__he4_mg24) - screened_rates(k_si28__p_al27);
    jac.set(Si28, Si28, scratch);

    scratch = screened_rates(k_si29__n_si28);
    jac.set(Si28, Si29, scratch);

    scratch = screened_rates(k_p29__p_si28);
    jac.set(Si28, P29, scratch);

    scratch = screened_rates(k_p_p31__he4_si28)*Y(H1)*state.rho;
    jac.set(Si28, P31, scratch);

    scratch = screened_rates(k_s32__he4_si28);
    jac.set(Si28, S32, scratch);

    scratch = screened_rates(k_n_p29__p_si29)*Y(P29)*state.rho + screened_rates(k_n_s32__he4_si29)*Y(S32)*state.rho + screened_rates(k_n_si28__si29)*Y(Si28)*state.rho - screened_rates(k_n_si29__he4_mg26)*Y(Si29)*state.rho - screened_rates(k_n_si29__si30)*Y(Si29)*state.rho;
    jac.set(Si29, N, scratch);

    scratch = screened_rates(k_p_p32__he4_si29)*Y(P32)*state.rho - screened_rates(k_p_si29__he4_al26)*Y(Si29)*state.rho - screened_rates(k_p_si29__n_p29)*Y(Si29)*state.rho - screened_rates(k_p_si29__p30)*Y(Si29)*state.rho;
    jac.set(Si29, H1, scratch);

    scratch = screened_rates(k_he4_al26__p_si29)*Y(Al26)*state.rho + screened_rates(k_he4_mg25__si29)*Y(Mg25)*state.rho + screened_rates(k_he4_mg26__n_si29)*Y(Mg26)*state.rho - screened_rates(k_he4_si29__n_s32)*Y(Si29)*state.rho - screened_rates(k_he4_si29__p_p32)*Y(Si29)*state.rho - screened_rates(k_he4_si29__s33)*Y(Si29)*state.rho;
    jac.set(Si29, He4, scratch);

    scratch = screened_rates(k_he4_mg25__si29)*Y(He4)*state.rho;
    jac.set(Si29, Mg25, scratch);

    scratch = screened_rates(k_he4_mg26__n_si29)*Y(He4)*state.rho;
    jac.set(Si29, Mg26, scratch);

    scratch = screened_rates(k_he4_al26__p_si29)*Y(He4)*state.rho;
    jac.set(Si29, Al26, scratch);

    scratch = screened_rates(k_n_si28__si29)*Y(N)*state.rho;
    jac.set(Si29, Si28, scratch);

    scratch = -screened_rates(k_he4_si29__n_s32)*Y(He4)*state.rho - screened_rates(k_he4_si29__p_p32)*Y(He4)*state.rho - screened_rates(k_he4_si29__s33)*Y(He4)*state.rho - screened_rates(k_n_si29__he4_mg26)*Y(N)*state.rho - screened_rates(k_n_si29__si30)*Y(N)*state.rho - screened_rates(k_p_si29__he4_al26)*Y(H1)*state.rho - screened_rates(k_p_si29__n_p29)*Y(H1)*state.rho - screened_rates(k_p_si29__p30)*Y(H1)*state.rho - screened_rates(k_si29__he4_mg25) - screened_rates(k_si29__n_si28);
    jac.set(Si29, Si29, scratch);

    scratch = screened_rates(k_si30__n_si29);
    jac.set(Si29, Si30, scratch);

    scratch = screened_rates(k_n_p29__p_si29)*Y(N)*state.rho + screened_rates(k_p29__si29__weak__wc12);
    jac.set(Si29, P29, scratch);

    scratch = screened_rates(k_p30__p_si29);
    jac.set(Si29, P30, scratch);

    scratch = screened_rates(k_p_p32__he4_si29)*Y(H1)*state.rho;
    jac.set(Si29, P32, scratch);

    scratch = screened_rates(k_n_s32__he4_si29)*Y(N)*state.rho;
    jac.set(Si29, S32, scratch);

    scratch = screened_rates(k_s33__he4_si29);
    jac.set(Si29, S33, scratch);

    scratch = screened_rates(k_n_p30__p_si30)*Y(P30)*state.rho + screened_rates(k_n_s33__he4_si30)*Y(S33)*state.rho + screened_rates(k_n_si29__si30)*Y(Si29)*state.rho - screened_rates(k_n_si30__si31)*Y(Si30)*state.rho;
    jac.set(Si30, N, scratch);

    scratch = screened_rates(k_p_p33__he4_si30)*Y(P33)*state.rho - screened_rates(k_p_si30__he4_al27)*Y(Si30)*state.rho - screened_rates(k_p_si30__n_p30)*Y(Si30)*state.rho - screened_rates(k_p_si30__p31)*Y(Si30)*state.rho;
    jac.set(Si30, H1, scratch);

    scratch = screened_rates(k_he4_al27__p_si30)*Y(Al27)*state.rho + screened_rates(k_he4_mg26__si30)*Y(Mg26)*state.rho - screened_rates(k_he4_si30__n_s33)*Y(Si30)*state.rho - screened_rates(k_he4_si30__p_p33)*Y(Si30)*state.rho - screened_rates(k_he4_si30__s34)*Y(Si30)*state.rho;
    jac.set(Si30, He4, scratch);

    scratch = screened_rates(k_he4_mg26__si30)*Y(He4)*state.rho;
    jac.set(Si30, Mg26, scratch);

    scratch = screened_rates(k_he4_al27__p_si30)*Y(He4)*state.rho;
    jac.set(Si30, Al27, scratch);

    scratch = screened_rates(k_n_si29__si30)*Y(N)*state.rho;
    jac.set(Si30, Si29, scratch);

    scratch = -screened_rates(k_he4_si30__n_s33)*Y(He4)*state.rho - screened_rates(k_he4_si30__p_p33)*Y(He4)*state.rho - screened_rates(k_he4_si30__s34)*Y(He4)*state.rho - screened_rates(k_n_si30__si31)*Y(N)*state.rho - screened_rates(k_p_si30__he4_al27)*Y(H1)*state.rho - screened_rates(k_p_si30__n_p30)*Y(H1)*state.rho - screened_rates(k_p_si30__p31)*Y(H1)*state.rho - screened_rates(k_si30__he4_mg26) - screened_rates(k_si30__n_si29);
    jac.set(Si30, Si30, scratch);

    scratch = screened_rates(k_si31__n_si30);
    jac.set(Si30, Si31, scratch);

    scratch = screened_rates(k_n_p30__p_si30)*Y(N)*state.rho + screened_rates(k_p30__si30__weak__wc12);
    jac.set(Si30, P30, scratch);

    scratch = screened_rates(k_p31__p_si30);
    jac.set(Si30, P31, scratch);

    scratch = screened_rates(k_p_p33__he4_si30)*Y(H1)*state.rho;
    jac.set(Si30, P33, scratch);

    scratch = screened_rates(k_n_s33__he4_si30)*Y(N)*state.rho;
    jac.set(Si30, S33, scratch);

    scratch = screened_rates(k_s34__he4_si30);
    jac.set(Si30, S34, scratch);

    scratch = screened_rates(k_n_p31__p_si31)*Y(P31)*state.rho + screened_rates(k_n_s34__he4_si31)*Y(S34)*state.rho + screened_rates(k_n_si30__si31)*Y(Si30)*state.rho - screened_rates(k_n_si31__si32)*Y(Si31)*state.rho;
    jac.set(Si31, N, scratch);

    scratch = -screened_rates(k_p_si31__n_p31)*Y(Si31)*state.rho - screened_rates(k_p_si31__p32)*Y(Si31)*state.rho;
    jac.set(Si31, H1, scratch);

    scratch = -screened_rates(k_he4_si31__n_s34)*Y(Si31)*state.rho - screened_rates(k_he4_si31__s35)*Y(Si31)*state.rho;
    jac.set(Si31, He4, scratch);

    scratch = screened_rates(k_n_si30__si31)*Y(N)*state.rho;
    jac.set(Si31, Si30, scratch);

    scratch = -screened_rates(k_he4_si31__n_s34)*Y(He4)*state.rho - screened_rates(k_he4_si31__s35)*Y(He4)*state.rho - screened_rates(k_n_si31__si32)*Y(N)*state.rho - screened_rates(k_p_si31__n_p31)*Y(H1)*state.rho - screened_rates(k_p_si31__p32)*Y(H1)*state.rho - screened_rates(k_si31__n_si30) - screened_rates(k_si31__p31__weak__wc12);
    jac.set(Si31, Si31, scratch);

    scratch = screened_rates(k_si32__n_si31);
    jac.set(Si31, Si32, scratch);

    scratch = screened_rates(k_n_p31__p_si31)*Y(N)*state.rho;
    jac.set(Si31, P31, scratch);

    scratch = screened_rates(k_p32__p_si31);
    jac.set(Si31, P32, scratch);

    scratch = screened_rates(k_n_s34__he4_si31)*Y(N)*state.rho;
    jac.set(Si31, S34, scratch);

    scratch = screened_rates(k_s35__he4_si31);
    jac.set(Si31, S35, scratch);

    scratch = screened_rates(k_n_p32__p_si32)*Y(P32)*state.rho + screened_rates(k_n_s35__he4_si32)*Y(S35)*state.rho + screened_rates(k_n_si31__si32)*Y(Si31)*state.rho;
    jac.set(Si32, N, scratch);

    scratch = -screened_rates(k_p_si32__n_p32)*Y(Si32)*state.rho - screened_rates(k_p_si32__p33)*Y(Si32)*state.rho;
    jac.set(Si32, H1, scratch);

    scratch = -screened_rates(k_he4_si32__n_s35)*Y(Si32)*state.rho - screened_rates(k_he4_si32__s36)*Y(Si32)*state.rho;
    jac.set(Si32, He4, scratch);

    scratch = screened_rates(k_n_si31__si32)*Y(N)*state.rho;
    jac.set(Si32, Si31, scratch);

    scratch = -screened_rates(k_he4_si32__n_s35)*Y(He4)*state.rho - screened_rates(k_he4_si32__s36)*Y(He4)*state.rho - screened_rates(k_p_si32__n_p32)*Y(H1)*state.rho - screened_rates(k_p_si32__p33)*Y(H1)*state.rho - screened_rates(k_si32__n_si31) - screened_rates(k_si32__p32__weak__wc12);
    jac.set(Si32, Si32, scratch);

    scratch = screened_rates(k_n_p32__p_si32)*Y(N)*state.rho;
    jac.set(Si32, P32, scratch);

    scratch = screened_rates(k_p33__p_si32);
    jac.set(Si32, P33, scratch);

    scratch = screened_rates(k_n_s35__he4_si32)*Y(N)*state.rho;
    jac.set(Si32, S35, scratch);

    scratch = screened_rates(k_s36__he4_si32);
    jac.set(Si32, S36, scratch);

    scratch = -screened_rates(k_n_p29__he4_al26)*Y(P29)*state.rho - screened_rates(k_n_p29__p30)*Y(P29)*state.rho - screened_rates(k_n_p29__p_si29)*Y(P29)*state.rho;
    jac.set(P29, N, scratch);

    scratch = screened_rates(k_p_s32__he4_p29)*Y(S32)*state.rho + screened_rates(k_p_si28__p29)*Y(Si28)*state.rho + screened_rates(k_p_si29__n_p29)*Y(Si29)*state.rho;
    jac.set(P29, H1, scratch);

    scratch = screened_rates(k_he4_al25__p29)*Y(Al25)*state.rho + screened_rates(k_he4_al26__n_p29)*Y(Al26)*state.rho - screened_rates(k_he4_p29__cl33)*Y(P29)*state.rho - screened_rates(k_he4_p29__p_s32)*Y(P29)*state.rho;
    jac.set(P29, He4, scratch);

    scratch = screened_rates(k_he4_al25__p29)*Y(He4)*state.rho;
    jac.set(P29, Al25, scratch);

    scratch = screened_rates(k_he4_al26__n_p29)*Y(He4)*state.rho;
    jac.set(P29, Al26, scratch);

    scratch = screened_rates(k_p_si28__p29)*Y(H1)*state.rho;
    jac.set(P29, Si28, scratch);

    scratch = screened_rates(k_p_si29__n_p29)*Y(H1)*state.rho;
    jac.set(P29, Si29, scratch);

    scratch = -screened_rates(k_he4_p29__cl33)*Y(He4)*state.rho - screened_rates(k_he4_p29__p_s32)*Y(He4)*state.rho - screened_rates(k_n_p29__he4_al26)*Y(N)*state.rho - screened_rates(k_n_p29__p30)*Y(N)*state.rho - screened_rates(k_n_p29__p_si29)*Y(N)*state.rho - screened_rates(k_p29__he4_al25) - screened_rates(k_p29__p_si28) - screened_rates(k_p29__si29__weak__wc12);
    jac.set(P29, P29, scratch);

    scratch = screened_rates(k_p30__n_p29);
    jac.set(P29, P30, scratch);

    scratch = screened_rates(k_p_s32__he4_p29)*Y(H1)*state.rho;
    jac.set(P29, S32, scratch);

    scratch = screened_rates(k_cl33__he4_p29);
    jac.set(P29, Cl33, scratch);

    scratch = screened_rates(k_n_cl33__he4_p30)*Y(Cl33)*state.rho + screened_rates(k_n_p29__p30)*Y(P29)*state.rho - screened_rates(k_n_p30__he4_al27)*Y(P30)*state.rho - screened_rates(k_n_p30__p31)*Y(P30)*state.rho - screened_rates(k_n_p30__p_si30)*Y(P30)*state.rho;
    jac.set(P30, N, scratch);

    scratch = screened_rates(k_p_s33__he4_p30)*Y(S33)*state.rho + screened_rates(k_p_si29__p30)*Y(Si29)*state.rho + screened_rates(k_p_si30__n_p30)*Y(Si30)*state.rho;
    jac.set(P30, H1, scratch);

    scratch = screened_rates(k_he4_al26__p30)*Y(Al26)*state.rho + screened_rates(k_he4_al27__n_p30)*Y(Al27)*state.rho - screened_rates(k_he4_p30__cl34)*Y(P30)*state.rho - screened_rates(k_he4_p30__n_cl33)*Y(P30)*state.rho - screened_rates(k_he4_p30__p_s33)*Y(P30)*state.rho;
    jac.set(P30, He4, scratch);

    scratch = screened_rates(k_he4_al26__p30)*Y(He4)*state.rho;
    jac.set(P30, Al26, scratch);

    scratch = screened_rates(k_he4_al27__n_p30)*Y(He4)*state.rho;
    jac.set(P30, Al27, scratch);

    scratch = screened_rates(k_p_si29__p30)*Y(H1)*state.rho;
    jac.set(P30, Si29, scratch);

    scratch = screened_rates(k_p_si30__n_p30)*Y(H1)*state.rho;
    jac.set(P30, Si30, scratch);

    scratch = screened_rates(k_n_p29__p30)*Y(N)*state.rho;
    jac.set(P30, P29, scratch);

    scratch = -screened_rates(k_he4_p30__cl34)*Y(He4)*state.rho - screened_rates(k_he4_p30__n_cl33)*Y(He4)*state.rho - screened_rates(k_he4_p30__p_s33)*Y(He4)*state.rho - screened_rates(k_n_p30__he4_al27)*Y(N)*state.rho - screened_rates(k_n_p30__p31)*Y(N)*state.rho - screened_rates(k_n_p30__p_si30)*Y(N)*state.rho - screened_rates(k_p30__he4_al26) - screened_rates(k_p30__n_p29) - screened_rates(k_p30__p_si29) - screened_rates(k_p30__si30__weak__wc12);
    jac.set(P30, P30, scratch);

    scratch = screened_rates(k_p31__n_p30);
    jac.set(P30, P31, scratch);

    scratch = screened_rates(k_p_s33__he4_p30)*Y(H1)*state.rho;
    jac.set(P30, S33, scratch);

    scratch = screened_rates(k_n_cl33__he4_p30)*Y(N)*state.rho;
    jac.set(P30, Cl33, scratch);

    scratch = screened_rates(k_cl34__he4_p30);
    jac.set(P30, Cl34, scratch);

    scratch = screened_rates(k_n_cl34__he4_p31)*Y(Cl34)*state.rho + screened_rates(k_n_p30__p31)*Y(P30)*state.rho - screened_rates(k_n_p31__p32)*Y(P31)*state.rho - screened_rates(k_n_p31__p_si31)*Y(P31)*state.rho;
    jac.set(P31, N, scratch);

    scratch = -screened_rates(k_p_p31__c12_ne20)*Y(P31)*state.rho - screened_rates(k_p_p31__he4_si28)*Y(P31)*state.rho - screened_rates(k_p_p31__o16_o16)*Y(P31)*state.rho - screened_rates(k_p_p31__s32)*Y(P31)*state.rho + screened_rates(k_p_s34__he4_p31)*Y(S34)*state.rho + screened_rates(k_p_si30__p31)*Y(Si30)*state.rho + screened_rates(k_p_si31__n_p31)*Y(Si31)*state.rho;
    jac.set(P31, H1, scratch);

    scratch = screened_rates(k_he4_al27__p31)*Y(Al27)*state.rho - screened_rates(k_he4_p31__cl35)*Y(P31)*state.rho - screened_rates(k_he4_p31__n_cl34)*Y(P31)*state.rho - screened_rates(k_he4_p31__p_s34)*Y(P31)*state.rho + screened_rates(k_he4_si28__p_p31)*Y(Si28)*state.rho;
    jac.set(P31, He4, scratch);

    scratch = screened_rates(k_c12_ne20__p_p31)*Y(Ne20)*state.rho;
    jac.set(P31, C12, scratch);

    scratch = 1.0*screened_rates(k_o16_o16__p_p31)*Y(O16)*state.rho;
    jac.set(P31, O16, scratch);

    scratch = screened_rates(k_c12_ne20__p_p31)*Y(C12)*state.rho;
    jac.set(P31, Ne20, scratch);

    scratch = screened_rates(k_he4_al27__p31)*Y(He4)*state.rho;
    jac.set(P31, Al27, scratch);

    scratch = screened_rates(k_he4_si28__p_p31)*Y(He4)*state.rho;
    jac.set(P31, Si28, scratch);

    scratch = screened_rates(k_p_si30__p31)*Y(H1)*state.rho;
    jac.set(P31, Si30, scratch);

    scratch = screened_rates(k_p_si31__n_p31)*Y(H1)*state.rho + screened_rates(k_si31__p31__weak__wc12);
    jac.set(P31, Si31, scratch);

    scratch = screened_rates(k_n_p30__p31)*Y(N)*state.rho;
    jac.set(P31, P30, scratch);

    scratch = -screened_rates(k_he4_p31__cl35)*Y(He4)*state.rho - screened_rates(k_he4_p31__n_cl34)*Y(He4)*state.rho - screened_rates(k_he4_p31__p_s34)*Y(He4)*state.rho - screened_rates(k_n_p31__p32)*Y(N)*state.rho - screened_rates(k_n_p31__p_si31)*Y(N)*state.rho - screened_rates(k_p31__he4_al27) - screened_rates(k_p31__n_p30) - screened_rates(k_p31__p_si30) - screened_rates(k_p_p31__c12_ne20)*Y(H1)*state.rho - screened_rates(k_p_p31__he4_si28)*Y(H1)*state.rho - screened_rates(k_p_p31__o16_o16)*Y(H1)*state.rho - screened_rates(k_p_p31__s32)*Y(H1)*state.rho;
    jac.set(P31, P31, scratch);

    scratch = screened_rates(k_p32__n_p31);
    jac.set(P31, P32, scratch);

    scratch = screened_rates(k_s32__p_p31);
    jac.set(P31, S32, scratch);

    scratch = screened_rates(k_p_s34__he4_p31)*Y(H1)*state.rho;
    jac.set(P31, S34, scratch);

    scratch = screened_rates(k_n_cl34__he4_p31)*Y(N)*state.rho;
    jac.set(P31, Cl34, scratch);

    scratch = screened_rates(k_cl35__he4_p31);
    jac.set(P31, Cl35, scratch);

    scratch = screened_rates(k_n_cl35__he4_p32)*Y(Cl35)*state.rho + screened_rates(k_n_p31__p32)*Y(P31)*state.rho - screened_rates(k_n_p32__p33)*Y(P32)*state.rho - screened_rates(k_n_p32__p_si32)*Y(P32)*state.rho + screened_rates(k_n_s32__p_p32)*Y(S32)*state.rho;
    jac.set(P32, N, scratch);

    scratch = -screened_rates(k_p_p32__he4_si29)*Y(P32)*state.rho - screened_rates(k_p_p32__n_s32)*Y(P32)*state.rho - screened_rates(k_p_p32__s33)*Y(P32)*state.rho + screened_rates(k_p_s35__he4_p32)*Y(S35)*state.rho + screened_rates(k_p_si31__p32)*Y(Si31)*state.rho + screened_rates(k_p_si32__n_p32)*Y(Si32)*state.rho;
    jac.set(P32, H1, scratch);

    scratch = -screened_rates(k_he4_p32__cl36)*Y(P32)*state.rho - screened_rates(k_he4_p32__n_cl35)*Y(P32)*state.rho - screened_rates(k_he4_p32__p_s35)*Y(P32)*state.rho + screened_rates(k_he4_si29__p_p32)*Y(Si29)*state.rho;
    jac.set(P32, He4, scratch);

    scratch = screened_rates(k_he4_si29__p_p32)*Y(He4)*state.rho;
    jac.set(P32, Si29, scratch);

    scratch = screened_rates(k_p_si31__p32)*Y(H1)*state.rho;
    jac.set(P32, Si31, scratch);

    scratch = screened_rates(k_p_si32__n_p32)*Y(H1)*state.rho + screened_rates(k_si32__p32__weak__wc12);
    jac.set(P32, Si32, scratch);

    scratch = screened_rates(k_n_p31__p32)*Y(N)*state.rho;
    jac.set(P32, P31, scratch);

    scratch = -screened_rates(k_he4_p32__cl36)*Y(He4)*state.rho - screened_rates(k_he4_p32__n_cl35)*Y(He4)*state.rho - screened_rates(k_he4_p32__p_s35)*Y(He4)*state.rho - screened_rates(k_n_p32__p33)*Y(N)*state.rho - screened_rates(k_n_p32__p_si32)*Y(N)*state.rho - screened_rates(k_p32__n_p31) - screened_rates(k_p32__p_si31) - screened_rates(k_p32__s32__weak__wc12) - screened_rates(k_p_p32__he4_si29)*Y(H1)*state.rho - screened_rates(k_p_p32__n_s32)*Y(H1)*state.rho - screened_rates(k_p_p32__s33)*Y(H1)*state.rho;
    jac.set(P32, P32, scratch);

    scratch = screened_rates(k_p33__n_p32);
    jac.set(P32, P33, scratch);

    scratch = screened_rates(k_n_s32__p_p32)*Y(N)*state.rho;
    jac.set(P32, S32, scratch);

    scratch = screened_rates(k_s33__p_p32);
    jac.set(P32, S33, scratch);

    scratch = screened_rates(k_p_s35__he4_p32)*Y(H1)*state.rho;
    jac.set(P32, S35, scratch);

    scratch = screened_rates(k_n_cl35__he4_p32)*Y(N)*state.rho;
    jac.set(P32, Cl35, scratch);

    scratch = screened_rates(k_cl36__he4_p32);
    jac.set(P32, Cl36, scratch);

    scratch = screened_rates(k_n_cl36__he4_p33)*Y(Cl36)*state.rho + screened_rates(k_n_p32__p33)*Y(P32)*state.rho + screened_rates(k_n_s33__p_p33)*Y(S33)*state.rho;
    jac.set(P33, N, scratch);

    scratch = -screened_rates(k_p_p33__he4_si30)*Y(P33)*state.rho - screened_rates(k_p_p33__n_s33)*Y(P33)*state.rho - screened_rates(k_p_p33__s34)*Y(P33)*state.rho + screened_rates(k_p_s36__he4_p33)*Y(S36)*state.rho + screened_rates(k_p_si32__p33)*Y(Si32)*state.rho;
    jac.set(P33, H1, scratch);

    scratch = -screened_rates(k_he4_p33__cl37)*Y(P33)*state.rho - screened_rates(k_he4_p33__n_cl36)*Y(P33)*state.rho - screened_rates(k_he4_p33__p_s36)*Y(P33)*state.rho + screened_rates(k_he4_si30__p_p33)*Y(Si30)*state.rho;
    jac.set(P33, He4, scratch);

    scratch = screened_rates(k_he4_si30__p_p33)*Y(He4)*state.rho;
    jac.set(P33, Si30, scratch);

    scratch = screened_rates(k_p_si32__p33)*Y(H1)*state.rho;
    jac.set(P33, Si32, scratch);

    scratch = screened_rates(k_n_p32__p33)*Y(N)*state.rho;
    jac.set(P33, P32, scratch);

    scratch = -screened_rates(k_he4_p33__cl37)*Y(He4)*state.rho - screened_rates(k_he4_p33__n_cl36)*Y(He4)*state.rho - screened_rates(k_he4_p33__p_s36)*Y(He4)*state.rho - screened_rates(k_p33__n_p32) - screened_rates(k_p33__p_si32) - screened_rates(k_p33__s33__weak__wc12) - screened_rates(k_p_p33__he4_si30)*Y(H1)*state.rho - screened_rates(k_p_p33__n_s33)*Y(H1)*state.rho - screened_rates(k_p_p33__s34)*Y(H1)*state.rho;
    jac.set(P33, P33, scratch);

    scratch = screened_rates(k_n_s33__p_p33)*Y(N)*state.rho;
    jac.set(P33, S33, scratch);

    scratch = screened_rates(k_s34__p_p33);
    jac.set(P33, S34, scratch);

    scratch = screened_rates(k_p_s36__he4_p33)*Y(H1)*state.rho;
    jac.set(P33, S36, scratch);

    scratch = screened_rates(k_n_cl36__he4_p33)*Y(N)*state.rho;
    jac.set(P33, Cl36, scratch);

    scratch = screened_rates(k_cl37__he4_p33);
    jac.set(P33, Cl37, scratch);

    scratch = -screened_rates(k_n_s32__he4_si29)*Y(S32)*state.rho - screened_rates(k_n_s32__p_p32)*Y(S32)*state.rho - screened_rates(k_n_s32__s33)*Y(S32)*state.rho;
    jac.set(S32, N, scratch);

    scratch = screened_rates(k_p_cl35__he4_s32)*Y(Cl35)*state.rho + screened_rates(k_p_p31__s32)*Y(P31)*state.rho + screened_rates(k_p_p32__n_s32)*Y(P32)*state.rho - screened_rates(k_p_s32__cl33)*Y(S32)*state.rho - screened_rates(k_p_s32__he4_p29)*Y(S32)*state.rho;
    jac.set(S32, H1, scratch);

    scratch = screened_rates(k_he4_p29__p_s32)*Y(P29)*state.rho - screened_rates(k_he4_s32__ar36)*Y(S32)*state.rho - screened_rates(k_he4_s32__p_cl35)*Y(S32)*state.rho + screened_rates(k_he4_si28__s32)*Y(Si28)*state.rho + screened_rates(k_he4_si29__n_s32)*Y(Si29)*state.rho;
    jac.set(S32, He4, scratch);

    scratch = screened_rates(k_he4_si28__s32)*Y(He4)*state.rho;
    jac.set(S32, Si28, scratch);

    scratch = screened_rates(k_he4_si29__n_s32)*Y(He4)*state.rho;
    jac.set(S32, Si29, scratch);

    scratch = screened_rates(k_he4_p29__p_s32)*Y(He4)*state.rho;
    jac.set(S32, P29, scratch);

    scratch = screened_rates(k_p_p31__s32)*Y(H1)*state.rho;
    jac.set(S32, P31, scratch);

    scratch = screened_rates(k_p32__s32__weak__wc12) + screened_rates(k_p_p32__n_s32)*Y(H1)*state.rho;
    jac.set(S32, P32, scratch);

    scratch = -screened_rates(k_he4_s32__ar36)*Y(He4)*state.rho - screened_rates(k_he4_s32__p_cl35)*Y(He4)*state.rho - screened_rates(k_n_s32__he4_si29)*Y(N)*state.rho - screened_rates(k_n_s32__p_p32)*Y(N)*state.rho - screened_rates(k_n_s32__s33)*Y(N)*state.rho - screened_rates(k_p_s32__cl33)*Y(H1)*state.rho - screened_rates(k_p_s32__he4_p29)*Y(H1)*state.rho - screened_rates(k_s32__he4_si28) - screened_rates(k_s32__p_p31);
    jac.set(S32, S32, scratch);

    scratch = screened_rates(k_s33__n_s32);
    jac.set(S32, S33, scratch);

    scratch = screened_rates(k_cl33__p_s32);
    jac.set(S32, Cl33, scratch);

    scratch = screened_rates(k_p_cl35__he4_s32)*Y(H1)*state.rho;
    jac.set(S32, Cl35, scratch);

    scratch = screened_rates(k_ar36__he4_s32);
    jac.set(S32, Ar36, scratch);

    scratch = screened_rates(k_n_ar36__he4_s33)*Y(Ar36)*state.rho + screened_rates(k_n_cl33__p_s33)*Y(Cl33)*state.rho + screened_rates(k_n_s32__s33)*Y(S32)*state.rho - screened_rates(k_n_s33__he4_si30)*Y(S33)*state.rho - screened_rates(k_n_s33__p_p33)*Y(S33)*state.rho - screened_rates(k_n_s33__s34)*Y(S33)*state.rho;
    jac.set(S33, N, scratch);

    scratch = screened_rates(k_p_cl36__he4_s33)*Y(Cl36)*state.rho + screened_rates(k_p_p32__s33)*Y(P32)*state.rho + screened_rates(k_p_p33__n_s33)*Y(P33)*state.rho - screened_rates(k_p_s33__cl34)*Y(S33)*state.rho - screened_rates(k_p_s33__he4_p30)*Y(S33)*state.rho - screened_rates(k_p_s33__n_cl33)*Y(S33)*state.rho;
    jac.set(S33, H1, scratch);

    scratch = screened_rates(k_he4_p30__p_s33)*Y(P30)*state.rho - screened_rates(k_he4_s33__ar37)*Y(S33)*state.rho - screened_rates(k_he4_s33__n_ar36)*Y(S33)*state.rho - screened_rates(k_he4_s33__p_cl36)*Y(S33)*state.rho + screened_rates(k_he4_si29__s33)*Y(Si29)*state.rho + screened_rates(k_he4_si30__n_s33)*Y(Si30)*state.rho;
    jac.set(S33, He4, scratch);

    scratch = screened_rates(k_he4_si29__s33)*Y(He4)*state.rho;
    jac.set(S33, Si29, scratch);

    scratch = screened_rates(k_he4_si30__n_s33)*Y(He4)*state.rho;
    jac.set(S33, Si30, scratch);

    scratch = screened_rates(k_he4_p30__p_s33)*Y(He4)*state.rho;
    jac.set(S33, P30, scratch);

    scratch = screened_rates(k_p_p32__s33)*Y(H1)*state.rho;
    jac.set(S33, P32, scratch);

    scratch = screened_rates(k_p33__s33__weak__wc12) + screened_rates(k_p_p33__n_s33)*Y(H1)*state.rho;
    jac.set(S33, P33, scratch);

    scratch = screened_rates(k_n_s32__s33)*Y(N)*state.rho;
    jac.set(S33, S32, scratch);

    scratch = -screened_rates(k_he4_s33__ar37)*Y(He4)*state.rho - screened_rates(k_he4_s33__n_ar36)*Y(He4)*state.rho - screened_rates(k_he4_s33__p_cl36)*Y(He4)*state.rho - screened_rates(k_n_s33__he4_si30)*Y(N)*state.rho - screened_rates(k_n_s33__p_p33)*Y(N)*state.rho - screened_rates(k_n_s33__s34)*Y(N)*state.rho - screened_rates(k_p_s33__cl34)*Y(H1)*state.rho - screened_rates(k_p_s33__he4_p30)*Y(H1)*state.rho - screened_rates(k_p_s33__n_cl33)*Y(H1)*state.rho - screened_rates(k_s33__he4_si29) - screened_rates(k_s33__n_s32) - screened_rates(k_s33__p_p32);
    jac.set(S33, S33, scratch);

    scratch = screened_rates(k_s34__n_s33);
    jac.set(S33, S34, scratch);

    scratch = screened_rates(k_cl33__s33__weak__wc12) + screened_rates(k_n_cl33__p_s33)*Y(N)*state.rho;
    jac.set(S33, Cl33, scratch);

    scratch = screened_rates(k_cl34__p_s33);
    jac.set(S33, Cl34, scratch);

    scratch = screened_rates(k_p_cl36__he4_s33)*Y(H1)*state.rho;
    jac.set(S33, Cl36, scratch);

    scratch = screened_rates(k_n_ar36__he4_s33)*Y(N)*state.rho;
    jac.set(S33, Ar36, scratch);

    scratch = screened_rates(k_ar37__he4_s33);
    jac.set(S33, Ar37, scratch);

    scratch = screened_rates(k_n_ar37__he4_s34)*Y(Ar37)*state.rho + screened_rates(k_n_cl34__p_s34)*Y(Cl34)*state.rho + screened_rates(k_n_s33__s34)*Y(S33)*state.rho - screened_rates(k_n_s34__he4_si31)*Y(S34)*state.rho - screened_rates(k_n_s34__s35)*Y(S34)*state.rho;
    jac.set(S34, N, scratch);

    scratch = screened_rates(k_p_cl37__he4_s34)*Y(Cl37)*state.rho + screened_rates(k_p_p33__s34)*Y(P33)*state.rho - screened_rates(k_p_s34__cl35)*Y(S34)*state.rho - screened_rates(k_p_s34__he4_p31)*Y(S34)*state.rho - screened_rates(k_p_s34__n_cl34)*Y(S34)*state.rho;
    jac.set(S34, H1, scratch);

    scratch = screened_rates(k_he4_p31__p_s34)*Y(P31)*state.rho - screened_rates(k_he4_s34__ar38)*Y(S34)*state.rho - screened_rates(k_he4_s34__n_ar37)*Y(S34)*state.rho - screened_rates(k_he4_s34__p_cl37)*Y(S34)*state.rho + screened_rates(k_he4_si30__s34)*Y(Si30)*state.rho + screened_rates(k_he4_si31__n_s34)*Y(Si31)*state.rho;
    jac.set(S34, He4, scratch);

    scratch = screened_rates(k_he4_si30__s34)*Y(He4)*state.rho;
    jac.set(S34, Si30, scratch);

    scratch = screened_rates(k_he4_si31__n_s34)*Y(He4)*state.rho;
    jac.set(S34, Si31, scratch);

    scratch = screened_rates(k_he4_p31__p_s34)*Y(He4)*state.rho;
    jac.set(S34, P31, scratch);

    scratch = screened_rates(k_p_p33__s34)*Y(H1)*state.rho;
    jac.set(S34, P33, scratch);

    scratch = screened_rates(k_n_s33__s34)*Y(N)*state.rho;
    jac.set(S34, S33, scratch);

    scratch = -screened_rates(k_he4_s34__ar38)*Y(He4)*state.rho - screened_rates(k_he4_s34__n_ar37)*Y(He4)*state.rho - screened_rates(k_he4_s34__p_cl37)*Y(He4)*state.rho - screened_rates(k_n_s34__he4_si31)*Y(N)*state.rho - screened_rates(k_n_s34__s35)*Y(N)*state.rho - screened_rates(k_p_s34__cl35)*Y(H1)*state.rho - screened_rates(k_p_s34__he4_p31)*Y(H1)*state.rho - screened_rates(k_p_s34__n_cl34)*Y(H1)*state.rho - screened_rates(k_s34__he4_si30) - screened_rates(k_s34__n_s33) - screened_rates(k_s34__p_p33);
    jac.set(S34, S34, scratch);

    scratch = screened_rates(k_s35__n_s34);
    jac.set(S34, S35, scratch);

    scratch = screened_rates(k_cl34__s34__weak__wc12) + screened_rates(k_n_cl34__p_s34)*Y(N)*state.rho;
    jac.set(S34, Cl34, scratch);

    scratch = screened_rates(k_cl35__p_s34);
    jac.set(S34, Cl35, scratch);

    scratch = screened_rates(k_p_cl37__he4_s34)*Y(H1)*state.rho;
    jac.set(S34, Cl37, scratch);

    scratch = screened_rates(k_n_ar37__he4_s34)*Y(N)*state.rho;
    jac.set(S34, Ar37, scratch);

    scratch = screened_rates(k_ar38__he4_s34);
    jac.set(S34, Ar38, scratch);

    scratch = screened_rates(k_n_ar38__he4_s35)*Y(Ar38)*state.rho + screened_rates(k_n_cl35__p_s35)*Y(Cl35)*state.rho + screened_rates(k_n_s34__s35)*Y(S34)*state.rho - screened_rates(k_n_s35__he4_si32)*Y(S35)*state.rho - screened_rates(k_n_s35__s36)*Y(S35)*state.rho;
    jac.set(S35, N, scratch);

    scratch = -screened_rates(k_p_s35__cl36)*Y(S35)*state.rho - screened_rates(k_p_s35__he4_p32)*Y(S35)*state.rho - screened_rates(k_p_s35__n_cl35)*Y(S35)*state.rho;
    jac.set(S35, H1, scratch);

    scratch = screened_rates(k_he4_p32__p_s35)*Y(P32)*state.rho - screened_rates(k_he4_s35__ar39)*Y(S35)*state.rho - screened_rates(k_he4_s35__n_ar38)*Y(S35)*state.rho + screened_rates(k_he4_si31__s35)*Y(Si31)*state.rho + screened_rates(k_he4_si32__n_s35)*Y(Si32)*state.rho;
    jac.set(S35, He4, scratch);

    scratch = screened_rates(k_he4_si31__s35)*Y(He4)*state.rho;
    jac.set(S35, Si31, scratch);

    scratch = screened_rates(k_he4_si32__n_s35)*Y(He4)*state.rho;
    jac.set(S35, Si32, scratch);

    scratch = screened_rates(k_he4_p32__p_s35)*Y(He4)*state.rho;
    jac.set(S35, P32, scratch);

    scratch = screened_rates(k_n_s34__s35)*Y(N)*state.rho;
    jac.set(S35, S34, scratch);

    scratch = -screened_rates(k_he4_s35__ar39)*Y(He4)*state.rho - screened_rates(k_he4_s35__n_ar38)*Y(He4)*state.rho - screened_rates(k_n_s35__he4_si32)*Y(N)*state.rho - screened_rates(k_n_s35__s36)*Y(N)*state.rho - screened_rates(k_p_s35__cl36)*Y(H1)*state.rho - screened_rates(k_p_s35__he4_p32)*Y(H1)*state.rho - screened_rates(k_p_s35__n_cl35)*Y(H1)*state.rho - screened_rates(k_s35__cl35__weak__wc12) - screened_rates(k_s35__he4_si31) - screened_rates(k_s35__n_s34);
    jac.set(S35, S35, scratch);

    scratch = screened_rates(k_s36__n_s35);
    jac.set(S35, S36, scratch);

    scratch = screened_rates(k_n_cl35__p_s35)*Y(N)*state.rho;
    jac.set(S35, Cl35, scratch);

    scratch = screened_rates(k_cl36__p_s35);
    jac.set(S35, Cl36, scratch);

    scratch = screened_rates(k_n_ar38__he4_s35)*Y(N)*state.rho;
    jac.set(S35, Ar38, scratch);

    scratch = screened_rates(k_ar39__he4_s35);
    jac.set(S35, Ar39, scratch);

    scratch = screened_rates(k_n_ar39__he4_s36)*Y(Ar39)*state.rho + screened_rates(k_n_cl36__p_s36)*Y(Cl36)*state.rho + screened_rates(k_n_s35__s36)*Y(S35)*state.rho;
    jac.set(S36, N, scratch);

    scratch = -screened_rates(k_p_s36__cl37)*Y(S36)*state.rho - screened_rates(k_p_s36__he4_p33)*Y(S36)*state.rho - screened_rates(k_p_s36__n_cl36)*Y(S36)*state.rho;
    jac.set(S36, H1, scratch);

    scratch = screened_rates(k_he4_p33__p_s36)*Y(P33)*state.rho - screened_rates(k_he4_s36__ar40)*Y(S36)*state.rho - screened_rates(k_he4_s36__n_ar39)*Y(S36)*state.rho + screened_rates(k_he4_si32__s36)*Y(Si32)*state.rho;
    jac.set(S36, He4, scratch);

    scratch = screened_rates(k_he4_si32__s36)*Y(He4)*state.rho;
    jac.set(S36, Si32, scratch);

    scratch = screened_rates(k_he4_p33__p_s36)*Y(He4)*state.rho;
    jac.set(S36, P33, scratch);

    scratch = screened_rates(k_n_s35__s36)*Y(N)*state.rho;
    jac.set(S36, S35, scratch);

    scratch = -screened_rates(k_he4_s36__ar40)*Y(He4)*state.rho - screened_rates(k_he4_s36__n_ar39)*Y(He4)*state.rho - screened_rates(k_p_s36__cl37)*Y(H1)*state.rho - screened_rates(k_p_s36__he4_p33)*Y(H1)*state.rho - screened_rates(k_p_s36__n_cl36)*Y(H1)*state.rho - screened_rates(k_s36__he4_si32) - screened_rates(k_s36__n_s35);
    jac.set(S36, S36, scratch);

    scratch = screened_rates(k_cl36__s36__weak__wc12) + screened_rates(k_n_cl36__p_s36)*Y(N)*state.rho;
    jac.set(S36, Cl36, scratch);

    scratch = screened_rates(k_cl37__p_s36);
    jac.set(S36, Cl37, scratch);

    scratch = screened_rates(k_n_ar39__he4_s36)*Y(N)*state.rho;
    jac.set(S36, Ar39, scratch);

    scratch = screened_rates(k_ar40__he4_s36);
    jac.set(S36, Ar40, scratch);

    scratch = -screened_rates(k_n_cl33__cl34)*Y(Cl33)*state.rho - screened_rates(k_n_cl33__he4_p30)*Y(Cl33)*state.rho - screened_rates(k_n_cl33__p_s33)*Y(Cl33)*state.rho;
    jac.set(Cl33, N, scratch);

    scratch = screened_rates(k_p_ar36__he4_cl33)*Y(Ar36)*state.rho + screened_rates(k_p_s32__cl33)*Y(S32)*state.rho + screened_rates(k_p_s33__n_cl33)*Y(S33)*state.rho;
    jac.set(Cl33, H1, scratch);

    scratch = -screened_rates(k_he4_cl33__k37)*Y(Cl33)*state.rho - screened_rates(k_he4_cl33__p_ar36)*Y(Cl33)*state.rho + screened_rates(k_he4_p29__cl33)*Y(P29)*state.rho + screened_rates(k_he4_p30__n_cl33)*Y(P30)*state.rho;
    jac.set(Cl33, He4, scratch);

    scratch = screened_rates(k_he4_p29__cl33)*Y(He4)*state.rho;
    jac.set(Cl33, P29, scratch);

    scratch = screened_rates(k_he4_p30__n_cl33)*Y(He4)*state.rho;
    jac.set(Cl33, P30, scratch);

    scratch = screened_rates(k_p_s32__cl33)*Y(H1)*state.rho;
    jac.set(Cl33, S32, scratch);

    scratch = screened_rates(k_p_s33__n_cl33)*Y(H1)*state.rho;
    jac.set(Cl33, S33, scratch);

    scratch = -screened_rates(k_cl33__he4_p29) - screened_rates(k_cl33__p_s32) - screened_rates(k_cl33__s33__weak__wc12) - screened_rates(k_he4_cl33__k37)*Y(He4)*state.rho - screened_rates(k_he4_cl33__p_ar36)*Y(He4)*state.rho - screened_rates(k_n_cl33__cl34)*Y(N)*state.rho - screened_rates(k_n_cl33__he4_p30)*Y(N)*state.rho - screened_rates(k_n_cl33__p_s33)*Y(N)*state.rho;
    jac.set(Cl33, Cl33, scratch);

    scratch = screened_rates(k_cl34__n_cl33);
    jac.set(Cl33, Cl34, scratch);

    scratch = screened_rates(k_p_ar36__he4_cl33)*Y(H1)*state.rho;
    jac.set(Cl33, Ar36, scratch);

    scratch = screened_rates(k_k37__he4_cl33);
    jac.set(Cl33, K37, scratch);

    scratch = screened_rates(k_n_cl33__cl34)*Y(Cl33)*state.rho - screened_rates(k_n_cl34__cl35)*Y(Cl34)*state.rho - screened_rates(k_n_cl34__he4_p31)*Y(Cl34)*state.rho - screened_rates(k_n_cl34__p_s34)*Y(Cl34)*state.rho + screened_rates(k_n_k37__he4_cl34)*Y(K37)*state.rho;
    jac.set(Cl34, N, scratch);

    scratch = screened_rates(k_p_ar37__he4_cl34)*Y(Ar37)*state.rho + screened_rates(k_p_s33__cl34)*Y(S33)*state.rho + screened_rates(k_p_s34__n_cl34)*Y(S34)*state.rho;
    jac.set(Cl34, H1, scratch);

    scratch = -screened_rates(k_he4_cl34__k38)*Y(Cl34)*state.rho - screened_rates(k_he4_cl34__n_k37)*Y(Cl34)*state.rho - screened_rates(k_he4_cl34__p_ar37)*Y(Cl34)*state.rho + screened_rates(k_he4_p30__cl34)*Y(P30)*state.rho + screened_rates(k_he4_p31__n_cl34)*Y(P31)*state.rho;
    jac.set(Cl34, He4, scratch);

    scratch = screened_rates(k_he4_p30__cl34)*Y(He4)*state.rho;
    jac.set(Cl34, P30, scratch);

    scratch = screened_rates(k_he4_p31__n_cl34)*Y(He4)*state.rho;
    jac.set(Cl34, P31, scratch);

    scratch = screened_rates(k_p_s33__cl34)*Y(H1)*state.rho;
    jac.set(Cl34, S33, scratch);

    scratch = screened_rates(k_p_s34__n_cl34)*Y(H1)*state.rho;
    jac.set(Cl34, S34, scratch);

    scratch = screened_rates(k_n_cl33__cl34)*Y(N)*state.rho;
    jac.set(Cl34, Cl33, scratch);

    scratch = -screened_rates(k_cl34__he4_p30) - screened_rates(k_cl34__n_cl33) - screened_rates(k_cl34__p_s33) - screened_rates(k_cl34__s34__weak__wc12) - screened_rates(k_he4_cl34__k38)*Y(He4)*state.rho - screened_rates(k_he4_cl34__n_k37)*Y(He4)*state.rho - screened_rates(k_he4_cl34__p_ar37)*Y(He4)*state.rho - screened_rates(k_n_cl34__cl35)*Y(N)*state.rho - screened_rates(k_n_cl34__he4_p31)*Y(N)*state.rho - screened_rates(k_n_cl34__p_s34)*Y(N)*state.rho;
    jac.set(Cl34, Cl34, scratch);

    scratch = screened_rates(k_cl35__n_cl34);
    jac.set(Cl34, Cl35, scratch);

    scratch = screened_rates(k_p_ar37__he4_cl34)*Y(H1)*state.rho;
    jac.set(Cl34, Ar37, scratch);

    scratch = screened_rates(k_n_k37__he4_cl34)*Y(N)*state.rho;
    jac.set(Cl34, K37, scratch);

    scratch = screened_rates(k_k38__he4_cl34);
    jac.set(Cl34, K38, scratch);

    scratch = screened_rates(k_n_cl34__cl35)*Y(Cl34)*state.rho - screened_rates(k_n_cl35__cl36)*Y(Cl35)*state.rho - screened_rates(k_n_cl35__he4_p32)*Y(Cl35)*state.rho - screened_rates(k_n_cl35__p_s35)*Y(Cl35)*state.rho + screened_rates(k_n_k38__he4_cl35)*Y(K38)*state.rho;
    jac.set(Cl35, N, scratch);

    scratch = screened_rates(k_p_ar38__he4_cl35)*Y(Ar38)*state.rho - screened_rates(k_p_cl35__ar36)*Y(Cl35)*state.rho - screened_rates(k_p_cl35__he4_s32)*Y(Cl35)*state.rho + screened_rates(k_p_s34__cl35)*Y(S34)*state.rho + screened_rates(k_p_s35__n_cl35)*Y(S35)*state.rho;
    jac.set(Cl35, H1, scratch);

    scratch = -screened_rates(k_he4_cl35__k39)*Y(Cl35)*state.rho - screened_rates(k_he4_cl35__n_k38)*Y(Cl35)*state.rho - screened_rates(k_he4_cl35__p_ar38)*Y(Cl35)*state.rho + screened_rates(k_he4_p31__cl35)*Y(P31)*state.rho + screened_rates(k_he4_p32__n_cl35)*Y(P32)*state.rho + screened_rates(k_he4_s32__p_cl35)*Y(S32)*state.rho;
    jac.set(Cl35, He4, scratch);

    scratch = screened_rates(k_he4_p31__cl35)*Y(He4)*state.rho;
    jac.set(Cl35, P31, scratch);

    scratch = screened_rates(k_he4_p32__n_cl35)*Y(He4)*state.rho;
    jac.set(Cl35, P32, scratch);

    scratch = screened_rates(k_he4_s32__p_cl35)*Y(He4)*state.rho;
    jac.set(Cl35, S32, scratch);

    scratch = screened_rates(k_p_s34__cl35)*Y(H1)*state.rho;
    jac.set(Cl35, S34, scratch);

    scratch = screened_rates(k_p_s35__n_cl35)*Y(H1)*state.rho + screened_rates(k_s35__cl35__weak__wc12);
    jac.set(Cl35, S35, scratch);

    scratch = screened_rates(k_n_cl34__cl35)*Y(N)*state.rho;
    jac.set(Cl35, Cl34, scratch);

    scratch = -screened_rates(k_cl35__he4_p31) - screened_rates(k_cl35__n_cl34) - screened_rates(k_cl35__p_s34) - screened_rates(k_he4_cl35__k39)*Y(He4)*state.rho - screened_rates(k_he4_cl35__n_k38)*Y(He4)*state.rho - screened_rates(k_he4_cl35__p_ar38)*Y(He4)*state.rho - screened_rates(k_n_cl35__cl36)*Y(N)*state.rho - screened_rates(k_n_cl35__he4_p32)*Y(N)*state.rho - screened_rates(k_n_cl35__p_s35)*Y(N)*state.rho - screened_rates(k_p_cl35__ar36)*Y(H1)*state.rho - screened_rates(k_p_cl35__he4_s32)*Y(H1)*state.rho;
    jac.set(Cl35, Cl35, scratch);

    scratch = screened_rates(k_cl36__n_cl35);
    jac.set(Cl35, Cl36, scratch);

    scratch = screened_rates(k_ar36__p_cl35);
    jac.set(Cl35, Ar36, scratch);

    scratch = screened_rates(k_p_ar38__he4_cl35)*Y(H1)*state.rho;
    jac.set(Cl35, Ar38, scratch);

    scratch = screened_rates(k_n_k38__he4_cl35)*Y(N)*state.rho;
    jac.set(Cl35, K38, scratch);

    scratch = screened_rates(k_k39__he4_cl35);
    jac.set(Cl35, K39, scratch);

    scratch = screened_rates(k_n_ar36__p_cl36)*Y(Ar36)*state.rho + screened_rates(k_n_cl35__cl36)*Y(Cl35)*state.rho - screened_rates(k_n_cl36__cl37)*Y(Cl36)*state.rho - screened_rates(k_n_cl36__he4_p33)*Y(Cl36)*state.rho - screened_rates(k_n_cl36__p_s36)*Y(Cl36)*state.rho + screened_rates(k_n_k39__he4_cl36)*Y(K39)*state.rho;
    jac.set(Cl36, N, scratch);

    scratch = screened_rates(k_p_ar39__he4_cl36)*Y(Ar39)*state.rho - screened_rates(k_p_cl36__ar37)*Y(Cl36)*state.rho - screened_rates(k_p_cl36__he4_s33)*Y(Cl36)*state.rho - screened_rates(k_p_cl36__n_ar36)*Y(Cl36)*state.rho + screened_rates(k_p_s35__cl36)*Y(S35)*state.rho + screened_rates(k_p_s36__n_cl36)*Y(S36)*state.rho;
    jac.set(Cl36, H1, scratch);

    scratch = -screened_rates(k_he4_cl36__k40)*Y(Cl36)*state.rho - screened_rates(k_he4_cl36__n_k39)*Y(Cl36)*state.rho - screened_rates(k_he4_cl36__p_ar39)*Y(Cl36)*state.rho + screened_rates(k_he4_p32__cl36)*Y(P32)*state.rho + screened_rates(k_he4_p33__n_cl36)*Y(P33)*state.rho + screened_rates(k_he4_s33__p_cl36)*Y(S33)*state.rho;
    jac.set(Cl36, He4, scratch);

    scratch = screened_rates(k_he4_p32__cl36)*Y(He4)*state.rho;
    jac.set(Cl36, P32, scratch);

    scratch = screened_rates(k_he4_p33__n_cl36)*Y(He4)*state.rho;
    jac.set(Cl36, P33, scratch);

    scratch = screened_rates(k_he4_s33__p_cl36)*Y(He4)*state.rho;
    jac.set(Cl36, S33, scratch);

    scratch = screened_rates(k_p_s35__cl36)*Y(H1)*state.rho;
    jac.set(Cl36, S35, scratch);

    scratch = screened_rates(k_p_s36__n_cl36)*Y(H1)*state.rho;
    jac.set(Cl36, S36, scratch);

    scratch = screened_rates(k_n_cl35__cl36)*Y(N)*state.rho;
    jac.set(Cl36, Cl35, scratch);

    scratch = -screened_rates(k_cl36__ar36__weak__wc12) - screened_rates(k_cl36__he4_p32) - screened_rates(k_cl36__n_cl35) - screened_rates(k_cl36__p_s35) - screened_rates(k_cl36__s36__weak__wc12) - screened_rates(k_he4_cl36__k40)*Y(He4)*state.rho - screened_rates(k_he4_cl36__n_k39)*Y(He4)*state.rho - screened_rates(k_he4_cl36__p_ar39)*Y(He4)*state.rho - screened_rates(k_n_cl36__cl37)*Y(N)*state.rho - screened_rates(k_n_cl36__he4_p33)*Y(N)*state.rho - screened_rates(k_n_cl36__p_s36)*Y(N)*state.rho - screened_rates(k_p_cl36__ar37)*Y(H1)*state.rho - screened_rates(k_p_cl36__he4_s33)*Y(H1)*state.rho - screened_rates(k_p_cl36__n_ar36)*Y(H1)*state.rho;
    jac.set(Cl36, Cl36, scratch);

    scratch = screened_rates(k_cl37__n_cl36);
    jac.set(Cl36, Cl37, scratch);

    scratch = screened_rates(k_n_ar36__p_cl36)*Y(N)*state.rho;
    jac.set(Cl36, Ar36, scratch);

    scratch = screened_rates(k_ar37__p_cl36);
    jac.set(Cl36, Ar37, scratch);

    scratch = screened_rates(k_p_ar39__he4_cl36)*Y(H1)*state.rho;
    jac.set(Cl36, Ar39, scratch);

    scratch = screened_rates(k_n_k39__he4_cl36)*Y(N)*state.rho;
    jac.set(Cl36, K39, scratch);

    scratch = screened_rates(k_k40__he4_cl36);
    jac.set(Cl36, K40, scratch);

    scratch = screened_rates(k_n_ar37__p_cl37)*Y(Ar37)*state.rho + screened_rates(k_n_cl36__cl37)*Y(Cl36)*state.rho + screened_rates(k_n_k40__he4_cl37)*Y(K40)*state.rho;
    jac.set(Cl37, N, scratch);

    scratch = screened_rates(k_p_ar40__he4_cl37)*Y(Ar40)*state.rho - screened_rates(k_p_cl37__ar38)*Y(Cl37)*state.rho - screened_rates(k_p_cl37__he4_s34)*Y(Cl37)*state.rho - screened_rates(k_p_cl37__n_ar37)*Y(Cl37)*state.rho + screened_rates(k_p_s36__cl37)*Y(S36)*state.rho;
    jac.set(Cl37, H1, scratch);

    scratch = -screened_rates(k_he4_cl37__k41)*Y(Cl37)*state.rho - screened_rates(k_he4_cl37__n_k40)*Y(Cl37)*state.rho - screened_rates(k_he4_cl37__p_ar40)*Y(Cl37)*state.rho + screened_rates(k_he4_p33__cl37)*Y(P33)*state.rho + screened_rates(k_he4_s34__p_cl37)*Y(S34)*state.rho;
    jac.set(Cl37, He4, scratch);

    scratch = screened_rates(k_he4_p33__cl37)*Y(He4)*state.rho;
    jac.set(Cl37, P33, scratch);

    scratch = screened_rates(k_he4_s34__p_cl37)*Y(He4)*state.rho;
    jac.set(Cl37, S34, scratch);

    scratch = screened_rates(k_p_s36__cl37)*Y(H1)*state.rho;
    jac.set(Cl37, S36, scratch);

    scratch = screened_rates(k_n_cl36__cl37)*Y(N)*state.rho;
    jac.set(Cl37, Cl36, scratch);

    scratch = -screened_rates(k_cl37__he4_p33) - screened_rates(k_cl37__n_cl36) - screened_rates(k_cl37__p_s36) - screened_rates(k_he4_cl37__k41)*Y(He4)*state.rho - screened_rates(k_he4_cl37__n_k40)*Y(He4)*state.rho - screened_rates(k_he4_cl37__p_ar40)*Y(He4)*state.rho - screened_rates(k_p_cl37__ar38)*Y(H1)*state.rho - screened_rates(k_p_cl37__he4_s34)*Y(H1)*state.rho - screened_rates(k_p_cl37__n_ar37)*Y(H1)*state.rho;
    jac.set(Cl37, Cl37, scratch);

    scratch = screened_rates(k_ar37__cl37__weak__wc12) + screened_rates(k_n_ar37__p_cl37)*Y(N)*state.rho;
    jac.set(Cl37, Ar37, scratch);

    scratch = screened_rates(k_ar38__p_cl37);
    jac.set(Cl37, Ar38, scratch);

    scratch = screened_rates(k_p_ar40__he4_cl37)*Y(H1)*state.rho;
    jac.set(Cl37, Ar40, scratch);

    scratch = screened_rates(k_n_k40__he4_cl37)*Y(N)*state.rho;
    jac.set(Cl37, K40, scratch);

    scratch = screened_rates(k_k41__he4_cl37);
    jac.set(Cl37, K41, scratch);

    scratch = -screened_rates(k_n_ar36__ar37)*Y(Ar36)*state.rho - screened_rates(k_n_ar36__he4_s33)*Y(Ar36)*state.rho - screened_rates(k_n_ar36__p_cl36)*Y(Ar36)*state.rho;
    jac.set(Ar36, N, scratch);

    scratch = -screened_rates(k_p_ar36__he4_cl33)*Y(Ar36)*state.rho - screened_rates(k_p_ar36__k37)*Y(Ar36)*state.rho + screened_rates(k_p_cl35__ar36)*Y(Cl35)*state.rho + screened_rates(k_p_cl36__n_ar36)*Y(Cl36)*state.rho + screened_rates(k_p_k39__he4_ar36)*Y(K39)*state.rho;
    jac.set(Ar36, H1, scratch);

    scratch = -screened_rates(k_he4_ar36__ca40)*Y(Ar36)*state.rho - screened_rates(k_he4_ar36__p_k39)*Y(Ar36)*state.rho + screened_rates(k_he4_cl33__p_ar36)*Y(Cl33)*state.rho + screened_rates(k_he4_s32__ar36)*Y(S32)*state.rho + screened_rates(k_he4_s33__n_ar36)*Y(S33)*state.rho;
    jac.set(Ar36, He4, scratch);

    scratch = screened_rates(k_he4_s32__ar36)*Y(He4)*state.rho;
    jac.set(Ar36, S32, scratch);

    scratch = screened_rates(k_he4_s33__n_ar36)*Y(He4)*state.rho;
    jac.set(Ar36, S33, scratch);

    scratch = screened_rates(k_he4_cl33__p_ar36)*Y(He4)*state.rho;
    jac.set(Ar36, Cl33, scratch);

    scratch = screened_rates(k_p_cl35__ar36)*Y(H1)*state.rho;
    jac.set(Ar36, Cl35, scratch);

    scratch = screened_rates(k_cl36__ar36__weak__wc12) + screened_rates(k_p_cl36__n_ar36)*Y(H1)*state.rho;
    jac.set(Ar36, Cl36, scratch);

    scratch = -screened_rates(k_ar36__he4_s32) - screened_rates(k_ar36__p_cl35) - screened_rates(k_he4_ar36__ca40)*Y(He4)*state.rho - screened_rates(k_he4_ar36__p_k39)*Y(He4)*state.rho - screened_rates(k_n_ar36__ar37)*Y(N)*state.rho - screened_rates(k_n_ar36__he4_s33)*Y(N)*state.rho - screened_rates(k_n_ar36__p_cl36)*Y(N)*state.rho - screened_rates(k_p_ar36__he4_cl33)*Y(H1)*state.rho - screened_rates(k_p_ar36__k37)*Y(H1)*state.rho;
    jac.set(Ar36, Ar36, scratch);

    scratch = screened_rates(k_ar37__n_ar36);
    jac.set(Ar36, Ar37, scratch);

    scratch = screened_rates(k_k37__p_ar36);
    jac.set(Ar36, K37, scratch);

    scratch = screened_rates(k_p_k39__he4_ar36)*Y(H1)*state.rho;
    jac.set(Ar36, K39, scratch);

    scratch = screened_rates(k_ca40__he4_ar36);
    jac.set(Ar36, Ca40, scratch);

    scratch = screened_rates(k_n_ar36__ar37)*Y(Ar36)*state.rho - screened_rates(k_n_ar37__ar38)*Y(Ar37)*state.rho - screened_rates(k_n_ar37__he4_s34)*Y(Ar37)*state.rho - screened_rates(k_n_ar37__p_cl37)*Y(Ar37)*state.rho + screened_rates(k_n_ca40__he4_ar37)*Y(Ca40)*state.rho + screened_rates(k_n_k37__p_ar37)*Y(K37)*state.rho;
    jac.set(Ar37, N, scratch);

    scratch = -screened_rates(k_p_ar37__he4_cl34)*Y(Ar37)*state.rho - screened_rates(k_p_ar37__k38)*Y(Ar37)*state.rho - screened_rates(k_p_ar37__n_k37)*Y(Ar37)*state.rho + screened_rates(k_p_cl36__ar37)*Y(Cl36)*state.rho + screened_rates(k_p_cl37__n_ar37)*Y(Cl37)*state.rho + screened_rates(k_p_k40__he4_ar37)*Y(K40)*state.rho;
    jac.set(Ar37, H1, scratch);

    scratch = -screened_rates(k_he4_ar37__ca41)*Y(Ar37)*state.rho - screened_rates(k_he4_ar37__n_ca40)*Y(Ar37)*state.rho - screened_rates(k_he4_ar37__p_k40)*Y(Ar37)*state.rho + screened_rates(k_he4_cl34__p_ar37)*Y(Cl34)*state.rho + screened_rates(k_he4_s33__ar37)*Y(S33)*state.rho + screened_rates(k_he4_s34__n_ar37)*Y(S34)*state.rho;
    jac.set(Ar37, He4, scratch);

    scratch = screened_rates(k_he4_s33__ar37)*Y(He4)*state.rho;
    jac.set(Ar37, S33, scratch);

    scratch = screened_rates(k_he4_s34__n_ar37)*Y(He4)*state.rho;
    jac.set(Ar37, S34, scratch);

    scratch = screened_rates(k_he4_cl34__p_ar37)*Y(He4)*state.rho;
    jac.set(Ar37, Cl34, scratch);

    scratch = screened_rates(k_p_cl36__ar37)*Y(H1)*state.rho;
    jac.set(Ar37, Cl36, scratch);

    scratch = screened_rates(k_p_cl37__n_ar37)*Y(H1)*state.rho;
    jac.set(Ar37, Cl37, scratch);

    scratch = screened_rates(k_n_ar36__ar37)*Y(N)*state.rho;
    jac.set(Ar37, Ar36, scratch);

    scratch = -screened_rates(k_ar37__cl37__weak__wc12) - screened_rates(k_ar37__he4_s33) - screened_rates(k_ar37__n_ar36) - screened_rates(k_ar37__p_cl36) - screened_rates(k_he4_ar37__ca41)*Y(He4)*state.rho - screened_rates(k_he4_ar37__n_ca40)*Y(He4)*state.rho - screened_rates(k_he4_ar37__p_k40)*Y(He4)*state.rho - screened_rates(k_n_ar37__ar38)*Y(N)*state.rho - screened_rates(k_n_ar37__he4_s34)*Y(N)*state.rho - screened_rates(k_n_ar37__p_cl37)*Y(N)*state.rho - screened_rates(k_p_ar37__he4_cl34)*Y(H1)*state.rho - screened_rates(k_p_ar37__k38)*Y(H1)*state.rho - screened_rates(k_p_ar37__n_k37)*Y(H1)*state.rho;
    jac.set(Ar37, Ar37, scratch);

    scratch = screened_rates(k_ar38__n_ar37);
    jac.set(Ar37, Ar38, scratch);

    scratch = screened_rates(k_k37__ar37__weak__wc12) + screened_rates(k_n_k37__p_ar37)*Y(N)*state.rho;
    jac.set(Ar37, K37, scratch);

    scratch = screened_rates(k_k38__p_ar37);
    jac.set(Ar37, K38, scratch);

    scratch = screened_rates(k_p_k40__he4_ar37)*Y(H1)*state.rho;
    jac.set(Ar37, K40, scratch);

    scratch = screened_rates(k_n_ca40__he4_ar37)*Y(N)*state.rho;
    jac.set(Ar37, Ca40, scratch);

    scratch = screened_rates(k_ca41__he4_ar37);
    jac.set(Ar37, Ca41, scratch);

    scratch = screened_rates(k_n_ar37__ar38)*Y(Ar37)*state.rho - screened_rates(k_n_ar38__ar39)*Y(Ar38)*state.rho - screened_rates(k_n_ar38__he4_s35)*Y(Ar38)*state.rho + screened_rates(k_n_ca41__he4_ar38)*Y(Ca41)*state.rho + screened_rates(k_n_k38__p_ar38)*Y(K38)*state.rho;
    jac.set(Ar38, N, scratch);

    scratch = -screened_rates(k_p_ar38__he4_cl35)*Y(Ar38)*state.rho - screened_rates(k_p_ar38__k39)*Y(Ar38)*state.rho - screened_rates(k_p_ar38__n_k38)*Y(Ar38)*state.rho + screened_rates(k_p_cl37__ar38)*Y(Cl37)*state.rho + screened_rates(k_p_k41__he4_ar38)*Y(K41)*state.rho;
    jac.set(Ar38, H1, scratch);

    scratch = -screened_rates(k_he4_ar38__ca42)*Y(Ar38)*state.rho - screened_rates(k_he4_ar38__n_ca41)*Y(Ar38)*state.rho - screened_rates(k_he4_ar38__p_k41)*Y(Ar38)*state.rho + screened_rates(k_he4_cl35__p_ar38)*Y(Cl35)*state.rho + screened_rates(k_he4_s34__ar38)*Y(S34)*state.rho + screened_rates(k_he4_s35__n_ar38)*Y(S35)*state.rho;
    jac.set(Ar38, He4, scratch);

    scratch = screened_rates(k_he4_s34__ar38)*Y(He4)*state.rho;
    jac.set(Ar38, S34, scratch);

    scratch = screened_rates(k_he4_s35__n_ar38)*Y(He4)*state.rho;
    jac.set(Ar38, S35, scratch);

    scratch = screened_rates(k_he4_cl35__p_ar38)*Y(He4)*state.rho;
    jac.set(Ar38, Cl35, scratch);

    scratch = screened_rates(k_p_cl37__ar38)*Y(H1)*state.rho;
    jac.set(Ar38, Cl37, scratch);

    scratch = screened_rates(k_n_ar37__ar38)*Y(N)*state.rho;
    jac.set(Ar38, Ar37, scratch);

    scratch = -screened_rates(k_ar38__he4_s34) - screened_rates(k_ar38__n_ar37) - screened_rates(k_ar38__p_cl37) - screened_rates(k_he4_ar38__ca42)*Y(He4)*state.rho - screened_rates(k_he4_ar38__n_ca41)*Y(He4)*state.rho - screened_rates(k_he4_ar38__p_k41)*Y(He4)*state.rho - screened_rates(k_n_ar38__ar39)*Y(N)*state.rho - screened_rates(k_n_ar38__he4_s35)*Y(N)*state.rho - screened_rates(k_p_ar38__he4_cl35)*Y(H1)*state.rho - screened_rates(k_p_ar38__k39)*Y(H1)*state.rho - screened_rates(k_p_ar38__n_k38)*Y(H1)*state.rho;
    jac.set(Ar38, Ar38, scratch);

    scratch = screened_rates(k_ar39__n_ar38);
    jac.set(Ar38, Ar39, scratch);

    scratch = screened_rates(k_k38__ar38__weak__wc12) + screened_rates(k_n_k38__p_ar38)*Y(N)*state.rho;
    jac.set(Ar38, K38, scratch);

    scratch = screened_rates(k_k39__p_ar38);
    jac.set(Ar38, K39, scratch);

    scratch = screened_rates(k_p_k41__he4_ar38)*Y(H1)*state.rho;
    jac.set(Ar38, K41, scratch);

    scratch = screened_rates(k_n_ca41__he4_ar38)*Y(N)*state.rho;
    jac.set(Ar38, Ca41, scratch);

    scratch = screened_rates(k_ca42__he4_ar38);
    jac.set(Ar38, Ca42, scratch);

    scratch = screened_rates(k_n_ar38__ar39)*Y(Ar38)*state.rho - screened_rates(k_n_ar39__ar40)*Y(Ar39)*state.rho - screened_rates(k_n_ar39__he4_s36)*Y(Ar39)*state.rho + screened_rates(k_n_ca42__he4_ar39)*Y(Ca42)*state.rho + screened_rates(k_n_k39__p_ar39)*Y(K39)*state.rho;
    jac.set(Ar39, N, scratch);

    scratch = -screened_rates(k_p_ar39__he4_cl36)*Y(Ar39)*state.rho - screened_rates(k_p_ar39__k40)*Y(Ar39)*state.rho - screened_rates(k_p_ar39__n_k39)*Y(Ar39)*state.rho;
    jac.set(Ar39, H1, scratch);

    scratch = -screened_rates(k_he4_ar39__ca43)*Y(Ar39)*state.rho - screened_rates(k_he4_ar39__n_ca42)*Y(Ar39)*state.rho + screened_rates(k_he4_cl36__p_ar39)*Y(Cl36)*state.rho + screened_rates(k_he4_s35__ar39)*Y(S35)*state.rho + screened_rates(k_he4_s36__n_ar39)*Y(S36)*state.rho;
    jac.set(Ar39, He4, scratch);

    scratch = screened_rates(k_he4_s35__ar39)*Y(He4)*state.rho;
    jac.set(Ar39, S35, scratch);

    scratch = screened_rates(k_he4_s36__n_ar39)*Y(He4)*state.rho;
    jac.set(Ar39, S36, scratch);

    scratch = screened_rates(k_he4_cl36__p_ar39)*Y(He4)*state.rho;
    jac.set(Ar39, Cl36, scratch);

    scratch = screened_rates(k_n_ar38__ar39)*Y(N)*state.rho;
    jac.set(Ar39, Ar38, scratch);

    scratch = -screened_rates(k_ar39__he4_s35) - screened_rates(k_ar39__k39__weak__wc12) - screened_rates(k_ar39__n_ar38) - screened_rates(k_he4_ar39__ca43)*Y(He4)*state.rho - screened_rates(k_he4_ar39__n_ca42)*Y(He4)*state.rho - screened_rates(k_n_ar39__ar40)*Y(N)*state.rho - screened_rates(k_n_ar39__he4_s36)*Y(N)*state.rho - screened_rates(k_p_ar39__he4_cl36)*Y(H1)*state.rho - screened_rates(k_p_ar39__k40)*Y(H1)*state.rho - screened_rates(k_p_ar39__n_k39)*Y(H1)*state.rho;
    jac.set(Ar39, Ar39, scratch);

    scratch = screened_rates(k_ar40__n_ar39);
    jac.set(Ar39, Ar40, scratch);

    scratch = screened_rates(k_n_k39__p_ar39)*Y(N)*state.rho;
    jac.set(Ar39, K39, scratch);

    scratch = screened_rates(k_k40__p_ar39);
    jac.set(Ar39, K40, scratch);

    scratch = screened_rates(k_n_ca42__he4_ar39)*Y(N)*state.rho;
    jac.set(Ar39, Ca42, scratch);

    scratch = screened_rates(k_ca43__he4_ar39);
    jac.set(Ar39, Ca43, scratch);

    scratch = screened_rates(k_n_ar39__ar40)*Y(Ar39)*state.rho + screened_rates(k_n_ca43__he4_ar40)*Y(Ca43)*state.rho + screened_rates(k_n_k40__p_ar40)*Y(K40)*state.rho;
    jac.set(Ar40, N, scratch);

    scratch = -screened_rates(k_p_ar40__he4_cl37)*Y(Ar40)*state.rho - screened_rates(k_p_ar40__k41)*Y(Ar40)*state.rho - screened_rates(k_p_ar40__n_k40)*Y(Ar40)*state.rho;
    jac.set(Ar40, H1, scratch);

    scratch = -screened_rates(k_he4_ar40__ca44)*Y(Ar40)*state.rho - screened_rates(k_he4_ar40__n_ca43)*Y(Ar40)*state.rho + screened_rates(k_he4_cl37__p_ar40)*Y(Cl37)*state.rho + screened_rates(k_he4_s36__ar40)*Y(S36)*state.rho;
    jac.set(Ar40, He4, scratch);

    scratch = screened_rates(k_he4_s36__ar40)*Y(He4)*state.rho;
    jac.set(Ar40, S36, scratch);

    scratch = screened_rates(k_he4_cl37__p_ar40)*Y(He4)*state.rho;
    jac.set(Ar40, Cl37, scratch);

    scratch = screened_rates(k_n_ar39__ar40)*Y(N)*state.rho;
    jac.set(Ar40, Ar39, scratch);

    scratch = -screened_rates(k_ar40__he4_s36) - screened_rates(k_ar40__n_ar39) - screened_rates(k_he4_ar40__ca44)*Y(He4)*state.rho - screened_rates(k_he4_ar40__n_ca43)*Y(He4)*state.rho - screened_rates(k_p_ar40__he4_cl37)*Y(H1)*state.rho - screened_rates(k_p_ar40__k41)*Y(H1)*state.rho - screened_rates(k_p_ar40__n_k40)*Y(H1)*state.rho;
    jac.set(Ar40, Ar40, scratch);

    scratch = screened_rates(k_k40__ar40__weak__wc12) + screened_rates(k_n_k40__p_ar40)*Y(N)*state.rho;
    jac.set(Ar40, K40, scratch);

    scratch = screened_rates(k_k41__p_ar40);
    jac.set(Ar40, K41, scratch);

    scratch = screened_rates(k_n_ca43__he4_ar40)*Y(N)*state.rho;
    jac.set(Ar40, Ca43, scratch);

    scratch = screened_rates(k_ca44__he4_ar40);
    jac.set(Ar40, Ca44, scratch);

    scratch = -screened_rates(k_n_k37__he4_cl34)*Y(K37)*state.rho - screened_rates(k_n_k37__k38)*Y(K37)*state.rho - screened_rates(k_n_k37__p_ar37)*Y(K37)*state.rho;
    jac.set(K37, N, scratch);

    scratch = screened_rates(k_p_ar36__k37)*Y(Ar36)*state.rho + screened_rates(k_p_ar37__n_k37)*Y(Ar37)*state.rho + screened_rates(k_p_ca40__he4_k37)*Y(Ca40)*state.rho;
    jac.set(K37, H1, scratch);

    scratch = screened_rates(k_he4_cl33__k37)*Y(Cl33)*state.rho + screened_rates(k_he4_cl34__n_k37)*Y(Cl34)*state.rho - screened_rates(k_he4_k37__p_ca40)*Y(K37)*state.rho;
    jac.set(K37, He4, scratch);

    scratch = screened_rates(k_he4_cl33__k37)*Y(He4)*state.rho;
    jac.set(K37, Cl33, scratch);

    scratch = screened_rates(k_he4_cl34__n_k37)*Y(He4)*state.rho;
    jac.set(K37, Cl34, scratch);

    scratch = screened_rates(k_p_ar36__k37)*Y(H1)*state.rho;
    jac.set(K37, Ar36, scratch);

    scratch = screened_rates(k_p_ar37__n_k37)*Y(H1)*state.rho;
    jac.set(K37, Ar37, scratch);

    scratch = -screened_rates(k_he4_k37__p_ca40)*Y(He4)*state.rho - screened_rates(k_k37__ar37__weak__wc12) - screened_rates(k_k37__he4_cl33) - screened_rates(k_k37__p_ar36) - screened_rates(k_n_k37__he4_cl34)*Y(N)*state.rho - screened_rates(k_n_k37__k38)*Y(N)*state.rho - screened_rates(k_n_k37__p_ar37)*Y(N)*state.rho;
    jac.set(K37, K37, scratch);

    scratch = screened_rates(k_k38__n_k37);
    jac.set(K37, K38, scratch);

    scratch = screened_rates(k_p_ca40__he4_k37)*Y(H1)*state.rho;
    jac.set(K37, Ca40, scratch);

    scratch = screened_rates(k_n_k37__k38)*Y(K37)*state.rho - screened_rates(k_n_k38__he4_cl35)*Y(K38)*state.rho - screened_rates(k_n_k38__k39)*Y(K38)*state.rho - screened_rates(k_n_k38__p_ar38)*Y(K38)*state.rho;
    jac.set(K38, N, scratch);

    scratch = screened_rates(k_p_ar37__k38)*Y(Ar37)*state.rho + screened_rates(k_p_ar38__n_k38)*Y(Ar38)*state.rho + screened_rates(k_p_ca41__he4_k38)*Y(Ca41)*state.rho;
    jac.set(K38, H1, scratch);

    scratch = screened_rates(k_he4_cl34__k38)*Y(Cl34)*state.rho + screened_rates(k_he4_cl35__n_k38)*Y(Cl35)*state.rho - screened_rates(k_he4_k38__p_ca41)*Y(K38)*state.rho;
    jac.set(K38, He4, scratch);

    scratch = screened_rates(k_he4_cl34__k38)*Y(He4)*state.rho;
    jac.set(K38, Cl34, scratch);

    scratch = screened_rates(k_he4_cl35__n_k38)*Y(He4)*state.rho;
    jac.set(K38, Cl35, scratch);

    scratch = screened_rates(k_p_ar37__k38)*Y(H1)*state.rho;
    jac.set(K38, Ar37, scratch);

    scratch = screened_rates(k_p_ar38__n_k38)*Y(H1)*state.rho;
    jac.set(K38, Ar38, scratch);

    scratch = screened_rates(k_n_k37__k38)*Y(N)*state.rho;
    jac.set(K38, K37, scratch);

    scratch = -screened_rates(k_he4_k38__p_ca41)*Y(He4)*state.rho - screened_rates(k_k38__ar38__weak__wc12) - screened_rates(k_k38__he4_cl34) - screened_rates(k_k38__n_k37) - screened_rates(k_k38__p_ar37) - screened_rates(k_n_k38__he4_cl35)*Y(N)*state.rho - screened_rates(k_n_k38__k39)*Y(N)*state.rho - screened_rates(k_n_k38__p_ar38)*Y(N)*state.rho;
    jac.set(K38, K38, scratch);

    scratch = screened_rates(k_k39__n_k38);
    jac.set(K38, K39, scratch);

    scratch = screened_rates(k_p_ca41__he4_k38)*Y(H1)*state.rho;
    jac.set(K38, Ca41, scratch);

    scratch = screened_rates(k_n_k38__k39)*Y(K38)*state.rho - screened_rates(k_n_k39__he4_cl36)*Y(K39)*state.rho - screened_rates(k_n_k39__k40)*Y(K39)*state.rho - screened_rates(k_n_k39__p_ar39)*Y(K39)*state.rho;
    jac.set(K39, N, scratch);

    scratch = screened_rates(k_p_ar38__k39)*Y(Ar38)*state.rho + screened_rates(k_p_ar39__n_k39)*Y(Ar39)*state.rho + screened_rates(k_p_ca42__he4_k39)*Y(Ca42)*state.rho - screened_rates(k_p_k39__ca40)*Y(K39)*state.rho - screened_rates(k_p_k39__he4_ar36)*Y(K39)*state.rho;
    jac.set(K39, H1, scratch);

    scratch = screened_rates(k_he4_ar36__p_k39)*Y(Ar36)*state.rho + screened_rates(k_he4_cl35__k39)*Y(Cl35)*state.rho + screened_rates(k_he4_cl36__n_k39)*Y(Cl36)*state.rho - screened_rates(k_he4_k39__p_ca42)*Y(K39)*state.rho - screened_rates(k_he4_k39__sc43)*Y(K39)*state.rho;
    jac.set(K39, He4, scratch);

    scratch = screened_rates(k_he4_cl35__k39)*Y(He4)*state.rho;
    jac.set(K39, Cl35, scratch);

    scratch = screened_rates(k_he4_cl36__n_k39)*Y(He4)*state.rho;
    jac.set(K39, Cl36, scratch);

    scratch = screened_rates(k_he4_ar36__p_k39)*Y(He4)*state.rho;
    jac.set(K39, Ar36, scratch);

    scratch = screened_rates(k_p_ar38__k39)*Y(H1)*state.rho;
    jac.set(K39, Ar38, scratch);

    scratch = screened_rates(k_ar39__k39__weak__wc12) + screened_rates(k_p_ar39__n_k39)*Y(H1)*state.rho;
    jac.set(K39, Ar39, scratch);

    scratch = screened_rates(k_n_k38__k39)*Y(N)*state.rho;
    jac.set(K39, K38, scratch);

    scratch = -screened_rates(k_he4_k39__p_ca42)*Y(He4)*state.rho - screened_rates(k_he4_k39__sc43)*Y(He4)*state.rho - screened_rates(k_k39__he4_cl35) - screened_rates(k_k39__n_k38) - screened_rates(k_k39__p_ar38) - screened_rates(k_n_k39__he4_cl36)*Y(N)*state.rho - screened_rates(k_n_k39__k40)*Y(N)*state.rho - screened_rates(k_n_k39__p_ar39)*Y(N)*state.rho - screened_rates(k_p_k39__ca40)*Y(H1)*state.rho - screened_rates(k_p_k39__he4_ar36)*Y(H1)*state.rho;
    jac.set(K39, K39, scratch);

    scratch = screened_rates(k_k40__n_k39);
    jac.set(K39, K40, scratch);

    scratch = screened_rates(k_ca40__p_k39);
    jac.set(K39, Ca40, scratch);

    scratch = screened_rates(k_p_ca42__he4_k39)*Y(H1)*state.rho;
    jac.set(K39, Ca42, scratch);

    scratch = screened_rates(k_sc43__he4_k39);
    jac.set(K39, Sc43, scratch);

    scratch = screened_rates(k_n_ca40__p_k40)*Y(Ca40)*state.rho + screened_rates(k_n_k39__k40)*Y(K39)*state.rho - screened_rates(k_n_k40__he4_cl37)*Y(K40)*state.rho - screened_rates(k_n_k40__k41)*Y(K40)*state.rho - screened_rates(k_n_k40__p_ar40)*Y(K40)*state.rho + screened_rates(k_n_sc43__he4_k40)*Y(Sc43)*state.rho;
    jac.set(K40, N, scratch);

    scratch = screened_rates(k_p_ar39__k40)*Y(Ar39)*state.rho + screened_rates(k_p_ar40__n_k40)*Y(Ar40)*state.rho + screened_rates(k_p_ca43__he4_k40)*Y(Ca43)*state.rho - screened_rates(k_p_k40__ca41)*Y(K40)*state.rho - screened_rates(k_p_k40__he4_ar37)*Y(K40)*state.rho - screened_rates(k_p_k40__n_ca40)*Y(K40)*state.rho;
    jac.set(K40, H1, scratch);

    scratch = screened_rates(k_he4_ar37__p_k40)*Y(Ar37)*state.rho + screened_rates(k_he4_cl36__k40)*Y(Cl36)*state.rho + screened_rates(k_he4_cl37__n_k40)*Y(Cl37)*state.rho - screened_rates(k_he4_k40__n_sc43)*Y(K40)*state.rho - screened_rates(k_he4_k40__p_ca43)*Y(K40)*state.rho - screened_rates(k_he4_k40__sc44)*Y(K40)*state.rho;
    jac.set(K40, He4, scratch);

    scratch = screened_rates(k_he4_cl36__k40)*Y(He4)*state.rho;
    jac.set(K40, Cl36, scratch);

    scratch = screened_rates(k_he4_cl37__n_k40)*Y(He4)*state.rho;
    jac.set(K40, Cl37, scratch);

    scratch = screened_rates(k_he4_ar37__p_k40)*Y(He4)*state.rho;
    jac.set(K40, Ar37, scratch);

    scratch = screened_rates(k_p_ar39__k40)*Y(H1)*state.rho;
    jac.set(K40, Ar39, scratch);

    scratch = screened_rates(k_p_ar40__n_k40)*Y(H1)*state.rho;
    jac.set(K40, Ar40, scratch);

    scratch = screened_rates(k_n_k39__k40)*Y(N)*state.rho;
    jac.set(K40, K39, scratch);

    scratch = -screened_rates(k_he4_k40__n_sc43)*Y(He4)*state.rho - screened_rates(k_he4_k40__p_ca43)*Y(He4)*state.rho - screened_rates(k_he4_k40__sc44)*Y(He4)*state.rho - screened_rates(k_k40__ar40__weak__wc12) - screened_rates(k_k40__ca40__weak__wc12) - screened_rates(k_k40__he4_cl36) - screened_rates(k_k40__n_k39) - screened_rates(k_k40__p_ar39) - screened_rates(k_n_k40__he4_cl37)*Y(N)*state.rho - screened_rates(k_n_k40__k41)*Y(N)*state.rho - screened_rates(k_n_k40__p_ar40)*Y(N)*state.rho - screened_rates(k_p_k40__ca41)*Y(H1)*state.rho - screened_rates(k_p_k40__he4_ar37)*Y(H1)*state.rho - screened_rates(k_p_k40__n_ca40)*Y(H1)*state.rho;
    jac.set(K40, K40, scratch);

    scratch = screened_rates(k_k41__n_k40);
    jac.set(K40, K41, scratch);

    scratch = screened_rates(k_n_ca40__p_k40)*Y(N)*state.rho;
    jac.set(K40, Ca40, scratch);

    scratch = screened_rates(k_ca41__p_k40);
    jac.set(K40, Ca41, scratch);

    scratch = screened_rates(k_p_ca43__he4_k40)*Y(H1)*state.rho;
    jac.set(K40, Ca43, scratch);

    scratch = screened_rates(k_n_sc43__he4_k40)*Y(N)*state.rho;
    jac.set(K40, Sc43, scratch);

    scratch = screened_rates(k_sc44__he4_k40);
    jac.set(K40, Sc44, scratch);

    scratch = screened_rates(k_n_ca41__p_k41)*Y(Ca41)*state.rho + screened_rates(k_n_k40__k41)*Y(K40)*state.rho + screened_rates(k_n_sc44__he4_k41)*Y(Sc44)*state.rho;
    jac.set(K41, N, scratch);

    scratch = screened_rates(k_p_ar40__k41)*Y(Ar40)*state.rho + screened_rates(k_p_ca44__he4_k41)*Y(Ca44)*state.rho - screened_rates(k_p_k41__ca42)*Y(K41)*state.rho - screened_rates(k_p_k41__he4_ar38)*Y(K41)*state.rho - screened_rates(k_p_k41__n_ca41)*Y(K41)*state.rho;
    jac.set(K41, H1, scratch);

    scratch = screened_rates(k_he4_ar38__p_k41)*Y(Ar38)*state.rho + screened_rates(k_he4_cl37__k41)*Y(Cl37)*state.rho - screened_rates(k_he4_k41__n_sc44)*Y(K41)*state.rho - screened_rates(k_he4_k41__p_ca44)*Y(K41)*state.rho - screened_rates(k_he4_k41__sc45)*Y(K41)*state.rho;
    jac.set(K41, He4, scratch);

    scratch = screened_rates(k_he4_cl37__k41)*Y(He4)*state.rho;
    jac.set(K41, Cl37, scratch);

    scratch = screened_rates(k_he4_ar38__p_k41)*Y(He4)*state.rho;
    jac.set(K41, Ar38, scratch);

    scratch = screened_rates(k_p_ar40__k41)*Y(H1)*state.rho;
    jac.set(K41, Ar40, scratch);

    scratch = screened_rates(k_n_k40__k41)*Y(N)*state.rho;
    jac.set(K41, K40, scratch);

    scratch = -screened_rates(k_he4_k41__n_sc44)*Y(He4)*state.rho - screened_rates(k_he4_k41__p_ca44)*Y(He4)*state.rho - screened_rates(k_he4_k41__sc45)*Y(He4)*state.rho - screened_rates(k_k41__he4_cl37) - screened_rates(k_k41__n_k40) - screened_rates(k_k41__p_ar40) - screened_rates(k_p_k41__ca42)*Y(H1)*state.rho - screened_rates(k_p_k41__he4_ar38)*Y(H1)*state.rho - screened_rates(k_p_k41__n_ca41)*Y(H1)*state.rho;
    jac.set(K41, K41, scratch);

    scratch = screened_rates(k_ca41__k41__weak__wc12) + screened_rates(k_n_ca41__p_k41)*Y(N)*state.rho;
    jac.set(K41, Ca41, scratch);

    scratch = screened_rates(k_ca42__p_k41);
    jac.set(K41, Ca42, scratch);

    scratch = screened_rates(k_p_ca44__he4_k41)*Y(H1)*state.rho;
    jac.set(K41, Ca44, scratch);

    scratch = screened_rates(k_n_sc44__he4_k41)*Y(N)*state.rho;
    jac.set(K41, Sc44, scratch);

    scratch = screened_rates(k_sc45__he4_k41);
    jac.set(K41, Sc45, scratch);

    scratch = -screened_rates(k_n_ca40__ca41)*Y(Ca40)*state.rho - screened_rates(k_n_ca40__he4_ar37)*Y(Ca40)*state.rho - screened_rates(k_n_ca40__p_k40)*Y(Ca40)*state.rho;
    jac.set(Ca40, N, scratch);

    scratch = -screened_rates(k_p_ca40__he4_k37)*Y(Ca40)*state.rho + screened_rates(k_p_k39__ca40)*Y(K39)*state.rho + screened_rates(k_p_k40__n_ca40)*Y(K40)*state.rho + screened_rates(k_p_sc43__he4_ca40)*Y(Sc43)*state.rho;
    jac.set(Ca40, H1, scratch);

    scratch = screened_rates(k_he4_ar36__ca40)*Y(Ar36)*state.rho + screened_rates(k_he4_ar37__n_ca40)*Y(Ar37)*state.rho - screened_rates(k_he4_ca40__p_sc43)*Y(Ca40)*state.rho - screened_rates(k_he4_ca40__ti44)*Y(Ca40)*state.rho + screened_rates(k_he4_k37__p_ca40)*Y(K37)*state.rho;
    jac.set(Ca40, He4, scratch);

    scratch = screened_rates(k_he4_ar36__ca40)*Y(He4)*state.rho;
    jac.set(Ca40, Ar36, scratch);

    scratch = screened_rates(k_he4_ar37__n_ca40)*Y(He4)*state.rho;
    jac.set(Ca40, Ar37, scratch);

    scratch = screened_rates(k_he4_k37__p_ca40)*Y(He4)*state.rho;
    jac.set(Ca40, K37, scratch);

    scratch = screened_rates(k_p_k39__ca40)*Y(H1)*state.rho;
    jac.set(Ca40, K39, scratch);

    scratch = screened_rates(k_k40__ca40__weak__wc12) + screened_rates(k_p_k40__n_ca40)*Y(H1)*state.rho;
    jac.set(Ca40, K40, scratch);

    scratch = -screened_rates(k_ca40__he4_ar36) - screened_rates(k_ca40__p_k39) - screened_rates(k_he4_ca40__p_sc43)*Y(He4)*state.rho - screened_rates(k_he4_ca40__ti44)*Y(He4)*state.rho - screened_rates(k_n_ca40__ca41)*Y(N)*state.rho - screened_rates(k_n_ca40__he4_ar37)*Y(N)*state.rho - screened_rates(k_n_ca40__p_k40)*Y(N)*state.rho - screened_rates(k_p_ca40__he4_k37)*Y(H1)*state.rho;
    jac.set(Ca40, Ca40, scratch);

    scratch = screened_rates(k_ca41__n_ca40);
    jac.set(Ca40, Ca41, scratch);

    scratch = screened_rates(k_p_sc43__he4_ca40)*Y(H1)*state.rho;
    jac.set(Ca40, Sc43, scratch);

    scratch = screened_rates(k_ti44__he4_ca40);
    jac.set(Ca40, Ti44, scratch);

    scratch = screened_rates(k_n_ca40__ca41)*Y(Ca40)*state.rho - screened_rates(k_n_ca41__ca42)*Y(Ca41)*state.rho - screened_rates(k_n_ca41__he4_ar38)*Y(Ca41)*state.rho - screened_rates(k_n_ca41__p_k41)*Y(Ca41)*state.rho + screened_rates(k_n_ti44__he4_ca41)*Y(Ti44)*state.rho;
    jac.set(Ca41, N, scratch);

    scratch = -screened_rates(k_p_ca41__he4_k38)*Y(Ca41)*state.rho + screened_rates(k_p_k40__ca41)*Y(K40)*state.rho + screened_rates(k_p_k41__n_ca41)*Y(K41)*state.rho + screened_rates(k_p_sc44__he4_ca41)*Y(Sc44)*state.rho;
    jac.set(Ca41, H1, scratch);

    scratch = screened_rates(k_he4_ar37__ca41)*Y(Ar37)*state.rho + screened_rates(k_he4_ar38__n_ca41)*Y(Ar38)*state.rho - screened_rates(k_he4_ca41__n_ti44)*Y(Ca41)*state.rho - screened_rates(k_he4_ca41__p_sc44)*Y(Ca41)*state.rho - screened_rates(k_he4_ca41__ti45)*Y(Ca41)*state.rho + screened_rates(k_he4_k38__p_ca41)*Y(K38)*state.rho;
    jac.set(Ca41, He4, scratch);

    scratch = screened_rates(k_he4_ar37__ca41)*Y(He4)*state.rho;
    jac.set(Ca41, Ar37, scratch);

    scratch = screened_rates(k_he4_ar38__n_ca41)*Y(He4)*state.rho;
    jac.set(Ca41, Ar38, scratch);

    scratch = screened_rates(k_he4_k38__p_ca41)*Y(He4)*state.rho;
    jac.set(Ca41, K38, scratch);

    scratch = screened_rates(k_p_k40__ca41)*Y(H1)*state.rho;
    jac.set(Ca41, K40, scratch);

    scratch = screened_rates(k_p_k41__n_ca41)*Y(H1)*state.rho;
    jac.set(Ca41, K41, scratch);

    scratch = screened_rates(k_n_ca40__ca41)*Y(N)*state.rho;
    jac.set(Ca41, Ca40, scratch);

    scratch = -screened_rates(k_ca41__he4_ar37) - screened_rates(k_ca41__k41__weak__wc12) - screened_rates(k_ca41__n_ca40) - screened_rates(k_ca41__p_k40) - screened_rates(k_he4_ca41__n_ti44)*Y(He4)*state.rho - screened_rates(k_he4_ca41__p_sc44)*Y(He4)*state.rho - screened_rates(k_he4_ca41__ti45)*Y(He4)*state.rho - screened_rates(k_n_ca41__ca42)*Y(N)*state.rho - screened_rates(k_n_ca41__he4_ar38)*Y(N)*state.rho - screened_rates(k_n_ca41__p_k41)*Y(N)*state.rho - screened_rates(k_p_ca41__he4_k38)*Y(H1)*state.rho;
    jac.set(Ca41, Ca41, scratch);

    scratch = screened_rates(k_ca42__n_ca41);
    jac.set(Ca41, Ca42, scratch);

    scratch = screened_rates(k_p_sc44__he4_ca41)*Y(H1)*state.rho;
    jac.set(Ca41, Sc44, scratch);

    scratch = screened_rates(k_n_ti44__he4_ca41)*Y(N)*state.rho;
    jac.set(Ca41, Ti44, scratch);

    scratch = screened_rates(k_ti45__he4_ca41);
    jac.set(Ca41, Ti45, scratch);

    scratch = screened_rates(k_n_ca41__ca42)*Y(Ca41)*state.rho - screened_rates(k_n_ca42__ca43)*Y(Ca42)*state.rho - screened_rates(k_n_ca42__he4_ar39)*Y(Ca42)*state.rho + screened_rates(k_n_ti45__he4_ca42)*Y(Ti45)*state.rho;
    jac.set(Ca42, N, scratch);

    scratch = -screened_rates(k_p_ca42__he4_k39)*Y(Ca42)*state.rho - screened_rates(k_p_ca42__sc43)*Y(Ca42)*state.rho + screened_rates(k_p_k41__ca42)*Y(K41)*state.rho + screened_rates(k_p_sc45__he4_ca42)*Y(Sc45)*state.rho;
    jac.set(Ca42, H1, scratch);

    scratch = screened_rates(k_he4_ar38__ca42)*Y(Ar38)*state.rho + screened_rates(k_he4_ar39__n_ca42)*Y(Ar39)*state.rho - screened_rates(k_he4_ca42__n_ti45)*Y(Ca42)*state.rho - screened_rates(k_he4_ca42__p_sc45)*Y(Ca42)*state.rho - screened_rates(k_he4_ca42__ti46)*Y(Ca42)*state.rho + screened_rates(k_he4_k39__p_ca42)*Y(K39)*state.rho;
    jac.set(Ca42, He4, scratch);

    scratch = screened_rates(k_he4_ar38__ca42)*Y(He4)*state.rho;
    jac.set(Ca42, Ar38, scratch);

    scratch = screened_rates(k_he4_ar39__n_ca42)*Y(He4)*state.rho;
    jac.set(Ca42, Ar39, scratch);

    scratch = screened_rates(k_he4_k39__p_ca42)*Y(He4)*state.rho;
    jac.set(Ca42, K39, scratch);

    scratch = screened_rates(k_p_k41__ca42)*Y(H1)*state.rho;
    jac.set(Ca42, K41, scratch);

    scratch = screened_rates(k_n_ca41__ca42)*Y(N)*state.rho;
    jac.set(Ca42, Ca41, scratch);

    scratch = -screened_rates(k_ca42__he4_ar38) - screened_rates(k_ca42__n_ca41) - screened_rates(k_ca42__p_k41) - screened_rates(k_he4_ca42__n_ti45)*Y(He4)*state.rho - screened_rates(k_he4_ca42__p_sc45)*Y(He4)*state.rho - screened_rates(k_he4_ca42__ti46)*Y(He4)*state.rho - screened_rates(k_n_ca42__ca43)*Y(N)*state.rho - screened_rates(k_n_ca42__he4_ar39)*Y(N)*state.rho - screened_rates(k_p_ca42__he4_k39)*Y(H1)*state.rho - screened_rates(k_p_ca42__sc43)*Y(H1)*state.rho;
    jac.set(Ca42, Ca42, scratch);

    scratch = screened_rates(k_ca43__n_ca42);
    jac.set(Ca42, Ca43, scratch);

    scratch = screened_rates(k_sc43__p_ca42);
    jac.set(Ca42, Sc43, scratch);

    scratch = screened_rates(k_p_sc45__he4_ca42)*Y(H1)*state.rho;
    jac.set(Ca42, Sc45, scratch);

    scratch = screened_rates(k_n_ti45__he4_ca42)*Y(N)*state.rho;
    jac.set(Ca42, Ti45, scratch);

    scratch = screened_rates(k_ti46__he4_ca42);
    jac.set(Ca42, Ti46, scratch);

    scratch = screened_rates(k_n_ca42__ca43)*Y(Ca42)*state.rho - screened_rates(k_n_ca43__ca44)*Y(Ca43)*state.rho - screened_rates(k_n_ca43__he4_ar40)*Y(Ca43)*state.rho + screened_rates(k_n_sc43__p_ca43)*Y(Sc43)*state.rho + screened_rates(k_n_ti46__he4_ca43)*Y(Ti46)*state.rho;
    jac.set(Ca43, N, scratch);

    scratch = -screened_rates(k_p_ca43__he4_k40)*Y(Ca43)*state.rho - screened_rates(k_p_ca43__n_sc43)*Y(Ca43)*state.rho - screened_rates(k_p_ca43__sc44)*Y(Ca43)*state.rho + screened_rates(k_p_sc46__he4_ca43)*Y(Sc46)*state.rho;
    jac.set(Ca43, H1, scratch);

    scratch = screened_rates(k_he4_ar39__ca43)*Y(Ar39)*state.rho + screened_rates(k_he4_ar40__n_ca43)*Y(Ar40)*state.rho - screened_rates(k_he4_ca43__n_ti46)*Y(Ca43)*state.rho - screened_rates(k_he4_ca43__p_sc46)*Y(Ca43)*state.rho - screened_rates(k_he4_ca43__ti47)*Y(Ca43)*state.rho + screened_rates(k_he4_k40__p_ca43)*Y(K40)*state.rho;
    jac.set(Ca43, He4, scratch);

    scratch = screened_rates(k_he4_ar39__ca43)*Y(He4)*state.rho;
    jac.set(Ca43, Ar39, scratch);

    scratch = screened_rates(k_he4_ar40__n_ca43)*Y(He4)*state.rho;
    jac.set(Ca43, Ar40, scratch);

    scratch = screened_rates(k_he4_k40__p_ca43)*Y(He4)*state.rho;
    jac.set(Ca43, K40, scratch);

    scratch = screened_rates(k_n_ca42__ca43)*Y(N)*state.rho;
    jac.set(Ca43, Ca42, scratch);

    scratch = -screened_rates(k_ca43__he4_ar39) - screened_rates(k_ca43__n_ca42) - screened_rates(k_he4_ca43__n_ti46)*Y(He4)*state.rho - screened_rates(k_he4_ca43__p_sc46)*Y(He4)*state.rho - screened_rates(k_he4_ca43__ti47)*Y(He4)*state.rho - screened_rates(k_n_ca43__ca44)*Y(N)*state.rho - screened_rates(k_n_ca43__he4_ar40)*Y(N)*state.rho - screened_rates(k_p_ca43__he4_k40)*Y(H1)*state.rho - screened_rates(k_p_ca43__n_sc43)*Y(H1)*state.rho - screened_rates(k_p_ca43__sc44)*Y(H1)*state.rho;
    jac.set(Ca43, Ca43, scratch);

    scratch = screened_rates(k_ca44__n_ca43);
    jac.set(Ca43, Ca44, scratch);

    scratch = screened_rates(k_n_sc43__p_ca43)*Y(N)*state.rho + screened_rates(k_sc43__ca43__weak__wc12);
    jac.set(Ca43, Sc43, scratch);

    scratch = screened_rates(k_sc44__p_ca43);
    jac.set(Ca43, Sc44, scratch);

    scratch = screened_rates(k_p_sc46__he4_ca43)*Y(H1)*state.rho;
    jac.set(Ca43, Sc46, scratch);

    scratch = screened_rates(k_n_ti46__he4_ca43)*Y(N)*state.rho;
    jac.set(Ca43, Ti46, scratch);

    scratch = screened_rates(k_ti47__he4_ca43);
    jac.set(Ca43, Ti47, scratch);

    scratch = screened_rates(k_n_ca43__ca44)*Y(Ca43)*state.rho - screened_rates(k_n_ca44__ca45)*Y(Ca44)*state.rho + screened_rates(k_n_sc44__p_ca44)*Y(Sc44)*state.rho + screened_rates(k_n_ti47__he4_ca44)*Y(Ti47)*state.rho;
    jac.set(Ca44, N, scratch);

    scratch = -screened_rates(k_p_ca44__he4_k41)*Y(Ca44)*state.rho - screened_rates(k_p_ca44__n_sc44)*Y(Ca44)*state.rho - screened_rates(k_p_ca44__sc45)*Y(Ca44)*state.rho + screened_rates(k_p_sc47__he4_ca44)*Y(Sc47)*state.rho;
    jac.set(Ca44, H1, scratch);

    scratch = screened_rates(k_he4_ar40__ca44)*Y(Ar40)*state.rho - screened_rates(k_he4_ca44__n_ti47)*Y(Ca44)*state.rho - screened_rates(k_he4_ca44__p_sc47)*Y(Ca44)*state.rho - screened_rates(k_he4_ca44__ti48)*Y(Ca44)*state.rho + screened_rates(k_he4_k41__p_ca44)*Y(K41)*state.rho;
    jac.set(Ca44, He4, scratch);

    scratch = screened_rates(k_he4_ar40__ca44)*Y(He4)*state.rho;
    jac.set(Ca44, Ar40, scratch);

    scratch = screened_rates(k_he4_k41__p_ca44)*Y(He4)*state.rho;
    jac.set(Ca44, K41, scratch);

    scratch = screened_rates(k_n_ca43__ca44)*Y(N)*state.rho;
    jac.set(Ca44, Ca43, scratch);

    scratch = -screened_rates(k_ca44__he4_ar40) - screened_rates(k_ca44__n_ca43) - screened_rates(k_he4_ca44__n_ti47)*Y(He4)*state.rho - screened_rates(k_he4_ca44__p_sc47)*Y(He4)*state.rho - screened_rates(k_he4_ca44__ti48)*Y(He4)*state.rho - screened_rates(k_n_ca44__ca45)*Y(N)*state.rho - screened_rates(k_p_ca44__he4_k41)*Y(H1)*state.rho - screened_rates(k_p_ca44__n_sc44)*Y(H1)*state.rho - screened_rates(k_p_ca44__sc45)*Y(H1)*state.rho;
    jac.set(Ca44, Ca44, scratch);

    scratch = screened_rates(k_ca45__n_ca44);
    jac.set(Ca44, Ca45, scratch);

    scratch = screened_rates(k_n_sc44__p_ca44)*Y(N)*state.rho + screened_rates(k_sc44__ca44__weak__wc12);
    jac.set(Ca44, Sc44, scratch);

    scratch = screened_rates(k_sc45__p_ca44);
    jac.set(Ca44, Sc45, scratch);

    scratch = screened_rates(k_p_sc47__he4_ca44)*Y(H1)*state.rho;
    jac.set(Ca44, Sc47, scratch);

    scratch = screened_rates(k_n_ti47__he4_ca44)*Y(N)*state.rho;
    jac.set(Ca44, Ti47, scratch);

    scratch = screened_rates(k_ti48__he4_ca44);
    jac.set(Ca44, Ti48, scratch);

    scratch = screened_rates(k_n_ca44__ca45)*Y(Ca44)*state.rho - screened_rates(k_n_ca45__ca46)*Y(Ca45)*state.rho + screened_rates(k_n_sc45__p_ca45)*Y(Sc45)*state.rho + screened_rates(k_n_ti48__he4_ca45)*Y(Ti48)*state.rho;
    jac.set(Ca45, N, scratch);

    scratch = -screened_rates(k_p_ca45__n_sc45)*Y(Ca45)*state.rho - screened_rates(k_p_ca45__sc46)*Y(Ca45)*state.rho + screened_rates(k_p_sc48__he4_ca45)*Y(Sc48)*state.rho;
    jac.set(Ca45, H1, scratch);

    scratch = -screened_rates(k_he4_ca45__n_ti48)*Y(Ca45)*state.rho - screened_rates(k_he4_ca45__p_sc48)*Y(Ca45)*state.rho - screened_rates(k_he4_ca45__ti49)*Y(Ca45)*state.rho;
    jac.set(Ca45, He4, scratch);

    scratch = screened_rates(k_n_ca44__ca45)*Y(N)*state.rho;
    jac.set(Ca45, Ca44, scratch);

    scratch = -screened_rates(k_ca45__n_ca44) - screened_rates(k_ca45__sc45__weak__wc12) - screened_rates(k_he4_ca45__n_ti48)*Y(He4)*state.rho - screened_rates(k_he4_ca45__p_sc48)*Y(He4)*state.rho - screened_rates(k_he4_ca45__ti49)*Y(He4)*state.rho - screened_rates(k_n_ca45__ca46)*Y(N)*state.rho - screened_rates(k_p_ca45__n_sc45)*Y(H1)*state.rho - screened_rates(k_p_ca45__sc46)*Y(H1)*state.rho;
    jac.set(Ca45, Ca45, scratch);

    scratch = screened_rates(k_ca46__n_ca45);
    jac.set(Ca45, Ca46, scratch);

    scratch = screened_rates(k_n_sc45__p_ca45)*Y(N)*state.rho;
    jac.set(Ca45, Sc45, scratch);

    scratch = screened_rates(k_sc46__p_ca45);
    jac.set(Ca45, Sc46, scratch);

    scratch = screened_rates(k_p_sc48__he4_ca45)*Y(H1)*state.rho;
    jac.set(Ca45, Sc48, scratch);

    scratch = screened_rates(k_n_ti48__he4_ca45)*Y(N)*state.rho;
    jac.set(Ca45, Ti48, scratch);

    scratch = screened_rates(k_ti49__he4_ca45);
    jac.set(Ca45, Ti49, scratch);

    scratch = screened_rates(k_n_ca45__ca46)*Y(Ca45)*state.rho - screened_rates(k_n_ca46__ca47)*Y(Ca46)*state.rho + screened_rates(k_n_sc46__p_ca46)*Y(Sc46)*state.rho + screened_rates(k_n_ti49__he4_ca46)*Y(Ti49)*state.rho;
    jac.set(Ca46, N, scratch);

    scratch = -screened_rates(k_p_ca46__n_sc46)*Y(Ca46)*state.rho - screened_rates(k_p_ca46__sc47)*Y(Ca46)*state.rho + screened_rates(k_p_sc49__he4_ca46)*Y(Sc49)*state.rho;
    jac.set(Ca46, H1, scratch);

    scratch = -screened_rates(k_he4_ca46__n_ti49)*Y(Ca46)*state.rho - screened_rates(k_he4_ca46__p_sc49)*Y(Ca46)*state.rho - screened_rates(k_he4_ca46__ti50)*Y(Ca46)*state.rho;
    jac.set(Ca46, He4, scratch);

    scratch = screened_rates(k_n_ca45__ca46)*Y(N)*state.rho;
    jac.set(Ca46, Ca45, scratch);

    scratch = -screened_rates(k_ca46__n_ca45) - screened_rates(k_he4_ca46__n_ti49)*Y(He4)*state.rho - screened_rates(k_he4_ca46__p_sc49)*Y(He4)*state.rho - screened_rates(k_he4_ca46__ti50)*Y(He4)*state.rho - screened_rates(k_n_ca46__ca47)*Y(N)*state.rho - screened_rates(k_p_ca46__n_sc46)*Y(H1)*state.rho - screened_rates(k_p_ca46__sc47)*Y(H1)*state.rho;
    jac.set(Ca46, Ca46, scratch);

    scratch = screened_rates(k_ca47__n_ca46);
    jac.set(Ca46, Ca47, scratch);

    scratch = screened_rates(k_n_sc46__p_ca46)*Y(N)*state.rho;
    jac.set(Ca46, Sc46, scratch);

    scratch = screened_rates(k_sc47__p_ca46);
    jac.set(Ca46, Sc47, scratch);

    scratch = screened_rates(k_p_sc49__he4_ca46)*Y(H1)*state.rho;
    jac.set(Ca46, Sc49, scratch);

    scratch = screened_rates(k_n_ti49__he4_ca46)*Y(N)*state.rho;
    jac.set(Ca46, Ti49, scratch);

    scratch = screened_rates(k_ti50__he4_ca46);
    jac.set(Ca46, Ti50, scratch);

    scratch = screened_rates(k_n_ca46__ca47)*Y(Ca46)*state.rho - screened_rates(k_n_ca47__ca48)*Y(Ca47)*state.rho + screened_rates(k_n_sc47__p_ca47)*Y(Sc47)*state.rho + screened_rates(k_n_ti50__he4_ca47)*Y(Ti50)*state.rho;
    jac.set(Ca47, N, scratch);

    scratch = -screened_rates(k_p_ca47__n_sc47)*Y(Ca47)*state.rho - screened_rates(k_p_ca47__sc48)*Y(Ca47)*state.rho;
    jac.set(Ca47, H1, scratch);

    scratch = -screened_rates(k_he4_ca47__n_ti50)*Y(Ca47)*state.rho - screened_rates(k_he4_ca47__ti51)*Y(Ca47)*state.rho;
    jac.set(Ca47, He4, scratch);

    scratch = screened_rates(k_n_ca46__ca47)*Y(N)*state.rho;
    jac.set(Ca47, Ca46, scratch);

    scratch = -screened_rates(k_ca47__n_ca46) - screened_rates(k_ca47__sc47__weak__wc12) - screened_rates(k_he4_ca47__n_ti50)*Y(He4)*state.rho - screened_rates(k_he4_ca47__ti51)*Y(He4)*state.rho - screened_rates(k_n_ca47__ca48)*Y(N)*state.rho - screened_rates(k_p_ca47__n_sc47)*Y(H1)*state.rho - screened_rates(k_p_ca47__sc48)*Y(H1)*state.rho;
    jac.set(Ca47, Ca47, scratch);

    scratch = screened_rates(k_ca48__n_ca47);
    jac.set(Ca47, Ca48, scratch);

    scratch = screened_rates(k_n_sc47__p_ca47)*Y(N)*state.rho;
    jac.set(Ca47, Sc47, scratch);

    scratch = screened_rates(k_sc48__p_ca47);
    jac.set(Ca47, Sc48, scratch);

    scratch = screened_rates(k_n_ti50__he4_ca47)*Y(N)*state.rho;
    jac.set(Ca47, Ti50, scratch);

    scratch = screened_rates(k_ti51__he4_ca47);
    jac.set(Ca47, Ti51, scratch);

    scratch = screened_rates(k_n_ca47__ca48)*Y(Ca47)*state.rho + screened_rates(k_n_sc48__p_ca48)*Y(Sc48)*state.rho + screened_rates(k_n_ti51__he4_ca48)*Y(Ti51)*state.rho;
    jac.set(Ca48, N, scratch);

    scratch = -screened_rates(k_p_ca48__n_sc48)*Y(Ca48)*state.rho - screened_rates(k_p_ca48__sc49)*Y(Ca48)*state.rho;
    jac.set(Ca48, H1, scratch);

    scratch = -screened_rates(k_he4_ca48__n_ti51)*Y(Ca48)*state.rho;
    jac.set(Ca48, He4, scratch);

    scratch = screened_rates(k_n_ca47__ca48)*Y(N)*state.rho;
    jac.set(Ca48, Ca47, scratch);

    scratch = -screened_rates(k_ca48__n_ca47) - screened_rates(k_ca48__sc48__weak__mo03) - screened_rates(k_he4_ca48__n_ti51)*Y(He4)*state.rho - screened_rates(k_p_ca48__n_sc48)*Y(H1)*state.rho - screened_rates(k_p_ca48__sc49)*Y(H1)*state.rho;
    jac.set(Ca48, Ca48, scratch);

    scratch = screened_rates(k_n_sc48__p_ca48)*Y(N)*state.rho;
    jac.set(Ca48, Sc48, scratch);

    scratch = screened_rates(k_sc49__p_ca48);
    jac.set(Ca48, Sc49, scratch);

    scratch = screened_rates(k_n_ti51__he4_ca48)*Y(N)*state.rho;
    jac.set(Ca48, Ti51, scratch);

    scratch = -screened_rates(k_n_sc43__he4_k40)*Y(Sc43)*state.rho - screened_rates(k_n_sc43__p_ca43)*Y(Sc43)*state.rho - screened_rates(k_n_sc43__sc44)*Y(Sc43)*state.rho + screened_rates(k_n_v46__he4_sc43)*Y(V46)*state.rho;
    jac.set(Sc43, N, scratch);

    scratch = screened_rates(k_p_ca42__sc43)*Y(Ca42)*state.rho + screened_rates(k_p_ca43__n_sc43)*Y(Ca43)*state.rho - screened_rates(k_p_sc43__he4_ca40)*Y(Sc43)*state.rho - screened_rates(k_p_sc43__ti44)*Y(Sc43)*state.rho + screened_rates(k_p_ti46__he4_sc43)*Y(Ti46)*state.rho;
    jac.set(Sc43, H1, scratch);

    scratch = screened_rates(k_he4_ca40__p_sc43)*Y(Ca40)*state.rho + screened_rates(k_he4_k39__sc43)*Y(K39)*state.rho + screened_rates(k_he4_k40__n_sc43)*Y(K40)*state.rho - screened_rates(k_he4_sc43__n_v46)*Y(Sc43)*state.rho - screened_rates(k_he4_sc43__p_ti46)*Y(Sc43)*state.rho - screened_rates(k_he4_sc43__v47)*Y(Sc43)*state.rho;
    jac.set(Sc43, He4, scratch);

    scratch = screened_rates(k_he4_k39__sc43)*Y(He4)*state.rho;
    jac.set(Sc43, K39, scratch);

    scratch = screened_rates(k_he4_k40__n_sc43)*Y(He4)*state.rho;
    jac.set(Sc43, K40, scratch);

    scratch = screened_rates(k_he4_ca40__p_sc43)*Y(He4)*state.rho;
    jac.set(Sc43, Ca40, scratch);

    scratch = screened_rates(k_p_ca42__sc43)*Y(H1)*state.rho;
    jac.set(Sc43, Ca42, scratch);

    scratch = screened_rates(k_p_ca43__n_sc43)*Y(H1)*state.rho;
    jac.set(Sc43, Ca43, scratch);

    scratch = -screened_rates(k_he4_sc43__n_v46)*Y(He4)*state.rho - screened_rates(k_he4_sc43__p_ti46)*Y(He4)*state.rho - screened_rates(k_he4_sc43__v47)*Y(He4)*state.rho - screened_rates(k_n_sc43__he4_k40)*Y(N)*state.rho - screened_rates(k_n_sc43__p_ca43)*Y(N)*state.rho - screened_rates(k_n_sc43__sc44)*Y(N)*state.rho - screened_rates(k_p_sc43__he4_ca40)*Y(H1)*state.rho - screened_rates(k_p_sc43__ti44)*Y(H1)*state.rho - screened_rates(k_sc43__ca43__weak__wc12) - screened_rates(k_sc43__he4_k39) - screened_rates(k_sc43__p_ca42);
    jac.set(Sc43, Sc43, scratch);

    scratch = screened_rates(k_sc44__n_sc43);
    jac.set(Sc43, Sc44, scratch);

    scratch = screened_rates(k_ti44__p_sc43);
    jac.set(Sc43, Ti44, scratch);

    scratch = screened_rates(k_p_ti46__he4_sc43)*Y(H1)*state.rho;
    jac.set(Sc43, Ti46, scratch);

    scratch = screened_rates(k_n_v46__he4_sc43)*Y(N)*state.rho;
    jac.set(Sc43, V46, scratch);

    scratch = screened_rates(k_v47__he4_sc43);
    jac.set(Sc43, V47, scratch);

    scratch = screened_rates(k_n_sc43__sc44)*Y(Sc43)*state.rho - screened_rates(k_n_sc44__he4_k41)*Y(Sc44)*state.rho - screened_rates(k_n_sc44__p_ca44)*Y(Sc44)*state.rho - screened_rates(k_n_sc44__sc45)*Y(Sc44)*state.rho + screened_rates(k_n_ti44__p_sc44)*Y(Ti44)*state.rho + screened_rates(k_n_v47__he4_sc44)*Y(V47)*state.rho;
    jac.set(Sc44, N, scratch);

    scratch = screened_rates(k_p_ca43__sc44)*Y(Ca43)*state.rho + screened_rates(k_p_ca44__n_sc44)*Y(Ca44)*state.rho - screened_rates(k_p_sc44__he4_ca41)*Y(Sc44)*state.rho - screened_rates(k_p_sc44__n_ti44)*Y(Sc44)*state.rho - screened_rates(k_p_sc44__ti45)*Y(Sc44)*state.rho + screened_rates(k_p_ti47__he4_sc44)*Y(Ti47)*state.rho;
    jac.set(Sc44, H1, scratch);

    scratch = screened_rates(k_he4_ca41__p_sc44)*Y(Ca41)*state.rho + screened_rates(k_he4_k40__sc44)*Y(K40)*state.rho + screened_rates(k_he4_k41__n_sc44)*Y(K41)*state.rho - screened_rates(k_he4_sc44__n_v47)*Y(Sc44)*state.rho - screened_rates(k_he4_sc44__p_ti47)*Y(Sc44)*state.rho - screened_rates(k_he4_sc44__v48)*Y(Sc44)*state.rho;
    jac.set(Sc44, He4, scratch);

    scratch = screened_rates(k_he4_k40__sc44)*Y(He4)*state.rho;
    jac.set(Sc44, K40, scratch);

    scratch = screened_rates(k_he4_k41__n_sc44)*Y(He4)*state.rho;
    jac.set(Sc44, K41, scratch);

    scratch = screened_rates(k_he4_ca41__p_sc44)*Y(He4)*state.rho;
    jac.set(Sc44, Ca41, scratch);

    scratch = screened_rates(k_p_ca43__sc44)*Y(H1)*state.rho;
    jac.set(Sc44, Ca43, scratch);

    scratch = screened_rates(k_p_ca44__n_sc44)*Y(H1)*state.rho;
    jac.set(Sc44, Ca44, scratch);

    scratch = screened_rates(k_n_sc43__sc44)*Y(N)*state.rho;
    jac.set(Sc44, Sc43, scratch);

    scratch = -screened_rates(k_he4_sc44__n_v47)*Y(He4)*state.rho - screened_rates(k_he4_sc44__p_ti47)*Y(He4)*state.rho - screened_rates(k_he4_sc44__v48)*Y(He4)*state.rho - screened_rates(k_n_sc44__he4_k41)*Y(N)*state.rho - screened_rates(k_n_sc44__p_ca44)*Y(N)*state.rho - screened_rates(k_n_sc44__sc45)*Y(N)*state.rho - screened_rates(k_p_sc44__he4_ca41)*Y(H1)*state.rho - screened_rates(k_p_sc44__n_ti44)*Y(H1)*state.rho - screened_rates(k_p_sc44__ti45)*Y(H1)*state.rho - screened_rates(k_sc44__ca44__weak__wc12) - screened_rates(k_sc44__he4_k40) - screened_rates(k_sc44__n_sc43) - screened_rates(k_sc44__p_ca43);
    jac.set(Sc44, Sc44, scratch);

    scratch = screened_rates(k_sc45__n_sc44);
    jac.set(Sc44, Sc45, scratch);

    scratch = screened_rates(k_n_ti44__p_sc44)*Y(N)*state.rho + screened_rates(k_ti44__sc44__weak__wc12);
    jac.set(Sc44, Ti44, scratch);

    scratch = screened_rates(k_ti45__p_sc44);
    jac.set(Sc44, Ti45, scratch);

    scratch = screened_rates(k_p_ti47__he4_sc44)*Y(H1)*state.rho;
    jac.set(Sc44, Ti47, scratch);

    scratch = screened_rates(k_n_v47__he4_sc44)*Y(N)*state.rho;
    jac.set(Sc44, V47, scratch);

    scratch = screened_rates(k_v48__he4_sc44);
    jac.set(Sc44, V48, scratch);

    scratch = screened_rates(k_n_sc44__sc45)*Y(Sc44)*state.rho - screened_rates(k_n_sc45__p_ca45)*Y(Sc45)*state.rho - screened_rates(k_n_sc45__sc46)*Y(Sc45)*state.rho + screened_rates(k_n_ti45__p_sc45)*Y(Ti45)*state.rho + screened_rates(k_n_v48__he4_sc45)*Y(V48)*state.rho;
    jac.set(Sc45, N, scratch);

    scratch = screened_rates(k_p_ca44__sc45)*Y(Ca44)*state.rho + screened_rates(k_p_ca45__n_sc45)*Y(Ca45)*state.rho - screened_rates(k_p_sc45__he4_ca42)*Y(Sc45)*state.rho - screened_rates(k_p_sc45__n_ti45)*Y(Sc45)*state.rho - screened_rates(k_p_sc45__ti46)*Y(Sc45)*state.rho + screened_rates(k_p_ti48__he4_sc45)*Y(Ti48)*state.rho;
    jac.set(Sc45, H1, scratch);

    scratch = screened_rates(k_he4_ca42__p_sc45)*Y(Ca42)*state.rho + screened_rates(k_he4_k41__sc45)*Y(K41)*state.rho - screened_rates(k_he4_sc45__n_v48)*Y(Sc45)*state.rho - screened_rates(k_he4_sc45__p_ti48)*Y(Sc45)*state.rho - screened_rates(k_he4_sc45__v49)*Y(Sc45)*state.rho;
    jac.set(Sc45, He4, scratch);

    scratch = screened_rates(k_he4_k41__sc45)*Y(He4)*state.rho;
    jac.set(Sc45, K41, scratch);

    scratch = screened_rates(k_he4_ca42__p_sc45)*Y(He4)*state.rho;
    jac.set(Sc45, Ca42, scratch);

    scratch = screened_rates(k_p_ca44__sc45)*Y(H1)*state.rho;
    jac.set(Sc45, Ca44, scratch);

    scratch = screened_rates(k_ca45__sc45__weak__wc12) + screened_rates(k_p_ca45__n_sc45)*Y(H1)*state.rho;
    jac.set(Sc45, Ca45, scratch);

    scratch = screened_rates(k_n_sc44__sc45)*Y(N)*state.rho;
    jac.set(Sc45, Sc44, scratch);

    scratch = -screened_rates(k_he4_sc45__n_v48)*Y(He4)*state.rho - screened_rates(k_he4_sc45__p_ti48)*Y(He4)*state.rho - screened_rates(k_he4_sc45__v49)*Y(He4)*state.rho - screened_rates(k_n_sc45__p_ca45)*Y(N)*state.rho - screened_rates(k_n_sc45__sc46)*Y(N)*state.rho - screened_rates(k_p_sc45__he4_ca42)*Y(H1)*state.rho - screened_rates(k_p_sc45__n_ti45)*Y(H1)*state.rho - screened_rates(k_p_sc45__ti46)*Y(H1)*state.rho - screened_rates(k_sc45__he4_k41) - screened_rates(k_sc45__n_sc44) - screened_rates(k_sc45__p_ca44);
    jac.set(Sc45, Sc45, scratch);

    scratch = screened_rates(k_sc46__n_sc45);
    jac.set(Sc45, Sc46, scratch);

    scratch = screened_rates(k_n_ti45__p_sc45)*Y(N)*state.rho + screened_rates(k_ti45__sc45__weak__wc12);
    jac.set(Sc45, Ti45, scratch);

    scratch = screened_rates(k_ti46__p_sc45);
    jac.set(Sc45, Ti46, scratch);

    scratch = screened_rates(k_p_ti48__he4_sc45)*Y(H1)*state.rho;
    jac.set(Sc45, Ti48, scratch);

    scratch = screened_rates(k_n_v48__he4_sc45)*Y(N)*state.rho;
    jac.set(Sc45, V48, scratch);

    scratch = screened_rates(k_v49__he4_sc45);
    jac.set(Sc45, V49, scratch);

    scratch = screened_rates(k_n_sc45__sc46)*Y(Sc45)*state.rho - screened_rates(k_n_sc46__p_ca46)*Y(Sc46)*state.rho - screened_rates(k_n_sc46__sc47)*Y(Sc46)*state.rho + screened_rates(k_n_ti46__p_sc46)*Y(Ti46)*state.rho + screened_rates(k_n_v49__he4_sc46)*Y(V49)*state.rho;
    jac.set(Sc46, N, scratch);

    scratch = screened_rates(k_p_ca45__sc46)*Y(Ca45)*state.rho + screened_rates(k_p_ca46__n_sc46)*Y(Ca46)*state.rho - screened_rates(k_p_sc46__he4_ca43)*Y(Sc46)*state.rho - screened_rates(k_p_sc46__n_ti46)*Y(Sc46)*state.rho - screened_rates(k_p_sc46__ti47)*Y(Sc46)*state.rho + screened_rates(k_p_ti49__he4_sc46)*Y(Ti49)*state.rho;
    jac.set(Sc46, H1, scratch);

    scratch = screened_rates(k_he4_ca43__p_sc46)*Y(Ca43)*state.rho - screened_rates(k_he4_sc46__n_v49)*Y(Sc46)*state.rho - screened_rates(k_he4_sc46__p_ti49)*Y(Sc46)*state.rho - screened_rates(k_he4_sc46__v50)*Y(Sc46)*state.rho;
    jac.set(Sc46, He4, scratch);

    scratch = screened_rates(k_he4_ca43__p_sc46)*Y(He4)*state.rho;
    jac.set(Sc46, Ca43, scratch);

    scratch = screened_rates(k_p_ca45__sc46)*Y(H1)*state.rho;
    jac.set(Sc46, Ca45, scratch);

    scratch = screened_rates(k_p_ca46__n_sc46)*Y(H1)*state.rho;
    jac.set(Sc46, Ca46, scratch);

    scratch = screened_rates(k_n_sc45__sc46)*Y(N)*state.rho;
    jac.set(Sc46, Sc45, scratch);

    scratch = -screened_rates(k_he4_sc46__n_v49)*Y(He4)*state.rho - screened_rates(k_he4_sc46__p_ti49)*Y(He4)*state.rho - screened_rates(k_he4_sc46__v50)*Y(He4)*state.rho - screened_rates(k_n_sc46__p_ca46)*Y(N)*state.rho - screened_rates(k_n_sc46__sc47)*Y(N)*state.rho - screened_rates(k_p_sc46__he4_ca43)*Y(H1)*state.rho - screened_rates(k_p_sc46__n_ti46)*Y(H1)*state.rho - screened_rates(k_p_sc46__ti47)*Y(H1)*state.rho - screened_rates(k_sc46__n_sc45) - screened_rates(k_sc46__p_ca45) - screened_rates(k_sc46__ti46__weak__wc12);
    jac.set(Sc46, Sc46, scratch);

    scratch = screened_rates(k_sc47__n_sc46);
    jac.set(Sc46, Sc47, scratch);

    scratch = screened_rates(k_n_ti46__p_sc46)*Y(N)*state.rho;
    jac.set(Sc46, Ti46, scratch);

    scratch = screened_rates(k_ti47__p_sc46);
    jac.set(Sc46, Ti47, scratch);

    scratch = screened_rates(k_p_ti49__he4_sc46)*Y(H1)*state.rho;
    jac.set(Sc46, Ti49, scratch);

    scratch = screened_rates(k_n_v49__he4_sc46)*Y(N)*state.rho;
    jac.set(Sc46, V49, scratch);

    scratch = screened_rates(k_v50__he4_sc46);
    jac.set(Sc46, V50, scratch);

    scratch = screened_rates(k_n_sc46__sc47)*Y(Sc46)*state.rho - screened_rates(k_n_sc47__p_ca47)*Y(Sc47)*state.rho - screened_rates(k_n_sc47__sc48)*Y(Sc47)*state.rho + screened_rates(k_n_ti47__p_sc47)*Y(Ti47)*state.rho + screened_rates(k_n_v50__he4_sc47)*Y(V50)*state.rho;
    jac.set(Sc47, N, scratch);

    scratch = screened_rates(k_p_ca46__sc47)*Y(Ca46)*state.rho + screened_rates(k_p_ca47__n_sc47)*Y(Ca47)*state.rho - screened_rates(k_p_sc47__he4_ca44)*Y(Sc47)*state.rho - screened_rates(k_p_sc47__n_ti47)*Y(Sc47)*state.rho - screened_rates(k_p_sc47__ti48)*Y(Sc47)*state.rho + screened_rates(k_p_ti50__he4_sc47)*Y(Ti50)*state.rho;
    jac.set(Sc47, H1, scratch);

    scratch = screened_rates(k_he4_ca44__p_sc47)*Y(Ca44)*state.rho - screened_rates(k_he4_sc47__n_v50)*Y(Sc47)*state.rho - screened_rates(k_he4_sc47__p_ti50)*Y(Sc47)*state.rho - screened_rates(k_he4_sc47__v51)*Y(Sc47)*state.rho;
    jac.set(Sc47, He4, scratch);

    scratch = screened_rates(k_he4_ca44__p_sc47)*Y(He4)*state.rho;
    jac.set(Sc47, Ca44, scratch);

    scratch = screened_rates(k_p_ca46__sc47)*Y(H1)*state.rho;
    jac.set(Sc47, Ca46, scratch);

    scratch = screened_rates(k_ca47__sc47__weak__wc12) + screened_rates(k_p_ca47__n_sc47)*Y(H1)*state.rho;
    jac.set(Sc47, Ca47, scratch);

    scratch = screened_rates(k_n_sc46__sc47)*Y(N)*state.rho;
    jac.set(Sc47, Sc46, scratch);

    scratch = -screened_rates(k_he4_sc47__n_v50)*Y(He4)*state.rho - screened_rates(k_he4_sc47__p_ti50)*Y(He4)*state.rho - screened_rates(k_he4_sc47__v51)*Y(He4)*state.rho - screened_rates(k_n_sc47__p_ca47)*Y(N)*state.rho - screened_rates(k_n_sc47__sc48)*Y(N)*state.rho - screened_rates(k_p_sc47__he4_ca44)*Y(H1)*state.rho - screened_rates(k_p_sc47__n_ti47)*Y(H1)*state.rho - screened_rates(k_p_sc47__ti48)*Y(H1)*state.rho - screened_rates(k_sc47__n_sc46) - screened_rates(k_sc47__p_ca46) - screened_rates(k_sc47__ti47__weak__wc12);
    jac.set(Sc47, Sc47, scratch);

    scratch = screened_rates(k_sc48__n_sc47);
    jac.set(Sc47, Sc48, scratch);

    scratch = screened_rates(k_n_ti47__p_sc47)*Y(N)*state.rho;
    jac.set(Sc47, Ti47, scratch);

    scratch = screened_rates(k_ti48__p_sc47);
    jac.set(Sc47, Ti48, scratch);

    scratch = screened_rates(k_p_ti50__he4_sc47)*Y(H1)*state.rho;
    jac.set(Sc47, Ti50, scratch);

    scratch = screened_rates(k_n_v50__he4_sc47)*Y(N)*state.rho;
    jac.set(Sc47, V50, scratch);

    scratch = screened_rates(k_v51__he4_sc47);
    jac.set(Sc47, V51, scratch);

    scratch = screened_rates(k_n_sc47__sc48)*Y(Sc47)*state.rho - screened_rates(k_n_sc48__p_ca48)*Y(Sc48)*state.rho - screened_rates(k_n_sc48__sc49)*Y(Sc48)*state.rho + screened_rates(k_n_ti48__p_sc48)*Y(Ti48)*state.rho + screened_rates(k_n_v51__he4_sc48)*Y(V51)*state.rho;
    jac.set(Sc48, N, scratch);

    scratch = screened_rates(k_p_ca47__sc48)*Y(Ca47)*state.rho + screened_rates(k_p_ca48__n_sc48)*Y(Ca48)*state.rho - screened_rates(k_p_sc48__he4_ca45)*Y(Sc48)*state.rho - screened_rates(k_p_sc48__n_ti48)*Y(Sc48)*state.rho - screened_rates(k_p_sc48__ti49)*Y(Sc48)*state.rho + screened_rates(k_p_ti51__he4_sc48)*Y(Ti51)*state.rho;
    jac.set(Sc48, H1, scratch);

    scratch = screened_rates(k_he4_ca45__p_sc48)*Y(Ca45)*state.rho - screened_rates(k_he4_sc48__n_v51)*Y(Sc48)*state.rho - screened_rates(k_he4_sc48__p_ti51)*Y(Sc48)*state.rho - screened_rates(k_he4_sc48__v52)*Y(Sc48)*state.rho;
    jac.set(Sc48, He4, scratch);

    scratch = screened_rates(k_he4_ca45__p_sc48)*Y(He4)*state.rho;
    jac.set(Sc48, Ca45, scratch);

    scratch = screened_rates(k_p_ca47__sc48)*Y(H1)*state.rho;
    jac.set(Sc48, Ca47, scratch);

    scratch = screened_rates(k_ca48__sc48__weak__mo03) + screened_rates(k_p_ca48__n_sc48)*Y(H1)*state.rho;
    jac.set(Sc48, Ca48, scratch);

    scratch = screened_rates(k_n_sc47__sc48)*Y(N)*state.rho;
    jac.set(Sc48, Sc47, scratch);

    scratch = -screened_rates(k_he4_sc48__n_v51)*Y(He4)*state.rho - screened_rates(k_he4_sc48__p_ti51)*Y(He4)*state.rho - screened_rates(k_he4_sc48__v52)*Y(He4)*state.rho - screened_rates(k_n_sc48__p_ca48)*Y(N)*state.rho - screened_rates(k_n_sc48__sc49)*Y(N)*state.rho - screened_rates(k_p_sc48__he4_ca45)*Y(H1)*state.rho - screened_rates(k_p_sc48__n_ti48)*Y(H1)*state.rho - screened_rates(k_p_sc48__ti49)*Y(H1)*state.rho - screened_rates(k_sc48__n_sc47) - screened_rates(k_sc48__p_ca47) - screened_rates(k_sc48__ti48__weak__wc12);
    jac.set(Sc48, Sc48, scratch);

    scratch = screened_rates(k_sc49__n_sc48);
    jac.set(Sc48, Sc49, scratch);

    scratch = screened_rates(k_n_ti48__p_sc48)*Y(N)*state.rho;
    jac.set(Sc48, Ti48, scratch);

    scratch = screened_rates(k_ti49__p_sc48);
    jac.set(Sc48, Ti49, scratch);

    scratch = screened_rates(k_p_ti51__he4_sc48)*Y(H1)*state.rho;
    jac.set(Sc48, Ti51, scratch);

    scratch = screened_rates(k_n_v51__he4_sc48)*Y(N)*state.rho;
    jac.set(Sc48, V51, scratch);

    scratch = screened_rates(k_v52__he4_sc48);
    jac.set(Sc48, V52, scratch);

    scratch = screened_rates(k_n_sc48__sc49)*Y(Sc48)*state.rho + screened_rates(k_n_ti49__p_sc49)*Y(Ti49)*state.rho + screened_rates(k_n_v52__he4_sc49)*Y(V52)*state.rho;
    jac.set(Sc49, N, scratch);

    scratch = screened_rates(k_p_ca48__sc49)*Y(Ca48)*state.rho - screened_rates(k_p_sc49__he4_ca46)*Y(Sc49)*state.rho - screened_rates(k_p_sc49__n_ti49)*Y(Sc49)*state.rho - screened_rates(k_p_sc49__ti50)*Y(Sc49)*state.rho;
    jac.set(Sc49, H1, scratch);

    scratch = screened_rates(k_he4_ca46__p_sc49)*Y(Ca46)*state.rho - screened_rates(k_he4_sc49__n_v52)*Y(Sc49)*state.rho;
    jac.set(Sc49, He4, scratch);

    scratch = screened_rates(k_he4_ca46__p_sc49)*Y(He4)*state.rho;
    jac.set(Sc49, Ca46, scratch);

    scratch = screened_rates(k_p_ca48__sc49)*Y(H1)*state.rho;
    jac.set(Sc49, Ca48, scratch);

    scratch = screened_rates(k_n_sc48__sc49)*Y(N)*state.rho;
    jac.set(Sc49, Sc48, scratch);

    scratch = -screened_rates(k_he4_sc49__n_v52)*Y(He4)*state.rho - screened_rates(k_p_sc49__he4_ca46)*Y(H1)*state.rho - screened_rates(k_p_sc49__n_ti49)*Y(H1)*state.rho - screened_rates(k_p_sc49__ti50)*Y(H1)*state.rho - screened_rates(k_sc49__n_sc48) - screened_rates(k_sc49__p_ca48) - screened_rates(k_sc49__ti49__weak__wc12);
    jac.set(Sc49, Sc49, scratch);

    scratch = screened_rates(k_n_ti49__p_sc49)*Y(N)*state.rho;
    jac.set(Sc49, Ti49, scratch);

    scratch = screened_rates(k_ti50__p_sc49);
    jac.set(Sc49, Ti50, scratch);

    scratch = screened_rates(k_n_v52__he4_sc49)*Y(N)*state.rho;
    jac.set(Sc49, V52, scratch);

    scratch = -screened_rates(k_n_ti44__he4_ca41)*Y(Ti44)*state.rho - screened_rates(k_n_ti44__p_sc44)*Y(Ti44)*state.rho - screened_rates(k_n_ti44__ti45)*Y(Ti44)*state.rho;
    jac.set(Ti44, N, scratch);

    scratch = screened_rates(k_p_sc43__ti44)*Y(Sc43)*state.rho + screened_rates(k_p_sc44__n_ti44)*Y(Sc44)*state.rho + screened_rates(k_p_v47__he4_ti44)*Y(V47)*state.rho;
    jac.set(Ti44, H1, scratch);

    scratch = screened_rates(k_he4_ca40__ti44)*Y(Ca40)*state.rho + screened_rates(k_he4_ca41__n_ti44)*Y(Ca41)*state.rho - screened_rates(k_he4_ti44__cr48)*Y(Ti44)*state.rho - screened_rates(k_he4_ti44__p_v47)*Y(Ti44)*state.rho;
    jac.set(Ti44, He4, scratch);

    scratch = screened_rates(k_he4_ca40__ti44)*Y(He4)*state.rho;
    jac.set(Ti44, Ca40, scratch);

    scratch = screened_rates(k_he4_ca41__n_ti44)*Y(He4)*state.rho;
    jac.set(Ti44, Ca41, scratch);

    scratch = screened_rates(k_p_sc43__ti44)*Y(H1)*state.rho;
    jac.set(Ti44, Sc43, scratch);

    scratch = screened_rates(k_p_sc44__n_ti44)*Y(H1)*state.rho;
    jac.set(Ti44, Sc44, scratch);

    scratch = -screened_rates(k_he4_ti44__cr48)*Y(He4)*state.rho - screened_rates(k_he4_ti44__p_v47)*Y(He4)*state.rho - screened_rates(k_n_ti44__he4_ca41)*Y(N)*state.rho - screened_rates(k_n_ti44__p_sc44)*Y(N)*state.rho - screened_rates(k_n_ti44__ti45)*Y(N)*state.rho - screened_rates(k_ti44__he4_ca40) - screened_rates(k_ti44__p_sc43) - screened_rates(k_ti44__sc44__weak__wc12);
    jac.set(Ti44, Ti44, scratch);

    scratch = screened_rates(k_ti45__n_ti44);
    jac.set(Ti44, Ti45, scratch);

    scratch = screened_rates(k_p_v47__he4_ti44)*Y(H1)*state.rho;
    jac.set(Ti44, V47, scratch);

    scratch = screened_rates(k_cr48__he4_ti44);
    jac.set(Ti44, Cr48, scratch);

    scratch = screened_rates(k_n_cr48__he4_ti45)*Y(Cr48)*state.rho + screened_rates(k_n_ti44__ti45)*Y(Ti44)*state.rho - screened_rates(k_n_ti45__he4_ca42)*Y(Ti45)*state.rho - screened_rates(k_n_ti45__p_sc45)*Y(Ti45)*state.rho - screened_rates(k_n_ti45__ti46)*Y(Ti45)*state.rho;
    jac.set(Ti45, N, scratch);

    scratch = screened_rates(k_p_sc44__ti45)*Y(Sc44)*state.rho + screened_rates(k_p_sc45__n_ti45)*Y(Sc45)*state.rho - screened_rates(k_p_ti45__v46)*Y(Ti45)*state.rho + screened_rates(k_p_v48__he4_ti45)*Y(V48)*state.rho;
    jac.set(Ti45, H1, scratch);

    scratch = screened_rates(k_he4_ca41__ti45)*Y(Ca41)*state.rho + screened_rates(k_he4_ca42__n_ti45)*Y(Ca42)*state.rho - screened_rates(k_he4_ti45__cr49)*Y(Ti45)*state.rho - screened_rates(k_he4_ti45__n_cr48)*Y(Ti45)*state.rho - screened_rates(k_he4_ti45__p_v48)*Y(Ti45)*state.rho;
    jac.set(Ti45, He4, scratch);

    scratch = screened_rates(k_he4_ca41__ti45)*Y(He4)*state.rho;
    jac.set(Ti45, Ca41, scratch);

    scratch = screened_rates(k_he4_ca42__n_ti45)*Y(He4)*state.rho;
    jac.set(Ti45, Ca42, scratch);

    scratch = screened_rates(k_p_sc44__ti45)*Y(H1)*state.rho;
    jac.set(Ti45, Sc44, scratch);

    scratch = screened_rates(k_p_sc45__n_ti45)*Y(H1)*state.rho;
    jac.set(Ti45, Sc45, scratch);

    scratch = screened_rates(k_n_ti44__ti45)*Y(N)*state.rho;
    jac.set(Ti45, Ti44, scratch);

    scratch = -screened_rates(k_he4_ti45__cr49)*Y(He4)*state.rho - screened_rates(k_he4_ti45__n_cr48)*Y(He4)*state.rho - screened_rates(k_he4_ti45__p_v48)*Y(He4)*state.rho - screened_rates(k_n_ti45__he4_ca42)*Y(N)*state.rho - screened_rates(k_n_ti45__p_sc45)*Y(N)*state.rho - screened_rates(k_n_ti45__ti46)*Y(N)*state.rho - screened_rates(k_p_ti45__v46)*Y(H1)*state.rho - screened_rates(k_ti45__he4_ca41) - screened_rates(k_ti45__n_ti44) - screened_rates(k_ti45__p_sc44) - screened_rates(k_ti45__sc45__weak__wc12);
    jac.set(Ti45, Ti45, scratch);

    scratch = screened_rates(k_ti46__n_ti45);
    jac.set(Ti45, Ti46, scratch);

    scratch = screened_rates(k_v46__p_ti45);
    jac.set(Ti45, V46, scratch);

    scratch = screened_rates(k_p_v48__he4_ti45)*Y(H1)*state.rho;
    jac.set(Ti45, V48, scratch);

    scratch = screened_rates(k_n_cr48__he4_ti45)*Y(N)*state.rho;
    jac.set(Ti45, Cr48, scratch);

    scratch = screened_rates(k_cr49__he4_ti45);
    jac.set(Ti45, Cr49, scratch);

    scratch = screened_rates(k_n_cr49__he4_ti46)*Y(Cr49)*state.rho + screened_rates(k_n_ti45__ti46)*Y(Ti45)*state.rho - screened_rates(k_n_ti46__he4_ca43)*Y(Ti46)*state.rho - screened_rates(k_n_ti46__p_sc46)*Y(Ti46)*state.rho - screened_rates(k_n_ti46__ti47)*Y(Ti46)*state.rho + screened_rates(k_n_v46__p_ti46)*Y(V46)*state.rho;
    jac.set(Ti46, N, scratch);

    scratch = screened_rates(k_p_sc45__ti46)*Y(Sc45)*state.rho + screened_rates(k_p_sc46__n_ti46)*Y(Sc46)*state.rho - screened_rates(k_p_ti46__he4_sc43)*Y(Ti46)*state.rho - screened_rates(k_p_ti46__n_v46)*Y(Ti46)*state.rho - screened_rates(k_p_ti46__v47)*Y(Ti46)*state.rho + screened_rates(k_p_v49__he4_ti46)*Y(V49)*state.rho;
    jac.set(Ti46, H1, scratch);

    scratch = screened_rates(k_he4_ca42__ti46)*Y(Ca42)*state.rho + screened_rates(k_he4_ca43__n_ti46)*Y(Ca43)*state.rho + screened_rates(k_he4_sc43__p_ti46)*Y(Sc43)*state.rho - screened_rates(k_he4_ti46__cr50)*Y(Ti46)*state.rho - screened_rates(k_he4_ti46__n_cr49)*Y(Ti46)*state.rho - screened_rates(k_he4_ti46__p_v49)*Y(Ti46)*state.rho;
    jac.set(Ti46, He4, scratch);

    scratch = screened_rates(k_he4_ca42__ti46)*Y(He4)*state.rho;
    jac.set(Ti46, Ca42, scratch);

    scratch = screened_rates(k_he4_ca43__n_ti46)*Y(He4)*state.rho;
    jac.set(Ti46, Ca43, scratch);

    scratch = screened_rates(k_he4_sc43__p_ti46)*Y(He4)*state.rho;
    jac.set(Ti46, Sc43, scratch);

    scratch = screened_rates(k_p_sc45__ti46)*Y(H1)*state.rho;
    jac.set(Ti46, Sc45, scratch);

    scratch = screened_rates(k_p_sc46__n_ti46)*Y(H1)*state.rho + screened_rates(k_sc46__ti46__weak__wc12);
    jac.set(Ti46, Sc46, scratch);

    scratch = screened_rates(k_n_ti45__ti46)*Y(N)*state.rho;
    jac.set(Ti46, Ti45, scratch);

    scratch = -screened_rates(k_he4_ti46__cr50)*Y(He4)*state.rho - screened_rates(k_he4_ti46__n_cr49)*Y(He4)*state.rho - screened_rates(k_he4_ti46__p_v49)*Y(He4)*state.rho - screened_rates(k_n_ti46__he4_ca43)*Y(N)*state.rho - screened_rates(k_n_ti46__p_sc46)*Y(N)*state.rho - screened_rates(k_n_ti46__ti47)*Y(N)*state.rho - screened_rates(k_p_ti46__he4_sc43)*Y(H1)*state.rho - screened_rates(k_p_ti46__n_v46)*Y(H1)*state.rho - screened_rates(k_p_ti46__v47)*Y(H1)*state.rho - screened_rates(k_ti46__he4_ca42) - screened_rates(k_ti46__n_ti45) - screened_rates(k_ti46__p_sc45);
    jac.set(Ti46, Ti46, scratch);

    scratch = screened_rates(k_ti47__n_ti46);
    jac.set(Ti46, Ti47, scratch);

    scratch = screened_rates(k_n_v46__p_ti46)*Y(N)*state.rho + screened_rates(k_v46__ti46__weak__wc12);
    jac.set(Ti46, V46, scratch);

    scratch = screened_rates(k_v47__p_ti46);
    jac.set(Ti46, V47, scratch);

    scratch = screened_rates(k_p_v49__he4_ti46)*Y(H1)*state.rho;
    jac.set(Ti46, V49, scratch);

    scratch = screened_rates(k_n_cr49__he4_ti46)*Y(N)*state.rho;
    jac.set(Ti46, Cr49, scratch);

    scratch = screened_rates(k_cr50__he4_ti46);
    jac.set(Ti46, Cr50, scratch);

    scratch = screened_rates(k_n_cr50__he4_ti47)*Y(Cr50)*state.rho + screened_rates(k_n_ti46__ti47)*Y(Ti46)*state.rho - screened_rates(k_n_ti47__he4_ca44)*Y(Ti47)*state.rho - screened_rates(k_n_ti47__p_sc47)*Y(Ti47)*state.rho - screened_rates(k_n_ti47__ti48)*Y(Ti47)*state.rho + screened_rates(k_n_v47__p_ti47)*Y(V47)*state.rho;
    jac.set(Ti47, N, scratch);

    scratch = screened_rates(k_p_sc46__ti47)*Y(Sc46)*state.rho + screened_rates(k_p_sc47__n_ti47)*Y(Sc47)*state.rho - screened_rates(k_p_ti47__he4_sc44)*Y(Ti47)*state.rho - screened_rates(k_p_ti47__n_v47)*Y(Ti47)*state.rho - screened_rates(k_p_ti47__v48)*Y(Ti47)*state.rho + screened_rates(k_p_v50__he4_ti47)*Y(V50)*state.rho;
    jac.set(Ti47, H1, scratch);

    scratch = screened_rates(k_he4_ca43__ti47)*Y(Ca43)*state.rho + screened_rates(k_he4_ca44__n_ti47)*Y(Ca44)*state.rho + screened_rates(k_he4_sc44__p_ti47)*Y(Sc44)*state.rho - screened_rates(k_he4_ti47__cr51)*Y(Ti47)*state.rho - screened_rates(k_he4_ti47__n_cr50)*Y(Ti47)*state.rho - screened_rates(k_he4_ti47__p_v50)*Y(Ti47)*state.rho;
    jac.set(Ti47, He4, scratch);

    scratch = screened_rates(k_he4_ca43__ti47)*Y(He4)*state.rho;
    jac.set(Ti47, Ca43, scratch);

    scratch = screened_rates(k_he4_ca44__n_ti47)*Y(He4)*state.rho;
    jac.set(Ti47, Ca44, scratch);

    scratch = screened_rates(k_he4_sc44__p_ti47)*Y(He4)*state.rho;
    jac.set(Ti47, Sc44, scratch);

    scratch = screened_rates(k_p_sc46__ti47)*Y(H1)*state.rho;
    jac.set(Ti47, Sc46, scratch);

    scratch = screened_rates(k_p_sc47__n_ti47)*Y(H1)*state.rho + screened_rates(k_sc47__ti47__weak__wc12);
    jac.set(Ti47, Sc47, scratch);

    scratch = screened_rates(k_n_ti46__ti47)*Y(N)*state.rho;
    jac.set(Ti47, Ti46, scratch);

    scratch = -screened_rates(k_he4_ti47__cr51)*Y(He4)*state.rho - screened_rates(k_he4_ti47__n_cr50)*Y(He4)*state.rho - screened_rates(k_he4_ti47__p_v50)*Y(He4)*state.rho - screened_rates(k_n_ti47__he4_ca44)*Y(N)*state.rho - screened_rates(k_n_ti47__p_sc47)*Y(N)*state.rho - screened_rates(k_n_ti47__ti48)*Y(N)*state.rho - screened_rates(k_p_ti47__he4_sc44)*Y(H1)*state.rho - screened_rates(k_p_ti47__n_v47)*Y(H1)*state.rho - screened_rates(k_p_ti47__v48)*Y(H1)*state.rho - screened_rates(k_ti47__he4_ca43) - screened_rates(k_ti47__n_ti46) - screened_rates(k_ti47__p_sc46);
    jac.set(Ti47, Ti47, scratch);

    scratch = screened_rates(k_ti48__n_ti47);
    jac.set(Ti47, Ti48, scratch);

    scratch = screened_rates(k_n_v47__p_ti47)*Y(N)*state.rho + screened_rates(k_v47__ti47__weak__wc12);
    jac.set(Ti47, V47, scratch);

    scratch = screened_rates(k_v48__p_ti47);
    jac.set(Ti47, V48, scratch);

    scratch = screened_rates(k_p_v50__he4_ti47)*Y(H1)*state.rho;
    jac.set(Ti47, V50, scratch);

    scratch = screened_rates(k_n_cr50__he4_ti47)*Y(N)*state.rho;
    jac.set(Ti47, Cr50, scratch);

    scratch = screened_rates(k_cr51__he4_ti47);
    jac.set(Ti47, Cr51, scratch);

    scratch = screened_rates(k_n_cr51__he4_ti48)*Y(Cr51)*state.rho + screened_rates(k_n_ti47__ti48)*Y(Ti47)*state.rho - screened_rates(k_n_ti48__he4_ca45)*Y(Ti48)*state.rho - screened_rates(k_n_ti48__p_sc48)*Y(Ti48)*state.rho - screened_rates(k_n_ti48__ti49)*Y(Ti48)*state.rho + screened_rates(k_n_v48__p_ti48)*Y(V48)*state.rho;
    jac.set(Ti48, N, scratch);

    scratch = screened_rates(k_p_sc47__ti48)*Y(Sc47)*state.rho + screened_rates(k_p_sc48__n_ti48)*Y(Sc48)*state.rho - screened_rates(k_p_ti48__he4_sc45)*Y(Ti48)*state.rho - screened_rates(k_p_ti48__n_v48)*Y(Ti48)*state.rho - screened_rates(k_p_ti48__v49)*Y(Ti48)*state.rho + screened_rates(k_p_v51__he4_ti48)*Y(V51)*state.rho;
    jac.set(Ti48, H1, scratch);

    scratch = screened_rates(k_he4_ca44__ti48)*Y(Ca44)*state.rho + screened_rates(k_he4_ca45__n_ti48)*Y(Ca45)*state.rho + screened_rates(k_he4_sc45__p_ti48)*Y(Sc45)*state.rho - screened_rates(k_he4_ti48__cr52)*Y(Ti48)*state.rho - screened_rates(k_he4_ti48__n_cr51)*Y(Ti48)*state.rho - screened_rates(k_he4_ti48__p_v51)*Y(Ti48)*state.rho;
    jac.set(Ti48, He4, scratch);

    scratch = screened_rates(k_he4_ca44__ti48)*Y(He4)*state.rho;
    jac.set(Ti48, Ca44, scratch);

    scratch = screened_rates(k_he4_ca45__n_ti48)*Y(He4)*state.rho;
    jac.set(Ti48, Ca45, scratch);

    scratch = screened_rates(k_he4_sc45__p_ti48)*Y(He4)*state.rho;
    jac.set(Ti48, Sc45, scratch);

    scratch = screened_rates(k_p_sc47__ti48)*Y(H1)*state.rho;
    jac.set(Ti48, Sc47, scratch);

    scratch = screened_rates(k_p_sc48__n_ti48)*Y(H1)*state.rho + screened_rates(k_sc48__ti48__weak__wc12);
    jac.set(Ti48, Sc48, scratch);

    scratch = screened_rates(k_n_ti47__ti48)*Y(N)*state.rho;
    jac.set(Ti48, Ti47, scratch);

    scratch = -screened_rates(k_he4_ti48__cr52)*Y(He4)*state.rho - screened_rates(k_he4_ti48__n_cr51)*Y(He4)*state.rho - screened_rates(k_he4_ti48__p_v51)*Y(He4)*state.rho - screened_rates(k_n_ti48__he4_ca45)*Y(N)*state.rho - screened_rates(k_n_ti48__p_sc48)*Y(N)*state.rho - screened_rates(k_n_ti48__ti49)*Y(N)*state.rho - screened_rates(k_p_ti48__he4_sc45)*Y(H1)*state.rho - screened_rates(k_p_ti48__n_v48)*Y(H1)*state.rho - screened_rates(k_p_ti48__v49)*Y(H1)*state.rho - screened_rates(k_ti48__he4_ca44) - screened_rates(k_ti48__n_ti47) - screened_rates(k_ti48__p_sc47);
    jac.set(Ti48, Ti48, scratch);

    scratch = screened_rates(k_ti49__n_ti48);
    jac.set(Ti48, Ti49, scratch);

    scratch = screened_rates(k_n_v48__p_ti48)*Y(N)*state.rho + screened_rates(k_v48__ti48__weak__wc12);
    jac.set(Ti48, V48, scratch);

    scratch = screened_rates(k_v49__p_ti48);
    jac.set(Ti48, V49, scratch);

    scratch = screened_rates(k_p_v51__he4_ti48)*Y(H1)*state.rho;
    jac.set(Ti48, V51, scratch);

    scratch = screened_rates(k_n_cr51__he4_ti48)*Y(N)*state.rho;
    jac.set(Ti48, Cr51, scratch);

    scratch = screened_rates(k_cr52__he4_ti48);
    jac.set(Ti48, Cr52, scratch);

    scratch = screened_rates(k_n_cr52__he4_ti49)*Y(Cr52)*state.rho + screened_rates(k_n_ti48__ti49)*Y(Ti48)*state.rho - screened_rates(k_n_ti49__he4_ca46)*Y(Ti49)*state.rho - screened_rates(k_n_ti49__p_sc49)*Y(Ti49)*state.rho - screened_rates(k_n_ti49__ti50)*Y(Ti49)*state.rho + screened_rates(k_n_v49__p_ti49)*Y(V49)*state.rho;
    jac.set(Ti49, N, scratch);

    scratch = screened_rates(k_p_sc48__ti49)*Y(Sc48)*state.rho + screened_rates(k_p_sc49__n_ti49)*Y(Sc49)*state.rho - screened_rates(k_p_ti49__he4_sc46)*Y(Ti49)*state.rho - screened_rates(k_p_ti49__n_v49)*Y(Ti49)*state.rho - screened_rates(k_p_ti49__v50)*Y(Ti49)*state.rho + screened_rates(k_p_v52__he4_ti49)*Y(V52)*state.rho;
    jac.set(Ti49, H1, scratch);

    scratch = screened_rates(k_he4_ca45__ti49)*Y(Ca45)*state.rho + screened_rates(k_he4_ca46__n_ti49)*Y(Ca46)*state.rho + screened_rates(k_he4_sc46__p_ti49)*Y(Sc46)*state.rho - screened_rates(k_he4_ti49__cr53)*Y(Ti49)*state.rho - screened_rates(k_he4_ti49__n_cr52)*Y(Ti49)*state.rho - screened_rates(k_he4_ti49__p_v52)*Y(Ti49)*state.rho;
    jac.set(Ti49, He4, scratch);

    scratch = screened_rates(k_he4_ca45__ti49)*Y(He4)*state.rho;
    jac.set(Ti49, Ca45, scratch);

    scratch = screened_rates(k_he4_ca46__n_ti49)*Y(He4)*state.rho;
    jac.set(Ti49, Ca46, scratch);

    scratch = screened_rates(k_he4_sc46__p_ti49)*Y(He4)*state.rho;
    jac.set(Ti49, Sc46, scratch);

    scratch = screened_rates(k_p_sc48__ti49)*Y(H1)*state.rho;
    jac.set(Ti49, Sc48, scratch);

    scratch = screened_rates(k_p_sc49__n_ti49)*Y(H1)*state.rho + screened_rates(k_sc49__ti49__weak__wc12);
    jac.set(Ti49, Sc49, scratch);

    scratch = screened_rates(k_n_ti48__ti49)*Y(N)*state.rho;
    jac.set(Ti49, Ti48, scratch);

    scratch = -screened_rates(k_he4_ti49__cr53)*Y(He4)*state.rho - screened_rates(k_he4_ti49__n_cr52)*Y(He4)*state.rho - screened_rates(k_he4_ti49__p_v52)*Y(He4)*state.rho - screened_rates(k_n_ti49__he4_ca46)*Y(N)*state.rho - screened_rates(k_n_ti49__p_sc49)*Y(N)*state.rho - screened_rates(k_n_ti49__ti50)*Y(N)*state.rho - screened_rates(k_p_ti49__he4_sc46)*Y(H1)*state.rho - screened_rates(k_p_ti49__n_v49)*Y(H1)*state.rho - screened_rates(k_p_ti49__v50)*Y(H1)*state.rho - screened_rates(k_ti49__he4_ca45) - screened_rates(k_ti49__n_ti48) - screened_rates(k_ti49__p_sc48);
    jac.set(Ti49, Ti49, scratch);

    scratch = screened_rates(k_ti50__n_ti49);
    jac.set(Ti49, Ti50, scratch);

    scratch = screened_rates(k_n_v49__p_ti49)*Y(N)*state.rho + screened_rates(k_v49__ti49__weak__wc12);
    jac.set(Ti49, V49, scratch);

    scratch = screened_rates(k_v50__p_ti49);
    jac.set(Ti49, V50, scratch);

    scratch = screened_rates(k_p_v52__he4_ti49)*Y(H1)*state.rho;
    jac.set(Ti49, V52, scratch);

    scratch = screened_rates(k_n_cr52__he4_ti49)*Y(N)*state.rho;
    jac.set(Ti49, Cr52, scratch);

    scratch = screened_rates(k_cr53__he4_ti49);
    jac.set(Ti49, Cr53, scratch)