#install.packages("optimx")
library(optimx)

#load data
d <- read.table("data.txt", header = TRUE, sep = "\t")

###remove Rumex from 2018
d$sort <- paste(d$year, d$focal_sp, sep = "_", collapse = NULL)
d <- subset(d, sort != "2018_rum_ace")

#sum of all neighbour cover measures
d$tot_nei <- apply(d[, 12:22], 1, sum, na.rm = TRUE)

#species names
splist <- unique(d$focal_sp)

#objects to save results
M4_lambda <- rep(NA, 8)
M4_alpha <- matrix(NA, 8, 8)
M5_lambda <- rep(NA, 8)
M5_alpha <- matrix(NA, 8, 8)
M5_theta <- rep(NA, 8)
M5_omega <- rep(NA, 8)
M6_lambda <- rep(NA, 8)
M6_alpha <- matrix(NA, 8, 8)
M6_theta <- rep(NA, 8)
M6_omega <- matrix(NA, 8, 8)

model <- rep(1:6, length(splist))
parameters <- rep(c(1, 2, 4, 12, 14, 24), length(splist))
species <- c(rep(splist[1], 6), rep(splist[2], 6), rep(splist[3], 6), rep(splist[4], 6),
             rep(splist[5], 6), rep(splist[6], 6), rep(splist[7], 6), rep(splist[8], 6))


##### MODELS:

#MODEL 1 - no competitive effects
compmodel1 <- function(par){

  #parameters
  lambda <- par[1] #mean of log_biomass
  sigma <- par[2] #sd of log_biomass
  
  #this is the predictive model - here is just fitting a horizontal line through the data:
  pred <- rep(lambda, times = length(log_biomass))
  
  #these are the log likelihoods of the data given the model + parameters
  llik <- dnorm(log_biomass, log(pred + 1), sd = sigma, log = TRUE)
  
  #return the sum of negative log likelihoods - what optim minimizes
  return(sum(-1 * llik)) 

} #end compmodel1()


### MODEL 2 --- competition, but no difference between species
compmodel2 <- function(par){
  
  lambda <- par[1] #same as model 1 (mean of log_biomass)
  alpha <- par[2] #new parameter introduced in model 2 (modulates total biomass)
  sigma <- par[3] #same as model 1 (sd of log_biomass)
  
  #predictive model:
  pred <- lambda / (1 + alpha * (data$tot_nei)) #uses the sumatory change in cover of all the neighbours
  
  #log likelihoods of data given the model + parameters:
  llik <- dnorm(log_biomass, log(pred + 1), sd = sigma, log = TRUE)
  
  ##return sum of negative log likelihoods:
  return(sum(-1 * llik))
  
} #end compmodel2()


### MODEL 3 --- competition, but no difference between species + SIZE
compmodel3 <- function(par){
  
  lambda <- par[1] #same as model 2 (mean of log_biomass)
  theta <- par[2] #new parameter introduced in model 3
  omega <- par[3] #new parameter introduced in model 3 (modulates size and tot_nei)
  alpha <- par[4] #same as model 2
  sigma <- par[5] #same as model 2 (sd of log_biomass)
  
  #predictive model:
  pred <- (lambda * (1 + theta * data$biomass_i)) / (1 + (alpha + omega * data$biomass_i) * data$tot_nei) #uses the sumatory change in cover of all the neighbours
  
  #log likelihoods of data given the model + parameters:
  llik <- dnorm(log_biomass, log(pred + 1), sd = sigma, log = TRUE)
  
  ##return sum of negative log likelihoods:
  return(sum(-1 * llik))
  
} #end compmodel3()


### MODEL 4 --- all species have different competitive effects
compmodel4 <- function(par){
  
  lambda <- par[1]
  a_ao <- par[2] #alpha from model 3
  a_cj <- par[3]
  a_cb <- par[4]
  a_dg <- par[5]
  a_pm <- par[6]
  a_ra <- par[7]
  a_sp <- par[8]
  a_to <- par[9]
  a_he <- par[10]
  a_gr <- par[11]
  a_le <- par[12]
  sigma <- par[13]
  
  #predictive model:
  pred <- lambda / (1 + a_ao * data$ant_odo + a_cj * data$cen_jac + a_cb * data$cre_bie + a_pm * data$pla_med +
                      a_ra * data$rum_ace + a_sp * data$sal_pra + a_to * data$tar_off + a_he * data$herbs +
                      a_gr * data$grasses + a_le * data$legumes)
  
  #likelihood as before:
  llik <- dnorm(log_biomass, log(pred + 1), sd = sigma, log = TRUE)
  
  #return sum of negative log likelihoods
  return(sum(-1 * llik)) #sum of negative log likelihoods
  
} #end compmodel4()


### MODEL 5 --- all species have different competitive effects but common size effect
compmodel5 <- function(par){
  
  lambda <- par[1] #same as model 2 (mean of log_biomass)
  theta <- par[2]
  omega <- par[3]
  a_ao <- par[4]	#new parameters - use alpha estimate from model 4 as the start value for fitting
  a_cj <- par[5]
  a_cb <- par[6]
  a_dg <- par[7]
  a_pm <- par[8]
  a_ra <- par[9]
  a_sp <- par[10]
  a_to <- par[11]
  a_he <- par[12]
  a_gr <- par[13]
  a_le <- par[14]
  sigma <- par[15] #same as model 2 (sd of log_biomass)
  
  #same form as model 1, but super long given all the species:
  pred <- (lambda * (1 + theta * data$biomass_i)) / (1 + (a_ao + omega * data$biomass_i) * data$ant_odo + (a_cj + omega * data$biomass_i) * data$cen_jac
                                                          + (a_cb + omega * data$biomass_i) * data$cre_bie + (a_dg + omega * data$biomass_i) * data$dac_glo
                                                          + (a_pm + omega * data$biomass_i) * data$pla_med + (a_ra + omega * data$biomass_i) * data$rum_ace
                                                          + (a_sp + omega * data$biomass_i) * data$sal_pra + (a_to + omega * data$biomass_i) * data$tar_off
                                                          + (a_he + omega * data$biomass_i) * data$herbs + (a_gr + omega * data$biomass_i) * data$grasses
                                                          + (a_le + omega * data$biomass_i) * data$legumes)
  
  #likelihood as before:
  llik <- dnorm(log_biomass, log(pred + 1), sd = sigma, log = TRUE)
  
  #return sum of negative log likelihoods
  return(sum(-1 * llik)) #sum of negative log likelihoods
  
} #end compmodel5()


### MODEL 6 --- all species have different competitive effects and different size effects
compmodel6 <- function(par){
  
  lambda <- par[1] #same as model 2 (mean of log_biomass)
  theta <- par[2]
  omega_ao <- par[3]	#new parameters - use alpha estimate from model 5 as the start value for fitting
  omega_cj <- par[4]
  omega_cb <- par[5]
  omega_dg <- par[6]
  omega_pm <- par[7]
  omega_ra <- par[8]
  omega_sp <- par[9]
  omega_to <- par[10]
  omega_he <- par[11]
  omega_gr <- par[12]
  omega_le <- par[13]
  a_ao <- par[14]
  a_cj <- par[15]
  a_cb <- par[16]
  a_dg <- par[17]
  a_pm <- par[18]
  a_ra <- par[19]
  a_sp <- par[20]
  a_to <- par[21]
  a_he <- par[22]
  a_gr <- par[23]
  a_le <- par[24]
  sigma <- par[25] #same as model 2 (sd of log_biomass)
  
  #same form as model 1, but super long given all the species:
  pred <- (lambda * (1 + theta * data$biomass_i)) / (1 + (a_ao + omega_ao * data$biomass_i) * data$ant_odo + (a_cj + omega_cj * data$biomass_i) * data$cen_jac
                                                          + (a_cb + omega_cb * data$biomass_i) * data$cre_bie + (a_dg + omega_dg * data$biomass_i) * data$dac_glo
                                                          + (a_pm + omega_pm * data$biomass_i) * data$pla_med + (a_ra + omega_ra * data$biomass_i) * data$rum_ace
                                                          + (a_sp + omega_sp * data$biomass_i) * data$sal_pra + (a_to + omega_to * data$biomass_i) * data$tar_off
                                                          + (a_he + omega_he * data$biomass_i) * data$herbs + (a_gr + omega_gr * data$biomass_i) * data$grasses
                                                          + (a_le + omega_le * data$biomass_i) * data$legumes)
  
  #likelihood as before:
  llik <- dnorm(log_biomass, log(pred + 1), sd = sigma, log = TRUE)
  
  #return sum of negative log likelihoods
  return(sum(-1 * llik)) #sum of negative log likelihoods
  
} #end compmodel6()


###APPLY optimx ####################################### for both years together, at first

#### CONTROL ####
dtreat <- subset(d, treatment == "control") #TREATMENT = CONTROL

loglikelihood <- NULL

for(i in 1:length(splist)){
  data <- subset(dtreat, focal_sp == splist[i])
  #data <- subset(data, year == 2017)
  log_biomass <- log(data$biomass_f + 1) - log(data$biomass_i + 1)
  
  #model 1
  par1 <- c(mean(log_biomass), sd(log_biomass))
  for(j in 1:length(par1)){
    if(par1[j] <= 0){
      par1[j] <- 0.000001
    }
  }
  result1 <- optimx(par = par1, fn = compmodel1, method = "Nelder-Mead")
  
  #model 2
  par2 <- c(result1$p1[1], 0.000001, result1$p2[1])
  for(j in 1:length(par2)){
    if(par2[j] <= 0){
      par2[j] <- 0.000001
    }
  }
  result2 <- optimx(par = par2, fn = compmodel2, method = "Nelder-Mead")
  
  #model 3
  par3 <- c(result2$p1[1], 0.000001, result2$p2[1], 0.000001, result2$p3[1])
  for(j in 1:length(par3)){
    if(par3[j] <= 0){
      par3[j] <- 0.000001
    }
  }
  result3 <- optimx(par = par3, fn = compmodel3, method = "Nelder-Mead")
  
  #model 4
  par4 <- c(result3$p1[1], rep(result3$p4[1], 11), result3$p5[1])
  for(j in 1:length(par4)){
    if(par4[j] <= 0){
      par4[j] <- 0.000001
    }
  }
  result4 <- optimx(par = par4, fn = compmodel4, method = "Nelder-Mead")
  
  #model 5
  par5 <- c(result4$p1[1], result3$p2[1], result3$p3[1], t(result4[1, 2:12]), result4$p13[1])
  for(j in 1:length(par5)){
    if(par5[j] <= 0){
      par5[j] <- 0.000001
    }
  }
  result5 <- optimx(par = par5, fn = compmodel5, method = "Nelder-Mead")
  
  #model 6
  par6 <- c(result5$p1[1], result5$p2[1], rep(result5$p3[1], 11), t(result5[1, 4:14]), result5$p15[1])
  for(j in 1:length(par6)){
    if(par6[j] <= 0){
      par6[j] <- 0.000001
    }
  }
  result6 <- optimx(par = par6, fn = compmodel6, method = "Nelder-Mead")
  
  M4_lambda[i] <- result4$p1
  M4_alpha[i,] <- t(result4[1, 2:9])
  M5_lambda[i] <- result5$p1
  M5_alpha[,i] <- t(result5[1, 4:11])
  M5_theta[i] <- result5$p2
  M5_omega[i] <- result5$p3
  M6_lambda[i] <- result6$p1
  M6_alpha[,i] <- t(result6[1, 14:21])
  M6_theta[i] <- result6$p2
  M6_omega[,i] <- t(result6[1, 3:10])
  loglikelihood <- c(loglikelihood, result1$value, result2$value, result3$value, result4$value, result5$value, result6$value)
  
} #end loop i

#format outputs
M4_alpha <- as.data.frame(t(M4_alpha))
M5_alpha <- as.data.frame(t(M5_alpha))
M6_alpha <- as.data.frame(t(M6_alpha))
M6_omega <- as.data.frame(t(M6_omega))
names(M4_lambda) <- splist
rownames(M4_alpha) <- splist
colnames(M4_alpha) <- splist
names(M5_lambda) <- splist
rownames(M5_alpha) <- splist
colnames(M5_alpha) <- splist
rownames(M6_alpha) <- splist
colnames(M6_alpha) <- splist
rownames(M6_omega) <- splist
colnames(M6_omega) <- splist

#model selection
AIC_Burnham_Anderson <- loglikelihood * 2 + parameters #method = Burnham & Anderson 2002
AIC <- data.frame(model, parameters, loglikelihood, AIC_Burnham_Anderson, species)

AIC <- subset(AIC, model > 3)
model_sel <- data.frame("species" = splist, "model" = rep(NA, length(splist)))
for (k in 1:length(splist)){
  subAIC <- subset(AIC, species == splist[k])
  minimum <- range(subAIC$AIC_Burnham_Anderson)[1]
  model_sel$model[k] <- subAIC$model[which(subAIC$AIC_Burnham_Anderson == minimum)]
}

#save
write.table(M4_lambda, file = "M4_lambda_control.txt", row.names = TRUE, sep = "\t")
write.table(M4_alpha, file = "M4_alpha_control.txt", row.names = TRUE, sep = "\t")
write.table(M5_lambda, file = "M5_lambda_control.txt", row.names = TRUE, sep = "\t")
write.table(M5_alpha, file = "M5_alpha_control.txt", row.names = TRUE, sep = "\t")
write.table(M5_theta, file = "M5_theta_control.txt", row.names = TRUE, sep = "\t")
write.table(M5_omega, file = "M5_omega_control.txt", row.names = TRUE, sep = "\t")
write.table(M6_lambda, file = "M6_lambda_control.txt", row.names = TRUE, sep = "\t")
write.table(M6_alpha, file = "M6_alpha_control.txt", row.names = TRUE, sep = "\t")
write.table(M6_theta, file = "M6_theta_control.txt", row.names = TRUE, sep = "\t")
write.table(M6_omega, file = "M6_omega_control.txt", row.names = TRUE, sep = "\t")
write.table(AIC, file = "AIC_control.txt", row.names = FALSE, sep = "\t")
write.table(model_sel, file = "model_sel_control.txt", row.names = FALSE, sep = "\t")


#### NITROGEN ####
dtreat <- subset(d, treatment == "nitrogen") #TREATMENT = NITROGEN

loglikelihood <- NULL

for(i in 1:length(splist)){
  data <- subset(dtreat, focal_sp == splist[i])
  #data <- subset(data, year == 2017)
  log_biomass <- log(data$biomass_f + 1) - log(data$biomass_i + 1)
  
  #model 1
  par1 <- c(mean(log_biomass), sd(log_biomass))
  for(j in 1:length(par1)){
    if(par1[j] <= 0){
      par1[j] <- 0.000001
    }
  }
  result1 <- optimx(par = par1, fn = compmodel1, method = "Nelder-Mead")
  
  #model 2
  par2 <- c(result1$p1[1], 0.000001, result1$p2[1])
  for(j in 1:length(par2)){
    if(par2[j] <= 0){
      par2[j] <- 0.000001
    }
  }
  result2 <- optimx(par = par2, fn = compmodel2, method = "Nelder-Mead")
  
  #model 3
  par3 <- c(result2$p1[1], 0.000001, result2$p2[1], 0.000001, result2$p3[1])
  for(j in 1:length(par3)){
    if(par3[j] <= 0){
      par3[j] <- 0.000001
    }
  }
  result3 <- optimx(par = par3, fn = compmodel3, method = "Nelder-Mead")
  
  #model 4
  par4 <- c(result3$p1[1], rep(result3$p4[1], 11), result3$p5[1])
  for(j in 1:length(par4)){
    if(par4[j] <= 0){
      par4[j] <- 0.000001
    }
  }
  result4 <- optimx(par = par4, fn = compmodel4, method = "Nelder-Mead")
  
  #model 5
  par5 <- c(result4$p1[1], result3$p2[1], result3$p3[1], t(result4[1, 2:12]), result4$p13[1])
  for(j in 1:length(par5)){
    if(par5[j] <= 0){
      par5[j] <- 0.000001
    }
  }
  result5 <- optimx(par = par5, fn = compmodel5, method = "Nelder-Mead")
  
  #model 6
  par6 <- c(result5$p1[1], result5$p2[1], rep(result5$p3[1], 11), t(result5[1, 4:14]), result5$p15[1])
  for(j in 1:length(par6)){
    if(par6[j] <= 0){
      par6[j] <- 0.000001
    }
  }
  result6 <- optimx(par = par6, fn = compmodel6, method = "Nelder-Mead")
  
  M4_lambda[i] <- result4$p1
  M4_alpha[i,] <- t(result4[1, 2:9])
  M5_lambda[i] <- result5$p1
  M5_alpha[,i] <- t(result5[1, 4:11])
  M5_theta[i] <- result5$p2
  M5_omega[i] <- result5$p3
  M6_lambda[i] <- result6$p1
  M6_alpha[,i] <- t(result6[1, 14:21])
  M6_theta[i] <- result6$p2
  M6_omega[,i] <- t(result6[1, 3:10])  
  loglikelihood <- c(loglikelihood, result1$value, result2$value, result3$value, result4$value, result5$value, result6$value)
  
} #end loop i

#format outputs
M4_alpha <- as.data.frame(t(M4_alpha))
M5_alpha <- as.data.frame(t(M5_alpha))
M6_alpha <- as.data.frame(t(M6_alpha))
M6_omega <- as.data.frame(t(M6_omega))
names(M4_lambda) <- splist
rownames(M4_alpha) <- splist
colnames(M4_alpha) <- splist
names(M5_lambda) <- splist
rownames(M5_alpha) <- splist
colnames(M5_alpha) <- splist
rownames(M6_alpha) <- splist
colnames(M6_alpha) <- splist
rownames(M6_omega) <- splist
colnames(M6_omega) <- splist

#model selection
AIC_Burnham_Anderson <- loglikelihood * 2 + parameters #method = Burnham & Anderson 2002
AIC <- data.frame(model, parameters, loglikelihood, AIC_Burnham_Anderson, species)

AIC <- subset(AIC, model > 3)
model_sel <- data.frame("species" = splist, "model" = rep(NA, length(splist)))
for (k in 1:length(splist)){
  subAIC <- subset(AIC, species == splist[k])
  minimum <- range(subAIC$AIC_Burnham_Anderson)[1]
  model_sel$model[k] <- subAIC$model[which(subAIC$AIC_Burnham_Anderson == minimum)]
}

#save
write.table(M4_lambda, file = "M4_lambda_nitrogen.txt", row.names = TRUE, sep = "\t")
write.table(M4_alpha, file = "M4_alpha_nitrogen.txt", row.names = TRUE, sep = "\t")
write.table(M5_lambda, file = "M5_lambda_nitrogen.txt", row.names = TRUE, sep = "\t")
write.table(M5_alpha, file = "M5_alpha_nitrogen.txt", row.names = TRUE, sep = "\t")
write.table(M5_theta, file = "M5_theta_nitrogen.txt", row.names = TRUE, sep = "\t")
write.table(M5_omega, file = "M5_omega_nitrogen.txt", row.names = TRUE, sep = "\t")
write.table(M6_lambda, file = "M6_lambda_nitrogen.txt", row.names = TRUE, sep = "\t")
write.table(M6_alpha, file = "M6_alpha_nitrogen.txt", row.names = TRUE, sep = "\t")
write.table(M6_theta, file = "M6_theta_nitrogen.txt", row.names = TRUE, sep = "\t")
write.table(M6_omega, file = "M6_omega_nitrogen.txt", row.names = TRUE, sep = "\t")
write.table(AIC, file = "AIC_nitrogen.txt", row.names = FALSE, sep = "\t")
write.table(model_sel, file = "model_sel_nitrogen.txt", row.names = FALSE, sep = "\t")


#### FUNGICIDE ####
dtreat <- subset(d, treatment == "fungicide") #TREATMENT = FUNGICIDE

loglikelihood <- NULL

for(i in 1:length(splist)){
  data <- subset(dtreat, focal_sp == splist[i])
  #data <- subset(data, year == 2017)
  log_biomass <- log(data$biomass_f + 1) - log(data$biomass_i + 1)
  
  #model 1
  par1 <- c(mean(log_biomass), sd(log_biomass))
  for(j in 1:length(par1)){
    if(par1[j] <= 0){
      par1[j] <- 0.000001
    }
  }
  result1 <- optimx(par = par1, fn = compmodel1, method = "Nelder-Mead")
  
  #model 2
  par2 <- c(result1$p1[1], 0.000001, result1$p2[1])
  for(j in 1:length(par2)){
    if(par2[j] <= 0){
      par2[j] <- 0.000001
    }
  }
  result2 <- optimx(par = par2, fn = compmodel2, method = "Nelder-Mead")
  
  #model 3
  par3 <- c(result2$p1[1], 0.000001, result2$p2[1], 0.000001, result2$p3[1])
  for(j in 1:length(par3)){
    if(par3[j] <= 0){
      par3[j] <- 0.000001
    }
  }
  result3 <- optimx(par = par3, fn = compmodel3, method = "Nelder-Mead")
  
  #model 4
  par4 <- c(result3$p1[1], rep(result3$p4[1], 11), result3$p5[1])
  for(j in 1:length(par4)){
    if(par4[j] <= 0){
      par4[j] <- 0.000001
    }
  }
  result4 <- optimx(par = par4, fn = compmodel4, method = "Nelder-Mead")
  
  #model 5
  par5 <- c(result4$p1[1], result3$p2[1], result3$p3[1], t(result4[1, 2:12]), result4$p13[1])
  for(j in 1:length(par5)){
    if(par5[j] <= 0){
      par5[j] <- 0.000001
    }
  }
  result5 <- optimx(par = par5, fn = compmodel5, method = "Nelder-Mead")
  
  #model 6
  par6 <- c(result5$p1[1], result5$p2[1], rep(result5$p3[1], 11), t(result5[1, 4:14]), result5$p15[1])
  for(j in 1:length(par6)){
    if(par6[j] <= 0){
      par6[j] <- 0.000001
    }
  }
  result6 <- optimx(par = par6, fn = compmodel6, method = "Nelder-Mead")
  
  M4_lambda[i] <- result4$p1
  M4_alpha[i,] <- t(result4[1, 2:9])
  M5_lambda[i] <- result5$p1
  M5_alpha[,i] <- t(result5[1, 4:11])
  M5_theta[i] <- result5$p2
  M5_omega[i] <- result5$p3
  M6_lambda[i] <- result6$p1
  M6_alpha[,i] <- t(result6[1, 14:21])
  M6_theta[i] <- result6$p2
  M6_omega[,i] <- t(result6[1, 3:10])  
  loglikelihood <- c(loglikelihood, result1$value, result2$value, result3$value, result4$value, result5$value, result6$value)
  
} #end loop i

#format outputs
M4_alpha <- as.data.frame(t(M4_alpha))
M5_alpha <- as.data.frame(t(M5_alpha))
M6_alpha <- as.data.frame(t(M6_alpha))
M6_omega <- as.data.frame(t(M6_omega))
names(M4_lambda) <- splist
rownames(M4_alpha) <- splist
colnames(M4_alpha) <- splist
names(M5_lambda) <- splist
rownames(M5_alpha) <- splist
colnames(M5_alpha) <- splist
rownames(M6_alpha) <- splist
colnames(M6_alpha) <- splist
rownames(M6_omega) <- splist
colnames(M6_omega) <- splist

#model selection
AIC_Burnham_Anderson <- loglikelihood * 2 + parameters #method = Burnham & Anderson 2002
AIC <- data.frame(model, parameters, loglikelihood, AIC_Burnham_Anderson, species)

AIC <- subset(AIC, model > 3)
model_sel <- data.frame("species" = splist, "model" = rep(NA, length(splist)))
for (k in 1:length(splist)){
  subAIC <- subset(AIC, species == splist[k])
  minimum <- range(subAIC$AIC_Burnham_Anderson)[1]
  model_sel$model[k] <- subAIC$model[which(subAIC$AIC_Burnham_Anderson == minimum)]
}

#save
write.table(M4_lambda, file = "M4_lambda_fungicide.txt", row.names = TRUE, sep = "\t")
write.table(M4_alpha, file = "M4_alpha_fungicide.txt", row.names = TRUE, sep = "\t")
write.table(M5_lambda, file = "M5_lambda_fungicide.txt", row.names = TRUE, sep = "\t")
write.table(M5_alpha, file = "M5_alpha_fungicide.txt", row.names = TRUE, sep = "\t")
write.table(M5_theta, file = "M5_theta_fungicide.txt", row.names = TRUE, sep = "\t")
write.table(M5_omega, file = "M5_omega_fungicide.txt", row.names = TRUE, sep = "\t")
write.table(M6_lambda, file = "M6_lambda_fungicide.txt", row.names = TRUE, sep = "\t")
write.table(M6_alpha, file = "M6_alpha_fungicide.txt", row.names = TRUE, sep = "\t")
write.table(M6_theta, file = "M6_theta_fungicide.txt", row.names = TRUE, sep = "\t")
write.table(M6_omega, file = "M6_omega_fungicide.txt", row.names = TRUE, sep = "\t")
write.table(AIC, file = "AIC_fungicide.txt", row.names = FALSE, sep = "\t")
write.table(model_sel, file = "model_sel_fungicide.txt", row.names = FALSE, sep = "\t")


#### COMBINED ####
dtreat <- subset(d, treatment == "combined") #TREATMENT = COMBINED

loglikelihood <- NULL

for(i in 1:length(splist)){
  data <- subset(dtreat, focal_sp == splist[i])
  #data <- subset(data, year == 2017)
  log_biomass <- log(data$biomass_f + 1) - log(data$biomass_i + 1)
  
  #model 1
  par1 <- c(mean(log_biomass), sd(log_biomass))
  for(j in 1:length(par1)){
    if(par1[j] <= 0){
      par1[j] <- 0.000001
    }
  }
  result1 <- optimx(par = par1, fn = compmodel1, method = "Nelder-Mead")
  
  #model 2
  par2 <- c(result1$p1[1], 0.000001, result1$p2[1])
  for(j in 1:length(par2)){
    if(par2[j] <= 0){
      par2[j] <- 0.000001
    }
  }
  result2 <- optimx(par = par2, fn = compmodel2, method = "Nelder-Mead")
  
  #model 3
  par3 <- c(result2$p1[1], 0.000001, result2$p2[1], 0.000001, result2$p3[1])
  for(j in 1:length(par3)){
    if(par3[j] <= 0){
      par3[j] <- 0.000001
    }
  }
  result3 <- optimx(par = par3, fn = compmodel3, method = "Nelder-Mead")
  
  #model 4
  par4 <- c(result3$p1[1], rep(result3$p4[1], 11), result3$p5[1])
  for(j in 1:length(par4)){
    if(par4[j] <= 0){
      par4[j] <- 0.000001
    }
  }
  result4 <- optimx(par = par4, fn = compmodel4, method = "Nelder-Mead")
  
  #model 5
  par5 <- c(result4$p1[1], result3$p2[1], result3$p3[1], t(result4[1, 2:12]), result4$p13[1])
  for(j in 1:length(par5)){
    if(par5[j] <= 0){
      par5[j] <- 0.000001
    }
  }
  result5 <- optimx(par = par5, fn = compmodel5, method = "Nelder-Mead")
  
  #model 6
  par6 <- c(result5$p1[1], result5$p2[1], rep(result5$p3[1], 11), t(result5[1, 4:14]), result5$p15[1])
  for(j in 1:length(par6)){
    if(par6[j] <= 0){
      par6[j] <- 0.000001
    }
  }
  result6 <- optimx(par = par6, fn = compmodel6, method = "Nelder-Mead")
  
  M4_lambda[i] <- result4$p1
  M4_alpha[i,] <- t(result4[1, 2:9])
  M5_lambda[i] <- result5$p1
  M5_alpha[,i] <- t(result5[1, 4:11])
  M5_theta[i] <- result5$p2
  M5_omega[i] <- result5$p3
  M6_lambda[i] <- result6$p1
  M6_alpha[,i] <- t(result6[1, 14:21])
  M6_theta[i] <- result6$p2
  M6_omega[,i] <- t(result6[1, 3:10])
  loglikelihood <- c(loglikelihood, result1$value, result2$value, result3$value, result4$value, result5$value, result6$value)
  
} #end loop i

#format outputs
M4_alpha <- as.data.frame(t(M4_alpha))
M5_alpha <- as.data.frame(t(M5_alpha))
M6_alpha <- as.data.frame(t(M6_alpha))
M6_omega <- as.data.frame(t(M6_omega))
names(M4_lambda) <- splist
rownames(M4_alpha) <- splist
colnames(M4_alpha) <- splist
names(M5_lambda) <- splist
rownames(M5_alpha) <- splist
colnames(M5_alpha) <- splist
rownames(M6_alpha) <- splist
colnames(M6_alpha) <- splist
rownames(M6_omega) <- splist
colnames(M6_omega) <- splist

#model selection
AIC_Burnham_Anderson <- loglikelihood * 2 + parameters #method = Burnham & Anderson 2002
AIC <- data.frame(model, parameters, loglikelihood, AIC_Burnham_Anderson, species)

AIC <- subset(AIC, model > 3)
model_sel <- data.frame("species" = splist, "model" = rep(NA, length(splist)))
for (k in 1:length(splist)){
  subAIC <- subset(AIC, species == splist[k])
  minimum <- range(subAIC$AIC_Burnham_Anderson)[1]
  model_sel$model[k] <- subAIC$model[which(subAIC$AIC_Burnham_Anderson == minimum)]
}

#save
write.table(M4_lambda, file = "M4_lambda_combined.txt", row.names = TRUE, sep = "\t")
write.table(M4_alpha, file = "M4_alpha_combined.txt", row.names = TRUE, sep = "\t")
write.table(M5_lambda, file = "M5_lambda_combined.txt", row.names = TRUE, sep = "\t")
write.table(M5_alpha, file = "M5_alpha_combined.txt", row.names = TRUE, sep = "\t")
write.table(M5_theta, file = "M5_theta_combined.txt", row.names = TRUE, sep = "\t")
write.table(M5_omega, file = "M5_omega_combined.txt", row.names = TRUE, sep = "\t")
write.table(M6_lambda, file = "M6_lambda_combined.txt", row.names = TRUE, sep = "\t")
write.table(M6_alpha, file = "M6_alpha_combined.txt", row.names = TRUE, sep = "\t")
write.table(M6_theta, file = "M6_theta_combined.txt", row.names = TRUE, sep = "\t")
write.table(M6_omega, file = "M6_omega_combined.txt", row.names = TRUE, sep = "\t")
write.table(AIC, file = "AIC_combined.txt", row.names = FALSE, sep = "\t")
write.table(model_sel, file = "model_sel_combined.txt", row.names = FALSE, sep = "\t")

#end of optimx.R