#!/usr/bin/python
"""
Solver wrapper for the pcYeast8 model
Pranas Grigaitis (p.grigaitis@vu.nl), 2020-2021
Systems Biology Lab, AIMMS, VU Amsterdam
"""
import os
import sys
import numpy as np
import pandas as pd
import cbmpy
import datetime
import subprocess
import time
import dynamicRunner as dr

def applyGrowthMedium(m, medium):
	"""
	Input: model instance, string with medium type (corresponds to the column name in the media definition spreadsheet)
	Output: update model instance
	Function to constrain the exchange reactions according to the composition of the growth medium
	"""
	if medium.find('Anaerobic') >= 0: # Reverse the ATPase
	    for i in m.getReactionIds('r_0226'):
                rxnStoich = model.getReaction(i).getStoichiometry()
                for j in rxnStoich: 
                    if j[1] == 's_0799': coef = -2.0
                    elif j[1] == 's_0794': coef = +3.0
                    else: coef = -1.0 * j[0]
                    model.getReaction(i).setStoichCoefficient(j[1], coef) 
        
	rxns = [x for x in m.getReactionIds('_rev') if m.getReaction(x).name.find('exchange') >= 0]
	for i in rxns: m.setReactionBounds(i, 0, 0)

	allMedia = pd.read_excel('pcYeast8_dynamicInfo.xls', sheet_name = 'GrowthMedia')
	for i in allMedia.index: 
	    m.setReactionBounds(allMedia['rid'][i], 0.0, allMedia[medium][i])
	    if allMedia['rid'][i] != 'r_1761_rev': m.setReactionBounds(allMedia['rid'][i].replace('rev', 'fwd'), 0.0, 0.0)
	return m

# Parsing arguments for the run
parameter = sys.argv[1]
initial_value = float(sys.argv[2])
step = float(sys.argv[3])
param_val = list(range(0, int(sys.argv[4])))
directory = sys.argv[5]

# Loading the SBML model
model = cbmpy.CBRead.loadModel('pcYeast8_static_v0.22.xml')
model.createReactionReagent('r_4041', 's_1475', -1e-4) # Add biotin to the biomass

# Creating the folder to put the results to
if not os.path.exists(directory): 
	os.mkdir(directory)
	os.mkdir('%s/results' %(directory))

growthMedium = 'SD_glc'
model = applyGrowthMedium(model, growthMedium)

# Setting the presence of an alternative oxidase
# Options: 'None', 'AOX1', 'NOX1'
altOxidase = 'None'

# Need to leave the proton exchange open both sides
for i in model.getReactionIds('r_1832'): model.setReactionBounds(i, 0, 10) 

# Blocking instances of undesired metabolism
# These can be futile cycles, unrealistic "cheaper" pathways etc.
for i in ['r_4658_fwd', 'r_2083_fwd', 'r_1793_fwd', 'r_2024_fwd', 'r_4530_fwd', 'r_2061_fwd', 'r_2001_fwd', 'r_1818_fwd', 'r_4659_fwd', 'r_4549_fwd', 'r_4525_fwd', 'r_4520_fwd', 'r_4519_fwd', 'r_1589_fwd', 'r_1841_fwd', 'r_4552_fwd', 'r_4531_fwd', 'r_4519_fwd', 'r_1867_fwd', 'r_1870_fwd', 'r_1862_fwd', 'r_1866_fwd', 'r_1598_fwd', 'r_1865_fwd', 'r_1581_fwd', 'r_1577_fwd', 'r_1550_fwd', 'r_1765_fwd', 'r_1580_fwd', 'r_1572_fwd', 'r_1547_fwd', 'r_1631_fwd', 'r_1873_fwd'] + model.getReactionIds('r_4332') + model.getReactionIds('r_1244__YNR013C') + model.getReactionIds('r_0718') + model.getReactionIds('r_4538') + model.getReactionIds('r_1652') + model.getReactionIds('r_1172') + model.getReactionIds('r_2129') + model.getReactionIds('r_1824') + model.getReactionIds('r_4161') + model.getReactionIds('r_4162') + model.getReactionIds('r_4163') + model.getReactionIds('YDR352W') + model.getReactionIds('YKL069W') + model.getReactionIds('r_1729__YDL166C_fwd') + model.getReactionIds('r_2116') + model.getReactionIds('r_2032') + model.getReactionIds('r_4527') + model.getReactionIds('r_0001') + model.getReactionIds('r_0547') + model.getReactionIds('r_1829_rev') + model.getReactionIds('r_1137') + model.getReactionIds('r_0487') + model.getReactionIds('r_0470') + model.getReactionIds('r_0731') + model.getReactionIds('r_0659') + model.getReactionIds('r_1668') + model.getReactionIds('r_1138') + model.getReactionIds('r_4262') + model.getReactionIds('r_4264') + [x for x in model.getReactionIds('r_0503') if x.find('rev') >= 0] + model.getReactionIds('r_4292') + model.getReactionIds('r_1549') + model.getReactionIds('r_0472') + model.getReactionIds('r_4331') + model.getReactionIds('r_0303') + model.getReactionIds('r_2305'): model.setReactionBounds(i, 0, 0)

for i in list(range(0, len(param_val))): param_val[i]= initial_value + (param_val[i] * step)
tolerance = 1e-4
mu_min = 0.00

print('Computation began at %s.' %(datetime.datetime.now()))

"""
Binary search algorithm.
In this, we loop through parameter values passed in the arguments when calling this script, 
and determine the highest growth rate which still results in a feasible linear program. 
First we formulate the LP for the specific growth rate (starting at the middle of the defined range (from 'mu_min' to 'mu_max'),
then call the SoPlex solver to determine whether it is feasible. Based on the result, the binary search window is shrunk and 
the process continues. Optimization ends when the feasible solution falls within the tolerance range below the 'mu_max'.
"""
for i in param_val:
	mu_max = 0.60
	mu_previous_feasible = 0.0
	mu_test = 0.0
	optimized = False
	iteration = 1
	while not optimized:
		mu_test = (mu_min + mu_max)/2.0
		problem_name = dr.dynamicModel(model, directory, mu_test, parameter, i, altOxidase)# function from the dynamic model generator
		if mu_test > 0.05: solver = subprocess.Popen('$HOME/software/soplex/bin/soplex -x -q -t500 -c -o1e-16 -f1e-16 %s > %s.out'%(problem_name, problem_name), shell = True)
		else: solver = subprocess.Popen('$HOME/software/soplex/bin/soplex -x -q -t500 -c -o1e-12 -f1e-12 %s > %s.out'%(problem_name, problem_name), shell = True)
		solver.wait()
		feasible = False
		no_attempts = 1
		while not feasible:
			feasible_status = subprocess.Popen("grep '\[optimal\]' '%s.out'" %(problem_name), shell = True, stdout = subprocess.PIPE)
			if len(feasible_status.communicate()[0]) > 0:
				feasible = True
				mu_previous_feasible = mu_test
				break
			elif os.popen("grep '\[optimal with unscaled violations\]\|error \[unspecified\]' '%s.out' | wc -l" %(problem_name)).read() in ['1\n','2\n']:
				subprocess.call('$HOME/software/soplex/bin/soplex -x -q -t250 -g5 -p5 -f1e-16 -o1e-16 %s > %s.out'%(problem_name, problem_name), shell = True)
				no_attempts = no_attempts + 1
				if no_attempts >= 3: break
			else: break
		if feasible:
			print('Model feasible at mu = %.6f' %(mu_test))
			mu_min = mu_test
			last_feasible = problem_name
			if iteration > 29: optimized = True
		else: 
			mu_max = mu_test
			print('Model infeasible at mu = %.6f' %(mu_test))
			if abs(mu_max - mu_min) <= tolerance: optimized = True
		print('Iteration %d: ended at %s' %(iteration, datetime.datetime.now()))
		iteration = iteration + 1
	print('Optimization for %s = %.3f ended: mu* = %.6f' %(parameter, i, mu_previous_feasible))
	os.system('cp %s.out %s/results' %(last_feasible, directory))
