import os
import sys
import numpy as np
import pandas as pd
import cbmpy
import math
import re

parameters = {
	'defaultKcat' : 71.0 * 3600.0, # median kcat for proteins in central carbon metabol; Nilsson 2016
	'gDW' : 1.7, # the volume which is taken up by 1 gDW of cells; Canelas 2011
	'kdeg' : 0.043, # kdeg of proteins in S. cerevisiae CEN.PK113-7D
	'ribosomeVoidFraction' : 0.053, # fraction of ribosomes which have to be produced at zero-growth; Metzl-Raz 2017
	'upMitochondrial': 0.28 # fraction of the unspecified protein (UP) in total mitochondrial protein
	}

# Loading datasets from the dynamicInfo spreadsheet
macroMolComplexes = pd.read_excel('pcYeast8_dynamicInfo.xls', sheet_name = 'macromolecularComplexes')
proteinData = pd.read_excel('pcYeast8_dynamicInfo.xls', sheet_name = 'proteinProperties')
pMemProteins = pd.read_excel('pcYeast8_dynamicInfo.xls', sheet_name = 'plasmaMembrane')
pCProteins = pd.read_excel('pcYeast8_dynamicInfo.xls', sheet_name = 'plasmaCarbon')
pNProteins = pd.read_excel('pcYeast8_dynamicInfo.xls', sheet_name = 'plasmaNitrogen')
pPProteins = pd.read_excel('pcYeast8_dynamicInfo.xls', sheet_name = 'plasmaPhosphorus')
pSProteins = pd.read_excel('pcYeast8_dynamicInfo.xls', sheet_name = 'plasmaSulfur')
kcatInfo = pd.read_excel('pcYeast8_dynamicInfo.xls', sheet_name = 'kcatReviewed')
cplxInfo = pd.read_excel('pcYeast8_dynamicInfo.xls', sheet_name = 'rxnGPRs')

def dynamicModel(m, directory, mu, parameter, par_val, aO):
	# Initial values of the titratable parameters
	riboa_f = 1.0
	sat_f = 1.0
	sat_f_n = 1.0
	sat_f_p = 1.0
	sat_f_s = 1.0
	upMinimum = 0.22
	mitoVol = 1.25
	pcArea = 0.13
	pnArea = 0.05
	mCherryFraction = 0.0
	gamValue = 24.0
	exec(parameter + '= par_val')

	if aO != 'None': m.setReactionBounds(m.getReactionIds('%s_complex_formation' %(aO))[0], 0.0, 1e-5)
	
	# Growth rate-dependent changes to the biomass equation
	m = biomassCorrection(m, mu, gamValue)
	cbmpy.CBCPLEX.cplx_analyzeModel(m)
	
	# Exporting the mu-specific model to the LP format
	problemName = '%s/%s_%.4f_mu_%.7f.lp' %(directory, parameter, par_val, mu) 
	cbmpy.CBCPLEX.cplx_constructLPfromFBA(m, fname = problemName.replace('.lp',''))
	
	with open (problemName, 'r') as f:
		model_lp = f.readlines()
		f.close()
	
	model_lp = ''.join(model_lp)
	model_lp = model_lp.split('\n')
	for i in list(range(0, len(model_lp))):
		if model_lp[i] == "Bounds":
			model_lp_metabolic = model_lp[0:i]
			model_lp_bounds = model_lp[i:]
	model_udc = []
	
	"""
	Adding metabolic coupling. 
	For this we use the GPRs of the intitial model to set up the 3 constraints for each metabolic enzyme complex:
	1. Complex usage (v = e * kcat) and the partitioning between 
	2. complex degradation (kdeg/(mu+kdeg)) and 
	3. complex dilution by growth (mu/(mu+kdeg)) of the 'spent' enzyme.
	"""
	# Determining the varied transporters
	if parameter == 'sat_f':
		proteinsSubject = pCProteins['Gene Systematic Name'].str.replace('-', '')
		paramVal = sat_f
	elif parameter == 'sat_f_n':
		proteinsSubject = pNProteins['Gene Systematic Name'].str.replace('-', '')
		paramVal = sat_f_n
	elif parameter == 'sat_f_p':
		proteinsSubject = pPProteins['Gene Systematic Name'].str.replace('-', '')
		paramVal = sat_f_p
	elif parameter == 'sat_f_s':
		proteinsSubject = pSProteins['Gene Systematic Name'].str.replace('-', '')
		paramVal = sat_f_s
	else:
		proteinsSubject = []
		paramVal = 1.0
	
	# Calling the metabolic coupling function
	model_udc = model_udc + addManualMetabolicCoupling(m, mu, parameters['kdeg'], parameters['defaultKcat'], paramVal, proteinsSubject)
	
	"""
	Macromolecular coupling factors. We do the coupling in three stages:
	(1) for complexes with exact stoichiometry
	(2) ribosomes and elongation factors (i.e. demand depends on the protein chain)
	(3) initiation factors (i.e. demand depends on the length of the 5'-UTR)
	"""
	# (1) Complexes with exact stoichiometry
	for i in macroMolComplexes.index:
		if macroMolComplexes['stoichiometry'][i] == 0: continue 
		searchStrings = macroMolComplexes['searchString'][i].split(';')
		catRxns = []
		for j in searchStrings: catRxns = catRxns + [[macroMolComplexes['stoichiometry'][i], x] for x in [y for y in m.getReactionIds() if re.search(j, y)]]
		if len([x for x in searchStrings if x.find('degradation') >= 0]) > 0:
			catRxns = [x for x in catRxns if x[1].find('complex') < 0]
		rxnsCplx = 'mmc_complex_formation_cprmt + mmc_complex_degradation_cprmt + mmc_complex_dilution_cprmt'.replace('mmc', macroMolComplexes['cplxName'][i]).replace('cprmt', macroMolComplexes['compartment'][i])
		model_udc = model_udc + addMacromolecularComplexBalance(catRxns, rxnsCplx.split(' + '), macroMolComplexes['kcat h-1'][i], mu, parameters['kdeg'], parameters['ribosomeVoidFraction'], riboa_f)
	
	# (2) Ribosomes and elongation factors
	cplxListRibo = ['Ribosome','_eEF1A', '_eEF1B', '_eEF1G', '_eEF2', '_eEF3A', '_eEF3B', 'MitoRibosome', 'mEF']
	for i in macroMolComplexes.loc[macroMolComplexes['cplxName'].isin(cplxListRibo)].index:
		searchStrings = macroMolComplexes['searchString'][i].split(';')
		catRxns = []
		for j in searchStrings: catRxns = catRxns + [[proteinData['Length'].loc[proteinData['Entry'] == x.split('_')[0]], x] for x in [y for y in m.getReactionIds() if re.search(j, y)]]
		rxnsCplx = 'mmc_complex_formation_cprmt + mmc_complex_degradation_cprmt + mmc_complex_dilution_cprmt'.replace('mmc', macroMolComplexes['cplxName'][i]).replace('cprmt', macroMolComplexes['compartment'][i])
		model_udc = model_udc + addMacromolecularComplexBalance(catRxns, rxnsCplx.split(' + '), macroMolComplexes['kcat h-1'][i], mu, parameters['kdeg'], parameters['ribosomeVoidFraction'], riboa_f)
	
	# (3) Initiation factors
	cplxListInit = ['_eIF1', '_eIF1A', '_eIF2', '_eIF2A', '_eIF2B', '_eIF3', '_eIF4A', '_eIF4B','_eIF4E', '_eIF4F', '_eIF5', '_eIF5A', '_eIF5B', '_eIF6', 'mIF']
	for i in macroMolComplexes.loc[macroMolComplexes['cplxName'].isin(cplxListInit)].index:
		searchStrings = macroMolComplexes['searchString'][i].split(';')
		catRxns = []
		for j in searchStrings: catRxns = catRxns + [[proteinData['5UTRlength'].loc[proteinData['Entry'] == x.split('_')[0]], x] for x in [y for y in m.getReactionIds() if re.search(j, y)]]
		rxnsCplx = 'mmc_complex_formation_cprmt + mmc_complex_degradation_cprmt + mmc_complex_dilution_cprmt'.replace('mmc', macroMolComplexes['cplxName'][i]).replace('cprmt', macroMolComplexes['compartment'][i])
		model_udc = model_udc + addMacromolecularComplexBalance(catRxns, rxnsCplx.split(' + '), macroMolComplexes['kcat h-1'][i], mu, parameters['kdeg'], parameters['ribosomeVoidFraction'], riboa_f)
	
	
	# Compartment-specific proteome constraints
	"""	
	The implementation is borrowed from the pcYeast:
	peptideArea * 1e-6 (for conversion nm2->um2)*6.022e20 (for particles in mmol) * cellMass
	cellMass is taken from the relation that 1.7 mL equals 1 gDW of cells. Thus:
	cellVol = cellSize(mu)
	cellCount = (cellVol/1e18)/(parameters['gDW']/1e6)
	the final dim: 
	coef [um^2 * gDW / cell mmol] * flux [mmol/gDW/h] * 1/(mu+kdeg) [/h] -> um^2/cell
	"""
	# Plasma membrane constraints
	modelRxns = m.getReactionIds('translation')
	cytosolicComplexes = m.getReactionIds('complex_formation_c')
	for i in [['PlasmaMembrane', pMemProteins['Gene Systematic Name'], 0.20], ['PlasmaMembraneCarbon', pCProteins['Gene Systematic Name'], pcArea], ['PlasmaMembraneNitrogen', pNProteins['Gene Systematic Name'], pnArea], ['PlasmaMembranePhosphorus', pPProteins['Gene Systematic Name'], 0.005], ['PlasmaMembraneSulfur', pSProteins['Gene Systematic Name'], 0.0015]]:
		areaIncrements = []
		allComplexes = []
		for j in i[1].unique(): allComplexes = allComplexes + [x for x in cytosolicComplexes if x.find(j.replace('-', '') + '_') >= 0]
		allComplexes = pd.Series(allComplexes).unique()
		for k in allComplexes:
			complexMemberUniProts = [x.replace('_c', '') for x in m.getReaction(k).getSubstrateIds()]
			massOfComplex = sum(proteinData['Mass'].loc[proteinData['Entry'].isin(complexMemberUniProts)])
			areaIncrements.append([(6.022e2*cellSize(mu)/parameters['gDW'])*peptideArea(massOfComplex)/(mu+parameters['kdeg']), k])
		model_udc.append(compartmentConstraint(i[0], areaIncrements, "<=", i[2] * cellArea(mu)))
	
	# Mitochondrial volume
	mitoTransportRxns = m.getReactionIds('_import_c_m') + m.getReactionIds('translation_MitoRibosome')
	volumeIncrements = []
	volumeIncrements = [[(6.022e-1*cellSize(mu)/parameters['gDW'])*peptideVolume(proteinData['Mass'].loc[proteinData['Entry'] == x.split('_')[0]])/(mu+parameters['kdeg']), x] for x in mitoTransportRxns]
	model_udc.append(compartmentConstraint('MitoVolume', volumeIncrements, "<=", mitoVol))
	
	# Proteome constraints
	# Total proteome mass
	massIncrements = [[1e-3*x[0]/(mu+parameters['kdeg']), [y for y in modelRxns if y.find(x[1]) >= 0]] for x in zip(proteinData['Mass'], proteinData['Entry'])]
	massIncrements = [[x[0],x[1][0]] for x in massIncrements if x[1] != []]
	model_udc.append(compartmentConstraint('TotalProteome', massIncrements, "=", proteinRatio(mu)))
	       
	# Mitochondrial maintenance rxn
	mitoMassIncrements = [[1e-3*proteinData['Mass'].loc[proteinData['Entry'] == x.split('_')[0]]/(mu+parameters['kdeg']), x] for x in mitoTransportRxns]
	mitoMassIncrements = ['%.15f %s' %(x[0],x[1]) for x in mitoMassIncrements]
	mitoMassIncrements = ' + '.join(mitoMassIncrements)
	mitoMassIncrements = 'MitochondrialMaintenance: %s - %.15f MitoMaintenance = 0.0' %(mitoMassIncrements, 1.0)
	model_udc.append(mitoMassIncrements)
	
	# The unspecified protein
	UPdefinition = [proteinData['Mass'].loc[proteinData['Entry'] == 'PDUMMY']/(mu+parameters['kdeg']), 'PDUMMY_folding']
	rxnsCplx = ['UP_complex_formation_c','UP_complex_degradation_c', 'UP_complex_dilution_c']
	model_udc = model_udc + addUPBalance(m, UPdefinition, rxnsCplx, upMinimum, mu, parameters['kdeg'], proteinData)
	
	# Ditto, for mitochondria
	UPdefinition[1] = 'PDUMMY_folding_m'
	mitoProteinIDs = [x.split('_')[0] for x in mitoTransportRxns]
	model_udc = model_udc + addUPBalance(m, UPdefinition, [x.replace('_c', '_m') for x in rxnsCplx], parameters['upMitochondrial'], mu, parameters['kdeg'], proteinData)
	
	# mCherry protein overexpression
	model_udc = model_udc + addGratuitousProteinBalance(m, 'PMCHERRY', mCherryFraction, mu, parameters['kdeg'], proteinData)

	# Writing model to file. For future: find out if there are less verbose ways to do the splitting of 8000+ char lines (SoPlex doesn't like them).
	for i in list(range(0, len(model_udc))): 
		if len(model_udc[i]) > 8000: 
			splits = (len(model_udc[i]) // 8000) + 1
			split_string = model_udc[i].split(' + ')
			back_string = ''
			k = 0
			for j in list(range(0, splits)):
				pos = math.trunc(len(model_udc[i])*(j+1)/splits)
				while (len(back_string) < pos):
					if (k == (len(split_string) - 1)): back_string = back_string + split_string[k]
					else: back_string = back_string + split_string[k] + ' + '
					k = k + 1
				if (k == len(split_string)): break
				back_string = back_string + '\n'
			model_udc[i] = back_string

			if len(model_udc[i]) > 8000: 
				splits = (len(model_udc[i]) // 8000) + 1
				split_string = model_udc[i].split(' - ')
				back_string = ''
				k = 0
				for j in list(range(0, splits)):
					pos = math.trunc(len(model_udc[i])*(j+1)/splits)
					while (len(back_string) < pos):
						if (k == (len(split_string) - 1)): back_string = back_string + split_string[k]
						else: back_string = back_string + split_string[k] + ' - '
						k = k + 1
					if (k == len(split_string)): break
					back_string = back_string + '\n'
				model_udc[i] = back_string
	
	with open (problemName, 'w') as f:
		model_description = ('\n'.join(model_lp_metabolic) + '\n' + '\n'.join(model_udc) + '\n' + '\n'.join(model_lp_bounds))
		f.write(model_description)
		f.close()
	
	return problemName

def addGratuitousProteinBalance(m, gProtein, gFraction, mu, kdeg, pData):
	"""
	Input: 
	m: model instance
	gProtein: ID of the gratuitous protein  
	gFraction: proteome fraction of the gratuitous protein (gram/gram protein), float
	mu: growth rate
	kdeg: median protein kdeg
	pData: protein definition spreadsheet
	Output: a list of 3 constraint expressions setting the minimal demand of the UP and defining the balance between its degradation and dilution.
	"""
	constraintExpressions = []
	
	coefficient = 1e-3 * pData['Mass'].loc[pData['Entry'] == gProtein]/(mu+kdeg)
	cRxns = [x for x in m.getReactionIds(gProtein) if x.find('complex') >= 0]
	
	constraintExpressions.append('%s_balance: %.15f %s %s %.15f' %(gProtein, coefficient, [x for x in cRxns if x.find('form') > 0][0], "=", gFraction * proteinRatio(mu)))
	constraintExpressions.append('%s_deg: %s - %.15f %s %s %.1f' %(gProtein,[x for x in cRxns if x.find('deg') > 0][0], (kdeg/(kdeg+mu)), [x for x in cRxns if x.find('form') > 0][0], "=", 0.0))
	constraintExpressions.append('%s_dil: %s - %.15f %s %s %.1f' %(gProtein, [x for x in cRxns if x.find('dil') > 0][0], (mu/(kdeg+mu)), [x for x in cRxns if x.find('form') > 0][0], "=", 0.0))
	
	return constraintExpressions

def addMacromolecularComplexBalance(catRxns, mainRxns, kcat, mu, kdeg, RVF, RAF):
	"""
	Input: 
	catRxns: nested lists of [coefficient, rxn]
	mainRxns: list of reaction names for complex formation, degradation, dilution
	kcat: kcat of the macromolecular complex
	mu: growth rate
	kdeg: median protein kdeg
	RVF: void fraction of ribosomes (amount of ribosomes [g/g protein] which have to be produced at zero-growth)
	RAF: the active fraction of ribosomes (default 1)
	Output: a list of constraint expressions for the macromolecular complex (balance, demand for catalysis; degradation/dilution ratio)
	"""
	constraintExpressions = []
	
	catRxns = ['%.5f %s' %(x[0], x[1]) for x in catRxns]
	usageConstraint = ' + '.join(catRxns)
	
	cplxName = mainRxns[0].split('_')[0]
	if len(cplxName) == 0: cplxName = mainRxns[0].split('_')[1]
	if cplxName == 'Ribosome': 
		kcat = kcat * RAF
		rhs = -1e3 * RVF * proteinRatio(mu) * kcat/1388488.0
	else: rhs = 0.0
	
	constraintExpressions.append('%s_balance: %s - %.15f %s %s %.15f' %(cplxName, usageConstraint, (kcat/(kdeg+mu)), mainRxns[0], "=", rhs))
	constraintExpressions.append('%s_deg: %s - %.15f %s %s %.1f' %(cplxName, mainRxns[1], (kdeg/(kdeg+mu)), mainRxns[0], "=", 0.0))
	constraintExpressions.append('%s_dil: %s - %.15f %s %s %.1f' %(cplxName, mainRxns[2], (mu/(kdeg+mu)), mainRxns[0], "=", 0.0))
	
	return constraintExpressions

def addManualMetabolicCoupling(m, mu, kdeg, defKcat, parameterVal, proteinsSubject):
	"""
	Input: 
	m: model instance
	mu: growth rate
	kdeg: median protein kdeg
	defKcat: default kcat value for complexes w/o known kcat
	parameterVal: value of titrated parameter, such as hexose transporter saturation
	proteinsSubject: spreadsheet with proteins which are subject to the multiplication of their kcat by parameterVal
	Output: a list of constraint expressions for each metabolic complex (balance, demand for catalysis; degradation/dilution ratio).
	
	The function to determine the metabolic coupling in the model. Uses kcats, collected 
	from BRENDA to establish the constraints, but every kcat < 1 1/s gets set to the
	base-line value of 1 1/s. Some of the enzymes with terribly low kcats do not permit 
	actual computation of the model (the demand is way too high to accomodate).
	"""
	constraintExpressions = []
	metabolicComplexes = [x for x in m.getReactionIds('complex_formation') if (x[0] in ['Y','Q', 'A', 'N'])]
	for cplx in metabolicComplexes:
		complexName = cplx.split('_complex')[0]
		complexComp = cplx.split('_')[-1]
		kcatToApply = float(kcatInfo['kcat(h-1)'].loc[(kcatInfo['gpr'] == complexName) & (kcatInfo['CompCode'] == complexComp)])
		if np.isnan(kcatToApply): kcatToApply = defKcat
		if kcatToApply < 3600: kcatToApply = 3600
		if len(proteinsSubject) > 0:
			if sum(pd.Series(complexName.split('__')).isin(proteinsSubject)) > 0: kcatToApply = parameterVal * kcatToApply
		metabolicReactions = cplxInfo['Reaction'].loc[(cplxInfo['GPR'] == complexName.replace('__', ';')) & (cplxInfo['Compartment'] == complexComp)]
		constraintExpressions.append('%s_%s_balance: %s - %.15f %s = %.1f' %(complexName, complexComp, ' + '.join(metabolicReactions), kcatToApply/(kdeg+mu), cplx, 0.0))
		constraintExpressions.append('%s_%s_deg: %s - %.15f %s = %.1f' %(complexName, complexComp, cplx.replace('formation', 'degradation'), 1.0 * kdeg/(kdeg+mu), cplx, 0.0))
		constraintExpressions.append('%s_%s_dil: %s - %.15f %s = %.1f' %(complexName, complexComp, cplx.replace('formation', 'dilution'), 1.0 * mu/(kdeg+mu), cplx, 0.0))
	
	return constraintExpressions

def addUPBalance(m, upDef, mainRxns, upMin, mu, kdeg, pData):
	"""
	Input: 
	m: model instance
	upDef: list of UP turnover reaction names (complex formation, degradation, dilution) 
	upMin: the value of UP minimum (gram UP/gram protein), float
	mu: growth rate
	kdeg: median protein kdeg
	pData: protein definition spreadsheet
	Output: a list of 3 constraint expressions setting the minimal demand of the UP and defining the balance between its degradation and dilution.
	
	Function for implementing the constraint of minimal UP amount, expression being 
	[weighted sum of all proteins] <= 1/upMin [weighted UP]
	Since UP appears on both sides of the constraint expression, the rearranged expression is
	[weighted sum of all proteins but UP] - (1 - 1/upMin) [weighted UP] <= 0.
	"""
	constraintExpressions = []
	
	if upDef[1].find('_m') > 0: translationRxns = [x for x in m.getReactionIds('_folding_m') if x.find('PDUMMY') < 0]
	elif upDef[1].find('MCHERRY') >= 0: translationRxns = m.getReactionIds('_translation')
	else: translationRxns = [x for x in m.getReactionIds('_folding') if (x.find('PDUMMY') < 0)] + ['PDUMMY_folding_m']
	
	translationRxns = ['%.15f %s' %(pData['Mass'].loc[pData['Entry'] == x.split('_')[0]]/(mu+kdeg), x) for x in translationRxns]
	usageConstraint = ' + '.join(translationRxns)
	
	cplxName = ('%s_%s' %(mainRxns[0].split('_')[0], mainRxns[0].split('_')[-1]))
	
	constraintExpressions.append('%s_balance: %s %.15f %s %s %.1f' %(cplxName, usageConstraint, (1.0 - 1.0/upMin)*upDef[0], upDef[1], "<=", 0.0))
	constraintExpressions.append('%s_deg: %s - %.15f %s %s %.1f' %(cplxName, mainRxns[1], (kdeg/(kdeg+mu)), mainRxns[0], "=", 0.0))
	constraintExpressions.append('%s_dil: %s - %.15f %s %s %.1f' %(cplxName, mainRxns[2], (mu/(kdeg+mu)), mainRxns[0], "=", 0.0))
	
	return constraintExpressions

def biomassCorrection(m, mu, g):
	"""
	Input: 
	m: model instance
	mu: growth rate, float
	g: GAM value [mmol/gDW], float
	Output: updated model instance
	
	Function for modifying the growth rate-dependent stoichiometric coefficients of the biomass equation.
	""""""
	Literature sources:
	Lipids: Canelas et al. 2011 (bulk), Ejsing et al. 2009 (ratios)
	
	Carbohydrates: Canelas et al. 2011 (ratios)
	
	RNA: Canelas et al. 2011 (bulk), Tada & Tada 1962 (molar ratio)
	RNA mass fractions (recomputed from Tada & Tada, 1962):
	AMP: 21.626% | GMP: 32.012% | CMP: 26.170% | UMP: 20.192%
	"""
	# Setting the biomass flux to the probed growth rate
	m.setReactionBounds('r_4041', mu, mu)

	# Modifying ATP maintenance
	atpCost = {'ngam':0.168, 'gam': g}
	maintenanceCoef = (atpCost['ngam']/mu) + (atpCost['gam'])
	maintenanceCoefMito = 2.2e-2/mu

	for i in [['s_0434', -1.0], ['s_0803', -1.0], ['s_0394', +1.0], ['s_1322', +1.0], ['s_0794', +1.0]]: m.getReaction('r_4041').setStoichCoefficient(i[0], i[1] * maintenanceCoef)
	
	# Correcting the phosphate coefficient for the biomass demand
	m.getReaction('r_4041').setStoichCoefficient('s_1322', m.getReaction('r_4041').getReagentWithSpeciesRef('s_1322').getCoefficient() - 0.006)
	
	# Lipid content
	mLipid = -0.045576*mu + 0.074549
	m.getReaction('r_4041').setStoichCoefficient('s_1096', -1.0 * mLipid)
	
	# Biomass carbohydrates
	m.getReaction('r_4048').setStoichCoefficient('s_1520', -1.0 * (12.61369*np.exp(-13.10818*mu)/100.0)*1e3/342.3) # Tre
	m.getReaction('r_4048').setStoichCoefficient('s_0773', -1.0 * (20.44753*np.exp(-9.593868*mu)/100.0)*1e3/162.0) # Glycogen
	m.getReaction('r_4048').setStoichCoefficient('s_0001', -1.0 * ((-13.7727 * mu + 18.8208)/100.0/2.0)*1e3/162.0) # Glucan 1-3
	m.getReaction('r_4048').setStoichCoefficient('s_0004', -1.0 * ((-13.7727 * mu + 18.8208)/100.0/2.0)*1e3/162.0) # Glucan 1-6
	m.getReaction('r_4048').setStoichCoefficient('s_1107', -1.0 * 0.09975*1e3/162.0) #Mannans
	
	# RNA content
	mRNA = 0.15249*mu + 0.049841
	for i in [[0.21626, 347.22, 's_0423'], [0.32011, 363.22, 's_0782'], 
		[0.261695, 323.1965, 's_0526'], [0.2019, 324.1813, 's_1545']]:
		 m.getReaction('r_4049').setStoichCoefficient(i[2], -1.0 * mRNA * i[0] * 1e3/ i[1])
	
	return m

def cellArea(mu):
	"""
	Input: 
	mu: growth rate, float
	Output: assuming a spheric cell, its surface area (based on cell volume), float
	"""
	cellVol = cellSize(mu)
	r = pow((3.0/4.0)*(1.0/np.pi)*cellVol, 1.0/3.0)
	return 4.0 * np.pi * pow(r, 2.0)

def cellSize(mu):
	"""
	Input: 
	mu: growth rate, float
	Output: cell size, based on the measurements of Tyson 1975, float
	"""
	cs = 29.3716 * mu + 20.55066
	return cs

def compartmentConstraint(compName, proteins, condition, rhs):
	"""
	Input: 
	proteins: nested lists of [coefficient, proteinRxn]
	condition: the evaluation sign ('=', '<=', '<', '>', '=>')
	rhs: the right-hand side
	Output: the expression of the compartment-specific protein constraint, string
	"""
	proteins = ['%.15f %s' %(x[0], x[1]) for x in proteins]
	compartmentConstraint = ' + '.join(proteins)
	
	return "%s_compartment: %s %s %.15f" %(compName, compartmentConstraint, condition, rhs)

def peptideArea(MW):
	"""
	Input: 
	MW: molecular weight of the protein, float
	Output: assuming a globular protein, the cross-section area of 1 molecule in nm^2, float
	"""
	r_min=0.066*pow(MW, 1.0/3.0)
	vol=np.pi*pow(r_min,2.0)
	return vol

def peptideVolume(MW):
	"""
	Input: 
	MW: molecular weight of the protein, float
	Output: assuming a globular protein, the volume of 1 molecule in nm^3, float
	"""
	r_min = 0.066*pow(MW, 1.0/3.0)
	vol = (4.0/3.0) *np.pi*pow(r_min,3.0)
	return vol

def proteinRatio(mu):
	"""
	Input: 
	mu: growth rate, float
	Output: protein content in gDW of CEN.PK113-7D cells, as determined by Canelas 2011, float
	"""
	return (0.42436 * mu + 0.35364)
