#!/usr/bin/python3
import os
import sys
import numpy as np
import pandas as pd
import math
import glob
import io
import matplotlib.pyplot as plt
from matplotlib import rc

pd.options.mode.chained_assignment = None
def cellArea(mu):
	"""
	Input: the growth rate
	"""
	cellVol = cellSize(mu)
	r = pow((3.0/4.0)*(1.0/np.pi)*cellVol, 1.0/3.0)
	return 4.0 * np.pi * pow(r, 2.0)

def cellSize(mu):
	"""
	Input: the growth rate
	"""
	cs = 29.3716 * mu + 20.55066
	return cs

def peptideVolume(MW):
	r_min = 0.066*pow(MW, 1.0/3.0)
	vol = (4.0/3.0) *np.pi*pow(r_min,3.0)
	return vol

def peptideArea(MW):
	r_min=0.066*pow(MW, 1.0/3.0)
	vol=np.pi*pow(r_min,2.0)
	return vol

def proteinRatio(mu):
	return (0.42436 * mu + 0.35364)

try: directory = sys.argv[1]
except IndexError: directory = '.'

kdeg = 0.043
mitoVol = 1.25
minUP = 0.22
minUPmito = 0.28
gDW = 1.7
pmcQuota = 0.11
pmnQuota = 0.055
pmpQuota = 0.007
pmsQuota = 0.0015
pmQuota = 0.20
os.chdir(directory)
outputs = sorted(glob.glob('*.out'))

parameters = ['mu', 'qGlc', 'qO2', 'qEtOH', 'qCO2', 'qGlyc', 'qPyr', 'HXTsat', 'Yxs', 'RQ',
 'SumProtein', 'RiboMass', 'RiboMassFraction', 'UPmass', 'UPmassFraction', 'RelUP',
'RelPlasmaMem', 'RelCarbonUpt', 'RelNitrogenUpt', 'RelPhosphorusUpt', 'RelSulfurUpt', 'MitoVol', 'RelMitoVol', 'MitoMassFraction']

df = pd.DataFrame(index = range(0,len(outputs)), columns = parameters)
proteinMass = pd.read_excel('/Users/pranas/surfdrive/MEmodels/pcYeast8/packageForMiha/pcYeast8_dynamicInfo.xls', sheet_name = 'proteinProperties')
riboProts = pd.read_excel('/Users/pranas/surfdrive/MEmodels/pcYeast8/packageForMiha/jupyterNotebook/pcYeast8_GPRs.xls', sheet_name = 'RiboSubunitsA')
pmProts = pd.read_excel('/Users/pranas/surfdrive/MEmodels/pcYeast8/packageForMiha/pcYeast8_dynamicInfo.xls', sheet_name = 'plasmaMembrane')
pmcProts = pd.read_excel('/Users/pranas/surfdrive/MEmodels/pcYeast8/packageForMiha/pcYeast8_dynamicInfo.xls', sheet_name = 'plasmaCarbon')
pmnProts = pd.read_excel('/Users/pranas/surfdrive/MEmodels/pcYeast8/packageForMiha/pcYeast8_dynamicInfo.xls', sheet_name = 'plasmaNitrogen')
pmpProts = pd.read_excel('/Users/pranas/surfdrive/MEmodels/pcYeast8/packageForMiha/pcYeast8_dynamicInfo.xls', sheet_name = 'plasmaPhosphorus')
pmsProts = pd.read_excel('/Users/pranas/surfdrive/MEmodels/pcYeast8/packageForMiha/pcYeast8_dynamicInfo.xls', sheet_name = 'plasmaSulfur')
cytoTable = proteinMass[['Entry', 'Mass']]

allRxns = pd.read_excel('/Users/pranas/surfdrive/MEmodels/pcYeast8/packageForMiha/yeastGEM.xlsx', sheet_name = 'RXNS')
exchangeFluxes = allRxns[['ID', 'NAME']].loc[(allRxns['NAME'].str.contains('exchange')) & (allRxns['EQUATION'].str.contains('\[e\]'))]
metabolicFluxes = pd.read_excel('/Users/pranas/surfdrive/MEmodels/pcYeast8/packageForMiha/jupyterNotebook/yeast_8.4.2_withCorrections.xls', sheet_name = 'reactions')
metabolicFluxes = metabolicFluxes[['id', 'name']]
metabolicFluxes = metabolicFluxes.merge(allRxns[['ID', 'EQUATION']], left_on = 'id', right_on = 'ID', how = 'left').drop('ID', axis = 1)

for i in range(0, len(outputs)): 
	with open(outputs[i], 'r') as f:
		soplex_output = f.readlines()
		f.close()
	for j in range(0, len(soplex_output)):
		if soplex_output[j] == "Primal solution (name, value):\n": begin_of_table = j + 1
		if soplex_output[j].find('All other variables') >=0:
			end_of_table = j
			break
	fluxes = soplex_output[begin_of_table:end_of_table]
	for j in range(0, len(fluxes)): fluxes[j] = fluxes[j].replace('    ','')
	flux_table = pd.read_csv(io.StringIO('\n'.join(fluxes)), sep = '\t', header=None)
	try: flux_table.columns = ['flux', 'value']
	except: continue
	
	df['mu'][i] = flux_table['value'].loc[flux_table['flux'] == 'r_4041'].values[0]
	df['qGlc'][i] = flux_table['value'].loc[flux_table['flux'].isin(['r_1714_rev', 'r_1710_rev', 'r_1931_rev'])].values[0]
	try: df['qO2'][i] = flux_table['value'].loc[flux_table['flux'] == 'r_1992_rev'].values[0]
	except: df['qO2'][i] = 0.0
	try: df['qEtOH'][i] = flux_table['value'].loc[flux_table['flux'] == 'r_1761_fwd'].values[0]
	except: df['qEtOH'][i] = 0.0 
	try: df['qCO2'][i] = flux_table['value'].loc[flux_table['flux'] == 'r_1672_fwd'].values[0]
	except: df['qCO2'][i] = 0.0 
	try: df['qGlyc'][i] = flux_table['value'].loc[flux_table['flux'] == 'r_1808_fwd'].values[0]
	except: df['qGlyc'][i] = 0.0 
	try: df['qPyr'][i] = flux_table['value'].loc[flux_table['flux'] == 'r_2033_fwd'].values[0]
	except: df['qPyr'][i] = 0.0 
	df['UPmass'][i] = flux_table['value'].loc[flux_table['flux'] == 'PDUMMY_folding'].values[0]*1e-3*proteinMass['Mass'].loc[proteinMass['Entry'] == 'PDUMMY'].values[0]/(df['mu'][i]+kdeg)
	try: df['HXTsat'][i] = float(outputs[i].replace('sat_f_', '').split('_')[0])
	except: df['HXTsat'][i] = 1.0
		
	growthRate = str(round(df['mu'][i], 6))
	
	efTable = flux_table.loc[flux_table['flux'].str.contains('|'.join(exchangeFluxes['ID']))].reset_index(drop=True)
	efTable['direction'] = efTable['flux'].apply(lambda x: -1.0 if x.find('_rev') > 0 else +1.0)
	efTable['value'] = efTable['value'] * efTable['direction']
	efTable.loc[:, 'flux'] = efTable['flux'].apply(lambda x: 'r_%s' %(x.split('_')[1]))
	efTable.rename({'value': growthRate, 'flux': 'ID'}, axis = 1, inplace = 1)
	efTable = efTable.groupby('ID')[growthRate].sum()
	efTable = pd.DataFrame(efTable.reset_index())
	exchangeFluxes = exchangeFluxes.merge(efTable[['ID', growthRate]], on = 'ID', how = 'left')
	
	mfTable = flux_table.loc[flux_table['flux'].str.contains('^r_')].reset_index(drop=True)
	mfTable['direction'] = mfTable['flux'].apply(lambda x: -1.0 if x.find('_rev') > 0 else +1.0)
	mfTable['value'] = mfTable['value'] * mfTable['direction']
	mfTable.loc[:, 'flux'] = mfTable['flux'].apply(lambda x: 'r_%s' %(x.split('_')[1]))
	mfTable.rename({'value': growthRate, 'flux': 'id'}, axis = 1, inplace = 1)
	mfTable = mfTable.groupby('id')[growthRate].sum()
	mfTable = pd.DataFrame(mfTable.reset_index())
	metabolicFluxes = metabolicFluxes.merge(mfTable[['id', growthRate]], on = 'id', how = 'left')
	
	translationTable = flux_table.loc[flux_table['flux'].str.contains('_translation_')]
	#The following line causes warning
	translationTable.loc[:,'flux'] = translationTable['flux'].apply(lambda x: x.split('_')[0])
	translationTable = translationTable.merge(cytoTable, left_on = 'flux', right_on = 'Entry', how = 'left').dropna()
	translationTable['volume'] = translationTable['Mass'].apply(peptideVolume)*(6.022e-1 *cellSize(df['mu'][i])/gDW)/(df['mu'][i] + kdeg)
	translationTable['area'] = translationTable['Mass'].apply(peptideArea)*(6.022e2 *cellSize(df['mu'][i])/gDW)/(df['mu'][i] + kdeg)
	translationTable['ProteinMass'] = translationTable['value'] * translationTable['Mass'] * 1e-3/(df['mu'][i]+kdeg) 
	
	importTable = flux_table.loc[flux_table['flux'].str.contains('_import_c_m|translation_MitoRibosome')]
	importTable.loc[:, 'flux'] = importTable['flux'].apply(lambda x: x.split('_')[0])
	importTable = importTable.merge(cytoTable, left_on = 'flux', right_on = 'Entry', how = 'left').dropna()
	importTable['volume'] = importTable['Mass'].apply(peptideVolume)*(6.022e-1 *cellSize(df['mu'][i])/gDW)/(df['mu'][i] + kdeg)
	importTable['area'] = importTable['Mass'].apply(peptideArea)*(6.022e2 *cellSize(df['mu'][i])/gDW)/(df['mu'][i] + kdeg)
	importTable['ProteinMass'] = importTable['value'] * importTable['Mass'] * 1e-3/(df['mu'][i]+kdeg)
	
	cytosolicComplexes = flux_table.loc[flux_table['flux'].str.contains('_complex_formation_c')]
	pmComplexes = []
	for j in pmProts['Gene Systematic Name'].unique(): pmComplexes = pmComplexes + [x for x in cytosolicComplexes['flux'] if x.find(j.replace('-', '') + '_') >= 0]
	pmComplexes = pd.DataFrame(pd.Series(pmComplexes).unique(), columns = ['flux']).merge(flux_table, on = 'flux')
	pmComplexes['Mass'] = 0.0
	for k in pmComplexes['flux']:
		complexMemberMappings = [x.replace('A$', '-A') for x in k.replace('_complex_formation_c', '').split('__')]
		pmComplexes.loc[pmComplexes['flux'] == k, 'Mass'] = sum(proteinMass['Mass'].loc[proteinMass['Entry'].isin(pmProts['Entry'].loc[pmProts['Gene Systematic Name'].isin(complexMemberMappings)])])
		
	pmComplexes['volume'] = pmComplexes['Mass'].apply(peptideVolume)*(6.022e-1 *cellSize(df['mu'][i])/gDW)/(df['mu'][i] + kdeg)
	pmComplexes['area'] = pmComplexes['Mass'].apply(peptideArea)*(6.022e2 *cellSize(df['mu'][i])/gDW)/(df['mu'][i] + kdeg)
	pmComplexes['ProteinMass'] = pmComplexes['value'] * pmComplexes['Mass'] * 1e-3/(df['mu'][i]+kdeg)
	
	df['SumProtein'][i] = sum(translationTable['ProteinMass'])
	df['MitoVol'][i] = sum(importTable['value']*importTable['volume'])
	df['RelPlasmaMem'][i] = sum(pmComplexes['value']*pmComplexes['area'])
	df['RelCarbonUpt'][i] = sum(pmComplexes['value'].loc[pmComplexes['flux'].str.contains('|'.join(pmcProts['Gene Systematic Name'].str.replace('-', '', regex = True)))]*pmComplexes['area'].loc[pmComplexes['flux'].str.contains('|'.join(pmcProts['Gene Systematic Name'].str.replace('-', '', regex = True)))])
	df['RelNitrogenUpt'][i] = sum(pmComplexes['value'].loc[pmComplexes['flux'].str.contains('|'.join(pmnProts['Gene Systematic Name'].str.replace('-', '', regex = True)))]*pmComplexes['area'].loc[pmComplexes['flux'].str.contains('|'.join(pmnProts['Gene Systematic Name'].str.replace('-', '', regex = True)))])
	df['RelPhosphorusUpt'][i] = sum(pmComplexes['value'].loc[pmComplexes['flux'].str.contains('|'.join(pmpProts['Gene Systematic Name'].str.replace('-', '', regex = True)))]*pmComplexes['area'].loc[pmComplexes['flux'].str.contains('|'.join(pmpProts['Gene Systematic Name'].str.replace('-', '', regex = True)))])
	df['RelSulfurUpt'][i] = sum(pmComplexes['value'].loc[pmComplexes['flux'].str.contains('|'.join(pmsProts['Gene Systematic Name'].str.replace('-', '', regex = True)))]*pmComplexes['area'].loc[pmComplexes['flux'].str.contains('|'.join(pmsProts['Gene Systematic Name'].str.replace('-', '', regex = True)))])
	
	riboTable = translationTable[translationTable['Entry'].isin(riboProts['Entry'].loc[~riboProts['SubunitName'].str.contains("^m")])]
	df['RiboMass'][i] = sum(riboTable['ProteinMass'])
	df['MitoMassFraction'][i] = sum(importTable['ProteinMass'])/df['SumProtein'][i]
	
	df['RiboMassFraction'][i] = df['RiboMass'][i]/df['SumProtein'][i]
	df['RelPlasmaMem'][i] = df['RelPlasmaMem'][i] / (cellArea(df['mu'][i]) * pmQuota)
	df['RelCarbonUpt'][i] = df['RelCarbonUpt'][i] / (cellArea(df['mu'][i]) * pmcQuota)
	df['RelNitrogenUpt'][i] = df['RelNitrogenUpt'][i] / (cellArea(df['mu'][i]) * pmnQuota)
	df['RelPhosphorusUpt'][i] = df['RelPhosphorusUpt'][i] / (cellArea(df['mu'][i]) * pmpQuota)
	df['RelSulfurUpt'][i] = df['RelSulfurUpt'][i] / (cellArea(df['mu'][i]) * pmsQuota)
	df['RelMitoVol'][i] = df['MitoVol'][i] / (mitoVol)
	df['UPmassFraction'][i] = df['UPmass'][i]/df['SumProtein'][i]
	df['RelUP'][i] = minUP/df['UPmassFraction'][i]
	print('Processed %d/%d output points.' %(i+1,len(outputs)))

df['Yxs'] = df['mu']/(df['qGlc']*180.1516/1000.0)
df['RQ'] = df['qCO2']/df['qO2']
df = df.sort_values('mu')
df.to_csv('analyzedResults.csv', index=False)

exchangeFluxes.fillna(0, inplace = True)
exchangeFluxes = exchangeFluxes.loc[(exchangeFluxes[exchangeFluxes.columns[2:]].sum(axis = 1) != 0), :]
exchangeFluxes.to_csv('exchangeFluxes.csv', index = False, float_format='%.7E')

metabolicFluxes.fillna(0, inplace = True)
metabolicFluxes = metabolicFluxes.loc[(metabolicFluxes[metabolicFluxes.columns[3:]].sum(axis = 1) != 0), :]
metabolicFluxes.to_csv('metabolicFluxes.csv', index = False, float_format='%.7E')

os.system('python3 /Users/pranas/surfdrive/MEmodels/pcYeast8/packageForMiha/proteinParsing.py %s' %(directory))

labels = [r'q$S_{x}$',r'q$O_{2}$', r'q$EtOH$', r'q$CO_{2}$', r'q$Glyc$', r'q$Pyr$']
ax = df.plot(x = 'mu', y = df.columns[1:7], label = labels, lw = 2.5)
ax.set_xlabel(r"Growth rate ($h^{-1}$)")
ax.set_ylabel(r"Flux ($mmol gDW^{-1}h^{-1}$)")
ax.set_ylim(0,30)
ax.set_xlim(0,0.5)
plt.show()
