
generateMaxentParam <- function(beta, fc, extraArgs = NULL) {
	# produce param vector for dismo::maxent() with beta mulitiplier and feature classes
	tuneTemplate <- c('betamultiplier=1', 'noautofeature', 'jackknife=false', 'linear=false', 'product=false', 'quadratic=false', 'hinge=false', 'threshold=false')
	param <- paste(tuneTemplate, collapse=',')
	fc <- toupper(fc)
	param <- gsub('betamultiplier=1', paste0('betamultiplier=', beta), param)
	if (grepl('L', fc)) param <- gsub('linear=false', 'linear=true', param)
	if (grepl('P', fc)) param <- gsub('product=false', 'product=true', param)
	if (grepl('Q', fc)) param <- gsub('quadratic=false', 'quadratic=true', param)
	if (grepl('H', fc)) param <- gsub('hinge=false', 'hinge=true', param)
	if (grepl('T', fc)) param <- gsub('threshold=false', 'threshold=true', param)
	param <- strsplit(param, ',')[[1]]
	if (!is.null(extraArgs)) {
		for (i in 1:length(extraArgs)) {
			param <- c(param, extraArgs[i])
		}
	}
	return(param)
}




maxentTuning <- function(RMvals, fc, selectedVar, clim, PA, PAbin, blocks = NULL, onlyAIC = FALSE, Boyce = TRUE, TSS = TRUE, SEDI = TRUE, extraArgs = NULL) {

	# RMvals: numeric vector of beta multipliers to test
	# fc: vector of feature class combinations to test
	# selectedVar: character vector of predictor names to be used
	# PA: SpatialPoints of presences (first), and pseudo-absences (second)
	# PAbin: vector of 1's and 0's, denoting which in PA are presences (1) and absences (0)
	# blocks: evaluation blocks as generated by blockCV::spatialBlock

	# if onlyAIC = TRUE and blocks = NULL, then only AIC tuning will be run.

	if (!onlyAIC & is.null(blocks)) {
		stop('blocks needed unless tuning with AIC only.')
	}

	if (all(RMvals == 'default') & all(fc == 'default')) {
		cat('\n\tEvaluating model without tuning.\n\n')
		defaultTuning <- TRUE
		tuneOptions <- 'default'
		names(tuneOptions) <- 'default'
	} else {
		defaultTuning <- FALSE
	}


	if (!defaultTuning) {
		# set up tuning options
		tuneTemplate <- c('betamultiplier=1', 'noautofeature', 'jackknife=false', 'linear=false', 'product=false', 'quadratic=false', 'hinge=false', 'threshold=false')
		counter <- 1
		tuneOptions <- vector('list', length = length(RMvals) * length(fc))
		for (i in 1:length(fc)) {
			for (j in 1:length(RMvals)) {
				param <- generateMaxentParam(beta = RMvals[j], fc = fc[i], extraArgs = extraArgs)
				tuneOptions[[counter]] <- param
				names(tuneOptions)[counter] <- paste0(fc[i], '_', RMvals[j])
				counter <- counter + 1
			}
		}	
	}

	cat('\n\tUsing the following', length(selectedVar), 'predictors:', paste(selectedVar, collapse=' '),'\n')

	if (onlyAIC) {
		cat('\n\tTesting', length(tuneOptions), 'tuning combinations and evaluating with AICc.\n\n')
	} else {

		# if eval blocks from blockCV package, reorganize to structure used by ENMeval::get.block
		if (inherits(blocks, 'SpatialBlock')) {
			folds <- list(occ.grp = numeric(length(which(PAbin == 1))), bg.grp = numeric(length(which(PAbin == 0))))
			for (i in 1:max(blocks$foldID)) {
				blockPres <- intersect(blocks$folds[[i]][[2]], which(PAbin == 1))
				blockBG <- intersect(blocks$folds[[i]][[2]], which(PAbin == 0)) - length(which(PAbin == 1))
				folds[[1]][blockPres] <- i
				folds[[2]][blockBG] <- i
			}
		} else {
			folds <- blocks
		}

		nfolds <- max(folds[[1]])
		cat('\n\tTesting', length(tuneOptions), 'tuning combinations and evaluating across', nfolds, 'folds.\n\n')
	}

	swd <- as.data.frame(extract(clim[[selectedVar]], PA))

	tuningEvals <- vector('list', length = length(tuneOptions))

	for (i in 1:length(tuneOptions)) {

		tune <- tuneOptions[[i]]
		# tune <- c(tune, 'noaddsamplestobackground')		

		if (onlyAIC) cat('\t', i, '\n')

		if (!onlyAIC) {
			boyceVals <- c()
			tssVals <- c()
			sediVals <- c()

			for (j in 1:nfolds) {
				
				cat('\t', i, '--', j, '\n')
				
				trainingPresInd <- which(folds[[1]] %in% setdiff(1:nfolds, j))
				trainingBGInd <- which(folds[[2]] %in% setdiff(1:nfolds, j))

				testingPresInd <- which(folds[[1]] == j)
				testingBGInd <- which(folds[[2]] == j)
				
				# adjust so it matches index of swd, which contains pres + BG
				trainingBGInd <- trainingBGInd + length(which(PAbin == 1))
				testingBGInd <- testingBGInd + length(which(PAbin == 1))

				presTestExtract1 <- swd[testingPresInd,]
				bgTestExtract1 <- swd[testingBGInd,]

				presExtract1 <- swd[trainingPresInd,]
				bgExtract1 <- swd[trainingBGInd,]
				bin <- c(rep(1, nrow(presExtract1)), rep(0, nrow(bgExtract1)))
				DFenv <- as.data.frame(rbind(presExtract1, bgExtract1))
				if (defaultTuning) {
					mod <- maxent(x=DFenv, p = bin)
				} else {
					mod <- maxent(x=DFenv, p = bin, args = tune)
				}
				predPres <- rmaxent::project(mod, presTestExtract1, quiet = TRUE)$prediction_cloglog
				predBg <- rmaxent::project(mod, bgTestExtract1, quiet = TRUE)$prediction_cloglog

				if (Boyce) {
					boyceVals[j] <- contBoyce(predPres, predBg, na.rm = TRUE)
				} else {
					boyceVals[j] <- NA
				}
				if (TSS) {
					tssVals[j] <- tssWeighted(predPres, predBg, na.rm = TRUE)
				} else {
					tssVals[j] <- NA
				}
				if (SEDI) {
					sediVals[j] <- max(sediWeighted(predPres, predBg, na.rm = TRUE), na.rm = TRUE)
				} else {
					sediVals[j] <- NA
				}
				gc()
			}

			tuningEvals[[i]][[1]] <- boyceVals
			tuningEvals[[i]][[2]] <- tssVals
			tuningEvals[[i]][[3]] <- sediVals
		} else {
			tuningEvals[[i]][[1]] <- NA
			tuningEvals[[i]][[2]] <- NA
			tuningEvals[[i]][[3]] <- NA
		}

		# calculate model with all data to calculate AICc
		presExtract1 <- swd[which(PAbin == 1),]
		bgExtract1 <- swd[which(PAbin == 0),]
		bin <- c(rep(1, nrow(presExtract1)), rep(0, nrow(bgExtract1)))
		DFenv <- as.data.frame(rbind(presExtract1, bgExtract1))
		if (defaultTuning) {
			mod <- maxent(x=DFenv, p = bin)
		} else {
			mod <- maxent(x=DFenv, p = bin, args = tune)
		}
		predPres <- rmaxent::project(mod, presExtract1)
		predBg <- rmaxent::project(mod, bgExtract1)
		rawSum <- sum(c(predPres$prediction_raw, predBg$prediction_raw), na.rm=TRUE)

		## calculate log likelihood
		ll <- sum(log(predPres$prediction_raw / rawSum), na.rm=TRUE)
		# calc number of parameters
		K <- ENMeval::get.params(mod)
		# AICc
		tuningEvals[[i]][[4]] <- -2 * ll + 2 * K + (2 * K * (K + 1)) / ( sum(bin) - K - 1)

		names(tuningEvals[[i]]) <- c('Boyce','TSS','SEDI','AICc')

		# clear temporary maxent files

	}

	if (onlyAIC) {
		res <- cbind.data.frame(tuning=names(tuneOptions), AICc=sapply(tuningEvals, function(x) x['AICc']), stringsAsFactors=FALSE)
	} else {
		res <- cbind.data.frame(tuning = names(tuneOptions), Boyce = sapply(tuningEvals, function(x) mean(x$Boyce)), TSS = sapply(tuningEvals, function(x) mean(x$TSS)), SEDI = sapply(tuningEvals, function(x) mean(x$SEDI)), AICc = sapply(tuningEvals, function(x) x$AICc), stringsAsFactors=FALSE)
	}
	res <- cbind.data.frame(res, dAICc = res$AICc - min(res$AICc), stringsAsFactors=FALSE)
	res <- cbind.data.frame(res, wAICc = exp(-0.5 * res$dAICc) / sum(exp(-0.5 * res$dAICc)), stringsAsFactors=FALSE)

	cat('\n')
	return(res)
}



# MaxNet/MaxEnt tuning on AICc only
maxentTuningAICc <- function(RMvals, fc, PA, PAbin, extraArgs = NULL, verbose = TRUE, ret = c('tuning', 'model'), cores = 1, addSamplesToBG = FALSE) {
	
	require(maxnet)
	require(foreach)
	require(doParallel)
		
	counter <- 1
	tuneOptions <- vector('list', length = length(RMvals) * length(fc))
	for (i in 1:length(RMvals)) {
		for (j in 1:length(fc)) {
			tuneOptions[[counter]] <- c(RMvals[i], fc[j])
			counter <- counter + 1
		}
	}

	cat('\n\tTesting', length(tuneOptions), 'tuning combinations and evaluating with AICc.\n\n')
		
	cl <- makeCluster(cores)
	registerDoParallel(cl)
	
	tuneList <- foreach(i = 1:length(tuneOptions), .packages = c('maxnet'), .export = c(), .verbose = FALSE) %dopar% {
		
		tuneVec <- character(6)
		names(tuneVec) <- c('regMult', 'fc', 'numClasses', 'numCoeff', 'logLik', 'AICc')


		if (verbose) message('\t\tbeta = ', tuneOptions[[i]][1], ', features = ', tuneOptions[[i]][2])

		mod <- try(maxnet(PAbin, PA, regmult = as.numeric(tuneOptions[[i]][1]), maxnet.formula(PAbin, PA, classes = tolower(tuneOptions[[i]][2])), addsamplestobackground = addSamplesToBG))

		if (inherits(mod, 'try-error')) {
			modelProblem <- TRUE
		} else {
			modelProblem <- FALSE
			
			predPres <- predict(mod, PA[which(PAbin == 1), ], type = 'exponential')[,1]
			predBg <- predict(mod, PA[which(PAbin == 0), ], type = 'exponential')[,1]

			# calc number of parameters
			K <- length(mod$betas)
			
		}
		
		if (modelProblem) {

			tuneVec['regMult'] <- tuneOptions[[i]][1]
			tuneVec['fc'] <- tuneOptions[[i]][2]
			tuneVec['numClasses'] <- nchar(tuneOptions[[i]][2])
			tuneVec['numCoeff'] <- NA
			tuneVec['logLik'] <- NA
			tuneVec['AICc'] <- NA

		} else {

			## calculate log likelihood
			rawSum <- sum(c(predPres, predBg), na.rm=TRUE)
			ll <- sum(log(predPres / rawSum), na.rm=TRUE)
			
			AICc <- -2 * ll + 2 * K + (2 * K * (K + 1)) / ( sum(PAbin) - K - 1)
			
			tuneVec['regMult'] <- tuneOptions[[i]][1]
			tuneVec['fc'] <- tuneOptions[[i]][2]
			tuneVec['numClasses'] <- nchar(tuneOptions[[i]][2])
			tuneVec['numCoeff'] <- K
			tuneVec['logLik'] <- ll
			tuneVec['AICc'] <- AICc
		}			
		return(tuneVec)
	}
	
	stopCluster(cl)
	
	tuningEvals <- do.call(rbind, tuneList)
	tuningEvals <- as.data.frame(tuningEvals, stringsAsFactors = FALSE)
	for (i in setdiff(colnames(tuningEvals), 'fc')) {
		tuningEvals[, i] <- as.numeric(tuningEvals[, i])
	}
	
	# if any tuning option led to a AIC of NA, drop those
	tuningEvals <- tuningEvals[which(!is.na(tuningEvals[, 'AICc'])), ]
	
	tuningEvals[, 'dAICc'] <- tuningEvals[, 'AICc'] - min(tuningEvals[, 'AICc'], na.rm = TRUE)
	tuningEvals[, 'wAICc'] <- exp(-0.5 * tuningEvals[, 'dAICc']) / sum(exp(-0.5 * tuningEvals[, 'dAICc']), na.rm = TRUE)	
	
	# sort by AIC performance
	tuningEvals <- tuningEvals[order(tuningEvals$AICc, tuningEvals$numClasses, tuningEvals$regMult, tuningEvals$numCoeff, decreasing=c(FALSE, FALSE, TRUE, FALSE)), ]
	rownames(tuningEvals) <- NULL
	
	if (all(ret == 'tuning')) {
		
		return(list(tuning = tuningEvals))
		
	} else if ('model' %in% ret) {
		
		mod <- maxnet(PAbin, PA, regmult = as.numeric(tuningEvals[1, 'regMult']), maxnet.formula(PAbin, PA, classes = tuningEvals[1, 'fc']), addsamplestobackground = addSamplesToBG)
		
		return(list(tuning = tuningEvals, model = mod))		
	}
}



# This version drops one variable at a time, or several if there are ties
backwardVarSelect <- function(rmVals, fc, PA, PAbin, varThresh = 1, verbose = TRUE, cores = 1, addSamplesToBG = FALSE) {
	
	# first pass
	tictoc::tic('\t\ttime spent on tuning')
	tuning <- maxentTuningAICc(rmVals, fc, PA, PAbin, ret = 'tuning', cores = cores, verbose = verbose, addSamplesToBG = addSamplesToBG)
	tictoc::toc()
	
	# get best params -> get all params that have best AICc (that way we preserve record of ties)
	topAIC <- which(tuning$tuning$dAICc < 4)
	tunedBeta <- as.numeric(tuning$tuning[topAIC, 'regMult'])
	tunedClasses <- tuning$tuning[topAIC, 'fc']

	# get variable contribution
	# it is not currently possible to get variable importance from maxnet model, therefore we will use the tuned parameters to fit a maxent model.
	mod <- dismo::maxent(x = PA, p = PAbin, args = generateMaxentParam(beta = tunedBeta[1], fc = tunedClasses[1]))
	maxentVarImport <- ENMeval::var.importance(mod)
	lowPermut <- which(maxentVarImport$permutation.importance < varThresh)
	thinnedVar <- colnames(PA)
	if (length(lowPermut) > 0) {
		lowest <- which(maxentVarImport[, 'permutation.importance'] == min(maxentVarImport[lowPermut, 'permutation.importance']))
		thinnedVar <- setdiff(thinnedVar, as.character(maxentVarImport[lowest, 'variable']))
	}
	
	masterRes <- list()
	tmp <- vector('list', 3)
	names(tmp) <- c('inputVariables', 'thinnedVariables', 'tuning')
	tmp[[1]] <- sort(colnames(PA))
	tmp[[2]] <- sort(thinnedVar)
	tmp[[3]] <- tuning$tuning
	masterRes[[1]] <- tmp
	counter <- 2
	
	gc()
	
	while (any(maxentVarImport$permutation.importance < varThresh)) {
		
		tictoc::tic('\t\ttime spent on tuning')
		
		if (verbose) message('\n\t\tround ', counter, ', ', length(thinnedVar), ' predictors...')
		
		tuning <- maxentTuningAICc(rmVals, fc, PA[, thinnedVar], PAbin, ret = 'tuning', cores = cores, verbose = verbose, addSamplesToBG = addSamplesToBG)
		
		tictoc::toc()
		
		# get best params -> get all params that have best AICc (that way we preserve record of ties)
		topAIC <- which(tuning$tuning$dAICc < 4)
		tunedBeta <- as.numeric(tuning$tuning[topAIC, 'regMult'])
		tunedClasses <- tuning$tuning[topAIC, 'fc']
	
		# get variable contribution
		inputVariables <- thinnedVar
		mod <- dismo::maxent(x = PA[, inputVariables], p = PAbin, args = generateMaxentParam(beta = tunedBeta[1], fc = tunedClasses[1]))
		maxentVarImport <- ENMeval::var.importance(mod)
		lowPermut <- which(maxentVarImport$permutation.importance < varThresh)
		if (length(lowPermut) > 0) {
			lowest <- which(maxentVarImport[, 'permutation.importance'] == min(maxentVarImport[lowPermut, 'permutation.importance']))
			thinnedVar <- setdiff(thinnedVar, as.character(maxentVarImport[lowest, 'variable']))
		}

		tmp <- vector('list', 3)
		names(tmp) <- c('inputVariables', 'thinnedVariables', 'tuning')
		tmp[[1]] <- sort(inputVariables)
		tmp[[2]] <- sort(thinnedVar)
		tmp[[3]] <- tuning$tuning
		masterRes[[counter]] <- tmp
		counter <- counter + 1
		gc()
		
	}
	
	return(masterRes)
	
}




geogCrossValidate <- function(rmVals, fc, PA, PAbin, foldsVec, verbose = TRUE, cores = 1, addSamplesToBG = FALSE) {
	# AUCdiff should reflect the degree of overfitting, with lower values showing less overfitting.
	# OR10: 10% omission rate: 

	counter <- 1
	tuneOptions <- vector('list', length = length(rmVals) * length(fc))
	for (i in 1:length(rmVals)) {
		for (j in 1:length(fc)) {
			tuneOptions[[counter]] <- c(rmVals[i], fc[j])
			counter <- counter + 1
		}
	}

	cat('\n\tTesting', length(tuneOptions), 'tuning combinations and evaluating via cross-validation.\n\n')
	
	nfolds <- max(foldsVec)
		
	cl <- makeCluster(cores)
	registerDoParallel(cl)

	tuneList <- foreach(i = 1:length(tuneOptions), .packages = c('maxnet', 'enmSdm'), .export = c('folds'), .verbose = F) %dopar% {

		if (verbose) message('\t\tbeta = ', tuneOptions[[i]][1], ', features = ', tuneOptions[[i]][2])
		
		tuningEvalVec <- character(11)
		names(tuningEvalVec) <- c('beta', 'fc', 'boyceTrain', 'boyceTest', 'boyceDiff', 'aucTrain', 'aucTest', 'aucDiff', 'sediTrain', 'sediTest', 'OR10')
		beta_i <- as.numeric(tuneOptions[[i]][1])
		fc_i <- tolower(tuneOptions[[i]][2])
				
		klist <- vector('list', nfolds)
		for (k in 1:nfolds) {
			
			ret <- rep(NA, 9)
			names(ret) <- c('boyceTrain', 'boyceTest', 'boyceDiff', 'aucTrain', 'aucTest', 'aucDiff', 'sediTrain', 'sediTest', 'OR10')

			trainPresInd <- intersect(which(foldsVec != k), which(PAbin == 1))
			trainBGind <- intersect(which(foldsVec != k), which(PAbin == 0))
			testPresInd <- intersect(which(foldsVec == k), which(PAbin == 1))
			testBGind <- intersect(which(foldsVec == k), which(PAbin == 0))
						
			if (verbose & k == 1) cat('\t\t\t')
			if (verbose) cat(paste0('fold_', k, '\t'))
						
			mod <- try(maxnet(PAbin[c(trainPresInd, trainBGind)], PA[c(trainPresInd, trainBGind), ], regmult = beta_i, maxnet.formula(PAbin[c(trainPresInd, trainBGind)], PA[c(trainPresInd, trainBGind), ], classes = fc_i), addsamplestobackground = addSamplesToBG))
			
			if (!inherits(mod, 'try-error')) {
				
				predTestPres <- predict(mod, PA[testPresInd, ], type = 'cloglog')[,1]
				predTestBG <- predict(mod, PA[testBGind, ], type = 'cloglog')[,1]
				
				predTrainPres <- predict(mod, PA[trainPresInd, ], type = 'cloglog')[,1]
				predTrainBG <- predict(mod, PA[trainBGind, ], type = 'cloglog')[,1]
	
				# Boyce_train and Boyce_test
				ret['boyceTest'] <- contBoyce(predTestPres, predTestBG, na.rm = TRUE)
				ret['boyceTrain'] <- contBoyce(predTrainPres, predTrainBG, na.rm = TRUE)
				ret['boyceDiff'] <- max(c(0, ret['boyceTrain'] - ret['boyceTest']), na.rm = TRUE)
				
				# AUC train and test
				ret['aucTest'] <- aucWeighted(predTestPres, predTestBG, na.rm = TRUE)
				ret['aucTrain'] <- aucWeighted(predTrainPres, predTrainBG, na.rm = TRUE)
				ret['aucDiff'] <- max(c(0, ret['aucTrain'] - ret['aucTest']), na.rm = TRUE)
				
				# SEDI
				ret['sediTest'] <- max(sediWeighted(predTestPres, predTestBG, na.rm = TRUE), na.rm = TRUE)
				ret['sediTrain'] <- max(sediWeighted(predTrainPres, predTrainBG, na.rm = TRUE), na.rm = TRUE)
				
				# 10% omission rate
				n90 <- ceiling(length(predTrainPres) * 0.9)
				train.thr.10 <- rev(sort(predTrainPres))[n90]
				ret['OR10'] <- mean(predTestPres < train.thr.10, na.rm = TRUE)
			}
			
			klist[[k]] <- ret
		}
		
		kfold <- do.call(rbind, klist)
			
		tuningEvalVec['beta'] <- beta_i
		tuningEvalVec['fc'] <- fc_i
		tuningEvalVec['boyceTrain'] <- mean(kfold[, 'boyceTrain'], na.rm = TRUE)
		tuningEvalVec['boyceTest'] <- mean(kfold[, 'boyceTest'], na.rm = TRUE)
		tuningEvalVec['boyceDiff'] <- mean(kfold[, 'boyceDiff'], na.rm = TRUE)
		tuningEvalVec['aucTrain'] <- mean(kfold[, 'aucTrain'], na.rm = TRUE)
		tuningEvalVec['aucTest'] <- mean(kfold[, 'aucTest'], na.rm = TRUE)
		tuningEvalVec['aucDiff'] <- mean(kfold[, 'aucDiff'], na.rm = TRUE)
		tuningEvalVec['sediTrain'] <- mean(kfold[, 'sediTrain'], na.rm = TRUE)
		tuningEvalVec['sediTest'] <- mean(kfold[, 'sediTest'], na.rm = TRUE)
		tuningEvalVec['OR10'] <- mean(kfold[, 'OR10'], na.rm = TRUE)
		
		if (verbose) cat('\n')
		
		return(tuningEvalVec)
	
	}
	
	stopCluster(cl)

	tuningEvals <- do.call(rbind, tuneList)
	tuningEvals <- as.data.frame(tuningEvals, stringsAsFactors = FALSE)
	for (i in setdiff(colnames(tuningEvals), 'fc')) {
		tuningEvals[,i] <- as.numeric(tuningEvals[, i])
	}
		
	return(tuningEvals)
}
	

# This version drops one variable at a time, or several if there are ties
backwardVarSelect_geogFolds <- function(rmVals, fc, PA, PAbin, foldsVec, rankMetric = 'boyceTest', varThresh = 1, verbose = TRUE, cores = 1, addSamplesToBG = FALSE) {
	
	if (!rankMetric %in% c('boyceTrain', 'boyceTest', 'boyceDiff', 'aucTrain', 'aucTest', 'aucDiff', 'sediTrain', 'sediTest', 'OR10')) {
		stop("rankMetric not recognized.")
	}
	
	PA <- as.data.frame(PA)
	
	# first pass
	tictoc::tic('\t\ttime spent on tuning')
	tuning <- geogCrossValidate(rmVals, fc, PA, PAbin, foldsVec, verbose = verbose, cores = cores, addSamplesToBG = addSamplesToBG)
	tictoc::toc()
	
	# Identify best model as having the highest Boyce test score, while being in the lower 50% quantile of omission rate
	lowerOR <- tuning[which(tuning$OR10 <= quantile(tuning$OR10, 0.5)),]
	topTune <- lowerOR[which(lowerOR[, rankMetric] == max(lowerOR[, rankMetric])),]
	tunedBeta <- topTune$beta
	tunedClasses <- topTune$fc

	# get variable contribution
	# it is not currently possible to get variable importance from maxnet model, therefore we will use the tuned parameters to fit a maxent model.
	mod <- dismo::maxent(x = PA, p = PAbin, args = generateMaxentParam(beta = tunedBeta[1], fc = tunedClasses[1]))
	maxentVarImport <- ENMeval::var.importance(mod)
	lowPermut <- which(maxentVarImport$permutation.importance < varThresh)
	thinnedVar <- colnames(PA)
	if (length(lowPermut) > 0) {
		lowest <- which(maxentVarImport[, 'permutation.importance'] == min(maxentVarImport[lowPermut, 'permutation.importance']))
		thinnedVar <- setdiff(thinnedVar, as.character(maxentVarImport[lowest, 'variable']))
	}
	
	masterRes <- list()
	tmp <- vector('list', 3)
	names(tmp) <- c('inputVariables', 'thinnedVariables', 'tuning')
	tmp[[1]] <- sort(colnames(PA))
	tmp[[2]] <- sort(thinnedVar)
	tmp[[3]] <- tuning
	masterRes[[1]] <- tmp
	counter <- 2
	
	gc()
	
	while (any(maxentVarImport$permutation.importance < varThresh)) {
		
		tictoc::tic('\t\ttime spent on tuning')
		if (verbose) message('\n\t\tround ', counter, ', ', length(thinnedVar), ' predictors...')
		
		tuning <- geogCrossValidate(rmVals, fc, PA[, thinnedVar], PAbin, foldsVec, verbose = verbose, cores = cores, addSamplesToBG = addSamplesToBG)

		tictoc::toc()
		
		# Identify best model as having the highest Boyce test score, while being in the lower 50% quantile of omission rate
		lowerOR <- tuning[which(tuning$OR10 <= quantile(tuning$OR10, 0.5)),]
		topTune <- lowerOR[which(lowerOR[, rankMetric] == max(lowerOR[, rankMetric])),]
		tunedBeta <- topTune$beta
		tunedClasses <- topTune$fc
	
		# get variable contribution
		inputVariables <- thinnedVar
		mod <- dismo::maxent(x = PA[, inputVariables], p = PAbin, args = generateMaxentParam(beta = tunedBeta[1], fc = tunedClasses[1]))
		maxentVarImport <- ENMeval::var.importance(mod)
		lowPermut <- which(maxentVarImport$permutation.importance < varThresh)
		if (length(lowPermut) > 0) {
			lowest <- which(maxentVarImport[, 'permutation.importance'] == min(maxentVarImport[lowPermut, 'permutation.importance']))
			thinnedVar <- setdiff(thinnedVar, as.character(maxentVarImport[lowest, 'variable']))
		}

		tmp <- vector('list', 3)
		names(tmp) <- c('inputVariables', 'thinnedVariables', 'tuning')
		tmp[[1]] <- sort(inputVariables)
		tmp[[2]] <- sort(thinnedVar)
		tmp[[3]] <- tuning
		masterRes[[counter]] <- tmp
		counter <- counter + 1
		gc()
		
	}
	
	return(masterRes)
	
}


