
# Linear discriminant analysis of morphogroups

library(raster)
library(rgeos)
library(rgdal)

climDir <- 'data/SDM/env/chelsa/current'
landcoverDir <- 'data/SDM/env/earthEnvLandCover'

# load occurrences
allOcc <- readRDS('data/SDM/occ5.rds')
allOcc <- allOcc[, c('decimalLongitude','decimalLatitude')]
colnames(allOcc) <- c('Longitude','Latitude')


load('data/SDM/region_list.Rdata')

presPtsList <- vector('list', 4)
names(presPtsList) <- c('coastal', 'interior', 'coastalInterior', 'rockies')
#################################
# Coastal

	subOcc <- rgeos::gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), region_list$coastal)
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL
	
	presPtsList[['coastal']] <- subOcc
	
#################################
# Interior

	subOcc <- rgeos::gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), region_list$interior)
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL
	
	presPtsList[['interior']] <- subOcc
	
#################################
# Coastal+Interior

	subOcc <- rgeos::gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), rgeos::gUnion(region_list$interior, region_list$coastal))
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL
	
	presPtsList[['coastalInterior']] <- subOcc

#################################
# Rockies

	subOcc <- rgeos::gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), region_list$rockies)
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL
	
	presPtsList[['rockies']] <- subOcc




# ----------------------------------------------
# climate only

vifVar <- c("CHELSA_bioclim_02", "CHELSA_bioclim_07", "CHELSA_bioclim_14", "CHELSA_bioclim_15", "CHELSA_bioclim_18", "CHELSA_bioclim_19", "climaticMoistureIndex", "embergerQ", "minTempWarmest", "PETDriestQuarter", "PETseasonality", "PETWettestQuarter")

vifVarLC <- c("CHELSA_bioclim_02", "CHELSA_bioclim_03", "CHELSA_bioclim_07", "CHELSA_bioclim_14", "CHELSA_bioclim_15", "CHELSA_bioclim_18", "CHELSA_bioclim_19", "climaticMoistureIndex", "embergerQ", "minTempWarmest", "PETDriestQuarter", "PETseasonality", "PETWettestQuarter", paste0("landcover", c(1,2,3,4,5,6,8,10,11)))


# load climate grids
clim <- list.files(climDir, full.names=T)
clim <- clim[ - grep('tmin|tmax|tmean|precip|solrad|monthlyPET', clim)] # remove monthly rasters

clim <- clim[ - grep('summer|winter', clim, ignore.case=TRUE)]
clim <- clim[ - grep('monthCountByTemp10', clim)] # this is a discrete variable which we won't use
clim <- clim[ - grep('aridityIndexThornthwaite', clim)] # this is redundant with climatic moisture index
clim <- stack(clim)

names(clim)

# Coarsen from 1x1 km to 2x2 km
clim <- aggregate(clim, fact = 2)


landcover <- list.files(landcoverDir, full.names = TRUE)
landcover <- landcover[order(as.numeric(gsub('(.+)(full_class_)(\\d?\\d)\\.tif$', '\\3', landcover)))]
landcover <- stack(landcover)

names(landcover) <- gsub('consensus_full_class_', 'landcover', names(landcover))
landcover <- landcover[[intersect(names(landcover), vifVarLC)]]

landcover <- resample(landcover, clim)

for (i in 1:nlayers(landcover)) {
	landcover[[i]][which(is.na(values(clim[[1]])))] <- NA
}

clim <- addLayer(clim, landcover)

rm(landcover)

climVar <- setdiff(names(clim), grep('landcover', names(clim), value = TRUE))


# generate background points: 100k across the region (sampling is biased to account for changing cell size with latitude)
bg <- enmSdm::sampleRast(clim[[1]], n = 100000, replace = FALSE, prob = FALSE)

# drop any records that fall outside of climate data
e <- extract(clim[[c(1,nlayers(clim))]], bg)
if (anyNA(e)) {
	bg <- bg[complete.cases(e),]
}	

for (i in 1:length(presPtsList)) {

	e <- extract(clim[[c(1,nlayers(clim))]], presPtsList[[i]])
	if (anyNA(e)) {
		presPtsList[[i]] <- presPtsList[[i]][complete.cases(e),]
	}	
	
}

coastal_pres <- extract(clim, presPtsList[['coastal']])
interior_pres <- extract(clim, presPtsList[['interior']])
coastalInterior_pres <- extract(clim, presPtsList[['coastalInterior']])
rockies_pres <- extract(clim, presPtsList[['rockies']])	
e_bg <- extract(clim, bg)

save(presPtsList, coastal_pres, interior_pres, coastalInterior_pres, rockies_pres, e_bg, file = 'climExtracts.Rdata')

# the file climExtracts.Rdata contains the environmental data at each occurrence record coordinate.


# can pick up from here without needing to execute the above code.
load('data/SDM/climExtracts.Rdata')

# collapse landcover types into closed and open habitats
# LC1+2+3+4 = forests (needleleaf, broadleaf, mixed/other trees)
# LC5+6+8+10+11 = open habitats (shrubs, herbaceous, flooded vegetation, snow/ice, barren)
closedLC <- paste0('landcover', 1:4)
openLC <- paste0('landcover', c(5,6,8)) # excluding 10 (snow/ice) and 11 (barren)
#coastal_pres <- cbind(coastal_pres, openLC = rowSums(coastal_pres[, openLC]))
coastal_pres <- cbind(coastal_pres, closedLC = rowSums(coastal_pres[, closedLC]))
#interior_pres <- cbind(interior_pres, openLC = rowSums(interior_pres[, openLC]))
interior_pres <- cbind(interior_pres, closedLC = rowSums(interior_pres[, closedLC]))
#rockies_pres <- cbind(rockies_pres, openLC = rowSums(rockies_pres[, openLC]))
rockies_pres <- cbind(rockies_pres, closedLC = rowSums(rockies_pres[, closedLC]))

#e_bg <- cbind(e_bg, openLC = rowSums(e_bg[, openLC]))
e_bg <- cbind(e_bg, closedLC = rowSums(e_bg[, closedLC]))

#e_allOcc <- cbind(e_allOcc, openLC = rowSums(e_allOcc[, openLC]))
e_allOcc <- cbind(e_allOcc, closedLC = rowSums(e_allOcc[, closedLC]))

# drop landcover types but keep open / closed
coastal_pres <- coastal_pres[, !grepl('landcover', colnames(coastal_pres))]
interior_pres <- interior_pres[, !grepl('landcover', colnames(interior_pres))]
rockies_pres <- rockies_pres[, !grepl('landcover', colnames(rockies_pres))]



cols <- c(fullSp = 'darkslateblue', coastal = 'yellow3', interior = 'skyblue3', coastalInterior = '#31a354', rockies = 'purple')
symbs <- c(fullSp = 25, coastal = 21, interior = 22, coastalInterior = 23, rockies = 24)
transp <- 0.5

# closed landcover and climatic moisture index
setEPS()
cairo_ps('output/Figure10_closedLandcoverDensities+CMI.eps', width = 14, height = 7)
#pdf('output/Figure10_closedLandcoverDensities+CMI.pdf', width = 14, height = 7)
	
	par(mfrow=c(1,2))		
	
	plot.new()
	xrange <- c(-1, 1)
	yrange <- range(density(coastal_pres[, 'climaticMoistureIndex'])$y, density(interior_pres[, 'climaticMoistureIndex'])$y, density(rockies_pres[, 'climaticMoistureIndex'])$y)
	plot.window(xlim = c(xrange[1]*0.99, xrange[2]*1.01), ylim = 	c(0, yrange[2]))
	axis(1, lwd = 0, lwd.ticks = 1)
	axis(2, lwd = 0, lwd.ticks = 1)
	box(which = 'plot', bty = 'l')
	mtext('climatic moisture index', side = 1, line = 2)
	mtext('density', side = 2, line = 2.4)

	polygon(density(coastal_pres[, 'climaticMoistureIndex']), col = adjustcolor(cols['coastal'], alpha.f = 0.5), border = adjustcolor(cols['coastal'], alpha.f = 0.7))
	polygon(density(interior_pres[, 'climaticMoistureIndex']), col = adjustcolor(cols['interior'], alpha.f = 0.5), border = adjustcolor(cols['interior'], alpha.f = 0.7))
	polygon(density(rockies_pres[, 'climaticMoistureIndex']), col = adjustcolor(cols['rockies'], alpha.f = 0.5), border = adjustcolor(cols['rockies'], alpha.f = 0.7))

	legend('topright', legend = c('Coastal', 'Interior', 'Rocky Mountain'), pt.bg = adjustcolor(cols[c('coastal', 'interior','rockies')], alpha.f = transp), pch = symbs[c('coastal', 'coastal', 'coastal')], bty = 'n')

	mtext('climatic moisture index', side = 3, line = 1, cex = 1, font = 2)
	mtext('A', side = 3, line = 1, cex = 1, font = 2, at = par('usr')[1])
	
	plot.new()
	plot.window(xlim = range(0, 100), ylim = c(0, 0.03))
	axis(1, lwd = 0, lwd.ticks = 1)
	axis(2, at = c(0, 0.01, 0.02, 0.03), lwd = 0, lwd.ticks = 1)
	box(which = 'plot', bty = 'l')
	mtext('percent cover', side = 1, line = 2)
	mtext('density', side = 2, line = 2.4)
	
	polygon(density(coastal_pres[, 'closedLC']), col = adjustcolor(cols['coastal'], alpha.f = 0.5), border = adjustcolor(cols['coastal'], alpha.f = 0.7))
	polygon(density(interior_pres[, 'closedLC']), col = adjustcolor(cols['interior'], alpha.f = 0.5), border = adjustcolor(cols['interior'], alpha.f = 0.7))
	polygon(density(rockies_pres[, 'closedLC']), col = adjustcolor(cols['rockies'], alpha.f = 0.5), border = adjustcolor(cols['rockies'], alpha.f = 0.7))
	
	mtext('closed landcover types', side = 3, line = 1, cex = 1, font = 2)
	mtext('B', side = 3, line = 1, cex = 1, font = 2, at = par('usr')[1])
	


dev.off()


varToPlot <- c('CHELSA_bioclim_07', 'minTempWarmest', 'CHELSA_bioclim_02', 'CHELSA_bioclim_03')
varToPlotLabels <- c(minTempWarmest='min temp of the warmest month', CHELSA_bioclim_02='bioclim 2: mean of monthly temp ranges', CHELSA_bioclim_03='bioclim 3: isothermality', CHELSA_bioclim_07='bioclim 7: annual temp range')

# setEPS()
# cairo_ps('output/FigureA14_LDA_topClim_densities.eps', width = 10, height = 10)
pdf('output/FigureA14_LDA_topClim_densities.pdf', width = 10, height = 10)
par(mfrow=c(2,2), mar = c(2,4,2,2))

for (i in 1:length(varToPlot)) {
	
	plot.new()
	xrange <- range(coastal_pres[, varToPlot[i]], interior_pres[, varToPlot[i]], rockies_pres[, varToPlot[i]])
	if (varToPlot[i] == 'climaticMoistureIndex') xrange <- c(-1,1)
	yrange <- range(density(coastal_pres[, varToPlot[i]])$y, density(interior_pres[, varToPlot[i]])$y, density(rockies_pres[, varToPlot[i]])$y)
	plot.window(xlim = c(xrange[1]*0.99, xrange[2]*1.01), ylim = 	c(0, yrange[2]))
	axis(1, lwd = 0, lwd.ticks = 1)
	axis(2, lwd = 0, lwd.ticks = 1)
	box(which = 'plot', bty = 'l')
	polygon(density(coastal_pres[, varToPlot[i]]), col = adjustcolor(cols['coastal'], alpha.f = 0.5), border = adjustcolor(cols['coastal'], alpha.f = 0.7))
	polygon(density(interior_pres[, varToPlot[i]]), col = adjustcolor(cols['interior'], alpha.f = 0.5), border = adjustcolor(cols['interior'], alpha.f = 0.7))
	polygon(density(rockies_pres[, varToPlot[i]]), col = adjustcolor(cols['rockies'], alpha.f = 0.5), border = adjustcolor(cols['rockies'], alpha.f = 0.7))
	title(main = varToPlotLabels[varToPlot[i]])
	mtext('density', side = 2, line = 2.2)
	if (i == 1) {
		legend('topleft', legend = c('Coastal', 'Interior', 'Rocky Mountain'), pt.bg = adjustcolor(cols[c('coastal', 'interior','rockies')], alpha.f = transp), pch = symbs[c('coastal', 'coastal', 'coastal')], bty = 'n')
	}
}
dev.off()


# isothermality: Isothermality quantifies how large the day-to-night temperatures oscillate relative to the summer-to-winter (annual) oscillations.


