# Combine clade-specific model tuning and generate final models for each clade


library(maptools)
library(rgeos)
library(rgdal)
library(maxnet)
library(raster)
library(dismo)


source('scripts/SDM/maxentTuningFxn.R')

# change temporary directory so I can more actively manage it
tmpPath <- '~/maxentTemp'

if (!dir.exists(tmpPath)) {
	dir.create(tmpPath)
}

tempdir()
unixtools::set.tempdir(tmpPath)

climDir <- 'data/SDM/env/chelsa/current'
climDirLGM <- 'data/SDM/env/chelsa/lgm_ccsm4'
climDirFuture <- 'data/SDM/env/chelsa/future'
anthropoDir <- 'data/SDM/env/anthropo/mosaics'
landcoverDir <- 'data/SDM/env/earthEnvLandCover'

outputDir <- 'data/SDM/finalSDMmodels'

tifOptions <- c("COMPRESS=DEFLATE", "PREDICTOR=2", "ZLEVEL=6")


# ----------------------------------------------------------------
# Processing occurrence records
## Occurrence filtering done in script prepOcc.R

# load occurrences
allOcc <- readRDS('data/SDM/occ5.rds')
allOcc <- allOcc[, c('decimalLongitude','decimalLatitude')]
colnames(allOcc) <- c('Longitude','Latitude')
# -----------------------------------------------------------------
# Process climatic predictors

# from script 4
vifVar <- c("CHELSA_bioclim_02", "CHELSA_bioclim_07", "CHELSA_bioclim_14", "CHELSA_bioclim_15", "CHELSA_bioclim_18", "CHELSA_bioclim_19", "climaticMoistureIndex", "embergerQ", "minTempWarmest", "PETDriestQuarter", "PETseasonality", "PETWettestQuarter")



# load climate grids
clim <- list.files(climDir, full.names=T)
clim <- clim[ - grep('tmin|tmax|tmean|precip|solrad|monthlyPET', clim)] # remove monthly rasters
clim <- clim[ - grep('summer|winter', clim, ignore.case=TRUE)]
clim <- clim[ - grep('monthCountByTemp10', clim)] # this is a discrete variable which we won't use
clim <- clim[ - grep('aridityIndexThornthwaite', clim)] # this is redundant with climatic moisture index
clim <- stack(clim)
clim <- clim[[vifVar]]

names(clim)

# Coarsen from 1x1 km to 2x2 km
clim <- aggregate(clim, fact = 2)

# ---------------------------------------------------------------
LGMclim <- list.files(climDirLGM, full.names=TRUE)
LGMclim <- LGMclim[ - grep('tmin|tmax|tmean|precip|solrad|monthlyPET', LGMclim)] # remove monthly rasters
LGMclim <- LGMclim[ - grep('summer|winter', LGMclim, ignore.case=TRUE)]
LGMclim <- LGMclim[ - grep('monthCountByTemp10', LGMclim)] # this is a discrete variable which we won't use
LGMclim <- LGMclim[ - grep('aridityIndexThornthwaite', LGMclim)] # this is redundant with climatic moisture index
LGMclim <- stack(LGMclim)

# Coarsen from 1x1 km to 2x2 km
LGMclim <- aggregate(LGMclim, fact = 2)

names(LGMclim) <- gsub('_LGM', '', names(LGMclim))



# ---------------------------------------------------------------
futureClim <- list.files(climDirFuture, full.names=TRUE)
futureClim <- grep('2070', futureClim, value = TRUE)
futureClim <- futureClim[ - grep('tmin|tmax|tmean|precip|solrad|monthlyPET', futureClim)] # remove monthly rasters
futureClim <- futureClim[ - grep('monthCountByTemp10', futureClim)] # this is a discrete variable which we won't use
futureClim <- futureClim[ - grep('aridityIndexThornthwaite', futureClim)] # this is redundant with climatic moisture index
futureClim <- stack(futureClim)

# Coarsen from 1x1 km to 2x2 km
futureClim <- aggregate(futureClim, fact = 2)

names(futureClim) <- gsub('_?FUTURE2070', '', names(futureClim))



modList <- vector('list', 5)
names(modList) <- c('fullSp', 'coastal', 'interior', 'coastalInterior', 'rockies')

################################################################################
################################################################################
# CLADE MODELING

bg <- readRDS('data/PA.rds')[[1]]
bg <- coordinates(bg)
bg <- tail(bg, 100000)
rownames(bg) <- NULL

# drop any records that fall outside of climate data
e <- extract(clim[[c(1,nlayers(clim))]], bg)
if (anyNA(e)) {
	bg <- bg[which(complete.cases(e)),]
}

load('data/SDM/region_list.Rdata')


### Load tuning results
rockiesTuning <- readRDS('data/SDM/rockies_tuning_v2.rds')
pacificTuning <- readRDS('data/SDM/pacific_tuning_v2.rds')
interiorTuning <- readRDS('data/SDM/interior_tuning_v2.rds')
pacificInteriorTuning <- readRDS('data/SDM/pacificInterior_tuning_v2.rds')


#################################
# Rockies

	clade <- 'rockies'
	varSelect1 <- rockiesTuning
	currentPXname <- paste0(clade, '_current_aicVarSelect')
	lgmPXname <- paste0(clade, '_LGM_aicVarSelect')
	futureLXname <- paste0(clade, '_FUTURE2070_aicVarSelect')
	

	subOcc <- gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), region_list$rockies)
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL

	# drop any records that fall outside of climate data
	e <- extract(clim[[c(1,nlayers(clim))]], subOcc)
	if (anyNA(e)) {
		subOcc <- subOcc[complete.cases(e),]
	}	
	
	PA <- rbind(subOcc, bg)
	PA <- SpatialPoints(PA, proj4string=CRS('+proj=longlat +datum=WGS84'))
	PAbin <- c(rep(1, nrow(subOcc)), rep(0, nrow(bg)))
	
	PA <- spTransform(PA, crs(clim))
	PA <- SpatialPointsDataFrame(PA, as.data.frame(PAbin))
	swd <- as.data.frame(extract(clim[[vifVar]], PA))
	
	# model tuned for beta and feature class, with set predictors
	tunedBeta <- varSelect1[[1]][1, 'regMult']
	tunedFC <- tolower(varSelect1[[1]][1, 'fc'])
	mod <- maxnet(PAbin, swd[, vifVar], regmult = tunedBeta, maxnet.formula(PAbin, swd[, vifVar], classes = tunedFC))
		
	# model tuned through backward variable selection, where each iteration was tuned via AICc
	thinnedVar <- varSelect1[[length(varSelect1)]][['inputVariables']]
	tunedBeta <- varSelect1[[length(varSelect1)]]$tuning[1, 'regMult']
	tunedFC <- tolower(varSelect1[[length(varSelect1)]]$tuning[1, 'fc'])
	mod <- maxnet(PAbin, swd[, thinnedVar], regmult = tunedBeta, maxnet.formula(PAbin, swd[, thinnedVar], classes = tunedFC))
	px <- ENMeval::maxnet.predictRaster(mod, clim[[colnames(swd)]], type = 'cloglog', clamp = TRUE)
	names(px) <- currentPXname
	
	modList[['rockies']] <- mod
	
	# projected to LGM
	pxLGM <- ENMeval::maxnet.predictRaster(mod, LGMclim[[colnames(swd)]], type = 'cloglog', clamp = TRUE)
	names(pxLGM) <- lgmPXname
	
	# project to future
	pxFuture <- ENMeval::maxnet.predictRaster(mod, futureClim[[colnames(swd)]], type = 'cloglog', clamp = TRUE)
	names(pxFuture) <- futureLXname

	writeRaster(px, paste0(outputDir, currentPXname, '.tif'), NAflag = -9999, options = tifOptions, overwrite=TRUE)
	writeRaster(pxLGM, paste0(outputDir, lgmPXname, '.tif'), NAflag = -9999, options = tifOptions, overwrite=TRUE)
	writeRaster(pxFuture, paste0(outputDir, futureLXname, '.tif'), NAflag = -9999, options = tifOptions, overwrite=TRUE)
	
	rm(clade, varSelect1, currentPXname, lgmPXname, pxFuture, subOcc, PA, PAbin, thinnedVar, swd, px, pxLGM, mod)
	gc()


#################################
# Coastal

	clade <- 'pacific'
	varSelect1 <- pacificTuning
	currentPXname <- paste0(clade, '_current_aicVarSelect')
	lgmPXname <- paste0(clade, '_LGM_aicVarSelect')
	futureLXname <- paste0(clade, '_FUTURE2070_aicVarSelect')
	

	subOcc <- gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), region_list$coastal)
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL

	# drop any records that fall outside of climate data
	e <- extract(clim[[c(1,nlayers(clim))]], subOcc)
	if (anyNA(e)) {
		subOcc <- subOcc[complete.cases(e),]
	}	
	
	PA <- rbind(subOcc, bg)
	PA <- SpatialPoints(PA, proj4string=CRS('+proj=longlat +datum=WGS84'))
	PAbin <- c(rep(1, nrow(subOcc)), rep(0, nrow(bg)))
	
	PA <- spTransform(PA, crs(clim))
	PA <- SpatialPointsDataFrame(PA, as.data.frame(PAbin))
	swd <- as.data.frame(extract(clim[[vifVar]], PA))
	
	# model tuned for beta and feature class, with set predictors
	tunedBeta <- varSelect1[[1]][1, 'regMult']
	tunedFC <- tolower(varSelect1[[1]][1, 'fc'])
	mod <- maxnet(PAbin, swd[, vifVar], regmult = tunedBeta, maxnet.formula(PAbin, swd[, vifVar], classes = tunedFC))	
	
	# model tuned through backward variable selection, where each iteration was tuned via AICc
	thinnedVar <- varSelect1[[length(varSelect1)]][['inputVariables']]
	tunedBeta <- varSelect1[[length(varSelect1)]]$tuning[1, 'regMult']
	tunedFC <- tolower(varSelect1[[length(varSelect1)]]$tuning[1, 'fc'])
	mod <- maxnet(PAbin, swd[, thinnedVar], regmult = tunedBeta, maxnet.formula(PAbin, swd[, thinnedVar], classes = tunedFC))
	px <- ENMeval::maxnet.predictRaster(mod, clim[[colnames(swd)]], type = 'cloglog', clamp = TRUE)
	names(px) <- currentPXname
	
	modList[['coastal']] <- mod
	
	# projected to LGM
	pxLGM <- ENMeval::maxnet.predictRaster(mod, LGMclim[[colnames(swd)]], type = 'cloglog', clamp = TRUE)
	names(pxLGM) <- lgmPXname
	
	# project to future
	pxFuture <- ENMeval::maxnet.predictRaster(mod, futureClim[[colnames(swd)]], type = 'cloglog', clamp = TRUE)
	names(pxFuture) <- futureLXname	
		
	writeRaster(px, paste0(outputDir, currentPXname, '.tif'), NAflag = -9999, options = tifOptions, overwrite=TRUE)
	writeRaster(pxLGM, paste0(outputDir, lgmPXname, '.tif'), NAflag = -9999, options = tifOptions, overwrite=TRUE)
	writeRaster(pxFuture, paste0(outputDir, futureLXname, '.tif'), NAflag = -9999, options = tifOptions, overwrite=TRUE)

	rm(clade, varSelect1, currentPXname, lgmPXname, pxFuture, subOcc, PA, PAbin, thinnedVar, swd, px, pxLGM, mod)	
	gc()

#################################
# Interior

	clade <- 'interior'
	varSelect1 <- interiorTuning
	currentPXname <- paste0(clade, '_current_aicVarSelect')
	lgmPXname <- paste0(clade, '_LGM_aicVarSelect')
	futureLXname <- paste0(clade, '_FUTURE2070_aicVarSelect')
	

	subOcc <- gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), region_list$interior)
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL

	# drop any records that fall outside of climate data
	e <- extract(clim[[c(1,nlayers(clim))]], subOcc)
	if (anyNA(e)) {
		subOcc <- subOcc[complete.cases(e),]
	}	
	
	PA <- rbind(subOcc, bg)
	PA <- SpatialPoints(PA, proj4string=CRS('+proj=longlat +datum=WGS84'))
	PAbin <- c(rep(1, nrow(subOcc)), rep(0, nrow(bg)))
	
	PA <- spTransform(PA, crs(clim))
	PA <- SpatialPointsDataFrame(PA, as.data.frame(PAbin))
	swd <- as.data.frame(extract(clim[[vifVar]], PA))
	
	# model tuned for beta and feature class, with set predictors
	tunedBeta <- varSelect1[[1]][1, 'regMult']
	tunedFC <- tolower(varSelect1[[1]][1, 'fc'])
	mod <- maxnet(PAbin, swd[, vifVar], regmult = tunedBeta, maxnet.formula(PAbin, swd[, vifVar], classes = tunedFC))

	# model tuned through backward variable selection, where each iteration was tuned via AICc
	thinnedVar <- varSelect1[[length(varSelect1)]][['inputVariables']]
	tunedBeta <- varSelect1[[length(varSelect1)]]$tuning[1, 'regMult']
	tunedFC <- tolower(varSelect1[[length(varSelect1)]]$tuning[1, 'fc'])
	mod <- maxnet(PAbin, swd[, thinnedVar], regmult = tunedBeta, maxnet.formula(PAbin, swd[, thinnedVar], classes = tunedFC))
	px <- ENMeval::maxnet.predictRaster(mod, clim[[colnames(swd)]], type = 'cloglog', clamp = TRUE)
	names(px) <- currentPXname
	
	modList[['interior']] <- mod
	
	# projected to LGM
	pxLGM <- ENMeval::maxnet.predictRaster(mod, LGMclim[[colnames(swd)]], type = 'cloglog', clamp = TRUE)
	names(pxLGM) <- lgmPXname

	# project to future
	pxFuture <- ENMeval::maxnet.predictRaster(mod, futureClim[[colnames(swd)]], type = 'cloglog', clamp = TRUE)
	names(pxFuture) <- futureLXname
		
	writeRaster(px, paste0(outputDir, currentPXname, '.tif'), NAflag = -9999, options = tifOptions, overwrite=TRUE)
	writeRaster(pxLGM, paste0(outputDir, lgmPXname, '.tif'), NAflag = -9999, options = tifOptions, overwrite=TRUE)
	writeRaster(pxFuture, paste0(outputDir, futureLXname, '.tif'), NAflag = -9999, options = tifOptions, overwrite=TRUE)
	
	rm(clade, varSelect1, currentPXname, lgmPXname, pxFuture, subOcc, PA, PAbin, thinnedVar, swd, px, pxLGM, mod)
	gc()

#################################
# Pacific+Interior

	clade <- 'pacificInterior'
	varSelect1 <- pacificInteriorTuning
	currentPXname <- paste0(clade, '_current_aicVarSelect')
	lgmPXname <- paste0(clade, '_LGM_aicVarSelect')
	futureLXname <- paste0(clade, '_FUTURE2070_aicVarSelect')
	
	subOcc <- gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), gUnion(region_list$interior, region_list$coastal))
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL

	# drop any records that fall outside of climate data
	e <- extract(clim[[c(1,nlayers(clim))]], subOcc)
	if (anyNA(e)) {
		subOcc <- subOcc[complete.cases(e),]
	}	
	
	PA <- rbind(subOcc, bg)
	PA <- SpatialPoints(PA, proj4string=CRS('+proj=longlat +datum=WGS84'))
	PAbin <- c(rep(1, nrow(subOcc)), rep(0, nrow(bg)))
	
	PA <- spTransform(PA, crs(clim))
	PA <- SpatialPointsDataFrame(PA, as.data.frame(PAbin))
	swd <- as.data.frame(extract(clim[[vifVar]], PA))
	
	# model tuned for beta and feature class, with set predictors
	tunedBeta <- varSelect1[[1]][1, 'regMult']
	tunedFC <- tolower(varSelect1[[1]][1, 'fc'])
	mod <- maxnet(PAbin, swd[, vifVar], regmult = tunedBeta, maxnet.formula(PAbin, swd[, vifVar], classes = tunedFC))
	
	# model tuned through backward variable selection, where each iteration was tuned via AICc
	thinnedVar <- varSelect1[[length(varSelect1)]][['inputVariables']]
	tunedBeta <- varSelect1[[length(varSelect1)]]$tuning[1, 'regMult']
	tunedFC <- tolower(varSelect1[[length(varSelect1)]]$tuning[1, 'fc'])
	mod <- maxnet(PAbin, swd[, thinnedVar], regmult = tunedBeta, maxnet.formula(PAbin, swd[, thinnedVar], classes = tunedFC))
	px <- ENMeval::maxnet.predictRaster(mod, clim[[colnames(swd)]], type = 'cloglog', clamp = TRUE)
	names(px) <- currentPXname
	
	modList[['coastalInterior']] <- mod
	
	# projected to LGM
	pxLGM <- ENMeval::maxnet.predictRaster(mod, LGMclim[[colnames(swd)]], type = 'cloglog', clamp = TRUE)
	names(pxLGM) <- lgmPXname
	
	# project to future
	pxFuture <- ENMeval::maxnet.predictRaster(mod, futureClim[[colnames(swd)]], type = 'cloglog', clamp = TRUE)
	names(pxFuture) <- futureLXname	
		
	writeRaster(px, paste0(outputDir, currentPXname, '.tif'), NAflag = -9999, options = tifOptions, overwrite=TRUE)
	writeRaster(pxLGM, paste0(outputDir, lgmPXname, '.tif'), NAflag = -9999, options = tifOptions, overwrite=TRUE)
	writeRaster(pxFuture, paste0(outputDir, futureLXname, '.tif'), NAflag = -9999, options = tifOptions, overwrite=TRUE)
	
	rm(clade, varSelect1, currentPXname, lgmPXname, pxFuture, subOcc, PA, PAbin, thinnedVar, swd, px, pxLGM, mod)
	gc()


# --------------------------------------------
rm(clim, LGMclim, futureClim)

saveRDS(modList, 'modList.rds')

#############################################################################################
# -------------------------------------------------------------------------------------------
#############################################################################################
# LANDCOVER

# from script 4
vifVar <- c("CHELSA_bioclim_02", "CHELSA_bioclim_03", "CHELSA_bioclim_07", "CHELSA_bioclim_14", "CHELSA_bioclim_15", "CHELSA_bioclim_18", "CHELSA_bioclim_19", "climaticMoistureIndex", "embergerQ", "minTempWarmest", "PETDriestQuarter", "PETseasonality", "PETWettestQuarter", paste0("landcover", c(1,2,3,4,5,6,8,10,11)))


# Process climatic predictors

# load climate grids
clim <- list.files(climDir, full.names=T)
clim <- clim[ - grep('tmin|tmax|tmean|precip|solrad|monthlyPET', clim)] # remove monthly rasters
clim <- clim[ - grep('summer|winter', clim, ignore.case=TRUE)]
clim <- clim[ - grep('monthCountByTemp10', clim)] # this is a discrete variable which we won't use
clim <- clim[ - grep('aridityIndexThornthwaite', clim)] # this is redundant with climatic moisture index
clim <- stack(clim)

clim <- clim[[intersect(names(clim), vifVar)]]
names(clim)

# Coarsen from 1x1 km to 2x2 km
clim <- aggregate(clim, fact = 2)


landcover <- list.files(landcoverDir, full.names = TRUE)
landcover <- landcover[order(as.numeric(gsub('(.+)(full_class_)(\\d?\\d)\\.tif$', '\\3', landcover)))]
landcover <- stack(landcover)

names(landcover) <- gsub('consensus_full_class_', 'landcover', names(landcover))
landcover <- landcover[[intersect(names(landcover), vifVar)]]

landcover <- resample(landcover, clim)

for (i in 1:nlayers(landcover)) {
	landcover[[i]][which(is.na(values(clim[[1]])))] <- NA
}

clim <- addLayer(clim, landcover)

rm(landcover)


################################################################################
################################################################################
# CLADE MODELING - INCLUDING LANDCOVER

bg <- readRDS('data/SDM/PA.rds')[[1]]
bg <- coordinates(bg)
bg <- tail(bg, 100000)
rownames(bg) <- NULL

# drop any records that fall outside of climate data
e <- extract(clim[[c(1,nlayers(clim))]], bg)
if (anyNA(e)) {
	bg <- bg[which(complete.cases(e)),]
}

load('data/SDM/region_list.Rdata')


### Load tuning results
rockiesTuning <- readRDS('data/SDM/rockies_withLC_tuning_v2.rds')
pacificTuning <- readRDS('data/SDM/pacific_withLC_tuning_v2.rds')
interiorTuning <- readRDS('data/SDM/interior_withLC_tuning_v2.rds')
pacificInteriorTuning <- readRDS('data/SDM/pacificInterior_withLC_tuning_v2.rds')


#################################
# Rockies

	clade <- 'rockies'
	varSelect1 <- rockiesTuning
	currentPXname <- paste0(clade, '_current_aicVarSelect_withLC')
	

	subOcc <- gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), region_list$rockies)
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL

	# drop any records that fall outside of climate data
	e <- extract(clim[[c(1,nlayers(clim))]], subOcc)
	if (anyNA(e)) {
		subOcc <- subOcc[complete.cases(e),]
	}	
	
	PA <- rbind(subOcc, bg)
	PA <- SpatialPoints(PA, proj4string=CRS('+proj=longlat +datum=WGS84'))
	PAbin <- c(rep(1, nrow(subOcc)), rep(0, nrow(bg)))
	
	PA <- spTransform(PA, crs(clim))
	PA <- SpatialPointsDataFrame(PA, as.data.frame(PAbin))
	swd <- as.data.frame(extract(clim[[vifVar]], PA))
	
	# model tuned through backward variable selection, where each iteration was tuned via AICc
	thinnedVar <- varSelect1[[length(varSelect1)]][['inputVariables']]
	tunedBeta <- varSelect1[[length(varSelect1)]]$tuning[1, 'regMult']
	tunedFC <- tolower(varSelect1[[length(varSelect1)]]$tuning[1, 'fc'])
	mod <- maxnet(PAbin, swd[, thinnedVar], regmult = tunedBeta, maxnet.formula(PAbin, swd[, thinnedVar], classes = tunedFC))
	px <- ENMeval::maxnet.predictRaster(mod, clim[[colnames(swd)]], type = 'cloglog', clamp = TRUE)
	names(px) <- currentPXname
			
	writeRaster(px, paste0(outputDir, currentPXname, '.tif'), NAflag = -9999, options = tifOptions, overwrite=TRUE)
	
	rm(clade, varSelect1, currentPXname, subOcc, PA, PAbin, thinnedVar, swd, px, mod)
	gc()


#################################
# Pacific

	clade <- 'pacific'
	varSelect1 <- pacificTuning
	currentPXname <- paste0(clade, '_current_aicVarSelect_withLC')
	

	subOcc <- gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), region_list$coastal)
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL

	# drop any records that fall outside of climate data
	e <- extract(clim[[c(1,nlayers(clim))]], subOcc)
	if (anyNA(e)) {
		subOcc <- subOcc[complete.cases(e),]
	}	
	
	PA <- rbind(subOcc, bg)
	PA <- SpatialPoints(PA, proj4string=CRS('+proj=longlat +datum=WGS84'))
	PAbin <- c(rep(1, nrow(subOcc)), rep(0, nrow(bg)))
	
	PA <- spTransform(PA, crs(clim))
	PA <- SpatialPointsDataFrame(PA, as.data.frame(PAbin))
	swd <- as.data.frame(extract(clim[[vifVar]], PA))
	
	# model tuned through backward variable selection, where each iteration was tuned via AICc
	thinnedVar <- varSelect1[[length(varSelect1)]][['inputVariables']]
	tunedBeta <- varSelect1[[length(varSelect1)]]$tuning[1, 'regMult']
	tunedFC <- tolower(varSelect1[[length(varSelect1)]]$tuning[1, 'fc'])
	mod <- maxnet(PAbin, swd[, thinnedVar], regmult = tunedBeta, maxnet.formula(PAbin, swd[, thinnedVar], classes = tunedFC))
	px <- ENMeval::maxnet.predictRaster(mod, clim[[colnames(swd)]], type = 'cloglog', clamp = TRUE)
	names(px) <- currentPXname
	
		
	writeRaster(px, paste0(outputDir, currentPXname, '.tif'), NAflag = -9999, options = tifOptions, overwrite=TRUE)

	rm(clade, varSelect1, currentPXname, subOcc, PA, PAbin, thinnedVar, swd, px, mod)	
	gc()

#################################
# Interior

	clade <- 'interior'
	varSelect1 <- interiorTuning
	currentPXname <- paste0(clade, '_current_aicVarSelect_withLC')
	

	subOcc <- gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), region_list$interior)
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL

	# drop any records that fall outside of climate data
	e <- extract(clim[[c(1,nlayers(clim))]], subOcc)
	if (anyNA(e)) {
		subOcc <- subOcc[complete.cases(e),]
	}	
	
	PA <- rbind(subOcc, bg)
	PA <- SpatialPoints(PA, proj4string=CRS('+proj=longlat +datum=WGS84'))
	PAbin <- c(rep(1, nrow(subOcc)), rep(0, nrow(bg)))
	
	PA <- spTransform(PA, crs(clim))
	PA <- SpatialPointsDataFrame(PA, as.data.frame(PAbin))
	swd <- as.data.frame(extract(clim[[vifVar]], PA))
	
	# model tuned through backward variable selection, where each iteration was tuned via AICc
	thinnedVar <- varSelect1[[length(varSelect1)]][['inputVariables']]
	tunedBeta <- varSelect1[[length(varSelect1)]]$tuning[1, 'regMult']
	tunedFC <- tolower(varSelect1[[length(varSelect1)]]$tuning[1, 'fc'])
	mod <- maxnet(PAbin, swd[, thinnedVar], regmult = tunedBeta, maxnet.formula(PAbin, swd[, thinnedVar], classes = tunedFC))
	px <- ENMeval::maxnet.predictRaster(mod, clim[[colnames(swd)]], type = 'cloglog', clamp = TRUE)
	names(px) <- currentPXname
	
		
	writeRaster(px, paste0(outputDir, currentPXname, '.tif'), NAflag = -9999, options = tifOptions, overwrite=TRUE)
	
	rm(clade, varSelect1, currentPXname, subOcc, PA, PAbin, thinnedVar, swd, px, mod)
	gc()

#################################
# Pacific+Interior

	clade <- 'pacificInterior'
	varSelect1 <- pacificInteriorTuning
	currentPXname <- paste0(clade, '_current_aicVarSelect_withLC')
	
	subOcc <- gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), gUnion(region_list$interior, region_list$coastal))
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL

	# drop any records that fall outside of climate data
	e <- extract(clim[[c(1,nlayers(clim))]], subOcc)
	if (anyNA(e)) {
		subOcc <- subOcc[complete.cases(e),]
	}	
	
	PA <- rbind(subOcc, bg)
	PA <- SpatialPoints(PA, proj4string=CRS('+proj=longlat +datum=WGS84'))
	PAbin <- c(rep(1, nrow(subOcc)), rep(0, nrow(bg)))
	
	PA <- spTransform(PA, crs(clim))
	PA <- SpatialPointsDataFrame(PA, as.data.frame(PAbin))
	swd <- as.data.frame(extract(clim[[vifVar]], PA))
	
	# model tuned through backward variable selection, where each iteration was tuned via AICc
	thinnedVar <- varSelect1[[length(varSelect1)]][['inputVariables']]
	tunedBeta <- varSelect1[[length(varSelect1)]]$tuning[1, 'regMult']
	tunedFC <- tolower(varSelect1[[length(varSelect1)]]$tuning[1, 'fc'])
	mod <- maxnet(PAbin, swd[, thinnedVar], regmult = tunedBeta, maxnet.formula(PAbin, swd[, thinnedVar], classes = tunedFC))
	px <- ENMeval::maxnet.predictRaster(mod, clim[[colnames(swd)]], type = 'cloglog', clamp = TRUE)
	names(px) <- currentPXname
	
		
	writeRaster(px, paste0(outputDir, currentPXname, '.tif'), NAflag = -9999, options = tifOptions, overwrite=TRUE)
	
	rm(clade, varSelect1, currentPXname, subOcc, PA, PAbin, thinnedVar, swd, px, mod)
	gc()


# --------------------------------------------







