##############################################
##############################################
# Separate SDM's for separate clades
##############################################

# require(ENMeval) # for maxent variable importance
library(rJava) # needed for maxent
library(maptools)
library(rgeos)
library(rgdal)
library(raster)
library(dismo) # for maxent called by R
library(maxnet)
library(enmSdm) # for sampling pseudo-absences, and Boyce index
library(viridis) # for color ramp
library(blockCV)
library(foreach)
library(doParallel)

source('scripts/SDM/maxentTuningFxn.R')

# change temporary directory so I can more actively manage it
tmpPath <- '~/maxentTemp'

if (!dir.exists(tmpPath)) {
	dir.create(tmpPath)
}

tempdir()
unixtools::set.tempdir(tmpPath)

climDir <- 'data/SDM/env/chelsa/current'
landcoverDir <- 'data/SDM/env/earthEnvLandCover'

# read in species range
jayRange <- readOGR('utility/Cyanocitta_stelleri_shp/Cyanocitta_stelleri.shp')
jayRange <- gSimplify(jayRange, tol = 0.1, topologyPreserve=T)


blueRamp <- colorRampPalette(c('gray97', 'blue'))

# convenience function for printing list of predictors
varPrint <- function(set) {
	cat('\n')
	for (i in 1:length(set)) cat('\t', set[i], '\n')
	cat('\n')
}

# ----------------------------------------------------------------
# Processing occurrence records
## Occurrence filtering done in script prepOcc.R

# load occurrences
allOcc <- readRDS('data/SDM/occ5.rds')
allOcc <- allOcc[, c('decimalLongitude','decimalLatitude')]
colnames(allOcc) <- c('Longitude','Latitude')
# -----------------------------------------------------------------
# Process climatic predictors

# from script 4
vifVar <- c("CHELSA_bioclim_02", "CHELSA_bioclim_07", "CHELSA_bioclim_14", "CHELSA_bioclim_15", "CHELSA_bioclim_18", "CHELSA_bioclim_19", "climaticMoistureIndex", "embergerQ", "minTempWarmest", "PETDriestQuarter", "PETseasonality", "PETWettestQuarter")



# load climate grids
clim <- list.files(climDir, full.names=T)
clim <- clim[ - grep('tmin|tmax|tmean|precip|solrad|monthlyPET', clim)] # remove monthly rasters
clim <- clim[ - grep('summer|winter', clim, ignore.case=TRUE)]
clim <- clim[ - grep('monthCountByTemp10', clim)] # this is a discrete variable which we won't use
clim <- clim[ - grep('aridityIndexThornthwaite', clim)] # this is redundant with climatic moisture index
clim <- stack(clim)
clim <- clim[[vifVar]]

names(clim)

# Coarsen from 1x1 km to 2x2 km
clim <- aggregate(clim, fact = 2)

bg <- readRDS('data/PA.rds')[[1]]
bg <- coordinates(bg)
bg <- tail(bg, 100000)
rownames(bg) <- NULL

# drop any records that fall outside of climate data
e <- extract(clim[[c(1,nlayers(clim))]], bg)
if (anyNA(e)) {
	bg <- bg[which(complete.cases(e)),]
}


# To ease comparison of different populations, we will use the same set of predictors, rather than reduce through population-specific tuning. 

load('data/SDM/region_list.Rdata')

rmVals <- seq(1, 12, by=0.5)
featureClasses <- c(c('L','Q','H'))
featureClasses <- unlist(lapply(1:length(featureClasses), function(x) combn(featureClasses, x, FUN=paste, collapse='')))

# number of tuning combinations
length(rmVals) * length(featureClasses)

nCores <- 10

#################################
# Rockies

	clade <- 'rockies'

	subOcc <- gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), region_list$rockies)
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL

	# drop any records that fall outside of climate data
	e <- extract(clim[[c(1,nlayers(clim))]], subOcc)
	if (anyNA(e)) {
		subOcc <- subOcc[complete.cases(e),]
	}	
	
	PA <- rbind(subOcc, bg)
	PA <- SpatialPoints(PA, proj4string=CRS('+proj=longlat +datum=WGS84'))
	PAbin <- c(rep(1, nrow(subOcc)), rep(0, nrow(bg)))
	
	PA <- spTransform(PA, crs(clim))
	PA <- SpatialPointsDataFrame(PA, as.data.frame(PAbin))
	
	##########################################################
	# Generate blocks for cross-validation
	PAdf <- cbind.data.frame(bin = PAbin, coordinates(PA))
	PAdf <- sf::st_as_sf(PAdf, coords = c("Longitude", "Latitude"), crs = crs(PA))
	
	co1 <- ncf::correlog(subOcc[,1], subOcc[,2], raster::extract(clim[[1]], subOcc),increment = 1, resamp = 0, latlon = T, na.rm=TRUE)
	zeroSA <- co1$x.intercept # this is in km
	
	blocks <- spatialBlock(PAdf, rasterLayer = clim[[1]], theRange = zeroSA*1000, k=5, biomod2Format=F, showBlocks=FALSE)
	
	# if eval blocks from blockCV package, reorganize to structure used by ENMeval::get.block
	if (inherits(blocks, 'SpatialBlock')) {
		folds <- list(occ.grp = numeric(length(which(PAbin == 1))), bg.grp = numeric(length(which(PAbin == 0))))
		for (i in 1:max(blocks$foldID)) {
			blockPres <- intersect(blocks$folds[[i]][[2]], which(PAbin == 1))
			blockBG <- intersect(blocks$folds[[i]][[2]], which(PAbin == 0)) - length(which(PAbin == 1))
			folds[[1]][blockPres] <- i
			folds[[2]][blockBG] <- i
		}
	} else {
		folds <- blocks
	}
	
	nfolds <- max(folds[[1]])
	
	##########################################################
		
	# Maxnet tuning
	swd <- as.data.frame(extract(clim[[vifVar]], PA))

	# variable selection and tuning via AICc
	varSelect1 <- backwardVarSelect(rmVals, featureClasses, swd, PAbin, varThresh = 1, verbose = TRUE, cores = nCores, addSamplesToBG = FALSE)

	saveRDS(varSelect1, paste0('data/SDM/', clade, '_tuning_v2.rds'))



##########################################################
##########################################################
# Pacific

	clade <- 'pacific'

	subOcc <- gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), region_list$coastal)
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL

	# drop any records that fall outside of climate data
	e <- extract(clim[[c(1,nlayers(clim))]], subOcc)
	if (anyNA(e)) {
		subOcc <- subOcc[complete.cases(e),]
	}

	PA <- rbind(subOcc, bg)
	PA <- SpatialPoints(PA, proj4string=CRS('+proj=longlat +datum=WGS84'))
	PAbin <- c(rep(1, nrow(subOcc)), rep(0, nrow(bg)))
	
	PA <- spTransform(PA, crs(clim))
	PA <- SpatialPointsDataFrame(PA, as.data.frame(PAbin))
	
	##########################################################
	# Generate blocks for cross-validation
	PAdf <- cbind.data.frame(bin = PAbin, coordinates(PA))
	PAdf <- sf::st_as_sf(PAdf, coords = c("Longitude", "Latitude"), crs = crs(PA))
	
	co1 <- ncf::correlog(subOcc[,1], subOcc[,2], raster::extract(clim[[1]], subOcc),increment = 1, resamp = 0, latlon = T, na.rm=TRUE)
	zeroSA <- co1$x.intercept # this is in km
	
	blocks <- spatialBlock(PAdf, rasterLayer = clim[[1]], theRange = zeroSA*1000, k=5, biomod2Format=F, showBlocks=FALSE)
	
	# if eval blocks from blockCV package, reorganize to structure used by ENMeval::get.block
	if (inherits(blocks, 'SpatialBlock')) {
		folds <- list(occ.grp = numeric(length(which(PAbin == 1))), bg.grp = numeric(length(which(PAbin == 0))))
		for (i in 1:max(blocks$foldID)) {
			blockPres <- intersect(blocks$folds[[i]][[2]], which(PAbin == 1))
			blockBG <- intersect(blocks$folds[[i]][[2]], which(PAbin == 0)) - length(which(PAbin == 1))
			folds[[1]][blockPres] <- i
			folds[[2]][blockBG] <- i
		}
	} else {
		folds <- blocks
	}
	
	nfolds <- max(folds[[1]])
	
	##########################################################
		
	# Maxnet tuning
	swd <- as.data.frame(extract(clim[[vifVar]], PA))

	# variable selection and tuning via AICc
	varSelect1 <- backwardVarSelect(rmVals, featureClasses, swd, PAbin, varThresh = 1, verbose = TRUE, cores = nCores, addSamplesToBG = FALSE)

	saveRDS(varSelect1, paste0('data/SDM/', clade, '_tuning_v2.rds'))





##########################################################
##########################################################
# Interior

	clade <- 'interior'

	subOcc <- gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), region_list$interior)
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL

	# drop any records that fall outside of climate data
	e <- extract(clim[[c(1,nlayers(clim))]], subOcc)
	if (anyNA(e)) {
		subOcc <- subOcc[complete.cases(e),]
	}	
	
	PA <- rbind(subOcc, bg)
	PA <- SpatialPoints(PA, proj4string=CRS('+proj=longlat +datum=WGS84'))
	PAbin <- c(rep(1, nrow(subOcc)), rep(0, nrow(bg)))
	
	PA <- spTransform(PA, crs(clim))
	PA <- SpatialPointsDataFrame(PA, as.data.frame(PAbin))
	
	##########################################################
	# Generate blocks for cross-validation
	PAdf <- cbind.data.frame(bin = PAbin, coordinates(PA))
	PAdf <- sf::st_as_sf(PAdf, coords = c("Longitude", "Latitude"), crs = crs(PA))
	
	co1 <- ncf::correlog(subOcc[,1], subOcc[,2], raster::extract(clim[[1]], subOcc),increment = 1, resamp = 0, latlon = T, na.rm=TRUE)
	zeroSA <- co1$x.intercept # this is in km
	
	blocks <- spatialBlock(PAdf, rasterLayer = clim[[1]], theRange = zeroSA*1000, k=5, biomod2Format=F, showBlocks=FALSE)
	
	# if eval blocks from blockCV package, reorganize to structure used by ENMeval::get.block
	if (inherits(blocks, 'SpatialBlock')) {
		folds <- list(occ.grp = numeric(length(which(PAbin == 1))), bg.grp = numeric(length(which(PAbin == 0))))
		for (i in 1:max(blocks$foldID)) {
			blockPres <- intersect(blocks$folds[[i]][[2]], which(PAbin == 1))
			blockBG <- intersect(blocks$folds[[i]][[2]], which(PAbin == 0)) - length(which(PAbin == 1))
			folds[[1]][blockPres] <- i
			folds[[2]][blockBG] <- i
		}
	} else {
		folds <- blocks
	}
	
	nfolds <- max(folds[[1]])
	
	##########################################################
		
	# Maxnet tuning
	swd <- as.data.frame(extract(clim[[vifVar]], PA))

	# variable selection and tuning via AICc
	varSelect1 <- backwardVarSelect(rmVals, featureClasses, swd, PAbin, varThresh = 1, verbose = TRUE, cores = nCores, addSamplesToBG = FALSE)

	saveRDS(varSelect1, paste0('data/SDM/', clade, '_tuning_v2.rds'))





##########################################################
##########################################################
# Pacific+Interior

	clade <- 'pacificInterior'

	subOcc <- gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), gUnion(region_list$interior, region_list$coastal))
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL

	# drop any records that fall outside of climate data
	e <- extract(clim[[c(1,nlayers(clim))]], subOcc)
	if (anyNA(e)) {
		subOcc <- subOcc[complete.cases(e),]
	}	
	
	PA <- rbind(subOcc, bg)
	PA <- SpatialPoints(PA, proj4string=CRS('+proj=longlat +datum=WGS84'))
	PAbin <- c(rep(1, nrow(subOcc)), rep(0, nrow(bg)))
	
	PA <- spTransform(PA, crs(clim))
	PA <- SpatialPointsDataFrame(PA, as.data.frame(PAbin))
	
	##########################################################
	# Generate blocks for cross-validation
	PAdf <- cbind.data.frame(bin = PAbin, coordinates(PA))
	PAdf <- sf::st_as_sf(PAdf, coords = c("Longitude", "Latitude"), crs = crs(PA))
	
	co1 <- ncf::correlog(subOcc[,1], subOcc[,2], raster::extract(clim[[1]], subOcc),increment = 1, resamp = 0, latlon = T, na.rm=TRUE)
	zeroSA <- co1$x.intercept # this is in km
	
	blocks <- spatialBlock(PAdf, rasterLayer = clim[[1]], theRange = zeroSA*1000, k=5, biomod2Format=F, showBlocks=FALSE)
	
	# if eval blocks from blockCV package, reorganize to structure used by ENMeval::get.block
	if (inherits(blocks, 'SpatialBlock')) {
		folds <- list(occ.grp = numeric(length(which(PAbin == 1))), bg.grp = numeric(length(which(PAbin == 0))))
		for (i in 1:max(blocks$foldID)) {
			blockPres <- intersect(blocks$folds[[i]][[2]], which(PAbin == 1))
			blockBG <- intersect(blocks$folds[[i]][[2]], which(PAbin == 0)) - length(which(PAbin == 1))
			folds[[1]][blockPres] <- i
			folds[[2]][blockBG] <- i
		}
	} else {
		folds <- blocks
	}
	
	nfolds <- max(folds[[1]])
	
	##########################################################
		
	# Maxnet tuning
	swd <- as.data.frame(extract(clim[[vifVar]], PA))

	# variable selection and tuning via AICc
	varSelect1 <- backwardVarSelect(rmVals, featureClasses, swd, PAbin, varThresh = 1, verbose = TRUE, cores = nCores, addSamplesToBG = FALSE)

	saveRDS(varSelect1, paste0('data/SDM/', clade, '_tuning_v2.rds'))





#############################################################################################
# -------------------------------------------------------------------------------------------
#############################################################################################
# LANDCOVER

# from script 4
vifVar <- c("CHELSA_bioclim_02", "CHELSA_bioclim_03", "CHELSA_bioclim_07", "CHELSA_bioclim_14", "CHELSA_bioclim_15", "CHELSA_bioclim_18", "CHELSA_bioclim_19", "climaticMoistureIndex", "embergerQ", "minTempWarmest", "PETDriestQuarter", "PETseasonality", "PETWettestQuarter", paste0("landcover", c(1,2,3,4,5,6,8,10,11)))


# Process climatic predictors

# load climate grids
clim <- list.files(climDir, full.names=T)
clim <- clim[ - grep('tmin|tmax|tmean|precip|solrad|monthlyPET', clim)] # remove monthly rasters
clim <- clim[ - grep('summer|winter', clim, ignore.case=TRUE)]
clim <- clim[ - grep('monthCountByTemp10', clim)] # this is a discrete variable which we won't use
clim <- clim[ - grep('aridityIndexThornthwaite', clim)] # this is redundant with climatic moisture index
clim <- stack(clim)

clim <- clim[[intersect(names(clim), vifVar)]]
names(clim)

# Coarsen from 1x1 km to 2x2 km
clim <- aggregate(clim, fact = 2)


landcover <- list.files(landcoverDir, full.names = TRUE)
landcover <- landcover[order(as.numeric(gsub('(.+)(full_class_)(\\d?\\d)\\.tif$', '\\3', landcover)))]
landcover <- stack(landcover)

names(landcover) <- gsub('consensus_full_class_', 'landcover', names(landcover))
landcover <- landcover[[intersect(names(landcover), vifVar)]]

landcover <- resample(landcover, clim)

for (i in 1:nlayers(landcover)) {
	landcover[[i]][which(is.na(values(clim[[1]])))] <- NA
}

clim <- addLayer(clim, landcover)


bg <- readRDS('data/PA.rds')[[1]]
bg <- coordinates(bg)
bg <- tail(bg, 100000)
rownames(bg) <- NULL


# drop any records that fall outside of climate data
e <- extract(clim[[c(1,nlayers(clim))]], bg)
if (anyNA(e)) {
	bg <- bg[which(!is.na(e)),]
}



# To ease comparison of different populations, we will use the same set of predictors, rather than reduce through population-specific tuning. 

load('data/SDM/region_list.Rdata')

rmVals <- seq(1, 12, by=0.5)
featureClasses <- c(c('L','Q','H'))
featureClasses <- unlist(lapply(1:length(featureClasses), function(x) combn(featureClasses, x, FUN=paste, collapse='')))

# number of tuning combinations
length(rmVals) * length(featureClasses)

#################################
# Rockies

	clade <- 'rockies'

	subOcc <- gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), region_list$rockies)
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL

	# drop any records that fall outside of climate data
	e <- extract(clim[[c(1,nlayers(clim))]], subOcc)
	if (anyNA(e)) {
		subOcc <- subOcc[complete.cases(e),]
	}	
	
	PA <- rbind(subOcc, bg)
	PA <- SpatialPoints(PA, proj4string=CRS('+proj=longlat +datum=WGS84'))
	PAbin <- c(rep(1, nrow(subOcc)), rep(0, nrow(bg)))
	
	PA <- spTransform(PA, crs(clim))
	PA <- SpatialPointsDataFrame(PA, as.data.frame(PAbin))
	
	##########################################################
	# Generate blocks for cross-validation
	PAdf <- cbind.data.frame(bin = PAbin, coordinates(PA))
	PAdf <- sf::st_as_sf(PAdf, coords = c("Longitude", "Latitude"), crs = crs(PA))
	
	co1 <- ncf::correlog(subOcc[,1], subOcc[,2], raster::extract(clim[[1]], subOcc),increment = 1, resamp = 0, latlon = T, na.rm=TRUE)
	zeroSA <- co1$x.intercept # this is in km
	
	blocks <- spatialBlock(PAdf, rasterLayer = clim[[1]], theRange = zeroSA*1000, k=5, biomod2Format=F, showBlocks=FALSE)
	
	# if eval blocks from blockCV package, reorganize to structure used by ENMeval::get.block
	if (inherits(blocks, 'SpatialBlock')) {
		folds <- list(occ.grp = numeric(length(which(PAbin == 1))), bg.grp = numeric(length(which(PAbin == 0))))
		for (i in 1:max(blocks$foldID)) {
			blockPres <- intersect(blocks$folds[[i]][[2]], which(PAbin == 1))
			blockBG <- intersect(blocks$folds[[i]][[2]], which(PAbin == 0)) - length(which(PAbin == 1))
			folds[[1]][blockPres] <- i
			folds[[2]][blockBG] <- i
		}
	} else {
		folds <- blocks
	}
	
	nfolds <- max(folds[[1]])
	
	##########################################################
		
	# Maxnet tuning
	swd <- as.data.frame(extract(clim[[vifVar]], PA))

	# variable selection and tuning via AICc
	varSelect1 <- backwardVarSelect(rmVals, featureClasses, swd, PAbin, varThresh = 1, verbose = TRUE, cores = nCores, addSamplesToBG = FALSE)

	saveRDS(varSelect1, paste0('data/SDM/', clade, '_withLC_tuning_v2.rds'))



##########################################################
##########################################################
# Pacific

	clade <- 'pacific'

	subOcc <- gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), region_list$coastal)
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL

	# drop any records that fall outside of climate data
	e <- extract(clim[[c(1,nlayers(clim))]], subOcc)
	if (anyNA(e)) {
		subOcc <- subOcc[complete.cases(e),]
	}	
	
	PA <- rbind(subOcc, bg)
	PA <- SpatialPoints(PA, proj4string=CRS('+proj=longlat +datum=WGS84'))
	PAbin <- c(rep(1, nrow(subOcc)), rep(0, nrow(bg)))
	
	PA <- spTransform(PA, crs(clim))
	PA <- SpatialPointsDataFrame(PA, as.data.frame(PAbin))
	
	##########################################################
	# Generate blocks for cross-validation
	PAdf <- cbind.data.frame(bin = PAbin, coordinates(PA))
	PAdf <- sf::st_as_sf(PAdf, coords = c("Longitude", "Latitude"), crs = crs(PA))
	
	co1 <- ncf::correlog(subOcc[,1], subOcc[,2], raster::extract(clim[[1]], subOcc),increment = 1, resamp = 0, latlon = T, na.rm=TRUE)
	zeroSA <- co1$x.intercept # this is in km
	
	blocks <- spatialBlock(PAdf, rasterLayer = clim[[1]], theRange = zeroSA*1000, k=5, biomod2Format=F, showBlocks=FALSE)
	
	# if eval blocks from blockCV package, reorganize to structure used by ENMeval::get.block
	if (inherits(blocks, 'SpatialBlock')) {
		folds <- list(occ.grp = numeric(length(which(PAbin == 1))), bg.grp = numeric(length(which(PAbin == 0))))
		for (i in 1:max(blocks$foldID)) {
			blockPres <- intersect(blocks$folds[[i]][[2]], which(PAbin == 1))
			blockBG <- intersect(blocks$folds[[i]][[2]], which(PAbin == 0)) - length(which(PAbin == 1))
			folds[[1]][blockPres] <- i
			folds[[2]][blockBG] <- i
		}
	} else {
		folds <- blocks
	}
	
	nfolds <- max(folds[[1]])
	
	##########################################################
		
	# Maxnet tuning
	swd <- as.data.frame(extract(clim[[vifVar]], PA))

	# variable selection and tuning via AICc
	varSelect1 <- backwardVarSelect(rmVals, featureClasses, swd, PAbin, varThresh = 1, verbose = TRUE, cores = nCores, addSamplesToBG = FALSE)

	saveRDS(varSelect1, paste0('data/SDM/', clade, '_withLC_tuning_v2.rds'))





##########################################################
##########################################################
# Interior

	clade <- 'interior'

	subOcc <- gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), region_list$interior)
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL

	# drop any records that fall outside of climate data
	e <- extract(clim[[c(1,nlayers(clim))]], subOcc)
	if (anyNA(e)) {
		subOcc <- subOcc[complete.cases(e),]
	}
	
	PA <- rbind(subOcc, bg)
	PA <- SpatialPoints(PA, proj4string=CRS('+proj=longlat +datum=WGS84'))
	PAbin <- c(rep(1, nrow(subOcc)), rep(0, nrow(bg)))
	
	PA <- spTransform(PA, crs(clim))
	PA <- SpatialPointsDataFrame(PA, as.data.frame(PAbin))
	
	##########################################################
	# Generate blocks for cross-validation
	PAdf <- cbind.data.frame(bin = PAbin, coordinates(PA))
	PAdf <- sf::st_as_sf(PAdf, coords = c("Longitude", "Latitude"), crs = crs(PA))
	
	co1 <- ncf::correlog(subOcc[,1], subOcc[,2], raster::extract(clim[[1]], subOcc),increment = 1, resamp = 0, latlon = T, na.rm=TRUE)
	zeroSA <- co1$x.intercept # this is in km
	
	blocks <- spatialBlock(PAdf, rasterLayer = clim[[1]], theRange = zeroSA*1000, k=5, biomod2Format=F, showBlocks=FALSE)
	
	# if eval blocks from blockCV package, reorganize to structure used by ENMeval::get.block
	if (inherits(blocks, 'SpatialBlock')) {
		folds <- list(occ.grp = numeric(length(which(PAbin == 1))), bg.grp = numeric(length(which(PAbin == 0))))
		for (i in 1:max(blocks$foldID)) {
			blockPres <- intersect(blocks$folds[[i]][[2]], which(PAbin == 1))
			blockBG <- intersect(blocks$folds[[i]][[2]], which(PAbin == 0)) - length(which(PAbin == 1))
			folds[[1]][blockPres] <- i
			folds[[2]][blockBG] <- i
		}
	} else {
		folds <- blocks
	}
	
	nfolds <- max(folds[[1]])
	
	##########################################################
		
	# Maxnet tuning
	swd <- as.data.frame(extract(clim[[vifVar]], PA))

	# variable selection and tuning via AICc
	varSelect1 <- backwardVarSelect(rmVals, featureClasses, swd, PAbin, varThresh = 1, verbose = TRUE, cores = nCores, addSamplesToBG = FALSE)

	saveRDS(varSelect1, paste0('data/SDM/', clade, '_withLC_tuning_v2.rds'))





##########################################################
##########################################################
# Pacific+Interior

	clade <- 'pacificInterior'

	subOcc <- gIntersection(SpatialPoints(allOcc, proj4string=CRS(proj4string(region_list$rockies))), gUnion(region_list$interior, region_list$coastal))
	subOcc <- coordinates(subOcc)
	colnames(subOcc) <- c('Longitude','Latitude')
	rownames(subOcc) <- NULL

	# drop any records that fall outside of climate data
	e <- extract(clim[[c(1,nlayers(clim))]], subOcc)
	if (anyNA(e)) {
		subOcc <- subOcc[complete.cases(e),]
	}
	
	PA <- rbind(subOcc, bg)
	PA <- SpatialPoints(PA, proj4string=CRS('+proj=longlat +datum=WGS84'))
	PAbin <- c(rep(1, nrow(subOcc)), rep(0, nrow(bg)))
	
	PA <- spTransform(PA, crs(clim))
	PA <- SpatialPointsDataFrame(PA, as.data.frame(PAbin))
	
	##########################################################
	# Generate blocks for cross-validation
	PAdf <- cbind.data.frame(bin = PAbin, coordinates(PA))
	PAdf <- sf::st_as_sf(PAdf, coords = c("Longitude", "Latitude"), crs = crs(PA))
	
	co1 <- ncf::correlog(subOcc[,1], subOcc[,2], raster::extract(clim[[1]], subOcc),increment = 1, resamp = 0, latlon = T, na.rm=TRUE)
	zeroSA <- co1$x.intercept # this is in km
	
	blocks <- spatialBlock(PAdf, rasterLayer = clim[[1]], theRange = zeroSA*1000, k=5, biomod2Format=F, showBlocks=FALSE)
	
	# if eval blocks from blockCV package, reorganize to structure used by ENMeval::get.block
	if (inherits(blocks, 'SpatialBlock')) {
		folds <- list(occ.grp = numeric(length(which(PAbin == 1))), bg.grp = numeric(length(which(PAbin == 0))))
		for (i in 1:max(blocks$foldID)) {
			blockPres <- intersect(blocks$folds[[i]][[2]], which(PAbin == 1))
			blockBG <- intersect(blocks$folds[[i]][[2]], which(PAbin == 0)) - length(which(PAbin == 1))
			folds[[1]][blockPres] <- i
			folds[[2]][blockBG] <- i
		}
	} else {
		folds <- blocks
	}
	
	nfolds <- max(folds[[1]])
	
	##########################################################
		
	# Maxnet tuning
	swd <- as.data.frame(extract(clim[[vifVar]], PA))

	# variable selection and tuning via AICc
	varSelect1 <- backwardVarSelect(rmVals, featureClasses, swd, PAbin, varThresh = 1, verbose = TRUE, cores = nCores, addSamplesToBG = FALSE)

	saveRDS(varSelect1, paste0('data/SDM/', clade, '_withLC_tuning_v2.rds'))





