# for 3 anthropogenic datasets (populated places, urban areas, roads), 
# given a raster grid, calculate the distance from each grid cell to the closest anthropogenic 
# feature. For instance, with the roads dataset, this will return a raster where each 
# grid cell value is the distance to the nearest road. 

library(Rfast)
library(raster)
library(rgdal)
library(maptools)
library(rgeos)


template <- raster('data/SDM/chelsa/current/annualPET.tif')


pop <- readOGR('data/SDM/env/anthropo/ne_10m_populated_places/ne_10m_populated_places.shp')

urban <- readOGR('data/SDM/env/anthropo/ne_10m_urban_areas/ne_10m_urban_areas.shp')

roads <- readOGR('data/SDM/env/anthropo/ne_10m_roads/ne_10m_roads.shp')

protected <- readOGR('data/SDM/env/anthropo/ne_10m_parks_and_protected_lands/ne_10m_parks_and_protected_lands_area.shp')
protected2 <- readOGR('data/SDM/env/anthropo/ne_10m_parks_and_protected_lands/ne_10m_parks_and_protected_lands_point.shp')
protected3 <- readOGR('data/SDM/env/anthropo/ne_10m_parks_and_protected_lands/ne_10m_parks_and_protected_lands_line.shp')

data(wrld_simpl)

# clip to extent

pop <- crop(pop, template)
urban <- crop(urban, template)
roads <- crop(roads, template)
protected <- crop(protected, template)
protected2 <- crop(protected2, template)
protected3 <- crop(protected3, template)

protectedRasterized1 <- rasterize(protected, template)
protectedRasterized2 <- rasterize(protected2, template)
protectedRasterized3 <- rasterize(protected3, template)

protectedRasterized4 <- calc(stack(protectedRasterized1, protectedRasterized2, protectedRasterized3), fun=sum, na.rm=TRUE)

protectedRasterized <- protectedRasterized4
rm(protectedRasterized1)
rm(protectedRasterized2)
rm(protectedRasterized3)
rm(protectedRasterized4)
protectedRasterized[protectedRasterized == 0] <- NA

roadsRasterized <- rasterize(roads, template)
popRasterized <- rasterize(pop, template, field = 'NAME', fun='count')
urbanRasterized <- rasterize(urban, template)


# ras: the raster that distances will be calculated from. This raster should have NA cells, except where the features are. 
# nTiles: a perfect square integer, raster will be split into this many tiles
# chunkSize: Distances are calculated from each raster point to X destination points at a time. 
# template is a raster that will be used to include/exclude cells for calculation. 

calcDistanceRaster <- function(ras, nTiles, chunkSize = 1000, template) {
	
	library(Rfast)
	library(raster)
	library(rgeos)	

	if (sqrt(nTiles) != round(sqrt(nTiles), 0)) {
		stop('Number of tiles must be a perfect square.')
	}

	s <- sqrt(nTiles)
	h <- ceiling(ncol(ras)/s)
	v <- ceiling(nrow(ras)/s)
	agg <- aggregate(ras,fact=c(h,v))
	agg[]    <- 1:ncell(agg)
	agg_poly <- rasterToPolygons(agg)
	
	# extract coordinates for all features
	featurePts <- rasterToPoints(ras)[,1:2]
	voronoi <- dismo::voronoi(featurePts, ext=extent(ras))
	proj4string(voronoi) <- crs(ras)
	
	distTiles <- vector('list', length = length(agg_poly))
	
	for (i in 1:length(agg_poly)) {
		
		cat('\tTile', i, 'of', length(agg_poly), '\n')
		tile <- crop(ras, agg_poly[i,])
		tile <- mask(tile, agg_poly[i,])
		
		ptsToCalc <- coordinates(tile)

		# keep only cells that have data in template
		# no need to calculate distances for them if they will just get masked later
		e <- extract(template, ptsToCalc)
		ptsToCalc <- ptsToCalc[which(!is.na(e)),]
		rm(e)
	
		if (nrow(ptsToCalc) > 0) {

			# We've calculated voronoi polygons based on the feature points, where each voronoi polygon provides the region where you would be closest to that feature point. Therefore, we just need to search within the voronoi polygons that intersect the convex hull of the calc points
			# This way, we can reduce the number of distances to test
			hull <- gConvexHull(SpatialPoints(ptsToCalc, proj4string=crs(ras)))
			voronoiSub <- gIntersects(voronoi, hull, byid=TRUE)
			voronoiSub <- voronoi[which(voronoiSub == TRUE),]
			featurePtsSp <- SpatialPoints(featurePts, proj4string=crs(ras))
			featurePtsSp <- gIntersection(featurePtsSp, voronoiSub)
			featurePtsSub <- coordinates(featurePtsSp)

			chunks <- split(1:nrow(featurePtsSub), ceiling(seq_along(1:nrow(featurePtsSub))/chunkSize))
		
			if (length(chunks) > 1) cat('\t\tProcessing in', length(chunks), 'chunks...\n')
		
			if (length(chunks) > 1) pb <- txtProgressBar(min=0, max = length(chunks))
			for (j in 1:length(chunks)) {
				
				if (length(chunks) > 1) setTxtProgressBar(pb, j)
				if (length(chunks[[j]]) > 1) {
					d1 <- dista(ptsToCalc, featurePtsSub[chunks[[j]],], type='euclidean', k=1)
				} else {
					d1 <- dista(ptsToCalc, featurePtsSub[c(chunks[[j]], chunks[[j]]),], type='euclidean', k=1)
				}
				d1 <- as.numeric(d1[,1])
		
				if (j == 1) {
					dmin <- d1
				} else {
					update <- d1 < dmin
					dmin[update] <- d1[update]
				}
		
				gc()
			}
			
			if (length(chunks) > 1) close(pb)

			ret <- raster(tile)
			ret[] <- NA
			ret[cellFromXY(ret, featurePts)] <- 0
			ret[cellFromXY(ret, ptsToCalc)] <- dmin
		} else {
			ret <- raster(tile)
			ret[] <- NA	
		}
	
		distTiles[[i]] <- ret
	}
	
	# reassemble
	mosaicList <- distTiles
	names(mosaicList)[1:2] <- c('x','y')
	mosaicList$fun <- min
	mosaicList.na.rm <- TRUE
	q <- do.call(mosaic, mosaicList)
	
	return(q)
}




popDistance <- calcDistanceRaster(popRasterized, nTiles = 25, chunkSize = 1000, template = template)
urbanDistance <- calcDistanceRaster(urbanRasterized, nTiles = 25, chunkSize = 1000, template = template)
protectedDistance <- calcDistanceRaster(protectedRasterized, nTiles = 25, chunkSize = 1000, template = template)
roadsDistance <- calcDistanceRaster(roadsRasterized, nTiles = 25, chunkSize = 1000, template = template)


names(popDistance) <- 'populatedPlacesDistance'
names(roadsDistance) <- 'roadsDistance'
names(urbanDistance) <- 'urbanDistance'
names(protectedDistance) <- 'protectedPlacesDistance'

tifOptions <- c("COMPRESS=DEFLATE", "PREDICTOR=2", "ZLEVEL=6")
dest <- 'data/SDM/env/anthropo/mosaics/'
writeRaster(roadsDistance, paste0(dest, names(roadsDistance), '.tif'), options=tifOptions)
writeRaster(protectedDistance, paste0(dest, names(protectedDistance), '.tif'), options=tifOptions)
writeRaster(urbanDistance, paste0(dest, names(urbanDistance), '.tif'), options=tifOptions)
writeRaster(popDistance, paste0(dest, names(popDistance), '.tif'), options=tifOptions)



