# Read in CHELSA monthly rasters
# Set units to deg C, mm
# Produce ENVIREM layers
# Generate 19 bioclim rasters (preference to generate them here so that they conform to units)
# write to disk

# CHELSA climate data precipitation is in kg m-2, which is equivalent to mm (since 1 L of water is close to 1 kg)

# UNITS in CHELSA:
# current: temp degC / 10, precip mm
# LGM: temp K / 10, precip mm / 10
# FUTURE: temp degC / 10, precip mm

library(envirem)
library(raster)
library(rgdal)
library(dismo)

climDir <- 'data/SDM/env/chelsa/current'
climDirLGM <- 'data/SDM/env/chelsa/lgm_ccsm4'
climDirFUTURE <- 'data/SDM/env/chelsa/future'

tifOptions <- c("COMPRESS=DEFLATE", "PREDICTOR=2", "ZLEVEL=6")


tmpPath <- '~/maxentTemp'

if (!dir.exists(tmpPath)) {
	dir.create(tmpPath)
}

tempdir()
unixtools::set.tempdir(tmpPath)

#################################################################
# current clim -- read in monthly rasters
setwd(climDir)

f <- list.files(pattern = '\\.tif$')

# identify min/max/mean temp and precipitation
	tmin <- grep('tmin', f, value = TRUE)
	tmax <- grep('tmax', f, value = TRUE)
	tmean <- grep('temp', f, value = TRUE)
	precip <- grep('prec', f, value = TRUE)

# fix ordering
	tmin <- tmin[order(as.numeric(gsub('(CHELSA_tmin_)(\\d\\d?)(_1979.2013_V1.2_land.tif)', '\\2', tmin)))]
	tmax <- tmax[order(as.numeric(gsub('(CHELSA_tmax_)(\\d\\d?)(_1979.2013_V1.2_land.tif)', '\\2', tmax)))]
	tmean <- tmean[order(as.numeric(gsub('(CHELSA_temp_)(\\d\\d?)(_1979.2013_V1.2_land.tif)', '\\2', tmean)))]
	precip <- precip[order(as.numeric(gsub('(CHELSA_prec_)(\\d\\d?)(_V1.2_land.tif)', '\\2', precip)))]

# project to equal area, adjust units, write

	for (i in 1:length(tmin)) {
		message('\t', i)
		tmp <- raster(tmin[i])

		tmp <- tmp / 10

		fn <- paste0(climDir, '/CHELSA_tmin_', ifelse(i < 10, paste0('0', i), i), '.tif')
		writeRaster(tmp, fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
	}
	gc()

	for (i in 1:length(tmax)) {
		message('\t', tmax[i])
		tmp <- raster(tmax[i])

		tmp <- tmp / 10

		fn <- paste0(climDir, '/CHELSA_tmax_', ifelse(i < 10, paste0('0', i), i), '.tif')
		writeRaster(tmp, fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
	}
	gc()

	for (i in 1:length(tmean)) {
		message('\t', tmean[i])
		tmp <- raster(tmean[i])

		tmp <- tmp / 10

		fn <- paste0(climDir, '/CHELSA_tmean_', ifelse(i < 10, paste0('0', i), i), '.tif')
		writeRaster(tmp, fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
	}
	gc()

	for (i in 1:length(precip)) {
		message('\t', precip[i])
		tmp <- raster(precip[i])

		fn <- paste0(climDir, '/CHELSA_precip_', ifelse(i < 10, paste0('0', i), i), '.tif')
		writeRaster(tmp, fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
	}
	gc()



#################################################################
# LGM clim -- read in monthly rasters
setwd(climDirLGM)

f <- list.files(pattern = '\\.tif$')

# identify min/max/mean temp and precipitation
	tmin <- grep('tmin', f, value = TRUE)
	tmax <- grep('tmax', f, value = TRUE)
	tmean <- grep('tmean', f, value = TRUE)
	precip <- grep('prec', f, value = TRUE)

# fix ordering
	tmin <- tmin[order(as.numeric(gsub('(CHELSA_PMIP_CCSM4_tmin_)(\\d\\d?)(_1.tif)', '\\2', tmin)))]
	tmax <- tmax[order(as.numeric(gsub('(CHELSA_PMIP_CCSM4_tmax_)(\\d\\d?)(_1.tif)', '\\2', tmax)))]
	tmean <- tmean[order(as.numeric(gsub('(CHELSA_PMIP_CCSM4_tmean_)(\\d\\d?)(_1.tif)', '\\2', tmean)))]
	precip <- precip[order(as.numeric(gsub('(CHELSA_PMIP_CCSM4_prec_)(\\d\\d?)(_1.tif)', '\\2', precip)))]

# project to equal area, adjust units, write

# For CHELSA LGM, "divide by 10 and add -273.15 to get degree Celsius"

	for (i in 1:length(tmin)) {
		message('\t', tmin[i])
		tmp <- raster(tmin[i])

		tmp <- tmp / 10 + -273.15

		fn <- paste0(climDirLGM, '/CHELSA_LGM_tmin_', ifelse(i < 10, paste0('0', i), i), '.tif')
		writeRaster(tmp, fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
	}
	gc()

	for (i in 1:length(tmax)) {
		message('\t', tmax[i])
		tmp <- raster(tmax[i])

		tmp <- tmp / 10 + -273.15

		fn <- paste0(climDirLGM, '/CHELSA_LGM_tmax_', ifelse(i < 10, paste0('0', i), i), '.tif')
		writeRaster(tmp, fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
	}
	gc()

	for (i in 1:length(tmean)) {
		message('\t', tmean[i])
		tmp <- raster(tmean[i])

		tmp <- tmp / 10 + -273.15

		fn <- paste0(climDirLGM, '/CHELSA_LGM_tmean_', ifelse(i < 10, paste0('0', i), i), '.tif')
		writeRaster(tmp, fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
	}
	gc()

	for (i in 1:length(precip)) {
		message('\t', precip[i])
		tmp <- raster(precip[i])
		
		tmp <- tmp / 10

		fn <- paste0(climDirLGM, '/CHELSA_LGM_precip_', ifelse(i < 10, paste0('0', i), i), '.tif')
		writeRaster(tmp, fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
	}
	gc()

#################################################################
# FUTURE clim -- read in monthly rasters
setwd(climDirFUTURE)

f <- list.files(pattern = '\\.tif$')
f <- grep('2061-2080', f, value = TRUE)

# identify min/max/mean temp and precipitation
	tmin <- grep('tasmin', f, value = TRUE)
	tmax <- grep('tasmax', f, value = TRUE)
	tmean <- grep('tas_', f, value = TRUE)
	precip <- grep('pr_', f, value = TRUE)

# fix ordering
	tmin <- tmin[order(as.numeric(gsub('(CHELSA_tasmin_mon_CCSM4_rcp45_r1i1p1_g025.nc_)(\\d\\d?)(_2061-2080_V1.2.tif)', '\\2', tmin)))]
	tmax <- tmax[order(as.numeric(gsub('(CHELSA_tasmax_mon_CCSM4_rcp45_r1i1p1_g025.nc_)(\\d\\d?)(_2061-2080_V1.2.tif)', '\\2', tmax)))]
	tmean <- tmean[order(as.numeric(gsub('(CHELSA_tas_mon_CCSM4_rcp45_r1i1p1_g025.nc_)(\\d\\d?)(_2061-2080_V1.2.tif)', '\\2', tmean)))]
	precip <- precip[order(as.numeric(gsub('(CHELSA_pr_mon_CCSM4_rcp45_r1i1p1_g025.nc_)(\\d\\d?)(_2061-2080.tif)', '\\2', precip)))]

# adjust units, write
	
# read template to match extent
	template <- raster(paste0(climDir, '/CHELSA_precip_01.tif'))

# For CHELSA PMIP5, temperature is in degC * 10, precip is in mm

	for (i in 1:length(tmin)) {
		message('\t', tmin[i])
		tmp <- raster(tmin[i])
		
		tmp <- crop(tmp, template)
		tmp <- mask(tmp, template)
		
		tmp <- tmp / 10

		fn <- paste0(climDirFUTURE, '/CHELSA_FUTURE2070_tmin_', ifelse(i < 10, paste0('0', i), i), '.tif')
		writeRaster(tmp, fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
	}
	gc()

	for (i in 1:length(tmax)) {
		message('\t', tmax[i])
		tmp <- raster(tmax[i])

		tmp <- crop(tmp, template)
		tmp <- mask(tmp, template)
		
		tmp <- tmp / 10

		fn <- paste0(climDirFUTURE, '/CHELSA_FUTURE2070_tmax_', ifelse(i < 10, paste0('0', i), i), '.tif')
		writeRaster(tmp, fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
	}
	gc()

	for (i in 1:length(tmean)) {
		message('\t', tmean[i])
		tmp <- raster(tmean[i])

		tmp <- crop(tmp, template)
		tmp <- mask(tmp, template)
		
		tmp <- tmp / 10

		fn <- paste0(climDirFUTURE, '/CHELSA_FUTURE2070_tmean_', ifelse(i < 10, paste0('0', i), i), '.tif')
		writeRaster(tmp, fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
	}
	gc()

	for (i in 1:length(precip)) {
		message('\t', precip[i])
		tmp <- raster(precip[i])

		tmp <- crop(tmp, template)
		tmp <- mask(tmp, template)
				
		fn <- paste0(climDirFUTURE, '/CHELSA_FUTURE2070_precip_', ifelse(i < 10, paste0('0', i), i), '.tif')
		writeRaster(tmp, fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
	}
	gc()

#################################################################
# ENVIREM CURRENT CLIM

# assign naming scheme
varnames()

assignNames(	tmax = "CHELSA_tmax_##",
				tmin = "CHELSA_tmin_##",
				tmean = "CHELSA_tmean_##",
				precip = "CHELSA_precip_##",
				solrad = "CHELSA_solrad_##",
)

# Generate solar radiation rasters
	setwd(climDir)

	# load a raster as template
	rasterTemplate <- raster(list.files(pattern = '\\.tif$')[1])
	
	ETsolradRasters(rasterTemplate = rasterTemplate, year = 30, outputDir = './')


# Generate ENVIREM variables
# check 
verifyFileStructure(returnFileNames = FALSE)

generateRasters(var = 'all', maindir = './', outputDir = './', nTiles = 9)

gc()

# calculate and write monthly PET rasters to disk
	f <- list.files(pattern = '\\.tif$')
	
	tmin <- grep('tmin', f, value = TRUE)
	tmax <- grep('tmax', f, value = TRUE)
	tmean <- grep('tmean', f, value = TRUE)
	solrad <- grep('solrad', f, value = TRUE)
	
	tmin <- tmin[order(as.numeric(gsub('(CHELSA_tmin_)(\\d\\d?)(.tif)', '\\2', tmin)))]
	tmax <- tmax[order(as.numeric(gsub('(CHELSA_tmax_)(\\d\\d?)(.tif)', '\\2', tmax)))]
	tmean <- tmean[order(as.numeric(gsub('(CHELSA_tmean_)(\\d\\d?)(.tif)', '\\2', tmean)))]
	solrad <- solrad[order(as.numeric(gsub('(CHELSA_solrad_)(\\d\\d?)(.tif)', '\\2', solrad)))]
	
	tmeanstack <- stack(tmean)
	solradstack <- stack(solrad)
	TD <- abs(stack(tmax) - stack(tmin))
	
	monthPET <- monthlyPET(Tmean = tmeanstack, RA = solradstack, TD = TD)
	
	for (i in 1:nlayers(monthPET)) {
		fn <- paste0('CHELSA_monthlyPET_', ifelse(i < 10, paste0('0', i), i), '.tif')
		message('\t', basename(fn))
		writeRaster(monthPET[[i]], fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
	}
	
	rm(tmeanstack)
	rm(solradstack)
	rm(TD)
	rm(monthPET)

#################################################################
# ENVIREM LGM CLIM

assignNames(	tmax = "CHELSA_LGM_tmax_##",
				tmin = "CHELSA_LGM_tmin_##",
				tmean = "CHELSA_LGM_tmean_##",
				precip = "CHELSA_LGM_precip_##",
				solrad = "CHELSA_solrad_##",
)

varnames()

# Generate solar radiation rasters
	setwd('../../../')
	setwd(climDirLGM)

	# load a raster as template
	rasterTemplate <- raster(list.files(pattern = '\\.tif$')[1])
	
	ETsolradRasters(rasterTemplate = rasterTemplate, year = -21500, outputDir = './')


# Generate ENVIREM variables
# check 
verifyFileStructure(returnFileNames = FALSE)

generateRasters(var = 'all', maindir = './', outputDir = './', nTiles = 4)


# calculate and write monthly PET rasters to disk
	f <- list.files(pattern = '\\.tif$')
	
	tmin <- grep('tmin', f, value = TRUE)
	tmax <- grep('tmax', f, value = TRUE)
	tmean <- grep('tmean', f, value = TRUE)
	solrad <- grep('solrad', f, value = TRUE)
	
	tmin <- tmin[order(as.numeric(gsub('(CHELSA_LGM_tmin_)(\\d\\d?)(.tif)', '\\2', tmin)))]
	tmax <- tmax[order(as.numeric(gsub('(CHELSA_LGM_tmax_)(\\d\\d?)(.tif)', '\\2', tmax)))]
	tmean <- tmean[order(as.numeric(gsub('(CHELSA_LGM_tmean_)(\\d\\d?)(.tif)', '\\2', tmean)))]
	solrad <- solrad[order(as.numeric(gsub('(CHELSA_solrad_)(\\d\\d?)(.tif)', '\\2', solrad)))]
	
	tmeanstack <- stack(tmean)
	solradstack <- stack(solrad)
	TD <- abs(stack(tmax) - stack(tmin))
	
	monthPET <- monthlyPET(Tmean = tmeanstack, RA = solradstack, TD = TD)
	
	for (i in 1:nlayers(monthPET)) {
		fn <- paste0('CHELSA_LGM_monthlyPET_', ifelse(i < 10, paste0('0', i), i), '.tif')
		message('\t', basename(fn))
		writeRaster(monthPET[[i]], fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
	}
	
	rm(tmeanstack)
	rm(solradstack)
	rm(TD)
	rm(monthPET)

#################################################################
# ENVIREM FUTURE CLIM

assignNames(	tmax = "CHELSA_FUTURE2070_tmax_##",
				tmin = "CHELSA_FUTURE2070_tmin_##",
				tmean = "CHELSA_FUTURE2070_tmean_##",
				precip = "CHELSA_FUTURE2070_precip_##",
				solrad = "CHELSA_solrad_##",
)

varnames()

# Generate solar radiation rasters
	setwd('../../../')
	setwd(climDirFUTURE)

	# load a raster as template
	rasterTemplate <- raster(list.files(pattern = '\\.tif$')[1])
	
	ETsolradRasters(rasterTemplate = rasterTemplate, year = 120, outputDir = './', overwrite = TRUE)


# Generate ENVIREM variables
# check 
verifyFileStructure(returnFileNames = FALSE)

generateRasters(var = 'all', maindir = './', outputDir = './', nTiles = 4, prefix = 'FUTURE2070')


# calculate and write monthly PET rasters to disk
	f <- list.files(pattern = '\\.tif$')
	f <- grep('2070', f, value = TRUE)
	
	tmin <- grep('tmin', f, value = TRUE)
	tmax <- grep('tmax', f, value = TRUE)
	tmean <- grep('tmean', f, value = TRUE)
	solrad <- grep('solrad', list.files(pattern = '\\.tif$'), value = TRUE)
	
	tmin <- tmin[order(as.numeric(gsub('(CHELSA_FUTURE2070_tmin_)(\\d\\d?)(.tif)', '\\2', tmin)))]
	tmax <- tmax[order(as.numeric(gsub('(CHELSA_FUTURE2070_tmax_)(\\d\\d?)(.tif)', '\\2', tmax)))]
	tmean <- tmean[order(as.numeric(gsub('(CHELSA_FUTURE2070_tmean_)(\\d\\d?)(.tif)', '\\2', tmean)))]
	solrad <- solrad[order(as.numeric(gsub('(CHELSA_solrad_)(\\d\\d?)(.tif)', '\\2', solrad)))]
	
	tmeanstack <- stack(tmean)
	solradstack <- stack(solrad)
	TD <- abs(stack(tmax) - stack(tmin))
	
	monthPET <- monthlyPET(Tmean = tmeanstack, RA = solradstack, TD = TD)
	
	for (i in 1:nlayers(monthPET)) {
		fn <- paste0('CHELSA_FUTURE2070_monthlyPET_', ifelse(i < 10, paste0('0', i), i), '.tif')
		message('\t', basename(fn))
		writeRaster(monthPET[[i]], fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
	}
	
	rm(tmeanstack)
	rm(solradstack)
	rm(TD)
	rm(monthPET)

	setwd('../../../')
#################################################################
# BIOCLIM CURRENT CLIM

setwd(climDir)

f <- list.files(pattern = '\\.tif$')

# identify min/max/mean temp and precipitation
	tmin <- grep('tmin', f, value = TRUE)
	tmax <- grep('tmax', f, value = TRUE)
	precip <- grep('precip', f, value = TRUE)

# fix ordering
	tmin <- tmin[order(as.numeric(gsub('(CHELSA_tmin_)(\\d\\d?)(.tif)', '\\2', tmin)))]
	tmax <- tmax[order(as.numeric(gsub('(CHELSA_tmax_)(\\d\\d?)(.tif)', '\\2', tmax)))]
	precip <- precip[order(as.numeric(gsub('(CHELSA_precip_)(\\d\\d?)(.tif)', '\\2', precip)))]

# read as rasterBricks
	tmin <- brick(stack(tmin))
	tmax <- brick(stack(tmax))
	precip <- brick(stack(precip))


gc()
bioclim <- biovars(precip, tmin, tmax)
gc()

# write to disk
for (i in 1:nlayers(bioclim)) {
	fn <- paste0('CHELSA_bioclim_', ifelse(i < 10, paste0('0', i), i), '.tif')
	message('\t', basename(fn))
	writeRaster(bioclim[[i]], fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
}


setwd('../../../')
#################################################################
# BIOCLIM LGM CLIM

setwd(climDirLGM)

f <- list.files(pattern = '\\.tif$')

# identify min/max/mean temp and precipitation
	tmin <- grep('tmin', f, value = TRUE)
	tmax <- grep('tmax', f, value = TRUE)
	precip <- grep('precip', f, value = TRUE)

# fix ordering
	tmin <- tmin[order(as.numeric(gsub('(CHELSA_LGM_tmin_)(\\d\\d?)(.tif)', '\\2', tmin)))]
	tmax <- tmax[order(as.numeric(gsub('(CHELSA_LGM_tmax_)(\\d\\d?)(.tif)', '\\2', tmax)))]
	precip <- precip[order(as.numeric(gsub('(CHELSA_LGM_precip_)(\\d\\d?)(.tif)', '\\2', precip)))]

# read as rasterBricks
	tmin <- brick(stack(tmin))
	tmax <- brick(stack(tmax))
	precip <- brick(stack(precip))


gc()
bioclim <- biovars(precip, tmin, tmax)
gc()

# write to disk
for (i in 1:nlayers(bioclim)) {
	fn <- paste0('CHELSA_LGM_bioclim_', ifelse(i < 10, paste0('0', i), i), '.tif')
	message('\t', basename(fn))
	writeRaster(bioclim[[i]], fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
}

setwd('../../../')
#################################################################
# BIOCLIM FUTURE CLIM

setwd(climDirFUTURE)

f <- list.files(pattern = '\\.tif$')
f <- grep('2070', f, value = TRUE)

# identify min/max/mean temp and precipitation
	tmin <- grep('tmin', f, value = TRUE)
	tmax <- grep('tmax', f, value = TRUE)
	precip <- grep('precip', f, value = TRUE)

# fix ordering
	tmin <- tmin[order(as.numeric(gsub('(CHELSA_FUTURE2070_tmin_)(\\d\\d?)(.tif)', '\\2', tmin)))]
	tmax <- tmax[order(as.numeric(gsub('(CHELSA_FUTURE2070_tmax_)(\\d\\d?)(.tif)', '\\2', tmax)))]
	precip <- precip[order(as.numeric(gsub('(CHELSA_FUTURE2070_precip_)(\\d\\d?)(.tif)', '\\2', precip)))]

# read as rasterBricks
	tmin <- brick(stack(tmin))
	tmax <- brick(stack(tmax))
	precip <- brick(stack(precip))


gc()
bioclim <- biovars(precip, tmin, tmax)
gc()

# write to disk
for (i in 1:nlayers(bioclim)) {
	fn <- paste0('CHELSA_FUTURE2070_bioclim_', ifelse(i < 10, paste0('0', i), i), '.tif')
	message('\t', basename(fn))
	writeRaster(bioclim[[i]], fn, NAflag = -9999, options = tifOptions, overwrite = TRUE)
}

setwd('../../../')
#######################################################################
# Instead of quarterly climate summaries, calculate temperature, precipitation and PET for summer and winter
# This allows us to avoid variables that are "splotchy"

# We will not replace variables based on quarterly summaries of precip, because North America will vary in terms of which 3 months those are.

## potential variables to avoid
# bio5: max temp warmest month <- fine
# bio6: min temp coldest month <- fine
# bio8: mean temp wettest quarter <- fine
# bio9: mean temp driest quarter <- fine
# bio10: mean temp warmest quarter <- replace with mean summer temp (June-July-Aug)
# bio11: mean temp coldest quarter <- replace with mean winter temp (Dec-Jan-Feb)
# bio13: precipitation wettest month <- fine
# bio14: precipitation driest month <- fine
# bio16: precip wettest quarter <- fine
# bio17: precip driest quarter <- fine
# bio18: precip warmest quarter <- replace with summer precip (June-July-Aug)
# bio19: precip coldest quarter <- replace with winter precip (Dec-Jan-Feb)

# PET coldest quarter <- replace with winter PET
# PET driest quarter <- fine
# PET warmest quarter <- replace with summer PET
# PET wettest quarter <- fine

setwd(climDir)

f <- list.files(pattern = '\\.tif$')

# identify min/max/mean temp and precipitation
	tmin <- grep('tmin', f, value = TRUE)
	tmax <- grep('tmax', f, value = TRUE)
	tmean <- grep('tmean', f, value = TRUE)
	precip <- grep('prec', f, value = TRUE)
	pet <- grep('monthlyPET', f, value = TRUE)
	

# fix ordering
	tmin <- tmin[order(as.numeric(gsub('(CHELSA_tmin_)(\\d\\d?)(.tif)', '\\2', tmin)))]
	tmax <- tmax[order(as.numeric(gsub('(CHELSA_tmax_)(\\d\\d?)(.tif)', '\\2', tmax)))]
	tmean <- tmean[order(as.numeric(gsub('(CHELSA_tmean_)(\\d\\d?)(.tif)', '\\2', tmean)))]
	precip <- precip[order(as.numeric(gsub('(CHELSA_precip_)(\\d\\d?)(.tif)', '\\2', precip)))]
	pet <- pet[order(as.numeric(gsub('(CHELSA_monthlyPET_)(\\d\\d?)(.tif)', '\\2', pet)))]


# mean temp variables
	tmeanStack <- stack(tmean)
	meanTempSummer <- calc(tmeanStack[[6:8]], mean) # replacement for bio10
	meanTempWinter <- calc(tmeanStack[[c(12,1,2)]], mean) # replacement for bio11
	names(meanTempSummer) <- 'meanTempSummer'
	names(meanTempWinter) <- 'meanTempWinter'
	
	writeRaster(meanTempSummer, 'meanTempSummer.tif', NAflag = -9999, options = tifOptions, overwrite = TRUE)
	writeRaster(meanTempWinter, 'meanTempWinter.tif', NAflag = -9999, options = tifOptions, overwrite = TRUE)
	
	rm(tmeanStack)
	rm(meanTempSummer)
	rm(meanTempWinter)

# precip variables
	precipStack <- stack(precip)
	precipSummer <- calc(precipStack[[6:8]], sum) # replacement for bio18
	precipWinter <- calc(precipStack[[c(12,1,2)]], sum) # replacement for bio19
	names(precipSummer) <- 'precipSummer'
	names(precipWinter) <- 'precipWinter'
	
	writeRaster(precipSummer, 'precipSummer.tif', NAflag = -9999, options = tifOptions, overwrite = TRUE)
	writeRaster(precipWinter, 'precipWinter.tif', NAflag = -9999, options = tifOptions, overwrite = TRUE)
	
	rm(precipStack)
	rm(precipSummer)
	rm(precipWinter)

# PET variables
	petStack <- stack(pet)
	summerPET <- calc(petStack[[6:8]], mean) # replacement for PET warmest quarter
	winterPET <- calc(petStack[[c(12,1,2)]], mean) # replacement for PET coldest quarter
	names(summerPET) <- 'summerPET'
	names(winterPET) <- 'winterPET'
	
	writeRaster(summerPET, 'summerPET.tif', NAflag = -9999, options = tifOptions, overwrite = TRUE)
	writeRaster(winterPET, 'winterPET.tif', NAflag = -9999, options = tifOptions, overwrite = TRUE)
	
	rm(petStack)
	rm(summerPET)
	rm(winterPET)


setwd('../../../')

## SAME FOR LGM
setwd(climDirLGM)

f <- list.files(pattern = '\\.tif$')

# identify min/max/mean temp and precipitation
	tmin <- grep('tmin', f, value = TRUE)
	tmax <- grep('tmax', f, value = TRUE)
	tmean <- grep('tmean', f, value = TRUE)
	precip <- grep('prec', f, value = TRUE)
	pet <- grep('monthlyPET', f, value = TRUE)
	

# fix ordering
	tmin <- tmin[order(as.numeric(gsub('(CHELSA_LGM_tmin_)(\\d\\d?)(.tif)', '\\2', tmin)))]
	tmax <- tmax[order(as.numeric(gsub('(CHELSA_LGM_tmax_)(\\d\\d?)(.tif)', '\\2', tmax)))]
	tmean <- tmean[order(as.numeric(gsub('(CHELSA_LGM_tmean_)(\\d\\d?)(.tif)', '\\2', tmean)))]
	precip <- precip[order(as.numeric(gsub('(CHELSA_LGM_precip_)(\\d\\d?)(.tif)', '\\2', precip)))]
	pet <- pet[order(as.numeric(gsub('(CHELSA_LGM_monthlyPET_)(\\d\\d?)(.tif)', '\\2', pet)))]

# mean temp variables
	tmeanStack <- stack(tmean)
	meanTempSummer <- calc(tmeanStack[[6:8]], mean) # replacement for bio10
	meanTempWinter <- calc(tmeanStack[[c(12,1,2)]], mean) # replacement for bio11
	names(meanTempSummer) <- 'meanTempSummer'
	names(meanTempWinter) <- 'meanTempWinter'
	
	writeRaster(meanTempSummer, 'meanTempSummer.tif', NAflag = -9999, options = tifOptions, overwrite = TRUE)
	writeRaster(meanTempWinter, 'meanTempWinter.tif', NAflag = -9999, options = tifOptions, overwrite = TRUE)
	
	rm(tmeanStack)
	rm(meanTempSummer)
	rm(meanTempWinter)

# precip variables
	precipStack <- stack(precip)
	precipSummer <- calc(precipStack[[6:8]], sum) # replacement for bio18
	precipWinter <- calc(precipStack[[c(12,1,2)]], sum) # replacement for bio19
	names(precipSummer) <- 'precipSummer'
	names(precipWinter) <- 'precipWinter'
	
	writeRaster(precipSummer, 'precipSummer.tif', NAflag = -9999, options = tifOptions, overwrite = TRUE)
	writeRaster(precipWinter, 'precipWinter.tif', NAflag = -9999, options = tifOptions, overwrite = TRUE)
	
	rm(precipStack)
	rm(precipSummer)
	rm(precipWinter)

# PET variables
	petStack <- stack(pet)
	summerPET <- calc(petStack[[6:8]], mean)
	winterPET <- calc(petStack[[c(12,1,2)]], mean)
	names(summerPET) <- 'summerPET'
	names(winterPET) <- 'winterPET'
	
	writeRaster(summerPET, 'summerPET.tif', NAflag = -9999, options = tifOptions, overwrite = TRUE)
	writeRaster(winterPET, 'winterPET.tif', NAflag = -9999, options = tifOptions, overwrite = TRUE)
	
	rm(petStack)
	rm(summerPET)
	rm(winterPET)


