### Cline Analysis on Cyanocitta mtDNA ###
library("hzar")
library("doMC")
registerDoMC()
library("coda")

### Set up seeds for MCMC ###
mainSeed = 
	list(
		A=c(978,544,99,596,528,124),
		B=c(544,99,596,528,124,978),
		C=c(99,596,528,124,978,544),
		D=c(596,528,124,978,544,99),
		E=c(528,124,978,544,99,596),
		F=c(124,978,544,99,596,528),
		G=c(784,496,315,392,481,268),
		H=c(496,315,392,481,268,784),
		I=c(315,392,481,268,784,496),
		J=c(392,481,268,784,496,315),
		K=c(481,268,784,496,315,392),
		L=c(268,784,496,315,392,481),
		M=c(372,285,115,914,377,418),
		N=c(285,115,914,377,418,372),
		O=c(115,914,377,418,372,285))

		
### Read in data files ###
JayND2 <-read.table("../data/hzar_mtDNA_input.txt",header=TRUE) # percentages of ND2 interior vs. rocky mtn haplotype in each pop

JayND2
head(JayND2)

## Blank out space in memory to hold mtDNA data analysis
if(length(apropos("^Cst$",ignore.case=FALSE)) == 0 ||
 !is.list(Cst) ) 
 
Cst <- list()
Cst$ND2 <- list();
Cst$ND2$obs <- list(); #space to hold observed data
Cst$ND2$models <- list(); #space to hold models to fit
Cst$ND2$fitRs <- list(); #space to hold compiled fit requests
Cst$ND2$runs <- list(); #space to hold the output data chains
Cst$ND2$analysis <- list(); #space to hold the analysed data

## Add observed data into Cst list
Cst$ND2$obs <-hzar.doMolecularData1DPops(JayND2$Distance,JayND2$mtDNA_interior,JayND2$n)

## Plot graph to quickly visualize observed data
hzar.plot.obsData(Cst$ND2$obs)

## Creating helper function ##
Cst.loadND2model <- function(scaling,tails,
							 id=paste(scaling,tails,sep="."))
	Cst$ND2$models[[id]] <<-
	 hzar.makeCline1DFreq(Cst$ND2$obs, scaling, tails)


### Now we name the models with the scaling and tail informations
Cst.loadND2model("none","none","modelI"); ## Pmax Pmin 0 and 1; no exp tail 
Cst.loadND2model("fixed" ,"none","modelII"); ## Pmax Pmin fixed to obs mean; no exp tail 
Cst.loadND2model("free" ,"none","modelIII"); ## Pmax Pmin free to vary; no exp tail 

Cst.loadND2model("none","right","modelIV"); ## Pmax Pmin 0 and 1; right exp tail
Cst.loadND2model("fixed" ,"right","modelV"); ## Pmax Pmin fixed to obs mean; right exp tail
Cst.loadND2model("free" ,"right","modelVI"); ## Pmax Pmin free to vary; right exp tail

Cst.loadND2model("none","left","modelVII"); 
Cst.loadND2model("fixed" ,"left","modelVIII");
Cst.loadND2model("free" ,"left","modelIX");

Cst.loadND2model("none","mirror","modelX");
Cst.loadND2model("fixed" ,"mirror","modelXI"); 
Cst.loadND2model("free" ,"mirror","modelXII");

Cst.loadND2model("none","both","modelXIII"); ## Pmax Pmin 0 and 1; 2 tails w/ independent params
Cst.loadND2model("fixed" ,"both","modelXIV"); ## Pmax Pmin fixed to obs mean; 2 tails w/ independent params
Cst.loadND2model("free" ,"both","modelXV"); ## Pmax Pmin free to vary; 2 tails w/ independent params


## Check the default settings
print(Cst$ND2$models)
length(Cst$ND2$models)

## Modify models to focus on region where observed data were collected. ##
Cst$ND2$models <- sapply(Cst$ND2$models,hzar.model.addBoxReq,-50,1350, simplify=FALSE)

## Also, due to the large number of free variables, it is prudent to reduce the tune setting of modelXV from 1.5 to 1.2 ---- (WHY?)
# hzar.meta.tune(Cst$ND2$models$modelXV)<-1.2

## Check updated settings 
# Cst$ND2$models

## Compile each of the models to prepare for fitting
Cst$ND2$fitRs$init <- sapply(Cst$ND2$models,hzar.first.fitRequest.old.ML,obsData=Cst$ND2$obs,verbose=FALSE,simplify=FALSE)

### Set length of chain and burnin ###
burnin<-5e5
chainLength<-1e6


## Update settings for the fitter is desired
class(Cst$ND2$fitRs$init)
for(i in 1:length(Cst$ND2$fitRs$init)){
	Cst$ND2$fitRs$init[[i]]$mcmcParam$chainLength<-chainLength
	Cst$ND2$fitRs$init[[i]]$mcmcParam$burnin<-burnin
	Cst$ND2$fitRs$init[[i]]$mcmcParam$seed[[1]] <- mainSeed[[i]]
}
names(Cst$ND2$fitRs$init)

## Check fit request settings
print(Cst$ND2$fitRs$init)

## Start running models for an initial chain
Cst$ND2$runs$init <- list()

for(i in 1:length(Cst$ND2$fitRs$init)){
	Cst$ND2$runs$init[[i]]<-hzar.doFit(Cst$ND2$fitRs$init[[i]])
}
names(Cst$ND2$runs$init)<-names(Cst$ND2$fitRs$init)

## Plotting the trace
for(i in 1:length(Cst$ND2$runs$init)){
	plot(hzar.mcmc.bindLL(Cst$ND2$runs$init[[i]]))
}

names(Cst$ND2$runs$init)<-names(Cst$ND2$fitRs$init)

## Then combine a new set of fit requests using the initial chain
Cst$ND2$fitRs$chains <- lapply(Cst$ND2$runs$init,hzar.next.fitRequest)

## Replicate each fit request 3 times, keeping the original seeds while switching to a new seed channel
Cst$ND2$fitRs$chains <-hzar.multiFitRequest(Cst$ND2$fitRs$chains,each=3,baseSeed=NULL)

param_names<-lapply(Cst$ND2$fitRs$chains,function(x) names(x$modelParam$init))


## To be thorough, randomize the initial value for each fit.
for(i in 1:length(Cst$ND2$fitRs$chains)){
	Cst$ND2$fitRs$chains[[i]]$modelParam$init["center"] <- runif(1,-50,1350)
	Cst$ND2$fitRs$chains[[i]]$modelParam$init["width"] <- runif(1,0,1400)
}

## Provide initial starting values for other parameters
for(i in 1:length(grep("pMin",param_names))){
	Cst$ND2$fitRs$chains[[grep("pMin",param_names)[i]]]$modelParam$init["pMin"]<-runif(1,0,1)
}
for(i in 1:length(grep("pMax",param_names))){
	Cst$ND2$fitRs$chains[[grep("pMax",param_names)[i]]]$modelParam$init["pMax"]<-runif(1,0,1)
}
for(i in 1:length(grep("deltaL",param_names))){
	Cst$ND2$fitRs$chains[[grep("deltaL",param_names)[i]]]$modelParam$init["deltaL"]<-runif(1,10,60)
}
for(i in 1:length(grep("tauL",param_names))){
	Cst$ND2$fitRs$chains[[grep("tauL",param_names)[i]]]$modelParam$init["tauL"]<-runif(1,0,1)
}
for(i in 1:length(grep("deltaR",param_names))){
	Cst$ND2$fitRs$chains[[grep("deltaR",param_names)[i]]]$modelParam$init["deltaR"]<-runif(1,10,60)
}
for(i in 1:length(grep("tauR",param_names))){
	Cst$ND2$fitRs$chains[[grep("tauR",param_names)[i]]]$modelParam$init["tauR"]<-runif(1,0,1)
}
for(i in 1:length(grep("deltaM",param_names))){
	Cst$ND2$fitRs$chains[[grep("deltaM",param_names)[i]]]$modelParam$init["deltaM"]<-runif(1,10,60)
}
for(i in 1:length(grep("tauM",param_names))){
	Cst$ND2$fitRs$chains[[grep("tauM",param_names)[i]]]$modelParam$init["tauM"]<-runif(1,0,1)
}


### Run a chain of 3 runs for every fit request
Cst$ND2$runs$chains <- hzar.doChain.multi(Cst$ND2$fitRs$chains,doPar=TRUE,inOrder=FALSE,count=3)

### Look at convergence metrics ###
for(i in 1:(45/3)){
	print(summary(do.call(mcmc.list,lapply(Cst$ND2$runs$chains[((i-1)*3+1):((i-1)*3+3)],function(x) hzar.mcmc.bindLL(x[[3]])))))
}

### All runs are convergent! Move on to data aggregation ###

## Create model data group (hzar.dataGroup object) for each model from initial runs
Cst$ND2$analysis$initDGs <- list(
nullModel = hzar.dataGroup.null(Cst$ND2$obs))

for(i in 1:15){
	Cst$ND2$analysis$initDGs[[i]]<-hzar.dataGroup.add(Cst$ND2$runs$init[[i]])
}

names(Cst$ND2$analysis$initDGs)<-names(Cst$ND2$fitRs$init)


### Check for 15 non-indentical models ###
for(i in 1:length(Cst$ND2$analysis$initDGs)){
	print(names(Cst$ND2$analysis$initDGs)[i])
	print(sapply(Cst$ND2$analysis$initDGs,function(x) hzar.sameModel(Cst$ND2$analysis$initDGs[[i]],x)))
}

### Create a hzar.obsDataGroup from the hzar.dataGroup we just created.
Cst$ND2$analysis$oDG <-hzar.make.obsDataGroup(Cst$ND2$analysis$initDGs)
?hzar.make.obsDataGroup


Cst$ND2$analysis$oDG <-hzar.copyModelLabels(Cst$ND2$analysis$initDGs,Cst$ND2$analysis$oDG)


## Convert all runs to hzar.dataGroup objects, adding them to the hzar.obsDataGroup object.
length(Cst$ND2$analysis$initDGs)
names(Cst$ND2$analysis$oDG[[1]])
length(Cst$ND2$analysis$oDG[[1]])

Cst$ND2$analysis$oDG <-hzar.make.obsDataGroup(lapply(Cst$ND2$runs$chains,hzar.dataGroup.add),Cst$ND2$analysis$oDG)
 
## Check the number of hzar.dataGroup objects in the hzar.obsDataGroup object
print(summary(Cst$ND2$analysis$oDG$data.groups))
 
## Compare the fitted clines graphically
hzar.plot.cline(Cst$ND2$analysis$oDG)

## Print out the model with the minimum AICc score
#str(Cst$ND2$analysis)
#print(Cst$ND2$analysis$model.name<-rownames(Cst$ND2$analysis$AICcTable)[[which.min(Cst$ND2$analysis$AICcTable$AICc)]])
print(Cst$ND2$analysis$AICcTable <- hzar.AICc.hzar.obsDataGroup(Cst$ND2$analysis$oDG))

### Save output so we do not have to rerun the analysis ###
save(Cst,file="../output/Cst_HZAR_ND2_output.Rdata")
load("../output/Cst_HZAR_ND2_output.Rdata")
Cst 

## Extract hzar.dataGroup object for selected model
Cst$ND2$analysis$model.selected<-
Cst$ND2$analysis$oDG$dataGroup[[Cst$ND2$analysis$model.name]]

## Looking at variation in parameters for the selected model
print(hzar.getLLCutParam(Cst$ND2$analysis$model.selected,
names(Cst$ND2$analysis$model.selected$data.param)))

## Print and plot ML cline for selected model
print(hzar.get.ML.cline(Cst$ND2$analysis$model.selected));
hzar.plot.cline(Cst$ND2$analysis$model.selected);
hzar.plot.fzCline(Cst$ND2$analysis$model.selected)

##
pdf("../output/HZAR_ND2.pdf")
hzar.plot.fzCline(Cst$msat$analysis$model.selected,ylab="% interior ND2 haplotype")
dev.off()