fprintf('Beginning SCSP LAMP simulation program.\n');
fprintf('For troubleshooting, contact Stephen Guth at sguth@mit.edu,\n');
fprintf('or the current maintainer at SAND Lab.\n\n');



z_star = 4.5;       % radius of structured samples
n_exp = 10;         % number of distinct wave episodes to simulate
n_scsp_repeats = 1; % set to >1 if you want to repeat each wave episode with
                    % multiple stochastic preludes

fixed_T = 40;       % length of wave episode
n_modes = 4;        % number of retained wave episode modes

lamp_retained_freq = 1024;  % fidelity of LAMP DFT reconstruction.
                            % Reducing this quantity may increase simulation 
                            % speed


%
% Investigate parallelization for your machine!  Each LAMP thread will use
% only one core.  MATLAB will keep one LAMP thread running for each thread
% allocated in the parallel pool, until the entire set is done
%

fprintf('Initializing parallel pool.\n')
parpool('local'); 

%
% `Initialization period' for the LAMP simulation.  Guth and Sapsis (2022)
% used quite long initializations--you may be able to increase simulation
% speed without sacrificing fidelity
%
     
ramp_interval = 100;
post_ramp_interval = 80 + fixed_T/2;


%
% Set up path and directories.  Be sure to set windows/linux!
%

output_path = sprintf('./output/');
mode_filename = sprintf('%sT=%d.mat', output_path, fixed_T);
control_filename = sprintf('%scontrol.mat', output_path);

os = 'linux';
%os = 'win';

if ~exist(output_path, 'dir')
    mkdir(output_path)
end

cd ../Working_Folder/;      
addpath './lamp_binaries';          % LAMP binaries not provided!
addpath './lamp_templates';
addpath '../LAMP_Wrapper';

fprintf('Output sent to %s.\n', output_path);




fprintf('Establishing parameters.\n')

j_par = JONSWAP_Parameters();

gpe_par = GPE_Parameters();
%gpe_par.n_memory = 512;    % good for fixed_T  = 40
gpe_par.n_memory = 1024;    % good for fixed_T  = 80
gpe_par.n_var_length = gpe_par.n_memory*2; % 2 for both sides extrapolation
gpe_par.extrap_forward = true;
gpe_par.extrap_backward = true;


% feed in one-sided spectrum
amp_of_cosine = @(S, w, dw) sqrt(2*S(w).*dw);






fprintf('Building KL basis.\n')

WW_kl = linspace(j_par.omega_min, j_par.omega_max, j_par.n_W)';
dW = WW_kl(2) - WW_kl(1);
AA_kl = amp_of_cosine(j_par.S, WW_kl, dW);

T_max_kl = fixed_T;
n_t_kl = gpe_par.n_memory;
TT_kl = linspace(0, T_max_kl, n_t_kl);
dt_kl = TT_kl(2) - TT_kl(1);

[ V_kl, D_kl ] = calc_direct_kl_modes(AA_kl, WW_kl, TT_kl);

kl_struct = struct;
kl_struct.T = TT_kl;
kl_struct.modes = V_kl;
kl_struct.variance = D_kl;
save(mode_filename, 'kl_struct');
fclose('all');




%
% Build the matrix of all the wave episodes to run, which is saved in
% X_design.
%


if exist(control_filename, 'file')
    fprintf('Loading experimental control file.\n')
    load(control_filename, 'X_design', 'n_exp', 'n_modes');
    fclose('all');
else
    fprintf('Building experimental control file.\n')
    
    II_cur = 1:n_modes;
    X_design = z_star*(1 - 2*lhsdesign(n_exp, n_modes, 'Smooth', 'on'));    

    save(control_filename, 'X_design', 'n_exp', 'n_modes');
end






fprintf('Building autocorrelation.\n')
[ R_xi ] = calc_autocorrelation_fft(j_par.S);
fprintf('Building gpe fixed coefficients.\n')
[ gpe_sig, gpe_B ] = calc_gpe_coeff(gpe_par, R_xi, dt_kl);







fprintf('Beginning main simulation loop.\n')

fprintf('  %d distinct parameter sets with %d distinct prelude repeats each.\n', n_exp, n_scsp_repeats);

parfor k_next = 1:n_exp

     cur_alpha = X_design(k_next, :);
     
     l_par = LAMP_Parameters(1, 0, os);
     l_par.template_path = 'lamp_templates/Template_sj.in';
     l_par.output_path =  output_path;
     l_par.nfreq = lamp_retained_freq;
     l_par.n_simulation = k_next;
     l_par.verbose = 0;
    
     l_par.initial_time = -(ramp_interval + post_ramp_interval);
     l_par.time = fixed_T;
     l_par.nstep = floor((l_par.time - l_par.initial_time)/ l_par.timestep) + 1;
     l_par.final_ramp_time = -post_ramp_interval;
     
     %
     % Lagrangian translation issue -- 
     % Adjusting this value changes when the vessel passes through the
     % the fixed region.  Currently, the ship exits at t=0.
     %
     % If you adjust this, BE SURE TO CAREFULLY EXAMINE THE OUTPUT TO
     % DETERMINE WHEN THE SHIP MEETS THE PRESCRIBED WAVE EPISODE!
     % 
    
     %l_par.xstart = (l_par.initial_time - l_par.time/2)*l_par.xvelstart;
     l_par.xstart = (l_par.initial_time)*l_par.xvelstart;
     
     for k_scsp = 1:n_scsp_repeats
        fprintf('Starting %d-%d.\n', k_next, k_scsp);
        
        [ Xi ] = build_scsp_signal( gpe_par, D_kl, V_kl, gpe_B, gpe_sig, cur_alpha );
        [ W, A, phi ] = discretize_scsp_signal( l_par, gpe_par, Xi, dt_kl );  
        
        l_par.filename_root = sprintf('%srun_%d-%d', output_path, l_par.n_simulation, k_scsp);

        wave_freq = W/(2*pi);
        phi_deg = phi/(2*pi)*360;
        A_hz = A;

        %
        % For diagnostic purposes, you could turn the sum-of-sinusoid
        % representation back into a time series
        %
        
        %ZZ = recover_scsp_signal( W, A, phi, TT);

        create_input_file(j_par, l_par, W, A, phi_deg);

        fclose('all');
        
        run_lamp_simulation( l_par );
        simulate_slamming( l_par );
        generate_motion_data( l_par );

        %
        % Convert the data from LAMP format to MATLAB matrices
        %

        suffix = sprintf('run-%d-%d-out', l_par.n_simulation, k_scsp);
        [ true_struct ] = process_data(l_par, suffix);
    
    end
     
end




fclose('all');
delete(gcp('nocreate')) % close parallel pool
