classdef JONSWAP_Parameters < handle
    %JONSWAP_PARAMETERS Summary of this class goes here
    %   Detailed explanation goes here
    
    properties
        alpha; % Phillip constant
        beta;
        gamma;
        tmodal;
        H_s;
        heading;
        g;
        delta_low;
        delta_high;
        freq_p;
        omega_m;
        
        f_spec_w;
        S;
        h;
        
        f_disp_w;
        f_disp_k;
        
        omega_min; % magic constants the keep the most variance
        omega_max;
        n_W;
    end
    
    methods
        function par = JONSWAP_Parameters()
            %JONSWAP_PARAMETERS Construct an instance of this class
            %   Detailed explanation goes here            
            
            
            par.alpha = 0.060; % JONSWAP Three-Parameter Spectrum Property
            par.tmodal = 10.0; % JONSWAP Three-Parameter Spetrum Property
            par.gamma = 3.0; % JONSWAP Three-Parameter Spectrum Property
            % 1.0 to 7.0, 3.3 typical
            
            par.H_s = 13.2;
            par.heading = 180.0;
            % 180 = Standard (Wave Direction relative to Bow of Hullform)

            par.g = 9.81; %[m/s^2] Gravity
            par.delta_low = 0.07; % Omega <= Omega_m
            par.delta_high = 0.09; % Omega > Omega_m

            par.n_W = 250;
            
            par.h = 20;  % arbitrary
            
            par.update_derived_parameters();
            par.update_dispersion_relation();
            par.update_spectrum_function();
        end
        
        function [ outcode ] = update_significant_wave_height( j_par, newHs)
            j_par.H_s = newHs;
            
            j_par.update_derived_parameters();
            j_par.update_dispersion_relation();
            j_par.update_spectrum_function();
            
            outcode = 1;
        end
        
        function [ outcode ] = update_water_depth( j_par, newh)
            j_par.h = newh;
            
            j_par.update_derived_parameters();
            j_par.update_dispersion_relation();
            j_par.update_spectrum_function();
            
            outcode = 1;
        end
        
        function [ outcode ] = update_modal_period( j_par, newT)
            j_par.tmodal = newT;
            
            j_par.alpha = (1/j_par.tmodal)^4 * ((5.058 * (j_par.H_s^2))*(1-(0.287*log(j_par.gamma))));
            
            j_par.update_derived_parameters();
            j_par.update_dispersion_relation();
            j_par.update_spectrum_function();
            
            outcode = 1;
        end
        
        
        
        function [ outcode ] = update_derived_parameters(j_par)
            j_par.freq_p = (j_par.alpha / ((5.058 * (j_par.H_s^2))*(1-(0.287*log(j_par.gamma)))))^(1/4);
            j_par.omega_m = 2 * pi * j_par.freq_p;
            
            j_par.tmodal = 1/j_par.freq_p;
             
            j_par.omega_min = 0.4417; % magic constants the keep the most variance
            j_par.omega_max = 2.2627; % currently, fixed
            
            outcode = 1;
        end
        
        function [ outcode ] = update_spectrum_function(j_par)
            j_par.f_spec_w = @(a, g, w,wm, gam, d) ...
                abs((a*g^2)./(w.^5)) .* ...
                exp( (-5/4).*(wm./w).^4).* ...
                gam.^exp(-(w - wm.*sign(w)).^2 ./ (2*d^2.*wm.^2));
            
            j_par.S = @(w) (j_par.f_spec_w(j_par.alpha, j_par.g, w, j_par.omega_m, j_par.gamma, j_par.delta_low).*(abs(w) <= j_par.omega_m)) + ...
                (j_par.f_spec_w(j_par.alpha, j_par.g, w, j_par.omega_m, j_par.gamma, j_par.delta_high).*(abs(w) > j_par.omega_m));
            
            outcode = 1;   
        end
        
        function [ outcode ] = update_dispersion_relation( j_par )
            j_par.f_disp_w = @(k) sqrt(j_par.g*k.*tanh(k*j_par.h));

            KK = linspace(0, 10, 1024);
            WW = j_par.f_disp_w(KK);

            j_par.f_disp_k = @(w) interp1(WW, KK, w, 'linear', 'extrap');
            
            outcode = 1;
        end
        
        
        function [ Hb ] = estimate_goda_height( j_par )
            A = 0.12; % lowball estimate for irregular waves
            
            k = j_par.f_disp_k(j_par.omega_m);
            Lp = 2*pi/j_par.f_disp_k(j_par.omega_m);
            Tp = 2*pi/j_par.omega_m ;
            
            
            Hb = j_par.g*Tp^2*A*(1-exp(-1.5*pi* j_par.h / Lp))*tanh(k*j_par.h) / (2*pi);
            
        end
        
          function [ outcode ] = plot_spectrum(j_par)
            
            WW = linspace(-j_par.omega_max, j_par.omega_max, 256);
            
            figure(201);
            clf;
            plot(WW, j_par.S(WW), 'LineWidth', 3);
            title('power spectrum', 'Interpreter', 'Latex');
            xlabel('$\omega$', 'Interpreter', 'Latex')
            xlabel('$S(\omega)$', 'Interpreter', 'Latex')
            set(gca, 'Fontsize', 14)
            
            outcode = 1;
        end
        

    end
end

