/*
 * Copyright (c) 2019 - 2022 Geode-solutions
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

#include <pybind11/iostream.h>
#include <pybind11/pybind11.h>
#include <pybind11/stl.h>

#include "barycentric_coordinates.h"
#include "basic_objects.h"
#include "bounding_box.h"
#include "distance.h"
#include "intersection.h"
#include "intersection_detection.h"
#include "mensuration.h"
#include "nn_search.h"
#include "perpendicular.h"
#include "point.h"
#include "points_sort.h"
#include "position.h"
#include "projection.h"
#include "rotation.h"
#include "sign.h"
#include "vector.h"

namespace pybind11
{
    namespace detail
    {
        template < typename Type >
        struct type_caster< absl::FixedArray< Type > >
            : list_caster< absl::FixedArray< Type >, Type >
        {
        };

        template < typename Type >
        struct type_caster< absl::Span< Type > >
            : list_caster< absl::Span< Type >, Type >
        {
            using value_conv = make_caster< Type >;

            bool load( handle src, bool convert )
            {
                cpp_.clear();
                auto s = reinterpret_borrow< sequence >( src );
                cpp_.reserve( s.size() );
                for( auto it : s )
                {
                    value_conv conv;
                    if( !conv.load( it, convert ) )
                        return false;
                    cpp_.push_back( cast_op< Type&& >( std::move( conv ) ) );
                }
                this->value = absl::MakeConstSpan( cpp_ );
                return true;
            }

            std::vector< typename std::remove_const< Type >::type > cpp_;
        };

        template < typename Type, size_t dimension >
        struct type_caster< absl::InlinedVector< Type, dimension > >
            : list_caster< absl::InlinedVector< Type, dimension >, Type >
        {
        };

        template < typename T >
        struct type_caster< absl::optional< T > >
            : public optional_caster< absl::optional< T > >
        {
        };
    } // namespace detail
} // namespace pybind11

PYBIND11_MODULE( opengeode_py_geometry, module )
{
    pybind11::add_ostream_redirect( module );
    module.doc() = "OpenGeode Python binding for geometry";
    pybind11::class_< geode::OpenGeodeGeometry >( module, "OpenGeodeGeometry" )
        .def( "initialize", &geode::OpenGeodeGeometry::initialize );
    geode::define_barycentric( module );
    geode::define_basic_objects( module );
    geode::define_bounding_box( module );
    geode::define_distance( module );
    geode::define_intersection( module );
    geode::define_intersection_detection( module );
    geode::define_mensuration( module );
    geode::define_nn_search( module );
    geode::define_perpendicular( module );
    geode::define_point( module );
    geode::define_points_sort( module );
    geode::define_position( module );
    geode::define_projection( module );
    geode::define_rotation( module );
    geode::define_sign( module );
    geode::define_vector( module );
}