from utils import Network as net

# uses the fixed routing map to remove non-changing switches from other routing maps
def removeNonChangingSwitches(network: net.Network):
    for source in network.routing.fixed:
        del network.routing.initial[source]
        del network.routing.final[source]

# use the fixed routing to find the next switch which is not fixed
def getNextNonFixedSwitch(routing: net.Network.Routing, switch: int):
    nextSwitch = switch
    while nextSwitch != None:
        switch = nextSwitch
        nextSwitch = routing.fixed.get(nextSwitch)
    return switch

# uses the fixed routing map to skip non-changing switches in other routing maps
def applyFixedRoutingToOtherRoutings(routing: net.Network.Routing):
    def applyFixedRouting(subrouting):
        for source, target in subrouting.items():
            subrouting[source] = getNextNonFixedSwitch(routing, target)
    applyFixedRouting(routing.initial)
    applyFixedRouting(routing.final)

# uses the fixed routing map to skip non-changing switches in node numbers
def applyFixedRoutingToNodes(network: net.Network):
    network.nodeStart = getNextNonFixedSwitch(network.routing, network.nodeStart)
    network.nodeFinal = getNextNonFixedSwitch(network.routing, network.nodeFinal)
    network.nodesWaypoint = {getNextNonFixedSwitch(network.routing, waypoint) for waypoint in network.nodesWaypoint}

# copies the fixed routing to the fixed routing tables for concurrent mode
def copyFixedRoutings(network: net.Network):
    network.routingBefore.fixed = network.routing.fixed.copy()
    network.routingIntermediate.fixed = network.routing.fixed.copy()
    network.routingAfter.fixed = network.routing.fixed.copy()
    network.routingSingle.fixed = network.routing.fixed.copy()

# copies the initial/final routing tables to the routing tables for concurrent mode
def copyRoutings(network: net.Network):
    network.routingBefore.initial = network.routing.initial.copy()
    network.routingIntermediate.initial = network.routing.initial.copy()
    network.routingAfter.initial = network.routing.initial.copy()
    network.routingSingle.initial = network.routing.initial.copy()
    network.routingBefore.final = network.routing.final.copy()
    network.routingIntermediate.final = network.routing.final.copy()
    network.routingAfter.final = network.routing.final.copy()
    network.routingSingle.final = network.routing.final.copy()

# convert the first and last switches to fixed routings in all routing tables
# first switches are fixed with their final routing (except in the before and single routing)
# last switches are fixed with their initial routing (except in the after and single  routing)
# they must be switched in the first/last step
def convertFirstLastSwitchesToFixedRoutings(network: net.Network):
    for switch in network.switchesFirst:
        target = network.routing.final.pop(switch)
        network.routing.fixed[switch] = target
        network.routingBefore.fixed[switch] = "xx"
        network.routingIntermediate.fixed[switch] = target
        network.routingAfter.fixed[switch] = target
        network.routingSingle.fixed[switch] = "xx"
    for switch in network.switchesLast:
        target = network.routing.initial.pop(switch)
        network.routing.fixed[switch] = target
        network.routingBefore.fixed[switch] = target
        network.routingIntermediate.fixed[switch] = target
        network.routingAfter.fixed[switch] = "xx"
        network.routingSingle.fixed[switch] = "xx"
    network.firstLastConvertedToFixed = True
