from collections import defaultdict
import json

from utils import Output

class Network:
    class Routing:
        def __init__(self):
            self.initial = dict()
            self.final = dict()
            self.fixed = dict()

        def copy(self):
            ret = Network.Routing()
            ret.initial = self.initial.copy()
            ret.final = self.final.copy()
            ret.fixed = self.fixed.copy()
            return ret

        def __str__(self):
            return f"""
        initial: {self.initial}
        final: {self.final}
        fixed: {self.fixed}"""


    def __init__(self, jsonFileName):
        file = open(jsonFileName)
        data = json.load(file)
        file.close()
        self.convertFromJson(data)
    
    def convertFromJson(self, jsonData):
        def convertArrayToMap(array):
            return {key: value for (key, value) in array}

        self.routing = self.Routing()
        self.routing.initial = convertArrayToMap(jsonData["Initial_routing"])
        self.routing.final = convertArrayToMap(jsonData["Final_routing"])
        self.nodeStart = jsonData["Properties"]["Waypoint"]["startNode"]
        self.nodeFinal = jsonData["Properties"]["Waypoint"]["finalNode"]
        self.nodesWaypoint = jsonData["Properties"]["Waypoint"]["waypoint"]

        # routing tables for the different phases of the concurrent search
        # before: before switching the first switches
        # intermediate: after switching the first, before switching the last
        # after: after switching the last switches
        # single: special case for 1 step only
        self.routingBefore = self.Routing()
        self.routingIntermediate = self.Routing()
        self.routingAfter = self.Routing()
        self.routingSingle = self.Routing()

        self.switchesChanging = set() # switches with different routing in initial and last
        self.switchesNotChanging = set() # switches with fixed routing
        self.switchesFirst = set() # switches with initial dead-end routing
        self.switchesLast = set() # switches with final dead-end routing
        self.switchesNormal = set() # changing switches which are not first or last
        self.switchesDeadEnd = set() # dead-end in both routing maps
        self.firstLastConvertedToFixed = False

        # support both: single waypoint and waypoint array
        # convert both to set
        if type(self.nodesWaypoint) is list:
            self.nodesWaypoint = set(self.nodesWaypoint)
        else:
            self.nodesWaypoint = {self.nodesWaypoint}

        if self.nodeStart != jsonData["Properties"]["Reachability"]["startNode"]:
            raise ValueError("StartNode (WP, REACH) ambiguous")
        # if self.nodeStart != jsonData["Properties"]["LoopFreedom"]["startNode"]:
        #     raise ValueError("StartNode (WP, LOOPFREE) ambiguous")
        if self.nodeFinal != jsonData["Properties"]["Reachability"]["finalNode"]:
            raise ValueError("FinalNode (WP, REACH) ambiguous")      

    def getLogOutput(self):
        ret = ""
        ret += "Don't Switch: " + Output.formatSet(self.switchesNotChanging, "s", ",") + "\n"
        ret += "Switch Before: " + Output.formatSet(self.switchesFirst, "s", ",") + "\n"
        ret += "Switch After: " + Output.formatSet(self.switchesLast, "s", ",") + "\n"
        ret += "Calculate: " + Output.formatSet(self.switchesNormal, "s", ",") + "\n"
        ret += "DeadEnd: " + Output.formatSet(self.switchesDeadEnd, "s", ",")
        return ret

    def __str__(self):
        return f"""Network:
    routing: {self.routing}
    routingBefore: {self.routingBefore}
    routingIntermediate: {self.routingIntermediate}
    routingAfter: {self.routingAfter}
    routingSingle: {self.routingSingle}
    switchesChanging: {self.switchesChanging}
    switchesNotChanging: {self.switchesNotChanging}
    switchesFirst: {self.switchesFirst}
    switchesLast: {self.switchesLast}
    switchesNormal: {self.switchesNormal}
    switchesDeadEnd: {self.switchesDeadEnd}
    nodeStart: {self.nodeStart}
    nodeFinal: {self.nodeFinal}
    nodesWaypoint: {self.nodesWaypoint}
    firstLastConvertedToFixed: {self.firstLastConvertedToFixed}"""
    