import sys

from utils import Network as net
from utils import Analyzer
from utils import Optimizer
from utils import PDDL

arg_options = {argument for argument in sys.argv[1:] if argument.startswith("-")}
arg_files = [argument for argument in sys.argv[1:] if not argument.startswith("-")]

if len(arg_options) > 0 or len(arg_files) != 2:
    sys.stderr.write(f"Usage: {sys.argv[0]} json-file output-file\n")
    sys.exit(-1)

inFileName = arg_files[0]
outFileName = arg_files[1]

# read the network from json into our data structure
network = net.Network(inFileName)

# analyze and optimize non-changing switches
Analyzer.generateFixedRouting(network)
Optimizer.removeNonChangingSwitches(network)
Optimizer.applyFixedRoutingToOtherRoutings(network.routing)

# generate the sets of all kind of switches
Analyzer.generateSwitchSets(network)

# analyze and optimize switches with first and last dead-end routes
Optimizer.copyFixedRoutings(network)
Optimizer.convertFirstLastSwitchesToFixedRoutings(network)

# apply the fixed routing tables to all other routing tables
Optimizer.copyRoutings(network)
Optimizer.applyFixedRoutingToOtherRoutings(network.routing)
Optimizer.applyFixedRoutingToOtherRoutings(network.routingBefore)
Optimizer.applyFixedRoutingToOtherRoutings(network.routingIntermediate)
Optimizer.applyFixedRoutingToOtherRoutings(network.routingAfter)
Optimizer.applyFixedRoutingToOtherRoutings(network.routingSingle)

# update start, final and waypoint nodes
Optimizer.applyFixedRoutingToNodes(network)

# check that start, target and waypoint nodes are known switches
Analyzer.checkNodeConsistency(network)

# maximum number of steps:
#   1 for every normal switch
# maximum number of steps with concurrent_first_last:
#   1 for every normal switch and 1 for the first and 1 for the last switches together
#   less steps could be used as multiple switches can be combined into 1 step
#   also in the first and last step other switches can be added
max_steps_necessary = len(network.switchesNormal)
if len(network.switchesFirst) > 0:
    max_steps_necessary += 1
if len(network.switchesLast) > 0:
    max_steps_necessary += 1

print(network.getLogOutput())
print("Step count:", max_steps_necessary)

out = PDDL.generatePddl(network, max_steps_necessary)
file = open(outFileName, "w")
file.write(out)
file.close()
