/*******************************************************************************

************
** Function: 
************

  Creates data extract that captures flows of inter-generational transfers, 
	including both inter-vivos transfers and bequests. 

************
** Inputs  : 
************

	- HRS cross-wave tracker file
	- RAND HRS longitudinal file
	- RAND HRS family respondent-level file
	- sample_single.dta (see Decedents_ProcessData.do)
	- CPI deflator (see GetCPIdeflator.do)

************
** Outputs : 
************

  - transfers_sample.dta

*******************************************************************************/

clear *
macro drop _all
set more off 

set scheme hack, permanently

********************************************************************************
** Directory names
********************************************************************************

do GetDirNames.do

********************************************************************************
** TRACKER
********************************************************************************

************
** Variables
************

** BIRTHMO:         Birthdate: Month
** BIRTHYR:         Birthdate: Year
** KNOWNDECEASEDMO: Known Deceased - Month
** KNOWNDECEASEDYR: Known Deceased - Year

local VARS ///
	HHID PN ?SUBHH ?IWTYPE ?WGTR ?WGTRNH BIRTHMO BIRTHYR KNOWNDECEASEDMO KNOWNDECEASEDYR

use `VARS' using ${tracker}, clear

renvars *MO *YR, l

drop A* B* C* D* E*
// -> 1992, 1993, 1994, 1995, 1996

foreach var in IWTYPE SUBHH WGTR WGTRNH {
  cap rename F`var' `var'4 
  cap rename G`var' `var'5 
  cap rename H`var' `var'6 
  cap rename J`var' `var'7 
  cap rename K`var' `var'8 
  cap rename L`var' `var'9 
  cap rename M`var' `var'10 
  cap rename N`var' `var'11 
  cap rename O`var' `var'12 
}

reshape long SUBHH@ IWTYPE@ WGTR@ WGTRNH@ , i(HHID PN) j(w)

lab def IWTYPE ///
1 "1.  Core interview obtained" ///
5 "5.  Core interview not obtained" ///
11 "11.  Exit interview obtained" ///
15 "15.  Exit interview not obtained" ///
21 "21.  Post-exit interview obtained" ///
25 "25.  Post-exit interview not obtained" ///
99 "99.  Not in the sample this wave"
lab val IWTYPE IWTYPE

lab def SUBHH ///
0 "0.  Original household" ///
1 "1.  Sub-household, split off from original" ///
2 "2.  Sub-household, split off from original" ///
3 "3.  Deceased respondent household" ///
4 "4.  Deceased respondent household" ///
5 "5.  Sub-household, split off a household that already split into a 1 and 2" ///
6 "6.  Sub-household, split off a household that already split into a 1 and 2" ///
7 "7.  Used when two respondents split and then recombine with each other" ///
8 "8.  Sub-household, split off a household that already split into a 1 and 2" ///
9 "9.  Not in the sample this wave or respondent not interviewed this wave"
destring SUBHH, gen(SUBHHnum)
lab val SUBHHnum SUBHH

* birthdate
replace birthmo=. if inlist(birthmo,99,98)     // set DK values to missing 
replace birthyr=. if inlist(birthyr,9998,9999) // set DK values to missing 
gen birthdate = mdy(birthmo,15,birthyr)		    
// date of birth in Stata format, (days since 1 Jan 1960). (15 is the assumed day)
format birthdate %td				             
label var birthdate "DATE OF BIRTH"

* deathdate
replace knowndeceasedmo=. if inlist(knowndeceasedmo,99,98)     // set DK values to missing 
replace knowndeceasedyr=. if inlist(knowndeceasedyr,9998,9999) // set DK values to missing 
gen deathdate = mdy(knowndeceasedmo,15,knowndeceasedyr)		    
// date of death in Stata format, (days since 1 Jan 1960). (15 is the assumed day)
format deathdate %td				             
label var deathdate "DATE OF DEATH (KNOWNDECEASEDMO,YR)"

* NOTE: Appears that the KNOWNDECEASED* variables supercede the EXDEATH* variables.
*       Whenever both are non-missing, the values are equal. And, when EXDEATH* is non-missing,
*       KNOWNDECEASED* is also non-missing. Can use just the KNOWNDECEASED* variables.

save ${save}/transfers_tracker, replace 

********************************************************************************
** CORE: RAND LONGITUDINAL FILE 
********************************************************************************

clear

local V1 hhidpn   rahhidpn ragender ///
         inw* h*hhidc  r*wtresp r*wtr_nh r*wthh ///
		     r*wtcrnh r*iwstat r*iwmid  ///
		     h*pickhh ///
				 h*afhous ///
         h*atotb  ///
				 h*itot   ///
		     r*agey_b h*cpl h*child  s*agey_b ///
		     r*nhmliv

local V2 inw@ h@hhidc  r@wtresp r@wtr_nh r@wthh ///
         r@wtcrnh r@iwstat r@iwmid  ///
		     h@pickhh ///
				 h@afhous ///
         h@atotb  ///
				 h@itot   ///
		     r@agey_b h@cpl h@child  s@agey_b  ///
		     r@nhmliv 
                 
use `V1' using ${randhrs}, clear
qui: reshape long `V2', i(hhidpn) j(w)

gen HHID      = substr(rahhidpn ,1,6)
gen PN        = substr(rahhidpn ,7,3)
gen SUBHH     = substr(hhhidc   ,7,1)   // extract sub-household id
sort HHID PN w

gen rwtall=rwtresp
replace rwtall=rwtr_nh if rnhmliv==1 // assign NH weights
// -> now provided by RAND (r*wtcrnh) 

* interview date, and previous interview date
ren riwmid iwdate
by HHID PN: gen pwiwdate = iwdate[_n-1]
by HHID PN: carryforward pwiwdate, gen(cpwiwdate)
format iwdate pwiwdate cpwiwdate %td

* carryforward individual weights, for decedents
by HHID PN: carryforward   rwtall, ///
					          gen( cfrwtall  )
	
* indicator for homeownership
recode hafhous (1/5=1) (6=0) (7/9=.m), gen(rownhm)

keep if w>=4  // begin with year 1998

sort HHID PN w
order HHID PN w 

unique HHID PN w  // #337,455 (INCLUDES waves 11 and 12)

save ${save}/transfers_randhrs, replace 

********************************************************************************
** CORE: RAND Family File (R File)
********************************************************************************

* Respondent level data (includes those without kids).
* Data from core IWs only.
* Uniquely ID'd by HHID PN w.

clear

local V1 hhidpn     rahhidpn    ///
         h*nkid     ///
         h*tcany h*tcamt h*tcflg
		 
local V2 ///
         h@nkid     ///
         h@tcany h@tcamt h@tcflg

use `V1' using ${randfamr}, clear
qui: reshape long `V2', i(hhidpn) j(w)

gen HHID = substr(rahhidpn ,1,6)
gen PN   = substr(rahhidpn ,7,3)

keep if w>=4   // begin with year 1998

sort HHID PN w
order HHID PN w

unique HHID PN w // #337,455 (INCLUDES waves 11 and 12)
 
save ${save}/transfers_randfamr, replace 

********************************************************************************
** MERGE TOGETHER FILES PREPARED ABOVE
********************************************************************************

use ${save}/transfers_tracker, clear

merge 1:1 HHID PN w using ${save}/transfers_randhrs
tab w _merge
drop if (_merge == 1)
drop _merge

merge 1:1 HHID PN w using ${save}/transfers_randfamr   , nogen
merge m:1         w using ${save}/CPIdeflator, keep(3) nogen keepusing(deflator10)

renvars HHID PN, l

********************************************************************************
** MERGE IN DATA PREPARED WITH Decedents_GetSample AND Decedents_ProcessData
********************************************************************************
* Fill in 2004-2012 estate data from single decedents dataset (sample_single)

merge 1:1 hhid pn w using ${save}/decedent_sample_single, ///
  gen(sample) ///
	keepusing(estVal10 valf xIW xnriw A019 A101 xsingle nochild)

* A019: age
* A101: number of children	

********************************************************************************
** Adjust nominal values
********************************************************************************

// convert nominal values to constant 2010 dollars
gen atotb10 = hatotb/deflator10 // net value of total wealth
gen tcamt10 = htcamt/deflator10 // transfers to children
gen itot10 = hitot/deflator10 // household income

* carryforward assets for decedents, to use for wealth thresholds
carryforward atotb10, gen(cfatotb10)
// in fact, we don't use this now. for exit interviews, use estate value.
// for core interviews, use current wealth.

********************************************************************************
** Checking age at death
********************************************************************************

* Check death date from tracker
count if (xIW==1) // -> 3,227 (single decedents, exit waves 7-11)
count if (xIW==1) & (nochild==0) // -> 2,869 (single decedent parents -- our sample for bequest data)
count if (xIW==1) & (nochild==0) & mi(deathdate) // -> 17 (missing deathdate)
count if (xIW==1) & (nochild==0) & ~mi(deathdate,cpwiwdate) & deathdate<cpwiwdate // -> 0 (death before last iw)

* If missing death date, use previous iw date
replace deathdate = cpwiwdate if (xIW==1) & mi(deathdate) & ~mi(cpwiwdate)
// -> 31 cases (17 parents, 14 childless)

* If death date precedes previous interview date, use previous iw date
replace deathdate = cpwiwdate if (xIW==1) & ~mi(deathdate) & ~mi(cpwiwdate) & (deathdate<cpwiwdate)
// -> 0 cases

* Age at death
gen xage = deathdate - birthdate
// -> in days
replace xage = floor(xage/365)
// -> switch to years

tabstat xage A019 if xIW==1, s(n mean min p5 p10 p25 p50 p75 p90 p95 p99 max)
// -> These measures are almost identical. 
//    Might as well just use A019 (as we have been doing elsewhere).

drop xage birthdate deathdate birthmo birthyr knowndeceasedyr knowndeceasedmo

********************************************************************************
** Find age of eldest in household
********************************************************************************

egen ageEldest = rowmax(ragey_b sagey_b)
// -> pick oldest age in the household

replace ageEldest = A019 if (xIW==1)
// -> obtain age for decedents from exit interview
//    (decedents are single, no need to consider sp/p age)

* Age bins (for an average dollar transferred histogram)
gen ageEldestCat = irecode(ageEldest,69,74,79,84,89) // 5-year age bins
lab def AGECAT 0 "65-69" 1 "70-74" 2 "75-79" 3 "80-84" 4 "85-89" 5 "90+"
lab val ageEldestCat AGECAT	
lab var ageEldestCat "Age of eldest household member (5-year bins)"	

********************************************************************************
** Dummy to identify (single) decedents (those who appear in our decedent sample only)
********************************************************************************

gen decedent = .
replace decedent = 0 if (riwstat==1)
replace decedent = 1 if (xIW==1    )
// -> = 0 if alive at interview, responds to survey
//    = 1 if part of our sample of single decedents with exit interviews in waves 7-11
//    = . neither alive/respondent or exit interview

tab w decedent [aw=cfrwtall], row nof
// -> shows % of respondents in each wave that have an exit interview and are
//    members of our single decent sample who gave exit interviews in waves 7-11
/*
           |       decedent
     w |         0          1 |     Total
-----------+----------------------+----------
         4 |    100.00       0.00 |    100.00 
         5 |    100.00       0.00 |    100.00 
         6 |    100.00       0.00 |    100.00 
         7 |     97.44       2.56 |    100.00 
         8 |     97.41       2.59 |    100.00 
         9 |     96.73       3.27 |    100.00 
        10 |     97.38       2.62 |    100.00 
        11 |     97.90       2.10 |    100.00 
        12 |    100.00       0.00 |    100.00 
-----------+----------------------+----------
     Total |     98.43       1.57 |    100.00 
*/

** How well do exit interviews line up with the time of death?

sort hhid pn
by hhid pn: egen dead = max(xIW==1)
// -> = 1 if member of decedent sample
//    = 0 otherwise 
by hhid pn: egen exitw = max(cond(xIW==1,w,.))
// -> = wave of exit interview
by hhid pn: egen deadw = max(cond(riwstat==5,w,.))
replace deadw = . if (dead~=1)
// -> = wave of death
//      (set to missing for those not part of decedent sample)

tab deadw exitw if (xIW==1)
/*
           |                         exitw
     deadw |         7          8          9         10         11 |     Total
-----------+-------------------------------------------------------+----------
         5 |         1          0          0          0          0 |         1 
         6 |         9          2          0          0          0 |        11 
         7 |       619          8          1          0          0 |       628 
         8 |         0        604         23          5          0 |       632 
         9 |         0          0        680          3          0 |       683 
        10 |         0          0          0        717         38 |       755 
        11 |         0          0          0          0        517 |       517 
-----------+-------------------------------------------------------+----------
     Total |       629        614        704        725        555 |     3,227 
*/
tab deadw exitw if (xIW==1), row nof
/*
           |                         exitw
     deadw |         7          8          9         10         11 |     Total
-----------+-------------------------------------------------------+----------
         5 |    100.00       0.00       0.00       0.00       0.00 |    100.00 
         6 |     81.82      18.18       0.00       0.00       0.00 |    100.00 
         7 |     98.57       1.27       0.16       0.00       0.00 |    100.00 
         8 |      0.00      95.57       3.64       0.79       0.00 |    100.00 
         9 |      0.00       0.00      99.56       0.44       0.00 |    100.00 
        10 |      0.00       0.00       0.00      94.97       5.03 |    100.00 
        11 |      0.00       0.00       0.00       0.00     100.00 |    100.00 
-----------+-------------------------------------------------------+----------
     Total |     19.49      19.03      21.82      22.47      17.20 |    100.00 
*/
// -> ~95%+ of those who die have exit interview in the wave in which they die. 

drop dead deadw exitw

********************************************************************************
** Note about RAND's H*PICKHH and decedents
********************************************************************************

tab hcpl hpickhh,m
/*
                  |        hpickhh
             hcpl | 0.drop fo  1.select  |     Total
------------------+----------------------+----------
0.not a couple hh |         0     61,512 |    61,512 
      1.couple hh |    55,336     59,430 |   114,766 
                . |   161,177          0 |   161,177 
------------------+----------------------+----------
            Total |   216,513    120,942 |   337,455
*/
// -> all singles are chosen (obviously)
//    interestingly, there are a different number of individuals with hpickhh=0
//    and hpickhh=1 among couples. 

tab hpickhh decedent,m
// -> our single decedents are (obviously) never selected
/*
                      |             decedent
              hpickhh |         0          1          . |     Total
----------------------+---------------------------------+----------
0.drop for hh-level f |    55,336      3,227    157,950 |   216,513 
1.select for hh-level |   120,942          0          0 |   120,942 
----------------------+---------------------------------+----------
                Total |   176,278      3,227    157,950 |   337,455 
*/

* In our analysis, we will select individuals for whom either hpickhh==1 or decedent==1.

********************************************************************************
** Save
********************************************************************************

save ${save}/transfers_sample, replace

********************************************************************************
** Erase temporary files
********************************************************************************

erase ${save}/transfers_tracker.dta
erase ${save}/transfers_randhrs.dta
erase ${save}/transfers_randfamr.dta

********************************************************************************
