/*******************************************************************************

************
** Function: 
************

  Computes bequest and IVT flows over a two-year period. Produces a transfer histogram
	for the text and statistics cited in the text on IVT-bequest ratios and the age at
	which the average dollar is transferred.

************
** Inputs  :
************
	
	- transfers_sample.dta (see Transfer_GetSample.do)
	
************
** Outputs : 
************
	
	- IVT_BEQUEST_RATIOS.txt
	- AGE_AVERAGE_DOLLAR_GIVEN.txt
	
	- FIG5_and_FIG12a.eps
	- FIGG1a.eps
	- FIGG1b.eps
	- FIGG1c.eps
	- FIGG1d.eps

*******************************************************************************/

clear *
macro drop _all
set more off 

set scheme hack, permanently

********************************************************************************
** Directory names
********************************************************************************

do GetDirNames.do

********************************************************************************
** DATA
********************************************************************************

use ${save}/transfers_sample, clear

********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************

********************************************************************************
** IVT-to-Bequest Ratios
********************************************************************************
/*

To compute IVT-to-bequest ratios:

    In all cases, we use individual-level sample weights.
		
    We compute adjusted weights to correct for having different numbers of core and
    exit interviews. Suppose the correct weight ratio of exit to core weights is
		given by (sum of exit weights 2004-2010) / (sum of core weights 2004-2010),
		i.e., the ratio of the sums of weights in the overlap years. Call this ratio
		'A'. Define the ratio 'B' as (sum exit weights 2004-2012) / (sum core weights 1998-2010).
		Expect that B < A because we have 7 core and 5 exit interviews. We want to 
		find an adjustment factor 'c' such that c*B = A, or c = A/B. If we adjust
		upward by the factor 'c' the exit interview weights, then we correct for 
		having different numbers of core/exit interviews. We can compute the IVT to
		bequest ratio as (7-wave sum of IVT) / (re-weighted 5-wave sum of bequests).
		This approach uses all of the data and easily extends to calculating the age
		at which the average dollar is transferred. 
				
*/
********************************************************************************
** Key variables defined
********************************************************************************

* Impose age 65+ restriction for both core/exit interviews to be consistent with model.

* IVT: Core interviews. All households whose eldest member is 65+. Respondent-level
*      weights, corrected for NH residents. Select one member per household.
*      Average wave-by-wave sums across core interviews 4-10 (years 1998-2010).

* BEQ: Exit interviews. All single decedents in our sample, gave exit interviews
*      in waves 7-11, single at time of death, dropping those with bad data.
*      Include only decedents with children (nochild is A101, corrected for cases 
*      where proxy is child)
*      Average wave-by-wave sums across exit interviews 7-11 (years 2004-2012).

gen ivt = tcamt10 if (hpickhh==1 ///
								    & inrange(w,4,10) ///
										& (ageEldest >= 65 & ageEldest < . ))

gen beq = estVal10 if (nochild==0) & (A019 >= 65 & A019 < .)
// A019 is age calculation taken from exit interview

egen xfr = rowtotal(ivt beq), m
// -> total transfer including inter-vivos transfers and bequests

gen wgt = .
replace wgt =   rwtall if (riwstat==1)
replace wgt = cfrwtall if (xIW==1)     // -> use carried forward weight for decedents only.

* Re-weighting scheme
* (upweight exit interviews b/c we have 5 exit iws and 7 core iws)
* A = (sum of exit weights 2004-2010) / (sum of core weights 2004-2010)
qui summ wgt if inrange(w,7,10) & (xIW==1)
local numA = r(sum)
qui summ wgt if inrange(w,7,10) & (riwstat==1)
local denA = r(sum)
local A = `numA' / `denA'

* B = (sum of exit weights 2004-2012) / (sum of core weights 1998-2010)
qui summ wgt if inrange(w,7,11) & (xIW==1)
local numB = r(sum)
qui summ wgt if inrange(w,4,10) & (riwstat==1)
local denB = r(sum)
local B = `numB' / `denB'

di "ratio A = " `A'
di "ratio B = " `B'
/*
ratio A = .02823202
ratio B = .0213956

B < A, as expected b/c more core than exit interviews
*/

* Find adjustment factor c such that c*B = A, or c = A/B.
local c = `A' / `B'
di "adjustment factor c = " `c'
/*
adjustment factor c = 1.3195247

for comparison, 7/5 = 1.4
(crude adjustment: inflate exit weights by 7/5 since we have 7 core, 5 exit interviews.)
*/

gen wgtadj = wgt
replace wgtadj = wgt * `c' if (xIW==1)
// NOTE: re-weight exit interviews (bequests) only.

*Check calculations
qui summ wgtadj if inrange(w,7,11) & (xIW==1)
local numC = r(sum)
qui summ wgtadj if inrange(w,4,10) & (riwstat==1)
local denC = r(sum)
local C = `numC' / `denC'

di "ratio C = " `C'

/*
ratio C = .02823202

compare to our target (above):

ratio A = .02823202
*/

* Modified weights for calculating age at which average transfer is given
* Unadjusted weights
gen wgtXivt = wgt * ivt
gen wgtXbeq = wgt * beq
gen wgtXxfr = wgt * xfr
* Adjusted weights
gen wgtadjXivt = wgtadj * ivt
gen wgtadjXbeq = wgtadj * beq
gen wgtadjXxfr = wgtadj * xfr

********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************

********************************************************************************
** IVT-Bequest Ratio and Sensitivity to Excluding Top of Wealth Distribution
********************************************************************************

* To exclude the top wealth percentiles, we drop individuals whose current wealth (core 
* interviews) or estate value (exit interviews) is above some threshold.

* find wealth percentiles in core interviews 1998-2010 (all respondents, not just elderly)
* (hrs only approx representative up to 95th percentile)
* (note: results for ages 65+ are very similar)
tabstat atotb10 [aw=rwtall] if (hpickhh==1 & inrange(w,4,10)) ///
	, s(p95) c(v) f(%12.0fc)		
	
/*
   stats |   atotb10
---------+----------
     p95 |   1,684,282
--------------------
*/

* input vector of wealth thresholds:
local THRESHOLDS 5.0e5   1.0e6  1.684e6   5.0e6   10e6      .
//                                 p95                    \infty (no threshold)

* variables to store thresholds and ivt/bequest ratios
* (needed only to make nice table output)
cap drop threshold
cap drop ratio

gen threshold = .
gen ratio = .

local i = 0
foreach THRESHOLD of local THRESHOLDS {

  local i = `i' + 1
  
  qui replace threshold = `THRESHOLD' in `i'
  
	qui: summ ivt [aw=wgtadj] if (atotb10 < `THRESHOLD')    // -> adjusted weights
	local ivtsum = r(sum)

	qui: summ beq [aw=wgtadj] if (beq < `THRESHOLD')  // -> adjusted weights
	local beqsum = r(sum)           // -> use estate value for decedents	
	
	* Ratio of IVT / BEQ
	local ratio = `ivtsum' / `beqsum'
	
	di "Threshold: " %10.0fc `THRESHOLD' ". IVT / Bequest ratio: " %11.10fc `ratio'
	
	qui replace ratio = `ratio' in `i'

}

format threshold %10.0fc
format ratio %4.3fc

* turn on log
cap log close
log using ${logs}/IVT_BEQUEST_RATIOS.txt, text replace

list threshold ratio in 1/6, clean noobs

/*
Result:

As a table, from the list command:

     threshold   ratio  
       500,000   0.346      // THESE ARE THE RESULTS APPEARING
     1,000,000   0.312      // IN APPENDIX TABLE G.3.
     1,684,000   0.299      // THE 0.299 FIGURE (THIS ROW) IS THE
     5,000,000   0.310      // RATIO CITED IN THE TEXT.
    10,000,000   0.329  
             .   0.266  
				
*/

* close log
cap log close

drop threshold ratio

********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************

********************************************************************************
** At what age is the average dollar given?
********************************************************************************
/*
To compute age at which average dollar is transferred, we calculate a weighted
average of age (eldest age in the household, or age at death) where the weights
are the product of the sample weight * the magnitude of the transfer. 
*/

** NOTE: Excluding the top 5 percent of wealth distribution

preserve

local THRESHOLD 1.684e6
// -> 95th percentile of wealth, core 1998-2010

drop if (riwstat==1 & atotb10 >= `THRESHOLD') | (xIW==1 & beq >= `THRESHOLD')
// -> use estate value for decedents

* Inter-vivos transfers
summ ageEldest [aw=wgtadjXivt]
local ivt = r(mean)

* Bequests
summ ageEldest [aw=wgtadjXbeq]
local beq = r(mean)

* Total transfers
summ ageEldest [aw=wgtadjXxfr]
local xfr = r(mean)

* turn on log
cap log close
log using ${logs}/AGE_AVERAGE_DOLLAR_GIVEN.txt, text replace

di          "Age at which average IVT given: " %3.1f `ivt' ///
   _newline "Age at which average BEQ given: " %3.1f `beq' ///
	 _newline "Age at which average XFR given: " %3.1f `xfr'

/*

Age at which average IVT given: 75.6
Age at which average BEQ given: 85.5
Age at which average XFR given: 83.2

*/

* close log
cap log close

restore

********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************

********************************************************************************
** Histograms of dollars transferred by age
********************************************************************************

** NOTE: Excluding the top 5 percent of wealth distribution

local THRESHOLD 1.684e6
// -> 95th percentile of wealth, core 1998-2010

preserve

drop if (riwstat==1 & atotb10 >= `THRESHOLD') | (xIW==1 & beq >= `THRESHOLD')
// -> drop the wealthy. for decedents, use estate value.

keep if (ageEldest >= 65 & ageEldest < .)
// -> age eligible

* For checks:
qui: summ ivt [aw=wgtadj]
local ivtsum = r(sum)
qui: summ beq [aw=wgtadj]
local beqsum = r(sum)

* Ratio of IVT / BEQ (check that is the same as above)
local ratio = `ivtsum' / `beqsum'
di "IVT / Bequest ratio: " %11.10fc `ratio'

* IVT share of all transfers (compare with output below) // -> Result: 0.2301607782
local share = `ivtsum' / (`ivtsum' + `beqsum')
di "IVT share of transfers: " %11.10fc `share'

** Among all dollars given as IVT, what is distribution across ages
tab ageEldestCat [aw=wgtadjXivt]

** Among all dollars given as bequests, what is distribution across ages                                                     
tab ageEldestCat [aw=wgtadjXbeq]

** Among all dollars given, what is distribution across ages                                                     
tab ageEldestCat [aw=wgtadjXxfr]

** Data for the graphs
collapse (sum) wgtadjXivt wgtadjXbeq wgtadjXxfr, by(ageEldestCat)									 
							
ren wgtadjXivt ivt
ren wgtadjXbeq beq
ren wgtadjXxfr xfr							
							
egen ivtTot = total(ivt)
gen ivtHist = ivt/ivtTot        // -> compare to tab above
egen beqTot = total(beq)
gen beqHist = beq/beqTot        // -> compare to tab above
egen xfrTot = total(xfr)
gen xfrHist = xfr/xfrTot        // -> compare to tab above
list ageEldestCat *Hist, sep(0)

* Shares
gen ivtShr = ivt/xfrTot  // -> ivt / total transfers
gen beqShr = beq/xfrTot  // -> beq / total transfers
gen xfrShr = xfr/xfrTot

list ageEldestCat *Shr, sep(0) clean noobs
/*
    ageEld~t     ivtShr     beqShr     xfrShr  
       65-69   .0627396   .0267735   .0895131  
       70-74    .049714   .0576484   .1073624  
       75-79   .0484184   .0880564   .1364748  
       80-84   .0338486   .1298947   .1637433  
       85-89   .0241357   .2086674   .2328031  
         90+   .0113044   .2587988   .2701032 
*/

* Check math
egen ivtShrSum = total(ivtShr)
egen beqShrSum = total(beqShr)
gen checkSum = ivtShrSum + beqShrSum
egen xfrShrSum = total(xfrShr)
list *Sum in 1, sep(0) clean noobs
/*
    ivtShr~m   beqShr~m   checkSum   xfrShr~m  
    .2301608   .7698392   .9999999   .9999999 
*/

* IVT/bequest ratio:
gen ivtBeqRat = ivtShrSum / beqShrSum
list ivtBeqRat in 1, sep(0) clean noobs
/*
    ivtBeq~t  
    .2989725 
*/

** Stacked bar (actually graph one bar (ivt) in front of the total (xfr))
gr tw ///
(bar xfrShr ageEldestCat, barwidth(0.5) ) ///
(bar ivtShr ageEldestCat, barwidth(0.5) ) ///
, legend(order(1 "Bequests" 2 "Inter-vivos transfers") rows(1)) ///
  ytitle("Fraction of all dollars transferred" "by parent households ages 65+") ///
	xtitle("Age of eldest household member (5-year bins)") ///
	xlab(0 "65-69" 1 "70-74" 2 "75-79" 3 "80-84" 4 "85-89" 5 "90+") ///
	ylab(0(.05).3, angle(0) gmax)

graph2tex, epsfile(${figs}/FIG5_and_FIG12a) 
	
restore

********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************
********************************************************************************

********************************************************************************
** Boxplots of dollars transferred by age categories
********************************************************************************

***************************************
********** WITHOUT wealth restriction:
***************************************

preserve

keep if (ageEldest >= 65 & ageEldest < .)
// -> age eligible

* simple summaries
summ ivt [aw=wgtadj] if (ivt>0), detail
summ beq [aw=wgtadj] if (beq>0), detail

* boxplots

gr box ivt [aw=wgtadj] if (ivt>0), over(ageEldestCat) yscale(log) ///
    ylabel(1e2 1e3 5e3 1e4 1e5 1e6 2e6,angle(0)) ///
		ytitle("Inter-vivos transfer amounts (if non-zero)" "by parent households ages 65+") ///
		name(figg1a)

graph2tex, epsfile(${figs}/FIGG1a)
		
gr box beq [aw=wgtadj] if (beq>0), over(ageEldestCat) yscale(log) ///
    ylabel(1e2 1e3 1e4 1e5 1e6 1e7 1e8,angle(0)) ///
		ytitle("Bequest amounts (if non-zero)" "by parent households ages 65+") ///
		name(figg1c)	

graph2tex, epsfile(${figs}/FIGG1c)		
		
restore

***************************************
********** WITH wealth restriction:
***************************************

local THRESHOLD 1.684e6
// -> 95th percentile of wealth, core 1998-2010

preserve

drop if (riwstat==1 & atotb10 >= `THRESHOLD') | (xIW==1 & beq >= `THRESHOLD')
// -> drop the wealthy. for decedents, use estate value.

keep if (ageEldest >= 65 & ageEldest < .)
// -> age eligible

* simple summaries
summ ivt [aw=wgtadj] if (ivt>0), detail
summ beq [aw=wgtadj] if (beq>0), detail

* boxplots

gr box ivt [aw=wgtadj] if (ivt>0), over(ageEldestCat) yscale(log) ///
    ylabel(1e2 1e3 5e3 1e4 1e5 1e6 2e6,angle(0)) ///
		ytitle("Inter-vivos transfer amounts (if non-zero)" "by parent households ages 65+") ///
		name(figg1b)

graph2tex, epsfile(${figs}/FIGG1b)		
		
gr box beq [aw=wgtadj] if (beq>0), over(ageEldestCat) yscale(log) ///
    ylabel(1e2 1e3 1e4 1e5 1e6 1e7 1e8,angle(0)) ///
		ytitle("Bequest amounts (if non-zero)" "by parent households ages 65+") ///
		name(figg1d)	

graph2tex, epsfile(${figs}/FIGG1d)		
		
restore

********************************************************************************
